# Makefile.PL #

use 5.006;
use strict;
use autodie ':all';
use warnings FATAL => 'all';

use Inline::MakeMaker;
use Getopt::Long;

my $name     = 'Systemd-Daemon';
my $version  = '0.01';
my $release  = '1';
my $abstract = 'Write systemd-aware daemons in Perl';
my $devel;

# ==================================================================================================
# Script body.
# ==================================================================================================

{
    local @ARGV = @ARGV;
    local $SIG{ __WARN__ } = sub { die @_; };
    GetOptions(
        'devel' => \$devel,
    );
}

# TODO: What if rpmbuild is not found?
my $rpmbuild = ( $ENV{ RPMBUILD } // 'rpmbuild' );
my $arch = qx{ $rpmbuild --eval '%{_arch}' };
my $dist = qx{ $rpmbuild --eval '%{dist}'  };
chomp( $arch, $dist );

WriteMakefile(
    NAME                => 'Systemd::Daemon',
    VERSION             => $version,
    LICENSE             => 'gpl_3',
    ABSTRACT            => $abstract,
    AUTHOR              => 'Van de Bugger <van.de.bugger@liamg.moc>',
    META_MERGE          => {
        release_status  => 'testing',
        no_index            => {
            directory           => [ 'bin', ],
        },
    },
    PM_FILTER           => "$^X -p -e \"s{__VERSION__}{$version}g; s{__ABSTRACT__}{$abstract}g;\"",
    CONFIGURE_REQUIRES  =>  {
        'autodie'               => '0',
        'CPAN::Meta'            => '0',
        'ExtUtils::MakeMaker'   => '6.52',
        'Getopt::Long'          => '0',
        'Inline::MakeMaker'     => '0.45',
        'IPC::System::Simple'   => '0',     # Required by autodie ':all'.
        'Path::Class'           => '0',
        'strict'                => '0',
        'warnings'              => '0',
    },
    BUILD_REQUIRES      => {
        'File::pushd'           => '1.009',
        'Inline'                => '0',
        'Inline::C'             => '0',
        'Inline::Filters'       => '0',     # POD is used in __DATA__.
    },
    TEST_REQUIRES       => {
        'autodie'               => '0',
        'File::Basename'        => '0',
        'File::Temp'            => '0',
        'FindBin'               => '0',
        'Getopt::Long'          => '0',
        'IPC::System::Simple'   => '0',     # Required by autodie ':all'.
        'Test::More'            => '0',
    },
    PREREQ_PM           => {
        'Exporter'              => '5.57',  # use Exporter qw{ import };
        'Inline'                => '0',
        'Inline::C'             => '0',
        'Inline::Filters'       => '0',     # POD is used in __DATA__.
        'strict'                => '0',
        'warnings'              => '0',
    },
    dist                => {
        COMPRESS    => 'gzip -f --best',
        SUFFIX      => '.gz',
    },
    clean               => {    # Temporary and intermediate files.
        FILES => [ '_Inline', '_rpmbuild', "$name-$version", "$name.inl" ],
    },
    realclean           => {    # Output files,
        FILES => [ "$name-*.tar.gz", "perl-$name-*.rpm" ],
    },
);

exit( 0 );

# ==================================================================================================
# Package MY — MakeMaker customizations.
# ==================================================================================================

{ package MY;

use Path::Class;

our $Text;


# --------------------------------------------------------------------------------------------------
# Some helper methods.
# --------------------------------------------------------------------------------------------------


#  Assemble a makefile rule from target, prerequisites, and commands.
sub rule($$@) {
    my ( $target, $prereqs, @commands ) = @_;
    my $text =
        $target . ' : ' . ( ref( $prereqs ) eq 'ARRAY' ? join( ' ', @$prereqs ) : $prereqs )  . "\n" .
        join( '', map( "\t$_\n", @commands ) );
    return $text;
}; # sub rule


sub add_rule($$@) {
    $Text .= &rule( @_ );
}; # sub add_rule


sub del_rule($) {
    my ( $target ) = @_;
    $Text =~ s{
        ^ $target \s* : .*? ( \\ \n .*? ) * \n
        ( \t .*? \n ) *
    }{}mx;
}; # sub del_rule


sub fix_rule($$@) {
    del_rule( $_[ 0 ] );
    &add_rule( @_ );
}; # sub fix_rule


# --------------------------------------------------------------------------------------------------
# MakeMaker sections customization.
# --------------------------------------------------------------------------------------------------

#   Customization methods should start with `my_` and/or `My_' prefix. When a method is invoked,
#   variable $Text is preloaded with original section text. The method may change the variable
#   value: append some more text to the end, change it, fully replace, or leave it as is.
#   Methods with `my_` prefix are called unconditionally, methods with `My_` prefix are called
#   only if the script is invoked with `--devel` option. If both methods are defined for the same
#   section (for example, `my_constants` and `My_constants`), `my_` method is callled first.
#
#   Thus, `my_` methods should be used for simple and safe changes, while tricky and potentially
#   unsafe customizations (e. g. customizations which rely on MakeMaker internals) should be done
#   in `My_` methods.


#   post_initialize
#   const_config

sub my_constants {
    $Text .= "RELEASE=$release\n";
};


#   platform_constants
#   tool_autosplit
#   tool_xsubpp


sub my_tools_other {
    $Text .=
        "RPMBUILD = rpmbuild\n" .
        "SAY = @\$(ECHO)\n";
};


#   makemakerdflt
#   dist
#   macro
#   depend
#   cflags
#   const_loadlibs
#   const_cccmd
#   post_constants
#   pasthru


sub My_special_targets {
    #   Let GNU make delete target file in case of error.
    #   TODO: Add it if make is GNU Make.
    add_rule( '.DELETE_ON_ERROR', '' );
};


#   c_o
#   xs_c
#   xs_o


sub my_top_targets {
    #   `help` target generated by MakeMaker shows help on MakeMaker itself. It is very long
    #   and not useful for a user who just wants to build the module.
    fix_rule( 'help', '',
        '$(SAY) ""',
        '$(SAY) "Targets:"',
        '$(SAY) "    Building:"',
        '$(SAY) "        all       -- build everything"',
        '$(SAY) "   Testing:"',
        '$(SAY) "        test      -- run all the tests"',
        '$(SAY) "        distcheck -- check manifest"',
        #~ '$(SAY) "        disttest  -- "',
        '$(SAY) "   Packaging:"',
        '$(SAY) "        dist      -- build tarball package"',
        ( $devel ? (
        '$(SAY) "        rpm       -- build rpm packages"',
        ) : () ),
        '$(SAY) "   Cleaning:"',
        '$(SAY) "        clean     -- delete temp files"',
        '$(SAY) "        realclean -- clean then delete packages"',
        '$(SAY) "        veryclean -- realclean then delete backup files"',
        '$(SAY) "        distclean -- realclean then distcheck"',
    );
};


#   blibdirs
#   linkext
#   dlsyms
#   dynamic_bs
#   dynamic
#   dynamic_lib
#   static
#   static_lib


sub my_manifypods {
    #   MakeMaker makes man pages from original pm modules located uder `lib/`
    #   directory. It is wrong, because original modules are not yet filtered
    #   and contain `__ABSTRACT__` and `__VERSION__` placeholders. We should use
    #   modules located under `blib/' directory.
    #   This change is a bit tricky, but we cannot perform it only in "devel"
    #   version of makefile because it will cause invalid man pages.
    $Text =~ s{(\s)(lib/\S*?\.pm)}{$1blib/$2}mgx;
};


#   processPL
#   installbin
#   subdirs
#   clean_subdirs
#   clean
#   realclean_subdirs
#   realclean


sub My_metafile {
    $Text =~ s{
        ^ metafile \s* : \s* create_distdir \s* \n
    }{metafile : \$(DISTVNAME)/META.yml \$(DISTVNAME)/META.json\n}mx;
    $Text =~ s{
        ( ^ \t \Q$(NOECHO) $(ECHO) Generating META.\E(yml|json) \n )
    }{\$(DISTVNAME)/META.$2 : \$(FIRSTMAKE)\n$1}gmx;
    $Text =~ s{^(\t)-(\Q$(NOECHO) $(MV)\E)}{$1$2}mg;
    $Text =~ s{META_new\.(yml|json)}{META.$1.tmp}g;
};


#   signature


sub My_dist_basics {
    # Delete `manifest` target — our `MANIFEST` is hand-made.
    del_rule( 'manifest' );
};


sub My_dist_core {
    # Do not delete directory.
    $Text =~ s{
        ^ \t \Q$(RM_RF) $(DISTVNAME)\E \n
    }{}mxg;
};


sub My_distdir {
    #   I do not want to delete and fully recreate distdir every time.
    #   Instead, I want to populate it with changed files and remove any
    #   extra files.
    fix_rule( 'create_distdir', '',
        '$(PERLRUN) bin/create_distdir.pl . $(DISTVNAME) $(DIST_CP)'
    );
};


#   dist_test


sub my_dist_ci {
    #   Disable "ci" target. I do not use archaic RCS.
    $Text = "# Disabled by " . file( __FILE__ )->basename . "\n";
};


sub My_distmeta {
    #   I do not like hidden commands.
    $Text =~ s{\Q$(NOECHO)\E}{}g;
    #   Temporary workaround for bug #103042
    #       https://rt.cpan.org/Ticket/Display.html?id=103042
    $Text =~ s{ or print }{ or die }g;
    $Text =~ s{\${4}}{\$\$}g;
};


#   distsignature
#   install
#   force
#   perldepend


sub My_makefile {

    #   Do not ignore command's exit status.
    $Text =~ s{^\t-}{\t}gm;

    #   Do not redirect command output to `/dev/null`.
    #   `DEV_NULL` macro contains redirection characters,
    #   so we just need to delete all occurenses of `$(DEV_NULL)`.
    $Text =~ s{\Q$(DEV_NULL)\E}{}gm;

    #   Pass Makefile.PL arguments to Makefile.PL called by make.
    $Text =~ s{
        ^ ( \t \Q$(PERLRUN) Makefile.PL\E ) \s* \n
    }{$1 @ARGV\n}mx;

    # GNU make automatically reloads Makefile if it was rebuilt.
    # No need to print a message and stop make. Let it continue.
    # TODO: Do replace if GNU Make.
    $Text =~ s{
        ^ \t \Q$(NOECHO) $(ECHO) "==> Your Makefile has been rebuilt. <=="\E \n
        ^ \t \Q$(NOECHO) $(ECHO) "==> Please rerun the $(MAKE) command.  <=="\E \n
        ^ \t \Q$(FALSE)\E \n
    }{}mx;

};


#   staticmake
#   makeaperl


sub My_test {
    #   I want verbose testing.
    $Text =~ s{^(TEST_VERBOSE=)0$}{${1}1}m;
    #   And release testing too.
    $Text =~ s{
        ^ \t ( PERL_DL_NONLAZY=1 \s+ \Q$(FULLPERLRUN)\E )
    }{\tRELEASE_TESTING=1 $1}gmx;
};


#   test
#   ppd
#   pm_to_blib
#   selfdocument


sub My_postamble {
    $Text .= rpm() . rules();
};


# --------------------------------------------------------------------------------------------------
# Makefile.PL sections.
# --------------------------------------------------------------------------------------------------


sub rules {
    my $text =
        "\n" .
        "# --- " . __FILE__ . " rules section:\n" .
        rule( '%/.dir', [], '$(MKPATH) $(dir $@)', '$(TOUCH) $@' ) .
        "";
    return $text;
};


sub rpm {
    my $D = sub {
        return '-D"' . $_[ 0 ] . ' ' . $_[ 1 ]->absolute . '" ';
    };
    my $dir = dir();
    my $topdir = $dir->subdir( '_rpmbuild' );
    my $srcdir = $dir;
    my $spec   = $topdir->subdir( 'SPECS' )->file( "perl-$name.spec" );
    my $brpm   = "perl-$name-$version-$release.$arch.rpm";
    my $srpm   = "perl-$name-$version-$release.src.rpm";
    my $text =
        "\n" .
        "# --- " . __FILE__ . " rpm section:\n" .
        rule( $spec, [ $dir->subdir( 'rpm' )->file( "perl-$name.spec.in" ), '$(FIRST_MAKEFILE)', $spec->dir->file( '.dir' ) ],
            "sed -e 's/\@VERSION\@/\$(VERSION)/g' -e 's/\@RELEASE\@/\$(RELEASE)/g' \$< > \$@"
        ) .
        rule( $srpm, [ $spec, 'dist' ],
            '$(RPMBUILD) ' . $D->( '_topdir', $topdir ) . $D->( '_sourcedir', $srcdir ) . '-bs $<',
            '$(CP) ' . $topdir->subdir( 'SRPMS' )->file( $srpm ) . ' $@'
        ) .
        rule( $brpm, [ $spec, 'dist' ],
            '$(RPMBUILD) ' . $D->( '_topdir', $topdir ) . $D->( '_sourcedir', $srcdir ) . '-bb $<',
            '$(CP) ' . $topdir->subdir( "RPMS/$arch" )->file( $brpm ) . ' $@'
        ) .
        rule( 'rpm', [ $srpm, $brpm ],
            '$(NOECHO) $(NOOP)'
        );
    return $text;
}; # rpm


# --------------------------------------------------------------------------------------------------
# Some magic.
# --------------------------------------------------------------------------------------------------


#   Customizing a make section requires non-trivial work for finding the original section text.
#   This method does all the stuff, so customizing methods (defined above) a free from repeating the
#   same boilerplate code: they just operate on $Text variable.
#
sub section {

    my ( $self, $name ) = @_;

    #   Find out superclass method name.
    #   Some superclass methods named differently, for example: not "matafile"
    #   but "metafile_taget". Handle such a case (MakeMaker does the same).
    my $super = "SUPER::$name";
    if ( not $self->can( $super ) ) {
        $super .= '_target';
    }; # if

    #   Load original section text into $Text variable.
    local $Text = $self->$super();

    #   Call customization method.
    {
        if ( exists( $MY::{ "my_$name" } ) ) {
            no strict 'refs';
            &{ "my_$name" }();
        }; # if
        if ( exists( $MY::{ "My_$name" } ) and $devel ) {
            no strict 'refs';
            &{ "My_$name" }();
        }; # if
    };

    #   Return customized text to MakeMaker.
    return $Text;

}; # sub section


BEGIN {
    my %methods;
    foreach my $symbol ( keys( %MY:: ) ) {
        if ( $symbol =~ m{^[Mm]y_(.*)$} ) {
            if ( not $methods{ lc( $symbol ) } ) {
                eval "sub $1 { section( \$_[ 0 ], \"$1\" ); };";
                $methods{ lc( $symbol ) } = 1;
            }; # if
        }; # if
    }
} # BEGIN


} # package MY

__END__

=comment

Using `Module::Install` looks nice but does not work. `make` finishes with error:

    make: *** No rule to make target 'Daemon.inl', needed by 'pure_all'.  Stop.

This error can be workarounded by moving `Daemon.pm` to the top-level directory, but in such case
`make` each run rebuilds almost everything because `make` does not create `Daemon.inl`. I believe
it is a bug in the `Module::Install::Inline`.

=cut

use 5.006;
use strict;
use warnings FATAL => 'all';
use inc::Module::Install;

name      'Systemd-Daemon';
all_from  'lib/Systemd/Daemon.pm';

configure_requires  'Inline::MakeMaker'     => '0.45',
configure_requires  'ExtUtils::MakeMaker'   => '6.52',

test_requires   'Test::More'    => 0;

requires  'Inline'          => '0.45';
requires  'Inline::C'       => 0;
requires  'Inline::Filters' => 0;

WriteAll( inline => 1 );

# end of file #
