# 

#pod =encoding UTF-8
#pod
#pod =head1 COPYRIGHT AND LICENSE
#pod
#pod Copyright © 2015 Van de Bugger
#pod
#pod This file is part of perl-Software-License-OrLaterPack.
#pod
#pod perl-Software-License-OrLaterPack is free software: you can redistribute it and/or modify it
#pod under the terms of the GNU General Public License as published by the Free Software Foundation,
#pod either version 3 of the License, or (at your option) any later version.
#pod
#pod perl-Software-License-OrLaterPack is distributed in the hope that it will be useful, but
#pod WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
#pod PARTICULAR PURPOSE. See the GNU General Public License for more details.
#pod
#pod You should have received a copy of the GNU General Public License along with
#pod perl-Software-License-OrLaterPack. If not, see <http://www.gnu.org/licenses/>.
#pod
#pod =cut

# --------------------------------------------------------------------------------------------------

#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt
#tt


#pod =for :this This is C<Software::License::LGPL_3_0::or_later> module documentation. Read this if you
#pod are going to hack or extend the module, or use the module directly.
#pod
#pod =for :those If you want to use GNU license with "or later" clause read the L<user
#pod manual|Software::License::OrLaterPack>. General topics like getting source, building, installing,
#pod bug reporting and some others are covered in the F<README> file.
#pod
#pod =head1 SYNOPSIS
#pod
#pod     $lic = Software::License::LGPL_3_0::or_later->new( {
#pod         holder  => 'John Doe',
#pod         year    => '2010',
#pod         program => 'Assa'
#pod     );
#pod
#pod     $lic->_abbr;    # 'LGPL'
#pod     $lic->abbr;     # 'LGPLv3.0+'
#pod     $lic->_name;    # ''
#pod     $lic->name;     # 'The  version 3.0 or later'
#pod
#pod     $lic->notice;
#pod         # Copyright statement and
#pod         # standard GNU 3-paragraph license notice.
#pod     $lic->notice( 'short' );
#pod         # Copyright statement, license name and
#pod         # two sentences about free software and warranties.
#pod
#pod     # + other methods inherited from Software::License::LGPL_3_0
#pod     # and Software::License.
#pod
#pod =head1 DESCRIPTION
#pod
#pod C<Software::License::LGPL_3_0::or_later> is a subclass of L<Software::License::LGPL_3_0>.
#pod It overrides few parent methods and introduces few own methods.
#pod
#pod See documentation on L<Software::License> for a general description of the class interface.
#pod
#pod =cut

# version:          3+
# base->version:    3
# name:             The GNU General Public License version 3 or later
# _name:            GNU General Public License
# _abbr:            GPL
# abbr:             GPLv3+
#
#
#
#

package Software::License::LGPL_3_0::or_later;

use strict;
use warnings;
use version 0.77;

# ABSTRACT: LGPLv3.0+ license for Software::License infrastructure
our $VERSION = 'v0.9.0_01'; # TRIAL VERSION

use parent 'Software::License::LGPL_3_0';
use Text::Wrap;

#pod =method _abbr
#pod
#pod Returns bare abbreviated license name, "LGPL".
#pod
#pod Note: this method is not inherited from base class.
#pod
#pod =cut

sub _abbr {
    return 'LGPL';
};

#pod =method abbr
#pod
#pod Returns abbreviated license name: concatenated bare abbreviated license name, 'v' character,
#pod and license version (with trailing plus sign).
#pod
#pod Note: this method is not inherited from base class.
#pod
#pod =cut

sub abbr {
    my ( $self ) = @_;
    return $self->_abbr . 'v' . $self->version;
};

#pod =method base
#pod
#pod Returns a reference to base license object, i. e. license without "or later" clause.
#pod
#pod Note: this method is not inherited from base class.
#pod
#pod =cut

sub base {
    my ( $self ) = @_;
    if ( not $self->{ base } ) {
        my %base = %$self;  # Create a copy, because `new` (re)blesses passed `HashRef`.
        $self->{ base } = Software::License::LGPL_3_0->new( \%base );
    };
    return $self->{ base };
};

#pod =method _name
#pod
#pod Returns bare name of the license, which is also bare name of the base license, because it does
#pod include neither definitive article ("The"), nor license version nor "or later" clause:
#pod "GNU Lesser General Public License".
#pod
#pod Note: this method is not inherited from base class.
#pod
#pod =cut

sub _name {
    return 'GNU Lesser General Public License';
};

#pod =method name
#pod
#pod This method meets <Software::License> specification: returned name starts with definitive
#pod capitalized article ("The"). Returned name also includes the base license version (like other
#pod C<Software::LIcense> licenses do) (without trailing plus sign) and "or later" clause.
#pod
#pod =cut

sub name {
    my ( $self ) = @_;
    return sprintf( "The %s version %s or later", $self->_name, $self->base->version );
};

#pod =method program
#pod
#pod Returns a program name as specified by the C<program> option in constructor, or the C<Program>
#pod option in constructor, in "this program". This form of program name is intended to be used in the
#pod middle of sentence.
#pod
#pod Note: this method is not inherited from base class.
#pod
#pod =cut

sub program {
    my ( $self ) = @_;
    return $self->{ program } || $self->{ Program } || 'this program';
};

#pod =method Program
#pod
#pod Returns a program name as specified by the C<Program> option in constructor, the C<program> option
#pod in constructor, "This program". This form of program name is intended to be used in the beginning
#pod of sentence.
#pod
#pod Note: this method is not inherited from base class.
#pod
#pod =cut

sub Program {
    my ( $self ) = @_;
    return $self->{ Program } || $self->{ program } || 'This program';
};

#pod =method notice
#pod
#pod This method overrides L<Software::License>'s C<notice>. Differences are:
#pod
#pod =for :list
#pod *   If the license object was created with C<program> option, notice will include real program name
#pod     instead of generic "this program".
#pod *   It returns copyright statement followed by standard GNU 3-paragraph license notice.
#pod *   Result is formatted with L<Text::Wrap::fill>.
#pod
#pod The method can be called with C<'short'> argument to get short version of notice. Short version
#pod includes: copyright statement, license name, and two sentences about free software and warranties.
#pod
#pod =cut

sub notice {
    my ( $self, $arg ) = @_;
    my $notice = Text::Wrap::fill( '', '', $self->_fill_in( $arg || 'NOTICE' ) );
    #   Documentation on `fill` says it deletes all trailing whitespace, but it looks like it
    #   may leave one space. Let us make sure notice ends with one newline.
    $notice =~ s{\s*\z}{\n}x;
    return $notice;
};

#pod =method version
#pod
#pod Returns base license version with appended plus sign.
#pod
#pod =cut

sub version {
    my ( $self ) = @_;
    return $self->base->version . '+';
};

1;

# doc/what.pod #

#pod =encoding UTF-8
#pod
#pod =head1 WHAT?
#pod
#pod C<Software-License-OrLaterPack> is an add-on for C<Software-License>, a set of licenses with "or later" clause
#pod (like C<GPL_3::or_later>). It allows Perl developers (who use C<Dist-Zilla>) to release their work
#pod under the terms of a I<License> version I<N> or (at user option) any later version.
#pod
#pod =cut

# end of file #

=pod

=encoding UTF-8

=head1 NAME

Software::License::LGPL_3_0::or_later - LGPLv3.0+ license for Software::License infrastructure

=head1 VERSION

Version v0.9.0_01, released on 2015-09-27 13:18 UTC.
This is a B<trial release>.

=head1 WHAT?

C<Software-License-OrLaterPack> is an add-on for C<Software-License>, a set of licenses with "or later" clause
(like C<GPL_3::or_later>). It allows Perl developers (who use C<Dist-Zilla>) to release their work
under the terms of a I<License> version I<N> or (at user option) any later version.

This is C<Software::License::LGPL_3_0::or_later> module documentation. Read this if you
are going to hack or extend the module, or use the module directly.

If you want to use GNU license with "or later" clause read the L<user
manual|Software::License::OrLaterPack>. General topics like getting source, building, installing,
bug reporting and some others are covered in the F<README> file.

=head1 SYNOPSIS

    $lic = Software::License::LGPL_3_0::or_later->new( {
        holder  => 'John Doe',
        year    => '2010',
        program => 'Assa'
    );

    $lic->_abbr;    # 'LGPL'
    $lic->abbr;     # 'LGPLv3.0+'
    $lic->_name;    # ''
    $lic->name;     # 'The  version 3.0 or later'

    $lic->notice;
        # Copyright statement and
        # standard GNU 3-paragraph license notice.
    $lic->notice( 'short' );
        # Copyright statement, license name and
        # two sentences about free software and warranties.

    # + other methods inherited from Software::License::LGPL_3_0
    # and Software::License.

=head1 DESCRIPTION

C<Software::License::LGPL_3_0::or_later> is a subclass of L<Software::License::LGPL_3_0>.
It overrides few parent methods and introduces few own methods.

See documentation on L<Software::License> for a general description of the class interface.

=head1 OBJECT METHODS

=head2 _abbr

Returns bare abbreviated license name, "LGPL".

Note: this method is not inherited from base class.

=head2 abbr

Returns abbreviated license name: concatenated bare abbreviated license name, 'v' character,
and license version (with trailing plus sign).

Note: this method is not inherited from base class.

=head2 base

Returns a reference to base license object, i. e. license without "or later" clause.

Note: this method is not inherited from base class.

=head2 _name

Returns bare name of the license, which is also bare name of the base license, because it does
include neither definitive article ("The"), nor license version nor "or later" clause:
"GNU Lesser General Public License".

Note: this method is not inherited from base class.

=head2 name

This method meets <Software::License> specification: returned name starts with definitive
capitalized article ("The"). Returned name also includes the base license version (like other
C<Software::LIcense> licenses do) (without trailing plus sign) and "or later" clause.

=head2 program

Returns a program name as specified by the C<program> option in constructor, or the C<Program>
option in constructor, in "this program". This form of program name is intended to be used in the
middle of sentence.

Note: this method is not inherited from base class.

=head2 Program

Returns a program name as specified by the C<Program> option in constructor, the C<program> option
in constructor, "This program". This form of program name is intended to be used in the beginning
of sentence.

Note: this method is not inherited from base class.

=head2 notice

This method overrides L<Software::License>'s C<notice>. Differences are:

=over 4

=item *

If the license object was created with C<program> option, notice will include real program name instead of generic "this program".

=item *

It returns copyright statement followed by standard GNU 3-paragraph license notice.

=item *

Result is formatted with L<Text::Wrap::fill>.

=back

The method can be called with C<'short'> argument to get short version of notice. Short version
includes: copyright statement, license name, and two sentences about free software and warranties.

=head2 version

Returns base license version with appended plus sign.

=head1 AUTHOR

Van de Bugger <van.de.bugger@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright © 2015 Van de Bugger

This file is part of perl-Software-License-OrLaterPack.

perl-Software-License-OrLaterPack is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.

perl-Software-License-OrLaterPack is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
perl-Software-License-OrLaterPack. If not, see <http://www.gnu.org/licenses/>.

=cut

__DATA__

__NOTICE__
Copyright (C) {{$self->year}} {{$self->holder}}

{{$self->Program}} is free software: you can redistribute it and/or modify it
under the terms of the {{$self->_name}} as published by the Free Software Foundation, either
version {{$self->base->version}} of the License, or (at your option) any later version.

{{$self->Program}} is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See the {{$self->_name}} for more details.

You should have received a copy of the {{$self->_name}} along with
{{$self->program}}. If not, see <http://www.gnu.org/licenses/>.

__short__
Copyright (C) {{$self->year}} {{$self->holder}}

License {{$self->abbr}}: {{$self->name}}
<{{$self->url}}>.

This is free software: you are free to change and redistribute it.
There is NO WARRANTY, to the extent permitted by law.

__END__

# end of file #
