#!perl
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: t/error-logger.t
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Role-ErrorLogger.
#
#   perl-Dist-Zilla-Role-ErrorLogger is free software: you can redistribute it and/or modify it
#   under the terms of the GNU General Public License as published by the Free Software Foundation,
#   either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Role-ErrorLogger is distributed in the hope that it will be useful, but WITHOUT
#   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#   PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Role-ErrorLogger. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

use strict;
use warnings;
use lib 't';        # Make ErrorLoggerTestPlugin accessible.

use Test::More;
use Test::Routine::Util;
use Test::Deep qw{ re };

plan tests => 6;

my $role     = 'ErrorLoggerTester';
my $aborting = 'Aborting...';

run_tests( 'successful build', $role, {
    hook => sub {
        my ( $self ) = @_;
        my $rc;
        subtest 'inner' => sub {
            plan tests => 6;
            is( $self->error_count, 0, 'error count inited to zero' );
            $self->abort_if_error();
            pass( 'not yet aborted' );
            $rc = $self->log_error( 'the first error' );
            ok( $rc, 'log_error returns true value' );
            is( $self->error_count, 1, 'error count bumped' );
            $rc = $self->log_error( 'another error' );
            ok( $rc, 'log_error returns true value again' );
            is( $self->error_count, 2, 'error count bumped again' );
            done_testing;
        };
    },
    exp_messages => [
        'the first error',
        'another error',
    ],
} );

run_tests( 'abort', $role, {
    #   `abort` with no arguments does not add messages to the log, it just dies.
    hook => sub {
        my ( $self ) = @_;
        $self->abort();
    },
    exp_exception => $aborting,
    exp_messages  => [],        # No messages.
} );

run_tests( 'abort', $role, {
    #   `abort` with an argument logs it, and then dies.
    hook => sub {
        my ( $self ) = @_;
        $self->abort( 'Oops!' );                            # Abort message.
    },
    exp_exception => $aborting,
    exp_messages  => [
        'Oops!',                                            # Abort message was logged
    ],
} );

run_tests( 'failure', $role, {
    #   This time we expect exception. If an exception is generated within subtest, it considered
    #   failed, so do not organize subtest here.
    hook => sub {
        my ( $self ) = @_;
        $self->abort_if_error();
        pass( 'not yet aborted' );
        $self->log_error( 'error' );                        # Error message
        $self->abort_if_error();
        fail( 'must be aborted before this point' );
    },
    exp_exception => $aborting,
    exp_messages  => [
        'error',                                            # Error message was logged
    ],
} );

run_tests( 'custom abort message', $role, {
    hook => sub {
        my ( $self ) = @_;
        $self->log_error( 'error' );                        # Error message
        $self->abort_if_error( 'Custom abort' );            # Custom abort message
        fail( 'must be aborted before this point' );
    },
    exp_exception => $aborting,
    exp_messages => [
        'error',                                            # Error message was logged.
        'Custom abort',                                     # Custom abort message was logged too
    ],
} );

run_tests( 'log arguments', $role, {
    hook => sub {
        my ( $self ) = @_;
        my ( $args, $orig );
        subtest 'inner' => sub {
            plan tests => 2;
            $args = { prefix => 'pfx1: ' };
            $orig = { %$args };
            $self->log_error( $args, 'error' );                 # Error message with custom prefix.
            is_deeply( $args, $orig, 'args are not changed' );
            $args = { prefix => 'pfx2: ', level => 'info' };
            $orig = { %$args };
            $self->log_error( $args, 'another' );               # With custom prefix and level.
            is_deeply( $args, $orig, 'args are not changed again' );
            done_testing;
        };
    },
    exp_messages => [
        'pfx1: error',                                          # Error message with custom prefix.
        'pfx2: another',                                        # With custom prefix and level.
    ],
} );

done_testing;

exit( 0 );

# end of file #
