package Finance::BankVal::International::IBANValidate;

use 5.008000;
use strict;
use warnings;

use vars qw($size $format $iban $userid $pin $error $ua $url);
use LWP::UserAgent;
use XML::Simple;
use JSON;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(ibanValidate new);
our $VERSION = '0.01';

my $format;		#response return format i.e. xml json csv
my $iban;          	#the IBAN to be validated
my $userid;
my $pin;
my $size;               #the number of parameters passed only 2 and 4 are valid
my $error;              #any error messages generated here, not from unifieds servers
my $responseString;     #the return value of ibanValidate either $errors or web service response
my $ua;
my $url;

#constructor
sub new {
        my $proto = shift;
        my $class = ref($proto) || $proto;
        my $self = {};
	bless ($self, $class);
        return $self;
}

# exportable sub takes parameter array of 2 or 4 elements
# see perldoc for this module for more details
# ibanValidate($format,$iban,$userid,pin);
# or ibanValidate($format,$iban);
sub ibanValidate{
	$error="";
	my @params = @_;
        $size = @params;
        #the following block checks to see if the first param is a reference
        #if it is then the sub was called as an object ref so size is reduced
        #accordingly
        my $refCheck = shift @_;       						#remove the leftmost array element
        if (ref($refCheck)){                                                    #check if its a reference
        	$size--;                                                        #if it is reduce the size value to account for it
        }else{                                                                  #otherwise
        	unshift(@_, $refCheck);                                         #put it back
    	}
        $format = lc($_[0]);
        $iban = $_[1];
        if ($size > 2){
        	$userid = $_[2];
                $pin = $_[3];
        }else{
                &loadUidPin;
        }
        #all params should now be present so call validate formats sub
        &validateFormat;
        #if invalid formats are found return error message
        if ($error){
           	$responseString = "$error";
                &formatErrorMsg;
                return $responseString;
        }
        #if all formats are ok call web service sub then return response
        &goValidate;
        return $responseString;

}

sub goValidate{
        #create user agent
	$ua = LWP::UserAgent->new();

	#build the URL
        my $baseUrl = 'https://www.unifiedsoftware.co.uk/services/bankvalint/ibanvalidator';
    	$url = "$baseUrl/userid/$userid/pin/$pin/iban/$iban/$format/";

	#call the service
	my $response = $ua->get($url);

	#Check the response code if its fail call backup server sub
	if($response->code<200||$response->code>399){
                &goFallOver;
	} else {
	        $responseString = $response->content();
	}
}

sub goFallOver{
	#build the URL
        my $baseUrl = 'https://www.unifiedservices.co.uk/services/bankvalint/ibanvalidator';
    	$url = "$baseUrl/userid/$userid/pin/$pin/iban/$iban/$format/";

	#call the service
	my $response = $ua->get($url);

	#Check the response code
	if($response->code<200||$response->code>399){
                $responseString .= $response->code;
	} else {
	        $responseString = $response->content();
	}
}

sub validateFormat{
        #Validate response format must match json, xml, or, csv
        if ($format !~ /^json$|^xml$|^csv$/){
                $error .= "INVALID - Result Format";
                return;
        }
        #Validate IBAN up to 34 chars A-Z 0-9
        if ($iban !~ /^[A-Z,0-9]{1,34}$/) {
          	$error .= "INVALID - FORMAT";
                return;
        }
        #Validate PIN all numeric 5 characters
        if ($pin){
                if ($pin !~ /^\d\d\d\d\d$/){
                        $error .= "ERROR -Invalid User ID/PIN";
                        return;
                }
        }
        #Validate UID must end with 3 numerics exactly and start with 3 Alpha variable length otherwise
        if ($userid){
                 if ($userid !~ /^[a-zA-Z\-_][a-zA-Z][a-zA-Z]*\D\d\d\d$/){
                    	$error .= "ERROR -Invalid User ID/PIN";
                        return;
                 }
        }
}

sub loadUidPin {
	my $fileOpened = open UIDCONF, "InternationalLoginConfig.txt";
        if ( ! $fileOpened){
                $error .= "No UserID / PIN supplied, please visit http://www.unifiedsoftware.co.uk/freetrial/free-trial-home.html: ";
        }else{
         	while (<UIDCONF>){
        		if ($_ =~ /^UserID/){
                        	chomp(my @line = split (/ |\t/,$_));
                                $userid = $line[-1];
                        }elsif($_ =~ /^PIN/){
                                chomp(my @line = split (/ |\t/,$_));
                                $pin = $line[-1];
                        }

                }
                #check to see if conf file has empty params - if so return error message directing to free trial page
                if (($userid !~ /\w/) || ($pin !~ /\w/)){
                        $error .= "No User ID / PIN supplied, please visit http://www.unifiedsoftware.co.uk/freetrial/free-trial-home.html: ";
                }
        close UIDCONF;
        }
}

sub formatErrorMsg{
 	if($format eq "xml"){
        	$responseString = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><iban><result>" . $responseString . "</result></iban>";
        }elsif($format eq "json"){
         	$responseString = "{\"result\":\"" . $responseString . "\"}";
        }
}

1;
__END__


=head1 NAME

Finance::BankVal::International::ibanValidate

=head1 SYNOPSIS

  use Finance::BankVal::International::IBANValidate qw(&ibanValidate);

  my $ans = ibanValidate('XML','ANIBANNUMBERGOESHERE','xmpl123','12345');

  =============================or for OO================================

  use Finance::BankVal::International::IBANValidate;

  $ibanObj = Finance::BankVal::International::IBANValidate->new();
  my $ans = $ibanObj->ibanValidate('XML','ANIBANNUMBERGOESHERE','xmpl123','12345');




=head1 DESCRIPTION

This module handles all of the restful web service calls to Unified Software's
BankValInternational IBAN validate service. It also handles fail over to the back up services
transparently to the calling script. It can be called in a procedural or OO fashion
(see synopsis)

The exportable method &ibanValidate(); takes a number of parameters including;

1: Format 	- the response format (either xml, json or csv)

2: IBAN		- the IBAN to be validated

3: UserID	- available from www.unifiedsoftware.co.uk

4: PIN		- available from www.unifiedsoftware.co.uk

(UserID and PIN are available from http://www.unifiedsoftware.co.uk/freetrial/free-trial-home.html)

The order of the parameters B<must> be as above. The UserID and PIN can be stored in the InternationalLoginConfig.txt
file bundled with this module, the use of this file saves passing the PIN and user ID data with each call to ibanValidate.
For example, a call to validate an IBAN passing the user ID and PIN as parameters
and printing the reply to console should follow this form:

=====================================================================

 #!/usr/bin/perl

 use Finance::BankVal::International::IBANValidate qw(&ibanValidate);

 my $ans = ibanValidate('XML','ANIBANNUMBERGOESHERE','xmpl123','12345');

 print $ans;

=============================OR for OO===============================

 use Finance::BankVal::International::IBANValidate;

 $ibanObj = Finance::BankVal::International::IBANValidate->new();
 my $ans = $ibanObj->ibanValidate('XML','ANIBANNUMBERGOESHERE','xmpl123','12345');

 print $ans;

=====================================================================

valid parameter lists are:-

ibanValidate('$format','$iban','$userID','$PIN');

ibanValidate('$format','$iban');

n.b. the last parameter list requires that the user ID and PIN are stored
in the I<InternationalLoginConfig.txt> file bundled with this module.

=head2 EXPORT

None by default.
&ibanValidate is exported on request i.e. "use Finance::BankVal::International::IBANValidate qw(&ibanValidate);"


=head1 SEE ALSO

Please see L<http://www.unifiedsoftware.co.uk> for full details on Unified Software's web services.


=head1 AUTHOR

A. Evans - Unified Software, E<lt>support@unifiedsoftware.co.ukE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010 by Unified Software Limited

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.08.0 or,
at your option, any later version of Perl 5 you may have available.


=cut