#!./perl -T
#
# Taint tests by Tom Phoenix <rootbeer@teleport.com>.
#
# I don't claim to know all about tainting. If anyone sees
# tests that I've missed here, please add them. But this is
# better than having no tests at all, right?
#

use strict;
use Config;
use File::Spec::Functions;

BEGIN { require './test.pl'; }
plan tests => 232;

$| = 1;

use vars qw($ipcsysv); # did we manage to load IPC::SysV?

my ($old_env_path, $old_env_dcl_path, $old_env_term);
BEGIN {
   $old_env_path = %ENV{'PATH'};
   $old_env_dcl_path = %ENV{'DCL$PATH'};
   $old_env_term = %ENV{'TERM'};
  if ($^O eq 'VMS' && !defined(%Config{d_setenv})) {
      %ENV{PATH} = %ENV{PATH};
      %ENV{TERM} = %ENV{TERM} ne ''? %ENV{TERM} : 'dummy';
  }
  if (%Config{'extensions'} =~ m/\bIPC\/SysV\b/
      && (%Config{d_shm} || %Config{d_msg})) {
      eval { require IPC::SysV };
      unless ($@) {
	  $ipcsysv++;
	  'IPC::SysV'->import(qw(IPC_PRIVATE IPC_RMID IPC_CREAT S_IRWXU IPC_NOWAIT));
      }
  }
}

my $Is_MacOS    = $^O eq 'MacOS';
my $Is_VMS      = $^O eq 'VMS';
my $Is_MSWin32  = $^O eq 'MSWin32';
my $Is_NetWare  = $^O eq 'NetWare';
my $Is_Dos      = $^O eq 'dos';
my $Is_Cygwin   = $^O eq 'cygwin';
my $Is_OpenBSD  = $^O eq 'openbsd';
my $Invoke_Perl = $Is_VMS      ? 'MCR Sys$Disk:[]Perl.exe' :
                  $Is_MSWin32  ? '.\perl'               :
                  $Is_MacOS    ? ':perl'                :
                  $Is_NetWare  ? 'perl'                 : 
                                 './perl'               ;
my @MoreEnv = qw/IFS CDPATH ENV BASH_ENV/;

if ($Is_VMS) {
    my (%old, $x);
    for $x ('DCL$PATH', @MoreEnv) {
	(%old{$x}) = %ENV{$x} =~ m/^(.*)$/ if exists %ENV{$x};
    }
    # VMS note:  PATH and TERM are automatically created by the C
    # library in VMS on reference to the their keys in %ENV.
    # There is currently no way to determine if they did not exist
    # before this test was run.
    eval <<EndOfCleanup;
	END \{
	    \$ENV\{PATH\} = \$old_env_path;
	    warn "# Note: logical name 'PATH' may have been created\n";
	    \$ENV\{'TERM'\} = \$old_env_term;
	    warn "# Note: logical name 'TERM' may have been created\n";
	    \@ENV\{keys \%old\} = values \%old;
	    if (defined \$old_env_dcl_path) \{
		\$ENV\{'DCL\$PATH'\} = \$old_env_dcl_path;
	    \} else \{
		delete \$ENV\{'DCL\$PATH'\};
	    \}
	\}
EndOfCleanup
}

# Sources of taint:
#   The empty tainted value, for tainting strings
my $TAINT = substr($^X, 0, 0);
#   A tainted zero, useful for tainting numbers
my $TAINT0;
{
    no warnings;
    $TAINT0 = 0 + $TAINT;
}

# This taints each argument passed. All must be lvalues.
# Side effect: It also stringifies them. :-(
sub taint_these (@) {
    for (@_) { $_ .= $TAINT }
}

# How to identify taint when you see it
sub any_tainted (@) {
    return scalar grep { tainted($_) } @_;
}
sub tainted ($) {
    my $tainted = not eval { @_[0], kill 0; 1};
    die if $@ and $@->message !~ m/^Insecure dependency in kill while running with -T switch/;
    return $tainted;
}
sub all_tainted (@) {
    for (@_) { return 0 unless tainted $_ }
    1;
}


sub test ($;$) {
    my($ok, $diag) = @_;

    my $curr_test = curr_test();

    if ($ok) {
	print "ok $curr_test\n";
    } else {
	print "not ok $curr_test\n";
        printf "# Failed test at line \%d\n", (caller)[[2]];
	for (split m/^/m, $diag) {
	    print "# $_";
	}
	print "\n" unless
	    $diag eq ''
	    or substr($diag, -1) eq "\n";
    }

    next_test();

    return $ok;
}

# We need an external program to call.
my $ECHO = ($Is_MSWin32 ? ".\\echo$$" : $Is_MacOS ? ":echo$$" : ($Is_NetWare ? "echo$$" : "./echo$$"));
END { unlink $ECHO }
open PROG, ">", "$ECHO" or die "Can't create $ECHO: $!";
print PROG 'print "@ARGV\n"', "\n";
close PROG;
my $echo = "$Invoke_Perl $ECHO";

my $TEST = catfile(curdir(), 'TEST');

# First, let's make sure that Perl is checking the dangerous
# environment variables. Maybe they aren't set yet, so we'll
# taint them ourselves.
{
    %ENV{'DCL$PATH'} = '' if $Is_VMS;

    if ($Is_MSWin32 && %Config{ccname} =~ m/bcc32/ && ! -f 'cc3250mt.dll') {
	my $bcc_dir;
	foreach my $dir (split m/%Config{path_sep}/, %ENV{PATH}) {
	    if (-f "$dir/cc3250mt.dll") {
		$bcc_dir = $dir and last;
	    }
	}
	if (defined $bcc_dir) {
	    require File::Copy;
	    File::Copy::copy("$bcc_dir/cc3250mt.dll", '.') or
		die "$0: failed to copy cc3250mt.dll: $!\n";
	    eval q{
		END { unlink "cc3250mt.dll" }
	    };
	}
    }
    %ENV{PATH} = ($Is_Cygwin) ? '/usr/bin' : '';
    delete %ENV{[@MoreEnv]};
    %ENV{TERM} = 'dumb';

    test eval { `$echo 1` } eq "1\n";

    SKIP: {
        skip "Environment tainting tests skipped", 4
          if $Is_MSWin32 || $Is_NetWare || $Is_VMS || $Is_Dos || $Is_MacOS;

	my @vars = ('PATH', @MoreEnv);
	while (my $v = @vars[0]) {
	    local %ENV{$v} = $TAINT;
	    last if eval { `$echo 1` };
	    last unless $@->{description} =~ m/^Insecure \$ENV{$v}/;
	    shift @vars;
	}
	test !@vars, "@vars";

	# tainted $TERM is unsafe only if it contains metachars
	local %ENV{TERM};
	%ENV{TERM} = 'e=mc2';
	test eval { `$echo 1` } eq "1\n";
	%ENV{TERM} = 'e=mc2' . $TAINT;
	test !eval { `$echo 1` };
	like( $@->{description}, qr/^Insecure \$ENV{TERM}/ );
    }

    my $tmp;
    if ($^O eq 'os2' || $^O eq 'amigaos' || $Is_MSWin32 || $Is_NetWare || $Is_Dos) {
	print "# all directories are writeable\n";
    }
    else {
	$tmp = (grep { defined and -d and (stat '_')[[2]] ^&^ 2 }
		     qw(sys$scratch /tmp /var/tmp /usr/tmp),
		     %ENV{[qw(TMP TEMP)]})[[0]]
	    or print "# can't find world-writeable directory to test PATH\n";
    }

    SKIP: {
        skip "all directories are writeable", 2 unless $tmp;

	local %ENV{PATH} = $tmp;
	test !eval { `$echo 1` };
	test $@->{description} =~ m/^Insecure directory in \$ENV{PATH}/, $@;
    }

    SKIP: {
        skip "This is not VMS", 4 unless $Is_VMS;

	%ENV{'DCL$PATH'} = $TAINT;
	test  eval { `$echo 1` } eq '';
	test $@->{description} =~ m/^Insecure \$ENV{DCL\$PATH}/, $@;
	SKIP: {
            skip q[can't find world-writeable directory to test DCL$PATH], 2
              unless $tmp;

	    %ENV{'DCL$PATH'} = $tmp;
	    test eval { `$echo 1` } eq '';
	    test $@->{description} =~ m/^Insecure directory in \$ENV{DCL\$PATH}/, $@;
	}
	%ENV{'DCL$PATH'} = '';
    }
}

# Let's see that we can taint and untaint as needed.
{
    my $foo = $TAINT;
    test tainted $foo;

    # That was a sanity check. If it failed, stop the insanity!
    die "Taint checks don't seem to be enabled" unless tainted $foo;

    $foo = "foo";
    test not tainted $foo;

    taint_these($foo);
    test tainted $foo;

    my @list = 1..10;
    test not any_tainted @list;
    taint_these @list[[1,3,5,7,9]];
    test any_tainted @list;
    test all_tainted @list[[1,3,5,7,9]];
    test not any_tainted @list[[0,2,4,6,8]];

    ($foo) = $foo =~ m/(.+)/;
    test not tainted $foo;

    $foo = $1 if ('bar' . $TAINT) =~ m/(.+)/;
    test not tainted $foo;
    test $foo eq 'bar';

    {
      use re 'taint';

      ($foo) = ('bar' . $TAINT) =~ m/(.+)/;
      test tainted $foo;
      test $foo eq 'bar';

      $foo = $1 if ('bar' . $TAINT) =~ m/(.+)/;
      test tainted $foo;
      test $foo eq 'bar';
    }

    $foo = $1 if 'bar' =~ m/(.+)$TAINT/;
    test tainted $foo;
    test $foo eq 'bar';

    my $pi = 4 * atan2(1,1) + $TAINT0;
    test tainted $pi;

    ($pi) = $pi =~ m/(\d+\.\d+)/;
    test not tainted $pi;
    test sprintf("\%.5f", $pi) eq '3.14159';
}

# How about command-line arguments? The problem is that we don't
# always get some, so we'll run another process with some.
SKIP: {
    my $arg = catfile(curdir(), "arg$$");
    open PROG, ">", "$arg" or die "Can't create $arg: $!";
    print PROG q{
	eval { join('', @ARGV), kill 0 };
	exit 0 if $@->{description} =~ m/^Insecure dependency/;
	print "# Oops: \$@ was [$@]\n";
	exit 1;
    };
    close PROG;
    print `$Invoke_Perl "-T" $arg and some suspect arguments`;
    test !$?, "Exited with status $?";
    unlink $arg;
}

# Reading from a file should be tainted
{
    test open(FILE, "<", $TEST), "Couldn't open '$TEST': $!";

    my $block;
    sysread(FILE, $block, 100);
    my $line = ~< *FILE;
    close FILE;
    test tainted $block;
    test tainted $line;
}

# Globs should be forbidden, except under VMS,
#   which doesn't spawn an external program.
SKIP: {
    skip "globs should be forbidden", 2 if 1 or $Is_VMS;

    my @globs = eval { glob("*") };
    test @globs == 0 && $@->{description} =~ m/^Insecure dependency/;

    @globs = eval { glob '*' };
    test @globs == 0 && $@->{description} =~ m/^Insecure dependency/;
}

# Output of commands should be tainted
{
    my $foo = `$echo abc`;
    test tainted $foo;
}

# Certain system variables should be tainted
{
    test all_tainted $^X, $0;
}

# Results of matching should all be untainted
{
    my $foo = "abcdefghi" . $TAINT;
    test tainted $foo;

    $foo =~ m/def/;
    test not any_tainted $`, $&, $';

    $foo =~ m/(...)(...)(...)/;
    test not any_tainted $1, $2, $3, $+;

    my @bar = $foo =~ m/(...)(...)(...)/;
    test not any_tainted @bar;

    test tainted $foo;	# $foo should still be tainted!
    test $foo eq "abcdefghi";
}

# Operations which affect files can't use tainted data.
{
    dies_like( sub { chmod 0, $TAINT }, qr/^Insecure dependency/);

    # There is no feature test in %Config{} for truncate,
    #   so we allow for the possibility that it's missing.
    dies_like( sub { truncate 'NoSuChFiLe', $TAINT0 }, 
               qr/^(?:Insecure dependency|truncate not implemented)/);

    dies_like( sub { rename '', $TAINT },
               qr/^Insecure dependency/);

    dies_like( sub { unlink $TAINT },
               qr/^Insecure dependency/);

    dies_like( sub { utime $TAINT },
               qr/^Insecure dependency/);

    SKIP: {
        skip "chown() is not available", 2 unless %Config{d_chown};

	dies_like( sub { chown -1, -1, $TAINT },
                   qr/^Insecure dependency/);
    }

    SKIP: {
        skip "link() is not available", 2 unless %Config{d_link};

	dies_like( sub { link $TAINT, '' },
                   qr/^Insecure dependency/);
    }

    SKIP: {
        skip "symlink() is not available", 2 unless %Config{d_symlink};

	dies_like( sub { symlink $TAINT, '' },
                   qr/^Insecure dependency/);
    }
}

# Operations which affect directories can't use tainted data.
{
    dies_like( sub { mkdir "foo".$TAINT, 0755.$TAINT0 },
               qr/^Insecure dependency/);

    dies_like( sub { rmdir $TAINT },
               qr/^Insecure dependency/);

    dies_like( sub { chdir "foo".$TAINT },
               qr/^Insecure dependency/);

    SKIP: {
        skip "chroot() is not available", 2 unless %Config{d_chroot};

	dies_like( sub { chroot $TAINT },
                   qr/^Insecure dependency/);
    }
}

# Some operations using files can't use tainted data.
{
    my $foo = "imaginary library" . $TAINT;
    dies_like( sub { require $foo },
               qr/^Insecure dependency/);

    my $filename = "./taintB$$";	# NB: $filename isn't tainted!
    END { unlink $filename if defined $filename }
    $foo = $filename . $TAINT;
    unlink $filename;	# in any case

    test !eval { open FOO, "<", $foo }, 'open for read';
    test $@ eq '', $@;		# NB: This should be allowed

    # Try first new style but allow also old style.
    # We do not want the whole taint.t to fail
    # just because Errno possibly failing.
    test eval('$!{ENOENT}') ||
	$! == 2 || # File not found
	($Is_Dos && $! == 22) ||
	($^O eq 'mint' && $! == 33);

    dies_like( sub { open FOO, ">", "$foo" },
               qr/^Insecure dependency/);
}

# Commands to the system can't use tainted data
{
    my $foo = $TAINT;

    SKIP: {
        skip "open('|') is not available", 4 if $^O eq 'amigaos';

	dies_like( sub { open FOO, "|-", "x$foo" },
                   qr/^Insecure dependency/);

	dies_like( sub { open FOO, "-|", "x$foo" },
                   qr/^Insecure dependency/);
    }

    dies_like( sub { exec $TAINT },
               qr/^Insecure dependency/);

    dies_like( sub { system $TAINT },
               qr/^Insecure dependency/);

    $foo = "*";
    taint_these $foo;

    dies_like( sub { `$echo 1$foo` },
               qr/^Insecure dependency/);

    SKIP: {
        # wildcard expansion doesn't invoke shell on VMS, so is safe
        skip "This is not VMS", 2 unless $Is_VMS;
    
	test join('', eval { glob $foo } ) ne '', 'globbing';
	test $@ eq '', $@;
    }
}

# Operations which affect processes can't use tainted data.
{
    dies_like( sub { kill 0, $TAINT },
               qr/^Insecure dependency/);

    SKIP: {
        skip "setpgrp() is not available", 2 unless %Config{d_setpgrp};

	dies_like( sub { setpgrp 0, $TAINT0 },
                   qr/^Insecure dependency/);
    }

    SKIP: {
        skip "setpriority() is not available", 2 unless %Config{d_setprior};

	dies_like( sub { setpriority 0, $TAINT0, $TAINT0 },
                   qr/^Insecure dependency/);
    }
}

# Some miscellaneous operations can't use tainted data.
{
    SKIP: {
        skip "syscall() is not available", 2 unless %Config{d_syscall};

	dies_like( sub { syscall $TAINT },
                   qr/^Insecure dependency/);
    }

    {
	my $foo = "x" x 979;
	taint_these $foo;
	local *FOO;
	my $temp = "./taintC$$";
	END { unlink $temp }
	test open(FOO, ">", "$temp"), "Couldn't open $temp for write: $!";

	dies_like( sub { ioctl FOO, $TAINT0, $foo },
                   qr/^Insecure dependency/);

        SKIP: {
            skip "fcntl() is not available", 2 unless %Config{d_fcntl};

	    dies_like( sub { fcntl FOO, $TAINT0, $foo },
                       qr/^Insecure dependency/);
	}

	close FOO;
    }
}

# Some tests involving references
{
    my $foo = 'abc' . $TAINT;
    my $fooref = \$foo;
    test not tainted $fooref;
    test tainted $$fooref;
    test tainted $foo;
}

# Some tests involving assignment
{
    my $foo = $TAINT0;
    my $bar = $foo;
    test all_tainted $foo, $bar;
    test tainted($foo = $bar);
    test tainted($bar = $bar);
    test tainted($bar += $bar);
    test tainted($bar -= $bar);
    test tainted($bar *= $bar);
    test tainted($bar++);
    test tainted($bar /= $bar);
    test tainted($bar += 0);
    test tainted($bar -= 2);
    test tainted($bar *= -1);
    test tainted($bar /= 1);
    test tainted($bar--);
    test $bar == 0;
}

# Test assignment and return of lists
{
    my @foo = ("A", "tainted" . $TAINT, "B");
    test not tainted @foo[0];
    test     tainted @foo[1];
    test not tainted @foo[2];
    my @bar = @foo;
    test not tainted @bar[0];
    test     tainted @bar[1];
    test not tainted @bar[2];
    my @baz = eval { "A", "tainted" . $TAINT, "B" };
    test not tainted @baz[0];
    test     tainted @baz[1];
    test not tainted @baz[2];
    my @plugh = eval q[ "A", "tainted" . $TAINT, "B" ];
    test not tainted @plugh[0];
    test     tainted @plugh[1];
    test not tainted @plugh[2];
    my $nautilus = sub { "A", "tainted" . $TAINT, "B" };
    test not tainted ((&$nautilus)[[0]]);
    test     tainted ((&$nautilus)[[1]]);
    test not tainted ((&$nautilus)[[2]]);
    my @xyzzy = &$nautilus;
    test not tainted @xyzzy[0];
    test     tainted @xyzzy[1];
    test not tainted @xyzzy[2];
    my $red_october = sub { return "A", "tainted" . $TAINT, "B" };
    test not tainted ((&$red_october)[[0]]);
    test     tainted ((&$red_october)[[1]]);
    test not tainted ((&$red_october)[[2]]);
    my @corge = &$red_october;
    test not tainted @corge[0];
    test     tainted @corge[1];
    test not tainted @corge[2];
}

# Test for system/library calls returning string data of dubious origin.
{
    # No reliable %Config check for getpw*
    SKIP: {
        skip "getpwent() is not available", 1 unless 
          eval { setpwent(); getpwent() };

	setpwent();
	my @getpwent = getpwent();
	die "getpwent: $!\n" unless (@getpwent);
	test (    not tainted @getpwent[0]
	          and     tainted @getpwent[1]
	          and not tainted @getpwent[2]
	          and not tainted @getpwent[3]
	          and not tainted @getpwent[4]
	          and not tainted @getpwent[5]
	          and     tainted @getpwent[6]		# ge?cos
	          and not tainted @getpwent[7]
		  and     tainted @getpwent[8]);	# shell
	endpwent();
    }

    SKIP: {
        # pretty hard to imagine not
        skip "readdir() is not available", 1 unless %Config{d_readdir};

	local(*D);
	opendir(D, "op") or die "opendir: $!\n";
	my $readdir = readdir(D);
	test tainted $readdir;
	closedir(D);
    }

    SKIP: {
        skip "readlink() or symlink() is not available" unless 
          %Config{d_readlink} && %Config{d_symlink};

	my $symlink = "sl$$";
	unlink($symlink);
	my $sl = "/something/naughty";
	# it has to be a real path on Mac OS
	$sl = MacPerl::MakePath((MacPerl::Volumes())[[0]]) if $Is_MacOS;
	symlink($sl, $symlink) or die "symlink: $!\n";
	my $readlink = readlink($symlink);
	test tainted $readlink;
	unlink($symlink);
    }
}

# test bitwise ops (regression bug)
{
    my $why = "y";
    my $j = "x" ^|^ $why;
    test not tainted $j;
    $why = $TAINT."y";
    $j = "x" ^|^ $why;
    test     tainted $j;
}

# test target of substitution (regression bug)
{
    my $why = $TAINT."y";
    $why =~ s/y/z/;
    test     tainted $why;

    my $z = "[z]";
    $why =~ s/$z/zee/;
    test     tainted $why;

    $why =~ s/e/{'-'.$$}/g;
    test     tainted $why;
}


SKIP: {
    skip "no IPC::SysV", 2 unless $ipcsysv;

    # test shmread
    SKIP: {
        skip "shm*() not available", 1 unless %Config{d_shm};

        no strict 'subs';
        my $sent = "foobar";
        my $rcvd;
        my $size = 2000;
        my $id = shmget(IPC_PRIVATE, $size, S_IRWXU);

        if (defined $id) {
            if (shmwrite($id, $sent, 0, 60)) {
                if (shmread($id, $rcvd, 0, 60)) {
                    substr($rcvd, index($rcvd, "\0"), undef, '');
                } else {
                    warn "# shmread failed: $!\n";
                }
            } else {
                warn "# shmwrite failed: $!\n";
            }
            shmctl($id, IPC_RMID, 0) or warn "# shmctl failed: $!\n";
        } else {
            warn "# shmget failed: $!\n";
        }

        skip "SysV shared memory operation failed", 1 unless 
          $rcvd eq $sent;

        test tainted $rcvd;
    }


    # test msgrcv
    SKIP: {
        skip "msg*() not available", 1 unless %Config{d_msg};

	no strict 'subs';
	my $id = msgget(IPC_PRIVATE, IPC_CREAT ^|^ S_IRWXU);

	my $sent      = "message";
	my $type_sent = 1234;
	my $rcvd;
	my $type_rcvd;

	if (defined $id) {
	    if (msgsnd($id, pack("l! a*", $type_sent, $sent), IPC_NOWAIT)) {
		if (msgrcv($id, $rcvd, 60, 0, IPC_NOWAIT)) {
		    ($type_rcvd, $rcvd) = unpack("l! a*", $rcvd);
		} else {
		    warn "# msgrcv failed: $!\n";
		}
	    } else {
		warn "# msgsnd failed: $!\n";
	    }
	    msgctl($id, IPC_RMID, 0) or warn "# msgctl failed: $!\n";
	} else {
	    warn "# msgget failed\n";
	}

        SKIP: {
            skip "SysV message queue operation failed", 1
              unless $rcvd eq $sent && $type_sent == $type_rcvd;

	    test tainted $rcvd;
	}
    }
}

{
    # bug id 20001004.006

    open IN, "<", $TEST or warn "$0: cannot read $TEST: $!" ;
    local $/;
    my $a = ~< *IN;
    my $b = ~< *IN;

    ok tainted($a) && tainted($b) && !defined($b);

    close IN;
}

{
    # bug id 20001004.007

    open IN, "<", $TEST or warn "$0: cannot read $TEST: $!" ;
    my $a = ~< *IN;

    my $c = { a => 42,
	      b => $a };

    ok !tainted($c->{a}) && tainted($c->{b});


    my $d = { a => $a,
	      b => 42 };
    ok tainted($d->{a}) && !tainted($d->{b});


    my $e = { a => 42,
	      b => { c => $a, d => 42 } };
    ok !tainted($e->{a}) &&
       !tainted($e->{b}) &&
	tainted($e->{b}->{c}) &&
       !tainted($e->{b}->{d});

    close IN;
}

{
    # bug id 20010519.003

    BEGIN {
	use vars qw($has_fcntl);
	eval { require Fcntl; Fcntl->import; };
	unless ($@) {
	    $has_fcntl = 1;
	}
    }

    SKIP: {
        skip "no Fcntl", 18 unless $has_fcntl;

	my $evil = "foo" . $TAINT;

	dies_not( sub { sysopen(my $ro, $evil, &O_RDONLY) } );
	
	dies_like( sub { sysopen(my $wo, $evil, &O_WRONLY) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $rw, $evil, &O_RDWR) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $ap, $evil, &O_APPEND) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $cr, $evil, &O_CREAT) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $tr, $evil, &O_TRUNC) },
                   qr/^Insecure dependency/);
	
        dies_not( sub { sysopen(my $ro, "foo", &O_RDONLY ^|^ $TAINT0) } );
	
	dies_like( sub { sysopen(my $wo, "foo", &O_WRONLY ^|^ $TAINT0) },
                   qr/^Insecure dependency/);

	dies_like( sub { sysopen(my $rw, "foo", &O_RDWR ^|^ $TAINT0) },
                   qr/^Insecure dependency/);

	dies_like( sub { sysopen(my $ap, "foo", &O_APPEND ^|^ $TAINT0) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $cr, "foo", &O_CREAT ^|^ $TAINT0) },
                   qr/^Insecure dependency/);

	dies_like( sub { sysopen(my $tr, "foo", &O_TRUNC ^|^ $TAINT0) },
                   qr/^Insecure dependency/);

	dies_not( sub { sysopen(my $ro, "foo", &O_RDONLY, $TAINT0) } );
	
	dies_like( sub { sysopen(my $wo, "foo", &O_WRONLY, $TAINT0) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $rw, "foo", &O_RDWR, $TAINT0) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $ap, "foo", &O_APPEND, $TAINT0) },
                   qr/^Insecure dependency/);
	
	dies_like( sub { sysopen(my $cr, "foo", &O_CREAT, $TAINT0) },
                   qr/^Insecure dependency/);

	dies_like( sub { sysopen(my $tr, "foo", &O_TRUNC, $TAINT0) },
                   qr/^Insecure dependency/);
	
	unlink("foo"); # not unlink($evil), because that would fail...
    }
}

{
    # bug 20010526.004

    use warnings;

    my $saw_warning = 0;
    local $^WARN_HOOK = sub { $saw_warning = 1 };

    sub fmi {
	my $divnum = shift()/1;
	sprintf("\%1.1f\n", $divnum);
    }

    fmi(21 . $TAINT);
    fmi(37);
    fmi(248);

    test !$saw_warning;
}


{
    # Bug ID 20010730.010

    my $i = 0;

    sub Tie::TIESCALAR {
        my $class =  shift;
        my $arg   =  shift;

        bless \$arg => $class;
    }

    sub Tie::FETCH {
        $i ++;
        ${@_ [0]}
    }

 
    package main;
 
    my $bar = "The Big Bright Green Pleasure Machine";
    taint_these $bar;
    tie my ($foo), Tie => $bar;

    my $baz = $foo;

    ok $i == 1;
}

{
    # Check that all environment variables are tainted.
    my @untainted;
    while (my ($k, $v) = each %ENV) {
	if (!tainted($v) &&
	    # These we have explicitly untainted or set earlier.
	    $k !~ m/^(BASH_ENV|CDPATH|ENV|IFS|PATH|PERL_CORE|TEMP|TERM|TMP)$/) {
	    push @untainted, "# '$k' = '$v'\n";
	}
    }
    test @untainted == 0, "untainted:\n @untainted";
}


ok( $^TAINT == 1, '$^TAINT is on' );

eval { $^TAINT = 0 };
ok( $^TAINT,  '$^TAINT is not assignable' );
ok( $@->{description} =~ m/^Modification of a read-only value attempted/,
    'Assigning to $^TAINT fails' );

{
    # bug 20011111.105
    
    my $re1 = qr/x$TAINT/;
    test tainted $re1;
    
    my $re2 = qr/^$re1\z/;
    test tainted $re2;
    
    my $re3 = "$re2";
    test tainted $re3;
}

SKIP: {
    skip "system \{\} has different semantics on Win32", 1 if $Is_MSWin32;

    # bug 20010221.005
    local %ENV{PATH} .= $TAINT;
    dies_like(sub { system { "echo" } "/arg0", "arg1" },
              qr/^Insecure \$ENV/);
}

TODO: {
    todo_skip 'tainted %ENV warning occludes tainted arguments warning', 22
      if $Is_VMS;

    # bug 20020208.005 plus some single arg exec/system extras
    my $err = qr/^Insecure dependency/ ;
    dies_like( sub { exec $TAINT, $TAINT },
               $err );
    dies_like( sub { exec $TAINT, $TAINT },
               $err );
    dies_like( sub { exec $TAINT, $TAINT, $TAINT },
               $err );
    dies_like( sub { exec $TAINT 'notaint' },
               $err );
    dies_like( sub { exec {'notaint'} $TAINT },
               $err );

    dies_like( sub { system $TAINT, $TAINT }, $err );
    dies_like( sub { system $TAINT $TAINT }, $err, 'system');
    dies_like( sub { system $TAINT $TAINT, $TAINT }, $err, 'system');
    dies_like( sub { system $TAINT 'notaint' }, $err, 'system');
    dies_like( sub { system {'notaint'} $TAINT }, $err, 'system');

    eval { 
        no warnings;
        system("lskdfj does not exist","with","args"); 
    };
    test !$@;

    SKIP: {
        skip "no exec() on MacOS Classic" if $Is_MacOS;

	eval { 
            no warnings;
            exec("lskdfj does not exist","with","args"); 
        };
	test !$@;
    }

    # If you add tests here update also the above skip block for VMS.
}

{
    # [ID 20020704.001] taint propagation failure
    use re 'taint';
    $TAINT =~ m/(.*)/;
    test tainted(my $foo = $1);
}

{
    # [perl #24291] this used to dump core
    our %nonmagicalenv = ( PATH => "util" );
    local *ENV = \%nonmagicalenv;
    dies_like(sub { system("lskdfj") },
              qr/^\%ENV is aliased to another variable while running with -T switch/);
    local *ENV = *nonmagicalenv;
    dies_like( sub { system("lskdfj"); },
               qr/^\%ENV is aliased to \%nonmagicalenv while running with -T switch/);
}
{
    # [perl #24248]
    $TAINT =~ m/(.*)/;
    test !tainted($1);
    my $notaint = $1;
    test !tainted($notaint);

    my $l;
    $notaint =~ m/($notaint)/;
    $l = $1;
    test !tainted($1);
    test !tainted($l);
    $notaint =~ m/($TAINT)/;
    $l = $1;
    test tainted($1);
    test tainted($l);

    $TAINT =~ m/($notaint)/;
    $l = $1;
    test !tainted($1);
    test !tainted($l);
    $TAINT =~ m/($TAINT)/;
    $l = $1;
    test tainted($1);
    test tainted($l);

    my $r;
    ($r = $TAINT) =~ m/($notaint)/;
    test !tainted($1);
    ($r = $TAINT) =~ m/($TAINT)/;
    test tainted($1);

    #  [perl #24674]
    # accessing $^O  shoudn't taint it as a side-effect;
    # assigning tainted data to it is now an error

    test !tainted($^O);
    if (!$^X) { } elsif ($^O eq 'bar') { }
    test !tainted($^O);
    eval_dies_like( '$^O = $^X',
                    qr/Insecure dependency in/ );
}

EFFECTIVELY_CONSTANTS: {
    my $tainted_number = 12 + $TAINT0;
    test tainted( $tainted_number );

    # Even though it's always 0, it's still tainted
    my $tainted_product = $tainted_number * 0;
    test tainted( $tainted_product );
    test $tainted_product == 0;
}

TERNARY_CONDITIONALS: {
    my $tainted_true  = $TAINT . "blah blah blah";
    my $tainted_false = $TAINT0;
    test tainted( $tainted_true );
    test tainted( $tainted_false );

    my $result = $tainted_true ? "True" : "False";
    test $result eq "True";
    test !tainted( $result );

    $result = $tainted_false ? "True" : "False";
    test $result eq "False";
    test !tainted( $result );

    my $untainted_whatever = "The Fabulous Johnny Cash";
    my $tainted_whatever = "Soft Cell" . $TAINT;

    $result = $tainted_true ? $tainted_whatever : $untainted_whatever;
    test $result eq "Soft Cell";
    test tainted( $result );

    $result = $tainted_false ? $tainted_whatever : $untainted_whatever;
    test $result eq "The Fabulous Johnny Cash";
    test !tainted( $result );
}

{
    # rt.perl.org 5900  $1 remains tainted if...
    # 1) The regular expression contains a scalar variable AND
    # 2) The regular expression appears in an elsif clause

    my $foo = "abcdefghi" . $TAINT;

    my $valid_chars = 'a-z';
    if ( $foo eq '' ) {
    }
    elsif ( $foo =~ m/([$valid_chars]+)/o ) {
        test not tainted $1;
    }

    if ( $foo eq '' ) {
    }
    elsif ( my @bar = $foo =~ m/([$valid_chars]+)/o ) {
        test not any_tainted @bar;
    }
}

# at scope exit, a restored localised value should have its old
# taint status, not the taint status of the current statement

{
    our $x99 = $^X;
    test tainted $x99;

    $x99 = '';
    test not tainted $x99;

    my $c = do { local $x99; $^X };
    test not tainted $x99;
}
{
    our $x99 = $^X;
    test tainted $x99;

    my $c = do { local $x99; '' };
    test tainted $x99;
}

# an mg_get of a tainted value during localization shouldn't taint the
# statement

{
    eval { local $0, eval '1' };
    test $@ eq '';
}

# [perl #8262] //g loops infinitely on tainted data

{
    my @a;
    local $::TODO = 1;
    @a[0] = $^X;
    my $i = 0;
    while(@a[0]=~ m/(.)/g ) {
	last if $i++ +> 10000;
    }
    cmp_ok $i, '+<', 10000, "infinite m//g";
}

SKIP:
{
    my $got_dualvar;
    eval 'use Scalar::Util "dualvar"; $got_dualvar++';
    skip "No Scalar::Util::dualvar" unless $got_dualvar;
    my $a = Scalar::Util::dualvar(3, $^X);
    my $b = $a + 5;
    is ($b, 8, "Arithmetic on tainted dualvars works");
}

# opening '|-' should not trigger $ENV{PATH} check

{
    SKIP: {
	skip "fork() is not available", 3 unless %Config{'d_fork'};
	skip "opening |- is not stable on threaded OpenBSD with taint", 3
            if %Config{useithreads} && $Is_OpenBSD;

	%ENV{'PATH'} = $TAINT;
	local %SIG{'PIPE'} = 'IGNORE';
	eval {
	    my $pid = open my $pipe, "|-", '-';
	    if (!defined $pid) {
		die "open failed: $!";
	    }
	    if (!$pid) {
		kill 'KILL', $$;	# child suicide
	    }
	    close $pipe;
	};
	test( ($@ && $@->{description}) !~ m/Insecure \$ENV/, 'fork triggers %ENV check');
	is $@, '',               'pipe/fork/open/close failed';
	eval {
	    open my $pipe, "|-", "$Invoke_Perl -e 1";
	    close $pipe;
	};
	test $@->{description} =~ m/Insecure \$ENV/, 'popen neglects %ENV check';
    }
}

{
    # tests for tainted format in s?printf
    dies_like( sub { printf($TAINT . "# \%s\n", "foo") },
               qr/^Insecure dependency in printf/, q/printf doesn't like tainted formats/);
    eval { printf("# \%s\n", $TAINT . "foo") };
    ok(!$@, q/printf accepts other tainted args/);
    dies_like( sub { sprintf($TAINT . "# \%s\n", "foo") },
               qr/^Insecure dependency in sprintf/, q/sprintf doesn't like tainted formats/);
    eval { sprintf("# \%s\n", $TAINT . "foo") };
    ok(!$@, q/sprintf accepts other tainted args/);
}

{
    # 40708
    my $n  = 7e9;
    8e9 - $n;

    my $val = $n;
    is ($val, '7000000000', 'Assignment to untainted variable');
    $val = $TAINT;
    $val = $n;
    is ($val, '7000000000', 'Assignment to tainted variable');
}

{
    my $val = 0;
    my $tainted = '1' . $TAINT;
    eval '$val = eval $tainted;';
    is ($val, 0, "eval doesn't like tainted strings");
    like ($@->{description}, qr/^Insecure dependency in eval/);

    # Rather nice code to get a tainted undef by from Rick Delaney
    open FH, "<", "test.pl" or die $!;
    seek FH, 0, 2 or die $!;
    $tainted = ~< *FH;

    eval 'eval $tainted';
    like ($@->{description}, qr/^Insecure dependency in eval/);
}

{
    use utf8;
    foreach my $ord (78, 163, 256) {
        # 47195
        my $line = 'A1' . $TAINT . chr $ord;
        chop $line;
        is($line, 'A1');
        $line =~ m/(A\S*)/;
        ok(!tainted($1), "\\S match with chr $ord");
    }
}

{
    my $value = "foo bar";
    my @values = split(m/\s+/, $value, 2);
    ok(!tainted(@values[1]), "result of split is not tainted if input was not tainted");
    my @values = split(m/\s+/, $value . $TAINT, 2);
    ok(tainted(@values[1]), "result of split is tainted if input was tainted");
}

# This may bomb out with the alarm signal so keep it last
SKIP: {
    skip "No alarm()"  unless %Config{d_alarm};
    # Test from RT #41831]
    # [PATCH] Bug & fix: hang when using study + taint mode (perl 5.6.1, 5.8.x)

    my $DATA = <<'END' . $TAINT;
line1 is here
line2 is here
line3 is here
line4 is here

END

    #study $DATA;

    ## don't set $SIG{ALRM}, since we'd never get to a user-level handler as
    ## perl is stuck in a regexp infinite loop!

    alarm(10);

    if ($DATA =~ m/^line2.*line4/m) {
	fail("Should not be a match")
    } else {
	pass("Match on tainted multiline data should fail promptly");
    }

    alarm(0);
}
__END__
# Keep the previous test last
