#!perl

###############################################################################

use strict;
use warnings;

use Test::More tests => 100;
use Math::BigSym;

###############################################################################
# general tests

my $mbn = 'Math::BigSym';

{    ## Some tests doesn't pass, yet.
    my $x = $mbn->new(1234);
    is($x, 1234);

    $x = $mbn->new("1234/1");
    is($x, 1234);

    $x = $mbn->new("1234/2");
    is($x, "617");

    $x = $mbn->new("100/1");
    is($x, 100);

    #$x = $mbn->new("10.0/1.0");
    #is($x, 10, qq|\$x = $mbn->new("10.0/1.0")|);

    #$x = $mbn->new("0.1/10");
    #is($x, "1/100", qq|\$x = $mbn->new("0.1/10")|);

    #$x = $mbn->new("0.1/0.1");
    #is($x, "1", qq|\$x = $mbn->new("0.1/0.1")|);

    #$x = $mbn->new("1e2/10");
    #is($x, 10, qq|\$x = $mbn->new("1e2/10")|);

    $x = $mbn->new("5/100");
    is($x, "1/20");

    #$x = $mbn->new("1e2/1e1");
    #is($x, 10, qq|\$x = $mbn->new("1e2/1e1")|);

    $x = $mbn->new("1/3");
    is("$x", "1/3");

    $x = $mbn->new("-1/3");
    is("$x", "-1/3");

    #$x = $mbn->new("1/");
    #is($x, "NaN", qq|\$x = $mbn->new("1/")|);

    #$x = $mbn->new("7e", 16);
    #is($x, 126);

    #$x = $mbn->new("1/1.2");
    #is($x, "5/6", qq|\$x = $mbn->new("1/1.2")|);

    #$x = $mbn->new("1.3/1.2");
    #is($x, "13/12", qq|\$x = $mbn->new("1.3/1.2")|);

    #$x = $mbn->new("1.2/1");
    #is($x, "6/5", qq|\$x = $mbn->new("1.2/1")|);
}

###############################################################################
# general tests

{
    use Math::BigSym qw(:constant);

    # see if rational constant works
    is((1 / 3),         '1/3',  qq|1/3 = '1/3'|);
    is((1 / 4 + 1 / 3), '7/12', qq|1/4+1/3 = '7/12'|);
    is((5 / 7 + 3 / 7), '8/7',  qq|5/7+3/7 = '8/7'|);

    is((3 / 7 + 1),     '10/7',   qq|3/7+1 = '10/7'|);
    is((3 / 7 + 1.1),   '107/70', qq|3/7+1.1 = '107/70'|);
    is((3 / 7 + 3 / 7), '6/7',    qq|3/7+3/7 = '6/7'|);

    is((3 / 7 - 1),     '-4/7',   qq|3/7-1 = '-4/7'|);
    is((3 / 7 - 1.1),   '-47/70', qq|3/7-1.1 = '-47/70'|);
    is((3 / 7 - 2 / 7), '1/7',    qq|3/7-2/7 = '1/7'|);

    is((1 + 3 / 7), '10/7', qq|1+3/7 = '10/7'|);

    is((1.1 + 3 / 7),   '107/70', qq|1.1+3/7 = '107/70'|);
    is((3 / 7 * 5 / 7), '15/49',  qq|3/7*5/7 = '15/49'|);
    is((3 / 7 / (5 / 7)), '3/5', qq|3/7 / (5/7) = '3/5'|);
    is((3 / 7 / 1),   '3/7', qq|3/7 / 1 = '3/7'|);
    is((3 / 7 / 1.5), '2/7', qq|3/7 / 1.5 = '2/7'|);
}

{
    use Math::BigSym qw(:constant);

    is(sqrt(2),   "sqrt(2)");
    is(log(100),  "log(100)");
    is(exp(10),   "exp(10)");
    is(abs(-4.5), 4.5);
    is(abs(10),   10);
    is(2.3**5.4,  'exp(27/5*log(23/10))');

    is(Math::BigSym->new("+2"), 2);
    is(Math::BigSym->new("-2"), -2);

    # Division by zero
    is(1 / 0,   -log(0));
    is(-1 / 0,  log(0));
    is(42 / 0,  "-log(0)");
    is(-42 / 0, "log(0)");

    is(-1 / 0 - 42 / 0, 2 * log(0));
    is(-1 / 0 + 42 / 0, 0);            # weird?
}

##############################################################################

my ($x, $y, $z);

$x = $mbn->new('1/4');
$y = $mbn->new('1/3');

is(($x + $y), '7/12');
is(($x * $y), '1/12');
is(($x / $y), '3/4');

$x = $mbn->new('7/5');
$x *= '3/2';
is($x, '21/10');
$x -= '0.1';
is($x, '2');

$x = $mbn->new('2/3');
$y = $mbn->new('3/2');
ok(!($x > $y));
ok($x < $y);
ok(!($x == $y));

$x = $mbn->new('-2/3');
$y = $mbn->new('3/2');
ok(!($x > $y));
ok($x < $y);
ok(!($x == $y));

$x = $mbn->new('-2/3');
$y = $mbn->new('-2/3');
ok(!($x > $y));
ok(!($x < $y));
ok($x == $y);

$x = $mbn->new('-2/3');
$y = $mbn->new('-1/3');
ok(!($x > $y));
ok($x < $y);
ok(!($x == $y));

$x = $mbn->new('-124');
$y = $mbn->new('-122');
is($x->cmp($y), -1);

$x = $mbn->new('3/7');
$y = $mbn->new('5/7');
is(($x + $y), '8/7');

$x = $mbn->new('3/7');
$y = $mbn->new('5/7');
is(($x * $y), '15/49');

$x = $mbn->new('3/5');
$y = $mbn->new('5/7');
is(($x * $y), '3/7');

$x = $mbn->new('3/5');
$y = $mbn->new('5/7');
is(($x / $y), '21/25');

## Doesn't work, yet.
#$x = $mbn->new('7/4');
#$y = $mbn->new('1');
#is(($x % $y), '3/4');

## Not exact, yet.
#$x = $mbn->new('7/4');
#$y = $mbn->new('5/13');
#is(($x % $y)->as_rat, '11/52');

## Not exact, yet.
#$x = $mbn->new('7/4');
#$y = $mbn->new('5/9');
#is(($x % $y)->as_rat, '1/12');

#$x = $mbn->new('-144/9')->bsqrt();
#is("$x", '4i');

$x = sqrt($mbn->new('144/9'));
like($x, qr{^(?:4|12/3)\z});

##############################################################################
# mixed arguments

is($mbn->new('3/7') +  (1),                '10/7');
is($mbn->new('3/10') + (1.1),              '7/5');
is($mbn->new('3/7') +  ($mbn->new(1)),     '10/7');
is($mbn->new('3/10') + ($mbn->new('1.1')), '7/5');

is($mbn->new('3/7') -  (1),                '-4/7');
is($mbn->new('3/10') - (1.1),              '-4/5');
is($mbn->new('3/7') -  ($mbn->new(1)),     '-4/7');
is($mbn->new('3/10') - ($mbn->new('1.1')), '-4/5');

is($mbn->new('3/7') *  (1),                '3/7');
is($mbn->new('3/10') * (1.1),              '33/100');
is($mbn->new('3/7') *  ($mbn->new(1)),     '3/7');
is($mbn->new('3/10') * ($mbn->new('1.1')), '33/100');

is($mbn->new('3/7') /  (1),                '3/7');
is($mbn->new('3/10') / (1.1),              '3/11');
is($mbn->new('3/7') /  ($mbn->new(1)),     '3/7');
is($mbn->new('3/10') / ($mbn->new('1.1')), '3/11');

##############################################################################
# power operator (**)

$x = $mbn->new('2/1');
$z = $x**'3';
is($x, "2");
is($z, '8');

$x = $mbn->new('1/2');
$z = $x**'3';
is($x, "1/2");
is($z, '1/8');    # 1/8

$x = $mbn->new('1/3');
$z = $x**'4';
is($z, 1 / $mbn->new(81));    # 1/81

$x = $mbn->new('2/3');
$z = $x**'4';
is($z, $mbn->new(16) / 81);    # 16/81

$x = $mbn->new('2/3');
$z = $x**'-4';
is($z, "81/16");               # 81/16

$x = $mbn->new('2/3');
$z = $x**$mbn->new('5/3');
is("$z", "exp(5/3*log(2/3))");

##############################################################################
# inc/dec (++/--)

$x = $mbn->new('3/2');
is(++$x, '5/2');
$x = $mbn->new('15/6');
is(--$x, '3/2');

##############################################################################
# sqrt()

$x = $mbn->new('144');
is(sqrt($x), '12');

$x = $mbn->new('144/16');    # new_rat() does canonalization
is(sqrt($x), '3');

##############################################################################
# broot(), blog(), bmodpow() and bmodinv()

$x = $mbn->new(2)**32;
$y = $mbn->new(4);

is($x->root($y), "exp(1/4*log(4294967296))");

$x = $mbn->new(8);
$y = $mbn->new(5033);
$z = $mbn->new(4404);

$x = $mbn->new('256.1');
is($x->log(2), "1/(log(2))*log(2561/10)");

$x = $mbn->new(144);
is(sqrt($x), 12);

$x = $mbn->new(12 * 12 * 12);

{
    my $re = join(
        '|',
        map { quotemeta($_) }
          qw(
          12
          12**(1/3)
          exp(1/3*log(1728))
          )
    );
    like($x->root('3'), qr/^(?:$re)\z/);
}

##############################################################################
# numify()

my @array = qw/1 2 3 4 5 6 7 8 9/;
$x = $mbn->new('8/8');
is($array[$x], 2);

$x = $mbn->new('16/8');
is($array[$x], 3);

$x = $mbn->new('17/8');
is($array[$x], 3);

$x = $mbn->new('33/8');
is($array[$x], 5);

$x = $mbn->new('-33/8');
is($array[$x], 6);

$x = $mbn->new('-8/1');
is($array[$x], 2);    # -8 => 2

$x = $mbn->new('33/8');
is($x->numify() * 1000, 4125);

$x = $mbn->new('-33/8');
is($x->numify() * 1000, -4125);

$x = $mbn->new('4/3');
is("$x", "4/3");

##############################################################################
# done

1;
