package Geo::Coordinates::OSTN02;
require Exporter;
use strict;
use warnings;

our @ISA = qw(Exporter);
our $VERSION = '2.02';
our @EXPORT = qw();
our @EXPORT_OK = qw(
    OSGB36_to_ETRS89
    ETRS89_to_OSGB36
);

our @ostn_data = <DATA>; # load all the data from below
our %ostn_shift_for=();

sub ETRS89_to_OSGB36 {
    return unless defined wantarray;

    my ($x,  $y,  $z ) = (@_, 0);
    my ($dx, $dy, $dz) = _find_OSTN02_shifts_at($x,$y);

    ($x, $y, $z) = _round_to_nearest_mm($x+$dx, $y+$dy, $z-$dz);

    return ($x, $y, $z) if wantarray;
    return "$x $y $z";
}

sub OSGB36_to_ETRS89 {
    return unless defined wantarray;

    my $epsilon = 0.00001;
    my ($x0, $y0, $z0) = (@_, 0);
    my ($dx, $dy, $dz) = _find_OSTN02_shifts_at($x0,$y0);
    my ($x,  $y,  $z ) = ($x0-$dx, $y0-$dy, $z0+$dz);
    my ($last_dx, $last_dy) = ($dx, $dy);
    APPROX:
    while (1) {
        ($dx, $dy, $dz) = _find_OSTN02_shifts_at($x,$y);
        ($x, $y) = ($x0-$dx, $y0-$dy);
        last APPROX if abs($dx-$last_dx)<$epsilon && abs($dy-$last_dy)<$epsilon;
        ($last_dx, $last_dy) = ($dx, $dy);
    }

    ($x, $y, $z) = _round_to_nearest_mm($x0-$dx, $y0-$dy, $z0+$dz);

    return ($x, $y, $z) if wantarray;
    return "$x $y $z";
}

sub _round_to_nearest_mm {
    my ($x,  $y,  $z ) = @_;
    $x = int($x*1000+0.5)/1000;
    $y = int($y*1000+0.5)/1000;
    $z = int($z*1000+0.5)/1000;
    return ($x, $y, $z);
}

sub _find_OSTN02_shifts_at {

    my ($x, $y) = @_;

    my $e_index = int($x/1000);
    my $n_index = int($y/1000);

    my $x0 = $e_index * 1000;
    my $y0 = $n_index * 1000;

    my $s0_ref = _get_ostn_ref($e_index+0, $n_index+0) || die "OSTN not defined for $x $y\n";
    my $s1_ref = _get_ostn_ref($e_index+1, $n_index+0) || die "OSTN not defined for $x $y\n";
    my $s2_ref = _get_ostn_ref($e_index+0, $n_index+1) || die "OSTN not defined for $x $y\n";
    my $s3_ref = _get_ostn_ref($e_index+1, $n_index+1) || die "OSTN not defined for $x $y\n";

    my $dx = $x - $x0; # offset within square
    my $dy = $y - $y0;

    my $t = $dx/1000;
    my $u = $dy/1000;

    my $f0 = (1-$t)*(1-$u);
    my $f1 =    $t *(1-$u);
    my $f2 = (1-$t)*   $u ;
    my $f3 =    $t *   $u ;

    my $se = $f0*$s0_ref->[0] + $f1*$s1_ref->[0] + $f2*$s2_ref->[0] + $f3*$s3_ref->[0];
    my $sn = $f0*$s0_ref->[1] + $f1*$s1_ref->[1] + $f2*$s2_ref->[1] + $f3*$s3_ref->[1];
    my $sg = $f0*$s0_ref->[2] + $f1*$s1_ref->[2] + $f2*$s2_ref->[2] + $f3*$s3_ref->[2];

    return ($se, $sn, $sg);
}

sub _get_ostn_ref {
    my $x = shift;
    my $y = shift;
    my $key = $x + ($y*701) + 1;

    return $ostn_shift_for{$key} if exists $ostn_shift_for{$key};

    my $lo = 0;
    my $hi = @ostn_data;
    my $mid;
    while (1) {
        $mid = ($hi+$lo)/2;
        my ($k, @data) = split ' ', $ostn_data[$mid];
        $ostn_shift_for{$k} = \@data;
        last        if $k == $key;
        $lo = $mid  if $k  < $key;
        $hi = $mid  if $k  > $key;
    }

    return $ostn_shift_for{$key};
}

1;

=pod

=head1 NAME

Geo::Coordinates::OSTN02 -- Implement the OSTN02 transformation


=head1 SYNOPSIS

  use Geo::Coordinates::OSTN02 qw/OSGB36_to_ETRS89 ETRS89_to_OSGB36/;

  ($x, $y, $z) = OSGB36_to_ETRS89($e, $n, $h);
  ($e, $n, $h) = ETRS89_to_OSGB36($x, $y, $z);

=head1 DESCRIPTION

The purpose and use of these modules is described briefly in the companion
L<Geo::Coordinates::OSGB> modules.  In essence they implement in Perl the
Ordnance Survey's OSTN02 transformation which is part of the current
definition of the British National Grid.  You are strongly recommended to
read the official public guides and other introductory documents that are
published by the Ordnance Survey of Great Britain with a wealth of other
technical information about the OSTN02 transformation.  The best starting
point on the web is
http://www.ordnancesurvey.co.uk/oswebsite/gps/information/index.html

=head1 FUNCTIONS

The following functions can be exported from the C<Geo::Coordinates::OSTN02>
module:

    OSGB36_to_ETRS89
    ETRS89_to_OSGB36

None is exported by default.

=over 4

=item OSGB36_to_ETRS89(easting, northing, [elevation])

Transforms from normal OSGB36 grid references to a pseudo-grid based on the WGS84/ETRS89 geoid
model, which can then be translated to lat/long coordinates using C<grid_to_ll()> with the 'WGS84'
parameter.

   my $elevation = '100'; # in metres
   my ($e, $n) = parse_grid_trad('TQ304293');
   my ($x, $y, $z) = OSGB36_to_ETRS89($e, $n, $elevation);
   my ($lat, $lon) = grid_to_ll($x, $y, 'ETRS89'); # or you can say 'WGS84' if you like - same thing

Elevation will default to 0 metres if you omit it.

=item ETRS89_to_OSGB36(x, y, [z])

Transforms WGS84/ETRS89 pseudo-grid coordinates into OSGB36 grid references.

  my ($lat, $lon, $height) = (51.5, -1, 10); # Somewhere in London
  my ($x, $y) = ll_to_grid($lat, $lon, 'ETRS89'); # or 'WGS84'
  my ($e, $n, $elevation) = ETRS89_to_OSGB36($x, $y, $height);

=back

=head1 IMPLEMENTATION NOTES

Since 2002 the British Ordnance Survey have defined the UK national grid not
as a projection from their own model of the earth (the Airy 1830 geoid,
revised 1936, known as OSGB36), but as a simple table of calculated
differences from a projection based on the European standard geoid ETRS89
(which is for Europe a functional equivalent of the international WGS84
geoid).  The idea is that you project your WGS84 latitude and longitude
coordinates into WGS84 (or ETRS89) grid references, and then look up the
appropriate three dimensional correction in the OS table, and add the results
to your grid reference to give you a standard British National Grid reference
that you can use with British OS maps.  Going back the other way is slightly
more complicated, as you have to work backwards, but a simple iteration will
do the job.  This package implements these lookups and adjustments.  You give
it a three dimensional grid reference (easting, northing, and altitude, all
in metres) and the package returns it corrected from one system to the other.

The problem in the implementation is that the table is huge, and most of it
is empty as the OS have chosen to leave the transformation undefined for
areas that are off shore.  So this package only works properly for grid
references that are actually on one or more OS maps.  The complete table
(including all the 0 lines) contains nearly 1 million lines with four data
points and a key.  In text form as supplied by the OS that is about 36M bytes
of table.  Even storing the numbers in binary form it would still need about
9M bytes.  About 2/3 of the points are blank though, so leaving them out
brings the amount of text down to about 10M bytes and binary data to about 3M
bytes.  For simplicity this package includes the text of the table with the
blank lines omitted.

A future version might compress the table into binary form.  The ranges of the corrections are
as follows:

  Max k: 861292 e: 665 n: 1228 x: 103.443 y: -50.168 z: 57.668
  Min k:     80 e:   2 n:    0 x:  86.275 y: -81.603 z: 43.982
  Range  861212    663    1228     17.168     31.435    13.686
  Bits       20                    15         15        14
  Total bits needed 64 = 8 bytes x 309792 lines = 2478336 = 2.4M bytes

Since we omit the blank lines, we cannot simply get the nth row for index n
so we keep two structures:  @ostn_data holds all the data in a big array, and
%ostn_shift_for is a hash that caches lines in the table that we have already
looked up.  To get a line, we calculate the index number from an (x,y) pair,
look for that in the hash, and use it if we can.  If it is not there, we then
search through the array using a simple binary chop, and cache the result.

This works pretty quickly, the only slow bit is loading the array at the
beginning, but it is much faster and need *much* less memory than loading all
the data into the hash (which would be simpler, and is what the original
version did, but it was too slow to be usable and meant that the tests failed
on many smaller systems as part of CPAN testing).

Loading the array takes about 1.7 seconds on my Windows machine (a 1.8G Hz
Intel Pentium M processor with 1G byte of memory) and just under 0.5 seconds
on my Linux system (a 2.8G Hz Intel Pentium 4 processor with 512M bytes of
memory).  I think this probably says more about speed of the disks I have
(and probably the efficiency of Perl under Linux), but your results should be
comparable.  Once the data is loaded, calling the routines is reasonably
quick.

=head1 BUGS

Please report any to toby@cpan.org

=head1 EXAMPLES

See the test routines included in the distribution.

=head1 AUTHOR

Toby Thurston --- 22 Sep 2008

web: http://www.wildfire.dircon.co.uk

=head1 SEE ALSO

See L<Geo::Coordinates::OSGB> for the main routines.
=cut
__DATA__
80 91.888 -81.603 53.184
81 91.903 -81.597 53.178
82 91.917 -81.592 53.173
83 91.936 -81.582 53.168
84 91.956 -81.569 53.164
85 91.976 -81.557 53.160
86 91.996 -81.545 53.157
87 92.016 -81.534 53.154
88 92.035 -81.522 53.151
89 92.056 -81.510 53.148
90 92.076 -81.497 53.143
91 92.096 -81.485 53.139
92 92.116 -81.473 53.135
93 92.139 -81.462 53.131
94 92.163 -81.450 53.127
95 92.188 -81.439 53.123
96 92.212 -81.428 53.117
97 92.236 -81.416 53.112
98 92.261 -81.405 53.105
99 92.285 -81.394 53.098
100 92.310 -81.382 53.092
780 91.873 -81.579 53.192
781 91.887 -81.574 53.184
782 91.902 -81.569 53.178
783 91.916 -81.563 53.172
784 91.932 -81.557 53.167
785 91.951 -81.545 53.162
786 91.971 -81.533 53.158
787 91.992 -81.521 53.155
788 92.012 -81.509 53.152
789 92.031 -81.497 53.148
790 92.052 -81.484 53.145
791 92.072 -81.473 53.140
792 92.091 -81.461 53.136
793 92.115 -81.450 53.132
794 92.140 -81.437 53.128
795 92.164 -81.426 53.124
796 92.189 -81.415 53.119
797 92.213 -81.403 53.114
798 92.237 -81.392 53.108
799 92.262 -81.381 53.102
800 92.286 -81.369 53.094
801 92.311 -81.358 53.089
802 92.336 -81.347 53.082
1480 91.858 -81.556 53.201
1481 91.872 -81.550 53.193
1482 91.887 -81.545 53.185
1483 91.901 -81.540 53.179
1484 91.916 -81.535 53.172
1485 91.930 -81.529 53.166
1486 91.947 -81.520 53.161
1487 91.967 -81.508 53.157
1488 91.988 -81.496 53.154
1489 92.007 -81.484 53.150
1490 92.027 -81.472 53.146
1491 92.048 -81.460 53.143
1492 92.068 -81.448 53.137
1493 92.092 -81.437 53.133
1494 92.117 -81.425 53.129
1495 92.141 -81.414 53.124
1496 92.165 -81.403 53.120
1497 92.190 -81.391 53.116
1498 92.214 -81.380 53.110
1499 92.239 -81.368 53.105
1500 92.263 -81.356 53.098
1501 92.288 -81.345 53.091
1502 92.312 -81.334 53.085
1503 92.337 -81.322 53.079
1504 92.356 -81.315 53.072
1569 93.499 -80.739 53.060
1570 93.513 -80.729 53.056
1571 93.520 -80.719 53.050
1572 93.528 -80.709 53.045
2181 91.857 -81.527 53.204
2182 91.872 -81.522 53.195
2183 91.886 -81.516 53.187
2184 91.900 -81.511 53.180
2185 91.915 -81.506 53.173
2186 91.929 -81.500 53.167
2187 91.944 -81.495 53.161
2188 91.963 -81.484 53.157
2189 91.984 -81.472 53.152
2190 92.003 -81.459 53.148
2191 92.023 -81.447 53.144
2192 92.044 -81.435 53.140
2193 92.069 -81.424 53.135
2194 92.093 -81.412 53.130
2195 92.118 -81.401 53.126
2196 92.142 -81.390 53.121
2197 92.166 -81.378 53.117
2198 92.191 -81.367 53.112
2199 92.215 -81.356 53.106
2200 92.240 -81.344 53.101
2201 92.264 -81.333 53.094
2202 92.289 -81.321 53.088
2203 92.313 -81.310 53.082
2204 92.333 -81.302 53.075
2205 92.351 -81.294 53.069
2206 92.369 -81.287 53.061
2226 92.739 -81.134 53.425
2227 92.758 -81.127 53.415
2228 92.776 -81.118 53.405
2229 92.794 -81.111 53.394
2230 92.813 -81.103 53.384
2269 93.478 -80.729 53.068
2270 93.494 -80.718 53.063
2271 93.510 -80.707 53.058
2272 93.518 -80.698 53.053
2273 93.525 -80.688 53.048
2274 93.533 -80.678 53.042
2275 93.540 -80.667 53.037
2276 93.547 -80.657 53.032
2277 93.554 -80.648 53.026
2278 93.561 -80.638 53.019
2279 93.568 -80.628 53.011
2280 93.575 -80.618 53.003
2881 91.843 -81.503 53.219
2882 91.857 -81.497 53.208
2883 91.871 -81.492 53.198
2884 91.885 -81.487 53.189
2885 91.900 -81.481 53.182
2886 91.915 -81.476 53.174
2887 91.929 -81.472 53.168
2888 91.943 -81.466 53.162
2889 91.959 -81.459 53.157
2890 91.979 -81.447 53.152
2891 91.999 -81.434 53.147
2892 92.021 -81.423 53.143
2893 92.045 -81.412 53.139
2894 92.070 -81.400 53.133
2895 92.094 -81.389 53.128
2896 92.119 -81.377 53.123
2897 92.143 -81.365 53.118
2898 92.167 -81.354 53.113
2899 92.192 -81.343 53.108
2900 92.217 -81.331 53.103
2901 92.241 -81.320 53.097
2902 92.266 -81.309 53.091
2903 92.290 -81.297 53.084
2904 92.309 -81.289 53.078
2905 92.328 -81.281 53.072
2906 92.346 -81.274 53.065
2907 92.365 -81.266 53.058
2926 92.716 -81.121 53.429
2927 92.734 -81.114 53.419
2928 92.753 -81.106 53.409
2929 92.771 -81.099 53.397
2930 92.790 -81.091 53.387
2931 92.808 -81.083 53.377
2932 92.827 -81.075 53.366
2933 92.845 -81.068 53.357
2934 92.864 -81.060 53.347
2935 92.882 -81.052 53.338
2936 92.901 -81.045 53.329
2969 93.457 -80.718 53.075
2970 93.473 -80.708 53.070
2971 93.489 -80.698 53.065
2972 93.505 -80.687 53.060
2973 93.516 -80.676 53.056
2974 93.523 -80.667 53.051
2975 93.531 -80.657 53.046
2976 93.538 -80.647 53.040
2977 93.545 -80.636 53.035
2978 93.552 -80.626 53.029
2979 93.559 -80.616 53.022
2980 93.566 -80.607 53.015
2981 93.573 -80.597 53.007
2982 93.579 -80.586 52.999
2983 93.586 -80.576 52.991
3581 91.827 -81.479 53.239
3582 91.842 -81.474 53.226
3583 91.856 -81.469 53.214
3584 91.871 -81.463 53.203
3585 91.885 -81.458 53.193
3586 91.899 -81.453 53.185
3587 91.914 -81.447 53.177
3588 91.928 -81.442 53.170
3589 91.943 -81.437 53.163
3590 91.957 -81.431 53.157
3591 91.975 -81.422 53.152
3592 91.997 -81.410 53.147
3593 92.022 -81.399 53.142
3594 92.047 -81.387 53.137
3595 92.071 -81.376 53.132
3596 92.095 -81.365 53.127
3597 92.120 -81.353 53.121
3598 92.144 -81.342 53.116
3599 92.169 -81.331 53.110
3600 92.193 -81.319 53.105
3601 92.218 -81.308 53.099
3602 92.242 -81.296 53.094
3603 92.267 -81.284 53.087
3604 92.286 -81.276 53.080
3605 92.305 -81.268 53.074
3606 92.323 -81.261 53.068
3607 92.342 -81.253 53.061
3608 92.360 -81.246 53.054
3609 92.379 -81.238 53.047
3626 92.693 -81.108 53.434
3627 92.711 -81.101 53.423
3628 92.730 -81.093 53.413
3629 92.748 -81.086 53.402
3630 92.767 -81.078 53.391
3631 92.785 -81.071 53.380
3632 92.804 -81.063 53.369
3633 92.822 -81.055 53.359
3634 92.841 -81.048 53.349
3635 92.859 -81.039 53.339
3636 92.877 -81.032 53.329
3637 92.896 -81.024 53.320
3638 92.915 -81.017 53.310
3639 92.933 -81.009 53.301
3669 93.435 -80.708 53.079
3670 93.451 -80.698 53.075
3671 93.468 -80.687 53.072
3672 93.484 -80.676 53.067
3673 93.500 -80.666 53.062
3674 93.514 -80.655 53.058
3675 93.521 -80.645 53.054
3676 93.529 -80.635 53.049
3677 93.536 -80.626 53.043
3678 93.543 -80.615 53.037
3679 93.550 -80.605 53.031
3680 93.557 -80.595 53.024
3681 93.564 -80.585 53.017
3682 93.571 -80.576 53.010
3683 93.577 -80.565 53.002
3684 93.584 -80.555 52.994
3685 93.591 -80.545 52.985
4282 91.827 -81.450 53.247
4283 91.841 -81.445 53.233
4284 91.856 -81.440 53.221
4285 91.870 -81.434 53.209
4286 91.884 -81.429 53.199
4287 91.899 -81.424 53.189
4288 91.913 -81.419 53.180
4289 91.928 -81.413 53.173
4290 91.942 -81.408 53.166
4291 91.956 -81.403 53.159
4292 91.974 -81.398 53.153
4293 91.999 -81.386 53.147
4294 92.023 -81.375 53.142
4295 92.048 -81.363 53.137
4296 92.072 -81.352 53.131
4297 92.096 -81.340 53.125
4298 92.121 -81.329 53.119
4299 92.145 -81.318 53.114
4300 92.170 -81.306 53.108
4301 92.195 -81.295 53.102
4302 92.219 -81.284 53.096
4303 92.243 -81.272 53.090
4304 92.263 -81.263 53.084
4305 92.282 -81.256 53.076
4306 92.300 -81.248 53.070
4307 92.319 -81.240 53.064
4308 92.337 -81.233 53.058
4309 92.356 -81.225 53.051
4310 92.374 -81.218 53.044
4311 92.392 -81.210 53.037
4327 92.688 -81.088 53.429
4328 92.707 -81.080 53.417
4329 92.725 -81.073 53.406
4330 92.743 -81.065 53.395
4331 92.762 -81.058 53.384
4332 92.781 -81.050 53.373
4333 92.799 -81.043 53.361
4334 92.817 -81.035 53.351
4335 92.836 -81.027 53.340
4336 92.855 -81.020 53.329
4337 92.873 -81.012 53.320
4338 92.892 -81.004 53.309
4339 92.910 -80.996 53.300
4340 92.928 -80.989 53.291
4369 93.414 -80.698 53.081
4370 93.430 -80.687 53.078
4371 93.446 -80.676 53.075
4372 93.463 -80.666 53.071
4373 93.479 -80.655 53.068
4374 93.495 -80.645 53.064
4375 93.512 -80.634 53.060
4376 93.519 -80.624 53.055
4377 93.526 -80.614 53.051
4378 93.534 -80.604 53.045
4379 93.541 -80.594 53.039
4380 93.548 -80.584 53.033
4381 93.555 -80.574 53.026
4382 93.562 -80.564 53.019
4383 93.569 -80.554 53.011
4384 93.575 -80.544 53.003
4385 93.582 -80.534 52.995
4386 93.589 -80.524 52.986
4387 93.593 -80.519 52.977
4983 91.826 -81.422 53.257
4984 91.841 -81.416 53.242
4985 91.855 -81.411 53.228
4986 91.869 -81.406 53.216
4987 91.884 -81.400 53.205
4988 91.898 -81.395 53.195
4989 91.913 -81.390 53.185
4990 91.927 -81.384 53.177
4991 91.940 -81.378 53.169
4992 91.954 -81.373 53.162
4993 91.968 -81.367 53.155
4994 92.000 -81.363 53.149
4995 92.024 -81.351 53.142
4996 92.049 -81.340 53.137
4997 92.073 -81.328 53.131
4998 92.097 -81.316 53.124
4999 92.122 -81.305 53.118
5000 92.147 -81.293 53.112
5001 92.171 -81.282 53.106
5002 92.196 -81.271 53.100
5003 92.220 -81.259 53.093
5004 92.240 -81.251 53.086
5005 92.259 -81.243 53.080
5006 92.277 -81.236 53.072
5007 92.296 -81.228 53.065
5008 92.314 -81.220 53.059
5009 92.333 -81.212 53.054
5010 92.351 -81.205 53.047
5011 92.369 -81.197 53.040
5012 92.388 -81.190 53.033
5027 92.665 -81.075 53.435
5028 92.684 -81.068 53.423
5029 92.702 -81.060 53.411
5030 92.721 -81.052 53.400
5031 92.739 -81.045 53.388
5032 92.758 -81.037 53.376
5033 92.776 -81.030 53.364
5034 92.794 -81.022 53.352
5035 92.813 -81.014 53.341
5036 92.832 -81.007 53.330
5037 92.850 -80.999 53.320
5038 92.868 -80.992 53.309
5039 92.887 -80.984 53.299
5040 92.905 -80.977 53.289
5041 92.924 -80.968 53.280
5042 92.942 -80.961 53.271
5069 93.392 -80.687 53.082
5070 93.409 -80.676 53.078
5071 93.425 -80.666 53.076
5072 93.441 -80.655 53.073
5073 93.458 -80.645 53.070
5074 93.474 -80.634 53.067
5075 93.490 -80.624 53.065
5076 93.507 -80.613 53.061
5077 93.517 -80.603 53.056
5078 93.524 -80.593 53.052
5079 93.532 -80.583 53.046
5080 93.539 -80.573 53.041
5081 93.546 -80.563 53.035
5082 93.553 -80.553 53.028
5083 93.560 -80.543 53.021
5084 93.567 -80.533 53.013
5085 93.573 -80.523 53.005
5086 93.580 -80.514 52.997
5087 93.583 -80.510 52.988
5088 93.586 -80.504 52.979
5089 93.590 -80.500 52.970
5090 93.593 -80.495 52.961
5683 91.811 -81.398 53.286
5684 91.825 -81.393 53.269
5685 91.840 -81.387 53.253
5686 91.855 -81.382 53.238
5687 91.869 -81.377 53.225
5688 91.883 -81.371 53.213
5689 91.896 -81.365 53.202
5690 91.911 -81.359 53.192
5691 91.925 -81.354 53.182
5692 91.939 -81.349 53.174
5693 91.953 -81.343 53.165
5694 91.966 -81.337 53.158
5695 92.000 -81.339 53.151
5696 92.026 -81.328 53.144
5697 92.050 -81.315 53.137
5698 92.074 -81.304 53.131
5699 92.099 -81.293 53.124
5700 92.123 -81.281 53.117
5701 92.148 -81.270 53.110
5702 92.172 -81.259 53.104
5703 92.197 -81.247 53.098
5704 92.217 -81.238 53.090
5705 92.236 -81.231 53.083
5706 92.254 -81.223 53.076
5707 92.273 -81.215 53.069
5708 92.291 -81.208 53.062
5709 92.309 -81.200 53.055
5710 92.328 -81.192 53.049
5711 92.347 -81.184 53.042
5712 92.365 -81.177 53.036
5713 92.383 -81.169 53.030
5727 92.642 -81.062 53.441
5728 92.660 -81.055 53.429
5729 92.679 -81.047 53.416
5730 92.698 -81.039 53.404
5731 92.716 -81.032 53.392
5732 92.734 -81.024 53.380
5733 92.753 -81.017 53.368
5734 92.772 -81.009 53.355
5735 92.790 -81.002 53.343
5736 92.808 -80.994 53.331
5737 92.827 -80.986 53.320
5738 92.845 -80.979 53.309
5739 92.864 -80.971 53.298
5740 92.882 -80.964 53.288
5741 92.901 -80.956 53.278
5742 92.919 -80.949 53.268
5743 92.938 -80.941 53.260
5769 93.371 -80.676 53.081
5770 93.387 -80.667 53.077
5771 93.404 -80.656 53.074
5772 93.420 -80.645 53.071
5773 93.436 -80.635 53.069
5774 93.452 -80.624 53.067
5775 93.469 -80.613 53.064
5776 93.485 -80.603 53.063
5777 93.502 -80.592 53.061
5778 93.515 -80.582 53.057
5779 93.522 -80.572 53.052
5780 93.530 -80.562 53.047
5781 93.537 -80.552 53.042
5782 93.544 -80.541 53.035
5783 93.551 -80.532 53.029
5784 93.558 -80.522 53.022
5785 93.565 -80.512 53.015
5786 93.570 -80.504 53.006
5787 93.573 -80.500 52.998
5788 93.576 -80.495 52.989
5789 93.580 -80.490 52.980
5790 93.583 -80.485 52.971
5791 93.586 -80.481 52.962
5792 93.590 -80.476 52.953
6384 91.811 -81.369 53.300
6385 91.825 -81.364 53.282
6386 91.839 -81.358 53.265
6387 91.853 -81.353 53.249
6388 91.867 -81.347 53.235
6389 91.881 -81.341 53.222
6390 91.895 -81.335 53.210
6391 91.909 -81.330 53.199
6392 91.923 -81.325 53.189
6393 91.937 -81.318 53.179
6394 91.951 -81.313 53.170
6395 91.965 -81.307 53.161
6396 91.985 -81.304 53.154
6397 92.028 -81.306 53.146
6398 92.052 -81.293 53.139
6399 92.076 -81.281 53.132
6400 92.100 -81.268 53.124
6401 92.125 -81.257 53.116
6402 92.149 -81.246 53.109
6403 92.173 -81.234 53.103
6404 92.194 -81.225 53.095
6405 92.213 -81.218 53.088
6406 92.231 -81.210 53.080
6407 92.249 -81.202 53.073
6408 92.268 -81.195 53.065
6409 92.286 -81.187 53.058
6410 92.305 -81.180 53.051
6411 92.323 -81.172 53.045
6412 92.342 -81.165 53.039
6413 92.360 -81.156 53.032
6414 92.379 -81.149 53.026
6415 92.398 -81.141 53.018
6428 92.638 -81.043 53.436
6429 92.656 -81.035 53.423
6430 92.674 -81.027 53.409
6431 92.693 -81.019 53.397
6432 92.711 -81.011 53.384
6433 92.730 -81.004 53.371
6434 92.748 -80.996 53.359
6435 92.767 -80.989 53.346
6436 92.785 -80.981 53.333
6437 92.804 -80.974 53.321
6438 92.822 -80.966 53.309
6439 92.841 -80.958 53.298
6440 92.859 -80.951 53.287
6441 92.878 -80.943 53.277
6442 92.896 -80.936 53.267
6443 92.915 -80.928 53.257
6444 92.933 -80.920 53.248
6445 92.951 -80.913 53.240
6470 93.366 -80.656 53.074
6471 93.382 -80.645 53.071
6472 93.399 -80.635 53.069
6473 93.415 -80.624 53.066
6474 93.431 -80.613 53.065
6475 93.447 -80.603 53.063
6476 93.464 -80.592 53.062
6477 93.480 -80.582 53.061
6478 93.497 -80.571 53.058
6479 93.513 -80.560 53.055
6480 93.520 -80.551 53.052
6481 93.528 -80.541 53.047
6482 93.535 -80.531 53.042
6483 93.542 -80.520 53.036
6484 93.549 -80.510 53.029
6485 93.556 -80.500 53.022
6486 93.560 -80.494 53.015
6487 93.563 -80.490 53.007
6488 93.567 -80.485 52.998
6489 93.570 -80.481 52.990
6490 93.573 -80.475 52.981
6491 93.576 -80.471 52.972
6492 93.580 -80.466 52.963
6493 93.583 -80.462 52.954
6494 93.586 -80.457 52.944
7085 91.809 -81.340 53.315
7086 91.823 -81.334 53.297
7087 91.837 -81.328 53.279
7088 91.851 -81.322 53.262
7089 91.865 -81.317 53.247
7090 91.879 -81.311 53.233
7091 91.893 -81.306 53.220
7092 91.907 -81.300 53.208
7093 91.921 -81.294 53.197
7094 91.935 -81.289 53.186
7095 91.949 -81.283 53.176
7096 91.963 -81.278 53.166
7097 91.977 -81.271 53.158
7098 92.024 -81.279 53.150
7099 92.054 -81.271 53.142
7100 92.078 -81.259 53.134
7101 92.102 -81.246 53.125
7102 92.126 -81.234 53.117
7103 92.150 -81.221 53.109
7104 92.171 -81.212 53.101
7105 92.190 -81.205 53.094
7106 92.208 -81.197 53.086
7107 92.226 -81.190 53.078
7108 92.245 -81.182 53.070
7109 92.264 -81.174 53.062
7110 92.282 -81.167 53.055
7111 92.300 -81.159 53.048
7112 92.319 -81.152 53.042
7113 92.337 -81.144 53.035
7114 92.356 -81.137 53.028
7115 92.374 -81.129 53.022
7116 92.392 -81.121 53.015
7129 92.633 -81.022 53.430
7130 92.651 -81.014 53.417
7131 92.670 -81.007 53.403
7132 92.689 -80.999 53.390
7133 92.707 -80.991 53.377
7134 92.725 -80.983 53.363
7135 92.744 -80.976 53.350
7136 92.762 -80.968 53.336
7137 92.781 -80.961 53.322
7138 92.799 -80.953 53.311
7139 92.817 -80.945 53.299
7140 92.836 -80.938 53.287
7141 92.855 -80.930 53.276
7142 92.873 -80.923 53.265
7143 92.892 -80.915 53.255
7144 92.910 -80.908 53.245
7145 92.928 -80.900 53.236
7146 92.947 -80.892 53.227
7170 93.345 -80.645 53.069
7171 93.361 -80.635 53.067
7172 93.377 -80.624 53.065
7173 93.394 -80.613 53.063
7174 93.410 -80.603 53.061
7175 93.426 -80.592 53.060
7176 93.442 -80.582 53.059
7177 93.459 -80.571 53.059
7178 93.475 -80.560 53.057
7179 93.492 -80.550 53.056
7180 93.508 -80.540 53.054
7181 93.518 -80.529 53.050
7182 93.525 -80.519 53.046
7183 93.533 -80.510 53.041
7184 93.540 -80.499 53.035
7185 93.547 -80.489 53.029
7186 93.550 -80.485 53.022
7187 93.553 -80.480 53.015
7188 93.557 -80.475 53.007
7189 93.560 -80.471 52.999
7190 93.563 -80.466 52.990
7191 93.567 -80.461 52.981
7192 93.570 -80.457 52.972
7193 93.573 -80.452 52.963
7194 93.577 -80.447 52.955
7195 93.580 -80.443 52.945
7196 93.583 -80.438 52.935
7786 91.807 -81.309 53.333
7787 91.821 -81.304 53.314
7788 91.835 -81.298 53.295
7789 91.849 -81.293 53.277
7790 91.863 -81.287 53.261
7791 91.877 -81.281 53.246
7792 91.891 -81.275 53.232
7793 91.905 -81.270 53.219
7794 91.919 -81.265 53.206
7795 91.933 -81.259 53.194
7796 91.947 -81.253 53.183
7797 91.961 -81.247 53.173
7798 91.975 -81.242 53.164
7799 92.009 -81.244 53.155
7800 92.056 -81.249 53.146
7801 92.079 -81.236 53.137
7802 92.103 -81.224 53.128
7803 92.128 -81.212 53.120
7804 92.147 -81.199 53.111
7805 92.165 -81.191 53.103
7806 92.185 -81.184 53.094
7807 92.203 -81.177 53.085
7808 92.222 -81.169 53.077
7809 92.240 -81.162 53.068
7810 92.259 -81.154 53.060
7811 92.277 -81.146 53.053
7812 92.296 -81.139 53.046
7813 92.314 -81.131 53.039
7814 92.333 -81.124 53.032
7815 92.351 -81.116 53.026
7816 92.370 -81.108 53.019
7817 92.388 -81.101 53.012
7830 92.628 -81.002 53.425
7831 92.647 -80.994 53.411
7832 92.665 -80.986 53.396
7833 92.684 -80.979 53.382
7834 92.702 -80.971 53.369
7835 92.721 -80.964 53.355
7836 92.739 -80.955 53.341
7837 92.758 -80.948 53.327
7838 92.776 -80.940 53.313
7839 92.795 -80.933 53.300
7840 92.813 -80.925 53.288
7841 92.832 -80.917 53.275
7842 92.850 -80.910 53.264
7843 92.868 -80.902 53.253
7844 92.887 -80.895 53.243
7845 92.906 -80.887 53.233
7846 92.924 -80.880 53.223
7847 92.942 -80.872 53.214
7848 92.961 -80.864 53.206
7870 93.323 -80.635 53.064
7871 93.339 -80.624 53.062
7872 93.356 -80.613 53.059
7873 93.372 -80.603 53.057
7874 93.389 -80.592 53.056
7875 93.405 -80.582 53.055
7876 93.421 -80.572 53.054
7877 93.437 -80.561 53.054
7878 93.454 -80.551 53.054
7879 93.470 -80.540 53.053
7880 93.487 -80.529 53.053
7881 93.503 -80.519 53.052
7882 93.516 -80.508 53.048
7883 93.523 -80.498 53.044
7884 93.531 -80.488 53.039
7885 93.537 -80.479 53.034
7886 93.540 -80.475 53.028
7887 93.543 -80.470 53.021
7888 93.547 -80.466 53.014
7889 93.550 -80.461 53.006
7890 93.553 -80.457 52.998
7891 93.557 -80.452 52.990
7892 93.560 -80.447 52.981
7893 93.563 -80.442 52.972
7894 93.567 -80.438 52.964
7895 93.570 -80.433 52.955
7896 93.573 -80.428 52.945
7897 93.577 -80.424 52.936
7898 93.580 -80.419 52.927
8486 91.791 -81.285 53.374
8487 91.805 -81.280 53.353
8488 91.819 -81.274 53.332
8489 91.834 -81.268 53.313
8490 91.847 -81.262 53.294
8491 91.861 -81.257 53.277
8492 91.875 -81.251 53.261
8493 91.889 -81.246 53.245
8494 91.904 -81.240 53.231
8495 91.917 -81.234 53.218
8496 91.931 -81.229 53.204
8497 91.945 -81.223 53.193
8498 91.959 -81.218 53.182
8499 91.974 -81.212 53.171
8500 91.994 -81.209 53.161
8501 92.048 -81.219 53.151
8502 92.081 -81.214 53.141
8503 92.105 -81.201 53.132
8504 92.126 -81.190 53.123
8505 92.142 -81.179 53.114
8506 92.158 -81.168 53.105
8507 92.177 -81.162 53.094
8508 92.197 -81.155 53.085
8509 92.216 -81.148 53.076
8510 92.236 -81.141 53.067
8511 92.254 -81.133 53.059
8512 92.273 -81.126 53.051
8513 92.291 -81.118 53.044
8514 92.309 -81.111 53.037
8515 92.328 -81.103 53.030
8516 92.347 -81.096 53.023
8517 92.365 -81.088 53.016
8518 92.383 -81.080 53.009
8531 92.624 -80.981 53.419
8532 92.642 -80.974 53.405
8533 92.661 -80.966 53.389
8534 92.679 -80.958 53.375
8535 92.698 -80.951 53.362
8536 92.716 -80.943 53.347
8537 92.734 -80.936 53.332
8538 92.753 -80.928 53.318
8539 92.772 -80.920 53.304
8540 92.790 -80.912 53.290
8541 92.808 -80.905 53.277
8542 92.827 -80.897 53.265
8543 92.845 -80.889 53.253
8544 92.864 -80.882 53.241
8545 92.882 -80.874 53.231
8546 92.901 -80.867 53.220
8547 92.919 -80.859 53.210
8548 92.938 -80.851 53.200
8549 92.956 -80.844 53.193
8550 92.975 -80.836 53.184
8551 92.991 -80.826 53.176
8571 93.318 -80.614 53.055
8572 93.334 -80.604 53.053
8573 93.351 -80.593 53.051
8574 93.367 -80.582 53.049
8575 93.383 -80.572 53.048
8576 93.400 -80.561 53.047
8577 93.416 -80.551 53.047
8578 93.432 -80.540 53.048
8579 93.449 -80.529 53.048
8580 93.465 -80.519 53.048
8581 93.482 -80.508 53.048
8582 93.498 -80.497 53.048
8583 93.514 -80.487 53.047
8584 93.521 -80.477 53.044
8585 93.526 -80.469 53.038
8586 93.530 -80.465 53.032
8587 93.533 -80.460 53.026
8588 93.537 -80.456 53.019
8589 93.540 -80.451 53.012
8590 93.543 -80.447 53.005
8591 93.547 -80.442 52.998
8592 93.550 -80.438 52.990
8593 93.553 -80.432 52.981
8594 93.557 -80.428 52.972
8595 93.560 -80.423 52.964
8596 93.563 -80.419 52.954
8597 93.567 -80.414 52.945
8598 93.570 -80.410 52.937
8599 93.573 -80.405 52.928
8600 93.577 -80.400 52.919
9187 91.790 -81.255 53.395
9188 91.803 -81.249 53.373
9189 91.817 -81.244 53.352
9190 91.831 -81.238 53.332
9191 91.846 -81.233 53.312
9192 91.860 -81.227 53.294
9193 91.873 -81.221 53.277
9194 91.887 -81.215 53.260
9195 91.901 -81.210 53.245
9196 91.916 -81.204 53.231
9197 91.930 -81.199 53.217
9198 91.943 -81.193 53.204
9199 91.957 -81.187 53.191
9200 91.971 -81.182 53.180
9201 91.986 -81.176 53.169
9202 92.032 -81.184 53.158
9203 92.083 -81.191 53.147
9204 92.104 -81.180 53.137
9205 92.120 -81.168 53.127
9206 92.136 -81.158 53.117
9207 92.152 -81.147 53.106
9208 92.170 -81.138 53.096
9209 92.190 -81.131 53.086
9210 92.209 -81.124 53.076
9211 92.228 -81.118 53.067
9212 92.248 -81.112 53.059
9213 92.268 -81.105 53.050
9214 92.287 -81.098 53.043
9215 92.305 -81.090 53.035
9216 92.323 -81.083 53.027
9217 92.342 -81.075 53.020
9218 92.360 -81.068 53.014
9219 92.379 -81.060 53.007
9232 92.619 -80.961 53.414
9233 92.638 -80.953 53.399
9234 92.656 -80.945 53.384
9235 92.674 -80.938 53.370
9236 92.693 -80.930 53.355
9237 92.712 -80.923 53.339
9238 92.730 -80.915 53.325
9239 92.748 -80.908 53.310
9240 92.767 -80.900 53.296
9241 92.785 -80.892 53.281
9242 92.804 -80.884 53.267
9243 92.823 -80.877 53.254
9244 92.841 -80.869 53.241
9245 92.859 -80.861 53.230
9246 92.878 -80.854 53.218
9247 92.896 -80.846 53.207
9248 92.915 -80.839 53.197
9249 92.933 -80.831 53.187
9250 92.951 -80.823 53.179
9251 92.970 -80.815 53.171
9252 92.986 -80.804 53.163
9253 93.002 -80.794 53.155
9254 93.018 -80.783 53.148
9255 93.035 -80.773 53.141
9256 93.051 -80.762 53.135
9257 93.068 -80.751 53.129
9272 93.313 -80.593 53.044
9273 93.329 -80.582 53.042
9274 93.346 -80.572 53.041
9275 93.362 -80.561 53.040
9276 93.378 -80.551 53.040
9277 93.395 -80.540 53.039
9278 93.411 -80.529 53.039
9279 93.427 -80.519 53.040
9280 93.444 -80.508 53.041
9281 93.460 -80.497 53.043
9282 93.477 -80.487 53.044
9283 93.493 -80.476 53.045
9284 93.509 -80.466 53.045
9285 93.516 -80.460 53.042
9286 93.520 -80.455 53.036
9287 93.523 -80.450 53.029
9288 93.526 -80.446 53.022
9289 93.530 -80.441 53.016
9290 93.533 -80.437 53.009
9291 93.537 -80.432 53.003
9292 93.540 -80.428 52.996
9293 93.543 -80.423 52.989
9294 93.547 -80.418 52.980
9295 93.550 -80.413 52.971
9296 93.553 -80.409 52.964
9297 93.557 -80.404 52.954
9298 93.560 -80.400 52.946
9299 93.563 -80.395 52.937
9300 93.567 -80.391 52.928
9301 93.570 -80.386 52.919
9302 93.574 -80.381 52.910
9888 91.788 -81.225 53.418
9889 91.802 -81.219 53.395
9890 91.815 -81.214 53.373
9891 91.829 -81.209 53.352
9892 91.844 -81.202 53.331
9893 91.858 -81.197 53.313
9894 91.872 -81.191 53.294
9895 91.885 -81.186 53.277
9896 91.899 -81.180 53.260
9897 91.914 -81.174 53.245
9898 91.928 -81.169 53.230
9899 91.942 -81.163 53.216
9900 91.955 -81.158 53.202
9901 91.969 -81.152 53.190
9902 91.984 -81.146 53.178
9903 92.018 -81.149 53.166
9904 92.067 -81.160 53.154
9905 92.091 -81.157 53.143
9906 92.111 -81.147 53.132
9907 92.131 -81.137 53.121
9908 92.147 -81.126 53.109
9909 92.163 -81.115 53.099
9910 92.182 -81.108 53.088
9911 92.202 -81.102 53.078
9912 92.221 -81.095 53.068
9913 92.240 -81.088 53.059
9914 92.260 -81.082 53.050
9915 92.279 -81.075 53.042
9916 92.299 -81.068 53.034
9917 92.318 -81.061 53.026
9918 92.338 -81.055 53.019
9919 92.356 -81.047 53.011
9920 92.374 -81.039 53.004
9933 92.615 -80.940 53.409
9934 92.633 -80.933 53.394
9935 92.651 -80.925 53.379
9936 92.670 -80.917 53.363
9937 92.689 -80.910 53.348
9938 92.707 -80.902 53.333
9939 92.725 -80.895 53.317
9940 92.744 -80.887 53.302
9941 92.762 -80.880 53.286
9942 92.781 -80.872 53.271
9943 92.799 -80.864 53.257
9944 92.818 -80.857 53.244
9945 92.836 -80.848 53.231
9946 92.855 -80.841 53.218
9947 92.873 -80.833 53.206
9948 92.892 -80.826 53.195
9949 92.910 -80.818 53.184
9950 92.929 -80.811 53.174
9951 92.947 -80.803 53.165
9952 92.965 -80.794 53.158
9953 92.981 -80.783 53.150
9954 92.997 -80.773 53.141
9955 93.013 -80.762 53.135
9956 93.030 -80.751 53.128
9957 93.046 -80.741 53.121
9958 93.063 -80.730 53.115
9959 93.079 -80.720 53.109
9960 93.095 -80.710 53.103
9961 93.112 -80.699 53.097
9962 93.128 -80.688 53.090
9972 93.292 -80.582 53.036
9973 93.308 -80.572 53.033
9974 93.324 -80.561 53.032
9975 93.341 -80.551 53.030
9976 93.357 -80.540 53.029
9977 93.373 -80.529 53.029
9978 93.390 -80.519 53.029
9979 93.406 -80.508 53.030
9980 93.422 -80.498 53.032
9981 93.439 -80.488 53.034
9982 93.455 -80.477 53.037
9983 93.468 -80.466 53.040
9984 93.481 -80.457 53.042
9985 93.497 -80.450 53.042
9986 93.509 -80.445 53.039
9987 93.513 -80.441 53.032
9988 93.516 -80.436 53.025
9989 93.520 -80.431 53.018
9990 93.523 -80.427 53.012
9991 93.526 -80.422 53.006
9992 93.530 -80.418 53.000
9993 93.534 -80.413 52.994
9994 93.537 -80.409 52.987
9995 93.540 -80.403 52.979
9996 93.543 -80.399 52.971
9997 93.547 -80.394 52.963
9998 93.550 -80.390 52.954
9999 93.553 -80.385 52.945
10000 93.557 -80.381 52.936
10001 93.560 -80.376 52.927
10002 93.564 -80.371 52.919
10003 93.574 -80.363 52.910
10589 91.786 -81.195 53.441
10590 91.800 -81.190 53.418
10591 91.814 -81.184 53.395
10592 91.828 -81.178 53.373
10593 91.842 -81.173 53.352
10594 91.856 -81.167 53.332
10595 91.870 -81.162 53.313
10596 91.884 -81.155 53.294
10597 91.898 -81.150 53.277
10598 91.912 -81.144 53.261
10599 91.926 -81.139 53.245
10600 91.940 -81.133 53.229
10601 91.954 -81.127 53.215
10602 91.968 -81.122 53.201
10603 91.982 -81.116 53.187
10604 92.003 -81.113 53.174
10605 92.038 -81.129 53.162
10606 92.066 -81.133 53.150
10607 92.086 -81.123 53.138
10608 92.107 -81.112 53.125
10609 92.127 -81.102 53.114
10610 92.150 -81.093 53.103
10611 92.174 -81.085 53.092
10612 92.194 -81.078 53.081
10613 92.213 -81.071 53.070
10614 92.233 -81.065 53.060
10615 92.252 -81.058 53.051
10616 92.272 -81.052 53.042
10617 92.291 -81.045 53.034
10618 92.311 -81.039 53.026
10619 92.331 -81.032 53.018
10620 92.350 -81.025 53.011
10621 92.369 -81.019 53.003
10634 92.610 -80.920 53.405
10635 92.629 -80.912 53.390
10636 92.647 -80.905 53.373
10637 92.665 -80.897 53.357
10638 92.684 -80.889 53.342
10639 92.702 -80.882 53.327
10640 92.721 -80.874 53.310
10641 92.739 -80.867 53.294
10642 92.758 -80.859 53.278
10643 92.776 -80.851 53.263
10644 92.795 -80.844 53.248
10645 92.813 -80.836 53.234
10646 92.832 -80.829 53.220
10647 92.850 -80.821 53.207
10648 92.868 -80.813 53.195
10649 92.887 -80.805 53.183
10650 92.906 -80.798 53.172
10651 92.924 -80.790 53.162
10652 92.942 -80.782 53.152
10653 92.959 -80.773 53.145
10654 92.976 -80.762 53.137
10655 92.992 -80.752 53.128
10656 93.008 -80.742 53.121
10657 93.025 -80.731 53.114
10658 93.041 -80.720 53.107
10659 93.058 -80.710 53.102
10660 93.074 -80.699 53.096
10661 93.090 -80.688 53.090
10662 93.107 -80.678 53.083
10663 93.123 -80.667 53.077
10664 93.139 -80.657 53.071
10665 93.156 -80.646 53.064
10673 93.286 -80.561 53.023
10674 93.303 -80.551 53.021
10675 93.319 -80.541 53.020
10676 93.336 -80.530 53.018
10677 93.352 -80.519 53.017
10678 93.368 -80.509 53.017
10679 93.384 -80.498 53.018
10680 93.401 -80.488 53.020
10681 93.414 -80.476 53.022
10682 93.426 -80.465 53.025
10683 93.439 -80.454 53.030
10684 93.452 -80.449 53.034
10685 93.468 -80.443 53.037
10686 93.485 -80.435 53.037
10687 93.498 -80.430 53.034
10688 93.506 -80.426 53.029
10689 93.510 -80.422 53.021
10690 93.513 -80.417 53.014
10691 93.516 -80.413 53.009
10692 93.520 -80.408 53.003
10693 93.523 -80.403 52.998
10694 93.526 -80.399 52.993
10695 93.530 -80.394 52.985
10696 93.534 -80.389 52.977
10697 93.537 -80.385 52.970
10698 93.540 -80.380 52.962
10699 93.543 -80.375 52.953
10700 93.547 -80.371 52.944
10701 93.550 -80.366 52.935
10702 93.554 -80.361 52.927
10703 93.564 -80.353 52.919
10704 93.574 -80.344 52.910
10705 93.584 -80.335 52.901
11291 91.798 -81.159 53.441
11292 91.812 -81.154 53.417
11293 91.826 -81.149 53.394
11294 91.840 -81.143 53.373
11295 91.854 -81.137 53.352
11296 91.868 -81.131 53.332
11297 91.882 -81.126 53.312
11298 91.896 -81.120 53.295
11299 91.910 -81.115 53.278
11300 91.924 -81.108 53.260
11301 91.938 -81.103 53.244
11302 91.952 -81.098 53.229
11303 91.965 -81.096 53.214
11304 91.979 -81.095 53.199
11305 91.992 -81.095 53.186
11306 92.010 -81.098 53.173
11307 92.040 -81.108 53.159
11308 92.061 -81.098 53.145
11309 92.082 -81.088 53.133
11310 92.112 -81.079 53.121
11311 92.143 -81.070 53.108
11312 92.164 -81.063 53.096
11313 92.184 -81.055 53.084
11314 92.205 -81.049 53.073
11315 92.224 -81.042 53.062
11316 92.244 -81.035 53.052
11317 92.265 -81.028 53.044
11318 92.284 -81.022 53.035
11319 92.303 -81.015 53.026
11320 92.323 -81.008 53.018
11321 92.342 -81.002 53.011
11322 92.362 -80.996 53.003
11336 92.624 -80.892 53.384
11337 92.642 -80.884 53.368
11338 92.661 -80.877 53.352
11339 92.679 -80.869 53.336
11340 92.698 -80.861 53.320
11341 92.716 -80.854 53.303
11342 92.735 -80.846 53.287
11343 92.753 -80.839 53.271
11344 92.772 -80.831 53.255
11345 92.790 -80.823 53.239
11346 92.808 -80.816 53.225
11347 92.827 -80.808 53.210
11348 92.846 -80.801 53.197
11349 92.864 -80.793 53.184
11350 92.882 -80.786 53.172
11351 92.901 -80.777 53.160
11352 92.919 -80.770 53.150
11353 92.938 -80.762 53.140
11354 92.954 -80.752 53.132
11355 92.971 -80.742 53.124
11356 92.987 -80.731 53.116
11357 93.003 -80.720 53.108
11358 93.020 -80.710 53.101
11359 93.036 -80.699 53.094
11360 93.053 -80.688 53.088
11361 93.069 -80.678 53.082
11362 93.085 -80.667 53.076
11363 93.101 -80.657 53.070
11364 93.118 -80.646 53.064
11365 93.134 -80.635 53.058
11366 93.151 -80.625 53.052
11367 93.167 -80.614 53.046
11374 93.281 -80.541 53.011
11375 93.298 -80.530 53.009
11376 93.314 -80.519 53.007
11377 93.331 -80.509 53.006
11378 93.347 -80.498 53.005
11379 93.360 -80.487 53.004
11380 93.372 -80.476 53.006
11381 93.385 -80.465 53.008
11382 93.398 -80.453 53.011
11383 93.410 -80.444 53.016
11384 93.424 -80.441 53.021
11385 93.440 -80.435 53.027
11386 93.456 -80.428 53.029
11387 93.473 -80.420 53.029
11388 93.486 -80.415 53.027
11389 93.496 -80.412 53.023
11390 93.503 -80.407 53.017
11391 93.507 -80.403 53.011
11392 93.510 -80.398 53.007
11393 93.513 -80.394 53.002
11394 93.516 -80.389 52.997
11395 93.520 -80.385 52.991
11396 93.523 -80.380 52.984
11397 93.526 -80.375 52.977
11398 93.530 -80.370 52.969
11399 93.534 -80.366 52.961
11400 93.537 -80.361 52.952
11401 93.540 -80.356 52.943
11402 93.544 -80.352 52.934
11403 93.554 -80.343 52.926
11404 93.564 -80.334 52.918
11405 93.574 -80.325 52.909
11406 93.584 -80.317 52.901
11407 93.594 -80.309 52.892
11992 91.796 -81.130 53.465
11993 91.810 -81.124 53.441
11994 91.824 -81.118 53.417
11995 91.838 -81.113 53.395
11996 91.852 -81.107 53.373
11997 91.866 -81.102 53.352
11998 91.880 -81.096 53.332
11999 91.894 -81.090 53.314
12000 91.908 -81.089 53.296
12001 91.921 -81.088 53.277
12002 91.934 -81.087 53.261
12003 91.947 -81.086 53.245
12004 91.961 -81.086 53.229
12005 91.974 -81.085 53.213
12006 91.987 -81.084 53.198
12007 92.001 -81.083 53.184
12008 92.015 -81.083 53.169
12009 92.040 -81.075 53.155
12010 92.065 -81.067 53.141
12011 92.090 -81.059 53.128
12012 92.115 -81.051 53.115
12013 92.140 -81.043 53.102
12014 92.165 -81.035 53.090
12015 92.190 -81.027 53.078
12016 92.215 -81.020 53.067
12017 92.235 -81.013 53.056
12018 92.255 -81.006 53.047
12019 92.275 -80.999 53.037
12020 92.295 -80.992 53.028
12021 92.315 -80.986 53.020
12022 92.335 -80.979 53.011
12023 92.354 -80.972 53.003
12038 92.638 -80.864 53.364
12039 92.656 -80.856 53.348
12040 92.674 -80.848 53.332
12041 92.693 -80.841 53.315
12042 92.712 -80.833 53.297
12043 92.730 -80.826 53.281
12044 92.748 -80.818 53.264
12045 92.767 -80.811 53.247
12046 92.785 -80.803 53.231
12047 92.804 -80.795 53.216
12048 92.823 -80.788 53.201
12049 92.841 -80.780 53.187
12050 92.859 -80.773 53.173
12051 92.878 -80.765 53.161
12052 92.897 -80.757 53.149
12053 92.915 -80.750 53.137
12054 92.933 -80.742 53.128
12055 92.949 -80.731 53.120
12056 92.966 -80.720 53.111
12057 92.982 -80.710 53.103
12058 92.998 -80.699 53.096
12059 93.015 -80.688 53.089
12060 93.031 -80.678 53.082
12061 93.048 -80.667 53.075
12062 93.064 -80.657 53.069
12063 93.080 -80.646 53.063
12064 93.096 -80.635 53.057
12065 93.113 -80.626 53.051
12066 93.129 -80.615 53.045
12067 93.146 -80.604 53.039
12068 93.162 -80.594 53.034
12069 93.179 -80.583 53.028
12070 93.195 -80.573 53.022
12075 93.276 -80.519 52.999
12076 93.293 -80.509 52.996
12077 93.306 -80.498 52.994
12078 93.319 -80.487 52.992
12079 93.331 -80.475 52.991
12080 93.343 -80.464 52.991
12081 93.356 -80.453 52.993
12082 93.368 -80.442 52.996
12083 93.382 -80.437 53.000
12084 93.396 -80.433 53.006
12085 93.412 -80.427 53.012
12086 93.428 -80.420 53.018
12087 93.444 -80.413 53.021
12088 93.460 -80.406 53.021
12089 93.475 -80.400 53.019
12090 93.485 -80.396 53.016
12091 93.494 -80.393 53.012
12092 93.500 -80.388 53.009
12093 93.503 -80.384 53.005
12094 93.507 -80.379 53.001
12095 93.510 -80.375 52.996
12096 93.513 -80.370 52.991
12097 93.517 -80.366 52.983
12098 93.520 -80.361 52.976
12099 93.523 -80.356 52.968
12100 93.526 -80.351 52.960
12101 93.530 -80.347 52.951
12102 93.534 -80.342 52.943
12103 93.544 -80.333 52.935
12104 93.554 -80.325 52.926
12105 93.564 -80.316 52.917
12106 93.574 -80.307 52.910
12107 93.584 -80.299 52.901
12108 93.594 -80.290 52.892
12693 91.794 -81.099 53.489
12694 91.808 -81.094 53.465
12695 91.822 -81.089 53.442
12696 91.836 -81.083 53.418
12697 91.850 -81.081 53.396
12698 91.863 -81.080 53.375
12699 91.876 -81.080 53.354
12700 91.889 -81.079 53.335
12701 91.903 -81.078 53.316
12702 91.916 -81.077 53.297
12703 91.930 -81.077 53.279
12704 91.943 -81.076 53.261
12705 91.956 -81.075 53.244
12706 91.970 -81.075 53.228
12707 91.985 -81.074 53.212
12708 91.999 -81.074 53.196
12709 92.013 -81.069 53.181
12710 92.027 -81.060 53.165
12711 92.042 -81.050 53.151
12712 92.060 -81.042 53.137
12713 92.082 -81.033 53.123
12714 92.107 -81.026 53.109
12715 92.132 -81.017 53.097
12716 92.157 -81.010 53.085
12717 92.181 -81.002 53.073
12718 92.207 -80.994 53.062
12719 92.232 -80.986 53.051
12720 92.258 -80.978 53.041
12721 92.284 -80.971 53.032
12722 92.306 -80.963 53.023
12723 92.325 -80.956 53.014
12724 92.346 -80.949 53.005
12740 92.652 -80.836 53.344
12741 92.670 -80.828 53.328
12742 92.689 -80.820 53.310
12743 92.707 -80.813 53.293
12744 92.725 -80.805 53.276
12745 92.744 -80.798 53.258
12746 92.763 -80.790 53.242
12747 92.781 -80.782 53.225
12748 92.799 -80.775 53.209
12749 92.818 -80.767 53.194
12750 92.836 -80.760 53.179
12751 92.855 -80.752 53.164
12752 92.873 -80.745 53.152
12753 92.892 -80.737 53.140
12754 92.910 -80.729 53.128
12755 92.928 -80.720 53.118
12756 92.944 -80.710 53.108
12757 92.960 -80.699 53.100
12758 92.977 -80.688 53.092
12759 92.993 -80.678 53.084
12760 93.010 -80.668 53.077
12761 93.026 -80.657 53.070
12762 93.043 -80.647 53.063
12763 93.059 -80.636 53.057
12764 93.075 -80.626 53.050
12765 93.091 -80.615 53.045
12766 93.108 -80.604 53.039
12767 93.124 -80.594 53.033
12768 93.141 -80.583 53.027
12769 93.157 -80.573 53.022
12770 93.174 -80.562 53.017
12771 93.190 -80.551 53.011
12772 93.206 -80.541 53.005
12773 93.222 -80.530 52.999
12774 93.238 -80.519 52.994
12775 93.252 -80.509 52.990
12776 93.265 -80.497 52.986
12777 93.277 -80.486 52.983
12778 93.289 -80.475 52.981
12779 93.302 -80.464 52.979
12780 93.314 -80.453 52.979
12781 93.327 -80.441 52.979
12782 93.340 -80.433 52.981
12783 93.353 -80.429 52.983
12784 93.367 -80.425 52.986
12785 93.383 -80.419 52.992
12786 93.400 -80.412 52.999
12787 93.416 -80.405 53.006
12788 93.432 -80.397 53.010
12789 93.448 -80.391 53.012
12790 93.463 -80.385 53.011
12791 93.473 -80.381 53.009
12792 93.483 -80.378 53.008
12793 93.493 -80.374 53.007
12794 93.497 -80.369 53.006
12795 93.500 -80.365 53.002
12796 93.503 -80.360 52.997
12797 93.507 -80.356 52.992
12798 93.510 -80.351 52.983
12799 93.513 -80.347 52.975
12800 93.517 -80.341 52.966
12801 93.520 -80.337 52.959
12802 93.523 -80.332 52.951
12803 93.534 -80.324 52.942
12804 93.544 -80.315 52.934
12805 93.554 -80.306 52.925
12806 93.564 -80.297 52.916
12807 93.574 -80.289 52.909
12808 93.584 -80.280 52.900
12809 93.594 -80.272 52.892
13394 91.792 -81.074 53.513
13395 91.805 -81.073 53.490
13396 91.818 -81.072 53.466
13397 91.832 -81.071 53.442
13398 91.845 -81.071 53.420
13399 91.858 -81.070 53.398
13400 91.872 -81.069 53.377
13401 91.885 -81.068 53.356
13402 91.898 -81.068 53.336
13403 91.913 -81.068 53.316
13404 91.927 -81.067 53.298
13405 91.942 -81.066 53.280
13406 91.956 -81.066 53.262
13407 91.970 -81.065 53.245
13408 91.985 -81.065 53.228
13409 91.999 -81.060 53.211
13410 92.012 -81.055 53.194
13411 92.026 -81.050 53.178
13412 92.040 -81.038 53.162
13413 92.054 -81.026 53.148
13414 92.070 -81.017 53.133
13415 92.086 -81.009 53.119
13416 92.106 -81.001 53.105
13417 92.127 -80.992 53.093
13418 92.149 -80.984 53.080
13419 92.173 -80.976 53.068
13420 92.199 -80.968 53.057
13421 92.223 -80.960 53.047
13422 92.248 -80.952 53.037
13423 92.274 -80.944 53.027
13424 92.299 -80.936 53.017
13442 92.665 -80.808 53.325
13443 92.684 -80.800 53.308
13444 92.702 -80.792 53.290
13445 92.721 -80.785 53.273
13446 92.739 -80.777 53.254
13447 92.758 -80.770 53.237
13448 92.776 -80.762 53.220
13449 92.795 -80.754 53.204
13450 92.813 -80.747 53.188
13451 92.832 -80.739 53.173
13452 92.850 -80.732 53.157
13453 92.869 -80.724 53.144
13454 92.887 -80.717 53.132
13455 92.906 -80.709 53.120
13456 92.923 -80.700 53.109
13457 92.939 -80.689 53.100
13458 92.955 -80.679 53.091
13459 92.972 -80.668 53.083
13460 92.988 -80.657 53.075
13461 93.005 -80.647 53.067
13462 93.021 -80.636 53.060
13463 93.038 -80.626 53.053
13464 93.054 -80.615 53.047
13465 93.070 -80.604 53.041
13466 93.086 -80.594 53.034
13467 93.103 -80.583 53.028
13468 93.119 -80.573 53.022
13469 93.136 -80.562 53.016
13470 93.152 -80.551 53.010
13471 93.168 -80.541 53.005
13472 93.185 -80.530 52.999
13473 93.198 -80.519 52.993
13474 93.211 -80.508 52.987
13475 93.223 -80.497 52.982
13476 93.235 -80.486 52.977
13477 93.248 -80.475 52.973
13478 93.260 -80.463 52.970
13479 93.273 -80.452 52.967
13480 93.285 -80.441 52.966
13481 93.298 -80.430 52.965
13482 93.311 -80.425 52.966
13483 93.325 -80.422 52.967
13484 93.339 -80.418 52.969
13485 93.355 -80.411 52.973
13486 93.371 -80.404 52.979
13487 93.388 -80.397 52.986
13488 93.404 -80.390 52.993
13489 93.420 -80.383 52.998
13490 93.436 -80.375 53.001
13491 93.451 -80.369 53.001
13492 93.462 -80.366 53.002
13493 93.472 -80.362 53.002
13494 93.481 -80.359 53.002
13495 93.490 -80.355 53.002
13496 93.493 -80.350 52.999
13497 93.497 -80.346 52.996
13498 93.500 -80.341 52.992
13499 93.503 -80.337 52.983
13500 93.507 -80.332 52.974
13501 93.510 -80.327 52.965
13502 93.513 -80.322 52.956
13503 93.523 -80.314 52.948
13504 93.534 -80.305 52.940
13505 93.544 -80.297 52.931
13506 93.554 -80.288 52.923
13507 93.564 -80.279 52.915
13508 93.574 -80.271 52.906
13509 93.584 -80.262 52.899
13510 93.594 -80.253 52.891
13511 93.604 -80.244 52.884
14096 91.801 -81.062 53.514
14097 91.814 -81.061 53.490
14098 91.827 -81.061 53.467
14099 91.841 -81.060 53.445
14100 91.856 -81.060 53.422
14101 91.870 -81.059 53.399
14102 91.884 -81.059 53.378
14103 91.898 -81.058 53.357
14104 91.913 -81.058 53.336
14105 91.927 -81.058 53.318
14106 91.942 -81.057 53.299
14107 91.956 -81.056 53.280
14108 91.970 -81.055 53.262
14109 91.984 -81.051 53.245
14110 91.998 -81.046 53.226
14111 92.011 -81.041 53.209
14112 92.025 -81.036 53.192
14113 92.039 -81.027 53.176
14114 92.053 -81.015 53.160
14115 92.067 -81.003 53.146
14116 92.082 -80.992 53.131
14117 92.098 -80.984 53.116
14118 92.114 -80.976 53.103
14119 92.131 -80.967 53.089
14120 92.152 -80.959 53.077
14121 92.173 -80.951 53.065
14122 92.194 -80.942 53.054
14123 92.215 -80.934 53.043
14124 92.240 -80.927 53.032
14125 92.265 -80.918 53.022
14144 92.680 -80.780 53.307
14145 92.698 -80.773 53.289
14146 92.716 -80.764 53.272
14147 92.735 -80.757 53.253
14148 92.753 -80.749 53.235
14149 92.772 -80.742 53.218
14150 92.790 -80.734 53.200
14151 92.808 -80.726 53.183
14152 92.827 -80.719 53.168
14153 92.846 -80.711 53.152
14154 92.864 -80.704 53.138
14155 92.882 -80.696 53.126
14156 92.901 -80.688 53.114
14157 92.918 -80.679 53.103
14158 92.934 -80.668 53.093
14159 92.950 -80.657 53.084
14160 92.967 -80.647 53.075
14161 92.983 -80.636 53.067
14162 93.000 -80.626 53.059
14163 93.016 -80.615 53.051
14164 93.032 -80.604 53.045
14165 93.049 -80.594 53.038
14166 93.065 -80.583 53.031
14167 93.081 -80.573 53.024
14168 93.098 -80.562 53.018
14169 93.114 -80.551 53.012
14170 93.131 -80.541 53.005
14171 93.144 -80.530 52.999
14172 93.156 -80.519 52.993
14173 93.169 -80.508 52.987
14174 93.182 -80.497 52.981
14175 93.194 -80.485 52.975
14176 93.206 -80.474 52.969
14177 93.219 -80.463 52.964
14178 93.231 -80.452 52.960
14179 93.243 -80.441 52.956
14180 93.256 -80.429 52.953
14181 93.269 -80.421 52.951
14182 93.283 -80.417 52.951
14183 93.297 -80.413 52.952
14184 93.310 -80.410 52.952
14185 93.327 -80.403 52.954
14186 93.343 -80.397 52.959
14187 93.359 -80.389 52.965
14188 93.375 -80.382 52.971
14189 93.392 -80.375 52.978
14190 93.408 -80.368 52.984
14191 93.424 -80.360 52.988
14192 93.440 -80.353 52.991
14193 93.450 -80.350 52.993
14194 93.460 -80.347 52.994
14195 93.470 -80.344 52.995
14196 93.480 -80.340 52.995
14197 93.487 -80.336 52.995
14198 93.490 -80.331 52.992
14199 93.493 -80.327 52.989
14200 93.497 -80.322 52.980
14201 93.500 -80.318 52.971
14202 93.503 -80.312 52.961
14203 93.513 -80.304 52.952
14204 93.523 -80.296 52.944
14205 93.534 -80.287 52.935
14206 93.544 -80.278 52.927
14207 93.554 -80.269 52.919
14208 93.564 -80.261 52.911
14209 93.574 -80.252 52.903
14210 93.584 -80.243 52.897
14211 93.594 -80.235 52.890
14212 93.604 -80.226 52.882
14797 91.798 -81.052 53.539
14798 91.812 -81.052 53.515
14799 91.826 -81.052 53.492
14800 91.841 -81.051 53.469
14801 91.856 -81.050 53.446
14802 91.870 -81.050 53.424
14803 91.884 -81.049 53.401
14804 91.898 -81.049 53.380
14805 91.913 -81.049 53.359
14806 91.927 -81.048 53.339
14807 91.942 -81.048 53.319
14808 91.956 -81.046 53.299
14809 91.969 -81.042 53.280
14810 91.984 -81.036 53.262
14811 91.997 -81.032 53.243
14812 92.011 -81.027 53.225
14813 92.024 -81.023 53.208
14814 92.038 -81.017 53.190
14815 92.052 -81.005 53.175
14816 92.066 -80.992 53.159
14817 92.080 -80.980 53.144
14818 92.094 -80.968 53.129
14819 92.109 -80.959 53.114
14820 92.126 -80.951 53.100
14821 92.142 -80.943 53.087
14822 92.158 -80.935 53.074
14823 92.177 -80.927 53.063
14824 92.198 -80.918 53.051
14825 92.219 -80.910 53.040
14826 92.239 -80.902 53.029
14846 92.693 -80.752 53.289
14847 92.712 -80.745 53.271
14848 92.730 -80.737 53.253
14849 92.748 -80.729 53.234
14850 92.767 -80.721 53.216
14851 92.786 -80.714 53.199
14852 92.804 -80.706 53.182
14853 92.823 -80.698 53.165
14854 92.841 -80.691 53.150
14855 92.859 -80.683 53.136
14856 92.878 -80.676 53.122
14857 92.897 -80.668 53.109
14858 92.913 -80.657 53.099
14859 92.929 -80.647 53.088
14860 92.945 -80.636 53.079
14861 92.962 -80.626 53.069
14862 92.978 -80.615 53.060
14863 92.995 -80.604 53.052
14864 93.011 -80.594 53.044
14865 93.027 -80.584 53.037
14866 93.044 -80.573 53.029
14867 93.060 -80.563 53.022
14868 93.076 -80.552 53.015
14869 93.090 -80.541 53.009
14870 93.102 -80.529 53.002
14871 93.115 -80.519 52.995
14872 93.128 -80.507 52.988
14873 93.140 -80.496 52.982
14874 93.152 -80.485 52.976
14875 93.165 -80.474 52.969
14876 93.178 -80.463 52.962
14877 93.190 -80.451 52.956
14878 93.202 -80.441 52.951
14879 93.215 -80.429 52.946
14880 93.227 -80.418 52.943
14881 93.240 -80.413 52.940
14882 93.255 -80.410 52.938
14883 93.268 -80.406 52.936
14884 93.282 -80.402 52.937
14885 93.298 -80.396 52.938
14886 93.314 -80.388 52.941
14887 93.331 -80.381 52.945
14888 93.347 -80.375 52.951
14889 93.363 -80.367 52.956
14890 93.379 -80.360 52.961
14891 93.396 -80.353 52.968
14892 93.412 -80.346 52.973
14893 93.428 -80.338 52.978
14894 93.438 -80.335 52.981
14895 93.448 -80.331 52.983
14896 93.458 -80.328 52.985
14897 93.468 -80.325 52.986
14898 93.478 -80.322 52.987
14899 93.483 -80.317 52.986
14900 93.487 -80.312 52.983
14901 93.490 -80.308 52.975
14902 93.493 -80.303 52.966
14903 93.503 -80.294 52.955
14904 93.513 -80.286 52.946
14905 93.523 -80.277 52.938
14906 93.534 -80.269 52.930
14907 93.544 -80.260 52.922
14908 93.554 -80.251 52.915
14909 93.564 -80.243 52.908
14910 93.574 -80.234 52.900
14911 93.584 -80.225 52.894
14912 93.594 -80.216 52.887
14913 93.604 -80.208 52.879
15498 91.798 -81.043 53.563
15499 91.812 -81.043 53.540
15500 91.826 -81.042 53.516
15501 91.841 -81.042 53.493
15502 91.856 -81.041 53.470
15503 91.870 -81.040 53.447
15504 91.884 -81.040 53.425
15505 91.898 -81.039 53.402
15506 91.913 -81.039 53.381
15507 91.927 -81.039 53.360
15508 91.942 -81.037 53.339
15509 91.955 -81.033 53.319
15510 91.969 -81.027 53.300
15511 91.983 -81.023 53.281
15512 91.996 -81.018 53.262
15513 92.010 -81.013 53.243
15514 92.023 -81.008 53.225
15515 92.037 -81.004 53.207
15516 92.051 -80.995 53.190
15517 92.065 -80.983 53.174
15518 92.079 -80.971 53.158
15519 92.093 -80.958 53.142
15520 92.107 -80.945 53.128
15521 92.122 -80.934 53.113
15522 92.138 -80.926 53.098
15523 92.154 -80.918 53.085
15524 92.170 -80.910 53.073
15525 92.186 -80.902 53.061
15526 92.202 -80.894 53.050
15548 92.707 -80.723 53.272
15549 92.725 -80.717 53.253
15550 92.744 -80.709 53.235
15551 92.763 -80.701 53.216
15552 92.781 -80.694 53.199
15553 92.799 -80.685 53.182
15554 92.818 -80.678 53.165
15555 92.837 -80.670 53.149
15556 92.855 -80.663 53.135
15557 92.873 -80.655 53.121
15558 92.892 -80.647 53.108
15559 92.908 -80.636 53.098
15560 92.924 -80.626 53.088
15561 92.940 -80.616 53.078
15562 92.957 -80.605 53.067
15563 92.973 -80.594 53.057
15564 92.990 -80.584 53.049
15565 93.006 -80.573 53.040
15566 93.022 -80.563 53.032
15567 93.036 -80.551 53.023
15568 93.049 -80.541 53.016
15569 93.061 -80.529 53.008
15570 93.074 -80.518 53.001
15571 93.086 -80.507 52.993
15572 93.098 -80.496 52.986
15573 93.111 -80.485 52.979
15574 93.123 -80.473 52.972
15575 93.136 -80.462 52.966
15576 93.148 -80.451 52.958
15577 93.160 -80.440 52.951
15578 93.173 -80.428 52.944
15579 93.186 -80.418 52.938
15580 93.198 -80.409 52.933
15581 93.212 -80.405 52.930
15582 93.226 -80.401 52.927
15583 93.240 -80.398 52.924
15584 93.254 -80.394 52.922
15585 93.270 -80.388 52.923
15586 93.286 -80.381 52.924
15587 93.302 -80.374 52.927
15588 93.319 -80.366 52.931
15589 93.335 -80.359 52.936
15590 93.351 -80.352 52.941
15591 93.367 -80.345 52.946
15592 93.383 -80.337 52.952
15593 93.400 -80.331 52.958
15594 93.416 -80.324 52.962
15595 93.427 -80.319 52.966
15596 93.437 -80.316 52.969
15597 93.446 -80.312 52.970
15598 93.457 -80.309 52.972
15599 93.467 -80.306 52.972
15600 93.477 -80.303 52.971
15601 93.480 -80.298 52.969
15602 93.483 -80.294 52.963
15603 93.493 -80.284 52.956
15604 93.503 -80.276 52.947
15605 93.513 -80.268 52.938
15606 93.523 -80.259 52.930
15607 93.534 -80.250 52.923
15608 93.544 -80.241 52.916
15609 93.554 -80.233 52.910
15610 93.564 -80.224 52.904
15611 93.574 -80.215 52.896
15612 93.584 -80.207 52.890
15613 93.594 -80.198 52.884
15614 93.604 -80.190 52.876
16200 91.812 -81.033 53.563
16201 91.826 -81.033 53.540
16202 91.841 -81.032 53.516
16203 91.856 -81.032 53.493
16204 91.870 -81.031 53.470
16205 91.884 -81.030 53.448
16206 91.898 -81.030 53.425
16207 91.913 -81.030 53.404
16208 91.927 -81.028 53.382
16209 91.941 -81.024 53.361
16210 91.954 -81.018 53.341
16211 91.968 -81.014 53.321
16212 91.982 -81.009 53.301
16213 91.996 -81.004 53.282
16214 92.009 -80.999 53.263
16215 92.023 -80.995 53.244
16216 92.036 -80.990 53.226
16217 92.050 -80.984 53.208
16218 92.064 -80.972 53.191
16219 92.078 -80.960 53.175
16220 92.092 -80.948 53.159
16221 92.106 -80.936 53.143
16222 92.121 -80.924 53.128
16223 92.135 -80.911 53.112
16224 92.150 -80.901 53.099
16225 92.165 -80.893 53.086
16226 92.181 -80.885 53.073
16227 92.198 -80.877 53.061
16250 92.719 -80.694 53.256
16251 92.738 -80.687 53.237
16252 92.758 -80.680 53.218
16253 92.776 -80.673 53.200
16254 92.795 -80.666 53.184
16255 92.813 -80.658 53.167
16256 92.832 -80.650 53.150
16257 92.850 -80.642 53.136
16258 92.869 -80.635 53.122
16259 92.886 -80.626 53.110
16260 92.903 -80.616 53.099
16261 92.919 -80.605 53.090
16262 92.935 -80.594 53.080
16263 92.952 -80.584 53.068
16264 92.968 -80.573 53.058
16265 92.982 -80.563 53.048
16266 92.995 -80.551 53.040
16267 93.007 -80.540 53.030
16268 93.019 -80.529 53.021
16269 93.032 -80.518 53.012
16270 93.045 -80.507 53.003
16271 93.057 -80.495 52.995
16272 93.069 -80.484 52.986
16273 93.082 -80.473 52.979
16274 93.094 -80.462 52.971
16275 93.107 -80.450 52.963
16276 93.119 -80.439 52.955
16277 93.132 -80.428 52.948
16278 93.144 -80.417 52.941
16279 93.156 -80.406 52.934
16280 93.170 -80.401 52.927
16281 93.184 -80.397 52.922
16282 93.198 -80.394 52.917
16283 93.212 -80.390 52.913
16284 93.225 -80.387 52.911
16285 93.241 -80.380 52.909
16286 93.258 -80.373 52.909
16287 93.274 -80.366 52.911
16288 93.290 -80.359 52.913
16289 93.306 -80.352 52.916
16290 93.323 -80.344 52.921
16291 93.339 -80.337 52.926
16292 93.355 -80.330 52.932
16293 93.371 -80.323 52.937
16294 93.388 -80.316 52.942
16295 93.404 -80.309 52.946
16296 93.415 -80.304 52.949
16297 93.425 -80.301 52.952
16298 93.435 -80.297 52.954
16299 93.445 -80.294 52.954
16300 93.455 -80.290 52.955
16301 93.465 -80.287 52.954
16302 93.474 -80.284 52.953
16303 93.483 -80.275 52.949
16304 93.493 -80.266 52.942
16305 93.503 -80.258 52.935
16306 93.513 -80.249 52.928
16307 93.523 -80.240 52.922
16308 93.534 -80.232 52.916
16309 93.544 -80.223 52.910
16310 93.554 -80.215 52.904
16311 93.564 -80.206 52.899
16312 93.574 -80.197 52.893
16313 93.584 -80.188 52.887
16314 93.594 -80.180 52.880
16315 93.604 -80.171 52.873
16316 93.614 -80.162 52.867
16902 91.826 -81.024 53.563
16903 91.841 -81.023 53.540
16904 91.856 -81.022 53.517
16905 91.870 -81.022 53.494
16906 91.884 -81.021 53.472
16907 91.898 -81.021 53.449
16908 91.913 -81.019 53.427
16909 91.926 -81.014 53.406
16910 91.940 -81.009 53.384
16911 91.954 -81.005 53.363
16912 91.967 -81.000 53.343
16913 91.981 -80.995 53.322
16914 91.995 -80.990 53.303
16915 92.008 -80.986 53.284
16916 92.022 -80.980 53.265
16917 92.035 -80.976 53.246
16918 92.050 -80.971 53.228
16919 92.063 -80.962 53.210
16920 92.077 -80.949 53.193
16921 92.091 -80.937 53.176
16922 92.105 -80.925 53.160
16923 92.120 -80.913 53.144
16924 92.134 -80.901 53.129
16925 92.148 -80.889 53.114
16926 92.162 -80.877 53.100
16927 92.177 -80.868 53.086
16951 92.689 -80.674 53.260
16952 92.715 -80.667 53.241
16953 92.741 -80.659 53.223
16954 92.768 -80.651 53.205
16955 92.789 -80.644 53.188
16956 92.809 -80.638 53.171
16957 92.827 -80.630 53.154
16958 92.846 -80.623 53.139
16959 92.864 -80.614 53.125
16960 92.881 -80.605 53.113
16961 92.898 -80.594 53.104
16962 92.914 -80.584 53.095
16963 92.928 -80.573 53.085
16964 92.941 -80.562 53.074
16965 92.953 -80.551 53.063
16966 92.966 -80.540 53.053
16967 92.978 -80.529 53.042
16968 92.991 -80.517 53.033
16969 93.003 -80.506 53.023
16970 93.015 -80.495 53.011
16971 93.028 -80.484 53.000
16972 93.040 -80.472 52.991
16973 93.053 -80.461 52.981
16974 93.065 -80.450 52.973
16975 93.077 -80.439 52.964
16976 93.090 -80.428 52.956
16977 93.102 -80.416 52.947
16978 93.115 -80.406 52.939
16979 93.128 -80.397 52.932
16980 93.142 -80.393 52.924
16981 93.155 -80.390 52.917
16982 93.169 -80.386 52.911
16983 93.183 -80.382 52.905
16984 93.197 -80.378 52.901
16985 93.213 -80.372 52.898
16986 93.229 -80.365 52.897
16987 93.246 -80.358 52.897
16988 93.262 -80.351 52.898
16989 93.278 -80.344 52.900
16990 93.294 -80.337 52.905
16991 93.310 -80.329 52.909
16992 93.327 -80.322 52.914
16993 93.343 -80.316 52.918
16994 93.359 -80.308 52.922
16995 93.375 -80.301 52.925
16996 93.392 -80.294 52.929
16997 93.404 -80.289 52.931
16998 93.414 -80.286 52.932
16999 93.424 -80.282 52.934
17000 93.433 -80.279 52.935
17001 93.443 -80.275 52.935
17002 93.453 -80.272 52.936
17003 93.463 -80.256 52.936
17004 93.483 -80.256 52.932
17005 93.493 -80.248 52.927
17006 93.503 -80.240 52.922
17007 93.513 -80.231 52.917
17008 93.523 -80.222 52.913
17009 93.534 -80.213 52.909
17010 93.544 -80.205 52.904
17011 93.554 -80.196 52.898
17012 93.564 -80.187 52.894
17013 93.574 -80.179 52.889
17014 93.584 -80.170 52.883
17015 93.594 -80.162 52.877
17016 93.604 -80.153 52.871
17017 93.614 -80.144 52.865
17058 93.990 -79.964 52.391
17059 93.999 -79.962 52.375
17060 94.007 -79.961 52.361
17061 94.016 -79.959 52.346
17062 94.025 -79.958 52.330
17063 94.033 -79.956 52.315
17064 94.042 -79.955 52.299
17065 94.050 -79.953 52.284
17066 94.059 -79.952 52.268
17603 91.826 -81.014 53.587
17604 91.841 -81.013 53.564
17605 91.856 -81.013 53.541
17606 91.870 -81.012 53.518
17607 91.884 -81.012 53.496
17608 91.898 -81.010 53.473
17609 91.912 -81.005 53.451
17610 91.926 -81.000 53.429
17611 91.939 -80.996 53.407
17612 91.953 -80.991 53.387
17613 91.966 -80.986 53.366
17614 91.980 -80.981 53.346
17615 91.994 -80.977 53.326
17616 92.008 -80.971 53.307
17617 92.021 -80.967 53.287
17618 92.035 -80.962 53.268
17619 92.049 -80.958 53.249
17620 92.062 -80.952 53.230
17621 92.076 -80.939 53.213
17622 92.090 -80.927 53.196
17623 92.104 -80.915 53.179
17624 92.119 -80.903 53.162
17625 92.133 -80.890 53.147
17626 92.147 -80.878 53.131
17627 92.161 -80.866 53.116
17652 92.651 -80.655 53.266
17653 92.677 -80.648 53.247
17654 92.704 -80.640 53.230
17655 92.730 -80.632 53.211
17656 92.757 -80.625 53.193
17657 92.783 -80.617 53.176
17658 92.809 -80.610 53.160
17659 92.836 -80.601 53.145
17660 92.860 -80.594 53.131
17661 92.874 -80.584 53.120
17662 92.887 -80.573 53.110
17663 92.899 -80.561 53.101
17664 92.912 -80.551 53.091
17665 92.924 -80.539 53.080
17666 92.936 -80.528 53.070
17667 92.949 -80.517 53.059
17668 92.961 -80.506 53.048
17669 92.974 -80.494 53.038
17670 92.986 -80.483 53.027
17671 92.999 -80.472 53.014
17672 93.011 -80.461 53.001
17673 93.023 -80.450 52.989
17674 93.036 -80.438 52.978
17675 93.049 -80.428 52.968
17676 93.061 -80.416 52.958
17677 93.073 -80.405 52.950
17678 93.086 -80.394 52.941
17679 93.099 -80.389 52.932
17680 93.113 -80.385 52.924
17681 93.127 -80.381 52.915
17682 93.141 -80.378 52.908
17683 93.155 -80.375 52.901
17684 93.168 -80.371 52.895
17685 93.185 -80.364 52.891
17686 93.201 -80.357 52.888
17687 93.217 -80.350 52.887
17688 93.233 -80.343 52.886
17689 93.250 -80.336 52.888
17690 93.266 -80.329 52.890
17691 93.282 -80.322 52.895
17692 93.298 -80.315 52.899
17693 93.314 -80.307 52.902
17694 93.331 -80.300 52.905
17695 93.347 -80.293 52.907
17696 93.363 -80.286 52.909
17697 93.379 -80.278 52.911
17698 93.392 -80.274 52.912
17699 93.402 -80.270 52.914
17700 93.412 -80.267 52.916
17701 93.422 -80.263 52.917
17702 93.432 -80.260 52.917
17703 93.442 -80.243 52.917
17704 93.451 -80.225 52.917
17705 93.465 -80.226 52.916
17706 93.488 -80.227 52.913
17707 93.503 -80.221 52.909
17708 93.513 -80.212 52.906
17709 93.523 -80.204 52.903
17710 93.534 -80.195 52.901
17711 93.544 -80.187 52.897
17712 93.554 -80.178 52.892
17713 93.564 -80.169 52.889
17714 93.574 -80.160 52.884
17715 93.584 -80.152 52.879
17716 93.594 -80.143 52.874
17717 93.604 -80.134 52.869
17718 93.614 -80.126 52.863
17719 93.624 -80.117 52.857
17758 93.981 -79.951 52.416
17759 93.989 -79.949 52.401
17760 93.998 -79.948 52.386
17761 94.006 -79.946 52.372
17762 94.015 -79.945 52.357
17763 94.024 -79.943 52.341
17764 94.032 -79.942 52.326
17765 94.041 -79.940 52.310
17766 94.049 -79.939 52.295
17767 94.058 -79.938 52.279
17768 94.066 -79.936 52.263
18305 91.841 -81.004 53.588
18306 91.856 -81.003 53.565
18307 91.870 -81.003 53.542
18308 91.884 -81.001 53.520
18309 91.897 -80.996 53.497
18310 91.911 -80.991 53.476
18311 91.925 -80.986 53.454
18312 91.939 -80.982 53.432
18313 91.952 -80.977 53.411
18314 91.966 -80.972 53.390
18315 91.980 -80.967 53.370
18316 91.993 -80.962 53.350
18317 92.007 -80.958 53.330
18318 92.020 -80.953 53.310
18319 92.034 -80.948 53.290
18320 92.048 -80.943 53.271
18321 92.062 -80.939 53.252
18322 92.075 -80.929 53.234
18323 92.089 -80.917 53.217
18324 92.103 -80.905 53.199
18325 92.118 -80.892 53.182
18326 92.132 -80.880 53.166
18327 92.146 -80.868 53.149
18353 92.616 -80.638 53.274
18354 92.641 -80.629 53.256
18355 92.666 -80.622 53.237
18356 92.693 -80.613 53.219
18357 92.719 -80.606 53.201
18358 92.745 -80.598 53.183
18359 92.772 -80.591 53.167
18360 92.798 -80.583 53.152
18361 92.824 -80.576 53.139
18362 92.850 -80.568 53.129
18363 92.868 -80.554 53.119
18364 92.882 -80.539 53.108
18365 92.895 -80.528 53.098
18366 92.908 -80.516 53.087
18367 92.920 -80.505 53.077
18368 92.932 -80.494 53.067
18369 92.945 -80.483 53.055
18370 92.957 -80.472 53.043
18371 92.970 -80.460 53.031
18372 92.982 -80.450 53.018
18373 92.995 -80.438 53.005
18374 93.007 -80.427 52.991
18375 93.019 -80.416 52.978
18376 93.032 -80.405 52.967
18377 93.044 -80.394 52.956
18378 93.057 -80.385 52.946
18379 93.071 -80.381 52.937
18380 93.085 -80.378 52.928
18381 93.098 -80.374 52.919
18382 93.113 -80.370 52.910
18383 93.126 -80.367 52.902
18384 93.140 -80.363 52.894
18385 93.156 -80.356 52.888
18386 93.172 -80.350 52.884
18387 93.189 -80.342 52.880
18388 93.205 -80.335 52.878
18389 93.221 -80.328 52.879
18390 93.237 -80.321 52.880
18391 93.254 -80.314 52.882
18392 93.270 -80.306 52.885
18393 93.286 -80.300 52.888
18394 93.302 -80.293 52.891
18395 93.319 -80.285 52.892
18396 93.335 -80.278 52.892
18397 93.351 -80.271 52.893
18398 93.367 -80.264 52.893
18399 93.380 -80.259 52.894
18400 93.391 -80.255 52.896
18401 93.401 -80.252 52.897
18402 93.410 -80.248 52.898
18403 93.420 -80.233 52.899
18404 93.430 -80.211 52.899
18405 93.439 -80.194 52.900
18406 93.445 -80.196 52.899
18407 93.470 -80.196 52.898
18408 93.493 -80.196 52.897
18409 93.513 -80.194 52.895
18410 93.523 -80.185 52.894
18411 93.534 -80.177 52.892
18412 93.544 -80.168 52.890
18413 93.554 -80.159 52.887
18414 93.564 -80.151 52.884
18415 93.574 -80.142 52.881
18416 93.584 -80.134 52.876
18417 93.594 -80.124 52.871
18418 93.604 -80.116 52.867
18419 93.614 -80.107 52.861
18420 93.624 -80.099 52.856
18421 93.634 -80.090 52.851
18458 93.971 -79.938 52.440
18459 93.980 -79.936 52.425
18460 93.988 -79.935 52.410
18461 93.997 -79.933 52.396
18462 94.005 -79.932 52.382
18463 94.014 -79.930 52.367
18464 94.023 -79.929 52.352
18465 94.031 -79.927 52.336
18466 94.040 -79.926 52.321
18467 94.048 -79.925 52.305
18468 94.057 -79.924 52.290
18469 94.065 -79.922 52.275
18470 94.074 -79.921 52.259
18497 94.231 -79.587 51.699
18498 94.244 -79.567 51.671
18499 94.257 -79.548 51.644
18500 94.269 -79.528 51.616
18501 94.282 -79.508 51.586
18502 94.295 -79.488 51.558
18503 94.308 -79.469 51.530
18504 94.320 -79.449 51.501
18505 94.337 -79.471 51.473
18506 94.353 -79.503 51.443
18507 94.370 -79.534 51.413
18508 94.388 -79.565 51.383
18509 94.405 -79.596 51.353
19007 91.856 -80.994 53.588
19008 91.869 -80.992 53.566
19009 91.883 -80.986 53.544
19010 91.897 -80.982 53.522
19011 91.911 -80.977 53.501
19012 91.924 -80.972 53.479
19013 91.938 -80.967 53.456
19014 91.951 -80.963 53.436
19015 91.965 -80.958 53.416
19016 91.979 -80.953 53.395
19017 91.993 -80.949 53.374
19018 92.006 -80.944 53.353
19019 92.020 -80.939 53.333
19020 92.033 -80.934 53.313
19021 92.047 -80.930 53.294
19022 92.061 -80.925 53.275
19023 92.074 -80.919 53.257
19024 92.088 -80.907 53.238
19025 92.102 -80.894 53.220
19026 92.117 -80.882 53.203
19027 92.131 -80.870 53.186
19054 92.598 -80.621 53.285
19055 92.619 -80.613 53.266
19056 92.640 -80.604 53.248
19057 92.661 -80.596 53.229
19058 92.683 -80.588 53.210
19059 92.708 -80.580 53.192
19060 92.734 -80.572 53.176
19061 92.760 -80.564 53.161
19062 92.786 -80.557 53.147
19063 92.812 -80.549 53.136
19064 92.839 -80.541 53.126
19065 92.859 -80.529 53.117
19066 92.873 -80.514 53.106
19067 92.888 -80.500 53.096
19068 92.902 -80.485 53.086
19069 92.916 -80.471 53.075
19070 92.928 -80.460 53.062
19071 92.940 -80.449 53.049
19072 92.953 -80.438 53.035
19073 92.966 -80.426 53.022
19074 92.978 -80.416 53.009
19075 92.990 -80.404 52.994
19076 93.003 -80.393 52.979
19077 93.015 -80.382 52.967
19078 93.029 -80.377 52.955
19079 93.043 -80.373 52.945
19080 93.057 -80.370 52.935
19081 93.070 -80.366 52.925
19082 93.084 -80.363 52.915
19083 93.098 -80.359 52.906
19084 93.112 -80.356 52.898
19085 93.128 -80.349 52.890
19086 93.144 -80.341 52.884
19087 93.160 -80.334 52.879
19088 93.177 -80.328 52.876
19089 93.193 -80.320 52.873
19090 93.209 -80.313 52.873
19091 93.225 -80.306 52.874
19092 93.241 -80.299 52.875
19093 93.258 -80.292 52.877
19094 93.274 -80.284 52.878
19095 93.290 -80.278 52.878
19096 93.306 -80.270 52.879
19097 93.323 -80.263 52.879
19098 93.339 -80.256 52.878
19099 93.355 -80.249 52.877
19100 93.369 -80.243 52.878
19101 93.379 -80.240 52.878
19102 93.389 -80.237 52.878
19103 93.399 -80.222 52.879
19104 93.409 -80.201 52.881
19105 93.419 -80.179 52.883
19106 93.427 -80.163 52.883
19107 93.433 -80.165 52.883
19108 93.450 -80.165 52.884
19109 93.475 -80.165 52.885
19110 93.498 -80.166 52.885
19111 93.522 -80.166 52.884
19112 93.534 -80.159 52.883
19113 93.544 -80.149 52.883
19114 93.554 -80.141 52.881
19115 93.564 -80.132 52.879
19116 93.574 -80.124 52.876
19117 93.584 -80.115 52.872
19118 93.594 -80.106 52.868
19119 93.604 -80.098 52.864
19120 93.614 -80.089 52.859
19121 93.624 -80.080 52.853
19122 93.634 -80.071 52.848
19123 93.644 -80.063 52.841
19158 93.962 -79.925 52.462
19159 93.970 -79.924 52.448
19160 93.979 -79.922 52.434
19161 93.987 -79.921 52.419
19162 93.996 -79.919 52.405
19163 94.004 -79.918 52.391
19164 94.013 -79.916 52.377
19165 94.022 -79.914 52.361
19166 94.030 -79.913 52.346
19167 94.039 -79.912 52.330
19168 94.047 -79.911 52.315
19169 94.056 -79.909 52.299
19170 94.064 -79.908 52.284
19171 94.073 -79.906 52.268
19196 94.251 -79.560 51.765
19197 94.241 -79.567 51.738
19198 94.229 -79.573 51.712
19199 94.241 -79.554 51.685
19200 94.254 -79.535 51.657
19201 94.266 -79.515 51.629
19202 94.279 -79.495 51.600
19203 94.292 -79.476 51.572
19204 94.305 -79.456 51.543
19205 94.317 -79.436 51.514
19206 94.332 -79.439 51.486
19207 94.349 -79.471 51.456
19208 94.366 -79.502 51.426
19209 94.383 -79.533 51.395
19210 94.400 -79.565 51.365
19211 94.417 -79.596 51.334
19212 94.434 -79.627 51.304
19709 91.869 -80.977 53.589
19710 91.882 -80.973 53.567
19711 91.896 -80.968 53.546
19712 91.910 -80.963 53.524
19713 91.923 -80.958 53.503
19714 91.937 -80.954 53.482
19715 91.951 -80.949 53.461
19716 91.964 -80.944 53.440
19717 91.978 -80.939 53.419
19718 91.992 -80.935 53.398
19719 92.005 -80.930 53.377
19720 92.019 -80.925 53.358
19721 92.032 -80.920 53.337
19722 92.047 -80.915 53.317
19723 92.060 -80.911 53.298
19724 92.074 -80.906 53.279
19725 92.087 -80.896 53.261
19726 92.101 -80.884 53.242
19727 92.116 -80.872 53.224
19755 92.581 -80.605 53.296
19756 92.602 -80.597 53.278
19757 92.623 -80.588 53.259
19758 92.644 -80.580 53.240
19759 92.665 -80.572 53.221
19760 92.686 -80.563 53.203
19761 92.707 -80.555 53.187
19762 92.727 -80.547 53.171
19763 92.749 -80.538 53.157
19764 92.775 -80.531 53.145
19765 92.801 -80.522 53.136
19766 92.828 -80.515 53.127
19767 92.850 -80.505 53.117
19768 92.864 -80.490 53.107
19769 92.879 -80.475 53.096
19770 92.894 -80.460 53.084
19771 92.908 -80.444 53.070
19772 92.922 -80.429 53.056
19773 92.936 -80.415 53.042
19774 92.949 -80.403 53.028
19775 92.961 -80.393 53.013
19776 92.974 -80.381 52.998
19777 92.987 -80.373 52.982
19778 93.000 -80.369 52.968
19779 93.014 -80.366 52.956
19780 93.028 -80.362 52.945
19781 93.042 -80.358 52.935
19782 93.056 -80.355 52.924
19783 93.070 -80.351 52.913
19784 93.083 -80.347 52.904
19785 93.099 -80.341 52.895
19786 93.116 -80.334 52.888
19787 93.132 -80.327 52.882
19788 93.148 -80.319 52.877
19789 93.164 -80.312 52.874
19790 93.181 -80.306 52.872
19791 93.197 -80.298 52.870
19792 93.213 -80.291 52.869
19793 93.229 -80.284 52.869
19794 93.246 -80.277 52.869
19795 93.262 -80.270 52.869
19796 93.278 -80.262 52.868
19797 93.294 -80.256 52.869
19798 93.311 -80.248 52.868
19799 93.327 -80.241 52.867
19800 93.343 -80.234 52.866
19801 93.357 -80.228 52.865
19802 93.367 -80.225 52.864
19803 93.377 -80.212 52.863
19804 93.387 -80.190 52.864
19805 93.397 -80.169 52.866
19806 93.407 -80.148 52.869
19807 93.416 -80.132 52.870
19808 93.421 -80.134 52.871
19809 93.432 -80.134 52.873
19810 93.455 -80.135 52.874
19811 93.479 -80.135 52.875
19812 93.503 -80.136 52.876
19813 93.527 -80.136 52.876
19814 93.544 -80.131 52.876
19815 93.554 -80.123 52.875
19816 93.564 -80.114 52.875
19817 93.574 -80.106 52.871
19818 93.584 -80.096 52.868
19819 93.594 -80.088 52.865
19820 93.604 -80.079 52.861
19821 93.614 -80.071 52.856
19822 93.624 -80.062 52.851
19823 93.634 -80.053 52.846
19824 93.644 -80.045 52.839
19825 93.654 -80.036 52.832
19858 93.953 -79.912 52.484
19859 93.961 -79.911 52.470
19860 93.970 -79.909 52.457
19861 93.978 -79.908 52.442
19862 93.987 -79.906 52.428
19863 93.995 -79.905 52.414
19864 94.004 -79.903 52.400
19865 94.012 -79.902 52.386
19866 94.021 -79.900 52.370
19867 94.030 -79.899 52.355
19868 94.038 -79.898 52.340
19869 94.047 -79.896 52.325
19870 94.055 -79.895 52.309
19871 94.064 -79.893 52.293
19872 94.072 -79.892 52.277
19873 94.081 -79.890 52.260
19896 94.265 -79.535 51.804
19897 94.255 -79.541 51.778
19898 94.244 -79.546 51.752
19899 94.232 -79.552 51.725
19900 94.238 -79.541 51.699
19901 94.251 -79.521 51.672
19902 94.263 -79.501 51.643
19903 94.276 -79.482 51.614
19904 94.289 -79.462 51.586
19905 94.301 -79.442 51.558
19906 94.315 -79.422 51.528
19907 94.328 -79.408 51.500
19908 94.345 -79.439 51.470
19909 94.361 -79.470 51.439
19910 94.379 -79.502 51.408
19911 94.396 -79.533 51.377
19912 94.413 -79.564 51.347
19913 94.429 -79.595 51.316
19914 94.447 -79.627 51.285
20412 91.895 -80.954 53.569
20413 91.909 -80.949 53.548
20414 91.923 -80.945 53.527
20415 91.936 -80.939 53.506
20416 91.950 -80.935 53.485
20417 91.963 -80.930 53.464
20418 91.977 -80.926 53.443
20419 91.991 -80.920 53.422
20420 92.005 -80.916 53.402
20421 92.018 -80.911 53.382
20422 92.032 -80.906 53.361
20423 92.046 -80.902 53.341
20424 92.059 -80.897 53.322
20425 92.073 -80.892 53.302
20426 92.087 -80.886 53.284
20427 92.101 -80.873 53.265
20456 92.565 -80.588 53.309
20457 92.586 -80.580 53.290
20458 92.606 -80.572 53.272
20459 92.628 -80.563 53.253
20460 92.648 -80.555 53.234
20461 92.669 -80.547 53.216
20462 92.690 -80.538 53.199
20463 92.711 -80.530 53.183
20464 92.732 -80.522 53.167
20465 92.753 -80.513 53.156
20466 92.773 -80.505 53.146
20467 92.794 -80.497 53.138
20468 92.816 -80.489 53.129
20469 92.841 -80.480 53.119
20470 92.855 -80.465 53.108
20471 92.870 -80.450 53.094
20472 92.884 -80.435 53.079
20473 92.899 -80.419 53.064
20474 92.913 -80.404 53.049
20475 92.930 -80.388 53.033
20476 92.946 -80.371 53.018
20477 92.962 -80.365 53.002
20478 92.977 -80.364 52.985
20479 92.992 -80.362 52.971
20480 93.007 -80.359 52.958
20481 93.022 -80.356 52.947
20482 93.038 -80.353 52.936
20483 93.053 -80.350 52.926
20484 93.066 -80.344 52.915
20485 93.078 -80.337 52.907
20486 93.091 -80.328 52.898
20487 93.105 -80.319 52.890
20488 93.120 -80.312 52.884
20489 93.136 -80.305 52.879
20490 93.152 -80.298 52.875
20491 93.169 -80.290 52.873
20492 93.185 -80.284 52.869
20493 93.201 -80.276 52.866
20494 93.217 -80.269 52.864
20495 93.234 -80.262 52.862
20496 93.250 -80.255 52.862
20497 93.266 -80.247 52.861
20498 93.282 -80.240 52.860
20499 93.298 -80.233 52.859
20500 93.315 -80.226 52.857
20501 93.331 -80.218 52.856
20502 93.346 -80.212 52.854
20503 93.356 -80.202 52.853
20504 93.365 -80.181 52.852
20505 93.375 -80.159 52.853
20506 93.386 -80.137 52.855
20507 93.395 -80.116 52.857
20508 93.404 -80.101 52.859
20509 93.410 -80.102 52.860
20510 93.415 -80.104 52.861
20511 93.436 -80.104 52.863
20512 93.461 -80.105 52.865
20513 93.484 -80.105 52.867
20514 93.508 -80.106 52.869
20515 93.532 -80.106 52.869
20516 93.554 -80.104 52.870
20517 93.564 -80.096 52.870
20518 93.573 -80.087 52.868
20519 93.583 -80.078 52.866
20520 93.593 -80.070 52.862
20521 93.604 -80.061 52.858
20522 93.614 -80.052 52.853
20523 93.624 -80.043 52.848
20524 93.634 -80.035 52.843
20525 93.644 -80.026 52.837
20526 93.654 -80.018 52.830
20527 93.664 -80.009 52.823
20557 93.934 -79.901 52.517
20558 93.943 -79.899 52.505
20559 93.952 -79.898 52.492
20560 93.960 -79.896 52.479
20561 93.969 -79.895 52.465
20562 93.977 -79.893 52.451
20563 93.986 -79.892 52.438
20564 93.994 -79.890 52.425
20565 94.003 -79.889 52.410
20566 94.011 -79.887 52.395
20567 94.020 -79.886 52.381
20568 94.029 -79.885 52.365
20569 94.037 -79.883 52.349
20570 94.046 -79.882 52.334
20571 94.054 -79.880 52.318
20572 94.063 -79.879 52.302
20573 94.071 -79.877 52.286
20574 94.080 -79.876 52.270
20575 94.088 -79.874 52.253
20596 94.268 -79.528 51.841
20597 94.269 -79.515 51.816
20598 94.258 -79.520 51.790
20599 94.247 -79.526 51.765
20600 94.236 -79.532 51.739
20601 94.235 -79.527 51.713
20602 94.248 -79.507 51.686
20603 94.260 -79.488 51.657
20604 94.273 -79.468 51.629
20605 94.286 -79.448 51.601
20606 94.298 -79.429 51.572
20607 94.312 -79.409 51.543
20608 94.324 -79.389 51.515
20609 94.340 -79.407 51.485
20610 94.357 -79.439 51.453
20611 94.374 -79.470 51.423
20612 94.391 -79.501 51.392
20613 94.408 -79.532 51.360
20614 94.425 -79.564 51.329
20615 94.442 -79.595 51.297
20616 94.459 -79.626 51.266
20617 94.476 -79.658 51.234
21115 91.922 -80.930 53.551
21116 91.936 -80.926 53.530
21117 91.949 -80.921 53.509
21118 91.963 -80.917 53.488
21119 91.977 -80.911 53.468
21120 91.990 -80.907 53.447
21121 92.004 -80.902 53.427
21122 92.017 -80.897 53.406
21123 92.031 -80.892 53.385
21124 92.045 -80.888 53.366
21125 92.059 -80.883 53.346
21126 92.072 -80.878 53.326
21157 92.548 -80.573 53.324
21158 92.569 -80.564 53.305
21159 92.590 -80.556 53.286
21160 92.611 -80.547 53.267
21161 92.632 -80.539 53.249
21162 92.652 -80.531 53.231
21163 92.673 -80.522 53.214
21164 92.694 -80.514 53.197
21165 92.715 -80.506 53.181
21166 92.736 -80.497 53.169
21167 92.757 -80.489 53.159
21168 92.778 -80.481 53.150
21169 92.798 -80.472 53.142
21170 92.819 -80.464 53.132
21171 92.840 -80.456 53.120
21172 92.863 -80.432 53.106
21173 92.883 -80.408 53.090
21174 92.901 -80.391 53.073
21175 92.917 -80.374 53.056
21176 92.934 -80.357 53.039
21177 92.951 -80.341 53.022
21178 92.966 -80.336 53.005
21179 92.981 -80.339 52.989
21180 92.995 -80.342 52.975
21181 93.009 -80.346 52.962
21182 93.023 -80.349 52.950
21183 93.036 -80.347 52.939
21184 93.048 -80.341 52.929
21185 93.060 -80.334 52.919
21186 93.072 -80.326 52.910
21187 93.085 -80.318 52.901
21188 93.099 -80.309 52.893
21189 93.113 -80.300 52.886
21190 93.126 -80.291 52.881
21191 93.140 -80.283 52.876
21192 93.157 -80.275 52.872
21193 93.173 -80.269 52.868
21194 93.189 -80.261 52.863
21195 93.205 -80.254 52.860
21196 93.221 -80.247 52.859
21197 93.238 -80.240 52.857
21198 93.255 -80.233 52.856
21199 93.271 -80.225 52.854
21200 93.288 -80.218 52.852
21201 93.304 -80.211 52.849
21202 93.321 -80.204 52.848
21203 93.334 -80.192 52.847
21204 93.344 -80.170 52.846
21205 93.354 -80.149 52.845
21206 93.364 -80.127 52.845
21207 93.373 -80.106 52.845
21208 93.383 -80.084 52.846
21209 93.392 -80.069 52.848
21210 93.398 -80.071 52.850
21211 93.404 -80.073 52.851
21212 93.418 -80.074 52.854
21213 93.441 -80.074 52.857
21214 93.466 -80.074 52.860
21215 93.489 -80.074 52.863
21216 93.513 -80.075 52.864
21217 93.537 -80.075 52.865
21218 93.561 -80.076 52.866
21219 93.573 -80.068 52.865
21220 93.583 -80.060 52.863
21221 93.593 -80.051 52.860
21222 93.604 -80.043 52.856
21223 93.614 -80.034 52.851
21224 93.624 -80.025 52.846
21225 93.634 -80.017 52.841
21226 93.644 -80.008 52.835
21227 93.654 -79.999 52.828
21228 93.664 -79.990 52.822
21229 93.674 -79.982 52.814
21230 93.684 -79.973 52.805
21231 93.694 -79.965 52.796
21232 93.704 -79.956 52.788
21233 93.714 -79.947 52.779
21234 93.724 -79.939 52.769
21235 93.734 -79.930 52.760
21236 93.745 -79.921 52.750
21258 93.933 -79.886 52.524
21259 93.942 -79.885 52.512
21260 93.951 -79.883 52.500
21261 93.959 -79.882 52.487
21262 93.968 -79.880 52.474
21263 93.976 -79.879 52.460
21264 93.985 -79.877 52.447
21265 93.993 -79.876 52.434
21266 94.002 -79.874 52.420
21267 94.010 -79.874 52.405
21268 94.019 -79.872 52.390
21269 94.028 -79.871 52.375
21270 94.036 -79.869 52.359
21271 94.045 -79.867 52.344
21272 94.053 -79.866 52.328
21273 94.062 -79.864 52.312
21274 94.070 -79.863 52.295
21275 94.079 -79.861 52.279
21276 94.087 -79.860 52.262
21296 94.259 -79.541 51.878
21297 94.267 -79.514 51.853
21298 94.272 -79.494 51.828
21299 94.261 -79.501 51.803
21300 94.250 -79.507 51.778
21301 94.239 -79.512 51.753
21302 94.232 -79.514 51.727
21303 94.245 -79.494 51.699
21304 94.258 -79.475 51.671
21305 94.270 -79.455 51.643
21306 94.283 -79.435 51.615
21307 94.296 -79.416 51.587
21308 94.309 -79.396 51.558
21309 94.321 -79.376 51.529
21310 94.336 -79.376 51.499
21311 94.353 -79.407 51.468
21312 94.369 -79.438 51.437
21313 94.387 -79.469 51.405
21314 94.404 -79.501 51.373
21315 94.421 -79.532 51.342
21316 94.437 -79.563 51.310
21317 94.455 -79.595 51.277
21318 94.472 -79.626 51.245
21319 94.485 -79.664 51.212
21320 94.493 -79.707 51.179
21819 91.962 -80.902 53.513
21820 91.976 -80.898 53.492
21821 91.990 -80.893 53.472
21822 92.003 -80.888 53.451
21823 92.017 -80.883 53.430
21858 92.533 -80.556 53.339
21859 92.553 -80.548 53.321
21860 92.573 -80.540 53.301
21861 92.594 -80.532 53.283
21862 92.615 -80.523 53.265
21863 92.636 -80.515 53.247
21864 92.657 -80.507 53.229
21865 92.677 -80.498 53.212
21866 92.698 -80.490 53.196
21867 92.719 -80.482 53.184
21868 92.740 -80.473 53.174
21869 92.761 -80.465 53.165
21870 92.782 -80.457 53.156
21871 92.802 -80.448 53.146
21872 92.824 -80.440 53.133
21873 92.844 -80.431 53.117
21874 92.866 -80.415 53.101
21875 92.889 -80.388 53.084
21876 92.912 -80.360 53.066
21877 92.935 -80.333 53.048
21878 92.955 -80.309 53.030
21879 92.971 -80.306 53.013
21880 92.983 -80.311 52.996
21881 92.994 -80.316 52.981
21882 93.006 -80.321 52.968
21883 93.018 -80.326 52.955
21884 93.030 -80.331 52.944
21885 93.042 -80.330 52.933
21886 93.054 -80.323 52.924
21887 93.067 -80.316 52.914
21888 93.080 -80.306 52.904
21889 93.093 -80.298 52.895
21890 93.107 -80.289 52.888
21891 93.119 -80.279 52.882
21892 93.136 -80.270 52.877
21893 93.161 -80.262 52.872
21894 93.178 -80.254 52.867
21895 93.194 -80.247 52.862
21896 93.211 -80.240 52.858
21897 93.227 -80.233 52.856
21898 93.244 -80.225 52.855
21899 93.261 -80.218 52.853
21900 93.277 -80.211 52.851
21901 93.294 -80.204 52.848
21902 93.310 -80.196 52.846
21903 93.320 -80.186 52.844
21904 93.325 -80.165 52.843
21905 93.333 -80.140 52.842
21906 93.342 -80.117 52.840
21907 93.352 -80.096 52.839
21908 93.362 -80.074 52.840
21909 93.372 -80.052 52.840
21910 93.380 -80.038 52.841
21911 93.386 -80.040 52.843
21912 93.392 -80.042 52.845
21913 93.399 -80.043 52.847
21914 93.423 -80.043 52.851
21915 93.446 -80.043 52.855
21916 93.471 -80.044 52.859
21917 93.494 -80.044 52.860
21918 93.518 -80.045 52.861
21919 93.542 -80.045 52.863
21920 93.566 -80.046 52.862
21921 93.583 -80.042 52.861
21922 93.593 -80.033 52.859
21923 93.604 -80.024 52.854
21924 93.614 -80.015 52.850
21925 93.624 -80.007 52.845
21926 93.634 -79.998 52.840
21927 93.644 -79.990 52.834
21928 93.654 -79.981 52.827
21929 93.664 -79.972 52.820
21930 93.674 -79.964 52.813
21931 93.684 -79.955 52.804
21932 93.694 -79.946 52.795
21933 93.704 -79.937 52.787
21934 93.714 -79.929 52.778
21935 93.724 -79.920 52.769
21936 93.734 -79.911 52.760
21937 93.745 -79.904 52.751
21938 93.753 -79.902 52.741
21959 93.933 -79.872 52.532
21960 93.941 -79.871 52.520
21961 93.950 -79.869 52.508
21962 93.959 -79.867 52.495
21963 93.967 -79.866 52.483
21964 93.976 -79.864 52.469
21965 93.984 -79.863 52.455
21966 93.993 -79.861 52.442
21967 94.001 -79.861 52.429
21968 94.010 -79.859 52.415
21969 94.018 -79.858 52.400
21970 94.027 -79.856 52.384
21971 94.035 -79.855 52.369
21972 94.044 -79.853 52.353
21973 94.052 -79.852 52.337
21974 94.061 -79.850 52.321
21975 94.069 -79.849 52.304
21976 94.078 -79.847 52.288
21977 94.086 -79.845 52.270
21978 94.095 -79.844 52.253
21995 94.241 -79.579 51.934
21996 94.249 -79.553 51.912
21997 94.258 -79.527 51.888
21998 94.266 -79.501 51.864
21999 94.275 -79.476 51.840
22000 94.264 -79.480 51.815
22001 94.253 -79.486 51.790
22002 94.243 -79.492 51.765
22003 94.231 -79.498 51.739
22004 94.242 -79.481 51.712
22005 94.255 -79.461 51.685
22006 94.267 -79.441 51.656
22007 94.280 -79.422 51.628
22008 94.293 -79.402 51.600
22009 94.306 -79.382 51.572
22010 94.318 -79.363 51.543
22011 94.331 -79.344 51.513
22012 94.348 -79.375 51.482
22013 94.365 -79.407 51.451
22014 94.382 -79.438 51.419
22015 94.399 -79.469 51.386
22016 94.416 -79.501 51.354
22017 94.433 -79.532 51.322
22018 94.450 -79.563 51.290
22019 94.465 -79.598 51.255
22020 94.474 -79.641 51.222
22021 94.483 -79.684 51.189
22022 94.491 -79.727 51.156
22559 92.529 -80.539 53.357
22560 92.546 -80.531 53.338
22561 92.562 -80.523 53.319
22562 92.578 -80.515 53.301
22563 92.598 -80.507 53.282
22564 92.619 -80.498 53.263
22565 92.640 -80.490 53.244
22566 92.661 -80.482 53.227
22567 92.682 -80.473 53.212
22568 92.703 -80.465 53.200
22569 92.723 -80.457 53.190
22570 92.744 -80.448 53.180
22571 92.765 -80.440 53.170
22572 92.786 -80.431 53.159
22573 92.807 -80.423 53.146
22574 92.828 -80.415 53.130
22575 92.848 -80.406 53.113
22576 92.869 -80.398 53.096
22577 92.892 -80.371 53.078
22578 92.915 -80.344 53.060
22579 92.938 -80.316 53.042
22580 92.961 -80.289 53.025
22581 92.976 -80.288 53.007
22582 92.989 -80.294 52.991
22583 93.001 -80.299 52.976
22584 93.013 -80.303 52.963
22585 93.024 -80.309 52.951
22586 93.037 -80.314 52.940
22587 93.049 -80.312 52.929
22588 93.061 -80.305 52.918
22589 93.074 -80.296 52.907
22590 93.087 -80.287 52.896
22591 93.098 -80.277 52.889
22592 93.109 -80.266 52.882
22593 93.131 -80.257 52.876
22594 93.156 -80.249 52.871
22595 93.181 -80.240 52.866
22596 93.200 -80.233 52.862
22597 93.216 -80.225 52.858
22598 93.233 -80.218 52.856
22599 93.250 -80.212 52.854
22600 93.266 -80.204 52.851
22601 93.283 -80.197 52.849
22602 93.299 -80.190 52.847
22603 93.309 -80.178 52.844
22604 93.307 -80.162 52.842
22605 93.315 -80.137 52.841
22606 93.324 -80.113 52.839
22607 93.332 -80.089 52.837
22608 93.341 -80.065 52.836
22609 93.350 -80.043 52.836
22610 93.360 -80.021 52.836
22611 93.368 -80.007 52.836
22612 93.374 -80.008 52.838
22613 93.380 -80.011 52.840
22614 93.386 -80.012 52.844
22615 93.404 -80.013 52.847
22616 93.428 -80.013 52.852
22617 93.451 -80.013 52.856
22618 93.476 -80.014 52.859
22619 93.499 -80.014 52.860
22620 93.523 -80.015 52.861
22621 93.547 -80.015 52.861
22622 93.571 -80.015 52.860
22623 93.593 -80.015 52.857
22624 93.604 -80.005 52.853
22625 93.614 -79.997 52.848
22626 93.624 -79.989 52.844
22627 93.634 -79.980 52.839
22628 93.644 -79.971 52.833
22629 93.654 -79.962 52.826
22630 93.664 -79.954 52.819
22631 93.674 -79.945 52.812
22632 93.684 -79.936 52.804
22633 93.694 -79.928 52.795
22634 93.704 -79.919 52.786
22635 93.714 -79.911 52.778
22636 93.724 -79.902 52.770
22637 93.734 -79.893 52.760
22638 93.744 -79.889 52.751
22639 93.752 -79.889 52.742
22640 93.761 -79.887 52.733
22661 93.940 -79.856 52.527
22662 93.949 -79.855 52.514
22663 93.958 -79.853 52.502
22664 93.966 -79.852 52.490
22665 93.975 -79.850 52.477
22666 93.983 -79.849 52.464
22667 93.992 -79.848 52.451
22668 94.000 -79.846 52.437
22669 94.009 -79.845 52.423
22670 94.017 -79.843 52.409
22671 94.026 -79.842 52.393
22672 94.035 -79.840 52.377
22673 94.043 -79.839 52.362
22674 94.052 -79.837 52.345
22675 94.060 -79.836 52.329
22676 94.069 -79.834 52.313
22677 94.077 -79.833 52.296
22678 94.086 -79.831 52.278
22679 94.095 -79.830 52.260
22695 94.231 -79.592 51.964
22696 94.240 -79.565 51.943
22697 94.248 -79.539 51.921
22698 94.257 -79.513 51.898
22699 94.265 -79.488 51.874
22700 94.274 -79.462 51.851
22701 94.267 -79.460 51.826
22702 94.257 -79.466 51.801
22703 94.246 -79.472 51.776
22704 94.235 -79.478 51.751
22705 94.239 -79.467 51.725
22706 94.252 -79.447 51.697
22707 94.264 -79.428 51.669
22708 94.277 -79.408 51.641
22709 94.290 -79.388 51.613
22710 94.302 -79.369 51.585
22711 94.316 -79.349 51.556
22712 94.328 -79.329 51.527
22713 94.344 -79.344 51.495
22714 94.361 -79.375 51.463
22715 94.378 -79.407 51.430
22716 94.395 -79.438 51.398
22717 94.412 -79.469 51.365
22718 94.429 -79.501 51.333
22719 94.445 -79.532 51.301
22720 94.454 -79.576 51.266
22721 94.463 -79.619 51.232
22722 94.472 -79.662 51.198
22723 94.481 -79.705 51.166
22724 94.489 -79.748 51.132
22725 94.498 -79.792 51.099
23260 92.525 -80.522 53.375
23261 92.542 -80.514 53.357
23262 92.558 -80.506 53.338
23263 92.573 -80.498 53.319
23264 92.589 -80.490 53.300
23265 92.605 -80.482 53.281
23266 92.623 -80.474 53.263
23267 92.644 -80.466 53.245
23268 92.665 -80.457 53.229
23269 92.686 -80.449 53.217
23270 92.707 -80.441 53.206
23271 92.727 -80.432 53.195
23272 92.748 -80.424 53.184
23273 92.769 -80.416 53.172
23274 92.790 -80.407 53.159
23275 92.811 -80.399 53.144
23276 92.832 -80.391 53.127
23277 92.853 -80.382 53.110
23278 92.873 -80.374 53.092
23279 92.896 -80.353 53.074
23280 92.919 -80.326 53.056
23281 92.941 -80.299 53.038
23282 92.965 -80.272 53.021
23283 92.979 -80.270 53.004
23284 92.992 -80.275 52.987
23285 93.005 -80.281 52.972
23286 93.017 -80.286 52.958
23287 93.030 -80.291 52.946
23288 93.043 -80.296 52.934
23289 93.055 -80.294 52.922
23290 93.066 -80.284 52.910
23291 93.077 -80.274 52.900
23292 93.088 -80.263 52.892
23293 93.101 -80.253 52.884
23294 93.126 -80.245 52.876
23295 93.151 -80.237 52.871
23296 93.176 -80.228 52.866
23297 93.201 -80.219 52.862
23298 93.222 -80.212 52.859
23299 93.239 -80.204 52.857
23300 93.256 -80.197 52.855
23301 93.272 -80.190 52.851
23302 93.289 -80.183 52.848
23303 93.297 -80.171 52.845
23304 93.296 -80.154 52.843
23305 93.297 -80.134 52.842
23306 93.305 -80.110 52.840
23307 93.314 -80.086 52.839
23308 93.323 -80.062 52.836
23309 93.331 -80.037 52.834
23310 93.340 -80.013 52.833
23311 93.348 -79.990 52.832
23312 93.357 -79.976 52.833
23313 93.362 -79.977 52.834
23314 93.370 -79.982 52.836
23315 93.381 -79.992 52.840
23316 93.395 -79.988 52.844
23317 93.409 -79.983 52.849
23318 93.433 -79.983 52.854
23319 93.457 -79.983 52.857
23320 93.481 -79.983 52.859
23321 93.504 -79.983 52.860
23322 93.529 -79.984 52.860
23323 93.552 -79.984 52.858
23324 93.576 -79.985 52.856
23325 93.600 -79.985 52.851
23326 93.614 -79.979 52.847
23327 93.624 -79.970 52.842
23328 93.634 -79.961 52.838
23329 93.644 -79.953 52.831
23330 93.654 -79.944 52.824
23331 93.664 -79.936 52.818
23332 93.674 -79.927 52.810
23333 93.684 -79.918 52.802
23334 93.694 -79.909 52.794
23335 93.704 -79.901 52.785
23336 93.714 -79.892 52.777
23337 93.724 -79.883 52.770
23338 93.734 -79.877 52.760
23339 93.743 -79.876 52.751
23340 93.751 -79.874 52.742
23341 93.760 -79.873 52.733
23362 93.939 -79.842 52.533
23363 93.948 -79.840 52.521
23364 93.957 -79.839 52.509
23365 93.965 -79.837 52.496
23366 93.974 -79.836 52.483
23367 93.982 -79.835 52.470
23368 93.991 -79.833 52.457
23369 93.999 -79.832 52.443
23370 94.008 -79.830 52.429
23371 94.016 -79.829 52.414
23372 94.025 -79.827 52.399
23373 94.034 -79.826 52.383
23374 94.042 -79.824 52.367
23375 94.051 -79.823 52.351
23376 94.059 -79.821 52.335
23377 94.068 -79.820 52.319
23378 94.076 -79.818 52.302
23379 94.085 -79.817 52.285
23380 94.094 -79.816 52.267
23381 94.102 -79.814 52.250
23395 94.222 -79.604 51.993
23396 94.230 -79.578 51.971
23397 94.239 -79.552 51.951
23398 94.247 -79.526 51.930
23399 94.256 -79.501 51.907
23400 94.264 -79.474 51.883
23401 94.273 -79.448 51.860
23402 94.271 -79.440 51.835
23403 94.260 -79.446 51.810
23404 94.249 -79.452 51.786
23405 94.238 -79.457 51.762
23406 94.236 -79.454 51.736
23407 94.249 -79.435 51.709
23408 94.261 -79.415 51.681
23409 94.274 -79.394 51.653
23410 94.287 -79.375 51.625
23411 94.299 -79.355 51.597
23412 94.313 -79.335 51.569
23413 94.325 -79.316 51.539
23414 94.339 -79.313 51.508
23415 94.356 -79.344 51.475
23416 94.373 -79.375 51.442
23417 94.390 -79.406 51.409
23418 94.407 -79.438 51.376
23419 94.424 -79.469 51.343
23420 94.434 -79.510 51.311
23421 94.443 -79.553 51.276
23422 94.452 -79.596 51.241
23423 94.461 -79.639 51.207
23424 94.470 -79.683 51.174
23425 94.479 -79.726 51.140
23426 94.487 -79.770 51.107
23961 92.521 -80.505 53.395
23962 92.537 -80.497 53.377
23963 92.553 -80.489 53.358
23964 92.569 -80.481 53.339
23965 92.585 -80.473 53.320
23966 92.601 -80.465 53.301
23967 92.618 -80.457 53.282
23968 92.634 -80.449 53.264
23969 92.649 -80.441 53.247
23970 92.669 -80.433 53.233
23971 92.690 -80.425 53.221
23972 92.711 -80.416 53.210
23973 92.732 -80.408 53.199
23974 92.753 -80.400 53.186
23975 92.773 -80.391 53.173
23976 92.794 -80.383 53.157
23977 92.815 -80.375 53.141
23978 92.836 -80.366 53.124
23979 92.857 -80.358 53.108
23980 92.877 -80.350 53.090
23981 92.899 -80.337 53.072
23982 92.922 -80.309 53.053
23983 92.945 -80.282 53.036
23984 92.968 -80.255 53.019
23985 92.983 -80.252 53.001
23986 92.996 -80.257 52.983
23987 93.008 -80.262 52.966
23988 93.021 -80.268 52.952
23989 93.034 -80.273 52.939
23990 93.047 -80.278 52.926
23991 93.057 -80.271 52.914
23992 93.067 -80.261 52.904
23993 93.078 -80.250 52.895
23994 93.096 -80.241 52.887
23995 93.121 -80.232 52.879
23996 93.146 -80.224 52.873
23997 93.171 -80.215 52.868
23998 93.196 -80.207 52.864
23999 93.221 -80.199 52.861
24000 93.245 -80.190 52.858
24001 93.262 -80.183 52.855
24002 93.278 -80.176 52.852
24003 93.286 -80.164 52.848
24004 93.284 -80.146 52.845
24005 93.283 -80.129 52.842
24006 93.288 -80.108 52.840
24007 93.296 -80.083 52.838
24008 93.304 -80.059 52.836
24009 93.313 -80.034 52.833
24010 93.322 -80.010 52.831
24011 93.330 -79.986 52.830
24012 93.339 -79.964 52.827
24013 93.354 -79.958 52.828
24014 93.365 -79.968 52.830
24015 93.375 -79.977 52.831
24016 93.387 -79.987 52.836
24017 93.404 -79.974 52.840
24018 93.418 -79.968 52.845
24019 93.431 -79.962 52.852
24020 93.444 -79.956 52.856
24021 93.462 -79.953 52.858
24022 93.485 -79.953 52.860
24023 93.509 -79.953 52.859
24024 93.533 -79.954 52.857
24025 93.557 -79.954 52.854
24026 93.581 -79.955 52.849
24027 93.605 -79.955 52.845
24028 93.624 -79.952 52.841
24029 93.634 -79.943 52.835
24030 93.644 -79.934 52.829
24031 93.654 -79.926 52.823
24032 93.664 -79.917 52.816
24033 93.674 -79.908 52.808
24034 93.684 -79.900 52.800
24035 93.694 -79.891 52.792
24036 93.704 -79.883 52.784
24037 93.714 -79.874 52.776
24038 93.724 -79.865 52.769
24039 93.733 -79.863 52.759
24040 93.742 -79.861 52.750
24041 93.751 -79.860 52.742
24042 93.759 -79.858 52.733
24043 93.768 -79.857 52.724
24063 93.938 -79.827 52.538
24064 93.947 -79.826 52.527
24065 93.956 -79.824 52.514
24066 93.964 -79.823 52.501
24067 93.973 -79.822 52.488
24068 93.981 -79.820 52.475
24069 93.990 -79.819 52.462
24070 93.998 -79.817 52.449
24071 94.007 -79.816 52.435
24072 94.015 -79.814 52.419
24073 94.024 -79.813 52.404
24074 94.033 -79.811 52.388
24075 94.041 -79.810 52.372
24076 94.050 -79.808 52.356
24077 94.058 -79.807 52.340
24078 94.067 -79.805 52.324
24079 94.075 -79.804 52.307
24080 94.084 -79.803 52.290
24081 94.092 -79.802 52.272
24082 94.101 -79.800 52.254
24083 94.110 -79.798 52.237
24094 94.204 -79.642 52.038
24095 94.212 -79.616 52.018
24096 94.221 -79.590 51.997
24097 94.229 -79.564 51.977
24098 94.238 -79.538 51.957
24099 94.247 -79.513 51.936
24100 94.255 -79.487 51.914
24101 94.264 -79.461 51.891
24102 94.272 -79.435 51.867
24103 94.274 -79.420 51.843
24104 94.263 -79.426 51.819
24105 94.252 -79.432 51.795
24106 94.241 -79.438 51.772
24107 94.233 -79.441 51.746
24108 94.246 -79.421 51.719
24109 94.259 -79.401 51.692
24110 94.271 -79.382 51.664
24111 94.284 -79.362 51.636
24112 94.297 -79.342 51.609
24113 94.310 -79.322 51.581
24114 94.322 -79.303 51.551
24115 94.335 -79.283 51.519
24116 94.352 -79.312 51.486
24117 94.369 -79.343 51.453
24118 94.386 -79.375 51.420
24119 94.403 -79.406 51.386
24120 94.415 -79.444 51.353
24121 94.424 -79.488 51.319
24122 94.432 -79.531 51.284
24123 94.441 -79.574 51.250
24124 94.450 -79.617 51.216
24125 94.459 -79.661 51.182
24126 94.468 -79.704 51.148
24127 94.477 -79.747 51.114
24662 92.517 -80.488 53.417
24663 92.533 -80.480 53.398
24664 92.549 -80.472 53.379
24665 92.565 -80.464 53.359
24666 92.581 -80.457 53.342
24667 92.597 -80.448 53.322
24668 92.614 -80.440 53.303
24669 92.629 -80.432 53.284
24670 92.645 -80.424 53.266
24671 92.661 -80.416 53.250
24672 92.677 -80.408 53.235
24673 92.694 -80.400 53.224
24674 92.715 -80.391 53.211
24675 92.736 -80.383 53.199
24676 92.757 -80.375 53.185
24677 92.778 -80.366 53.170
24678 92.798 -80.358 53.154
24679 92.819 -80.350 53.138
24680 92.840 -80.341 53.122
24681 92.861 -80.333 53.105
24682 92.882 -80.325 53.087
24683 92.903 -80.316 53.068
24684 92.925 -80.292 53.051
24685 92.948 -80.265 53.033
24686 92.971 -80.237 53.014
24687 92.987 -80.234 52.995
24688 92.999 -80.239 52.978
24689 93.012 -80.244 52.961
24690 93.025 -80.250 52.947
24691 93.038 -80.255 52.933
24692 93.048 -80.256 52.921
24693 93.058 -80.248 52.910
24694 93.068 -80.237 52.901
24695 93.091 -80.228 52.892
24696 93.116 -80.220 52.884
24697 93.141 -80.212 52.877
24698 93.166 -80.203 52.871
24699 93.191 -80.194 52.866
24700 93.216 -80.186 52.862
24701 93.241 -80.178 52.859
24702 93.266 -80.169 52.857
24703 93.275 -80.156 52.853
24704 93.273 -80.139 52.848
24705 93.271 -80.122 52.844
24706 93.270 -80.105 52.842
24707 93.278 -80.080 52.838
24708 93.287 -80.056 52.835
24709 93.295 -80.032 52.833
24710 93.304 -80.008 52.830
24711 93.313 -79.990 52.828
24712 93.322 -79.971 52.826
24713 93.334 -79.957 52.823
24714 93.358 -79.952 52.823
24715 93.370 -79.962 52.825
24716 93.381 -79.971 52.828
24717 93.392 -79.981 52.833
24718 93.414 -79.961 52.837
24719 93.427 -79.955 52.842
24720 93.440 -79.949 52.848
24721 93.453 -79.943 52.852
24722 93.467 -79.936 52.856
24723 93.480 -79.930 52.857
24724 93.493 -79.924 52.856
24725 93.514 -79.923 52.854
24726 93.538 -79.924 52.850
24727 93.562 -79.924 52.846
24728 93.586 -79.924 52.842
24729 93.610 -79.924 52.837
24730 93.634 -79.924 52.832
24731 93.644 -79.916 52.826
24732 93.654 -79.908 52.821
24733 93.664 -79.899 52.815
24734 93.674 -79.890 52.807
24735 93.684 -79.881 52.798
24736 93.694 -79.873 52.791
24737 93.704 -79.864 52.783
24738 93.714 -79.855 52.775
24739 93.724 -79.850 52.767
24740 93.732 -79.849 52.758
24741 93.741 -79.847 52.749
24742 93.750 -79.845 52.741
24743 93.758 -79.844 52.732
24744 93.767 -79.842 52.723
24764 93.938 -79.813 52.541
24765 93.946 -79.811 52.530
24766 93.955 -79.811 52.517
24767 93.964 -79.809 52.505
24768 93.972 -79.808 52.493
24769 93.981 -79.806 52.479
24770 93.989 -79.805 52.466
24771 93.998 -79.803 52.453
24772 94.006 -79.802 52.439
24773 94.015 -79.800 52.423
24774 94.024 -79.798 52.408
24775 94.032 -79.797 52.392
24776 94.041 -79.795 52.376
24777 94.049 -79.794 52.360
24778 94.058 -79.792 52.344
24779 94.066 -79.791 52.328
24780 94.075 -79.790 52.310
24781 94.083 -79.789 52.293
24782 94.092 -79.787 52.275
24783 94.101 -79.786 52.258
24784 94.109 -79.784 52.241
24785 94.117 -79.783 52.223
24794 94.194 -79.654 52.064
24795 94.203 -79.629 52.043
24796 94.211 -79.603 52.023
24797 94.220 -79.577 52.003
24798 94.228 -79.551 51.983
24799 94.237 -79.525 51.963
24800 94.246 -79.499 51.942
24801 94.254 -79.473 51.921
24802 94.263 -79.447 51.898
24803 94.271 -79.422 51.875
24804 94.277 -79.400 51.851
24805 94.266 -79.406 51.828
24806 94.255 -79.412 51.805
24807 94.245 -79.418 51.782
24808 94.234 -79.423 51.757
24809 94.243 -79.407 51.730
24810 94.256 -79.388 51.702
24811 94.268 -79.368 51.676
24812 94.281 -79.348 51.649
24813 94.294 -79.328 51.622
24814 94.307 -79.309 51.594
24815 94.319 -79.289 51.563
24816 94.332 -79.269 51.530
24817 94.347 -79.280 51.496
24818 94.364 -79.312 51.463
24819 94.382 -79.343 51.430
24820 94.395 -79.379 51.396
24821 94.404 -79.422 51.363
24822 94.413 -79.465 51.329
24823 94.422 -79.508 51.294
24824 94.430 -79.551 51.259
24825 94.439 -79.595 51.225
24826 94.448 -79.639 51.190
24827 94.457 -79.682 51.156
24828 94.466 -79.725 51.122
25363 92.513 -80.472 53.438
25364 92.529 -80.463 53.420
25365 92.545 -80.455 53.401
25366 92.561 -80.447 53.382
25367 92.577 -80.439 53.363
25368 92.593 -80.431 53.343
25369 92.609 -80.423 53.323
25370 92.625 -80.415 53.304
25371 92.641 -80.407 53.285
25372 92.657 -80.399 53.267
25373 92.673 -80.391 53.250
25374 92.690 -80.383 53.234
25375 92.705 -80.375 53.221
25376 92.721 -80.367 53.208
25377 92.740 -80.359 53.195
25378 92.761 -80.350 53.182
25379 92.782 -80.342 53.167
25380 92.802 -80.334 53.152
25381 92.824 -80.325 53.136
25382 92.844 -80.317 53.119
25383 92.865 -80.309 53.101
25384 92.886 -80.300 53.083
25385 92.907 -80.292 53.065
25386 92.929 -80.275 53.047
25387 92.951 -80.248 53.026
25388 92.975 -80.221 53.007
25389 92.990 -80.216 52.988
25390 93.003 -80.222 52.970
25391 93.016 -80.226 52.954
25392 93.028 -80.231 52.940
25393 93.041 -80.236 52.927
25394 93.048 -80.233 52.915
25395 93.061 -80.225 52.906
25396 93.086 -80.216 52.896
25397 93.111 -80.208 52.888
25398 93.135 -80.199 52.880
25399 93.157 -80.189 52.874
25400 93.180 -80.178 52.869
25401 93.202 -80.168 52.865
25402 93.224 -80.159 52.862
25403 93.247 -80.148 52.859
25404 93.261 -80.137 52.855
25405 93.260 -80.115 52.850
25406 93.258 -80.097 52.845
25407 93.260 -80.077 52.841
25408 93.269 -80.053 52.837
25409 93.278 -80.034 52.833
25410 93.287 -80.015 52.829
25411 93.296 -79.996 52.827
25412 93.305 -79.977 52.825
25413 93.317 -79.961 52.822
25414 93.331 -79.952 52.819
25415 93.352 -79.947 52.819
25416 93.376 -79.957 52.821
25417 93.387 -79.966 52.825
25418 93.398 -79.976 52.829
25419 93.421 -79.949 52.834
25420 93.436 -79.940 52.839
25421 93.449 -79.934 52.844
25422 93.463 -79.928 52.849
25423 93.476 -79.923 52.853
25424 93.489 -79.917 52.854
25425 93.502 -79.911 52.853
25426 93.516 -79.905 52.851
25427 93.529 -79.899 52.846
25428 93.543 -79.893 52.843
25429 93.567 -79.893 52.840
25430 93.591 -79.894 52.835
25431 93.615 -79.894 52.829
25432 93.639 -79.895 52.823
25433 93.654 -79.889 52.817
25434 93.664 -79.880 52.810
25435 93.674 -79.872 52.803
25436 93.684 -79.863 52.796
25437 93.694 -79.855 52.789
25438 93.704 -79.845 52.781
25439 93.714 -79.837 52.772
25440 93.723 -79.836 52.764
25441 93.731 -79.834 52.757
25442 93.740 -79.833 52.748
25443 93.749 -79.831 52.739
25444 93.757 -79.830 52.731
25445 93.766 -79.828 52.722
25446 93.774 -79.827 52.713
25465 93.937 -79.798 52.544
25466 93.945 -79.798 52.533
25467 93.954 -79.796 52.520
25468 93.963 -79.795 52.509
25469 93.971 -79.793 52.496
25470 93.980 -79.792 52.483
25471 93.988 -79.790 52.469
25472 93.997 -79.789 52.456
25473 94.005 -79.787 52.442
25474 94.014 -79.786 52.427
25475 94.023 -79.784 52.411
25476 94.031 -79.783 52.395
25477 94.040 -79.781 52.379
25478 94.048 -79.780 52.363
25479 94.057 -79.778 52.346
25480 94.065 -79.777 52.331
25481 94.074 -79.776 52.313
25482 94.082 -79.774 52.295
25483 94.091 -79.773 52.278
25484 94.100 -79.771 52.261
25485 94.108 -79.770 52.244
25486 94.117 -79.768 52.228
25487 94.125 -79.767 52.211
25492 94.168 -79.719 52.122
25493 94.177 -79.692 52.104
25494 94.185 -79.667 52.087
25495 94.194 -79.641 52.068
25496 94.202 -79.615 52.048
25497 94.210 -79.589 52.028
25498 94.219 -79.563 52.008
25499 94.227 -79.538 51.988
25500 94.236 -79.512 51.968
25501 94.245 -79.486 51.948
25502 94.253 -79.460 51.927
25503 94.262 -79.434 51.904
25504 94.270 -79.408 51.882
25505 94.279 -79.382 51.860
25506 94.269 -79.386 51.838
25507 94.259 -79.391 51.816
25508 94.248 -79.397 51.794
25509 94.237 -79.404 51.769
25510 94.240 -79.394 51.742
25511 94.253 -79.374 51.714
25512 94.265 -79.354 51.688
25513 94.278 -79.335 51.663
25514 94.291 -79.315 51.637
25515 94.303 -79.295 51.609
25516 94.316 -79.275 51.577
25517 94.329 -79.256 51.542
25518 94.343 -79.249 51.508
25519 94.360 -79.280 51.475
25520 94.376 -79.313 51.441
25521 94.384 -79.357 51.406
25522 94.393 -79.400 51.372
25523 94.402 -79.443 51.338
25524 94.411 -79.486 51.303
25525 94.419 -79.529 51.268
25526 94.428 -79.573 51.233
25527 94.437 -79.616 51.198
25528 94.446 -79.659 51.164
25529 94.455 -79.702 51.129
26065 92.525 -80.447 53.442
26066 92.541 -80.438 53.423
26067 92.557 -80.430 53.404
26068 92.573 -80.422 53.385
26069 92.589 -80.414 53.365
26070 92.605 -80.406 53.345
26071 92.621 -80.398 53.325
26072 92.637 -80.390 53.305
26073 92.653 -80.382 53.285
26074 92.669 -80.374 53.266
26075 92.685 -80.366 53.248
26076 92.701 -80.358 53.232
26077 92.717 -80.350 53.216
26078 92.733 -80.342 53.201
26079 92.749 -80.334 53.187
26080 92.766 -80.326 53.174
26081 92.786 -80.318 53.160
26082 92.807 -80.309 53.146
26083 92.828 -80.301 53.131
26084 92.849 -80.293 53.115
26085 92.869 -80.284 53.097
26086 92.890 -80.276 53.077
26087 92.911 -80.268 53.058
26088 92.932 -80.259 53.037
26089 92.955 -80.231 53.017
26090 92.978 -80.204 52.998
26091 92.994 -80.198 52.978
26092 93.007 -80.203 52.961
26093 93.019 -80.209 52.946
26094 93.032 -80.214 52.933
26095 93.042 -80.213 52.921
26096 93.050 -80.208 52.910
26097 93.068 -80.196 52.901
26098 93.090 -80.186 52.892
26099 93.112 -80.176 52.884
26100 93.135 -80.166 52.877
26101 93.157 -80.156 52.871
26102 93.180 -80.146 52.867
26103 93.202 -80.136 52.863
26104 93.224 -80.125 52.859
26105 93.248 -80.120 52.855
26106 93.247 -80.099 52.850
26107 93.245 -80.078 52.844
26108 93.252 -80.060 52.840
26109 93.261 -80.041 52.835
26110 93.270 -80.022 52.830
26111 93.279 -80.002 52.824
26112 93.288 -79.983 52.822
26113 93.298 -79.967 52.819
26114 93.313 -79.958 52.817
26115 93.328 -79.949 52.815
26116 93.347 -79.942 52.815
26117 93.381 -79.952 52.817
26118 93.393 -79.961 52.820
26119 93.403 -79.971 52.824
26120 93.428 -79.940 52.830
26121 93.445 -79.927 52.834
26122 93.459 -79.921 52.840
26123 93.472 -79.914 52.845
26124 93.485 -79.908 52.849
26125 93.498 -79.902 52.851
26126 93.512 -79.896 52.851
26127 93.525 -79.891 52.849
26128 93.538 -79.885 52.844
26129 93.552 -79.879 52.841
26130 93.565 -79.873 52.838
26131 93.578 -79.867 52.833
26132 93.596 -79.864 52.827
26133 93.620 -79.864 52.821
26134 93.644 -79.864 52.814
26135 93.664 -79.862 52.807
26136 93.674 -79.853 52.800
26137 93.684 -79.845 52.793
26138 93.694 -79.836 52.787
26139 93.704 -79.827 52.778
26140 93.713 -79.823 52.770
26141 93.722 -79.821 52.762
26142 93.730 -79.820 52.755
26143 93.739 -79.818 52.746
26144 93.748 -79.817 52.738
26145 93.756 -79.815 52.730
26146 93.765 -79.814 52.721
26147 93.773 -79.812 52.712
26148 93.782 -79.811 52.704
26166 93.936 -79.785 52.546
26167 93.944 -79.783 52.535
26168 93.953 -79.782 52.523
26169 93.962 -79.780 52.511
26170 93.970 -79.779 52.499
26171 93.979 -79.777 52.486
26172 93.987 -79.776 52.472
26173 93.996 -79.774 52.458
26174 94.004 -79.773 52.445
26175 94.013 -79.771 52.430
26176 94.022 -79.770 52.414
26177 94.030 -79.768 52.398
26178 94.039 -79.767 52.383
26179 94.047 -79.765 52.366
26180 94.056 -79.764 52.349
26181 94.064 -79.763 52.334
26182 94.073 -79.761 52.316
26183 94.081 -79.760 52.298
26184 94.090 -79.758 52.281
26185 94.099 -79.757 52.264
26186 94.107 -79.755 52.248
26187 94.116 -79.754 52.231
26188 94.124 -79.752 52.214
26189 94.133 -79.751 52.196
26190 94.141 -79.749 52.178
26191 94.150 -79.748 52.160
26192 94.158 -79.731 52.142
26193 94.167 -79.705 52.125
26194 94.176 -79.679 52.108
26195 94.184 -79.654 52.091
26196 94.193 -79.628 52.071
26197 94.201 -79.601 52.051
26198 94.210 -79.576 52.032
26199 94.218 -79.550 52.012
26200 94.227 -79.524 51.992
26201 94.236 -79.498 51.973
26202 94.244 -79.473 51.953
26203 94.252 -79.447 51.932
26204 94.261 -79.421 51.910
26205 94.269 -79.395 51.889
26206 94.278 -79.369 51.868
26207 94.273 -79.366 51.848
26208 94.262 -79.372 51.829
26209 94.251 -79.378 51.807
26210 94.240 -79.383 51.783
26211 94.237 -79.381 51.754
26212 94.250 -79.361 51.727
26213 94.262 -79.341 51.702
26214 94.275 -79.322 51.678
26215 94.288 -79.302 51.653
26216 94.300 -79.282 51.624
26217 94.314 -79.263 51.593
26218 94.326 -79.243 51.556
26219 94.339 -79.223 51.521
26220 94.355 -79.248 51.487
26221 94.364 -79.291 51.451
26222 94.373 -79.334 51.415
26223 94.382 -79.377 51.381
26224 94.391 -79.420 51.348
26225 94.400 -79.463 51.312
26226 94.409 -79.507 51.276
26227 94.417 -79.551 51.241
26228 94.426 -79.594 51.206
26229 94.435 -79.637 51.171
26230 94.444 -79.680 51.137
26766 92.521 -80.429 53.465
26767 92.537 -80.422 53.446
26768 92.553 -80.413 53.426
26769 92.569 -80.405 53.407
26770 92.585 -80.397 53.387
26771 92.601 -80.389 53.367
26772 92.617 -80.381 53.347
26773 92.633 -80.373 53.325
26774 92.649 -80.365 53.305
26775 92.665 -80.357 53.284
26776 92.681 -80.349 53.265
26777 92.697 -80.341 53.245
26778 92.713 -80.333 53.227
26779 92.729 -80.325 53.210
26780 92.745 -80.317 53.195
26781 92.761 -80.309 53.180
26782 92.777 -80.301 53.166
26783 92.793 -80.293 53.153
26784 92.811 -80.284 53.139
26785 92.832 -80.276 53.124
26786 92.853 -80.268 53.107
26787 92.873 -80.259 53.088
26788 92.895 -80.251 53.068
26789 92.915 -80.243 53.047
26790 92.936 -80.234 53.027
26791 92.958 -80.214 53.007
26792 92.981 -80.187 52.987
26793 92.998 -80.180 52.968
26794 93.010 -80.185 52.952
26795 93.023 -80.190 52.938
26796 93.035 -80.193 52.926
26797 93.042 -80.190 52.915
26798 93.056 -80.177 52.905
26799 93.070 -80.164 52.896
26800 93.089 -80.153 52.888
26801 93.112 -80.143 52.880
26802 93.134 -80.133 52.873
26803 93.156 -80.123 52.867
26804 93.179 -80.113 52.862
26805 93.201 -80.103 52.859
26806 93.225 -80.096 52.854
26807 93.233 -80.083 52.849
26808 93.231 -80.062 52.844
26809 93.244 -80.047 52.839
26810 93.253 -80.028 52.833
26811 93.262 -80.009 52.827
26812 93.271 -79.990 52.821
26813 93.281 -79.971 52.817
26814 93.295 -79.962 52.814
26815 93.310 -79.953 52.811
26816 93.325 -79.944 52.810
26817 93.341 -79.936 52.811
26818 93.387 -79.946 52.813
26819 93.398 -79.955 52.816
26820 93.410 -79.961 52.821
26821 93.435 -79.930 52.826
26822 93.454 -79.913 52.830
26823 93.468 -79.907 52.837
26824 93.481 -79.901 52.842
26825 93.494 -79.895 52.846
26826 93.508 -79.889 52.849
26827 93.521 -79.883 52.849
26828 93.534 -79.877 52.848
26829 93.548 -79.871 52.844
26830 93.561 -79.864 52.841
26831 93.574 -79.858 52.838
26832 93.587 -79.853 52.832
26833 93.601 -79.847 52.825
26834 93.614 -79.841 52.818
26835 93.627 -79.835 52.812
26836 93.649 -79.834 52.805
26837 93.673 -79.834 52.798
26838 93.684 -79.827 52.791
26839 93.694 -79.817 52.784
26840 93.704 -79.810 52.776
26841 93.713 -79.808 52.768
26842 93.721 -79.807 52.761
26843 93.730 -79.805 52.753
26844 93.739 -79.804 52.745
26845 93.747 -79.802 52.737
26846 93.756 -79.801 52.729
26847 93.764 -79.799 52.720
26848 93.773 -79.798 52.712
26849 93.781 -79.796 52.704
26867 93.935 -79.770 52.549
26868 93.943 -79.769 52.538
26869 93.952 -79.767 52.526
26870 93.961 -79.766 52.514
26871 93.969 -79.764 52.501
26872 93.978 -79.763 52.488
26873 93.986 -79.761 52.475
26874 93.995 -79.760 52.461
26875 94.003 -79.758 52.447
26876 94.012 -79.757 52.432
26877 94.020 -79.755 52.416
26878 94.029 -79.754 52.400
26879 94.038 -79.752 52.385
26880 94.046 -79.751 52.369
26881 94.055 -79.750 52.352
26882 94.063 -79.748 52.336
26883 94.072 -79.747 52.318
26884 94.080 -79.745 52.300
26885 94.089 -79.744 52.284
26886 94.098 -79.742 52.267
26887 94.106 -79.741 52.250
26888 94.115 -79.739 52.233
26889 94.123 -79.738 52.216
26890 94.132 -79.736 52.198
26891 94.140 -79.735 52.180
26892 94.149 -79.733 52.162
26893 94.157 -79.717 52.144
26894 94.166 -79.692 52.127
26895 94.175 -79.666 52.111
26896 94.183 -79.640 52.093
26897 94.192 -79.614 52.074
26898 94.200 -79.588 52.054
26899 94.209 -79.563 52.034
26900 94.217 -79.537 52.014
26901 94.226 -79.510 51.995
26902 94.235 -79.485 51.976
26903 94.243 -79.459 51.956
26904 94.252 -79.433 51.936
26905 94.260 -79.407 51.914
26906 94.269 -79.382 51.895
26907 94.277 -79.356 51.876
26908 94.276 -79.346 51.858
26909 94.265 -79.352 51.839
26910 94.265 -79.353 51.817
26911 94.278 -79.352 51.792
26912 94.290 -79.350 51.764
26913 94.302 -79.348 51.737
26914 94.313 -79.338 51.712
26915 94.319 -79.316 51.688
26916 94.326 -79.296 51.662
26917 94.332 -79.275 51.634
26918 94.337 -79.254 51.604
26919 94.339 -79.233 51.569
26920 94.340 -79.212 51.534
26921 94.345 -79.225 51.498
26922 94.354 -79.269 51.460
26923 94.362 -79.312 51.424
26924 94.371 -79.355 51.389
26925 94.380 -79.398 51.355
26926 94.389 -79.441 51.319
26927 94.398 -79.485 51.284
26928 94.407 -79.528 51.249
26929 94.415 -79.571 51.215
26930 94.424 -79.614 51.179
26931 94.433 -79.657 51.145
27468 92.533 -80.404 53.469
27469 92.549 -80.397 53.449
27470 92.565 -80.388 53.429
27471 92.581 -80.380 53.410
27472 92.597 -80.372 53.389
27473 92.613 -80.364 53.368
27474 92.629 -80.356 53.347
27475 92.645 -80.348 53.325
27476 92.661 -80.340 53.304
27477 92.677 -80.332 53.283
27478 92.693 -80.324 53.262
27479 92.709 -80.316 53.243
27480 92.725 -80.308 53.223
27481 92.741 -80.300 53.204
27482 92.757 -80.292 53.187
27483 92.773 -80.284 53.172
27484 92.789 -80.276 53.159
27485 92.805 -80.268 53.145
27486 92.821 -80.259 53.131
27487 92.838 -80.252 53.115
27488 92.857 -80.243 53.096
27489 92.878 -80.235 53.077
27490 92.899 -80.227 53.056
27491 92.919 -80.218 53.035
27492 92.940 -80.210 53.015
27493 92.961 -80.197 52.995
27494 92.985 -80.170 52.975
27495 93.001 -80.162 52.957
27496 93.014 -80.167 52.943
27497 93.027 -80.172 52.932
27498 93.035 -80.171 52.920
27499 93.048 -80.159 52.910
27500 93.062 -80.146 52.901
27501 93.075 -80.134 52.893
27502 93.089 -80.121 52.884
27503 93.111 -80.111 52.876
27504 93.134 -80.100 52.869
27505 93.156 -80.090 52.863
27506 93.178 -80.080 52.859
27507 93.202 -80.073 52.855
27508 93.219 -80.066 52.850
27509 93.222 -80.047 52.845
27510 93.236 -80.035 52.838
27511 93.246 -80.015 52.831
27512 93.255 -79.996 52.825
27513 93.264 -79.977 52.819
27514 93.278 -79.968 52.813
27515 93.292 -79.958 52.808
27516 93.307 -79.949 52.805
27517 93.322 -79.940 52.805
27518 93.337 -79.931 52.807
27519 93.386 -79.940 52.809
27520 93.404 -79.950 52.814
27521 93.417 -79.952 52.819
27522 93.442 -79.921 52.823
27523 93.464 -79.899 52.828
27524 93.477 -79.892 52.834
27525 93.490 -79.886 52.839
27526 93.503 -79.880 52.844
27527 93.517 -79.875 52.846
27528 93.530 -79.869 52.846
27529 93.544 -79.863 52.846
27530 93.557 -79.857 52.844
27531 93.570 -79.851 52.842
27532 93.583 -79.845 52.839
27533 93.596 -79.839 52.834
27534 93.610 -79.833 52.826
27535 93.623 -79.827 52.818
27536 93.636 -79.820 52.811
27537 93.650 -79.815 52.805
27538 93.663 -79.809 52.798
27539 93.678 -79.804 52.790
27540 93.694 -79.799 52.783
27541 93.703 -79.795 52.776
27542 93.712 -79.794 52.768
27543 93.720 -79.792 52.759
27544 93.729 -79.791 52.752
27545 93.737 -79.789 52.744
27546 93.746 -79.788 52.736
27547 93.755 -79.786 52.728
27548 93.763 -79.785 52.720
27549 93.772 -79.783 52.712
27550 93.780 -79.782 52.705
27551 93.789 -79.780 52.697
27568 93.934 -79.756 52.550
27569 93.943 -79.755 52.539
27570 93.952 -79.753 52.527
27571 93.960 -79.751 52.514
27572 93.969 -79.750 52.501
27573 93.977 -79.748 52.489
27574 93.986 -79.747 52.475
27575 93.994 -79.745 52.461
27576 94.003 -79.744 52.447
27577 94.011 -79.742 52.432
27578 94.020 -79.741 52.417
27579 94.029 -79.739 52.401
27580 94.037 -79.739 52.385
27581 94.046 -79.737 52.368
27582 94.054 -79.736 52.352
27583 94.063 -79.734 52.336
27584 94.071 -79.733 52.318
27585 94.080 -79.731 52.301
27586 94.088 -79.730 52.285
27587 94.097 -79.728 52.267
27588 94.106 -79.726 52.250
27589 94.114 -79.725 52.234
27590 94.123 -79.723 52.217
27591 94.131 -79.722 52.199
27592 94.140 -79.720 52.181
27593 94.148 -79.720 52.163
27594 94.157 -79.704 52.146
27595 94.165 -79.678 52.129
27596 94.174 -79.652 52.112
27597 94.182 -79.626 52.094
27598 94.191 -79.601 52.075
27599 94.199 -79.575 52.055
27600 94.208 -79.549 52.034
27601 94.216 -79.523 52.016
27602 94.225 -79.498 51.997
27603 94.234 -79.472 51.977
27604 94.242 -79.446 51.959
27605 94.251 -79.419 51.939
27606 94.259 -79.394 51.918
27607 94.268 -79.368 51.900
27608 94.276 -79.342 51.882
27609 94.286 -79.325 51.865
27610 94.293 -79.325 51.847
27611 94.301 -79.326 51.825
27612 94.309 -79.326 51.799
27613 94.317 -79.326 51.771
27614 94.325 -79.326 51.743
27615 94.329 -79.330 51.718
27616 94.329 -79.316 51.693
27617 94.331 -79.294 51.667
27618 94.332 -79.273 51.640
27619 94.334 -79.253 51.611
27620 94.335 -79.231 51.577
27621 94.337 -79.210 51.542
27622 94.334 -79.209 51.505
27623 94.343 -79.246 51.467
27624 94.352 -79.289 51.431
27625 94.360 -79.333 51.396
27626 94.369 -79.376 51.362
27627 94.378 -79.419 51.327
27628 94.387 -79.463 51.291
27629 94.396 -79.506 51.256
27630 94.405 -79.549 51.222
27631 94.413 -79.592 51.186
27632 94.422 -79.636 51.151
28169 92.528 -80.388 53.493
28170 92.545 -80.379 53.473
28171 92.561 -80.372 53.453
28172 92.577 -80.363 53.433
28173 92.593 -80.355 53.412
28174 92.609 -80.347 53.390
28175 92.625 -80.339 53.369
28176 92.641 -80.331 53.347
28177 92.657 -80.323 53.325
28178 92.673 -80.315 53.304
28179 92.689 -80.307 53.283
28180 92.705 -80.299 53.261
28181 92.721 -80.291 53.240
28182 92.737 -80.283 53.220
28183 92.753 -80.275 53.200
28184 92.769 -80.267 53.182
28185 92.785 -80.259 53.165
28186 92.801 -80.251 53.149
28187 92.817 -80.243 53.135
28188 92.833 -80.234 53.120
28189 92.849 -80.227 53.104
28190 92.865 -80.218 53.085
28191 92.882 -80.211 53.065
28192 92.903 -80.203 53.044
28193 92.924 -80.194 53.022
28194 92.944 -80.186 53.000
28195 92.965 -80.178 52.981
28196 92.988 -80.153 52.964
28197 93.005 -80.144 52.949
28198 93.018 -80.149 52.937
28199 93.028 -80.151 52.926
28200 93.040 -80.141 52.915
28201 93.054 -80.128 52.907
28202 93.067 -80.115 52.898
28203 93.081 -80.102 52.888
28204 93.094 -80.090 52.879
28205 93.111 -80.077 52.872
28206 93.133 -80.068 52.865
28207 93.155 -80.058 52.859
28208 93.179 -80.049 52.855
28209 93.206 -80.049 52.850
28210 93.213 -80.034 52.845
28211 93.227 -80.021 52.838
28212 93.237 -80.003 52.831
28213 93.247 -79.984 52.823
28214 93.260 -79.973 52.816
28215 93.275 -79.964 52.811
28216 93.289 -79.955 52.807
28217 93.304 -79.946 52.804
28218 93.319 -79.936 52.805
28219 93.334 -79.927 52.806
28220 93.380 -79.935 52.808
28221 93.409 -79.945 52.813
28222 93.424 -79.943 52.818
28223 93.449 -79.912 52.823
28224 93.473 -79.885 52.827
28225 93.486 -79.879 52.832
28226 93.499 -79.873 52.836
28227 93.513 -79.867 52.841
28228 93.526 -79.861 52.842
28229 93.540 -79.855 52.843
28230 93.553 -79.849 52.843
28231 93.566 -79.842 52.844
28232 93.579 -79.837 52.843
28233 93.592 -79.831 52.840
28234 93.606 -79.825 52.835
28235 93.619 -79.819 52.827
28236 93.632 -79.813 52.819
28237 93.646 -79.807 52.810
28238 93.659 -79.801 52.804
28239 93.672 -79.795 52.797
28240 93.684 -79.787 52.790
28241 93.694 -79.777 52.784
28242 93.708 -79.770 52.776
28243 93.715 -79.772 52.767
28244 93.721 -79.775 52.758
28245 93.728 -79.776 52.750
28246 93.736 -79.775 52.742
28247 93.745 -79.773 52.734
28248 93.754 -79.772 52.726
28249 93.762 -79.770 52.718
28250 93.771 -79.769 52.710
28251 93.779 -79.767 52.702
28252 93.788 -79.766 52.695
28269 93.933 -79.742 52.550
28270 93.942 -79.740 52.539
28271 93.951 -79.739 52.526
28272 93.959 -79.737 52.513
28273 93.968 -79.736 52.500
28274 93.976 -79.734 52.489
28275 93.985 -79.733 52.475
28276 93.993 -79.731 52.461
28277 94.002 -79.730 52.446
28278 94.010 -79.728 52.432
28279 94.019 -79.726 52.416
28280 94.028 -79.726 52.401
28281 94.036 -79.724 52.385
28282 94.045 -79.723 52.368
28283 94.053 -79.721 52.351
28284 94.062 -79.720 52.336
28285 94.070 -79.718 52.318
28286 94.079 -79.717 52.301
28287 94.087 -79.715 52.284
28288 94.096 -79.714 52.267
28289 94.105 -79.712 52.250
28290 94.113 -79.711 52.233
28291 94.122 -79.709 52.217
28292 94.130 -79.708 52.199
28293 94.139 -79.707 52.181
28294 94.147 -79.705 52.163
28295 94.156 -79.691 52.145
28296 94.165 -79.665 52.128
28297 94.173 -79.639 52.111
28298 94.182 -79.614 52.094
28299 94.190 -79.588 52.074
28300 94.199 -79.561 52.054
28301 94.207 -79.535 52.034
28302 94.215 -79.510 52.016
28303 94.224 -79.484 51.997
28304 94.232 -79.458 51.978
28305 94.241 -79.432 51.959
28306 94.250 -79.407 51.940
28307 94.258 -79.381 51.921
28308 94.266 -79.354 51.903
28309 94.274 -79.328 51.886
28310 94.288 -79.313 51.869
28311 94.296 -79.313 51.850
28312 94.305 -79.313 51.827
28313 94.312 -79.313 51.802
28314 94.320 -79.313 51.774
28315 94.328 -79.313 51.745
28316 94.326 -79.322 51.718
28317 94.326 -79.313 51.693
28318 94.327 -79.293 51.667
28319 94.329 -79.272 51.641
28320 94.331 -79.250 51.613
28321 94.332 -79.229 51.581
28322 94.331 -79.214 51.546
28323 94.323 -79.217 51.509
28324 94.332 -79.224 51.472
28325 94.341 -79.267 51.437
28326 94.350 -79.310 51.403
28327 94.358 -79.353 51.370
28328 94.367 -79.397 51.334
28329 94.376 -79.440 51.298
28330 94.385 -79.483 51.262
28331 94.394 -79.526 51.228
28332 94.403 -79.570 51.193
28333 94.411 -79.613 51.157
28871 92.541 -80.363 53.498
28872 92.557 -80.354 53.478
28873 92.573 -80.347 53.457
28874 92.589 -80.338 53.435
28875 92.605 -80.330 53.413
28876 92.621 -80.322 53.392
28877 92.637 -80.314 53.370
28878 92.653 -80.306 53.347
28879 92.669 -80.298 53.326
28880 92.685 -80.290 53.304
28881 92.701 -80.282 53.282
28882 92.717 -80.274 53.259
28883 92.733 -80.266 53.238
28884 92.749 -80.258 53.217
28885 92.765 -80.250 53.197
28886 92.781 -80.242 53.178
28887 92.797 -80.234 53.159
28888 92.813 -80.226 53.142
28889 92.829 -80.218 53.126
28890 92.845 -80.209 53.109
28891 92.861 -80.202 53.091
28892 92.877 -80.193 53.071
28893 92.894 -80.186 53.050
28894 92.909 -80.178 53.029
28895 92.928 -80.169 53.007
28896 92.948 -80.161 52.988
28897 92.970 -80.153 52.972
28898 92.991 -80.136 52.956
28899 93.009 -80.126 52.943
28900 93.021 -80.131 52.932
28901 93.031 -80.124 52.922
28902 93.046 -80.111 52.914
28903 93.059 -80.098 52.902
28904 93.073 -80.085 52.891
28905 93.086 -80.072 52.883
28906 93.100 -80.059 52.874
28907 93.114 -80.046 52.867
28908 93.133 -80.035 52.860
28909 93.155 -80.025 52.855
28910 93.183 -80.026 52.850
28911 93.204 -80.019 52.844
28912 93.219 -80.007 52.838
28913 93.230 -79.990 52.830
28914 93.242 -79.977 52.822
28915 93.257 -79.968 52.816
28916 93.272 -79.959 52.811
28917 93.286 -79.950 52.807
28918 93.301 -79.941 52.805
28919 93.316 -79.932 52.805
28920 93.331 -79.923 52.806
28921 93.374 -79.930 52.808
28922 93.415 -79.939 52.814
28923 93.431 -79.933 52.818
28924 93.457 -79.902 52.822
28925 93.482 -79.871 52.825
28926 93.495 -79.864 52.828
28927 93.509 -79.859 52.831
28928 93.522 -79.853 52.835
28929 93.535 -79.847 52.836
28930 93.549 -79.841 52.838
28931 93.562 -79.835 52.839
28932 93.575 -79.829 52.841
28933 93.588 -79.823 52.841
28934 93.602 -79.817 52.838
28935 93.615 -79.811 52.835
28936 93.628 -79.805 52.828
28937 93.641 -79.799 52.820
28938 93.653 -79.791 52.811
28939 93.663 -79.781 52.804
28940 93.673 -79.771 52.797
28941 93.683 -79.761 52.790
28942 93.693 -79.751 52.785
28943 93.706 -79.743 52.776
28944 93.721 -79.737 52.765
28945 93.734 -79.733 52.756
28946 93.741 -79.735 52.746
28947 93.748 -79.737 52.738
28948 93.754 -79.740 52.730
28949 93.761 -79.742 52.723
28950 93.767 -79.745 52.715
28951 93.774 -79.747 52.707
28952 93.780 -79.750 52.700
28953 93.787 -79.752 52.693
28954 93.795 -79.751 52.686
28956 93.813 -79.748 52.671
28957 93.821 -79.746 52.663
28958 93.830 -79.745 52.655
28959 93.838 -79.743 52.648
28960 93.847 -79.742 52.640
28961 93.855 -79.740 52.632
28962 93.864 -79.739 52.623
28963 93.872 -79.737 52.615
28964 93.881 -79.736 52.607
28965 93.890 -79.734 52.597
28970 93.932 -79.727 52.548
28971 93.941 -79.726 52.536
28972 93.949 -79.724 52.524
28973 93.958 -79.723 52.511
28974 93.967 -79.721 52.499
28975 93.975 -79.720 52.486
28976 93.984 -79.718 52.472
28977 93.992 -79.717 52.459
28978 94.001 -79.715 52.445
28979 94.009 -79.714 52.430
28980 94.018 -79.713 52.415
28981 94.027 -79.711 52.399
28982 94.035 -79.710 52.384
28983 94.044 -79.708 52.367
28984 94.052 -79.707 52.350
28985 94.061 -79.705 52.334
28986 94.069 -79.704 52.317
28987 94.078 -79.702 52.299
28988 94.086 -79.701 52.283
28989 94.095 -79.699 52.266
28990 94.104 -79.698 52.249
28991 94.112 -79.696 52.232
28992 94.121 -79.695 52.216
28993 94.129 -79.694 52.198
28994 94.138 -79.692 52.180
28995 94.146 -79.691 52.162
28996 94.155 -79.677 52.144
28997 94.164 -79.651 52.127
28998 94.172 -79.626 52.110
28999 94.181 -79.600 52.092
29000 94.189 -79.574 52.073
29001 94.198 -79.548 52.053
29002 94.206 -79.523 52.033
29003 94.215 -79.497 52.014
29004 94.223 -79.470 51.996
29005 94.232 -79.444 51.977
29006 94.241 -79.419 51.959
29007 94.249 -79.393 51.940
29008 94.257 -79.366 51.921
29009 94.264 -79.340 51.903
29010 94.273 -79.314 51.886
29011 94.289 -79.304 51.868
29012 94.300 -79.300 51.848
29013 94.308 -79.300 51.826
29014 94.316 -79.300 51.801
29015 94.323 -79.300 51.773
29016 94.327 -79.303 51.745
29017 94.324 -79.315 51.719
29018 94.323 -79.312 51.694
29019 94.324 -79.291 51.668
29020 94.326 -79.270 51.642
29021 94.327 -79.249 51.614
29022 94.329 -79.228 51.584
29023 94.323 -79.219 51.550
29024 94.311 -79.225 51.514
29025 94.321 -79.219 51.478
29026 94.330 -79.245 51.443
29027 94.339 -79.288 51.409
29028 94.348 -79.332 51.376
29029 94.356 -79.375 51.340
29030 94.365 -79.418 51.304
29031 94.374 -79.461 51.269
29032 94.383 -79.504 51.235
29033 94.392 -79.548 51.199
29034 94.401 -79.591 51.164
29035 94.409 -79.634 51.128
29572 92.538 -80.349 53.524
29573 92.553 -80.337 53.503
29574 92.569 -80.329 53.482
29575 92.585 -80.322 53.460
29576 92.600 -80.313 53.439
29577 92.617 -80.305 53.417
29578 92.633 -80.297 53.394
29579 92.649 -80.289 53.372
29580 92.665 -80.281 53.350
29581 92.681 -80.273 53.327
29582 92.697 -80.265 53.304
29583 92.713 -80.257 53.280
29584 92.729 -80.249 53.258
29585 92.745 -80.241 53.236
29586 92.761 -80.233 53.215
29587 92.777 -80.225 53.194
29588 92.793 -80.217 53.173
29589 92.809 -80.209 53.153
29590 92.825 -80.201 53.134
29591 92.841 -80.193 53.116
29592 92.857 -80.184 53.098
29593 92.873 -80.177 53.078
29594 92.889 -80.168 53.058
29595 92.905 -80.161 53.037
29596 92.921 -80.153 53.016
29597 92.937 -80.144 52.998
29598 92.953 -80.137 52.981
29599 92.974 -80.128 52.965
29600 92.995 -80.119 52.951
29601 93.012 -80.108 52.939
29602 93.023 -80.106 52.928
29603 93.037 -80.093 52.918
29604 93.051 -80.080 52.907
29605 93.065 -80.067 52.896
29606 93.078 -80.054 52.886
29607 93.092 -80.041 52.877
29608 93.106 -80.028 52.869
29609 93.119 -80.015 52.861
29610 93.133 -80.002 52.855
29611 93.172 -80.017 52.849
29612 93.190 -80.005 52.842
29613 93.208 -79.993 52.836
29614 93.224 -79.983 52.829
29615 93.239 -79.974 52.823
29616 93.254 -79.965 52.816
29617 93.269 -79.955 52.811
29618 93.283 -79.946 52.807
29619 93.298 -79.937 52.805
29620 93.313 -79.928 52.805
29621 93.328 -79.919 52.806
29622 93.369 -79.924 52.809
29623 93.418 -79.934 52.815
29624 93.438 -79.924 52.818
29625 93.464 -79.893 52.821
29626 93.489 -79.862 52.822
29627 93.505 -79.851 52.824
29628 93.518 -79.845 52.827
29629 93.531 -79.839 52.831
29630 93.545 -79.833 52.832
29631 93.558 -79.827 52.833
29632 93.571 -79.821 52.834
29633 93.584 -79.815 52.835
29634 93.597 -79.809 52.836
29635 93.611 -79.803 52.834
29636 93.623 -79.795 52.831
29637 93.632 -79.785 52.825
29638 93.642 -79.775 52.819
29639 93.652 -79.765 52.810
29640 93.662 -79.755 52.803
29641 93.672 -79.745 52.797
29642 93.682 -79.735 52.791
29643 93.692 -79.725 52.784
29644 93.704 -79.717 52.774
29645 93.719 -79.710 52.762
29646 93.734 -79.704 52.752
29647 93.750 -79.698 52.742
29648 93.761 -79.695 52.734
29649 93.767 -79.698 52.725
29650 93.774 -79.700 52.718
29651 93.780 -79.702 52.710
29652 93.787 -79.705 52.703
29653 93.793 -79.708 52.697
29654 93.800 -79.710 52.690
29655 93.806 -79.713 52.683
29656 93.813 -79.715 52.675
29657 93.820 -79.717 52.668
29658 93.826 -79.720 52.661
29659 93.833 -79.723 52.653
29660 93.840 -79.725 52.645
29661 93.846 -79.727 52.638
29662 93.855 -79.726 52.629
29663 93.863 -79.724 52.621
29664 93.872 -79.723 52.613
29665 93.880 -79.721 52.604
29666 93.889 -79.720 52.595
29667 93.898 -79.719 52.586
29668 93.906 -79.717 52.576
29669 93.915 -79.716 52.566
29670 93.923 -79.714 52.556
29671 93.932 -79.713 52.545
29672 93.940 -79.711 52.533
29673 93.949 -79.710 52.521
29674 93.958 -79.708 52.508
29675 93.966 -79.707 52.495
29676 93.975 -79.705 52.482
29677 93.983 -79.704 52.469
29678 93.992 -79.702 52.455
29679 94.000 -79.701 52.442
29680 94.009 -79.700 52.427
29681 94.017 -79.698 52.411
29682 94.026 -79.697 52.395
29683 94.035 -79.695 52.379
29684 94.043 -79.694 52.363
29685 94.051 -79.692 52.347
29686 94.060 -79.691 52.332
29687 94.068 -79.689 52.315
29688 94.077 -79.688 52.297
29689 94.085 -79.686 52.280
29690 94.094 -79.685 52.264
29691 94.103 -79.683 52.247
29692 94.111 -79.682 52.230
29693 94.120 -79.681 52.214
29694 94.128 -79.679 52.197
29695 94.137 -79.678 52.179
29696 94.145 -79.676 52.160
29697 94.154 -79.664 52.142
29698 94.163 -79.638 52.124
29699 94.171 -79.612 52.107
29700 94.180 -79.586 52.091
29701 94.188 -79.560 52.072
29702 94.197 -79.535 52.051
29703 94.205 -79.509 52.032
29704 94.214 -79.483 52.013
29705 94.222 -79.457 51.994
29706 94.231 -79.431 51.975
29707 94.239 -79.405 51.957
29708 94.247 -79.379 51.939
29709 94.254 -79.352 51.920
29710 94.262 -79.325 51.903
29711 94.273 -79.305 51.885
29712 94.289 -79.294 51.866
29713 94.303 -79.286 51.847
29714 94.311 -79.287 51.825
29715 94.319 -79.287 51.801
29716 94.327 -79.287 51.773
29717 94.325 -79.296 51.746
29718 94.321 -79.307 51.721
29719 94.319 -79.310 51.697
29720 94.321 -79.289 51.671
29721 94.322 -79.268 51.645
29722 94.324 -79.247 51.617
29723 94.325 -79.226 51.587
29724 94.316 -79.225 51.554
29725 94.302 -79.231 51.519
29726 94.310 -79.226 51.483
29727 94.319 -79.222 51.449
29728 94.328 -79.266 51.415
29729 94.337 -79.309 51.382
29730 94.346 -79.352 51.347
29731 94.354 -79.395 51.311
29732 94.363 -79.438 51.276
29733 94.372 -79.482 51.241
29734 94.381 -79.525 51.206
29735 94.390 -79.569 51.170
29736 94.398 -79.612 51.135
29737 94.407 -79.655 51.098
29738 94.416 -79.698 51.061
30274 92.552 -80.327 53.528
30275 92.565 -80.315 53.507
30276 92.580 -80.304 53.485
30277 92.596 -80.297 53.464
30278 92.613 -80.288 53.442
30279 92.629 -80.280 53.419
30280 92.645 -80.272 53.396
30281 92.661 -80.264 53.374
30282 92.676 -80.256 53.351
30283 92.693 -80.248 53.327
30284 92.709 -80.240 53.304
30285 92.725 -80.232 53.281
30286 92.741 -80.224 53.258
30287 92.757 -80.216 53.235
30288 92.773 -80.208 53.213
30289 92.789 -80.200 53.191
30290 92.805 -80.192 53.169
30291 92.821 -80.184 53.148
30292 92.837 -80.176 53.127
30293 92.853 -80.168 53.106
30294 92.869 -80.159 53.086
30295 92.885 -80.152 53.066
30296 92.901 -80.143 53.047
30297 92.917 -80.136 53.027
30298 92.933 -80.127 53.009
30299 92.949 -80.119 52.991
30300 92.965 -80.112 52.975
30301 92.981 -80.103 52.960
30302 92.999 -80.095 52.946
30303 93.015 -80.088 52.935
30304 93.029 -80.075 52.923
30305 93.043 -80.062 52.911
30306 93.059 -80.050 52.900
30307 93.083 -80.040 52.890
30308 93.108 -80.030 52.880
30309 93.132 -80.020 52.871
30310 93.157 -80.009 52.863
30311 93.182 -79.999 52.855
30312 93.200 -79.990 52.848
30313 93.211 -79.983 52.841
30314 93.222 -79.975 52.835
30315 93.233 -79.968 52.829
30316 93.246 -79.960 52.822
30317 93.259 -79.953 52.816
30318 93.272 -79.946 52.811
30319 93.285 -79.938 52.808
30320 93.297 -79.931 52.807
30321 93.310 -79.924 52.808
30322 93.325 -79.914 52.810
30323 93.363 -79.918 52.812
30324 93.413 -79.929 52.814
30325 93.445 -79.914 52.816
30326 93.471 -79.883 52.818
30327 93.496 -79.853 52.820
30328 93.514 -79.837 52.821
30329 93.527 -79.831 52.823
30330 93.541 -79.825 52.827
30331 93.554 -79.819 52.829
30332 93.567 -79.813 52.830
30333 93.580 -79.807 52.829
30334 93.592 -79.799 52.829
30335 93.602 -79.789 52.829
30336 93.612 -79.779 52.829
30337 93.622 -79.769 52.825
30338 93.632 -79.759 52.820
30339 93.641 -79.749 52.815
30340 93.651 -79.739 52.807
30341 93.661 -79.729 52.800
30342 93.671 -79.719 52.794
30343 93.681 -79.709 52.788
30344 93.691 -79.699 52.779
30345 93.702 -79.690 52.769
30346 93.717 -79.683 52.758
30347 93.732 -79.677 52.746
30348 93.748 -79.670 52.736
30349 93.763 -79.664 52.727
30350 93.778 -79.657 52.720
30351 93.787 -79.657 52.713
30352 93.793 -79.661 52.705
30353 93.800 -79.663 52.698
30354 93.806 -79.665 52.692
30355 93.813 -79.668 52.685
30356 93.820 -79.670 52.678
30357 93.826 -79.673 52.671
30358 93.833 -79.675 52.664
30359 93.840 -79.678 52.657
30360 93.846 -79.680 52.650
30361 93.853 -79.683 52.641
30362 93.859 -79.686 52.634
30363 93.866 -79.688 52.626
30364 93.872 -79.690 52.618
30365 93.879 -79.692 52.610
30366 93.886 -79.695 52.601
30367 93.892 -79.698 52.591
30368 93.899 -79.700 52.582
30369 93.905 -79.703 52.573
30370 93.914 -79.701 52.562
30371 93.922 -79.700 52.552
30372 93.931 -79.698 52.541
30373 93.939 -79.697 52.530
30374 93.948 -79.695 52.517
30375 93.957 -79.694 52.505
30376 93.965 -79.692 52.491
30377 93.974 -79.691 52.478
30378 93.982 -79.689 52.465
30379 93.991 -79.688 52.452
30380 93.999 -79.687 52.437
30381 94.008 -79.686 52.423
30382 94.016 -79.684 52.408
30383 94.025 -79.683 52.392
30384 94.034 -79.681 52.376
30385 94.042 -79.679 52.360
30386 94.051 -79.678 52.344
30387 94.059 -79.676 52.329
30388 94.068 -79.675 52.312
30389 94.076 -79.673 52.295
30390 94.085 -79.672 52.279
30391 94.094 -79.670 52.262
30392 94.102 -79.669 52.245
30393 94.111 -79.668 52.229
30394 94.119 -79.667 52.213
30395 94.128 -79.665 52.195
30396 94.136 -79.664 52.177
30397 94.145 -79.662 52.159
30398 94.153 -79.651 52.141
30399 94.162 -79.625 52.123
30400 94.171 -79.599 52.106
30401 94.179 -79.573 52.088
30402 94.187 -79.547 52.069
30403 94.196 -79.521 52.049
30404 94.204 -79.495 52.030
30405 94.213 -79.469 52.011
30406 94.221 -79.443 51.992
30407 94.228 -79.416 51.974
30408 94.237 -79.390 51.957
30409 94.245 -79.363 51.939
30410 94.252 -79.338 51.920
30411 94.260 -79.311 51.902
30412 94.274 -79.295 51.884
30413 94.289 -79.285 51.866
30414 94.306 -79.275 51.848
30415 94.314 -79.273 51.827
30416 94.322 -79.273 51.803
30417 94.326 -79.278 51.775
30418 94.322 -79.288 51.749
30419 94.319 -79.300 51.724
30420 94.316 -79.308 51.700
30421 94.317 -79.288 51.675
30422 94.319 -79.266 51.649
30423 94.320 -79.245 51.621
30424 94.321 -79.225 51.592
30425 94.308 -79.231 51.559
30426 94.294 -79.237 51.525
30427 94.298 -79.234 51.489
30428 94.308 -79.228 51.456
30429 94.317 -79.244 51.422
30430 94.326 -79.287 51.388
30431 94.335 -79.330 51.354
30432 94.343 -79.373 51.319
30433 94.352 -79.416 51.284
30434 94.361 -79.460 51.249
30435 94.370 -79.503 51.213
30436 94.379 -79.546 51.177
30437 94.388 -79.589 51.142
30438 94.396 -79.632 51.105
30439 94.405 -79.676 51.068
30440 94.414 -79.719 51.030
30441 94.423 -79.762 50.993
30977 92.578 -80.292 53.510
30978 92.592 -80.280 53.489
30979 92.609 -80.272 53.467
30980 92.625 -80.263 53.444
30981 92.641 -80.255 53.421
30982 92.656 -80.247 53.398
30983 92.672 -80.239 53.375
30984 92.689 -80.231 53.351
30985 92.705 -80.223 53.328
30986 92.721 -80.215 53.304
30987 92.737 -80.207 53.281
30988 92.753 -80.199 53.257
30989 92.769 -80.191 53.234
30990 92.785 -80.183 53.211
30991 92.801 -80.174 53.188
30992 92.817 -80.167 53.166
30993 92.833 -80.159 53.144
30994 92.849 -80.151 53.122
30995 92.865 -80.143 53.101
30996 92.881 -80.134 53.081
30997 92.897 -80.127 53.061
30998 92.913 -80.118 53.040
30999 92.929 -80.111 53.021
31000 92.945 -80.102 53.003
31001 92.961 -80.094 52.986
31002 92.977 -80.087 52.970
31003 92.993 -80.078 52.955
31004 93.009 -80.071 52.942
31005 93.023 -80.062 52.928
31006 93.044 -80.052 52.916
31007 93.064 -80.041 52.904
31008 93.085 -80.030 52.893
31009 93.106 -80.020 52.883
31010 93.126 -80.009 52.874
31011 93.147 -79.999 52.864
31012 93.167 -79.988 52.855
31013 93.188 -79.977 52.847
31014 93.213 -79.968 52.841
31015 93.231 -79.960 52.834
31016 93.243 -79.952 52.828
31017 93.256 -79.945 52.822
31018 93.269 -79.937 52.817
31019 93.281 -79.930 52.813
31020 93.295 -79.922 52.811
31021 93.308 -79.914 52.810
31022 93.322 -79.906 52.811
31023 93.335 -79.898 52.811
31024 93.359 -79.895 52.811
31025 93.406 -79.902 52.813
31026 93.442 -79.889 52.814
31027 93.478 -79.874 52.815
31028 93.503 -79.843 52.817
31029 93.523 -79.823 52.818
31030 93.537 -79.817 52.820
31031 93.550 -79.811 52.822
31032 93.561 -79.803 52.825
31033 93.571 -79.793 52.827
31034 93.581 -79.783 52.827
31035 93.591 -79.773 52.824
31036 93.601 -79.763 52.823
31037 93.611 -79.753 52.821
31038 93.621 -79.743 52.817
31039 93.631 -79.733 52.813
31040 93.640 -79.723 52.809
31041 93.650 -79.713 52.802
31042 93.660 -79.703 52.796
31043 93.670 -79.693 52.790
31044 93.680 -79.683 52.782
31045 93.690 -79.673 52.773
31046 93.700 -79.664 52.763
31047 93.715 -79.657 52.751
31048 93.730 -79.651 52.740
31049 93.746 -79.644 52.730
31050 93.761 -79.638 52.721
31051 93.776 -79.631 52.713
31052 93.791 -79.625 52.707
31053 93.806 -79.618 52.700
31054 93.813 -79.620 52.693
31055 93.820 -79.623 52.687
31056 93.826 -79.626 52.680
31057 93.833 -79.628 52.673
31058 93.840 -79.630 52.667
31059 93.846 -79.633 52.660
31060 93.853 -79.636 52.653
31061 93.859 -79.638 52.646
31062 93.866 -79.640 52.638
31063 93.872 -79.643 52.630
31064 93.879 -79.645 52.623
31065 93.886 -79.648 52.615
31066 93.892 -79.651 52.607
31067 93.899 -79.653 52.598
31068 93.905 -79.655 52.588
31069 93.912 -79.657 52.578
31070 93.918 -79.661 52.569
31071 93.925 -79.663 52.558
31072 93.931 -79.665 52.547
31073 93.938 -79.668 52.537
31074 93.944 -79.670 52.526
31075 93.951 -79.673 52.513
31076 93.958 -79.675 52.501
31077 93.965 -79.678 52.488
31078 93.973 -79.676 52.475
31079 93.981 -79.675 52.462
31080 93.990 -79.674 52.448
31081 93.998 -79.673 52.434
31082 94.007 -79.671 52.419
31083 94.015 -79.670 52.404
31084 94.024 -79.668 52.388
31085 94.033 -79.667 52.373
31086 94.041 -79.665 52.357
31087 94.050 -79.664 52.342
31088 94.058 -79.662 52.327
31089 94.067 -79.661 52.310
31090 94.075 -79.659 52.293
31091 94.084 -79.657 52.277
31092 94.092 -79.656 52.259
31093 94.101 -79.655 52.243
31094 94.110 -79.654 52.227
31095 94.118 -79.652 52.211
31096 94.127 -79.651 52.193
31097 94.135 -79.649 52.175
31098 94.144 -79.648 52.157
31099 94.152 -79.637 52.140
31100 94.161 -79.611 52.122
31101 94.170 -79.585 52.105
31102 94.178 -79.560 52.087
31103 94.187 -79.534 52.068
31104 94.195 -79.508 52.048
31105 94.203 -79.482 52.029
31106 94.211 -79.455 52.010
31107 94.219 -79.429 51.991
31108 94.226 -79.402 51.974
31109 94.235 -79.375 51.958
31110 94.242 -79.349 51.940
31111 94.250 -79.322 51.922
31112 94.258 -79.297 51.904
31113 94.274 -79.286 51.886
31114 94.290 -79.275 51.868
31115 94.306 -79.265 51.850
31116 94.318 -79.260 51.829
31117 94.325 -79.260 51.805
31118 94.323 -79.270 51.779
31119 94.320 -79.281 51.754
31120 94.316 -79.292 51.729
31121 94.313 -79.303 51.706
31122 94.314 -79.285 51.682
31123 94.315 -79.265 51.657
31124 94.317 -79.244 51.630
31125 94.314 -79.231 51.599
31126 94.300 -79.237 51.567
31127 94.287 -79.243 51.533
31128 94.286 -79.242 51.497
31129 94.296 -79.235 51.463
31130 94.306 -79.228 51.430
31131 94.315 -79.264 51.397
31132 94.324 -79.307 51.362
31133 94.333 -79.350 51.328
31134 94.341 -79.394 51.293
31135 94.350 -79.438 51.256
31136 94.359 -79.481 51.220
31137 94.368 -79.524 51.185
31138 94.377 -79.567 51.149
31139 94.386 -79.610 51.112
31140 94.394 -79.654 51.074
31141 94.403 -79.697 51.037
31142 94.412 -79.740 50.999
31143 94.424 -79.760 50.962
31680 92.605 -80.257 53.492
31681 92.621 -80.247 53.469
31682 92.636 -80.238 53.447
31683 92.652 -80.231 53.424
31684 92.668 -80.222 53.401
31685 92.685 -80.214 53.377
31686 92.701 -80.206 53.351
31687 92.717 -80.198 53.327
31688 92.732 -80.190 53.304
31689 92.748 -80.182 53.279
31690 92.765 -80.174 53.255
31691 92.781 -80.166 53.232
31692 92.797 -80.158 53.208
31693 92.813 -80.149 53.185
31694 92.829 -80.142 53.162
31695 92.845 -80.134 53.140
31696 92.861 -80.126 53.118
31697 92.877 -80.118 53.097
31698 92.893 -80.109 53.076
31699 92.909 -80.102 53.055
31700 92.925 -80.093 53.035
31701 92.941 -80.086 53.017
31702 92.957 -80.077 52.998
31703 92.973 -80.069 52.981
31704 92.989 -80.062 52.965
31705 93.005 -80.053 52.949
31706 93.021 -80.046 52.934
31707 93.036 -80.036 52.920
31708 93.047 -80.030 52.908
31709 93.068 -80.019 52.895
31710 93.088 -80.008 52.885
31711 93.109 -79.998 52.876
31712 93.129 -79.987 52.866
31713 93.150 -79.977 52.856
31714 93.170 -79.966 52.847
31715 93.194 -79.956 52.840
31716 93.219 -79.947 52.834
31717 93.241 -79.938 52.828
31718 93.256 -79.930 52.823
31719 93.271 -79.921 52.820
31720 93.286 -79.913 52.817
31721 93.301 -79.905 52.814
31722 93.315 -79.896 52.814
31723 93.330 -79.888 52.813
31724 93.345 -79.880 52.812
31725 93.360 -79.871 52.812
31726 93.393 -79.859 52.813
31727 93.428 -79.846 52.813
31728 93.465 -79.833 52.814
31729 93.500 -79.820 52.815
31730 93.531 -79.807 52.817
31731 93.541 -79.797 52.818
31732 93.550 -79.787 52.819
31733 93.560 -79.777 52.821
31734 93.570 -79.767 52.823
31735 93.580 -79.757 52.823
31736 93.590 -79.747 52.820
31737 93.600 -79.737 52.817
31738 93.610 -79.727 52.813
31739 93.620 -79.717 52.810
31740 93.630 -79.707 52.806
31741 93.640 -79.697 52.802
31742 93.649 -79.687 52.797
31743 93.659 -79.677 52.790
31744 93.670 -79.667 52.783
31745 93.679 -79.657 52.776
31746 93.689 -79.647 52.765
31747 93.699 -79.637 52.754
31748 93.713 -79.630 52.742
31749 93.728 -79.624 52.733
31750 93.744 -79.617 52.723
31751 93.759 -79.611 52.714
31752 93.774 -79.604 52.707
31753 93.789 -79.598 52.700
31754 93.804 -79.592 52.693
31755 93.820 -79.585 52.688
31756 93.833 -79.581 52.682
31757 93.840 -79.583 52.676
31758 93.846 -79.585 52.669
31759 93.853 -79.588 52.663
31760 93.859 -79.591 52.657
31761 93.866 -79.593 52.649
31762 93.872 -79.595 52.642
31763 93.879 -79.598 52.635
31764 93.886 -79.601 52.627
31765 93.892 -79.603 52.620
31766 93.899 -79.606 52.612
31767 93.905 -79.608 52.604
31768 93.912 -79.610 52.595
31769 93.918 -79.613 52.585
31770 93.925 -79.616 52.575
31771 93.931 -79.618 52.565
31772 93.938 -79.620 52.554
31773 93.944 -79.623 52.544
31774 93.951 -79.626 52.533
31775 93.958 -79.628 52.522
31776 93.965 -79.630 52.509
31777 93.971 -79.633 52.497
31778 93.978 -79.636 52.485
31779 93.984 -79.638 52.471
31780 93.991 -79.641 52.458
31781 93.997 -79.643 52.444
31782 94.004 -79.645 52.431
31783 94.010 -79.648 52.416
31784 94.017 -79.651 52.401
31785 94.024 -79.653 52.386
31786 94.032 -79.652 52.370
31787 94.040 -79.651 52.354
31788 94.049 -79.649 52.339
31789 94.057 -79.648 52.324
31790 94.066 -79.646 52.308
31791 94.074 -79.645 52.291
31792 94.083 -79.643 52.274
31793 94.091 -79.642 52.257
31794 94.100 -79.641 52.241
31795 94.109 -79.639 52.225
31796 94.117 -79.638 52.209
31797 94.126 -79.636 52.192
31798 94.134 -79.635 52.174
31799 94.143 -79.633 52.156
31800 94.151 -79.624 52.139
31801 94.160 -79.598 52.122
31802 94.169 -79.572 52.104
31803 94.177 -79.546 52.086
31804 94.185 -79.520 52.067
31805 94.193 -79.494 52.047
31806 94.201 -79.467 52.029
31807 94.209 -79.441 52.010
31808 94.217 -79.414 51.991
31809 94.224 -79.388 51.975
31810 94.232 -79.361 51.959
31811 94.240 -79.335 51.943
31812 94.248 -79.308 51.925
31813 94.259 -79.287 51.906
31814 94.274 -79.276 51.888
31815 94.290 -79.266 51.870
31816 94.307 -79.256 51.851
31817 94.321 -79.247 51.830
31818 94.324 -79.251 51.808
31819 94.321 -79.263 51.784
31820 94.317 -79.273 51.760
31821 94.314 -79.285 51.736
31822 94.310 -79.296 51.713
31823 94.310 -79.284 51.690
31824 94.312 -79.263 51.665
31825 94.314 -79.242 51.639
31826 94.306 -79.237 51.608
31827 94.293 -79.243 51.575
31828 94.279 -79.249 51.542
31829 94.275 -79.250 51.507
31830 94.285 -79.244 51.472
31831 94.294 -79.237 51.439
31832 94.304 -79.242 51.407
31833 94.313 -79.285 51.372
31834 94.322 -79.328 51.337
31835 94.331 -79.372 51.302
31836 94.339 -79.415 51.265
31837 94.348 -79.458 51.229
31838 94.357 -79.501 51.193
31839 94.366 -79.545 51.157
31840 94.374 -79.588 51.119
31841 94.384 -79.631 51.082
31842 94.392 -79.674 51.044
31843 94.403 -79.708 51.006
31844 94.416 -79.720 50.968
31845 94.429 -79.734 50.931
31846 94.442 -79.747 50.894
32383 92.633 -80.222 53.472
32384 92.648 -80.213 53.448
32385 92.664 -80.206 53.425
32386 92.681 -80.197 53.400
32387 92.697 -80.189 53.376
32388 92.712 -80.181 53.352
32389 92.728 -80.173 53.327
32390 92.744 -80.165 53.303
32391 92.761 -80.157 53.278
32392 92.777 -80.149 53.254
32393 92.793 -80.141 53.229
32394 92.808 -80.133 53.205
32395 92.825 -80.124 53.182
32396 92.841 -80.117 53.160
32397 92.857 -80.109 53.137
32398 92.873 -80.101 53.115
32399 92.889 -80.093 53.093
32400 92.905 -80.084 53.071
32401 92.921 -80.077 53.051
32402 92.937 -80.068 53.031
32403 92.953 -80.061 53.012
32404 92.969 -80.052 52.993
32405 92.985 -80.044 52.975
32406 93.001 -80.037 52.959
32407 93.017 -80.028 52.942
32408 93.033 -80.021 52.927
32409 93.049 -80.008 52.913
32410 93.056 -80.005 52.900
32411 93.071 -79.997 52.889
32412 93.091 -79.987 52.878
32413 93.112 -79.976 52.868
32414 93.133 -79.965 52.859
32415 93.153 -79.955 52.851
32416 93.174 -79.944 52.842
32417 93.199 -79.935 52.836
32418 93.222 -79.926 52.831
32419 93.237 -79.918 52.826
32420 93.252 -79.909 52.824
32421 93.267 -79.901 52.821
32422 93.282 -79.892 52.819
32423 93.296 -79.884 52.818
32424 93.311 -79.876 52.817
32425 93.330 -79.867 52.815
32426 93.353 -79.857 52.814
32427 93.379 -79.846 52.814
32428 93.415 -79.834 52.814
32429 93.451 -79.821 52.815
32430 93.487 -79.808 52.816
32431 93.519 -79.795 52.817
32432 93.534 -79.778 52.818
32433 93.549 -79.762 52.817
32434 93.559 -79.751 52.818
32435 93.569 -79.741 52.818
32436 93.579 -79.731 52.818
32437 93.589 -79.721 52.815
32438 93.599 -79.711 52.811
32439 93.609 -79.701 52.807
32440 93.619 -79.691 52.803
32441 93.629 -79.681 52.799
32442 93.639 -79.671 52.795
32443 93.648 -79.661 52.789
32444 93.658 -79.651 52.782
32445 93.669 -79.642 52.774
32446 93.679 -79.631 52.767
32447 93.688 -79.621 52.756
32448 93.698 -79.611 52.745
32449 93.712 -79.604 52.733
32450 93.727 -79.597 52.724
32451 93.742 -79.591 52.715
32452 93.757 -79.585 52.708
32453 93.772 -79.578 52.701
32454 93.787 -79.572 52.694
32455 93.802 -79.565 52.688
32456 93.818 -79.559 52.683
32457 93.833 -79.552 52.678
32458 93.848 -79.546 52.672
32459 93.859 -79.544 52.665
32460 93.866 -79.546 52.660
32461 93.872 -79.548 52.653
32462 93.879 -79.551 52.646
32463 93.886 -79.554 52.640
32464 93.892 -79.556 52.632
32465 93.899 -79.558 52.625
32466 93.905 -79.561 52.617
32467 93.912 -79.563 52.610
32468 93.918 -79.566 52.601
32469 93.925 -79.568 52.592
32470 93.931 -79.571 52.582
32471 93.938 -79.573 52.572
32472 93.944 -79.576 52.562
32473 93.951 -79.579 52.552
32474 93.958 -79.581 52.541
32475 93.965 -79.583 52.530
32476 93.971 -79.585 52.518
32477 93.978 -79.588 52.506
32478 93.984 -79.591 52.493
32479 93.991 -79.593 52.480
32480 93.997 -79.596 52.467
32481 94.004 -79.598 52.454
32482 94.010 -79.601 52.440
32483 94.017 -79.603 52.427
32484 94.024 -79.606 52.412
32485 94.030 -79.608 52.397
32486 94.037 -79.610 52.383
32487 94.043 -79.614 52.367
32488 94.050 -79.616 52.352
32489 94.056 -79.618 52.336
32490 94.063 -79.620 52.321
32491 94.069 -79.623 52.305
32492 94.076 -79.626 52.288
32493 94.083 -79.628 52.272
32494 94.091 -79.628 52.255
32495 94.100 -79.626 52.239
32496 94.108 -79.625 52.223
32497 94.117 -79.623 52.208
32498 94.125 -79.622 52.191
32499 94.134 -79.620 52.174
32500 94.142 -79.619 52.156
32501 94.150 -79.610 52.140
32502 94.159 -79.585 52.124
32503 94.168 -79.558 52.106
32504 94.175 -79.532 52.087
32505 94.183 -79.506 52.068
32506 94.191 -79.479 52.048
32507 94.199 -79.453 52.029
32508 94.207 -79.426 52.010
32509 94.214 -79.400 51.994
32510 94.222 -79.373 51.978
32511 94.230 -79.347 51.962
32512 94.238 -79.320 51.946
32513 94.246 -79.294 51.927
32514 94.259 -79.278 51.908
32515 94.275 -79.267 51.889
32516 94.291 -79.256 51.871
32517 94.307 -79.246 51.852
32518 94.323 -79.235 51.832
32519 94.322 -79.244 51.811
32520 94.318 -79.255 51.789
32521 94.315 -79.266 51.767
32522 94.311 -79.277 51.744
32523 94.308 -79.288 51.721
32524 94.307 -79.282 51.698
32525 94.309 -79.261 51.674
32526 94.310 -79.240 51.648
32527 94.298 -79.243 51.617
32528 94.285 -79.249 51.585
32529 94.272 -79.254 51.552
32530 94.264 -79.258 51.518
32531 94.273 -79.251 51.484
32532 94.283 -79.244 51.450
32533 94.293 -79.238 51.417
32534 94.302 -79.263 51.382
32535 94.311 -79.306 51.346
32536 94.320 -79.350 51.311
32537 94.329 -79.393 51.275
32538 94.337 -79.436 51.239
32539 94.346 -79.479 51.202
32540 94.355 -79.523 51.165
32541 94.364 -79.566 51.128
32542 94.372 -79.609 51.090
32543 94.382 -79.652 51.051
32544 94.394 -79.668 51.013
32545 94.407 -79.682 50.974
32546 94.420 -79.695 50.937
32547 94.433 -79.708 50.899
32548 94.447 -79.720 50.861
33085 92.646 -80.200 53.473
33086 92.660 -80.188 53.449
33087 92.676 -80.181 53.425
33088 92.692 -80.172 53.400
33089 92.708 -80.164 53.376
33090 92.724 -80.156 53.351
33091 92.740 -80.148 53.326
33092 92.757 -80.140 53.301
33093 92.773 -80.132 53.276
33094 92.788 -80.124 53.252
33095 92.804 -80.116 53.228
33096 92.820 -80.108 53.204
33097 92.837 -80.100 53.180
33098 92.853 -80.092 53.157
33099 92.869 -80.084 53.134
33100 92.885 -80.076 53.111
33101 92.901 -80.068 53.089
33102 92.917 -80.059 53.068
33103 92.933 -80.052 53.046
33104 92.949 -80.043 53.026
33105 92.965 -80.036 53.006
33106 92.981 -80.027 52.987
33107 92.997 -80.019 52.970
33108 93.013 -80.012 52.952
33109 93.029 -80.003 52.936
33110 93.046 -79.994 52.920
33111 93.063 -79.981 52.907
33112 93.070 -79.977 52.893
33113 93.076 -79.974 52.880
33114 93.095 -79.965 52.871
33115 93.116 -79.954 52.863
33116 93.136 -79.943 52.854
33117 93.156 -79.933 52.846
33118 93.180 -79.923 52.840
33119 93.203 -79.914 52.835
33120 93.218 -79.905 52.831
33121 93.233 -79.897 52.829
33122 93.248 -79.889 52.827
33123 93.263 -79.880 52.824
33124 93.277 -79.872 52.823
33125 93.301 -79.862 52.821
33126 93.324 -79.852 52.819
33127 93.347 -79.843 52.818
33128 93.370 -79.833 52.818
33129 93.402 -79.821 52.818
33130 93.438 -79.808 52.818
33131 93.474 -79.795 52.819
33132 93.506 -79.783 52.819
33133 93.521 -79.766 52.819
33134 93.537 -79.750 52.817
33135 93.551 -79.734 52.815
33136 93.566 -79.718 52.814
33137 93.578 -79.705 52.812
33138 93.588 -79.695 52.809
33139 93.598 -79.686 52.805
33140 93.608 -79.675 52.801
33141 93.618 -79.665 52.796
33142 93.628 -79.655 52.791
33143 93.638 -79.645 52.786
33144 93.648 -79.636 52.779
33145 93.657 -79.625 52.772
33146 93.667 -79.615 52.764
33147 93.678 -79.605 52.756
33148 93.687 -79.595 52.746
33149 93.697 -79.585 52.736
33150 93.710 -79.577 52.725
33151 93.725 -79.570 52.716
33152 93.740 -79.564 52.708
33153 93.755 -79.557 52.701
33154 93.770 -79.551 52.694
33155 93.785 -79.545 52.688
33156 93.800 -79.538 52.682
33157 93.816 -79.532 52.678
33158 93.831 -79.526 52.672
33159 93.846 -79.520 52.667
33160 93.861 -79.513 52.661
33161 93.876 -79.507 52.656
33162 93.886 -79.506 52.649
33163 93.892 -79.509 52.642
33164 93.899 -79.511 52.636
33165 93.905 -79.513 52.629
33166 93.912 -79.516 52.622
33167 93.918 -79.519 52.614
33168 93.925 -79.521 52.606
33169 93.931 -79.523 52.598
33170 93.938 -79.526 52.590
33171 93.944 -79.529 52.580
33172 93.951 -79.531 52.570
33173 93.958 -79.533 52.559
33174 93.965 -79.536 52.548
33175 93.971 -79.538 52.537
33176 93.978 -79.541 52.525
33177 93.984 -79.544 52.514
33178 93.991 -79.546 52.501
33179 93.997 -79.548 52.488
33180 94.004 -79.551 52.475
33181 94.010 -79.554 52.463
33182 94.017 -79.556 52.450
33183 94.024 -79.558 52.436
33184 94.030 -79.561 52.423
33185 94.037 -79.563 52.408
33186 94.043 -79.566 52.393
33187 94.050 -79.569 52.379
33188 94.056 -79.571 52.364
33189 94.063 -79.573 52.349
33190 94.069 -79.576 52.333
33191 94.076 -79.579 52.319
33192 94.082 -79.581 52.302
33193 94.089 -79.583 52.285
33194 94.096 -79.585 52.269
33195 94.103 -79.588 52.253
33196 94.109 -79.591 52.237
33197 94.116 -79.593 52.222
33198 94.122 -79.596 52.206
33199 94.129 -79.598 52.190
33200 94.135 -79.601 52.174
33201 94.142 -79.604 52.158
33202 94.150 -79.597 52.142
33203 94.157 -79.570 52.125
33204 94.166 -79.544 52.107
33205 94.173 -79.517 52.089
33206 94.181 -79.491 52.070
33207 94.189 -79.464 52.050
33208 94.197 -79.438 52.030
33209 94.205 -79.412 52.012
33210 94.212 -79.385 51.996
33211 94.220 -79.359 51.980
33212 94.228 -79.332 51.965
33213 94.236 -79.306 51.949
33214 94.244 -79.279 51.930
33215 94.260 -79.268 51.910
33216 94.275 -79.257 51.890
33217 94.291 -79.247 51.870
33218 94.300 -79.244 51.851
33219 94.308 -79.242 51.832
33220 94.317 -79.239 51.814
33221 94.316 -79.247 51.794
33222 94.312 -79.258 51.774
33223 94.309 -79.269 51.753
33224 94.305 -79.281 51.730
33225 94.303 -79.280 51.707
33226 94.305 -79.259 51.684
33227 94.305 -79.242 51.660
33228 94.291 -79.248 51.630
33229 94.277 -79.254 51.599
33230 94.264 -79.260 51.566
33231 94.252 -79.266 51.533
33232 94.262 -79.259 51.498
33233 94.272 -79.253 51.464
33234 94.281 -79.246 51.430
33235 94.291 -79.241 51.393
33236 94.300 -79.284 51.357
33237 94.309 -79.327 51.321
33238 94.318 -79.370 51.284
33239 94.327 -79.413 51.249
33240 94.335 -79.457 51.213
33241 94.344 -79.500 51.176
33242 94.353 -79.543 51.137
33243 94.362 -79.586 51.099
33244 94.372 -79.616 51.059
33245 94.386 -79.629 51.020
33246 94.399 -79.642 50.981
33247 94.412 -79.655 50.943
33248 94.425 -79.668 50.905
33249 94.438 -79.681 50.867
33788 92.673 -80.165 53.450
33789 92.688 -80.156 53.425
33790 92.704 -80.147 53.400
33791 92.720 -80.139 53.375
33792 92.736 -80.131 53.350
33793 92.753 -80.123 53.325
33794 92.768 -80.115 53.299
33795 92.784 -80.107 53.275
33796 92.800 -80.099 53.251
33797 92.816 -80.091 53.226
33798 92.833 -80.083 53.202
33799 92.849 -80.075 53.178
33800 92.865 -80.067 53.154
33801 92.880 -80.059 53.130
33802 92.897 -80.051 53.107
33803 92.913 -80.043 53.085
33804 92.929 -80.035 53.064
33805 92.945 -80.027 53.043
33806 92.961 -80.018 53.022
33807 92.977 -80.011 53.002
33808 92.993 -80.002 52.982
33809 93.009 -79.994 52.964
33810 93.025 -79.987 52.947
33811 93.041 -79.978 52.930
33812 93.058 -79.968 52.915
33813 93.076 -79.954 52.900
33814 93.083 -79.950 52.887
33815 93.090 -79.946 52.875
33816 93.098 -79.943 52.866
33817 93.119 -79.932 52.857
33818 93.139 -79.921 52.851
33819 93.160 -79.911 52.846
33820 93.184 -79.902 52.841
33821 93.199 -79.893 52.837
33822 93.214 -79.885 52.835
33823 93.228 -79.877 52.833
33824 93.248 -79.867 52.831
33825 93.271 -79.858 52.829
33826 93.295 -79.848 52.826
33827 93.318 -79.839 52.824
33828 93.341 -79.829 52.822
33829 93.364 -79.819 52.821
33830 93.389 -79.809 52.821
33831 93.425 -79.796 52.821
33832 93.461 -79.783 52.822
33833 93.494 -79.770 52.822
33834 93.509 -79.754 52.820
33835 93.524 -79.738 52.817
33836 93.539 -79.722 52.814
33837 93.554 -79.706 52.810
33838 93.569 -79.689 52.807
33839 93.584 -79.673 52.804
33840 93.597 -79.659 52.800
33841 93.607 -79.649 52.795
33842 93.617 -79.639 52.788
33843 93.627 -79.629 52.782
33844 93.637 -79.620 52.777
33845 93.647 -79.609 52.770
33846 93.657 -79.599 52.762
33847 93.666 -79.589 52.754
33848 93.677 -79.579 52.746
33849 93.687 -79.570 52.736
33850 93.696 -79.559 52.727
33851 93.708 -79.551 52.717
33852 93.723 -79.544 52.708
33853 93.738 -79.538 52.700
33854 93.753 -79.531 52.692
33855 93.768 -79.525 52.686
33856 93.783 -79.518 52.680
33857 93.798 -79.512 52.675
33858 93.814 -79.505 52.672
33859 93.829 -79.499 52.667
33860 93.844 -79.492 52.662
33861 93.859 -79.486 52.657
33862 93.874 -79.479 52.652
33863 93.890 -79.473 52.646
33864 93.905 -79.466 52.639
33865 93.912 -79.469 52.633
33866 93.918 -79.471 52.627
33867 93.925 -79.474 52.620
33868 93.931 -79.476 52.612
33869 93.938 -79.479 52.604
33870 93.944 -79.482 52.595
33871 93.951 -79.484 52.587
33872 93.958 -79.486 52.577
33873 93.965 -79.488 52.566
33874 93.971 -79.491 52.555
33875 93.978 -79.494 52.544
33876 93.984 -79.496 52.533
33877 93.991 -79.499 52.522
33878 93.997 -79.501 52.510
33879 94.004 -79.504 52.497
33880 94.010 -79.506 52.484
33881 94.017 -79.509 52.471
33882 94.024 -79.511 52.458
33883 94.030 -79.513 52.445
33884 94.037 -79.516 52.432
33885 94.043 -79.519 52.419
33886 94.050 -79.521 52.404
33887 94.056 -79.523 52.390
33888 94.063 -79.526 52.375
33889 94.069 -79.529 52.361
33890 94.076 -79.531 52.346
33891 94.082 -79.534 52.332
33892 94.089 -79.536 52.317
33893 94.096 -79.538 52.300
33894 94.103 -79.541 52.283
33895 94.105 -79.546 52.267
33896 94.106 -79.551 52.250
33897 94.107 -79.557 52.234
33898 94.109 -79.562 52.219
33899 94.110 -79.568 52.205
33900 94.113 -79.573 52.190
33901 94.122 -79.574 52.174
33902 94.135 -79.571 52.159
33903 94.146 -79.558 52.143
33904 94.154 -79.550 52.126
33905 94.162 -79.524 52.109
33906 94.170 -79.499 52.092
33907 94.178 -79.474 52.073
33908 94.187 -79.448 52.053
33909 94.195 -79.423 52.032
33910 94.202 -79.397 52.014
33911 94.210 -79.370 51.997
33912 94.218 -79.344 51.982
33913 94.226 -79.318 51.967
33914 94.234 -79.291 51.951
33915 94.244 -79.269 51.932
33916 94.260 -79.259 51.911
33917 94.268 -79.256 51.890
33918 94.276 -79.254 51.870
33919 94.284 -79.251 51.851
33920 94.293 -79.249 51.834
33921 94.301 -79.247 51.817
33922 94.309 -79.244 51.798
33923 94.310 -79.250 51.780
33924 94.306 -79.262 51.761
33925 94.302 -79.273 51.739
33926 94.300 -79.278 51.717
33927 94.301 -79.257 51.694
33928 94.296 -79.248 51.671
33929 94.283 -79.254 51.642
33930 94.270 -79.260 51.611
33931 94.257 -79.266 51.579
33932 94.243 -79.272 51.547
33933 94.251 -79.267 51.514
33934 94.260 -79.260 51.479
33935 94.270 -79.253 51.445
33936 94.280 -79.247 51.408
33937 94.289 -79.262 51.370
33938 94.298 -79.305 51.333
33939 94.307 -79.348 51.296
33940 94.316 -79.391 51.259
33941 94.325 -79.435 51.223
33942 94.333 -79.478 51.185
33943 94.342 -79.521 51.147
33944 94.351 -79.563 51.108
33945 94.364 -79.577 51.068
33946 94.377 -79.590 51.028
33947 94.391 -79.603 50.989
33948 94.404 -79.616 50.951
33949 94.417 -79.629 50.912
33950 94.430 -79.642 50.873
34491 92.700 -80.131 53.424
34492 92.716 -80.122 53.399
34493 92.732 -80.114 53.374
34494 92.748 -80.106 53.348
34495 92.764 -80.098 53.324
34496 92.780 -80.090 53.299
34497 92.796 -80.082 53.274
34498 92.812 -80.074 53.249
34499 92.829 -80.066 53.223
34500 92.845 -80.058 53.199
34501 92.860 -80.050 53.175
34502 92.876 -80.042 53.150
34503 92.893 -80.034 53.126
34504 92.909 -80.026 53.103
34505 92.925 -80.018 53.081
34506 92.941 -80.010 53.059
34507 92.956 -80.002 53.038
34508 92.973 -79.993 53.017
34509 92.989 -79.986 52.996
34510 93.005 -79.977 52.978
34511 93.021 -79.970 52.959
34512 93.037 -79.961 52.942
34513 93.053 -79.953 52.925
34514 93.070 -79.942 52.909
34515 93.088 -79.928 52.894
34516 93.097 -79.923 52.881
34517 93.103 -79.919 52.870
34518 93.111 -79.915 52.860
34519 93.122 -79.909 52.854
34520 93.143 -79.899 52.850
34521 93.165 -79.889 52.846
34522 93.180 -79.881 52.842
34523 93.195 -79.872 52.840
34524 93.219 -79.863 52.838
34525 93.242 -79.853 52.835
34526 93.265 -79.843 52.834
34527 93.288 -79.834 52.831
34528 93.312 -79.824 52.828
34529 93.335 -79.814 52.825
34530 93.358 -79.805 52.825
34531 93.382 -79.795 52.825
34532 93.412 -79.784 52.824
34533 93.447 -79.771 52.824
34534 93.482 -79.758 52.823
34535 93.497 -79.742 52.822
34536 93.512 -79.726 52.818
34537 93.527 -79.710 52.813
34538 93.542 -79.694 52.808
34539 93.557 -79.677 52.804
34540 93.572 -79.661 52.800
34541 93.586 -79.645 52.796
34542 93.602 -79.629 52.789
34543 93.616 -79.614 52.781
34544 93.626 -79.604 52.773
34545 93.636 -79.593 52.768
34546 93.646 -79.583 52.760
34547 93.656 -79.573 52.752
34548 93.665 -79.563 52.744
34549 93.676 -79.554 52.735
34550 93.686 -79.543 52.727
34551 93.695 -79.533 52.718
34552 93.706 -79.524 52.710
34553 93.721 -79.517 52.701
34554 93.736 -79.511 52.692
34555 93.751 -79.504 52.684
34556 93.767 -79.498 52.678
34557 93.782 -79.491 52.673
34558 93.797 -79.485 52.669
34559 93.812 -79.479 52.666
34560 93.827 -79.473 52.662
34561 93.842 -79.466 52.657
34562 93.857 -79.460 52.653
34563 93.872 -79.453 52.648
34564 93.888 -79.447 52.643
34565 93.903 -79.440 52.637
34566 93.918 -79.434 52.631
34567 93.931 -79.429 52.625
34568 93.938 -79.432 52.618
34569 93.944 -79.434 52.610
34570 93.951 -79.437 52.602
34571 93.958 -79.439 52.593
34572 93.965 -79.441 52.584
34573 93.971 -79.444 52.574
34574 93.978 -79.447 52.562
34575 93.984 -79.449 52.551
34576 93.991 -79.451 52.540
34577 93.997 -79.454 52.529
34578 94.004 -79.457 52.518
34579 94.010 -79.459 52.505
34580 94.017 -79.461 52.493
34581 94.024 -79.464 52.480
34582 94.030 -79.466 52.467
34583 94.037 -79.469 52.454
34584 94.043 -79.472 52.440
34585 94.050 -79.474 52.427
34586 94.056 -79.476 52.414
34587 94.059 -79.481 52.401
34588 94.060 -79.487 52.386
34589 94.061 -79.492 52.371
34590 94.062 -79.498 52.357
34591 94.063 -79.504 52.344
34592 94.064 -79.509 52.331
34593 94.065 -79.514 52.316
34594 94.067 -79.520 52.299
34595 94.068 -79.526 52.281
34596 94.069 -79.531 52.265
34597 94.070 -79.536 52.248
34598 94.075 -79.540 52.232
34599 94.084 -79.542 52.219
34600 94.093 -79.544 52.205
34601 94.102 -79.546 52.190
34602 94.112 -79.548 52.174
34603 94.128 -79.507 52.158
34604 94.143 -79.506 52.140
34605 94.149 -79.506 52.123
34606 94.158 -79.496 52.107
34607 94.167 -79.476 52.090
34608 94.175 -79.451 52.073
34609 94.183 -79.426 52.054
34610 94.191 -79.400 52.034
34611 94.199 -79.375 52.014
34612 94.207 -79.350 51.996
34613 94.215 -79.325 51.981
34614 94.223 -79.299 51.966
34615 94.231 -79.274 51.950
34616 94.238 -79.267 51.931
34617 94.245 -79.266 51.911
34618 94.253 -79.263 51.890
34619 94.261 -79.261 51.870
34620 94.269 -79.259 51.853
34621 94.277 -79.256 51.837
34622 94.285 -79.254 51.821
34623 94.294 -79.251 51.802
34624 94.302 -79.249 51.785
34625 94.303 -79.254 51.768
34626 94.300 -79.266 51.748
34627 94.296 -79.276 51.727
34628 94.298 -79.256 51.704
34629 94.289 -79.253 51.681
34630 94.276 -79.259 51.652
34631 94.262 -79.266 51.622
34632 94.249 -79.272 51.591
34633 94.236 -79.278 51.561
34634 94.239 -79.275 51.529
34635 94.249 -79.268 51.495
34636 94.259 -79.262 51.461
34637 94.268 -79.255 51.423
34638 94.278 -79.248 51.385
34639 94.287 -79.282 51.347
34640 94.296 -79.325 51.309
34641 94.305 -79.369 51.272
34642 94.314 -79.412 51.234
34643 94.322 -79.455 51.196
34644 94.331 -79.499 51.157
34645 94.342 -79.525 51.118
34646 94.356 -79.538 51.077
34647 94.369 -79.551 51.037
34648 94.382 -79.563 50.998
34649 94.395 -79.576 50.959
34650 94.408 -79.589 50.919
34651 94.421 -79.602 50.880
34652 94.434 -79.616 50.840
35195 92.744 -80.089 53.372
35196 92.760 -80.081 53.347
35197 92.776 -80.073 53.321
35198 92.792 -80.065 53.296
35199 92.808 -80.057 53.271
35200 92.825 -80.049 53.245
35201 92.840 -80.041 53.221
35202 92.856 -80.033 53.196
35203 92.872 -80.025 53.171
35204 92.888 -80.017 53.147
35205 92.905 -80.008 53.123
35206 92.921 -80.001 53.099
35207 92.936 -79.993 53.076
35208 92.952 -79.985 53.054
35209 92.969 -79.977 53.033
35210 92.985 -79.968 53.011
35211 93.001 -79.961 52.992
35212 93.017 -79.952 52.973
35213 93.032 -79.945 52.953
35214 93.049 -79.936 52.935
35215 93.065 -79.928 52.918
35216 93.082 -79.916 52.901
35217 93.100 -79.902 52.888
35218 93.111 -79.896 52.875
35219 93.117 -79.892 52.864
35220 93.124 -79.888 52.858
35221 93.131 -79.884 52.854
35222 93.146 -79.877 52.851
35223 93.166 -79.868 52.847
35224 93.189 -79.858 52.844
35225 93.213 -79.849 52.841
35226 93.236 -79.839 52.838
35227 93.259 -79.830 52.836
35228 93.282 -79.820 52.833
35229 93.306 -79.810 52.830
35230 93.329 -79.801 52.828
35231 93.352 -79.791 52.827
35232 93.375 -79.781 52.826
35233 93.399 -79.772 52.826
35234 93.434 -79.759 52.826
35235 93.470 -79.746 52.825
35236 93.485 -79.730 52.823
35237 93.500 -79.714 52.818
35238 93.515 -79.698 52.813
35239 93.530 -79.682 52.807
35240 93.545 -79.665 52.802
35241 93.559 -79.649 52.796
35242 93.574 -79.633 52.790
35243 93.589 -79.617 52.782
35244 93.604 -79.601 52.773
35245 93.619 -79.585 52.765
35246 93.634 -79.569 52.758
35247 93.645 -79.557 52.750
35248 93.655 -79.548 52.742
35249 93.665 -79.538 52.734
35250 93.675 -79.527 52.725
35251 93.685 -79.517 52.718
35252 93.695 -79.507 52.710
35253 93.704 -79.498 52.701
35254 93.719 -79.491 52.692
35255 93.734 -79.484 52.684
35256 93.750 -79.478 52.676
35257 93.765 -79.472 52.670
35258 93.780 -79.465 52.665
35259 93.795 -79.459 52.662
35260 93.810 -79.452 52.660
35261 93.825 -79.446 52.657
35262 93.840 -79.439 52.653
35263 93.855 -79.433 52.649
35264 93.870 -79.426 52.646
35265 93.886 -79.420 52.642
35266 93.901 -79.413 52.636
35267 93.916 -79.407 52.630
35268 93.931 -79.400 52.625
35269 93.946 -79.394 52.619
35270 93.958 -79.392 52.612
35271 93.965 -79.394 52.603
35272 93.971 -79.397 52.593
35273 93.978 -79.399 52.584
35274 93.984 -79.402 52.572
35275 93.991 -79.404 52.561
35276 93.997 -79.407 52.549
35277 94.004 -79.410 52.537
35278 94.010 -79.412 52.525
35279 94.013 -79.416 52.513
35280 94.014 -79.422 52.500
35281 94.015 -79.428 52.488
35282 94.016 -79.433 52.476
35283 94.017 -79.438 52.463
35284 94.018 -79.444 52.449
35285 94.019 -79.450 52.436
35286 94.020 -79.455 52.423
35287 94.022 -79.460 52.411
35288 94.023 -79.466 52.397
35289 94.024 -79.472 52.383
35290 94.025 -79.477 52.368
35291 94.026 -79.483 52.354
35292 94.027 -79.488 52.341
35293 94.028 -79.494 52.331
35294 94.029 -79.500 52.316
35295 94.030 -79.505 52.298
35296 94.037 -79.508 52.279
35297 94.046 -79.510 52.262
35298 94.055 -79.512 52.246
35299 94.064 -79.513 52.230
35300 94.073 -79.516 52.218
35301 94.083 -79.517 52.205
35302 94.092 -79.519 52.190
35303 94.128 -79.478 52.173
35304 94.139 -79.432 52.156
35305 94.141 -79.453 52.137
35306 94.145 -79.463 52.119
35307 94.154 -79.453 52.104
35308 94.163 -79.443 52.087
35309 94.171 -79.429 52.070
35310 94.179 -79.403 52.052
35311 94.187 -79.378 52.033
35312 94.195 -79.352 52.013
35313 94.203 -79.327 51.993
35314 94.211 -79.302 51.976
35315 94.219 -79.276 51.962
35316 94.227 -79.257 51.947
35317 94.234 -79.267 51.929
35318 94.239 -79.266 51.910
35319 94.245 -79.266 51.892
35320 94.250 -79.266 51.874
35321 94.255 -79.265 51.858
35322 94.262 -79.263 51.843
35323 94.270 -79.261 51.826
35324 94.278 -79.259 51.807
35325 94.286 -79.256 51.790
35326 94.295 -79.254 51.773
35327 94.297 -79.258 51.755
35328 94.294 -79.269 51.734
35329 94.294 -79.254 51.712
35330 94.281 -79.259 51.687
35331 94.268 -79.266 51.660
35332 94.255 -79.272 51.632
35333 94.242 -79.278 51.603
35334 94.228 -79.284 51.573
35335 94.227 -79.283 51.542
35336 94.238 -79.276 51.510
35337 94.247 -79.269 51.478
35338 94.257 -79.263 51.441
35339 94.267 -79.256 51.403
35340 94.276 -79.260 51.363
35341 94.285 -79.303 51.326
35342 94.294 -79.347 51.287
35343 94.302 -79.390 51.247
35344 94.312 -79.433 51.208
35345 94.321 -79.473 51.168
35346 94.334 -79.485 51.127
35347 94.347 -79.498 51.086
35348 94.360 -79.511 51.047
35349 94.373 -79.524 51.007
35350 94.387 -79.537 50.967
35351 94.400 -79.550 50.928
35352 94.413 -79.563 50.887
35353 94.426 -79.576 50.847
35899 92.788 -80.048 53.318
35900 92.804 -80.040 53.292
35901 92.820 -80.032 53.267
35902 92.836 -80.024 53.242
35903 92.852 -80.016 53.216
35904 92.868 -80.008 53.192
35905 92.884 -80.000 53.166
35906 92.901 -79.992 53.141
35907 92.916 -79.983 53.117
35908 92.932 -79.976 53.094
35909 92.948 -79.968 53.071
35910 92.965 -79.960 53.048
35911 92.981 -79.952 53.027
35912 92.997 -79.943 53.007
35913 93.012 -79.936 52.986
35914 93.028 -79.927 52.965
35915 93.045 -79.920 52.947
35916 93.061 -79.911 52.929
35917 93.077 -79.903 52.911
35918 93.095 -79.889 52.896
35919 93.113 -79.876 52.883
35920 93.124 -79.868 52.871
35921 93.131 -79.864 52.863
35922 93.138 -79.861 52.859
35923 93.144 -79.857 52.856
35924 93.160 -79.851 52.851
35925 93.184 -79.843 52.847
35926 93.207 -79.835 52.843
35927 93.230 -79.825 52.839
35928 93.253 -79.815 52.837
35929 93.276 -79.805 52.834
35930 93.300 -79.796 52.831
35931 93.323 -79.786 52.830
35932 93.346 -79.776 52.829
35933 93.369 -79.767 52.828
35934 93.393 -79.758 52.828
35935 93.421 -79.746 52.828
35936 93.457 -79.734 52.826
35937 93.473 -79.717 52.823
35938 93.488 -79.701 52.819
35939 93.502 -79.686 52.813
35940 93.517 -79.670 52.806
35941 93.533 -79.653 52.799
35942 93.547 -79.637 52.792
35943 93.562 -79.621 52.785
35944 93.577 -79.605 52.777
35945 93.592 -79.588 52.767
35946 93.607 -79.573 52.758
35947 93.622 -79.557 52.751
35948 93.637 -79.541 52.743
35949 93.651 -79.524 52.735
35950 93.664 -79.511 52.727
35951 93.674 -79.501 52.717
35952 93.684 -79.491 52.708
35953 93.694 -79.482 52.699
35954 93.704 -79.472 52.690
35955 93.717 -79.464 52.682
35956 93.732 -79.457 52.674
35957 93.748 -79.451 52.668
35958 93.763 -79.444 52.662
35959 93.778 -79.438 52.658
35960 93.793 -79.432 52.656
35961 93.808 -79.426 52.655
35962 93.823 -79.419 52.653
35963 93.838 -79.413 52.650
35964 93.854 -79.407 52.647
35965 93.869 -79.400 52.645
35966 93.884 -79.394 52.642
35967 93.899 -79.387 52.637
35968 93.914 -79.381 52.631
35969 93.929 -79.374 52.627
35970 93.944 -79.368 52.623
35971 93.960 -79.363 52.616
35972 93.971 -79.360 52.607
35973 93.969 -79.363 52.597
35974 93.970 -79.369 52.586
35975 93.971 -79.374 52.574
35976 93.972 -79.379 52.561
35977 93.973 -79.385 52.548
35978 93.975 -79.391 52.536
35979 93.976 -79.396 52.523
35980 93.977 -79.402 52.510
35981 93.978 -79.407 52.498
35982 93.979 -79.413 52.486
35983 93.980 -79.418 52.473
35984 93.981 -79.424 52.460
35985 93.982 -79.429 52.447
35986 93.983 -79.435 52.434
35987 93.984 -79.441 52.421
35988 93.985 -79.446 52.408
35989 93.986 -79.451 52.394
35990 93.987 -79.457 52.379
35991 93.988 -79.463 52.366
35992 93.989 -79.468 52.353
35993 93.991 -79.474 52.342
35994 93.999 -79.476 52.330
35995 94.008 -79.478 52.315
35996 94.017 -79.479 52.297
35997 94.027 -79.482 52.278
35998 94.036 -79.483 52.260
35999 94.045 -79.485 52.243
36000 94.054 -79.487 52.228
36001 94.063 -79.488 52.216
36002 94.072 -79.491 52.203
36003 94.121 -79.447 52.189
36004 94.099 -79.427 52.171
36005 94.136 -79.379 52.153
36006 94.138 -79.400 52.134
36007 94.141 -79.419 52.114
36008 94.149 -79.409 52.099
36009 94.158 -79.399 52.083
36010 94.167 -79.390 52.067
36011 94.175 -79.380 52.048
36012 94.183 -79.355 52.030
36013 94.191 -79.329 52.010
36014 94.199 -79.304 51.990
36015 94.207 -79.279 51.972
36016 94.215 -79.253 51.957
36017 94.223 -79.253 51.943
36018 94.230 -79.263 51.925
36019 94.236 -79.266 51.909
36020 94.241 -79.266 51.893
36021 94.246 -79.265 51.877
36022 94.251 -79.265 51.863
36023 94.257 -79.264 51.848
36024 94.262 -79.264 51.832
36025 94.267 -79.263 51.812
36026 94.272 -79.263 51.795
36027 94.279 -79.261 51.779
36028 94.287 -79.259 51.761
36029 94.291 -79.261 51.742
36030 94.287 -79.259 51.720
36031 94.274 -79.266 51.696
36032 94.261 -79.272 51.670
36033 94.247 -79.278 51.643
36034 94.234 -79.284 51.615
36035 94.220 -79.290 51.585
36036 94.216 -79.291 51.555
36037 94.226 -79.284 51.523
36038 94.236 -79.278 51.490
36039 94.246 -79.271 51.454
36040 94.255 -79.264 51.417
36041 94.265 -79.257 51.379
36042 94.274 -79.281 51.343
36043 94.283 -79.324 51.304
36044 94.292 -79.368 51.263
36045 94.300 -79.411 51.222
36046 94.313 -79.433 51.181
36047 94.326 -79.446 51.139
36048 94.339 -79.459 51.098
36049 94.352 -79.472 51.057
36050 94.365 -79.485 51.016
36051 94.378 -79.498 50.976
36052 94.392 -79.511 50.936
36053 94.405 -79.524 50.896
36054 94.418 -79.537 50.855
36055 94.431 -79.550 50.815
36602 92.816 -80.015 53.289
36603 92.832 -80.007 53.263
36604 92.848 -79.999 53.237
36605 92.864 -79.991 53.212
36606 92.880 -79.983 53.186
36607 92.896 -79.975 53.160
36608 92.912 -79.967 53.136
36609 92.928 -79.958 53.112
36610 92.944 -79.951 53.089
36611 92.960 -79.943 53.065
36612 92.977 -79.935 53.043
36613 92.992 -79.927 53.022
36614 93.008 -79.918 53.000
36615 93.024 -79.911 52.979
36616 93.041 -79.902 52.959
36617 93.057 -79.895 52.940
36618 93.073 -79.886 52.922
36619 93.089 -79.877 52.906
36620 93.108 -79.863 52.892
36621 93.126 -79.849 52.879
36622 93.138 -79.841 52.870
36623 93.144 -79.837 52.863
36624 93.151 -79.833 52.857
36625 93.158 -79.830 52.852
36626 93.179 -79.823 52.848
36627 93.202 -79.814 52.844
36628 93.224 -79.807 52.840
36629 93.248 -79.798 52.837
36630 93.271 -79.791 52.835
36631 93.294 -79.782 52.832
36632 93.317 -79.772 52.830
36633 93.340 -79.763 52.830
36634 93.363 -79.753 52.830
36635 93.387 -79.743 52.830
36636 93.410 -79.734 52.829
36637 93.444 -79.721 52.827
36638 93.461 -79.705 52.824
36639 93.476 -79.689 52.820
36640 93.490 -79.673 52.813
36641 93.505 -79.657 52.805
36642 93.520 -79.641 52.797
36643 93.535 -79.625 52.789
36644 93.550 -79.609 52.781
36645 93.565 -79.593 52.772
36646 93.580 -79.576 52.762
36647 93.594 -79.560 52.753
36648 93.610 -79.545 52.744
36649 93.625 -79.529 52.737
36650 93.639 -79.512 52.729
36651 93.654 -79.496 52.721
36652 93.669 -79.480 52.711
36653 93.683 -79.466 52.702
36654 93.693 -79.456 52.693
36655 93.703 -79.445 52.684
36656 93.715 -79.438 52.676
36657 93.730 -79.431 52.669
36658 93.746 -79.425 52.662
36659 93.761 -79.418 52.656
36660 93.776 -79.412 52.652
36661 93.791 -79.405 52.650
36662 93.806 -79.399 52.651
36663 93.822 -79.392 52.650
36664 93.837 -79.386 52.650
36665 93.852 -79.380 52.647
36666 93.866 -79.378 52.645
36667 93.882 -79.375 52.643
36668 93.896 -79.372 52.640
36669 93.911 -79.369 52.636
36670 93.926 -79.366 52.633
36671 93.941 -79.363 52.630
36672 93.946 -79.360 52.623
36673 93.951 -79.357 52.614
36674 93.946 -79.356 52.603
36675 93.938 -79.357 52.591
36676 93.935 -79.359 52.579
36677 93.936 -79.365 52.565
36678 93.937 -79.370 52.550
36679 93.938 -79.375 52.536
36680 93.939 -79.382 52.523
36681 93.940 -79.387 52.509
36682 93.941 -79.392 52.498
36683 93.942 -79.398 52.485
36684 93.943 -79.404 52.472
36685 93.944 -79.409 52.459
36686 93.945 -79.415 52.447
36687 93.946 -79.420 52.433
36688 93.947 -79.426 52.420
36689 93.949 -79.432 52.407
36690 93.950 -79.437 52.393
36691 93.952 -79.442 52.379
36692 93.961 -79.444 52.366
36693 93.970 -79.446 52.354
36694 93.979 -79.447 52.342
36695 93.988 -79.449 52.329
36696 93.998 -79.451 52.313
36697 94.007 -79.453 52.296
36698 94.016 -79.455 52.277
36699 94.025 -79.457 52.258
36700 94.035 -79.458 52.241
36701 94.044 -79.460 52.225
36702 94.053 -79.462 52.212
36703 94.096 -79.425 52.199
36704 94.107 -79.390 52.184
36705 94.109 -79.345 52.167
36706 94.134 -79.326 52.148
36707 94.136 -79.347 52.129
36708 94.138 -79.369 52.110
36709 94.145 -79.366 52.093
36710 94.153 -79.356 52.077
36711 94.162 -79.346 52.061
36712 94.171 -79.336 52.042
36713 94.179 -79.326 52.024
36714 94.187 -79.306 52.006
36715 94.195 -79.281 51.987
36716 94.203 -79.256 51.968
36717 94.211 -79.238 51.951
36718 94.218 -79.249 51.937
36719 94.225 -79.259 51.921
36720 94.232 -79.266 51.906
36721 94.237 -79.265 51.893
36722 94.243 -79.265 51.880
36723 94.248 -79.264 51.867
36724 94.253 -79.264 51.854
36725 94.258 -79.263 51.838
36726 94.263 -79.263 51.818
36727 94.269 -79.263 51.801
36728 94.274 -79.262 51.786
36729 94.279 -79.262 51.767
36730 94.284 -79.261 51.749
36731 94.285 -79.261 51.728
36732 94.267 -79.271 51.705
36733 94.253 -79.277 51.679
36734 94.240 -79.283 51.653
36735 94.226 -79.289 51.626
36736 94.213 -79.295 51.597
36737 94.205 -79.298 51.566
36738 94.214 -79.292 51.534
36739 94.224 -79.285 51.501
36740 94.234 -79.278 51.464
36741 94.244 -79.272 51.429
36742 94.254 -79.266 51.394
36743 94.263 -79.259 51.359
36744 94.272 -79.302 51.320
36745 94.281 -79.345 51.278
36746 94.291 -79.381 51.235
36747 94.304 -79.394 51.193
36748 94.317 -79.407 51.151
36749 94.330 -79.419 51.109
36750 94.344 -79.432 51.068
36751 94.357 -79.446 51.026
36752 94.370 -79.459 50.985
36753 94.383 -79.472 50.945
36754 94.396 -79.485 50.904
36755 94.409 -79.498 50.864
36756 94.422 -79.510 50.824
37308 92.892 -79.958 53.180
37309 92.908 -79.950 53.156
37310 92.924 -79.942 53.131
37311 92.940 -79.933 53.106
37312 92.956 -79.926 53.083
37313 92.972 -79.918 53.060
37314 92.988 -79.910 53.037
37315 93.004 -79.902 53.015
37316 93.020 -79.893 52.993
37317 93.037 -79.886 52.973
37318 93.053 -79.877 52.953
37319 93.068 -79.870 52.935
37320 93.084 -79.861 52.918
37321 93.101 -79.851 52.902
37322 93.120 -79.837 52.889
37323 93.138 -79.823 52.877
37324 93.151 -79.814 52.868
37325 93.158 -79.810 52.860
37326 93.165 -79.806 52.854
37327 93.174 -79.802 52.850
37328 93.197 -79.794 52.847
37329 93.220 -79.786 52.843
37330 93.242 -79.778 52.839
37331 93.266 -79.770 52.836
37332 93.289 -79.762 52.834
37333 93.311 -79.755 52.834
37334 93.335 -79.746 52.834
37335 93.358 -79.739 52.834
37336 93.381 -79.730 52.834
37337 93.404 -79.720 52.832
37338 93.431 -79.709 52.830
37339 93.448 -79.693 52.826
37340 93.463 -79.677 52.821
37341 93.478 -79.661 52.814
37342 93.493 -79.645 52.806
37343 93.508 -79.629 52.796
37344 93.523 -79.613 52.787
37345 93.538 -79.597 52.778
37346 93.553 -79.581 52.768
37347 93.568 -79.564 52.758
37348 93.582 -79.548 52.748
37349 93.597 -79.532 52.740
37350 93.612 -79.516 52.732
37351 93.627 -79.500 52.725
37352 93.642 -79.484 52.717
37353 93.657 -79.468 52.708
37354 93.672 -79.452 52.699
37355 93.687 -79.435 52.690
37356 93.702 -79.419 52.681
37357 93.713 -79.411 52.673
37358 93.728 -79.404 52.665
37359 93.744 -79.398 52.658
37360 93.759 -79.395 52.652
37361 93.773 -79.393 52.649
37362 93.788 -79.390 52.647
37363 93.803 -79.387 52.648
37364 93.818 -79.385 52.649
37365 93.833 -79.382 52.649
37366 93.848 -79.379 52.647
37367 93.862 -79.375 52.645
37368 93.877 -79.373 52.643
37369 93.892 -79.370 52.641
37370 93.907 -79.367 52.638
37371 93.918 -79.364 52.636
37372 93.922 -79.360 52.634
37373 93.926 -79.357 52.628
37374 93.931 -79.353 52.619
37375 93.929 -79.352 52.609
37376 93.921 -79.352 52.597
37377 93.913 -79.353 52.585
37378 93.905 -79.353 52.571
37379 93.901 -79.356 52.555
37380 93.902 -79.361 52.539
37381 93.903 -79.366 52.525
37382 93.904 -79.372 52.512
37383 93.905 -79.378 52.499
37384 93.906 -79.383 52.487
37385 93.907 -79.389 52.473
37386 93.908 -79.394 52.460
37387 93.909 -79.400 52.447
37388 93.910 -79.406 52.434
37389 93.914 -79.410 52.420
37390 93.923 -79.412 52.408
37391 93.932 -79.413 52.394
37392 93.941 -79.416 52.381
37393 93.951 -79.417 52.368
37394 93.960 -79.419 52.356
37395 93.969 -79.421 52.342
37396 93.978 -79.422 52.327
37397 93.987 -79.425 52.311
37398 93.996 -79.426 52.294
37399 94.006 -79.428 52.275
37400 94.015 -79.430 52.256
37401 94.024 -79.432 52.238
37402 94.033 -79.434 52.221
37403 94.071 -79.402 52.207
37404 94.115 -79.353 52.193
37405 94.084 -79.346 52.178
37406 94.085 -79.346 52.161
37407 94.116 -79.328 52.142
37408 94.133 -79.295 52.124
37409 94.135 -79.316 52.106
37410 94.140 -79.322 52.088
37411 94.149 -79.313 52.072
37412 94.157 -79.303 52.056
37413 94.166 -79.293 52.037
37414 94.175 -79.283 52.019
37415 94.183 -79.273 52.002
37416 94.191 -79.259 51.984
37417 94.200 -79.234 51.964
37418 94.207 -79.234 51.947
37419 94.214 -79.244 51.932
37420 94.221 -79.255 51.917
37421 94.228 -79.265 51.904
37422 94.234 -79.265 51.892
37423 94.239 -79.264 51.881
37424 94.244 -79.264 51.868
37425 94.249 -79.263 51.855
37426 94.254 -79.263 51.841
37427 94.260 -79.263 51.822
37428 94.265 -79.262 51.805
37429 94.270 -79.262 51.790
37430 94.275 -79.261 51.772
37431 94.281 -79.259 51.754
37432 94.289 -79.254 51.733
37433 94.297 -79.250 51.711
37434 94.278 -79.259 51.686
37435 94.260 -79.269 51.660
37436 94.242 -79.278 51.633
37437 94.223 -79.288 51.605
37438 94.204 -79.298 51.575
37439 94.206 -79.297 51.543
37440 94.213 -79.293 51.510
37441 94.222 -79.287 51.474
37442 94.232 -79.280 51.439
37443 94.242 -79.273 51.407
37444 94.252 -79.267 51.372
37445 94.261 -79.280 51.333
37446 94.270 -79.323 51.291
37447 94.282 -79.341 51.247
37448 94.295 -79.354 51.204
37449 94.309 -79.367 51.162
37450 94.322 -79.380 51.120
37451 94.335 -79.394 51.078
37452 94.348 -79.407 51.036
37453 94.361 -79.419 50.995
37454 94.374 -79.432 50.954
37455 94.388 -79.445 50.913
37456 94.401 -79.458 50.873
37457 94.414 -79.471 50.833
38013 92.952 -79.908 53.102
38014 92.968 -79.901 53.078
38015 92.984 -79.892 53.054
38016 93.000 -79.885 53.031
38017 93.016 -79.877 53.009
38018 93.032 -79.868 52.989
38019 93.048 -79.861 52.969
38020 93.064 -79.852 52.948
38021 93.080 -79.845 52.930
38022 93.096 -79.836 52.914
38023 93.114 -79.824 52.899
38024 93.132 -79.811 52.886
38025 93.150 -79.797 52.875
38026 93.165 -79.786 52.865
38027 93.171 -79.783 52.858
38028 93.179 -79.779 52.853
38029 93.192 -79.773 52.850
38030 93.215 -79.766 52.847
38031 93.238 -79.758 52.843
38032 93.261 -79.750 52.840
38033 93.284 -79.742 52.838
38034 93.307 -79.734 52.838
38035 93.330 -79.726 52.839
38036 93.353 -79.718 52.839
38037 93.376 -79.710 52.838
38038 93.399 -79.702 52.836
38039 93.422 -79.694 52.833
38040 93.434 -79.685 52.829
38041 93.448 -79.669 52.823
38042 93.463 -79.654 52.816
38043 93.477 -79.638 52.808
38044 93.492 -79.622 52.797
38045 93.506 -79.606 52.787
38046 93.521 -79.590 52.777
38047 93.536 -79.575 52.767
38048 93.550 -79.559 52.757
38049 93.565 -79.543 52.747
38050 93.579 -79.527 52.740
38051 93.594 -79.511 52.734
38052 93.609 -79.496 52.727
38053 93.623 -79.480 52.720
38054 93.638 -79.464 52.710
38055 93.652 -79.448 52.701
38056 93.667 -79.432 52.692
38057 93.682 -79.417 52.683
38058 93.707 -79.404 52.673
38059 93.725 -79.399 52.664
38060 93.740 -79.397 52.656
38061 93.755 -79.394 52.650
38062 93.769 -79.391 52.646
38063 93.784 -79.388 52.644
38064 93.799 -79.385 52.646
38065 93.814 -79.382 52.648
38066 93.829 -79.380 52.648
38067 93.844 -79.377 52.647
38068 93.858 -79.374 52.645
38069 93.873 -79.372 52.642
38070 93.888 -79.369 52.640
38071 93.893 -79.365 52.639
38072 93.898 -79.361 52.638
38073 93.902 -79.358 52.636
38074 93.907 -79.354 52.630
38075 93.911 -79.350 52.622
38076 93.912 -79.348 52.613
38077 93.904 -79.348 52.601
38078 93.896 -79.349 52.588
38079 93.888 -79.350 52.574
38080 93.880 -79.350 52.558
38081 93.872 -79.350 52.543
38082 93.866 -79.352 52.529
38083 93.867 -79.357 52.516
38084 93.868 -79.363 52.504
38085 93.869 -79.369 52.491
38086 93.870 -79.374 52.478
38087 93.875 -79.378 52.466
38088 93.885 -79.379 52.453
38089 93.894 -79.382 52.439
38090 93.903 -79.383 52.425
38091 93.912 -79.385 52.412
38092 93.922 -79.387 52.398
38093 93.931 -79.388 52.383
38094 93.940 -79.391 52.369
38095 93.949 -79.392 52.355
38096 93.959 -79.394 52.340
38097 93.968 -79.396 52.323
38098 93.977 -79.397 52.306
38099 93.986 -79.400 52.288
38100 93.995 -79.401 52.270
38101 94.004 -79.404 52.252
38102 94.014 -79.405 52.234
38103 94.047 -79.379 52.217
38104 94.101 -79.328 52.200
38105 94.109 -79.325 52.186
38106 94.094 -79.332 52.170
38107 94.080 -79.338 52.152
38108 94.094 -79.334 52.135
38109 94.117 -79.293 52.117
38110 94.133 -79.263 52.100
38111 94.136 -79.278 52.084
38112 94.144 -79.269 52.068
38113 94.153 -79.259 52.051
38114 94.161 -79.249 52.032
38115 94.170 -79.239 52.014
38116 94.179 -79.229 51.997
38117 94.187 -79.219 51.980
38118 94.195 -79.219 51.960
38119 94.202 -79.230 51.943
38120 94.209 -79.240 51.928
38121 94.217 -79.250 51.914
38122 94.224 -79.260 51.902
38123 94.229 -79.264 51.891
38124 94.235 -79.264 51.881
38125 94.240 -79.263 51.868
38126 94.246 -79.263 51.855
38127 94.251 -79.263 51.842
38128 94.256 -79.262 51.824
38129 94.261 -79.262 51.808
38130 94.266 -79.261 51.792
38131 94.272 -79.260 51.774
38132 94.280 -79.255 51.755
38133 94.287 -79.250 51.735
38134 94.295 -79.245 51.713
38135 94.303 -79.241 51.689
38136 94.290 -79.247 51.663
38137 94.272 -79.257 51.636
38138 94.253 -79.267 51.609
38139 94.235 -79.276 51.579
38140 94.226 -79.281 51.548
38141 94.229 -79.280 51.516
38142 94.231 -79.278 51.482
38143 94.234 -79.278 51.448
38144 94.237 -79.276 51.417
38145 94.241 -79.275 51.382
38146 94.250 -79.268 51.343
38147 94.261 -79.289 51.302
38148 94.274 -79.302 51.259
38149 94.287 -79.315 51.215
38150 94.300 -79.328 51.172
38151 94.314 -79.341 51.131
38152 94.327 -79.354 51.089
38153 94.340 -79.367 51.047
38154 94.353 -79.380 51.005
38155 94.366 -79.393 50.964
38156 94.380 -79.406 50.922
38157 94.393 -79.419 50.881
38158 94.406 -79.432 50.841
38716 92.980 -79.876 53.072
38717 92.996 -79.867 53.049
38718 93.012 -79.860 53.027
38719 93.028 -79.852 53.006
38720 93.044 -79.843 52.985
38721 93.060 -79.836 52.965
38722 93.076 -79.827 52.946
38723 93.092 -79.820 52.929
38724 93.109 -79.811 52.913
38725 93.126 -79.798 52.899
38726 93.144 -79.785 52.887
38727 93.162 -79.770 52.875
38728 93.179 -79.759 52.866
38729 93.185 -79.755 52.859
38730 93.192 -79.751 52.853
38731 93.210 -79.745 52.851
38732 93.233 -79.737 52.848
38733 93.256 -79.729 52.846
38734 93.279 -79.721 52.844
38735 93.302 -79.714 52.844
38736 93.325 -79.705 52.844
38737 93.348 -79.698 52.844
38738 93.371 -79.689 52.842
38739 93.394 -79.682 52.840
38740 93.416 -79.675 52.837
38741 93.436 -79.667 52.833
38742 93.444 -79.657 52.826
38743 93.452 -79.645 52.818
38744 93.461 -79.635 52.810
38745 93.469 -79.623 52.799
38746 93.482 -79.608 52.789
38747 93.497 -79.592 52.778
38748 93.512 -79.576 52.768
38749 93.526 -79.561 52.758
38750 93.541 -79.545 52.748
38751 93.556 -79.529 52.742
38752 93.570 -79.513 52.737
38753 93.585 -79.498 52.731
38754 93.600 -79.482 52.724
38755 93.614 -79.466 52.715
38756 93.629 -79.451 52.706
38757 93.643 -79.435 52.696
38758 93.665 -79.421 52.687
38759 93.693 -79.410 52.677
38760 93.721 -79.397 52.667
38761 93.736 -79.395 52.657
38762 93.751 -79.392 52.651
38763 93.766 -79.389 52.645
38764 93.780 -79.387 52.641
38765 93.795 -79.384 52.643
38766 93.810 -79.381 52.645
38767 93.825 -79.379 52.645
38768 93.840 -79.375 52.643
38769 93.855 -79.372 52.642
38770 93.864 -79.369 52.641
38771 93.868 -79.366 52.638
38772 93.873 -79.362 52.637
38773 93.877 -79.358 52.636
38774 93.882 -79.355 52.636
38775 93.887 -79.351 52.630
38776 93.891 -79.347 52.623
38777 93.894 -79.344 52.614
38778 93.886 -79.344 52.602
38779 93.879 -79.345 52.589
38780 93.871 -79.345 52.576
38781 93.864 -79.346 52.560
38782 93.857 -79.346 52.548
38783 93.849 -79.346 52.535
38784 93.842 -79.347 52.522
38785 93.840 -79.346 52.511
38786 93.848 -79.347 52.498
38787 93.856 -79.349 52.485
38788 93.865 -79.351 52.473
38789 93.874 -79.353 52.460
38790 93.884 -79.355 52.445
38791 93.893 -79.357 52.430
38792 93.902 -79.358 52.415
38793 93.911 -79.360 52.400
38794 93.921 -79.362 52.384
38795 93.930 -79.364 52.368
38796 93.939 -79.366 52.353
38797 93.948 -79.367 52.337
38798 93.958 -79.369 52.320
38799 93.967 -79.371 52.303
38800 93.976 -79.373 52.285
38801 93.985 -79.375 52.267
38802 93.994 -79.376 52.250
38803 94.023 -79.357 52.232
38804 94.053 -79.338 52.215
38805 94.082 -79.335 52.198
38806 94.087 -79.318 52.181
38807 94.087 -79.319 52.164
38808 94.090 -79.325 52.146
38809 94.076 -79.332 52.128
38810 94.096 -79.303 52.111
38811 94.117 -79.256 52.093
38812 94.132 -79.232 52.077
38813 94.140 -79.225 52.061
38814 94.148 -79.216 52.044
38815 94.157 -79.212 52.025
38816 94.165 -79.222 52.007
38817 94.174 -79.231 51.989
38818 94.180 -79.228 51.973
38819 94.184 -79.222 51.955
38820 94.198 -79.225 51.939
38821 94.205 -79.236 51.924
38822 94.212 -79.246 51.912
38823 94.219 -79.256 51.900
38824 94.226 -79.264 51.890
38825 94.231 -79.263 51.881
38826 94.237 -79.263 51.869
38827 94.242 -79.263 51.855
38828 94.247 -79.262 51.841
38829 94.252 -79.262 51.825
38830 94.258 -79.261 51.809
38831 94.263 -79.261 51.793
38832 94.270 -79.256 51.774
38833 94.278 -79.252 51.754
38834 94.286 -79.247 51.734
38835 94.293 -79.241 51.713
38836 94.301 -79.237 51.690
38837 94.310 -79.231 51.665
38838 94.302 -79.235 51.638
38839 94.283 -79.245 51.610
38840 94.265 -79.255 51.581
38841 94.247 -79.265 51.551
38842 94.249 -79.263 51.520
38843 94.252 -79.263 51.487
38844 94.255 -79.261 51.455
38845 94.257 -79.260 51.424
38846 94.260 -79.259 51.389
38847 94.258 -79.262 51.351
38848 94.260 -79.259 51.311
38849 94.269 -79.266 51.269
38850 94.279 -79.275 51.226
38851 94.292 -79.289 51.182
38852 94.305 -79.302 51.141
38853 94.318 -79.315 51.099
38854 94.331 -79.328 51.057
38855 94.345 -79.341 51.015
38856 94.358 -79.353 50.973
38857 94.371 -79.366 50.932
38858 94.384 -79.380 50.890
38859 94.397 -79.393 50.849
39418 92.992 -79.851 53.069
39419 93.008 -79.842 53.046
39420 93.024 -79.835 53.024
39421 93.040 -79.827 53.003
39422 93.056 -79.818 52.982
39423 93.072 -79.811 52.963
39424 93.088 -79.802 52.945
39425 93.105 -79.795 52.929
39426 93.121 -79.786 52.914
39427 93.139 -79.772 52.900
39428 93.157 -79.758 52.888
39429 93.175 -79.745 52.877
39430 93.192 -79.732 52.868
39431 93.199 -79.728 52.862
39432 93.206 -79.724 52.858
39433 93.228 -79.717 52.856
39434 93.251 -79.708 52.854
39435 93.274 -79.701 52.852
39436 93.297 -79.692 52.850
39437 93.320 -79.685 52.848
39438 93.343 -79.676 52.847
39439 93.366 -79.669 52.845
39440 93.389 -79.661 52.842
39441 93.411 -79.655 52.838
39442 93.433 -79.648 52.832
39443 93.447 -79.640 52.826
39444 93.455 -79.629 52.819
39445 93.463 -79.618 52.811
39446 93.471 -79.607 52.801
39447 93.479 -79.596 52.791
39448 93.487 -79.585 52.780
39449 93.495 -79.574 52.771
39450 93.503 -79.563 52.760
39451 93.517 -79.548 52.750
39452 93.532 -79.532 52.745
39453 93.546 -79.516 52.741
39454 93.561 -79.500 52.736
39455 93.575 -79.485 52.729
39456 93.590 -79.469 52.720
39457 93.605 -79.453 52.711
39458 93.623 -79.438 52.701
39459 93.651 -79.426 52.691
39460 93.679 -79.414 52.681
39461 93.707 -79.403 52.671
39462 93.732 -79.393 52.660
39463 93.747 -79.391 52.652
39464 93.762 -79.388 52.646
39465 93.776 -79.385 52.640
39466 93.791 -79.382 52.641
39467 93.806 -79.379 52.642
39468 93.821 -79.376 52.642
39469 93.835 -79.374 52.640
39470 93.839 -79.370 52.639
39471 93.844 -79.366 52.638
39472 93.848 -79.363 52.637
39473 93.853 -79.359 52.636
39474 93.857 -79.356 52.635
39475 93.865 -79.350 52.634
39476 93.877 -79.343 52.629
39477 93.890 -79.335 52.622
39478 93.896 -79.332 52.613
39479 93.895 -79.333 52.601
39480 93.894 -79.334 52.590
39481 93.888 -79.335 52.577
39482 93.880 -79.335 52.563
39483 93.873 -79.335 52.552
39484 93.865 -79.335 52.541
39485 93.858 -79.335 52.531
39486 93.860 -79.337 52.519
39487 93.863 -79.339 52.507
39488 93.870 -79.336 52.494
39489 93.878 -79.332 52.481
39490 93.887 -79.330 52.467
39491 93.894 -79.332 52.452
39492 93.902 -79.333 52.435
39493 93.910 -79.335 52.418
39494 93.918 -79.337 52.401
39495 93.925 -79.338 52.383
39496 93.933 -79.340 52.366
39497 93.941 -79.341 52.350
39498 93.949 -79.343 52.334
39499 93.957 -79.344 52.318
39500 93.965 -79.346 52.301
39501 93.972 -79.347 52.283
39502 93.980 -79.349 52.264
39503 93.987 -79.355 52.248
39504 94.015 -79.351 52.230
39505 94.044 -79.347 52.213
39506 94.073 -79.344 52.196
39507 94.068 -79.321 52.177
39508 94.059 -79.306 52.159
39509 94.062 -79.313 52.139
39510 94.065 -79.319 52.122
39511 94.078 -79.320 52.104
39512 94.100 -79.275 52.086
39513 94.122 -79.229 52.070
39514 94.135 -79.218 52.054
39515 94.143 -79.228 52.037
39516 94.151 -79.237 52.018
39517 94.159 -79.246 52.000
39518 94.166 -79.237 51.982
39519 94.173 -79.229 51.967
39520 94.175 -79.226 51.950
39521 94.185 -79.227 51.935
39522 94.201 -79.231 51.921
39523 94.208 -79.242 51.910
39524 94.215 -79.252 51.900
39525 94.222 -79.263 51.890
39526 94.227 -79.263 51.880
39527 94.232 -79.263 51.868
39528 94.238 -79.262 51.855
39529 94.243 -79.262 51.842
39530 94.249 -79.261 51.826
39531 94.254 -79.261 51.809
39532 94.260 -79.258 51.792
39533 94.268 -79.253 51.773
39534 94.276 -79.248 51.752
39535 94.284 -79.243 51.731
39536 94.292 -79.238 51.711
39537 94.299 -79.233 51.688
39538 94.308 -79.228 51.663
39539 94.316 -79.223 51.637
39540 94.314 -79.224 51.609
39541 94.295 -79.234 51.580
39542 94.276 -79.243 51.549
39543 94.269 -79.247 51.520
39544 94.272 -79.246 51.490
39545 94.275 -79.244 51.459
39546 94.277 -79.244 51.428
39547 94.280 -79.242 51.394
39548 94.254 -79.263 51.357
39549 94.262 -79.253 51.318
39550 94.277 -79.247 51.278
39551 94.285 -79.253 51.236
39552 94.294 -79.261 51.193
39553 94.302 -79.269 51.150
39554 94.311 -79.276 51.108
39555 94.323 -79.288 51.066
39556 94.336 -79.301 51.024
39557 94.349 -79.314 50.983
39558 94.362 -79.328 50.941
39559 94.376 -79.341 50.899
39560 94.389 -79.353 50.857
39561 94.402 -79.366 50.817
40120 93.004 -79.826 53.066
40121 93.020 -79.817 53.043
40122 93.036 -79.810 53.022
40123 93.052 -79.802 53.000
40124 93.068 -79.793 52.980
40125 93.084 -79.786 52.962
40126 93.100 -79.777 52.946
40127 93.116 -79.770 52.929
40128 93.133 -79.760 52.915
40129 93.151 -79.746 52.902
40130 93.169 -79.732 52.890
40131 93.187 -79.718 52.880
40132 93.206 -79.704 52.872
40133 93.212 -79.701 52.868
40134 93.223 -79.696 52.864
40135 93.246 -79.688 52.862
40136 93.269 -79.680 52.859
40137 93.292 -79.672 52.857
40138 93.315 -79.664 52.854
40139 93.338 -79.656 52.851
40140 93.361 -79.648 52.848
40141 93.383 -79.642 52.844
40142 93.405 -79.636 52.839
40143 93.427 -79.629 52.834
40144 93.450 -79.623 52.827
40145 93.458 -79.612 52.820
40146 93.466 -79.601 52.812
40147 93.474 -79.590 52.803
40148 93.482 -79.579 52.794
40149 93.490 -79.568 52.783
40150 93.498 -79.557 52.774
40151 93.506 -79.546 52.765
40152 93.514 -79.535 52.757
40153 93.521 -79.524 52.751
40154 93.530 -79.513 52.746
40155 93.538 -79.502 52.741
40156 93.552 -79.486 52.735
40157 93.566 -79.471 52.726
40158 93.581 -79.455 52.716
40159 93.609 -79.443 52.705
40160 93.637 -79.432 52.695
40161 93.665 -79.419 52.685
40162 93.693 -79.408 52.674
40163 93.721 -79.396 52.663
40164 93.743 -79.389 52.654
40165 93.758 -79.386 52.647
40166 93.772 -79.383 52.642
40167 93.787 -79.381 52.639
40168 93.802 -79.378 52.639
40169 93.810 -79.375 52.639
40170 93.815 -79.371 52.637
40171 93.819 -79.367 52.637
40172 93.824 -79.363 52.636
40173 93.835 -79.357 52.636
40174 93.847 -79.349 52.634
40175 93.860 -79.342 52.633
40176 93.872 -79.335 52.633
40177 93.885 -79.327 52.628
40178 93.897 -79.319 52.621
40179 93.898 -79.319 52.612
40180 93.896 -79.320 52.601
40181 93.895 -79.322 52.590
40182 93.894 -79.322 52.579
40183 93.893 -79.323 52.567
40184 93.889 -79.324 52.556
40185 93.882 -79.325 52.546
40186 93.877 -79.325 52.536
40187 93.880 -79.327 52.524
40188 93.884 -79.328 52.511
40189 93.888 -79.330 52.497
40190 93.893 -79.329 52.483
40191 93.901 -79.325 52.468
40192 93.909 -79.322 52.452
40193 93.917 -79.319 52.434
40194 93.925 -79.315 52.416
40195 93.933 -79.314 52.399
40196 93.941 -79.316 52.381
40197 93.948 -79.317 52.365
40198 93.957 -79.319 52.349
40199 93.964 -79.321 52.332
40200 93.972 -79.322 52.316
40201 93.980 -79.324 52.299
40202 93.988 -79.325 52.281
40203 94.000 -79.329 52.262
40204 94.006 -79.342 52.246
40205 94.006 -79.360 52.229
40206 94.035 -79.357 52.211
40207 94.063 -79.353 52.194
40208 94.048 -79.325 52.175
40209 94.031 -79.294 52.155
40210 94.034 -79.300 52.134
40211 94.051 -79.301 52.115
40212 94.071 -79.301 52.097
40213 94.091 -79.300 52.079
40214 94.113 -79.269 52.064
40215 94.129 -79.243 52.047
40216 94.137 -79.253 52.030
40217 94.145 -79.256 52.012
40218 94.151 -79.247 51.993
40219 94.158 -79.239 51.975
40220 94.164 -79.232 51.961
40221 94.165 -79.230 51.946
40222 94.172 -79.229 51.933
40223 94.189 -79.232 51.921
40224 94.203 -79.237 51.911
40225 94.210 -79.247 51.901
40226 94.218 -79.258 51.890
40227 94.224 -79.263 51.879
40228 94.229 -79.262 51.867
40229 94.235 -79.262 51.855
40230 94.240 -79.261 51.842
40231 94.245 -79.261 51.826
40232 94.251 -79.259 51.809
40233 94.259 -79.254 51.791
40234 94.266 -79.249 51.772
40235 94.274 -79.244 51.752
40236 94.282 -79.239 51.732
40237 94.290 -79.234 51.712
40238 94.298 -79.229 51.688
40239 94.306 -79.225 51.663
40240 94.314 -79.219 51.637
40241 94.322 -79.215 51.609
40242 94.325 -79.212 51.581
40243 94.307 -79.222 51.552
40244 94.290 -79.231 51.524
40245 94.292 -79.229 51.494
40246 94.295 -79.228 51.464
40247 94.297 -79.227 51.432
40248 94.266 -79.257 51.397
40249 94.255 -79.259 51.361
40250 94.260 -79.251 51.324
40251 94.282 -79.238 51.285
40252 94.294 -79.234 51.244
40253 94.302 -79.241 51.203
40254 94.311 -79.248 51.161
40255 94.319 -79.256 51.118
40256 94.327 -79.263 51.075
40257 94.335 -79.271 51.034
40258 94.344 -79.278 50.992
40259 94.354 -79.288 50.949
40260 94.367 -79.301 50.907
40261 94.381 -79.314 50.866
40262 94.394 -79.327 50.825
40822 93.016 -79.801 53.063
40823 93.032 -79.792 53.041
40824 93.048 -79.785 53.020
40825 93.064 -79.776 52.999
40826 93.080 -79.768 52.980
40827 93.096 -79.761 52.963
40828 93.112 -79.752 52.947
40829 93.128 -79.745 52.932
40830 93.145 -79.733 52.918
40831 93.163 -79.720 52.906
40832 93.182 -79.706 52.895
40833 93.200 -79.692 52.885
40834 93.218 -79.678 52.880
40835 93.226 -79.673 52.875
40836 93.241 -79.667 52.871
40837 93.264 -79.660 52.866
40838 93.287 -79.651 52.863
40839 93.310 -79.644 52.861
40840 93.333 -79.636 52.857
40841 93.356 -79.629 52.853
40842 93.378 -79.623 52.847
40843 93.400 -79.616 52.841
40844 93.422 -79.610 52.836
40845 93.444 -79.604 52.829
40846 93.461 -79.595 52.821
40847 93.469 -79.584 52.814
40848 93.477 -79.573 52.805
40849 93.485 -79.562 52.796
40850 93.493 -79.551 52.786
40851 93.500 -79.540 52.778
40852 93.508 -79.529 52.769
40853 93.516 -79.518 52.762
40854 93.524 -79.507 52.755
40855 93.533 -79.496 52.749
40856 93.540 -79.485 52.744
40857 93.549 -79.475 52.739
40858 93.559 -79.467 52.730
40859 93.572 -79.463 52.720
40860 93.596 -79.450 52.708
40861 93.623 -79.436 52.696
40862 93.651 -79.425 52.685
40863 93.679 -79.413 52.674
40864 93.707 -79.401 52.665
40865 93.735 -79.389 52.657
40866 93.754 -79.385 52.651
40867 93.768 -79.382 52.646
40868 93.781 -79.379 52.641
40869 93.786 -79.375 52.638
40870 93.791 -79.371 52.637
40871 93.804 -79.363 52.635
40872 93.817 -79.356 52.635
40873 93.829 -79.348 52.635
40874 93.842 -79.341 52.634
40875 93.854 -79.334 52.634
40876 93.867 -79.326 52.633
40877 93.880 -79.319 52.632
40878 93.892 -79.311 52.627
40879 93.901 -79.306 52.621
40880 93.899 -79.307 52.612
40881 93.898 -79.308 52.601
40882 93.897 -79.309 52.591
40883 93.895 -79.310 52.581
40884 93.894 -79.311 52.570
40885 93.893 -79.312 52.560
40886 93.890 -79.315 52.550
40887 93.896 -79.316 52.540
40888 93.902 -79.316 52.528
40889 93.905 -79.319 52.514
40890 93.908 -79.320 52.498
40891 93.912 -79.322 52.484
40892 93.916 -79.323 52.468
40893 93.924 -79.319 52.451
40894 93.932 -79.316 52.432
40895 93.940 -79.312 52.416
40896 93.948 -79.308 52.399
40897 93.956 -79.304 52.381
40898 93.964 -79.300 52.366
40899 93.972 -79.297 52.349
40900 93.980 -79.298 52.332
40901 93.987 -79.300 52.315
40902 93.995 -79.302 52.297
40903 94.009 -79.307 52.280
40904 94.025 -79.313 52.262
40905 94.026 -79.329 52.245
40906 94.017 -79.353 52.228
40907 94.026 -79.366 52.211
40908 94.047 -79.358 52.193
40909 94.029 -79.328 52.173
40910 94.013 -79.297 52.153
40911 94.026 -79.287 52.132
40912 94.045 -79.284 52.113
40913 94.064 -79.281 52.094
40914 94.084 -79.281 52.075
40915 94.104 -79.281 52.058
40916 94.124 -79.268 52.042
40917 94.131 -79.266 52.025
40918 94.137 -79.257 52.006
40919 94.143 -79.249 51.988
40920 94.150 -79.241 51.971
40921 94.154 -79.235 51.958
40922 94.156 -79.233 51.946
40923 94.158 -79.231 51.936
40924 94.175 -79.234 51.926
40925 94.192 -79.238 51.914
40926 94.206 -79.244 51.902
40927 94.213 -79.253 51.891
40928 94.220 -79.262 51.879
40929 94.225 -79.262 51.867
40930 94.230 -79.261 51.855
40931 94.236 -79.261 51.841
40932 94.241 -79.260 51.826
40933 94.249 -79.255 51.810
40934 94.257 -79.250 51.792
40935 94.265 -79.245 51.772
40936 94.272 -79.241 51.752
40937 94.280 -79.235 51.734
40938 94.288 -79.231 51.715
40939 94.296 -79.225 51.691
40940 94.304 -79.221 51.665
40941 94.312 -79.216 51.639
40942 94.320 -79.211 51.612
40943 94.328 -79.206 51.584
40944 94.335 -79.201 51.557
40945 94.318 -79.209 51.530
40946 94.313 -79.213 51.500
40947 94.315 -79.212 51.468
40948 94.296 -79.231 51.435
40949 94.261 -79.261 51.400
40950 94.255 -79.256 51.364
40951 94.258 -79.248 51.329
40952 94.285 -79.231 51.291
40953 94.299 -79.225 51.252
40954 94.311 -79.222 51.211
40955 94.319 -79.228 51.169
40956 94.327 -79.235 51.127
40957 94.335 -79.243 51.085
40958 94.344 -79.250 51.043
40959 94.352 -79.258 51.001
40960 94.360 -79.266 50.958
40961 94.368 -79.273 50.916
40962 94.377 -79.280 50.875
40963 94.385 -79.288 50.834
41524 93.028 -79.776 53.061
41525 93.044 -79.767 53.040
41526 93.060 -79.760 53.019
41527 93.076 -79.751 53.000
41528 93.092 -79.744 52.982
41529 93.108 -79.736 52.965
41530 93.124 -79.727 52.949
41531 93.140 -79.720 52.935
41532 93.157 -79.708 52.921
41533 93.176 -79.693 52.910
41534 93.194 -79.679 52.900
41535 93.212 -79.666 52.893
41536 93.230 -79.652 52.887
41537 93.240 -79.646 52.881
41538 93.259 -79.639 52.876
41539 93.282 -79.631 52.872
41540 93.305 -79.623 52.868
41541 93.328 -79.616 52.862
41542 93.350 -79.609 52.856
41543 93.372 -79.603 52.850
41544 93.395 -79.597 52.843
41545 93.417 -79.590 52.838
41546 93.439 -79.584 52.831
41547 93.461 -79.577 52.824
41548 93.472 -79.567 52.815
41549 93.480 -79.556 52.807
41550 93.487 -79.545 52.799
41551 93.495 -79.534 52.790
41552 93.503 -79.523 52.782
41553 93.511 -79.512 52.773
41554 93.521 -79.504 52.765
41555 93.531 -79.497 52.758
41556 93.541 -79.489 52.752
41557 93.551 -79.482 52.747
41558 93.561 -79.474 52.741
41559 93.571 -79.467 52.732
41560 93.587 -79.464 52.722
41561 93.604 -79.461 52.711
41562 93.621 -79.457 52.697
41563 93.647 -79.441 52.686
41564 93.672 -79.426 52.676
41565 93.698 -79.411 52.668
41566 93.723 -79.396 52.662
41567 93.750 -79.383 52.656
41568 93.761 -79.377 52.650
41569 93.774 -79.369 52.644
41570 93.786 -79.363 52.639
41571 93.799 -79.355 52.636
41572 93.811 -79.347 52.636
41573 93.824 -79.340 52.636
41574 93.836 -79.332 52.636
41575 93.849 -79.325 52.635
41576 93.861 -79.318 52.634
41577 93.874 -79.310 52.634
41578 93.887 -79.303 52.634
41579 93.899 -79.295 52.630
41580 93.902 -79.294 52.623
41581 93.901 -79.294 52.614
41582 93.898 -79.297 52.605
41583 93.895 -79.300 52.595
41584 93.891 -79.303 52.584
41585 93.888 -79.307 52.572
41586 93.885 -79.310 52.563
41587 93.888 -79.310 52.553
41588 93.896 -79.309 52.543
41589 93.905 -79.310 52.530
41590 93.917 -79.310 52.515
41591 93.929 -79.310 52.500
41592 93.933 -79.312 52.484
41593 93.936 -79.313 52.467
41594 93.939 -79.315 52.450
41595 93.946 -79.313 52.432
41596 93.955 -79.309 52.415
41597 93.963 -79.306 52.399
41598 93.971 -79.302 52.383
41599 93.978 -79.298 52.368
41600 93.986 -79.294 52.351
41601 93.994 -79.291 52.332
41602 94.005 -79.290 52.314
41603 94.019 -79.294 52.296
41604 94.034 -79.297 52.279
41605 94.047 -79.300 52.261
41606 94.045 -79.315 52.244
41607 94.038 -79.341 52.228
41608 94.029 -79.365 52.211
41609 94.028 -79.361 52.192
41610 94.019 -79.328 52.173
41611 94.011 -79.296 52.153
41612 94.026 -79.289 52.131
41613 94.046 -79.286 52.111
41614 94.065 -79.284 52.092
41615 94.084 -79.281 52.073
41616 94.104 -79.278 52.057
41617 94.112 -79.268 52.040
41618 94.123 -79.267 52.024
41619 94.129 -79.259 52.005
41620 94.136 -79.250 51.987
41621 94.142 -79.243 51.972
41622 94.144 -79.239 51.960
41623 94.146 -79.237 51.949
41624 94.148 -79.234 51.940
41625 94.161 -79.237 51.931
41626 94.179 -79.240 51.919
41627 94.196 -79.244 51.906
41628 94.209 -79.250 51.893
41629 94.216 -79.259 51.879
41630 94.221 -79.261 51.867
41631 94.227 -79.261 51.855
41632 94.232 -79.260 51.843
41633 94.240 -79.256 51.827
41634 94.247 -79.251 51.811
41635 94.255 -79.247 51.794
41636 94.263 -79.241 51.774
41637 94.271 -79.237 51.755
41638 94.278 -79.231 51.737
41639 94.286 -79.227 51.718
41640 94.294 -79.222 51.694
41641 94.302 -79.217 51.669
41642 94.310 -79.212 51.642
41643 94.318 -79.207 51.616
41644 94.326 -79.202 51.590
41645 94.333 -79.197 51.565
41646 94.341 -79.192 51.536
41647 94.333 -79.197 51.506
41648 94.327 -79.203 51.473
41649 94.274 -79.253 51.438
41650 94.261 -79.257 51.403
41651 94.256 -79.252 51.368
41652 94.255 -79.245 51.333
41653 94.282 -79.228 51.296
41654 94.305 -79.216 51.258
41655 94.316 -79.212 51.219
41656 94.327 -79.209 51.178
41657 94.336 -79.216 51.136
41658 94.344 -79.222 51.095
41659 94.352 -79.230 51.052
41660 94.360 -79.237 51.010
41661 94.368 -79.245 50.968
41662 94.377 -79.253 50.925
41663 94.385 -79.260 50.884
41664 94.394 -79.268 50.843
41724 95.166 -80.028 48.741
41725 95.186 -80.034 48.705
41726 95.210 -80.035 48.668
41727 95.233 -80.037 48.631
41728 95.257 -80.038 48.595
41729 95.280 -80.040 48.560
41730 95.304 -80.041 48.524
41731 95.328 -80.043 48.489
42226 93.040 -79.751 53.061
42227 93.056 -79.742 53.040
42228 93.072 -79.735 53.020
42229 93.088 -79.726 53.001
42230 93.104 -79.719 52.984
42231 93.120 -79.711 52.967
42232 93.136 -79.702 52.952
42233 93.152 -79.695 52.938
42234 93.170 -79.681 52.926
42235 93.188 -79.667 52.915
42236 93.206 -79.654 52.906
42237 93.224 -79.639 52.899
42238 93.242 -79.626 52.893
42239 93.255 -79.618 52.887
42240 93.277 -79.610 52.881
42241 93.300 -79.602 52.875
42242 93.323 -79.596 52.868
42243 93.345 -79.590 52.861
42244 93.367 -79.583 52.855
42245 93.389 -79.577 52.848
42246 93.411 -79.570 52.843
42247 93.433 -79.564 52.837
42248 93.455 -79.558 52.829
42249 93.474 -79.551 52.822
42250 93.484 -79.541 52.813
42251 93.493 -79.534 52.804
42252 93.503 -79.526 52.796
42253 93.513 -79.519 52.787
42254 93.523 -79.511 52.778
42255 93.533 -79.504 52.769
42256 93.543 -79.496 52.761
42257 93.553 -79.488 52.754
42258 93.563 -79.481 52.748
42259 93.573 -79.473 52.741
42260 93.587 -79.469 52.732
42261 93.603 -79.466 52.722
42262 93.619 -79.463 52.710
42263 93.635 -79.460 52.699
42264 93.651 -79.457 52.689
42265 93.673 -79.447 52.679
42266 93.698 -79.432 52.673
42267 93.723 -79.418 52.667
42268 93.745 -79.404 52.661
42269 93.758 -79.396 52.654
42270 93.771 -79.387 52.648
42271 93.784 -79.379 52.642
42272 93.797 -79.369 52.638
42273 93.809 -79.361 52.638
42274 93.823 -79.353 52.639
42275 93.836 -79.344 52.639
42276 93.848 -79.335 52.637
42277 93.861 -79.326 52.637
42278 93.873 -79.310 52.637
42279 93.885 -79.295 52.637
42280 93.897 -79.286 52.633
42281 93.897 -79.289 52.626
42282 93.893 -79.293 52.618
42283 93.890 -79.296 52.609
42284 93.886 -79.299 52.599
42285 93.883 -79.303 52.587
42286 93.880 -79.306 52.574
42287 93.878 -79.308 52.564
42288 93.889 -79.303 52.554
42289 93.897 -79.301 52.543
42290 93.903 -79.303 52.530
42291 93.913 -79.303 52.515
42292 93.926 -79.303 52.501
42293 93.938 -79.303 52.484
42294 93.951 -79.304 52.468
42295 93.961 -79.305 52.451
42296 93.964 -79.306 52.434
42297 93.969 -79.306 52.417
42298 93.977 -79.303 52.401
42299 93.985 -79.299 52.385
42300 93.993 -79.295 52.369
42301 94.001 -79.291 52.351
42302 94.016 -79.295 52.332
42303 94.030 -79.298 52.313
42304 94.032 -79.299 52.295
42305 94.034 -79.300 52.277
42306 94.036 -79.300 52.260
42307 94.038 -79.300 52.243
42308 94.036 -79.316 52.227
42309 94.030 -79.343 52.210
42310 94.024 -79.360 52.192
42311 94.016 -79.327 52.174
42312 94.008 -79.294 52.154
42313 94.027 -79.291 52.131
42314 94.046 -79.288 52.111
42315 94.065 -79.286 52.092
42316 94.084 -79.284 52.073
42317 94.091 -79.272 52.057
42318 94.097 -79.259 52.041
42319 94.115 -79.269 52.024
42320 94.121 -79.261 52.006
42321 94.128 -79.253 51.989
42322 94.133 -79.245 51.974
42323 94.135 -79.243 51.963
42324 94.137 -79.241 51.954
42325 94.138 -79.238 51.945
42326 94.148 -79.238 51.937
42327 94.165 -79.242 51.924
42328 94.182 -79.245 51.911
42329 94.199 -79.249 51.896
42330 94.212 -79.255 51.881
42331 94.218 -79.261 51.868
42332 94.223 -79.260 51.856
42333 94.229 -79.258 51.845
42334 94.238 -79.253 51.829
42335 94.246 -79.248 51.812
42336 94.253 -79.243 51.795
42337 94.261 -79.238 51.776
42338 94.269 -79.233 51.757
42339 94.277 -79.228 51.738
42340 94.284 -79.223 51.718
42341 94.292 -79.218 51.695
42342 94.300 -79.213 51.670
42343 94.308 -79.208 51.644
42344 94.316 -79.203 51.621
42345 94.324 -79.198 51.597
42346 94.332 -79.194 51.572
42347 94.339 -79.189 51.544
42348 94.345 -79.187 51.512
42349 94.305 -79.226 51.479
42350 94.267 -79.258 51.443
42351 94.262 -79.253 51.409
42352 94.257 -79.248 51.374
42353 94.253 -79.243 51.340
42354 94.280 -79.226 51.304
42355 94.307 -79.209 51.266
42356 94.321 -79.203 51.227
42357 94.332 -79.200 51.187
42358 94.344 -79.196 51.146
42359 94.352 -79.203 51.104
42360 94.360 -79.209 51.062
42361 94.369 -79.217 51.019
42362 94.377 -79.225 50.977
42363 94.386 -79.232 50.935
42364 94.394 -79.240 50.893
42365 94.402 -79.247 50.852
42424 95.144 -79.976 48.779
42425 95.158 -79.989 48.743
42426 95.176 -79.996 48.707
42427 95.200 -79.997 48.671
42428 95.224 -79.999 48.634
42429 95.247 -80.000 48.597
42430 95.271 -80.002 48.561
42431 95.294 -80.003 48.526
42432 95.318 -80.005 48.490
42433 95.341 -80.006 48.455
42434 95.365 -80.008 48.419
42928 93.052 -79.726 53.062
42929 93.068 -79.717 53.041
42930 93.084 -79.710 53.022
42931 93.100 -79.701 53.004
42932 93.116 -79.694 52.987
42933 93.132 -79.686 52.971
42934 93.148 -79.677 52.957
42935 93.164 -79.669 52.944
42936 93.183 -79.655 52.932
42937 93.201 -79.641 52.922
42938 93.219 -79.627 52.912
42939 93.237 -79.614 52.904
42940 93.255 -79.600 52.898
42941 93.272 -79.590 52.891
42942 93.295 -79.583 52.883
42943 93.317 -79.576 52.875
42944 93.339 -79.570 52.868
42945 93.361 -79.564 52.862
42946 93.382 -79.558 52.856
42947 93.403 -79.553 52.850
42948 93.425 -79.547 52.844
42949 93.446 -79.541 52.836
42950 93.467 -79.535 52.829
42951 93.483 -79.532 52.820
42952 93.499 -79.529 52.811
42953 93.515 -79.526 52.803
42954 93.525 -79.518 52.794
42955 93.536 -79.510 52.785
42956 93.546 -79.503 52.776
42957 93.555 -79.495 52.767
42958 93.565 -79.488 52.760
42959 93.575 -79.480 52.752
42960 93.586 -79.473 52.746
42961 93.602 -79.470 52.737
42962 93.618 -79.468 52.726
42963 93.634 -79.465 52.713
42964 93.650 -79.462 52.704
42965 93.666 -79.459 52.695
42966 93.682 -79.457 52.686
42967 93.698 -79.454 52.679
42968 93.723 -79.439 52.672
42969 93.741 -79.429 52.666
42970 93.754 -79.419 52.660
42971 93.766 -79.411 52.653
42972 93.779 -79.403 52.648
42973 93.791 -79.387 52.644
42974 93.802 -79.371 52.643
42975 93.814 -79.356 52.643
42976 93.826 -79.340 52.642
42977 93.838 -79.324 52.641
42978 93.849 -79.309 52.641
42979 93.861 -79.293 52.641
42980 93.874 -79.286 52.640
42981 93.888 -79.285 52.636
42982 93.888 -79.288 52.630
42983 93.885 -79.291 52.621
42984 93.881 -79.295 52.613
42985 93.878 -79.298 52.602
42986 93.874 -79.301 52.589
42987 93.871 -79.305 52.577
42988 93.880 -79.301 52.566
42989 93.891 -79.296 52.555
42990 93.899 -79.294 52.544
42991 93.904 -79.296 52.530
42992 93.910 -79.297 52.516
42993 93.922 -79.297 52.502
42994 93.934 -79.297 52.485
42995 93.947 -79.297 52.469
42996 93.960 -79.298 52.452
42997 93.972 -79.298 52.436
42998 93.985 -79.299 52.419
42999 93.992 -79.300 52.403
43000 94.000 -79.296 52.386
43001 94.011 -79.297 52.368
43002 94.014 -79.297 52.350
43003 94.016 -79.298 52.331
43004 94.018 -79.299 52.311
43005 94.020 -79.299 52.293
43006 94.023 -79.300 52.275
43007 94.025 -79.300 52.257
43008 94.027 -79.301 52.240
43009 94.029 -79.301 52.224
43010 94.026 -79.318 52.208
43011 94.020 -79.344 52.192
43012 94.013 -79.325 52.175
43013 94.008 -79.296 52.155
43014 94.027 -79.293 52.132
43015 94.047 -79.291 52.111
43016 94.066 -79.288 52.092
43017 94.071 -79.275 52.073
43018 94.077 -79.263 52.056
43019 94.082 -79.251 52.041
43020 94.100 -79.259 52.027
43021 94.113 -79.263 52.009
43022 94.120 -79.254 51.993
43023 94.124 -79.249 51.979
43024 94.125 -79.247 51.969
43025 94.127 -79.244 51.960
43026 94.129 -79.241 51.951
43027 94.134 -79.241 51.942
43028 94.151 -79.244 51.929
43029 94.168 -79.247 51.914
43030 94.185 -79.251 51.900
43031 94.202 -79.254 51.884
43032 94.214 -79.260 51.871
43033 94.220 -79.259 51.858
43034 94.228 -79.254 51.847
43035 94.236 -79.249 51.831
43036 94.244 -79.244 51.813
43037 94.251 -79.239 51.795
43038 94.259 -79.234 51.777
43039 94.267 -79.229 51.757
43040 94.273 -79.226 51.737
43041 94.280 -79.222 51.717
43042 94.286 -79.219 51.694
43043 94.293 -79.216 51.670
43044 94.299 -79.212 51.647
43045 94.306 -79.208 51.625
43046 94.313 -79.205 51.603
43047 94.319 -79.201 51.578
43048 94.321 -79.203 51.551
43049 94.317 -79.213 51.520
43050 94.282 -79.249 51.485
43051 94.268 -79.254 51.449
43052 94.263 -79.250 51.415
43053 94.258 -79.244 51.381
43054 94.252 -79.239 51.347
43055 94.278 -79.223 51.312
43056 94.305 -79.206 51.275
43057 94.327 -79.194 51.235
43058 94.338 -79.190 51.196
43059 94.349 -79.187 51.155
43060 94.360 -79.183 51.113
43061 94.369 -79.190 51.071
43062 94.377 -79.197 51.029
43063 94.386 -79.204 50.987
43064 94.394 -79.212 50.945
43065 94.402 -79.219 50.903
43066 94.410 -79.227 50.862
43124 95.123 -79.924 48.818
43125 95.136 -79.936 48.781
43126 95.149 -79.949 48.745
43127 95.167 -79.958 48.709
43128 95.190 -79.959 48.672
43129 95.214 -79.961 48.636
43130 95.237 -79.962 48.599
43131 95.261 -79.964 48.563
43132 95.284 -79.965 48.527
43133 95.308 -79.967 48.492
43134 95.332 -79.968 48.456
43135 95.355 -79.970 48.421
43136 95.379 -79.971 48.386
43629 93.048 -79.709 53.084
43630 93.064 -79.701 53.063
43631 93.080 -79.692 53.043
43632 93.095 -79.685 53.025
43633 93.112 -79.676 53.008
43634 93.128 -79.669 52.992
43635 93.144 -79.661 52.976
43636 93.160 -79.652 52.962
43637 93.177 -79.642 52.950
43638 93.195 -79.629 52.939
43639 93.213 -79.615 52.928
43640 93.231 -79.601 52.918
43641 93.249 -79.587 52.910
43642 93.267 -79.573 52.901
43643 93.286 -79.566 52.892
43644 93.307 -79.560 52.886
43645 93.329 -79.554 52.879
43646 93.350 -79.548 52.872
43647 93.371 -79.542 52.866
43648 93.392 -79.536 52.859
43649 93.413 -79.531 52.852
43650 93.435 -79.525 52.845
43651 93.456 -79.520 52.837
43652 93.474 -79.515 52.829
43653 93.490 -79.512 52.819
43654 93.505 -79.508 52.811
43655 93.521 -79.505 52.802
43656 93.537 -79.502 52.793
43657 93.553 -79.499 52.785
43658 93.567 -79.494 52.776
43659 93.577 -79.487 52.768
43660 93.587 -79.479 52.760
43661 93.601 -79.475 52.753
43662 93.617 -79.473 52.743
43663 93.633 -79.469 52.732
43664 93.649 -79.466 52.720
43665 93.665 -79.463 52.710
43666 93.681 -79.460 52.702
43667 93.694 -79.456 52.694
43668 93.707 -79.451 52.686
43669 93.720 -79.446 52.679
43670 93.732 -79.432 52.673
43671 93.744 -79.416 52.666
43672 93.756 -79.400 52.659
43673 93.767 -79.385 52.654
43674 93.779 -79.369 52.650
43675 93.791 -79.353 52.649
43676 93.802 -79.338 52.649
43677 93.814 -79.322 52.649
43678 93.826 -79.306 52.647
43679 93.837 -79.291 52.647
43680 93.851 -79.286 52.646
43681 93.865 -79.285 52.644
43682 93.879 -79.285 52.639
43683 93.880 -79.287 52.633
43684 93.876 -79.291 52.625
43685 93.873 -79.294 52.616
43686 93.869 -79.297 52.605
43687 93.866 -79.300 52.592
43688 93.870 -79.299 52.579
43689 93.881 -79.294 52.568
43690 93.892 -79.288 52.558
43691 93.900 -79.288 52.545
43692 93.906 -79.288 52.533
43693 93.911 -79.290 52.519
43694 93.918 -79.291 52.504
43695 93.930 -79.291 52.487
43696 93.943 -79.291 52.470
43697 93.956 -79.292 52.453
43698 93.968 -79.292 52.438
43699 93.981 -79.292 52.422
43700 93.994 -79.292 52.404
43701 94.004 -79.292 52.385
43702 94.003 -79.295 52.367
43703 94.003 -79.297 52.348
43704 94.004 -79.298 52.329
43705 94.007 -79.299 52.310
43706 94.009 -79.300 52.291
43707 94.011 -79.300 52.273
43708 94.013 -79.300 52.255
43709 94.015 -79.301 52.238
43710 94.017 -79.302 52.223
43711 94.019 -79.302 52.207
43712 94.017 -79.319 52.192
43713 94.010 -79.324 52.175
43714 94.008 -79.298 52.155
43715 94.028 -79.295 52.133
43716 94.046 -79.292 52.113
43717 94.051 -79.280 52.094
43718 94.057 -79.267 52.077
43719 94.063 -79.255 52.062
43720 94.068 -79.242 52.048
43721 94.084 -79.249 52.035
43722 94.103 -79.259 52.019
43723 94.112 -79.256 52.004
43724 94.114 -79.252 51.992
43725 94.116 -79.250 51.982
43726 94.118 -79.247 51.973
43727 94.120 -79.245 51.964
43728 94.121 -79.243 51.953
43729 94.137 -79.246 51.939
43730 94.154 -79.250 51.923
43731 94.172 -79.253 51.907
43732 94.189 -79.256 51.891
43733 94.208 -79.258 51.876
43734 94.214 -79.259 51.862
43735 94.221 -79.256 51.850
43736 94.227 -79.252 51.833
43737 94.234 -79.248 51.815
43738 94.241 -79.245 51.796
43739 94.247 -79.241 51.776
43740 94.254 -79.237 51.755
43741 94.260 -79.234 51.735
43742 94.267 -79.231 51.715
43743 94.273 -79.227 51.692
43744 94.280 -79.224 51.670
43745 94.286 -79.220 51.650
43746 94.292 -79.216 51.630
43747 94.299 -79.213 51.608
43748 94.297 -79.221 51.585
43749 94.293 -79.231 51.558
43750 94.290 -79.241 51.527
43751 94.274 -79.256 51.493
43752 94.269 -79.250 51.458
43753 94.264 -79.245 51.423
43754 94.258 -79.241 51.389
43755 94.253 -79.235 51.355
43756 94.275 -79.220 51.321
43757 94.302 -79.203 51.284
43758 94.330 -79.187 51.245
43759 94.343 -79.181 51.204
43760 94.354 -79.178 51.164
43761 94.365 -79.174 51.123
43762 94.377 -79.170 51.081
43763 94.386 -79.177 51.039
43764 94.394 -79.184 50.997
43765 94.402 -79.192 50.954
43766 94.410 -79.199 50.913
43767 94.419 -79.206 50.873
43824 95.102 -79.871 48.858
43825 95.115 -79.884 48.821
43826 95.128 -79.897 48.784
43827 95.141 -79.910 48.748
43828 95.157 -79.921 48.711
43829 95.180 -79.922 48.674
43830 95.204 -79.924 48.637
43831 95.228 -79.925 48.601
43832 95.251 -79.926 48.565
43833 95.275 -79.927 48.529
43834 95.298 -79.929 48.493
43835 95.322 -79.930 48.457
43836 95.345 -79.932 48.423
43837 95.369 -79.933 48.387
44331 93.060 -79.684 53.086
44332 93.075 -79.676 53.066
44333 93.091 -79.667 53.047
44334 93.108 -79.660 53.030
44335 93.124 -79.651 53.013
44336 93.140 -79.644 52.997
44337 93.156 -79.636 52.982
44338 93.171 -79.627 52.969
44339 93.189 -79.617 52.956
44340 93.207 -79.602 52.944
44341 93.225 -79.588 52.933
44342 93.243 -79.575 52.924
44343 93.262 -79.561 52.914
44344 93.280 -79.550 52.904
44345 93.298 -79.544 52.898
44346 93.318 -79.538 52.890
44347 93.339 -79.532 52.884
44348 93.360 -79.526 52.878
44349 93.381 -79.520 52.872
44350 93.402 -79.515 52.865
44351 93.423 -79.509 52.858
44352 93.445 -79.503 52.850
44353 93.465 -79.498 52.842
44354 93.480 -79.494 52.833
44355 93.496 -79.491 52.824
44356 93.512 -79.488 52.814
44357 93.528 -79.485 52.805
44358 93.544 -79.482 52.796
44359 93.559 -79.479 52.789
44360 93.575 -79.476 52.780
44361 93.588 -79.472 52.772
44362 93.603 -79.467 52.761
44363 93.616 -79.462 52.750
44364 93.628 -79.457 52.739
44365 93.641 -79.452 52.729
44366 93.654 -79.447 52.719
44367 93.667 -79.443 52.711
44368 93.680 -79.438 52.704
44369 93.693 -79.433 52.695
44370 93.707 -79.426 52.688
44371 93.720 -79.414 52.681
44372 93.732 -79.398 52.673
44373 93.744 -79.382 52.666
44374 93.756 -79.367 52.660
44375 93.767 -79.351 52.657
44376 93.779 -79.336 52.656
44377 93.790 -79.320 52.656
44378 93.802 -79.304 52.655
44379 93.814 -79.289 52.654
44380 93.827 -79.285 52.653
44381 93.841 -79.285 52.651
44382 93.855 -79.285 52.648
44383 93.869 -79.284 52.643
44384 93.871 -79.286 52.636
44385 93.867 -79.289 52.627
44386 93.864 -79.293 52.619
44387 93.861 -79.296 52.608
44388 93.860 -79.297 52.595
44389 93.871 -79.292 52.583
44390 93.883 -79.287 52.572
44391 93.894 -79.281 52.561
44392 93.902 -79.280 52.548
44393 93.907 -79.281 52.535
44394 93.913 -79.283 52.521
44395 93.919 -79.284 52.506
44396 93.926 -79.285 52.489
44397 93.939 -79.285 52.472
44398 93.952 -79.285 52.456
44399 93.964 -79.286 52.439
44400 93.980 -79.285 52.421
44401 93.995 -79.283 52.403
44402 94.004 -79.283 52.385
44403 94.004 -79.286 52.367
44404 94.003 -79.288 52.348
44405 94.003 -79.291 52.329
44406 94.003 -79.294 52.310
44407 94.002 -79.296 52.291
44408 94.002 -79.298 52.272
44409 94.002 -79.300 52.254
44410 94.004 -79.301 52.238
44411 94.006 -79.302 52.222
44412 94.008 -79.302 52.207
44413 94.010 -79.303 52.192
44414 94.007 -79.320 52.175
44415 94.009 -79.300 52.156
44416 94.026 -79.296 52.135
44417 94.031 -79.284 52.116
44418 94.037 -79.271 52.098
44419 94.043 -79.259 52.083
44420 94.048 -79.247 52.070
44421 94.054 -79.234 52.057
44422 94.069 -79.239 52.045
44423 94.087 -79.250 52.031
44424 94.103 -79.258 52.018
44425 94.105 -79.256 52.007
44426 94.107 -79.253 51.997
44427 94.109 -79.250 51.987
44428 94.112 -79.247 51.977
44429 94.114 -79.243 51.965
44430 94.129 -79.243 51.949
44431 94.148 -79.244 51.932
44432 94.167 -79.246 51.914
44433 94.186 -79.247 51.898
44434 94.197 -79.253 51.882
44435 94.202 -79.265 51.866
44436 94.208 -79.264 51.852
44437 94.214 -79.260 51.834
44438 94.221 -79.256 51.816
44439 94.227 -79.253 51.797
44440 94.234 -79.250 51.776
44441 94.241 -79.246 51.755
44442 94.247 -79.242 51.735
44443 94.253 -79.239 51.716
44444 94.260 -79.235 51.694
44445 94.266 -79.231 51.672
44446 94.273 -79.228 51.652
44447 94.277 -79.228 51.632
44448 94.273 -79.238 51.611
44449 94.270 -79.248 51.588
44450 94.266 -79.257 51.564
44451 94.268 -79.251 51.536
44452 94.275 -79.252 51.503
44453 94.270 -79.247 51.468
44454 94.264 -79.241 51.433
44455 94.259 -79.237 51.399
44456 94.254 -79.231 51.366
44457 94.273 -79.218 51.331
44458 94.300 -79.201 51.294
44459 94.327 -79.184 51.255
44460 94.348 -79.172 51.214
44461 94.360 -79.169 51.174
44462 94.371 -79.165 51.132
44463 94.383 -79.161 51.091
44464 94.394 -79.158 51.049
44465 94.402 -79.164 51.007
44466 94.410 -79.172 50.964
44467 94.419 -79.179 50.923
44468 94.427 -79.186 50.883
44524 95.080 -79.819 48.897
44525 95.093 -79.832 48.861
44526 95.106 -79.845 48.824
44527 95.119 -79.858 48.787
44528 95.132 -79.871 48.750
44529 95.147 -79.883 48.713
44530 95.171 -79.884 48.676
44531 95.194 -79.886 48.639
44532 95.218 -79.887 48.603
44533 95.241 -79.889 48.567
44534 95.265 -79.890 48.531
44535 95.288 -79.892 48.495
44536 95.312 -79.893 48.459
44537 95.336 -79.895 48.424
44538 95.359 -79.896 48.389
44539 95.383 -79.897 48.353
45033 93.071 -79.659 53.089
45034 93.087 -79.651 53.070
45035 93.103 -79.642 53.051
45036 93.120 -79.635 53.034
45037 93.136 -79.626 53.017
45038 93.152 -79.619 53.002
45039 93.167 -79.610 52.988
45040 93.184 -79.602 52.975
45041 93.201 -79.590 52.962
45042 93.219 -79.576 52.950
45043 93.237 -79.563 52.939
45044 93.256 -79.548 52.928
45045 93.274 -79.535 52.919
45046 93.292 -79.529 52.910
45047 93.311 -79.523 52.903
45048 93.330 -79.516 52.896
45049 93.349 -79.510 52.892
45050 93.370 -79.504 52.886
45051 93.391 -79.498 52.880
45052 93.412 -79.493 52.873
45053 93.433 -79.487 52.866
45054 93.455 -79.482 52.857
45055 93.471 -79.478 52.848
45056 93.487 -79.475 52.839
45057 93.502 -79.472 52.829
45058 93.518 -79.468 52.819
45059 93.534 -79.465 52.810
45060 93.550 -79.462 52.801
45061 93.560 -79.457 52.792
45062 93.570 -79.453 52.782
45063 93.586 -79.448 52.770
45064 93.602 -79.444 52.759
45065 93.615 -79.439 52.749
45066 93.628 -79.435 52.739
45067 93.640 -79.429 52.729
45068 93.653 -79.425 52.721
45069 93.666 -79.419 52.713
45070 93.680 -79.415 52.705
45071 93.694 -79.407 52.698
45072 93.708 -79.396 52.690
45073 93.720 -79.381 52.682
45074 93.732 -79.365 52.674
45075 93.744 -79.349 52.667
45076 93.755 -79.334 52.665
45077 93.767 -79.318 52.664
45078 93.778 -79.303 52.664
45079 93.790 -79.287 52.662
45080 93.804 -79.285 52.660
45081 93.818 -79.285 52.658
45082 93.832 -79.284 52.656
45083 93.846 -79.283 52.651
45084 93.860 -79.282 52.645
45085 93.862 -79.285 52.638
45086 93.859 -79.288 52.629
45087 93.856 -79.291 52.621
45088 93.852 -79.294 52.610
45089 93.862 -79.291 52.598
45090 93.873 -79.285 52.587
45091 93.885 -79.279 52.576
45092 93.896 -79.274 52.565
45093 93.903 -79.273 52.552
45094 93.909 -79.275 52.538
45095 93.914 -79.275 52.524
45096 93.920 -79.277 52.508
45097 93.926 -79.278 52.492
45098 93.935 -79.279 52.475
45099 93.949 -79.278 52.458
45100 93.965 -79.277 52.441
45101 93.981 -79.275 52.422
45102 93.996 -79.274 52.404
45103 94.005 -79.274 52.386
45104 94.005 -79.277 52.368
45105 94.004 -79.279 52.350
45106 94.004 -79.281 52.331
45107 94.003 -79.285 52.311
45108 94.003 -79.287 52.293
45109 94.002 -79.289 52.275
45110 94.002 -79.292 52.256
45111 94.002 -79.294 52.239
45112 94.001 -79.297 52.223
45113 94.001 -79.300 52.209
45114 94.000 -79.302 52.194
45115 94.001 -79.303 52.178
45116 94.006 -79.300 52.159
45117 94.011 -79.288 52.138
45118 94.017 -79.275 52.120
45119 94.023 -79.263 52.103
45120 94.028 -79.250 52.090
45121 94.034 -79.237 52.079
45122 94.040 -79.225 52.069
45123 94.054 -79.229 52.058
45124 94.072 -79.240 52.047
45125 94.098 -79.256 52.036
45126 94.101 -79.254 52.025
45127 94.104 -79.250 52.013
45128 94.107 -79.245 52.002
45129 94.110 -79.240 51.990
45130 94.113 -79.234 51.976
45131 94.125 -79.234 51.960
45132 94.144 -79.235 51.942
45133 94.164 -79.237 51.923
45134 94.181 -79.239 51.905
45135 94.185 -79.250 51.887
45136 94.190 -79.262 51.870
45137 94.195 -79.272 51.853
45138 94.201 -79.269 51.835
45139 94.207 -79.265 51.816
45140 94.214 -79.261 51.796
45141 94.220 -79.258 51.775
45142 94.227 -79.254 51.755
45143 94.234 -79.250 51.736
45144 94.240 -79.247 51.718
45145 94.247 -79.244 51.697
45146 94.253 -79.240 51.675
45147 94.253 -79.246 51.655
45148 94.249 -79.255 51.634
45149 94.246 -79.265 51.614
45150 94.248 -79.256 51.592
45151 94.251 -79.244 51.569
45152 94.258 -79.227 51.542
45153 94.276 -79.206 51.511
45154 94.270 -79.243 51.477
45155 94.265 -79.237 51.443
45156 94.260 -79.233 51.409
45157 94.255 -79.228 51.377
45158 94.270 -79.215 51.342
45159 94.298 -79.198 51.304
45160 94.325 -79.181 51.264
45161 94.352 -79.165 51.224
45162 94.365 -79.159 51.183
45163 94.377 -79.156 51.142
45164 94.388 -79.152 51.100
45165 94.399 -79.149 51.059
45166 94.410 -79.145 51.018
45167 94.419 -79.151 50.975
45168 94.427 -79.159 50.934
45169 94.435 -79.166 50.894
45224 95.058 -79.767 48.938
45225 95.071 -79.780 48.901
45226 95.085 -79.792 48.864
45227 95.098 -79.805 48.827
45228 95.111 -79.818 48.791
45229 95.124 -79.832 48.753
45230 95.137 -79.845 48.715
45231 95.161 -79.846 48.678
45232 95.184 -79.848 48.642
45233 95.208 -79.849 48.605
45234 95.232 -79.851 48.568
45235 95.255 -79.852 48.533
45236 95.279 -79.854 48.498
45237 95.302 -79.855 48.462
45238 95.326 -79.857 48.426
45239 95.349 -79.858 48.390
45240 95.373 -79.860 48.355
45241 95.397 -79.861 48.320
45735 93.083 -79.634 53.092
45736 93.099 -79.626 53.074
45737 93.116 -79.617 53.056
45738 93.132 -79.610 53.039
45739 93.147 -79.601 53.023
45740 93.163 -79.594 53.008
45741 93.180 -79.585 52.995
45742 93.196 -79.577 52.982
45743 93.214 -79.563 52.970
45744 93.232 -79.550 52.958
45745 93.250 -79.536 52.947
45746 93.268 -79.523 52.936
45747 93.286 -79.513 52.928
45748 93.305 -79.507 52.920
45749 93.324 -79.501 52.913
45750 93.342 -79.494 52.908
45751 93.360 -79.488 52.904
45752 93.380 -79.482 52.898
45753 93.401 -79.477 52.892
45754 93.422 -79.471 52.883
45755 93.444 -79.465 52.874
45756 93.462 -79.461 52.864
45757 93.477 -79.458 52.855
45758 93.493 -79.454 52.845
45759 93.509 -79.451 52.835
45760 93.521 -79.447 52.825
45761 93.532 -79.443 52.815
45762 93.541 -79.438 52.805
45763 93.552 -79.434 52.793
45764 93.568 -79.429 52.781
45765 93.585 -79.426 52.770
45766 93.601 -79.421 52.760
45767 93.614 -79.416 52.749
45768 93.627 -79.411 52.740
45769 93.640 -79.407 52.732
45770 93.653 -79.401 52.723
45771 93.666 -79.397 52.715
45772 93.681 -79.386 52.707
45773 93.695 -79.376 52.700
45774 93.708 -79.363 52.692
45775 93.720 -79.347 52.685
45776 93.731 -79.332 52.680
45777 93.743 -79.316 52.675
45778 93.755 -79.300 52.672
45779 93.767 -79.285 52.670
45780 93.781 -79.285 52.667
45781 93.795 -79.284 52.665
45782 93.809 -79.284 52.662
45783 93.823 -79.283 52.658
45784 93.837 -79.282 52.653
45785 93.851 -79.281 52.647
45786 93.854 -79.284 52.640
45787 93.851 -79.287 52.632
45788 93.847 -79.291 52.623
45789 93.852 -79.288 52.612
45790 93.864 -79.283 52.600
45791 93.875 -79.278 52.590
45792 93.886 -79.272 52.579
45793 93.897 -79.267 52.567
45794 93.905 -79.266 52.555
45795 93.910 -79.267 52.541
45796 93.916 -79.269 52.526
45797 93.922 -79.270 52.510
45798 93.927 -79.271 52.494
45799 93.936 -79.270 52.478
45800 93.951 -79.269 52.461
45801 93.966 -79.268 52.443
45802 93.982 -79.266 52.424
45803 93.997 -79.265 52.407
45804 94.006 -79.266 52.389
45805 94.005 -79.268 52.371
45806 94.005 -79.270 52.352
45807 94.004 -79.273 52.333
45808 94.004 -79.275 52.313
45809 94.004 -79.278 52.295
45810 94.003 -79.281 52.278
45811 94.003 -79.283 52.260
45812 94.002 -79.285 52.243
45813 94.002 -79.288 52.226
45814 94.001 -79.291 52.211
45815 94.001 -79.293 52.196
45816 94.003 -79.292 52.180
45817 94.004 -79.288 52.163
45818 94.002 -79.278 52.143
45819 94.002 -79.266 52.126
45820 94.008 -79.254 52.111
45821 94.014 -79.241 52.098
45822 94.019 -79.229 52.088
45823 94.025 -79.217 52.079
45824 94.039 -79.219 52.072
45825 94.061 -79.233 52.062
45826 94.085 -79.246 52.052
45827 94.099 -79.247 52.042
45828 94.102 -79.242 52.029
45829 94.105 -79.237 52.017
45830 94.108 -79.231 52.004
45831 94.111 -79.226 51.990
45832 94.122 -79.225 51.972
45833 94.141 -79.226 51.953
45834 94.160 -79.228 51.933
45835 94.169 -79.236 51.913
45836 94.174 -79.247 51.893
45837 94.178 -79.259 51.874
45838 94.183 -79.269 51.855
45839 94.188 -79.277 51.835
45840 94.194 -79.273 51.814
45841 94.201 -79.269 51.795
45842 94.207 -79.266 51.775
45843 94.214 -79.263 51.756
45844 94.220 -79.259 51.738
45845 94.227 -79.256 51.719
45846 94.232 -79.253 51.700
45847 94.229 -79.263 51.679
45848 94.225 -79.272 51.658
45849 94.228 -79.263 51.637
45850 94.231 -79.249 51.617
45851 94.235 -79.236 51.597
45852 94.238 -79.222 51.573
45853 94.253 -79.199 51.546
45854 94.273 -79.169 51.517
45855 94.271 -79.226 51.484
45856 94.266 -79.234 51.451
45857 94.261 -79.228 51.419
45858 94.255 -79.224 51.387
45859 94.268 -79.212 51.352
45860 94.295 -79.195 51.314
45861 94.323 -79.178 51.273
45862 94.350 -79.162 51.233
45863 94.370 -79.150 51.192
45864 94.382 -79.147 51.150
45865 94.393 -79.143 51.110
45866 94.404 -79.140 51.069
45867 94.416 -79.136 51.028
45868 94.427 -79.132 50.985
45869 94.435 -79.138 50.945
45870 94.444 -79.146 50.906
45924 95.037 -79.714 48.978
45925 95.050 -79.727 48.941
45926 95.063 -79.740 48.904
45927 95.076 -79.753 48.867
45928 95.090 -79.767 48.830
45929 95.103 -79.780 48.793
45930 95.116 -79.792 48.756
45931 95.129 -79.805 48.718
45932 95.151 -79.808 48.682
45933 95.175 -79.810 48.645
45934 95.198 -79.811 48.608
45935 95.222 -79.813 48.571
45936 95.245 -79.814 48.537
45937 95.269 -79.816 48.501
45938 95.293 -79.817 48.465
45939 95.316 -79.819 48.429
45940 95.340 -79.820 48.393
45941 95.363 -79.822 48.358
45942 95.387 -79.823 48.324
46436 93.079 -79.617 53.115
46437 93.095 -79.609 53.096
46438 93.112 -79.601 53.078
46439 93.127 -79.592 53.061
46440 93.143 -79.585 53.045
46441 93.159 -79.576 53.030
46442 93.176 -79.569 53.016
46443 93.192 -79.560 53.002
46444 93.208 -79.551 52.990
46445 93.226 -79.538 52.978
46446 93.244 -79.524 52.966
46447 93.263 -79.510 52.955
46448 93.281 -79.498 52.946
46449 93.299 -79.491 52.938
46450 93.318 -79.485 52.931
46451 93.336 -79.479 52.927
46452 93.355 -79.473 52.922
46453 93.373 -79.467 52.916
46454 93.392 -79.461 52.909
46455 93.411 -79.455 52.901
46456 93.432 -79.449 52.893
46457 93.452 -79.444 52.885
46458 93.468 -79.441 52.875
46459 93.484 -79.438 52.865
46460 93.493 -79.433 52.854
46461 93.503 -79.429 52.843
46462 93.513 -79.424 52.832
46463 93.523 -79.419 52.820
46464 93.534 -79.415 52.807
46465 93.551 -79.411 52.795
46466 93.567 -79.407 52.784
46467 93.584 -79.402 52.773
46468 93.600 -79.398 52.761
46469 93.613 -79.393 52.752
46470 93.626 -79.388 52.743
46471 93.639 -79.383 52.734
46472 93.652 -79.376 52.725
46473 93.667 -79.366 52.717
46474 93.682 -79.357 52.710
46475 93.696 -79.345 52.704
46476 93.708 -79.329 52.699
46477 93.720 -79.314 52.694
46478 93.731 -79.298 52.688
46479 93.744 -79.285 52.683
46480 93.758 -79.285 52.678
46481 93.771 -79.284 52.674
46482 93.785 -79.283 52.671
46483 93.799 -79.282 52.667
46484 93.814 -79.282 52.663
46485 93.828 -79.281 52.657
46486 93.842 -79.281 52.651
46487 93.846 -79.282 52.644
46488 93.842 -79.286 52.636
46489 93.843 -79.287 52.627
46490 93.854 -79.281 52.616
46491 93.865 -79.276 52.605
46492 93.876 -79.270 52.595
46493 93.888 -79.265 52.584
46494 93.899 -79.259 52.573
46495 93.906 -79.259 52.560
46496 93.912 -79.260 52.547
46497 93.917 -79.262 52.532
46498 93.923 -79.263 52.517
46499 93.931 -79.263 52.500
46500 93.941 -79.261 52.482
46501 93.952 -79.260 52.464
46502 93.967 -79.259 52.446
46503 93.983 -79.257 52.429
46504 93.998 -79.256 52.412
46505 94.007 -79.256 52.394
46506 94.006 -79.259 52.375
46507 94.006 -79.261 52.356
46508 94.005 -79.264 52.335
46509 94.005 -79.266 52.316
46510 94.004 -79.269 52.299
46511 94.004 -79.272 52.283
46512 94.004 -79.274 52.266
46513 94.003 -79.276 52.248
46514 94.003 -79.279 52.230
46515 94.004 -79.279 52.213
46516 94.007 -79.278 52.197
46517 94.010 -79.277 52.182
46518 94.011 -79.273 52.167
46519 94.008 -79.263 52.149
46520 94.006 -79.253 52.133
46521 94.004 -79.243 52.119
46522 94.001 -79.232 52.107
46523 94.005 -79.221 52.097
46524 94.010 -79.208 52.089
46525 94.025 -79.210 52.085
46526 94.047 -79.220 52.076
46527 94.065 -79.225 52.066
46528 94.083 -79.230 52.056
46529 94.100 -79.234 52.043
46530 94.103 -79.228 52.031
46531 94.106 -79.223 52.017
46532 94.109 -79.218 52.002
46533 94.119 -79.216 51.983
46534 94.138 -79.217 51.963
46535 94.153 -79.222 51.942
46536 94.157 -79.232 51.920
46537 94.162 -79.244 51.899
46538 94.167 -79.255 51.877
46539 94.171 -79.266 51.857
46540 94.176 -79.278 51.835
46541 94.181 -79.281 51.815
46542 94.188 -79.278 51.796
46543 94.194 -79.274 51.778
46544 94.200 -79.271 51.761
46545 94.207 -79.267 51.742
46546 94.209 -79.270 51.724
46547 94.205 -79.280 51.705
46548 94.208 -79.268 51.685
46549 94.211 -79.255 51.664
46550 94.215 -79.241 51.643
46551 94.218 -79.228 51.622
46552 94.221 -79.215 51.601
46553 94.226 -79.200 51.576
46554 94.247 -79.170 51.550
46555 94.268 -79.140 51.522
46556 94.273 -79.171 51.491
46557 94.267 -79.230 51.460
46558 94.261 -79.225 51.429
46559 94.256 -79.220 51.396
46560 94.266 -79.209 51.359
46561 94.293 -79.193 51.320
46562 94.320 -79.176 51.280
46563 94.347 -79.159 51.240
46564 94.374 -79.142 51.199
46565 94.387 -79.137 51.159
46566 94.399 -79.134 51.119
46567 94.410 -79.130 51.079
46568 94.421 -79.127 51.038
46569 94.432 -79.123 50.996
46570 94.443 -79.119 50.957
46571 94.452 -79.125 50.917
46625 95.028 -79.675 48.981
46626 95.042 -79.688 48.944
46627 95.055 -79.701 48.908
46628 95.068 -79.714 48.871
46629 95.081 -79.727 48.834
46630 95.094 -79.740 48.796
46631 95.107 -79.753 48.759
46632 95.120 -79.766 48.722
46633 95.141 -79.770 48.685
46634 95.165 -79.772 48.648
46635 95.188 -79.773 48.612
46636 95.212 -79.775 48.575
46637 95.236 -79.776 48.540
46638 95.259 -79.778 48.504
46639 95.282 -79.780 48.467
46640 95.306 -79.781 48.432
46641 95.330 -79.783 48.396
46642 95.353 -79.784 48.361
46643 95.377 -79.786 48.326
46644 95.401 -79.787 48.291
46711 96.850 -79.882 46.251
46712 96.872 -79.886 46.227
46713 96.894 -79.891 46.203
47138 93.091 -79.592 53.120
47139 93.107 -79.584 53.102
47140 93.123 -79.576 53.084
47141 93.139 -79.567 53.067
47142 93.155 -79.560 53.052
47143 93.171 -79.551 53.038
47144 93.188 -79.544 53.024
47145 93.203 -79.535 53.010
47146 93.220 -79.525 52.998
47147 93.238 -79.511 52.987
47148 93.257 -79.498 52.976
47149 93.275 -79.484 52.966
47150 93.293 -79.476 52.957
47151 93.311 -79.469 52.951
47152 93.330 -79.463 52.946
47153 93.349 -79.457 52.941
47154 93.367 -79.451 52.935
47155 93.386 -79.445 52.928
47156 93.404 -79.439 52.921
47157 93.423 -79.433 52.914
47158 93.442 -79.427 52.907
47159 93.455 -79.423 52.897
47160 93.465 -79.419 52.887
47161 93.475 -79.414 52.876
47162 93.485 -79.410 52.864
47163 93.495 -79.405 52.852
47164 93.504 -79.400 52.840
47165 93.516 -79.396 52.827
47166 93.533 -79.392 52.814
47167 93.549 -79.388 52.801
47168 93.566 -79.384 52.789
47169 93.582 -79.379 52.777
47170 93.599 -79.375 52.766
47171 93.612 -79.370 52.757
47172 93.625 -79.366 52.746
47173 93.639 -79.357 52.739
47174 93.654 -79.347 52.734
47175 93.669 -79.337 52.729
47176 93.684 -79.326 52.722
47177 93.696 -79.312 52.716
47178 93.708 -79.296 52.710
47179 93.720 -79.285 52.704
47180 93.734 -79.284 52.698
47181 93.748 -79.284 52.692
47182 93.762 -79.283 52.686
47183 93.776 -79.282 52.680
47184 93.790 -79.281 52.675
47185 93.804 -79.281 52.670
47186 93.819 -79.280 52.663
47187 93.833 -79.280 52.656
47188 93.837 -79.281 52.649
47189 93.834 -79.285 52.642
47190 93.845 -79.279 52.632
47191 93.856 -79.274 52.622
47192 93.867 -79.269 52.612
47193 93.878 -79.263 52.601
47194 93.889 -79.258 52.590
47195 93.900 -79.253 52.579
47196 93.907 -79.252 52.566
47197 93.913 -79.253 52.553
47198 93.919 -79.254 52.538
47199 93.927 -79.254 52.523
47200 93.936 -79.253 52.506
47201 93.946 -79.252 52.488
47202 93.956 -79.250 52.469
47203 93.968 -79.250 52.452
47204 93.984 -79.248 52.436
47205 93.999 -79.247 52.420
47206 94.007 -79.247 52.402
47207 94.007 -79.250 52.383
47208 94.006 -79.253 52.362
47209 94.006 -79.255 52.341
47210 94.006 -79.257 52.322
47211 94.005 -79.260 52.304
47212 94.005 -79.263 52.288
47213 94.004 -79.265 52.272
47214 94.005 -79.266 52.253
47215 94.008 -79.264 52.235
47216 94.011 -79.263 52.217
47217 94.013 -79.263 52.201
47218 94.016 -79.261 52.187
47219 94.018 -79.259 52.172
47220 94.015 -79.249 52.155
47221 94.013 -79.238 52.140
47222 94.010 -79.228 52.126
47223 94.008 -79.218 52.116
47224 94.006 -79.208 52.108
47225 94.003 -79.199 52.101
47226 94.008 -79.195 52.096
47227 94.026 -79.200 52.087
47228 94.044 -79.204 52.078
47229 94.062 -79.209 52.068
47230 94.080 -79.213 52.056
47231 94.099 -79.219 52.042
47232 94.105 -79.215 52.028
47233 94.107 -79.209 52.012
47234 94.116 -79.206 51.993
47235 94.134 -79.208 51.972
47236 94.141 -79.218 51.951
47237 94.145 -79.229 51.927
47238 94.150 -79.241 51.904
47239 94.154 -79.252 51.881
47240 94.159 -79.263 51.858
47241 94.164 -79.275 51.837
47242 94.168 -79.286 51.817
47243 94.174 -79.286 51.799
47244 94.181 -79.282 51.783
47245 94.187 -79.279 51.766
47246 94.185 -79.287 51.748
47247 94.188 -79.274 51.731
47248 94.192 -79.260 51.711
47249 94.195 -79.247 51.691
47250 94.198 -79.234 51.670
47251 94.202 -79.221 51.649
47252 94.205 -79.207 51.627
47253 94.208 -79.194 51.605
47254 94.221 -79.172 51.580
47255 94.242 -79.142 51.554
47256 94.262 -79.112 51.526
47257 94.274 -79.116 51.497
47258 94.268 -79.190 51.468
47259 94.262 -79.221 51.438
47260 94.257 -79.216 51.405
47261 94.263 -79.206 51.367
47262 94.290 -79.190 51.327
47263 94.318 -79.173 51.287
47264 94.345 -79.156 51.248
47265 94.372 -79.140 51.208
47266 94.393 -79.128 51.168
47267 94.404 -79.125 51.129
47268 94.415 -79.121 51.089
47269 94.426 -79.118 51.049
47270 94.437 -79.114 51.008
47271 94.449 -79.110 50.968
47272 94.460 -79.106 50.929
47326 95.020 -79.636 48.985
47327 95.033 -79.648 48.948
47328 95.046 -79.662 48.912
47329 95.059 -79.675 48.875
47330 95.072 -79.688 48.837
47331 95.086 -79.701 48.800
47332 95.099 -79.714 48.762
47333 95.112 -79.726 48.725
47334 95.131 -79.733 48.689
47335 95.155 -79.734 48.652
47336 95.179 -79.736 48.615
47337 95.202 -79.737 48.579
47338 95.226 -79.739 48.542
47339 95.249 -79.740 48.506
47340 95.273 -79.742 48.471
47341 95.297 -79.743 48.435
47342 95.320 -79.745 48.399
47343 95.343 -79.746 48.364
47344 95.367 -79.748 48.329
47345 95.391 -79.749 48.294
47346 95.414 -79.751 48.259
47411 96.823 -79.859 46.274
47412 96.846 -79.864 46.251
47413 96.868 -79.868 46.227
47414 96.891 -79.874 46.202
47415 96.913 -79.878 46.177
47416 96.936 -79.883 46.153
47417 96.958 -79.887 46.128
47418 96.997 -79.887 46.105
47419 97.043 -79.884 46.081
47420 97.089 -79.882 46.058
47421 97.135 -79.880 46.035
47840 93.104 -79.569 53.126
47841 93.119 -79.559 53.108
47842 93.135 -79.551 53.091
47843 93.151 -79.542 53.076
47844 93.167 -79.535 53.061
47845 93.183 -79.526 53.047
47846 93.199 -79.519 53.034
47847 93.215 -79.510 53.021
47848 93.233 -79.499 53.009
47849 93.251 -79.485 52.998
47850 93.269 -79.471 52.989
47851 93.287 -79.460 52.980
47852 93.306 -79.454 52.972
47853 93.324 -79.448 52.966
47854 93.343 -79.442 52.960
47855 93.361 -79.436 52.955
47856 93.380 -79.430 52.950
47857 93.398 -79.424 52.943
47858 93.417 -79.418 52.937
47859 93.434 -79.412 52.931
47860 93.443 -79.407 52.921
47861 93.451 -79.402 52.911
47862 93.460 -79.397 52.899
47863 93.468 -79.391 52.887
47864 93.476 -79.386 52.875
47865 93.486 -79.382 52.862
47866 93.498 -79.377 52.849
47867 93.515 -79.373 52.836
47868 93.532 -79.369 52.822
47869 93.548 -79.365 52.808
47870 93.565 -79.360 52.795
47871 93.581 -79.357 52.784
47872 93.598 -79.352 52.774
47873 93.612 -79.347 52.764
47874 93.626 -79.337 52.758
47875 93.641 -79.327 52.753
47876 93.656 -79.316 52.749
47877 93.671 -79.306 52.742
47878 93.684 -79.294 52.736
47879 93.697 -79.285 52.729
47880 93.711 -79.284 52.722
47881 93.725 -79.283 52.715
47882 93.739 -79.282 52.707
47883 93.753 -79.281 52.699
47884 93.767 -79.281 52.692
47885 93.781 -79.281 52.685
47886 93.795 -79.280 52.679
47887 93.809 -79.279 52.672
47888 93.823 -79.278 52.665
47889 93.829 -79.280 52.657
47890 93.835 -79.278 52.649
47891 93.846 -79.272 52.641
47892 93.857 -79.267 52.631
47893 93.868 -79.262 52.621
47894 93.880 -79.256 52.610
47895 93.891 -79.250 52.598
47896 93.902 -79.245 52.586
47897 93.909 -79.245 52.573
47898 93.915 -79.246 52.559
47899 93.922 -79.247 52.545
47900 93.931 -79.245 52.531
47901 93.941 -79.244 52.514
47902 93.951 -79.243 52.495
47903 93.961 -79.241 52.476
47904 93.971 -79.241 52.460
47905 93.985 -79.239 52.445
47906 94.000 -79.237 52.429
47907 94.008 -79.238 52.411
47908 94.008 -79.241 52.391
47909 94.007 -79.244 52.370
47910 94.007 -79.246 52.348
47911 94.006 -79.248 52.328
47912 94.006 -79.251 52.309
47913 94.006 -79.252 52.294
47914 94.009 -79.251 52.277
47915 94.012 -79.250 52.259
47916 94.014 -79.249 52.240
47917 94.017 -79.248 52.223
47918 94.019 -79.247 52.208
47919 94.023 -79.246 52.194
47920 94.025 -79.244 52.180
47921 94.023 -79.234 52.165
47922 94.019 -79.228 52.150
47923 94.015 -79.220 52.137
47924 94.012 -79.213 52.127
47925 94.009 -79.206 52.119
47926 94.005 -79.200 52.112
47927 94.010 -79.195 52.106
47928 94.022 -79.194 52.097
47929 94.034 -79.194 52.088
47930 94.046 -79.192 52.079
47931 94.060 -79.193 52.067
47932 94.077 -79.197 52.054
47933 94.096 -79.202 52.040
47934 94.106 -79.201 52.023
47935 94.112 -79.197 52.003
47936 94.125 -79.203 51.980
47937 94.129 -79.215 51.956
47938 94.134 -79.226 51.932
47939 94.138 -79.237 51.908
47940 94.143 -79.249 51.883
47941 94.147 -79.260 51.861
47942 94.152 -79.272 51.840
47943 94.156 -79.282 51.821
47944 94.161 -79.294 51.804
47945 94.165 -79.293 51.788
47946 94.169 -79.279 51.772
47947 94.172 -79.266 51.756
47948 94.175 -79.253 51.738
47949 94.179 -79.240 51.719
47950 94.182 -79.226 51.698
47951 94.185 -79.213 51.677
47952 94.189 -79.200 51.655
47953 94.192 -79.187 51.632
47954 94.195 -79.173 51.609
47955 94.216 -79.143 51.583
47956 94.237 -79.114 51.557
47957 94.253 -79.093 51.530
47958 94.267 -79.089 51.503
47959 94.269 -79.136 51.475
47960 94.263 -79.210 51.446
47961 94.258 -79.212 51.413
47962 94.261 -79.204 51.374
47963 94.288 -79.187 51.334
47964 94.316 -79.170 51.295
47965 94.343 -79.153 51.256
47966 94.370 -79.137 51.217
47967 94.397 -79.120 51.178
47968 94.409 -79.115 51.138
47969 94.420 -79.112 51.099
47970 94.432 -79.109 51.060
47971 94.443 -79.105 51.020
47972 94.454 -79.101 50.979
47973 94.466 -79.097 50.940
48027 95.012 -79.596 48.990
48028 95.025 -79.610 48.952
48029 95.038 -79.623 48.915
48030 95.051 -79.636 48.879
48031 95.064 -79.648 48.841
48032 95.077 -79.661 48.804
48033 95.091 -79.674 48.767
48034 95.104 -79.687 48.731
48035 95.122 -79.695 48.694
48036 95.145 -79.696 48.657
48037 95.169 -79.698 48.620
48038 95.192 -79.699 48.583
48039 95.216 -79.701 48.547
48040 95.240 -79.702 48.511
48041 95.263 -79.704 48.475
48042 95.286 -79.705 48.440
48043 95.310 -79.707 48.403
48044 95.334 -79.708 48.367
48045 95.357 -79.710 48.332
48046 95.381 -79.711 48.297
48047 95.404 -79.713 48.262
48069 95.851 -79.581 47.504
48070 95.874 -79.594 47.472
48071 95.896 -79.607 47.440
48072 95.919 -79.620 47.409
48073 95.942 -79.633 47.376
48074 95.965 -79.646 47.345
48075 95.988 -79.659 47.313
48076 96.011 -79.673 47.281
48077 96.033 -79.679 47.251
48111 96.797 -79.836 46.298
48112 96.820 -79.841 46.273
48113 96.842 -79.846 46.250
48114 96.865 -79.851 46.226
48115 96.887 -79.855 46.202
48116 96.910 -79.860 46.177
48117 96.932 -79.864 46.152
48118 96.955 -79.869 46.128
48119 96.993 -79.869 46.105
48120 97.039 -79.866 46.081
48121 97.086 -79.864 46.058
48122 97.133 -79.861 46.035
48123 97.132 -79.873 46.013
48124 97.131 -79.884 45.989
48125 97.131 -79.897 45.965
48126 97.134 -79.926 45.941
48127 97.138 -79.954 45.920
48542 93.117 -79.546 53.133
48543 93.131 -79.534 53.115
48544 93.147 -79.526 53.100
48545 93.163 -79.518 53.085
48546 93.179 -79.510 53.071
48547 93.195 -79.501 53.058
48548 93.211 -79.494 53.045
48549 93.227 -79.485 53.033
48550 93.245 -79.473 53.022
48551 93.263 -79.459 53.012
48552 93.281 -79.445 53.003
48553 93.300 -79.438 52.995
48554 93.319 -79.432 52.988
48555 93.337 -79.426 52.983
48556 93.355 -79.420 52.979
48557 93.374 -79.414 52.972
48558 93.393 -79.408 52.966
48559 93.411 -79.402 52.959
48560 93.426 -79.397 52.951
48561 93.437 -79.392 52.941
48562 93.445 -79.387 52.930
48563 93.453 -79.382 52.919
48564 93.462 -79.377 52.909
48565 93.470 -79.372 52.897
48566 93.478 -79.366 52.885
48567 93.489 -79.361 52.872
48568 93.505 -79.357 52.860
48569 93.521 -79.352 52.846
48570 93.538 -79.347 52.831
48571 93.554 -79.344 52.817
48572 93.570 -79.339 52.804
48573 93.586 -79.335 52.794
48574 93.602 -79.329 52.786
48575 93.617 -79.319 52.779
48576 93.631 -79.309 52.774
48577 93.646 -79.298 52.770
48578 93.660 -79.288 52.763
48579 93.675 -79.280 52.756
48580 93.689 -79.279 52.749
48581 93.702 -79.279 52.740
48582 93.716 -79.279 52.732
48583 93.730 -79.279 52.724
48584 93.744 -79.279 52.715
48585 93.758 -79.279 52.707
48586 93.772 -79.279 52.699
48587 93.786 -79.279 52.692
48588 93.800 -79.278 52.685
48589 93.814 -79.278 52.677
48590 93.826 -79.275 52.669
48591 93.837 -79.270 52.661
48592 93.848 -79.265 52.652
48593 93.859 -79.259 52.643
48594 93.870 -79.254 52.633
48595 93.881 -79.249 52.622
48596 93.892 -79.244 52.610
48597 93.903 -79.238 52.598
48598 93.910 -79.237 52.585
48599 93.917 -79.238 52.571
48600 93.927 -79.237 52.557
48601 93.936 -79.236 52.541
48602 93.946 -79.234 52.523
48603 93.956 -79.234 52.504
48604 93.966 -79.232 52.485
48605 93.975 -79.231 52.471
48606 93.986 -79.230 52.455
48607 94.001 -79.228 52.437
48608 94.009 -79.229 52.419
48609 94.008 -79.232 52.398
48610 94.008 -79.234 52.377
48611 94.007 -79.237 52.354
48612 94.007 -79.239 52.333
48613 94.010 -79.237 52.314
48614 94.013 -79.237 52.297
48615 94.015 -79.235 52.281
48616 94.018 -79.234 52.264
48617 94.020 -79.234 52.246
48618 94.023 -79.232 52.230
48619 94.023 -79.232 52.217
48620 94.023 -79.231 52.204
48621 94.023 -79.231 52.191
48622 94.025 -79.234 52.176
48623 94.021 -79.228 52.162
48624 94.018 -79.221 52.149
48625 94.014 -79.214 52.139
48626 94.011 -79.206 52.131
48627 94.007 -79.200 52.124
48628 94.011 -79.196 52.118
48629 94.024 -79.194 52.109
48630 94.036 -79.194 52.100
48631 94.048 -79.192 52.091
48632 94.059 -79.191 52.079
48633 94.071 -79.190 52.066
48634 94.083 -79.189 52.051
48635 94.096 -79.188 52.034
48636 94.109 -79.189 52.012
48637 94.113 -79.200 51.989
48638 94.118 -79.212 51.963
48639 94.123 -79.223 51.938
48640 94.128 -79.234 51.913
48641 94.133 -79.247 51.889
48642 94.138 -79.258 51.867
48643 94.143 -79.269 51.846
48644 94.148 -79.281 51.827
48645 94.150 -79.289 51.810
48646 94.148 -79.288 51.795
48647 94.151 -79.273 51.780
48648 94.155 -79.257 51.764
48649 94.159 -79.241 51.746
48650 94.164 -79.225 51.727
48651 94.168 -79.209 51.705
48652 94.172 -79.194 51.683
48653 94.176 -79.179 51.661
48654 94.179 -79.165 51.638
48655 94.189 -79.146 51.614
48656 94.204 -79.131 51.587
48657 94.219 -79.115 51.561
48658 94.234 -79.100 51.534
48659 94.247 -79.100 51.510
48660 94.259 -79.118 51.483
48661 94.265 -79.156 51.454
48662 94.258 -79.208 51.420
48663 94.259 -79.201 51.382
48664 94.286 -79.184 51.342
48665 94.313 -79.168 51.302
48666 94.340 -79.151 51.264
48667 94.367 -79.134 51.225
48668 94.395 -79.117 51.186
48669 94.415 -79.106 51.148
48670 94.426 -79.103 51.110
48671 94.437 -79.100 51.071
48672 94.449 -79.096 51.031
48673 94.460 -79.092 50.991
48674 94.471 -79.088 50.953
48728 95.003 -79.557 48.994
48729 95.016 -79.570 48.956
48730 95.030 -79.583 48.920
48731 95.043 -79.596 48.883
48732 95.056 -79.609 48.846
48733 95.069 -79.622 48.808
48734 95.082 -79.635 48.772
48735 95.095 -79.648 48.736
48736 95.112 -79.657 48.699
48737 95.135 -79.659 48.662
48738 95.159 -79.661 48.626
48739 95.183 -79.662 48.589
48740 95.206 -79.664 48.552
48741 95.230 -79.664 48.516
48742 95.253 -79.666 48.480
48743 95.277 -79.667 48.444
48744 95.301 -79.669 48.408
48745 95.324 -79.670 48.372
48746 95.347 -79.672 48.337
48747 95.371 -79.673 48.301
48748 95.395 -79.675 48.265
48749 95.418 -79.676 48.230
48758 95.630 -79.690 47.913
48759 95.654 -79.692 47.879
48760 95.678 -79.693 47.845
48761 95.693 -79.673 47.810
48762 95.708 -79.651 47.776
48763 95.723 -79.630 47.742
48764 95.738 -79.609 47.709
48765 95.753 -79.588 47.675
48766 95.768 -79.566 47.641
48767 95.783 -79.545 47.608
48768 95.802 -79.540 47.575
48769 95.824 -79.553 47.542
48770 95.847 -79.566 47.509
48771 95.870 -79.579 47.477
48772 95.893 -79.592 47.445
48773 95.916 -79.605 47.413
48774 95.939 -79.618 47.380
48775 95.961 -79.631 47.348
48776 95.984 -79.645 47.317
48777 96.007 -79.656 47.285
48778 96.030 -79.661 47.254
48779 96.052 -79.665 47.224
48811 96.771 -79.814 46.324
48812 96.794 -79.818 46.298
48813 96.816 -79.823 46.273
48814 96.839 -79.828 46.250
48815 96.861 -79.833 46.226
48816 96.884 -79.837 46.201
48817 96.906 -79.842 46.177
48818 96.929 -79.846 46.152
48819 96.951 -79.851 46.128
48820 96.990 -79.851 46.104
48821 97.036 -79.849 46.081
48822 97.083 -79.845 46.058
48823 97.129 -79.843 46.035
48824 97.130 -79.854 46.013
48825 97.129 -79.865 45.989
48826 97.128 -79.877 45.965
48827 97.132 -79.906 45.942
48828 97.135 -79.935 45.920
48829 97.139 -79.964 45.899
49244 93.130 -79.523 53.141
49245 93.144 -79.511 53.125
49246 93.159 -79.501 53.110
49247 93.175 -79.493 53.096
49248 93.191 -79.485 53.083
49249 93.207 -79.476 53.071
49250 93.223 -79.469 53.059
49251 93.239 -79.460 53.046
49252 93.258 -79.447 53.036
49253 93.276 -79.432 53.027
49254 93.294 -79.423 53.018
49255 93.312 -79.417 53.012
49256 93.331 -79.411 53.007
49257 93.349 -79.405 53.002
49258 93.368 -79.399 52.994
49259 93.386 -79.393 52.987
49260 93.404 -79.387 52.979
49261 93.419 -79.382 52.970
49262 93.431 -79.378 52.959
49263 93.439 -79.372 52.949
49264 93.447 -79.367 52.938
49265 93.455 -79.362 52.927
49266 93.463 -79.357 52.916
49267 93.471 -79.352 52.904
49268 93.485 -79.346 52.891
49269 93.500 -79.341 52.878
49270 93.516 -79.337 52.864
49271 93.532 -79.332 52.849
49272 93.548 -79.328 52.836
49273 93.564 -79.323 52.826
49274 93.581 -79.319 52.817
49275 93.596 -79.314 52.810
49276 93.611 -79.304 52.802
49277 93.626 -79.294 52.796
49278 93.640 -79.284 52.790
49279 93.655 -79.273 52.784
49280 93.669 -79.267 52.777
49281 93.683 -79.263 52.769
49282 93.697 -79.263 52.759
49283 93.711 -79.263 52.750
49284 93.725 -79.263 52.741
49285 93.739 -79.263 52.733
49286 93.753 -79.263 52.724
49287 93.767 -79.263 52.716
49288 93.780 -79.263 52.707
49289 93.794 -79.263 52.700
49290 93.807 -79.264 52.692
49291 93.820 -79.263 52.684
49292 93.833 -79.259 52.675
49293 93.846 -79.256 52.666
49294 93.859 -79.252 52.657
49295 93.871 -79.247 52.647
49296 93.883 -79.241 52.636
49297 93.894 -79.236 52.625
49298 93.905 -79.231 52.613
49299 93.912 -79.231 52.599
49300 93.922 -79.229 52.585
49301 93.932 -79.228 52.569
49302 93.941 -79.227 52.552
49303 93.951 -79.225 52.534
49304 93.961 -79.225 52.515
49305 93.971 -79.223 52.498
49306 93.980 -79.222 52.483
49307 93.990 -79.221 52.467
49308 94.002 -79.219 52.449
49309 94.009 -79.220 52.429
49310 94.009 -79.223 52.408
49311 94.009 -79.225 52.386
49312 94.011 -79.225 52.364
49313 94.014 -79.223 52.344
49314 94.016 -79.222 52.324
49315 94.017 -79.222 52.307
49316 94.017 -79.221 52.290
49317 94.017 -79.220 52.274
49318 94.017 -79.220 52.258
49319 94.017 -79.219 52.243
49320 94.017 -79.219 52.229
49321 94.017 -79.219 52.216
49322 94.017 -79.218 52.203
49323 94.019 -79.222 52.189
49324 94.024 -79.228 52.176
49325 94.020 -79.221 52.164
49326 94.017 -79.214 52.155
49327 94.013 -79.207 52.147
49328 94.010 -79.200 52.139
49329 94.013 -79.196 52.131
49330 94.025 -79.194 52.123
49331 94.037 -79.194 52.113
49332 94.049 -79.193 52.102
49333 94.061 -79.191 52.091
49334 94.073 -79.190 52.077
49335 94.085 -79.189 52.062
49336 94.094 -79.187 52.044
49337 94.100 -79.188 52.022
49338 94.107 -79.200 51.998
49339 94.112 -79.211 51.971
49340 94.117 -79.223 51.945
49341 94.122 -79.234 51.919
49342 94.127 -79.246 51.896
49343 94.132 -79.257 51.875
49344 94.137 -79.269 51.855
49345 94.138 -79.276 51.835
49346 94.138 -79.282 51.818
49347 94.136 -79.280 51.803
49348 94.133 -79.277 51.788
49349 94.135 -79.265 51.772
49350 94.139 -79.249 51.754
49351 94.143 -79.233 51.734
49352 94.147 -79.217 51.711
49353 94.151 -79.201 51.689
49354 94.155 -79.184 51.666
49355 94.160 -79.169 51.642
49356 94.170 -79.152 51.618
49357 94.185 -79.137 51.591
49358 94.199 -79.121 51.565
49359 94.214 -79.106 51.539
49360 94.227 -79.112 51.515
49361 94.239 -79.130 51.489
49362 94.251 -79.147 51.460
49363 94.260 -79.175 51.427
49364 94.256 -79.198 51.390
49365 94.283 -79.181 51.351
49366 94.311 -79.165 51.311
49367 94.338 -79.148 51.273
49368 94.365 -79.131 51.235
49369 94.393 -79.115 51.197
49370 94.420 -79.097 51.159
49371 94.431 -79.093 51.121
49372 94.442 -79.090 51.083
49373 94.454 -79.087 51.043
49374 94.465 -79.083 51.004
49375 94.476 -79.079 50.966
49376 94.488 -79.075 50.928
49429 94.995 -79.518 48.999
49430 95.008 -79.531 48.961
49431 95.021 -79.544 48.924
49432 95.034 -79.557 48.887
49433 95.047 -79.570 48.850
49434 95.060 -79.582 48.814
49435 95.073 -79.596 48.777
49436 95.087 -79.609 48.741
49437 95.102 -79.620 48.704
49438 95.126 -79.621 48.668
49439 95.149 -79.623 48.632
49440 95.173 -79.624 48.596
49441 95.196 -79.626 48.559
49442 95.220 -79.627 48.522
49443 95.244 -79.629 48.486
49444 95.267 -79.630 48.449
49445 95.290 -79.632 48.413
49446 95.314 -79.633 48.377
49447 95.338 -79.634 48.341
49448 95.361 -79.636 48.305
49449 95.385 -79.637 48.269
49450 95.408 -79.639 48.234
49451 95.432 -79.640 48.199
49452 95.456 -79.642 48.164
49453 95.479 -79.643 48.129
49454 95.503 -79.645 48.093
49455 95.526 -79.646 48.057
49456 95.550 -79.648 48.022
49457 95.573 -79.649 47.987
49458 95.597 -79.651 47.952
49459 95.621 -79.652 47.918
49460 95.644 -79.654 47.883
49461 95.668 -79.655 47.849
49462 95.686 -79.642 47.815
49463 95.701 -79.621 47.781
49464 95.716 -79.600 47.748
49465 95.731 -79.579 47.714
49466 95.746 -79.557 47.680
49467 95.761 -79.536 47.647
49468 95.776 -79.515 47.614
49469 95.798 -79.525 47.580
49470 95.821 -79.538 47.547
49471 95.844 -79.551 47.514
49472 95.867 -79.564 47.481
49473 95.890 -79.577 47.450
49474 95.912 -79.590 47.418
49475 95.935 -79.603 47.385
49476 95.958 -79.617 47.353
49477 95.981 -79.629 47.321
49478 96.004 -79.638 47.289
49479 96.026 -79.643 47.258
49480 96.048 -79.648 47.228
49511 96.745 -79.791 46.349
49512 96.768 -79.796 46.323
49513 96.790 -79.801 46.297
49514 96.813 -79.805 46.273
49515 96.835 -79.810 46.249
49516 96.858 -79.814 46.225
49517 96.880 -79.819 46.201
49518 96.902 -79.823 46.177
49519 96.925 -79.828 46.152
49520 96.948 -79.833 46.128
49521 96.986 -79.833 46.104
49522 97.033 -79.830 46.081
49523 97.079 -79.827 46.058
49524 97.125 -79.825 46.035
49525 97.127 -79.836 46.013
49526 97.126 -79.847 45.989
49527 97.125 -79.858 45.965
49528 97.129 -79.887 45.943
49529 97.133 -79.915 45.921
49530 97.136 -79.944 45.899
49531 97.140 -79.973 45.878
49532 97.143 -80.002 45.855
49945 93.129 -79.513 53.166
49946 93.143 -79.501 53.151
49947 93.157 -79.489 53.137
49948 93.171 -79.477 53.122
49949 93.187 -79.468 53.108
49950 93.203 -79.460 53.096
49951 93.219 -79.451 53.084
49952 93.235 -79.444 53.072
49953 93.252 -79.434 53.061
49954 93.270 -79.420 53.052
49955 93.288 -79.407 53.042
49956 93.306 -79.401 53.036
49957 93.325 -79.395 53.029
49958 93.344 -79.389 53.022
49959 93.362 -79.383 53.015
49960 93.380 -79.377 53.007
49961 93.396 -79.372 52.998
49962 93.411 -79.368 52.988
49963 93.424 -79.363 52.977
49964 93.432 -79.358 52.966
49965 93.440 -79.353 52.955
49966 93.448 -79.347 52.944
49967 93.457 -79.342 52.933
49968 93.466 -79.337 52.921
49969 93.481 -79.332 52.908
49970 93.495 -79.326 52.894
49971 93.510 -79.321 52.880
49972 93.526 -79.316 52.867
49973 93.543 -79.313 52.855
49974 93.559 -79.308 52.846
49975 93.575 -79.303 52.838
49976 93.591 -79.300 52.832
49977 93.606 -79.289 52.824
49978 93.621 -79.279 52.816
49979 93.635 -79.269 52.810
49980 93.649 -79.259 52.804
49981 93.663 -79.254 52.796
49982 93.678 -79.249 52.787
49983 93.692 -79.246 52.778
49984 93.706 -79.246 52.769
49985 93.719 -79.246 52.760
49986 93.733 -79.246 52.751
49987 93.747 -79.246 52.743
49988 93.761 -79.246 52.735
49989 93.775 -79.246 52.726
49990 93.789 -79.246 52.718
49991 93.799 -79.253 52.709
49992 93.813 -79.250 52.701
49993 93.826 -79.246 52.692
49994 93.839 -79.242 52.683
49995 93.852 -79.238 52.674
49996 93.865 -79.234 52.664
49997 93.878 -79.230 52.654
49998 93.892 -79.226 52.643
49999 93.905 -79.222 52.631
50000 93.915 -79.221 52.616
50001 93.925 -79.220 52.600
50002 93.936 -79.219 52.582
50003 93.946 -79.218 52.565
50004 93.956 -79.216 52.547
50005 93.966 -79.216 52.529
50006 93.975 -79.214 52.513
50007 93.985 -79.213 52.497
50008 93.995 -79.212 52.481
50009 94.004 -79.210 52.464
50010 94.010 -79.211 52.442
50011 94.012 -79.211 52.421
50012 94.013 -79.210 52.399
50013 94.013 -79.210 52.377
50014 94.013 -79.209 52.357
50015 94.013 -79.209 52.338
50016 94.013 -79.208 52.320
50017 94.013 -79.208 52.303
50018 94.013 -79.207 52.287
50019 94.013 -79.206 52.272
50020 94.012 -79.206 52.258
50021 94.012 -79.206 52.245
50022 94.012 -79.205 52.232
50023 94.012 -79.205 52.219
50024 94.014 -79.206 52.205
50025 94.018 -79.213 52.193
50026 94.022 -79.220 52.182
50027 94.019 -79.215 52.174
50028 94.015 -79.207 52.165
50029 94.012 -79.200 52.157
50030 94.014 -79.196 52.149
50031 94.027 -79.195 52.139
50032 94.039 -79.194 52.128
50033 94.051 -79.193 52.116
50034 94.063 -79.191 52.102
50035 94.074 -79.190 52.088
50036 94.080 -79.187 52.072
50037 94.085 -79.184 52.053
50038 94.092 -79.187 52.031
50039 94.099 -79.200 52.007
50040 94.106 -79.211 51.980
50041 94.111 -79.222 51.953
50042 94.116 -79.234 51.929
50043 94.121 -79.245 51.906
50044 94.126 -79.257 51.885
50045 94.126 -79.263 51.864
50046 94.126 -79.269 51.845
50047 94.126 -79.275 51.827
50048 94.124 -79.272 51.811
50049 94.121 -79.269 51.795
50050 94.119 -79.266 51.779
50051 94.119 -79.256 51.761
50052 94.123 -79.241 51.740
50053 94.127 -79.224 51.717
50054 94.131 -79.208 51.694
50055 94.135 -79.192 51.671
50056 94.143 -79.175 51.646
50057 94.153 -79.159 51.620
50058 94.165 -79.142 51.595
50059 94.180 -79.127 51.570
50060 94.194 -79.112 51.545
50061 94.207 -79.125 51.520
50062 94.219 -79.142 51.494
50063 94.231 -79.159 51.466
50064 94.243 -79.177 51.435
50065 94.255 -79.195 51.399
50066 94.281 -79.179 51.361
50067 94.309 -79.162 51.321
50068 94.336 -79.145 51.284
50069 94.363 -79.128 51.246
50070 94.390 -79.112 51.208
50071 94.417 -79.095 51.170
50072 94.436 -79.084 51.133
50073 94.448 -79.081 51.095
50074 94.459 -79.078 51.056
50075 94.470 -79.074 51.018
50076 94.482 -79.070 50.980
50077 94.493 -79.066 50.941
50078 94.504 -79.062 50.904
50130 94.986 -79.479 49.004
50131 94.999 -79.491 48.967
50132 95.013 -79.504 48.930
50133 95.026 -79.517 48.894
50134 95.039 -79.530 48.857
50135 95.052 -79.544 48.820
50136 95.065 -79.557 48.783
50137 95.078 -79.570 48.746
50138 95.092 -79.582 48.710
50139 95.116 -79.583 48.675
50140 95.139 -79.585 48.640
50141 95.163 -79.586 48.604
50142 95.187 -79.588 48.565
50143 95.210 -79.589 48.528
50144 95.234 -79.591 48.492
50145 95.257 -79.592 48.455
50146 95.281 -79.594 48.419
50147 95.305 -79.595 48.383
50148 95.328 -79.597 48.346
50149 95.351 -79.598 48.310
50150 95.375 -79.600 48.274
50151 95.399 -79.601 48.238
50152 95.422 -79.603 48.203
50153 95.446 -79.604 48.168
50154 95.469 -79.605 48.133
50155 95.493 -79.607 48.097
50156 95.517 -79.608 48.062
50157 95.540 -79.610 48.027
50158 95.564 -79.611 47.992
50159 95.587 -79.613 47.957
50160 95.611 -79.614 47.922
50161 95.634 -79.616 47.888
50162 95.658 -79.617 47.854
50163 95.679 -79.612 47.821
50164 95.694 -79.591 47.787
50165 95.709 -79.570 47.754
50166 95.724 -79.548 47.720
50167 95.739 -79.527 47.687
50168 95.754 -79.506 47.653
50169 95.772 -79.497 47.619
50170 95.795 -79.510 47.586
50171 95.818 -79.523 47.553
50172 95.840 -79.536 47.520
50173 95.863 -79.549 47.487
50174 95.886 -79.562 47.456
50175 95.909 -79.575 47.424
50176 95.932 -79.588 47.390
50177 95.955 -79.601 47.358
50178 95.977 -79.614 47.326
50179 96.000 -79.620 47.294
50180 96.022 -79.625 47.263
50181 96.045 -79.629 47.231
50182 96.067 -79.634 47.200
50211 96.719 -79.769 46.375
50212 96.742 -79.773 46.350
50213 96.764 -79.778 46.324
50214 96.787 -79.783 46.298
50215 96.809 -79.787 46.272
50216 96.831 -79.792 46.249
50217 96.854 -79.796 46.225
50218 96.877 -79.801 46.201
50219 96.899 -79.806 46.176
50220 96.922 -79.811 46.152
50221 96.944 -79.815 46.127
50222 96.982 -79.814 46.104
50223 97.029 -79.812 46.081
50224 97.076 -79.809 46.058
50225 97.122 -79.807 46.035
50226 97.125 -79.817 46.013
50227 97.124 -79.829 45.989
50228 97.123 -79.840 45.965
50229 97.126 -79.867 45.942
50230 97.130 -79.896 45.920
50231 97.134 -79.925 45.899
50232 97.137 -79.953 45.877
50233 97.141 -79.982 45.855
50234 97.144 -80.010 45.834
50647 93.142 -79.491 53.178
50648 93.156 -79.479 53.163
50649 93.170 -79.466 53.149
50650 93.185 -79.454 53.135
50651 93.199 -79.443 53.122
50652 93.215 -79.435 53.110
50653 93.231 -79.426 53.098
50654 93.247 -79.419 53.087
50655 93.264 -79.408 53.077
50656 93.282 -79.394 53.067
50657 93.300 -79.385 53.059
50658 93.319 -79.379 53.051
50659 93.338 -79.374 53.044
50660 93.356 -79.368 53.036
50661 93.373 -79.362 53.027
50662 93.388 -79.357 53.018
50663 93.403 -79.353 53.007
50664 93.418 -79.349 52.996
50665 93.426 -79.344 52.985
50666 93.434 -79.338 52.972
50667 93.442 -79.333 52.962
50668 93.450 -79.328 52.950
50669 93.462 -79.322 52.937
50670 93.476 -79.317 52.924
50671 93.491 -79.312 52.910
50672 93.506 -79.306 52.896
50673 93.521 -79.301 52.883
50674 93.538 -79.297 52.874
50675 93.554 -79.293 52.866
50676 93.570 -79.288 52.859
50677 93.586 -79.284 52.852
50678 93.601 -79.274 52.844
50679 93.615 -79.264 52.837
50680 93.630 -79.254 52.829
50681 93.644 -79.247 52.822
50682 93.658 -79.241 52.814
50683 93.672 -79.236 52.805
50684 93.686 -79.231 52.796
50685 93.700 -79.229 52.787
50686 93.714 -79.229 52.779
50687 93.728 -79.229 52.770
50688 93.742 -79.229 52.762
50689 93.756 -79.229 52.754
50690 93.770 -79.229 52.745
50691 93.780 -79.237 52.736
50692 93.792 -79.240 52.728
50693 93.805 -79.236 52.719
50694 93.819 -79.232 52.710
50695 93.832 -79.228 52.701
50696 93.845 -79.224 52.691
50697 93.858 -79.220 52.683
50698 93.871 -79.216 52.672
50699 93.885 -79.213 52.660
50700 93.899 -79.213 52.647
50701 93.899 -79.210 52.630
50702 93.910 -79.209 52.613
50703 93.921 -79.209 52.596
50704 93.931 -79.207 52.579
50705 93.942 -79.206 52.562
50706 93.954 -79.205 52.546
50707 93.964 -79.204 52.530
50708 93.975 -79.203 52.515
50709 93.986 -79.202 52.498
50710 93.997 -79.200 52.481
50711 94.007 -79.203 52.460
50712 94.008 -79.200 52.438
50713 94.008 -79.197 52.417
50714 94.008 -79.197 52.395
50715 94.008 -79.196 52.375
50716 94.008 -79.196 52.355
50717 94.008 -79.195 52.337
50718 94.008 -79.194 52.321
50719 94.008 -79.194 52.305
50720 94.008 -79.194 52.290
50721 94.008 -79.193 52.275
50722 94.008 -79.193 52.262
50723 94.008 -79.192 52.250
50724 94.008 -79.191 52.238
50725 94.008 -79.191 52.224
50726 94.012 -79.198 52.212
50727 94.016 -79.205 52.202
50728 94.020 -79.212 52.194
50729 94.018 -79.208 52.185
50730 94.014 -79.201 52.176
50731 94.016 -79.196 52.168
50732 94.028 -79.195 52.157
50733 94.040 -79.194 52.144
50734 94.052 -79.193 52.130
50735 94.060 -79.190 52.114
50736 94.066 -79.187 52.098
50737 94.071 -79.184 52.081
50738 94.077 -79.181 52.062
50739 94.084 -79.187 52.040
50740 94.091 -79.199 52.015
50741 94.098 -79.210 51.989
50742 94.105 -79.222 51.963
50743 94.110 -79.234 51.939
50744 94.114 -79.244 51.918
50745 94.114 -79.250 51.898
50746 94.114 -79.256 51.878
50747 94.114 -79.261 51.860
50748 94.114 -79.266 51.842
50749 94.111 -79.263 51.824
50750 94.109 -79.260 51.807
50751 94.106 -79.257 51.789
50752 94.104 -79.254 51.769
50753 94.103 -79.247 51.746
50754 94.107 -79.231 51.722
50755 94.111 -79.216 51.698
50756 94.117 -79.200 51.673
50757 94.127 -79.182 51.648
50758 94.137 -79.165 51.623
50759 94.147 -79.149 51.599
50760 94.160 -79.133 51.575
50761 94.175 -79.118 51.551
50762 94.187 -79.136 51.525
50763 94.199 -79.153 51.499
50764 94.211 -79.172 51.473
50765 94.235 -79.179 51.442
50766 94.260 -79.175 51.408
50767 94.280 -79.171 51.372
50768 94.306 -79.159 51.333
50769 94.333 -79.143 51.295
50770 94.360 -79.126 51.257
50771 94.388 -79.109 51.219
50772 94.415 -79.092 51.182
50773 94.442 -79.075 51.145
50774 94.453 -79.072 51.107
50775 94.465 -79.068 51.069
50776 94.476 -79.065 51.031
50777 94.487 -79.061 50.992
50778 94.498 -79.057 50.955
50779 94.509 -79.053 50.917
50780 94.521 -79.049 50.880
50781 94.532 -79.046 50.843
50829 94.952 -79.413 49.083
50830 94.965 -79.426 49.046
50831 94.978 -79.439 49.010
50832 94.991 -79.452 48.973
50833 95.004 -79.465 48.937
50834 95.017 -79.478 48.900
50835 95.031 -79.491 48.863
50836 95.044 -79.504 48.826
50837 95.057 -79.517 48.789
50838 95.070 -79.530 48.753
50839 95.083 -79.543 48.717
50840 95.106 -79.545 48.683
50841 95.130 -79.547 48.648
50842 95.153 -79.548 48.614
50843 95.177 -79.550 48.575
50844 95.200 -79.551 48.537
50845 95.224 -79.553 48.499
50846 95.248 -79.554 48.461
50847 95.271 -79.556 48.425
50848 95.295 -79.557 48.388
50849 95.318 -79.559 48.352
50850 95.342 -79.560 48.315
50851 95.366 -79.562 48.279
50852 95.389 -79.563 48.243
50853 95.412 -79.565 48.209
50854 95.436 -79.567 48.173
50855 95.460 -79.568 48.138
50856 95.483 -79.570 48.102
50857 95.507 -79.571 48.067
50858 95.530 -79.573 48.032
50859 95.554 -79.574 47.997
50860 95.578 -79.575 47.962
50861 95.601 -79.576 47.928
50862 95.625 -79.578 47.894
50863 95.648 -79.579 47.860
50864 95.672 -79.581 47.827
50865 95.687 -79.561 47.793
50866 95.702 -79.540 47.760
50867 95.717 -79.518 47.727
50868 95.732 -79.497 47.693
50869 95.747 -79.476 47.660
50870 95.768 -79.482 47.626
50871 95.792 -79.495 47.593
50872 95.814 -79.508 47.560
50873 95.837 -79.521 47.527
50874 95.860 -79.534 47.494
50875 95.883 -79.548 47.462
50876 95.905 -79.560 47.430
50877 95.928 -79.573 47.397
50878 95.951 -79.586 47.364
50879 95.974 -79.598 47.331
50880 95.996 -79.602 47.299
50881 96.019 -79.607 47.268
50882 96.041 -79.611 47.236
50883 96.063 -79.616 47.205
50884 96.086 -79.620 47.174
50911 96.693 -79.746 46.401
50912 96.716 -79.751 46.375
50913 96.738 -79.755 46.350
50914 96.760 -79.760 46.324
50915 96.783 -79.764 46.298
50916 96.806 -79.769 46.273
50917 96.828 -79.773 46.249
50918 96.851 -79.779 46.226
50919 96.873 -79.783 46.201
50920 96.895 -79.788 46.177
50921 96.918 -79.792 46.152
50922 96.940 -79.797 46.127
50923 96.979 -79.797 46.104
50924 97.026 -79.794 46.082
50925 97.072 -79.792 46.059
50926 97.118 -79.789 46.036
50927 97.123 -79.798 46.013
50928 97.122 -79.811 45.990
50929 97.121 -79.822 45.966
50930 97.124 -79.849 45.942
50931 97.127 -79.877 45.921
50932 97.131 -79.905 45.899
50933 97.135 -79.934 45.877
50934 97.138 -79.963 45.855
50935 97.145 -79.975 45.834
50936 97.152 -79.983 45.812
51348 93.141 -79.481 53.204
51349 93.155 -79.469 53.190
51350 93.169 -79.456 53.176
51351 93.184 -79.444 53.163
51352 93.198 -79.432 53.149
51353 93.212 -79.419 53.136
51354 93.227 -79.410 53.124
51355 93.243 -79.401 53.113
51356 93.259 -79.394 53.103
51357 93.276 -79.382 53.093
51358 93.295 -79.370 53.083
51359 93.313 -79.364 53.074
51360 93.332 -79.358 53.066
51361 93.350 -79.352 53.057
51362 93.365 -79.347 53.047
51363 93.380 -79.343 53.037
51364 93.395 -79.338 53.026
51365 93.410 -79.334 53.015
51366 93.420 -79.328 53.002
51367 93.428 -79.324 52.990
51368 93.436 -79.319 52.979
51369 93.444 -79.313 52.967
51370 93.457 -79.308 52.954
51371 93.472 -79.303 52.942
51372 93.487 -79.297 52.928
51373 93.502 -79.291 52.915
51374 93.516 -79.286 52.904
51375 93.532 -79.281 52.894
51376 93.548 -79.277 52.886
51377 93.564 -79.272 52.879
51378 93.580 -79.269 52.872
51379 93.595 -79.259 52.865
51380 93.610 -79.249 52.857
51381 93.625 -79.239 52.848
51382 93.638 -79.234 52.839
51383 93.652 -79.228 52.830
51384 93.666 -79.223 52.822
51385 93.681 -79.218 52.813
51386 93.695 -79.212 52.805
51387 93.709 -79.212 52.798
51388 93.722 -79.212 52.789
51389 93.736 -79.212 52.780
51390 93.751 -79.212 52.772
51391 93.762 -79.219 52.764
51392 93.772 -79.228 52.755
51393 93.785 -79.226 52.747
51394 93.798 -79.222 52.738
51395 93.811 -79.219 52.728
51396 93.824 -79.215 52.719
51397 93.837 -79.210 52.709
51398 93.850 -79.206 52.701
51399 93.864 -79.204 52.689
51400 93.878 -79.204 52.676
51401 93.893 -79.204 52.663
51402 93.890 -79.201 52.645
51403 93.894 -79.199 52.628
51404 93.905 -79.198 52.611
51405 93.916 -79.197 52.596
51406 93.926 -79.196 52.580
51407 93.937 -79.194 52.565
51408 93.948 -79.194 52.549
51409 93.960 -79.192 52.532
51410 93.970 -79.191 52.515
51411 93.990 -79.192 52.496
51412 93.997 -79.195 52.473
51413 93.999 -79.199 52.451
51414 94.001 -79.198 52.429
51415 94.002 -79.192 52.407
51416 94.003 -79.186 52.388
51417 94.003 -79.182 52.370
51418 94.003 -79.182 52.355
51419 94.003 -79.181 52.339
51420 94.003 -79.181 52.323
51421 94.003 -79.181 52.308
51422 94.003 -79.180 52.295
51423 94.003 -79.179 52.283
51424 94.003 -79.178 52.271
51425 94.003 -79.178 52.260
51426 94.003 -79.178 52.247
51427 94.006 -79.183 52.235
51428 94.010 -79.190 52.224
51429 94.014 -79.197 52.216
51430 94.018 -79.203 52.207
51431 94.017 -79.201 52.197
51432 94.018 -79.196 52.188
51433 94.030 -79.195 52.176
51434 94.041 -79.194 52.162
51435 94.046 -79.190 52.145
51436 94.052 -79.187 52.127
51437 94.057 -79.184 52.109
51438 94.063 -79.181 52.090
51439 94.068 -79.177 52.070
51440 94.075 -79.187 52.048
51441 94.083 -79.199 52.025
51442 94.090 -79.210 51.999
51443 94.097 -79.222 51.975
51444 94.102 -79.231 51.952
51445 94.102 -79.237 51.931
51446 94.102 -79.242 51.912
51447 94.102 -79.248 51.894
51448 94.102 -79.254 51.876
51449 94.102 -79.258 51.857
51450 94.099 -79.255 51.838
51451 94.097 -79.252 51.819
51452 94.094 -79.249 51.799
51453 94.091 -79.246 51.777
51454 94.089 -79.243 51.753
51455 94.086 -79.239 51.727
51456 94.091 -79.223 51.702
51457 94.101 -79.206 51.676
51458 94.111 -79.190 51.651
51459 94.121 -79.172 51.627
51460 94.130 -79.156 51.604
51461 94.140 -79.139 51.581
51462 94.154 -79.131 51.558
51463 94.167 -79.148 51.534
51464 94.188 -79.158 51.508
51465 94.216 -79.163 51.483
51466 94.243 -79.169 51.453
51467 94.266 -79.153 51.420
51468 94.283 -79.128 51.385
51469 94.306 -79.147 51.346
51470 94.331 -79.140 51.308
51471 94.358 -79.123 51.270
51472 94.386 -79.106 51.232
51473 94.413 -79.090 51.195
51474 94.440 -79.072 51.158
51475 94.459 -79.063 51.120
51476 94.470 -79.059 51.081
51477 94.481 -79.056 51.043
51478 94.492 -79.052 51.005
51479 94.504 -79.048 50.967
51480 94.515 -79.044 50.930
51481 94.526 -79.040 50.893
51482 94.538 -79.037 50.856
51483 94.549 -79.034 50.819
51529 94.933 -79.365 49.126
51530 94.943 -79.374 49.089
51531 94.956 -79.387 49.052
51532 94.969 -79.400 49.015
51533 94.983 -79.413 48.979
51534 94.996 -79.426 48.943
51535 95.009 -79.439 48.907
51536 95.022 -79.452 48.870
51537 95.035 -79.465 48.833
51538 95.048 -79.478 48.797
51539 95.061 -79.491 48.760
51540 95.074 -79.504 48.724
51541 95.096 -79.507 48.690
51542 95.120 -79.509 48.657
51543 95.143 -79.510 48.624
51544 95.167 -79.512 48.586
51545 95.191 -79.513 48.546
51546 95.214 -79.515 48.506
51547 95.238 -79.516 48.468
51548 95.261 -79.518 48.430
51549 95.285 -79.520 48.394
51550 95.309 -79.521 48.357
51551 95.332 -79.523 48.321
51552 95.355 -79.524 48.285
51553 95.379 -79.526 48.249
51554 95.403 -79.527 48.214
51555 95.426 -79.529 48.178
51556 95.450 -79.530 48.143
51557 95.473 -79.532 48.107
51558 95.497 -79.533 48.072
51559 95.521 -79.535 48.037
51560 95.544 -79.536 48.002
51561 95.568 -79.538 47.968
51562 95.591 -79.539 47.934
51563 95.615 -79.541 47.900
51564 95.638 -79.542 47.867
51565 95.662 -79.544 47.833
51566 95.680 -79.531 47.800
51567 95.695 -79.510 47.767
51568 95.710 -79.488 47.734
51569 95.726 -79.467 47.701
51570 95.742 -79.454 47.667
51571 95.765 -79.467 47.634
51572 95.788 -79.480 47.601
51573 95.811 -79.493 47.568
51574 95.833 -79.506 47.534
51575 95.856 -79.520 47.502
51576 95.879 -79.532 47.469
51577 95.902 -79.545 47.436
51578 95.925 -79.558 47.403
51579 95.948 -79.571 47.370
51580 95.970 -79.579 47.337
51581 95.992 -79.584 47.304
51582 96.015 -79.588 47.273
51583 96.038 -79.593 47.241
51584 96.060 -79.598 47.210
51585 96.083 -79.603 47.178
51611 96.667 -79.723 46.426
51612 96.689 -79.728 46.401
51613 96.712 -79.733 46.376
51614 96.735 -79.737 46.350
51615 96.757 -79.742 46.323
51616 96.780 -79.746 46.299
51617 96.802 -79.751 46.274
51618 96.824 -79.756 46.250
51619 96.847 -79.761 46.226
51620 96.869 -79.765 46.202
51621 96.892 -79.770 46.177
51622 96.914 -79.774 46.152
51623 96.937 -79.779 46.128
51624 96.976 -79.779 46.105
51625 97.022 -79.776 46.082
51626 97.069 -79.773 46.060
51627 97.115 -79.770 46.037
51628 97.120 -79.780 46.013
51629 97.120 -79.792 45.990
51630 97.119 -79.803 45.966
51631 97.122 -79.829 45.943
51632 97.125 -79.858 45.922
51633 97.129 -79.886 45.899
51634 97.132 -79.915 45.878
51635 97.138 -79.933 45.856
51636 97.145 -79.941 45.834
51637 97.152 -79.949 45.812
51638 97.159 -79.956 45.791
51639 97.166 -79.964 45.771
52050 93.154 -79.458 53.217
52051 93.168 -79.446 53.203
52052 93.183 -79.434 53.190
52053 93.197 -79.422 53.177
52054 93.211 -79.410 53.164
52055 93.225 -79.397 53.151
52056 93.239 -79.385 53.140
52057 93.255 -79.376 53.129
52058 93.271 -79.369 53.117
52059 93.289 -79.356 53.107
52060 93.307 -79.348 53.097
52061 93.326 -79.342 53.088
52062 93.342 -79.338 53.077
52063 93.357 -79.333 53.067
52064 93.372 -79.328 53.057
52065 93.387 -79.324 53.045
52066 93.402 -79.319 53.033
52067 93.413 -79.314 53.020
52068 93.421 -79.309 53.008
52069 93.429 -79.303 52.996
52070 93.438 -79.299 52.984
52071 93.453 -79.293 52.972
52072 93.468 -79.288 52.959
52073 93.483 -79.282 52.946
52074 93.497 -79.277 52.934
52075 93.512 -79.272 52.923
52076 93.527 -79.266 52.915
52077 93.543 -79.262 52.907
52078 93.559 -79.257 52.899
52079 93.575 -79.253 52.893
52080 93.590 -79.244 52.885
52081 93.605 -79.234 52.877
52082 93.619 -79.227 52.868
52083 93.633 -79.222 52.858
52084 93.647 -79.216 52.848
52085 93.661 -79.210 52.839
52086 93.675 -79.205 52.831
52087 93.689 -79.200 52.824
52088 93.703 -79.196 52.817
52089 93.717 -79.196 52.809
52090 93.731 -79.196 52.800
52091 93.743 -79.200 52.792
52092 93.753 -79.209 52.783
52093 93.764 -79.216 52.775
52094 93.777 -79.212 52.767
52095 93.790 -79.209 52.757
52096 93.803 -79.205 52.748
52097 93.817 -79.201 52.738
52098 93.830 -79.197 52.730
52099 93.844 -79.196 52.720
52100 93.858 -79.195 52.707
52101 93.873 -79.195 52.693
52102 93.888 -79.195 52.679
52103 93.884 -79.192 52.661
52104 93.878 -79.188 52.644
52105 93.889 -79.187 52.629
52106 93.900 -79.186 52.614
52107 93.911 -79.185 52.600
52108 93.922 -79.184 52.585
52109 93.932 -79.183 52.569
52110 93.943 -79.181 52.551
52111 93.962 -79.182 52.531
52112 93.981 -79.184 52.510
52113 93.988 -79.186 52.486
52114 93.990 -79.190 52.463
52115 93.992 -79.195 52.441
52116 93.994 -79.197 52.422
52117 93.995 -79.190 52.405
52118 93.996 -79.184 52.389
52119 93.997 -79.178 52.376
52120 93.997 -79.172 52.360
52121 93.998 -79.168 52.345
52122 93.998 -79.167 52.330
52123 93.998 -79.166 52.317
52124 93.998 -79.166 52.306
52125 93.998 -79.165 52.295
52126 93.998 -79.165 52.284
52127 93.998 -79.165 52.272
52128 94.000 -79.169 52.260
52129 94.004 -79.175 52.249
52130 94.008 -79.182 52.241
52131 94.012 -79.189 52.232
52132 94.016 -79.196 52.222
52133 94.019 -79.196 52.211
52134 94.027 -79.194 52.197
52135 94.032 -79.190 52.180
52136 94.038 -79.187 52.161
52137 94.043 -79.184 52.142
52138 94.049 -79.180 52.122
52139 94.054 -79.177 52.102
52140 94.060 -79.175 52.081
52141 94.067 -79.187 52.060
52142 94.074 -79.198 52.036
52143 94.082 -79.209 52.013
52144 94.086 -79.218 51.990
52145 94.089 -79.225 51.969
52146 94.089 -79.230 51.949
52147 94.089 -79.235 51.931
52148 94.089 -79.241 51.913
52149 94.090 -79.247 51.894
52150 94.089 -79.250 51.874
52151 94.086 -79.247 51.853
52152 94.084 -79.244 51.832
52153 94.082 -79.241 51.809
52154 94.079 -79.237 51.784
52155 94.077 -79.234 51.759
52156 94.077 -79.231 51.732
52157 94.081 -79.223 51.706
52158 94.090 -79.207 51.680
52159 94.099 -79.191 51.655
52160 94.108 -79.175 51.631
52161 94.117 -79.159 51.610
52162 94.126 -79.144 51.589
52163 94.142 -79.136 51.568
52164 94.170 -79.142 51.545
52165 94.197 -79.147 51.521
52166 94.224 -79.153 51.496
52167 94.252 -79.159 51.466
52168 94.269 -79.131 51.434
52169 94.282 -79.103 51.400
52170 94.308 -79.107 51.361
52171 94.332 -79.122 51.322
52172 94.356 -79.120 51.284
52173 94.383 -79.103 51.245
52174 94.410 -79.087 51.208
52175 94.437 -79.070 51.170
52176 94.464 -79.053 51.133
52177 94.475 -79.050 51.094
52178 94.486 -79.046 51.055
52179 94.498 -79.043 51.018
52180 94.509 -79.039 50.980
52181 94.521 -79.035 50.943
52182 94.532 -79.031 50.906
52183 94.543 -79.028 50.869
52184 94.554 -79.024 50.832
52185 94.565 -79.021 50.795
52186 94.577 -79.017 50.757
52228 94.925 -79.329 49.206
52229 94.933 -79.337 49.169
52230 94.942 -79.344 49.132
52231 94.950 -79.352 49.095
52232 94.958 -79.360 49.059
52233 94.967 -79.366 49.023
52234 94.975 -79.374 48.987
52235 94.987 -79.387 48.950
52236 95.000 -79.400 48.914
52237 95.014 -79.413 48.878
52238 95.027 -79.426 48.842
52239 95.040 -79.438 48.805
52240 95.053 -79.451 48.768
52241 95.066 -79.464 48.732
52242 95.087 -79.469 48.697
52243 95.110 -79.471 48.664
52244 95.133 -79.473 48.630
52245 95.157 -79.474 48.591
52246 95.181 -79.476 48.552
52247 95.204 -79.477 48.513
52248 95.228 -79.479 48.475
52249 95.252 -79.480 48.438
52250 95.275 -79.482 48.401
52251 95.299 -79.483 48.364
52252 95.322 -79.485 48.327
52253 95.346 -79.486 48.291
52254 95.370 -79.488 48.255
52255 95.393 -79.489 48.219
52256 95.416 -79.491 48.184
52257 95.440 -79.492 48.148
52258 95.464 -79.494 48.112
52259 95.487 -79.495 48.077
52260 95.511 -79.497 48.042
52261 95.534 -79.498 48.007
52262 95.558 -79.500 47.974
52263 95.582 -79.501 47.940
52264 95.605 -79.503 47.907
52265 95.629 -79.504 47.874
52266 95.652 -79.506 47.841
52267 95.673 -79.501 47.808
52268 95.688 -79.479 47.775
52269 95.703 -79.458 47.741
52270 95.719 -79.437 47.708
52271 95.739 -79.439 47.674
52272 95.761 -79.452 47.641
52273 95.784 -79.465 47.609
52274 95.807 -79.478 47.575
52275 95.830 -79.491 47.542
52276 95.853 -79.504 47.509
52277 95.876 -79.517 47.475
52278 95.898 -79.530 47.442
52279 95.921 -79.543 47.410
52280 95.944 -79.557 47.376
52281 95.967 -79.561 47.343
52282 95.989 -79.566 47.310
52283 96.012 -79.570 47.277
52284 96.034 -79.576 47.245
52285 96.056 -79.580 47.214
52286 96.079 -79.585 47.182
52287 96.102 -79.589 47.150
52308 96.574 -79.687 46.532
52309 96.596 -79.692 46.505
52310 96.618 -79.696 46.477
52311 96.641 -79.701 46.452
52312 96.664 -79.705 46.427
52313 96.686 -79.710 46.403
52314 96.709 -79.714 46.376
52315 96.731 -79.719 46.350
52316 96.753 -79.723 46.325
52317 96.776 -79.729 46.300
52318 96.798 -79.733 46.276
52319 96.821 -79.738 46.252
52320 96.843 -79.742 46.227
52321 96.866 -79.747 46.203
52322 96.888 -79.751 46.179
52323 96.911 -79.756 46.154
52324 96.933 -79.761 46.130
52325 96.972 -79.761 46.106
52326 97.019 -79.758 46.084
52327 97.065 -79.755 46.062
52328 97.112 -79.752 46.039
52329 97.118 -79.762 46.016
52330 97.117 -79.773 45.992
52331 97.117 -79.785 45.968
52332 97.119 -79.810 45.944
52333 97.123 -79.839 45.922
52334 97.126 -79.867 45.900
52335 97.131 -79.891 45.878
52336 97.138 -79.899 45.857
52337 97.145 -79.906 45.834
52338 97.152 -79.914 45.812
52339 97.159 -79.921 45.791
52340 97.166 -79.929 45.771
52341 97.173 -79.936 45.750
52751 93.153 -79.447 53.244
52752 93.167 -79.435 53.230
52753 93.182 -79.423 53.217
52754 93.196 -79.411 53.204
52755 93.210 -79.399 53.191
52756 93.224 -79.387 53.178
52757 93.238 -79.375 53.166
52758 93.252 -79.363 53.155
52759 93.267 -79.351 53.142
52760 93.283 -79.343 53.131
52761 93.301 -79.333 53.121
52762 93.320 -79.327 53.110
52763 93.335 -79.322 53.099
52764 93.350 -79.318 53.088
52765 93.364 -79.313 53.077
52766 93.379 -79.309 53.066
52767 93.395 -79.304 53.054
52768 93.407 -79.300 53.040
52769 93.415 -79.294 53.026
52770 93.423 -79.289 53.013
52771 93.434 -79.284 53.000
52772 93.448 -79.278 52.988
52773 93.464 -79.273 52.976
52774 93.478 -79.268 52.964
52775 93.493 -79.263 52.953
52776 93.508 -79.257 52.943
52777 93.523 -79.252 52.935
52778 93.538 -79.247 52.927
52779 93.553 -79.242 52.919
52780 93.570 -79.237 52.913
52781 93.585 -79.230 52.905
52782 93.600 -79.219 52.897
52783 93.614 -79.214 52.887
52784 93.628 -79.209 52.877
52785 93.642 -79.203 52.867
52786 93.655 -79.198 52.858
52787 93.670 -79.193 52.850
52788 93.684 -79.187 52.843
52789 93.698 -79.182 52.836
52790 93.712 -79.178 52.829
52791 93.724 -79.182 52.821
52792 93.734 -79.191 52.812
52793 93.745 -79.200 52.804
52794 93.757 -79.203 52.796
52795 93.770 -79.199 52.787
52796 93.783 -79.195 52.778
52797 93.796 -79.191 52.768
52798 93.809 -79.187 52.759
52799 93.824 -79.187 52.751
52800 93.838 -79.187 52.741
52801 93.853 -79.186 52.728
52802 93.867 -79.186 52.713
52803 93.882 -79.186 52.698
52804 93.879 -79.183 52.682
52805 93.870 -79.179 52.666
52806 93.873 -79.177 52.651
52807 93.884 -79.175 52.637
52808 93.895 -79.175 52.623
52809 93.906 -79.173 52.607
52810 93.917 -79.172 52.589
52811 93.934 -79.172 52.570
52812 93.955 -79.174 52.549
52813 93.970 -79.175 52.527
52814 93.978 -79.178 52.503
52815 93.981 -79.181 52.480
52816 93.983 -79.186 52.459
52817 93.985 -79.190 52.442
52818 93.987 -79.194 52.426
52819 93.988 -79.190 52.412
52820 93.989 -79.184 52.399
52821 93.990 -79.177 52.384
52822 93.991 -79.171 52.369
52823 93.991 -79.165 52.356
52824 93.992 -79.159 52.342
52825 93.993 -79.153 52.331
52826 93.993 -79.153 52.321
52827 93.993 -79.152 52.310
52828 93.993 -79.151 52.299
52829 93.994 -79.153 52.287
52830 93.998 -79.160 52.276
52831 94.002 -79.167 52.269
52832 94.006 -79.174 52.259
52833 94.011 -79.184 52.247
52834 94.016 -79.193 52.235
52835 94.019 -79.190 52.219
52836 94.024 -79.187 52.201
52837 94.029 -79.184 52.180
52838 94.035 -79.180 52.160
52839 94.040 -79.177 52.139
52840 94.046 -79.174 52.118
52841 94.052 -79.175 52.098
52842 94.059 -79.187 52.076
52843 94.065 -79.197 52.053
52844 94.069 -79.204 52.030
52845 94.073 -79.212 52.008
52846 94.076 -79.219 51.987
52847 94.076 -79.224 51.967
52848 94.077 -79.228 51.950
52849 94.077 -79.234 51.932
52850 94.078 -79.239 51.912
52851 94.077 -79.241 51.891
52852 94.074 -79.238 51.868
52853 94.073 -79.235 51.844
52854 94.073 -79.231 51.818
52855 94.074 -79.228 51.792
52856 94.075 -79.225 51.765
52857 94.076 -79.221 51.736
52858 94.079 -79.213 51.709
52859 94.088 -79.197 51.684
52860 94.097 -79.182 51.661
52861 94.106 -79.166 51.640
52862 94.115 -79.150 51.620
52863 94.127 -79.132 51.602
52864 94.150 -79.112 51.583
52865 94.178 -79.123 51.563
52866 94.205 -79.134 51.540
52867 94.232 -79.143 51.516
52868 94.256 -79.137 51.487
52869 94.269 -79.109 51.455
52870 94.281 -79.081 51.420
52871 94.305 -79.078 51.379
52872 94.332 -79.086 51.338
52873 94.358 -79.096 51.298
52874 94.381 -79.100 51.259
52875 94.408 -79.084 51.221
52876 94.435 -79.067 51.183
52877 94.463 -79.050 51.146
52878 94.481 -79.041 51.107
52879 94.492 -79.037 51.069
52880 94.503 -79.034 51.030
52881 94.514 -79.030 50.993
52882 94.526 -79.026 50.956
52883 94.537 -79.022 50.919
52884 94.548 -79.019 50.883
52885 94.559 -79.015 50.845
52886 94.571 -79.012 50.807
52887 94.582 -79.008 50.770
52888 94.594 -79.004 50.733
52928 94.925 -79.302 49.249
52929 94.933 -79.309 49.213
52930 94.942 -79.316 49.177
52931 94.950 -79.324 49.140
52932 94.958 -79.332 49.103
52933 94.967 -79.339 49.067
52934 94.975 -79.347 49.030
52935 94.983 -79.354 48.994
52936 94.991 -79.361 48.958
52937 94.999 -79.369 48.921
52938 95.008 -79.376 48.886
52939 95.019 -79.386 48.850
52940 95.032 -79.399 48.813
52941 95.045 -79.412 48.777
52942 95.058 -79.426 48.741
52943 95.076 -79.432 48.705
52944 95.100 -79.433 48.670
52945 95.124 -79.435 48.635
52946 95.147 -79.436 48.597
52947 95.171 -79.438 48.559
52948 95.194 -79.439 48.520
52949 95.218 -79.441 48.483
52950 95.242 -79.442 48.445
52951 95.265 -79.444 48.408
52952 95.289 -79.445 48.370
52953 95.313 -79.447 48.334
52954 95.336 -79.448 48.297
52955 95.359 -79.450 48.260
52956 95.383 -79.451 48.225
52957 95.407 -79.453 48.190
52958 95.430 -79.454 48.154
52959 95.454 -79.456 48.118
52960 95.477 -79.457 48.083
52961 95.501 -79.459 48.048
52962 95.525 -79.460 48.014
52963 95.548 -79.462 47.980
52964 95.572 -79.463 47.946
52965 95.595 -79.465 47.913
52966 95.619 -79.466 47.880
52967 95.642 -79.468 47.847
52968 95.666 -79.469 47.815
52969 95.682 -79.449 47.782
52970 95.696 -79.428 47.750
52971 95.712 -79.411 47.717
52972 95.735 -79.424 47.684
52973 95.758 -79.437 47.650
52974 95.781 -79.450 47.618
52975 95.804 -79.463 47.584
52976 95.827 -79.476 47.550
52977 95.849 -79.489 47.517
52978 95.872 -79.502 47.483
52979 95.895 -79.516 47.450
52980 95.918 -79.529 47.416
52981 95.941 -79.538 47.382
52982 95.963 -79.543 47.348
52983 95.985 -79.548 47.315
52984 96.008 -79.553 47.282
52985 96.031 -79.557 47.250
52986 96.053 -79.562 47.218
52987 96.076 -79.567 47.186
52988 96.098 -79.571 47.154
53005 96.480 -79.650 46.640
53006 96.503 -79.654 46.613
53007 96.525 -79.659 46.586
53008 96.547 -79.664 46.559
53009 96.570 -79.669 46.532
53010 96.592 -79.673 46.505
53011 96.615 -79.678 46.478
53012 96.638 -79.683 46.452
53013 96.660 -79.687 46.428
53014 96.682 -79.692 46.403
53015 96.705 -79.696 46.378
53016 96.727 -79.701 46.351
53017 96.750 -79.706 46.326
53018 96.772 -79.711 46.301
53019 96.795 -79.715 46.277
53020 96.817 -79.720 46.253
53021 96.840 -79.724 46.229
53022 96.862 -79.729 46.205
53023 96.885 -79.733 46.181
53024 96.907 -79.739 46.157
53025 96.930 -79.743 46.132
53026 96.969 -79.742 46.109
53027 97.015 -79.740 46.087
53028 97.062 -79.737 46.065
53029 97.108 -79.735 46.042
53030 97.116 -79.743 46.018
53031 97.115 -79.755 45.995
53032 97.114 -79.767 45.970
53033 97.117 -79.791 45.946
53034 97.120 -79.819 45.924
53035 97.124 -79.848 45.901
53036 97.131 -79.857 45.879
53037 97.138 -79.864 45.857
53038 97.145 -79.872 45.834
53039 97.152 -79.880 45.812
53040 97.159 -79.887 45.791
53041 97.166 -79.895 45.770
53042 97.173 -79.902 45.749
53043 97.180 -79.910 45.729
53452 93.153 -79.438 53.271
53453 93.166 -79.425 53.257
53454 93.181 -79.413 53.244
53455 93.195 -79.400 53.231
53456 93.209 -79.388 53.218
53457 93.223 -79.376 53.205
53458 93.237 -79.364 53.193
53459 93.251 -79.352 53.181
53460 93.265 -79.340 53.167
53461 93.279 -79.328 53.156
53462 93.295 -79.317 53.144
53463 93.312 -79.313 53.132
53464 93.327 -79.308 53.121
53465 93.342 -79.303 53.109
53466 93.357 -79.299 53.098
53467 93.371 -79.294 53.087
53468 93.386 -79.290 53.074
53469 93.401 -79.285 53.061
53470 93.409 -79.280 53.047
53471 93.417 -79.275 53.034
53472 93.429 -79.269 53.022
53473 93.444 -79.264 53.010
53474 93.459 -79.259 52.997
53475 93.474 -79.253 52.985
53476 93.489 -79.248 52.974
53477 93.504 -79.242 52.963
53478 93.518 -79.237 52.955
53479 93.533 -79.231 52.947
53480 93.548 -79.226 52.940
53481 93.564 -79.222 52.932
53482 93.579 -79.215 52.924
53483 93.594 -79.206 52.916
53484 93.608 -79.201 52.907
53485 93.622 -79.196 52.897
53486 93.636 -79.190 52.888
53487 93.650 -79.185 52.879
53488 93.664 -79.180 52.871
53489 93.678 -79.175 52.864
53490 93.692 -79.170 52.856
53491 93.705 -79.167 52.849
53492 93.716 -79.173 52.842
53493 93.726 -79.182 52.834
53494 93.736 -79.190 52.826
53495 93.749 -79.189 52.817
53496 93.762 -79.185 52.808
53497 93.775 -79.181 52.799
53498 93.789 -79.178 52.790
53499 93.803 -79.178 52.781
53500 93.818 -79.178 52.772
53501 93.833 -79.178 52.761
53502 93.847 -79.177 52.749
53503 93.862 -79.177 52.735
53504 93.876 -79.177 52.719
53505 93.873 -79.174 52.703
53506 93.864 -79.170 52.688
53507 93.857 -79.166 52.674
53508 93.868 -79.165 52.660
53509 93.879 -79.164 52.644
53510 93.890 -79.162 52.628
53511 93.906 -79.162 52.609
53512 93.926 -79.164 52.589
53513 93.946 -79.165 52.569
53514 93.960 -79.167 52.548
53515 93.967 -79.170 52.526
53516 93.972 -79.173 52.504
53517 93.974 -79.178 52.484
53518 93.976 -79.181 52.468
53519 93.978 -79.186 52.453
53520 93.980 -79.190 52.440
53521 93.981 -79.188 52.427
53522 93.982 -79.182 52.412
53523 93.983 -79.176 52.398
53524 93.984 -79.170 52.385
53525 93.985 -79.164 52.371
53526 93.986 -79.157 52.359
53527 93.986 -79.151 52.348
53528 93.987 -79.145 52.339
53529 93.988 -79.139 52.328
53530 93.989 -79.138 52.318
53531 93.993 -79.147 52.307
53532 93.998 -79.156 52.297
53533 94.003 -79.166 52.287
53534 94.008 -79.176 52.274
53535 94.013 -79.186 52.261
53536 94.017 -79.185 52.244
53537 94.020 -79.182 52.224
53538 94.024 -79.179 52.201
53539 94.028 -79.176 52.180
53540 94.032 -79.174 52.159
53541 94.037 -79.170 52.138
53542 94.044 -79.175 52.117
53543 94.048 -79.183 52.095
53544 94.052 -79.190 52.073
53545 94.056 -79.198 52.050
53546 94.060 -79.206 52.028
53547 94.063 -79.213 52.007
53548 94.064 -79.218 51.986
53549 94.064 -79.223 51.969
53550 94.065 -79.228 51.950
53551 94.065 -79.233 51.930
53552 94.070 -79.233 51.908
53553 94.070 -79.229 51.883
53554 94.071 -79.225 51.857
53555 94.072 -79.222 51.830
53556 94.073 -79.218 51.802
53557 94.073 -79.215 51.773
53558 94.074 -79.211 51.743
53559 94.077 -79.204 51.716
53560 94.086 -79.188 51.691
53561 94.095 -79.172 51.670
53562 94.104 -79.156 51.651
53563 94.114 -79.140 51.634
53564 94.135 -79.115 51.618
53565 94.157 -79.090 51.600
53566 94.185 -79.099 51.580
53567 94.212 -79.109 51.559
53568 94.240 -79.120 51.535
53569 94.257 -79.112 51.507
53570 94.269 -79.087 51.474
53571 94.281 -79.059 51.438
53572 94.301 -79.048 51.396
53573 94.328 -79.057 51.354
53574 94.355 -79.065 51.313
53575 94.383 -79.074 51.273
53576 94.406 -79.081 51.234
53577 94.433 -79.065 51.197
53578 94.460 -79.047 51.159
53579 94.486 -79.032 51.121
53580 94.497 -79.028 51.082
53581 94.508 -79.024 51.044
53582 94.520 -79.021 51.007
53583 94.531 -79.017 50.970
53584 94.542 -79.013 50.933
53585 94.554 -79.010 50.896
53586 94.565 -79.006 50.858
53587 94.576 -79.002 50.821
53588 94.588 -78.999 50.783
53589 94.599 -78.995 50.746
53590 94.610 -78.991 50.709
53591 94.619 -78.997 50.672
53628 94.925 -79.274 49.293
53629 94.933 -79.281 49.256
53630 94.942 -79.289 49.220
53631 94.950 -79.296 49.184
53632 94.958 -79.303 49.149
53633 94.967 -79.311 49.111
53634 94.975 -79.319 49.075
53635 94.983 -79.326 49.039
53636 94.991 -79.334 49.002
53637 95.000 -79.341 48.966
53638 95.008 -79.349 48.930
53639 95.017 -79.356 48.894
53640 95.025 -79.363 48.858
53641 95.033 -79.371 48.822
53642 95.041 -79.379 48.786
53643 95.049 -79.386 48.750
53644 95.067 -79.394 48.713
53645 95.091 -79.396 48.677
53646 95.114 -79.397 48.640
53647 95.137 -79.399 48.603
53648 95.161 -79.400 48.566
53649 95.185 -79.402 48.528
53650 95.208 -79.403 48.490
53651 95.232 -79.404 48.452
53652 95.255 -79.406 48.415
53653 95.279 -79.407 48.378
53654 95.303 -79.409 48.341
53655 95.326 -79.410 48.303
53656 95.350 -79.412 48.266
53657 95.374 -79.413 48.232
53658 95.397 -79.415 48.195
53659 95.420 -79.416 48.159
53660 95.444 -79.418 48.124
53661 95.468 -79.419 48.089
53662 95.491 -79.421 48.055
53663 95.515 -79.422 48.020
53664 95.538 -79.424 47.986
53665 95.562 -79.425 47.952
53666 95.586 -79.427 47.920
53667 95.609 -79.429 47.887
53668 95.633 -79.430 47.854
53669 95.656 -79.432 47.822
53670 95.675 -79.419 47.790
53671 95.690 -79.398 47.759
53672 95.709 -79.396 47.727
53673 95.732 -79.409 47.694
53674 95.755 -79.422 47.662
53675 95.777 -79.435 47.628
53676 95.800 -79.448 47.594
53677 95.823 -79.461 47.559
53678 95.846 -79.474 47.525
53679 95.869 -79.488 47.491
53680 95.892 -79.501 47.457
53681 95.914 -79.513 47.423
53682 95.937 -79.521 47.388
53683 95.959 -79.526 47.353
53684 95.982 -79.530 47.319
53685 96.005 -79.535 47.286
53686 96.027 -79.539 47.254
53687 96.049 -79.544 47.222
53688 96.072 -79.548 47.190
53689 96.094 -79.553 47.157
53704 96.432 -79.623 46.698
53705 96.454 -79.627 46.669
53706 96.476 -79.632 46.640
53707 96.499 -79.637 46.613
53708 96.521 -79.642 46.586
53709 96.544 -79.646 46.559
53710 96.567 -79.651 46.532
53711 96.589 -79.655 46.506
53712 96.611 -79.660 46.479
53713 96.634 -79.664 46.453
53714 96.656 -79.669 46.429
53715 96.679 -79.674 46.404
53716 96.701 -79.679 46.379
53717 96.724 -79.683 46.353
53718 96.746 -79.688 46.327
53719 96.768 -79.692 46.302
53720 96.791 -79.697 46.278
53721 96.814 -79.701 46.254
53722 96.836 -79.706 46.231
53723 96.859 -79.711 46.207
53724 96.881 -79.716 46.183
53725 96.903 -79.720 46.161
53726 96.926 -79.725 46.137
53727 96.965 -79.724 46.114
53728 97.012 -79.722 46.093
53729 97.058 -79.719 46.069
53730 97.105 -79.717 46.045
53731 97.114 -79.725 46.021
53732 97.113 -79.736 45.998
53733 97.112 -79.748 45.973
53734 97.114 -79.771 45.949
53735 97.118 -79.800 45.925
53736 97.124 -79.814 45.903
53737 97.131 -79.822 45.880
53738 97.138 -79.830 45.856
53739 97.145 -79.837 45.834
53740 97.152 -79.845 45.812
53741 97.159 -79.852 45.790
53742 97.166 -79.860 45.770
53743 97.173 -79.867 45.749
53744 97.180 -79.875 45.728
53745 97.187 -79.883 45.707
54154 93.166 -79.415 53.285
54155 93.180 -79.403 53.272
54156 93.194 -79.391 53.258
54157 93.208 -79.379 53.245
54158 93.222 -79.366 53.232
54159 93.236 -79.354 53.219
54160 93.250 -79.341 53.206
54161 93.264 -79.329 53.193
54162 93.278 -79.317 53.181
54163 93.292 -79.305 53.168
54164 93.304 -79.297 53.156
54165 93.319 -79.293 53.142
54166 93.334 -79.288 53.130
54167 93.349 -79.284 53.119
54168 93.364 -79.279 53.108
54169 93.379 -79.275 53.095
54170 93.394 -79.270 53.081
54171 93.402 -79.266 53.068
54172 93.410 -79.260 53.056
54173 93.425 -79.255 53.045
54174 93.440 -79.250 53.033
54175 93.455 -79.244 53.020
54176 93.470 -79.238 53.009
54177 93.485 -79.233 52.997
54178 93.499 -79.228 52.987
54179 93.514 -79.222 52.977
54180 93.529 -79.217 52.969
54181 93.544 -79.212 52.962
54182 93.559 -79.206 52.954
54183 93.574 -79.200 52.946
54184 93.588 -79.194 52.938
54185 93.603 -79.188 52.928
54186 93.617 -79.183 52.918
54187 93.630 -79.178 52.909
54188 93.643 -79.175 52.901
54189 93.656 -79.172 52.893
54190 93.670 -79.169 52.886
54191 93.683 -79.165 52.880
54192 93.696 -79.162 52.873
54193 93.707 -79.164 52.865
54194 93.718 -79.172 52.857
54195 93.728 -79.179 52.848
54196 93.742 -79.175 52.840
54197 93.755 -79.172 52.831
54198 93.769 -79.169 52.822
54199 93.783 -79.169 52.812
54200 93.798 -79.169 52.802
54201 93.813 -79.169 52.793
54202 93.827 -79.169 52.781
54203 93.842 -79.168 52.768
54204 93.856 -79.168 52.755
54205 93.870 -79.168 52.741
54206 93.868 -79.165 52.726
54207 93.859 -79.161 52.712
54208 93.850 -79.157 52.698
54209 93.852 -79.154 52.682
54210 93.863 -79.153 52.666
54211 93.878 -79.153 52.648
54212 93.898 -79.154 52.629
54213 93.919 -79.156 52.610
54214 93.939 -79.156 52.591
54215 93.949 -79.159 52.572
54216 93.957 -79.162 52.550
54217 93.963 -79.165 52.530
54218 93.965 -79.169 52.512
54219 93.967 -79.173 52.497
54220 93.969 -79.178 52.483
54221 93.971 -79.181 52.470
54222 93.973 -79.186 52.457
54223 93.974 -79.187 52.443
54224 93.975 -79.181 52.429
54225 93.976 -79.175 52.416
54226 93.977 -79.169 52.402
54227 93.978 -79.162 52.389
54228 93.979 -79.156 52.377
54229 93.980 -79.150 52.368
54230 93.980 -79.144 52.358
54231 93.984 -79.139 52.348
54232 93.990 -79.139 52.338
54233 93.995 -79.149 52.327
54234 94.000 -79.159 52.316
54235 94.005 -79.169 52.302
54236 94.010 -79.178 52.289
54237 94.014 -79.181 52.271
54238 94.017 -79.178 52.250
54239 94.021 -79.175 52.227
54240 94.025 -79.172 52.205
54241 94.029 -79.169 52.183
54242 94.032 -79.166 52.161
54243 94.031 -79.169 52.139
54244 94.035 -79.177 52.117
54245 94.039 -79.184 52.093
54246 94.043 -79.192 52.071
54247 94.047 -79.200 52.048
54248 94.050 -79.207 52.026
54249 94.051 -79.212 52.005
54250 94.051 -79.217 51.986
54251 94.052 -79.222 51.967
54252 94.059 -79.223 51.947
54253 94.070 -79.222 51.924
54254 94.070 -79.219 51.898
54255 94.070 -79.216 51.871
54256 94.070 -79.212 51.843
54257 94.071 -79.208 51.814
54258 94.072 -79.204 51.783
54259 94.072 -79.201 51.752
54260 94.075 -79.194 51.726
54261 94.084 -79.179 51.704
54262 94.093 -79.163 51.683
54263 94.102 -79.147 51.665
54264 94.122 -79.123 51.651
54265 94.144 -79.098 51.636
54266 94.166 -79.072 51.618
54267 94.192 -79.074 51.599
54268 94.220 -79.084 51.577
54269 94.248 -79.095 51.554
54270 94.261 -79.080 51.525
54271 94.271 -79.059 51.492
54272 94.281 -79.037 51.454
54273 94.298 -79.019 51.412
54274 94.325 -79.028 51.370
54275 94.352 -79.036 51.328
54276 94.380 -79.044 51.287
54277 94.407 -79.053 51.248
54278 94.430 -79.062 51.210
54279 94.458 -79.045 51.173
54280 94.485 -79.028 51.135
54281 94.502 -79.019 51.096
54282 94.514 -79.015 51.058
54283 94.525 -79.012 51.020
54284 94.537 -79.008 50.983
54285 94.548 -79.004 50.946
54286 94.559 -79.001 50.909
54287 94.570 -78.997 50.871
54288 94.581 -78.993 50.834
54289 94.593 -78.990 50.796
54290 94.604 -78.986 50.759
54291 94.615 -78.982 50.722
54292 94.627 -78.978 50.685
54293 94.635 -78.984 50.647
54327 94.917 -79.238 49.375
54328 94.925 -79.246 49.338
54329 94.933 -79.253 49.301
54330 94.942 -79.261 49.264
54331 94.950 -79.269 49.228
54332 94.959 -79.276 49.193
54333 94.967 -79.284 49.157
54334 94.975 -79.291 49.120
54335 94.983 -79.298 49.083
54336 94.991 -79.306 49.048
54337 95.000 -79.313 49.012
54338 95.008 -79.321 48.975
54339 95.017 -79.328 48.939
54340 95.025 -79.336 48.903
54341 95.033 -79.343 48.867
54342 95.041 -79.350 48.832
54343 95.052 -79.366 48.796
54344 95.068 -79.385 48.760
54345 95.082 -79.385 48.723
54346 95.105 -79.389 48.686
54347 95.136 -79.394 48.649
54348 95.164 -79.394 48.611
54349 95.181 -79.390 48.574
54350 95.198 -79.385 48.537
54351 95.215 -79.381 48.499
54352 95.232 -79.376 48.460
54353 95.249 -79.372 48.423
54354 95.269 -79.370 48.386
54355 95.293 -79.372 48.348
54356 95.316 -79.372 48.310
54357 95.340 -79.374 48.272
54358 95.364 -79.375 48.237
54359 95.387 -79.377 48.201
54360 95.411 -79.379 48.165
54361 95.435 -79.380 48.130
54362 95.458 -79.382 48.095
54363 95.481 -79.383 48.059
54364 95.505 -79.385 48.025
54365 95.529 -79.386 47.991
54366 95.552 -79.388 47.958
54367 95.576 -79.389 47.927
54368 95.599 -79.391 47.895
54369 95.623 -79.392 47.863
54370 95.647 -79.394 47.831
54371 95.668 -79.389 47.798
54372 95.683 -79.368 47.769
54373 95.705 -79.381 47.739
54374 95.729 -79.394 47.708
54375 95.751 -79.407 47.676
54376 95.774 -79.420 47.641
54377 95.797 -79.433 47.605
54378 95.820 -79.446 47.569
54379 95.842 -79.460 47.534
54380 95.865 -79.473 47.499
54381 95.888 -79.485 47.464
54382 95.911 -79.498 47.429
54383 95.934 -79.503 47.393
54384 95.956 -79.507 47.357
54385 95.978 -79.512 47.324
54386 96.001 -79.516 47.290
54387 96.023 -79.521 47.257
54388 96.046 -79.526 47.225
54389 96.068 -79.530 47.192
54390 96.091 -79.535 47.160
54391 96.113 -79.540 47.128
54404 96.405 -79.600 46.729
54405 96.428 -79.604 46.699
54406 96.450 -79.610 46.668
54407 96.473 -79.614 46.640
54408 96.496 -79.619 46.613
54409 96.518 -79.623 46.586
54410 96.540 -79.628 46.559
54411 96.563 -79.632 46.532
54412 96.585 -79.637 46.505
54413 96.608 -79.642 46.479
54414 96.630 -79.647 46.454
54415 96.653 -79.651 46.429
54416 96.675 -79.656 46.405
54417 96.697 -79.661 46.380
54418 96.720 -79.665 46.355
54419 96.743 -79.670 46.330
54420 96.765 -79.674 46.304
54421 96.788 -79.679 46.279
54422 96.810 -79.684 46.256
54423 96.832 -79.689 46.232
54424 96.855 -79.693 46.209
54425 96.878 -79.698 46.187
54426 96.900 -79.702 46.167
54427 96.923 -79.707 46.148
54428 96.962 -79.706 46.128
54429 97.009 -79.704 46.105
54430 97.055 -79.701 46.082
54431 97.103 -79.698 46.057
54432 97.111 -79.707 46.030
54433 97.111 -79.718 46.003
54434 97.110 -79.730 45.976
54435 97.112 -79.752 45.951
54436 97.117 -79.772 45.927
54437 97.124 -79.780 45.905
54438 97.131 -79.787 45.881
54439 97.138 -79.795 45.857
54440 97.145 -79.803 45.834
54441 97.152 -79.811 45.812
54442 97.159 -79.818 45.791
54443 97.166 -79.826 45.770
54444 97.173 -79.833 45.748
54445 97.180 -79.841 45.728
54446 97.188 -79.849 45.707
54447 97.195 -79.856 45.686
54855 93.165 -79.404 53.312
54856 93.179 -79.392 53.299
54857 93.193 -79.380 53.285
54858 93.207 -79.368 53.271
54859 93.221 -79.356 53.258
54860 93.235 -79.344 53.245
54861 93.249 -79.332 53.232
54862 93.263 -79.319 53.219
54863 93.277 -79.307 53.206
54864 93.291 -79.295 53.192
54865 93.303 -79.290 53.179
54866 93.314 -79.283 53.165
54867 93.326 -79.274 53.152
54868 93.341 -79.269 53.141
54869 93.356 -79.265 53.129
54870 93.371 -79.260 53.116
54871 93.386 -79.256 53.103
54872 93.396 -79.250 53.090
54873 93.406 -79.245 53.079
54874 93.421 -79.240 53.068
54875 93.436 -79.234 53.057
54876 93.450 -79.229 53.045
54877 93.466 -79.224 53.034
54878 93.480 -79.219 53.022
54879 93.495 -79.213 53.013
54880 93.510 -79.208 53.003
54881 93.524 -79.203 52.993
54882 93.540 -79.197 52.986
54883 93.554 -79.191 52.978
54884 93.569 -79.187 52.970
54885 93.582 -79.184 52.961
54886 93.595 -79.180 52.950
54887 93.608 -79.177 52.940
54888 93.621 -79.174 52.931
54889 93.634 -79.170 52.924
54890 93.647 -79.167 52.916
54891 93.660 -79.164 52.910
54892 93.673 -79.161 52.904
54893 93.686 -79.157 52.897
54894 93.699 -79.154 52.889
54895 93.709 -79.163 52.882
54896 93.721 -79.165 52.873
54897 93.734 -79.162 52.864
54898 93.749 -79.160 52.855
54899 93.763 -79.160 52.846
54900 93.778 -79.159 52.836
54901 93.792 -79.159 52.826
54902 93.806 -79.159 52.816
54903 93.821 -79.159 52.803
54904 93.836 -79.159 52.791
54905 93.850 -79.159 52.778
54906 93.865 -79.159 52.764
54907 93.862 -79.156 52.750
54908 93.853 -79.152 52.737
54909 93.845 -79.148 52.724
54910 93.837 -79.143 52.706
54911 93.850 -79.143 52.689
54912 93.870 -79.144 52.671
54913 93.891 -79.146 52.651
54914 93.911 -79.147 52.632
54915 93.931 -79.148 52.613
54916 93.939 -79.150 52.595
54917 93.946 -79.153 52.576
54918 93.954 -79.156 52.558
54919 93.956 -79.160 52.542
54920 93.958 -79.164 52.529
54921 93.960 -79.169 52.515
54922 93.962 -79.173 52.502
54923 93.964 -79.177 52.489
54924 93.966 -79.181 52.476
54925 93.968 -79.186 52.462
54926 93.968 -79.180 52.450
54927 93.969 -79.174 52.435
54928 93.970 -79.167 52.421
54929 93.971 -79.161 52.408
54930 93.972 -79.155 52.398
54931 93.973 -79.149 52.387
54932 93.979 -79.145 52.378
54933 93.985 -79.142 52.368
54934 93.991 -79.141 52.357
54935 93.996 -79.151 52.345
54936 94.001 -79.161 52.331
54937 94.006 -79.171 52.317
54938 94.011 -79.176 52.299
54939 94.014 -79.173 52.277
54940 94.018 -79.170 52.255
54941 94.022 -79.168 52.232
54942 94.026 -79.165 52.209
54943 94.026 -79.166 52.186
54944 94.025 -79.169 52.163
54945 94.024 -79.173 52.140
54946 94.026 -79.178 52.116
54947 94.030 -79.186 52.093
54948 94.034 -79.194 52.070
54949 94.038 -79.202 52.047
54950 94.038 -79.206 52.025
54951 94.039 -79.212 52.006
54952 94.044 -79.213 51.986
54953 94.055 -79.212 51.963
54954 94.066 -79.212 51.939
54955 94.071 -79.209 51.913
54956 94.070 -79.206 51.885
54957 94.069 -79.202 51.856
54958 94.069 -79.198 51.826
54959 94.070 -79.194 51.796
54960 94.071 -79.191 51.766
54961 94.073 -79.185 51.743
54962 94.082 -79.169 51.722
54963 94.091 -79.153 51.703
54964 94.109 -79.131 51.687
54965 94.131 -79.106 51.671
54966 94.153 -79.081 51.656
54967 94.175 -79.055 51.638
54968 94.200 -79.049 51.618
54969 94.227 -79.060 51.597
54970 94.254 -79.068 51.572
54971 94.264 -79.048 51.541
54972 94.274 -79.027 51.507
54973 94.284 -79.006 51.469
54974 94.297 -78.995 51.427
54975 94.322 -78.998 51.384
54976 94.349 -79.006 51.343
54977 94.377 -79.015 51.302
54978 94.404 -79.024 51.263
54979 94.431 -79.032 51.224
54980 94.455 -79.042 51.186
54981 94.482 -79.025 51.148
54982 94.508 -79.010 51.110
54983 94.520 -79.006 51.072
54984 94.531 -79.002 51.034
54985 94.542 -78.999 50.997
54986 94.553 -78.995 50.959
54987 94.564 -78.991 50.922
54988 94.575 -78.988 50.884
54989 94.587 -78.984 50.847
54990 94.598 -78.981 50.809
54991 94.610 -78.977 50.772
54992 94.621 -78.973 50.734
54993 94.632 -78.969 50.697
54994 94.643 -78.966 50.660
54995 94.652 -78.971 50.622
54996 94.661 -78.979 50.584
54997 94.669 -78.987 50.547
54998 94.677 -78.994 50.508
55027 94.917 -79.211 49.421
55028 94.925 -79.219 49.384
55029 94.933 -79.225 49.346
55030 94.942 -79.233 49.310
55031 94.950 -79.241 49.274
55032 94.959 -79.248 49.238
55033 94.967 -79.256 49.202
55034 94.975 -79.263 49.166
55035 94.983 -79.271 49.129
55036 94.992 -79.278 49.093
55037 95.000 -79.285 49.057
55038 95.008 -79.293 49.022
55039 95.017 -79.300 48.986
55040 95.025 -79.308 48.950
55041 95.035 -79.321 48.914
55042 95.047 -79.339 48.878
55043 95.060 -79.357 48.843
55044 95.072 -79.375 48.808
55045 95.111 -79.400 48.772
55046 95.101 -79.382 48.734
55047 95.125 -79.385 48.697
55048 95.148 -79.389 48.659
55049 95.172 -79.393 48.622
55050 95.199 -79.397 48.584
55051 95.237 -79.403 48.547
55052 95.273 -79.408 48.509
55053 95.292 -79.404 48.470
55054 95.309 -79.400 48.432
55055 95.326 -79.395 48.394
55056 95.343 -79.391 48.356
55057 95.360 -79.386 48.318
55058 95.378 -79.382 48.281
55059 95.395 -79.377 48.245
55060 95.412 -79.372 48.208
55061 95.429 -79.368 48.173
55062 95.447 -79.363 48.138
55063 95.464 -79.359 48.102
55064 95.481 -79.354 48.066
55065 95.498 -79.350 48.031
55066 95.519 -79.348 47.998
55067 95.542 -79.350 47.966
55068 95.566 -79.351 47.934
55069 95.590 -79.353 47.903
55070 95.613 -79.354 47.872
55071 95.636 -79.356 47.841
55072 95.660 -79.357 47.812
55073 95.679 -79.353 47.781
55074 95.702 -79.366 47.752
55075 95.725 -79.379 47.722
55076 95.748 -79.392 47.689
55077 95.770 -79.405 47.654
55078 95.794 -79.418 47.618
55079 95.816 -79.432 47.583
55080 95.839 -79.444 47.547
55081 95.862 -79.457 47.510
55082 95.885 -79.470 47.472
55083 95.907 -79.480 47.434
55084 95.930 -79.485 47.398
55085 95.952 -79.489 47.362
55086 95.975 -79.494 47.327
55087 95.997 -79.498 47.293
55088 96.020 -79.503 47.260
55089 96.042 -79.508 47.228
55090 96.064 -79.513 47.194
55091 96.087 -79.517 47.162
55092 96.110 -79.522 47.129
55103 96.357 -79.573 46.788
55104 96.379 -79.577 46.758
55105 96.402 -79.582 46.728
55106 96.425 -79.587 46.698
55107 96.447 -79.592 46.668
55108 96.469 -79.596 46.640
55109 96.492 -79.601 46.612
55110 96.514 -79.605 46.585
55111 96.537 -79.610 46.559
55112 96.559 -79.614 46.532
55113 96.582 -79.620 46.505
55114 96.604 -79.624 46.479
55115 96.626 -79.629 46.454
55116 96.650 -79.632 46.430
55117 96.674 -79.636 46.407
55118 96.697 -79.640 46.382
55119 96.722 -79.644 46.357
55120 96.745 -79.648 46.332
55121 96.769 -79.651 46.306
55122 96.793 -79.655 46.281
55123 96.817 -79.660 46.258
55124 96.841 -79.664 46.235
55125 96.865 -79.667 46.213
55126 96.888 -79.671 46.192
55127 96.912 -79.675 46.173
55128 96.927 -79.685 46.156
55129 96.972 -79.683 46.136
55130 97.021 -79.679 46.114
55131 97.070 -79.676 46.091
55132 97.098 -79.672 46.067
55133 97.120 -79.668 46.042
55134 97.115 -79.690 46.014
55135 97.108 -79.711 45.986
55136 97.110 -79.730 45.958
55137 97.117 -79.738 45.932
55138 97.124 -79.745 45.908
55139 97.131 -79.753 45.883
55140 97.138 -79.761 45.859
55141 97.145 -79.768 45.835
55142 97.152 -79.776 45.813
55143 97.160 -79.783 45.791
55144 97.167 -79.791 45.769
55145 97.174 -79.798 45.748
55146 97.181 -79.806 45.727
55147 97.188 -79.814 45.707
55148 97.195 -79.822 45.685
55149 97.202 -79.830 45.664
55556 93.164 -79.394 53.339
55557 93.178 -79.382 53.325
55558 93.192 -79.370 53.312
55559 93.206 -79.358 53.298
55560 93.220 -79.345 53.284
55561 93.234 -79.333 53.271
55562 93.248 -79.321 53.257
55563 93.262 -79.309 53.244
55564 93.276 -79.297 53.231
55565 93.290 -79.285 53.217
55566 93.304 -79.279 53.204
55567 93.313 -79.277 53.189
55568 93.324 -79.269 53.176
55569 93.336 -79.259 53.164
55570 93.348 -79.250 53.151
55571 93.363 -79.246 53.138
55572 93.378 -79.241 53.125
55573 93.390 -79.236 53.113
55574 93.402 -79.231 53.103
55575 93.417 -79.225 53.092
55576 93.431 -79.220 53.081
55577 93.446 -79.215 53.070
55578 93.461 -79.209 53.059
55579 93.476 -79.204 53.049
55580 93.491 -79.199 53.040
55581 93.505 -79.194 53.030
55582 93.520 -79.188 53.021
55583 93.534 -79.184 53.013
55584 93.549 -79.179 53.004
55585 93.563 -79.174 52.994
55586 93.576 -79.172 52.984
55587 93.588 -79.170 52.973
55588 93.600 -79.169 52.963
55589 93.612 -79.167 52.955
55590 93.625 -79.165 52.947
55591 93.638 -79.162 52.940
55592 93.650 -79.159 52.934
55593 93.663 -79.156 52.929
55594 93.677 -79.153 52.922
55595 93.690 -79.150 52.915
55596 93.701 -79.154 52.908
55597 93.714 -79.152 52.900
55598 93.728 -79.151 52.890
55599 93.743 -79.151 52.881
55600 93.757 -79.151 52.871
55601 93.772 -79.150 52.861
55602 93.786 -79.150 52.851
55603 93.801 -79.150 52.840
55604 93.816 -79.150 52.827
55605 93.830 -79.150 52.815
55606 93.845 -79.150 52.802
55607 93.859 -79.150 52.789
55608 93.857 -79.147 52.776
55609 93.848 -79.143 52.763
55610 93.839 -79.139 52.748
55611 93.832 -79.135 52.732
55612 93.842 -79.134 52.714
55613 93.863 -79.136 52.695
55614 93.883 -79.137 52.676
55615 93.904 -79.138 52.656
55616 93.921 -79.140 52.638
55617 93.928 -79.142 52.621
55618 93.936 -79.145 52.604
55619 93.943 -79.147 52.588
55620 93.946 -79.151 52.575
55621 93.948 -79.156 52.562
55622 93.951 -79.160 52.550
55623 93.953 -79.164 52.537
55624 93.955 -79.169 52.525
55625 93.957 -79.173 52.512
55626 93.958 -79.177 52.498
55627 93.960 -79.181 52.485
55628 93.962 -79.178 52.470
55629 93.963 -79.172 52.455
55630 93.963 -79.166 52.440
55631 93.964 -79.160 52.428
55632 93.968 -79.156 52.417
55633 93.974 -79.152 52.407
55634 93.981 -79.148 52.396
55635 93.987 -79.145 52.385
55636 93.993 -79.143 52.373
55637 93.998 -79.154 52.360
55638 94.003 -79.164 52.345
55639 94.008 -79.172 52.327
55640 94.012 -79.169 52.306
55641 94.015 -79.166 52.283
55642 94.019 -79.163 52.260
55643 94.020 -79.163 52.236
55644 94.019 -79.166 52.213
55645 94.019 -79.169 52.188
55646 94.018 -79.173 52.163
55647 94.017 -79.176 52.139
55648 94.016 -79.180 52.116
55649 94.020 -79.188 52.092
55650 94.025 -79.196 52.069
55651 94.025 -79.200 52.047
55652 94.030 -79.203 52.026
55653 94.041 -79.203 52.004
55654 94.052 -79.201 51.978
55655 94.063 -79.200 51.952
55656 94.072 -79.199 51.926
55657 94.071 -79.195 51.898
55658 94.070 -79.192 51.868
55659 94.068 -79.188 51.839
55660 94.069 -79.184 51.810
55661 94.069 -79.181 51.784
55662 94.071 -79.176 51.763
55663 94.080 -79.160 51.743
55664 94.096 -79.140 51.725
55665 94.118 -79.114 51.709
55666 94.139 -79.089 51.692
55667 94.161 -79.063 51.675
55668 94.183 -79.038 51.657
55669 94.207 -79.025 51.637
55670 94.235 -79.036 51.615
55671 94.257 -79.037 51.589
55672 94.267 -79.015 51.556
55673 94.277 -78.995 51.521
55674 94.289 -78.981 51.482
55675 94.309 -78.991 51.440
55676 94.329 -79.000 51.398
55677 94.351 -78.997 51.356
55678 94.374 -78.990 51.315
55679 94.400 -78.994 51.276
55680 94.427 -79.002 51.237
55681 94.455 -79.011 51.200
55682 94.480 -79.022 51.162
55683 94.507 -79.006 51.124
55684 94.525 -78.997 51.086
55685 94.536 -78.993 51.049
55686 94.547 -78.990 51.011
55687 94.558 -78.986 50.974
55688 94.570 -78.982 50.936
55689 94.581 -78.979 50.898
55690 94.593 -78.975 50.860
55691 94.604 -78.971 50.822
55692 94.615 -78.968 50.785
55693 94.626 -78.964 50.747
55694 94.637 -78.960 50.710
55695 94.648 -78.957 50.672
55696 94.660 -78.953 50.634
55697 94.669 -78.959 50.597
55698 94.677 -78.966 50.559
55699 94.685 -78.974 50.521
55700 94.694 -78.981 50.483
55701 94.702 -78.989 50.445
55702 94.710 -78.996 50.407
55703 94.718 -79.003 50.369
55704 94.726 -79.011 50.331
55705 94.735 -79.018 50.294
55706 94.743 -79.026 50.257
55727 94.917 -79.183 49.467
55728 94.925 -79.190 49.430
55729 94.934 -79.198 49.392
55730 94.942 -79.206 49.356
55731 94.951 -79.213 49.319
55732 94.959 -79.220 49.283
55733 94.967 -79.228 49.247
55734 94.975 -79.235 49.211
55735 94.983 -79.243 49.175
55736 94.992 -79.250 49.139
55737 95.000 -79.258 49.103
55738 95.008 -79.266 49.067
55739 95.018 -79.276 49.032
55740 95.030 -79.294 48.996
55741 95.042 -79.313 48.961
55742 95.055 -79.331 48.925
55743 95.067 -79.349 48.890
55744 95.079 -79.367 48.855
55745 95.115 -79.391 48.820
55746 95.135 -79.400 48.783
55747 95.120 -79.378 48.746
55748 95.144 -79.382 48.708
55749 95.168 -79.385 48.670
55750 95.192 -79.389 48.633
55751 95.215 -79.393 48.595
55752 95.239 -79.397 48.557
55753 95.263 -79.401 48.519
55754 95.300 -79.407 48.481
55755 95.337 -79.412 48.442
55756 95.374 -79.417 48.403
55757 95.403 -79.419 48.365
55758 95.420 -79.414 48.328
55759 95.438 -79.410 48.291
55760 95.455 -79.405 48.253
55761 95.472 -79.400 48.217
55762 95.489 -79.396 48.182
55763 95.506 -79.391 48.146
55764 95.523 -79.387 48.110
55765 95.540 -79.382 48.074
55766 95.558 -79.378 48.039
55767 95.575 -79.373 48.005
55768 95.592 -79.369 47.974
55769 95.609 -79.364 47.942
55770 95.626 -79.360 47.911
55771 95.642 -79.354 47.881
55772 95.652 -79.346 47.852
55773 95.661 -79.338 47.825
55774 95.670 -79.340 47.794
55775 95.691 -79.352 47.763
55776 95.718 -79.364 47.732
55777 95.744 -79.377 47.699
55778 95.767 -79.390 47.664
55779 95.790 -79.404 47.628
55780 95.813 -79.416 47.592
55781 95.835 -79.429 47.555
55782 95.859 -79.442 47.517
55783 95.881 -79.456 47.477
55784 95.904 -79.462 47.438
55785 95.926 -79.466 47.401
55786 95.949 -79.471 47.365
55787 95.971 -79.476 47.329
55788 95.993 -79.481 47.295
55789 96.016 -79.485 47.262
55790 96.039 -79.490 47.229
55791 96.061 -79.494 47.196
55792 96.084 -79.499 47.163
55793 96.106 -79.504 47.130
55803 96.340 -79.545 46.818
55804 96.365 -79.548 46.788
55805 96.388 -79.552 46.757
55806 96.412 -79.556 46.727
55807 96.436 -79.560 46.697
55808 96.460 -79.563 46.667
55809 96.484 -79.567 46.639
55810 96.508 -79.571 46.612
55811 96.532 -79.575 46.584
55812 96.555 -79.579 46.558
55813 96.579 -79.582 46.532
55814 96.603 -79.586 46.505
55815 96.627 -79.590 46.479
55816 96.651 -79.594 46.455
55817 96.675 -79.598 46.431
55818 96.698 -79.601 46.408
55819 96.723 -79.605 46.384
55820 96.746 -79.609 46.359
55821 96.770 -79.613 46.333
55822 96.794 -79.617 46.308
55823 96.818 -79.620 46.283
55824 96.842 -79.624 46.260
55825 96.866 -79.628 46.238
55826 96.890 -79.632 46.217
55827 96.909 -79.639 46.197
55828 96.920 -79.651 46.180
55829 96.939 -79.661 46.161
55830 96.987 -79.657 46.141
55831 97.036 -79.653 46.119
55832 97.068 -79.649 46.096
55833 97.090 -79.645 46.075
55834 97.111 -79.642 46.054
55835 97.134 -79.638 46.028
55836 97.130 -79.659 45.999
55837 97.116 -79.690 45.970
55838 97.117 -79.703 45.940
55839 97.124 -79.711 45.912
55840 97.132 -79.718 45.886
55841 97.139 -79.726 45.861
55842 97.146 -79.733 45.836
55843 97.153 -79.742 45.813
55844 97.160 -79.749 45.790
55845 97.167 -79.757 45.769
55846 97.174 -79.764 45.747
55847 97.181 -79.772 45.726
55848 97.188 -79.780 45.706
55849 97.195 -79.787 45.684
55850 97.202 -79.795 45.663
55851 97.209 -79.802 45.642
56258 93.177 -79.372 53.352
56259 93.191 -79.360 53.338
56260 93.205 -79.347 53.324
56261 93.219 -79.335 53.311
56262 93.233 -79.323 53.297
56263 93.247 -79.311 53.283
56264 93.261 -79.299 53.269
56265 93.275 -79.286 53.256
56266 93.289 -79.274 53.243
56267 93.304 -79.269 53.229
56268 93.313 -79.266 53.214
56269 93.323 -79.263 53.200
56270 93.334 -79.255 53.187
56271 93.346 -79.244 53.174
56272 93.357 -79.234 53.161
56273 93.370 -79.226 53.148
56274 93.383 -79.222 53.137
56275 93.398 -79.216 53.126
56276 93.412 -79.211 53.115
56277 93.427 -79.206 53.103
56278 93.441 -79.200 53.093
56279 93.456 -79.196 53.083
56280 93.471 -79.190 53.073
56281 93.485 -79.186 53.064
56282 93.499 -79.181 53.055
56283 93.514 -79.176 53.046
56284 93.529 -79.172 53.038
56285 93.543 -79.166 53.029
56286 93.557 -79.162 53.019
56287 93.571 -79.158 53.008
56288 93.583 -79.156 52.997
56289 93.595 -79.155 52.988
56290 93.608 -79.153 52.979
56291 93.620 -79.152 52.972
56292 93.632 -79.150 52.966
56293 93.644 -79.149 52.960
56294 93.656 -79.147 52.954
56295 93.669 -79.146 52.948
56296 93.681 -79.144 52.941
56297 93.693 -79.143 52.935
56298 93.708 -79.142 52.927
56299 93.722 -79.142 52.918
56300 93.737 -79.142 52.908
56301 93.752 -79.142 52.898
56302 93.766 -79.141 52.887
56303 93.781 -79.141 52.876
56304 93.795 -79.141 52.865
56305 93.810 -79.141 52.853
56306 93.825 -79.141 52.840
56307 93.839 -79.140 52.827
56308 93.854 -79.140 52.815
56309 93.851 -79.137 52.802
56310 93.843 -79.134 52.788
56311 93.834 -79.130 52.773
56312 93.837 -79.128 52.757
56313 93.842 -79.127 52.740
56314 93.855 -79.128 52.720
56315 93.875 -79.128 52.701
56316 93.896 -79.130 52.682
56317 93.911 -79.131 52.664
56318 93.918 -79.134 52.648
56319 93.925 -79.137 52.633
56320 93.933 -79.139 52.619
56321 93.937 -79.143 52.608
56322 93.939 -79.147 52.595
56323 93.941 -79.151 52.583
56324 93.943 -79.156 52.571
56325 93.945 -79.160 52.559
56326 93.947 -79.164 52.546
56327 93.949 -79.169 52.532
56328 93.951 -79.173 52.518
56329 93.953 -79.177 52.503
56330 93.955 -79.178 52.487
56331 93.956 -79.171 52.472
56332 93.957 -79.165 52.459
56333 93.963 -79.162 52.446
56334 93.970 -79.158 52.434
56335 93.976 -79.155 52.423
56336 93.983 -79.151 52.412
56337 93.989 -79.147 52.400
56338 93.995 -79.147 52.387
56339 94.000 -79.156 52.372
56340 94.005 -79.166 52.355
56341 94.009 -79.164 52.335
56342 94.012 -79.162 52.311
56343 94.015 -79.159 52.287
56344 94.014 -79.163 52.262
56345 94.013 -79.166 52.237
56346 94.012 -79.170 52.213
56347 94.011 -79.173 52.188
56348 94.011 -79.176 52.162
56349 94.010 -79.180 52.138
56350 94.009 -79.183 52.115
56351 94.011 -79.190 52.092
56352 94.015 -79.194 52.068
56353 94.026 -79.193 52.045
56354 94.037 -79.191 52.021
56355 94.048 -79.190 51.992
56356 94.059 -79.190 51.965
56357 94.070 -79.188 51.936
56358 94.072 -79.185 51.908
56359 94.071 -79.182 51.879
56360 94.069 -79.178 51.852
56361 94.068 -79.175 51.826
56362 94.068 -79.171 51.803
56363 94.069 -79.166 51.786
56364 94.082 -79.147 51.768
56365 94.104 -79.122 51.749
56366 94.126 -79.096 51.730
56367 94.148 -79.071 51.713
56368 94.170 -79.046 51.695
56369 94.192 -79.021 51.676
56370 94.214 -79.000 51.656
56371 94.242 -79.011 51.633
56372 94.260 -79.005 51.605
56373 94.270 -78.984 51.571
56374 94.281 -78.967 51.535
56375 94.301 -78.977 51.495
56376 94.322 -78.987 51.452
56377 94.341 -78.996 51.410
56378 94.361 -79.006 51.369
56379 94.383 -79.006 51.328
56380 94.406 -78.998 51.289
56381 94.429 -78.990 51.251
56382 94.452 -78.983 51.215
56383 94.479 -78.990 51.178
56384 94.506 -78.999 51.140
56385 94.530 -78.988 51.102
56386 94.541 -78.984 51.064
56387 94.553 -78.981 51.026
56388 94.564 -78.977 50.988
56389 94.575 -78.973 50.950
56390 94.586 -78.970 50.912
56391 94.598 -78.966 50.873
56392 94.609 -78.962 50.835
56393 94.620 -78.959 50.798
56394 94.631 -78.955 50.760
56395 94.643 -78.951 50.723
56396 94.654 -78.948 50.685
56397 94.666 -78.944 50.647
56398 94.677 -78.940 50.609
56399 94.685 -78.946 50.571
56400 94.694 -78.953 50.533
56401 94.702 -78.961 50.495
56402 94.710 -78.968 50.457
56403 94.718 -78.976 50.419
56404 94.727 -78.984 50.381
56405 94.735 -78.991 50.343
56406 94.743 -78.998 50.305
56407 94.752 -79.006 50.268
56408 94.760 -79.013 50.230
56409 94.768 -79.021 50.192
56410 94.776 -79.028 50.154
56411 94.785 -79.036 50.117
56425 94.901 -79.140 49.590
56426 94.909 -79.148 49.553
56427 94.917 -79.156 49.514
56428 94.925 -79.162 49.477
56429 94.934 -79.170 49.440
56430 94.942 -79.178 49.403
56431 94.951 -79.185 49.366
56432 94.959 -79.193 49.329
56433 94.967 -79.200 49.293
56434 94.975 -79.208 49.256
56435 94.983 -79.215 49.221
56436 94.992 -79.222 49.185
56437 95.001 -79.231 49.149
56438 95.013 -79.250 49.114
56439 95.025 -79.268 49.078
56440 95.038 -79.286 49.043
56441 95.050 -79.304 49.008
56442 95.062 -79.322 48.973
56443 95.074 -79.341 48.938
56444 95.087 -79.359 48.903
56445 95.119 -79.382 48.868
56446 95.120 -79.374 48.832
56447 95.126 -79.370 48.795
56448 95.137 -79.372 48.758
56449 95.162 -79.376 48.720
56450 95.187 -79.382 48.682
56451 95.211 -79.385 48.644
56452 95.235 -79.389 48.605
56453 95.259 -79.393 48.567
56454 95.282 -79.397 48.530
56455 95.306 -79.401 48.492
56456 95.330 -79.405 48.453
56457 95.364 -79.410 48.415
56458 95.401 -79.415 48.376
56459 95.438 -79.421 48.339
56460 95.475 -79.426 48.302
56461 95.512 -79.432 48.264
56462 95.532 -79.428 48.228
56463 95.549 -79.423 48.192
56464 95.566 -79.419 48.155
56465 95.583 -79.414 48.119
56466 95.593 -79.407 48.083
56467 95.603 -79.398 48.047
56468 95.612 -79.390 48.015
56469 95.621 -79.382 47.983
56470 95.631 -79.373 47.951
56471 95.640 -79.365 47.921
56472 95.650 -79.357 47.891
56473 95.659 -79.349 47.863
56474 95.669 -79.341 47.835
56475 95.678 -79.336 47.804
56476 95.687 -79.346 47.772
56477 95.696 -79.357 47.739
56478 95.705 -79.367 47.704
56479 95.730 -79.379 47.668
56480 95.756 -79.391 47.631
56481 95.783 -79.404 47.592
56482 95.810 -79.417 47.554
56483 95.837 -79.429 47.516
56484 95.877 -79.434 47.478
56485 95.906 -79.441 47.440
56486 95.925 -79.452 47.403
56487 95.949 -79.457 47.367
56488 95.974 -79.460 47.331
56489 95.999 -79.462 47.296
56490 96.024 -79.464 47.262
56491 96.049 -79.466 47.230
56492 96.074 -79.469 47.197
56493 96.099 -79.472 47.164
56494 96.124 -79.474 47.131
56495 96.150 -79.476 47.099
56502 96.318 -79.501 46.877
56503 96.342 -79.505 46.848
56504 96.366 -79.509 46.818
56505 96.389 -79.513 46.788
56506 96.413 -79.516 46.758
56507 96.437 -79.520 46.727
56508 96.461 -79.524 46.696
56509 96.485 -79.528 46.665
56510 96.509 -79.532 46.637
56511 96.533 -79.535 46.610
56512 96.556 -79.539 46.583
56513 96.581 -79.543 46.557
56514 96.604 -79.547 46.531
56515 96.628 -79.551 46.506
56516 96.652 -79.554 46.480
56517 96.676 -79.558 46.456
56518 96.700 -79.562 46.432
56519 96.724 -79.567 46.408
56520 96.747 -79.570 46.384
56521 96.771 -79.574 46.360
56522 96.795 -79.578 46.335
56523 96.819 -79.582 46.310
56524 96.843 -79.585 46.286
56525 96.867 -79.589 46.263
56526 96.891 -79.593 46.242
56527 96.902 -79.605 46.221
56528 96.913 -79.617 46.201
56529 96.924 -79.630 46.182
56530 96.954 -79.635 46.161
56531 97.003 -79.631 46.140
56532 97.038 -79.627 46.117
56533 97.060 -79.623 46.097
56534 97.082 -79.620 46.077
56535 97.103 -79.616 46.056
56536 97.125 -79.612 46.029
56537 97.147 -79.608 46.004
56538 97.158 -79.616 45.977
56539 97.149 -79.639 45.947
56540 97.124 -79.676 45.916
56541 97.132 -79.684 45.888
56542 97.139 -79.692 45.862
56543 97.146 -79.699 45.837
56544 97.153 -79.707 45.813
56545 97.160 -79.714 45.790
56546 97.167 -79.722 45.769
56547 97.174 -79.730 45.747
56548 97.181 -79.737 45.726
56549 97.188 -79.745 45.705
56550 97.195 -79.752 45.684
56551 97.203 -79.760 45.662
56552 97.210 -79.768 45.642
56553 97.217 -79.776 45.623
56959 93.176 -79.362 53.379
56960 93.190 -79.350 53.365
56961 93.204 -79.337 53.351
56962 93.218 -79.325 53.337
56963 93.232 -79.313 53.323
56964 93.247 -79.300 53.310
56965 93.260 -79.288 53.296
56966 93.274 -79.276 53.282
56967 93.288 -79.264 53.268
56968 93.304 -79.257 53.253
56969 93.314 -79.255 53.239
56970 93.324 -79.253 53.224
56971 93.333 -79.250 53.212
56972 93.345 -79.241 53.198
56973 93.356 -79.230 53.185
56974 93.367 -79.220 53.173
56975 93.379 -79.209 53.160
56976 93.393 -79.203 53.148
56977 93.407 -79.198 53.136
56978 93.421 -79.193 53.125
56979 93.436 -79.188 53.114
56980 93.450 -79.183 53.104
56981 93.465 -79.178 53.094
56982 93.479 -79.174 53.086
56983 93.494 -79.169 53.077
56984 93.508 -79.164 53.069
56985 93.522 -79.159 53.062
56986 93.537 -79.154 53.053
56987 93.551 -79.149 53.044
56988 93.566 -79.144 53.033
56989 93.578 -79.142 53.022
56990 93.590 -79.140 53.013
56991 93.603 -79.139 53.004
56992 93.615 -79.137 52.998
56993 93.627 -79.136 52.991
56994 93.639 -79.134 52.985
56995 93.651 -79.133 52.980
56996 93.663 -79.131 52.974
56997 93.676 -79.131 52.968
56998 93.687 -79.134 52.962
56999 93.702 -79.134 52.954
57000 93.716 -79.133 52.945
57001 93.731 -79.133 52.934
57002 93.746 -79.133 52.924
57003 93.761 -79.132 52.913
57004 93.775 -79.132 52.902
57005 93.790 -79.132 52.890
57006 93.804 -79.132 52.879
57007 93.819 -79.132 52.866
57008 93.834 -79.131 52.854
57009 93.848 -79.131 52.842
57010 93.846 -79.128 52.830
57011 93.837 -79.125 52.816
57012 93.838 -79.123 52.800
57013 93.843 -79.122 52.783
57014 93.847 -79.121 52.766
57015 93.852 -79.119 52.747
57016 93.868 -79.120 52.728
57017 93.889 -79.121 52.709
57018 93.900 -79.123 52.692
57019 93.908 -79.126 52.676
57020 93.915 -79.128 52.662
57021 93.922 -79.131 52.650
57022 93.928 -79.134 52.639
57023 93.930 -79.138 52.627
57024 93.932 -79.143 52.615
57025 93.934 -79.147 52.603
57026 93.936 -79.151 52.591
57027 93.938 -79.156 52.578
57028 93.940 -79.159 52.564
57029 93.942 -79.164 52.549
57030 93.944 -79.169 52.534
57031 93.946 -79.172 52.519
57032 93.948 -79.176 52.504
57033 93.953 -79.172 52.489
57034 93.959 -79.168 52.475
57035 93.965 -79.165 52.460
57036 93.972 -79.161 52.449
57037 93.978 -79.157 52.438
57038 93.984 -79.154 52.426
57039 93.991 -79.150 52.414
57040 93.997 -79.149 52.400
57041 94.002 -79.159 52.382
57042 94.006 -79.159 52.362
57043 94.010 -79.157 52.338
57044 94.009 -79.160 52.314
57045 94.008 -79.163 52.288
57046 94.007 -79.166 52.262
57047 94.006 -79.170 52.238
57048 94.005 -79.173 52.213
57049 94.004 -79.177 52.186
57050 94.003 -79.180 52.161
57051 94.003 -79.182 52.136
57052 94.003 -79.184 52.112
57053 94.011 -79.183 52.086
57054 94.022 -79.181 52.061
57055 94.033 -79.181 52.034
57056 94.044 -79.180 52.005
57057 94.055 -79.178 51.976
57058 94.066 -79.178 51.947
57059 94.073 -79.175 51.918
57060 94.072 -79.172 51.892
57061 94.070 -79.169 51.868
57062 94.069 -79.165 51.847
57063 94.068 -79.161 51.827
57064 94.069 -79.155 51.810
57065 94.091 -79.131 51.792
57066 94.113 -79.105 51.770
57067 94.135 -79.080 51.752
57068 94.156 -79.054 51.733
57069 94.179 -79.029 51.714
57070 94.200 -79.003 51.694
57071 94.222 -78.978 51.673
57072 94.250 -78.987 51.649
57073 94.263 -78.972 51.621
57074 94.274 -78.953 51.586
57075 94.294 -78.963 51.548
57076 94.314 -78.973 51.507
57077 94.334 -78.982 51.465
57078 94.354 -78.992 51.423
57079 94.373 -79.002 51.381
57080 94.394 -79.012 51.341
57081 94.415 -79.015 51.303
57082 94.437 -79.006 51.266
57083 94.461 -78.999 51.231
57084 94.484 -78.991 51.195
57085 94.506 -78.984 51.158
57086 94.530 -78.977 51.119
57087 94.547 -78.975 51.081
57088 94.558 -78.971 51.042
57089 94.569 -78.968 51.003
57090 94.580 -78.964 50.965
57091 94.592 -78.961 50.926
57092 94.603 -78.957 50.887
57093 94.614 -78.953 50.849
57094 94.626 -78.949 50.811
57095 94.637 -78.946 50.773
57096 94.648 -78.942 50.736
57097 94.660 -78.939 50.697
57098 94.671 -78.935 50.659
57099 94.682 -78.931 50.621
57100 94.693 -78.927 50.583
57101 94.702 -78.933 50.545
57102 94.710 -78.940 50.507
57103 94.718 -78.948 50.469
57104 94.727 -78.955 50.431
57105 94.735 -78.963 50.393
57106 94.744 -78.971 50.355
57107 94.752 -78.978 50.317
57108 94.760 -78.985 50.280
57109 94.768 -78.993 50.242
57110 94.776 -79.000 50.204
57111 94.785 -79.008 50.166
57112 94.793 -79.015 50.128
57113 94.802 -79.023 50.090
57114 94.810 -79.031 50.053
57115 94.818 -79.038 50.015
57116 94.826 -79.045 49.978
57117 94.835 -79.053 49.939
57123 94.884 -79.097 49.714
57124 94.893 -79.105 49.676
57125 94.901 -79.112 49.638
57126 94.909 -79.120 49.601
57127 94.917 -79.128 49.564
57128 94.926 -79.135 49.526
57129 94.934 -79.143 49.489
57130 94.942 -79.150 49.451
57131 94.951 -79.157 49.413
57132 94.959 -79.165 49.376
57133 94.967 -79.172 49.340
57134 94.975 -79.180 49.304
57135 94.984 -79.187 49.267
57136 94.996 -79.205 49.232
57137 95.008 -79.223 49.196
57138 95.020 -79.241 49.160
57139 95.033 -79.259 49.125
57140 95.045 -79.278 49.090
57141 95.057 -79.296 49.056
57142 95.069 -79.315 49.021
57143 95.082 -79.333 48.986
57144 95.090 -79.347 48.952
57145 95.104 -79.357 48.917
57146 95.116 -79.358 48.882
57147 95.127 -79.360 48.846
57148 95.139 -79.361 48.809
57149 95.151 -79.363 48.772
57150 95.175 -79.366 48.734
57151 95.199 -79.372 48.695
57152 95.224 -79.376 48.657
57153 95.249 -79.381 48.619
57154 95.274 -79.386 48.581
57155 95.299 -79.391 48.543
57156 95.323 -79.395 48.505
57157 95.348 -79.400 48.467
57158 95.373 -79.405 48.429
57159 95.397 -79.409 48.392
57160 95.427 -79.413 48.356
57161 95.465 -79.419 48.320
57162 95.503 -79.423 48.281
57163 95.542 -79.428 48.244
57164 95.570 -79.425 48.207
57165 95.580 -79.419 48.169
57166 95.590 -79.413 48.131
57167 95.600 -79.408 48.094
57168 95.610 -79.401 48.058
57169 95.620 -79.393 48.027
57170 95.629 -79.385 47.994
57171 95.638 -79.376 47.961
57172 95.648 -79.368 47.930
57173 95.657 -79.360 47.898
57174 95.667 -79.352 47.867
57175 95.676 -79.344 47.837
57176 95.686 -79.335 47.804
57177 95.695 -79.342 47.770
57178 95.704 -79.353 47.735
57179 95.713 -79.363 47.701
57180 95.722 -79.373 47.666
57181 95.732 -79.383 47.630
57182 95.741 -79.394 47.594
57183 95.768 -79.407 47.557
57184 95.818 -79.404 47.519
57185 95.872 -79.401 47.481
57186 95.925 -79.397 47.442
57187 95.945 -79.408 47.403
57188 95.965 -79.419 47.366
57189 95.984 -79.430 47.330
57190 96.004 -79.441 47.296
57191 96.023 -79.452 47.262
57192 96.043 -79.463 47.229
57193 96.062 -79.474 47.196
57194 96.084 -79.482 47.163
57195 96.108 -79.485 47.130
57196 96.133 -79.488 47.098
57197 96.159 -79.490 47.066
57198 96.184 -79.492 47.034
57199 96.209 -79.494 47.002
57200 96.234 -79.497 46.970
57201 96.259 -79.499 46.939
57202 96.285 -79.501 46.908
57203 96.310 -79.504 46.875
57204 96.334 -79.507 46.845
57205 96.360 -79.509 46.816
57206 96.385 -79.511 46.785
57207 96.410 -79.513 46.755
57208 96.435 -79.516 46.725
57209 96.460 -79.518 46.694
57210 96.485 -79.521 46.663
57211 96.511 -79.523 46.635
57212 96.536 -79.526 46.608
57213 96.560 -79.528 46.581
57214 96.586 -79.530 46.556
57215 96.611 -79.532 46.530
57216 96.636 -79.535 46.504
57217 96.661 -79.538 46.479
57218 96.686 -79.540 46.455
57219 96.712 -79.542 46.432
57220 96.737 -79.545 46.409
57221 96.762 -79.547 46.385
57222 96.787 -79.549 46.361
57223 96.812 -79.551 46.337
57224 96.837 -79.554 46.311
57225 96.862 -79.557 46.288
57226 96.884 -79.559 46.267
57227 96.895 -79.571 46.246
57228 96.906 -79.584 46.225
57229 96.917 -79.596 46.204
57230 96.928 -79.609 46.182
57231 96.969 -79.608 46.160
57232 97.008 -79.604 46.138
57233 97.030 -79.601 46.115
57234 97.051 -79.598 46.095
57235 97.073 -79.594 46.075
57236 97.095 -79.590 46.053
57237 97.119 -79.585 46.027
57238 97.147 -79.579 46.003
57239 97.174 -79.568 45.977
57240 97.202 -79.557 45.948
57241 97.181 -79.589 45.917
57242 97.155 -79.628 45.889
57243 97.139 -79.657 45.863
57244 97.146 -79.664 45.838
57245 97.153 -79.672 45.813
57246 97.160 -79.679 45.790
57247 97.167 -79.688 45.769
57248 97.174 -79.695 45.747
57249 97.181 -79.703 45.725
57250 97.188 -79.711 45.704
57251 97.195 -79.718 45.682
57252 97.203 -79.726 45.661
57253 97.210 -79.733 45.641
57254 97.217 -79.741 45.623
57255 97.224 -79.748 45.605
57660 93.175 -79.351 53.405
57661 93.189 -79.339 53.392
57662 93.203 -79.327 53.377
57663 93.217 -79.315 53.363
57664 93.231 -79.303 53.349
57665 93.246 -79.291 53.336
57666 93.260 -79.278 53.322
57667 93.273 -79.266 53.307
57668 93.288 -79.253 53.293
57669 93.304 -79.247 53.278
57670 93.314 -79.244 53.263
57671 93.324 -79.241 53.249
57672 93.334 -79.239 53.236
57673 93.343 -79.237 53.223
57674 93.355 -79.226 53.209
57675 93.366 -79.216 53.196
57676 93.377 -79.206 53.182
57677 93.389 -79.196 53.169
57678 93.396 -79.194 53.156
57679 93.415 -79.181 53.144
57680 93.430 -79.175 53.134
57681 93.444 -79.171 53.123
57682 93.459 -79.166 53.111
57683 93.473 -79.161 53.102
57684 93.488 -79.156 53.093
57685 93.502 -79.151 53.086
57686 93.516 -79.147 53.081
57687 93.531 -79.141 53.075
57688 93.546 -79.137 53.068
57689 93.560 -79.131 53.058
57690 93.573 -79.128 53.047
57691 93.585 -79.127 53.038
57692 93.598 -79.125 53.030
57693 93.610 -79.124 53.023
57694 93.622 -79.122 53.016
57695 93.634 -79.121 53.010
57696 93.646 -79.119 53.007
57697 93.658 -79.118 53.001
57698 93.666 -79.123 52.995
57699 93.679 -79.125 52.988
57700 93.695 -79.125 52.980
57701 93.710 -79.125 52.971
57702 93.725 -79.125 52.960
57703 93.740 -79.124 52.950
57704 93.755 -79.124 52.939
57705 93.769 -79.123 52.928
57706 93.784 -79.123 52.916
57707 93.798 -79.123 52.905
57708 93.813 -79.123 52.892
57709 93.828 -79.122 52.880
57710 93.842 -79.122 52.869
57711 93.840 -79.119 52.856
57712 93.839 -79.117 52.843
57713 93.844 -79.116 52.827
57714 93.848 -79.115 52.810
57715 93.853 -79.114 52.792
57716 93.857 -79.112 52.774
57717 93.862 -79.112 52.755
57718 93.881 -79.112 52.736
57719 93.890 -79.115 52.718
57720 93.897 -79.118 52.701
57721 93.905 -79.120 52.688
57722 93.912 -79.122 52.675
57723 93.919 -79.125 52.665
57724 93.921 -79.130 52.652
57725 93.923 -79.134 52.639
57726 93.925 -79.138 52.626
57727 93.927 -79.143 52.614
57728 93.929 -79.147 52.601
57729 93.931 -79.151 52.587
57730 93.933 -79.156 52.574
57731 93.935 -79.158 52.561
57732 93.939 -79.160 52.547
57733 93.942 -79.163 52.532
57734 93.946 -79.169 52.518
57735 93.953 -79.168 52.503
57736 93.960 -79.167 52.488
57737 93.967 -79.165 52.476
57738 93.973 -79.164 52.463
57739 93.980 -79.160 52.451
57740 93.986 -79.157 52.441
57741 93.992 -79.153 52.427
57742 93.999 -79.151 52.409
57743 94.003 -79.156 52.388
57744 94.004 -79.156 52.365
57745 94.003 -79.160 52.341
57746 94.002 -79.163 52.314
57747 94.001 -79.166 52.288
57748 94.001 -79.168 52.263
57749 94.001 -79.169 52.237
57750 94.002 -79.170 52.209
57751 94.002 -79.172 52.183
57752 94.002 -79.173 52.156
57753 94.002 -79.175 52.129
57754 94.010 -79.173 52.103
57755 94.019 -79.171 52.076
57756 94.030 -79.169 52.048
57757 94.041 -79.169 52.017
57758 94.052 -79.168 51.989
57759 94.063 -79.166 51.960
57760 94.074 -79.165 51.932
57761 94.073 -79.162 51.908
57762 94.072 -79.158 51.888
57763 94.070 -79.155 51.869
57764 94.069 -79.151 51.851
57765 94.082 -79.131 51.834
57766 94.101 -79.106 51.814
57767 94.122 -79.084 51.792
57768 94.143 -79.062 51.772
57769 94.165 -79.037 51.753
57770 94.187 -79.012 51.734
57771 94.209 -78.987 51.713
57772 94.231 -78.961 51.691
57773 94.257 -78.961 51.665
57774 94.267 -78.940 51.636
57775 94.286 -78.949 51.599
57776 94.306 -78.959 51.560
57777 94.326 -78.968 51.519
57778 94.346 -78.978 51.476
57779 94.366 -78.988 51.435
57780 94.386 -78.998 51.394
57781 94.406 -79.007 51.354
57782 94.426 -79.017 51.317
57783 94.447 -79.023 51.281
57784 94.469 -79.015 51.247
57785 94.492 -79.007 51.212
57786 94.514 -78.996 51.176
57787 94.535 -78.970 51.139
57788 94.553 -78.971 51.100
57789 94.563 -78.964 51.060
57790 94.574 -78.959 51.020
57791 94.586 -78.955 50.980
57792 94.597 -78.952 50.941
57793 94.609 -78.948 50.902
57794 94.620 -78.944 50.864
57795 94.631 -78.940 50.825
57796 94.642 -78.937 50.787
57797 94.653 -78.933 50.749
57798 94.665 -78.930 50.710
57799 94.676 -78.926 50.672
57800 94.687 -78.922 50.634
57801 94.699 -78.918 50.596
57802 94.710 -78.915 50.558
57803 94.718 -78.921 50.521
57804 94.727 -78.927 50.482
57805 94.735 -78.935 50.443
57806 94.744 -78.943 50.405
57807 94.752 -78.950 50.367
57808 94.760 -78.958 50.329
57809 94.768 -78.965 50.292
57810 94.777 -78.973 50.253
57811 94.785 -78.980 50.215
57812 94.793 -78.987 50.178
57813 94.802 -78.995 50.139
57814 94.810 -79.002 50.101
57815 94.818 -79.010 50.065
57816 94.826 -79.018 50.027
57817 94.835 -79.025 49.990
57818 94.843 -79.032 49.953
57819 94.851 -79.040 49.915
57820 94.859 -79.047 49.877
57821 94.868 -79.055 49.840
57822 94.876 -79.062 49.802
57823 94.884 -79.070 49.765
57824 94.893 -79.078 49.727
57825 94.901 -79.085 49.689
57826 94.909 -79.092 49.650
57827 94.917 -79.100 49.612
57828 94.926 -79.107 49.575
57829 94.934 -79.115 49.538
57830 94.943 -79.122 49.501
57831 94.951 -79.130 49.463
57832 94.959 -79.137 49.424
57833 94.967 -79.144 49.387
57834 94.979 -79.160 49.351
57835 94.991 -79.178 49.315
57836 95.003 -79.197 49.279
57837 95.016 -79.215 49.243
57838 95.028 -79.233 49.208
57839 95.040 -79.251 49.173
57840 95.052 -79.270 49.138
57841 95.062 -79.286 49.104
57842 95.069 -79.299 49.069
57843 95.075 -79.311 49.035
57844 95.082 -79.324 49.001
57845 95.096 -79.337 48.967
57846 95.114 -79.345 48.933
57847 95.129 -79.348 48.897
57848 95.141 -79.350 48.861
57849 95.153 -79.351 48.823
57850 95.164 -79.353 48.785
57851 95.187 -79.357 48.748
57852 95.212 -79.362 48.710
57853 95.237 -79.366 48.672
57854 95.261 -79.371 48.634
57855 95.286 -79.376 48.597
57856 95.311 -79.381 48.558
57857 95.336 -79.385 48.520
57858 95.361 -79.390 48.483
57859 95.386 -79.395 48.445
57860 95.410 -79.400 48.409
57861 95.430 -79.400 48.373
57862 95.461 -79.408 48.337
57863 95.500 -79.413 48.298
57864 95.540 -79.416 48.259
57865 95.569 -79.412 48.221
57866 95.583 -79.404 48.182
57867 95.593 -79.399 48.143
57868 95.603 -79.394 48.105
57869 95.613 -79.388 48.068
57870 95.623 -79.382 48.035
57871 95.633 -79.377 48.002
57872 95.644 -79.372 47.968
57873 95.654 -79.366 47.935
57874 95.664 -79.360 47.902
57875 95.674 -79.355 47.869
57876 95.684 -79.347 47.836
57877 95.693 -79.338 47.802
57878 95.702 -79.338 47.767
57879 95.712 -79.349 47.731
57880 95.721 -79.359 47.697
57881 95.731 -79.369 47.663
57882 95.741 -79.379 47.628
57883 95.751 -79.389 47.593
57884 95.774 -79.395 47.557
57885 95.827 -79.391 47.520
57886 95.879 -79.386 47.482
57887 95.932 -79.382 47.442
57888 95.961 -79.378 47.403
57889 95.981 -79.386 47.365
57890 96.002 -79.394 47.329
57891 96.022 -79.402 47.295
57892 96.042 -79.410 47.261
57893 96.062 -79.419 47.228
57894 96.082 -79.431 47.194
57895 96.101 -79.441 47.161
57896 96.121 -79.453 47.128
57897 96.141 -79.463 47.096
57898 96.160 -79.475 47.063
57899 96.180 -79.485 47.031
57900 96.199 -79.497 47.000
57901 96.219 -79.507 46.968
57902 96.243 -79.510 46.937
57903 96.268 -79.513 46.905
57904 96.294 -79.515 46.873
57905 96.319 -79.517 46.843
57906 96.344 -79.520 46.813
57907 96.369 -79.523 46.782
57908 96.394 -79.525 46.752
57909 96.419 -79.527 46.722
57910 96.444 -79.529 46.691
57911 96.469 -79.532 46.660
57912 96.495 -79.534 46.632
57913 96.520 -79.537 46.605
57914 96.545 -79.539 46.578
57915 96.570 -79.541 46.553
57916 96.595 -79.544 46.528
57917 96.620 -79.546 46.502
57918 96.645 -79.548 46.477
57919 96.670 -79.551 46.454
57920 96.695 -79.554 46.432
57921 96.721 -79.556 46.409
57922 96.746 -79.558 46.386
57923 96.771 -79.560 46.362
57924 96.796 -79.563 46.338
57925 96.821 -79.565 46.313
57926 96.851 -79.563 46.291
57927 96.877 -79.558 46.270
57928 96.897 -79.554 46.249
57929 96.910 -79.563 46.228
57930 96.921 -79.576 46.206
57931 96.936 -79.586 46.182
57932 96.978 -79.582 46.159
57933 97.000 -79.579 46.135
57934 97.021 -79.575 46.113
57935 97.045 -79.570 46.091
57936 97.073 -79.564 46.069
57937 97.100 -79.557 46.047
57938 97.127 -79.551 46.023
57939 97.155 -79.544 45.998
57940 97.182 -79.532 45.972
57941 97.199 -79.530 45.944
57942 97.214 -79.536 45.915
57943 97.213 -79.541 45.888
57944 97.187 -79.578 45.862
57945 97.161 -79.616 45.838
57946 97.153 -79.638 45.813
57947 97.160 -79.645 45.789
57948 97.167 -79.653 45.766
57949 97.174 -79.661 45.745
57950 97.181 -79.668 45.724
57951 97.188 -79.676 45.702
57952 97.195 -79.683 45.681
57953 97.203 -79.691 45.660
57954 97.210 -79.698 45.640
57955 97.217 -79.707 45.622
57956 97.224 -79.714 45.604
57957 97.231 -79.722 45.585
58362 93.188 -79.328 53.417
58363 93.202 -79.316 53.404
58364 93.216 -79.304 53.389
58365 93.230 -79.292 53.375
58366 93.245 -79.280 53.361
58367 93.259 -79.268 53.347
58368 93.273 -79.256 53.332
58369 93.287 -79.244 53.317
58370 93.305 -79.235 53.302
58371 93.314 -79.233 53.288
58372 93.324 -79.231 53.273
58373 93.334 -79.228 53.260
58374 93.344 -79.225 53.247
58375 93.353 -79.222 53.233
58376 93.365 -79.212 53.219
58377 93.376 -79.202 53.205
58378 93.388 -79.191 53.191
58379 93.398 -79.182 53.178
58380 93.406 -79.178 53.165
58381 93.417 -79.175 53.153
58382 93.436 -79.162 53.142
58383 93.453 -79.153 53.130
58384 93.468 -79.149 53.120
58385 93.482 -79.143 53.111
58386 93.496 -79.139 53.104
58387 93.511 -79.134 53.099
58388 93.525 -79.129 53.095
58389 93.540 -79.125 53.091
58390 93.554 -79.119 53.083
58391 93.569 -79.115 53.074
58392 93.581 -79.113 53.065
58393 93.593 -79.112 53.058
58394 93.605 -79.110 53.051
58395 93.617 -79.109 53.045
58396 93.630 -79.107 53.039
58397 93.642 -79.106 53.034
58398 93.649 -79.110 53.029
58399 93.657 -79.115 53.023
58400 93.671 -79.117 53.016
58401 93.688 -79.116 53.007
58402 93.703 -79.116 52.997
58403 93.718 -79.116 52.987
58404 93.733 -79.115 52.976
58405 93.748 -79.115 52.966
58406 93.763 -79.115 52.955
58407 93.777 -79.115 52.943
58408 93.792 -79.114 52.931
58409 93.807 -79.114 52.919
58410 93.822 -79.113 52.906
58411 93.837 -79.113 52.894
58412 93.840 -79.112 52.882
58413 93.845 -79.110 52.869
58414 93.849 -79.109 52.855
58415 93.854 -79.108 52.838
58416 93.858 -79.107 52.819
58417 93.863 -79.106 52.800
58418 93.867 -79.105 52.781
58419 93.872 -79.104 52.762
58420 93.880 -79.106 52.743
58421 93.887 -79.109 52.727
58422 93.894 -79.112 52.713
58423 93.902 -79.114 52.701
58424 93.909 -79.117 52.689
58425 93.912 -79.121 52.677
58426 93.914 -79.125 52.663
58427 93.915 -79.130 52.649
58428 93.918 -79.133 52.637
58429 93.921 -79.135 52.624
58430 93.924 -79.137 52.611
58431 93.928 -79.140 52.599
58432 93.931 -79.142 52.586
58433 93.934 -79.144 52.573
58434 93.937 -79.147 52.559
58435 93.941 -79.155 52.546
58436 93.946 -79.160 52.531
58437 93.953 -79.159 52.516
58438 93.960 -79.158 52.504
58439 93.966 -79.157 52.492
58440 93.973 -79.156 52.479
58441 93.980 -79.154 52.468
58442 93.987 -79.153 52.453
58443 93.994 -79.152 52.436
58444 94.000 -79.151 52.414
58445 94.000 -79.153 52.391
58446 94.000 -79.154 52.366
58447 94.000 -79.156 52.339
58448 94.000 -79.157 52.312
58449 94.000 -79.159 52.285
58450 94.000 -79.159 52.258
58451 94.000 -79.161 52.228
58452 94.001 -79.162 52.201
58453 94.001 -79.164 52.172
58454 94.002 -79.165 52.143
58455 94.010 -79.163 52.116
58456 94.019 -79.161 52.089
58457 94.028 -79.159 52.060
58458 94.037 -79.158 52.030
58459 94.048 -79.156 52.003
58460 94.059 -79.156 51.976
58461 94.070 -79.155 51.949
58462 94.074 -79.152 51.928
58463 94.073 -79.148 51.910
58464 94.071 -79.144 51.893
58465 94.076 -79.134 51.875
58466 94.095 -79.108 51.856
58467 94.113 -79.081 51.836
58468 94.133 -79.057 51.813
58469 94.154 -79.035 51.792
58470 94.176 -79.013 51.773
58471 94.197 -78.991 51.754
58472 94.218 -78.969 51.732
58473 94.240 -78.944 51.708
58474 94.260 -78.929 51.681
58475 94.278 -78.935 51.649
58476 94.298 -78.945 51.611
58477 94.319 -78.954 51.571
58478 94.338 -78.964 51.531
58479 94.358 -78.974 51.488
58480 94.378 -78.984 51.447
58481 94.398 -78.993 51.406
58482 94.418 -79.003 51.367
58483 94.438 -79.013 51.331
58484 94.458 -79.023 51.297
58485 94.478 -79.030 51.264
58486 94.495 -78.984 51.229
58487 94.516 -78.968 51.194
58488 94.538 -78.959 51.157
58489 94.556 -78.961 51.118
58490 94.570 -78.962 51.078
58491 94.581 -78.955 51.037
58492 94.592 -78.949 50.998
58493 94.603 -78.943 50.958
58494 94.614 -78.939 50.918
58495 94.625 -78.935 50.879
58496 94.636 -78.931 50.840
58497 94.647 -78.927 50.801
58498 94.659 -78.924 50.763
58499 94.670 -78.920 50.725
58500 94.682 -78.917 50.687
58501 94.693 -78.913 50.648
58502 94.704 -78.909 50.610
58503 94.715 -78.905 50.572
58504 94.726 -78.902 50.535
58505 94.736 -78.908 50.496
58506 94.744 -78.915 50.456
58507 94.752 -78.922 50.418
58508 94.760 -78.930 50.380
58509 94.768 -78.937 50.342
58510 94.777 -78.945 50.305
58511 94.785 -78.952 50.267
58512 94.793 -78.960 50.229
58513 94.802 -78.968 50.191
58514 94.810 -78.974 50.153
58515 94.818 -78.982 50.115
58516 94.827 -78.990 50.078
58517 94.835 -78.997 50.041
58518 94.843 -79.005 50.003
58519 94.851 -79.012 49.966
58520 94.859 -79.020 49.928
58521 94.868 -79.027 49.890
58522 94.876 -79.034 49.852
58523 94.885 -79.042 49.816
58524 94.893 -79.049 49.778
58525 94.901 -79.057 49.740
58526 94.909 -79.065 49.702
58527 94.917 -79.072 49.663
58528 94.926 -79.079 49.625
58529 94.934 -79.087 49.588
58530 94.943 -79.094 49.551
58531 94.951 -79.102 49.512
58532 94.961 -79.115 49.474
58533 94.974 -79.134 49.437
58534 94.986 -79.152 49.400
58535 94.998 -79.170 49.364
58536 95.011 -79.188 49.328
58537 95.023 -79.207 49.292
58538 95.034 -79.225 49.256
58539 95.041 -79.237 49.221
58540 95.047 -79.250 49.187
58541 95.054 -79.263 49.153
58542 95.060 -79.275 49.119
58543 95.066 -79.288 49.085
58544 95.074 -79.300 49.051
58545 95.089 -79.313 49.017
58546 95.103 -79.325 48.983
58547 95.122 -79.333 48.948
58548 95.142 -79.338 48.912
58549 95.154 -79.340 48.875
58550 95.166 -79.341 48.837
58551 95.178 -79.343 48.801
58552 95.199 -79.347 48.764
58553 95.225 -79.352 48.726
58554 95.249 -79.357 48.689
58555 95.274 -79.361 48.652
58556 95.299 -79.366 48.614
58557 95.324 -79.371 48.576
58558 95.348 -79.375 48.539
58559 95.373 -79.381 48.502
58560 95.398 -79.385 48.465
58561 95.416 -79.380 48.427
58562 95.431 -79.388 48.389
58563 95.458 -79.397 48.350
58564 95.497 -79.401 48.311
58565 95.537 -79.406 48.271
58566 95.566 -79.400 48.232
58567 95.585 -79.390 48.193
58568 95.595 -79.384 48.153
58569 95.606 -79.379 48.113
58570 95.616 -79.373 48.074
58571 95.626 -79.368 48.039
58572 95.636 -79.362 48.004
58573 95.647 -79.357 47.969
58574 95.657 -79.351 47.935
58575 95.667 -79.346 47.903
58576 95.677 -79.340 47.870
58577 95.687 -79.335 47.836
58578 95.697 -79.329 47.800
58579 95.713 -79.332 47.764
58580 95.724 -79.344 47.728
58581 95.734 -79.354 47.695
58582 95.744 -79.365 47.660
58583 95.754 -79.375 47.625
58584 95.764 -79.385 47.589
58585 95.789 -79.390 47.553
58586 95.841 -79.385 47.516
58587 95.894 -79.381 47.478
58588 95.923 -79.360 47.439
58589 95.942 -79.344 47.401
58590 95.989 -79.366 47.364
58591 96.014 -79.375 47.328
58592 96.034 -79.384 47.293
58593 96.054 -79.392 47.259
58594 96.075 -79.400 47.226
58595 96.095 -79.408 47.192
58596 96.116 -79.416 47.159
58597 96.136 -79.425 47.126
58598 96.157 -79.432 47.093
58599 96.177 -79.441 47.061
58600 96.197 -79.449 47.029
58601 96.218 -79.457 46.997
58602 96.238 -79.465 46.966
58603 96.258 -79.475 46.934
58604 96.277 -79.485 46.902
58605 96.297 -79.497 46.871
58606 96.316 -79.507 46.840
58607 96.336 -79.519 46.809
58608 96.356 -79.529 46.779
58609 96.378 -79.535 46.748
58610 96.403 -79.538 46.717
58611 96.429 -79.541 46.687
58612 96.454 -79.543 46.656
58613 96.478 -79.545 46.629
58614 96.504 -79.548 46.602
58615 96.529 -79.550 46.575
58616 96.554 -79.552 46.550
58617 96.579 -79.555 46.525
58618 96.604 -79.557 46.500
58619 96.629 -79.560 46.476
58620 96.655 -79.562 46.453
58621 96.680 -79.564 46.431
58622 96.704 -79.567 46.408
58623 96.730 -79.569 46.386
58624 96.755 -79.572 46.363
58625 96.780 -79.574 46.341
58626 96.811 -79.571 46.316
58627 96.844 -79.566 46.294
58628 96.869 -79.561 46.274
58629 96.888 -79.557 46.252
58630 96.908 -79.553 46.230
58631 96.925 -79.554 46.208
58632 96.948 -79.560 46.184
58633 96.970 -79.556 46.158
58634 96.998 -79.550 46.133
58635 97.025 -79.544 46.110
58636 97.053 -79.537 46.086
58637 97.081 -79.531 46.063
58638 97.108 -79.525 46.040
58639 97.136 -79.518 46.016
58640 97.163 -79.507 45.991
58641 97.183 -79.502 45.966
58642 97.196 -79.505 45.940
58643 97.208 -79.508 45.913
58644 97.223 -79.517 45.887
58645 97.222 -79.522 45.862
58646 97.220 -79.527 45.837
58647 97.193 -79.566 45.812
58648 97.167 -79.604 45.788
58649 97.167 -79.618 45.765
58650 97.174 -79.626 45.743
58651 97.181 -79.634 45.721
58652 97.188 -79.642 45.700
58653 97.195 -79.649 45.679
58654 97.203 -79.657 45.658
58655 97.210 -79.664 45.638
58656 97.217 -79.672 45.620
58657 97.224 -79.679 45.602
58658 97.231 -79.687 45.584
58659 97.238 -79.695 45.566
58673 97.337 -79.802 45.313
59064 93.201 -79.306 53.429
59065 93.215 -79.294 53.415
59066 93.229 -79.281 53.401
59067 93.243 -79.269 53.386
59068 93.258 -79.257 53.372
59069 93.272 -79.245 53.357
59070 93.286 -79.233 53.342
59071 93.305 -79.225 53.327
59072 93.314 -79.222 53.312
59073 93.325 -79.219 53.297
59074 93.334 -79.217 53.284
59075 93.344 -79.215 53.270
59076 93.353 -79.212 53.257
59077 93.363 -79.208 53.243
59078 93.375 -79.198 53.228
59079 93.387 -79.187 53.214
59080 93.398 -79.177 53.200
59081 93.407 -79.169 53.187
59082 93.416 -79.162 53.174
59083 93.422 -79.165 53.161
59084 93.438 -79.156 53.150
59085 93.457 -79.143 53.140
59086 93.476 -79.131 53.130
59087 93.490 -79.126 53.122
59088 93.505 -79.122 53.117
59089 93.519 -79.117 53.115
59090 93.534 -79.112 53.113
59091 93.548 -79.107 53.109
59092 93.563 -79.102 53.101
59093 93.576 -79.099 53.094
59094 93.588 -79.097 53.086
59095 93.601 -79.096 53.080
59096 93.613 -79.094 53.074
59097 93.625 -79.093 53.068
59098 93.633 -79.097 53.063
59099 93.640 -79.102 53.057
59100 93.648 -79.107 53.051
59101 93.663 -79.108 53.043
59102 93.681 -79.108 53.034
59103 93.697 -79.108 53.025
59104 93.711 -79.107 53.015
59105 93.726 -79.107 53.005
59106 93.741 -79.106 52.994
59107 93.756 -79.106 52.984
59108 93.771 -79.106 52.972
59109 93.786 -79.106 52.959
59110 93.800 -79.106 52.947
59111 93.816 -79.105 52.934
59112 93.830 -79.103 52.921
59113 93.837 -79.101 52.908
59114 93.841 -79.100 52.894
59115 93.845 -79.099 52.880
59116 93.849 -79.097 52.864
59117 93.853 -79.096 52.845
59118 93.856 -79.094 52.827
59119 93.860 -79.093 52.808
59120 93.868 -79.090 52.788
59121 93.875 -79.092 52.769
59122 93.882 -79.096 52.753
59123 93.888 -79.100 52.739
59124 93.894 -79.103 52.726
59125 93.900 -79.106 52.714
59126 93.904 -79.109 52.700
59127 93.907 -79.112 52.686
59128 93.910 -79.114 52.672
59129 93.914 -79.116 52.661
59130 93.917 -79.118 52.648
59131 93.920 -79.122 52.636
59132 93.923 -79.124 52.623
59133 93.926 -79.126 52.611
59134 93.929 -79.128 52.599
59135 93.933 -79.131 52.586
59136 93.936 -79.140 52.573
59137 93.940 -79.148 52.560
59138 93.946 -79.152 52.545
59139 93.953 -79.150 52.533
59140 93.960 -79.150 52.521
59141 93.966 -79.148 52.508
59142 93.973 -79.147 52.495
59143 93.980 -79.146 52.480
59144 93.993 -79.140 52.461
59145 94.000 -79.139 52.440
59146 94.000 -79.141 52.416
59147 93.999 -79.143 52.390
59148 93.999 -79.146 52.362
59149 93.999 -79.148 52.334
59150 93.999 -79.149 52.306
59151 93.999 -79.150 52.276
59152 93.999 -79.152 52.244
59153 93.999 -79.153 52.215
59154 94.000 -79.155 52.185
59155 94.002 -79.156 52.155
59156 94.010 -79.153 52.127
59157 94.019 -79.152 52.100
59158 94.028 -79.150 52.072
59159 94.036 -79.148 52.043
59160 94.045 -79.146 52.017
59161 94.056 -79.144 51.993
59162 94.067 -79.143 51.970
59163 94.075 -79.142 51.951
59164 94.074 -79.138 51.934
59165 94.072 -79.134 51.917
59166 94.088 -79.111 51.899
59167 94.107 -79.084 51.879
59168 94.126 -79.057 51.856
59169 94.144 -79.031 51.832
59170 94.165 -79.009 51.811
59171 94.186 -78.987 51.791
59172 94.207 -78.965 51.772
59173 94.228 -78.943 51.750
59174 94.250 -78.921 51.724
59175 94.266 -78.911 51.694
59176 94.285 -78.919 51.660
59177 94.304 -78.927 51.620
59178 94.324 -78.937 51.580
59179 94.343 -78.945 51.541
59180 94.362 -78.953 51.500
59181 94.382 -78.962 51.460
59182 94.401 -78.971 51.419
59183 94.420 -78.979 51.381
59184 94.439 -78.987 51.346
59185 94.459 -78.996 51.313
59186 94.476 -78.971 51.281
59187 94.498 -78.965 51.247
59188 94.520 -78.956 51.212
59189 94.542 -78.949 51.175
59190 94.559 -78.951 51.136
59191 94.575 -78.956 51.095
59192 94.588 -78.953 51.055
59193 94.599 -78.947 51.016
59194 94.609 -78.940 50.976
59195 94.620 -78.934 50.936
59196 94.631 -78.927 50.895
59197 94.642 -78.922 50.856
59198 94.653 -78.918 50.817
59199 94.664 -78.915 50.778
59200 94.676 -78.911 50.740
59201 94.687 -78.908 50.702
59202 94.698 -78.904 50.663
59203 94.709 -78.900 50.625
59204 94.720 -78.896 50.587
59205 94.732 -78.893 50.549
59206 94.743 -78.889 50.511
59207 94.752 -78.895 50.471
59208 94.760 -78.902 50.433
59209 94.769 -78.909 50.395
59210 94.777 -78.917 50.356
59211 94.785 -78.924 50.318
59212 94.793 -78.932 50.281
59213 94.802 -78.940 50.243
59214 94.810 -78.947 50.204
59215 94.818 -78.955 50.167
59216 94.827 -78.962 50.129
59217 94.835 -78.969 50.092
59218 94.843 -78.977 50.054
59219 94.851 -78.984 50.017
59220 94.859 -78.992 49.979
59221 94.868 -78.999 49.942
59222 94.876 -79.007 49.904
59223 94.885 -79.015 49.866
59224 94.893 -79.021 49.830
59225 94.901 -79.029 49.792
59226 94.909 -79.037 49.754
59227 94.918 -79.044 49.715
59228 94.926 -79.052 49.677
59229 94.934 -79.059 49.639
59230 94.944 -79.071 49.601
59231 94.957 -79.089 49.564
59232 94.969 -79.107 49.525
59233 94.981 -79.125 49.487
59234 94.993 -79.143 49.450
59235 95.005 -79.162 49.413
59236 95.013 -79.176 49.377
59237 95.020 -79.188 49.341
59238 95.026 -79.201 49.305
59239 95.032 -79.214 49.271
59240 95.039 -79.227 49.237
59241 95.045 -79.239 49.202
59242 95.051 -79.252 49.169
59243 95.058 -79.265 49.136
59244 95.067 -79.277 49.102
59245 95.082 -79.290 49.068
59246 95.096 -79.303 49.033
59247 95.111 -79.315 48.999
59248 95.131 -79.321 48.963
59249 95.151 -79.326 48.926
59250 95.168 -79.331 48.889
59251 95.179 -79.332 48.852
59252 95.191 -79.333 48.816
59253 95.212 -79.338 48.780
59254 95.237 -79.342 48.743
59255 95.262 -79.347 48.705
59256 95.286 -79.351 48.667
59257 95.311 -79.357 48.630
59258 95.336 -79.361 48.592
59259 95.361 -79.366 48.553
59260 95.384 -79.368 48.515
59261 95.402 -79.362 48.476
59262 95.419 -79.360 48.436
59263 95.432 -79.375 48.396
59264 95.456 -79.386 48.356
59265 95.495 -79.391 48.316
59266 95.534 -79.394 48.276
59267 95.563 -79.389 48.237
59268 95.586 -79.375 48.197
59269 95.598 -79.369 48.156
59270 95.608 -79.364 48.116
59271 95.618 -79.358 48.075
59272 95.628 -79.353 48.038
59273 95.639 -79.347 48.002
59274 95.649 -79.342 47.966
59275 95.659 -79.336 47.932
59276 95.670 -79.331 47.899
59277 95.680 -79.325 47.865
59278 95.690 -79.320 47.831
59279 95.700 -79.314 47.794
59280 95.723 -79.317 47.757
59281 95.736 -79.340 47.721
59282 95.746 -79.350 47.687
59283 95.757 -79.360 47.652
59284 95.767 -79.370 47.616
59285 95.777 -79.381 47.581
59286 95.804 -79.385 47.544
59287 95.856 -79.380 47.508
59288 95.909 -79.375 47.470
59289 95.912 -79.337 47.433
59290 95.922 -79.310 47.396
59291 95.970 -79.332 47.359
59292 96.018 -79.353 47.324
59293 96.046 -79.366 47.289
59294 96.066 -79.373 47.255
59295 96.087 -79.382 47.222
59296 96.107 -79.390 47.188
59297 96.128 -79.398 47.155
59298 96.148 -79.407 47.122
59299 96.169 -79.414 47.088
59300 96.189 -79.422 47.056
59301 96.209 -79.431 47.024
59302 96.230 -79.439 46.993
59303 96.250 -79.447 46.962
59304 96.270 -79.455 46.929
59305 96.291 -79.463 46.897
59306 96.311 -79.472 46.865
59307 96.331 -79.479 46.835
59308 96.352 -79.488 46.804
59309 96.372 -79.496 46.774
59310 96.393 -79.504 46.743
59311 96.413 -79.512 46.713
59312 96.434 -79.520 46.682
59313 96.453 -79.530 46.652
59314 96.473 -79.541 46.625
59315 96.493 -79.552 46.598
59316 96.513 -79.561 46.571
59317 96.538 -79.563 46.546
59318 96.563 -79.566 46.521
59319 96.588 -79.568 46.497
59320 96.613 -79.571 46.473
59321 96.639 -79.573 46.450
59322 96.664 -79.576 46.428
59323 96.689 -79.578 46.407
59324 96.714 -79.580 46.386
59325 96.739 -79.582 46.365
59326 96.770 -79.579 46.344
59327 96.804 -79.574 46.321
59328 96.837 -79.568 46.300
59329 96.860 -79.564 46.279
59330 96.880 -79.560 46.256
59331 96.900 -79.554 46.233
59332 96.925 -79.543 46.209
59333 96.951 -79.529 46.185
59334 96.978 -79.523 46.156
59335 97.006 -79.516 46.130
59336 97.033 -79.510 46.107
59337 97.061 -79.504 46.081
59338 97.089 -79.498 46.057
59339 97.116 -79.491 46.034
59340 97.144 -79.483 46.010
59341 97.168 -79.475 45.986
59342 97.180 -79.477 45.962
59343 97.192 -79.480 45.937
59344 97.205 -79.483 45.911
59345 97.217 -79.486 45.886
59346 97.232 -79.498 45.860
59347 97.231 -79.503 45.835
59348 97.230 -79.508 45.810
59349 97.226 -79.516 45.786
59350 97.200 -79.554 45.764
59351 97.174 -79.592 45.741
59352 97.182 -79.599 45.719
59353 97.189 -79.607 45.697
59354 97.196 -79.614 45.677
59355 97.203 -79.622 45.656
59356 97.210 -79.629 45.637
59357 97.217 -79.637 45.618
59358 97.224 -79.645 45.601
59359 97.231 -79.653 45.583
59360 97.238 -79.661 45.564
59361 97.245 -79.668 45.547
59362 97.252 -79.676 45.528
59363 97.259 -79.683 45.509
59364 97.266 -79.691 45.492
59365 97.274 -79.698 45.474
59366 97.281 -79.706 45.456
59367 97.288 -79.714 45.438
59368 97.295 -79.721 45.419
59369 97.302 -79.729 45.400
59370 97.309 -79.736 45.382
59371 97.316 -79.744 45.365
59372 97.323 -79.751 45.348
59373 97.330 -79.760 45.331
59374 97.337 -79.767 45.314
59375 97.345 -79.775 45.296
59766 93.214 -79.284 53.440
59767 93.228 -79.272 53.426
59768 93.242 -79.259 53.411
59769 93.257 -79.247 53.397
59770 93.271 -79.235 53.382
59771 93.286 -79.223 53.368
59772 93.305 -79.214 53.353
59773 93.315 -79.211 53.337
59774 93.325 -79.209 53.322
59775 93.334 -79.206 53.308
59776 93.344 -79.204 53.295
59777 93.354 -79.201 53.281
59778 93.363 -79.199 53.267
59779 93.373 -79.194 53.252
59780 93.385 -79.184 53.238
59781 93.397 -79.173 53.224
59782 93.408 -79.163 53.210
59783 93.417 -79.156 53.197
59784 93.425 -79.148 53.184
59785 93.432 -79.148 53.172
59786 93.439 -79.150 53.161
59787 93.459 -79.138 53.150
59788 93.478 -79.125 53.141
59789 93.497 -79.112 53.137
59790 93.513 -79.104 53.135
59791 93.528 -79.100 53.134
59792 93.543 -79.094 53.133
59793 93.557 -79.090 53.129
59794 93.571 -79.085 53.123
59795 93.583 -79.084 53.116
59796 93.595 -79.082 53.110
59797 93.608 -79.081 53.104
59798 93.616 -79.084 53.097
59799 93.624 -79.090 53.092
59800 93.631 -79.094 53.086
59801 93.639 -79.099 53.079
59802 93.656 -79.100 53.071
59803 93.674 -79.100 53.062
59804 93.690 -79.100 53.053
59805 93.705 -79.099 53.043
59806 93.719 -79.099 53.034
59807 93.734 -79.098 53.023
59808 93.749 -79.098 53.012
59809 93.764 -79.097 53.000
59810 93.779 -79.097 52.987
59811 93.794 -79.096 52.974
59812 93.808 -79.095 52.960
59813 93.823 -79.091 52.946
59814 93.830 -79.090 52.932
59815 93.834 -79.088 52.918
59816 93.838 -79.087 52.904
59817 93.841 -79.085 52.888
59818 93.845 -79.084 52.870
59819 93.849 -79.083 52.850
59820 93.856 -79.081 52.831
59821 93.864 -79.078 52.813
59822 93.872 -79.075 52.794
59823 93.878 -79.079 52.778
59824 93.885 -79.083 52.765
59825 93.891 -79.087 52.752
59826 93.897 -79.090 52.739
59827 93.900 -79.093 52.726
59828 93.903 -79.096 52.713
59829 93.906 -79.098 52.701
59830 93.909 -79.100 52.689
59831 93.912 -79.103 52.676
59832 93.915 -79.105 52.662
59833 93.918 -79.107 52.650
59834 93.921 -79.110 52.638
59835 93.925 -79.112 52.625
59836 93.928 -79.116 52.612
59837 93.932 -79.125 52.600
59838 93.936 -79.132 52.588
59839 93.940 -79.140 52.574
59840 93.945 -79.143 52.562
59841 93.953 -79.142 52.550
59842 93.959 -79.140 52.536
59843 93.966 -79.140 52.522
59844 93.981 -79.134 52.506
59845 93.998 -79.126 52.486
59846 94.001 -79.127 52.464
59847 94.001 -79.129 52.440
59848 94.000 -79.131 52.413
59849 94.000 -79.134 52.385
59850 93.999 -79.137 52.354
59851 93.999 -79.139 52.324
59852 93.998 -79.141 52.293
59853 93.998 -79.143 52.260
59854 93.998 -79.144 52.228
59855 93.998 -79.146 52.196
59856 94.002 -79.146 52.166
59857 94.010 -79.144 52.138
59858 94.019 -79.142 52.111
59859 94.028 -79.140 52.084
59860 94.036 -79.138 52.057
59861 94.045 -79.136 52.033
59862 94.053 -79.134 52.011
59863 94.063 -79.133 51.992
59864 94.074 -79.131 51.975
59865 94.075 -79.128 51.958
59866 94.082 -79.114 51.941
59867 94.101 -79.087 51.921
59868 94.120 -79.060 51.899
59869 94.138 -79.034 51.875
59870 94.156 -79.007 51.850
59871 94.176 -78.982 51.829
59872 94.197 -78.960 51.809
59873 94.218 -78.938 51.789
59874 94.240 -78.916 51.765
59875 94.261 -78.894 51.736
59876 94.274 -78.895 51.704
59877 94.285 -78.891 51.668
59878 94.304 -78.900 51.630
59879 94.324 -78.908 51.591
59880 94.343 -78.917 51.552
59881 94.362 -78.926 51.512
59882 94.382 -78.934 51.472
59883 94.401 -78.943 51.433
59884 94.420 -78.951 51.395
59885 94.439 -78.960 51.361
59886 94.458 -78.954 51.329
59887 94.478 -78.942 51.298
59888 94.501 -78.952 51.265
59889 94.524 -78.946 51.230
59890 94.546 -78.937 51.193
59891 94.562 -78.942 51.154
59892 94.578 -78.946 51.114
59893 94.595 -78.952 51.074
59894 94.605 -78.945 51.033
59895 94.616 -78.939 50.995
59896 94.627 -78.932 50.955
59897 94.637 -78.926 50.914
59898 94.648 -78.919 50.874
59899 94.659 -78.913 50.834
59900 94.670 -78.906 50.795
59901 94.681 -78.902 50.757
59902 94.692 -78.899 50.718
59903 94.703 -78.895 50.680
59904 94.715 -78.891 50.641
59905 94.726 -78.887 50.603
59906 94.737 -78.883 50.565
59907 94.749 -78.880 50.526
59908 94.760 -78.877 50.486
59909 94.769 -78.882 50.448
59910 94.777 -78.890 50.409
59911 94.785 -78.897 50.371
59912 94.793 -78.904 50.332
59913 94.802 -78.912 50.294
59914 94.810 -78.919 50.257
59915 94.818 -78.927 50.218
59916 94.827 -78.934 50.180
59917 94.835 -78.942 50.143
59918 94.843 -78.949 50.106
59919 94.851 -78.956 50.068
59920 94.860 -78.964 50.031
59921 94.868 -78.971 49.993
59922 94.877 -78.979 49.956
59923 94.885 -78.987 49.919
59924 94.893 -78.994 49.881
59925 94.901 -79.002 49.844
59926 94.909 -79.009 49.806
59927 94.918 -79.016 49.768
59928 94.927 -79.026 49.729
59929 94.939 -79.044 49.691
59930 94.952 -79.062 49.653
59931 94.964 -79.081 49.615
59932 94.976 -79.100 49.577
59933 94.985 -79.115 49.539
59934 94.991 -79.127 49.501
59935 94.998 -79.140 49.463
59936 95.004 -79.153 49.427
59937 95.011 -79.165 49.391
59938 95.017 -79.178 49.355
59939 95.024 -79.190 49.320
59940 95.030 -79.203 49.287
59941 95.037 -79.216 49.254
59942 95.043 -79.228 49.220
59943 95.049 -79.241 49.188
59944 95.060 -79.254 49.155
59945 95.074 -79.266 49.121
59946 95.089 -79.279 49.086
59947 95.103 -79.291 49.050
59948 95.119 -79.303 49.015
59949 95.139 -79.308 48.979
59950 95.160 -79.314 48.942
59951 95.180 -79.320 48.905
59952 95.192 -79.322 48.868
59953 95.204 -79.324 48.832
59954 95.225 -79.328 48.796
59955 95.249 -79.332 48.760
59956 95.274 -79.337 48.721
59957 95.299 -79.341 48.682
59958 95.324 -79.347 48.644
59959 95.348 -79.351 48.605
59960 95.370 -79.350 48.565
59961 95.388 -79.344 48.524
59962 95.405 -79.338 48.484
59963 95.420 -79.347 48.442
59964 95.433 -79.362 48.401
59965 95.453 -79.375 48.360
59966 95.492 -79.379 48.320
59967 95.531 -79.384 48.279
59968 95.560 -79.378 48.239
59969 95.583 -79.365 48.199
59970 95.601 -79.354 48.158
59971 95.611 -79.349 48.117
59972 95.621 -79.344 48.077
59973 95.631 -79.338 48.038
59974 95.641 -79.332 48.000
59975 95.652 -79.327 47.964
59976 95.662 -79.322 47.930
59977 95.672 -79.316 47.895
59978 95.682 -79.310 47.860
59979 95.692 -79.305 47.825
59980 95.703 -79.300 47.788
59981 95.730 -79.305 47.752
59982 95.750 -79.325 47.715
59983 95.759 -79.346 47.679
59984 95.769 -79.356 47.644
59985 95.779 -79.366 47.609
59986 95.790 -79.376 47.573
59987 95.819 -79.380 47.536
59988 95.872 -79.375 47.500
59989 95.905 -79.357 47.463
59990 95.901 -79.313 47.427
59991 95.903 -79.277 47.391
59992 95.951 -79.298 47.355
59993 95.999 -79.319 47.320
59994 96.046 -79.341 47.285
59995 96.079 -79.355 47.251
59996 96.099 -79.363 47.218
59997 96.120 -79.372 47.185
59998 96.140 -79.380 47.151
59999 96.160 -79.388 47.117
60000 96.181 -79.396 47.084
60001 96.201 -79.404 47.051
60002 96.222 -79.413 47.020
60003 96.242 -79.420 46.988
60004 96.262 -79.429 46.956
60005 96.283 -79.437 46.924
60006 96.303 -79.445 46.891
60007 96.323 -79.453 46.859
60008 96.344 -79.461 46.829
60009 96.364 -79.469 46.799
60010 96.384 -79.478 46.769
60011 96.405 -79.485 46.738
60012 96.425 -79.494 46.708
60013 96.446 -79.502 46.677
60014 96.466 -79.510 46.647
60015 96.486 -79.518 46.620
60016 96.507 -79.526 46.595
60017 96.527 -79.535 46.569
60018 96.547 -79.543 46.543
60019 96.568 -79.551 46.518
60020 96.588 -79.559 46.493
60021 96.608 -79.567 46.469
60022 96.629 -79.576 46.447
60023 96.649 -79.585 46.426
60024 96.673 -79.589 46.406
60025 96.698 -79.592 46.387
60026 96.730 -79.588 46.367
60027 96.763 -79.582 46.346
60028 96.796 -79.573 46.324
60029 96.829 -79.563 46.303
60030 96.859 -79.555 46.282
60031 96.883 -79.544 46.258
60032 96.907 -79.532 46.234
60033 96.932 -79.521 46.209
60034 96.957 -79.507 46.183
60035 96.986 -79.490 46.153
60036 97.014 -79.483 46.126
60037 97.042 -79.477 46.102
60038 97.069 -79.470 46.077
60039 97.097 -79.464 46.052
60040 97.124 -79.458 46.028
60041 97.152 -79.447 46.005
60042 97.164 -79.450 45.982
60043 97.176 -79.452 45.958
60044 97.188 -79.455 45.933
60045 97.201 -79.458 45.909
60046 97.213 -79.461 45.885
60047 97.231 -79.469 45.859
60048 97.241 -79.479 45.833
60049 97.240 -79.484 45.808
60050 97.239 -79.488 45.784
60051 97.237 -79.494 45.761
60052 97.232 -79.504 45.740
60053 97.206 -79.542 45.717
60054 97.189 -79.573 45.695
60055 97.196 -79.580 45.675
60056 97.203 -79.588 45.654
60057 97.210 -79.595 45.634
60058 97.217 -79.603 45.617
60059 97.224 -79.610 45.598
60060 97.231 -79.618 45.580
60061 97.238 -79.626 45.563
60062 97.245 -79.633 45.545
60063 97.253 -79.641 45.527
60064 97.260 -79.648 45.509
60065 97.267 -79.656 45.491
60066 97.274 -79.664 45.474
60067 97.281 -79.672 45.456
60068 97.288 -79.679 45.437
60069 97.295 -79.687 45.419
60070 97.302 -79.695 45.401
60071 97.309 -79.702 45.382
60072 97.316 -79.710 45.366
60073 97.323 -79.717 45.349
60074 97.330 -79.725 45.332
60075 97.337 -79.733 45.315
60076 97.345 -79.740 45.297
60077 97.352 -79.748 45.280
60469 93.241 -79.249 53.436
60470 93.256 -79.237 53.421
60471 93.270 -79.225 53.407
60472 93.285 -79.212 53.393
60473 93.305 -79.203 53.378
60474 93.315 -79.200 53.363
60475 93.325 -79.198 53.348
60476 93.335 -79.195 53.333
60477 93.344 -79.193 53.319
60478 93.354 -79.190 53.305
60479 93.364 -79.188 53.290
60480 93.373 -79.185 53.277
60481 93.383 -79.180 53.262
60482 93.395 -79.169 53.248
60483 93.407 -79.159 53.234
60484 93.418 -79.149 53.220
60485 93.426 -79.142 53.207
60486 93.434 -79.135 53.195
60487 93.442 -79.131 53.183
60488 93.447 -79.136 53.172
60489 93.460 -79.132 53.163
60490 93.480 -79.119 53.157
60491 93.499 -79.106 53.155
60492 93.518 -79.093 53.154
60493 93.537 -79.082 53.155
60494 93.551 -79.077 53.153
60495 93.565 -79.072 53.150
60496 93.578 -79.069 53.145
60497 93.590 -79.068 53.139
60498 93.599 -79.071 53.134
60499 93.607 -79.077 53.128
60500 93.615 -79.081 53.122
60501 93.622 -79.087 53.116
60502 93.630 -79.091 53.109
60503 93.648 -79.091 53.101
60504 93.666 -79.091 53.092
60505 93.683 -79.090 53.082
60506 93.698 -79.090 53.072
60507 93.713 -79.090 53.063
60508 93.727 -79.090 53.053
60509 93.743 -79.089 53.041
60510 93.757 -79.089 53.029
60511 93.772 -79.089 53.015
60512 93.787 -79.087 53.001
60513 93.801 -79.083 52.985
60514 93.817 -79.079 52.970
60515 93.823 -79.078 52.956
60516 93.827 -79.076 52.941
60517 93.830 -79.075 52.927
60518 93.834 -79.074 52.911
60519 93.838 -79.072 52.894
60520 93.844 -79.070 52.874
60521 93.852 -79.068 52.855
60522 93.860 -79.065 52.837
60523 93.868 -79.062 52.820
60524 93.875 -79.063 52.805
60525 93.881 -79.067 52.792
60526 93.888 -79.071 52.779
60527 93.894 -79.075 52.765
60528 93.896 -79.077 52.752
60529 93.898 -79.080 52.740
60530 93.901 -79.082 52.730
60531 93.904 -79.084 52.718
60532 93.907 -79.087 52.705
60533 93.911 -79.089 52.691
60534 93.914 -79.091 52.679
60535 93.917 -79.093 52.666
60536 93.920 -79.096 52.653
60537 93.923 -79.101 52.640
60538 93.927 -79.109 52.627
60539 93.931 -79.117 52.615
60540 93.935 -79.125 52.603
60541 93.939 -79.134 52.591
60542 93.945 -79.134 52.579
60543 93.953 -79.133 52.565
60544 93.969 -79.126 52.549
60545 93.986 -79.119 52.531
60546 94.002 -79.112 52.511
60547 94.002 -79.115 52.488
60548 94.001 -79.117 52.462
60549 94.001 -79.119 52.434
60550 94.000 -79.122 52.405
60551 94.000 -79.125 52.375
60552 93.999 -79.127 52.343
60553 93.999 -79.129 52.309
60554 93.998 -79.132 52.275
60555 93.998 -79.134 52.241
60556 93.997 -79.137 52.209
60557 94.002 -79.136 52.179
60558 94.010 -79.134 52.150
60559 94.019 -79.132 52.123
60560 94.028 -79.131 52.098
60561 94.036 -79.128 52.074
60562 94.045 -79.127 52.051
60563 94.053 -79.125 52.031
60564 94.062 -79.122 52.014
60565 94.071 -79.121 51.998
60566 94.076 -79.118 51.981
60567 94.095 -79.090 51.964
60568 94.114 -79.064 51.942
60569 94.132 -79.037 51.918
60570 94.150 -79.010 51.894
60571 94.169 -78.984 51.869
60572 94.188 -78.956 51.847
60573 94.208 -78.933 51.826
60574 94.229 -78.911 51.806
60575 94.250 -78.889 51.778
60576 94.269 -78.876 51.747
60577 94.282 -78.878 51.712
60578 94.294 -78.882 51.676
60579 94.304 -78.872 51.639
60580 94.324 -78.880 51.601
60581 94.343 -78.890 51.564
60582 94.362 -78.898 51.524
60583 94.382 -78.906 51.485
60584 94.401 -78.915 51.446
60585 94.420 -78.924 51.410
60586 94.439 -78.932 51.377
60587 94.458 -78.912 51.345
60588 94.481 -78.921 51.314
60589 94.504 -78.931 51.282
60590 94.527 -78.935 51.248
60591 94.549 -78.927 51.211
60592 94.565 -78.932 51.172
60593 94.581 -78.937 51.132
60594 94.598 -78.942 51.092
60595 94.612 -78.943 51.053
60596 94.623 -78.937 51.014
60597 94.633 -78.930 50.975
60598 94.644 -78.924 50.934
60599 94.655 -78.918 50.893
60600 94.666 -78.911 50.853
60601 94.677 -78.904 50.814
60602 94.687 -78.898 50.775
60603 94.698 -78.891 50.736
60604 94.709 -78.886 50.697
60605 94.720 -78.882 50.658
60606 94.732 -78.878 50.620
60607 94.743 -78.874 50.581
60608 94.754 -78.871 50.542
60609 94.765 -78.868 50.502
60610 94.776 -78.864 50.464
60611 94.785 -78.869 50.425
60612 94.793 -78.877 50.387
60613 94.802 -78.884 50.348
60614 94.810 -78.891 50.310
60615 94.819 -78.899 50.271
60616 94.827 -78.906 50.233
60617 94.835 -78.914 50.196
60618 94.843 -78.921 50.159
60619 94.851 -78.929 50.122
60620 94.860 -78.937 50.084
60621 94.868 -78.944 50.047
60622 94.877 -78.951 50.009
60623 94.885 -78.959 49.971
60624 94.893 -78.966 49.933
60625 94.901 -78.974 49.896
60626 94.910 -78.981 49.858
60627 94.922 -78.999 49.821
60628 94.934 -79.018 49.783
60629 94.947 -79.036 49.743
60630 94.957 -79.053 49.705
60631 94.964 -79.065 49.667
60632 94.970 -79.078 49.629
60633 94.976 -79.091 49.592
60634 94.983 -79.104 49.554
60635 94.989 -79.116 49.516
60636 94.996 -79.129 49.479
60637 95.002 -79.142 49.442
60638 95.008 -79.155 49.406
60639 95.015 -79.167 49.371
60640 95.022 -79.180 49.338
60641 95.028 -79.193 49.305
60642 95.034 -79.205 49.274
60643 95.041 -79.218 49.244
60644 95.053 -79.231 49.211
60645 95.067 -79.243 49.177
60646 95.082 -79.256 49.142
60647 95.096 -79.268 49.106
60648 95.110 -79.281 49.070
60649 95.127 -79.290 49.035
60650 95.148 -79.296 48.997
60651 95.168 -79.302 48.960
60652 95.188 -79.308 48.923
60653 95.206 -79.313 48.886
60654 95.217 -79.314 48.850
60655 95.237 -79.318 48.814
60656 95.262 -79.322 48.777
60657 95.286 -79.327 48.738
60658 95.312 -79.332 48.697
60659 95.336 -79.337 48.656
60660 95.355 -79.332 48.614
60661 95.373 -79.326 48.573
60662 95.391 -79.320 48.531
60663 95.408 -79.318 48.488
60664 95.421 -79.334 48.446
60665 95.434 -79.350 48.404
60666 95.450 -79.364 48.363
60667 95.489 -79.368 48.321
60668 95.528 -79.372 48.280
60669 95.558 -79.366 48.239
60670 95.580 -79.353 48.199
60671 95.603 -79.340 48.158
60672 95.613 -79.335 48.118
60673 95.624 -79.328 48.079
60674 95.634 -79.323 48.038
60675 95.644 -79.318 48.001
60676 95.654 -79.313 47.964
60677 95.665 -79.306 47.928
60678 95.675 -79.301 47.891
60679 95.685 -79.296 47.855
60680 95.695 -79.291 47.819
60681 95.705 -79.285 47.783
60682 95.737 -79.293 47.748
60683 95.763 -79.308 47.711
60684 95.772 -79.336 47.674
60685 95.782 -79.351 47.638
60686 95.792 -79.362 47.602
60687 95.802 -79.372 47.566
60688 95.834 -79.375 47.529
60689 95.886 -79.370 47.493
60690 95.894 -79.334 47.458
60691 95.890 -79.290 47.421
60692 95.886 -79.246 47.386
60693 95.932 -79.265 47.351
60694 95.979 -79.286 47.316
60695 96.027 -79.307 47.281
60696 96.075 -79.329 47.248
60697 96.111 -79.345 47.215
60698 96.132 -79.353 47.181
60699 96.152 -79.361 47.147
60700 96.173 -79.369 47.112
60701 96.193 -79.378 47.079
60702 96.214 -79.386 47.047
60703 96.234 -79.394 47.015
60704 96.254 -79.402 46.982
60705 96.274 -79.410 46.950
60706 96.295 -79.419 46.918
60707 96.315 -79.427 46.886
60708 96.335 -79.435 46.854
60709 96.356 -79.443 46.823
60710 96.376 -79.451 46.793
60711 96.397 -79.460 46.763
60712 96.417 -79.467 46.733
60713 96.438 -79.476 46.702
60714 96.458 -79.484 46.672
60715 96.478 -79.492 46.642
60716 96.499 -79.500 46.616
60717 96.519 -79.508 46.592
60718 96.539 -79.516 46.568
60719 96.559 -79.525 46.542
60720 96.580 -79.532 46.515
60721 96.600 -79.541 46.490
60722 96.621 -79.549 46.466
60723 96.641 -79.557 46.445
60724 96.662 -79.566 46.425
60725 96.681 -79.576 46.407
60726 96.699 -79.576 46.386
60727 96.719 -79.555 46.365
60728 96.752 -79.545 46.342
60729 96.785 -79.535 46.320
60730 96.818 -79.525 46.299
60731 96.851 -79.515 46.278
60732 96.891 -79.522 46.253
60733 96.915 -79.510 46.228
60734 96.940 -79.498 46.202
60735 96.964 -79.487 46.178
60736 96.994 -79.463 46.148
60737 97.022 -79.450 46.121
60738 97.050 -79.444 46.097
60739 97.078 -79.438 46.072
60740 97.105 -79.431 46.048
60741 97.133 -79.422 46.024
60742 97.148 -79.422 46.000
60743 97.161 -79.425 45.976
60744 97.173 -79.427 45.953
60745 97.185 -79.430 45.928
60746 97.198 -79.435 45.905
60747 97.215 -79.439 45.882
60748 97.232 -79.446 45.858
60749 97.248 -79.454 45.833
60750 97.250 -79.460 45.808
60751 97.249 -79.464 45.783
60752 97.248 -79.469 45.759
60753 97.247 -79.475 45.737
60754 97.245 -79.479 45.715
60755 97.239 -79.492 45.693
60756 97.212 -79.531 45.672
60757 97.203 -79.553 45.652
60758 97.210 -79.560 45.632
60759 97.217 -79.568 45.615
60760 97.225 -79.576 45.597
60761 97.232 -79.583 45.579
60762 97.239 -79.592 45.561
60763 97.246 -79.599 45.544
60764 97.253 -79.607 45.526
60765 97.260 -79.614 45.508
60766 97.267 -79.622 45.491
60767 97.274 -79.629 45.473
60768 97.281 -79.637 45.455
60769 97.288 -79.645 45.437
60770 97.295 -79.652 45.419
60771 97.302 -79.660 45.401
60772 97.309 -79.667 45.383
60773 97.316 -79.675 45.367
60774 97.324 -79.683 45.349
60775 97.331 -79.690 45.332
60776 97.338 -79.698 45.316
60777 97.345 -79.706 45.298
60778 97.352 -79.714 45.281
60779 97.371 -79.729 45.265
60835 98.445 -80.585 44.578
60836 98.464 -80.600 44.572
60837 98.487 -80.607 44.565
60838 98.512 -80.609 44.558
61172 93.269 -79.214 53.433
61173 93.285 -79.202 53.418
61174 93.306 -79.192 53.402
61175 93.315 -79.190 53.388
61176 93.325 -79.187 53.373
61177 93.335 -79.184 53.358
61178 93.345 -79.182 53.344
61179 93.354 -79.180 53.330
61180 93.364 -79.177 53.315
61181 93.373 -79.175 53.300
61182 93.383 -79.172 53.286
61183 93.394 -79.165 53.272
61184 93.405 -79.155 53.258
61185 93.417 -79.145 53.244
61186 93.427 -79.135 53.231
61187 93.436 -79.128 53.219
61188 93.444 -79.122 53.207
61189 93.452 -79.115 53.195
61190 93.458 -79.119 53.185
61191 93.464 -79.125 53.178
61192 93.481 -79.113 53.174
61193 93.500 -79.100 53.173
61194 93.519 -79.088 53.175
61195 93.539 -79.075 53.176
61196 93.558 -79.062 53.175
61197 93.574 -79.056 53.172
61198 93.582 -79.059 53.167
61199 93.590 -79.064 53.162
61200 93.598 -79.068 53.157
61201 93.606 -79.074 53.152
61202 93.613 -79.078 53.145
61203 93.623 -79.082 53.139
61204 93.640 -79.082 53.130
61205 93.658 -79.082 53.121
61206 93.676 -79.082 53.112
61207 93.691 -79.082 53.102
61208 93.706 -79.081 53.091
61209 93.721 -79.081 53.080
61210 93.736 -79.081 53.068
61211 93.751 -79.081 53.054
61212 93.765 -79.078 53.039
61213 93.780 -79.075 53.024
61214 93.795 -79.071 53.008
61215 93.809 -79.068 52.993
61216 93.815 -79.065 52.978
61217 93.819 -79.065 52.964
61218 93.823 -79.063 52.949
61219 93.827 -79.062 52.934
61220 93.832 -79.060 52.917
61221 93.840 -79.057 52.898
61222 93.848 -79.054 52.879
61223 93.856 -79.052 52.862
61224 93.864 -79.049 52.847
61225 93.871 -79.047 52.834
61226 93.878 -79.051 52.821
61227 93.884 -79.054 52.808
61228 93.891 -79.058 52.792
61229 93.893 -79.061 52.780
61230 93.894 -79.063 52.769
61231 93.896 -79.065 52.758
61232 93.900 -79.068 52.747
61233 93.903 -79.071 52.735
61234 93.906 -79.073 52.720
61235 93.909 -79.075 52.707
61236 93.912 -79.078 52.694
61237 93.915 -79.080 52.680
61238 93.919 -79.085 52.666
61239 93.923 -79.093 52.653
61240 93.927 -79.102 52.641
61241 93.931 -79.110 52.629
61242 93.935 -79.118 52.619
61243 93.939 -79.125 52.606
61244 93.957 -79.119 52.591
61245 93.974 -79.112 52.574
61246 93.991 -79.105 52.556
61247 94.003 -79.100 52.535
61248 94.003 -79.103 52.510
61249 94.002 -79.105 52.483
61250 94.002 -79.108 52.454
61251 94.001 -79.110 52.424
61252 94.001 -79.112 52.392
61253 94.000 -79.115 52.359
61254 94.000 -79.117 52.324
61255 93.999 -79.120 52.288
61256 94.000 -79.117 52.254
61257 94.001 -79.114 52.222
61258 94.006 -79.112 52.191
61259 94.014 -79.110 52.164
61260 94.023 -79.109 52.138
61261 94.032 -79.107 52.114
61262 94.040 -79.106 52.091
61263 94.048 -79.104 52.070
61264 94.057 -79.103 52.051
61265 94.065 -79.102 52.035
61266 94.074 -79.100 52.018
61267 94.086 -79.095 52.001
61268 94.107 -79.068 51.982
61269 94.126 -79.040 51.958
61270 94.145 -79.013 51.933
61271 94.163 -78.987 51.908
61272 94.182 -78.959 51.883
61273 94.200 -78.933 51.862
61274 94.219 -78.906 51.841
61275 94.240 -78.884 51.820
61276 94.261 -78.862 51.790
61277 94.277 -78.858 51.756
61278 94.290 -78.861 51.721
61279 94.303 -78.864 51.686
61280 94.314 -78.867 51.649
61281 94.324 -78.853 51.612
61282 94.343 -78.861 51.575
61283 94.362 -78.870 51.536
61284 94.382 -78.879 51.497
61285 94.401 -78.887 51.460
61286 94.420 -78.896 51.425
61287 94.438 -78.880 51.392
61288 94.461 -78.890 51.360
61289 94.484 -78.901 51.330
61290 94.507 -78.911 51.298
61291 94.531 -78.921 51.264
61292 94.552 -78.918 51.230
61293 94.568 -78.922 51.190
61294 94.584 -78.927 51.151
61295 94.601 -78.932 51.111
61296 94.617 -78.937 51.073
61297 94.629 -78.935 51.035
61298 94.640 -78.928 50.996
61299 94.651 -78.922 50.957
61300 94.662 -78.915 50.915
61301 94.672 -78.909 50.874
61302 94.683 -78.902 50.835
61303 94.694 -78.896 50.795
61304 94.705 -78.890 50.756
61305 94.715 -78.883 50.717
61306 94.726 -78.877 50.678
61307 94.737 -78.870 50.639
61308 94.748 -78.865 50.600
61309 94.759 -78.861 50.560
61310 94.771 -78.858 50.521
61311 94.782 -78.855 50.481
61312 94.793 -78.851 50.442
61313 94.802 -78.856 50.404
61314 94.810 -78.864 50.365
61315 94.819 -78.871 50.326
61316 94.827 -78.879 50.287
61317 94.835 -78.886 50.249
61318 94.843 -78.893 50.212
61319 94.852 -78.901 50.175
61320 94.860 -78.908 50.138
61321 94.868 -78.916 50.100
61322 94.877 -78.924 50.063
61323 94.885 -78.931 50.025
61324 94.893 -78.938 49.987
61325 94.905 -78.955 49.949
61326 94.917 -78.973 49.912
61327 94.929 -78.992 49.874
61328 94.936 -79.004 49.836
61329 94.942 -79.017 49.798
61330 94.949 -79.030 49.759
61331 94.955 -79.043 49.720
61332 94.962 -79.055 49.682
61333 94.968 -79.068 49.644
61334 94.974 -79.081 49.607
61335 94.981 -79.093 49.569
61336 94.987 -79.106 49.532
61337 94.993 -79.118 49.495
61338 95.000 -79.131 49.459
61339 95.006 -79.143 49.423
61340 95.013 -79.156 49.390
61341 95.016 -79.165 49.359
61342 95.019 -79.174 49.328
61343 95.025 -79.193 49.298
61344 95.044 -79.203 49.267
61345 95.059 -79.218 49.234
61346 95.075 -79.232 49.201
61347 95.089 -79.245 49.166
61348 95.103 -79.257 49.130
61349 95.118 -79.270 49.093
61350 95.136 -79.278 49.056
61351 95.156 -79.284 49.018
61352 95.177 -79.290 48.979
61353 95.197 -79.296 48.941
61354 95.217 -79.302 48.903
61355 95.231 -79.304 48.865
61356 95.250 -79.308 48.827
61357 95.274 -79.313 48.788
61358 95.299 -79.317 48.747
61359 95.323 -79.321 48.705
61360 95.341 -79.315 48.663
61361 95.359 -79.309 48.620
61362 95.377 -79.303 48.577
61363 95.395 -79.297 48.535
61364 95.409 -79.306 48.492
61365 95.422 -79.321 48.448
61366 95.435 -79.337 48.406
61367 95.448 -79.353 48.364
61368 95.486 -79.357 48.323
61369 95.526 -79.362 48.282
61370 95.555 -79.355 48.240
61371 95.577 -79.342 48.200
61372 95.600 -79.328 48.159
61373 95.616 -79.319 48.119
61374 95.626 -79.314 48.079
61375 95.636 -79.309 48.040
61376 95.647 -79.303 48.002
61377 95.657 -79.297 47.965
61378 95.667 -79.292 47.928
61379 95.677 -79.287 47.891
61380 95.687 -79.281 47.854
61381 95.698 -79.275 47.817
61382 95.709 -79.271 47.780
61383 95.743 -79.281 47.744
61384 95.776 -79.291 47.708
61385 95.785 -79.319 47.671
61386 95.795 -79.347 47.634
61387 95.805 -79.357 47.598
61388 95.815 -79.367 47.562
61389 95.849 -79.369 47.525
61390 95.887 -79.355 47.490
61391 95.883 -79.311 47.455
61392 95.879 -79.267 47.419
61393 95.875 -79.223 47.383
61394 95.912 -79.231 47.347
61395 95.960 -79.253 47.312
61396 96.008 -79.274 47.278
61397 96.055 -79.295 47.244
61398 96.103 -79.317 47.210
61399 96.144 -79.335 47.176
61400 96.164 -79.343 47.142
61401 96.185 -79.351 47.108
61402 96.205 -79.360 47.074
61403 96.226 -79.368 47.041
61404 96.246 -79.375 47.009
61405 96.266 -79.384 46.977
61406 96.287 -79.392 46.945
61407 96.307 -79.400 46.913
61408 96.327 -79.408 46.881
61409 96.347 -79.416 46.849
61410 96.368 -79.425 46.818
61411 96.388 -79.433 46.787
61412 96.409 -79.441 46.757
61413 96.429 -79.449 46.727
61414 96.450 -79.457 46.697
61415 96.470 -79.466 46.667
61416 96.490 -79.473 46.637
61417 96.511 -79.482 46.610
61418 96.531 -79.490 46.585
61419 96.551 -79.498 46.560
61420 96.572 -79.506 46.535
61421 96.592 -79.514 46.510
61422 96.613 -79.523 46.484
61423 96.633 -79.534 46.460
61424 96.654 -79.544 46.439
61425 96.670 -79.549 46.419
61426 96.684 -79.548 46.399
61427 96.707 -79.551 46.377
61428 96.726 -79.531 46.354
61429 96.744 -79.510 46.331
61430 96.773 -79.497 46.309
61431 96.806 -79.487 46.289
61432 96.839 -79.476 46.269
61433 96.875 -79.472 46.245
61434 96.920 -79.485 46.220
61435 96.947 -79.476 46.194
61436 96.971 -79.464 46.169
61437 97.000 -79.444 46.141
61438 97.030 -79.418 46.116
61439 97.058 -79.410 46.093
61440 97.086 -79.404 46.069
61441 97.113 -79.397 46.045
61442 97.133 -79.394 46.021
61443 97.145 -79.397 45.997
61444 97.157 -79.399 45.973
61445 97.170 -79.404 45.950
61446 97.184 -79.408 45.927
61447 97.198 -79.413 45.904
61448 97.220 -79.416 45.880
61449 97.235 -79.424 45.855
61450 97.255 -79.432 45.830
61451 97.261 -79.441 45.805
61452 97.257 -79.446 45.781
61453 97.258 -79.445 45.756
61454 97.257 -79.451 45.734
61455 97.256 -79.455 45.713
61456 97.254 -79.460 45.692
61457 97.253 -79.466 45.671
61458 97.245 -79.480 45.650
61459 97.218 -79.519 45.631
61460 97.218 -79.534 45.613
61461 97.225 -79.541 45.596
61462 97.232 -79.549 45.578
61463 97.239 -79.557 45.560
61464 97.246 -79.564 45.543
61465 97.253 -79.572 45.525
61466 97.260 -79.579 45.507
61467 97.267 -79.587 45.490
61468 97.274 -79.595 45.473
61469 97.281 -79.602 45.455
61470 97.288 -79.610 45.438
61471 97.296 -79.618 45.419
61472 97.303 -79.626 45.401
61473 97.310 -79.633 45.384
61474 97.317 -79.641 45.368
61475 97.324 -79.648 45.350
61476 97.331 -79.656 45.333
61477 97.338 -79.664 45.317
61478 97.345 -79.671 45.299
61479 97.358 -79.683 45.282
61480 97.377 -79.698 45.266
61481 97.396 -79.713 45.251
61534 98.412 -80.522 44.601
61535 98.431 -80.538 44.592
61536 98.450 -80.554 44.586
61537 98.469 -80.569 44.579
61538 98.492 -80.576 44.572
61539 98.517 -80.578 44.566
61540 98.542 -80.581 44.559
61541 98.567 -80.583 44.551
61542 98.591 -80.585 44.543
61543 98.616 -80.588 44.535
61544 98.641 -80.591 44.528
61545 98.665 -80.593 44.522
61546 98.690 -80.596 44.515
61547 98.715 -80.599 44.507
61548 98.741 -80.604 44.500
61549 98.767 -80.607 44.495
61550 98.792 -80.611 44.488
61551 98.817 -80.615 44.481
61552 98.843 -80.620 44.476
61553 98.867 -80.627 44.471
61874 93.284 -79.192 53.443
61875 93.306 -79.181 53.427
61876 93.315 -79.178 53.413
61877 93.326 -79.176 53.398
61878 93.335 -79.174 53.384
61879 93.345 -79.171 53.369
61880 93.354 -79.169 53.355
61881 93.364 -79.166 53.340
61882 93.374 -79.164 53.326
61883 93.383 -79.161 53.311
61884 93.393 -79.159 53.296
61885 93.404 -79.151 53.282
61886 93.415 -79.141 53.268
61887 93.427 -79.131 53.255
61888 93.437 -79.122 53.242
61889 93.445 -79.115 53.229
61890 93.453 -79.108 53.217
61891 93.462 -79.101 53.207
61892 93.468 -79.102 53.199
61893 93.474 -79.107 53.195
61894 93.483 -79.108 53.192
61895 93.502 -79.095 53.195
61896 93.521 -79.082 53.197
61897 93.541 -79.069 53.198
61898 93.560 -79.056 53.196
61899 93.573 -79.053 53.193
61900 93.581 -79.056 53.190
61901 93.588 -79.061 53.185
61902 93.596 -79.065 53.180
61903 93.604 -79.071 53.174
61904 93.615 -79.074 53.168
61905 93.633 -79.074 53.159
61906 93.650 -79.074 53.150
61907 93.668 -79.074 53.140
61908 93.685 -79.073 53.130
61909 93.699 -79.073 53.118
61910 93.714 -79.073 53.105
61911 93.729 -79.072 53.093
61912 93.744 -79.069 53.078
61913 93.759 -79.066 53.063
61914 93.773 -79.062 53.047
61915 93.788 -79.059 53.031
61916 93.802 -79.056 53.016
61917 93.808 -79.054 53.001
61918 93.812 -79.053 52.986
61919 93.816 -79.051 52.972
61920 93.820 -79.049 52.957
61921 93.828 -79.047 52.940
61922 93.836 -79.044 52.922
61923 93.844 -79.041 52.905
61924 93.852 -79.039 52.889
61925 93.860 -79.036 52.874
61926 93.868 -79.033 52.862
61927 93.874 -79.034 52.850
61928 93.881 -79.038 52.837
61929 93.887 -79.042 52.823
61930 93.890 -79.045 52.809
61931 93.891 -79.047 52.797
61932 93.892 -79.049 52.785
61933 93.895 -79.052 52.775
61934 93.898 -79.054 52.763
61935 93.901 -79.057 52.749
61936 93.904 -79.059 52.734
61937 93.908 -79.062 52.719
61938 93.911 -79.064 52.704
61939 93.914 -79.070 52.689
61940 93.918 -79.078 52.677
61941 93.922 -79.087 52.666
61942 93.926 -79.094 52.654
61943 93.930 -79.103 52.643
61944 93.936 -79.103 52.630
61945 93.962 -79.105 52.614
61946 93.979 -79.098 52.598
61947 93.995 -79.090 52.579
61948 94.004 -79.088 52.557
61949 94.003 -79.090 52.530
61950 94.003 -79.093 52.502
61951 94.002 -79.096 52.472
61952 94.002 -79.098 52.441
61953 94.001 -79.100 52.408
61954 94.002 -79.099 52.374
61955 94.003 -79.095 52.339
61956 94.004 -79.092 52.302
61957 94.005 -79.088 52.268
61958 94.006 -79.085 52.235
61959 94.012 -79.083 52.204
61960 94.020 -79.081 52.178
61961 94.029 -79.080 52.154
61962 94.037 -79.078 52.131
61963 94.046 -79.077 52.110
61964 94.054 -79.075 52.090
61965 94.062 -79.074 52.071
61966 94.071 -79.072 52.054
61967 94.080 -79.071 52.036
61968 94.095 -79.063 52.017
61969 94.112 -79.053 51.997
61970 94.132 -79.025 51.972
61971 94.153 -78.995 51.947
61972 94.175 -78.965 51.923
61973 94.194 -78.936 51.899
61974 94.213 -78.909 51.878
61975 94.231 -78.882 51.857
61976 94.251 -78.858 51.833
61977 94.271 -78.839 51.802
61978 94.285 -78.841 51.768
61979 94.298 -78.843 51.733
61980 94.312 -78.846 51.697
61981 94.323 -78.851 51.660
61982 94.330 -78.840 51.623
61983 94.343 -78.834 51.587
61984 94.362 -78.843 51.548
61985 94.382 -78.851 51.510
61986 94.401 -78.859 51.474
61987 94.420 -78.861 51.439
61988 94.441 -78.860 51.407
61989 94.465 -78.870 51.376
61990 94.487 -78.880 51.346
61991 94.510 -78.890 51.314
61992 94.534 -78.900 51.282
61993 94.555 -78.908 51.248
61994 94.571 -78.912 51.209
61995 94.588 -78.918 51.170
61996 94.604 -78.923 51.132
61997 94.620 -78.927 51.094
61998 94.636 -78.933 51.057
61999 94.647 -78.927 51.019
62000 94.657 -78.920 50.982
62001 94.668 -78.914 50.939
62002 94.679 -78.907 50.899
62003 94.690 -78.901 50.859
62004 94.700 -78.894 50.821
62005 94.711 -78.887 50.781
62006 94.722 -78.881 50.741
62007 94.733 -78.874 50.701
62008 94.744 -78.868 50.661
62009 94.754 -78.861 50.622
62010 94.765 -78.855 50.583
62011 94.776 -78.849 50.542
62012 94.787 -78.846 50.501
62013 94.799 -78.842 50.461
62014 94.810 -78.838 50.422
62015 94.819 -78.843 50.382
62016 94.827 -78.851 50.343
62017 94.835 -78.858 50.305
62018 94.843 -78.866 50.266
62019 94.852 -78.874 50.228
62020 94.860 -78.880 50.191
62021 94.868 -78.888 50.154
62022 94.877 -78.896 50.116
62023 94.888 -78.910 50.079
62024 94.900 -78.929 50.041
62025 94.908 -78.943 50.003
62026 94.914 -78.955 49.965
62027 94.921 -78.968 49.928
62028 94.927 -78.981 49.890
62029 94.933 -78.993 49.853
62030 94.940 -79.006 49.814
62031 94.947 -79.019 49.775
62032 94.953 -79.031 49.736
62033 94.959 -79.044 49.698
62034 94.963 -79.053 49.660
62035 94.965 -79.062 49.622
62036 94.968 -79.070 49.585
62037 94.970 -79.078 49.548
62038 94.973 -79.087 49.511
62039 94.975 -79.095 49.476
62040 94.978 -79.103 49.442
62041 94.979 -79.117 49.410
62042 94.975 -79.158 49.380
62043 95.007 -79.165 49.350
62044 95.029 -79.173 49.320
62045 95.045 -79.182 49.288
62046 95.061 -79.197 49.255
62047 95.076 -79.211 49.221
62048 95.092 -79.225 49.187
62049 95.108 -79.241 49.152
62050 95.123 -79.255 49.114
62051 95.144 -79.265 49.075
62052 95.165 -79.272 49.036
62053 95.185 -79.278 48.996
62054 95.205 -79.284 48.957
62055 95.226 -79.290 48.917
62056 95.244 -79.295 48.877
62057 95.262 -79.298 48.837
62058 95.287 -79.303 48.796
62059 95.309 -79.303 48.754
62060 95.327 -79.297 48.712
62061 95.344 -79.291 48.669
62062 95.362 -79.285 48.625
62063 95.380 -79.279 48.582
62064 95.397 -79.277 48.539
62065 95.410 -79.293 48.496
62066 95.423 -79.309 48.452
62067 95.436 -79.324 48.409
62068 95.449 -79.340 48.368
62069 95.483 -79.346 48.327
62070 95.523 -79.350 48.285
62071 95.552 -79.344 48.243
62072 95.574 -79.331 48.202
62073 95.597 -79.318 48.162
62074 95.619 -79.305 48.122
62075 95.629 -79.299 48.082
62076 95.639 -79.294 48.043
62077 95.650 -79.288 48.005
62078 95.660 -79.283 47.967
62079 95.670 -79.277 47.929
62080 95.680 -79.272 47.892
62081 95.690 -79.266 47.854
62082 95.700 -79.261 47.817
62083 95.716 -79.259 47.779
62084 95.750 -79.268 47.742
62085 95.783 -79.278 47.706
62086 95.799 -79.301 47.669
62087 95.808 -79.329 47.632
62088 95.818 -79.353 47.596
62089 95.828 -79.363 47.559
62090 95.864 -79.364 47.522
62091 95.876 -79.332 47.487
62092 95.872 -79.288 47.451
62093 95.868 -79.244 47.416
62094 95.864 -79.200 47.380
62095 95.893 -79.197 47.344
62096 95.941 -79.219 47.309
62097 95.988 -79.241 47.274
62098 96.036 -79.262 47.240
62099 96.084 -79.283 47.205
62100 96.131 -79.305 47.171
62101 96.177 -79.325 47.137
62102 96.197 -79.333 47.103
62103 96.217 -79.341 47.069
62104 96.238 -79.349 47.036
62105 96.258 -79.357 47.003
62106 96.278 -79.366 46.971
62107 96.299 -79.374 46.939
62108 96.319 -79.382 46.907
62109 96.339 -79.390 46.875
62110 96.360 -79.398 46.843
62111 96.380 -79.407 46.812
62112 96.400 -79.414 46.781
62113 96.421 -79.422 46.751
62114 96.441 -79.431 46.721
62115 96.462 -79.439 46.691
62116 96.482 -79.447 46.661
62117 96.503 -79.455 46.631
62118 96.523 -79.463 46.604
62119 96.543 -79.472 46.578
62120 96.564 -79.481 46.552
62121 96.585 -79.491 46.528
62122 96.605 -79.501 46.503
62123 96.626 -79.512 46.477
62124 96.647 -79.523 46.453
62125 96.658 -79.523 46.431
62126 96.669 -79.521 46.410
62127 96.692 -79.523 46.388
62128 96.715 -79.526 46.365
62129 96.734 -79.507 46.341
62130 96.752 -79.486 46.317
62131 96.771 -79.465 46.295
62132 96.795 -79.448 46.275
62133 96.827 -79.438 46.256
62134 96.861 -79.428 46.234
62135 96.899 -79.429 46.210
62136 96.945 -79.442 46.185
62137 96.979 -79.442 46.159
62138 97.005 -79.427 46.133
62139 97.036 -79.400 46.109
62140 97.066 -79.377 46.087
62141 97.094 -79.371 46.065
62142 97.116 -79.366 46.042
62143 97.129 -79.369 46.018
62144 97.143 -79.373 45.995
62145 97.156 -79.378 45.971
62146 97.170 -79.382 45.948
62147 97.184 -79.386 45.927
62148 97.203 -79.391 45.904
62149 97.223 -79.395 45.877
62150 97.239 -79.403 45.850
62151 97.268 -79.410 45.825
62152 97.279 -79.419 45.801
62153 97.272 -79.429 45.777
62154 97.264 -79.438 45.755
62155 97.261 -79.441 45.733
62156 97.266 -79.432 45.711
62157 97.265 -79.436 45.690
62158 97.263 -79.441 45.670
62159 97.262 -79.447 45.650
62160 97.261 -79.451 45.631
62161 97.251 -79.469 45.614
62162 97.225 -79.507 45.596
62163 97.232 -79.514 45.578
62164 97.239 -79.522 45.561
62165 97.246 -79.529 45.543
62166 97.253 -79.538 45.525
62167 97.260 -79.545 45.507
62168 97.267 -79.553 45.490
62169 97.275 -79.560 45.473
62170 97.282 -79.568 45.456
62171 97.289 -79.576 45.438
62172 97.296 -79.583 45.420
62173 97.303 -79.591 45.402
62174 97.310 -79.598 45.385
62175 97.317 -79.606 45.369
62176 97.324 -79.614 45.352
62177 97.331 -79.621 45.335
62178 97.338 -79.629 45.319
62179 97.345 -79.637 45.301
62180 97.363 -79.651 45.285
62181 97.382 -79.667 45.269
62182 97.401 -79.682 45.254
62233 98.379 -80.461 44.623
62234 98.398 -80.476 44.616
62235 98.417 -80.491 44.609
62236 98.436 -80.507 44.601
62237 98.455 -80.522 44.594
62238 98.475 -80.538 44.587
62239 98.498 -80.544 44.581
62240 98.522 -80.547 44.574
62241 98.547 -80.549 44.566
62242 98.572 -80.552 44.559
62243 98.597 -80.554 44.552
62244 98.622 -80.557 44.543
62245 98.646 -80.560 44.536
62246 98.671 -80.562 44.530
62247 98.696 -80.565 44.523
62248 98.721 -80.568 44.516
62249 98.746 -80.572 44.508
62250 98.772 -80.576 44.501
62251 98.797 -80.580 44.495
62252 98.823 -80.584 44.489
62253 98.848 -80.589 44.484
62254 98.871 -80.600 44.478
62577 93.316 -79.168 53.438
62578 93.326 -79.165 53.423
62579 93.335 -79.162 53.409
62580 93.345 -79.160 53.395
62581 93.355 -79.158 53.379
62582 93.364 -79.156 53.365
62583 93.374 -79.153 53.351
62584 93.384 -79.150 53.336
62585 93.394 -79.148 53.321
62586 93.403 -79.146 53.306
62587 93.414 -79.137 53.292
62588 93.425 -79.127 53.278
62589 93.437 -79.116 53.265
62590 93.446 -79.109 53.252
62591 93.454 -79.101 53.239
62592 93.463 -79.094 53.229
62593 93.471 -79.087 53.221
62594 93.479 -79.085 53.214
62595 93.484 -79.090 53.210
62596 93.490 -79.095 53.211
62597 93.504 -79.089 53.212
62598 93.523 -79.076 53.213
62599 93.542 -79.063 53.213
62600 93.561 -79.051 53.212
62601 93.572 -79.051 53.210
62602 93.579 -79.056 53.207
62603 93.587 -79.059 53.203
62604 93.595 -79.064 53.198
62605 93.608 -79.065 53.192
62606 93.625 -79.065 53.185
62607 93.643 -79.065 53.176
62608 93.660 -79.065 53.166
62609 93.678 -79.065 53.156
62610 93.693 -79.065 53.143
62611 93.707 -79.064 53.129
62612 93.722 -79.061 53.115
62613 93.737 -79.057 53.101
62614 93.752 -79.054 53.085
62615 93.766 -79.050 53.069
62616 93.781 -79.047 53.054
62617 93.795 -79.043 53.039
62618 93.800 -79.042 53.024
62619 93.804 -79.040 53.009
62620 93.808 -79.039 52.995
62621 93.816 -79.037 52.980
62622 93.824 -79.034 52.964
62623 93.832 -79.031 52.947
62624 93.840 -79.028 52.931
62625 93.848 -79.025 52.916
62626 93.856 -79.023 52.902
62627 93.863 -79.020 52.891
62628 93.871 -79.018 52.880
62629 93.877 -79.022 52.867
62630 93.884 -79.026 52.853
62631 93.887 -79.028 52.839
62632 93.888 -79.031 52.825
62633 93.889 -79.034 52.813
62634 93.890 -79.036 52.802
62635 93.893 -79.038 52.789
62636 93.897 -79.040 52.774
62637 93.900 -79.043 52.759
62638 93.903 -79.045 52.743
62639 93.906 -79.048 52.727
62640 93.910 -79.055 52.712
62641 93.914 -79.063 52.700
62642 93.918 -79.071 52.687
62643 93.922 -79.079 52.674
62644 93.922 -79.080 52.662
62645 93.933 -79.080 52.649
62646 93.959 -79.082 52.634
62647 93.984 -79.084 52.616
62648 94.000 -79.076 52.596
62649 94.005 -79.076 52.573
62650 94.004 -79.078 52.547
62651 94.004 -79.081 52.519
62652 94.004 -79.081 52.489
62653 94.005 -79.077 52.457
62654 94.006 -79.074 52.423
62655 94.007 -79.070 52.389
62656 94.008 -79.066 52.355
62657 94.010 -79.063 52.318
62658 94.011 -79.059 52.282
62659 94.012 -79.056 52.249
62660 94.017 -79.054 52.218
62661 94.026 -79.053 52.193
62662 94.034 -79.051 52.171
62663 94.043 -79.049 52.150
62664 94.051 -79.048 52.130
62665 94.060 -79.046 52.111
62666 94.069 -79.045 52.092
62667 94.078 -79.043 52.072
62668 94.087 -79.041 52.052
62669 94.104 -79.031 52.031
62670 94.120 -79.021 52.009
62671 94.137 -79.012 51.984
62672 94.158 -78.983 51.960
62673 94.179 -78.952 51.937
62674 94.200 -78.921 51.916
62675 94.221 -78.891 51.894
62676 94.243 -78.861 51.872
62677 94.262 -78.832 51.847
62678 94.279 -78.821 51.815
62679 94.292 -78.824 51.780
62680 94.306 -78.826 51.745
62681 94.320 -78.829 51.708
62682 94.332 -78.833 51.671
62683 94.343 -78.838 51.635
62684 94.346 -78.814 51.600
62685 94.362 -78.814 51.562
62686 94.382 -78.824 51.525
62687 94.401 -78.832 51.489
62688 94.421 -78.829 51.454
62689 94.445 -78.839 51.422
62690 94.468 -78.849 51.391
62691 94.491 -78.859 51.361
62692 94.515 -78.876 51.330
62693 94.536 -78.893 51.298
62694 94.550 -78.901 51.265
62695 94.569 -78.906 51.228
62696 94.591 -78.908 51.190
62697 94.607 -78.913 51.151
62698 94.623 -78.918 51.114
62699 94.639 -78.923 51.077
62700 94.653 -78.924 51.041
62701 94.664 -78.918 51.005
62702 94.675 -78.912 50.964
62703 94.686 -78.905 50.924
62704 94.696 -78.899 50.886
62705 94.707 -78.893 50.849
62706 94.718 -78.886 50.811
62707 94.729 -78.879 50.771
62708 94.740 -78.873 50.731
62709 94.750 -78.866 50.690
62710 94.761 -78.860 50.650
62711 94.772 -78.853 50.609
62712 94.782 -78.847 50.568
62713 94.793 -78.840 50.525
62714 94.804 -78.833 50.483
62715 94.815 -78.829 50.443
62716 94.826 -78.825 50.402
62717 94.835 -78.830 50.363
62718 94.844 -78.838 50.325
62719 94.852 -78.846 50.286
62720 94.860 -78.853 50.248
62721 94.871 -78.865 50.210
62722 94.880 -78.881 50.172
62723 94.887 -78.894 50.134
62724 94.893 -78.907 50.096
62725 94.899 -78.919 50.058
62726 94.906 -78.932 50.020
62727 94.909 -78.942 49.982
62728 94.912 -78.950 49.945
62729 94.914 -78.959 49.907
62730 94.917 -78.966 49.869
62731 94.919 -78.974 49.831
62732 94.922 -78.983 49.791
62733 94.924 -78.991 49.753
62734 94.927 -78.999 49.715
62735 94.929 -79.008 49.677
62736 94.932 -79.016 49.639
62737 94.934 -79.024 49.602
62738 94.936 -79.032 49.566
62739 94.939 -79.040 49.530
62740 94.935 -79.076 49.496
62741 94.929 -79.118 49.464
62742 94.957 -79.129 49.432
62743 94.989 -79.136 49.400
62744 95.013 -79.143 49.368
62745 95.030 -79.153 49.335
62746 95.047 -79.162 49.302
62747 95.063 -79.175 49.269
62748 95.078 -79.190 49.235
62749 95.094 -79.204 49.201
62750 95.109 -79.219 49.166
62751 95.126 -79.231 49.129
62752 95.148 -79.241 49.090
62753 95.170 -79.250 49.050
62754 95.191 -79.259 49.009
62755 95.212 -79.269 48.968
62756 95.234 -79.278 48.928
62757 95.254 -79.284 48.887
62758 95.274 -79.288 48.845
62759 95.294 -79.285 48.802
62760 95.312 -79.279 48.759
62761 95.330 -79.273 48.717
62762 95.348 -79.267 48.674
62763 95.366 -79.261 48.632
62764 95.384 -79.255 48.588
62765 95.398 -79.265 48.544
62766 95.411 -79.280 48.501
62767 95.424 -79.296 48.458
62768 95.437 -79.312 48.416
62769 95.450 -79.327 48.374
62770 95.481 -79.335 48.331
62771 95.520 -79.339 48.289
62772 95.549 -79.332 48.247
62773 95.571 -79.319 48.205
62774 95.594 -79.306 48.166
62775 95.616 -79.293 48.127
62776 95.631 -79.285 48.087
62777 95.641 -79.279 48.049
62778 95.652 -79.273 48.010
62779 95.662 -79.268 47.971
62780 95.672 -79.263 47.932
62781 95.683 -79.257 47.895
62782 95.693 -79.251 47.857
62783 95.703 -79.246 47.819
62784 95.723 -79.247 47.780
62785 95.757 -79.256 47.742
62786 95.790 -79.266 47.705
62787 95.812 -79.284 47.668
62788 95.821 -79.312 47.632
62789 95.831 -79.341 47.595
62790 95.840 -79.359 47.557
62791 95.869 -79.352 47.521
62792 95.865 -79.308 47.485
62793 95.861 -79.264 47.449
62794 95.857 -79.220 47.413
62795 95.853 -79.176 47.376
62796 95.874 -79.164 47.341
62797 95.921 -79.185 47.305
62798 95.969 -79.206 47.270
62799 96.017 -79.228 47.236
62800 96.064 -79.250 47.201
62801 96.112 -79.271 47.166
62802 96.160 -79.293 47.132
62803 96.207 -79.314 47.097
62804 96.230 -79.322 47.062
62805 96.250 -79.331 47.030
62806 96.270 -79.339 46.997
62807 96.291 -79.347 46.965
62808 96.311 -79.356 46.932
62809 96.331 -79.363 46.900
62810 96.352 -79.372 46.868
62811 96.372 -79.380 46.835
62812 96.392 -79.388 46.805
62813 96.413 -79.396 46.775
62814 96.433 -79.404 46.745
62815 96.454 -79.413 46.714
62816 96.474 -79.421 46.684
62817 96.494 -79.429 46.655
62818 96.515 -79.438 46.626
62819 96.536 -79.448 46.598
62820 96.556 -79.459 46.571
62821 96.577 -79.469 46.545
62822 96.598 -79.479 46.519
62823 96.619 -79.490 46.494
62824 96.637 -79.498 46.470
62825 96.646 -79.496 46.445
62826 96.655 -79.494 46.422
62827 96.677 -79.496 46.399
62828 96.699 -79.498 46.376
62829 96.722 -79.501 46.352
62830 96.742 -79.484 46.328
62831 96.760 -79.463 46.305
62832 96.779 -79.442 46.281
62833 96.798 -79.421 46.259
62834 96.816 -79.400 46.239
62835 96.849 -79.390 46.217
62836 96.882 -79.379 46.195
62837 96.924 -79.387 46.173
62838 96.969 -79.400 46.148
62839 97.011 -79.408 46.122
62840 97.041 -79.382 46.098
62841 97.072 -79.356 46.077
62842 97.101 -79.338 46.056
62843 97.114 -79.342 46.034
62844 97.128 -79.347 46.011
62845 97.142 -79.351 45.987
62846 97.156 -79.356 45.963
62847 97.169 -79.360 45.941
62848 97.186 -79.364 45.919
62849 97.208 -79.368 45.894
62850 97.226 -79.374 45.870
62851 97.251 -79.382 45.845
62852 97.281 -79.389 45.820
62853 97.296 -79.398 45.797
62854 97.289 -79.407 45.774
62855 97.282 -79.417 45.752
62856 97.275 -79.427 45.731
62857 97.267 -79.437 45.710
62858 97.265 -79.437 45.689
62859 97.271 -79.424 45.669
62860 97.273 -79.422 45.649
62861 97.272 -79.427 45.630
62862 97.271 -79.432 45.613
62863 97.269 -79.438 45.596
62864 97.257 -79.457 45.578
62865 97.239 -79.488 45.560
62866 97.246 -79.495 45.543
62867 97.253 -79.503 45.525
62868 97.260 -79.510 45.507
62869 97.267 -79.518 45.490
62870 97.275 -79.526 45.473
62871 97.282 -79.533 45.456
62872 97.289 -79.541 45.438
62873 97.296 -79.548 45.421
62874 97.303 -79.557 45.403
62875 97.310 -79.564 45.386
62876 97.317 -79.572 45.369
62877 97.324 -79.579 45.353
62878 97.331 -79.587 45.337
62879 97.338 -79.595 45.320
62880 97.349 -79.604 45.304
62881 97.368 -79.620 45.288
62882 97.387 -79.635 45.272
62883 97.406 -79.651 45.256
62884 97.426 -79.666 45.241
62931 98.327 -80.384 44.654
62932 98.346 -80.399 44.646
62933 98.365 -80.415 44.639
62934 98.384 -80.430 44.632
62935 98.404 -80.445 44.625
62936 98.423 -80.460 44.618
62937 98.442 -80.475 44.610
62938 98.461 -80.491 44.603
62939 98.480 -80.506 44.596
62940 98.503 -80.513 44.590
62941 98.528 -80.516 44.583
62942 98.553 -80.518 44.576
62943 98.577 -80.520 44.568
62944 98.602 -80.523 44.560
62945 98.627 -80.526 44.551
62946 98.652 -80.529 44.544
62947 98.676 -80.531 44.538
62948 98.701 -80.533 44.531
62949 98.726 -80.537 44.524
62950 98.751 -80.541 44.516
62951 98.777 -80.544 44.509
62952 98.803 -80.548 44.503
62953 98.828 -80.553 44.498
62954 98.852 -80.562 44.492
62955 98.875 -80.573 44.486
62956 98.899 -80.583 44.481
63280 93.336 -79.152 53.433
63281 93.345 -79.150 53.418
63282 93.355 -79.147 53.403
63283 93.365 -79.144 53.389
63284 93.374 -79.142 53.374
63285 93.384 -79.140 53.359
63286 93.394 -79.137 53.343
63287 93.404 -79.134 53.330
63288 93.413 -79.132 53.316
63289 93.424 -79.123 53.301
63290 93.436 -79.112 53.289
63291 93.447 -79.102 53.275
63292 93.455 -79.095 53.262
63293 93.464 -79.088 53.251
63294 93.472 -79.081 53.241
63295 93.481 -79.074 53.234
63296 93.489 -79.068 53.228
63297 93.494 -79.073 53.226
63298 93.500 -79.078 53.225
63299 93.505 -79.083 53.226
63300 93.524 -79.070 53.227
63301 93.544 -79.058 53.227
63302 93.562 -79.046 53.227
63303 93.570 -79.050 53.225
63304 93.578 -79.054 53.222
63305 93.586 -79.059 53.218
63306 93.601 -79.059 53.213
63307 93.618 -79.058 53.206
63308 93.635 -79.057 53.197
63309 93.653 -79.056 53.187
63310 93.671 -79.056 53.178
63311 93.686 -79.056 53.164
63312 93.701 -79.053 53.150
63313 93.715 -79.049 53.135
63314 93.730 -79.046 53.121
63315 93.745 -79.042 53.106
63316 93.759 -79.038 53.090
63317 93.774 -79.035 53.077
63318 93.788 -79.031 53.063
63319 93.793 -79.030 53.049
63320 93.797 -79.029 53.034
63321 93.803 -79.027 53.021
63322 93.812 -79.024 53.006
63323 93.820 -79.021 52.990
63324 93.828 -79.018 52.974
63325 93.836 -79.015 52.958
63326 93.844 -79.012 52.944
63327 93.851 -79.010 52.931
63328 93.859 -79.007 52.921
63329 93.867 -79.004 52.910
63330 93.874 -79.006 52.897
63331 93.880 -79.009 52.883
63332 93.883 -79.012 52.869
63333 93.884 -79.015 52.855
63334 93.885 -79.017 52.842
63335 93.887 -79.020 52.828
63336 93.889 -79.022 52.814
63337 93.892 -79.024 52.799
63338 93.895 -79.027 52.783
63339 93.898 -79.029 52.767
63340 93.901 -79.031 52.751
63341 93.905 -79.040 52.735
63342 93.909 -79.047 52.721
63343 93.913 -79.056 52.708
63344 93.915 -79.059 52.693
63345 93.913 -79.056 52.680
63346 93.929 -79.057 52.666
63347 93.956 -79.059 52.651
63348 93.981 -79.062 52.632
63349 94.005 -79.062 52.612
63350 94.006 -79.062 52.589
63351 94.007 -79.059 52.562
63352 94.008 -79.055 52.533
63353 94.009 -79.052 52.504
63354 94.010 -79.048 52.473
63355 94.012 -79.044 52.439
63356 94.013 -79.041 52.406
63357 94.014 -79.037 52.372
63358 94.015 -79.034 52.334
63359 94.016 -79.031 52.300
63360 94.017 -79.028 52.268
63361 94.023 -79.024 52.238
63362 94.031 -79.024 52.213
63363 94.040 -79.022 52.191
63364 94.048 -79.021 52.171
63365 94.057 -79.018 52.149
63366 94.067 -79.017 52.129
63367 94.076 -79.015 52.109
63368 94.085 -79.013 52.088
63369 94.096 -79.009 52.066
63370 94.113 -78.999 52.045
63371 94.129 -78.990 52.024
63372 94.146 -78.981 52.001
63373 94.162 -78.971 51.977
63374 94.183 -78.940 51.954
63375 94.204 -78.909 51.932
63376 94.226 -78.879 51.911
63377 94.247 -78.849 51.888
63378 94.268 -78.818 51.862
63379 94.285 -78.808 51.828
63380 94.300 -78.806 51.793
63381 94.314 -78.809 51.757
63382 94.327 -78.811 51.720
63383 94.340 -78.814 51.684
63384 94.352 -78.820 51.648
63385 94.363 -78.824 51.613
63386 94.363 -78.788 51.577
63387 94.382 -78.796 51.540
63388 94.402 -78.798 51.504
63389 94.424 -78.808 51.470
63390 94.448 -78.820 51.437
63391 94.474 -78.845 51.406
63392 94.500 -78.869 51.377
63393 94.515 -78.877 51.346
63394 94.530 -78.886 51.314
63395 94.544 -78.894 51.281
63396 94.558 -78.902 51.247
63397 94.578 -78.906 51.210
63398 94.601 -78.908 51.171
63399 94.623 -78.910 51.134
63400 94.642 -78.914 51.097
63401 94.658 -78.918 51.062
63402 94.671 -78.916 51.027
63403 94.682 -78.910 50.988
63404 94.692 -78.903 50.950
63405 94.703 -78.897 50.912
63406 94.714 -78.890 50.877
63407 94.724 -78.884 50.839
63408 94.736 -78.877 50.800
63409 94.746 -78.871 50.760
63410 94.757 -78.865 50.719
63411 94.768 -78.858 50.677
63412 94.778 -78.852 50.635
63413 94.789 -78.845 50.593
63414 94.800 -78.838 50.549
63415 94.811 -78.832 50.505
63416 94.822 -78.825 50.464
63417 94.832 -78.819 50.425
63418 94.843 -78.812 50.387
63419 94.852 -78.820 50.349
63420 94.856 -78.830 50.311
63421 94.858 -78.838 50.272
63422 94.861 -78.846 50.233
63423 94.863 -78.855 50.194
63424 94.866 -78.863 50.154
63425 94.868 -78.871 50.115
63426 94.871 -78.879 50.076
63427 94.874 -78.887 50.037
63428 94.876 -78.896 49.999
63429 94.879 -78.904 49.962
63430 94.881 -78.912 49.924
63431 94.884 -78.921 49.886
63432 94.886 -78.929 49.848
63433 94.888 -78.937 49.809
63434 94.891 -78.945 49.770
63435 94.893 -78.953 49.732
63436 94.896 -78.962 49.694
63437 94.898 -78.970 49.657
63438 94.897 -78.992 49.620
63439 94.892 -79.034 49.585
63440 94.886 -79.077 49.549
63441 94.907 -79.093 49.515
63442 94.939 -79.100 49.482
63443 94.972 -79.107 49.449
63444 94.997 -79.115 49.415
63445 95.014 -79.124 49.382
63446 95.031 -79.134 49.349
63447 95.048 -79.143 49.315
63448 95.064 -79.153 49.282
63449 95.079 -79.168 49.249
63450 95.095 -79.182 49.215
63451 95.111 -79.197 49.178
63452 95.130 -79.208 49.141
63453 95.152 -79.217 49.102
63454 95.174 -79.226 49.062
63455 95.195 -79.235 49.021
63456 95.216 -79.245 48.979
63457 95.238 -79.254 48.938
63458 95.261 -79.262 48.895
63459 95.281 -79.266 48.853
63460 95.298 -79.261 48.810
63461 95.316 -79.255 48.766
63462 95.334 -79.250 48.723
63463 95.351 -79.244 48.680
63464 95.370 -79.237 48.637
63465 95.386 -79.236 48.593
63466 95.399 -79.252 48.550
63467 95.412 -79.267 48.507
63468 95.425 -79.283 48.463
63469 95.438 -79.299 48.420
63470 95.451 -79.314 48.378
63471 95.478 -79.324 48.335
63472 95.517 -79.328 48.293
63473 95.546 -79.322 48.252
63474 95.568 -79.308 48.212
63475 95.591 -79.295 48.172
63476 95.613 -79.281 48.133
63477 95.634 -79.269 48.093
63478 95.644 -79.264 48.053
63479 95.655 -79.259 48.014
63480 95.665 -79.253 47.975
63481 95.675 -79.247 47.936
63482 95.685 -79.242 47.898
63483 95.695 -79.237 47.859
63484 95.705 -79.231 47.821
63485 95.730 -79.234 47.782
63486 95.763 -79.244 47.744
63487 95.797 -79.253 47.707
63488 95.825 -79.266 47.669
63489 95.835 -79.294 47.632
63490 95.844 -79.323 47.594
63491 95.853 -79.351 47.556
63492 95.858 -79.328 47.520
63493 95.853 -79.285 47.483
63494 95.849 -79.241 47.447
63495 95.846 -79.197 47.411
63496 95.842 -79.153 47.375
63497 95.855 -79.130 47.339
63498 95.902 -79.152 47.303
63499 95.950 -79.173 47.267
63500 95.997 -79.194 47.232
63501 96.045 -79.216 47.197
63502 96.093 -79.237 47.162
63503 96.141 -79.259 47.128
63504 96.188 -79.280 47.093
63505 96.236 -79.302 47.058
63506 96.262 -79.313 47.025
63507 96.283 -79.321 46.993
63508 96.303 -79.329 46.960
63509 96.323 -79.337 46.927
63510 96.343 -79.345 46.895
63511 96.364 -79.353 46.863
63512 96.384 -79.362 46.831
63513 96.404 -79.369 46.800
63514 96.425 -79.378 46.770
63515 96.445 -79.386 46.740
63516 96.466 -79.395 46.710
63517 96.486 -79.406 46.680
63518 96.508 -79.416 46.650
63519 96.528 -79.426 46.621
63520 96.549 -79.437 46.593
63521 96.570 -79.447 46.565
63522 96.591 -79.457 46.538
63523 96.611 -79.468 46.512
63524 96.625 -79.471 46.487
63525 96.635 -79.469 46.462
63526 96.644 -79.467 46.435
63527 96.662 -79.468 46.411
63528 96.684 -79.471 46.389
63529 96.707 -79.473 46.365
63530 96.730 -79.476 46.341
63531 96.750 -79.460 46.316
63532 96.768 -79.439 46.292
63533 96.787 -79.419 46.268
63534 96.806 -79.397 46.246
63535 96.824 -79.376 46.225
63536 96.843 -79.355 46.203
63537 96.870 -79.341 46.181
63538 96.903 -79.332 46.159
63539 96.949 -79.344 46.136
63540 96.994 -79.357 46.111
63541 97.043 -79.347 46.088
63542 97.058 -79.307 46.067
63543 97.098 -79.323 46.047
63544 97.113 -79.324 46.025
63545 97.127 -79.325 46.002
63546 97.141 -79.329 45.978
63547 97.155 -79.334 45.954
63548 97.169 -79.338 45.932
63549 97.191 -79.342 45.912
63550 97.213 -79.346 45.887
63551 97.234 -79.353 45.864
63552 97.264 -79.360 45.841
63553 97.290 -79.371 45.817
63554 97.307 -79.375 45.793
63555 97.306 -79.386 45.771
63556 97.299 -79.396 45.749
63557 97.292 -79.406 45.729
63558 97.285 -79.415 45.708
63559 97.278 -79.425 45.688
63560 97.271 -79.435 45.668
63561 97.270 -79.432 45.648
63562 97.276 -79.419 45.630
63563 97.281 -79.408 45.613
63564 97.280 -79.413 45.595
63565 97.278 -79.418 45.577
63566 97.277 -79.423 45.559
63567 97.263 -79.445 45.542
63568 97.253 -79.468 45.524
63569 97.260 -79.476 45.507
63570 97.267 -79.484 45.490
63571 97.275 -79.491 45.473
63572 97.282 -79.499 45.456
63573 97.289 -79.507 45.439
63574 97.296 -79.514 45.422
63575 97.303 -79.522 45.404
63576 97.310 -79.529 45.386
63577 97.317 -79.537 45.370
63578 97.324 -79.545 45.354
63579 97.331 -79.552 45.338
63580 97.338 -79.560 45.322
63581 97.355 -79.573 45.306
63582 97.374 -79.588 45.290
63583 97.393 -79.604 45.274
63584 97.412 -79.619 45.259
63585 97.431 -79.635 45.244
63586 97.450 -79.650 45.229
63587 97.469 -79.665 45.213
63588 97.489 -79.680 45.198
63630 98.294 -80.322 44.680
63631 98.313 -80.337 44.672
63632 98.332 -80.353 44.662
63633 98.351 -80.368 44.655
63634 98.370 -80.383 44.648
63635 98.390 -80.398 44.641
63636 98.409 -80.414 44.634
63637 98.428 -80.429 44.627
63638 98.447 -80.444 44.619
63639 98.466 -80.460 44.612
63640 98.486 -80.475 44.605
63641 98.508 -80.482 44.599
63642 98.533 -80.484 44.592
63643 98.558 -80.487 44.585
63644 98.583 -80.489 44.578
63645 98.607 -80.492 44.569
63646 98.632 -80.494 44.560
63647 98.657 -80.497 44.553
63648 98.682 -80.500 44.547
63649 98.707 -80.502 44.539
63650 98.731 -80.505 44.532
63651 98.757 -80.509 44.525
63652 98.783 -80.513 44.518
63653 98.808 -80.517 44.512
63654 98.833 -80.524 44.506
63655 98.856 -80.535 44.502
63656 98.879 -80.545 44.496
63657 98.903 -80.556 44.491
63658 98.926 -80.566 44.485
63983 93.355 -79.136 53.426
63984 93.365 -79.134 53.412
63985 93.374 -79.131 53.397
63986 93.384 -79.128 53.382
63987 93.394 -79.126 53.367
63988 93.404 -79.124 53.353
63989 93.413 -79.122 53.338
63990 93.423 -79.118 53.325
63991 93.434 -79.109 53.311
63992 93.446 -79.098 53.297
63993 93.457 -79.088 53.284
63994 93.465 -79.081 53.272
63995 93.474 -79.075 53.261
63996 93.482 -79.068 53.252
63997 93.490 -79.061 53.246
63998 93.498 -79.053 53.240
63999 93.505 -79.056 53.237
64000 93.510 -79.061 53.237
64001 93.515 -79.066 53.238
64002 93.526 -79.065 53.239
64003 93.546 -79.052 53.240
64004 93.561 -79.044 53.240
64005 93.569 -79.049 53.238
64006 93.577 -79.053 53.235
64007 93.594 -79.052 53.232
64008 93.612 -79.051 53.225
64009 93.629 -79.050 53.217
64010 93.646 -79.049 53.207
64011 93.664 -79.046 53.198
64012 93.680 -79.043 53.185
64013 93.694 -79.040 53.171
64014 93.708 -79.037 53.155
64015 93.723 -79.034 53.142
64016 93.737 -79.030 53.128
64017 93.752 -79.027 53.113
64018 93.767 -79.023 53.101
64019 93.782 -79.020 53.088
64020 93.786 -79.018 53.075
64021 93.791 -79.016 53.060
64022 93.799 -79.014 53.046
64023 93.807 -79.011 53.032
64024 93.816 -79.008 53.017
64025 93.824 -79.005 53.001
64026 93.832 -79.002 52.987
64027 93.839 -78.999 52.974
64028 93.847 -78.996 52.961
64029 93.855 -78.994 52.951
64030 93.863 -78.991 52.941
64031 93.871 -78.990 52.928
64032 93.877 -78.993 52.914
64033 93.880 -78.996 52.900
64034 93.881 -78.999 52.885
64035 93.882 -79.001 52.872
64036 93.883 -79.003 52.858
64037 93.884 -79.006 52.843
64038 93.887 -79.009 52.826
64039 93.890 -79.011 52.809
64040 93.894 -79.013 52.791
64041 93.897 -79.016 52.775
64042 93.901 -79.024 52.758
64043 93.905 -79.032 52.743
64044 93.908 -79.039 52.727
64045 93.906 -79.036 52.712
64046 93.905 -79.033 52.697
64047 93.926 -79.034 52.681
64048 93.952 -79.037 52.665
64049 93.978 -79.039 52.646
64050 93.997 -79.043 52.626
64051 94.008 -79.044 52.603
64052 94.010 -79.038 52.576
64053 94.012 -79.032 52.546
64054 94.014 -79.026 52.517
64055 94.016 -79.020 52.489
64056 94.017 -79.015 52.456
64057 94.018 -79.012 52.423
64058 94.019 -79.009 52.388
64059 94.020 -79.006 52.352
64060 94.022 -79.002 52.319
64061 94.023 -78.999 52.289
64062 94.028 -78.996 52.260
64063 94.037 -78.995 52.236
64064 94.046 -78.993 52.214
64065 94.055 -78.991 52.193
64066 94.064 -78.989 52.170
64067 94.073 -78.987 52.148
64068 94.082 -78.986 52.126
64069 94.092 -78.984 52.105
64070 94.105 -78.977 52.083
64071 94.121 -78.968 52.062
64072 94.138 -78.959 52.042
64073 94.154 -78.949 52.018
64074 94.171 -78.939 51.995
64075 94.188 -78.927 51.973
64076 94.209 -78.897 51.951
64077 94.230 -78.867 51.928
64078 94.251 -78.836 51.903
64079 94.272 -78.805 51.875
64080 94.289 -78.800 51.842
64081 94.305 -78.797 51.807
64082 94.321 -78.795 51.770
64083 94.335 -78.794 51.733
64084 94.348 -78.797 51.698
64085 94.360 -78.802 51.663
64086 94.372 -78.807 51.628
64087 94.380 -78.799 51.593
64088 94.382 -78.767 51.557
64089 94.407 -78.789 51.520
64090 94.433 -78.814 51.487
64091 94.460 -78.838 51.454
64092 94.480 -78.854 51.422
64093 94.494 -78.862 51.392
64094 94.509 -78.871 51.361
64095 94.525 -78.877 51.330
64096 94.543 -78.881 51.298
64097 94.562 -78.885 51.265
64098 94.581 -78.890 51.228
64099 94.600 -78.895 51.190
64100 94.619 -78.899 51.153
64101 94.638 -78.904 51.118
64102 94.657 -78.909 51.083
64103 94.677 -78.914 51.050
64104 94.688 -78.907 51.012
64105 94.698 -78.899 50.973
64106 94.709 -78.892 50.934
64107 94.719 -78.884 50.898
64108 94.730 -78.877 50.862
64109 94.741 -78.869 50.824
64110 94.751 -78.861 50.784
64111 94.762 -78.854 50.743
64112 94.772 -78.846 50.701
64113 94.782 -78.839 50.660
64114 94.793 -78.831 50.617
64115 94.804 -78.824 50.573
64116 94.814 -78.816 50.529
64117 94.821 -78.812 50.487
64118 94.825 -78.811 50.450
64119 94.834 -78.812 50.414
64120 94.857 -78.814 50.378
64121 94.855 -78.815 50.339
64122 94.849 -78.814 50.299
64123 94.843 -78.814 50.259
64124 94.836 -78.814 50.217
64125 94.833 -78.817 50.176
64126 94.835 -78.825 50.135
64127 94.838 -78.833 50.095
64128 94.840 -78.842 50.056
64129 94.842 -78.849 50.017
64130 94.845 -78.858 49.980
64131 94.847 -78.866 49.943
64132 94.850 -78.874 49.904
64133 94.852 -78.883 49.865
64134 94.855 -78.891 49.827
64135 94.857 -78.899 49.788
64136 94.859 -78.908 49.750
64137 94.854 -78.950 49.712
64138 94.848 -78.993 49.675
64139 94.845 -79.032 49.638
64140 94.854 -79.059 49.603
64141 94.889 -79.065 49.569
64142 94.922 -79.071 49.534
64143 94.954 -79.078 49.499
64144 94.982 -79.085 49.465
64145 94.998 -79.095 49.431
64146 95.016 -79.104 49.398
64147 95.032 -79.114 49.365
64148 95.049 -79.124 49.331
64149 95.065 -79.133 49.297
64150 95.081 -79.146 49.263
64151 95.097 -79.161 49.229
64152 95.113 -79.175 49.192
64153 95.134 -79.184 49.153
64154 95.156 -79.194 49.114
64155 95.177 -79.203 49.074
64156 95.199 -79.212 49.032
64157 95.222 -79.220 48.990
64158 95.247 -79.225 48.947
64159 95.273 -79.228 48.904
64160 95.296 -79.233 48.861
64161 95.301 -79.237 48.817
64162 95.319 -79.231 48.773
64163 95.337 -79.225 48.730
64164 95.355 -79.219 48.686
64165 95.373 -79.213 48.643
64166 95.387 -79.223 48.599
64167 95.400 -79.239 48.555
64168 95.413 -79.255 48.512
64169 95.426 -79.270 48.468
64170 95.439 -79.286 48.425
64171 95.452 -79.302 48.383
64172 95.475 -79.313 48.340
64173 95.515 -79.317 48.299
64174 95.543 -79.310 48.258
64175 95.565 -79.297 48.219
64176 95.588 -79.284 48.178
64177 95.610 -79.271 48.138
64178 95.633 -79.257 48.097
64179 95.647 -79.250 48.057
64180 95.657 -79.244 48.018
64181 95.667 -79.238 47.979
64182 95.678 -79.233 47.940
64183 95.688 -79.228 47.901
64184 95.698 -79.222 47.862
64185 95.708 -79.216 47.823
64186 95.736 -79.222 47.785
64187 95.770 -79.231 47.746
64188 95.804 -79.241 47.708
64189 95.837 -79.250 47.670
64190 95.848 -79.278 47.632
64191 95.858 -79.306 47.594
64192 95.866 -79.332 47.556
64193 95.861 -79.310 47.519
64194 95.843 -79.262 47.483
64195 95.838 -79.218 47.446
64196 95.835 -79.174 47.411
64197 95.831 -79.130 47.375
64198 95.835 -79.096 47.339
64199 95.883 -79.118 47.302
64200 95.931 -79.140 47.266
64201 95.978 -79.161 47.230
64202 96.026 -79.182 47.195
64203 96.074 -79.204 47.159
64204 96.121 -79.225 47.124
64205 96.169 -79.247 47.089
64206 96.217 -79.268 47.055
64207 96.264 -79.290 47.021
64208 96.295 -79.303 46.988
64209 96.315 -79.310 46.955
64210 96.335 -79.319 46.922
64211 96.356 -79.327 46.890
64212 96.376 -79.335 46.858
64213 96.396 -79.343 46.826
64214 96.417 -79.353 46.795
64215 96.438 -79.363 46.765
64216 96.459 -79.373 46.735
64217 96.479 -79.384 46.705
64218 96.500 -79.394 46.675
64219 96.521 -79.404 46.645
64220 96.542 -79.415 46.616
64221 96.562 -79.426 46.588
64222 96.583 -79.435 46.560
64223 96.604 -79.446 46.532
64224 96.614 -79.444 46.506
64225 96.623 -79.443 46.479
64226 96.631 -79.441 46.454
64227 96.647 -79.441 46.426
64228 96.669 -79.443 46.402
64229 96.692 -79.445 46.379
64230 96.715 -79.448 46.355
64231 96.738 -79.451 46.331
64232 96.758 -79.437 46.306
64233 96.777 -79.416 46.280
64234 96.795 -79.395 46.256
64235 96.814 -79.374 46.233
64236 96.832 -79.353 46.212
64237 96.851 -79.332 46.190
64238 96.870 -79.311 46.168
64239 96.892 -79.293 46.147
64240 96.928 -79.289 46.125
64241 96.974 -79.301 46.100
64242 97.007 -79.277 46.078
64243 97.056 -79.294 46.058
64244 97.094 -79.312 46.037
64245 97.109 -79.313 46.015
64246 97.124 -79.313 45.992
64247 97.139 -79.314 45.968
64248 97.154 -79.315 45.945
64249 97.174 -79.316 45.924
64250 97.196 -79.320 45.904
64251 97.218 -79.324 45.880
64252 97.220 -79.350 45.859
64253 97.209 -79.381 45.837
64254 97.246 -79.363 45.814
64255 97.280 -79.353 45.790
64256 97.296 -79.357 45.768
64257 97.300 -79.370 45.747
64258 97.304 -79.382 45.726
64259 97.302 -79.394 45.707
64260 97.295 -79.404 45.687
64261 97.288 -79.413 45.667
64262 97.281 -79.422 45.647
64263 97.274 -79.432 45.629
64264 97.274 -79.428 45.613
64265 97.280 -79.415 45.595
64266 97.285 -79.403 45.577
64267 97.288 -79.399 45.559
64268 97.286 -79.404 45.542
64269 97.285 -79.409 45.524
64270 97.269 -79.434 45.506
64271 97.267 -79.449 45.490
64272 97.275 -79.457 45.473
64273 97.282 -79.464 45.456
64274 97.289 -79.472 45.439
64275 97.296 -79.479 45.422
64276 97.303 -79.487 45.405
64277 97.310 -79.494 45.387
64278 97.317 -79.503 45.372
64279 97.324 -79.510 45.355
64280 97.331 -79.518 45.339
64281 97.341 -79.527 45.325
64282 97.360 -79.542 45.308
64283 97.379 -79.557 45.292
64284 97.398 -79.573 45.277
64285 97.418 -79.588 45.261
64286 97.437 -79.603 45.247
64287 97.456 -79.618 45.232
64288 97.475 -79.634 45.216
64289 97.494 -79.649 45.202
64290 97.513 -79.664 45.187
64291 97.532 -79.679 45.172
64292 97.551 -79.695 45.158
64329 98.261 -80.260 44.706
64330 98.280 -80.275 44.697
64331 98.299 -80.291 44.689
64332 98.318 -80.306 44.680
64333 98.338 -80.322 44.671
64334 98.357 -80.337 44.664
64335 98.376 -80.352 44.657
64336 98.395 -80.367 44.650
64337 98.414 -80.382 44.643
64338 98.433 -80.398 44.636
64339 98.452 -80.413 44.629
64340 98.472 -80.428 44.621
64341 98.491 -80.444 44.615
64342 98.514 -80.450 44.609
64343 98.538 -80.453 44.602
64344 98.564 -80.455 44.595
64345 98.588 -80.458 44.587
64346 98.613 -80.460 44.578
64347 98.638 -80.463 44.569
64348 98.662 -80.466 44.562
64349 98.687 -80.468 44.556
64350 98.712 -80.471 44.549
64351 98.737 -80.474 44.542
64352 98.763 -80.478 44.535
64353 98.788 -80.482 44.528
64354 98.813 -80.486 44.521
64355 98.837 -80.497 44.515
64356 98.860 -80.507 44.510
64357 98.883 -80.518 44.505
64358 98.907 -80.529 44.499
64359 98.930 -80.539 44.493
64360 98.953 -80.549 44.488
64686 93.375 -79.120 53.420
64687 93.384 -79.118 53.405
64688 93.394 -79.115 53.390
64689 93.404 -79.112 53.376
64690 93.414 -79.110 53.362
64691 93.423 -79.108 53.348
64692 93.433 -79.105 53.333
64693 93.444 -79.094 53.319
64694 93.456 -79.084 53.306
64695 93.466 -79.075 53.294
64696 93.475 -79.068 53.283
64697 93.483 -79.061 53.273
64698 93.491 -79.054 53.265
64699 93.500 -79.047 53.258
64700 93.508 -79.040 53.253
64701 93.515 -79.039 53.250
64702 93.520 -79.044 53.249
64703 93.526 -79.049 53.249
64704 93.532 -79.054 53.251
64705 93.547 -79.046 53.253
64706 93.560 -79.043 53.254
64707 93.570 -79.046 53.253
64708 93.587 -79.045 53.250
64709 93.605 -79.044 53.243
64710 93.624 -79.040 53.235
64711 93.643 -79.034 53.226
64712 93.661 -79.030 53.217
64713 93.675 -79.028 53.205
64714 93.689 -79.025 53.191
64715 93.703 -79.023 53.176
64716 93.716 -79.021 53.164
64717 93.730 -79.018 53.151
64718 93.746 -79.015 53.138
64719 93.760 -79.011 53.126
64720 93.775 -79.008 53.114
64721 93.779 -79.006 53.101
64722 93.787 -79.003 53.087
64723 93.795 -79.000 53.073
64724 93.803 -78.998 53.058
64725 93.812 -78.995 53.044
64726 93.820 -78.992 53.029
64727 93.827 -78.990 53.016
64728 93.835 -78.987 53.004
64729 93.843 -78.984 52.992
64730 93.851 -78.981 52.982
64731 93.859 -78.978 52.971
64732 93.867 -78.975 52.959
64733 93.874 -78.977 52.945
64734 93.876 -78.980 52.931
64735 93.877 -78.983 52.916
64736 93.879 -78.985 52.901
64737 93.880 -78.987 52.887
64738 93.881 -78.990 52.872
64739 93.883 -78.992 52.855
64740 93.886 -78.995 52.837
64741 93.889 -78.997 52.819
64742 93.892 -79.001 52.802
64743 93.896 -79.009 52.784
64744 93.900 -79.017 52.766
64745 93.899 -79.015 52.749
64746 93.898 -79.012 52.731
64747 93.897 -79.009 52.714
64748 93.923 -79.012 52.696
64749 93.949 -79.014 52.677
64750 93.968 -79.019 52.658
64751 93.985 -79.024 52.637
64752 94.003 -79.029 52.615
64753 94.009 -79.027 52.588
64754 94.011 -79.021 52.559
64755 94.013 -79.015 52.530
64756 94.015 -79.009 52.502
64757 94.017 -79.002 52.471
64758 94.019 -78.996 52.439
64759 94.022 -78.990 52.406
64760 94.024 -78.984 52.372
64761 94.026 -78.978 52.341
64762 94.028 -78.972 52.311
64763 94.034 -78.967 52.284
64764 94.044 -78.965 52.261
64765 94.053 -78.963 52.239
64766 94.062 -78.962 52.216
64767 94.071 -78.959 52.192
64768 94.080 -78.958 52.168
64769 94.089 -78.956 52.144
64770 94.099 -78.954 52.123
64771 94.114 -78.946 52.102
64772 94.130 -78.937 52.081
64773 94.146 -78.927 52.061
64774 94.163 -78.917 52.037
64775 94.180 -78.907 52.014
64776 94.196 -78.897 51.993
64777 94.213 -78.885 51.969
64778 94.235 -78.855 51.945
64779 94.256 -78.824 51.919
64780 94.276 -78.796 51.889
64781 94.292 -78.793 51.856
64782 94.308 -78.789 51.821
64783 94.324 -78.786 51.784
64784 94.340 -78.784 51.748
64785 94.356 -78.781 51.714
64786 94.368 -78.786 51.680
64787 94.379 -78.793 51.646
64788 94.391 -78.802 51.610
64789 94.395 -78.786 51.574
64790 94.419 -78.807 51.538
64791 94.445 -78.830 51.503
64792 94.459 -78.839 51.470
64793 94.475 -78.845 51.439
64794 94.494 -78.849 51.408
64795 94.512 -78.853 51.377
64796 94.531 -78.857 51.346
64797 94.550 -78.861 51.315
64798 94.568 -78.865 51.282
64799 94.587 -78.870 51.246
64800 94.607 -78.874 51.209
64801 94.626 -78.880 51.172
64802 94.645 -78.884 51.137
64803 94.663 -78.888 51.103
64804 94.677 -78.889 51.070
64805 94.690 -78.885 51.034
64806 94.700 -78.877 50.995
64807 94.711 -78.870 50.957
64808 94.721 -78.862 50.921
64809 94.732 -78.855 50.885
64810 94.742 -78.847 50.847
64811 94.753 -78.839 50.807
64812 94.763 -78.832 50.765
64813 94.774 -78.824 50.723
64814 94.784 -78.817 50.681
64815 94.789 -78.815 50.638
64816 94.793 -78.814 50.594
64817 94.797 -78.814 50.550
64818 94.801 -78.813 50.507
64819 94.805 -78.813 50.469
64820 94.827 -78.815 50.434
64821 94.850 -78.818 50.399
64822 94.874 -78.821 50.362
64823 94.871 -78.821 50.324
64824 94.864 -78.821 50.285
64825 94.858 -78.821 50.243
64826 94.851 -78.821 50.200
64827 94.845 -78.820 50.158
64828 94.839 -78.820 50.116
64829 94.832 -78.820 50.076
64830 94.826 -78.820 50.037
64831 94.820 -78.820 49.999
64832 94.814 -78.820 49.961
64833 94.817 -78.828 49.924
64834 94.819 -78.836 49.886
64835 94.817 -78.866 49.847
64836 94.824 -78.895 49.809
64837 94.832 -78.923 49.771
64838 94.840 -78.951 49.733
64839 94.848 -78.979 49.694
64840 94.856 -79.007 49.658
64841 94.863 -79.035 49.623
64842 94.896 -79.045 49.588
64843 94.935 -79.049 49.553
64844 94.966 -79.056 49.518
64845 94.983 -79.065 49.485
64846 95.000 -79.075 49.452
64847 95.017 -79.085 49.419
64848 95.033 -79.094 49.386
64849 95.050 -79.104 49.352
64850 95.066 -79.113 49.316
64851 95.083 -79.125 49.280
64852 95.098 -79.139 49.243
64853 95.117 -79.151 49.206
64854 95.138 -79.160 49.167
64855 95.160 -79.169 49.126
64856 95.182 -79.178 49.085
64857 95.207 -79.183 49.043
64858 95.233 -79.187 49.000
64859 95.259 -79.191 48.957
64860 95.284 -79.196 48.913
64861 95.310 -79.200 48.869
64862 95.314 -79.206 48.824
64863 95.323 -79.208 48.781
64864 95.341 -79.202 48.737
64865 95.358 -79.196 48.694
64866 95.376 -79.195 48.650
64867 95.388 -79.211 48.606
64868 95.401 -79.226 48.562
64869 95.414 -79.242 48.517
64870 95.427 -79.258 48.474
64871 95.440 -79.273 48.431
64872 95.453 -79.289 48.389
64873 95.472 -79.302 48.348
64874 95.512 -79.306 48.306
64875 95.540 -79.299 48.265
64876 95.562 -79.285 48.225
64877 95.585 -79.272 48.184
64878 95.607 -79.259 48.142
64879 95.630 -79.246 48.100
64880 95.650 -79.234 48.060
64881 95.660 -79.229 48.020
64882 95.670 -79.224 47.981
64883 95.680 -79.218 47.943
64884 95.690 -79.212 47.904
64885 95.700 -79.207 47.865
64886 95.711 -79.202 47.826
64887 95.743 -79.210 47.787
64888 95.776 -79.219 47.748
64889 95.810 -79.228 47.710
64890 95.844 -79.238 47.672
64891 95.862 -79.260 47.634
64892 95.871 -79.288 47.595
64893 95.875 -79.297 47.557
64894 95.870 -79.278 47.520
64895 95.861 -79.247 47.484
64896 95.843 -79.200 47.448
64897 95.825 -79.151 47.412
64898 95.820 -79.106 47.376
64899 95.816 -79.063 47.340
64900 95.864 -79.084 47.303
64901 95.911 -79.106 47.266
64902 95.959 -79.128 47.230
64903 96.007 -79.149 47.194
64904 96.054 -79.170 47.158
64905 96.102 -79.192 47.122
64906 96.150 -79.213 47.087
64907 96.197 -79.234 47.052
64908 96.245 -79.256 47.017
64909 96.293 -79.278 46.984
64910 96.327 -79.292 46.951
64911 96.347 -79.300 46.918
64912 96.368 -79.310 46.885
64913 96.389 -79.320 46.853
64914 96.409 -79.331 46.821
64915 96.431 -79.341 46.791
64916 96.451 -79.351 46.760
64917 96.472 -79.362 46.730
64918 96.493 -79.372 46.700
64919 96.514 -79.382 46.671
64920 96.534 -79.393 46.641
64921 96.555 -79.404 46.612
64922 96.576 -79.414 46.584
64923 96.593 -79.420 46.556
64924 96.602 -79.418 46.527
64925 96.611 -79.416 46.500
64926 96.620 -79.414 46.473
64927 96.631 -79.413 46.445
64928 96.654 -79.416 46.418
64929 96.677 -79.418 46.393
64930 96.700 -79.420 46.369
64931 96.723 -79.423 46.345
64932 96.745 -79.426 46.321
64933 96.765 -79.413 46.297
64934 96.784 -79.393 46.271
64935 96.803 -79.372 46.246
64936 96.821 -79.350 46.223
64937 96.840 -79.329 46.201
64938 96.859 -79.309 46.179
64939 96.877 -79.288 46.159
64940 96.896 -79.266 46.137
64941 96.914 -79.245 46.115
64942 96.955 -79.245 46.091
64943 97.000 -79.255 46.069
64944 97.058 -79.287 46.047
64945 97.091 -79.300 46.026
64946 97.105 -79.301 46.005
64947 97.120 -79.302 45.983
64948 97.135 -79.303 45.961
64949 97.157 -79.304 45.939
64950 97.179 -79.306 45.919
64951 97.197 -79.307 45.898
64952 97.205 -79.313 45.875
64953 97.171 -79.351 45.854
64954 97.180 -79.355 45.832
64955 97.215 -79.345 45.809
64956 97.249 -79.335 45.786
64957 97.282 -79.327 45.764
64958 97.286 -79.340 45.744
64959 97.290 -79.352 45.724
64960 97.294 -79.365 45.705
64961 97.298 -79.378 45.685
64962 97.302 -79.391 45.666
64963 97.298 -79.401 45.647
64964 97.291 -79.411 45.629
64965 97.284 -79.420 45.612
64966 97.277 -79.430 45.594
64967 97.279 -79.423 45.576
64968 97.284 -79.410 45.560
64969 97.290 -79.398 45.542
64970 97.295 -79.385 45.524
64971 97.294 -79.390 45.507
64972 97.293 -79.395 45.489
64973 97.276 -79.422 45.474
64974 97.282 -79.430 45.457
64975 97.289 -79.438 45.440
64976 97.296 -79.445 45.423
64977 97.303 -79.453 45.405
64978 97.310 -79.460 45.389
64979 97.317 -79.468 45.373
64980 97.324 -79.476 45.357
64981 97.331 -79.483 45.341
64982 97.346 -79.495 45.326
64983 97.365 -79.511 45.311
64984 97.384 -79.526 45.295
64985 97.403 -79.541 45.280
64986 97.423 -79.557 45.265
64987 97.442 -79.572 45.250
64988 97.461 -79.587 45.235
64989 97.480 -79.602 45.220
64990 97.500 -79.618 45.205
64991 97.519 -79.633 45.191
64992 97.538 -79.648 45.176
64993 97.557 -79.664 45.163
64994 97.576 -79.679 45.149
64995 97.595 -79.694 45.135
64996 97.614 -79.710 45.121
64997 97.634 -79.725 45.107
65028 98.228 -80.199 44.732
65029 98.247 -80.214 44.724
65030 98.266 -80.229 44.714
65031 98.286 -80.244 44.706
65032 98.305 -80.259 44.697
65033 98.324 -80.275 44.689
65034 98.343 -80.290 44.681
65035 98.362 -80.306 44.673
65036 98.381 -80.321 44.666
65037 98.401 -80.336 44.659
65038 98.420 -80.351 44.653
65039 98.439 -80.366 44.646
65040 98.458 -80.382 44.638
65041 98.477 -80.397 44.631
65042 98.496 -80.413 44.625
65043 98.519 -80.419 44.618
65044 98.544 -80.422 44.611
65045 98.569 -80.424 44.604
65046 98.593 -80.427 44.597
65047 98.619 -80.429 44.588
65048 98.643 -80.431 44.579
65049 98.668 -80.435 44.572
65050 98.693 -80.437 44.565
65051 98.717 -80.440 44.558
65052 98.742 -80.442 44.551
65053 98.768 -80.447 44.544
65054 98.793 -80.450 44.538
65055 98.817 -80.459 44.531
65056 98.841 -80.469 44.525
65057 98.864 -80.480 44.519
65058 98.887 -80.491 44.514
65059 98.911 -80.501 44.508
65060 98.934 -80.512 44.503
65061 98.957 -80.522 44.497
65062 98.981 -80.532 44.491
65388 93.385 -79.106 53.428
65389 93.395 -79.104 53.413
65390 93.404 -79.102 53.399
65391 93.414 -79.100 53.385
65392 93.424 -79.096 53.371
65393 93.433 -79.094 53.356
65394 93.443 -79.090 53.342
65395 93.454 -79.080 53.329
65396 93.466 -79.070 53.316
65397 93.476 -79.062 53.305
65398 93.484 -79.055 53.294
65399 93.492 -79.047 53.285
65400 93.501 -79.040 53.277
65401 93.509 -79.034 53.270
65402 93.517 -79.027 53.266
65403 93.525 -79.022 53.263
65404 93.531 -79.027 53.263
65405 93.536 -79.032 53.264
65406 93.542 -79.037 53.268
65407 93.549 -79.040 53.270
65408 93.564 -79.039 53.271
65409 93.583 -79.034 53.268
65410 93.602 -79.028 53.262
65411 93.621 -79.024 53.254
65412 93.640 -79.018 53.244
65413 93.656 -79.015 53.235
65414 93.671 -79.012 53.223
65415 93.684 -79.010 53.211
65416 93.698 -79.008 53.199
65417 93.712 -79.006 53.187
65418 93.726 -79.003 53.176
65419 93.740 -79.001 53.164
65420 93.754 -78.999 53.153
65421 93.768 -78.996 53.141
65422 93.775 -78.993 53.128
65423 93.783 -78.990 53.114
65424 93.791 -78.987 53.101
65425 93.799 -78.985 53.087
65426 93.807 -78.982 53.072
65427 93.815 -78.979 53.058
65428 93.823 -78.977 53.046
65429 93.831 -78.974 53.035
65430 93.839 -78.971 53.024
65431 93.847 -78.968 53.013
65432 93.855 -78.965 53.001
65433 93.863 -78.962 52.989
65434 93.871 -78.961 52.976
65435 93.873 -78.964 52.962
65436 93.874 -78.966 52.947
65437 93.875 -78.968 52.931
65438 93.876 -78.971 52.917
65439 93.877 -78.974 52.902
65440 93.878 -78.976 52.884
65441 93.881 -78.978 52.866
65442 93.884 -78.981 52.848
65443 93.888 -78.985 52.830
65444 93.892 -78.993 52.810
65445 93.892 -78.994 52.791
65446 93.890 -78.991 52.771
65447 93.889 -78.988 52.751
65448 93.894 -78.987 52.731
65449 93.920 -78.989 52.710
65450 93.938 -78.994 52.689
65451 93.956 -78.999 52.668
65452 93.973 -79.004 52.647
65453 93.991 -79.009 52.624
65454 94.009 -79.015 52.598
65455 94.011 -79.009 52.571
65456 94.013 -79.002 52.542
65457 94.015 -78.997 52.514
65458 94.017 -78.991 52.484
65459 94.019 -78.985 52.454
65460 94.022 -78.979 52.424
65461 94.023 -78.973 52.392
65462 94.027 -78.965 52.363
65463 94.037 -78.955 52.335
65464 94.045 -78.944 52.310
65465 94.053 -78.940 52.286
65466 94.060 -78.934 52.263
65467 94.068 -78.932 52.241
65468 94.078 -78.930 52.216
65469 94.087 -78.928 52.191
65470 94.096 -78.927 52.168
65471 94.106 -78.924 52.145
65472 94.122 -78.915 52.123
65473 94.139 -78.905 52.101
65474 94.155 -78.895 52.080
65475 94.172 -78.885 52.056
65476 94.188 -78.875 52.033
65477 94.205 -78.866 52.010
65478 94.221 -78.856 51.986
65479 94.239 -78.843 51.962
65480 94.260 -78.811 51.935
65481 94.280 -78.788 51.905
65482 94.296 -78.785 51.871
65483 94.312 -78.782 51.836
65484 94.328 -78.780 51.801
65485 94.343 -78.780 51.765
65486 94.359 -78.781 51.731
65487 94.374 -78.776 51.699
65488 94.386 -78.782 51.665
65489 94.397 -78.791 51.629
65490 94.409 -78.805 51.592
65491 94.426 -78.813 51.557
65492 94.445 -78.817 51.521
65493 94.463 -78.821 51.487
65494 94.482 -78.825 51.455
65495 94.500 -78.829 51.424
65496 94.519 -78.833 51.393
65497 94.538 -78.837 51.361
65498 94.556 -78.841 51.331
65499 94.575 -78.846 51.298
65500 94.594 -78.849 51.264
65501 94.613 -78.855 51.228
65502 94.632 -78.859 51.190
65503 94.648 -78.861 51.155
65504 94.663 -78.862 51.121
65505 94.678 -78.863 51.089
65506 94.691 -78.862 51.054
65507 94.702 -78.855 51.017
65508 94.712 -78.847 50.979
65509 94.723 -78.839 50.943
65510 94.734 -78.832 50.907
65511 94.744 -78.824 50.869
65512 94.753 -78.818 50.827
65513 94.757 -78.818 50.786
65514 94.761 -78.818 50.743
65515 94.765 -78.817 50.699
65516 94.769 -78.816 50.656
65517 94.773 -78.815 50.612
65518 94.777 -78.814 50.569
65519 94.781 -78.814 50.525
65520 94.796 -78.816 50.487
65521 94.820 -78.818 50.451
65522 94.843 -78.821 50.417
65523 94.866 -78.824 50.382
65524 94.890 -78.827 50.345
65525 94.886 -78.827 50.307
65526 94.880 -78.827 50.266
65527 94.873 -78.826 50.223
65528 94.864 -78.826 50.181
65529 94.857 -78.825 50.139
65530 94.849 -78.825 50.097
65531 94.841 -78.824 50.058
65532 94.833 -78.824 50.020
65533 94.825 -78.824 49.983
65534 94.817 -78.823 49.946
65535 94.816 -78.824 49.908
65536 94.827 -78.842 49.867
65537 94.835 -78.870 49.829
65538 94.843 -78.898 49.790
65539 94.850 -78.926 49.750
65540 94.858 -78.954 49.711
65541 94.866 -78.982 49.674
65542 94.874 -79.010 49.638
65543 94.898 -79.026 49.604
65544 94.937 -79.031 49.569
65545 94.968 -79.037 49.534
65546 94.985 -79.046 49.501
65547 95.001 -79.056 49.469
65548 95.018 -79.065 49.436
65549 95.034 -79.075 49.404
65550 95.051 -79.084 49.370
65551 95.067 -79.093 49.333
65552 95.084 -79.103 49.296
65553 95.100 -79.117 49.258
65554 95.120 -79.128 49.219
65555 95.142 -79.137 49.178
65556 95.168 -79.141 49.137
65557 95.193 -79.146 49.095
65558 95.219 -79.150 49.052
65559 95.245 -79.154 49.009
65560 95.270 -79.159 48.965
65561 95.296 -79.163 48.922
65562 95.321 -79.167 48.878
65563 95.329 -79.173 48.835
65564 95.332 -79.179 48.791
65565 95.344 -79.178 48.747
65566 95.362 -79.172 48.703
65567 95.377 -79.182 48.660
65568 95.389 -79.198 48.616
65569 95.402 -79.214 48.571
65570 95.415 -79.229 48.526
65571 95.428 -79.245 48.483
65572 95.441 -79.261 48.441
65573 95.454 -79.276 48.398
65574 95.470 -79.291 48.356
65575 95.509 -79.295 48.314
65576 95.537 -79.288 48.272
65577 95.559 -79.275 48.230
65578 95.582 -79.261 48.189
65579 95.604 -79.248 48.147
65580 95.627 -79.234 48.105
65581 95.649 -79.222 48.065
65582 95.663 -79.214 48.025
65583 95.673 -79.209 47.986
65584 95.683 -79.203 47.947
65585 95.693 -79.198 47.907
65586 95.703 -79.192 47.868
65587 95.716 -79.188 47.829
65588 95.750 -79.198 47.790
65589 95.783 -79.207 47.751
65590 95.817 -79.216 47.712
65591 95.850 -79.225 47.673
65592 95.875 -79.243 47.635
65593 95.884 -79.271 47.596
65594 95.883 -79.262 47.558
65595 95.878 -79.242 47.520
65596 95.873 -79.222 47.484
65597 95.861 -79.185 47.449
65598 95.843 -79.137 47.413
65599 95.826 -79.089 47.377
65600 95.808 -79.040 47.340
65601 95.848 -79.058 47.303
65602 95.898 -79.082 47.266
65603 95.948 -79.106 47.229
65604 95.997 -79.131 47.192
65605 96.046 -79.156 47.157
65606 96.096 -79.180 47.120
65607 96.145 -79.204 47.084
65608 96.194 -79.229 47.049
65609 96.244 -79.253 47.014
65610 96.294 -79.278 46.980
65611 96.325 -79.235 46.947
65612 96.356 -79.283 46.915
65613 96.381 -79.298 46.882
65614 96.402 -79.309 46.850
65615 96.423 -79.319 46.819
65616 96.444 -79.329 46.789
65617 96.465 -79.340 46.758
65618 96.485 -79.350 46.728
65619 96.506 -79.361 46.699
65620 96.527 -79.371 46.669
65621 96.548 -79.382 46.639
65622 96.569 -79.392 46.609
65623 96.581 -79.394 46.580
65624 96.590 -79.391 46.552
65625 96.599 -79.390 46.524
65626 96.608 -79.388 46.496
65627 96.617 -79.385 46.468
65628 96.639 -79.388 46.440
65629 96.662 -79.391 46.413
65630 96.685 -79.393 46.387
65631 96.708 -79.395 46.362
65632 96.730 -79.398 46.337
65633 96.753 -79.400 46.313
65634 96.773 -79.391 46.288
65635 96.792 -79.369 46.262
65636 96.811 -79.348 46.237
65637 96.829 -79.327 46.215
65638 96.848 -79.306 46.193
65639 96.867 -79.285 46.172
65640 96.885 -79.264 46.150
65641 96.904 -79.243 46.129
65642 96.923 -79.222 46.107
65643 96.950 -79.216 46.084
65644 97.002 -79.247 46.061
65645 97.060 -79.279 46.040
65646 97.087 -79.289 46.019
65647 97.102 -79.290 45.998
65648 97.117 -79.291 45.976
65649 97.140 -79.292 45.954
65650 97.159 -79.294 45.932
65651 97.173 -79.294 45.911
65652 97.188 -79.294 45.890
65653 97.197 -79.298 45.869
65654 97.163 -79.338 45.848
65655 97.164 -79.336 45.826
65656 97.183 -79.327 45.804
65657 97.218 -79.317 45.782
65658 97.252 -79.307 45.761
65659 97.271 -79.310 45.742
65660 97.275 -79.322 45.723
65661 97.279 -79.335 45.703
65662 97.283 -79.347 45.684
65663 97.287 -79.360 45.665
65664 97.291 -79.373 45.647
65665 97.295 -79.385 45.629
65666 97.299 -79.398 45.611
65667 97.294 -79.409 45.594
65668 97.287 -79.418 45.577
65669 97.280 -79.428 45.560
65670 97.283 -79.419 45.542
65671 97.288 -79.407 45.524
65672 97.294 -79.394 45.507
65673 97.300 -79.381 45.490
65674 97.302 -79.376 45.474
65675 97.301 -79.381 45.458
65676 97.289 -79.403 45.441
65677 97.296 -79.410 45.424
65678 97.303 -79.418 45.407
65679 97.310 -79.426 45.390
65680 97.318 -79.433 45.374
65681 97.325 -79.441 45.359
65682 97.332 -79.449 45.343
65683 97.352 -79.464 45.328
65684 97.371 -79.479 45.313
65685 97.390 -79.495 45.298
65686 97.409 -79.510 45.283
65687 97.428 -79.526 45.268
65688 97.447 -79.541 45.253
65689 97.466 -79.556 45.238
65690 97.486 -79.571 45.224
65691 97.505 -79.586 45.209
65692 97.524 -79.602 45.195
65693 97.543 -79.617 45.181
65694 97.563 -79.632 45.167
65695 97.582 -79.648 45.153
65696 97.601 -79.663 45.139
65697 97.620 -79.678 45.126
65698 97.639 -79.694 45.112
65699 97.658 -79.709 45.099
65700 97.677 -79.724 45.085
65701 97.697 -79.739 45.073
65702 97.716 -79.755 45.060
65703 97.735 -79.770 45.046
65704 97.754 -79.786 45.034
65705 97.773 -79.801 45.021
65724 98.138 -80.091 44.788
65725 98.157 -80.106 44.779
65726 98.176 -80.121 44.769
65727 98.195 -80.137 44.759
65728 98.214 -80.152 44.750
65729 98.233 -80.168 44.741
65730 98.252 -80.183 44.732
65731 98.272 -80.198 44.723
65732 98.291 -80.213 44.715
65733 98.310 -80.228 44.707
65734 98.329 -80.243 44.698
65735 98.348 -80.259 44.690
65736 98.367 -80.275 44.683
65737 98.387 -80.290 44.676
65738 98.406 -80.305 44.669
65739 98.425 -80.320 44.662
65740 98.444 -80.335 44.655
65741 98.463 -80.350 44.648
65742 98.483 -80.366 44.641
65743 98.502 -80.381 44.635
65744 98.525 -80.388 44.628
65745 98.550 -80.391 44.621
65746 98.574 -80.393 44.614
65747 98.599 -80.395 44.607
65748 98.624 -80.398 44.598
65749 98.649 -80.400 44.590
65750 98.673 -80.403 44.582
65751 98.698 -80.406 44.575
65752 98.723 -80.408 44.568
65753 98.747 -80.411 44.561
65754 98.773 -80.415 44.554
65755 98.798 -80.421 44.547
65756 98.821 -80.431 44.540
65757 98.845 -80.442 44.534
65758 98.868 -80.453 44.528
65759 98.891 -80.463 44.523
65760 98.915 -80.474 44.518
65761 98.938 -80.485 44.513
65762 98.961 -80.495 44.507
65763 98.985 -80.505 44.501
65764 99.008 -80.516 44.496
66091 93.405 -79.091 53.422
66092 93.414 -79.088 53.408
66093 93.424 -79.086 53.393
66094 93.433 -79.084 53.379
66095 93.443 -79.081 53.366
66096 93.453 -79.076 53.352
66097 93.465 -79.066 53.340
66098 93.476 -79.056 53.328
66099 93.485 -79.048 53.317
66100 93.493 -79.041 53.307
66101 93.502 -79.034 53.298
66102 93.510 -79.027 53.290
66103 93.518 -79.020 53.285
66104 93.527 -79.013 53.280
66105 93.535 -79.006 53.279
66106 93.541 -79.010 53.279
66107 93.547 -79.015 53.283
66108 93.552 -79.020 53.286
66109 93.561 -79.022 53.287
66110 93.580 -79.017 53.285
66111 93.599 -79.012 53.279
66112 93.618 -79.007 53.272
66113 93.637 -79.002 53.263
66114 93.652 -78.999 53.253
66115 93.666 -78.997 53.243
66116 93.680 -78.995 53.232
66117 93.694 -78.993 53.222
66118 93.707 -78.990 53.211
66119 93.721 -78.988 53.199
66120 93.735 -78.986 53.188
66121 93.751 -78.983 53.178
66122 93.763 -78.981 53.166
66123 93.770 -78.981 53.154
66124 93.777 -78.981 53.141
66125 93.785 -78.978 53.128
66126 93.793 -78.974 53.114
66127 93.801 -78.971 53.099
66128 93.810 -78.968 53.085
66129 93.818 -78.965 53.074
66130 93.826 -78.962 53.064
66131 93.834 -78.959 53.053
66132 93.843 -78.956 53.042
66133 93.851 -78.952 53.031
66134 93.859 -78.949 53.019
66135 93.867 -78.946 53.006
66136 93.870 -78.948 52.992
66137 93.871 -78.950 52.978
66138 93.872 -78.952 52.962
66139 93.873 -78.955 52.947
66140 93.874 -78.958 52.932
66141 93.875 -78.960 52.915
66142 93.876 -78.962 52.896
66143 93.880 -78.965 52.877
66144 93.883 -78.970 52.858
66145 93.885 -78.974 52.837
66146 93.883 -78.971 52.816
66147 93.882 -78.968 52.794
66148 93.880 -78.965 52.772
66149 93.891 -78.965 52.749
66150 93.908 -78.969 52.725
66151 93.926 -78.974 52.701
66152 93.944 -78.980 52.679
66153 93.962 -78.984 52.656
66154 93.979 -78.990 52.633
66155 93.997 -78.995 52.607
66156 94.011 -78.997 52.580
66157 94.013 -78.991 52.553
66158 94.015 -78.985 52.525
66159 94.017 -78.979 52.496
66160 94.019 -78.973 52.468
66161 94.021 -78.967 52.440
66162 94.029 -78.957 52.411
66163 94.038 -78.946 52.384
66164 94.047 -78.937 52.358
66165 94.056 -78.926 52.333
66166 94.064 -78.918 52.309
66167 94.072 -78.913 52.285
66168 94.079 -78.908 52.263
66169 94.086 -78.903 52.237
66170 94.094 -78.899 52.213
66171 94.103 -78.897 52.189
66172 94.115 -78.893 52.166
66173 94.131 -78.883 52.142
66174 94.147 -78.873 52.119
66175 94.164 -78.863 52.097
66176 94.181 -78.854 52.072
66177 94.197 -78.844 52.048
66178 94.214 -78.834 52.025
66179 94.230 -78.824 52.002
66180 94.247 -78.814 51.977
66181 94.264 -78.799 51.950
66182 94.283 -78.780 51.920
66183 94.299 -78.777 51.886
66184 94.315 -78.778 51.851
66185 94.331 -78.779 51.816
66186 94.346 -78.780 51.781
66187 94.362 -78.781 51.749
66188 94.378 -78.782 51.717
66189 94.393 -78.771 51.683
66190 94.404 -78.780 51.647
66191 94.419 -78.788 51.610
66192 94.437 -78.793 51.573
66193 94.455 -78.796 51.538
66194 94.472 -78.801 51.503
66195 94.490 -78.805 51.470
66196 94.507 -78.809 51.440
66197 94.526 -78.813 51.408
66198 94.544 -78.818 51.376
66199 94.563 -78.821 51.345
66200 94.581 -78.826 51.313
66201 94.600 -78.830 51.280
66202 94.619 -78.834 51.245
66203 94.634 -78.835 51.207
66204 94.648 -78.836 51.171
66205 94.663 -78.837 51.137
66206 94.678 -78.838 51.105
66207 94.692 -78.839 51.073
66208 94.703 -78.833 51.037
66209 94.714 -78.825 50.999
66210 94.721 -78.821 50.964
66211 94.725 -78.821 50.927
66212 94.729 -78.821 50.887
66213 94.733 -78.820 50.846
66214 94.737 -78.819 50.803
66215 94.741 -78.818 50.759
66216 94.745 -78.818 50.714
66217 94.749 -78.817 50.672
66218 94.753 -78.817 50.629
66219 94.757 -78.816 50.585
66220 94.766 -78.816 50.541
66221 94.790 -78.819 50.503
66222 94.813 -78.821 50.467
66223 94.836 -78.824 50.433
66224 94.860 -78.827 50.398
66225 94.889 -78.821 50.363
66226 94.893 -78.830 50.327
66227 94.885 -78.829 50.287
66228 94.878 -78.829 50.245
66229 94.870 -78.828 50.203
66230 94.861 -78.828 50.162
66231 94.853 -78.827 50.119
66232 94.845 -78.827 50.078
66233 94.838 -78.827 50.041
66234 94.830 -78.826 50.005
66235 94.822 -78.825 49.968
66236 94.817 -78.825 49.929
66237 94.833 -78.828 49.887
66238 94.845 -78.845 49.847
66239 94.853 -78.873 49.807
66240 94.861 -78.901 49.766
66241 94.868 -78.929 49.727
66242 94.877 -78.957 49.689
66243 94.885 -78.985 49.653
66244 94.900 -79.007 49.618
66245 94.939 -79.012 49.583
66246 94.971 -79.018 49.549
66247 94.987 -79.028 49.516
66248 95.003 -79.037 49.484
66249 95.020 -79.046 49.452
66250 95.036 -79.056 49.420
66251 95.052 -79.065 49.385
66252 95.069 -79.075 49.348
66253 95.085 -79.084 49.310
66254 95.103 -79.094 49.271
66255 95.128 -79.100 49.230
66256 95.154 -79.104 49.188
66257 95.179 -79.108 49.146
66258 95.205 -79.112 49.104
66259 95.231 -79.117 49.061
66260 95.256 -79.122 49.018
66261 95.282 -79.125 48.975
66262 95.307 -79.130 48.931
66263 95.333 -79.134 48.888
66264 95.345 -79.140 48.845
66265 95.347 -79.146 48.801
66266 95.350 -79.153 48.757
66267 95.365 -79.154 48.713
66268 95.378 -79.169 48.670
66269 95.390 -79.185 48.625
66270 95.403 -79.201 48.580
66271 95.416 -79.217 48.535
66272 95.429 -79.232 48.492
66273 95.442 -79.248 48.449
66274 95.455 -79.264 48.406
66275 95.468 -79.279 48.364
66276 95.506 -79.285 48.322
66277 95.534 -79.276 48.280
66278 95.556 -79.263 48.237
66279 95.579 -79.250 48.195
66280 95.601 -79.237 48.153
66281 95.624 -79.224 48.112
66282 95.646 -79.210 48.071
66283 95.665 -79.200 48.031
66284 95.675 -79.194 47.991
66285 95.685 -79.188 47.952
66286 95.696 -79.183 47.912
66287 95.706 -79.178 47.873
66288 95.723 -79.176 47.834
66289 95.756 -79.185 47.794
66290 95.790 -79.195 47.755
66291 95.824 -79.204 47.716
66292 95.857 -79.213 47.676
66293 95.888 -79.225 47.637
66294 95.895 -79.247 47.599
66295 95.891 -79.227 47.560
66296 95.886 -79.207 47.522
66297 95.881 -79.187 47.486
66298 95.877 -79.167 47.450
66299 95.863 -79.130 47.414
66300 95.849 -79.093 47.377
66301 95.836 -79.057 47.340
66302 95.867 -79.069 47.303
66303 95.917 -79.103 47.265
66304 95.969 -79.136 47.228
66305 96.019 -79.169 47.191
66306 96.068 -79.193 47.155
66307 96.118 -79.217 47.119
66308 96.168 -79.241 47.082
66309 96.217 -79.266 47.046
66310 96.252 -79.239 47.012
66311 96.281 -79.188 46.978
66312 96.310 -79.137 46.945
66313 96.341 -79.176 46.913
66314 96.372 -79.235 46.881
66315 96.403 -79.285 46.849
66316 96.431 -79.303 46.818
66317 96.457 -79.318 46.788
66318 96.478 -79.328 46.758
66319 96.499 -79.339 46.728
66320 96.520 -79.349 46.698
66321 96.540 -79.360 46.667
66322 96.560 -79.369 46.637
66323 96.570 -79.367 46.606
66324 96.579 -79.365 46.577
66325 96.588 -79.363 46.549
66326 96.596 -79.361 46.521
66327 96.605 -79.359 46.492
66328 96.624 -79.360 46.464
66329 96.647 -79.363 46.436
66330 96.670 -79.365 46.408
66331 96.692 -79.368 46.382
66332 96.715 -79.370 46.356
66333 96.738 -79.372 46.331
66334 96.761 -79.375 46.306
66335 96.782 -79.367 46.280
66336 96.800 -79.346 46.255
66337 96.819 -79.325 46.230
66338 96.837 -79.303 46.207
66339 96.856 -79.283 46.186
66340 96.875 -79.262 46.164
66341 96.893 -79.241 46.142
66342 96.912 -79.219 46.121
66343 96.934 -79.203 46.099
66344 96.971 -79.214 46.077
66345 97.004 -79.240 46.055
66346 97.062 -79.272 46.034
66347 97.083 -79.278 46.014
66348 97.100 -79.278 45.992
66349 97.120 -79.280 45.970
66350 97.135 -79.280 45.948
66351 97.150 -79.281 45.925
66352 97.165 -79.281 45.904
66353 97.179 -79.281 45.883
66354 97.190 -79.284 45.864
66355 97.156 -79.323 45.842
66356 97.164 -79.315 45.821
66357 97.166 -79.308 45.800
66358 97.186 -79.300 45.779
66359 97.221 -79.289 45.759
66360 97.255 -79.279 45.740
66361 97.260 -79.291 45.721
66362 97.264 -79.304 45.702
66363 97.268 -79.317 45.683
66364 97.272 -79.330 45.665
66365 97.276 -79.343 45.646
66366 97.280 -79.355 45.629
66367 97.284 -79.368 45.612
66368 97.288 -79.381 45.595
66369 97.292 -79.394 45.578
66370 97.296 -79.407 45.560
66371 97.290 -79.416 45.543
66372 97.283 -79.425 45.525
66373 97.287 -79.414 45.508
66374 97.293 -79.402 45.491
66375 97.298 -79.389 45.476
66376 97.304 -79.376 45.460
66377 97.309 -79.363 45.443
66378 97.309 -79.367 45.426
66379 97.304 -79.384 45.410
66380 97.311 -79.391 45.393
66381 97.318 -79.399 45.377
66382 97.325 -79.407 45.362
66383 97.337 -79.418 45.347
66384 97.357 -79.433 45.331
66385 97.376 -79.448 45.316
66386 97.395 -79.463 45.301
66387 97.415 -79.479 45.286
66388 97.434 -79.494 45.272
66389 97.453 -79.510 45.258
66390 97.472 -79.525 45.243
66391 97.491 -79.540 45.228
66392 97.510 -79.555 45.214
66393 97.529 -79.570 45.200
66394 97.548 -79.586 45.186
66395 97.568 -79.601 45.172
66396 97.587 -79.617 45.158
66397 97.606 -79.632 45.145
66398 97.625 -79.647 45.132
66399 97.645 -79.662 45.119
66400 97.664 -79.678 45.105
66401 97.683 -79.693 45.092
66402 97.702 -79.708 45.080
66403 97.721 -79.723 45.067
66404 97.740 -79.739 45.054
66405 97.759 -79.754 45.041
66406 97.779 -79.770 45.028
66407 97.798 -79.785 45.016
66408 97.817 -79.800 45.003
66409 97.836 -79.815 44.990
66410 97.855 -79.830 44.977
66411 97.874 -79.845 44.963
66412 97.893 -79.861 44.950
66413 97.913 -79.877 44.936
66414 97.932 -79.892 44.923
66415 97.951 -79.907 44.910
66416 97.970 -79.922 44.898
66417 97.990 -79.937 44.886
66418 98.009 -79.952 44.874
66419 98.028 -79.968 44.862
66420 98.047 -79.983 44.850
66421 98.066 -79.999 44.839
66422 98.085 -80.014 44.828
66423 98.104 -80.029 44.818
66424 98.124 -80.044 44.807
66425 98.143 -80.060 44.797
66426 98.162 -80.075 44.788
66427 98.181 -80.090 44.778
66428 98.201 -80.106 44.768
66429 98.220 -80.121 44.759
66430 98.239 -80.136 44.750
66431 98.258 -80.152 44.742
66432 98.277 -80.167 44.733
66433 98.296 -80.182 44.725
66434 98.315 -80.197 44.716
66435 98.335 -80.212 44.708
66436 98.354 -80.228 44.700
66437 98.373 -80.243 44.692
66438 98.392 -80.259 44.685
66439 98.411 -80.274 44.678
66440 98.430 -80.289 44.672
66441 98.449 -80.304 44.665
66442 98.469 -80.319 44.658
66443 98.488 -80.334 44.652
66444 98.507 -80.350 44.645
66445 98.530 -80.356 44.639
66446 98.555 -80.359 44.633
66447 98.580 -80.362 44.625
66448 98.604 -80.364 44.617
66449 98.629 -80.366 44.609
66450 98.654 -80.369 44.600
66451 98.678 -80.372 44.593
66452 98.704 -80.375 44.585
66453 98.728 -80.377 44.578
66454 98.753 -80.380 44.571
66455 98.779 -80.384 44.564
66456 98.802 -80.394 44.557
66457 98.826 -80.404 44.550
66458 98.849 -80.415 44.544
66459 98.872 -80.425 44.539
66460 98.896 -80.436 44.533
66461 98.919 -80.447 44.527
66462 98.942 -80.457 44.522
66463 98.966 -80.468 44.517
66464 98.989 -80.478 44.510
66465 99.012 -80.488 44.505
66466 99.035 -80.499 44.501
66794 93.424 -79.075 53.416
66795 93.434 -79.072 53.402
66796 93.443 -79.070 53.389
66797 93.453 -79.068 53.376
66798 93.464 -79.062 53.363
66799 93.475 -79.052 53.351
66800 93.486 -79.041 53.340
66801 93.495 -79.034 53.329
66802 93.503 -79.028 53.320
66803 93.511 -79.021 53.312
66804 93.519 -79.014 53.304
66805 93.528 -79.007 53.299
66806 93.536 -78.999 53.296
66807 93.545 -78.993 53.294
66808 93.551 -78.993 53.295
66809 93.557 -78.998 53.298
66810 93.562 -79.002 53.300
66811 93.577 -79.001 53.299
66812 93.596 -78.996 53.295
66813 93.615 -78.991 53.289
66814 93.633 -78.986 53.282
66815 93.647 -78.984 53.273
66816 93.661 -78.981 53.263
66817 93.675 -78.979 53.253
66818 93.689 -78.977 53.245
66819 93.703 -78.974 53.234
66820 93.717 -78.972 53.222
66821 93.734 -78.970 53.212
66822 93.750 -78.968 53.202
66823 93.759 -78.967 53.191
66824 93.766 -78.967 53.179
66825 93.773 -78.967 53.166
66826 93.780 -78.967 53.153
66827 93.786 -78.967 53.139
66828 93.794 -78.964 53.124
66829 93.802 -78.961 53.111
66830 93.811 -78.958 53.100
66831 93.819 -78.954 53.090
66832 93.827 -78.951 53.080
66833 93.836 -78.948 53.069
66834 93.844 -78.945 53.057
66835 93.852 -78.942 53.045
66836 93.862 -78.933 53.032
66837 93.865 -78.933 53.018
66838 93.866 -78.936 53.004
66839 93.868 -78.937 52.989
66840 93.869 -78.939 52.974
66841 93.871 -78.941 52.959
66842 93.872 -78.943 52.942
66843 93.873 -78.946 52.923
66844 93.875 -78.949 52.904
66845 93.877 -78.953 52.884
66846 93.876 -78.950 52.861
66847 93.874 -78.947 52.838
66848 93.872 -78.944 52.815
66849 93.869 -78.940 52.791
66850 93.880 -78.943 52.765
66851 93.897 -78.949 52.740
66852 93.914 -78.955 52.715
66853 93.932 -78.960 52.690
66854 93.949 -78.965 52.665
66855 93.967 -78.970 52.640
66856 93.985 -78.975 52.614
66857 94.003 -78.981 52.588
66858 94.013 -78.980 52.561
66859 94.015 -78.974 52.534
66860 94.017 -78.968 52.506
66861 94.022 -78.960 52.479
66862 94.031 -78.949 52.454
66863 94.040 -78.939 52.428
66864 94.049 -78.928 52.402
66865 94.058 -78.918 52.376
66866 94.067 -78.907 52.353
66867 94.076 -78.896 52.329
66868 94.083 -78.892 52.306
66869 94.091 -78.886 52.283
66870 94.098 -78.882 52.257
66871 94.105 -78.877 52.232
66872 94.113 -78.872 52.209
66873 94.125 -78.866 52.185
66874 94.142 -78.856 52.161
66875 94.158 -78.846 52.137
66876 94.175 -78.836 52.113
66877 94.191 -78.826 52.087
66878 94.207 -78.816 52.062
66879 94.224 -78.806 52.039
66880 94.241 -78.796 52.015
66881 94.257 -78.786 51.990
66882 94.273 -78.777 51.963
66883 94.287 -78.774 51.933
66884 94.302 -78.776 51.899
66885 94.318 -78.778 51.865
66886 94.334 -78.779 51.829
66887 94.349 -78.780 51.796
66888 94.365 -78.781 51.764
66889 94.381 -78.782 51.733
66890 94.397 -78.774 51.700
66891 94.413 -78.765 51.664
66892 94.430 -78.770 51.626
66893 94.448 -78.774 51.589
66894 94.466 -78.778 51.553
66895 94.483 -78.783 51.518
66896 94.500 -78.787 51.485
66897 94.518 -78.791 51.454
66898 94.535 -78.796 51.422
66899 94.553 -78.800 51.390
66900 94.570 -78.805 51.359
66901 94.588 -78.808 51.327
66902 94.608 -78.814 51.295
66903 94.626 -78.816 51.260
66904 94.640 -78.818 51.223
66905 94.655 -78.819 51.186
66906 94.671 -78.820 51.151
66907 94.685 -78.821 51.119
66908 94.700 -78.822 51.088
66909 94.698 -78.826 51.054
66910 94.697 -78.823 51.018
66911 94.701 -78.823 50.983
66912 94.705 -78.822 50.944
66913 94.709 -78.821 50.904
66914 94.713 -78.821 50.861
66915 94.717 -78.820 50.817
66916 94.721 -78.819 50.773
66917 94.725 -78.819 50.729
66918 94.729 -78.818 50.687
66919 94.733 -78.818 50.643
66920 94.737 -78.817 50.599
66921 94.760 -78.819 50.558
66922 94.783 -78.822 50.518
66923 94.806 -78.825 50.481
66924 94.833 -78.822 50.447
66925 94.864 -78.814 50.413
66926 94.887 -78.809 50.379
66927 94.892 -78.818 50.344
66928 94.890 -78.832 50.305
66929 94.882 -78.831 50.264
66930 94.874 -78.831 50.222
66931 94.866 -78.830 50.182
66932 94.858 -78.830 50.139
66933 94.850 -78.830 50.097
66934 94.842 -78.829 50.058
66935 94.834 -78.828 50.020
66936 94.826 -78.828 49.983
66937 94.819 -78.827 49.944
66938 94.834 -78.830 49.902
66939 94.851 -78.833 49.863
66940 94.863 -78.848 49.822
66941 94.872 -78.876 49.779
66942 94.879 -78.904 49.741
66943 94.887 -78.932 49.703
66944 94.895 -78.961 49.666
66945 94.903 -78.989 49.631
66946 94.942 -78.993 49.596
66947 94.973 -78.999 49.561
66948 94.990 -79.009 49.528
66949 95.006 -79.018 49.498
66950 95.023 -79.028 49.466
66951 95.039 -79.037 49.434
66952 95.055 -79.046 49.398
66953 95.071 -79.056 49.360
66954 95.089 -79.065 49.321
66955 95.119 -79.071 49.281
66956 95.143 -79.074 49.240
66957 95.168 -79.076 49.198
66958 95.192 -79.078 49.155
66959 95.217 -79.081 49.112
66960 95.242 -79.084 49.070
66961 95.268 -79.088 49.027
66962 95.293 -79.093 48.985
66963 95.319 -79.097 48.942
66964 95.344 -79.101 48.898
66965 95.360 -79.106 48.854
66966 95.362 -79.113 48.811
66967 95.365 -79.125 48.768
66968 95.368 -79.141 48.724
66969 95.379 -79.157 48.679
66970 95.391 -79.172 48.633
66971 95.404 -79.188 48.589
66972 95.417 -79.204 48.545
66973 95.430 -79.219 48.501
66974 95.443 -79.235 48.457
66975 95.456 -79.251 48.413
66976 95.469 -79.266 48.372
66977 95.503 -79.273 48.329
66978 95.531 -79.265 48.287
66979 95.553 -79.252 48.245
66980 95.576 -79.238 48.203
66981 95.598 -79.225 48.162
66982 95.621 -79.212 48.120
66983 95.643 -79.199 48.079
66984 95.666 -79.186 48.038
66985 95.678 -79.179 47.998
66986 95.688 -79.174 47.957
66987 95.698 -79.169 47.918
66988 95.708 -79.162 47.879
66989 95.729 -79.164 47.840
66990 95.763 -79.173 47.800
66991 95.797 -79.183 47.761
66992 95.830 -79.192 47.722
66993 95.864 -79.201 47.682
66994 95.898 -79.210 47.641
66995 95.903 -79.211 47.602
66996 95.899 -79.191 47.563
66997 95.894 -79.182 47.525
66998 95.890 -79.173 47.487
66999 95.886 -79.164 47.451
67000 95.891 -79.147 47.414
67001 95.878 -79.110 47.378
67002 95.864 -79.074 47.341
67003 95.884 -79.074 47.303
67004 95.935 -79.108 47.265
67005 95.986 -79.141 47.227
67006 96.037 -79.175 47.190
67007 96.088 -79.209 47.153
67008 96.138 -79.243 47.117
67009 96.182 -79.247 47.080
67010 96.209 -79.194 47.044
67011 96.238 -79.141 47.010
67012 96.266 -79.090 46.977
67013 96.295 -79.039 46.945
67014 96.326 -79.068 46.912
67015 96.357 -79.128 46.880
67016 96.388 -79.187 46.849
67017 96.420 -79.246 46.818
67018 96.450 -79.288 46.788
67019 96.478 -79.304 46.758
67020 96.506 -79.322 46.729
67021 96.533 -79.338 46.698
67022 96.549 -79.343 46.667
67023 96.558 -79.341 46.635
67024 96.567 -79.338 46.605
67025 96.576 -79.337 46.576
67026 96.585 -79.335 46.547
67027 96.594 -79.332 46.517
67028 96.609 -79.332 46.489
67029 96.631 -79.335 46.461
67030 96.655 -79.338 46.432
67031 96.677 -79.340 46.404
67032 96.700 -79.343 46.378
67033 96.723 -79.345 46.352
67034 96.746 -79.347 46.327
67035 96.768 -79.350 46.301
67036 96.790 -79.344 46.274
67037 96.808 -79.322 46.248
67038 96.827 -79.301 46.224
67039 96.846 -79.280 46.201
67040 96.864 -79.259 46.179
67041 96.883 -79.238 46.156
67042 96.901 -79.217 46.135
67043 96.920 -79.196 46.114
67044 96.955 -79.203 46.092
67045 96.992 -79.212 46.071
67046 97.029 -79.223 46.050
67047 97.064 -79.265 46.029
67048 97.082 -79.266 46.010
67049 97.097 -79.266 45.988
67050 97.112 -79.267 45.965
67051 97.126 -79.267 45.943
67052 97.141 -79.267 45.921
67053 97.156 -79.267 45.900
67054 97.171 -79.267 45.880
67055 97.183 -79.269 45.861
67056 97.152 -79.306 45.840
67057 97.165 -79.294 45.819
67058 97.166 -79.287 45.798
67059 97.168 -79.280 45.777
67060 97.190 -79.272 45.758
67061 97.224 -79.261 45.740
67062 97.245 -79.261 45.722
67063 97.249 -79.274 45.703
67064 97.253 -79.287 45.685
67065 97.257 -79.300 45.666
67066 97.261 -79.313 45.647
67067 97.265 -79.325 45.629
67068 97.270 -79.338 45.613
67069 97.274 -79.350 45.597
67070 97.278 -79.363 45.580
67071 97.282 -79.376 45.562
67072 97.286 -79.388 45.544
67073 97.290 -79.401 45.527
67074 97.293 -79.414 45.510
67075 97.286 -79.422 45.494
67076 97.291 -79.410 45.478
67077 97.297 -79.397 45.462
67078 97.303 -79.385 45.446
67079 97.308 -79.372 45.429
67080 97.314 -79.359 45.413
67081 97.317 -79.353 45.397
67082 97.318 -79.364 45.381
67083 97.325 -79.372 45.366
67084 97.343 -79.387 45.351
67085 97.362 -79.402 45.336
67086 97.381 -79.417 45.320
67087 97.400 -79.432 45.305
67088 97.420 -79.447 45.290
67089 97.439 -79.463 45.276
67090 97.458 -79.479 45.262
67091 97.477 -79.494 45.248
67092 97.497 -79.509 45.233
67093 97.516 -79.524 45.219
67094 97.535 -79.539 45.206
67095 97.554 -79.554 45.192
67096 97.573 -79.570 45.178
67097 97.592 -79.585 45.165
67098 97.611 -79.601 45.152
67099 97.631 -79.616 45.139
67100 97.650 -79.631 45.125
67101 97.669 -79.646 45.112
67102 97.688 -79.662 45.100
67103 97.707 -79.677 45.087
67104 97.727 -79.692 45.074
67105 97.746 -79.708 45.061
67106 97.765 -79.723 45.049
67107 97.784 -79.738 45.036
67108 97.803 -79.754 45.024
67109 97.822 -79.769 45.011
67110 97.842 -79.784 44.999
67111 97.861 -79.799 44.986
67112 97.880 -79.814 44.973
67113 97.899 -79.830 44.959
67114 97.918 -79.845 44.945
67115 97.937 -79.861 44.932
67116 97.956 -79.876 44.919
67117 97.975 -79.891 44.907
67118 97.995 -79.906 44.894
67119 98.014 -79.921 44.882
67120 98.033 -79.936 44.870
67121 98.053 -79.952 44.859
67122 98.072 -79.968 44.848
67123 98.091 -79.983 44.837
67124 98.110 -79.998 44.827
67125 98.129 -80.013 44.816
67126 98.148 -80.028 44.806
67127 98.167 -80.044 44.797
67128 98.186 -80.059 44.787
67129 98.206 -80.074 44.778
67130 98.225 -80.090 44.769
67131 98.244 -80.105 44.761
67132 98.263 -80.120 44.752
67133 98.283 -80.135 44.743
67134 98.302 -80.151 44.735
67135 98.321 -80.166 44.727
67136 98.340 -80.181 44.719
67137 98.359 -80.196 44.711
67138 98.378 -80.212 44.703
67139 98.397 -80.227 44.696
67140 98.417 -80.243 44.690
67141 98.436 -80.258 44.683
67142 98.455 -80.273 44.676
67143 98.474 -80.288 44.669
67144 98.493 -80.303 44.662
67145 98.512 -80.319 44.656
67146 98.535 -80.325 44.650
67147 98.560 -80.328 44.644
67148 98.585 -80.330 44.636
67149 98.610 -80.333 44.629
67150 98.635 -80.335 44.621
67151 98.659 -80.338 44.612
67152 98.684 -80.341 44.605
67153 98.709 -80.343 44.598
67154 98.734 -80.346 44.591
67155 98.759 -80.348 44.583
67156 98.783 -80.356 44.576
67157 98.806 -80.366 44.568
67158 98.830 -80.377 44.560
67159 98.853 -80.388 44.554
67160 98.876 -80.398 44.549
67161 98.900 -80.409 44.543
67162 98.923 -80.419 44.537
67163 98.946 -80.430 44.532
67164 98.970 -80.441 44.526
67165 98.993 -80.451 44.520
67166 99.016 -80.461 44.515
67167 99.039 -80.472 44.510
67168 99.063 -80.482 44.506
67496 93.434 -79.062 53.425
67497 93.444 -79.059 53.412
67498 93.453 -79.057 53.399
67499 93.463 -79.054 53.386
67500 93.474 -79.048 53.374
67501 93.485 -79.037 53.363
67502 93.496 -79.028 53.352
67503 93.504 -79.021 53.342
67504 93.512 -79.014 53.333
67505 93.521 -79.007 53.325
67506 93.529 -79.000 53.319
67507 93.538 -78.993 53.314
67508 93.546 -78.987 53.310
67509 93.554 -78.979 53.309
67510 93.562 -78.976 53.310
67511 93.567 -78.981 53.313
67512 93.574 -78.985 53.313
67513 93.593 -78.980 53.309
67514 93.612 -78.974 53.304
67515 93.629 -78.971 53.299
67516 93.643 -78.968 53.291
67517 93.656 -78.966 53.281
67518 93.671 -78.964 53.272
67519 93.685 -78.962 53.264
67520 93.701 -78.959 53.254
67521 93.718 -78.957 53.244
67522 93.735 -78.955 53.233
67523 93.745 -78.953 53.223
67524 93.755 -78.952 53.212
67525 93.763 -78.952 53.200
67526 93.769 -78.952 53.188
67527 93.776 -78.952 53.176
67528 93.782 -78.952 53.162
67529 93.789 -78.952 53.147
67530 93.796 -78.952 53.135
67531 93.803 -78.949 53.124
67532 93.812 -78.946 53.114
67533 93.820 -78.943 53.104
67534 93.828 -78.940 53.093
67535 93.837 -78.937 53.081
67536 93.846 -78.929 53.068
67537 93.857 -78.918 53.055
67538 93.859 -78.920 53.041
67539 93.860 -78.922 53.028
67540 93.862 -78.924 53.014
67541 93.863 -78.926 52.999
67542 93.865 -78.927 52.983
67543 93.867 -78.929 52.966
67544 93.868 -78.931 52.947
67545 93.870 -78.933 52.927
67546 93.866 -78.929 52.906
67547 93.863 -78.925 52.881
67548 93.860 -78.922 52.857
67549 93.857 -78.919 52.833
67550 93.855 -78.916 52.807
67551 93.872 -78.922 52.780
67552 93.889 -78.927 52.754
67553 93.906 -78.933 52.728
67554 93.923 -78.938 52.701
67555 93.940 -78.944 52.675
67556 93.957 -78.949 52.650
67557 93.974 -78.955 52.623
67558 93.991 -78.961 52.596
67559 94.008 -78.965 52.570
67560 94.014 -78.962 52.544
67561 94.024 -78.952 52.517
67562 94.033 -78.941 52.491
67563 94.042 -78.930 52.468
67564 94.051 -78.920 52.443
67565 94.060 -78.909 52.418
67566 94.069 -78.899 52.393
67567 94.078 -78.888 52.370
67568 94.087 -78.878 52.346
67569 94.096 -78.871 52.323
67570 94.103 -78.865 52.299
67571 94.110 -78.861 52.273
67572 94.117 -78.855 52.248
67573 94.125 -78.851 52.223
67574 94.139 -78.848 52.199
67575 94.155 -78.838 52.175
67576 94.172 -78.828 52.150
67577 94.189 -78.818 52.126
67578 94.205 -78.808 52.100
67579 94.221 -78.798 52.074
67580 94.238 -78.788 52.049
67581 94.254 -78.778 52.025
67582 94.271 -78.768 52.000
67583 94.282 -78.763 51.972
67584 94.292 -78.761 51.943
67585 94.308 -78.763 51.909
67586 94.323 -78.765 51.875
67587 94.339 -78.767 51.840
67588 94.354 -78.770 51.807
67589 94.369 -78.771 51.777
67590 94.385 -78.774 51.747
67591 94.400 -78.773 51.715
67592 94.416 -78.757 51.679
67593 94.433 -78.761 51.641
67594 94.451 -78.765 51.603
67595 94.468 -78.770 51.567
67596 94.486 -78.774 51.533
67597 94.503 -78.778 51.499
67598 94.521 -78.783 51.468
67599 94.538 -78.787 51.436
67600 94.555 -78.791 51.403
67601 94.573 -78.796 51.372
67602 94.593 -78.801 51.341
67603 94.620 -78.808 51.309
67604 94.646 -78.814 51.275
67605 94.661 -78.815 51.238
67606 94.676 -78.817 51.200
67607 94.691 -78.818 51.164
67608 94.706 -78.819 51.132
67609 94.718 -78.821 51.102
67610 94.716 -78.830 51.070
67611 94.712 -78.837 51.035
67612 94.690 -78.827 50.999
67613 94.685 -78.823 50.961
67614 94.689 -78.822 50.920
67615 94.693 -78.821 50.876
67616 94.697 -78.821 50.831
67617 94.701 -78.821 50.787
67618 94.705 -78.820 50.744
67619 94.709 -78.819 50.702
67620 94.713 -78.818 50.658
67621 94.729 -78.820 50.614
67622 94.753 -78.823 50.574
67623 94.777 -78.824 50.534
67624 94.808 -78.816 50.495
67625 94.839 -78.808 50.460
67626 94.866 -78.802 50.426
67627 94.885 -78.798 50.393
67628 94.891 -78.807 50.359
67629 94.890 -78.822 50.321
67630 94.887 -78.834 50.281
67631 94.879 -78.833 50.239
67632 94.871 -78.833 50.198
67633 94.863 -78.833 50.155
67634 94.855 -78.832 50.113
67635 94.847 -78.832 50.073
67636 94.839 -78.831 50.033
67637 94.831 -78.830 49.995
67638 94.823 -78.830 49.956
67639 94.835 -78.833 49.916
67640 94.852 -78.836 49.875
67641 94.868 -78.839 49.834
67642 94.882 -78.851 49.792
67643 94.890 -78.879 49.755
67644 94.897 -78.908 49.717
67645 94.905 -78.936 49.680
67646 94.913 -78.964 49.644
67647 94.944 -78.974 49.609
67648 94.976 -78.981 49.574
67649 94.992 -78.990 49.541
67650 95.008 -78.999 49.510
67651 95.025 -79.009 49.477
67652 95.041 -79.018 49.444
67653 95.057 -79.028 49.407
67654 95.076 -79.036 49.369
67655 95.106 -79.043 49.329
67656 95.135 -79.049 49.289
67657 95.162 -79.055 49.248
67658 95.187 -79.057 49.206
67659 95.211 -79.059 49.164
67660 95.236 -79.062 49.122
67661 95.260 -79.064 49.080
67662 95.285 -79.066 49.038
67663 95.309 -79.068 48.996
67664 95.334 -79.071 48.952
67665 95.358 -79.073 48.909
67666 95.375 -79.080 48.865
67667 95.378 -79.096 48.822
67668 95.380 -79.112 48.778
67669 95.382 -79.128 48.733
67670 95.385 -79.144 48.688
67671 95.392 -79.160 48.643
67672 95.405 -79.175 48.598
67673 95.418 -79.191 48.554
67674 95.431 -79.207 48.510
67675 95.444 -79.222 48.467
67676 95.457 -79.238 48.424
67677 95.470 -79.254 48.381
67678 95.500 -79.262 48.338
67679 95.528 -79.253 48.296
67680 95.550 -79.241 48.254
67681 95.572 -79.228 48.211
67682 95.595 -79.214 48.169
67683 95.618 -79.201 48.127
67684 95.640 -79.187 48.086
67685 95.663 -79.175 48.045
67686 95.680 -79.165 48.004
67687 95.691 -79.159 47.964
67688 95.701 -79.153 47.925
67689 95.711 -79.148 47.886
67690 95.736 -79.152 47.847
67691 95.769 -79.161 47.807
67692 95.803 -79.170 47.767
67693 95.837 -79.180 47.727
67694 95.871 -79.189 47.687
67695 95.899 -79.197 47.647
67696 95.912 -79.202 47.608
67697 95.908 -79.193 47.567
67698 95.903 -79.184 47.529
67699 95.899 -79.175 47.491
67700 95.895 -79.166 47.453
67701 95.893 -79.157 47.415
67702 95.905 -79.128 47.377
67703 95.892 -79.091 47.339
67704 95.901 -79.079 47.301
67705 95.952 -79.113 47.263
67706 96.003 -79.147 47.225
67707 96.054 -79.181 47.188
67708 96.105 -79.214 47.151
67709 96.151 -79.231 47.114
67710 96.179 -79.178 47.079
67711 96.207 -79.125 47.044
67712 96.235 -79.071 47.010
67713 96.263 -79.021 46.977
67714 96.297 -79.008 46.944
67715 96.327 -79.037 46.911
67716 96.356 -79.083 46.880
67717 96.384 -79.128 46.849
67718 96.412 -79.175 46.818
67719 96.441 -79.220 46.788
67720 96.469 -79.266 46.758
67721 96.496 -79.288 46.728
67722 96.523 -79.303 46.698
67723 96.546 -79.314 46.667
67724 96.555 -79.312 46.635
67725 96.564 -79.310 46.604
67726 96.573 -79.308 46.574
67727 96.582 -79.306 46.545
67728 96.594 -79.305 46.516
67729 96.616 -79.307 46.487
67730 96.640 -79.310 46.458
67731 96.662 -79.313 46.429
67732 96.685 -79.315 46.401
67733 96.708 -79.318 46.374
67734 96.731 -79.320 46.348
67735 96.753 -79.322 46.322
67736 96.776 -79.325 46.296
67737 96.798 -79.320 46.271
67738 96.816 -79.299 46.245
67739 96.835 -79.278 46.219
67740 96.854 -79.257 46.196
67741 96.872 -79.236 46.174
67742 96.891 -79.215 46.152
67743 96.909 -79.194 46.130
67744 96.939 -79.190 46.108
67745 96.976 -79.201 46.087
67746 97.013 -79.211 46.066
67747 97.050 -79.222 46.045
67748 97.076 -79.228 46.024
67749 97.074 -79.253 46.004
67750 97.088 -79.253 45.982
67751 97.103 -79.253 45.960
67752 97.117 -79.253 45.938
67753 97.133 -79.254 45.917
67754 97.147 -79.254 45.897
67755 97.162 -79.254 45.878
67756 97.175 -79.256 45.858
67757 97.151 -79.285 45.838
67758 97.165 -79.273 45.818
67759 97.167 -79.266 45.797
67760 97.168 -79.259 45.777
67761 97.170 -79.252 45.758
67762 97.193 -79.244 45.740
67763 97.228 -79.233 45.722
67764 97.235 -79.244 45.704
67765 97.239 -79.256 45.686
67766 97.243 -79.269 45.667
67767 97.247 -79.282 45.648
67768 97.251 -79.294 45.631
67769 97.255 -79.307 45.615
67770 97.259 -79.320 45.598
67771 97.263 -79.333 45.581
67772 97.267 -79.346 45.564
67773 97.271 -79.358 45.546
67774 97.275 -79.371 45.529
67775 97.279 -79.384 45.512
67776 97.283 -79.397 45.496
67777 97.287 -79.410 45.480
67778 97.290 -79.418 45.464
67779 97.296 -79.406 45.449
67780 97.302 -79.392 45.433
67781 97.309 -79.379 45.417
67782 97.317 -79.364 45.401
67783 97.325 -79.350 45.386
67784 97.329 -79.341 45.371
67785 97.349 -79.355 45.356
67786 97.368 -79.370 45.341
67787 97.387 -79.386 45.325
67788 97.406 -79.401 45.310
67789 97.425 -79.416 45.295
67790 97.444 -79.432 45.281
67791 97.463 -79.447 45.267
67792 97.483 -79.463 45.253
67793 97.502 -79.478 45.239
67794 97.521 -79.493 45.225
67795 97.540 -79.508 45.212
67796 97.560 -79.523 45.198
67797 97.579 -79.538 45.185
67798 97.598 -79.554 45.172
67799 97.617 -79.570 45.159
67800 97.636 -79.585 45.145
67801 97.655 -79.600 45.132
67802 97.674 -79.615 45.119
67803 97.693 -79.630 45.107
67804 97.713 -79.646 45.095
67805 97.732 -79.661 45.082
67806 97.751 -79.676 45.070
67807 97.770 -79.692 45.058
67808 97.789 -79.707 45.045
67809 97.809 -79.722 45.033
67810 97.828 -79.737 45.020
67811 97.847 -79.753 45.008
67812 97.866 -79.768 44.995
67813 97.885 -79.783 44.982
67814 97.904 -79.798 44.968
67815 97.924 -79.814 44.955
67816 97.943 -79.829 44.942
67817 97.962 -79.845 44.930
67818 97.981 -79.860 44.917
67819 98.000 -79.875 44.904
67820 98.019 -79.890 44.891
67821 98.038 -79.905 44.880
67822 98.058 -79.921 44.869
67823 98.077 -79.936 44.858
67824 98.096 -79.952 44.846
67825 98.115 -79.967 44.837
67826 98.135 -79.982 44.827
67827 98.154 -79.997 44.817
67828 98.173 -80.012 44.808
67829 98.192 -80.028 44.798
67830 98.211 -80.043 44.788
67831 98.230 -80.059 44.779
67832 98.249 -80.074 44.771
67833 98.269 -80.089 44.762
67834 98.288 -80.104 44.754
67835 98.307 -80.119 44.745
67836 98.326 -80.135 44.738
67837 98.345 -80.150 44.730
67838 98.365 -80.165 44.723
67839 98.384 -80.181 44.715
67840 98.403 -80.196 44.708
67841 98.422 -80.211 44.701
67842 98.441 -80.227 44.695
67843 98.460 -80.242 44.688
67844 98.480 -80.257 44.681
67845 98.499 -80.272 44.674
67846 98.518 -80.287 44.668
67847 98.541 -80.294 44.662
67848 98.566 -80.297 44.656
67849 98.590 -80.299 44.648
67850 98.616 -80.302 44.641
67851 98.640 -80.304 44.634
67852 98.665 -80.306 44.626
67853 98.690 -80.309 44.619
67854 98.714 -80.312 44.613
67855 98.739 -80.314 44.606
67856 98.762 -80.319 44.597
67857 98.787 -80.328 44.589
67858 98.810 -80.339 44.581
67859 98.834 -80.350 44.572
67860 98.857 -80.360 44.564
67861 98.880 -80.371 44.558
67862 98.904 -80.381 44.554
67863 98.927 -80.392 44.548
67864 98.950 -80.403 44.542
67865 98.974 -80.413 44.535
67866 98.997 -80.424 44.530
67867 99.020 -80.435 44.525
67868 99.043 -80.444 44.519
67869 99.067 -80.455 44.514
67870 99.090 -80.466 44.510
67871 99.113 -80.476 44.506
68198 93.444 -79.048 53.434
68199 93.453 -79.046 53.421
68200 93.464 -79.043 53.408
68201 93.473 -79.041 53.396
68202 93.484 -79.034 53.385
68203 93.495 -79.023 53.375
68204 93.505 -79.015 53.364
68205 93.513 -79.008 53.355
68206 93.522 -79.001 53.346
68207 93.530 -78.993 53.340
68208 93.539 -78.987 53.333
68209 93.547 -78.980 53.328
68210 93.555 -78.973 53.324
68211 93.564 -78.966 53.324
68212 93.572 -78.959 53.325
68213 93.577 -78.964 53.325
68214 93.590 -78.963 53.323
68215 93.609 -78.959 53.319
68216 93.624 -78.955 53.314
68217 93.638 -78.953 53.306
68218 93.652 -78.951 53.297
68219 93.669 -78.949 53.288
68220 93.685 -78.946 53.280
68221 93.702 -78.943 53.272
68222 93.719 -78.941 53.262
68223 93.731 -78.940 53.251
68224 93.741 -78.939 53.242
68225 93.751 -78.937 53.231
68226 93.759 -78.937 53.219
68227 93.765 -78.937 53.207
68228 93.772 -78.937 53.195
68229 93.779 -78.937 53.182
68230 93.785 -78.937 53.168
68231 93.792 -78.937 53.157
68232 93.798 -78.937 53.146
68233 93.805 -78.937 53.136
68234 93.813 -78.935 53.126
68235 93.821 -78.932 53.115
68236 93.831 -78.924 53.102
68237 93.842 -78.913 53.089
68238 93.851 -78.905 53.075
68239 93.853 -78.906 53.062
68240 93.854 -78.908 53.048
68241 93.856 -78.910 53.034
68242 93.857 -78.912 53.020
68243 93.859 -78.914 53.005
68244 93.861 -78.916 52.987
68245 93.861 -78.916 52.967
68246 93.857 -78.911 52.946
68247 93.852 -78.905 52.922
68248 93.848 -78.901 52.897
68249 93.845 -78.897 52.872
68250 93.842 -78.894 52.848
68251 93.847 -78.894 52.820
68252 93.864 -78.900 52.793
68253 93.881 -78.905 52.767
68254 93.898 -78.911 52.741
68255 93.915 -78.917 52.714
68256 93.932 -78.922 52.687
68257 93.948 -78.927 52.661
68258 93.968 -78.931 52.632
68259 93.988 -78.933 52.605
68260 94.008 -78.936 52.580
68261 94.024 -78.936 52.554
68262 94.034 -78.932 52.529
68263 94.044 -78.923 52.505
68264 94.053 -78.912 52.482
68265 94.062 -78.902 52.458
68266 94.071 -78.891 52.434
68267 94.080 -78.880 52.409
68268 94.089 -78.870 52.386
68269 94.099 -78.859 52.363
68270 94.108 -78.849 52.339
68271 94.115 -78.844 52.314
68272 94.122 -78.839 52.289
68273 94.129 -78.834 52.262
68274 94.140 -78.830 52.236
68275 94.154 -78.828 52.211
68276 94.170 -78.820 52.187
68277 94.186 -78.810 52.162
68278 94.202 -78.800 52.137
68279 94.219 -78.790 52.110
68280 94.236 -78.780 52.084
68281 94.252 -78.770 52.057
68282 94.268 -78.760 52.032
68283 94.281 -78.754 52.005
68284 94.290 -78.752 51.977
68285 94.298 -78.749 51.947
68286 94.313 -78.750 51.915
68287 94.328 -78.752 51.882
68288 94.343 -78.755 51.847
68289 94.359 -78.757 51.815
68290 94.374 -78.758 51.784
68291 94.390 -78.761 51.754
68292 94.403 -78.766 51.722
68293 94.418 -78.748 51.688
68294 94.436 -78.752 51.652
68295 94.454 -78.756 51.615
68296 94.471 -78.761 51.581
68297 94.488 -78.765 51.547
68298 94.506 -78.770 51.514
68299 94.523 -78.774 51.483
68300 94.541 -78.778 51.450
68301 94.558 -78.783 51.418
68302 94.577 -78.787 51.386
68303 94.604 -78.794 51.354
68304 94.631 -78.801 51.322
68305 94.659 -78.808 51.288
68306 94.682 -78.813 51.251
68307 94.696 -78.814 51.214
68308 94.711 -78.815 51.178
68309 94.726 -78.817 51.145
68310 94.733 -78.822 51.114
68311 94.730 -78.831 51.082
68312 94.728 -78.840 51.049
68313 94.725 -78.849 51.014
68314 94.704 -78.839 50.975
68315 94.683 -78.828 50.934
68316 94.673 -78.823 50.890
68317 94.677 -78.822 50.847
68318 94.681 -78.821 50.804
68319 94.685 -78.821 50.761
68320 94.689 -78.820 50.718
68321 94.699 -78.821 50.675
68322 94.722 -78.823 50.632
68323 94.751 -78.818 50.590
68324 94.782 -78.811 50.549
68325 94.813 -78.802 50.509
68326 94.844 -78.795 50.471
68327 94.864 -78.790 50.438
68328 94.883 -78.787 50.405
68329 94.889 -78.795 50.372
68330 94.888 -78.811 50.335
68331 94.887 -78.826 50.296
68332 94.884 -78.836 50.255
68333 94.876 -78.836 50.211
68334 94.867 -78.835 50.170
68335 94.859 -78.835 50.129
68336 94.852 -78.834 50.088
68337 94.844 -78.833 50.046
68338 94.836 -78.833 50.006
68339 94.828 -78.833 49.967
68340 94.836 -78.834 49.928
68341 94.852 -78.837 49.887
68342 94.870 -78.840 49.846
68343 94.886 -78.843 49.806
68344 94.900 -78.855 49.768
68345 94.908 -78.883 49.730
68346 94.916 -78.911 49.692
68347 94.923 -78.939 49.657
68348 94.946 -78.955 49.622
68349 94.978 -78.962 49.587
68350 94.995 -78.971 49.554
68351 95.011 -78.981 49.521
68352 95.028 -78.990 49.487
68353 95.044 -78.999 49.453
68354 95.063 -79.008 49.415
68355 95.093 -79.015 49.376
68356 95.122 -79.021 49.336
68357 95.151 -79.028 49.297
68358 95.181 -79.035 49.256
68359 95.205 -79.037 49.215
68360 95.230 -79.040 49.174
68361 95.255 -79.042 49.132
68362 95.279 -79.044 49.091
68363 95.304 -79.046 49.049
68364 95.328 -79.049 49.006
68365 95.353 -79.051 48.963
68366 95.376 -79.054 48.919
68367 95.385 -79.059 48.876
68368 95.390 -79.078 48.832
68369 95.394 -79.099 48.787
68370 95.397 -79.115 48.742
68371 95.399 -79.131 48.698
68372 95.402 -79.147 48.653
68373 95.406 -79.163 48.608
68374 95.419 -79.178 48.564
68375 95.432 -79.194 48.520
68376 95.445 -79.210 48.477
68377 95.458 -79.225 48.434
68378 95.471 -79.241 48.391
68379 95.497 -79.251 48.347
68380 95.525 -79.242 48.304
68381 95.547 -79.229 48.261
68382 95.569 -79.216 48.219
68383 95.592 -79.203 48.177
68384 95.615 -79.190 48.135
68385 95.637 -79.177 48.093
68386 95.660 -79.163 48.051
68387 95.682 -79.150 48.010
68388 95.693 -79.144 47.970
68389 95.703 -79.139 47.930
68390 95.713 -79.133 47.891
68391 95.743 -79.140 47.852
68392 95.776 -79.149 47.811
68393 95.810 -79.158 47.771
68394 95.843 -79.168 47.731
68395 95.871 -79.174 47.690
68396 95.883 -79.175 47.651
68397 95.895 -79.175 47.611
68398 95.912 -79.189 47.571
68399 95.912 -79.186 47.531
68400 95.908 -79.178 47.492
68401 95.903 -79.169 47.454
68402 95.899 -79.159 47.415
68403 95.933 -79.144 47.376
68404 95.919 -79.108 47.337
68405 95.918 -79.084 47.298
68406 95.969 -79.118 47.260
68407 96.020 -79.152 47.223
68408 96.071 -79.186 47.186
68409 96.121 -79.216 47.149
68410 96.149 -79.162 47.113
68411 96.176 -79.109 47.078
68412 96.204 -79.056 47.043
68413 96.237 -79.032 47.009
68414 96.271 -79.019 46.976
68415 96.305 -79.006 46.943
68416 96.335 -79.037 46.911
68417 96.364 -79.082 46.880
68418 96.392 -79.128 46.849
68419 96.420 -79.174 46.818
68420 96.449 -79.219 46.788
68421 96.475 -79.256 46.758
68422 96.489 -79.257 46.727
68423 96.511 -79.265 46.697
68424 96.537 -79.280 46.666
68425 96.552 -79.284 46.634
68426 96.561 -79.281 46.603
68427 96.570 -79.279 46.573
68428 96.579 -79.278 46.543
68429 96.601 -79.280 46.514
68430 96.624 -79.282 46.485
68431 96.647 -79.285 46.456
68432 96.670 -79.288 46.427
68433 96.693 -79.290 46.399
68434 96.716 -79.292 46.372
68435 96.738 -79.295 46.345
68436 96.761 -79.297 46.319
68437 96.784 -79.300 46.294
68438 96.806 -79.297 46.268
68439 96.824 -79.275 46.242
68440 96.843 -79.254 46.216
68441 96.862 -79.234 46.193
68442 96.880 -79.212 46.171
68443 96.899 -79.191 46.149
68444 96.923 -79.179 46.126
68445 96.960 -79.190 46.104
68446 96.997 -79.200 46.082
68447 97.034 -79.210 46.061
68448 97.051 -79.213 46.040
68449 97.079 -79.218 46.020
68450 97.080 -79.219 45.999
68451 97.079 -79.240 45.978
68452 97.094 -79.241 45.957
68453 97.109 -79.241 45.935
68454 97.123 -79.241 45.914
68455 97.139 -79.241 45.895
68456 97.153 -79.241 45.876
68457 97.168 -79.241 45.857
68458 97.150 -79.265 45.837
68459 97.166 -79.252 45.817
68460 97.167 -79.245 45.796
68461 97.169 -79.238 45.777
68462 97.171 -79.231 45.759
68463 97.172 -79.225 45.741
68464 97.196 -79.216 45.723
68465 97.220 -79.213 45.705
68466 97.224 -79.226 45.687
68467 97.228 -79.239 45.669
68468 97.232 -79.252 45.650
68469 97.236 -79.264 45.632
68470 97.240 -79.277 45.616
68471 97.244 -79.290 45.600
68472 97.248 -79.303 45.583
68473 97.252 -79.316 45.566
68474 97.256 -79.328 45.549
68475 97.260 -79.341 45.531
68476 97.264 -79.353 45.514
68477 97.277 -79.370 45.499
68478 97.291 -79.387 45.484
68479 97.293 -79.405 45.468
68480 97.321 -79.393 45.452
68481 97.346 -79.379 45.437
68482 97.354 -79.364 45.421
68483 97.360 -79.352 45.405
68484 97.366 -79.339 45.390
68485 97.372 -79.327 45.375
68486 97.374 -79.325 45.360
68487 97.361 -79.356 45.346
68488 97.395 -79.351 45.330
68489 97.415 -79.366 45.315
68490 97.434 -79.382 45.300
68491 97.453 -79.397 45.286
68492 97.472 -79.413 45.273
68493 97.491 -79.428 45.259
68494 97.510 -79.444 45.244
68495 97.529 -79.459 45.231
68496 97.548 -79.474 45.217
68497 97.567 -79.490 45.204
68498 97.586 -79.505 45.191
68499 97.605 -79.520 45.178
68500 97.625 -79.536 45.165
68501 97.644 -79.551 45.152
68502 97.662 -79.567 45.139
68503 97.681 -79.582 45.127
68504 97.701 -79.598 45.115
68505 97.720 -79.613 45.102
68506 97.739 -79.628 45.090
68507 97.758 -79.644 45.078
68508 97.777 -79.659 45.066
68509 97.796 -79.674 45.054
68510 97.815 -79.690 45.042
68511 97.834 -79.705 45.029
68512 97.853 -79.720 45.017
68513 97.872 -79.736 45.005
68514 97.891 -79.751 44.991
68515 97.911 -79.767 44.978
68516 97.930 -79.783 44.965
68517 97.948 -79.798 44.952
68518 97.967 -79.813 44.939
68519 97.987 -79.828 44.926
68520 98.006 -79.844 44.913
68521 98.025 -79.859 44.901
68522 98.044 -79.874 44.890
68523 98.063 -79.889 44.879
68524 98.082 -79.905 44.868
68525 98.101 -79.921 44.856
68526 98.121 -79.936 44.847
68527 98.140 -79.951 44.838
68528 98.159 -79.966 44.828
68529 98.178 -79.981 44.818
68530 98.198 -79.996 44.808
68531 98.217 -80.012 44.799
68532 98.236 -80.027 44.789
68533 98.255 -80.043 44.781
68534 98.274 -80.058 44.773
68535 98.293 -80.073 44.765
68536 98.312 -80.088 44.757
68537 98.332 -80.103 44.749
68538 98.351 -80.119 44.742
68539 98.370 -80.134 44.734
68540 98.389 -80.149 44.728
68541 98.408 -80.165 44.721
68542 98.427 -80.180 44.713
68543 98.447 -80.195 44.707
68544 98.466 -80.211 44.700
68545 98.485 -80.226 44.693
68546 98.504 -80.241 44.686
68547 98.523 -80.256 44.680
68548 98.547 -80.262 44.674
68549 98.571 -80.265 44.668
68550 98.596 -80.268 44.662
68551 98.621 -80.270 44.655
68552 98.644 -80.274 44.649
68553 98.666 -80.278 44.643
68554 98.688 -80.282 44.638
68555 98.709 -80.287 44.633
68556 98.730 -80.291 44.626
68557 98.752 -80.297 44.617
68558 98.773 -80.307 44.606
68559 98.814 -80.312 44.594
68560 98.838 -80.322 44.584
68561 98.861 -80.333 44.576
68562 98.884 -80.344 44.570
68563 98.908 -80.354 44.564
68564 98.931 -80.365 44.558
68565 98.954 -80.375 44.552
68566 98.977 -80.386 44.546
68567 99.001 -80.397 44.540
68568 99.024 -80.407 44.534
68569 99.047 -80.417 44.529
68570 99.071 -80.428 44.524
68571 99.094 -80.438 44.520
68572 99.117 -80.449 44.515
68573 99.141 -80.460 44.510
68574 99.164 -80.470 44.505
68575 99.187 -80.481 44.501
68900 93.454 -79.035 53.442
68901 93.464 -79.033 53.429
68902 93.473 -79.030 53.417
68903 93.483 -79.028 53.406
68904 93.494 -79.019 53.395
68905 93.505 -79.009 53.385
68906 93.514 -79.001 53.375
68907 93.523 -78.994 53.365
68908 93.532 -78.987 53.358
68909 93.540 -78.981 53.352
68910 93.548 -78.973 53.345
68911 93.556 -78.966 53.341
68912 93.565 -78.959 53.338
68913 93.573 -78.952 53.337
68914 93.581 -78.946 53.336
68915 93.588 -78.946 53.335
68916 93.606 -78.942 53.333
68917 93.620 -78.940 53.327
68918 93.635 -78.937 53.319
68919 93.652 -78.935 53.311
68920 93.669 -78.933 53.303
68921 93.686 -78.930 53.294
68922 93.703 -78.928 53.286
68923 93.718 -78.926 53.278
68924 93.727 -78.925 53.268
68925 93.737 -78.924 53.259
68926 93.747 -78.923 53.249
68927 93.755 -78.922 53.237
68928 93.762 -78.922 53.225
68929 93.768 -78.922 53.213
68930 93.775 -78.922 53.201
68931 93.781 -78.922 53.189
68932 93.788 -78.922 53.177
68933 93.795 -78.922 53.167
68934 93.801 -78.922 53.156
68935 93.808 -78.922 53.146
68936 93.815 -78.921 53.135
68937 93.827 -78.908 53.123
68938 93.838 -78.897 53.109
68939 93.845 -78.891 53.095
68940 93.846 -78.893 53.081
68941 93.848 -78.895 53.067
68942 93.850 -78.896 53.053
68943 93.851 -78.899 53.038
68944 93.853 -78.900 53.023
68945 93.852 -78.899 53.003
68946 93.848 -78.893 52.982
68947 93.843 -78.889 52.959
68948 93.839 -78.883 52.935
68949 93.834 -78.878 52.910
68950 93.830 -78.873 52.885
68951 93.827 -78.869 52.860
68952 93.839 -78.873 52.832
68953 93.856 -78.878 52.805
68954 93.873 -78.883 52.779
68955 93.893 -78.886 52.753
68956 93.913 -78.888 52.726
68957 93.934 -78.890 52.699
68958 93.954 -78.893 52.672
68959 93.974 -78.895 52.644
68960 93.995 -78.897 52.617
68961 94.015 -78.899 52.591
68962 94.033 -78.899 52.567
68963 94.043 -78.896 52.543
68964 94.053 -78.892 52.520
68965 94.063 -78.888 52.497
68966 94.073 -78.883 52.473
68967 94.082 -78.872 52.448
68968 94.091 -78.861 52.424
68969 94.100 -78.851 52.401
68970 94.109 -78.840 52.378
68971 94.118 -78.830 52.354
68972 94.127 -78.823 52.329
68973 94.134 -78.818 52.303
68974 94.141 -78.813 52.276
68975 94.154 -78.810 52.247
68976 94.169 -78.808 52.222
68977 94.184 -78.802 52.197
68978 94.200 -78.792 52.172
68979 94.216 -78.782 52.147
68980 94.233 -78.772 52.120
68981 94.249 -78.762 52.092
68982 94.266 -78.752 52.064
68983 94.280 -78.744 52.037
68984 94.288 -78.742 52.009
68985 94.297 -78.740 51.981
68986 94.306 -78.738 51.951
68987 94.317 -78.737 51.920
68988 94.333 -78.739 51.888
68989 94.348 -78.742 51.855
68990 94.364 -78.743 51.823
68991 94.379 -78.745 51.792
68992 94.393 -78.749 51.761
68993 94.406 -78.755 51.729
68994 94.421 -78.741 51.696
68995 94.438 -78.743 51.663
68996 94.456 -78.748 51.627
68997 94.473 -78.752 51.595
68998 94.491 -78.756 51.562
68999 94.508 -78.761 51.530
69000 94.526 -78.765 51.498
69001 94.543 -78.770 51.465
69002 94.561 -78.774 51.433
69003 94.588 -78.781 51.400
69004 94.615 -78.788 51.367
69005 94.642 -78.795 51.334
69006 94.670 -78.802 51.300
69007 94.697 -78.808 51.264
69008 94.717 -78.812 51.227
69009 94.732 -78.813 51.191
69010 94.747 -78.814 51.157
69011 94.747 -78.823 51.125
69012 94.744 -78.831 51.092
69013 94.742 -78.840 51.059
69014 94.740 -78.849 51.025
69015 94.737 -78.858 50.988
69016 94.718 -78.851 50.949
69017 94.697 -78.840 50.907
69018 94.675 -78.830 50.864
69019 94.661 -78.822 50.821
69020 94.665 -78.821 50.778
69021 94.669 -78.821 50.735
69022 94.694 -78.821 50.692
69023 94.725 -78.813 50.649
69024 94.756 -78.805 50.608
69025 94.787 -78.797 50.566
69026 94.818 -78.789 50.525
69027 94.844 -78.783 50.484
69028 94.862 -78.779 50.448
69029 94.881 -78.776 50.413
69030 94.888 -78.783 50.379
69031 94.887 -78.799 50.344
69032 94.886 -78.814 50.307
69033 94.884 -78.830 50.268
69034 94.880 -78.838 50.227
69035 94.872 -78.838 50.186
69036 94.864 -78.837 50.145
69037 94.856 -78.836 50.103
69038 94.848 -78.836 50.060
69039 94.840 -78.836 50.019
69040 94.833 -78.835 49.978
69041 94.837 -78.836 49.939
69042 94.853 -78.839 49.899
69043 94.870 -78.843 49.859
69044 94.887 -78.846 49.821
69045 94.904 -78.849 49.781
69046 94.918 -78.858 49.743
69047 94.926 -78.886 49.705
69048 94.934 -78.914 49.670
69049 94.949 -78.937 49.635
69050 94.981 -78.943 49.600
69051 94.997 -78.952 49.567
69052 95.014 -78.962 49.531
69053 95.030 -78.971 49.496
69054 95.050 -78.980 49.461
69055 95.079 -78.987 49.422
69056 95.109 -78.993 49.383
69057 95.138 -79.000 49.343
69058 95.168 -79.007 49.303
69059 95.197 -79.014 49.263
69060 95.225 -79.018 49.224
69061 95.249 -79.021 49.185
69062 95.273 -79.023 49.143
69063 95.298 -79.025 49.101
69064 95.323 -79.028 49.059
69065 95.347 -79.030 49.017
69066 95.372 -79.032 48.974
69067 95.385 -79.037 48.931
69068 95.394 -79.043 48.886
69069 95.400 -79.057 48.842
69070 95.404 -79.077 48.796
69071 95.409 -79.097 48.750
69072 95.414 -79.118 48.707
69073 95.416 -79.134 48.663
69074 95.419 -79.150 48.619
69075 95.421 -79.166 48.575
69076 95.433 -79.181 48.531
69077 95.446 -79.197 48.487
69078 95.459 -79.213 48.444
69079 95.472 -79.228 48.401
69080 95.495 -79.240 48.357
69081 95.522 -79.231 48.312
69082 95.544 -79.218 48.267
69083 95.566 -79.205 48.227
69084 95.589 -79.191 48.185
69085 95.612 -79.178 48.142
69086 95.634 -79.165 48.099
69087 95.657 -79.152 48.057
69088 95.679 -79.139 48.016
69089 95.696 -79.129 47.976
69090 95.706 -79.124 47.935
69091 95.716 -79.118 47.894
69092 95.749 -79.128 47.855
69093 95.783 -79.137 47.813
69094 95.817 -79.146 47.772
69095 95.842 -79.152 47.732
69096 95.854 -79.153 47.692
69097 95.867 -79.153 47.652
69098 95.879 -79.154 47.612
69099 95.891 -79.155 47.573
69100 95.909 -79.174 47.532
69101 95.917 -79.179 47.492
69102 95.912 -79.171 47.453
69103 95.908 -79.162 47.414
69104 95.919 -79.150 47.374
69105 95.947 -79.125 47.335
69106 95.935 -79.090 47.296
69107 95.986 -79.124 47.258
69108 96.037 -79.157 47.221
69109 96.088 -79.190 47.184
69110 96.118 -79.147 47.148
69111 96.146 -79.093 47.112
69112 96.176 -79.056 47.077
69113 96.210 -79.043 47.042
69114 96.245 -79.031 47.008
69115 96.279 -79.018 46.975
69116 96.313 -79.005 46.943
69117 96.343 -79.036 46.911
69118 96.372 -79.082 46.880
69119 96.400 -79.128 46.849
69120 96.429 -79.173 46.819
69121 96.457 -79.219 46.788
69122 96.477 -79.236 46.758
69123 96.491 -79.237 46.727
69124 96.506 -79.237 46.696
69125 96.525 -79.243 46.664
69126 96.549 -79.255 46.632
69127 96.558 -79.253 46.601
69128 96.568 -79.251 46.572
69129 96.586 -79.252 46.542
69130 96.609 -79.255 46.512
69131 96.632 -79.257 46.483
69132 96.655 -79.259 46.454
69133 96.678 -79.263 46.426
69134 96.700 -79.265 46.397
69135 96.723 -79.267 46.370
69136 96.746 -79.270 46.343
69137 96.769 -79.272 46.318
69138 96.792 -79.275 46.292
69139 96.814 -79.273 46.266
69140 96.832 -79.252 46.240
69141 96.851 -79.231 46.215
69142 96.870 -79.210 46.191
69143 96.888 -79.189 46.168
69144 96.907 -79.168 46.145
69145 96.944 -79.178 46.123
69146 96.981 -79.188 46.101
69147 97.018 -79.198 46.078
69148 97.021 -79.213 46.056
69149 97.056 -79.179 46.035
69150 97.081 -79.208 46.015
69151 97.077 -79.209 45.996
69152 97.085 -79.209 45.975
69153 97.085 -79.227 45.955
69154 97.100 -79.227 45.934
69155 97.115 -79.227 45.913
69156 97.130 -79.228 45.893
69157 97.145 -79.228 45.874
69158 97.159 -79.228 45.856
69159 97.149 -79.244 45.837
69160 97.166 -79.231 45.817
69161 97.168 -79.225 45.797
69162 97.169 -79.217 45.778
69163 97.171 -79.210 45.759
69164 97.173 -79.203 45.742
69165 97.174 -79.197 45.724
69166 97.199 -79.187 45.706
69167 97.210 -79.196 45.688
69168 97.214 -79.209 45.669
69169 97.218 -79.222 45.652
69170 97.221 -79.234 45.635
69171 97.225 -79.247 45.619
69172 97.229 -79.259 45.602
69173 97.233 -79.272 45.585
69174 97.237 -79.285 45.568
69175 97.251 -79.302 45.551
69176 97.265 -79.319 45.534
69177 97.280 -79.335 45.518
69178 97.294 -79.353 45.503
69179 97.308 -79.369 45.488
69180 97.297 -79.388 45.472
69181 97.318 -79.380 45.456
69182 97.353 -79.366 45.441
69183 97.374 -79.353 45.425
69184 97.387 -79.341 45.409
69185 97.400 -79.329 45.394
69186 97.412 -79.317 45.379
69187 97.418 -79.305 45.365
69188 97.418 -79.307 45.350
69189 97.404 -79.339 45.335
69190 97.391 -79.370 45.320
69191 97.433 -79.357 45.306
69192 97.461 -79.363 45.292
69193 97.480 -79.378 45.278
69194 97.500 -79.394 45.265
69195 97.519 -79.409 45.250
69196 97.537 -79.424 45.237
69197 97.557 -79.440 45.224
69198 97.576 -79.455 45.210
69199 97.595 -79.470 45.198
69200 97.614 -79.486 45.185
69201 97.633 -79.501 45.172
69202 97.652 -79.516 45.160
69203 97.671 -79.532 45.147
69204 97.690 -79.548 45.135
69205 97.709 -79.563 45.122
69206 97.728 -79.578 45.110
69207 97.747 -79.594 45.098
69208 97.767 -79.609 45.086
69209 97.786 -79.624 45.074
69210 97.805 -79.640 45.062
69211 97.823 -79.655 45.050
69212 97.843 -79.670 45.038
69213 97.862 -79.686 45.027
69214 97.881 -79.701 45.014
69215 97.900 -79.717 45.001
69216 97.919 -79.733 44.987
69217 97.938 -79.748 44.974
69218 97.957 -79.763 44.961
69219 97.976 -79.778 44.948
69220 97.995 -79.794 44.935
69221 98.014 -79.809 44.923
69222 98.033 -79.824 44.911
69223 98.053 -79.840 44.899
69224 98.072 -79.855 44.888
69225 98.090 -79.871 44.877
69226 98.109 -79.886 44.867
69227 98.129 -79.902 44.857
69228 98.148 -79.917 44.848
69229 98.167 -79.933 44.839
69230 98.186 -79.948 44.829
69231 98.205 -79.963 44.819
69232 98.224 -79.978 44.810
69233 98.243 -79.994 44.800
69234 98.262 -80.009 44.793
69235 98.281 -80.024 44.785
69236 98.300 -80.040 44.777
69237 98.319 -80.055 44.769
69238 98.339 -80.071 44.762
69239 98.357 -80.087 44.754
69240 98.376 -80.102 44.746
69241 98.395 -80.117 44.739
69242 98.415 -80.133 44.732
69243 98.434 -80.148 44.725
69244 98.453 -80.163 44.719
69245 98.472 -80.178 44.713
69246 98.491 -80.194 44.706
69247 98.508 -80.211 44.700
69248 98.526 -80.227 44.695
69249 98.550 -80.233 44.689
69250 98.570 -80.237 44.683
69251 98.591 -80.242 44.678
69252 98.613 -80.247 44.673
69253 98.635 -80.251 44.668
69254 98.656 -80.255 44.662
69255 98.678 -80.259 44.656
69256 98.700 -80.264 44.650
69257 98.721 -80.269 44.644
69258 98.742 -80.275 44.633
69259 98.762 -80.286 44.621
69260 98.786 -80.294 44.607
69261 98.836 -80.293 44.596
69262 98.865 -80.306 44.588
69263 98.888 -80.316 44.581
69264 98.912 -80.327 44.574
69265 98.935 -80.337 44.568
69266 98.958 -80.348 44.562
69267 98.981 -80.359 44.557
69268 99.005 -80.369 44.550
69269 99.028 -80.380 44.544
69270 99.051 -80.391 44.539
69271 99.075 -80.400 44.535
69272 99.098 -80.411 44.530
69273 99.121 -80.422 44.525
69274 99.144 -80.432 44.520
69275 99.168 -80.443 44.514
69276 99.191 -80.453 44.510
69277 99.214 -80.464 44.507
69278 99.238 -80.475 44.503
69602 93.464 -79.021 53.450
69603 93.474 -79.019 53.438
69604 93.483 -79.017 53.426
69605 93.493 -79.014 53.415
69606 93.504 -79.006 53.405
69607 93.515 -78.995 53.395
69608 93.524 -78.987 53.385
69609 93.533 -78.981 53.377
69610 93.541 -78.974 53.370
69611 93.549 -78.967 53.363
69612 93.557 -78.960 53.357
69613 93.566 -78.953 53.353
69614 93.574 -78.946 53.349
69615 93.582 -78.939 53.347
69616 93.591 -78.932 53.345
69617 93.602 -78.927 53.342
69618 93.619 -78.924 53.337
69619 93.636 -78.922 53.330
69620 93.653 -78.920 53.323
69621 93.670 -78.918 53.315
69622 93.687 -78.915 53.307
69623 93.704 -78.912 53.299
69624 93.713 -78.912 53.291
69625 93.723 -78.910 53.282
69626 93.733 -78.909 53.274
69627 93.743 -78.908 53.265
69628 93.751 -78.907 53.255
69629 93.758 -78.907 53.243
69630 93.764 -78.907 53.231
69631 93.771 -78.907 53.219
69632 93.778 -78.907 53.209
69633 93.784 -78.907 53.198
69634 93.791 -78.908 53.187
69635 93.798 -78.907 53.175
69636 93.805 -78.905 53.165
69637 93.812 -78.902 53.155
69638 93.822 -78.893 53.144
69639 93.833 -78.881 53.129
69640 93.839 -78.877 53.115
69641 93.840 -78.879 53.100
69642 93.842 -78.881 53.085
69643 93.844 -78.883 53.069
69644 93.845 -78.885 53.054
69645 93.843 -78.882 53.038
69646 93.839 -78.877 53.017
69647 93.834 -78.871 52.994
69648 93.830 -78.866 52.971
69649 93.825 -78.861 52.947
69650 93.821 -78.855 52.921
69651 93.817 -78.850 52.896
69652 93.821 -78.846 52.871
69653 93.839 -78.843 52.843
69654 93.859 -78.845 52.816
69655 93.880 -78.847 52.790
69656 93.900 -78.849 52.764
69657 93.920 -78.852 52.737
69658 93.940 -78.854 52.710
69659 93.961 -78.856 52.683
69660 93.981 -78.858 52.656
69661 94.002 -78.860 52.629
69662 94.022 -78.862 52.604
69663 94.041 -78.864 52.580
69664 94.051 -78.860 52.556
69665 94.061 -78.856 52.533
69666 94.071 -78.852 52.510
69667 94.081 -78.849 52.486
69668 94.091 -78.845 52.462
69669 94.102 -78.840 52.438
69670 94.111 -78.833 52.414
69671 94.120 -78.822 52.390
69672 94.129 -78.811 52.366
69673 94.138 -78.801 52.342
69674 94.145 -78.796 52.315
69675 94.155 -78.793 52.288
69676 94.170 -78.789 52.260
69677 94.183 -78.787 52.233
69678 94.197 -78.783 52.207
69679 94.214 -78.774 52.182
69680 94.230 -78.764 52.156
69681 94.247 -78.754 52.128
69682 94.263 -78.744 52.099
69683 94.278 -78.735 52.071
69684 94.287 -78.733 52.042
69685 94.296 -78.730 52.013
69686 94.305 -78.728 51.984
69687 94.313 -78.727 51.955
69688 94.322 -78.724 51.924
69689 94.338 -78.727 51.893
69690 94.353 -78.729 51.862
69691 94.368 -78.730 51.831
69692 94.384 -78.733 51.800
69693 94.397 -78.738 51.769
69694 94.410 -78.742 51.736
69695 94.424 -78.737 51.705
69696 94.441 -78.735 51.674
69697 94.459 -78.739 51.642
69698 94.476 -78.743 51.609
69699 94.493 -78.748 51.578
69700 94.511 -78.752 51.548
69701 94.529 -78.756 51.514
69702 94.546 -78.761 51.481
69703 94.572 -78.767 51.447
69704 94.600 -78.774 51.413
69705 94.627 -78.781 51.380
69706 94.654 -78.788 51.347
69707 94.681 -78.795 51.313
69708 94.708 -78.802 51.277
69709 94.736 -78.809 51.241
69710 94.752 -78.811 51.205
69711 94.763 -78.814 51.170
69712 94.761 -78.823 51.136
69713 94.759 -78.832 51.102
69714 94.756 -78.841 51.069
69715 94.754 -78.850 51.035
69716 94.752 -78.859 50.999
69717 94.748 -78.863 50.962
69718 94.728 -78.853 50.922
69719 94.706 -78.843 50.879
69720 94.684 -78.832 50.836
69721 94.662 -78.821 50.793
69722 94.668 -78.815 50.750
69723 94.699 -78.807 50.708
69724 94.731 -78.799 50.665
69725 94.762 -78.791 50.623
69726 94.792 -78.783 50.581
69727 94.824 -78.775 50.539
69728 94.843 -78.771 50.497
69729 94.861 -78.768 50.458
69730 94.879 -78.764 50.421
69731 94.887 -78.771 50.386
69732 94.885 -78.787 50.352
69733 94.884 -78.802 50.316
69734 94.883 -78.818 50.279
69735 94.882 -78.833 50.240
69736 94.877 -78.840 50.200
69737 94.869 -78.839 50.159
69738 94.861 -78.839 50.117
69739 94.853 -78.839 50.074
69740 94.845 -78.838 50.032
69741 94.837 -78.838 49.991
69742 94.837 -78.839 49.952
69743 94.854 -78.842 49.913
69744 94.871 -78.845 49.873
69745 94.888 -78.848 49.835
69746 94.904 -78.851 49.795
69747 94.921 -78.854 49.756
69748 94.936 -78.861 49.719
69749 94.945 -78.889 49.683
69750 94.952 -78.917 49.648
69751 94.983 -78.924 49.612
69752 95.000 -78.933 49.578
69753 95.016 -78.943 49.544
69754 95.037 -78.951 49.508
69755 95.066 -78.958 49.471
69756 95.096 -78.965 49.432
69757 95.125 -78.971 49.392
69758 95.155 -78.978 49.351
69759 95.184 -78.985 49.311
69760 95.213 -78.992 49.272
69761 95.243 -78.999 49.233
69762 95.268 -79.002 49.194
69763 95.293 -79.004 49.152
69764 95.317 -79.006 49.111
69765 95.341 -79.009 49.069
69766 95.366 -79.011 49.026
69767 95.386 -79.014 48.983
69768 95.394 -79.020 48.939
69769 95.403 -79.026 48.895
69770 95.410 -79.035 48.851
69771 95.415 -79.056 48.807
69772 95.419 -79.076 48.762
69773 95.424 -79.096 48.718
69774 95.428 -79.116 48.675
69775 95.433 -79.136 48.631
69776 95.436 -79.153 48.585
69777 95.439 -79.169 48.542
69778 95.447 -79.184 48.498
69779 95.460 -79.200 48.455
69780 95.473 -79.216 48.412
69781 95.492 -79.229 48.368
69782 95.519 -79.220 48.324
69783 95.541 -79.206 48.279
69784 95.563 -79.194 48.237
69785 95.586 -79.181 48.194
69786 95.609 -79.167 48.151
69787 95.631 -79.154 48.108
69788 95.654 -79.140 48.066
69789 95.676 -79.128 48.023
69790 95.698 -79.115 47.981
69791 95.709 -79.109 47.939
69792 95.723 -79.106 47.898
69793 95.756 -79.115 47.856
69794 95.790 -79.125 47.815
69795 95.813 -79.130 47.774
69796 95.826 -79.131 47.732
69797 95.838 -79.131 47.692
69798 95.850 -79.132 47.651
69799 95.862 -79.133 47.611
69800 95.875 -79.134 47.572
69801 95.888 -79.137 47.531
69802 95.907 -79.158 47.491
69803 95.921 -79.172 47.451
69804 95.917 -79.164 47.412
69805 95.913 -79.155 47.373
69806 95.960 -79.137 47.335
69807 95.961 -79.105 47.297
69808 96.003 -79.129 47.258
69809 96.054 -79.162 47.221
69810 96.088 -79.131 47.184
69811 96.116 -79.080 47.148
69812 96.150 -79.068 47.111
69813 96.184 -79.055 47.075
69814 96.219 -79.042 47.041
69815 96.253 -79.029 47.008
69816 96.287 -79.016 46.975
69817 96.321 -79.003 46.942
69818 96.352 -79.035 46.912
69819 96.380 -79.081 46.881
69820 96.408 -79.127 46.850
69821 96.437 -79.172 46.819
69822 96.464 -79.215 46.789
69823 96.479 -79.216 46.758
69824 96.493 -79.216 46.728
69825 96.508 -79.216 46.695
69826 96.523 -79.217 46.663
69827 96.540 -79.220 46.631
69828 96.555 -79.225 46.601
69829 96.572 -79.225 46.571
69830 96.594 -79.227 46.541
69831 96.617 -79.230 46.511
69832 96.640 -79.232 46.482
69833 96.663 -79.234 46.454
69834 96.685 -79.237 46.427
69835 96.708 -79.240 46.398
69836 96.731 -79.242 46.370
69837 96.754 -79.245 46.344
69838 96.777 -79.247 46.318
69839 96.799 -79.250 46.292
69840 96.822 -79.250 46.266
69841 96.840 -79.228 46.240
69842 96.859 -79.208 46.214
69843 96.878 -79.187 46.191
69844 96.896 -79.165 46.168
69845 96.928 -79.166 46.145
69846 96.965 -79.177 46.122
69847 96.999 -79.189 46.099
69848 96.991 -79.213 46.076
69849 97.026 -79.177 46.054
69850 97.064 -79.174 46.033
69851 97.084 -79.198 46.014
69852 97.080 -79.199 45.994
69853 97.081 -79.200 45.974
69854 97.090 -79.200 45.954
69855 97.099 -79.200 45.933
69856 97.106 -79.214 45.913
69857 97.121 -79.214 45.894
69858 97.136 -79.214 45.875
69859 97.151 -79.215 45.856
69860 97.148 -79.224 45.837
69861 97.167 -79.210 45.818
69862 97.168 -79.203 45.799
69863 97.170 -79.197 45.779
69864 97.172 -79.190 45.761
69865 97.173 -79.182 45.743
69866 97.175 -79.175 45.726
69867 97.177 -79.169 45.708
69868 97.194 -79.165 45.690
69869 97.198 -79.178 45.672
69870 97.203 -79.191 45.654
69871 97.207 -79.203 45.637
69872 97.212 -79.217 45.621
69873 97.226 -79.234 45.604
69874 97.240 -79.250 45.588
69875 97.254 -79.268 45.571
69876 97.268 -79.285 45.554
69877 97.283 -79.301 45.537
69878 97.297 -79.319 45.522
69879 97.311 -79.335 45.507
69880 97.316 -79.356 45.491
69881 97.300 -79.372 45.476
69882 97.316 -79.366 45.460
69883 97.350 -79.352 45.445
69884 97.373 -79.339 45.429
69885 97.386 -79.327 45.413
69886 97.399 -79.316 45.398
69887 97.412 -79.303 45.384
69888 97.425 -79.292 45.370
69889 97.447 -79.312 45.355
69890 97.475 -79.402 45.340
69891 97.461 -79.400 45.325
69892 97.443 -79.396 45.312
69893 97.424 -79.392 45.297
69894 97.461 -79.374 45.284
69895 97.508 -79.359 45.270
69896 97.527 -79.374 45.257
69897 97.546 -79.390 45.244
69898 97.565 -79.405 45.230
69899 97.584 -79.420 45.217
69900 97.603 -79.436 45.204
69901 97.623 -79.451 45.192
69902 97.642 -79.466 45.180
69903 97.661 -79.482 45.167
69904 97.680 -79.498 45.155
69905 97.699 -79.513 45.142
69906 97.718 -79.529 45.130
69907 97.737 -79.544 45.118
69908 97.756 -79.559 45.106
69909 97.775 -79.574 45.094
69910 97.794 -79.590 45.082
69911 97.813 -79.605 45.071
69912 97.832 -79.620 45.059
69913 97.851 -79.636 45.047
69914 97.870 -79.651 45.035
69915 97.889 -79.667 45.022
69916 97.909 -79.683 45.009
69917 97.928 -79.698 44.996
69918 97.947 -79.713 44.983
69919 97.965 -79.729 44.970
69920 97.985 -79.744 44.957
69921 98.004 -79.759 44.945
69922 98.023 -79.774 44.933
69923 98.042 -79.790 44.921
69924 98.061 -79.805 44.909
69925 98.080 -79.820 44.899
69926 98.098 -79.837 44.888
69927 98.114 -79.855 44.878
69928 98.132 -79.871 44.869
69929 98.149 -79.889 44.859
69930 98.166 -79.905 44.849
69931 98.182 -79.923 44.840
69932 98.200 -79.940 44.831
69933 98.217 -79.957 44.822
69934 98.233 -79.974 44.813
69935 98.250 -79.991 44.805
69936 98.268 -80.008 44.797
69937 98.285 -80.025 44.789
69938 98.302 -80.042 44.781
69939 98.319 -80.058 44.774
69940 98.336 -80.074 44.767
69941 98.354 -80.090 44.758
69942 98.371 -80.106 44.751
69943 98.388 -80.121 44.744
69944 98.406 -80.137 44.738
69945 98.423 -80.153 44.732
69946 98.446 -80.168 44.726
69947 98.469 -80.182 44.719
69948 98.503 -80.179 44.713
69949 98.530 -80.194 44.709
69950 98.557 -80.206 44.704
69951 98.577 -80.211 44.698
69952 98.598 -80.215 44.693
69953 98.620 -80.220 44.689
69954 98.640 -80.225 44.685
69955 98.661 -80.229 44.679
69956 98.683 -80.234 44.673
69957 98.704 -80.237 44.666
69958 98.728 -80.240 44.659
69959 98.754 -80.249 44.648
69960 98.779 -80.260 44.634
69961 98.790 -80.269 44.620
69962 98.797 -80.275 44.608
69963 98.845 -80.275 44.600
69964 98.892 -80.289 44.592
69965 98.915 -80.300 44.585
69966 98.939 -80.310 44.579
69967 98.962 -80.321 44.572
69968 98.985 -80.331 44.566
69969 99.009 -80.342 44.559
69970 99.032 -80.353 44.554
69971 99.055 -80.363 44.549
69972 99.079 -80.374 44.544
69973 99.102 -80.384 44.539
69974 99.125 -80.394 44.534
69975 99.148 -80.405 44.529
69976 99.172 -80.416 44.524
69977 99.195 -80.426 44.520
69978 99.218 -80.437 44.516
69979 99.242 -80.447 44.512
69980 99.265 -80.458 44.509
69981 99.288 -80.469 44.505
69982 99.311 -80.479 44.501
70304 93.474 -79.009 53.458
70305 93.484 -79.006 53.446
70306 93.493 -79.003 53.435
70307 93.503 -79.001 53.425
70308 93.514 -78.991 53.415
70309 93.525 -78.981 53.406
70310 93.534 -78.974 53.397
70311 93.542 -78.967 53.388
70312 93.550 -78.960 53.380
70313 93.559 -78.953 53.374
70314 93.567 -78.946 53.368
70315 93.575 -78.940 53.361
70316 93.583 -78.933 53.358
70317 93.592 -78.925 53.355
70318 93.609 -78.915 53.351
70319 93.620 -78.908 53.346
70320 93.637 -78.906 53.340
70321 93.654 -78.904 53.333
70322 93.671 -78.902 53.326
70323 93.688 -78.899 53.318
70324 93.700 -78.898 53.311
70325 93.709 -78.896 53.304
70326 93.719 -78.896 53.296
70327 93.729 -78.894 53.289
70328 93.739 -78.893 53.280
70329 93.747 -78.893 53.271
70330 93.754 -78.893 53.260
70331 93.761 -78.893 53.249
70332 93.767 -78.893 53.239
70333 93.774 -78.893 53.227
70334 93.781 -78.893 53.216
70335 93.787 -78.890 53.206
70336 93.794 -78.888 53.195
70337 93.801 -78.886 53.184
70338 93.808 -78.883 53.173
70339 93.818 -78.877 53.161
70340 93.829 -78.865 53.147
70341 93.833 -78.864 53.133
70342 93.834 -78.866 53.118
70343 93.836 -78.868 53.101
70344 93.838 -78.869 53.084
70345 93.834 -78.865 53.068
70346 93.830 -78.860 53.050
70347 93.825 -78.855 53.028
70348 93.821 -78.849 53.005
70349 93.816 -78.844 52.981
70350 93.817 -78.839 52.955
70351 93.822 -78.834 52.930
70352 93.828 -78.830 52.906
70353 93.837 -78.825 52.882
70354 93.852 -78.819 52.854
70355 93.868 -78.812 52.826
70356 93.887 -78.811 52.800
70357 93.907 -78.813 52.774
70358 93.927 -78.814 52.747
70359 93.947 -78.817 52.721
70360 93.968 -78.819 52.694
70361 93.988 -78.821 52.667
70362 94.009 -78.824 52.642
70363 94.029 -78.826 52.618
70364 94.049 -78.828 52.593
70365 94.059 -78.824 52.569
70366 94.069 -78.821 52.546
70367 94.079 -78.816 52.523
70368 94.090 -78.812 52.500
70369 94.100 -78.808 52.475
70370 94.110 -78.805 52.451
70371 94.120 -78.801 52.426
70372 94.130 -78.797 52.401
70373 94.140 -78.793 52.378
70374 94.149 -78.782 52.354
70375 94.157 -78.775 52.327
70376 94.171 -78.772 52.300
70377 94.184 -78.769 52.273
70378 94.198 -78.767 52.245
70379 94.212 -78.764 52.219
70380 94.228 -78.755 52.194
70381 94.244 -78.745 52.168
70382 94.261 -78.736 52.139
70383 94.277 -78.726 52.110
70384 94.286 -78.724 52.081
70385 94.294 -78.721 52.052
70386 94.303 -78.719 52.023
70387 94.312 -78.717 51.994
70388 94.320 -78.714 51.965
70389 94.329 -78.712 51.935
70390 94.343 -78.714 51.904
70391 94.358 -78.715 51.873
70392 94.373 -78.717 51.842
70393 94.387 -78.722 51.812
70394 94.400 -78.727 51.781
70395 94.414 -78.731 51.750
70396 94.427 -78.733 51.718
70397 94.443 -78.726 51.687
70398 94.461 -78.730 51.656
70399 94.479 -78.735 51.625
70400 94.496 -78.739 51.595
70401 94.513 -78.743 51.565
70402 94.531 -78.748 51.533
70403 94.557 -78.755 51.498
70404 94.584 -78.761 51.462
70405 94.611 -78.768 51.427
70406 94.638 -78.775 51.393
70407 94.666 -78.782 51.359
70408 94.693 -78.789 51.325
70409 94.720 -78.796 51.290
70410 94.747 -78.802 51.255
70411 94.773 -78.809 51.219
70412 94.777 -78.814 51.183
70413 94.775 -78.824 51.148
70414 94.771 -78.826 51.113
70415 94.768 -78.827 51.079
70416 94.764 -78.828 51.044
70417 94.760 -78.830 51.009
70418 94.756 -78.830 50.973
70419 94.737 -78.821 50.934
70420 94.715 -78.811 50.893
70421 94.693 -78.800 50.851
70422 94.676 -78.801 50.807
70423 94.679 -78.803 50.765
70424 94.705 -78.793 50.723
70425 94.736 -78.785 50.681
70426 94.767 -78.777 50.637
70427 94.798 -78.769 50.594
70428 94.823 -78.764 50.551
70429 94.841 -78.760 50.510
70430 94.859 -78.757 50.470
70431 94.877 -78.754 50.431
70432 94.885 -78.760 50.394
70433 94.884 -78.775 50.360
70434 94.883 -78.791 50.324
70435 94.882 -78.806 50.287
70436 94.881 -78.822 50.251
70437 94.879 -78.837 50.211
70438 94.874 -78.843 50.171
70439 94.866 -78.842 50.130
70440 94.858 -78.841 50.088
70441 94.850 -78.841 50.047
70442 94.842 -78.840 50.006
70443 94.838 -78.840 49.967
70444 94.855 -78.843 49.928
70445 94.872 -78.846 49.888
70446 94.889 -78.849 49.849
70447 94.905 -78.852 49.808
70448 94.922 -78.855 49.769
70449 94.939 -78.858 49.732
70450 94.955 -78.864 49.696
70451 94.963 -78.892 49.660
70452 94.986 -78.905 49.625
70453 95.002 -78.915 49.590
70454 95.024 -78.923 49.555
70455 95.054 -78.930 49.518
70456 95.083 -78.937 49.481
70457 95.112 -78.943 49.441
70458 95.142 -78.950 49.400
70459 95.171 -78.957 49.359
70460 95.200 -78.964 49.318
70461 95.230 -78.971 49.280
70462 95.259 -78.977 49.242
70463 95.287 -78.983 49.202
70464 95.311 -78.985 49.160
70465 95.336 -78.987 49.120
70466 95.360 -78.990 49.079
70467 95.385 -78.992 49.036
70468 95.395 -78.997 48.992
70469 95.403 -79.003 48.949
70470 95.412 -79.009 48.905
70471 95.420 -79.015 48.861
70472 95.425 -79.034 48.817
70473 95.429 -79.054 48.773
70474 95.434 -79.075 48.729
70475 95.439 -79.094 48.686
70476 95.444 -79.115 48.642
70477 95.448 -79.134 48.596
70478 95.453 -79.155 48.553
70479 95.456 -79.172 48.510
70480 95.461 -79.187 48.467
70481 95.474 -79.203 48.423
70482 95.489 -79.218 48.379
70483 95.516 -79.209 48.335
70484 95.538 -79.195 48.291
70485 95.560 -79.182 48.247
70486 95.583 -79.169 48.203
70487 95.606 -79.156 48.161
70488 95.628 -79.143 48.117
70489 95.651 -79.130 48.074
70490 95.673 -79.116 48.031
70491 95.696 -79.103 47.988
70492 95.711 -79.094 47.944
70493 95.729 -79.093 47.901
70494 95.763 -79.103 47.859
70495 95.784 -79.108 47.817
70496 95.797 -79.109 47.776
70497 95.809 -79.109 47.734
70498 95.821 -79.110 47.693
70499 95.834 -79.111 47.652
70500 95.846 -79.112 47.611
70501 95.858 -79.112 47.571
70502 95.871 -79.113 47.532
70503 95.886 -79.122 47.491
70504 95.905 -79.143 47.451
70505 95.925 -79.165 47.412
70506 95.921 -79.157 47.373
70507 95.945 -79.143 47.335
70508 95.989 -79.122 47.297
70509 96.020 -79.134 47.259
70510 96.057 -79.115 47.221
70511 96.090 -79.091 47.184
70512 96.124 -79.079 47.148
70513 96.158 -79.066 47.111
70514 96.192 -79.053 47.076
70515 96.227 -79.040 47.042
70516 96.261 -79.028 47.009
70517 96.295 -79.015 46.976
70518 96.329 -79.002 46.943
70519 96.359 -79.035 46.913
70520 96.388 -79.081 46.882
70521 96.416 -79.126 46.851
70522 96.445 -79.172 46.819
70523 96.466 -79.194 46.788
70524 96.480 -79.195 46.757
70525 96.496 -79.195 46.726
70526 96.510 -79.196 46.694
70527 96.525 -79.196 46.662
70528 96.540 -79.197 46.631
70529 96.554 -79.197 46.602
70530 96.579 -79.200 46.571
70531 96.602 -79.202 46.541
70532 96.624 -79.204 46.512
70533 96.648 -79.207 46.483
70534 96.670 -79.209 46.455
70535 96.693 -79.212 46.427
70536 96.716 -79.215 46.399
70537 96.739 -79.217 46.372
70538 96.761 -79.219 46.345
70539 96.784 -79.222 46.318
70540 96.807 -79.225 46.293
70541 96.830 -79.226 46.267
70542 96.849 -79.205 46.241
70543 96.867 -79.184 46.215
70544 96.886 -79.163 46.191
70545 96.912 -79.155 46.168
70546 96.949 -79.165 46.145
70547 96.972 -79.184 46.122
70548 96.960 -79.212 46.098
70549 96.996 -79.177 46.075
70550 97.031 -79.140 46.053
70551 97.072 -79.169 46.032
70552 97.087 -79.188 46.013
70553 97.083 -79.189 45.993
70554 97.078 -79.190 45.974
70555 97.086 -79.190 45.954
70556 97.095 -79.191 45.933
70557 97.104 -79.191 45.913
70558 97.112 -79.201 45.895
70559 97.127 -79.201 45.877
70560 97.142 -79.201 45.857
70561 97.147 -79.203 45.838
70562 97.167 -79.190 45.820
70563 97.169 -79.182 45.801
70564 97.171 -79.175 45.781
70565 97.172 -79.169 45.763
70566 97.174 -79.162 45.745
70567 97.175 -79.155 45.728
70568 97.177 -79.147 45.710
70569 97.179 -79.140 45.693
70570 97.186 -79.149 45.675
70571 97.201 -79.165 45.657
70572 97.215 -79.183 45.640
70573 97.229 -79.200 45.624
70574 97.243 -79.216 45.607
70575 97.257 -79.234 45.591
70576 97.272 -79.250 45.574
70577 97.286 -79.267 45.557
70578 97.300 -79.284 45.541
70579 97.314 -79.301 45.526
70580 97.328 -79.318 45.511
70581 97.306 -79.350 45.495
70582 97.304 -79.357 45.479
70583 97.319 -79.350 45.464
70584 97.347 -79.338 45.449
70585 97.373 -79.325 45.433
70586 97.385 -79.313 45.418
70587 97.398 -79.302 45.404
70588 97.412 -79.290 45.389
70589 97.428 -79.285 45.374
70590 97.458 -79.342 45.360
70591 97.475 -79.388 45.345
70592 97.482 -79.397 45.331
70593 97.478 -79.400 45.317
70594 97.460 -79.397 45.303
70595 97.471 -79.390 45.290
70596 97.496 -79.382 45.276
70597 97.520 -79.375 45.263
70598 97.543 -79.368 45.250
70599 97.574 -79.370 45.237
70600 97.593 -79.386 45.224
70601 97.612 -79.401 45.212
70602 97.631 -79.416 45.199
70603 97.650 -79.432 45.187
70604 97.669 -79.447 45.174
70605 97.687 -79.463 45.162
70606 97.704 -79.481 45.150
70607 97.721 -79.498 45.138
70608 97.738 -79.515 45.126
70609 97.755 -79.532 45.114
70610 97.772 -79.549 45.103
70611 97.789 -79.567 45.091
70612 97.806 -79.583 45.080
70613 97.823 -79.601 45.068
70614 97.840 -79.617 45.056
70615 97.857 -79.635 45.044
70616 97.874 -79.652 45.031
70617 97.891 -79.669 45.018
70618 97.908 -79.686 45.006
70619 97.925 -79.703 44.992
70620 97.942 -79.720 44.980
70621 97.959 -79.737 44.967
70622 97.976 -79.755 44.955
70623 97.993 -79.771 44.942
70624 98.010 -79.789 44.931
70625 98.027 -79.806 44.919
70626 98.043 -79.823 44.909
70627 98.061 -79.840 44.899
70628 98.078 -79.856 44.889
70629 98.095 -79.872 44.879
70630 98.112 -79.888 44.870
70631 98.130 -79.905 44.860
70632 98.147 -79.921 44.852
70633 98.164 -79.936 44.843
70634 98.181 -79.952 44.834
70635 98.199 -79.968 44.825
70636 98.216 -79.984 44.817
70637 98.233 -80.000 44.809
70638 98.251 -80.016 44.801
70639 98.268 -80.032 44.794
70640 98.285 -80.048 44.787
70641 98.303 -80.064 44.780
70642 98.320 -80.080 44.772
70643 98.342 -80.095 44.764
70644 98.366 -80.109 44.758
70645 98.389 -80.123 44.751
70646 98.414 -80.137 44.745
70647 98.449 -80.125 44.739
70648 98.480 -80.133 44.733
70649 98.507 -80.148 44.726
70650 98.534 -80.162 44.722
70651 98.562 -80.178 44.717
70652 98.585 -80.184 44.711
70653 98.605 -80.189 44.707
70654 98.627 -80.193 44.702
70655 98.648 -80.198 44.697
70656 98.672 -80.201 44.692
70657 98.696 -80.203 44.688
70658 98.720 -80.206 44.682
70659 98.745 -80.213 44.674
70660 98.772 -80.224 44.659
70661 98.797 -80.234 44.645
70662 98.803 -80.240 44.632
70663 98.805 -80.243 44.619
70664 98.821 -80.253 44.611
70665 98.858 -80.262 44.603
70666 98.917 -80.271 44.595
70667 98.943 -80.283 44.589
70668 98.966 -80.294 44.582
70669 98.989 -80.304 44.575
70670 99.013 -80.315 44.569
70671 99.036 -80.325 44.564
70672 99.059 -80.336 44.559
70673 99.082 -80.347 44.554
70674 99.106 -80.356 44.549
70675 99.129 -80.367 44.544
70676 99.152 -80.378 44.538
70677 99.176 -80.388 44.533
70678 99.199 -80.399 44.529
70679 99.222 -80.410 44.525
70680 99.246 -80.420 44.521
70681 99.269 -80.431 44.518
70682 99.292 -80.441 44.514
70683 99.315 -80.452 44.510
70684 99.342 -80.461 44.507
70685 99.381 -80.465 44.503
71006 93.484 -78.995 53.465
71007 93.493 -78.993 53.454
71008 93.503 -78.990 53.444
71009 93.513 -78.987 53.434
71010 93.524 -78.977 53.424
71011 93.535 -78.968 53.415
71012 93.543 -78.961 53.406
71013 93.551 -78.954 53.397
71014 93.560 -78.946 53.390
71015 93.568 -78.940 53.383
71016 93.576 -78.933 53.377
71017 93.585 -78.926 53.371
71018 93.593 -78.919 53.365
71019 93.602 -78.912 53.359
71020 93.622 -78.893 53.353
71021 93.638 -78.891 53.348
71022 93.654 -78.889 53.343
71023 93.672 -78.886 53.338
71024 93.686 -78.884 53.329
71025 93.696 -78.883 53.322
71026 93.705 -78.882 53.316
71027 93.715 -78.880 53.311
71028 93.725 -78.880 53.304
71029 93.734 -78.878 53.296
71030 93.744 -78.877 53.287
71031 93.750 -78.877 53.278
71032 93.757 -78.877 53.268
71033 93.764 -78.877 53.258
71034 93.770 -78.877 53.246
71035 93.777 -78.874 53.235
71036 93.784 -78.872 53.224
71037 93.791 -78.870 53.213
71038 93.798 -78.868 53.202
71039 93.805 -78.865 53.190
71040 93.813 -78.861 53.178
71041 93.825 -78.849 53.163
71042 93.827 -78.850 53.148
71043 93.828 -78.852 53.133
71044 93.829 -78.853 53.116
71045 93.825 -78.848 53.098
71046 93.821 -78.843 53.080
71047 93.816 -78.837 53.061
71048 93.814 -78.832 53.038
71049 93.819 -78.827 53.014
71050 93.824 -78.824 52.990
71051 93.830 -78.819 52.964
71052 93.835 -78.814 52.939
71053 93.840 -78.810 52.914
71054 93.852 -78.805 52.889
71055 93.868 -78.798 52.861
71056 93.884 -78.792 52.834
71057 93.900 -78.786 52.808
71058 93.916 -78.780 52.783
71059 93.934 -78.778 52.756
71060 93.955 -78.780 52.730
71061 93.975 -78.783 52.704
71062 93.995 -78.785 52.677
71063 94.015 -78.787 52.653
71064 94.036 -78.789 52.630
71065 94.056 -78.792 52.606
71066 94.068 -78.788 52.583
71067 94.078 -78.784 52.560
71068 94.088 -78.780 52.537
71069 94.098 -78.777 52.513
71070 94.108 -78.773 52.489
71071 94.118 -78.769 52.463
71072 94.128 -78.765 52.437
71073 94.138 -78.761 52.413
71074 94.149 -78.758 52.389
71075 94.159 -78.753 52.366
71076 94.173 -78.754 52.339
71077 94.185 -78.752 52.312
71078 94.199 -78.749 52.286
71079 94.213 -78.746 52.258
71080 94.226 -78.744 52.233
71081 94.242 -78.737 52.207
71082 94.258 -78.727 52.181
71083 94.275 -78.717 52.152
71084 94.284 -78.714 52.123
71085 94.293 -78.712 52.094
71086 94.301 -78.710 52.064
71087 94.310 -78.708 52.035
71088 94.319 -78.705 52.007
71089 94.328 -78.703 51.979
71090 94.336 -78.701 51.949
71091 94.347 -78.701 51.918
71092 94.363 -78.702 51.888
71093 94.378 -78.705 51.857
71094 94.391 -78.710 51.826
71095 94.404 -78.715 51.795
71096 94.417 -78.720 51.764
71097 94.430 -78.724 51.733
71098 94.447 -78.717 51.702
71099 94.464 -78.721 51.671
71100 94.481 -78.726 51.641
71101 94.499 -78.730 51.611
71102 94.516 -78.735 51.581
71103 94.541 -78.741 51.549
71104 94.568 -78.748 51.514
71105 94.596 -78.755 51.477
71106 94.623 -78.761 51.441
71107 94.650 -78.768 51.406
71108 94.677 -78.775 51.372
71109 94.704 -78.783 51.338
71110 94.732 -78.789 51.304
71111 94.755 -78.795 51.270
71112 94.773 -78.796 51.234
71113 94.784 -78.795 51.198
71114 94.783 -78.792 51.162
71115 94.779 -78.793 51.126
71116 94.775 -78.794 51.091
71117 94.771 -78.796 51.055
71118 94.768 -78.796 51.020
71119 94.764 -78.798 50.984
71120 94.747 -78.789 50.946
71121 94.724 -78.779 50.906
71122 94.705 -78.773 50.865
71123 94.693 -78.790 50.822
71124 94.695 -78.793 50.781
71125 94.714 -78.781 50.739
71126 94.741 -78.771 50.696
71127 94.772 -78.763 50.653
71128 94.803 -78.755 50.610
71129 94.821 -78.752 50.567
71130 94.839 -78.749 50.527
71131 94.857 -78.745 50.485
71132 94.875 -78.742 50.444
71133 94.884 -78.749 50.404
71134 94.883 -78.764 50.367
71135 94.882 -78.779 50.330
71136 94.880 -78.795 50.293
71137 94.879 -78.810 50.256
71138 94.878 -78.826 50.217
71139 94.877 -78.841 50.179
71140 94.871 -78.845 50.141
71141 94.862 -78.844 50.102
71142 94.854 -78.843 50.062
71143 94.847 -78.843 50.023
71144 94.839 -78.843 49.984
71145 94.856 -78.846 49.944
71146 94.873 -78.849 49.903
71147 94.890 -78.852 49.863
71148 94.906 -78.855 49.821
71149 94.923 -78.858 49.781
71150 94.939 -78.861 49.744
71151 94.957 -78.864 49.708
71152 94.973 -78.868 49.674
71153 94.989 -78.886 49.639
71154 95.012 -78.895 49.603
71155 95.041 -78.902 49.565
71156 95.070 -78.908 49.527
71157 95.100 -78.915 49.490
71158 95.129 -78.922 49.449
71159 95.158 -78.929 49.408
71160 95.188 -78.936 49.366
71161 95.217 -78.943 49.325
71162 95.246 -78.949 49.288
71163 95.276 -78.956 49.250
71164 95.305 -78.963 49.210
71165 95.330 -78.965 49.168
71166 95.355 -78.968 49.127
71167 95.379 -78.970 49.086
71168 95.395 -78.974 49.044
71169 95.404 -78.981 49.001
71170 95.412 -78.986 48.958
71171 95.421 -78.992 48.915
71172 95.429 -78.998 48.871
71173 95.435 -79.012 48.827
71174 95.440 -79.033 48.784
71175 95.445 -79.053 48.740
71176 95.449 -79.073 48.697
71177 95.454 -79.093 48.653
71178 95.458 -79.113 48.608
71179 95.463 -79.133 48.566
71180 95.467 -79.153 48.523
71181 95.472 -79.173 48.479
71182 95.475 -79.190 48.435
71183 95.488 -79.206 48.391
71184 95.513 -79.197 48.347
71185 95.535 -79.184 48.302
71186 95.557 -79.171 48.257
71187 95.580 -79.158 48.213
71188 95.603 -79.144 48.170
71189 95.625 -79.131 48.127
71190 95.648 -79.118 48.083
71191 95.670 -79.105 48.039
71192 95.693 -79.092 47.995
71193 95.714 -79.080 47.949
71194 95.736 -79.081 47.905
71195 95.756 -79.085 47.862
71196 95.768 -79.086 47.820
71197 95.780 -79.087 47.778
71198 95.793 -79.087 47.736
71199 95.805 -79.088 47.695
71200 95.817 -79.089 47.654
71201 95.830 -79.090 47.613
71202 95.842 -79.090 47.572
71203 95.854 -79.091 47.533
71204 95.867 -79.092 47.494
71205 95.883 -79.106 47.453
71206 95.903 -79.128 47.414
71207 95.923 -79.149 47.375
71208 95.930 -79.150 47.336
71209 95.986 -79.131 47.298
71210 96.029 -79.115 47.260
71211 96.063 -79.103 47.222
71212 96.098 -79.090 47.184
71213 96.132 -79.078 47.148
71214 96.166 -79.065 47.112
71215 96.200 -79.052 47.077
71216 96.235 -79.039 47.043
71217 96.269 -79.026 47.010
71218 96.303 -79.014 46.977
71219 96.337 -79.001 46.944
71220 96.367 -79.034 46.914
71221 96.396 -79.080 46.883
71222 96.424 -79.126 46.851
71223 96.453 -79.172 46.819
71224 96.468 -79.174 46.788
71225 96.482 -79.174 46.756
71226 96.498 -79.175 46.724
71227 96.512 -79.175 46.693
71228 96.527 -79.175 46.662
71229 96.544 -79.180 46.632
71230 96.568 -79.191 46.602
71231 96.573 -79.181 46.571
71232 96.600 -79.178 46.541
71233 96.632 -79.179 46.513
71234 96.655 -79.182 46.483
71235 96.678 -79.184 46.455
71236 96.701 -79.187 46.428
71237 96.724 -79.190 46.400
71238 96.746 -79.192 46.374
71239 96.769 -79.194 46.348
71240 96.792 -79.197 46.320
71241 96.815 -79.200 46.295
71242 96.837 -79.202 46.269
71243 96.857 -79.182 46.243
71244 96.875 -79.161 46.218
71245 96.896 -79.143 46.193
71246 96.933 -79.153 46.169
71247 96.945 -79.178 46.146
71248 96.930 -79.212 46.123
71249 96.965 -79.176 46.099
71250 97.001 -79.140 46.075
71251 97.037 -79.118 46.053
71252 97.079 -79.163 46.033
71253 97.090 -79.178 46.014
71254 97.085 -79.179 45.994
71255 97.081 -79.180 45.974
71256 97.081 -79.181 45.955
71257 97.091 -79.181 45.935
71258 97.100 -79.181 45.916
71259 97.109 -79.182 45.897
71260 97.118 -79.187 45.879
71261 97.133 -79.187 45.859
71262 97.146 -79.184 45.840
71263 97.168 -79.169 45.822
71264 97.169 -79.162 45.804
71265 97.171 -79.155 45.784
71266 97.173 -79.147 45.766
71267 97.174 -79.145 45.749
71268 97.175 -79.144 45.731
71269 97.176 -79.143 45.714
71270 97.177 -79.142 45.696
71271 97.195 -79.138 45.679
71272 97.218 -79.148 45.660
71273 97.232 -79.165 45.643
71274 97.246 -79.182 45.627
71275 97.260 -79.199 45.611
71276 97.275 -79.216 45.595
71277 97.289 -79.233 45.578
71278 97.303 -79.250 45.562
71279 97.317 -79.266 45.546
71280 97.331 -79.284 45.530
71281 97.346 -79.300 45.515
71282 97.296 -79.344 45.500
71283 97.308 -79.340 45.485
71284 97.323 -79.334 45.469
71285 97.344 -79.325 45.454
71286 97.372 -79.311 45.438
71287 97.385 -79.300 45.424
71288 97.397 -79.288 45.409
71289 97.411 -79.277 45.395
71290 97.438 -79.288 45.381
71291 97.469 -79.368 45.367
71292 97.474 -79.375 45.352
71293 97.480 -79.383 45.338
71294 97.489 -79.392 45.324
71295 97.496 -79.400 45.310
71296 97.505 -79.394 45.297
71297 97.527 -79.385 45.284
71298 97.543 -79.373 45.270
71299 97.558 -79.361 45.258
71300 97.562 -79.357 45.245
71301 97.567 -79.367 45.232
71302 97.582 -79.398 45.220
71303 97.599 -79.415 45.207
71304 97.616 -79.432 45.195
71305 97.633 -79.449 45.182
71306 97.650 -79.466 45.170
71307 97.667 -79.483 45.158
71308 97.684 -79.501 45.146
71309 97.701 -79.518 45.135
71310 97.718 -79.535 45.123
71311 97.735 -79.552 45.112
71312 97.752 -79.569 45.101
71313 97.769 -79.586 45.089
71314 97.786 -79.603 45.077
71315 97.803 -79.620 45.064
71316 97.820 -79.637 45.052
71317 97.837 -79.654 45.040
71318 97.854 -79.671 45.027
71319 97.871 -79.687 45.016
71320 97.888 -79.703 45.002
71321 97.906 -79.719 44.989
71322 97.923 -79.735 44.977
71323 97.941 -79.751 44.964
71324 97.958 -79.767 44.952
71325 97.975 -79.783 44.941
71326 97.993 -79.798 44.930
71327 98.010 -79.814 44.920
71328 98.027 -79.830 44.910
71329 98.044 -79.846 44.900
71330 98.062 -79.863 44.890
71331 98.079 -79.879 44.881
71332 98.096 -79.895 44.872
71333 98.113 -79.911 44.863
71334 98.131 -79.927 44.854
71335 98.148 -79.943 44.845
71336 98.165 -79.958 44.837
71337 98.182 -79.974 44.830
71338 98.200 -79.990 44.823
71339 98.217 -80.006 44.815
71340 98.238 -80.021 44.807
71341 98.262 -80.036 44.800
71342 98.286 -80.050 44.793
71343 98.309 -80.065 44.786
71344 98.333 -80.078 44.780
71345 98.357 -80.093 44.774
71346 98.385 -80.098 44.768
71347 98.429 -80.071 44.760
71348 98.456 -80.087 44.753
71349 98.484 -80.102 44.747
71350 98.511 -80.116 44.741
71351 98.538 -80.131 44.737
71352 98.566 -80.146 44.731
71353 98.592 -80.158 44.725
71354 98.615 -80.161 44.719
71355 98.640 -80.164 44.715
71356 98.664 -80.167 44.710
71357 98.688 -80.169 44.705
71358 98.712 -80.172 44.700
71359 98.737 -80.178 44.694
71360 98.763 -80.187 44.685
71361 98.789 -80.198 44.670
71362 98.810 -80.209 44.655
71363 98.819 -80.210 44.642
71364 98.822 -80.212 44.630
71365 98.834 -80.221 44.622
71366 98.850 -80.232 44.614
71367 98.869 -80.243 44.606
71368 98.917 -80.243 44.599
71369 98.970 -80.266 44.592
71370 98.993 -80.277 44.585
71371 99.017 -80.287 44.580
71372 99.040 -80.298 44.574
71373 99.063 -80.309 44.568
71374 99.086 -80.319 44.564
71375 99.110 -80.330 44.558
71376 99.133 -80.340 44.553
71377 99.156 -80.350 44.548
71378 99.180 -80.361 44.542
71379 99.203 -80.372 44.538
71380 99.226 -80.382 44.534
71381 99.250 -80.393 44.530
71382 99.273 -80.403 44.528
71383 99.296 -80.414 44.525
71384 99.320 -80.425 44.521
71385 99.353 -80.431 44.516
71386 99.392 -80.435 44.513
71387 99.430 -80.438 44.509
71388 99.469 -80.442 44.506
71708 93.494 -78.981 53.473
71709 93.503 -78.979 53.462
71710 93.513 -78.977 53.452
71711 93.523 -78.973 53.442
71712 93.535 -78.962 53.433
71713 93.544 -78.954 53.424
71714 93.552 -78.947 53.415
71715 93.561 -78.940 53.407
71716 93.569 -78.933 53.399
71717 93.577 -78.927 53.392
71718 93.586 -78.919 53.385
71719 93.594 -78.912 53.378
71720 93.603 -78.905 53.371
71721 93.655 -78.879 53.365
71722 93.638 -78.875 53.359
71723 93.655 -78.873 53.354
71724 93.672 -78.871 53.349
71725 93.682 -78.869 53.342
71726 93.692 -78.868 53.336
71727 93.701 -78.868 53.331
71728 93.711 -78.866 53.325
71729 93.720 -78.865 53.320
71730 93.730 -78.864 53.314
71731 93.740 -78.863 53.305
71732 93.746 -78.863 53.297
71733 93.753 -78.863 53.287
71734 93.760 -78.861 53.277
71735 93.767 -78.858 53.265
71736 93.774 -78.855 53.254
71737 93.781 -78.853 53.242
71738 93.788 -78.851 53.230
71739 93.795 -78.849 53.219
71740 93.802 -78.846 53.207
71741 93.809 -78.844 53.194
71742 93.819 -78.835 53.178
71743 93.821 -78.836 53.162
71744 93.820 -78.836 53.147
71745 93.816 -78.831 53.130
71746 93.812 -78.826 53.111
71747 93.816 -78.821 53.091
71748 93.821 -78.817 53.071
71749 93.826 -78.812 53.048
71750 93.831 -78.808 53.023
71751 93.837 -78.803 52.999
71752 93.842 -78.799 52.973
71753 93.847 -78.795 52.947
71754 93.852 -78.790 52.922
71755 93.867 -78.784 52.897
71756 93.884 -78.777 52.869
71757 93.899 -78.771 52.842
71758 93.915 -78.765 52.815
71759 93.931 -78.758 52.790
71760 93.947 -78.752 52.764
71761 93.963 -78.746 52.738
71762 93.982 -78.746 52.714
71763 94.002 -78.748 52.688
71764 94.023 -78.750 52.664
71765 94.043 -78.752 52.642
71766 94.063 -78.755 52.618
71767 94.076 -78.752 52.595
71768 94.086 -78.749 52.573
71769 94.097 -78.745 52.551
71770 94.107 -78.741 52.527
71771 94.117 -78.737 52.503
71772 94.127 -78.733 52.477
71773 94.137 -78.729 52.451
71774 94.145 -78.730 52.426
71775 94.149 -78.737 52.402
71776 94.169 -78.730 52.379
71777 94.196 -78.734 52.352
71778 94.206 -78.731 52.325
71779 94.216 -78.729 52.298
71780 94.227 -78.726 52.272
71781 94.242 -78.724 52.247
71782 94.256 -78.720 52.222
71783 94.272 -78.709 52.195
71784 94.283 -78.705 52.167
71785 94.291 -78.703 52.139
71786 94.300 -78.701 52.109
71787 94.309 -78.698 52.079
71788 94.318 -78.696 52.051
71789 94.326 -78.694 52.023
71790 94.335 -78.692 51.996
71791 94.343 -78.689 51.966
71792 94.352 -78.687 51.936
71793 94.368 -78.689 51.905
71794 94.382 -78.694 51.874
71795 94.395 -78.698 51.842
71796 94.408 -78.703 51.811
71797 94.421 -78.708 51.781
71798 94.434 -78.713 51.750
71799 94.449 -78.708 51.718
71800 94.467 -78.713 51.687
71801 94.484 -78.717 51.656
71802 94.501 -78.721 51.626
71803 94.526 -78.727 51.596
71804 94.553 -78.734 51.564
71805 94.580 -78.742 51.528
71806 94.607 -78.749 51.491
71807 94.634 -78.755 51.454
71808 94.662 -78.762 51.419
71809 94.687 -78.768 51.385
71810 94.705 -78.771 51.352
71811 94.723 -78.772 51.318
71812 94.741 -78.774 51.284
71813 94.759 -78.777 51.250
71814 94.770 -78.779 51.215
71815 94.778 -78.774 51.178
71816 94.784 -78.764 51.142
71817 94.783 -78.761 51.106
71818 94.779 -78.762 51.069
71819 94.775 -78.764 51.032
71820 94.772 -78.764 50.995
71821 94.756 -78.758 50.957
71822 94.734 -78.747 50.918
71823 94.721 -78.762 50.878
71824 94.710 -78.779 50.836
71825 94.711 -78.783 50.796
71826 94.730 -78.771 50.753
71827 94.749 -78.758 50.709
71828 94.777 -78.749 50.667
71829 94.801 -78.745 50.625
71830 94.819 -78.741 50.582
71831 94.837 -78.738 50.541
71832 94.855 -78.735 50.499
71833 94.873 -78.731 50.457
71834 94.883 -78.736 50.415
71835 94.881 -78.752 50.375
71836 94.880 -78.767 50.336
71837 94.879 -78.783 50.299
71838 94.878 -78.799 50.263
71839 94.877 -78.814 50.226
71840 94.876 -78.830 50.188
71841 94.874 -78.845 50.152
71842 94.867 -78.846 50.114
71843 94.859 -78.846 50.076
71844 94.851 -78.846 50.037
71845 94.843 -78.845 49.998
71846 94.857 -78.848 49.958
71847 94.874 -78.851 49.917
71848 94.890 -78.854 49.876
71849 94.907 -78.857 49.834
71850 94.924 -78.860 49.793
71851 94.941 -78.863 49.755
71852 94.957 -78.866 49.719
71853 94.974 -78.869 49.685
71854 94.998 -78.868 49.650
71855 95.028 -78.874 49.614
71856 95.057 -78.880 49.575
71857 95.087 -78.886 49.537
71858 95.116 -78.893 49.498
71859 95.145 -78.900 49.456
71860 95.175 -78.907 49.415
71861 95.204 -78.914 49.374
71862 95.234 -78.921 49.334
71863 95.263 -78.927 49.295
71864 95.292 -78.934 49.257
71865 95.322 -78.941 49.217
71866 95.349 -78.946 49.176
71867 95.374 -78.949 49.135
71868 95.396 -78.952 49.095
71869 95.404 -78.958 49.051
71870 95.413 -78.963 49.009
71871 95.421 -78.969 48.967
71872 95.429 -78.975 48.924
71873 95.438 -78.981 48.880
71874 95.446 -78.991 48.837
71875 95.450 -79.011 48.795
71876 95.455 -79.031 48.753
71877 95.459 -79.051 48.710
71878 95.464 -79.071 48.666
71879 95.468 -79.091 48.622
71880 95.473 -79.112 48.578
71881 95.478 -79.131 48.535
71882 95.482 -79.152 48.492
71883 95.487 -79.172 48.447
71884 95.491 -79.192 48.403
71885 95.511 -79.187 48.358
71886 95.533 -79.173 48.313
71887 95.556 -79.160 48.268
71888 95.578 -79.147 48.223
71889 95.601 -79.134 48.179
71890 95.623 -79.120 48.134
71891 95.645 -79.107 48.090
71892 95.668 -79.094 48.045
71893 95.690 -79.081 48.000
71894 95.712 -79.068 47.955
71895 95.728 -79.063 47.911
71896 95.740 -79.064 47.867
71897 95.752 -79.065 47.824
71898 95.764 -79.065 47.782
71899 95.776 -79.066 47.741
71900 95.789 -79.067 47.699
71901 95.801 -79.068 47.657
71902 95.813 -79.068 47.616
71903 95.825 -79.069 47.575
71904 95.838 -79.070 47.537
71905 95.850 -79.071 47.497
71906 95.862 -79.071 47.457
71907 95.881 -79.090 47.418
71908 95.901 -79.112 47.378
71909 95.921 -79.134 47.338
71910 95.947 -79.106 47.300
71911 96.035 -79.106 47.261
71912 96.072 -79.101 47.223
71913 96.106 -79.089 47.186
71914 96.140 -79.076 47.149
71915 96.174 -79.063 47.114
71916 96.208 -79.050 47.079
71917 96.243 -79.037 47.043
71918 96.277 -79.024 47.011
71919 96.311 -79.012 46.978
71920 96.345 -78.999 46.945
71921 96.375 -79.034 46.914
71922 96.404 -79.079 46.883
71923 96.432 -79.125 46.852
71924 96.455 -79.153 46.820
71925 96.470 -79.153 46.789
71926 96.484 -79.153 46.758
71927 96.499 -79.154 46.726
71928 96.514 -79.154 46.696
71929 96.532 -79.159 46.664
71930 96.556 -79.172 46.632
71931 96.580 -79.183 46.601
71932 96.593 -79.184 46.570
71933 96.591 -79.166 46.541
71934 96.608 -79.159 46.514
71935 96.645 -79.159 46.484
71936 96.681 -79.160 46.457
71937 96.709 -79.162 46.430
71938 96.731 -79.165 46.402
71939 96.754 -79.167 46.376
71940 96.777 -79.169 46.350
71941 96.800 -79.172 46.323
71942 96.822 -79.175 46.297
71943 96.846 -79.177 46.271
71944 96.865 -79.159 46.245
71945 96.883 -79.137 46.220
71946 96.916 -79.142 46.196
71947 96.919 -79.173 46.172
71948 96.899 -79.212 46.149
71949 96.935 -79.176 46.126
71950 96.970 -79.140 46.103
71951 97.006 -79.103 46.079
71952 97.044 -79.112 46.056
71953 97.087 -79.158 46.036
71954 97.093 -79.169 46.016
71955 97.088 -79.169 45.996
71956 97.084 -79.170 45.977
71957 97.080 -79.171 45.958
71958 97.086 -79.172 45.939
71959 97.096 -79.172 45.919
71960 97.105 -79.172 45.901
71961 97.114 -79.173 45.883
71962 97.124 -79.175 45.864
71963 97.145 -79.163 45.845
71964 97.164 -79.151 45.826
71965 97.171 -79.142 45.808
71966 97.171 -79.143 45.788
71967 97.172 -79.143 45.770
71968 97.173 -79.143 45.753
71969 97.173 -79.144 45.737
71970 97.174 -79.145 45.719
71971 97.175 -79.148 45.702
71972 97.185 -79.143 45.684
71973 97.223 -79.141 45.665
71974 97.249 -79.148 45.648
71975 97.263 -79.165 45.633
71976 97.278 -79.181 45.617
71977 97.292 -79.199 45.600
71978 97.306 -79.216 45.584
71979 97.320 -79.232 45.568
71980 97.334 -79.250 45.552
71981 97.349 -79.266 45.536
71982 97.341 -79.292 45.521
71983 97.296 -79.330 45.506
71984 97.312 -79.324 45.491
71985 97.327 -79.317 45.475
71986 97.343 -79.311 45.460
71987 97.371 -79.297 45.445
71988 97.384 -79.286 45.430
71989 97.397 -79.274 45.416
71990 97.421 -79.281 45.402
71991 97.449 -79.298 45.388
71992 97.468 -79.354 45.374
71993 97.474 -79.363 45.359
71994 97.479 -79.370 45.345
71995 97.486 -79.379 45.331
71996 97.495 -79.388 45.317
71997 97.510 -79.380 45.305
71998 97.526 -79.368 45.292
71999 97.536 -79.359 45.278
72000 97.540 -79.354 45.266
72001 97.544 -79.350 45.253
72002 97.547 -79.345 45.240
72003 97.551 -79.341 45.228
72004 97.559 -79.438 45.215
72005 97.575 -79.462 45.203
72006 97.596 -79.452 45.191
72007 97.613 -79.469 45.178
72008 97.630 -79.485 45.167
72009 97.648 -79.501 45.155
72010 97.665 -79.517 45.144
72011 97.682 -79.534 45.134
72012 97.700 -79.550 45.122
72013 97.717 -79.566 45.110
72014 97.734 -79.582 45.098
72015 97.751 -79.598 45.086
72016 97.769 -79.614 45.074
72017 97.786 -79.629 45.062
72018 97.803 -79.645 45.049
72019 97.820 -79.661 45.037
72020 97.838 -79.677 45.025
72021 97.855 -79.693 45.011
72022 97.872 -79.709 44.999
72023 97.890 -79.725 44.988
72024 97.907 -79.741 44.975
72025 97.924 -79.757 44.963
72026 97.942 -79.773 44.952
72027 97.959 -79.789 44.941
72028 97.976 -79.805 44.931
72029 97.994 -79.821 44.921
72030 98.011 -79.837 44.912
72031 98.028 -79.853 44.903
72032 98.045 -79.869 44.893
72033 98.063 -79.885 44.884
72034 98.080 -79.901 44.877
72035 98.097 -79.917 44.868
72036 98.114 -79.933 44.860
72037 98.134 -79.948 44.852
72038 98.158 -79.962 44.845
72039 98.181 -79.977 44.838
72040 98.205 -79.991 44.831
72041 98.229 -80.005 44.823
72042 98.253 -80.020 44.817
72043 98.276 -80.034 44.810
72044 98.302 -80.047 44.804
72045 98.334 -80.046 44.796
72046 98.346 -80.048 44.790
72047 98.387 -80.033 44.783
72048 98.434 -80.044 44.775
72049 98.462 -80.059 44.768
72050 98.489 -80.074 44.762
72051 98.517 -80.089 44.757
72052 98.544 -80.104 44.752
72053 98.572 -80.119 44.746
72054 98.600 -80.127 44.739
72055 98.630 -80.130 44.731
72056 98.655 -80.133 44.727
72057 98.679 -80.135 44.723
72058 98.704 -80.138 44.717
72059 98.728 -80.142 44.711
72060 98.754 -80.152 44.704
72061 98.776 -80.163 44.694
72062 98.789 -80.177 44.679
72063 98.808 -80.188 44.665
72064 98.831 -80.187 44.652
72065 98.839 -80.182 44.641
72066 98.847 -80.188 44.632
72067 98.863 -80.200 44.624
72068 98.880 -80.211 44.616
72069 98.896 -80.222 44.609
72070 98.929 -80.231 44.602
72071 98.997 -80.250 44.596
72072 99.021 -80.260 44.590
72073 99.044 -80.271 44.584
72074 99.067 -80.281 44.579
72075 99.090 -80.292 44.573
72076 99.114 -80.303 44.568
72077 99.137 -80.312 44.563
72078 99.160 -80.323 44.557
72079 99.184 -80.334 44.552
72080 99.207 -80.344 44.548
72081 99.230 -80.355 44.544
72082 99.254 -80.366 44.540
72083 99.277 -80.376 44.537
72084 99.300 -80.387 44.534
72085 99.326 -80.397 44.530
72086 99.364 -80.400 44.526
72087 99.403 -80.404 44.523
72088 99.441 -80.408 44.520
72089 99.480 -80.413 44.516
72090 99.518 -80.416 44.512
72091 99.557 -80.420 44.509
72410 93.504 -78.968 53.481
72411 93.513 -78.965 53.470
72412 93.523 -78.963 53.460
72413 93.533 -78.959 53.450
72414 93.545 -78.949 53.441
72415 93.554 -78.940 53.432
72416 93.562 -78.933 53.423
72417 93.570 -78.927 53.415
72418 93.578 -78.920 53.408
72419 93.587 -78.913 53.399
72420 93.595 -78.906 53.391
72421 93.604 -78.899 53.384
72422 93.643 -78.878 53.377
72423 93.660 -78.860 53.371
72424 93.656 -78.858 53.366
72425 93.668 -78.856 53.361
72426 93.678 -78.855 53.356
72427 93.688 -78.854 53.351
72428 93.697 -78.852 53.346
72429 93.707 -78.852 53.341
72430 93.716 -78.850 53.336
72431 93.726 -78.849 53.330
72432 93.736 -78.848 53.322
72433 93.743 -78.846 53.313
72434 93.750 -78.844 53.304
72435 93.757 -78.842 53.295
72436 93.764 -78.839 53.283
72437 93.771 -78.837 53.271
72438 93.778 -78.835 53.259
72439 93.785 -78.833 53.247
72440 93.792 -78.830 53.236
72441 93.799 -78.827 53.223
72442 93.806 -78.825 53.209
72443 93.813 -78.821 53.193
72444 93.811 -78.819 53.176
72445 93.812 -78.814 53.157
72446 93.818 -78.810 53.140
72447 93.823 -78.805 53.120
72448 93.828 -78.801 53.101
72449 93.833 -78.796 53.081
72450 93.839 -78.793 53.057
72451 93.844 -78.788 53.033
72452 93.849 -78.783 53.007
72453 93.854 -78.779 52.981
72454 93.860 -78.774 52.954
72455 93.867 -78.770 52.929
72456 93.883 -78.764 52.904
72457 93.899 -78.757 52.876
72458 93.915 -78.751 52.849
72459 93.930 -78.745 52.822
72460 93.946 -78.738 52.797
72461 93.963 -78.732 52.772
72462 93.978 -78.726 52.747
72463 93.994 -78.719 52.723
72464 94.010 -78.713 52.699
72465 94.030 -78.714 52.675
72466 94.050 -78.716 52.653
72467 94.070 -78.718 52.630
72468 94.084 -78.717 52.607
72469 94.095 -78.713 52.585
72470 94.105 -78.709 52.563
72471 94.115 -78.705 52.540
72472 94.125 -78.701 52.516
72473 94.130 -78.707 52.491
72474 94.134 -78.715 52.465
72475 94.139 -78.723 52.439
72476 94.153 -78.702 52.414
72477 94.205 -78.715 52.391
72478 94.219 -78.714 52.365
72479 94.229 -78.711 52.338
72480 94.239 -78.708 52.311
72481 94.249 -78.706 52.286
72482 94.259 -78.703 52.262
72483 94.270 -78.701 52.238
72484 94.281 -78.696 52.212
72485 94.290 -78.694 52.183
72486 94.299 -78.692 52.154
72487 94.308 -78.689 52.125
72488 94.316 -78.687 52.096
72489 94.325 -78.685 52.069
72490 94.334 -78.683 52.042
72491 94.342 -78.680 52.014
72492 94.351 -78.678 51.986
72493 94.359 -78.676 51.956
72494 94.372 -78.677 51.927
72495 94.385 -78.683 51.895
72496 94.398 -78.687 51.863
72497 94.411 -78.692 51.831
72498 94.424 -78.696 51.799
72499 94.437 -78.702 51.769
72500 94.452 -78.700 51.739
72501 94.469 -78.704 51.708
72502 94.486 -78.708 51.675
72503 94.509 -78.714 51.642
72504 94.537 -78.721 51.609
72505 94.564 -78.728 51.576
72506 94.591 -78.735 51.541
72507 94.618 -78.742 51.503
72508 94.636 -78.744 51.467
72509 94.654 -78.745 51.432
72510 94.672 -78.748 51.398
72511 94.690 -78.750 51.364
72512 94.708 -78.752 51.332
72513 94.726 -78.755 51.298
72514 94.745 -78.757 51.265
72515 94.755 -78.759 51.232
72516 94.766 -78.761 51.196
72517 94.773 -78.754 51.159
72518 94.779 -78.744 51.124
72519 94.784 -78.733 51.085
72520 94.783 -78.730 51.046
72521 94.779 -78.732 51.006
72522 94.765 -78.726 50.968
72523 94.750 -78.734 50.930
72524 94.739 -78.751 50.891
72525 94.727 -78.767 50.850
72526 94.726 -78.774 50.810
72527 94.745 -78.761 50.767
72528 94.764 -78.749 50.722
72529 94.781 -78.736 50.680
72530 94.799 -78.733 50.638
72531 94.817 -78.730 50.596
72532 94.835 -78.727 50.555
72533 94.853 -78.724 50.513
72534 94.872 -78.720 50.471
72535 94.881 -78.725 50.427
72536 94.880 -78.740 50.384
72537 94.879 -78.756 50.344
72538 94.878 -78.771 50.306
72539 94.877 -78.786 50.271
72540 94.875 -78.802 50.234
72541 94.874 -78.818 50.198
72542 94.873 -78.833 50.161
72543 94.872 -78.849 50.124
72544 94.864 -78.849 50.087
72545 94.856 -78.848 50.048
72546 94.848 -78.848 50.008
72547 94.857 -78.849 49.969
72548 94.875 -78.852 49.929
72549 94.891 -78.855 49.888
72550 94.908 -78.858 49.846
72551 94.924 -78.861 49.805
72552 94.942 -78.865 49.767
72553 94.958 -78.868 49.731
72554 94.987 -78.863 49.696
72555 95.016 -78.858 49.660
72556 95.045 -78.858 49.623
72557 95.074 -78.865 49.584
72558 95.103 -78.870 49.544
72559 95.132 -78.876 49.505
72560 95.162 -78.881 49.462
72561 95.191 -78.887 49.421
72562 95.220 -78.893 49.380
72563 95.250 -78.899 49.341
72564 95.279 -78.906 49.302
72565 95.309 -78.913 49.263
72566 95.338 -78.920 49.224
72567 95.368 -78.927 49.183
72568 95.393 -78.930 49.142
72569 95.405 -78.935 49.101
72570 95.413 -78.940 49.057
72571 95.421 -78.946 49.016
72572 95.430 -78.952 48.975
72573 95.439 -78.958 48.932
72574 95.447 -78.964 48.889
72575 95.456 -78.970 48.847
72576 95.460 -78.990 48.805
72577 95.465 -79.009 48.763
72578 95.469 -79.030 48.721
72579 95.474 -79.049 48.677
72580 95.479 -79.070 48.633
72581 95.483 -79.090 48.589
72582 95.488 -79.110 48.546
72583 95.492 -79.130 48.503
72584 95.497 -79.150 48.458
72585 95.511 -79.165 48.414
72586 95.526 -79.178 48.370
72587 95.548 -79.166 48.324
72588 95.570 -79.153 48.278
72589 95.593 -79.140 48.232
72590 95.615 -79.127 48.186
72591 95.637 -79.114 48.141
72592 95.660 -79.100 48.095
72593 95.682 -79.087 48.050
72594 95.705 -79.074 48.004
72595 95.725 -79.062 47.960
72596 95.734 -79.059 47.916
72597 95.741 -79.056 47.872
72598 95.747 -79.052 47.830
72599 95.753 -79.048 47.788
72600 95.760 -79.045 47.747
72601 95.772 -79.046 47.704
72602 95.784 -79.046 47.662
72603 95.797 -79.047 47.620
72604 95.809 -79.048 47.579
72605 95.821 -79.049 47.540
72606 95.834 -79.049 47.500
72607 95.846 -79.050 47.460
72608 95.859 -79.054 47.420
72609 95.879 -79.075 47.380
72610 95.899 -79.093 47.340
72611 95.941 -79.084 47.302
72612 96.037 -79.088 47.263
72613 96.077 -79.091 47.225
72614 96.114 -79.087 47.188
72615 96.148 -79.075 47.151
72616 96.182 -79.062 47.116
72617 96.217 -79.049 47.081
72618 96.251 -79.036 47.044
72619 96.285 -79.024 47.012
72620 96.320 -79.011 46.979
72621 96.354 -78.998 46.947
72622 96.383 -79.033 46.915
72623 96.412 -79.079 46.884
72624 96.440 -79.125 46.853
72625 96.457 -79.132 46.822
72626 96.472 -79.132 46.790
72627 96.486 -79.133 46.759
72628 96.501 -79.134 46.728
72629 96.520 -79.139 46.697
72630 96.544 -79.151 46.665
72631 96.568 -79.162 46.633
72632 96.591 -79.175 46.601
72633 96.614 -79.185 46.571
72634 96.612 -79.169 46.541
72635 96.610 -79.152 46.514
72636 96.617 -79.138 46.486
72637 96.653 -79.139 46.458
72638 96.690 -79.140 46.431
72639 96.726 -79.141 46.403
72640 96.762 -79.142 46.377
72641 96.785 -79.144 46.351
72642 96.807 -79.147 46.326
72643 96.830 -79.150 46.299
72644 96.853 -79.152 46.274
72645 96.872 -79.135 46.249
72646 96.901 -79.131 46.224
72647 96.892 -79.168 46.200
72648 96.869 -79.212 46.177
72649 96.904 -79.175 46.154
72650 96.940 -79.140 46.130
72651 96.975 -79.103 46.106
72652 97.011 -79.067 46.082
72653 97.052 -79.107 46.060
72654 97.094 -79.152 46.040
72655 97.095 -79.158 46.020
72656 97.091 -79.159 46.000
72657 97.087 -79.160 45.981
72658 97.082 -79.161 45.962
72659 97.082 -79.162 45.944
72660 97.091 -79.162 45.924
72661 97.100 -79.163 45.905
72662 97.104 -79.171 45.887
72663 97.108 -79.174 45.869
72664 97.124 -79.161 45.850
72665 97.145 -79.146 45.831
72666 97.168 -79.127 45.812
72667 97.184 -79.219 45.793
72668 97.181 -79.128 45.775
72669 97.180 -79.098 45.759
72670 97.181 -79.099 45.743
72671 97.170 -79.112 45.725
72672 97.168 -79.148 45.708
72673 97.178 -79.147 45.690
72674 97.213 -79.146 45.671
72675 97.250 -79.144 45.655
72676 97.281 -79.147 45.639
72677 97.295 -79.165 45.623
72678 97.309 -79.181 45.607
72679 97.323 -79.198 45.591
72680 97.337 -79.215 45.575
72681 97.352 -79.232 45.559
72682 97.366 -79.249 45.544
72683 97.330 -79.286 45.529
72684 97.300 -79.314 45.513
72685 97.315 -79.307 45.498
72686 97.331 -79.301 45.483
72687 97.347 -79.294 45.468
72688 97.370 -79.284 45.452
72689 97.383 -79.272 45.438
72690 97.404 -79.273 45.425
72691 97.432 -79.285 45.410
72692 97.461 -79.328 45.396
72693 97.468 -79.341 45.382
72694 97.473 -79.349 45.368
72695 97.479 -79.357 45.353
72696 97.485 -79.365 45.339
72697 97.493 -79.374 45.325
72698 97.508 -79.363 45.313
72699 97.514 -79.357 45.300
72700 97.518 -79.352 45.287
72701 97.522 -79.347 45.274
72702 97.525 -79.343 45.262
72703 97.529 -79.338 45.250
72704 97.533 -79.334 45.237
72705 97.537 -79.329 45.225
72706 97.543 -79.410 45.213
72707 97.566 -79.400 45.200
72708 97.585 -79.435 45.188
72709 97.605 -79.469 45.177
72710 97.620 -79.523 45.166
72711 97.635 -79.524 45.154
72712 97.649 -79.525 45.143
72713 97.666 -79.540 45.131
72714 97.683 -79.556 45.120
72715 97.701 -79.572 45.108
72716 97.718 -79.588 45.097
72717 97.735 -79.604 45.085
72718 97.752 -79.620 45.073
72719 97.770 -79.636 45.061
72720 97.787 -79.651 45.048
72721 97.804 -79.667 45.036
72722 97.821 -79.683 45.023
72723 97.839 -79.699 45.010
72724 97.856 -79.715 44.998
72725 97.873 -79.731 44.987
72726 97.891 -79.747 44.975
72727 97.908 -79.763 44.964
72728 97.925 -79.779 44.954
72729 97.943 -79.795 44.944
72730 97.960 -79.811 44.934
72731 97.977 -79.827 44.925
72732 97.995 -79.843 44.916
72733 98.012 -79.859 44.907
72734 98.029 -79.875 44.898
72735 98.054 -79.889 44.891
72736 98.077 -79.903 44.883
72737 98.101 -79.918 44.875
72738 98.125 -79.932 44.868
72739 98.148 -79.947 44.861
72740 98.171 -79.961 44.855
72741 98.196 -79.974 44.849
72742 98.222 -79.987 44.842
72743 98.253 -79.991 44.836
72744 98.266 -79.993 44.829
72745 98.278 -79.994 44.821
72746 98.290 -79.996 44.813
72747 98.319 -80.002 44.805
72748 98.371 -80.012 44.797
72749 98.423 -80.021 44.790
72750 98.467 -80.032 44.783
72751 98.495 -80.047 44.778
72752 98.522 -80.062 44.774
72753 98.550 -80.077 44.767
72754 98.578 -80.091 44.760
72755 98.607 -80.094 44.752
72756 98.637 -80.098 44.744
72757 98.667 -80.101 44.739
72758 98.696 -80.104 44.736
72759 98.720 -80.107 44.731
72760 98.742 -80.117 44.722
72761 98.756 -80.131 44.712
72762 98.769 -80.144 44.702
72763 98.782 -80.157 44.687
72764 98.807 -80.168 44.674
72765 98.833 -80.175 44.662
72766 98.849 -80.159 44.652
72767 98.859 -80.156 44.643
72768 98.876 -80.167 44.634
72769 98.892 -80.178 44.627
72770 98.909 -80.190 44.620
72771 98.926 -80.201 44.613
72772 98.942 -80.212 44.606
72773 98.995 -80.219 44.600
72774 99.048 -80.243 44.594
72775 99.071 -80.254 44.589
72776 99.094 -80.265 44.584
72777 99.118 -80.275 44.578
72778 99.141 -80.286 44.572
72779 99.164 -80.296 44.567
72780 99.188 -80.306 44.562
72781 99.211 -80.317 44.558
72782 99.234 -80.328 44.554
72783 99.258 -80.338 44.550
72784 99.281 -80.349 44.546
72785 99.304 -80.359 44.543
72786 99.336 -80.366 44.540
72787 99.375 -80.370 44.536
72788 99.413 -80.374 44.533
72789 99.452 -80.378 44.529
72790 99.491 -80.382 44.525
72791 99.529 -80.386 44.521
72792 99.564 -80.391 44.518
72793 99.567 -80.414 44.514
72794 99.570 -80.436 44.510
72795 99.574 -80.458 44.506
73112 93.513 -78.955 53.486
73113 93.523 -78.952 53.475
73114 93.533 -78.949 53.465
73115 93.543 -78.944 53.455
73116 93.555 -78.934 53.447
73117 93.563 -78.927 53.439
73118 93.571 -78.921 53.430
73119 93.580 -78.913 53.421
73120 93.588 -78.906 53.413
73121 93.596 -78.899 53.405
73122 93.605 -78.893 53.397
73123 93.631 -78.877 53.389
73124 93.697 -78.845 53.383
73125 93.666 -78.841 53.378
73126 93.662 -78.836 53.374
73127 93.672 -78.833 53.369
73128 93.682 -78.832 53.365
73129 93.691 -78.830 53.362
73130 93.700 -78.828 53.357
73131 93.710 -78.827 53.352
73132 93.721 -78.830 53.345
73133 93.732 -78.839 53.337
73134 93.739 -78.836 53.329
73135 93.746 -78.833 53.320
73136 93.753 -78.830 53.311
73137 93.760 -78.827 53.298
73138 93.767 -78.825 53.286
73139 93.774 -78.822 53.274
73140 93.781 -78.819 53.262
73141 93.788 -78.816 53.250
73142 93.795 -78.814 53.237
73143 93.802 -78.811 53.223
73144 93.814 -78.803 53.206
73145 93.820 -78.799 53.187
73146 93.825 -78.794 53.168
73147 93.830 -78.789 53.149
73148 93.835 -78.786 53.128
73149 93.841 -78.781 53.107
73150 93.846 -78.777 53.087
73151 93.851 -78.772 53.063
73152 93.856 -78.767 53.039
73153 93.862 -78.763 53.014
73154 93.867 -78.759 52.988
73155 93.872 -78.755 52.962
73156 93.883 -78.749 52.936
73157 93.898 -78.742 52.910
73158 93.914 -78.736 52.883
73159 93.930 -78.730 52.856
73160 93.946 -78.724 52.830
73161 93.962 -78.717 52.805
73162 93.978 -78.711 52.779
73163 93.994 -78.705 52.755
73164 94.009 -78.698 52.732
73165 94.026 -78.692 52.708
73166 94.041 -78.686 52.685
73167 94.057 -78.680 52.664
73168 94.077 -78.681 52.641
73169 94.092 -78.681 52.619
73170 94.103 -78.677 52.596
73171 94.111 -78.677 52.573
73172 94.115 -78.685 52.551
73173 94.120 -78.692 52.528
73174 94.124 -78.700 52.504
73175 94.128 -78.708 52.478
73176 94.144 -78.684 52.452
73177 94.189 -78.686 52.426
73178 94.232 -78.696 52.402
73179 94.242 -78.694 52.377
73180 94.252 -78.691 52.351
73181 94.262 -78.689 52.325
73182 94.272 -78.686 52.300
73183 94.281 -78.683 52.277
73184 94.282 -78.688 52.254
73185 94.290 -78.685 52.228
73186 94.299 -78.683 52.199
73187 94.308 -78.680 52.170
73188 94.316 -78.678 52.142
73189 94.325 -78.676 52.114
73190 94.333 -78.673 52.087
73191 94.342 -78.671 52.061
73192 94.351 -78.669 52.034
73193 94.359 -78.667 52.006
73194 94.368 -78.664 51.977
73195 94.380 -78.665 51.948
73196 94.390 -78.670 51.917
73197 94.402 -78.676 51.885
73198 94.415 -78.680 51.853
73199 94.428 -78.685 51.820
73200 94.441 -78.690 51.790
73201 94.455 -78.694 51.760
73202 94.472 -78.695 51.729
73203 94.494 -78.701 51.696
73204 94.521 -78.708 51.661
73205 94.548 -78.714 51.625
73206 94.567 -78.717 51.590
73207 94.585 -78.720 51.553
73208 94.603 -78.721 51.515
73209 94.621 -78.724 51.479
73210 94.639 -78.726 51.444
73211 94.657 -78.728 51.411
73212 94.676 -78.730 51.378
73213 94.694 -78.733 51.346
73214 94.712 -78.735 51.314
73215 94.730 -78.737 51.282
73216 94.740 -78.739 51.250
73217 94.751 -78.741 51.215
73218 94.762 -78.743 51.178
73219 94.768 -78.733 51.141
73220 94.773 -78.724 51.102
73221 94.779 -78.713 51.062
73222 94.784 -78.702 51.022
73223 94.779 -78.706 50.983
73224 94.767 -78.723 50.944
73225 94.756 -78.739 50.905
73226 94.744 -78.756 50.865
73227 94.742 -78.764 50.823
73228 94.761 -78.751 50.780
73229 94.778 -78.739 50.735
73230 94.787 -78.730 50.692
73231 94.796 -78.722 50.651
73232 94.815 -78.719 50.610
73233 94.833 -78.715 50.569
73234 94.851 -78.712 50.528
73235 94.870 -78.709 50.485
73236 94.880 -78.713 50.440
73237 94.879 -78.729 50.397
73238 94.878 -78.744 50.355
73239 94.876 -78.760 50.315
73240 94.875 -78.775 50.278
73241 94.874 -78.790 50.241
73242 94.873 -78.806 50.204
73243 94.872 -78.821 50.167
73244 94.871 -78.837 50.130
73245 94.868 -78.851 50.092
73246 94.860 -78.851 50.052
73247 94.853 -78.850 50.014
73248 94.858 -78.852 49.975
73249 94.875 -78.855 49.935
73250 94.892 -78.858 49.895
73251 94.909 -78.861 49.855
73252 94.925 -78.864 49.817
73253 94.947 -78.864 49.781
73254 94.975 -78.859 49.745
73255 95.004 -78.855 49.708
73256 95.033 -78.850 49.670
73257 95.061 -78.846 49.632
73258 95.091 -78.849 49.591
73259 95.120 -78.855 49.550
73260 95.149 -78.861 49.509
73261 95.179 -78.867 49.467
73262 95.208 -78.872 49.426
73263 95.237 -78.878 49.386
73264 95.266 -78.883 49.346
73265 95.296 -78.890 49.307
73266 95.325 -78.895 49.268
73267 95.354 -78.901 49.230
73268 95.384 -78.906 49.190
73269 95.405 -78.912 49.149
73270 95.414 -78.918 49.106
73271 95.422 -78.924 49.063
73272 95.430 -78.930 49.022
73273 95.439 -78.935 48.981
73274 95.448 -78.941 48.939
73275 95.456 -78.947 48.897
73276 95.465 -78.952 48.855
73277 95.471 -78.968 48.813
73278 95.475 -78.988 48.771
73279 95.480 -79.008 48.729
73280 95.484 -79.028 48.686
73281 95.489 -79.048 48.640
73282 95.493 -79.068 48.598
73283 95.498 -79.088 48.556
73284 95.502 -79.109 48.513
73285 95.516 -79.123 48.469
73286 95.531 -79.137 48.425
73287 95.546 -79.150 48.380
73288 95.562 -79.160 48.334
73289 95.585 -79.147 48.286
73290 95.607 -79.134 48.239
73291 95.629 -79.120 48.192
73292 95.652 -79.107 48.146
73293 95.674 -79.094 48.100
73294 95.697 -79.081 48.055
73295 95.719 -79.068 48.009
73296 95.733 -79.061 47.964
73297 95.742 -79.057 47.920
73298 95.750 -79.053 47.877
73299 95.756 -79.049 47.836
73300 95.763 -79.046 47.795
73301 95.769 -79.043 47.753
73302 95.776 -79.039 47.711
73303 95.782 -79.035 47.667
73304 95.789 -79.031 47.625
73305 95.795 -79.028 47.583
73306 95.805 -79.027 47.543
73307 95.817 -79.028 47.502
73308 95.830 -79.028 47.462
73309 95.842 -79.029 47.422
73310 95.857 -79.038 47.382
73311 95.877 -79.043 47.343
73312 95.943 -79.061 47.304
73313 96.030 -79.071 47.266
73314 96.079 -79.073 47.228
73315 96.118 -79.076 47.190
73316 96.156 -79.073 47.153
73317 96.190 -79.060 47.118
73318 96.225 -79.047 47.083
73319 96.259 -79.035 47.047
73320 96.294 -79.022 47.013
73321 96.328 -79.009 46.981
73322 96.362 -78.996 46.949
73323 96.391 -79.033 46.918
73324 96.419 -79.078 46.887
73325 96.444 -79.111 46.856
73326 96.459 -79.112 46.824
73327 96.474 -79.112 46.792
73328 96.488 -79.112 46.760
73329 96.508 -79.119 46.729
73330 96.532 -79.131 46.697
73331 96.555 -79.143 46.666
73332 96.579 -79.154 46.635
73333 96.603 -79.166 46.603
73334 96.626 -79.178 46.572
73335 96.633 -79.170 46.543
73336 96.631 -79.153 46.515
73337 96.629 -79.137 46.488
73338 96.627 -79.120 46.460
73339 96.662 -79.119 46.433
73340 96.698 -79.120 46.406
73341 96.735 -79.121 46.379
73342 96.771 -79.122 46.353
73343 96.808 -79.123 46.328
73344 96.838 -79.125 46.304
73345 96.861 -79.127 46.279
73346 96.885 -79.118 46.254
73347 96.865 -79.162 46.229
73348 96.840 -79.209 46.206
73349 96.874 -79.175 46.183
73350 96.909 -79.139 46.160
73351 96.945 -79.103 46.135
73352 96.980 -79.067 46.110
73353 97.018 -79.057 46.086
73354 97.060 -79.102 46.065
73355 97.102 -79.147 46.045
73356 97.098 -79.148 46.026
73357 97.094 -79.150 46.006
73358 97.090 -79.150 45.986
73359 97.085 -79.151 45.967
73360 97.074 -79.146 45.948
73361 97.076 -79.167 45.929
73362 97.079 -79.175 45.910
73363 97.082 -79.184 45.892
73364 97.088 -79.173 45.874
73365 97.104 -79.159 45.856
73366 97.126 -79.140 45.837
73367 97.149 -79.118 45.818
73368 97.171 -79.127 45.800
73369 97.191 -79.244 45.782
73370 97.196 -79.235 45.766
73371 97.197 -79.145 45.750
73372 97.170 -79.119 45.732
73373 97.149 -79.137 45.714
73374 97.176 -79.153 45.696
73375 97.203 -79.150 45.679
73376 97.240 -79.148 45.662
73377 97.278 -79.147 45.646
73378 97.312 -79.147 45.630
73379 97.326 -79.164 45.615
73380 97.341 -79.181 45.599
73381 97.355 -79.198 45.583
73382 97.369 -79.215 45.567
73383 97.375 -79.235 45.551
73384 97.320 -79.280 45.537
73385 97.304 -79.297 45.521
73386 97.319 -79.291 45.506
73387 97.334 -79.285 45.491
73388 97.350 -79.278 45.477
73389 97.369 -79.270 45.462
73390 97.387 -79.266 45.447
73391 97.414 -79.277 45.434
73392 97.442 -79.289 45.419
73393 97.462 -79.320 45.405
73394 97.467 -79.328 45.391
73395 97.473 -79.336 45.376
73396 97.478 -79.344 45.362
73397 97.484 -79.352 45.349
73398 97.491 -79.360 45.335
73399 97.495 -79.356 45.322
73400 97.499 -79.352 45.309
73401 97.503 -79.347 45.297
73402 97.507 -79.343 45.284
73403 97.511 -79.338 45.271
73404 97.515 -79.334 45.259
73405 97.519 -79.330 45.247
73406 97.523 -79.325 45.234
73407 97.527 -79.321 45.222
73408 97.540 -79.342 45.210
73409 97.557 -79.372 45.198
73410 97.576 -79.407 45.187
73411 97.595 -79.441 45.176
73412 97.614 -79.476 45.164
73413 97.634 -79.520 45.153
73414 97.648 -79.607 45.142
73415 97.662 -79.607 45.130
73416 97.676 -79.609 45.119
73417 97.691 -79.610 45.107
73418 97.705 -79.611 45.096
73419 97.719 -79.612 45.084
73420 97.736 -79.626 45.072
73421 97.753 -79.642 45.060
73422 97.771 -79.657 45.047
73423 97.788 -79.673 45.035
73424 97.805 -79.689 45.022
73425 97.822 -79.705 45.010
73426 97.840 -79.721 44.999
73427 97.857 -79.737 44.988
73428 97.874 -79.754 44.977
73429 97.892 -79.770 44.967
73430 97.909 -79.786 44.958
73431 97.926 -79.802 44.948
73432 97.949 -79.816 44.939
73433 97.973 -79.830 44.931
73434 97.997 -79.845 44.922
73435 98.020 -79.859 44.913
73436 98.043 -79.874 44.906
73437 98.066 -79.889 44.899
73438 98.090 -79.902 44.892
73439 98.116 -79.914 44.886
73440 98.143 -79.927 44.880
73441 98.171 -79.935 44.874
73442 98.184 -79.936 44.868
73443 98.199 -79.939 44.862
73444 98.222 -79.948 44.855
73445 98.246 -79.955 44.847
73446 98.270 -79.964 44.838
73447 98.295 -79.969 44.828
73448 98.324 -79.977 44.819
73449 98.362 -79.989 44.811
73450 98.406 -79.999 44.804
73451 98.458 -80.009 44.798
73452 98.501 -80.021 44.794
73453 98.528 -80.036 44.790
73454 98.556 -80.051 44.783
73455 98.585 -80.059 44.775
73456 98.614 -80.062 44.765
73457 98.644 -80.065 44.755
73458 98.673 -80.068 44.748
73459 98.703 -80.073 44.743
73460 98.723 -80.083 44.737
73461 98.735 -80.098 44.727
73462 98.748 -80.112 44.717
73463 98.761 -80.124 44.706
73464 98.779 -80.137 44.693
73465 98.805 -80.147 44.682
73466 98.833 -80.156 44.672
73467 98.861 -80.145 44.662
73468 98.875 -80.131 44.653
73469 98.889 -80.135 44.645
73470 98.905 -80.146 44.638
73471 98.922 -80.157 44.631
73472 98.938 -80.168 44.624
73473 98.955 -80.180 44.617
73474 98.971 -80.191 44.610
73475 99.001 -80.199 44.604
73476 99.075 -80.227 44.599
73477 99.098 -80.237 44.594
73478 99.122 -80.248 44.588
73479 99.145 -80.259 44.582
73480 99.168 -80.269 44.577
73481 99.192 -80.279 44.572
73482 99.215 -80.290 44.568
73483 99.238 -80.300 44.563
73484 99.262 -80.311 44.559
73485 99.285 -80.322 44.556
73486 99.308 -80.332 44.552
73487 99.347 -80.336 44.549
73488 99.385 -80.340 44.546
73489 99.424 -80.344 44.542
73490 99.463 -80.348 44.538
73491 99.501 -80.352 44.535
73492 99.540 -80.356 44.531
73493 99.561 -80.369 44.527
73494 99.564 -80.391 44.524
73495 99.567 -80.413 44.519
73496 99.570 -80.435 44.515
73497 99.573 -80.457 44.510
73498 99.577 -80.479 44.505
73814 93.523 -78.941 53.490
73815 93.533 -78.939 53.479
73816 93.543 -78.937 53.470
73817 93.553 -78.930 53.461
73818 93.564 -78.921 53.452
73819 93.572 -78.914 53.443
73820 93.581 -78.907 53.435
73821 93.589 -78.900 53.426
73822 93.597 -78.893 53.418
73823 93.606 -78.886 53.410
73824 93.619 -78.877 53.403
73825 93.688 -78.843 53.397
73826 93.703 -78.826 53.391
73827 93.699 -78.820 53.386
73828 93.695 -78.814 53.383
73829 93.691 -78.808 53.380
73830 93.687 -78.802 53.377
73831 93.694 -78.805 53.372
73832 93.707 -78.818 53.367
73833 93.720 -78.830 53.361
73834 93.728 -78.831 53.352
73835 93.735 -78.829 53.343
73836 93.742 -78.826 53.335
73837 93.749 -78.823 53.325
73838 93.756 -78.821 53.313
73839 93.763 -78.818 53.300
73840 93.770 -78.814 53.288
73841 93.777 -78.811 53.275
73842 93.784 -78.809 53.262
73843 93.791 -78.806 53.248
73844 93.807 -78.797 53.235
73845 93.813 -78.789 53.216
73846 93.827 -78.781 53.197
73847 93.837 -78.774 53.177
73848 93.843 -78.770 53.157
73849 93.848 -78.765 53.136
73850 93.853 -78.761 53.114
73851 93.858 -78.757 53.093
73852 93.863 -78.752 53.069
73853 93.869 -78.748 53.045
73854 93.874 -78.743 53.020
73855 93.880 -78.739 52.995
73856 93.885 -78.734 52.969
73857 93.898 -78.729 52.943
73858 93.914 -78.722 52.918
73859 93.930 -78.716 52.890
73860 93.945 -78.710 52.864
73861 93.962 -78.703 52.839
73862 93.977 -78.697 52.813
73863 93.993 -78.691 52.788
73864 94.009 -78.685 52.764
73865 94.025 -78.678 52.742
73866 94.041 -78.672 52.718
73867 94.057 -78.666 52.696
73868 94.073 -78.659 52.675
73869 94.088 -78.653 52.653
73870 94.097 -78.655 52.631
73871 94.101 -78.662 52.608
73872 94.105 -78.670 52.585
73873 94.109 -78.678 52.561
73874 94.114 -78.686 52.537
73875 94.118 -78.694 52.512
73876 94.135 -78.665 52.488
73877 94.174 -78.658 52.463
73878 94.225 -78.671 52.438
73879 94.255 -78.677 52.414
73880 94.265 -78.673 52.389
73881 94.275 -78.671 52.364
73882 94.285 -78.668 52.339
73883 94.294 -78.666 52.315
73884 94.296 -78.670 52.292
73885 94.288 -78.682 52.269
73886 94.293 -78.681 52.244
73887 94.308 -78.671 52.215
73888 94.317 -78.669 52.186
73889 94.325 -78.667 52.159
73890 94.334 -78.664 52.132
73891 94.343 -78.662 52.105
73892 94.351 -78.660 52.079
73893 94.360 -78.658 52.054
73894 94.368 -78.655 52.026
73895 94.377 -78.653 51.998
73896 94.391 -78.652 51.969
73897 94.403 -78.658 51.939
73898 94.413 -78.662 51.907
73899 94.424 -78.667 51.874
73900 94.435 -78.673 51.842
73901 94.445 -78.678 51.811
73902 94.458 -78.683 51.781
73903 94.478 -78.688 51.750
73904 94.498 -78.691 51.716
73905 94.516 -78.693 51.680
73906 94.534 -78.695 51.642
73907 94.552 -78.697 51.604
73908 94.570 -78.699 51.566
73909 94.589 -78.702 51.528
73910 94.607 -78.704 51.492
73911 94.625 -78.706 51.458
73912 94.643 -78.708 51.425
73913 94.661 -78.711 51.392
73914 94.679 -78.712 51.360
73915 94.697 -78.714 51.330
73916 94.714 -78.717 51.299
73917 94.725 -78.719 51.266
73918 94.736 -78.721 51.232
73919 94.747 -78.724 51.197
73920 94.757 -78.724 51.159
73921 94.762 -78.714 51.119
73922 94.768 -78.703 51.079
73923 94.773 -78.696 51.038
73924 94.776 -78.705 50.998
73925 94.780 -78.714 50.957
73926 94.773 -78.728 50.918
73927 94.761 -78.745 50.879
73928 94.758 -78.754 50.837
73929 94.776 -78.741 50.793
73930 94.785 -78.733 50.748
73931 94.794 -78.724 50.705
73932 94.803 -78.716 50.663
73933 94.813 -78.708 50.623
73934 94.831 -78.705 50.583
73935 94.849 -78.701 50.540
73936 94.867 -78.698 50.497
73937 94.879 -78.702 50.452
73938 94.877 -78.717 50.408
73939 94.876 -78.733 50.365
73940 94.875 -78.748 50.323
73941 94.874 -78.764 50.285
73942 94.873 -78.779 50.247
73943 94.872 -78.794 50.209
73944 94.870 -78.810 50.172
73945 94.869 -78.825 50.134
73946 94.867 -78.841 50.095
73947 94.865 -78.853 50.056
73948 94.857 -78.853 50.017
73949 94.859 -78.854 49.977
73950 94.876 -78.857 49.938
73951 94.893 -78.860 49.899
73952 94.910 -78.863 49.860
73953 94.935 -78.860 49.822
73954 94.964 -78.855 49.785
73955 94.992 -78.851 49.748
73956 95.021 -78.846 49.710
73957 95.050 -78.842 49.672
73958 95.078 -78.837 49.632
73959 95.107 -78.835 49.591
73960 95.137 -78.840 49.550
73961 95.166 -78.846 49.510
73962 95.195 -78.852 49.470
73963 95.225 -78.858 49.430
73964 95.254 -78.863 49.389
73965 95.283 -78.869 49.349
73966 95.313 -78.874 49.308
73967 95.342 -78.880 49.268
73968 95.370 -78.886 49.227
73969 95.389 -78.899 49.189
73970 95.407 -78.905 49.149
73971 95.421 -78.903 49.109
73972 95.431 -78.907 49.067
73973 95.440 -78.912 49.026
73974 95.448 -78.918 48.984
73975 95.457 -78.924 48.941
73976 95.465 -78.930 48.900
73977 95.473 -78.936 48.857
73978 95.481 -78.946 48.815
73979 95.485 -78.966 48.774
73980 95.490 -78.987 48.732
73981 95.494 -79.006 48.689
73982 95.499 -79.027 48.646
73983 95.503 -79.046 48.606
73984 95.509 -79.067 48.565
73985 95.521 -79.082 48.522
73986 95.536 -79.095 48.478
73987 95.551 -79.109 48.433
73988 95.566 -79.122 48.387
73989 95.582 -79.134 48.341
73990 95.599 -79.140 48.293
73991 95.621 -79.127 48.245
73992 95.644 -79.114 48.199
73993 95.666 -79.100 48.153
73994 95.689 -79.087 48.107
73995 95.711 -79.074 48.061
73996 95.731 -79.062 48.014
73997 95.740 -79.059 47.970
73998 95.749 -79.055 47.927
73999 95.758 -79.051 47.884
74000 95.766 -79.048 47.842
74001 95.772 -79.044 47.800
74002 95.779 -79.040 47.758
74003 95.785 -79.037 47.715
74004 95.792 -79.033 47.672
74005 95.798 -79.030 47.630
74006 95.805 -79.026 47.588
74007 95.811 -79.022 47.547
74008 95.817 -79.018 47.507
74009 95.824 -79.015 47.466
74010 95.830 -79.012 47.426
74011 95.838 -79.009 47.386
74012 95.877 -79.019 47.347
74013 95.944 -79.037 47.306
74014 96.017 -79.053 47.268
74015 96.081 -79.055 47.230
74016 96.120 -79.058 47.193
74017 96.160 -79.060 47.157
74018 96.198 -79.059 47.122
74019 96.233 -79.046 47.086
74020 96.267 -79.034 47.051
74021 96.302 -79.021 47.016
74022 96.336 -79.008 46.984
74023 96.370 -78.995 46.953
74024 96.399 -79.032 46.922
74025 96.428 -79.078 46.891
74026 96.446 -79.090 46.859
74027 96.461 -79.091 46.827
74028 96.476 -79.091 46.795
74029 96.496 -79.099 46.763
74030 96.520 -79.110 46.732
74031 96.543 -79.122 46.699
74032 96.567 -79.134 46.668
74033 96.590 -79.146 46.637
74034 96.614 -79.158 46.605
74035 96.638 -79.169 46.575
74036 96.654 -79.172 46.547
74037 96.652 -79.156 46.519
74038 96.650 -79.138 46.491
74039 96.648 -79.122 46.463
74040 96.646 -79.105 46.435
74041 96.670 -79.100 46.408
74042 96.707 -79.100 46.381
74043 96.743 -79.101 46.355
74044 96.780 -79.102 46.331
74045 96.816 -79.103 46.307
74046 96.863 -79.106 46.282
74047 96.838 -79.157 46.258
74048 96.813 -79.204 46.233
74049 96.844 -79.175 46.210
74050 96.879 -79.139 46.187
74051 96.914 -79.103 46.162
74052 96.950 -79.066 46.138
74053 96.985 -79.030 46.114
74054 97.025 -79.052 46.092
74055 97.068 -79.096 46.070
74056 97.105 -79.137 46.050
74057 97.101 -79.138 46.031
74058 97.097 -79.140 46.013
74059 97.092 -79.140 45.993
74060 97.060 -79.115 45.974
74061 97.068 -79.181 45.954
74062 97.064 -79.186 45.935
74063 97.060 -79.190 45.917
74064 97.060 -79.197 45.899
74065 97.068 -79.172 45.881
74066 97.084 -79.156 45.863
74067 97.107 -79.134 45.844
74068 97.130 -79.113 45.825
74069 97.153 -79.092 45.807
74070 97.174 -79.126 45.790
74071 97.173 -79.229 45.774
74072 97.225 -79.238 45.757
74073 97.277 -79.247 45.740
74074 97.233 -79.188 45.722
74075 97.165 -79.149 45.705
74076 97.193 -79.154 45.688
74077 97.231 -79.153 45.671
74078 97.267 -79.151 45.654
74079 97.305 -79.150 45.638
74080 97.342 -79.148 45.622
74081 97.358 -79.163 45.606
74082 97.372 -79.181 45.591
74083 97.386 -79.197 45.575
74084 97.365 -79.229 45.559
74085 97.311 -79.274 45.545
74086 97.307 -79.281 45.530
74087 97.323 -79.275 45.515
74088 97.338 -79.269 45.501
74089 97.354 -79.262 45.486
74090 97.364 -79.258 45.471
74091 97.397 -79.269 45.457
74092 97.425 -79.281 45.443
74093 97.452 -79.293 45.429
74094 97.461 -79.307 45.415
74095 97.460 -79.318 45.401
74096 97.454 -79.331 45.385
74097 97.463 -79.338 45.371
74098 97.477 -79.344 45.358
74099 97.491 -79.350 45.345
74100 97.508 -79.361 45.332
74101 97.522 -79.369 45.319
74102 97.526 -79.365 45.306
74103 97.530 -79.360 45.294
74104 97.534 -79.356 45.281
74105 97.538 -79.351 45.268
74106 97.542 -79.347 45.256
74107 97.546 -79.343 45.243
74108 97.550 -79.338 45.231
74109 97.564 -79.362 45.219
74110 97.579 -79.390 45.208
74111 97.594 -79.417 45.197
74112 97.608 -79.445 45.186
74113 97.624 -79.473 45.174
74114 97.639 -79.501 45.165
74115 97.647 -79.513 45.154
74116 97.658 -79.598 45.141
74117 97.672 -79.648 45.129
74118 97.684 -79.654 45.117
74119 97.698 -79.662 45.106
74120 97.710 -79.669 45.094
74121 97.723 -79.676 45.082
74122 97.736 -79.683 45.070
74123 97.748 -79.691 45.058
74124 97.761 -79.697 45.046
74125 97.776 -79.698 45.034
74126 97.790 -79.699 45.023
74127 97.806 -79.712 45.013
74128 97.823 -79.728 45.002
74129 97.845 -79.742 44.991
74130 97.868 -79.758 44.981
74131 97.891 -79.772 44.972
74132 97.915 -79.786 44.962
74133 97.938 -79.801 44.954
74134 97.961 -79.816 44.946
74135 97.985 -79.830 44.938
74136 98.011 -79.842 44.931
74137 98.037 -79.855 44.925
74138 98.064 -79.867 44.920
74139 98.090 -79.879 44.914
74140 98.113 -79.886 44.908
74141 98.137 -79.894 44.901
74142 98.160 -79.902 44.894
74143 98.184 -79.911 44.887
74144 98.208 -79.919 44.880
74145 98.231 -79.927 44.872
74146 98.256 -79.933 44.862
74147 98.282 -79.939 44.852
74148 98.307 -79.944 44.841
74149 98.335 -79.951 44.832
74150 98.373 -79.963 44.825
74151 98.411 -79.974 44.820
74152 98.449 -79.987 44.815
74153 98.493 -79.997 44.809
74154 98.534 -80.009 44.804
74155 98.562 -80.024 44.797
74156 98.591 -80.027 44.787
74157 98.621 -80.030 44.776
74158 98.650 -80.035 44.764
74159 98.679 -80.040 44.756
74160 98.708 -80.045 44.749
74161 98.719 -80.060 44.742
74162 98.731 -80.074 44.732
74163 98.743 -80.090 44.721
74164 98.755 -80.105 44.711
74165 98.786 -80.108 44.700
74166 98.821 -80.108 44.691
74167 98.857 -80.107 44.682
74168 98.882 -80.117 44.673
74169 98.896 -80.118 44.664
74170 98.905 -80.116 44.656
74171 98.921 -80.121 44.649
74172 98.936 -80.127 44.641
74173 98.951 -80.136 44.634
74174 98.968 -80.147 44.627
74175 98.984 -80.158 44.620
74176 99.001 -80.169 44.615
74177 99.017 -80.181 44.610
74178 99.075 -80.197 44.604
74179 99.126 -80.221 44.599
74180 99.149 -80.231 44.593
74181 99.172 -80.242 44.587
74182 99.196 -80.252 44.582
74183 99.219 -80.262 44.579
74184 99.242 -80.273 44.575
74185 99.266 -80.284 44.570
74186 99.289 -80.294 44.566
74187 99.320 -80.302 44.562
74188 99.358 -80.306 44.559
74189 99.397 -80.309 44.555
74190 99.435 -80.314 44.551
74191 99.474 -80.318 44.548
74192 99.512 -80.322 44.545
74193 99.551 -80.325 44.541
74194 99.557 -80.346 44.537
74195 99.560 -80.368 44.532
74196 99.563 -80.391 44.528
74197 99.567 -80.413 44.523
74198 99.570 -80.435 44.519
74199 99.573 -80.457 44.513
74200 99.576 -80.478 44.508
74516 93.534 -78.928 53.494
74517 93.543 -78.925 53.484
74518 93.553 -78.923 53.474
74519 93.563 -78.916 53.465
74520 93.573 -78.907 53.456
74521 93.582 -78.900 53.448
74522 93.590 -78.893 53.439
74523 93.599 -78.886 53.431
74524 93.607 -78.880 53.423
74525 93.615 -78.873 53.416
74526 93.677 -78.842 53.410
74527 93.736 -78.812 53.404
74528 93.725 -78.809 53.400
74529 93.714 -78.805 53.397
74530 93.704 -78.802 53.394
74531 93.693 -78.799 53.392
74532 93.693 -78.804 53.388
74533 93.706 -78.817 53.383
74534 93.717 -78.824 53.376
74535 93.724 -78.824 53.367
74536 93.731 -78.821 53.358
74537 93.738 -78.818 53.349
74538 93.745 -78.816 53.338
74539 93.752 -78.813 53.327
74540 93.759 -78.810 53.314
74541 93.766 -78.808 53.300
74542 93.773 -78.805 53.288
74543 93.780 -78.802 53.273
74544 93.795 -78.794 53.258
74545 93.809 -78.783 53.244
74546 93.813 -78.774 53.225
74547 93.826 -78.767 53.205
74548 93.840 -78.758 53.184
74549 93.854 -78.750 53.165
74550 93.860 -78.745 53.143
74551 93.865 -78.741 53.122
74552 93.871 -78.736 53.101
74553 93.876 -78.732 53.076
74554 93.882 -78.727 53.051
74555 93.887 -78.724 53.025
74556 93.892 -78.719 53.000
74557 93.898 -78.714 52.976
74558 93.913 -78.708 52.951
74559 93.929 -78.702 52.926
74560 93.945 -78.695 52.898
74561 93.961 -78.689 52.872
74562 93.977 -78.683 52.848
74563 93.993 -78.677 52.822
74564 94.009 -78.670 52.798
74565 94.025 -78.664 52.774
74566 94.041 -78.658 52.752
74567 94.056 -78.651 52.730
74568 94.069 -78.648 52.708
74569 94.079 -78.646 52.687
74570 94.089 -78.645 52.665
74571 94.091 -78.648 52.642
74572 94.095 -78.655 52.620
74573 94.099 -78.664 52.596
74574 94.103 -78.671 52.572
74575 94.107 -78.679 52.547
74576 94.126 -78.647 52.522
74577 94.158 -78.630 52.498
74578 94.209 -78.643 52.474
74579 94.261 -78.656 52.450
74580 94.278 -78.656 52.427
74581 94.288 -78.654 52.403
74582 94.298 -78.651 52.378
74583 94.308 -78.648 52.355
74584 94.310 -78.652 52.332
74585 94.302 -78.664 52.308
74586 94.294 -78.676 52.285
74587 94.295 -78.680 52.260
74588 94.312 -78.667 52.233
74589 94.326 -78.658 52.204
74590 94.335 -78.655 52.178
74591 94.343 -78.653 52.151
74592 94.352 -78.651 52.125
74593 94.360 -78.648 52.098
74594 94.369 -78.646 52.072
74595 94.378 -78.644 52.045
74596 94.387 -78.642 52.017
74597 94.400 -78.642 51.987
74598 94.415 -78.644 51.957
74599 94.425 -78.649 51.925
74600 94.436 -78.655 51.892
74601 94.447 -78.660 51.859
74602 94.458 -78.665 51.829
74603 94.468 -78.672 51.799
74604 94.482 -78.683 51.767
74605 94.497 -78.683 51.733
74606 94.511 -78.683 51.696
74607 94.527 -78.682 51.658
74608 94.542 -78.681 51.619
74609 94.557 -78.680 51.580
74610 94.574 -78.682 51.542
74611 94.592 -78.684 51.506
74612 94.610 -78.686 51.472
74613 94.628 -78.688 51.439
74614 94.646 -78.690 51.406
74615 94.665 -78.692 51.376
74616 94.683 -78.695 51.345
74617 94.699 -78.697 51.314
74618 94.710 -78.699 51.282
74619 94.720 -78.702 51.249
74620 94.732 -78.704 51.214
74621 94.742 -78.706 51.177
74622 94.751 -78.704 51.136
74623 94.757 -78.693 51.095
74624 94.760 -78.702 51.053
74625 94.764 -78.710 51.012
74626 94.767 -78.718 50.971
74627 94.771 -78.727 50.930
74628 94.774 -78.736 50.892
74629 94.773 -78.744 50.850
74630 94.783 -78.735 50.807
74631 94.792 -78.727 50.762
74632 94.801 -78.717 50.718
74633 94.810 -78.709 50.676
74634 94.819 -78.701 50.636
74635 94.829 -78.693 50.596
74636 94.847 -78.690 50.552
74637 94.865 -78.686 50.508
74638 94.877 -78.690 50.463
74639 94.876 -78.705 50.419
74640 94.875 -78.720 50.375
74641 94.874 -78.736 50.332
74642 94.873 -78.752 50.292
74643 94.871 -78.767 50.253
74644 94.870 -78.783 50.214
74645 94.868 -78.798 50.177
74646 94.867 -78.814 50.138
74647 94.866 -78.829 50.099
74648 94.865 -78.845 50.059
74649 94.862 -78.855 50.019
74650 94.860 -78.856 49.980
74651 94.877 -78.859 49.941
74652 94.895 -78.861 49.902
74653 94.923 -78.856 49.863
74654 94.952 -78.852 49.824
74655 94.981 -78.847 49.786
74656 95.010 -78.843 49.749
74657 95.038 -78.838 49.710
74658 95.067 -78.833 49.670
74659 95.096 -78.829 49.629
74660 95.124 -78.824 49.588
74661 95.154 -78.826 49.548
74662 95.183 -78.831 49.509
74663 95.212 -78.837 49.469
74664 95.242 -78.843 49.430
74665 95.271 -78.849 49.389
74666 95.300 -78.854 49.349
74667 95.329 -78.860 49.307
74668 95.350 -78.871 49.266
74669 95.370 -78.882 49.224
74670 95.390 -78.894 49.186
74671 95.407 -78.902 49.148
74672 95.421 -78.899 49.108
74673 95.435 -78.897 49.067
74674 95.449 -78.896 49.026
74675 95.457 -78.902 48.984
74676 95.466 -78.907 48.941
74677 95.474 -78.913 48.900
74678 95.482 -78.919 48.858
74679 95.491 -78.925 48.816
74680 95.495 -78.945 48.775
74681 95.500 -78.965 48.734
74682 95.505 -78.985 48.692
74683 95.510 -79.005 48.651
74684 95.514 -79.025 48.612
74685 95.526 -79.040 48.571
74686 95.541 -79.053 48.529
74687 95.556 -79.067 48.484
74688 95.571 -79.080 48.439
74689 95.587 -79.093 48.393
74690 95.602 -79.106 48.346
74691 95.617 -79.120 48.298
74692 95.636 -79.120 48.251
74693 95.659 -79.107 48.204
74694 95.681 -79.093 48.158
74695 95.703 -79.081 48.111
74696 95.726 -79.068 48.065
74697 95.739 -79.061 48.020
74698 95.748 -79.057 47.977
74699 95.757 -79.053 47.934
74700 95.766 -79.049 47.891
74701 95.775 -79.046 47.849
74702 95.782 -79.042 47.806
74703 95.789 -79.038 47.763
74704 95.795 -79.035 47.720
74705 95.801 -79.031 47.677
74706 95.808 -79.028 47.635
74707 95.814 -79.024 47.594
74708 95.821 -79.021 47.553
74709 95.827 -79.017 47.512
74710 95.833 -79.013 47.472
74711 95.840 -79.009 47.431
74712 95.851 -79.006 47.391
74713 95.888 -79.002 47.351
74714 95.946 -79.014 47.310
74715 96.012 -79.031 47.272
74716 96.083 -79.037 47.234
74717 96.123 -79.040 47.197
74718 96.163 -79.042 47.161
74719 96.202 -79.045 47.125
74720 96.241 -79.045 47.090
74721 96.275 -79.032 47.054
74722 96.310 -79.019 47.020
74723 96.344 -79.006 46.988
74724 96.378 -78.993 46.957
74725 96.407 -79.031 46.927
74726 96.434 -79.069 46.896
74727 96.448 -79.070 46.864
74728 96.463 -79.070 46.832
74729 96.484 -79.078 46.799
74730 96.508 -79.090 46.767
74731 96.531 -79.102 46.736
74732 96.554 -79.114 46.703
74733 96.578 -79.125 46.672
74734 96.602 -79.137 46.641
74735 96.625 -79.149 46.609
74736 96.649 -79.161 46.580
74737 96.673 -79.172 46.551
74738 96.672 -79.157 46.524
74739 96.671 -79.140 46.495
74740 96.669 -79.123 46.467
74741 96.667 -79.106 46.439
74742 96.665 -79.090 46.411
74743 96.679 -79.080 46.384
74744 96.716 -79.081 46.358
74745 96.768 -79.085 46.334
74746 96.827 -79.090 46.310
74747 96.885 -79.095 46.285
74748 96.832 -79.167 46.261
74749 96.814 -79.175 46.236
74750 96.849 -79.138 46.213
74751 96.884 -79.103 46.189
74752 96.920 -79.066 46.165
74753 96.955 -79.030 46.141
74754 96.991 -79.001 46.119
74755 97.033 -79.046 46.097
74756 97.075 -79.091 46.076
74757 97.108 -79.128 46.056
74758 97.104 -79.128 46.038
74759 97.085 -79.115 46.020
74760 97.050 -79.087 46.001
74761 97.020 -79.062 45.981
74762 97.058 -79.201 45.961
74763 97.054 -79.206 45.942
74764 97.050 -79.210 45.924
74765 97.049 -79.194 45.906
74766 97.049 -79.171 45.889
74767 97.065 -79.150 45.871
74768 97.066 -79.140 45.853
74769 97.076 -79.130 45.834
74770 97.087 -79.118 45.816
74771 97.097 -79.107 45.799
74772 97.119 -79.096 45.783
74773 97.169 -79.121 45.767
74774 97.219 -79.146 45.749
74775 97.268 -79.171 45.732
74776 97.221 -79.158 45.715
74777 97.187 -79.158 45.697
74778 97.221 -79.157 45.680
74779 97.258 -79.156 45.664
74780 97.295 -79.154 45.647
74781 97.332 -79.153 45.631
74782 97.369 -79.151 45.615
74783 97.389 -79.163 45.599
74784 97.403 -79.180 45.583
74785 97.355 -79.222 45.568
74786 97.301 -79.268 45.554
74787 97.311 -79.265 45.539
74788 97.327 -79.259 45.525
74789 97.331 -79.256 45.510
74790 97.330 -79.258 45.495
74791 97.354 -79.259 45.480
74792 97.394 -79.263 45.466
74793 97.412 -79.279 45.452
74794 97.424 -79.300 45.439
74795 97.423 -79.311 45.424
74796 97.436 -79.317 45.410
74797 97.450 -79.322 45.395
74798 97.463 -79.328 45.380
74799 97.477 -79.334 45.367
74800 97.491 -79.340 45.354
74801 97.506 -79.348 45.341
74802 97.524 -79.362 45.328
74803 97.543 -79.376 45.315
74804 97.552 -79.378 45.303
74805 97.557 -79.373 45.290
74806 97.561 -79.369 45.277
74807 97.565 -79.365 45.264
74808 97.569 -79.360 45.252
74809 97.573 -79.356 45.241
74810 97.587 -79.382 45.229
74811 97.602 -79.410 45.217
74812 97.617 -79.438 45.206
74813 97.632 -79.465 45.196
74814 97.645 -79.488 45.185
74815 97.651 -79.494 45.175
74816 97.657 -79.499 45.165
74817 97.668 -79.578 45.154
74818 97.683 -79.623 45.142
74819 97.696 -79.630 45.130
74820 97.709 -79.638 45.119
74821 97.721 -79.645 45.108
74822 97.734 -79.651 45.096
74823 97.747 -79.659 45.085
74824 97.759 -79.666 45.073
74825 97.773 -79.673 45.061
74826 97.785 -79.680 45.049
74827 97.798 -79.688 45.037
74828 97.811 -79.697 45.026
74829 97.828 -79.709 45.016
74830 97.850 -79.723 45.006
74831 97.872 -79.738 44.997
74832 97.894 -79.753 44.987
74833 97.916 -79.768 44.978
74834 97.941 -79.789 44.970
74835 97.975 -79.818 44.962
74836 98.004 -79.833 44.956
74837 98.032 -79.849 44.949
74838 98.058 -79.856 44.943
74839 98.078 -79.852 44.938
74840 98.099 -79.853 44.933
74841 98.123 -79.863 44.928
74842 98.146 -79.873 44.921
74843 98.169 -79.882 44.913
74844 98.193 -79.889 44.905
74845 98.218 -79.896 44.897
74846 98.243 -79.902 44.887
74847 98.269 -79.908 44.876
74848 98.294 -79.913 44.865
74849 98.319 -79.919 44.852
74850 98.346 -79.926 44.845
74851 98.384 -79.937 44.839
74852 98.422 -79.949 44.835
74853 98.459 -79.961 44.831
74854 98.497 -79.972 44.824
74855 98.535 -79.984 44.816
74856 98.568 -79.992 44.808
74857 98.598 -79.996 44.797
74858 98.627 -80.002 44.784
74859 98.656 -80.006 44.771
74860 98.686 -80.011 44.761
74861 98.704 -80.021 44.752
74862 98.716 -80.037 44.745
74863 98.727 -80.052 44.737
74864 98.743 -80.065 44.727
74865 98.769 -80.073 44.717
74866 98.799 -80.077 44.708
74867 98.834 -80.077 44.700
74868 98.866 -80.080 44.691
74869 98.887 -80.089 44.683
74870 98.910 -80.099 44.674
74871 98.921 -80.098 44.666
74872 98.937 -80.102 44.659
74873 98.953 -80.108 44.652
74874 98.969 -80.112 44.645
74875 98.984 -80.120 44.638
74876 98.999 -80.128 44.632
74877 99.013 -80.137 44.626
74878 99.030 -80.148 44.620
74879 99.047 -80.159 44.615
74880 99.073 -80.167 44.610
74881 99.153 -80.204 44.605
74882 99.177 -80.215 44.599
74883 99.200 -80.225 44.593
74884 99.223 -80.235 44.589
74885 99.246 -80.246 44.585
74886 99.270 -80.256 44.580
74887 99.293 -80.267 44.577
74888 99.330 -80.272 44.573
74889 99.369 -80.275 44.569
74890 99.407 -80.279 44.565
74891 99.446 -80.284 44.562
74892 99.485 -80.287 44.558
74893 99.523 -80.291 44.555
74894 99.550 -80.301 44.550
74895 99.554 -80.323 44.546
74896 99.557 -80.345 44.541
74897 99.560 -80.367 44.536
74898 99.563 -80.390 44.531
74899 99.566 -80.412 44.526
74900 99.570 -80.434 44.522
74901 99.573 -80.456 44.516
74902 99.576 -80.478 44.510
75218 93.544 -78.915 53.497
75219 93.553 -78.912 53.488
75220 93.563 -78.910 53.478
75221 93.574 -78.902 53.469
75222 93.583 -78.894 53.461
75223 93.591 -78.886 53.452
75224 93.600 -78.880 53.443
75225 93.608 -78.873 53.437
75226 93.616 -78.866 53.429
75227 93.664 -78.842 53.423
75228 93.725 -78.810 53.417
75229 93.723 -78.808 53.414
75230 93.714 -78.805 53.412
75231 93.703 -78.802 53.410
75232 93.692 -78.799 53.407
75233 93.692 -78.803 53.404
75234 93.705 -78.816 53.398
75235 93.713 -78.817 53.391
75236 93.720 -78.816 53.382
75237 93.727 -78.814 53.373
75238 93.734 -78.811 53.363
75239 93.741 -78.808 53.352
75240 93.748 -78.805 53.341
75241 93.755 -78.803 53.327
75242 93.762 -78.800 53.314
75243 93.769 -78.797 53.301
75244 93.781 -78.791 53.286
75245 93.806 -78.777 53.270
75246 93.810 -78.768 53.254
75247 93.814 -78.760 53.235
75248 93.825 -78.752 53.214
75249 93.839 -78.744 53.193
75250 93.853 -78.736 53.172
75251 93.867 -78.728 53.151
75252 93.878 -78.721 53.130
75253 93.884 -78.717 53.109
75254 93.889 -78.712 53.085
75255 93.894 -78.708 53.060
75256 93.899 -78.703 53.034
75257 93.905 -78.698 53.009
75258 93.913 -78.694 52.984
75259 93.929 -78.687 52.958
75260 93.944 -78.681 52.933
75261 93.961 -78.675 52.907
75262 93.976 -78.669 52.881
75263 93.992 -78.662 52.855
75264 94.008 -78.656 52.831
75265 94.024 -78.650 52.808
75266 94.039 -78.645 52.787
75267 94.048 -78.643 52.765
75268 94.058 -78.642 52.743
75269 94.068 -78.639 52.721
75270 94.078 -78.638 52.700
75271 94.088 -78.636 52.677
75272 94.088 -78.642 52.654
75273 94.088 -78.649 52.631
75274 94.092 -78.657 52.608
75275 94.097 -78.664 52.583
75276 94.117 -78.629 52.558
75277 94.142 -78.601 52.533
75278 94.194 -78.614 52.509
75279 94.246 -78.628 52.486
75280 94.291 -78.639 52.463
75281 94.301 -78.636 52.440
75282 94.311 -78.633 52.417
75283 94.321 -78.631 52.393
75284 94.324 -78.634 52.371
75285 94.316 -78.646 52.349
75286 94.309 -78.658 52.326
75287 94.300 -78.670 52.302
75288 94.297 -78.677 52.278
75289 94.314 -78.665 52.251
75290 94.330 -78.653 52.224
75291 94.344 -78.644 52.197
75292 94.352 -78.642 52.172
75293 94.361 -78.639 52.145
75294 94.370 -78.637 52.117
75295 94.379 -78.636 52.091
75296 94.387 -78.633 52.064
75297 94.396 -78.631 52.036
75298 94.410 -78.630 52.005
75299 94.427 -78.631 51.974
75300 94.438 -78.636 51.942
75301 94.449 -78.642 51.909
75302 94.459 -78.647 51.876
75303 94.470 -78.651 51.846
75304 94.480 -78.675 51.815
75305 94.493 -78.688 51.781
75306 94.508 -78.687 51.746
75307 94.524 -78.686 51.710
75308 94.539 -78.686 51.672
75309 94.553 -78.686 51.632
75310 94.568 -78.685 51.593
75311 94.583 -78.684 51.555
75312 94.599 -78.683 51.519
75313 94.614 -78.683 51.484
75314 94.629 -78.682 51.452
75315 94.643 -78.681 51.420
75316 94.659 -78.680 51.390
75317 94.674 -78.680 51.361
75318 94.686 -78.679 51.331
75319 94.697 -78.681 51.297
75320 94.708 -78.683 51.262
75321 94.718 -78.686 51.226
75322 94.730 -78.688 51.189
75323 94.740 -78.690 51.149
75324 94.748 -78.695 51.109
75325 94.754 -78.703 51.067
75326 94.760 -78.711 51.026
75327 94.765 -78.718 50.984
75328 94.771 -78.726 50.942
75329 94.776 -78.733 50.903
75330 94.783 -78.733 50.862
75331 94.791 -78.724 50.819
75332 94.799 -78.720 50.774
75333 94.807 -78.711 50.730
75334 94.816 -78.703 50.689
75335 94.825 -78.695 50.649
75336 94.834 -78.686 50.608
75337 94.845 -78.679 50.564
75338 94.863 -78.676 50.519
75339 94.876 -78.678 50.474
75340 94.875 -78.694 50.429
75341 94.874 -78.709 50.385
75342 94.872 -78.724 50.342
75343 94.871 -78.740 50.302
75344 94.870 -78.755 50.261
75345 94.868 -78.771 50.221
75346 94.867 -78.786 50.183
75347 94.866 -78.802 50.144
75348 94.865 -78.818 50.105
75349 94.864 -78.833 50.064
75350 94.862 -78.849 50.023
75351 94.861 -78.858 49.984
75352 94.883 -78.858 49.945
75353 94.912 -78.852 49.905
75354 94.941 -78.848 49.863
75355 94.969 -78.843 49.823
75356 94.998 -78.839 49.784
75357 95.027 -78.834 49.747
75358 95.055 -78.830 49.707
75359 95.084 -78.825 49.666
75360 95.113 -78.821 49.625
75361 95.141 -78.816 49.585
75362 95.170 -78.811 49.545
75363 95.200 -78.817 49.504
75364 95.229 -78.822 49.466
75365 95.258 -78.828 49.426
75366 95.287 -78.833 49.386
75367 95.312 -78.843 49.346
75368 95.331 -78.854 49.304
75369 95.351 -78.866 49.263
75370 95.370 -78.877 49.221
75371 95.390 -78.890 49.183
75372 95.408 -78.898 49.144
75373 95.422 -78.896 49.104
75374 95.436 -78.893 49.064
75375 95.450 -78.891 49.023
75376 95.464 -78.888 48.982
75377 95.474 -78.890 48.941
75378 95.483 -78.896 48.899
75379 95.491 -78.902 48.858
75380 95.500 -78.908 48.817
75381 95.506 -78.923 48.776
75382 95.511 -78.943 48.734
75383 95.515 -78.963 48.694
75384 95.520 -78.984 48.655
75385 95.531 -78.999 48.615
75386 95.546 -79.012 48.576
75387 95.562 -79.025 48.535
75388 95.577 -79.039 48.490
75389 95.592 -79.052 48.444
75390 95.607 -79.065 48.397
75391 95.622 -79.078 48.350
75392 95.637 -79.092 48.303
75393 95.653 -79.105 48.256
75394 95.673 -79.100 48.209
75395 95.695 -79.087 48.161
75396 95.718 -79.074 48.115
75397 95.737 -79.062 48.070
75398 95.746 -79.059 48.026
75399 95.755 -79.055 47.984
75400 95.765 -79.051 47.942
75401 95.774 -79.047 47.899
75402 95.783 -79.044 47.856
75403 95.792 -79.040 47.812
75404 95.798 -79.037 47.768
75405 95.805 -79.033 47.726
75406 95.811 -79.029 47.683
75407 95.817 -79.025 47.641
75408 95.824 -79.022 47.599
75409 95.830 -79.018 47.558
75410 95.836 -79.015 47.518
75411 95.843 -79.011 47.478
75412 95.849 -79.008 47.437
75413 95.882 -79.004 47.396
75414 95.918 -78.999 47.355
75415 95.955 -78.996 47.316
75416 96.014 -79.008 47.277
75417 96.085 -79.018 47.239
75418 96.125 -79.021 47.201
75419 96.165 -79.024 47.165
75420 96.204 -79.027 47.129
75421 96.244 -79.030 47.094
75422 96.284 -79.031 47.059
75423 96.318 -79.018 47.025
75424 96.352 -79.005 46.992
75425 96.386 -78.992 46.962
75426 96.415 -79.031 46.932
75427 96.435 -79.049 46.901
75428 96.450 -79.049 46.870
75429 96.471 -79.058 46.839
75430 96.495 -79.070 46.805
75431 96.519 -79.082 46.772
75432 96.542 -79.093 46.741
75433 96.566 -79.106 46.709
75434 96.590 -79.117 46.678
75435 96.613 -79.129 46.647
75436 96.637 -79.140 46.615
75437 96.661 -79.153 46.585
75438 96.684 -79.164 46.557
75439 96.693 -79.159 46.529
75440 96.691 -79.142 46.499
75441 96.689 -79.125 46.471
75442 96.687 -79.109 46.443
75443 96.688 -79.090 46.415
75444 96.690 -79.071 46.388
75445 96.751 -79.039 46.362
75446 96.791 -79.075 46.336
75447 96.849 -79.080 46.312
75448 96.908 -79.085 46.288
75449 96.876 -79.131 46.264
75450 96.853 -79.125 46.240
75451 96.876 -79.093 46.216
75452 96.900 -79.062 46.192
75453 96.925 -79.030 46.168
75454 96.960 -78.993 46.146
75455 96.999 -78.996 46.124
75456 97.041 -79.041 46.102
75457 97.083 -79.086 46.083
75458 97.110 -79.117 46.064
75459 97.075 -79.088 46.045
75460 97.040 -79.060 46.028
75461 97.005 -79.032 46.009
75462 97.011 -79.048 45.989
75463 97.049 -79.222 45.969
75464 97.045 -79.226 45.950
75465 97.043 -79.211 45.932
75466 97.044 -79.191 45.915
75467 97.050 -79.182 45.898
75468 97.058 -79.172 45.881
75469 97.069 -79.161 45.864
75470 97.080 -79.150 45.848
75471 97.091 -79.138 45.830
75472 97.100 -79.124 45.814
75473 97.104 -79.101 45.798
75474 97.124 -79.086 45.781
75475 97.162 -79.079 45.763
75476 97.207 -79.092 45.745
75477 97.224 -79.104 45.727
75478 97.214 -79.122 45.708
75479 97.231 -79.159 45.690
75480 97.262 -79.162 45.674
75481 97.296 -79.159 45.657
75482 97.329 -79.158 45.640
75483 97.363 -79.156 45.624
75484 97.397 -79.153 45.609
75485 97.399 -79.172 45.593
75486 97.345 -79.216 45.578
75487 97.300 -79.255 45.564
75488 97.309 -79.255 45.550
75489 97.320 -79.255 45.536
75490 97.331 -79.256 45.522
75491 97.343 -79.256 45.507
75492 97.373 -79.259 45.493
75493 97.404 -79.266 45.478
75494 97.402 -79.291 45.465
75495 97.410 -79.297 45.451
75496 97.423 -79.301 45.436
75497 97.436 -79.307 45.422
75498 97.450 -79.313 45.407
75499 97.463 -79.319 45.393
75500 97.477 -79.324 45.379
75501 97.490 -79.330 45.366
75502 97.504 -79.335 45.353
75503 97.522 -79.349 45.340
75504 97.541 -79.363 45.327
75505 97.560 -79.377 45.314
75506 97.579 -79.391 45.301
75507 97.583 -79.387 45.288
75508 97.587 -79.382 45.275
75509 97.591 -79.378 45.263
75510 97.595 -79.375 45.251
75511 97.610 -79.402 45.240
75512 97.626 -79.430 45.228
75513 97.640 -79.457 45.217
75514 97.649 -79.469 45.207
75515 97.655 -79.474 45.197
75516 97.661 -79.479 45.187
75517 97.667 -79.485 45.178
75518 97.677 -79.557 45.168
75519 97.693 -79.589 45.157
75520 97.707 -79.606 45.145
75521 97.720 -79.613 45.134
75522 97.733 -79.620 45.123
75523 97.745 -79.627 45.111
75524 97.758 -79.635 45.100
75525 97.771 -79.642 45.088
75526 97.784 -79.649 45.076
75527 97.796 -79.656 45.064
75528 97.810 -79.666 45.052
75529 97.825 -79.677 45.040
75530 97.841 -79.689 45.031
75531 97.862 -79.702 45.022
75532 97.883 -79.717 45.013
75533 97.906 -79.732 45.004
75534 97.927 -79.746 44.995
75535 97.964 -79.786 44.987
75536 98.003 -79.831 44.980
75537 98.027 -79.835 44.974
75538 98.047 -79.831 44.967
75539 98.068 -79.827 44.959
75540 98.088 -79.823 44.952
75541 98.110 -79.828 44.948
75542 98.133 -79.838 44.944
75543 98.157 -79.848 44.939
75544 98.180 -79.860 44.932
75545 98.205 -79.866 44.923
75546 98.230 -79.871 44.912
75547 98.255 -79.877 44.902
75548 98.281 -79.883 44.889
75549 98.306 -79.888 44.875
75550 98.331 -79.894 44.862
75551 98.357 -79.900 44.853
75552 98.395 -79.911 44.847
75553 98.433 -79.924 44.842
75554 98.471 -79.935 44.836
75555 98.508 -79.947 44.830
75556 98.542 -79.956 44.821
75557 98.574 -79.963 44.812
75558 98.603 -79.968 44.801
75559 98.633 -79.973 44.790
75560 98.662 -79.977 44.778
75561 98.689 -79.983 44.768
75562 98.701 -79.999 44.759
75563 98.712 -80.014 44.752
75564 98.733 -80.024 44.745
75565 98.759 -80.033 44.736
75566 98.784 -80.041 44.726
75567 98.812 -80.047 44.718
75568 98.847 -80.047 44.709
75569 98.874 -80.053 44.701
75570 98.895 -80.062 44.692
75571 98.916 -80.072 44.684
75572 98.938 -80.082 44.677
75573 98.952 -80.085 44.669
75574 98.969 -80.090 44.663
75575 98.985 -80.094 44.657
75576 99.001 -80.099 44.650
75577 99.017 -80.104 44.643
75578 99.031 -80.112 44.638
75579 99.046 -80.121 44.632
75580 99.061 -80.129 44.627
75581 99.076 -80.138 44.622
75582 99.092 -80.149 44.616
75583 99.156 -80.176 44.611
75584 99.204 -80.198 44.605
75585 99.227 -80.208 44.601
75586 99.251 -80.218 44.596
75587 99.274 -80.229 44.591
75588 99.302 -80.237 44.588
75589 99.341 -80.241 44.584
75590 99.379 -80.245 44.580
75591 99.418 -80.250 44.576
75592 99.457 -80.253 44.573
75593 99.495 -80.257 44.569
75594 99.534 -80.261 44.564
75595 99.547 -80.278 44.560
75596 99.550 -80.300 44.556
75597 99.553 -80.322 44.551
75598 99.556 -80.344 44.546
75599 99.560 -80.367 44.541
75600 99.563 -80.389 44.535
75601 99.566 -80.411 44.530
75602 99.569 -80.433 44.524
75603 99.572 -80.455 44.518
75604 99.576 -80.478 44.512
75605 99.579 -80.500 44.505
75606 99.582 -80.522 44.498
75607 99.585 -80.544 44.491
75608 99.588 -80.566 44.483
75609 99.592 -80.588 44.476
75610 99.595 -80.610 44.468
75611 99.598 -80.632 44.460
75612 99.602 -80.654 44.452
75613 99.605 -80.677 44.443
75614 99.608 -80.699 44.434
75615 99.611 -80.721 44.426
75616 99.615 -80.743 44.416
75617 99.632 -80.760 44.407
75919 93.544 -78.904 53.510
75920 93.553 -78.901 53.501
75921 93.563 -78.899 53.491
75922 93.573 -78.896 53.481
75923 93.584 -78.887 53.474
75924 93.592 -78.880 53.465
75925 93.601 -78.874 53.456
75926 93.609 -78.866 53.450
75927 93.618 -78.859 53.442
75928 93.652 -78.841 53.437
75929 93.717 -78.808 53.433
75930 93.712 -78.806 53.430
75931 93.710 -78.805 53.428
75932 93.703 -78.802 53.427
75933 93.692 -78.799 53.426
75934 93.691 -78.802 53.420
75935 93.702 -78.809 53.414
75936 93.709 -78.809 53.407
75937 93.716 -78.808 53.398
75938 93.723 -78.807 53.387
75939 93.730 -78.804 53.377
75940 93.737 -78.801 53.368
75941 93.744 -78.799 53.355
75942 93.751 -78.796 53.342
75943 93.758 -78.793 53.328
75944 93.768 -78.788 53.314
75945 93.793 -78.774 53.298
75946 93.808 -78.763 53.282
75947 93.812 -78.754 53.265
75948 93.815 -78.745 53.245
75949 93.824 -78.737 53.224
75950 93.838 -78.730 53.202
75951 93.852 -78.721 53.181
75952 93.866 -78.714 53.160
75953 93.880 -78.705 53.140
75954 93.894 -78.697 53.118
75955 93.901 -78.692 53.095
75956 93.906 -78.688 53.070
75957 93.912 -78.683 53.044
75958 93.917 -78.679 53.018
75959 93.928 -78.673 52.993
75960 93.944 -78.667 52.967
75961 93.960 -78.661 52.941
75962 93.976 -78.655 52.916
75963 93.992 -78.648 52.891
75964 94.008 -78.642 52.866
75965 94.018 -78.640 52.844
75966 94.028 -78.639 52.822
75967 94.038 -78.636 52.800
75968 94.048 -78.635 52.781
75969 94.058 -78.633 52.758
75970 94.067 -78.632 52.735
75971 94.077 -78.630 52.713
75972 94.085 -78.630 52.690
75973 94.084 -78.636 52.668
75974 94.083 -78.642 52.645
75975 94.088 -78.646 52.622
75976 94.108 -78.611 52.597
75977 94.128 -78.574 52.572
75978 94.179 -78.586 52.547
75979 94.230 -78.599 52.522
75980 94.282 -78.613 52.499
75981 94.314 -78.619 52.476
75982 94.324 -78.616 52.455
75983 94.334 -78.614 52.432
75984 94.338 -78.616 52.409
75985 94.330 -78.628 52.387
75986 94.322 -78.640 52.366
75987 94.315 -78.652 52.344
75988 94.307 -78.664 52.321
75989 94.299 -78.676 52.297
75990 94.316 -78.664 52.271
75991 94.332 -78.651 52.244
75992 94.349 -78.639 52.218
75993 94.362 -78.631 52.193
75994 94.370 -78.629 52.167
75995 94.379 -78.626 52.139
75996 94.388 -78.624 52.112
75997 94.396 -78.622 52.084
75998 94.405 -78.620 52.055
75999 94.419 -78.619 52.024
76000 94.437 -78.620 51.992
76001 94.450 -78.623 51.959
76002 94.461 -78.628 51.926
76003 94.472 -78.633 51.893
76004 94.482 -78.644 51.862
76005 94.492 -78.677 51.830
76006 94.505 -78.692 51.795
76007 94.520 -78.692 51.759
76008 94.535 -78.691 51.723
76009 94.550 -78.690 51.684
76010 94.565 -78.690 51.644
76011 94.580 -78.689 51.605
76012 94.596 -78.689 51.566
76013 94.610 -78.688 51.531
76014 94.625 -78.687 51.496
76015 94.640 -78.686 51.464
76016 94.653 -78.683 51.433
76017 94.665 -78.680 51.403
76018 94.677 -78.676 51.374
76019 94.688 -78.673 51.344
76020 94.699 -78.675 51.310
76021 94.710 -78.677 51.274
76022 94.720 -78.680 51.237
76023 94.732 -78.682 51.199
76024 94.744 -78.684 51.160
76025 94.758 -78.689 51.120
76026 94.764 -78.697 51.080
76027 94.769 -78.705 51.038
76028 94.775 -78.712 50.996
76029 94.780 -78.720 50.954
76030 94.786 -78.727 50.914
76031 94.793 -78.724 50.873
76032 94.803 -78.711 50.832
76033 94.812 -78.698 50.787
76034 94.819 -78.695 50.742
76035 94.825 -78.692 50.700
76036 94.832 -78.688 50.660
76037 94.841 -78.680 50.618
76038 94.849 -78.671 50.574
76039 94.862 -78.664 50.529
76040 94.875 -78.667 50.484
76041 94.873 -78.682 50.438
76042 94.872 -78.698 50.393
76043 94.871 -78.713 50.349
76044 94.870 -78.728 50.310
76045 94.868 -78.744 50.270
76046 94.867 -78.759 50.231
76047 94.866 -78.775 50.192
76048 94.865 -78.790 50.153
76049 94.867 -78.804 50.113
76050 94.868 -78.818 50.071
76051 94.871 -78.833 50.028
76052 94.876 -78.844 49.987
76053 94.896 -78.846 49.946
76054 94.926 -78.842 49.904
76055 94.956 -78.838 49.862
76056 94.986 -78.834 49.821
76057 95.015 -78.830 49.781
76058 95.044 -78.826 49.743
76059 95.072 -78.821 49.704
76060 95.101 -78.817 49.663
76061 95.130 -78.812 49.622
76062 95.159 -78.808 49.583
76063 95.187 -78.803 49.543
76064 95.216 -78.802 49.504
76065 95.246 -78.808 49.464
76066 95.273 -78.814 49.424
76067 95.293 -78.826 49.383
76068 95.312 -78.838 49.344
76069 95.331 -78.849 49.303
76070 95.351 -78.861 49.261
76071 95.371 -78.873 49.220
76072 95.390 -78.885 49.181
76073 95.408 -78.893 49.142
76074 95.422 -78.891 49.102
76075 95.436 -78.889 49.062
76076 95.450 -78.886 49.021
76077 95.464 -78.884 48.981
76078 95.478 -78.882 48.942
76079 95.492 -78.880 48.901
76080 95.500 -78.885 48.859
76081 95.509 -78.891 48.817
76082 95.516 -78.902 48.776
76083 95.521 -78.921 48.735
76084 95.525 -78.942 48.694
76085 95.537 -78.958 48.654
76086 95.552 -78.971 48.614
76087 95.567 -78.984 48.574
76088 95.582 -78.997 48.534
76089 95.597 -79.011 48.492
76090 95.612 -79.024 48.447
76091 95.627 -79.037 48.401
76092 95.643 -79.050 48.356
76093 95.658 -79.064 48.309
76094 95.673 -79.077 48.261
76095 95.688 -79.090 48.212
76096 95.710 -79.081 48.164
76097 95.732 -79.068 48.119
76098 95.745 -79.061 48.076
76099 95.754 -79.057 48.032
76100 95.763 -79.053 47.991
76101 95.772 -79.049 47.949
76102 95.781 -79.046 47.906
76103 95.791 -79.042 47.862
76104 95.800 -79.038 47.819
76105 95.808 -79.034 47.776
76106 95.814 -79.031 47.734
76107 95.820 -79.028 47.691
76108 95.827 -79.024 47.648
76109 95.833 -79.020 47.605
76110 95.840 -79.016 47.565
76111 95.846 -79.013 47.524
76112 95.852 -79.009 47.483
76113 95.877 -79.006 47.442
76114 95.913 -79.002 47.401
76115 95.950 -78.997 47.361
76116 95.986 -78.993 47.321
76117 96.023 -78.990 47.282
76118 96.087 -79.000 47.245
76119 96.127 -79.003 47.208
76120 96.167 -79.006 47.171
76121 96.206 -79.009 47.135
76122 96.246 -79.012 47.099
76123 96.286 -79.014 47.063
76124 96.326 -79.016 47.031
76125 96.360 -79.003 46.998
76126 96.394 -78.991 46.967
76127 96.422 -79.028 46.937
76128 96.437 -79.028 46.906
76129 96.459 -79.038 46.875
76130 96.483 -79.049 46.844
76131 96.507 -79.062 46.813
76132 96.530 -79.073 46.780
76133 96.554 -79.085 46.748
76134 96.578 -79.096 46.716
76135 96.601 -79.109 46.684
76136 96.625 -79.120 46.653
76137 96.649 -79.132 46.621
76138 96.672 -79.143 46.592
76139 96.696 -79.156 46.563
76140 96.714 -79.161 46.534
76141 96.716 -79.142 46.505
76142 96.718 -79.123 46.476
76143 96.720 -79.104 46.447
76144 96.722 -79.085 46.419
76145 96.724 -79.066 46.393
76146 96.788 -79.032 46.367
76147 96.862 -78.996 46.342
76148 96.915 -79.001 46.318
76149 96.949 -79.045 46.293
76150 96.938 -79.068 46.269
76151 96.911 -79.065 46.244
76152 96.915 -79.045 46.220
76153 96.939 -79.012 46.197
76154 96.958 -78.983 46.174
76155 96.970 -78.958 46.153
76156 96.978 -78.976 46.131
76157 96.979 -78.987 46.110
76158 97.012 -79.026 46.091
76159 97.034 -79.056 46.073
76160 96.991 -79.025 46.055
76161 96.979 -79.006 46.037
76162 96.976 -78.987 46.018
76163 97.007 -79.090 45.999
76164 97.039 -79.241 45.979
76165 97.038 -79.228 45.960
76166 97.043 -79.219 45.942
76167 97.049 -79.211 45.925
76168 97.055 -79.202 45.908
76169 97.063 -79.193 45.892
76170 97.074 -79.181 45.876
76171 97.085 -79.170 45.859
76172 97.095 -79.159 45.841
76173 97.101 -79.139 45.824
76174 97.106 -79.117 45.808
76175 97.110 -79.094 45.792
76176 97.131 -79.078 45.774
76177 97.168 -79.071 45.756
76178 97.205 -79.064 45.738
76179 97.230 -79.094 45.719
76180 97.254 -79.125 45.701
76181 97.275 -79.159 45.685
76182 97.303 -79.166 45.669
76183 97.334 -79.161 45.652
76184 97.366 -79.156 45.636
76185 97.397 -79.150 45.620
76186 97.407 -79.162 45.605
76187 97.401 -79.182 45.591
76188 97.349 -79.227 45.577
76189 97.322 -79.253 45.562
76190 97.333 -79.253 45.548
76191 97.344 -79.253 45.534
76192 97.355 -79.254 45.520
76193 97.392 -79.257 45.506
76194 97.399 -79.277 45.492
76195 97.404 -79.288 45.478
76196 97.414 -79.291 45.464
76197 97.425 -79.294 45.449
76198 97.436 -79.297 45.435
76199 97.450 -79.303 45.421
76200 97.463 -79.309 45.406
76201 97.476 -79.314 45.393
76202 97.490 -79.320 45.380
76203 97.504 -79.325 45.367
76204 97.520 -79.336 45.354
76205 97.539 -79.350 45.340
76206 97.558 -79.364 45.326
76207 97.577 -79.378 45.313
76208 97.596 -79.392 45.300
76209 97.609 -79.400 45.288
76210 97.613 -79.395 45.274
76211 97.619 -79.394 45.262
76212 97.634 -79.422 45.250
76213 97.647 -79.444 45.239
76214 97.653 -79.450 45.229
76215 97.658 -79.455 45.220
76216 97.664 -79.460 45.211
76217 97.670 -79.466 45.201
76218 97.676 -79.472 45.192
76219 97.687 -79.525 45.183
76220 97.703 -79.556 45.174
76221 97.719 -79.582 45.162
76222 97.731 -79.588 45.150
76223 97.744 -79.596 45.139
76224 97.757 -79.603 45.127
76225 97.770 -79.610 45.115
76226 97.782 -79.617 45.103
76227 97.795 -79.624 45.090
76228 97.809 -79.635 45.078
76229 97.823 -79.647 45.067
76230 97.838 -79.658 45.056
76231 97.853 -79.670 45.048
76232 97.874 -79.683 45.040
76233 97.895 -79.696 45.032
76234 97.917 -79.710 45.021
76235 97.947 -79.739 45.013
76236 97.961 -79.744 45.005
76237 97.971 -79.741 44.998
76238 97.996 -79.752 44.991
76239 98.030 -79.766 44.984
76240 98.064 -79.780 44.977
76241 98.097 -79.794 44.971
76242 98.121 -79.805 44.963
76243 98.144 -79.817 44.957
76244 98.167 -79.829 44.951
76245 98.192 -79.839 44.943
76246 98.217 -79.843 44.934
76247 98.242 -79.848 44.923
76248 98.267 -79.852 44.910
76249 98.293 -79.858 44.896
76250 98.318 -79.863 44.882
76251 98.343 -79.869 44.870
76252 98.368 -79.875 44.860
76253 98.406 -79.886 44.853
76254 98.443 -79.898 44.846
76255 98.481 -79.910 44.840
76256 98.515 -79.918 44.832
76257 98.547 -79.925 44.824
76258 98.578 -79.933 44.815
76259 98.609 -79.939 44.805
76260 98.639 -79.944 44.795
76261 98.668 -79.949 44.785
76262 98.685 -79.961 44.776
76263 98.698 -79.975 44.767
76264 98.723 -79.983 44.759
76265 98.748 -79.992 44.753
76266 98.774 -80.000 44.745
76267 98.799 -80.008 44.736
76268 98.825 -80.017 44.728
76269 98.860 -80.018 44.719
76270 98.881 -80.027 44.710
76271 98.903 -80.036 44.701
76272 98.924 -80.044 44.694
76273 98.944 -80.043 44.687
76274 98.961 -80.047 44.680
76275 98.981 -80.061 44.675
76276 99.000 -80.075 44.669
76277 99.016 -80.081 44.662
76278 99.032 -80.086 44.655
76279 99.049 -80.091 44.651
76280 99.065 -80.096 44.646
76281 99.079 -80.105 44.641
76282 99.093 -80.113 44.635
76283 99.108 -80.121 44.629
76284 99.123 -80.130 44.624
76285 99.153 -80.145 44.619
76286 99.231 -80.181 44.613
76287 99.255 -80.191 44.608
76288 99.278 -80.202 44.604
76289 99.313 -80.207 44.600
76290 99.352 -80.211 44.596
76291 99.391 -80.215 44.592
76292 99.429 -80.219 44.587
76293 99.468 -80.223 44.583
76294 99.506 -80.227 44.579
76295 99.540 -80.233 44.575
76296 99.543 -80.256 44.570
76297 99.547 -80.278 44.566
76298 99.550 -80.300 44.561
76299 99.553 -80.322 44.556
76300 99.556 -80.344 44.550
76301 99.559 -80.366 44.544
76302 99.563 -80.388 44.538
76303 99.566 -80.410 44.532
76304 99.569 -80.432 44.526
76305 99.572 -80.455 44.520
76306 99.575 -80.477 44.513
76307 99.579 -80.499 44.505
76308 99.582 -80.521 44.497
76309 99.585 -80.543 44.490
76310 99.588 -80.566 44.482
76311 99.591 -80.588 44.474
76312 99.595 -80.610 44.466
76313 99.598 -80.632 44.458
76314 99.601 -80.654 44.449
76315 99.605 -80.676 44.440
76316 99.608 -80.698 44.431
76317 99.611 -80.720 44.421
76318 99.640 -80.733 44.412
76319 99.688 -80.739 44.402
76320 99.736 -80.744 44.391
76321 99.784 -80.750 44.381
76620 93.544 -78.893 53.523
76621 93.554 -78.890 53.513
76622 93.563 -78.888 53.503
76623 93.573 -78.885 53.493
76624 93.582 -78.883 53.486
76625 93.593 -78.874 53.478
76626 93.602 -78.867 53.470
76627 93.610 -78.860 53.463
76628 93.619 -78.853 53.456
76629 93.641 -78.840 53.451
76630 93.709 -78.806 53.448
76631 93.700 -78.805 53.447
76632 93.698 -78.802 53.446
76633 93.697 -78.801 53.445
76634 93.692 -78.799 53.443
76635 93.690 -78.802 53.437
76636 93.698 -78.802 53.430
76637 93.705 -78.802 53.422
76638 93.711 -78.801 53.412
76639 93.718 -78.799 53.402
76640 93.726 -78.797 53.393
76641 93.733 -78.794 53.382
76642 93.740 -78.791 53.370
76643 93.747 -78.788 53.356
76644 93.755 -78.785 53.342
76645 93.779 -78.771 53.327
76646 93.804 -78.757 53.310
76647 93.809 -78.749 53.293
76648 93.813 -78.739 53.275
76649 93.816 -78.731 53.255
76650 93.823 -78.723 53.234
76651 93.837 -78.715 53.213
76652 93.851 -78.707 53.191
76653 93.865 -78.699 53.172
76654 93.879 -78.691 53.152
76655 93.893 -78.683 53.130
76656 93.907 -78.675 53.106
76657 93.919 -78.667 53.080
76658 93.924 -78.663 53.054
76659 93.929 -78.659 53.029
76660 93.943 -78.653 53.003
76661 93.960 -78.646 52.977
76662 93.976 -78.640 52.951
76663 93.987 -78.637 52.926
76664 93.997 -78.636 52.901
76665 94.007 -78.633 52.878
76666 94.017 -78.632 52.857
76667 94.027 -78.630 52.836
76668 94.037 -78.629 52.815
76669 94.047 -78.627 52.796
76670 94.057 -78.626 52.773
76671 94.067 -78.623 52.750
76672 94.077 -78.622 52.728
76673 94.082 -78.623 52.705
76674 94.081 -78.630 52.682
76675 94.079 -78.636 52.659
76676 94.099 -78.592 52.636
76677 94.119 -78.556 52.611
76678 94.163 -78.557 52.586
76679 94.214 -78.571 52.560
76680 94.266 -78.584 52.536
76681 94.318 -78.598 52.513
76682 94.337 -78.598 52.492
76683 94.347 -78.596 52.472
76684 94.351 -78.598 52.450
76685 94.344 -78.610 52.427
76686 94.336 -78.622 52.406
76687 94.328 -78.634 52.384
76688 94.321 -78.646 52.362
76689 94.313 -78.658 52.339
76690 94.305 -78.670 52.316
76691 94.318 -78.661 52.292
76692 94.334 -78.649 52.266
76693 94.351 -78.637 52.240
76694 94.367 -78.624 52.214
76695 94.380 -78.617 52.188
76696 94.388 -78.615 52.161
76697 94.397 -78.613 52.135
76698 94.405 -78.611 52.107
76699 94.414 -78.608 52.077
76700 94.428 -78.608 52.046
76701 94.447 -78.608 52.012
76702 94.462 -78.610 51.978
76703 94.473 -78.615 51.945
76704 94.484 -78.620 51.911
76705 94.494 -78.646 51.878
76706 94.504 -78.680 51.845
76707 94.517 -78.697 51.810
76708 94.532 -78.696 51.773
76709 94.547 -78.695 51.735
76710 94.562 -78.695 51.696
76711 94.577 -78.694 51.656
76712 94.592 -78.693 51.617
76713 94.607 -78.693 51.577
76714 94.620 -78.689 51.541
76715 94.631 -78.686 51.507
76716 94.643 -78.682 51.475
76717 94.655 -78.678 51.444
76718 94.667 -78.674 51.414
76719 94.679 -78.670 51.384
76720 94.690 -78.667 51.354
76721 94.701 -78.669 51.320
76722 94.712 -78.671 51.283
76723 94.723 -78.673 51.245
76724 94.734 -78.676 51.208
76725 94.751 -78.679 51.169
76726 94.768 -78.683 51.130
76727 94.773 -78.691 51.091
76728 94.779 -78.698 51.050
76729 94.784 -78.705 51.008
76730 94.790 -78.713 50.966
76731 94.795 -78.720 50.924
76732 94.804 -78.714 50.884
76733 94.813 -78.701 50.843
76734 94.822 -78.688 50.799
76735 94.832 -78.675 50.754
76736 94.839 -78.670 50.711
76737 94.845 -78.666 50.669
76738 94.852 -78.663 50.628
76739 94.858 -78.659 50.583
76740 94.865 -78.655 50.538
76741 94.873 -78.655 50.493
76742 94.872 -78.670 50.445
76743 94.873 -78.685 50.400
76744 94.875 -78.698 50.356
76745 94.877 -78.713 50.316
76746 94.879 -78.727 50.278
76747 94.881 -78.741 50.240
76748 94.882 -78.755 50.200
76749 94.884 -78.769 50.161
76750 94.886 -78.783 50.120
76751 94.888 -78.797 50.076
76752 94.890 -78.811 50.032
76753 94.900 -78.819 49.989
76754 94.914 -78.824 49.946
76755 94.934 -78.825 49.903
76756 94.964 -78.822 49.860
76757 94.993 -78.818 49.819
76758 95.024 -78.814 49.779
76759 95.053 -78.811 49.740
76760 95.083 -78.807 49.701
76761 95.113 -78.804 49.662
76762 95.143 -78.800 49.622
76763 95.173 -78.796 49.582
76764 95.202 -78.793 49.543
76765 95.233 -78.789 49.504
76766 95.254 -78.798 49.463
76767 95.273 -78.810 49.423
76768 95.293 -78.821 49.382
76769 95.312 -78.833 49.342
76770 95.332 -78.845 49.302
76771 95.351 -78.857 49.261
76772 95.371 -78.868 49.220
76773 95.390 -78.880 49.182
76774 95.409 -78.890 49.142
76775 95.422 -78.887 49.102
76776 95.437 -78.885 49.061
76777 95.451 -78.883 49.021
76778 95.464 -78.880 48.982
76779 95.478 -78.878 48.943
76780 95.492 -78.876 48.902
76781 95.506 -78.873 48.860
76782 95.518 -78.874 48.818
76783 95.526 -78.880 48.777
76784 95.531 -78.900 48.736
76785 95.542 -78.916 48.695
76786 95.557 -78.930 48.654
76787 95.572 -78.943 48.613
76788 95.587 -78.956 48.573
76789 95.602 -78.969 48.532
76790 95.618 -78.983 48.492
76791 95.633 -78.996 48.450
76792 95.648 -79.009 48.406
76793 95.663 -79.022 48.360
76794 95.678 -79.035 48.312
76795 95.693 -79.049 48.264
76796 95.708 -79.062 48.216
76797 95.724 -79.074 48.170
76798 95.744 -79.063 48.126
76799 95.753 -79.059 48.083
76800 95.762 -79.056 48.039
76801 95.771 -79.052 47.998
76802 95.780 -79.048 47.956
76803 95.789 -79.044 47.912
76804 95.798 -79.040 47.869
76805 95.807 -79.037 47.827
76806 95.816 -79.033 47.785
76807 95.824 -79.029 47.742
76808 95.830 -79.025 47.699
76809 95.836 -79.021 47.656
76810 95.843 -79.018 47.612
76811 95.849 -79.015 47.572
76812 95.855 -79.011 47.530
76813 95.871 -79.007 47.489
76814 95.907 -79.003 47.448
76815 95.944 -78.999 47.408
76816 95.980 -78.996 47.368
76817 96.017 -78.991 47.328
76818 96.053 -78.987 47.289
76819 96.090 -78.984 47.251
76820 96.129 -78.985 47.214
76821 96.169 -78.987 47.177
76822 96.208 -78.990 47.141
76823 96.248 -78.993 47.105
76824 96.288 -78.996 47.070
76825 96.328 -78.999 47.037
76826 96.368 -79.001 47.005
76827 96.402 -78.990 46.974
76828 96.425 -79.007 46.943
76829 96.447 -79.018 46.913
76830 96.471 -79.030 46.882
76831 96.495 -79.041 46.851
76832 96.518 -79.053 46.820
76833 96.542 -79.065 46.789
76834 96.566 -79.077 46.757
76835 96.584 -79.086 46.724
76836 96.603 -79.096 46.691
76837 96.621 -79.105 46.660
76838 96.640 -79.115 46.629
76839 96.658 -79.124 46.599
76840 96.677 -79.134 46.570
76841 96.707 -79.138 46.539
76842 96.743 -79.134 46.510
76843 96.752 -79.118 46.481
76844 96.754 -79.099 46.452
76845 96.756 -79.080 46.425
76846 96.758 -79.061 46.398
76847 96.823 -79.027 46.373
76848 96.897 -78.990 46.348
76849 96.972 -78.954 46.324
76850 96.989 -78.949 46.299
76851 96.965 -78.977 46.274
76852 96.942 -79.006 46.249
76853 96.935 -79.006 46.226
76854 96.946 -78.981 46.204
76855 96.958 -78.955 46.182
76856 96.969 -78.943 46.161
76857 96.977 -78.963 46.140
76858 96.985 -78.993 46.119
76859 96.993 -79.021 46.100
76860 96.983 -79.019 46.082
76861 96.959 -79.006 46.065
76862 96.976 -78.987 46.046
76863 96.994 -78.979 46.027
76864 97.045 -79.140 46.008
76865 97.037 -79.256 45.989
76866 97.043 -79.247 45.970
76867 97.048 -79.239 45.953
76868 97.054 -79.231 45.936
76869 97.061 -79.222 45.918
76870 97.068 -79.213 45.902
76871 97.078 -79.202 45.886
76872 97.089 -79.190 45.869
76873 97.099 -79.177 45.850
76874 97.103 -79.154 45.833
76875 97.107 -79.132 45.817
76876 97.112 -79.109 45.801
76877 97.116 -79.087 45.785
76878 97.137 -79.071 45.767
76879 97.169 -79.078 45.749
76880 97.193 -79.109 45.730
76881 97.229 -79.134 45.713
76882 97.268 -79.156 45.697
76883 97.301 -79.159 45.682
76884 97.332 -79.153 45.666
76885 97.364 -79.148 45.650
76886 97.385 -79.151 45.633
76887 97.391 -79.165 45.619
76888 97.398 -79.179 45.605
76889 97.404 -79.193 45.590
76890 97.362 -79.232 45.576
76891 97.346 -79.251 45.562
76892 97.357 -79.252 45.548
76893 97.371 -79.253 45.533
76894 97.397 -79.263 45.520
76895 97.400 -79.276 45.505
76896 97.407 -79.282 45.491
76897 97.417 -79.286 45.478
76898 97.428 -79.288 45.464
76899 97.439 -79.291 45.450
76900 97.450 -79.294 45.436
76901 97.463 -79.299 45.422
76902 97.476 -79.304 45.409
76903 97.490 -79.310 45.396
76904 97.504 -79.316 45.383
76905 97.518 -79.322 45.369
76906 97.537 -79.337 45.356
76907 97.556 -79.350 45.342
76908 97.575 -79.365 45.329
76909 97.594 -79.379 45.315
76910 97.612 -79.393 45.301
76911 97.632 -79.407 45.288
76912 97.642 -79.414 45.275
76913 97.650 -79.425 45.262
76914 97.656 -79.431 45.251
76915 97.662 -79.436 45.243
76916 97.668 -79.441 45.234
76917 97.674 -79.447 45.225
76918 97.680 -79.452 45.214
76919 97.686 -79.457 45.205
76920 97.697 -79.491 45.196
76921 97.713 -79.523 45.185
76922 97.729 -79.554 45.174
76923 97.743 -79.564 45.163
76924 97.755 -79.571 45.153
76925 97.768 -79.579 45.141
76926 97.781 -79.585 45.129
76927 97.794 -79.593 45.117
76928 97.808 -79.604 45.104
76929 97.822 -79.616 45.091
76930 97.837 -79.627 45.080
76931 97.852 -79.639 45.072
76932 97.866 -79.651 45.065
76933 97.886 -79.664 45.057
76934 97.908 -79.676 45.048
76935 97.928 -79.689 45.039
76936 97.943 -79.695 45.031
76937 97.956 -79.701 45.024
76938 97.973 -79.709 45.016
76939 97.981 -79.733 45.007
76940 98.026 -79.735 45.000
76941 98.066 -79.752 44.992
76942 98.105 -79.773 44.985
76943 98.132 -79.786 44.977
76944 98.155 -79.798 44.969
76945 98.178 -79.811 44.961
76946 98.204 -79.817 44.952
76947 98.229 -79.820 44.943
76948 98.254 -79.825 44.932
76949 98.279 -79.829 44.917
76950 98.304 -79.833 44.902
76951 98.330 -79.838 44.889
76952 98.355 -79.844 44.877
76953 98.380 -79.850 44.868
76954 98.417 -79.861 44.859
76955 98.454 -79.872 44.851
76956 98.487 -79.880 44.843
76957 98.519 -79.888 44.835
76958 98.551 -79.895 44.826
76959 98.582 -79.902 44.818
76960 98.614 -79.909 44.809
76961 98.644 -79.915 44.800
76962 98.670 -79.922 44.793
76963 98.688 -79.935 44.784
76964 98.713 -79.943 44.775
76965 98.738 -79.952 44.767
76966 98.764 -79.960 44.761
76967 98.789 -79.968 44.754
76968 98.814 -79.977 44.746
76969 98.840 -79.985 44.738
76970 98.867 -79.991 44.729
76971 98.888 -79.996 44.721
76972 98.909 -79.996 44.712
76973 98.929 -79.994 44.705
76974 98.950 -79.993 44.698
76975 98.964 -79.988 44.692
76976 98.983 -80.002 44.687
76977 99.002 -80.016 44.681
76978 99.021 -80.030 44.675
76979 99.040 -80.044 44.668
76980 99.059 -80.059 44.664
76981 99.078 -80.072 44.659
76982 99.096 -80.083 44.654
76983 99.112 -80.088 44.649
76984 99.127 -80.097 44.643
76985 99.209 -80.128 44.638
76986 99.214 -80.128 44.632
76987 99.219 -80.129 44.627
76988 99.243 -80.153 44.622
76989 99.286 -80.173 44.617
76990 99.324 -80.177 44.613
76991 99.363 -80.181 44.609
76992 99.402 -80.185 44.605
76993 99.440 -80.189 44.599
76994 99.479 -80.193 44.595
76995 99.517 -80.196 44.591
76996 99.537 -80.210 44.586
76997 99.540 -80.232 44.581
76998 99.543 -80.255 44.576
76999 99.546 -80.277 44.571
77000 99.550 -80.299 44.565
77001 99.553 -80.321 44.559
77002 99.556 -80.344 44.553
77003 99.559 -80.366 44.546
77004 99.562 -80.388 44.540
77005 99.566 -80.410 44.534
77006 99.569 -80.431 44.527
77007 99.572 -80.454 44.520
77008 99.575 -80.476 44.512
77009 99.578 -80.498 44.504
77010 99.581 -80.520 44.496
77011 99.585 -80.542 44.489
77012 99.588 -80.565 44.481
77013 99.591 -80.587 44.472
77014 99.594 -80.609 44.464
77015 99.597 -80.631 44.455
77016 99.601 -80.654 44.445
77017 99.604 -80.676 44.436
77018 99.608 -80.698 44.426
77019 99.648 -80.706 44.416
77020 99.697 -80.711 44.406
77021 99.745 -80.717 44.396
77022 99.792 -80.723 44.386
77023 99.840 -80.728 44.375
77024 99.888 -80.734 44.363
77025 99.936 -80.739 44.353
77321 93.544 -78.882 53.534
77322 93.554 -78.880 53.525
77323 93.564 -78.877 53.515
77324 93.573 -78.874 53.506
77325 93.583 -78.872 53.498
77326 93.592 -78.870 53.490
77327 93.603 -78.860 53.483
77328 93.611 -78.853 53.476
77329 93.620 -78.846 53.469
77330 93.629 -78.839 53.465
77331 93.698 -78.805 53.463
77332 93.688 -78.802 53.464
77333 93.687 -78.801 53.464
77334 93.685 -78.799 53.462
77335 93.684 -78.797 53.459
77336 93.687 -78.795 53.453
77337 93.694 -78.795 53.446
77338 93.701 -78.794 53.438
77339 93.707 -78.793 53.427
77340 93.714 -78.793 53.417
77341 93.721 -78.789 53.407
77342 93.728 -78.786 53.396
77343 93.736 -78.783 53.383
77344 93.743 -78.781 53.370
77345 93.766 -78.768 53.355
77346 93.791 -78.754 53.338
77347 93.807 -78.742 53.321
77348 93.811 -78.734 53.303
77349 93.814 -78.725 53.285
77350 93.817 -78.717 53.265
77351 93.822 -78.708 53.244
77352 93.836 -78.701 53.223
77353 93.850 -78.692 53.202
77354 93.864 -78.684 53.182
77355 93.878 -78.677 53.163
77356 93.892 -78.668 53.141
77357 93.906 -78.661 53.117
77358 93.920 -78.652 53.091
77359 93.934 -78.645 53.065
77360 93.943 -78.639 53.040
77361 93.957 -78.634 53.013
77362 93.967 -78.633 52.986
77363 93.977 -78.631 52.960
77364 93.987 -78.629 52.936
77365 93.996 -78.627 52.913
77366 94.006 -78.626 52.890
77367 94.016 -78.624 52.869
77368 94.027 -78.623 52.849
77369 94.036 -78.620 52.830
77370 94.046 -78.619 52.810
77371 94.056 -78.617 52.788
77372 94.066 -78.616 52.765
77373 94.076 -78.614 52.742
77374 94.079 -78.617 52.720
77375 94.077 -78.624 52.697
77376 94.076 -78.606 52.674
77377 94.105 -78.551 52.650
77378 94.147 -78.529 52.625
77379 94.199 -78.542 52.599
77380 94.251 -78.556 52.573
77381 94.302 -78.570 52.551
77382 94.350 -78.582 52.529
77383 94.360 -78.579 52.508
77384 94.365 -78.580 52.489
77385 94.357 -78.592 52.467
77386 94.350 -78.604 52.445
77387 94.342 -78.617 52.424
77388 94.334 -78.628 52.402
77389 94.327 -78.640 52.380
77390 94.319 -78.652 52.357
77391 94.312 -78.664 52.335
77392 94.320 -78.660 52.312
77393 94.336 -78.647 52.288
77394 94.353 -78.635 52.261
77395 94.369 -78.623 52.235
77396 94.386 -78.611 52.209
77397 94.397 -78.604 52.184
77398 94.406 -78.601 52.158
77399 94.415 -78.599 52.129
77400 94.423 -78.597 52.098
77401 94.437 -78.596 52.065
77402 94.456 -78.597 52.030
77403 94.474 -78.598 51.996
77404 94.485 -78.602 51.964
77405 94.496 -78.616 51.929
77406 94.506 -78.649 51.894
77407 94.517 -78.683 51.859
77408 94.529 -78.702 51.825
77409 94.544 -78.701 51.787
77410 94.559 -78.700 51.748
77411 94.574 -78.699 51.708
77412 94.586 -78.696 51.668
77413 94.598 -78.692 51.629
77414 94.610 -78.689 51.589
77415 94.622 -78.685 51.554
77416 94.633 -78.680 51.520
77417 94.645 -78.677 51.486
77418 94.657 -78.673 51.455
77419 94.669 -78.669 51.423
77420 94.681 -78.665 51.392
77421 94.692 -78.661 51.361
77422 94.703 -78.663 51.325
77423 94.714 -78.665 51.288
77424 94.725 -78.667 51.251
77425 94.739 -78.670 51.214
77426 94.757 -78.673 51.176
77427 94.776 -78.677 51.138
77428 94.783 -78.684 51.099
77429 94.788 -78.692 51.059
77430 94.794 -78.699 51.018
77431 94.800 -78.707 50.977
77432 94.806 -78.714 50.935
77433 94.814 -78.704 50.894
77434 94.824 -78.691 50.853
77435 94.833 -78.678 50.809
77436 94.842 -78.665 50.764
77437 94.851 -78.652 50.720
77438 94.859 -78.645 50.677
77439 94.865 -78.641 50.636
77440 94.872 -78.637 50.591
77441 94.879 -78.634 50.546
77442 94.886 -78.638 50.500
77443 94.891 -78.649 50.452
77444 94.892 -78.664 50.405
77445 94.894 -78.677 50.361
77446 94.896 -78.692 50.322
77447 94.898 -78.705 50.284
77448 94.899 -78.720 50.246
77449 94.901 -78.733 50.206
77450 94.903 -78.748 50.166
77451 94.905 -78.762 50.124
77452 94.907 -78.776 50.080
77453 94.909 -78.789 50.034
77454 94.923 -78.794 49.989
77455 94.938 -78.799 49.945
77456 94.953 -78.803 49.902
77457 94.972 -78.805 49.860
77458 95.001 -78.802 49.819
77459 95.031 -78.799 49.779
77460 95.061 -78.795 49.738
77461 95.091 -78.791 49.700
77462 95.121 -78.787 49.662
77463 95.151 -78.783 49.623
77464 95.181 -78.780 49.584
77465 95.210 -78.777 49.544
77466 95.232 -78.774 49.504
77467 95.252 -78.793 49.464
77468 95.273 -78.805 49.423
77469 95.293 -78.817 49.383
77470 95.312 -78.829 49.341
77471 95.332 -78.840 49.301
77472 95.351 -78.852 49.262
77473 95.371 -78.864 49.222
77474 95.390 -78.876 49.183
77475 95.409 -78.886 49.143
77476 95.423 -78.883 49.103
77477 95.437 -78.881 49.062
77478 95.451 -78.878 49.023
77479 95.465 -78.876 48.983
77480 95.479 -78.874 48.943
77481 95.492 -78.871 48.903
77482 95.507 -78.869 48.862
77483 95.520 -78.867 48.820
77484 95.534 -78.865 48.779
77485 95.547 -78.875 48.737
77486 95.562 -78.888 48.696
77487 95.577 -78.902 48.655
77488 95.592 -78.915 48.613
77489 95.606 -78.928 48.572
77490 95.620 -78.941 48.531
77491 95.633 -78.955 48.492
77492 95.648 -78.968 48.451
77493 95.662 -78.981 48.408
77494 95.675 -78.994 48.362
77495 95.689 -79.008 48.313
77496 95.703 -79.021 48.265
77497 95.717 -79.034 48.219
77498 95.731 -79.048 48.176
77499 95.747 -79.056 48.133
77500 95.760 -79.057 48.090
77501 95.769 -79.053 48.046
77502 95.778 -79.049 48.005
77503 95.788 -79.046 47.963
77504 95.797 -79.042 47.919
77505 95.806 -79.038 47.877
77506 95.815 -79.034 47.835
77507 95.824 -79.031 47.793
77508 95.833 -79.027 47.750
77509 95.839 -79.023 47.707
77510 95.846 -79.020 47.664
77511 95.852 -79.016 47.620
77512 95.859 -79.012 47.578
77513 95.866 -79.009 47.537
77514 95.902 -79.005 47.496
77515 95.939 -79.001 47.456
77516 95.975 -78.997 47.416
77517 96.012 -78.993 47.376
77518 96.048 -78.990 47.336
77519 96.072 -78.984 47.297
77520 96.095 -78.977 47.259
77521 96.129 -78.976 47.222
77522 96.169 -78.977 47.184
77523 96.209 -78.979 47.148
77524 96.249 -78.981 47.112
77525 96.290 -78.982 47.078
77526 96.329 -78.984 47.044
77527 96.370 -78.985 47.012
77528 96.409 -78.987 46.981
77529 96.429 -78.995 46.950
77530 96.447 -79.004 46.920
77531 96.466 -79.014 46.890
77532 96.484 -79.023 46.858
77533 96.503 -79.033 46.828
77534 96.522 -79.042 46.797
77535 96.540 -79.052 46.766
77536 96.559 -79.061 46.732
77537 96.578 -79.071 46.700
77538 96.596 -79.080 46.668
77539 96.615 -79.090 46.637
77540 96.633 -79.100 46.607
77541 96.648 -79.106 46.576
77542 96.687 -79.109 46.545
77543 96.735 -79.111 46.516
77544 96.764 -79.103 46.487
77545 96.788 -79.093 46.459
77546 96.790 -79.075 46.431
77547 96.792 -79.056 46.405
77548 96.859 -79.021 46.380
77549 96.929 -78.984 46.355
77550 96.961 -78.938 46.330
77551 96.981 -78.954 46.305
77552 96.953 -78.981 46.280
77553 96.930 -79.009 46.255
77554 96.934 -78.979 46.233
77555 96.946 -78.954 46.212
77556 96.957 -78.928 46.191
77557 96.967 -78.933 46.170
77558 96.975 -78.952 46.149
77559 96.983 -78.980 46.129
77560 96.991 -79.009 46.108
77561 96.999 -79.037 46.091
77562 96.984 -79.002 46.074
77563 96.967 -78.962 46.056
77564 97.006 -78.972 46.037
77565 97.014 -79.100 46.019
77566 97.029 -79.228 46.000
77567 97.048 -79.267 45.982
77568 97.054 -79.259 45.965
77569 97.060 -79.250 45.948
77570 97.066 -79.242 45.928
77571 97.072 -79.234 45.911
77572 97.083 -79.222 45.895
77573 97.093 -79.211 45.878
77574 97.100 -79.192 45.859
77575 97.105 -79.170 45.843
77576 97.109 -79.147 45.828
77577 97.114 -79.125 45.811
77578 97.118 -79.102 45.796
77579 97.123 -79.079 45.779
77580 97.150 -79.084 45.761
77581 97.189 -79.106 45.742
77582 97.228 -79.129 45.726
77583 97.267 -79.152 45.711
77584 97.299 -79.151 45.695
77585 97.330 -79.146 45.680
77586 97.362 -79.140 45.664
77587 97.369 -79.154 45.649
77588 97.375 -79.169 45.634
77589 97.382 -79.182 45.621
77590 97.388 -79.197 45.609
77591 97.395 -79.210 45.596
77592 97.375 -79.237 45.582
77593 97.369 -79.250 45.567
77594 97.391 -79.251 45.552
77595 97.396 -79.265 45.537
77596 97.403 -79.271 45.523
77597 97.410 -79.278 45.509
77598 97.421 -79.280 45.496
77599 97.432 -79.283 45.483
77600 97.443 -79.286 45.469
77601 97.454 -79.288 45.455
77602 97.465 -79.291 45.441
77603 97.476 -79.294 45.427
77604 97.490 -79.300 45.414
77605 97.504 -79.306 45.400
77606 97.517 -79.312 45.387
77607 97.535 -79.324 45.373
77608 97.554 -79.338 45.358
77609 97.573 -79.352 45.344
77610 97.592 -79.366 45.330
77611 97.610 -79.380 45.317
77612 97.630 -79.393 45.303
77613 97.648 -79.400 45.288
77614 97.661 -79.407 45.275
77615 97.667 -79.413 45.264
77616 97.672 -79.419 45.256
77617 97.678 -79.424 45.247
77618 97.684 -79.430 45.239
77619 97.690 -79.436 45.229
77620 97.696 -79.442 45.219
77621 97.707 -79.457 45.209
77622 97.723 -79.488 45.198
77623 97.739 -79.520 45.187
77624 97.754 -79.540 45.176
77625 97.767 -79.547 45.168
77626 97.780 -79.554 45.155
77627 97.793 -79.562 45.142
77628 97.807 -79.573 45.129
77629 97.821 -79.585 45.115
77630 97.836 -79.597 45.102
77631 97.851 -79.608 45.092
77632 97.865 -79.620 45.087
77633 97.879 -79.632 45.081
77634 97.898 -79.644 45.074
77635 97.919 -79.656 45.065
77636 97.940 -79.667 45.057
77637 97.956 -79.675 45.049
77638 97.968 -79.679 45.041
77639 97.981 -79.684 45.033
77640 98.000 -79.693 45.023
77641 98.035 -79.711 45.013
77642 98.074 -79.732 45.005
77643 98.112 -79.752 44.997
77644 98.143 -79.767 44.989
77645 98.166 -79.780 44.980
77646 98.190 -79.789 44.970
77647 98.215 -79.793 44.960
77648 98.241 -79.798 44.949
77649 98.266 -79.802 44.937
77650 98.291 -79.806 44.923
77651 98.316 -79.811 44.910
77652 98.342 -79.815 44.898
77653 98.367 -79.819 44.887
77654 98.392 -79.825 44.880
77655 98.427 -79.835 44.871
77656 98.460 -79.843 44.862
77657 98.492 -79.850 44.851
77658 98.523 -79.858 44.841
77659 98.555 -79.864 44.832
77660 98.586 -79.872 44.824
77661 98.618 -79.879 44.816
77662 98.650 -79.886 44.809
77663 98.675 -79.895 44.802
77664 98.700 -79.903 44.794
77665 98.724 -79.912 44.785
77666 98.749 -79.921 44.777
77667 98.773 -79.930 44.770
77668 98.798 -79.938 44.764
77669 98.822 -79.947 44.757
77670 98.847 -79.955 44.749
77671 98.870 -79.956 44.740
77672 98.892 -79.952 44.731
77673 98.914 -79.947 44.724
77674 98.935 -79.943 44.717
77675 98.956 -79.942 44.711
77676 98.966 -79.930 44.705
77677 98.985 -79.943 44.700
77678 99.005 -79.960 44.694
77679 99.024 -79.977 44.688
77680 99.044 -79.994 44.682
77681 99.063 -80.012 44.677
77682 99.082 -80.028 44.672
77683 99.101 -80.046 44.667
77684 99.121 -80.062 44.663
77685 99.147 -80.077 44.657
77686 99.211 -80.102 44.652
77687 99.248 -80.121 44.646
77688 99.255 -80.126 44.640
77689 99.267 -80.126 44.635
77690 99.297 -80.150 44.631
77691 99.337 -80.151 44.628
77692 99.373 -80.151 44.623
77693 99.412 -80.155 44.617
77694 99.451 -80.159 44.613
77695 99.489 -80.162 44.608
77696 99.528 -80.166 44.602
77697 99.533 -80.187 44.597
77698 99.536 -80.209 44.592
77699 99.540 -80.231 44.586
77700 99.543 -80.254 44.581
77701 99.546 -80.276 44.575
77702 99.549 -80.298 44.567
77703 99.552 -80.320 44.561
77704 99.556 -80.343 44.555
77705 99.559 -80.365 44.548
77706 99.562 -80.387 44.540
77707 99.565 -80.409 44.533
77708 99.568 -80.431 44.527
77709 99.572 -80.453 44.519
77710 99.575 -80.475 44.510
77711 99.578 -80.497 44.503
77712 99.581 -80.519 44.495
77713 99.584 -80.542 44.487
77714 99.588 -80.564 44.478
77715 99.591 -80.586 44.469
77716 99.594 -80.608 44.460
77717 99.597 -80.630 44.451
77718 99.600 -80.653 44.441
77719 99.609 -80.673 44.431
77720 99.657 -80.679 44.421
77721 99.705 -80.684 44.411
77722 99.753 -80.690 44.402
77723 99.800 -80.695 44.390
77724 99.849 -80.701 44.379
77725 99.897 -80.707 44.367
77726 99.944 -80.712 44.356
77727 99.992 -80.718 44.346
77728 100.040 -80.723 44.335
78022 93.545 -78.871 53.545
78023 93.554 -78.868 53.536
78024 93.564 -78.866 53.527
78025 93.573 -78.864 53.518
78026 93.583 -78.861 53.509
78027 93.593 -78.858 53.501
78028 93.603 -78.855 53.494
78029 93.613 -78.847 53.488
78030 93.621 -78.840 53.481
78031 93.629 -78.833 53.478
78032 93.686 -78.805 53.477
78033 93.676 -78.801 53.478
78034 93.675 -78.799 53.478
78035 93.674 -78.797 53.477
78036 93.678 -78.795 53.474
78037 93.683 -78.790 53.468
78038 93.690 -78.787 53.461
78039 93.697 -78.786 53.453
78040 93.703 -78.786 53.443
78041 93.710 -78.785 53.432
78042 93.717 -78.782 53.421
78043 93.724 -78.780 53.410
78044 93.731 -78.777 53.397
78045 93.753 -78.765 53.382
78046 93.777 -78.751 53.366
78047 93.802 -78.737 53.349
78048 93.808 -78.728 53.331
78049 93.812 -78.720 53.313
78050 93.815 -78.711 53.295
78051 93.819 -78.702 53.274
78052 93.822 -78.694 53.254
78053 93.835 -78.686 53.233
78054 93.849 -78.678 53.212
78055 93.863 -78.670 53.192
78056 93.877 -78.662 53.172
78057 93.891 -78.654 53.150
78058 93.905 -78.646 53.126
78059 93.919 -78.638 53.101
78060 93.933 -78.630 53.075
78061 93.945 -78.619 53.048
78062 93.956 -78.625 53.021
78063 93.966 -78.624 52.995
78064 93.976 -78.623 52.970
78065 93.986 -78.621 52.946
78066 93.996 -78.620 52.923
78067 94.006 -78.617 52.900
78068 94.016 -78.616 52.881
78069 94.026 -78.614 52.862
78070 94.036 -78.613 52.843
78071 94.046 -78.611 52.824
78072 94.055 -78.610 52.802
78073 94.065 -78.608 52.778
78074 94.075 -78.606 52.754
78075 94.075 -78.611 52.731
78076 94.073 -78.616 52.708
78077 94.081 -78.567 52.685
78078 94.115 -78.522 52.661
78079 94.176 -78.524 52.636
78080 94.235 -78.527 52.610
78081 94.287 -78.541 52.586
78082 94.339 -78.554 52.565
78083 94.373 -78.561 52.544
78084 94.379 -78.562 52.525
78085 94.371 -78.574 52.506
78086 94.363 -78.586 52.485
78087 94.356 -78.598 52.463
78088 94.348 -78.611 52.441
78089 94.340 -78.623 52.419
78090 94.333 -78.634 52.397
78091 94.325 -78.646 52.374
78092 94.318 -78.658 52.355
78093 94.322 -78.658 52.332
78094 94.339 -78.645 52.308
78095 94.355 -78.633 52.282
78096 94.371 -78.620 52.256
78097 94.388 -78.608 52.231
78098 94.404 -78.596 52.207
78099 94.415 -78.590 52.179
78100 94.424 -78.588 52.150
78101 94.432 -78.586 52.118
78102 94.447 -78.585 52.083
78103 94.465 -78.586 52.049
78104 94.483 -78.586 52.015
78105 94.497 -78.589 51.982
78106 94.508 -78.618 51.946
78107 94.519 -78.651 51.909
78108 94.529 -78.685 51.872
78109 94.541 -78.706 51.839
78110 94.553 -78.702 51.801
78111 94.564 -78.698 51.761
78112 94.576 -78.695 51.720
78113 94.588 -78.691 51.681
78114 94.600 -78.687 51.641
78115 94.612 -78.683 51.602
78116 94.624 -78.680 51.566
78117 94.635 -78.676 51.532
78118 94.647 -78.672 51.498
78119 94.659 -78.668 51.465
78120 94.671 -78.664 51.432
78121 94.683 -78.660 51.399
78122 94.694 -78.656 51.366
78123 94.705 -78.657 51.331
78124 94.716 -78.659 51.295
78125 94.727 -78.661 51.258
78126 94.745 -78.664 51.221
78127 94.764 -78.668 51.183
78128 94.782 -78.672 51.145
78129 94.792 -78.678 51.105
78130 94.798 -78.686 51.066
78131 94.804 -78.693 51.026
78132 94.810 -78.701 50.986
78133 94.815 -78.706 50.944
78134 94.825 -78.693 50.903
78135 94.834 -78.680 50.861
78136 94.843 -78.667 50.818
78137 94.852 -78.655 50.773
78138 94.861 -78.642 50.728
78139 94.871 -78.629 50.684
78140 94.879 -78.619 50.642
78141 94.886 -78.616 50.597
78142 94.893 -78.612 50.550
78143 94.899 -78.622 50.503
78144 94.906 -78.631 50.455
78145 94.911 -78.642 50.409
78146 94.913 -78.656 50.364
78147 94.915 -78.670 50.324
78148 94.917 -78.684 50.285
78149 94.918 -78.698 50.246
78150 94.920 -78.713 50.205
78151 94.922 -78.727 50.164
78152 94.924 -78.741 50.121
78153 94.926 -78.755 50.078
78154 94.933 -78.765 50.032
78155 94.948 -78.769 49.988
78156 94.962 -78.774 49.945
78157 94.977 -78.778 49.902
78158 94.992 -78.783 49.860
78159 95.010 -78.786 49.819
78160 95.039 -78.782 49.779
78161 95.069 -78.778 49.739
78162 95.099 -78.775 49.700
78163 95.129 -78.771 49.662
78164 95.159 -78.767 49.624
78165 95.189 -78.764 49.586
78166 95.218 -78.760 49.546
78167 95.215 -78.761 49.505
78168 95.244 -78.780 49.465
78169 95.273 -78.800 49.424
78170 95.293 -78.812 49.384
78171 95.313 -78.824 49.343
78172 95.332 -78.836 49.303
78173 95.351 -78.848 49.263
78174 95.371 -78.859 49.223
78175 95.391 -78.871 49.185
78176 95.409 -78.881 49.146
78177 95.423 -78.879 49.106
78178 95.438 -78.877 49.066
78179 95.451 -78.874 49.025
78180 95.465 -78.872 48.986
78181 95.479 -78.870 48.946
78182 95.493 -78.868 48.905
78183 95.507 -78.865 48.864
78184 95.521 -78.863 48.823
78185 95.529 -78.855 48.781
78186 95.545 -78.852 48.739
78187 95.551 -78.863 48.698
78188 95.562 -78.875 48.657
78189 95.576 -78.888 48.614
78190 95.590 -78.902 48.572
78191 95.604 -78.915 48.530
78192 95.617 -78.928 48.488
78193 95.631 -78.942 48.445
78194 95.645 -78.955 48.401
78195 95.659 -78.968 48.357
78196 95.673 -78.981 48.310
78197 95.687 -78.995 48.265
78198 95.701 -79.008 48.222
78199 95.715 -79.021 48.181
78200 95.730 -79.031 48.140
78201 95.747 -79.034 48.097
78202 95.764 -79.038 48.054
78203 95.780 -79.042 48.013
78204 95.796 -79.044 47.970
78205 95.805 -79.040 47.928
78206 95.814 -79.037 47.884
78207 95.823 -79.033 47.842
78208 95.832 -79.029 47.800
78209 95.841 -79.025 47.757
78210 95.849 -79.021 47.714
78211 95.855 -79.018 47.672
78212 95.862 -79.014 47.628
78213 95.869 -79.010 47.586
78214 95.897 -79.006 47.545
78215 95.933 -79.002 47.504
78216 95.970 -78.999 47.465
78217 96.005 -78.995 47.425
78218 96.028 -78.989 47.385
78219 96.051 -78.983 47.345
78220 96.075 -78.976 47.307
78221 96.098 -78.970 47.268
78222 96.129 -78.968 47.231
78223 96.169 -78.969 47.193
78224 96.209 -78.971 47.157
78225 96.249 -78.972 47.120
78226 96.289 -78.974 47.085
78227 96.329 -78.975 47.050
78228 96.369 -78.977 47.018
78229 96.404 -78.971 46.988
78230 96.416 -78.971 46.957
78231 96.422 -78.979 46.926
78232 96.441 -78.989 46.896
78233 96.460 -78.998 46.865
78234 96.478 -79.008 46.835
78235 96.497 -79.018 46.804
78236 96.516 -79.027 46.773
78237 96.534 -79.037 46.741
78238 96.553 -79.046 46.709
78239 96.572 -79.056 46.677
78240 96.590 -79.065 46.645
78241 96.605 -79.072 46.614
78242 96.620 -79.079 46.583
78243 96.668 -79.081 46.552
78244 96.717 -79.083 46.522
78245 96.756 -79.081 46.494
78246 96.785 -79.072 46.466
78247 96.814 -79.065 46.438
78248 96.826 -79.050 46.412
78249 96.879 -79.011 46.386
78250 96.911 -78.963 46.361
78251 96.937 -78.947 46.336
78252 96.957 -78.963 46.311
78253 96.946 -78.985 46.287
78254 96.924 -78.996 46.263
78255 96.934 -78.953 46.242
78256 96.945 -78.927 46.221
78257 96.956 -78.905 46.201
78258 96.965 -78.924 46.181
78259 96.974 -78.942 46.161
78260 96.982 -78.967 46.141
78261 96.990 -78.996 46.120
78262 96.990 -79.002 46.101
78263 96.974 -78.964 46.083
78264 96.970 -78.967 46.065
78265 96.945 -79.078 46.046
78266 97.004 -79.047 46.028
78267 97.019 -79.192 46.009
78268 97.037 -79.231 45.992
78269 97.056 -79.269 45.976
78270 97.065 -79.270 45.959
78271 97.071 -79.262 45.941
78272 97.077 -79.253 45.924
78273 97.087 -79.242 45.907
78274 97.097 -79.230 45.890
78275 97.102 -79.207 45.871
78276 97.106 -79.185 45.855
78277 97.111 -79.162 45.839
78278 97.115 -79.140 45.823
78279 97.124 -79.109 45.807
78280 97.133 -79.078 45.791
78281 97.149 -79.079 45.774
78282 97.188 -79.102 45.757
78283 97.227 -79.125 45.741
78284 97.265 -79.147 45.726
78285 97.297 -79.143 45.710
78286 97.328 -79.138 45.696
78287 97.346 -79.144 45.680
78288 97.353 -79.158 45.664
78289 97.359 -79.172 45.650
78290 97.366 -79.186 45.637
78291 97.372 -79.200 45.625
78292 97.379 -79.214 45.613
78293 97.385 -79.228 45.600
78294 97.388 -79.244 45.586
78295 97.393 -79.253 45.571
78296 97.399 -79.259 45.555
78297 97.406 -79.266 45.540
78298 97.414 -79.272 45.527
78299 97.425 -79.275 45.515
78300 97.436 -79.278 45.502
78301 97.447 -79.280 45.489
78302 97.457 -79.283 45.475
78303 97.468 -79.285 45.460
78304 97.479 -79.288 45.446
78305 97.491 -79.291 45.433
78306 97.504 -79.296 45.419
78307 97.517 -79.302 45.405
78308 97.532 -79.310 45.390
78309 97.551 -79.325 45.374
78310 97.571 -79.338 45.359
78311 97.589 -79.353 45.346
78312 97.608 -79.363 45.333
78313 97.627 -79.370 45.317
78314 97.646 -79.378 45.301
78315 97.664 -79.385 45.287
78316 97.677 -79.391 45.277
78317 97.683 -79.397 45.268
78318 97.689 -79.404 45.260
78319 97.695 -79.410 45.252
78320 97.703 -79.418 45.243
78321 97.714 -79.433 45.233
78322 97.725 -79.448 45.221
78323 97.736 -79.463 45.211
78324 97.749 -79.486 45.200
78325 97.766 -79.515 45.191
78326 97.778 -79.523 45.182
78327 97.791 -79.531 45.171
78328 97.806 -79.542 45.156
78329 97.820 -79.554 45.140
78330 97.835 -79.566 45.125
78331 97.849 -79.577 45.112
78332 97.864 -79.589 45.103
78333 97.878 -79.601 45.099
78334 97.892 -79.613 45.095
78335 97.909 -79.625 45.089
78336 97.930 -79.636 45.081
78337 97.950 -79.647 45.074
78338 97.973 -79.657 45.065
78339 97.992 -79.667 45.056
78340 98.012 -79.676 45.046
78341 98.031 -79.686 45.037
78342 98.051 -79.695 45.028
78343 98.082 -79.711 45.020
78344 98.121 -79.732 45.012
78345 98.154 -79.749 45.003
78346 98.177 -79.761 44.993
78347 98.202 -79.767 44.981
78348 98.227 -79.770 44.972
78349 98.252 -79.775 44.960
78350 98.278 -79.780 44.948
78351 98.303 -79.783 44.935
78352 98.329 -79.788 44.922
78353 98.354 -79.792 44.908
78354 98.379 -79.796 44.898
78355 98.405 -79.802 44.890
78356 98.435 -79.811 44.882
78357 98.468 -79.820 44.871
78358 98.501 -79.830 44.860
78359 98.533 -79.839 44.849
78360 98.566 -79.848 44.840
78361 98.598 -79.857 44.832
78362 98.631 -79.867 44.825
78363 98.662 -79.874 44.818
78364 98.679 -79.872 44.811
78365 98.700 -79.880 44.803
78366 98.721 -79.887 44.795
78367 98.743 -79.896 44.788
78368 98.768 -79.904 44.781
78369 98.793 -79.913 44.774
78370 98.817 -79.921 44.768
78371 98.842 -79.930 44.760
78372 98.865 -79.931 44.751
78373 98.887 -79.927 44.743
78374 98.909 -79.922 44.736
78375 98.932 -79.918 44.730
78376 98.957 -79.919 44.724
78377 98.973 -79.914 44.718
78378 98.992 -79.926 44.713
78379 99.011 -79.943 44.707
78380 99.030 -79.960 44.701
78381 99.050 -79.977 44.696
78382 99.069 -79.994 44.691
78383 99.089 -80.011 44.686
78384 99.108 -80.028 44.681
78385 99.128 -80.045 44.675
78386 99.157 -80.059 44.669
78387 99.189 -80.071 44.664
78388 99.243 -80.093 44.658
78389 99.272 -80.109 44.653
78390 99.277 -80.118 44.650
78391 99.305 -80.127 44.647
78392 99.353 -80.137 44.644
78393 99.396 -80.142 44.639
78394 99.425 -80.128 44.634
78395 99.462 -80.128 44.631
78396 99.500 -80.132 44.626
78397 99.526 -80.143 44.620
78398 99.529 -80.165 44.612
78399 99.533 -80.187 44.604
78400 99.536 -80.209 44.597
78401 99.539 -80.231 44.590
78402 99.543 -80.253 44.583
78403 99.546 -80.275 44.575
78404 99.549 -80.297 44.568
78405 99.552 -80.319 44.562
78406 99.555 -80.342 44.555
78407 99.559 -80.364 44.547
78408 99.562 -80.386 44.539
78409 99.565 -80.408 44.533
78410 99.568 -80.431 44.524
78411 99.571 -80.453 44.516
78412 99.575 -80.475 44.509
78413 99.578 -80.497 44.501
78414 99.581 -80.519 44.492
78415 99.584 -80.541 44.484
78416 99.587 -80.563 44.475
78417 99.591 -80.585 44.466
78418 99.594 -80.607 44.457
78419 99.597 -80.630 44.446
78420 99.617 -80.646 44.436
78421 99.665 -80.651 44.427
78422 99.713 -80.657 44.417
78423 99.761 -80.663 44.406
78424 99.809 -80.668 44.394
78425 99.857 -80.674 44.383
78426 99.905 -80.679 44.372
78427 99.952 -80.685 44.361
78428 100.001 -80.691 44.350
78429 100.049 -80.696 44.339
78430 100.097 -80.701 44.328
78724 93.554 -78.858 53.546
78725 93.564 -78.855 53.537
78726 93.574 -78.852 53.528
78727 93.583 -78.850 53.520
78728 93.593 -78.848 53.512
78729 93.603 -78.846 53.504
78730 93.613 -78.841 53.499
78731 93.622 -78.833 53.493
78732 93.630 -78.827 53.490
78733 93.674 -78.804 53.490
78734 93.664 -78.799 53.492
78735 93.665 -78.796 53.491
78736 93.670 -78.794 53.490
78737 93.675 -78.791 53.487
78738 93.680 -78.786 53.482
78739 93.686 -78.782 53.475
78740 93.692 -78.779 53.467
78741 93.699 -78.779 53.457
78742 93.706 -78.777 53.446
78743 93.713 -78.775 53.435
78744 93.720 -78.772 53.423
78745 93.739 -78.762 53.409
78746 93.764 -78.748 53.393
78747 93.788 -78.734 53.377
78748 93.806 -78.722 53.360
78749 93.809 -78.714 53.342
78750 93.813 -78.705 53.324
78751 93.816 -78.697 53.306
78752 93.820 -78.688 53.285
78753 93.823 -78.680 53.263
78754 93.834 -78.671 53.242
78755 93.848 -78.664 53.221
78756 93.862 -78.655 53.200
78757 93.876 -78.648 53.180
78758 93.890 -78.639 53.159
78759 93.904 -78.632 53.135
78760 93.918 -78.622 53.110
78761 93.930 -78.609 53.083
78762 93.943 -78.596 53.056
78763 93.954 -78.601 53.029
78764 93.965 -78.608 53.003
78765 93.975 -78.614 52.979
78766 93.985 -78.613 52.956
78767 93.995 -78.611 52.933
78768 94.005 -78.610 52.911
78769 94.015 -78.608 52.892
78770 94.025 -78.607 52.874
78771 94.035 -78.604 52.855
78772 94.045 -78.603 52.837
78773 94.055 -78.601 52.815
78774 94.065 -78.600 52.791
78775 94.073 -78.599 52.766
78776 94.072 -78.606 52.743
78777 94.065 -78.582 52.720
78778 94.085 -78.529 52.696
78779 94.140 -78.520 52.672
78780 94.201 -78.523 52.647
78781 94.262 -78.525 52.622
78782 94.323 -78.526 52.599
78783 94.374 -78.539 52.579
78784 94.393 -78.544 52.560
78785 94.385 -78.556 52.541
78786 94.378 -78.568 52.522
78787 94.369 -78.580 52.502
78788 94.362 -78.592 52.480
78789 94.354 -78.604 52.458
78790 94.347 -78.617 52.435
78791 94.339 -78.629 52.414
78792 94.331 -78.640 52.392
78793 94.324 -78.652 52.372
78794 94.324 -78.656 52.350
78795 94.341 -78.644 52.325
78796 94.357 -78.631 52.301
78797 94.373 -78.619 52.276
78798 94.390 -78.607 52.251
78799 94.406 -78.594 52.227
78800 94.422 -78.582 52.199
78801 94.433 -78.576 52.169
78802 94.442 -78.575 52.137
78803 94.456 -78.574 52.102
78804 94.474 -78.574 52.068
78805 94.492 -78.575 52.034
78806 94.506 -78.580 51.998
78807 94.515 -78.611 51.961
78808 94.524 -78.642 51.924
78809 94.533 -78.673 51.886
78810 94.542 -78.692 51.850
78811 94.554 -78.692 51.813
78812 94.566 -78.691 51.774
78813 94.578 -78.689 51.733
78814 94.590 -78.686 51.693
78815 94.602 -78.682 51.654
78816 94.614 -78.678 51.615
78817 94.626 -78.674 51.578
78818 94.637 -78.670 51.543
78819 94.649 -78.667 51.509
78820 94.661 -78.663 51.474
78821 94.673 -78.659 51.440
78822 94.685 -78.655 51.406
78823 94.696 -78.651 51.372
78824 94.708 -78.651 51.338
78825 94.718 -78.653 51.302
78826 94.733 -78.656 51.265
78827 94.751 -78.659 51.227
78828 94.770 -78.663 51.190
78829 94.789 -78.667 51.151
78830 94.802 -78.672 51.111
78831 94.808 -78.680 51.073
78832 94.814 -78.687 51.034
78833 94.819 -78.694 50.993
78834 94.826 -78.696 50.953
78835 94.835 -78.683 50.911
78836 94.844 -78.670 50.869
78837 94.853 -78.658 50.827
78838 94.862 -78.645 50.781
78839 94.872 -78.632 50.735
78840 94.881 -78.619 50.691
78841 94.890 -78.606 50.647
78842 94.899 -78.594 50.601
78843 94.906 -78.596 50.554
78844 94.913 -78.605 50.506
78845 94.920 -78.615 50.458
78846 94.927 -78.624 50.411
78847 94.932 -78.635 50.366
78848 94.934 -78.649 50.325
78849 94.935 -78.664 50.284
78850 94.937 -78.677 50.244
78851 94.939 -78.692 50.202
78852 94.941 -78.705 50.160
78853 94.943 -78.720 50.118
78854 94.945 -78.733 50.074
78855 94.957 -78.740 50.030
78856 94.972 -78.745 49.987
78857 94.986 -78.749 49.944
78858 95.001 -78.753 49.902
78859 95.016 -78.758 49.860
78860 95.031 -78.762 49.819
78861 95.047 -78.766 49.779
78862 95.077 -78.762 49.741
78863 95.107 -78.758 49.702
78864 95.137 -78.755 49.664
78865 95.167 -78.751 49.627
78866 95.196 -78.748 49.589
78867 95.227 -78.744 49.549
78868 95.207 -78.749 49.507
78869 95.236 -78.769 49.466
78870 95.265 -78.789 49.426
78871 95.294 -78.808 49.386
78872 95.313 -78.820 49.345
78873 95.332 -78.831 49.306
78874 95.352 -78.843 49.267
78875 95.371 -78.855 49.228
78876 95.391 -78.866 49.189
78877 95.410 -78.877 49.149
78878 95.424 -78.875 49.109
78879 95.438 -78.873 49.069
78880 95.452 -78.871 49.029
78881 95.466 -78.868 48.989
78882 95.479 -78.866 48.949
78883 95.493 -78.864 48.908
78884 95.508 -78.861 48.867
78885 95.516 -78.854 48.825
78886 95.520 -78.843 48.783
78887 95.535 -78.839 48.741
78888 95.554 -78.843 48.699
78889 95.559 -78.854 48.657
78890 95.565 -78.864 48.615
78891 95.573 -78.875 48.572
78892 95.588 -78.889 48.530
78893 95.601 -78.902 48.487
78894 95.615 -78.915 48.443
78895 95.629 -78.928 48.399
78896 95.643 -78.942 48.355
78897 95.657 -78.955 48.312
78898 95.671 -78.968 48.270
78899 95.685 -78.982 48.228
78900 95.699 -78.995 48.188
78901 95.714 -79.006 48.146
78902 95.731 -79.009 48.105
78903 95.747 -79.013 48.063
78904 95.764 -79.017 48.021
78905 95.781 -79.021 47.979
78906 95.798 -79.024 47.935
78907 95.815 -79.028 47.893
78908 95.830 -79.031 47.850
78909 95.839 -79.027 47.807
78910 95.848 -79.023 47.764
78911 95.858 -79.019 47.721
78912 95.865 -79.015 47.679
78913 95.872 -79.012 47.636
78914 95.891 -79.009 47.595
78915 95.927 -79.005 47.554
78916 95.961 -79.000 47.513
78917 95.984 -78.994 47.474
78918 96.007 -78.987 47.433
78919 96.030 -78.981 47.393
78920 96.053 -78.975 47.354
78921 96.077 -78.969 47.315
78922 96.100 -78.963 47.277
78923 96.129 -78.960 47.240
78924 96.169 -78.962 47.201
78925 96.209 -78.963 47.165
78926 96.249 -78.965 47.129
78927 96.289 -78.966 47.092
78928 96.329 -78.968 47.058
78929 96.365 -78.964 47.026
78930 96.398 -78.956 46.995
78931 96.424 -78.951 46.964
78932 96.419 -78.957 46.933
78933 96.416 -78.964 46.902
78934 96.435 -78.974 46.872
78935 96.454 -78.983 46.842
78936 96.472 -78.993 46.812
78937 96.491 -79.002 46.780
78938 96.510 -79.012 46.749
78939 96.528 -79.021 46.717
78940 96.547 -79.031 46.685
78941 96.562 -79.038 46.653
78942 96.576 -79.045 46.621
78943 96.601 -79.050 46.589
78944 96.649 -79.052 46.558
78945 96.697 -79.054 46.529
78946 96.745 -79.056 46.500
78947 96.778 -79.049 46.472
78948 96.806 -79.042 46.444
78949 96.844 -79.036 46.418
78950 96.861 -78.990 46.392
78951 96.892 -78.943 46.367
78952 96.913 -78.956 46.342
78953 96.933 -78.972 46.318
78954 96.942 -78.990 46.294
78955 96.939 -78.993 46.271
78956 96.934 -78.929 46.251
78957 96.945 -78.899 46.231
78958 96.955 -78.895 46.212
78959 96.963 -78.914 46.193
78960 96.972 -78.932 46.173
78961 96.980 -78.954 46.153
78962 96.988 -78.983 46.131
78963 96.979 -78.964 46.112
78964 96.973 -78.959 46.093
78965 96.973 -78.974 46.074
78966 96.973 -78.990 46.056
78967 96.997 -79.008 46.038
78968 97.008 -79.155 46.020
78969 97.027 -79.194 46.003
78970 97.046 -79.233 45.987
78971 97.065 -79.272 45.971
78972 97.076 -79.281 45.955
78973 97.082 -79.273 45.937
78974 97.092 -79.263 45.919
78975 97.099 -79.245 45.901
78976 97.103 -79.223 45.883
78977 97.108 -79.200 45.867
78978 97.116 -79.171 45.852
78979 97.125 -79.140 45.836
78980 97.134 -79.109 45.820
78981 97.143 -79.078 45.804
78982 97.158 -79.081 45.788
78983 97.186 -79.097 45.771
78984 97.226 -79.120 45.756
78985 97.264 -79.141 45.741
78986 97.295 -79.136 45.726
78987 97.323 -79.134 45.712
78988 97.330 -79.147 45.697
78989 97.336 -79.162 45.681
78990 97.343 -79.175 45.666
78991 97.350 -79.190 45.653
78992 97.356 -79.203 45.639
78993 97.363 -79.217 45.626
78994 97.371 -79.231 45.616
78995 97.384 -79.240 45.604
78996 97.395 -79.248 45.590
78997 97.402 -79.255 45.572
78998 97.409 -79.261 45.557
78999 97.417 -79.266 45.546
79000 97.428 -79.269 45.536
79001 97.439 -79.272 45.523
79002 97.450 -79.275 45.509
79003 97.461 -79.278 45.495
79004 97.472 -79.280 45.481
79005 97.483 -79.283 45.467
79006 97.494 -79.285 45.452
79007 97.505 -79.288 45.437
79008 97.517 -79.292 45.422
79009 97.530 -79.297 45.407
79010 97.549 -79.311 45.393
79011 97.568 -79.325 45.376
79012 97.587 -79.334 45.361
79013 97.606 -79.341 45.345
79014 97.625 -79.348 45.328
79015 97.643 -79.355 45.312
79016 97.662 -79.363 45.299
79017 97.681 -79.370 45.291
79018 97.694 -79.376 45.282
79019 97.700 -79.382 45.274
79020 97.710 -79.395 45.266
79021 97.721 -79.410 45.257
79022 97.732 -79.426 45.247
79023 97.743 -79.441 45.236
79024 97.754 -79.456 45.224
79025 97.766 -79.471 45.212
79026 97.777 -79.486 45.202
79027 97.790 -79.500 45.193
79028 97.804 -79.513 45.177
79029 97.817 -79.527 45.161
79030 97.831 -79.541 45.145
79031 97.845 -79.554 45.131
79032 97.858 -79.569 45.118
79033 97.871 -79.582 45.109
79034 97.885 -79.596 45.104
79035 97.897 -79.608 45.098
79036 97.911 -79.620 45.092
79037 97.931 -79.631 45.086
79038 97.952 -79.642 45.079
79039 97.975 -79.652 45.073
79040 98.004 -79.661 45.066
79041 98.032 -79.670 45.057
79042 98.061 -79.679 45.049
79043 98.083 -79.689 45.041
79044 98.102 -79.698 45.033
79045 98.129 -79.711 45.025
79046 98.165 -79.730 45.016
79047 98.189 -79.739 45.006
79048 98.214 -79.744 44.995
79049 98.239 -79.748 44.984
79050 98.265 -79.752 44.972
79051 98.290 -79.757 44.959
79052 98.316 -79.763 44.946
79053 98.343 -79.770 44.931
79054 98.369 -79.776 44.918
79055 98.395 -79.783 44.907
79056 98.422 -79.789 44.899
79057 98.449 -79.800 44.890
79058 98.482 -79.809 44.879
79059 98.515 -79.818 44.869
79060 98.548 -79.828 44.858
79061 98.580 -79.837 44.849
79062 98.613 -79.846 44.841
79063 98.646 -79.855 44.834
79064 98.670 -79.858 44.827
79065 98.685 -79.852 44.821
79066 98.706 -79.858 44.812
79067 98.727 -79.866 44.806
79068 98.748 -79.874 44.799
79069 98.769 -79.880 44.793
79070 98.790 -79.888 44.785
79071 98.812 -79.896 44.779
79072 98.837 -79.905 44.772
79073 98.860 -79.906 44.764
79074 98.883 -79.904 44.757
79075 98.908 -79.904 44.751
79076 98.934 -79.905 44.744
79077 98.959 -79.905 44.739
79078 98.981 -79.904 44.734
79079 98.998 -79.909 44.728
79080 99.018 -79.926 44.721
79081 99.038 -79.943 44.714
79082 99.057 -79.960 44.708
79083 99.076 -79.977 44.704
79084 99.095 -79.994 44.699
79085 99.115 -80.011 44.694
79086 99.135 -80.027 44.689
79087 99.167 -80.040 44.682
79088 99.199 -80.052 44.676
79089 99.231 -80.065 44.671
79090 99.276 -80.083 44.667
79091 99.296 -80.096 44.664
79092 99.303 -80.098 44.661
79093 99.349 -80.108 44.658
79094 99.404 -80.122 44.654
79095 99.454 -80.133 44.650
79096 99.484 -80.119 44.647
79097 99.513 -80.106 44.642
79098 99.523 -80.120 44.635
79099 99.526 -80.142 44.627
79100 99.529 -80.164 44.617
79101 99.533 -80.186 44.608
79102 99.536 -80.208 44.599
79103 99.539 -80.231 44.591
79104 99.542 -80.253 44.583
79105 99.545 -80.275 44.576
79106 99.549 -80.297 44.569
79107 99.552 -80.319 44.561
79108 99.555 -80.341 44.554
79109 99.558 -80.363 44.546
79110 99.561 -80.385 44.539
79111 99.565 -80.407 44.530
79112 99.568 -80.430 44.521
79113 99.571 -80.452 44.513
79114 99.574 -80.474 44.506
79115 99.577 -80.496 44.498
79116 99.581 -80.519 44.489
79117 99.584 -80.541 44.481
79118 99.587 -80.563 44.471
79119 99.590 -80.585 44.462
79120 99.593 -80.607 44.451
79121 99.625 -80.619 44.441
79122 99.674 -80.624 44.431
79123 99.721 -80.629 44.421
79124 99.769 -80.635 44.410
79125 99.817 -80.641 44.399
79126 99.865 -80.647 44.387
79127 99.913 -80.652 44.376
79128 99.961 -80.657 44.365
79129 100.009 -80.663 44.353
79130 100.057 -80.669 44.342
79131 100.105 -80.675 44.331
79425 93.555 -78.846 53.555
79426 93.564 -78.844 53.545
79427 93.574 -78.842 53.537
79428 93.584 -78.839 53.529
79429 93.593 -78.836 53.521
79430 93.603 -78.834 53.514
79431 93.613 -78.832 53.510
79432 93.623 -78.827 53.505
79433 93.631 -78.820 53.502
79434 93.662 -78.803 53.503
79435 93.658 -78.796 53.505
79436 93.662 -78.794 53.504
79437 93.666 -78.791 53.502
79438 93.672 -78.788 53.499
79439 93.677 -78.783 53.495
79440 93.683 -78.778 53.489
79441 93.689 -78.774 53.481
79442 93.695 -78.771 53.471
79443 93.702 -78.771 53.460
79444 93.709 -78.767 53.448
79445 93.726 -78.759 53.435
79446 93.751 -78.745 53.420
79447 93.775 -78.731 53.404
79448 93.800 -78.717 53.388
79449 93.807 -78.708 53.370
79450 93.811 -78.699 53.352
79451 93.814 -78.691 53.333
79452 93.818 -78.683 53.315
79453 93.821 -78.673 53.294
79454 93.824 -78.665 53.272
79455 93.833 -78.657 53.251
79456 93.847 -78.649 53.230
79457 93.861 -78.641 53.209
79458 93.875 -78.633 53.188
79459 93.889 -78.625 53.167
79460 93.903 -78.614 53.143
79461 93.915 -78.601 53.118
79462 93.928 -78.588 53.091
79463 93.940 -78.575 53.064
79464 93.952 -78.579 53.038
79465 93.963 -78.585 53.013
79466 93.973 -78.592 52.990
79467 93.984 -78.598 52.967
79468 93.994 -78.604 52.944
79469 94.004 -78.601 52.923
79470 94.014 -78.600 52.904
79471 94.024 -78.598 52.886
79472 94.034 -78.597 52.868
79473 94.044 -78.595 52.848
79474 94.054 -78.594 52.827
79475 94.064 -78.592 52.804
79476 94.070 -78.593 52.779
79477 94.067 -78.590 52.755
79478 94.062 -78.546 52.731
79479 94.105 -78.517 52.707
79480 94.166 -78.520 52.682
79481 94.226 -78.521 52.658
79482 94.287 -78.523 52.634
79483 94.348 -78.526 52.612
79484 94.400 -78.521 52.594
79485 94.399 -78.539 52.575
79486 94.391 -78.550 52.556
79487 94.384 -78.562 52.538
79488 94.376 -78.574 52.518
79489 94.368 -78.586 52.496
79490 94.360 -78.598 52.474
79491 94.353 -78.611 52.451
79492 94.345 -78.623 52.430
79493 94.337 -78.635 52.409
79494 94.330 -78.646 52.389
79495 94.326 -78.655 52.366
79496 94.343 -78.642 52.343
79497 94.359 -78.630 52.319
79498 94.376 -78.617 52.295
79499 94.392 -78.604 52.271
79500 94.408 -78.592 52.245
79501 94.424 -78.580 52.217
79502 94.441 -78.567 52.187
79503 94.451 -78.564 52.155
79504 94.465 -78.563 52.121
79505 94.483 -78.563 52.086
79506 94.495 -78.570 52.051
79507 94.507 -78.578 52.014
79508 94.518 -78.594 51.976
79509 94.527 -78.624 51.937
79510 94.536 -78.655 51.899
79511 94.544 -78.673 51.862
79512 94.555 -78.675 51.824
79513 94.567 -78.674 51.786
79514 94.579 -78.673 51.747
79515 94.591 -78.673 51.708
79516 94.604 -78.673 51.670
79517 94.616 -78.672 51.631
79518 94.628 -78.669 51.593
79519 94.639 -78.665 51.557
79520 94.651 -78.661 51.521
79521 94.663 -78.658 51.486
79522 94.675 -78.654 51.450
79523 94.687 -78.650 51.414
79524 94.698 -78.646 51.378
79525 94.710 -78.645 51.344
79526 94.720 -78.647 51.308
79527 94.739 -78.651 51.271
79528 94.758 -78.654 51.232
79529 94.777 -78.658 51.194
79530 94.795 -78.661 51.156
79531 94.812 -78.665 51.116
79532 94.818 -78.673 51.078
79533 94.823 -78.680 51.039
79534 94.829 -78.688 50.999
79535 94.836 -78.686 50.959
79536 94.845 -78.673 50.918
79537 94.854 -78.661 50.876
79538 94.864 -78.648 50.833
79539 94.873 -78.635 50.787
79540 94.882 -78.622 50.742
79541 94.892 -78.609 50.697
79542 94.901 -78.596 50.652
79543 94.919 -78.587 50.606
79544 94.929 -78.586 50.558
79545 94.927 -78.589 50.510
79546 94.934 -78.598 50.461
79547 94.941 -78.608 50.413
79548 94.948 -78.617 50.365
79549 94.953 -78.628 50.322
79550 94.955 -78.642 50.281
79551 94.957 -78.656 50.240
79552 94.958 -78.670 50.199
79553 94.960 -78.684 50.157
79554 94.962 -78.698 50.114
79555 94.966 -78.711 50.071
79556 94.981 -78.715 50.028
79557 94.995 -78.720 49.985
79558 95.011 -78.724 49.943
79559 95.025 -78.728 49.900
79560 95.040 -78.733 49.860
79561 95.055 -78.737 49.820
79562 95.069 -78.742 49.781
79563 95.085 -78.745 49.744
79564 95.115 -78.742 49.706
79565 95.144 -78.739 49.668
79566 95.175 -78.735 49.631
79567 95.204 -78.731 49.592
79568 95.226 -78.729 49.551
79569 95.199 -78.738 49.510
79570 95.228 -78.758 49.468
79571 95.257 -78.777 49.428
79572 95.286 -78.797 49.388
79573 95.313 -78.815 49.348
79574 95.332 -78.827 49.310
79575 95.352 -78.838 49.272
79576 95.372 -78.850 49.234
79577 95.391 -78.861 49.194
79578 95.410 -78.874 49.154
79579 95.424 -78.871 49.114
79580 95.438 -78.868 49.073
79581 95.452 -78.866 49.034
79582 95.466 -78.864 48.994
79583 95.480 -78.861 48.953
79584 95.494 -78.859 48.912
79585 95.502 -78.852 48.870
79586 95.507 -78.841 48.828
79587 95.510 -78.830 48.786
79588 95.525 -78.825 48.743
79589 95.548 -78.827 48.700
79590 95.562 -78.834 48.658
79591 95.568 -78.844 48.616
79592 95.574 -78.855 48.573
79593 95.580 -78.865 48.530
79594 95.586 -78.876 48.486
79595 95.599 -78.889 48.443
79596 95.613 -78.902 48.399
79597 95.627 -78.915 48.356
79598 95.641 -78.929 48.316
79599 95.655 -78.942 48.276
79600 95.669 -78.955 48.235
79601 95.683 -78.968 48.194
79602 95.697 -78.980 48.154
79603 95.714 -78.984 48.113
79604 95.731 -78.987 48.072
79605 95.748 -78.991 48.030
79606 95.765 -78.995 47.987
79607 95.781 -78.999 47.943
79608 95.799 -79.002 47.901
79609 95.815 -79.006 47.858
79610 95.832 -79.010 47.814
79611 95.849 -79.014 47.772
79612 95.866 -79.018 47.730
79613 95.875 -79.014 47.687
79614 95.886 -79.010 47.645
79615 95.916 -79.006 47.605
79616 95.940 -78.999 47.564
79617 95.963 -78.993 47.523
79618 95.986 -78.987 47.482
79619 96.009 -78.981 47.442
79620 96.032 -78.974 47.402
79621 96.055 -78.968 47.362
79622 96.079 -78.962 47.323
79623 96.102 -78.956 47.285
79624 96.129 -78.952 47.248
79625 96.169 -78.953 47.209
79626 96.209 -78.955 47.172
79627 96.249 -78.956 47.136
79628 96.289 -78.958 47.100
79629 96.326 -78.956 47.066
79630 96.359 -78.948 47.033
79631 96.392 -78.940 47.003
79632 96.425 -78.933 46.971
79633 96.427 -78.937 46.939
79634 96.422 -78.943 46.909
79635 96.417 -78.949 46.879
79636 96.429 -78.959 46.849
79637 96.447 -78.968 46.819
79638 96.466 -78.977 46.788
79639 96.484 -78.987 46.757
79640 96.503 -78.996 46.724
79641 96.519 -79.004 46.692
79642 96.533 -79.011 46.660
79643 96.547 -79.018 46.627
79644 96.582 -79.021 46.595
79645 96.630 -79.024 46.564
79646 96.678 -79.025 46.535
79647 96.726 -79.028 46.506
79648 96.773 -79.030 46.477
79649 96.806 -79.049 46.450
79650 96.825 -79.057 46.423
79651 96.845 -78.996 46.397
79652 96.871 -78.946 46.373
79653 96.892 -78.961 46.349
79654 96.912 -78.976 46.325
79655 96.931 -78.986 46.302
79656 96.937 -78.968 46.280
79657 96.933 -78.905 46.260
79658 96.944 -78.872 46.241
79659 96.953 -78.885 46.223
79660 96.961 -78.904 46.204
79661 96.970 -78.923 46.185
79662 96.979 -78.941 46.164
79663 96.984 -78.965 46.142
79664 96.976 -78.951 46.122
79665 96.976 -78.967 46.103
79666 96.976 -78.983 46.084
79667 96.976 -78.999 46.066
79668 96.974 -79.024 46.048
79669 96.999 -79.103 46.031
79670 97.017 -79.159 46.014
79671 97.036 -79.197 45.997
79672 97.055 -79.235 45.982
79673 97.074 -79.274 45.968
79674 97.088 -79.293 45.950
79675 97.096 -79.283 45.932
79676 97.101 -79.260 45.912
79677 97.108 -79.232 45.895
79678 97.117 -79.201 45.880
79679 97.126 -79.170 45.865
79680 97.136 -79.139 45.849
79681 97.145 -79.109 45.833
79682 97.154 -79.078 45.818
79683 97.168 -79.084 45.803
79684 97.185 -79.093 45.786
79685 97.224 -79.115 45.770
79686 97.262 -79.134 45.755
79687 97.293 -79.129 45.741
79688 97.307 -79.137 45.727
79689 97.314 -79.151 45.713
79690 97.320 -79.165 45.698
79691 97.327 -79.179 45.684
79692 97.333 -79.193 45.669
79693 97.343 -79.205 45.656
79694 97.357 -79.215 45.643
79695 97.371 -79.224 45.630
79696 97.385 -79.234 45.617
79697 97.398 -79.244 45.603
79698 97.405 -79.250 45.587
79699 97.412 -79.256 45.572
79700 97.421 -79.261 45.559
79701 97.432 -79.263 45.547
79702 97.443 -79.266 45.534
79703 97.454 -79.269 45.521
79704 97.465 -79.272 45.507
79705 97.476 -79.275 45.494
79706 97.487 -79.277 45.480
79707 97.498 -79.280 45.467
79708 97.509 -79.283 45.454
79709 97.520 -79.285 45.440
79710 97.531 -79.288 45.425
79711 97.547 -79.297 45.409
79712 97.566 -79.304 45.392
79713 97.585 -79.311 45.374
79714 97.603 -79.319 45.355
79715 97.624 -79.327 45.336
79716 97.643 -79.335 45.320
79717 97.663 -79.344 45.307
79718 97.683 -79.353 45.299
79719 97.700 -79.360 45.291
79720 97.713 -79.374 45.283
79721 97.723 -79.390 45.274
79722 97.733 -79.405 45.264
79723 97.744 -79.420 45.254
79724 97.754 -79.435 45.242
79725 97.764 -79.451 45.230
79726 97.775 -79.466 45.218
79727 97.785 -79.482 45.208
79728 97.797 -79.497 45.197
79729 97.810 -79.510 45.181
79730 97.823 -79.524 45.166
79731 97.837 -79.538 45.151
79732 97.851 -79.551 45.137
79733 97.864 -79.566 45.125
79734 97.877 -79.579 45.115
79735 97.889 -79.592 45.107
79736 97.901 -79.604 45.100
79737 97.913 -79.615 45.093
79738 97.933 -79.626 45.087
79739 97.953 -79.637 45.080
79740 97.978 -79.647 45.076
79741 98.006 -79.656 45.072
79742 98.034 -79.665 45.066
79743 98.063 -79.674 45.058
79744 98.091 -79.683 45.051
79745 98.120 -79.692 45.045
79746 98.148 -79.701 45.036
79747 98.175 -79.711 45.027
79748 98.201 -79.717 45.018
79749 98.227 -79.724 45.009
79750 98.253 -79.730 44.997
79751 98.280 -79.737 44.984
79752 98.306 -79.743 44.969
79753 98.333 -79.750 44.953
79754 98.359 -79.756 44.938
79755 98.385 -79.763 44.924
79756 98.412 -79.770 44.913
79757 98.438 -79.778 44.904
79758 98.465 -79.789 44.895
79759 98.496 -79.798 44.887
79760 98.529 -79.808 44.877
79761 98.562 -79.817 44.867
79762 98.594 -79.826 44.858
79763 98.627 -79.836 44.851
79764 98.660 -79.845 44.845
79765 98.678 -79.842 44.839
79766 98.693 -79.835 44.833
79767 98.711 -79.836 44.826
79768 98.732 -79.844 44.819
79769 98.754 -79.852 44.812
79770 98.775 -79.859 44.806
79771 98.796 -79.867 44.799
79772 98.817 -79.874 44.792
79773 98.838 -79.882 44.785
79774 98.860 -79.889 44.779
79775 98.885 -79.890 44.773
79776 98.911 -79.890 44.766
79777 98.936 -79.891 44.759
79778 98.961 -79.892 44.753
79779 98.987 -79.892 44.748
79780 99.002 -79.893 44.741
79781 99.024 -79.908 44.734
79782 99.044 -79.925 44.727
79783 99.063 -79.943 44.721
79784 99.083 -79.959 44.717
79785 99.102 -79.977 44.713
79786 99.122 -79.993 44.708
79787 99.145 -80.008 44.702
79788 99.178 -80.021 44.695
79789 99.209 -80.034 44.689
79790 99.241 -80.046 44.684
79791 99.273 -80.059 44.681
79792 99.308 -80.073 44.678
79793 99.322 -80.077 44.675
79794 99.336 -80.074 44.671
79795 99.381 -80.087 44.668
79796 99.427 -80.099 44.664
79797 99.472 -80.111 44.660
79798 99.502 -80.103 44.655
79799 99.526 -80.090 44.648
79800 99.526 -80.113 44.639
79801 99.526 -80.141 44.628
79802 99.529 -80.163 44.618
79803 99.532 -80.185 44.608
79804 99.536 -80.208 44.599
79805 99.539 -80.230 44.590
79806 99.542 -80.252 44.583
79807 99.545 -80.274 44.575
79808 99.548 -80.296 44.567
79809 99.552 -80.319 44.559
79810 99.555 -80.341 44.552
79811 99.558 -80.363 44.545
79812 99.561 -80.385 44.536
79813 99.564 -80.407 44.527
79814 99.568 -80.429 44.519
79815 99.571 -80.451 44.511
79816 99.574 -80.473 44.503
79817 99.577 -80.495 44.494
79818 99.580 -80.518 44.486
79819 99.584 -80.540 44.476
79820 99.587 -80.562 44.467
79821 99.590 -80.584 44.456
79822 99.634 -80.591 44.445
79823 99.682 -80.597 44.435
79824 99.729 -80.603 44.425
79825 99.777 -80.608 44.415
79826 99.826 -80.613 44.403
79827 99.873 -80.620 44.391
79828 99.921 -80.625 44.380
79829 99.969 -80.631 44.368
79830 100.017 -80.636 44.357
79831 100.065 -80.641 44.345
79832 100.113 -80.648 44.334
80126 93.555 -78.836 53.563
80127 93.565 -78.833 53.554
80128 93.574 -78.831 53.546
80129 93.584 -78.828 53.538
80130 93.593 -78.826 53.531
80131 93.604 -78.824 53.524
80132 93.613 -78.821 53.519
80133 93.623 -78.818 53.515
80134 93.632 -78.814 53.513
80135 93.650 -78.802 53.515
80136 93.654 -78.794 53.518
80137 93.658 -78.791 53.518
80138 93.663 -78.789 53.517
80139 93.669 -78.784 53.514
80140 93.674 -78.780 53.510
80141 93.680 -78.775 53.503
80142 93.686 -78.770 53.494
80143 93.692 -78.765 53.485
80144 93.698 -78.763 53.474
80145 93.712 -78.756 53.461
80146 93.737 -78.742 53.448
80147 93.762 -78.728 53.432
80148 93.786 -78.714 53.415
80149 93.805 -78.702 53.399
80150 93.808 -78.693 53.381
80151 93.812 -78.685 53.362
80152 93.815 -78.677 53.343
80153 93.819 -78.668 53.323
80154 93.822 -78.659 53.302
80155 93.826 -78.651 53.281
80156 93.832 -78.642 53.261
80157 93.846 -78.635 53.240
80158 93.860 -78.626 53.218
80159 93.874 -78.618 53.196
80160 93.887 -78.606 53.175
80161 93.900 -78.593 53.151
80162 93.913 -78.580 53.126
80163 93.925 -78.567 53.099
80164 93.938 -78.554 53.073
80165 93.949 -78.555 53.048
80166 93.960 -78.562 53.023
80167 93.971 -78.568 53.001
80168 93.982 -78.575 52.978
80169 93.992 -78.581 52.955
80170 94.003 -78.588 52.935
80171 94.013 -78.592 52.917
80172 94.024 -78.591 52.899
80173 94.034 -78.589 52.881
80174 94.043 -78.587 52.861
80175 94.053 -78.586 52.839
80176 94.063 -78.584 52.815
80177 94.066 -78.587 52.791
80178 94.058 -78.557 52.766
80179 94.070 -78.514 52.741
80180 94.131 -78.517 52.717
80181 94.192 -78.518 52.692
80182 94.245 -78.515 52.668
80183 94.290 -78.506 52.646
80184 94.334 -78.496 52.626
80185 94.379 -78.487 52.607
80186 94.390 -78.535 52.589
80187 94.397 -78.545 52.571
80188 94.390 -78.557 52.553
80189 94.382 -78.568 52.533
80190 94.374 -78.580 52.511
80191 94.366 -78.592 52.488
80192 94.359 -78.604 52.466
80193 94.351 -78.617 52.445
80194 94.343 -78.629 52.425
80195 94.336 -78.641 52.404
80196 94.328 -78.652 52.383
80197 94.345 -78.640 52.360
80198 94.361 -78.627 52.336
80199 94.378 -78.615 52.313
80200 94.394 -78.603 52.289
80201 94.410 -78.590 52.263
80202 94.426 -78.578 52.233
80203 94.443 -78.566 52.203
80204 94.459 -78.554 52.172
80205 94.473 -78.552 52.138
80206 94.485 -78.560 52.103
80207 94.496 -78.568 52.067
80208 94.507 -78.576 52.029
80209 94.519 -78.583 51.990
80210 94.529 -78.608 51.951
80211 94.538 -78.638 51.913
80212 94.546 -78.652 51.875
80213 94.555 -78.658 51.838
80214 94.567 -78.658 51.801
80215 94.579 -78.657 51.764
80216 94.592 -78.656 51.726
80217 94.604 -78.656 51.687
80218 94.616 -78.655 51.649
80219 94.628 -78.655 51.609
80220 94.641 -78.655 51.572
80221 94.653 -78.654 51.535
80222 94.665 -78.652 51.499
80223 94.677 -78.648 51.461
80224 94.689 -78.645 51.424
80225 94.700 -78.641 51.386
80226 94.712 -78.639 51.351
80227 94.726 -78.642 51.314
80228 94.746 -78.645 51.276
80229 94.764 -78.648 51.237
80230 94.783 -78.652 51.199
80231 94.802 -78.656 51.160
80232 94.821 -78.659 51.121
80233 94.827 -78.667 51.083
80234 94.833 -78.674 51.044
80235 94.838 -78.682 51.004
80236 94.846 -78.676 50.965
80237 94.855 -78.664 50.924
80238 94.865 -78.651 50.881
80239 94.874 -78.638 50.838
80240 94.883 -78.625 50.793
80241 94.893 -78.612 50.747
80242 94.902 -78.599 50.702
80243 94.923 -78.592 50.656
80244 94.961 -78.590 50.608
80245 94.972 -78.589 50.559
80246 94.964 -78.589 50.509
80247 94.956 -78.588 50.460
80248 94.955 -78.592 50.413
80249 94.962 -78.601 50.366
80250 94.969 -78.610 50.323
80251 94.974 -78.621 50.281
80252 94.976 -78.635 50.239
80253 94.977 -78.649 50.197
80254 94.979 -78.663 50.155
80255 94.981 -78.677 50.112
80256 94.990 -78.686 50.069
80257 95.005 -78.690 50.026
80258 95.020 -78.695 49.984
80259 95.035 -78.699 49.941
80260 95.049 -78.704 49.900
80261 95.064 -78.708 49.861
80262 95.079 -78.712 49.823
80263 95.094 -78.717 49.786
80264 95.109 -78.721 49.749
80265 95.123 -78.726 49.712
80266 95.153 -78.722 49.674
80267 95.183 -78.718 49.635
80268 95.212 -78.715 49.595
80269 95.209 -78.715 49.554
80270 95.191 -78.726 49.513
80271 95.219 -78.745 49.472
80272 95.249 -78.766 49.431
80273 95.278 -78.786 49.391
80274 95.307 -78.805 49.351
80275 95.333 -78.822 49.313
80276 95.352 -78.833 49.276
80277 95.372 -78.846 49.239
80278 95.391 -78.857 49.200
80279 95.411 -78.869 49.160
80280 95.425 -78.867 49.120
80281 95.439 -78.865 49.080
80282 95.453 -78.862 49.039
80283 95.466 -78.860 48.999
80284 95.480 -78.858 48.959
80285 95.489 -78.851 48.917
80286 95.493 -78.839 48.874
80287 95.497 -78.828 48.831
80288 95.500 -78.817 48.789
80289 95.515 -78.812 48.745
80290 95.538 -78.814 48.702
80291 95.561 -78.816 48.661
80292 95.570 -78.824 48.617
80293 95.577 -78.835 48.574
80294 95.582 -78.846 48.530
80295 95.588 -78.856 48.488
80296 95.594 -78.867 48.445
80297 95.600 -78.877 48.402
80298 95.611 -78.889 48.360
80299 95.625 -78.902 48.320
80300 95.639 -78.916 48.281
80301 95.653 -78.929 48.241
80302 95.667 -78.943 48.202
80303 95.681 -78.954 48.163
80304 95.698 -78.958 48.124
80305 95.715 -78.962 48.082
80306 95.732 -78.965 48.040
80307 95.748 -78.970 47.997
80308 95.765 -78.974 47.953
80309 95.782 -78.977 47.910
80310 95.799 -78.981 47.867
80311 95.816 -78.985 47.824
80312 95.832 -78.989 47.781
80313 95.852 -78.993 47.739
80314 95.876 -78.999 47.698
80315 95.895 -79.003 47.656
80316 95.918 -78.998 47.615
80317 95.942 -78.992 47.574
80318 95.965 -78.986 47.532
80319 95.988 -78.980 47.491
80320 96.012 -78.974 47.451
80321 96.035 -78.968 47.411
80322 96.058 -78.962 47.370
80323 96.081 -78.955 47.332
80324 96.104 -78.949 47.294
80325 96.128 -78.943 47.255
80326 96.169 -78.945 47.217
80327 96.208 -78.946 47.180
80328 96.249 -78.948 47.143
80329 96.288 -78.949 47.107
80330 96.321 -78.940 47.073
80331 96.354 -78.933 47.041
80332 96.386 -78.925 47.009
80333 96.417 -78.917 46.979
80334 96.430 -78.916 46.948
80335 96.430 -78.922 46.917
80336 96.425 -78.929 46.886
80337 96.420 -78.936 46.856
80338 96.422 -78.943 46.826
80339 96.441 -78.952 46.795
80340 96.460 -78.962 46.762
80341 96.476 -78.970 46.730
80342 96.490 -78.977 46.697
80343 96.504 -78.984 46.664
80344 96.518 -78.990 46.632
80345 96.562 -78.993 46.602
80346 96.610 -78.995 46.572
80347 96.659 -78.996 46.542
80348 96.704 -79.028 46.514
80349 96.739 -79.048 46.485
80350 96.764 -79.046 46.457
80351 96.790 -79.045 46.431
80352 96.814 -79.043 46.405
80353 96.853 -78.967 46.381
80354 96.884 -78.946 46.358
80355 96.900 -78.949 46.335
80356 96.910 -78.937 46.313
80357 96.922 -78.929 46.290
80358 96.933 -78.917 46.270
80359 96.943 -78.857 46.252
80360 96.951 -78.876 46.234
80361 96.960 -78.894 46.216
80362 96.968 -78.913 46.196
80363 96.977 -78.931 46.176
80364 96.979 -78.943 46.153
80365 96.979 -78.959 46.133
80366 96.979 -78.974 46.114
80367 96.979 -78.990 46.096
80368 96.979 -79.006 46.078
80369 96.979 -79.022 46.061
80370 96.989 -79.049 46.043
80371 97.007 -79.122 46.026
80372 97.026 -79.161 46.010
80373 97.045 -79.199 45.995
80374 97.064 -79.237 45.978
80375 97.083 -79.276 45.962
80376 97.100 -79.294 45.945
80377 97.109 -79.263 45.926
80378 97.118 -79.231 45.909
80379 97.127 -79.200 45.895
80380 97.137 -79.169 45.880
80381 97.146 -79.139 45.864
80382 97.155 -79.108 45.850
80383 97.164 -79.079 45.835
80384 97.179 -79.087 45.819
80385 97.194 -79.095 45.803
80386 97.223 -79.112 45.787
80387 97.260 -79.127 45.772
80388 97.285 -79.126 45.757
80389 97.291 -79.140 45.742
80390 97.298 -79.154 45.728
80391 97.304 -79.169 45.714
80392 97.315 -79.179 45.699
80393 97.329 -79.189 45.685
80394 97.343 -79.199 45.671
80395 97.357 -79.209 45.658
80396 97.371 -79.218 45.644
80397 97.385 -79.228 45.630
80398 97.398 -79.237 45.616
80399 97.408 -79.245 45.601
80400 97.415 -79.251 45.586
80401 97.424 -79.255 45.572
80402 97.435 -79.258 45.559
80403 97.446 -79.261 45.546
80404 97.457 -79.263 45.533
80405 97.468 -79.266 45.520
80406 97.479 -79.269 45.506
80407 97.491 -79.272 45.493
80408 97.502 -79.275 45.481
80409 97.513 -79.277 45.468
80410 97.526 -79.280 45.454
80411 97.541 -79.283 45.438
80412 97.560 -79.290 45.421
80413 97.579 -79.299 45.402
80414 97.599 -79.307 45.383
80415 97.619 -79.316 45.362
80416 97.639 -79.324 45.342
80417 97.659 -79.332 45.326
80418 97.678 -79.341 45.314
80419 97.691 -79.346 45.304
80420 97.691 -79.346 45.296
80421 97.706 -79.362 45.287
80422 97.724 -79.382 45.277
80423 97.740 -79.401 45.267
80424 97.750 -79.416 45.255
80425 97.761 -79.432 45.243
80426 97.771 -79.447 45.231
80427 97.782 -79.463 45.220
80428 97.792 -79.478 45.208
80429 97.802 -79.494 45.196
80430 97.816 -79.507 45.183
80431 97.829 -79.521 45.170
80432 97.843 -79.535 45.158
80433 97.856 -79.548 45.144
80434 97.870 -79.563 45.131
80435 97.882 -79.575 45.120
80436 97.893 -79.586 45.112
80437 97.905 -79.598 45.103
80438 97.917 -79.610 45.096
80439 97.934 -79.621 45.090
80440 97.955 -79.632 45.084
80441 97.980 -79.642 45.078
80442 98.008 -79.651 45.074
80443 98.036 -79.660 45.070
80444 98.065 -79.669 45.064
80445 98.093 -79.678 45.057
80446 98.122 -79.686 45.049
80447 98.150 -79.695 45.040
80448 98.186 -79.689 45.030
80449 98.218 -79.704 45.020
80450 98.244 -79.711 45.009
80451 98.270 -79.717 44.997
80452 98.296 -79.724 44.985
80453 98.323 -79.730 44.972
80454 98.349 -79.737 44.957
80455 98.375 -79.743 44.944
80456 98.402 -79.750 44.931
80457 98.428 -79.757 44.921
80458 98.455 -79.767 44.911
80459 98.483 -79.777 44.903
80460 98.511 -79.787 44.895
80461 98.544 -79.797 44.886
80462 98.576 -79.806 44.878
80463 98.609 -79.815 44.870
80464 98.641 -79.824 44.863
80465 98.672 -79.832 44.857
80466 98.687 -79.825 44.852
80467 98.701 -79.818 44.846
80468 98.717 -79.815 44.839
80469 98.738 -79.823 44.832
80470 98.759 -79.830 44.825
80471 98.780 -79.837 44.820
80472 98.801 -79.845 44.813
80473 98.822 -79.852 44.806
80474 98.844 -79.860 44.800
80475 98.865 -79.868 44.794
80476 98.887 -79.876 44.788
80477 98.913 -79.877 44.782
80478 98.938 -79.877 44.775
80479 98.963 -79.878 44.768
80480 98.989 -79.879 44.761
80481 99.005 -79.878 44.753
80482 99.029 -79.892 44.746
80483 99.051 -79.908 44.741
80484 99.070 -79.925 44.736
80485 99.089 -79.942 44.730
80486 99.109 -79.959 44.726
80487 99.128 -79.976 44.721
80488 99.155 -79.990 44.714
80489 99.188 -80.002 44.707
80490 99.219 -80.015 44.703
80491 99.251 -80.027 44.699
80492 99.283 -80.040 44.696
80493 99.314 -80.052 44.692
80494 99.337 -80.057 44.687
80495 99.360 -80.062 44.681
80496 99.372 -80.060 44.675
80497 99.385 -80.059 44.669
80498 99.429 -80.070 44.663
80499 99.474 -80.082 44.657
80500 99.514 -80.087 44.651
80501 99.538 -80.072 44.643
80502 99.540 -80.095 44.634
80503 99.531 -80.133 44.624
80504 99.528 -80.162 44.615
80505 99.532 -80.184 44.605
80506 99.535 -80.207 44.597
80507 99.539 -80.229 44.589
80508 99.542 -80.251 44.581
80509 99.545 -80.273 44.573
80510 99.548 -80.295 44.564
80511 99.551 -80.318 44.557
80512 99.554 -80.340 44.549
80513 99.558 -80.362 44.541
80514 99.561 -80.384 44.533
80515 99.564 -80.406 44.525
80516 99.567 -80.428 44.516
80517 99.570 -80.450 44.508
80518 99.574 -80.472 44.499
80519 99.577 -80.494 44.491
80520 99.580 -80.517 44.481
80521 99.583 -80.539 44.472
80522 99.594 -80.559 44.461
80523 99.642 -80.564 44.450
80524 99.690 -80.569 44.440
80525 99.738 -80.576 44.430
80526 99.786 -80.581 44.419
80527 99.834 -80.586 44.407
80528 99.881 -80.592 44.395
80529 99.929 -80.598 44.383
80530 99.978 -80.604 44.372
80531 100.025 -80.609 44.361
80532 100.073 -80.614 44.350
80533 100.121 -80.620 44.338
80827 93.555 -78.825 53.571
80828 93.565 -78.822 53.563
80829 93.574 -78.820 53.555
80830 93.584 -78.818 53.547
80831 93.594 -78.815 53.540
80832 93.604 -78.812 53.534
80833 93.613 -78.810 53.529
80834 93.623 -78.808 53.525
80835 93.633 -78.805 53.524
80836 93.642 -78.800 53.526
80837 93.651 -78.792 53.529
80838 93.655 -78.789 53.530
80839 93.659 -78.785 53.529
80840 93.665 -78.780 53.527
80841 93.672 -78.776 53.522
80842 93.677 -78.771 53.516
80843 93.683 -78.767 53.508
80844 93.689 -78.762 53.498
80845 93.699 -78.755 53.486
80846 93.724 -78.739 53.474
80847 93.749 -78.725 53.460
80848 93.773 -78.711 53.444
80849 93.798 -78.697 53.427
80850 93.806 -78.687 53.410
80851 93.809 -78.679 53.391
80852 93.813 -78.670 53.371
80853 93.817 -78.662 53.351
80854 93.820 -78.654 53.331
80855 93.823 -78.645 53.311
80856 93.827 -78.636 53.290
80857 93.832 -78.628 53.270
80858 93.845 -78.620 53.249
80859 93.859 -78.611 53.227
80860 93.872 -78.598 53.204
80861 93.885 -78.586 53.182
80862 93.898 -78.573 53.159
80863 93.910 -78.560 53.134
80864 93.923 -78.547 53.108
80865 93.935 -78.534 53.083
80866 93.947 -78.532 53.059
80867 93.958 -78.539 53.033
80868 93.969 -78.545 53.011
80869 93.979 -78.551 52.988
80870 93.990 -78.557 52.966
80871 94.001 -78.564 52.947
80872 94.011 -78.570 52.929
80873 94.023 -78.577 52.911
80874 94.033 -78.581 52.893
80875 94.043 -78.579 52.873
80876 94.053 -78.577 52.851
80877 94.062 -78.576 52.826
80878 94.060 -78.565 52.802
80879 94.050 -78.523 52.777
80880 94.090 -78.509 52.752
80881 94.135 -78.500 52.727
80882 94.180 -78.491 52.703
80883 94.224 -78.481 52.680
80884 94.269 -78.472 52.660
80885 94.313 -78.462 52.641
80886 94.357 -78.454 52.622
80887 94.365 -78.537 52.604
80888 94.380 -78.544 52.587
80889 94.395 -78.551 52.569
80890 94.388 -78.563 52.549
80891 94.381 -78.574 52.527
80892 94.372 -78.586 52.505
80893 94.365 -78.598 52.483
80894 94.357 -78.611 52.461
80895 94.350 -78.623 52.440
80896 94.342 -78.635 52.420
80897 94.334 -78.647 52.399
80898 94.347 -78.638 52.376
80899 94.363 -78.626 52.352
80900 94.380 -78.614 52.329
80901 94.396 -78.601 52.305
80902 94.412 -78.589 52.279
80903 94.428 -78.576 52.251
80904 94.445 -78.564 52.222
80905 94.460 -78.551 52.190
80906 94.471 -78.552 52.156
80907 94.484 -78.558 52.120
80908 94.497 -78.566 52.083
80909 94.508 -78.573 52.044
80910 94.520 -78.581 52.006
80911 94.531 -78.590 51.968
80912 94.540 -78.620 51.930
80913 94.548 -78.632 51.893
80914 94.556 -78.641 51.857
80915 94.568 -78.641 51.820
80916 94.580 -78.640 51.782
80917 94.592 -78.639 51.745
80918 94.605 -78.639 51.706
80919 94.617 -78.639 51.668
80920 94.629 -78.638 51.628
80921 94.641 -78.637 51.588
80922 94.653 -78.637 51.550
80923 94.666 -78.636 51.512
80924 94.678 -78.636 51.474
80925 94.690 -78.635 51.435
80926 94.702 -78.635 51.396
80927 94.714 -78.633 51.359
80928 94.733 -78.636 51.321
80929 94.752 -78.640 51.282
80930 94.771 -78.643 51.243
80931 94.789 -78.647 51.204
80932 94.809 -78.651 51.165
80933 94.827 -78.654 51.127
80934 94.837 -78.661 51.088
80935 94.842 -78.668 51.049
80936 94.848 -78.676 51.009
80937 94.857 -78.666 50.970
80938 94.866 -78.653 50.929
80939 94.875 -78.640 50.886
80940 94.885 -78.627 50.842
80941 94.894 -78.614 50.798
80942 94.903 -78.601 50.752
80943 94.928 -78.595 50.706
80944 94.965 -78.595 50.658
80945 95.002 -78.593 50.608
80946 95.014 -78.592 50.558
80947 95.006 -78.592 50.507
80948 94.998 -78.591 50.459
80949 94.991 -78.591 50.412
80950 94.983 -78.590 50.366
80951 94.982 -78.594 50.323
80952 94.989 -78.603 50.280
80953 94.994 -78.614 50.237
80954 94.996 -78.628 50.196
80955 94.998 -78.642 50.154
80956 95.000 -78.656 50.111
80957 95.014 -78.661 50.068
80958 95.029 -78.665 50.026
80959 95.044 -78.670 49.984
80960 95.058 -78.674 49.941
80961 95.073 -78.679 49.901
80962 95.088 -78.683 49.863
80963 95.103 -78.687 49.826
80964 95.118 -78.692 49.790
80965 95.132 -78.696 49.753
80966 95.147 -78.701 49.716
80967 95.162 -78.705 49.678
80968 95.190 -78.702 49.638
80969 95.220 -78.698 49.598
80970 95.192 -78.702 49.557
80971 95.182 -78.714 49.516
80972 95.211 -78.734 49.476
80973 95.241 -78.754 49.436
80974 95.270 -78.774 49.396
80975 95.299 -78.793 49.357
80976 95.328 -78.813 49.318
80977 95.352 -78.829 49.280
80978 95.372 -78.841 49.242
80979 95.391 -78.852 49.204
80980 95.411 -78.865 49.165
80981 95.425 -78.863 49.126
80982 95.439 -78.861 49.086
80983 95.453 -78.858 49.046
80984 95.467 -78.856 49.005
80985 95.476 -78.849 48.965
80986 95.479 -78.838 48.923
80987 95.483 -78.827 48.880
80988 95.487 -78.815 48.836
80989 95.491 -78.804 48.793
80990 95.504 -78.799 48.749
80991 95.528 -78.800 48.705
80992 95.551 -78.802 48.664
80993 95.573 -78.805 48.620
80994 95.579 -78.815 48.576
80995 95.585 -78.826 48.533
80996 95.591 -78.836 48.492
80997 95.597 -78.847 48.449
80998 95.603 -78.857 48.406
80999 95.609 -78.868 48.365
81000 95.614 -78.878 48.325
81001 95.623 -78.890 48.286
81002 95.637 -78.902 48.247
81003 95.651 -78.916 48.209
81004 95.665 -78.929 48.171
81005 95.682 -78.933 48.133
81006 95.698 -78.937 48.092
81007 95.716 -78.940 48.050
81008 95.732 -78.944 48.008
81009 95.749 -78.948 47.964
81010 95.768 -78.952 47.920
81011 95.792 -78.958 47.877
81012 95.816 -78.963 47.834
81013 95.839 -78.968 47.791
81014 95.864 -78.974 47.749
81015 95.875 -78.988 47.708
81016 95.894 -78.991 47.666
81017 95.918 -78.987 47.625
81018 95.943 -78.983 47.584
81019 95.967 -78.979 47.543
81020 95.990 -78.973 47.501
81021 96.014 -78.967 47.461
81022 96.037 -78.961 47.421
81023 96.060 -78.954 47.381
81024 96.083 -78.948 47.342
81025 96.106 -78.942 47.303
81026 96.129 -78.936 47.263
81027 96.168 -78.937 47.225
81028 96.208 -78.939 47.188
81029 96.249 -78.940 47.151
81030 96.282 -78.933 47.115
81031 96.315 -78.925 47.081
81032 96.347 -78.917 47.048
81033 96.373 -78.908 47.017
81034 96.398 -78.899 46.986
81035 96.414 -78.896 46.956
81036 96.415 -78.902 46.925
81037 96.416 -78.908 46.894
81038 96.418 -78.915 46.864
81039 96.419 -78.921 46.833
81040 96.418 -78.928 46.802
81041 96.433 -78.936 46.769
81042 96.447 -78.943 46.737
81043 96.461 -78.949 46.704
81044 96.475 -78.956 46.670
81045 96.495 -78.963 46.639
81046 96.543 -78.965 46.609
81047 96.589 -78.986 46.579
81048 96.632 -78.986 46.551
81049 96.675 -78.978 46.522
81050 96.717 -78.971 46.494
81051 96.759 -78.962 46.465
81052 96.802 -78.955 46.440
81053 96.839 -78.949 46.415
81054 96.855 -78.940 46.392
81055 96.866 -78.943 46.369
81056 96.889 -78.932 46.347
81057 96.901 -78.926 46.325
81058 96.913 -78.919 46.302
81059 96.925 -78.910 46.282
81060 96.927 -78.857 46.263
81061 96.916 -78.865 46.245
81062 96.924 -78.882 46.227
81063 96.939 -78.902 46.207
81064 96.967 -78.924 46.186
81065 96.986 -78.945 46.165
81066 96.985 -78.962 46.145
81067 96.984 -78.980 46.126
81068 96.984 -78.996 46.108
81069 96.983 -79.014 46.091
81070 96.982 -79.031 46.073
81071 96.982 -79.046 46.056
81072 96.997 -79.086 46.039
81073 97.017 -79.129 46.023
81074 97.038 -79.176 46.007
81075 97.059 -79.222 45.990
81076 97.080 -79.269 45.975
81077 97.100 -79.269 45.959
81078 97.119 -79.262 45.942
81079 97.129 -79.231 45.926
81080 97.138 -79.200 45.911
81081 97.147 -79.169 45.896
81082 97.156 -79.138 45.881
81083 97.165 -79.107 45.867
81084 97.175 -79.082 45.852
81085 97.189 -79.090 45.837
81086 97.205 -79.097 45.820
81087 97.222 -79.107 45.805
81088 97.258 -79.119 45.790
81089 97.268 -79.130 45.773
81090 97.275 -79.143 45.757
81091 97.288 -79.154 45.743
81092 97.302 -79.164 45.730
81093 97.316 -79.173 45.714
81094 97.329 -79.183 45.700
81095 97.344 -79.193 45.687
81096 97.357 -79.202 45.673
81097 97.371 -79.212 45.659
81098 97.385 -79.222 45.644
81099 97.398 -79.231 45.629
81100 97.411 -79.240 45.614
81101 97.418 -79.247 45.600
81102 97.428 -79.250 45.587
81103 97.439 -79.253 45.573
81104 97.450 -79.255 45.558
81105 97.461 -79.258 45.545
81106 97.472 -79.261 45.533
81107 97.483 -79.263 45.519
81108 97.496 -79.266 45.507
81109 97.511 -79.269 45.495
81110 97.525 -79.272 45.479
81111 97.540 -79.275 45.462
81112 97.555 -79.278 45.444
81113 97.575 -79.287 45.424
81114 97.595 -79.295 45.404
81115 97.615 -79.304 45.384
81116 97.635 -79.313 45.366
81117 97.654 -79.321 45.346
81118 97.674 -79.329 45.332
81119 97.682 -79.332 45.319
81120 97.682 -79.332 45.307
81121 97.682 -79.332 45.296
81122 97.698 -79.349 45.286
81123 97.717 -79.369 45.277
81124 97.735 -79.390 45.268
81125 97.753 -79.410 45.257
81126 97.768 -79.429 45.246
81127 97.778 -79.444 45.236
81128 97.789 -79.459 45.224
81129 97.799 -79.475 45.212
81130 97.809 -79.490 45.199
81131 97.822 -79.504 45.187
81132 97.836 -79.518 45.175
81133 97.849 -79.532 45.162
81134 97.862 -79.545 45.149
81135 97.874 -79.558 45.137
81136 97.886 -79.570 45.127
81137 97.897 -79.582 45.118
81138 97.909 -79.593 45.109
81139 97.921 -79.605 45.102
81140 97.936 -79.617 45.095
81141 97.956 -79.627 45.088
81142 97.982 -79.637 45.081
81143 98.010 -79.646 45.075
81144 98.038 -79.655 45.071
81145 98.067 -79.664 45.065
81146 98.095 -79.673 45.058
81147 98.124 -79.682 45.050
81148 98.160 -79.674 45.041
81149 98.197 -79.665 45.031
81150 98.231 -79.686 45.021
81151 98.261 -79.698 45.010
81152 98.287 -79.704 44.999
81153 98.313 -79.711 44.987
81154 98.340 -79.717 44.975
81155 98.366 -79.724 44.962
81156 98.392 -79.730 44.951
81157 98.418 -79.737 44.940
81158 98.445 -79.745 44.929
81159 98.472 -79.755 44.920
81160 98.499 -79.766 44.912
81161 98.526 -79.776 44.904
81162 98.558 -79.786 44.896
81163 98.590 -79.795 44.889
81164 98.623 -79.805 44.883
81165 98.656 -79.814 44.876
81166 98.680 -79.816 44.871
81167 98.695 -79.809 44.865
81168 98.709 -79.802 44.860
81169 98.723 -79.795 44.853
81170 98.743 -79.801 44.846
81171 98.765 -79.808 44.840
81172 98.786 -79.816 44.834
81173 98.807 -79.824 44.828
81174 98.828 -79.831 44.822
81175 98.849 -79.839 44.815
81176 98.871 -79.847 44.810
81177 98.893 -79.855 44.804
81178 98.915 -79.863 44.797
81179 98.940 -79.864 44.789
81180 98.965 -79.864 44.782
81181 98.991 -79.865 44.775
81182 99.009 -79.863 44.768
81183 99.033 -79.877 44.761
81184 99.057 -79.891 44.755
81185 99.077 -79.908 44.749
81186 99.096 -79.925 44.745
81187 99.116 -79.942 44.740
81188 99.135 -79.958 44.734
81189 99.166 -79.971 44.727
81190 99.198 -79.984 44.720
81191 99.229 -79.996 44.714
81192 99.261 -80.009 44.709
81193 99.290 -80.019 44.705
81194 99.313 -80.024 44.699
81195 99.337 -80.028 44.693
81196 99.360 -80.034 44.687
81197 99.383 -80.038 44.679
81198 99.406 -80.043 44.673
81199 99.426 -80.046 44.667
81200 99.439 -80.044 44.660
81201 99.476 -80.053 44.654
81202 99.521 -80.065 44.647
81203 99.550 -80.055 44.638
81204 99.552 -80.077 44.630
81205 99.544 -80.115 44.621
81206 99.536 -80.152 44.611
81207 99.532 -80.184 44.602
81208 99.535 -80.206 44.594
81209 99.538 -80.228 44.586
81210 99.541 -80.250 44.578
81211 99.545 -80.272 44.570
81212 99.548 -80.295 44.561
81213 99.551 -80.317 44.553
81214 99.554 -80.339 44.546
81215 99.557 -80.361 44.538
81216 99.561 -80.383 44.530
81217 99.564 -80.406 44.522
81218 99.567 -80.428 44.513
81219 99.570 -80.450 44.505
81220 99.573 -80.472 44.496
81221 99.577 -80.494 44.486
81222 99.580 -80.516 44.476
81223 99.603 -80.532 44.466
81224 99.650 -80.537 44.456
81225 99.698 -80.542 44.445
81226 99.746 -80.548 44.435
81227 99.794 -80.554 44.424
81228 99.842 -80.560 44.412
81229 99.890 -80.565 44.400
81230 99.938 -80.570 44.388
81231 99.986 -80.576 44.376
81232 100.034 -80.582 44.365
81233 100.081 -80.588 44.353
81234 100.130 -80.593 44.341
81528 93.556 -78.814 53.581
81529 93.565 -78.811 53.571
81530 93.575 -78.809 53.563
81531 93.584 -78.807 53.556
81532 93.594 -78.804 53.549
81533 93.604 -78.802 53.543
81534 93.614 -78.799 53.539
81535 93.623 -78.797 53.535
81536 93.633 -78.794 53.534
81537 93.642 -78.792 53.535
81538 93.647 -78.789 53.539
81539 93.650 -78.784 53.541
81540 93.654 -78.780 53.540
81541 93.663 -78.778 53.538
81542 93.669 -78.774 53.534
81543 93.675 -78.769 53.528
81544 93.680 -78.764 53.519
81545 93.688 -78.759 53.509
81546 93.712 -78.742 53.498
81547 93.736 -78.727 53.485
81548 93.760 -78.710 53.470
81549 93.784 -78.694 53.454
81550 93.804 -78.682 53.437
81551 93.807 -78.673 53.419
81552 93.811 -78.664 53.399
81553 93.814 -78.656 53.379
81554 93.818 -78.648 53.359
81555 93.821 -78.639 53.339
81556 93.825 -78.630 53.319
81557 93.828 -78.622 53.298
81558 93.831 -78.614 53.277
81559 93.844 -78.604 53.255
81560 93.857 -78.591 53.232
81561 93.869 -78.578 53.210
81562 93.882 -78.565 53.187
81563 93.895 -78.552 53.164
81564 93.908 -78.539 53.140
81565 93.920 -78.526 53.115
81566 93.933 -78.514 53.090
81567 93.945 -78.509 53.066
81568 93.956 -78.515 53.041
81569 93.967 -78.522 53.018
81570 93.977 -78.528 52.996
81571 93.988 -78.535 52.975
81572 93.999 -78.541 52.957
81573 94.009 -78.548 52.939
81574 94.020 -78.554 52.921
81575 94.031 -78.561 52.903
81576 94.042 -78.567 52.885
81577 94.052 -78.570 52.863
81578 94.060 -78.569 52.838
81579 94.055 -78.535 52.813
81580 94.060 -78.511 52.788
81581 94.074 -78.480 52.762
81582 94.114 -78.466 52.736
81583 94.158 -78.456 52.714
81584 94.203 -78.447 52.693
81585 94.248 -78.438 52.674
81586 94.292 -78.428 52.656
81587 94.336 -78.421 52.637
81588 94.343 -78.511 52.619
81589 94.355 -78.545 52.602
81590 94.370 -78.552 52.584
81591 94.385 -78.559 52.565
81592 94.387 -78.569 52.544
81593 94.379 -78.580 52.521
81594 94.371 -78.592 52.499
81595 94.363 -78.604 52.478
81596 94.356 -78.617 52.456
81597 94.348 -78.629 52.436
81598 94.340 -78.641 52.414
81599 94.349 -78.636 52.390
81600 94.365 -78.624 52.367
81601 94.382 -78.611 52.343
81602 94.398 -78.599 52.319
81603 94.414 -78.587 52.294
81604 94.430 -78.574 52.268
81605 94.445 -78.561 52.239
81606 94.453 -78.547 52.208
81607 94.466 -78.553 52.173
81608 94.479 -78.560 52.137
81609 94.493 -78.566 52.100
81610 94.506 -78.573 52.062
81611 94.520 -78.579 52.024
81612 94.532 -78.586 51.987
81613 94.542 -78.602 51.951
81614 94.550 -78.611 51.915
81615 94.558 -78.620 51.878
81616 94.568 -78.623 51.841
81617 94.580 -78.623 51.804
81618 94.593 -78.623 51.767
81619 94.605 -78.622 51.729
81620 94.617 -78.621 51.690
81621 94.629 -78.621 51.648
81622 94.642 -78.620 51.606
81623 94.654 -78.620 51.566
81624 94.666 -78.620 51.527
81625 94.679 -78.620 51.488
81626 94.692 -78.623 51.448
81627 94.705 -78.624 51.409
81628 94.718 -78.626 51.371
81629 94.740 -78.631 51.332
81630 94.758 -78.634 51.293
81631 94.777 -78.638 51.253
81632 94.796 -78.642 51.214
81633 94.815 -78.645 51.174
81634 94.834 -78.648 51.133
81635 94.846 -78.654 51.093
81636 94.852 -78.661 51.053
81637 94.858 -78.669 51.013
81638 94.867 -78.656 50.974
81639 94.876 -78.643 50.933
81640 94.886 -78.630 50.891
81641 94.895 -78.617 50.847
81642 94.904 -78.604 50.802
81643 94.932 -78.600 50.756
81644 94.970 -78.598 50.708
81645 95.007 -78.598 50.659
81646 95.044 -78.596 50.608
81647 95.056 -78.595 50.556
81648 95.048 -78.595 50.507
81649 95.041 -78.595 50.459
81650 95.033 -78.594 50.412
81651 95.025 -78.593 50.366
81652 95.018 -78.592 50.323
81653 95.010 -78.592 50.280
81654 95.010 -78.596 50.237
81655 95.016 -78.607 50.194
81656 95.017 -78.620 50.153
81657 95.023 -78.632 50.111
81658 95.038 -78.636 50.069
81659 95.053 -78.640 50.027
81660 95.068 -78.645 49.985
81661 95.083 -78.649 49.944
81662 95.097 -78.654 49.903
81663 95.112 -78.658 49.865
81664 95.127 -78.663 49.829
81665 95.142 -78.667 49.793
81666 95.157 -78.671 49.757
81667 95.171 -78.676 49.719
81668 95.186 -78.680 49.680
81669 95.201 -78.685 49.640
81670 95.219 -78.683 49.600
81671 95.176 -78.689 49.560
81672 95.174 -78.703 49.521
81673 95.203 -78.723 49.480
81674 95.233 -78.742 49.440
81675 95.262 -78.762 49.401
81676 95.290 -78.782 49.363
81677 95.320 -78.802 49.324
81678 95.349 -78.821 49.284
81679 95.372 -78.836 49.245
81680 95.392 -78.848 49.207
81681 95.411 -78.860 49.169
81682 95.425 -78.858 49.130
81683 95.440 -78.856 49.091
81684 95.453 -78.854 49.051
81685 95.462 -78.847 49.011
81686 95.466 -78.836 48.971
81687 95.470 -78.824 48.929
81688 95.474 -78.813 48.886
81689 95.478 -78.802 48.841
81690 95.482 -78.790 48.798
81691 95.494 -78.785 48.754
81692 95.517 -78.787 48.710
81693 95.541 -78.789 48.667
81694 95.564 -78.791 48.624
81695 95.581 -78.796 48.581
81696 95.587 -78.806 48.538
81697 95.593 -78.817 48.496
81698 95.599 -78.827 48.454
81699 95.605 -78.837 48.411
81700 95.611 -78.848 48.371
81701 95.617 -78.858 48.331
81702 95.623 -78.869 48.292
81703 95.629 -78.879 48.254
81704 95.635 -78.890 48.217
81705 95.649 -78.903 48.180
81706 95.665 -78.907 48.143
81707 95.683 -78.912 48.102
81708 95.707 -78.917 48.061
81709 95.731 -78.922 48.019
81710 95.755 -78.927 47.976
81711 95.779 -78.933 47.932
81712 95.803 -78.938 47.888
81713 95.827 -78.943 47.845
81714 95.850 -78.949 47.802
81715 95.862 -78.963 47.760
81716 95.874 -78.978 47.718
81717 95.894 -78.980 47.676
81718 95.918 -78.975 47.636
81719 95.943 -78.971 47.595
81720 95.967 -78.967 47.554
81721 95.991 -78.963 47.513
81722 96.016 -78.959 47.473
81723 96.039 -78.953 47.433
81724 96.062 -78.947 47.392
81725 96.085 -78.941 47.352
81726 96.108 -78.935 47.312
81727 96.131 -78.929 47.273
81728 96.168 -78.929 47.234
81729 96.208 -78.930 47.197
81730 96.243 -78.925 47.160
81731 96.276 -78.918 47.124
81732 96.304 -78.908 47.090
81733 96.329 -78.899 47.057
81734 96.354 -78.890 47.025
81735 96.379 -78.881 46.994
81736 96.398 -78.877 46.963
81737 96.399 -78.883 46.932
81738 96.401 -78.889 46.902
81739 96.402 -78.896 46.871
81740 96.403 -78.902 46.841
81741 96.406 -78.910 46.810
81742 96.417 -78.921 46.777
81743 96.427 -78.924 46.744
81744 96.437 -78.927 46.712
81745 96.446 -78.930 46.678
81746 96.474 -78.935 46.647
81747 96.518 -78.930 46.617
81748 96.563 -78.930 46.587
81749 96.608 -78.930 46.560
81750 96.654 -78.930 46.531
81751 96.699 -78.930 46.503
81752 96.744 -78.930 46.475
81753 96.790 -78.930 46.450
81754 96.832 -78.927 46.427
81755 96.851 -78.923 46.404
81756 96.861 -78.922 46.382
81757 96.871 -78.926 46.360
81758 96.892 -78.916 46.338
81759 96.904 -78.909 46.316
81760 96.916 -78.902 46.295
81761 96.915 -78.864 46.275
81762 96.905 -78.871 46.256
81763 96.907 -78.878 46.237
81764 96.938 -78.899 46.217
81765 96.966 -78.921 46.196
81766 96.995 -78.943 46.176
81767 96.996 -78.961 46.157
81768 96.995 -78.978 46.138
81769 96.994 -78.995 46.121
81770 96.993 -79.012 46.103
81771 96.990 -79.028 46.086
81772 96.983 -79.043 46.069
81773 96.994 -79.077 46.051
81774 97.015 -79.123 46.035
81775 97.036 -79.170 46.019
81776 97.057 -79.216 46.003
81777 97.078 -79.245 45.989
81778 97.098 -79.240 45.974
81779 97.118 -79.234 45.959
81780 97.139 -79.229 45.944
81781 97.148 -79.200 45.928
81782 97.157 -79.169 45.913
81783 97.166 -79.137 45.900
81784 97.175 -79.107 45.885
81785 97.185 -79.085 45.869
81786 97.201 -79.093 45.854
81787 97.215 -79.100 45.837
81788 97.230 -79.109 45.823
81789 97.247 -79.119 45.808
81790 97.260 -79.128 45.791
81791 97.275 -79.138 45.774
81792 97.288 -79.148 45.759
81793 97.302 -79.157 45.745
81794 97.316 -79.167 45.728
81795 97.329 -79.177 45.714
81796 97.344 -79.187 45.701
81797 97.357 -79.196 45.687
81798 97.371 -79.206 45.673
81799 97.385 -79.216 45.658
81800 97.399 -79.225 45.642
81801 97.413 -79.234 45.627
81802 97.421 -79.241 45.613
81803 97.432 -79.244 45.600
81804 97.443 -79.247 45.585
81805 97.454 -79.250 45.572
81806 97.465 -79.253 45.559
81807 97.480 -79.256 45.546
81808 97.495 -79.258 45.532
81809 97.510 -79.261 45.519
81810 97.524 -79.264 45.505
81811 97.539 -79.267 45.488
81812 97.554 -79.270 45.470
81813 97.571 -79.275 45.450
81814 97.591 -79.284 45.428
81815 97.610 -79.292 45.407
81816 97.630 -79.300 45.388
81817 97.650 -79.309 45.370
81818 97.670 -79.318 45.353
81819 97.674 -79.319 45.339
81820 97.674 -79.319 45.325
81821 97.674 -79.318 45.312
81822 97.674 -79.318 45.301
81823 97.690 -79.336 45.290
81824 97.709 -79.357 45.280
81825 97.727 -79.377 45.270
81826 97.745 -79.397 45.261
81827 97.764 -79.418 45.251
81828 97.782 -79.438 45.241
81829 97.796 -79.456 45.229
81830 97.806 -79.471 45.216
81831 97.816 -79.486 45.205
81832 97.828 -79.501 45.192
81833 97.842 -79.515 45.180
81834 97.855 -79.529 45.168
81835 97.867 -79.541 45.155
81836 97.878 -79.553 45.144
81837 97.890 -79.564 45.134
81838 97.901 -79.576 45.125
81839 97.913 -79.588 45.116
81840 97.925 -79.600 45.109
81841 97.937 -79.611 45.101
81842 97.958 -79.623 45.093
81843 97.984 -79.632 45.085
81844 98.012 -79.641 45.077
81845 98.040 -79.650 45.071
81846 98.069 -79.659 45.065
81847 98.097 -79.667 45.057
81848 98.134 -79.659 45.049
81849 98.171 -79.650 45.041
81850 98.208 -79.641 45.032
81851 98.241 -79.661 45.022
81852 98.277 -79.685 45.012
81853 98.303 -79.692 45.002
81854 98.329 -79.698 44.991
81855 98.356 -79.704 44.981
81856 98.382 -79.711 44.969
81857 98.409 -79.717 44.960
81858 98.435 -79.724 44.950
81859 98.462 -79.734 44.940
81860 98.489 -79.745 44.930
81861 98.516 -79.755 44.922
81862 98.544 -79.765 44.914
81863 98.572 -79.775 44.908
81864 98.604 -79.784 44.902
81865 98.637 -79.794 44.896
81866 98.670 -79.803 44.889
81867 98.688 -79.799 44.884
81868 98.703 -79.792 44.879
81869 98.717 -79.786 44.873
81870 98.731 -79.779 44.867
81871 98.749 -79.780 44.861
81872 98.770 -79.787 44.854
81873 98.791 -79.794 44.849
81874 98.812 -79.802 44.843
81875 98.833 -79.809 44.837
81876 98.855 -79.817 44.830
81877 98.877 -79.825 44.824
81878 98.899 -79.833 44.818
81879 98.920 -79.842 44.812
81880 98.942 -79.849 44.804
81881 98.968 -79.851 44.797
81882 98.993 -79.852 44.790
81883 99.015 -79.851 44.783
81884 99.037 -79.861 44.776
81885 99.061 -79.875 44.769
81886 99.083 -79.890 44.763
81887 99.102 -79.908 44.758
81888 99.122 -79.924 44.752
81889 99.144 -79.940 44.746
81890 99.176 -79.953 44.739
81891 99.208 -79.965 44.731
81892 99.239 -79.978 44.724
81893 99.266 -79.986 44.718
81894 99.289 -79.990 44.711
81895 99.312 -79.996 44.704
81896 99.336 -80.000 44.697
81897 99.359 -80.005 44.689
81898 99.382 -80.009 44.683
81899 99.406 -80.015 44.676
81900 99.429 -80.019 44.670
81901 99.452 -80.024 44.664
81902 99.475 -80.028 44.658
81903 99.493 -80.030 44.650
81904 99.523 -80.036 44.642
81905 99.562 -80.039 44.634
81906 99.565 -80.059 44.625
81907 99.556 -80.096 44.616
81908 99.548 -80.134 44.608
81909 99.540 -80.171 44.598
81910 99.535 -80.206 44.590
81911 99.538 -80.228 44.583
81912 99.541 -80.250 44.574
81913 99.544 -80.272 44.566
81914 99.548 -80.294 44.558
81915 99.551 -80.316 44.551
81916 99.554 -80.338 44.543
81917 99.557 -80.360 44.535
81918 99.560 -80.383 44.527
81919 99.564 -80.405 44.518
81920 99.567 -80.427 44.510
81921 99.570 -80.449 44.501
81922 99.573 -80.471 44.491
81923 99.576 -80.494 44.481
81924 99.611 -80.504 44.471
81925 99.658 -80.510 44.462
81926 99.706 -80.516 44.451
81927 99.755 -80.521 44.439
81928 99.802 -80.526 44.428
81929 99.850 -80.532 44.416
81930 99.898 -80.538 44.404
81931 99.946 -80.543 44.393
81932 99.994 -80.549 44.380
81933 100.042 -80.554 44.369
81934 100.092 -80.565 44.357
81935 100.143 -80.578 44.344
82229 93.556 -78.803 53.589
82230 93.565 -78.801 53.581
82231 93.575 -78.798 53.572
82232 93.585 -78.796 53.565
82233 93.594 -78.793 53.558
82234 93.604 -78.791 53.552
82235 93.614 -78.788 53.547
82236 93.624 -78.786 53.544
82237 93.633 -78.783 53.542
82238 93.641 -78.780 53.541
82239 93.641 -78.783 53.545
82240 93.644 -78.778 53.547
82241 93.648 -78.774 53.547
82242 93.658 -78.774 53.548
82243 93.666 -78.772 53.544
82244 93.672 -78.767 53.537
82245 93.678 -78.763 53.529
82246 93.700 -78.747 53.520
82247 93.724 -78.731 53.508
82248 93.749 -78.714 53.494
82249 93.770 -78.698 53.478
82250 93.789 -78.681 53.462
82251 93.801 -78.667 53.444
82252 93.804 -78.659 53.425
82253 93.808 -78.651 53.406
82254 93.812 -78.642 53.387
82255 93.815 -78.634 53.367
82256 93.818 -78.625 53.347
82257 93.822 -78.617 53.326
82258 93.825 -78.608 53.304
82259 93.828 -78.596 53.282
82260 93.840 -78.589 53.259
82261 93.852 -78.577 53.236
82262 93.865 -78.564 53.214
82263 93.878 -78.551 53.191
82264 93.891 -78.539 53.167
82265 93.903 -78.526 53.143
82266 93.916 -78.514 53.119
82267 93.928 -78.501 53.094
82268 93.950 -78.498 53.070
82269 93.960 -78.504 53.046
82270 93.970 -78.510 53.025
82271 93.980 -78.516 53.004
82272 93.991 -78.521 52.983
82273 94.001 -78.527 52.965
82274 94.011 -78.532 52.946
82275 94.022 -78.539 52.928
82276 94.032 -78.545 52.911
82277 94.043 -78.550 52.892
82278 94.053 -78.556 52.871
82279 94.066 -78.556 52.847
82280 94.072 -78.532 52.822
82281 94.077 -78.509 52.797
82282 94.087 -78.481 52.771
82283 94.107 -78.445 52.745
82284 94.137 -78.422 52.724
82285 94.182 -78.413 52.704
82286 94.226 -78.403 52.685
82287 94.271 -78.394 52.668
82288 94.314 -78.388 52.650
82289 94.321 -78.479 52.632
82290 94.330 -78.547 52.616
82291 94.345 -78.554 52.598
82292 94.360 -78.560 52.578
82293 94.375 -78.567 52.557
82294 94.385 -78.575 52.536
82295 94.377 -78.586 52.515
82296 94.369 -78.598 52.494
82297 94.362 -78.611 52.473
82298 94.354 -78.623 52.451
82299 94.346 -78.635 52.428
82300 94.351 -78.634 52.404
82301 94.367 -78.622 52.381
82302 94.384 -78.610 52.357
82303 94.400 -78.598 52.333
82304 94.416 -78.585 52.309
82305 94.431 -78.572 52.284
82306 94.439 -78.557 52.255
82307 94.448 -78.548 52.224
82308 94.461 -78.554 52.189
82309 94.474 -78.561 52.154
82310 94.488 -78.567 52.118
82311 94.501 -78.574 52.080
82312 94.514 -78.580 52.043
82313 94.530 -78.585 52.006
82314 94.546 -78.589 51.973
82315 94.552 -78.592 51.937
82316 94.560 -78.600 51.901
82317 94.569 -78.607 51.864
82318 94.581 -78.606 51.826
82319 94.593 -78.606 51.789
82320 94.606 -78.605 51.751
82321 94.618 -78.604 51.712
82322 94.630 -78.606 51.668
82323 94.643 -78.608 51.624
82324 94.657 -78.611 51.581
82325 94.670 -78.612 51.541
82326 94.683 -78.614 51.500
82327 94.695 -78.617 51.460
82328 94.708 -78.618 51.419
82329 94.722 -78.620 51.380
82330 94.743 -78.625 51.341
82331 94.764 -78.629 51.301
82332 94.784 -78.633 51.260
82333 94.803 -78.636 51.219
82334 94.821 -78.640 51.179
82335 94.840 -78.643 51.137
82336 94.856 -78.648 51.095
82337 94.862 -78.655 51.054
82338 94.868 -78.659 51.013
82339 94.878 -78.646 50.973
82340 94.887 -78.633 50.931
82341 94.896 -78.620 50.889
82342 94.905 -78.608 50.846
82343 94.937 -78.604 50.800
82344 94.974 -78.603 50.752
82345 95.012 -78.601 50.704
82346 95.049 -78.601 50.654
82347 95.086 -78.600 50.604
82348 95.098 -78.598 50.555
82349 95.091 -78.598 50.506
82350 95.083 -78.598 50.459
82351 95.075 -78.597 50.413
82352 95.067 -78.596 50.367
82353 95.060 -78.595 50.323
82354 95.052 -78.595 50.279
82355 95.044 -78.594 50.236
82356 95.037 -78.593 50.195
82357 95.036 -78.600 50.154
82358 95.047 -78.607 50.112
82359 95.062 -78.611 50.069
82360 95.077 -78.616 50.028
82361 95.092 -78.620 49.986
82362 95.107 -78.624 49.945
82363 95.121 -78.629 49.905
82364 95.131 -78.632 49.867
82365 95.140 -78.636 49.830
82366 95.149 -78.639 49.795
82367 95.159 -78.642 49.758
82368 95.169 -78.645 49.720
82369 95.178 -78.648 49.680
82370 95.188 -78.652 49.642
82371 95.196 -78.657 49.603
82372 95.177 -78.664 49.564
82373 95.192 -78.686 49.524
82374 95.208 -78.708 49.483
82375 95.225 -78.731 49.443
82376 95.253 -78.751 49.404
82377 95.282 -78.771 49.367
82378 95.312 -78.790 49.329
82379 95.341 -78.810 49.291
82380 95.370 -78.830 49.252
82381 95.392 -78.843 49.213
82382 95.411 -78.855 49.173
82383 95.426 -78.855 49.134
82384 95.440 -78.852 49.095
82385 95.449 -78.846 49.056
82386 95.453 -78.834 49.016
82387 95.457 -78.823 48.975
82388 95.460 -78.811 48.934
82389 95.464 -78.800 48.892
82390 95.468 -78.789 48.848
82391 95.472 -78.777 48.804
82392 95.483 -78.771 48.760
82393 95.507 -78.774 48.716
82394 95.530 -78.775 48.672
82395 95.554 -78.777 48.629
82396 95.577 -78.779 48.587
82397 95.590 -78.786 48.544
82398 95.596 -78.797 48.502
82399 95.602 -78.808 48.460
82400 95.607 -78.818 48.419
82401 95.613 -78.828 48.378
82402 95.619 -78.839 48.339
82403 95.625 -78.849 48.300
82404 95.631 -78.860 48.263
82405 95.637 -78.870 48.226
82406 95.645 -78.878 48.191
82407 95.670 -78.886 48.154
82408 95.694 -78.892 48.114
82409 95.718 -78.897 48.073
82410 95.742 -78.902 48.031
82411 95.766 -78.908 47.988
82412 95.790 -78.913 47.945
82413 95.814 -78.918 47.901
82414 95.837 -78.924 47.858
82415 95.849 -78.938 47.815
82416 95.861 -78.953 47.772
82417 95.872 -78.968 47.730
82418 95.893 -78.968 47.687
82419 95.918 -78.964 47.648
82420 95.942 -78.959 47.608
82421 95.966 -78.955 47.567
82422 95.990 -78.951 47.527
82423 96.015 -78.947 47.487
82424 96.039 -78.943 47.447
82425 96.063 -78.939 47.405
82426 96.088 -78.934 47.364
82427 96.111 -78.928 47.323
82428 96.134 -78.922 47.283
82429 96.168 -78.921 47.245
82430 96.204 -78.918 47.208
82431 96.234 -78.909 47.171
82432 96.259 -78.900 47.133
82433 96.285 -78.891 47.099
82434 96.310 -78.882 47.066
82435 96.335 -78.873 47.033
82436 96.360 -78.864 47.002
82437 96.382 -78.856 46.971
82438 96.384 -78.863 46.939
82439 96.385 -78.869 46.909
82440 96.386 -78.876 46.880
82441 96.388 -78.882 46.849
82442 96.397 -78.897 46.818
82443 96.408 -78.914 46.785
82444 96.418 -78.916 46.752
82445 96.428 -78.919 46.719
82446 96.437 -78.921 46.686
82447 96.466 -78.923 46.655
82448 96.512 -78.922 46.625
82449 96.557 -78.922 46.596
82450 96.602 -78.922 46.568
82451 96.648 -78.922 46.540
82452 96.693 -78.922 46.512
82453 96.738 -78.921 46.485
82454 96.781 -78.919 46.461
82455 96.822 -78.915 46.438
82456 96.847 -78.911 46.417
82457 96.858 -78.908 46.395
82458 96.868 -78.905 46.373
82459 96.878 -78.906 46.351
82460 96.895 -78.899 46.331
82461 96.907 -78.893 46.309
82462 96.904 -78.870 46.288
82463 96.909 -78.872 46.268
82464 96.918 -78.878 46.247
82465 96.934 -78.900 46.228
82466 96.964 -78.919 46.208
82467 96.995 -78.940 46.188
82468 97.006 -78.959 46.170
82469 97.005 -78.976 46.152
82470 96.999 -78.991 46.135
82471 96.993 -79.006 46.118
82472 96.986 -79.021 46.100
82473 96.980 -79.036 46.082
82474 96.991 -79.071 46.065
82475 97.012 -79.117 46.050
82476 97.033 -79.164 46.035
82477 97.054 -79.210 46.020
82478 97.075 -79.215 46.006
82479 97.095 -79.210 45.990
82480 97.115 -79.205 45.975
82481 97.136 -79.200 45.960
82482 97.156 -79.194 45.944
82483 97.167 -79.169 45.930
82484 97.176 -79.137 45.916
82485 97.185 -79.106 45.901
82486 97.196 -79.088 45.886
82487 97.211 -79.096 45.870
82488 97.226 -79.103 45.855
82489 97.238 -79.109 45.840
82490 97.248 -79.115 45.825
82491 97.261 -79.125 45.808
82492 97.275 -79.134 45.791
82493 97.289 -79.143 45.776
82494 97.303 -79.153 45.761
82495 97.316 -79.162 45.744
82496 97.330 -79.172 45.731
82497 97.344 -79.181 45.717
82498 97.358 -79.190 45.702
82499 97.371 -79.200 45.688
82500 97.385 -79.209 45.673
82501 97.399 -79.219 45.658
82502 97.413 -79.228 45.642
82503 97.424 -79.236 45.627
82504 97.435 -79.238 45.612
82505 97.450 -79.241 45.596
82506 97.465 -79.244 45.582
82507 97.479 -79.247 45.569
82508 97.494 -79.250 45.554
82509 97.509 -79.253 45.539
82510 97.523 -79.256 45.524
82511 97.538 -79.259 45.509
82512 97.553 -79.262 45.492
82513 97.568 -79.265 45.474
82514 97.586 -79.272 45.453
82515 97.606 -79.281 45.431
82516 97.626 -79.289 45.411
82517 97.646 -79.297 45.393
82518 97.665 -79.306 45.376
82519 97.665 -79.306 45.361
82520 97.665 -79.305 45.346
82521 97.665 -79.304 45.332
82522 97.665 -79.304 45.319
82523 97.665 -79.303 45.307
82524 97.682 -79.323 45.296
82525 97.701 -79.344 45.286
82526 97.719 -79.364 45.275
82527 97.737 -79.385 45.265
82528 97.756 -79.405 45.256
82529 97.774 -79.426 45.246
82530 97.792 -79.446 45.234
82531 97.811 -79.466 45.222
82532 97.823 -79.483 45.211
82533 97.834 -79.498 45.198
82534 97.847 -79.512 45.186
82535 97.860 -79.524 45.175
82536 97.871 -79.536 45.163
82537 97.882 -79.548 45.152
82538 97.894 -79.560 45.142
82539 97.906 -79.571 45.133
82540 97.917 -79.582 45.123
82541 97.929 -79.595 45.116
82542 97.940 -79.606 45.108
82543 97.959 -79.617 45.099
82544 97.986 -79.627 45.091
82545 98.014 -79.636 45.083
82546 98.042 -79.645 45.076
82547 98.071 -79.653 45.069
82548 98.108 -79.644 45.061
82549 98.145 -79.635 45.053
82550 98.182 -79.626 45.046
82551 98.218 -79.616 45.037
82552 98.251 -79.636 45.027
82553 98.287 -79.661 45.018
82554 98.320 -79.679 45.009
82555 98.346 -79.685 44.999
82556 98.372 -79.692 44.989
82557 98.398 -79.698 44.978
82558 98.425 -79.704 44.968
82559 98.451 -79.712 44.959
82560 98.479 -79.723 44.949
82561 98.506 -79.733 44.941
82562 98.533 -79.743 44.933
82563 98.560 -79.754 44.925
82564 98.587 -79.764 44.920
82565 98.619 -79.773 44.914
82566 98.652 -79.783 44.908
82567 98.681 -79.789 44.902
82568 98.696 -79.783 44.897
82569 98.710 -79.776 44.892
82570 98.725 -79.770 44.888
82571 98.739 -79.763 44.882
82572 98.755 -79.758 44.875
82573 98.776 -79.765 44.869
82574 98.796 -79.773 44.864
82575 98.817 -79.780 44.858
82576 98.839 -79.788 44.852
82577 98.861 -79.796 44.845
82578 98.882 -79.804 44.839
82579 98.905 -79.812 44.833
82580 98.926 -79.820 44.826
82581 98.948 -79.828 44.819
82582 98.970 -79.836 44.813
82583 98.995 -79.838 44.806
82584 99.020 -79.839 44.799
82585 99.041 -79.846 44.792
82586 99.065 -79.860 44.784
82587 99.089 -79.874 44.776
82588 99.110 -79.890 44.770
82589 99.129 -79.907 44.764
82590 99.154 -79.922 44.758
82591 99.186 -79.934 44.750
82592 99.218 -79.947 44.742
82593 99.242 -79.952 44.733
82594 99.265 -79.958 44.724
82595 99.288 -79.962 44.717
82596 99.311 -79.967 44.710
82597 99.335 -79.971 44.702
82598 99.358 -79.977 44.694
82599 99.381 -79.981 44.687
82600 99.405 -79.986 44.681
82601 99.428 -79.990 44.674
82602 99.451 -79.995 44.668
82603 99.474 -80.000 44.661
82604 99.497 -80.005 44.654
82605 99.521 -80.009 44.647
82606 99.544 -80.014 44.639
82607 99.571 -80.019 44.630
82608 99.577 -80.040 44.621
82609 99.569 -80.078 44.612
82610 99.561 -80.115 44.603
82611 99.553 -80.153 44.595
82612 99.544 -80.191 44.587
82613 99.538 -80.227 44.579
82614 99.541 -80.249 44.571
82615 99.544 -80.272 44.563
82616 99.547 -80.294 44.556
82617 99.550 -80.316 44.548
82618 99.554 -80.337 44.540
82619 99.557 -80.359 44.532
82620 99.560 -80.382 44.523
82621 99.563 -80.404 44.515
82622 99.566 -80.426 44.507
82623 99.570 -80.448 44.497
82624 99.573 -80.471 44.487
82625 99.619 -80.477 44.477
82626 99.667 -80.482 44.467
82627 99.715 -80.488 44.456
82628 99.763 -80.494 44.444
82629 99.810 -80.499 44.432
82630 99.858 -80.505 44.420
82631 99.906 -80.510 44.409
82632 99.955 -80.516 44.397
82633 100.005 -80.527 44.384
82634 100.056 -80.540 44.373
82635 100.107 -80.553 44.361
82636 100.158 -80.566 44.349
82930 93.556 -78.793 53.597
82931 93.566 -78.789 53.589
82932 93.575 -78.787 53.581
82933 93.585 -78.785 53.573
82934 93.594 -78.783 53.566
82935 93.605 -78.780 53.560
82936 93.614 -78.777 53.555
82937 93.624 -78.775 53.551
82938 93.633 -78.773 53.548
82939 93.638 -78.767 53.546
82940 93.637 -78.758 53.549
82941 93.638 -78.773 53.552
82942 93.642 -78.769 53.554
82943 93.652 -78.768 53.554
82944 93.661 -78.767 53.551
82945 93.669 -78.765 53.545
82946 93.689 -78.752 53.536
82947 93.708 -78.735 53.525
82948 93.727 -78.718 53.513
82949 93.747 -78.701 53.499
82950 93.766 -78.684 53.484
82951 93.785 -78.667 53.466
82952 93.796 -78.655 53.448
82953 93.799 -78.645 53.429
82954 93.803 -78.637 53.410
82955 93.806 -78.629 53.391
82956 93.809 -78.620 53.372
82957 93.813 -78.612 53.351
82958 93.817 -78.603 53.331
82959 93.819 -78.589 53.309
82960 93.822 -78.575 53.286
82961 93.835 -78.579 53.262
82962 93.847 -78.566 53.238
82963 93.860 -78.554 53.216
82964 93.873 -78.541 53.193
82965 93.886 -78.529 53.169
82966 93.898 -78.516 53.143
82967 93.911 -78.503 53.120
82968 93.939 -78.488 53.096
82969 93.960 -78.486 53.073
82970 93.964 -78.495 53.051
82971 93.974 -78.501 53.031
82972 93.985 -78.507 53.011
82973 93.995 -78.512 52.990
82974 94.005 -78.518 52.972
82975 94.016 -78.523 52.953
82976 94.027 -78.529 52.935
82977 94.037 -78.535 52.917
82978 94.048 -78.540 52.898
82979 94.062 -78.539 52.876
82980 94.077 -78.537 52.853
82981 94.089 -78.529 52.827
82982 94.094 -78.506 52.802
82983 94.100 -78.482 52.777
82984 94.119 -78.447 52.752
82985 94.139 -78.411 52.731
82986 94.161 -78.379 52.711
82987 94.205 -78.369 52.694
82988 94.250 -78.360 52.677
82989 94.292 -78.356 52.660
82990 94.299 -78.445 52.643
82991 94.307 -78.535 52.626
82992 94.321 -78.555 52.608
82993 94.335 -78.561 52.589
82994 94.350 -78.568 52.568
82995 94.365 -78.575 52.547
82996 94.380 -78.582 52.527
82997 94.376 -78.593 52.507
82998 94.368 -78.604 52.486
82999 94.360 -78.617 52.464
83000 94.352 -78.629 52.442
83001 94.353 -78.633 52.418
83002 94.369 -78.620 52.395
83003 94.386 -78.608 52.371
83004 94.402 -78.595 52.347
83005 94.418 -78.582 52.323
83006 94.425 -78.568 52.298
83007 94.432 -78.553 52.269
83008 94.442 -78.549 52.238
83009 94.456 -78.555 52.204
83010 94.470 -78.561 52.170
83011 94.485 -78.566 52.135
83012 94.501 -78.570 52.097
83013 94.517 -78.574 52.061
83014 94.532 -78.579 52.026
83015 94.548 -78.583 51.992
83016 94.557 -78.586 51.958
83017 94.563 -78.588 51.923
83018 94.570 -78.590 51.886
83019 94.582 -78.592 51.848
83020 94.595 -78.594 51.810
83021 94.608 -78.596 51.771
83022 94.621 -78.598 51.731
83023 94.634 -78.600 51.688
83024 94.647 -78.602 51.643
83025 94.660 -78.604 51.596
83026 94.673 -78.606 51.554
83027 94.686 -78.608 51.512
83028 94.699 -78.610 51.470
83029 94.712 -78.612 51.428
83030 94.725 -78.614 51.387
83031 94.747 -78.619 51.346
83032 94.768 -78.623 51.305
83033 94.789 -78.627 51.262
83034 94.809 -78.631 51.221
83035 94.828 -78.635 51.180
83036 94.846 -78.638 51.138
83037 94.865 -78.642 51.095
83038 94.872 -78.649 51.052
83039 94.879 -78.648 51.009
83040 94.888 -78.636 50.967
83041 94.897 -78.623 50.925
83042 94.906 -78.611 50.882
83043 94.942 -78.608 50.840
83044 94.979 -78.608 50.794
83045 95.016 -78.606 50.746
83046 95.053 -78.605 50.698
83047 95.072 -78.605 50.649
83048 95.075 -78.607 50.601
83049 95.069 -78.606 50.553
83050 95.064 -78.605 50.505
83051 95.058 -78.604 50.459
83052 95.052 -78.604 50.413
83053 95.047 -78.603 50.367
83054 95.041 -78.602 50.324
83055 95.036 -78.601 50.279
83056 95.030 -78.601 50.236
83057 95.030 -78.595 50.194
83058 95.046 -78.580 50.154
83059 95.061 -78.585 50.112
83060 95.072 -78.589 50.069
83061 95.083 -78.592 50.027
83062 95.094 -78.596 49.986
83063 95.104 -78.599 49.945
83064 95.115 -78.603 49.906
83065 95.126 -78.607 49.867
83066 95.136 -78.611 49.830
83067 95.147 -78.614 49.794
83068 95.158 -78.618 49.757
83069 95.168 -78.622 49.719
83070 95.179 -78.625 49.681
83071 95.190 -78.629 49.643
83072 95.214 -78.641 49.606
83073 95.224 -78.652 49.567
83074 95.232 -78.670 49.527
83075 95.246 -78.691 49.486
83076 95.259 -78.712 49.446
83077 95.274 -78.733 49.407
83078 95.290 -78.756 49.370
83079 95.305 -78.778 49.333
83080 95.333 -78.799 49.297
83081 95.361 -78.818 49.259
83082 95.391 -78.838 49.220
83083 95.411 -78.850 49.181
83084 95.426 -78.851 49.140
83085 95.436 -78.844 49.100
83086 95.439 -78.833 49.061
83087 95.443 -78.821 49.020
83088 95.447 -78.810 48.980
83089 95.451 -78.799 48.939
83090 95.455 -78.787 48.897
83091 95.459 -78.776 48.854
83092 95.463 -78.764 48.809
83093 95.473 -78.758 48.765
83094 95.496 -78.760 48.721
83095 95.520 -78.761 48.677
83096 95.543 -78.764 48.635
83097 95.567 -78.766 48.593
83098 95.590 -78.767 48.550
83099 95.598 -78.777 48.508
83100 95.604 -78.788 48.467
83101 95.610 -78.798 48.427
83102 95.616 -78.808 48.387
83103 95.622 -78.819 48.348
83104 95.628 -78.830 48.310
83105 95.633 -78.840 48.273
83106 95.641 -78.849 48.237
83107 95.654 -78.852 48.202
83108 95.678 -78.862 48.166
83109 95.705 -78.872 48.127
83110 95.729 -78.877 48.086
83111 95.753 -78.883 48.045
83112 95.777 -78.888 48.001
83113 95.801 -78.893 47.958
83114 95.825 -78.899 47.915
83115 95.837 -78.913 47.872
83116 95.848 -78.928 47.828
83117 95.859 -78.943 47.786
83118 95.873 -78.953 47.743
83119 95.894 -78.952 47.700
83120 95.918 -78.949 47.660
83121 95.942 -78.945 47.619
83122 95.966 -78.942 47.579
83123 95.990 -78.939 47.540
83124 96.015 -78.935 47.500
83125 96.039 -78.931 47.459
83126 96.063 -78.927 47.416
83127 96.088 -78.923 47.374
83128 96.112 -78.919 47.334
83129 96.136 -78.915 47.294
83130 96.165 -78.910 47.256
83131 96.190 -78.901 47.219
83132 96.215 -78.892 47.181
83133 96.240 -78.883 47.144
83134 96.265 -78.874 47.108
83135 96.291 -78.865 47.074
83136 96.316 -78.855 47.042
83137 96.340 -78.846 47.011
83138 96.366 -78.837 46.980
83139 96.368 -78.843 46.948
83140 96.369 -78.849 46.916
83141 96.371 -78.855 46.886
83142 96.376 -78.868 46.854
83143 96.388 -78.885 46.822
83144 96.399 -78.902 46.791
83145 96.409 -78.908 46.758
83146 96.418 -78.911 46.726
83147 96.429 -78.914 46.694
83148 96.461 -78.915 46.664
83149 96.506 -78.915 46.634
83150 96.551 -78.914 46.606
83151 96.596 -78.914 46.577
83152 96.642 -78.914 46.550
83153 96.687 -78.914 46.523
83154 96.731 -78.912 46.497
83155 96.771 -78.907 46.473
83156 96.812 -78.902 46.451
83157 96.844 -78.898 46.430
83158 96.854 -78.895 46.408
83159 96.864 -78.893 46.386
83160 96.874 -78.890 46.365
83161 96.885 -78.887 46.345
83162 96.898 -78.883 46.322
83163 96.901 -78.873 46.301
83164 96.922 -78.869 46.280
83165 96.929 -78.877 46.259
83166 96.929 -78.902 46.239
83167 96.960 -78.921 46.220
83168 96.990 -78.946 46.201
83169 97.008 -78.954 46.184
83170 97.002 -78.969 46.167
83171 96.996 -78.984 46.150
83172 96.990 -78.999 46.132
83173 96.983 -79.013 46.115
83174 96.977 -79.028 46.097
83175 96.989 -79.065 46.080
83176 97.010 -79.111 46.066
83177 97.031 -79.158 46.052
83178 97.052 -79.190 46.038
83179 97.073 -79.185 46.022
83180 97.093 -79.180 46.006
83181 97.113 -79.175 45.991
83182 97.133 -79.170 45.976
83183 97.153 -79.165 45.962
83184 97.174 -79.159 45.948
83185 97.186 -79.137 45.933
83186 97.195 -79.106 45.918
83187 97.207 -79.091 45.903
83188 97.222 -79.099 45.888
83189 97.232 -79.104 45.873
83190 97.239 -79.107 45.857
83191 97.249 -79.112 45.842
83192 97.262 -79.122 45.825
83193 97.277 -79.131 45.809
83194 97.290 -79.140 45.793
83195 97.304 -79.150 45.777
83196 97.317 -79.159 45.761
83197 97.331 -79.169 45.746
83198 97.345 -79.178 45.733
83199 97.359 -79.187 45.717
83200 97.372 -79.197 45.702
83201 97.386 -79.206 45.687
83202 97.401 -79.215 45.673
83203 97.416 -79.223 45.657
83204 97.429 -79.225 45.640
83205 97.449 -79.234 45.622
83206 97.464 -79.237 45.604
83207 97.478 -79.239 45.589
83208 97.493 -79.242 45.574
83209 97.508 -79.245 45.558
83210 97.523 -79.248 45.541
83211 97.537 -79.251 45.523
83212 97.552 -79.254 45.506
83213 97.567 -79.257 45.489
83214 97.582 -79.260 45.471
83215 97.602 -79.269 45.453
83216 97.621 -79.278 45.435
83217 97.642 -79.286 45.417
83218 97.657 -79.292 45.401
83219 97.657 -79.292 45.385
83220 97.657 -79.291 45.370
83221 97.657 -79.291 45.355
83222 97.656 -79.291 45.341
83223 97.656 -79.290 45.328
83224 97.657 -79.291 45.315
83225 97.675 -79.310 45.303
83226 97.693 -79.331 45.293
83227 97.712 -79.351 45.282
83228 97.730 -79.372 45.273
83229 97.748 -79.392 45.263
83230 97.767 -79.413 45.252
83231 97.785 -79.433 45.241
83232 97.803 -79.454 45.230
83233 97.821 -79.473 45.219
83234 97.840 -79.494 45.207
83235 97.852 -79.507 45.195
83236 97.863 -79.519 45.183
83237 97.875 -79.531 45.172
83238 97.886 -79.542 45.161
83239 97.898 -79.554 45.150
83240 97.910 -79.566 45.141
83241 97.921 -79.578 45.131
83242 97.933 -79.589 45.123
83243 97.944 -79.601 45.115
83244 97.961 -79.612 45.106
83245 97.988 -79.622 45.097
83246 98.016 -79.631 45.089
83247 98.045 -79.638 45.082
83248 98.082 -79.629 45.075
83249 98.119 -79.620 45.067
83250 98.156 -79.611 45.060
83251 98.194 -79.602 45.053
83252 98.229 -79.592 45.044
83253 98.262 -79.610 45.035
83254 98.298 -79.636 45.027
83255 98.334 -79.660 45.017
83256 98.363 -79.672 45.008
83257 98.389 -79.679 44.998
83258 98.415 -79.685 44.988
83259 98.441 -79.692 44.978
83260 98.468 -79.701 44.968
83261 98.496 -79.711 44.960
83262 98.523 -79.722 44.952
83263 98.550 -79.732 44.944
83264 98.577 -79.742 44.937
83265 98.604 -79.752 44.931
83266 98.633 -79.763 44.925
83267 98.666 -79.772 44.920
83268 98.690 -79.773 44.915
83269 98.704 -79.767 44.910
83270 98.718 -79.760 44.907
83271 98.733 -79.753 44.903
83272 98.747 -79.746 44.897
83273 98.762 -79.739 44.890
83274 98.781 -79.744 44.884
83275 98.802 -79.751 44.878
83276 98.823 -79.758 44.873
83277 98.845 -79.767 44.867
83278 98.867 -79.774 44.860
83279 98.888 -79.783 44.854
83280 98.910 -79.791 44.847
83281 98.932 -79.798 44.841
83282 98.954 -79.807 44.835
83283 98.976 -79.815 44.828
83284 98.997 -79.823 44.821
83285 99.022 -79.825 44.814
83286 99.045 -79.830 44.807
83287 99.069 -79.845 44.798
83288 99.093 -79.858 44.789
83289 99.116 -79.873 44.783
83290 99.135 -79.889 44.776
83291 99.164 -79.903 44.769
83292 99.195 -79.914 44.760
83293 99.218 -79.920 44.750
83294 99.241 -79.924 44.740
83295 99.264 -79.929 44.730
83296 99.287 -79.933 44.722
83297 99.311 -79.938 44.714
83298 99.334 -79.943 44.706
83299 99.357 -79.948 44.699
83300 99.380 -79.952 44.692
83301 99.404 -79.957 44.686
83302 99.427 -79.962 44.679
83303 99.450 -79.967 44.672
83304 99.474 -79.971 44.665
83305 99.497 -79.976 44.658
83306 99.520 -79.982 44.651
83307 99.544 -79.988 44.643
83308 99.568 -79.995 44.634
83309 99.588 -79.998 44.625
83310 99.587 -80.026 44.616
83311 99.582 -80.060 44.607
83312 99.573 -80.097 44.599
83313 99.565 -80.135 44.591
83314 99.557 -80.173 44.583
83315 99.549 -80.210 44.576
83316 99.541 -80.248 44.569
83317 99.544 -80.271 44.561
83318 99.547 -80.293 44.553
83319 99.550 -80.315 44.545
83320 99.553 -80.337 44.537
83321 99.557 -80.359 44.529
83322 99.560 -80.381 44.520
83323 99.563 -80.403 44.512
83324 99.566 -80.425 44.502
83325 99.579 -80.444 44.493
83326 99.627 -80.450 44.483
83327 99.675 -80.455 44.471
83328 99.723 -80.461 44.461
83329 99.771 -80.466 44.450
83330 99.819 -80.472 44.439
83331 99.867 -80.478 44.426
83332 99.917 -80.489 44.414
83333 99.969 -80.501 44.403
83334 100.020 -80.514 44.390
83335 100.071 -80.527 44.378
83336 100.122 -80.540 44.366
83337 100.174 -80.553 44.353
83632 93.566 -78.779 53.597
83633 93.576 -78.777 53.588
83634 93.585 -78.774 53.579
83635 93.595 -78.771 53.573
83636 93.605 -78.769 53.568
83637 93.614 -78.767 53.562
83638 93.624 -78.764 53.557
83639 93.634 -78.761 53.553
83640 93.635 -78.754 53.551
83641 93.633 -78.745 53.554
83642 93.632 -78.767 53.557
83643 93.636 -78.763 53.560
83644 93.645 -78.762 53.559
83645 93.655 -78.761 53.556
83646 93.667 -78.749 53.550
83647 93.685 -78.738 53.541
83648 93.704 -78.720 53.530
83649 93.723 -78.704 53.517
83650 93.743 -78.686 53.503
83651 93.762 -78.670 53.488
83652 93.781 -78.653 53.470
83653 93.791 -78.641 53.452
83654 93.794 -78.633 53.433
83655 93.797 -78.623 53.414
83656 93.801 -78.615 53.395
83657 93.804 -78.607 53.375
83658 93.807 -78.596 53.355
83659 93.811 -78.582 53.333
83660 93.813 -78.568 53.310
83661 93.816 -78.554 53.286
83662 93.824 -78.550 53.262
83663 93.842 -78.555 53.238
83664 93.855 -78.542 53.215
83665 93.868 -78.530 53.192
83666 93.880 -78.517 53.168
83667 93.893 -78.505 53.143
83668 93.908 -78.492 53.119
83669 93.952 -78.474 53.096
83670 93.970 -78.474 53.075
83671 93.971 -78.484 53.055
83672 93.979 -78.492 53.038
83673 93.989 -78.497 53.019
83674 94.000 -78.503 52.998
83675 94.010 -78.508 52.978
83676 94.020 -78.514 52.959
83677 94.031 -78.520 52.941
83678 94.043 -78.523 52.922
83679 94.058 -78.521 52.902
83680 94.072 -78.520 52.880
83681 94.087 -78.518 52.856
83682 94.102 -78.517 52.830
83683 94.111 -78.503 52.804
83684 94.117 -78.479 52.780
83685 94.132 -78.448 52.757
83686 94.151 -78.412 52.736
83687 94.171 -78.376 52.717
83688 94.190 -78.341 52.701
83689 94.229 -78.326 52.686
83690 94.271 -78.322 52.669
83691 94.278 -78.413 52.652
83692 94.285 -78.503 52.635
83693 94.295 -78.556 52.617
83694 94.311 -78.563 52.598
83695 94.325 -78.570 52.577
83696 94.340 -78.576 52.557
83697 94.355 -78.583 52.538
83698 94.370 -78.589 52.519
83699 94.374 -78.599 52.498
83700 94.366 -78.611 52.477
83701 94.359 -78.623 52.455
83702 94.355 -78.631 52.432
83703 94.371 -78.618 52.409
83704 94.388 -78.606 52.385
83705 94.404 -78.594 52.361
83706 94.411 -78.579 52.337
83707 94.418 -78.564 52.310
83708 94.427 -78.549 52.281
83709 94.441 -78.547 52.249
83710 94.457 -78.551 52.216
83711 94.472 -78.556 52.183
83712 94.488 -78.560 52.148
83713 94.503 -78.564 52.112
83714 94.519 -78.569 52.076
83715 94.535 -78.573 52.041
83716 94.550 -78.577 52.008
83717 94.563 -78.581 51.976
83718 94.569 -78.582 51.942
83719 94.575 -78.585 51.905
83720 94.585 -78.583 51.866
83721 94.599 -78.587 51.828
83722 94.612 -78.590 51.789
83723 94.625 -78.592 51.748
83724 94.638 -78.594 51.705
83725 94.651 -78.596 51.658
83726 94.664 -78.598 51.611
83727 94.677 -78.600 51.566
83728 94.690 -78.602 51.522
83729 94.703 -78.604 51.478
83730 94.716 -78.606 51.434
83731 94.729 -78.608 51.389
83732 94.750 -78.612 51.345
83733 94.771 -78.617 51.301
83734 94.792 -78.621 51.258
83735 94.814 -78.625 51.215
83736 94.834 -78.630 51.173
83737 94.853 -78.633 51.130
83738 94.872 -78.636 51.085
83739 94.881 -78.643 51.042
83740 94.889 -78.639 50.999
83741 94.898 -78.626 50.955
83742 94.909 -78.614 50.913
83743 94.946 -78.613 50.871
83744 94.983 -78.611 50.829
83745 95.021 -78.611 50.784
83746 95.030 -78.612 50.738
83747 95.034 -78.614 50.692
83748 95.039 -78.616 50.644
83749 95.034 -78.615 50.597
83750 95.029 -78.614 50.551
83751 95.023 -78.614 50.504
83752 95.017 -78.613 50.459
83753 95.012 -78.612 50.414
83754 95.006 -78.611 50.369
83755 95.000 -78.611 50.324
83756 95.006 -78.599 50.280
83757 95.021 -78.581 50.237
83758 95.035 -78.564 50.194
83759 95.054 -78.561 50.152
83760 95.074 -78.566 50.110
83761 95.086 -78.570 50.068
83762 95.097 -78.573 50.026
83763 95.108 -78.576 49.984
83764 95.118 -78.580 49.943
83765 95.129 -78.584 49.905
83766 95.139 -78.588 49.866
83767 95.150 -78.592 49.828
83768 95.161 -78.595 49.791
83769 95.172 -78.598 49.754
83770 95.182 -78.602 49.717
83771 95.193 -78.606 49.680
83772 95.208 -78.612 49.644
83773 95.241 -78.630 49.606
83774 95.274 -78.646 49.568
83775 95.275 -78.656 49.528
83776 95.289 -78.677 49.488
83777 95.303 -78.698 49.448
83778 95.317 -78.720 49.409
83779 95.331 -78.741 49.372
83780 95.345 -78.762 49.337
83781 95.358 -78.783 49.302
83782 95.373 -78.804 49.266
83783 95.387 -78.825 49.228
83784 95.412 -78.846 49.189
83785 95.422 -78.843 49.148
83786 95.426 -78.831 49.107
83787 95.429 -78.820 49.066
83788 95.433 -78.808 49.026
83789 95.438 -78.797 48.985
83790 95.442 -78.786 48.944
83791 95.445 -78.774 48.902
83792 95.449 -78.763 48.859
83793 95.453 -78.752 48.815
83794 95.463 -78.744 48.770
83795 95.486 -78.746 48.725
83796 95.510 -78.749 48.682
83797 95.533 -78.750 48.641
83798 95.556 -78.752 48.599
83799 95.580 -78.754 48.557
83800 95.600 -78.758 48.515
83801 95.606 -78.768 48.475
83802 95.612 -78.779 48.436
83803 95.618 -78.789 48.397
83804 95.624 -78.799 48.358
83805 95.630 -78.810 48.320
83806 95.637 -78.819 48.283
83807 95.650 -78.824 48.248
83808 95.661 -78.827 48.213
83809 95.684 -78.836 48.178
83810 95.713 -78.849 48.139
83811 95.740 -78.858 48.099
83812 95.764 -78.863 48.058
83813 95.788 -78.868 48.014
83814 95.812 -78.874 47.971
83815 95.824 -78.888 47.929
83816 95.835 -78.903 47.886
83817 95.850 -78.908 47.843
83818 95.868 -78.910 47.800
83819 95.883 -78.914 47.757
83820 95.896 -78.917 47.714
83821 95.920 -78.914 47.673
83822 95.945 -78.911 47.631
83823 95.969 -78.907 47.590
83824 95.993 -78.904 47.551
83825 96.017 -78.901 47.511
83826 96.041 -78.898 47.469
83827 96.065 -78.894 47.427
83828 96.090 -78.891 47.385
83829 96.114 -78.888 47.344
83830 96.138 -78.885 47.305
83831 96.156 -78.896 47.267
83832 96.172 -78.883 47.230
83833 96.196 -78.874 47.193
83834 96.221 -78.865 47.155
83835 96.246 -78.855 47.118
83836 96.271 -78.846 47.083
83837 96.297 -78.837 47.051
83838 96.321 -78.828 47.020
83839 96.346 -78.820 46.989
83840 96.352 -78.824 46.956
83841 96.353 -78.830 46.923
83842 96.356 -78.837 46.891
83843 96.367 -78.855 46.858
83844 96.378 -78.872 46.826
83845 96.389 -78.890 46.796
83846 96.400 -78.900 46.764
83847 96.410 -78.903 46.732
83848 96.419 -78.905 46.702
83849 96.455 -78.906 46.672
83850 96.500 -78.906 46.643
83851 96.545 -78.906 46.616
83852 96.591 -78.905 46.587
83853 96.636 -78.905 46.560
83854 96.680 -78.905 46.534
83855 96.721 -78.899 46.508
83856 96.761 -78.895 46.485
83857 96.802 -78.890 46.463
83858 96.840 -78.885 46.443
83859 96.851 -78.883 46.421
83860 96.861 -78.880 46.399
83861 96.871 -78.877 46.379
83862 96.881 -78.874 46.357
83863 96.891 -78.871 46.335
83864 96.911 -78.870 46.313
83865 96.932 -78.866 46.292
83866 96.938 -78.876 46.271
83867 96.925 -78.905 46.251
83868 96.954 -78.952 46.232
83869 96.982 -78.993 46.214
83870 96.995 -78.968 46.198
83871 96.999 -78.962 46.181
83872 96.992 -78.976 46.165
83873 96.986 -78.991 46.147
83874 96.979 -79.006 46.129
83875 96.973 -79.021 46.112
83876 96.987 -79.059 46.096
83877 97.008 -79.105 46.082
83878 97.029 -79.152 46.069
83879 97.049 -79.160 46.056
83880 97.070 -79.156 46.038
83881 97.090 -79.150 46.022
83882 97.110 -79.145 46.008
83883 97.131 -79.140 45.993
83884 97.151 -79.134 45.980
83885 97.171 -79.129 45.966
83886 97.191 -79.125 45.953
83887 97.206 -79.106 45.938
83888 97.218 -79.094 45.924
83889 97.227 -79.099 45.909
83890 97.234 -79.101 45.892
83891 97.240 -79.104 45.875
83892 97.250 -79.110 45.859
83893 97.264 -79.119 45.843
83894 97.278 -79.128 45.827
83895 97.292 -79.138 45.810
83896 97.305 -79.147 45.793
83897 97.319 -79.156 45.774
83898 97.333 -79.165 45.759
83899 97.348 -79.175 45.744
83900 97.363 -79.183 45.729
83901 97.378 -79.192 45.711
83902 97.393 -79.200 45.693
83903 97.408 -79.209 45.674
83904 97.419 -79.208 45.657
83905 97.431 -79.207 45.641
83906 97.452 -79.217 45.624
83907 97.474 -79.228 45.607
83908 97.492 -79.234 45.591
83909 97.507 -79.237 45.575
83910 97.522 -79.240 45.559
83911 97.536 -79.243 45.542
83912 97.551 -79.246 45.526
83913 97.566 -79.249 45.509
83914 97.580 -79.252 45.493
83915 97.597 -79.257 45.476
83916 97.617 -79.266 45.459
83917 97.637 -79.274 45.441
83918 97.648 -79.278 45.425
83919 97.648 -79.278 45.409
83920 97.648 -79.278 45.394
83921 97.648 -79.278 45.380
83922 97.648 -79.277 45.366
83923 97.648 -79.276 45.351
83924 97.648 -79.276 45.338
83925 97.649 -79.278 45.325
83926 97.667 -79.297 45.313
83927 97.685 -79.318 45.303
83928 97.704 -79.338 45.293
83929 97.722 -79.359 45.283
83930 97.740 -79.379 45.272
83931 97.758 -79.400 45.260
83932 97.777 -79.420 45.248
83933 97.795 -79.441 45.237
83934 97.813 -79.461 45.226
83935 97.840 -79.480 45.215
83936 97.854 -79.492 45.204
83937 97.866 -79.507 45.192
83938 97.878 -79.523 45.180
83939 97.890 -79.537 45.169
83940 97.902 -79.549 45.159
83941 97.914 -79.560 45.149
83942 97.925 -79.573 45.140
83943 97.937 -79.584 45.131
83944 97.948 -79.596 45.123
83945 97.962 -79.607 45.114
83946 97.989 -79.617 45.105
83947 98.019 -79.623 45.097
83948 98.056 -79.614 45.090
83949 98.093 -79.605 45.082
83950 98.130 -79.596 45.075
83951 98.167 -79.587 45.068
83952 98.205 -79.578 45.062
83953 98.239 -79.567 45.053
83954 98.272 -79.585 45.045
83955 98.308 -79.610 45.036
83956 98.344 -79.635 45.027
83957 98.379 -79.659 45.017
83958 98.406 -79.666 45.008
83959 98.432 -79.672 44.999
83960 98.458 -79.679 44.988
83961 98.486 -79.689 44.979
83962 98.512 -79.700 44.971
83963 98.539 -79.711 44.963
83964 98.567 -79.720 44.956
83965 98.594 -79.731 44.949
83966 98.621 -79.742 44.942
83967 98.648 -79.751 44.937
83968 98.680 -79.761 44.932
83969 98.698 -79.757 44.928
83970 98.712 -79.750 44.923
83971 98.726 -79.744 44.919
83972 98.741 -79.737 44.915
83973 98.755 -79.730 44.910
83974 98.770 -79.723 44.904
83975 98.786 -79.722 44.898
83976 98.807 -79.730 44.893
83977 98.829 -79.737 44.888
83978 98.851 -79.745 44.882
83979 98.872 -79.753 44.876
83980 98.894 -79.761 44.869
83981 98.916 -79.769 44.863
83982 98.938 -79.777 44.857
83983 98.959 -79.786 44.850
83984 98.981 -79.793 44.843
83985 99.003 -79.802 44.837
83986 99.025 -79.810 44.829
83987 99.049 -79.815 44.821
83988 99.072 -79.829 44.811
83989 99.096 -79.843 44.802
83990 99.121 -79.857 44.795
83991 99.142 -79.872 44.788
83992 99.170 -79.881 44.778
83993 99.194 -79.886 44.768
83994 99.217 -79.891 44.757
83995 99.240 -79.896 44.746
83996 99.264 -79.900 44.736
83997 99.287 -79.905 44.727
83998 99.310 -79.912 44.719
83999 99.334 -79.918 44.711
84000 99.358 -79.924 44.704
84001 99.382 -79.930 44.697
84002 99.406 -79.937 44.691
84003 99.429 -79.943 44.685
84004 99.453 -79.949 44.676
84005 99.477 -79.955 44.668
84006 99.500 -79.962 44.661
84007 99.524 -79.968 44.654
84008 99.548 -79.974 44.646
84009 99.570 -79.980 44.637
84010 99.584 -79.977 44.628
84011 99.582 -80.005 44.618
84012 99.581 -80.034 44.610
84013 99.580 -80.062 44.602
84014 99.579 -80.090 44.595
84015 99.577 -80.118 44.587
84016 99.570 -80.155 44.580
84017 99.561 -80.192 44.572
84018 99.553 -80.230 44.564
84019 99.545 -80.268 44.557
84020 99.547 -80.292 44.549
84021 99.550 -80.314 44.541
84022 99.553 -80.336 44.533
84023 99.556 -80.359 44.525
84024 99.559 -80.381 44.517
84025 99.563 -80.403 44.507
84026 99.587 -80.417 44.498
84027 99.635 -80.422 44.487
84028 99.683 -80.428 44.476
84029 99.731 -80.434 44.466
84030 99.779 -80.439 44.455
84031 99.830 -80.450 44.444
84032 99.881 -80.463 44.432
84033 99.932 -80.476 44.420
84034 99.984 -80.489 44.408
84035 100.035 -80.502 44.396
84036 100.086 -80.515 44.383
84037 100.137 -80.527 44.370
84038 100.189 -80.540 44.358
84039 100.240 -80.553 44.346
84333 93.566 -78.767 53.603
84334 93.576 -78.765 53.595
84335 93.585 -78.763 53.587
84336 93.595 -78.761 53.579
84337 93.605 -78.758 53.573
84338 93.615 -78.755 53.568
84339 93.624 -78.753 53.562
84340 93.633 -78.750 53.558
84341 93.631 -78.740 53.555
84342 93.630 -78.731 53.556
84343 93.628 -78.739 53.559
84344 93.630 -78.757 53.562
84345 93.639 -78.756 53.562
84346 93.650 -78.739 53.559
84347 93.665 -78.727 53.553
84348 93.683 -78.717 53.544
84349 93.700 -78.706 53.533
84350 93.719 -78.689 53.521
84351 93.739 -78.673 53.506
84352 93.758 -78.655 53.491
84353 93.777 -78.639 53.474
84354 93.785 -78.627 53.455
84355 93.789 -78.619 53.436
84356 93.792 -78.611 53.417
84357 93.796 -78.601 53.397
84358 93.799 -78.589 53.377
84359 93.801 -78.575 53.357
84360 93.804 -78.561 53.334
84361 93.807 -78.547 53.310
84362 93.809 -78.532 53.285
84363 93.812 -78.518 53.260
84364 93.837 -78.544 53.236
84365 93.850 -78.532 53.213
84366 93.863 -78.519 53.190
84367 93.875 -78.507 53.166
84368 93.888 -78.494 53.141
84369 93.921 -78.478 53.118
84370 93.965 -78.460 53.097
84371 93.980 -78.462 53.078
84372 93.981 -78.472 53.060
84373 93.984 -78.482 53.044
84374 93.994 -78.488 53.025
84375 94.004 -78.494 53.004
84376 94.015 -78.499 52.984
84377 94.025 -78.505 52.964
84378 94.039 -78.504 52.944
84379 94.054 -78.503 52.923
84380 94.068 -78.501 52.903
84381 94.082 -78.501 52.880
84382 94.097 -78.499 52.855
84383 94.112 -78.498 52.828
84384 94.126 -78.496 52.804
84385 94.133 -78.476 52.782
84386 94.144 -78.448 52.762
84387 94.164 -78.413 52.743
84388 94.183 -78.377 52.725
84389 94.203 -78.341 52.708
84390 94.222 -78.306 52.692
84391 94.249 -78.290 52.676
84392 94.256 -78.380 52.659
84393 94.263 -78.470 52.641
84394 94.271 -78.557 52.623
84395 94.286 -78.564 52.604
84396 94.300 -78.571 52.584
84397 94.316 -78.578 52.565
84398 94.330 -78.584 52.546
84399 94.345 -78.591 52.527
84400 94.360 -78.598 52.508
84401 94.372 -78.605 52.489
84402 94.365 -78.617 52.468
84403 94.357 -78.629 52.444
84404 94.373 -78.617 52.422
84405 94.390 -78.604 52.399
84406 94.397 -78.589 52.375
84407 94.406 -78.574 52.349
84408 94.423 -78.560 52.322
84409 94.441 -78.545 52.292
84410 94.450 -78.545 52.259
84411 94.460 -78.545 52.227
84412 94.475 -78.550 52.194
84413 94.490 -78.554 52.160
84414 94.506 -78.558 52.124
84415 94.522 -78.563 52.088
84416 94.537 -78.567 52.054
84417 94.553 -78.571 52.022
84418 94.568 -78.576 51.991
84419 94.574 -78.577 51.957
84420 94.580 -78.579 51.921
84421 94.591 -78.578 51.882
84422 94.601 -78.577 51.844
84423 94.615 -78.581 51.804
84424 94.628 -78.585 51.763
84425 94.641 -78.588 51.719
84426 94.654 -78.590 51.672
84427 94.668 -78.592 51.623
84428 94.680 -78.594 51.577
84429 94.693 -78.596 51.530
84430 94.706 -78.598 51.485
84431 94.719 -78.600 51.438
84432 94.733 -78.602 51.391
84433 94.754 -78.606 51.345
84434 94.775 -78.611 51.299
84435 94.796 -78.615 51.255
84436 94.817 -78.619 51.211
84437 94.839 -78.623 51.166
84438 94.859 -78.627 51.121
84439 94.878 -78.631 51.076
84440 94.891 -78.636 51.032
84441 94.899 -78.629 50.989
84442 94.913 -78.618 50.945
84443 94.951 -78.617 50.903
84444 94.982 -78.616 50.861
84445 94.987 -78.618 50.818
84446 94.991 -78.620 50.775
84447 94.996 -78.622 50.732
84448 95.000 -78.623 50.688
84449 94.999 -78.624 50.641
84450 94.993 -78.623 50.595
84451 94.987 -78.623 50.550
84452 94.982 -78.622 50.505
84453 94.976 -78.620 50.460
84454 94.971 -78.620 50.415
84455 94.983 -78.604 50.370
84456 94.997 -78.586 50.325
84457 95.012 -78.568 50.281
84458 95.026 -78.550 50.238
84459 95.041 -78.535 50.194
84460 95.062 -78.540 50.151
84461 95.082 -78.545 50.109
84462 95.100 -78.550 50.067
84463 95.111 -78.554 50.025
84464 95.121 -78.557 49.984
84465 95.132 -78.561 49.942
84466 95.143 -78.565 49.903
84467 95.154 -78.569 49.865
84468 95.164 -78.572 49.827
84469 95.175 -78.576 49.790
84470 95.186 -78.579 49.753
84471 95.196 -78.583 49.716
84472 95.207 -78.587 49.679
84473 95.235 -78.601 49.641
84474 95.268 -78.617 49.602
84475 95.301 -78.635 49.564
84476 95.333 -78.651 49.526
84477 95.333 -78.664 49.487
84478 95.346 -78.686 49.448
84479 95.360 -78.706 49.410
84480 95.375 -78.727 49.374
84481 95.388 -78.749 49.340
84482 95.402 -78.770 49.306
84483 95.416 -78.791 49.271
84484 95.430 -78.811 49.234
84485 95.438 -78.825 49.196
84486 95.429 -78.824 49.156
84487 95.416 -78.818 49.115
84488 95.420 -78.807 49.074
84489 95.424 -78.796 49.033
84490 95.428 -78.784 48.992
84491 95.432 -78.773 48.951
84492 95.436 -78.761 48.908
84493 95.440 -78.750 48.864
84494 95.444 -78.739 48.819
84495 95.453 -78.731 48.775
84496 95.476 -78.733 48.731
84497 95.499 -78.735 48.688
84498 95.523 -78.736 48.647
84499 95.546 -78.739 48.606
84500 95.570 -78.741 48.564
84501 95.593 -78.742 48.523
84502 95.609 -78.749 48.483
84503 95.615 -78.759 48.444
84504 95.621 -78.769 48.406
84505 95.626 -78.780 48.367
84506 95.633 -78.789 48.330
84507 95.645 -78.794 48.294
84508 95.657 -78.798 48.260
84509 95.669 -78.802 48.225
84510 95.690 -78.811 48.189
84511 95.719 -78.822 48.150
84512 95.748 -78.834 48.111
84513 95.775 -78.843 48.070
84514 95.799 -78.849 48.026
84515 95.812 -78.863 47.984
84516 95.828 -78.865 47.942
84517 95.845 -78.866 47.900
84518 95.863 -78.867 47.858
84519 95.879 -78.870 47.815
84520 95.894 -78.875 47.772
84521 95.901 -78.881 47.729
84522 95.923 -78.879 47.686
84523 95.948 -78.876 47.645
84524 95.972 -78.873 47.603
84525 95.996 -78.869 47.562
84526 96.020 -78.866 47.520
84527 96.044 -78.863 47.478
84528 96.068 -78.860 47.436
84529 96.093 -78.856 47.395
84530 96.117 -78.853 47.355
84531 96.137 -78.861 47.317
84532 96.155 -78.879 47.278
84533 96.172 -78.873 47.242
84534 96.188 -78.861 47.205
84535 96.205 -78.849 47.167
84536 96.227 -78.838 47.130
84537 96.252 -78.829 47.094
84538 96.277 -78.820 47.060
84539 96.302 -78.811 47.028
84540 96.327 -78.802 46.997
84541 96.336 -78.803 46.965
84542 96.337 -78.810 46.930
84543 96.346 -78.824 46.897
84544 96.358 -78.843 46.864
84545 96.369 -78.860 46.832
84546 96.380 -78.877 46.801
84547 96.391 -78.893 46.771
84548 96.401 -78.895 46.740
84549 96.410 -78.898 46.710
84550 96.449 -78.898 46.681
84551 96.494 -78.898 46.653
84552 96.539 -78.898 46.626
84553 96.585 -78.898 46.598
84554 96.629 -78.897 46.572
84555 96.670 -78.893 46.546
84556 96.711 -78.887 46.520
84557 96.751 -78.883 46.496
84558 96.791 -78.878 46.474
84559 96.832 -78.873 46.454
84560 96.848 -78.870 46.432
84561 96.858 -78.867 46.410
84562 96.868 -78.865 46.387
84563 96.878 -78.861 46.366
84564 96.890 -78.862 46.345
84565 96.911 -78.865 46.324
84566 96.932 -78.861 46.304
84567 96.940 -78.869 46.284
84568 96.930 -78.900 46.265
84569 96.956 -78.957 46.246
84570 96.971 -78.996 46.228
84571 96.980 -78.984 46.212
84572 96.993 -78.959 46.195
84573 96.989 -78.968 46.178
84574 96.982 -78.984 46.162
84575 96.976 -78.998 46.146
84576 96.970 -79.013 46.129
84577 96.984 -79.053 46.113
84578 97.006 -79.099 46.098
84579 97.027 -79.136 46.084
84580 97.047 -79.131 46.069
84581 97.067 -79.125 46.054
84582 97.087 -79.120 46.038
84583 97.108 -79.115 46.023
84584 97.128 -79.110 46.009
84585 97.148 -79.105 45.996
84586 97.168 -79.100 45.983
84587 97.188 -79.094 45.971
84588 97.209 -79.089 45.957
84589 97.223 -79.093 45.942
84590 97.229 -79.096 45.926
84591 97.235 -79.099 45.909
84592 97.242 -79.102 45.892
84593 97.251 -79.107 45.875
84594 97.265 -79.117 45.858
84595 97.280 -79.125 45.839
84596 97.295 -79.134 45.821
84597 97.310 -79.143 45.803
84598 97.325 -79.152 45.783
84599 97.341 -79.160 45.766
84600 97.356 -79.169 45.750
84601 97.371 -79.178 45.734
84602 97.385 -79.187 45.716
84603 97.400 -79.193 45.697
84604 97.411 -79.192 45.678
84605 97.423 -79.191 45.660
84606 97.434 -79.190 45.644
84607 97.453 -79.198 45.628
84608 97.475 -79.209 45.612
84609 97.498 -79.220 45.595
84610 97.520 -79.231 45.578
84611 97.535 -79.235 45.562
84612 97.550 -79.238 45.546
84613 97.565 -79.241 45.530
84614 97.580 -79.244 45.514
84615 97.594 -79.247 45.499
84616 97.613 -79.254 45.482
84617 97.633 -79.263 45.465
84618 97.640 -79.266 45.449
84619 97.640 -79.265 45.434
84620 97.640 -79.264 45.419
84621 97.639 -79.264 45.405
84622 97.639 -79.263 45.392
84623 97.639 -79.263 45.377
84624 97.639 -79.263 45.363
84625 97.639 -79.262 45.349
84626 97.641 -79.265 45.335
84627 97.659 -79.285 45.325
84628 97.678 -79.305 45.315
84629 97.696 -79.325 45.303
84630 97.714 -79.346 45.293
84631 97.733 -79.366 45.281
84632 97.751 -79.387 45.268
84633 97.769 -79.407 45.255
84634 97.788 -79.428 45.244
84635 97.816 -79.450 45.233
84636 97.840 -79.463 45.223
84637 97.855 -79.473 45.212
84638 97.867 -79.488 45.201
84639 97.879 -79.504 45.190
84640 97.892 -79.519 45.178
84641 97.904 -79.534 45.168
84642 97.916 -79.549 45.158
84643 97.929 -79.564 45.150
84644 97.941 -79.579 45.140
84645 97.952 -79.591 45.131
84646 97.964 -79.602 45.123
84647 97.993 -79.608 45.115
84648 98.030 -79.599 45.107
84649 98.067 -79.590 45.099
84650 98.104 -79.581 45.092
84651 98.142 -79.572 45.085
84652 98.178 -79.563 45.077
84653 98.216 -79.554 45.070
84654 98.249 -79.542 45.062
84655 98.282 -79.560 45.054
84656 98.318 -79.585 45.046
84657 98.354 -79.610 45.037
84658 98.389 -79.635 45.027
84659 98.422 -79.653 45.018
84660 98.448 -79.659 45.009
84661 98.475 -79.668 45.000
84662 98.502 -79.679 44.991
84663 98.529 -79.689 44.983
84664 98.557 -79.699 44.976
84665 98.584 -79.709 44.968
84666 98.611 -79.720 44.961
84667 98.638 -79.730 44.954
84668 98.665 -79.740 44.949
84669 98.692 -79.748 44.944
84670 98.706 -79.741 44.940
84671 98.720 -79.734 44.936
84672 98.734 -79.727 44.932
84673 98.749 -79.720 44.928
84674 98.763 -79.714 44.923
84675 98.778 -79.707 44.919
84676 98.792 -79.701 44.913
84677 98.813 -79.708 44.908
84678 98.835 -79.716 44.903
84679 98.856 -79.723 44.897
84680 98.878 -79.732 44.891
84681 98.900 -79.740 44.884
84682 98.922 -79.748 44.879
84683 98.943 -79.756 44.872
84684 98.966 -79.764 44.865
84685 98.987 -79.772 44.858
84686 99.008 -79.781 44.851
84687 99.027 -79.792 44.843
84688 99.058 -79.799 44.834
84689 99.082 -79.813 44.824
84690 99.106 -79.827 44.814
84691 99.130 -79.841 44.806
84692 99.148 -79.855 44.797
84693 99.172 -79.861 44.786
84694 99.196 -79.867 44.774
84695 99.219 -79.874 44.763
84696 99.243 -79.880 44.752
84697 99.267 -79.886 44.743
84698 99.291 -79.892 44.735
84699 99.314 -79.899 44.727
84700 99.338 -79.905 44.719
84701 99.362 -79.911 44.711
84702 99.385 -79.917 44.703
84703 99.409 -79.924 44.697
84704 99.433 -79.930 44.690
84705 99.457 -79.936 44.681
84706 99.481 -79.942 44.673
84707 99.504 -79.949 44.665
84708 99.528 -79.955 44.657
84709 99.552 -79.961 44.648
84710 99.567 -79.956 44.641
84711 99.579 -79.957 44.632
84712 99.578 -79.985 44.622
84713 99.576 -80.013 44.614
84714 99.575 -80.041 44.605
84715 99.574 -80.069 44.597
84716 99.573 -80.097 44.590
84717 99.571 -80.125 44.583
84718 99.570 -80.153 44.575
84719 99.569 -80.181 44.567
84720 99.566 -80.212 44.560
84721 99.558 -80.250 44.553
84722 99.549 -80.287 44.545
84723 99.550 -80.313 44.538
84724 99.553 -80.335 44.530
84725 99.556 -80.358 44.521
84726 99.559 -80.380 44.512
84727 99.595 -80.390 44.502
84728 99.644 -80.395 44.492
84729 99.692 -80.401 44.481
84730 99.742 -80.413 44.471
84731 99.793 -80.425 44.460
84732 99.845 -80.438 44.449
84733 99.896 -80.450 44.437
84734 99.947 -80.463 44.425
84735 99.999 -80.476 44.413
84736 100.050 -80.489 44.401
84737 100.101 -80.502 44.387
84738 100.152 -80.515 44.375
84739 100.198 -80.530 44.363
84740 100.243 -80.547 44.349
85034 93.566 -78.757 53.610
85035 93.576 -78.755 53.602
85036 93.586 -78.752 53.594
85037 93.595 -78.749 53.586
85038 93.605 -78.747 53.578
85039 93.615 -78.745 53.573
85040 93.625 -78.742 53.567
85041 93.629 -78.736 53.562
85042 93.628 -78.727 53.559
85043 93.627 -78.718 53.559
85044 93.625 -78.709 53.561
85045 93.624 -78.749 53.565
85046 93.635 -78.732 53.564
85047 93.646 -78.715 53.560
85048 93.663 -78.705 53.555
85049 93.681 -78.694 53.546
85050 93.698 -78.683 53.535
85051 93.716 -78.673 53.522
85052 93.734 -78.658 53.508
85053 93.754 -78.642 53.492
85054 93.773 -78.624 53.475
85055 93.780 -78.614 53.456
85056 93.784 -78.605 53.437
85057 93.787 -78.596 53.418
85058 93.790 -78.582 53.398
85059 93.792 -78.568 53.377
85060 93.795 -78.554 53.355
85061 93.798 -78.540 53.332
85062 93.800 -78.526 53.308
85063 93.803 -78.511 53.283
85064 93.806 -78.497 53.258
85065 93.827 -78.517 53.233
85066 93.845 -78.520 53.209
85067 93.858 -78.508 53.186
85068 93.870 -78.495 53.162
85069 93.889 -78.482 53.137
85070 93.933 -78.464 53.117
85071 93.977 -78.447 53.098
85072 93.990 -78.450 53.081
85073 93.991 -78.460 53.065
85074 93.992 -78.470 53.048
85075 93.998 -78.479 53.030
85076 94.009 -78.484 53.009
85077 94.020 -78.489 52.988
85078 94.035 -78.487 52.966
85079 94.049 -78.485 52.945
85080 94.064 -78.484 52.923
85081 94.078 -78.482 52.902
85082 94.093 -78.481 52.879
85083 94.108 -78.479 52.853
85084 94.122 -78.478 52.827
85085 94.137 -78.476 52.805
85086 94.150 -78.474 52.785
85087 94.157 -78.449 52.767
85088 94.177 -78.413 52.749
85089 94.196 -78.379 52.731
85090 94.215 -78.343 52.714
85091 94.233 -78.310 52.697
85092 94.236 -78.304 52.680
85093 94.237 -78.360 52.662
85094 94.242 -78.438 52.643
85095 94.249 -78.528 52.626
85096 94.261 -78.566 52.608
85097 94.276 -78.573 52.589
85098 94.290 -78.579 52.571
85099 94.306 -78.586 52.552
85100 94.321 -78.592 52.534
85101 94.335 -78.599 52.515
85102 94.350 -78.606 52.497
85103 94.365 -78.612 52.478
85104 94.363 -78.623 52.455
85105 94.376 -78.614 52.433
85106 94.385 -78.600 52.410
85107 94.402 -78.586 52.385
85108 94.420 -78.570 52.360
85109 94.437 -78.556 52.332
85110 94.453 -78.545 52.302
85111 94.462 -78.545 52.269
85112 94.471 -78.545 52.236
85113 94.480 -78.546 52.202
85114 94.493 -78.548 52.167
85115 94.508 -78.552 52.132
85116 94.524 -78.557 52.098
85117 94.540 -78.561 52.064
85118 94.555 -78.565 52.032
85119 94.571 -78.570 52.002
85120 94.579 -78.572 51.969
85121 94.586 -78.574 51.932
85122 94.596 -78.573 51.895
85123 94.607 -78.572 51.856
85124 94.617 -78.571 51.816
85125 94.631 -78.574 51.774
85126 94.644 -78.578 51.730
85127 94.657 -78.582 51.684
85128 94.671 -78.586 51.635
85129 94.684 -78.588 51.587
85130 94.697 -78.590 51.538
85131 94.710 -78.592 51.489
85132 94.723 -78.594 51.442
85133 94.736 -78.596 51.393
85134 94.757 -78.600 51.345
85135 94.778 -78.604 51.298
85136 94.800 -78.608 51.252
85137 94.821 -78.613 51.206
85138 94.842 -78.617 51.160
85139 94.863 -78.621 51.114
85140 94.885 -78.626 51.069
85141 94.901 -78.630 51.025
85142 94.918 -78.622 50.982
85143 94.939 -78.623 50.937
85144 94.944 -78.624 50.894
85145 94.949 -78.626 50.852
85146 94.953 -78.628 50.809
85147 94.958 -78.630 50.767
85148 94.962 -78.632 50.725
85149 94.964 -78.633 50.684
85150 94.958 -78.632 50.639
85151 94.952 -78.631 50.594
85152 94.947 -78.630 50.549
85153 94.945 -78.626 50.505
85154 94.959 -78.608 50.461
85155 94.974 -78.591 50.416
85156 94.988 -78.573 50.371
85157 95.002 -78.554 50.327
85158 95.017 -78.537 50.283
85159 95.031 -78.519 50.238
85160 95.049 -78.515 50.195
85161 95.069 -78.520 50.152
85162 95.090 -78.526 50.109
85163 95.111 -78.530 50.067
85164 95.125 -78.535 50.025
85165 95.135 -78.539 49.983
85166 95.146 -78.542 49.942
85167 95.157 -78.545 49.902
85168 95.168 -78.549 49.864
85169 95.178 -78.553 49.827
85170 95.189 -78.557 49.789
85171 95.199 -78.561 49.752
85172 95.210 -78.564 49.715
85173 95.229 -78.573 49.677
85174 95.261 -78.589 49.639
85175 95.295 -78.606 49.600
85176 95.328 -78.623 49.562
85177 95.360 -78.640 49.524
85178 95.381 -78.654 49.487
85179 95.390 -78.672 49.449
85180 95.404 -78.693 49.410
85181 95.418 -78.714 49.376
85182 95.432 -78.736 49.342
85183 95.446 -78.756 49.308
85184 95.460 -78.777 49.273
85185 95.471 -78.796 49.238
85186 95.465 -78.795 49.201
85187 95.460 -78.794 49.163
85188 95.450 -78.792 49.122
85189 95.432 -78.786 49.082
85190 95.415 -78.782 49.040
85191 95.418 -78.771 48.999
85192 95.422 -78.759 48.957
85193 95.426 -78.748 48.913
85194 95.430 -78.737 48.868
85195 95.434 -78.726 48.824
85196 95.443 -78.717 48.780
85197 95.466 -78.719 48.736
85198 95.489 -78.721 48.694
85199 95.513 -78.724 48.653
85200 95.536 -78.725 48.612
85201 95.559 -78.727 48.572
85202 95.583 -78.729 48.531
85203 95.606 -78.731 48.492
85204 95.617 -78.739 48.453
85205 95.623 -78.749 48.414
85206 95.629 -78.760 48.376
85207 95.641 -78.764 48.339
85208 95.653 -78.768 48.303
85209 95.665 -78.773 48.268
85210 95.677 -78.777 48.233
85211 95.695 -78.784 48.197
85212 95.725 -78.796 48.159
85213 95.753 -78.808 48.120
85214 95.782 -78.820 48.079
85215 95.806 -78.821 48.038
85216 95.823 -78.822 47.998
85217 95.841 -78.823 47.956
85218 95.858 -78.824 47.914
85219 95.875 -78.826 47.872
85220 95.890 -78.831 47.829
85221 95.904 -78.836 47.787
85222 95.911 -78.843 47.744
85223 95.926 -78.844 47.702
85224 95.950 -78.841 47.659
85225 95.974 -78.838 47.616
85226 95.999 -78.835 47.573
85227 96.023 -78.831 47.531
85228 96.047 -78.828 47.489
85229 96.071 -78.825 47.447
85230 96.095 -78.822 47.407
85231 96.119 -78.821 47.367
85232 96.136 -78.839 47.329
85233 96.154 -78.856 47.291
85234 96.171 -78.862 47.254
85235 96.188 -78.850 47.217
85236 96.204 -78.838 47.179
85237 96.221 -78.825 47.142
85238 96.237 -78.813 47.106
85239 96.258 -78.802 47.071
85240 96.283 -78.793 47.039
85241 96.308 -78.784 47.007
85242 96.320 -78.783 46.974
85243 96.325 -78.794 46.939
85244 96.337 -78.812 46.906
85245 96.348 -78.830 46.874
85246 96.360 -78.847 46.842
85247 96.371 -78.865 46.811
85248 96.382 -78.883 46.781
85249 96.392 -78.887 46.750
85250 96.401 -78.890 46.720
85251 96.443 -78.890 46.691
85252 96.488 -78.890 46.664
85253 96.533 -78.890 46.637
85254 96.579 -78.890 46.610
85255 96.619 -78.885 46.584
85256 96.660 -78.880 46.558
85257 96.700 -78.875 46.531
85258 96.741 -78.871 46.508
85259 96.781 -78.866 46.486
85260 96.822 -78.861 46.466
85261 96.844 -78.857 46.444
85262 96.855 -78.855 46.421
85263 96.865 -78.852 46.398
85264 96.875 -78.849 46.378
85265 96.890 -78.853 46.357
85266 96.910 -78.861 46.337
85267 96.931 -78.857 46.317
85268 96.941 -78.862 46.297
85269 96.934 -78.896 46.278
85270 96.960 -78.952 46.260
85271 96.971 -78.978 46.242
85272 96.966 -78.993 46.226
85273 96.977 -78.976 46.210
85274 96.985 -78.961 46.193
85275 96.979 -78.976 46.177
85276 96.973 -78.990 46.161
85277 96.966 -79.006 46.145
85278 96.982 -79.046 46.129
85279 97.004 -79.093 46.114
85280 97.024 -79.106 46.099
85281 97.044 -79.101 46.084
85282 97.065 -79.096 46.069
85283 97.085 -79.090 46.053
85284 97.105 -79.085 46.037
85285 97.125 -79.080 46.023
85286 97.146 -79.075 46.011
85287 97.166 -79.070 45.999
85288 97.189 -79.073 45.986
85289 97.213 -79.078 45.972
85290 97.230 -79.087 45.956
85291 97.237 -79.091 45.938
85292 97.243 -79.095 45.922
85293 97.248 -79.098 45.906
85294 97.257 -79.103 45.889
85295 97.272 -79.112 45.870
85296 97.287 -79.120 45.849
85297 97.302 -79.129 45.829
85298 97.317 -79.137 45.810
85299 97.332 -79.147 45.791
85300 97.348 -79.155 45.772
85301 97.363 -79.164 45.755
85302 97.378 -79.173 45.738
85303 97.391 -79.178 45.720
85304 97.403 -79.177 45.702
85305 97.414 -79.176 45.683
85306 97.426 -79.175 45.666
85307 97.437 -79.175 45.650
85308 97.454 -79.179 45.634
85309 97.476 -79.190 45.618
85310 97.498 -79.201 45.602
85311 97.520 -79.212 45.585
85312 97.542 -79.223 45.568
85313 97.564 -79.233 45.552
85314 97.579 -79.236 45.536
85315 97.593 -79.239 45.521
85316 97.608 -79.243 45.506
85317 97.629 -79.251 45.489
85318 97.631 -79.252 45.473
85319 97.631 -79.251 45.458
85320 97.631 -79.250 45.444
85321 97.631 -79.250 45.430
85322 97.631 -79.250 45.417
85323 97.631 -79.250 45.404
85324 97.630 -79.249 45.389
85325 97.630 -79.249 45.375
85326 97.630 -79.248 45.360
85327 97.634 -79.252 45.346
85328 97.652 -79.272 45.335
85329 97.670 -79.292 45.325
85330 97.688 -79.313 45.312
85331 97.707 -79.333 45.300
85332 97.725 -79.353 45.288
85333 97.743 -79.374 45.276
85334 97.761 -79.394 45.263
85335 97.788 -79.416 45.252
85336 97.823 -79.436 45.241
85337 97.840 -79.444 45.231
85338 97.856 -79.454 45.220
85339 97.868 -79.469 45.210
85340 97.880 -79.485 45.200
85341 97.892 -79.500 45.189
85342 97.905 -79.515 45.179
85343 97.917 -79.530 45.170
85344 97.930 -79.545 45.160
85345 97.942 -79.561 45.152
85346 97.954 -79.576 45.143
85347 97.967 -79.591 45.134
85348 97.995 -79.595 45.125
85349 98.041 -79.575 45.117
85350 98.078 -79.566 45.109
85351 98.115 -79.557 45.102
85352 98.153 -79.548 45.095
85353 98.190 -79.538 45.087
85354 98.227 -79.529 45.080
85355 98.260 -79.517 45.071
85356 98.293 -79.535 45.063
85357 98.329 -79.560 45.055
85358 98.364 -79.585 45.047
85359 98.400 -79.610 45.038
85360 98.436 -79.634 45.029
85361 98.465 -79.646 45.020
85362 98.492 -79.657 45.011
85363 98.519 -79.667 45.003
85364 98.546 -79.677 44.996
85365 98.573 -79.688 44.988
85366 98.598 -79.697 44.980
85367 98.624 -79.706 44.973
85368 98.649 -79.715 44.966
85369 98.675 -79.724 44.960
85370 98.690 -79.738 44.956
85371 98.707 -79.729 44.952
85372 98.724 -79.720 44.948
85373 98.741 -79.712 44.944
85374 98.757 -79.704 44.940
85375 98.771 -79.698 44.937
85376 98.786 -79.691 44.933
85377 98.800 -79.684 44.929
85378 98.818 -79.686 44.923
85379 98.841 -79.695 44.919
85380 98.862 -79.702 44.913
85381 98.884 -79.711 44.907
85382 98.905 -79.720 44.900
85383 98.924 -79.731 44.894
85384 98.944 -79.742 44.887
85385 98.963 -79.754 44.879
85386 98.982 -79.765 44.870
85387 99.001 -79.776 44.862
85388 99.037 -79.780 44.853
85389 99.076 -79.783 44.844
85390 99.100 -79.796 44.834
85391 99.125 -79.811 44.822
85392 99.148 -79.824 44.810
85393 99.157 -79.839 44.800
85394 99.176 -79.847 44.789
85395 99.200 -79.853 44.778
85396 99.223 -79.859 44.768
85397 99.247 -79.866 44.758
85398 99.271 -79.872 44.749
85399 99.294 -79.878 44.742
85400 99.319 -79.884 44.734
85401 99.342 -79.891 44.726
85402 99.366 -79.897 44.717
85403 99.390 -79.903 44.709
85404 99.413 -79.909 44.701
85405 99.437 -79.916 44.694
85406 99.461 -79.922 44.685
85407 99.484 -79.928 44.677
85408 99.508 -79.935 44.669
85409 99.532 -79.941 44.660
85410 99.549 -79.938 44.651
85411 99.564 -79.932 44.644
85412 99.574 -79.936 44.636
85413 99.573 -79.965 44.626
85414 99.572 -79.993 44.617
85415 99.571 -80.021 44.608
85416 99.569 -80.049 44.600
85417 99.568 -80.077 44.593
85418 99.567 -80.105 44.585
85419 99.566 -80.133 44.577
85420 99.564 -80.161 44.571
85421 99.563 -80.189 44.563
85422 99.562 -80.216 44.555
85423 99.561 -80.244 44.549
85424 99.559 -80.272 44.541
85425 99.554 -80.306 44.533
85426 99.553 -80.335 44.525
85427 99.556 -80.357 44.517
85428 99.604 -80.363 44.507
85429 99.655 -80.374 44.497
85430 99.706 -80.387 44.487
85431 99.757 -80.400 44.476
85432 99.808 -80.413 44.465
85433 99.860 -80.425 44.453
85434 99.911 -80.438 44.441
85435 99.963 -80.451 44.430
85436 100.014 -80.463 44.417
85437 100.065 -80.476 44.405
85438 100.112 -80.492 44.392
85439 100.156 -80.508 44.379
85440 100.201 -80.525 44.367
85441 100.245 -80.541 44.354
85735 93.567 -78.746 53.617
85736 93.576 -78.743 53.609
85737 93.586 -78.741 53.600
85738 93.596 -78.739 53.592
85739 93.606 -78.736 53.585
85740 93.615 -78.733 53.578
85741 93.625 -78.731 53.571
85742 93.626 -78.724 53.565
85743 93.625 -78.714 53.563
85744 93.623 -78.705 53.561
85745 93.622 -78.695 53.562
85746 93.620 -78.721 53.565
85747 93.630 -78.707 53.563
85748 93.644 -78.692 53.560
85749 93.661 -78.682 53.554
85750 93.679 -78.671 53.545
85751 93.697 -78.661 53.534
85752 93.714 -78.651 53.522
85753 93.731 -78.640 53.508
85754 93.750 -78.627 53.491
85755 93.769 -78.611 53.474
85756 93.775 -78.600 53.454
85757 93.778 -78.589 53.434
85758 93.781 -78.576 53.414
85759 93.784 -78.561 53.393
85760 93.786 -78.547 53.372
85761 93.789 -78.532 53.350
85762 93.792 -78.519 53.326
85763 93.794 -78.504 53.302
85764 93.797 -78.490 53.277
85765 93.800 -78.476 53.252
85766 93.815 -78.485 53.227
85767 93.840 -78.510 53.203
85768 93.853 -78.497 53.180
85769 93.865 -78.485 53.158
85770 93.902 -78.468 53.135
85771 93.945 -78.451 53.116
85772 93.989 -78.433 53.098
85773 94.000 -78.438 53.081
85774 94.001 -78.448 53.066
85775 94.002 -78.458 53.051
85776 94.004 -78.469 53.033
85777 94.015 -78.471 53.012
85778 94.030 -78.470 52.989
85779 94.045 -78.468 52.966
85780 94.059 -78.467 52.943
85781 94.074 -78.465 52.922
85782 94.088 -78.463 52.900
85783 94.103 -78.462 52.876
85784 94.118 -78.460 52.850
85785 94.132 -78.459 52.827
85786 94.147 -78.457 52.808
85787 94.161 -78.456 52.790
85788 94.173 -78.448 52.772
85789 94.189 -78.415 52.755
85790 94.208 -78.379 52.737
85791 94.220 -78.357 52.718
85792 94.222 -78.351 52.701
85793 94.225 -78.344 52.683
85794 94.227 -78.384 52.665
85795 94.228 -78.445 52.646
85796 94.229 -78.507 52.629
85797 94.236 -78.567 52.612
85798 94.251 -78.574 52.594
85799 94.266 -78.580 52.576
85800 94.281 -78.587 52.558
85801 94.296 -78.594 52.540
85802 94.312 -78.599 52.522
85803 94.328 -78.605 52.505
85804 94.345 -78.611 52.486
85805 94.361 -78.617 52.464
85806 94.381 -78.611 52.442
85807 94.399 -78.596 52.419
85808 94.416 -78.582 52.395
85809 94.433 -78.567 52.370
85810 94.451 -78.552 52.342
85811 94.464 -78.545 52.312
85812 94.473 -78.545 52.279
85813 94.483 -78.546 52.245
85814 94.492 -78.546 52.211
85815 94.501 -78.546 52.176
85816 94.510 -78.546 52.141
85817 94.526 -78.551 52.108
85818 94.542 -78.555 52.075
85819 94.557 -78.559 52.042
85820 94.573 -78.564 52.010
85821 94.585 -78.567 51.976
85822 94.591 -78.569 51.941
85823 94.602 -78.567 51.904
85824 94.612 -78.567 51.866
85825 94.622 -78.565 51.826
85826 94.633 -78.564 51.784
85827 94.647 -78.568 51.739
85828 94.660 -78.572 51.693
85829 94.674 -78.576 51.644
85830 94.687 -78.579 51.594
85831 94.700 -78.582 51.545
85832 94.713 -78.586 51.496
85833 94.726 -78.588 51.446
85834 94.740 -78.589 51.396
85835 94.761 -78.594 51.347
85836 94.782 -78.598 51.299
85837 94.803 -78.602 51.251
85838 94.824 -78.607 51.204
85839 94.845 -78.611 51.157
85840 94.867 -78.615 51.110
85841 94.888 -78.619 51.065
85842 94.904 -78.622 51.020
85843 94.901 -78.630 50.976
85844 94.906 -78.633 50.931
85845 94.910 -78.634 50.888
85846 94.915 -78.636 50.845
85847 94.919 -78.638 50.803
85848 94.924 -78.640 50.761
85849 94.928 -78.642 50.720
85850 94.922 -78.641 50.679
85851 94.917 -78.640 50.636
85852 94.921 -78.631 50.592
85853 94.935 -78.613 50.549
85854 94.950 -78.595 50.506
85855 94.964 -78.577 50.462
85856 94.979 -78.559 50.417
85857 94.993 -78.542 50.372
85858 95.007 -78.523 50.328
85859 95.022 -78.506 50.284
85860 95.037 -78.490 50.239
85861 95.057 -78.495 50.195
85862 95.077 -78.500 50.152
85863 95.098 -78.505 50.109
85864 95.118 -78.510 50.065
85865 95.139 -78.516 50.024
85866 95.149 -78.519 49.982
85867 95.160 -78.523 49.941
85868 95.171 -78.526 49.902
85869 95.181 -78.530 49.864
85870 95.192 -78.534 49.826
85871 95.203 -78.538 49.788
85872 95.213 -78.542 49.751
85873 95.224 -78.545 49.713
85874 95.255 -78.561 49.674
85875 95.288 -78.577 49.637
85876 95.321 -78.595 49.599
85877 95.354 -78.611 49.561
85878 95.387 -78.628 49.523
85879 95.420 -78.645 49.486
85880 95.433 -78.659 49.449
85881 95.448 -78.680 49.412
85882 95.462 -78.701 49.377
85883 95.475 -78.722 49.343
85884 95.489 -78.743 49.309
85885 95.503 -78.764 49.276
85886 95.498 -78.764 49.242
85887 95.493 -78.764 49.206
85888 95.488 -78.764 49.169
85889 95.483 -78.763 49.129
85890 95.469 -78.759 49.089
85891 95.452 -78.755 49.047
85892 95.435 -78.749 49.006
85893 95.418 -78.745 48.963
85894 95.416 -78.735 48.918
85895 95.420 -78.724 48.872
85896 95.424 -78.712 48.828
85897 95.432 -78.704 48.785
85898 95.456 -78.706 48.742
85899 95.479 -78.708 48.700
85900 95.502 -78.710 48.659
85901 95.526 -78.711 48.619
85902 95.549 -78.714 48.580
85903 95.572 -78.716 48.540
85904 95.596 -78.717 48.501
85905 95.619 -78.720 48.462
85906 95.625 -78.730 48.423
85907 95.637 -78.735 48.386
85908 95.649 -78.739 48.350
85909 95.661 -78.743 48.313
85910 95.673 -78.748 48.277
85911 95.685 -78.752 48.241
85912 95.701 -78.758 48.204
85913 95.730 -78.770 48.167
85914 95.759 -78.782 48.127
85915 95.788 -78.786 48.087
85916 95.818 -78.780 48.047
85917 95.836 -78.780 48.007
85918 95.853 -78.781 47.967
85919 95.870 -78.782 47.925
85920 95.886 -78.787 47.883
85921 95.901 -78.793 47.841
85922 95.914 -78.798 47.799
85923 95.921 -78.803 47.757
85924 95.930 -78.809 47.715
85925 95.953 -78.806 47.672
85926 95.977 -78.803 47.629
85927 96.002 -78.800 47.585
85928 96.026 -78.797 47.544
85929 96.050 -78.793 47.502
85930 96.074 -78.790 47.460
85931 96.098 -78.787 47.420
85932 96.117 -78.799 47.380
85933 96.135 -78.816 47.341
85934 96.153 -78.834 47.303
85935 96.171 -78.852 47.266
85936 96.187 -78.839 47.229
85937 96.203 -78.827 47.192
85938 96.220 -78.815 47.155
85939 96.237 -78.802 47.118
85940 96.253 -78.790 47.083
85941 96.269 -78.777 47.049
85942 96.289 -78.767 47.017
85943 96.305 -78.764 46.984
85944 96.316 -78.782 46.950
85945 96.327 -78.799 46.917
85946 96.339 -78.818 46.885
85947 96.350 -78.835 46.854
85948 96.361 -78.852 46.823
85949 96.373 -78.870 46.792
85950 96.383 -78.879 46.761
85951 96.393 -78.882 46.731
85952 96.437 -78.882 46.703
85953 96.482 -78.882 46.676
85954 96.527 -78.881 46.649
85955 96.569 -78.877 46.623
85956 96.609 -78.873 46.597
85957 96.650 -78.868 46.570
85958 96.690 -78.863 46.542
85959 96.731 -78.858 46.520
85960 96.771 -78.853 46.499
85961 96.812 -78.849 46.478
85962 96.841 -78.845 46.456
85963 96.851 -78.842 46.433
85964 96.861 -78.839 46.411
85965 96.871 -78.836 46.391
85966 96.889 -78.844 46.371
85967 96.909 -78.855 46.351
85968 96.930 -78.852 46.332
85969 96.943 -78.855 46.311
85970 96.938 -78.890 46.292
85971 96.965 -78.947 46.273
85972 96.971 -78.959 46.256
85973 96.958 -78.977 46.240
85974 96.963 -78.988 46.224
85975 96.975 -78.968 46.209
85976 96.975 -78.968 46.193
85977 96.969 -78.983 46.177
85978 96.963 -78.998 46.161
85979 96.980 -79.040 46.144
85980 97.001 -79.081 46.130
85981 97.022 -79.076 46.115
85982 97.042 -79.071 46.101
85983 97.062 -79.066 46.083
85984 97.082 -79.061 46.066
85985 97.102 -79.056 46.050
85986 97.122 -79.050 46.035
85987 97.146 -79.053 46.025
85988 97.169 -79.057 46.012
85989 97.193 -79.062 45.996
85990 97.216 -79.066 45.982
85991 97.232 -79.078 45.965
85992 97.246 -79.087 45.947
85993 97.251 -79.090 45.931
85994 97.257 -79.093 45.915
85995 97.264 -79.097 45.897
85996 97.280 -79.106 45.877
85997 97.295 -79.115 45.856
85998 97.310 -79.124 45.836
85999 97.325 -79.132 45.817
86000 97.340 -79.141 45.798
86001 97.355 -79.150 45.781
86002 97.370 -79.159 45.763
86003 97.383 -79.162 45.745
86004 97.394 -79.161 45.727
86005 97.406 -79.160 45.709
86006 97.417 -79.159 45.691
86007 97.429 -79.159 45.675
86008 97.440 -79.158 45.659
86009 97.454 -79.160 45.644
86010 97.476 -79.172 45.628
86011 97.499 -79.182 45.612
86012 97.521 -79.194 45.595
86013 97.543 -79.204 45.578
86014 97.566 -79.215 45.562
86015 97.588 -79.226 45.546
86016 97.607 -79.234 45.531
86017 97.623 -79.238 45.516
86018 97.623 -79.238 45.501
86019 97.623 -79.237 45.486
86020 97.622 -79.237 45.471
86021 97.622 -79.237 45.456
86022 97.622 -79.236 45.444
86023 97.622 -79.236 45.431
86024 97.621 -79.235 45.418
86025 97.621 -79.235 45.402
86026 97.621 -79.234 45.386
86027 97.621 -79.234 45.370
86028 97.626 -79.239 45.355
86029 97.644 -79.259 45.343
86030 97.662 -79.279 45.332
86031 97.680 -79.300 45.320
86032 97.699 -79.320 45.308
86033 97.717 -79.341 45.296
86034 97.735 -79.361 45.284
86035 97.759 -79.382 45.272
86036 97.799 -79.406 45.260
86037 97.823 -79.419 45.249
86038 97.840 -79.427 45.239
86039 97.856 -79.435 45.229
86040 97.869 -79.451 45.219
86041 97.881 -79.466 45.209
86042 97.893 -79.481 45.200
86043 97.906 -79.496 45.191
86044 97.918 -79.511 45.181
86045 97.931 -79.526 45.171
86046 97.943 -79.541 45.163
86047 97.955 -79.557 45.155
86048 97.975 -79.569 45.145
86049 97.999 -79.576 45.136
86050 98.024 -79.584 45.128
86051 98.055 -79.583 45.120
86052 98.108 -79.555 45.113
86053 98.161 -79.527 45.106
86054 98.201 -79.514 45.099
86055 98.238 -79.505 45.091
86056 98.271 -79.493 45.082
86057 98.303 -79.510 45.073
86058 98.339 -79.535 45.066
86059 98.374 -79.560 45.057
86060 98.414 -79.591 45.049
86061 98.444 -79.620 45.041
86062 98.476 -79.632 45.033
86063 98.502 -79.641 45.024
86064 98.527 -79.650 45.016
86065 98.553 -79.659 45.007
86066 98.578 -79.668 44.999
86067 98.604 -79.677 44.991
86068 98.629 -79.686 44.985
86069 98.655 -79.695 44.978
86070 98.680 -79.704 44.972
86071 98.691 -79.723 44.968
86072 98.701 -79.723 44.964
86073 98.718 -79.714 44.959
86074 98.735 -79.705 44.956
86075 98.752 -79.696 44.952
86076 98.770 -79.688 44.949
86077 98.787 -79.679 44.946
86078 98.803 -79.670 44.942
86079 98.821 -79.670 44.938
86080 98.840 -79.681 44.933
86081 98.860 -79.692 44.927
86082 98.879 -79.704 44.921
86083 98.898 -79.715 44.913
86084 98.917 -79.726 44.905
86085 98.936 -79.739 44.897
86086 98.955 -79.750 44.888
86087 98.975 -79.761 44.878
86088 99.013 -79.764 44.868
86089 99.055 -79.764 44.860
86090 99.094 -79.766 44.851
86091 99.119 -79.780 44.840
86092 99.143 -79.794 44.828
86093 99.164 -79.808 44.815
86094 99.168 -79.822 44.803
86095 99.180 -79.833 44.792
86096 99.204 -79.839 44.781
86097 99.227 -79.845 44.773
86098 99.251 -79.852 44.764
86099 99.275 -79.858 44.755
86100 99.298 -79.864 44.748
86101 99.322 -79.871 44.740
86102 99.346 -79.877 44.732
86103 99.369 -79.883 44.723
86104 99.394 -79.889 44.713
86105 99.417 -79.896 44.706
86106 99.441 -79.902 44.697
86107 99.465 -79.908 44.688
86108 99.488 -79.914 44.681
86109 99.512 -79.921 44.673
86110 99.531 -79.921 44.663
86111 99.546 -79.914 44.655
86112 99.561 -79.908 44.647
86113 99.570 -79.916 44.638
86114 99.568 -79.944 44.628
86115 99.567 -79.972 44.620
86116 99.566 -80.000 44.612
86117 99.565 -80.028 44.603
86118 99.564 -80.056 44.595
86119 99.562 -80.084 44.587
86120 99.561 -80.112 44.580
86121 99.560 -80.140 44.574
86122 99.559 -80.168 44.566
86123 99.557 -80.196 44.559
86124 99.556 -80.224 44.552
86125 99.555 -80.252 44.544
86126 99.554 -80.280 44.537
86127 99.552 -80.308 44.529
86128 99.567 -80.337 44.521
86129 99.619 -80.349 44.511
86130 99.669 -80.362 44.501
86131 99.721 -80.375 44.491
86132 99.772 -80.387 44.481
86133 99.824 -80.400 44.470
86134 99.875 -80.413 44.458
86135 99.926 -80.425 44.446
86136 99.978 -80.438 44.434
86137 100.025 -80.453 44.422
86138 100.069 -80.469 44.409
86139 100.114 -80.486 44.396
86140 100.158 -80.502 44.384
86141 100.203 -80.519 44.371
86142 100.248 -80.535 44.358
86436 93.567 -78.735 53.624
86437 93.577 -78.733 53.615
86438 93.586 -78.730 53.606
86439 93.596 -78.728 53.597
86440 93.606 -78.725 53.590
86441 93.616 -78.723 53.582
86442 93.624 -78.720 53.575
86443 93.623 -78.711 53.569
86444 93.621 -78.701 53.565
86445 93.620 -78.692 53.561
86446 93.619 -78.683 53.560
86447 93.617 -78.688 53.564
86448 93.625 -78.681 53.562
86449 93.642 -78.670 53.557
86450 93.659 -78.660 53.550
86451 93.677 -78.649 53.541
86452 93.695 -78.639 53.532
86453 93.712 -78.628 53.520
86454 93.729 -78.617 53.505
86455 93.747 -78.607 53.488
86456 93.765 -78.596 53.469
86457 93.770 -78.582 53.449
86458 93.772 -78.568 53.427
86459 93.775 -78.554 53.406
86460 93.778 -78.540 53.384
86461 93.780 -78.526 53.363
86462 93.783 -78.512 53.341
86463 93.786 -78.498 53.318
86464 93.788 -78.483 53.295
86465 93.791 -78.469 53.270
86466 93.794 -78.455 53.245
86467 93.803 -78.453 53.221
86468 93.830 -78.485 53.198
86469 93.848 -78.486 53.175
86470 93.870 -78.472 53.154
86471 93.914 -78.454 53.132
86472 93.958 -78.437 53.115
86473 94.002 -78.420 53.098
86474 94.010 -78.426 53.081
86475 94.011 -78.436 53.065
86476 94.012 -78.447 53.050
86477 94.008 -78.459 53.032
86478 94.018 -78.463 53.010
86479 94.034 -78.460 52.986
86480 94.049 -78.457 52.963
86481 94.065 -78.454 52.940
86482 94.080 -78.451 52.918
86483 94.096 -78.448 52.896
86484 94.112 -78.445 52.873
86485 94.127 -78.443 52.849
86486 94.142 -78.440 52.829
86487 94.157 -78.438 52.811
86488 94.172 -78.437 52.795
86489 94.186 -78.435 52.778
86490 94.202 -78.416 52.760
86491 94.207 -78.404 52.741
86492 94.209 -78.398 52.721
86493 94.212 -78.391 52.703
86494 94.215 -78.385 52.685
86495 94.217 -78.408 52.668
86496 94.218 -78.470 52.649
86497 94.219 -78.532 52.633
86498 94.231 -78.557 52.617
86499 94.247 -78.562 52.599
86500 94.263 -78.568 52.582
86501 94.280 -78.573 52.564
86502 94.296 -78.579 52.546
86503 94.312 -78.586 52.529
86504 94.329 -78.591 52.511
86505 94.345 -78.597 52.493
86506 94.362 -78.600 52.471
86507 94.386 -78.593 52.450
86508 94.408 -78.587 52.428
86509 94.429 -78.578 52.404
86510 94.447 -78.564 52.379
86511 94.464 -78.548 52.351
86512 94.476 -78.545 52.321
86513 94.485 -78.545 52.289
86514 94.494 -78.546 52.255
86515 94.503 -78.546 52.220
86516 94.513 -78.546 52.185
86517 94.522 -78.546 52.151
86518 94.532 -78.547 52.119
86519 94.544 -78.549 52.086
86520 94.560 -78.553 52.051
86521 94.575 -78.557 52.017
86522 94.590 -78.561 51.983
86523 94.597 -78.564 51.948
86524 94.607 -78.562 51.912
86525 94.617 -78.561 51.874
86526 94.628 -78.560 51.835
86527 94.638 -78.558 51.793
86528 94.649 -78.558 51.748
86529 94.663 -78.562 51.702
86530 94.676 -78.566 51.653
86531 94.690 -78.569 51.602
86532 94.703 -78.573 51.552
86533 94.716 -78.576 51.502
86534 94.730 -78.580 51.451
86535 94.743 -78.583 51.401
86536 94.764 -78.587 51.351
86537 94.784 -78.590 51.301
86538 94.806 -78.594 51.252
86539 94.827 -78.597 51.204
86540 94.847 -78.601 51.155
86541 94.868 -78.604 51.108
86542 94.889 -78.608 51.062
86543 94.893 -78.611 51.017
86544 94.896 -78.616 50.972
86545 94.890 -78.622 50.928
86546 94.886 -78.634 50.884
86547 94.881 -78.646 50.842
86548 94.886 -78.648 50.800
86549 94.890 -78.650 50.758
86550 94.887 -78.650 50.717
86551 94.898 -78.636 50.677
86552 94.912 -78.617 50.635
86553 94.926 -78.600 50.592
86554 94.941 -78.582 50.549
86555 94.955 -78.564 50.506
86556 94.969 -78.546 50.463
86557 94.984 -78.528 50.419
86558 94.998 -78.510 50.375
86559 95.013 -78.492 50.330
86560 95.027 -78.475 50.285
86561 95.045 -78.470 50.240
86562 95.065 -78.475 50.196
86563 95.086 -78.480 50.153
86564 95.106 -78.485 50.110
86565 95.126 -78.490 50.066
86566 95.147 -78.495 50.024
86567 95.163 -78.500 49.982
86568 95.174 -78.504 49.941
86569 95.185 -78.507 49.903
86570 95.195 -78.511 49.864
86571 95.206 -78.515 49.826
86572 95.217 -78.518 49.787
86573 95.228 -78.522 49.750
86574 95.249 -78.532 49.712
86575 95.282 -78.549 49.675
86576 95.315 -78.566 49.638
86577 95.348 -78.582 49.600
86578 95.381 -78.600 49.562
86579 95.414 -78.617 49.524
86580 95.447 -78.633 49.488
86581 95.480 -78.650 49.452
86582 95.491 -78.667 49.416
86583 95.505 -78.688 49.380
86584 95.519 -78.709 49.346
86585 95.533 -78.730 49.312
86586 95.532 -78.734 49.280
86587 95.527 -78.733 49.247
86588 95.522 -78.733 49.213
86589 95.516 -78.733 49.176
86590 95.511 -78.732 49.136
86591 95.506 -78.732 49.096
86592 95.489 -78.727 49.054
86593 95.472 -78.722 49.012
86594 95.455 -78.717 48.969
86595 95.438 -78.712 48.923
86596 95.421 -78.708 48.877
86597 95.415 -78.699 48.834
86598 95.422 -78.690 48.792
86599 95.445 -78.692 48.749
86600 95.469 -78.694 48.709
86601 95.492 -78.696 48.668
86602 95.516 -78.698 48.629
86603 95.539 -78.700 48.589
86604 95.563 -78.702 48.550
86605 95.588 -78.702 48.511
86606 95.613 -78.702 48.472
86607 95.632 -78.705 48.435
86608 95.644 -78.710 48.397
86609 95.657 -78.714 48.361
86610 95.669 -78.718 48.325
86611 95.681 -78.723 48.289
86612 95.692 -78.727 48.251
86613 95.707 -78.732 48.212
86614 95.736 -78.744 48.172
86615 95.765 -78.755 48.132
86616 95.794 -78.748 48.091
86617 95.824 -78.742 48.051
86618 95.848 -78.738 48.011
86619 95.866 -78.739 47.971
86620 95.881 -78.743 47.931
86621 95.896 -78.749 47.889
86622 95.911 -78.753 47.848
86623 95.924 -78.759 47.807
86624 95.932 -78.764 47.766
86625 95.940 -78.771 47.725
86626 95.956 -78.772 47.684
86627 95.980 -78.769 47.642
86628 96.004 -78.765 47.599
86629 96.028 -78.762 47.557
86630 96.052 -78.759 47.516
86631 96.077 -78.755 47.474
86632 96.099 -78.759 47.434
86633 96.116 -78.777 47.393
86634 96.134 -78.794 47.354
86635 96.152 -78.811 47.316
86636 96.170 -78.829 47.279
86637 96.186 -78.829 47.241
86638 96.203 -78.817 47.204
86639 96.220 -78.805 47.167
86640 96.236 -78.792 47.130
86641 96.252 -78.780 47.094
86642 96.269 -78.767 47.061
86643 96.286 -78.755 47.028
86644 96.297 -78.759 46.995
86645 96.307 -78.769 46.962
86646 96.318 -78.787 46.930
86647 96.329 -78.805 46.898
86648 96.340 -78.822 46.866
86649 96.352 -78.840 46.836
86650 96.363 -78.858 46.805
86651 96.374 -78.871 46.775
86652 96.385 -78.874 46.745
86653 96.431 -78.874 46.717
86654 96.476 -78.874 46.690
86655 96.518 -78.871 46.665
86656 96.559 -78.865 46.638
86657 96.599 -78.861 46.612
86658 96.640 -78.856 46.585
86659 96.680 -78.851 46.557
86660 96.721 -78.846 46.534
86661 96.761 -78.841 46.512
86662 96.802 -78.836 46.490
86663 96.837 -78.832 46.469
86664 96.848 -78.829 46.448
86665 96.858 -78.827 46.426
86666 96.869 -78.824 46.405
86667 96.889 -78.835 46.386
86668 96.909 -78.846 46.367
86669 96.930 -78.847 46.349
86670 96.945 -78.849 46.328
86671 96.943 -78.886 46.308
86672 96.966 -78.935 46.289
86673 96.971 -78.941 46.272
86674 96.958 -78.959 46.257
86675 96.952 -78.973 46.241
86676 96.960 -78.984 46.226
86677 96.972 -78.961 46.209
86678 96.966 -78.976 46.194
86679 96.959 -78.990 46.178
86680 96.978 -79.034 46.161
86681 96.999 -79.052 46.146
86682 97.019 -79.046 46.132
86683 97.039 -79.041 46.117
86684 97.060 -79.036 46.100
86685 97.080 -79.031 46.082
86686 97.102 -79.033 46.064
86687 97.126 -79.037 46.049
86688 97.150 -79.042 46.037
86689 97.173 -79.046 46.024
86690 97.197 -79.051 46.009
86691 97.219 -79.057 45.995
86692 97.234 -79.068 45.978
86693 97.250 -79.079 45.961
86694 97.260 -79.086 45.943
86695 97.266 -79.089 45.924
86696 97.272 -79.092 45.905
86697 97.287 -79.101 45.885
86698 97.302 -79.109 45.866
86699 97.317 -79.118 45.846
86700 97.332 -79.127 45.827
86701 97.348 -79.136 45.808
86702 97.363 -79.145 45.791
86703 97.375 -79.147 45.773
86704 97.386 -79.146 45.756
86705 97.398 -79.145 45.738
86706 97.409 -79.144 45.720
86707 97.421 -79.143 45.703
86708 97.432 -79.143 45.686
86709 97.443 -79.141 45.671
86710 97.455 -79.141 45.655
86711 97.477 -79.153 45.639
86712 97.500 -79.163 45.624
86713 97.522 -79.175 45.608
86714 97.544 -79.185 45.591
86715 97.566 -79.196 45.576
86716 97.588 -79.207 45.560
86717 97.610 -79.219 45.544
86718 97.614 -79.225 45.528
86719 97.613 -79.224 45.514
86720 97.613 -79.224 45.499
86721 97.613 -79.223 45.483
86722 97.613 -79.223 45.468
86723 97.613 -79.222 45.455
86724 97.613 -79.222 45.443
86725 97.613 -79.222 45.430
86726 97.613 -79.221 45.413
86727 97.612 -79.221 45.396
86728 97.612 -79.220 45.379
86729 97.618 -79.226 45.364
86730 97.636 -79.247 45.351
86731 97.655 -79.266 45.339
86732 97.673 -79.287 45.327
86733 97.691 -79.307 45.316
86734 97.709 -79.328 45.305
86735 97.731 -79.349 45.293
86736 97.771 -79.372 45.281
86737 97.806 -79.393 45.269
86738 97.823 -79.401 45.258
86739 97.840 -79.410 45.248
86740 97.856 -79.418 45.239
86741 97.870 -79.432 45.229
86742 97.882 -79.447 45.220
86743 97.894 -79.462 45.211
86744 97.907 -79.477 45.202
86745 97.919 -79.492 45.193
86746 97.932 -79.507 45.182
86747 97.944 -79.523 45.174
86748 97.959 -79.537 45.166
86749 97.982 -79.547 45.158
86750 98.006 -79.554 45.149
86751 98.031 -79.562 45.142
86752 98.056 -79.569 45.135
86753 98.080 -79.576 45.128
86754 98.114 -79.572 45.120
86755 98.167 -79.544 45.111
86756 98.220 -79.516 45.103
86757 98.283 -79.508 45.095
86758 98.329 -79.528 45.087
86759 98.354 -79.551 45.078
86760 98.379 -79.574 45.069
86761 98.406 -79.595 45.060
86762 98.439 -79.604 45.051
86763 98.472 -79.613 45.042
86764 98.505 -79.621 45.033
86765 98.533 -79.630 45.025
86766 98.558 -79.639 45.018
86767 98.584 -79.648 45.010
86768 98.609 -79.657 45.003
86769 98.634 -79.667 44.997
86770 98.660 -79.676 44.990
86771 98.685 -79.686 44.984
86772 98.692 -79.705 44.980
86773 98.696 -79.716 44.975
86774 98.713 -79.708 44.972
86775 98.730 -79.698 44.968
86776 98.747 -79.690 44.964
86777 98.764 -79.681 44.961
86778 98.781 -79.673 44.957
86779 98.798 -79.665 44.953
86780 98.814 -79.665 44.949
86781 98.833 -79.676 44.944
86782 98.852 -79.689 44.938
86783 98.872 -79.700 44.931
86784 98.891 -79.711 44.923
86785 98.910 -79.723 44.914
86786 98.929 -79.734 44.905
86787 98.949 -79.745 44.895
86788 98.990 -79.747 44.885
86789 99.031 -79.748 44.874
86790 99.072 -79.748 44.864
86791 99.113 -79.750 44.856
86792 99.137 -79.764 44.844
86793 99.161 -79.777 44.831
86794 99.175 -79.792 44.819
86795 99.179 -79.806 44.806
86796 99.184 -79.820 44.795
86797 99.207 -79.826 44.784
86798 99.231 -79.832 44.776
86799 99.255 -79.839 44.768
86800 99.279 -79.845 44.760
86801 99.302 -79.851 44.752
86802 99.326 -79.857 44.745
86803 99.350 -79.864 44.736
86804 99.373 -79.870 44.727
86805 99.397 -79.876 44.718
86806 99.421 -79.882 44.709
86807 99.445 -79.889 44.701
86808 99.469 -79.895 44.692
86809 99.492 -79.901 44.684
86810 99.512 -79.902 44.676
86811 99.527 -79.897 44.667
86812 99.543 -79.891 44.659
86813 99.558 -79.885 44.651
86814 99.565 -79.896 44.642
86815 99.564 -79.924 44.632
86816 99.563 -79.952 44.624
86817 99.561 -79.980 44.616
86818 99.560 -80.008 44.607
86819 99.559 -80.036 44.599
86820 99.558 -80.064 44.591
86821 99.556 -80.092 44.584
86822 99.555 -80.120 44.577
86823 99.554 -80.148 44.570
86824 99.553 -80.175 44.562
86825 99.551 -80.203 44.555
86826 99.550 -80.231 44.548
86827 99.549 -80.259 44.541
86828 99.565 -80.284 44.533
86829 99.590 -80.314 44.524
86830 99.628 -80.355 44.515
86831 99.685 -80.349 44.505
86832 99.736 -80.362 44.495
86833 99.787 -80.375 44.485
86834 99.839 -80.388 44.474
86835 99.890 -80.400 44.463
86836 99.938 -80.415 44.451
86837 99.982 -80.431 44.439
86838 100.027 -80.447 44.427
86839 100.071 -80.464 44.414
86840 100.116 -80.480 44.401
86841 100.160 -80.497 44.389
86842 100.205 -80.513 44.375
86843 100.250 -80.529 44.362
87137 93.567 -78.724 53.631
87138 93.577 -78.722 53.621
87139 93.586 -78.719 53.611
87140 93.596 -78.717 53.602
87141 93.606 -78.714 53.594
87142 93.616 -78.712 53.586
87143 93.620 -78.706 53.579
87144 93.619 -78.697 53.571
87145 93.618 -78.688 53.566
87146 93.617 -78.679 53.561
87147 93.615 -78.670 53.559
87148 93.614 -78.661 53.560
87149 93.623 -78.658 53.557
87150 93.640 -78.648 53.553
87151 93.658 -78.637 53.545
87152 93.675 -78.626 53.536
87153 93.694 -78.616 53.524
87154 93.713 -78.606 53.510
87155 93.733 -78.595 53.495
87156 93.753 -78.585 53.476
87157 93.767 -78.579 53.456
87158 93.775 -78.570 53.434
87159 93.774 -78.553 53.412
87160 93.772 -78.535 53.390
87161 93.771 -78.519 53.368
87162 93.774 -78.504 53.348
87163 93.777 -78.491 53.328
87164 93.779 -78.476 53.307
87165 93.782 -78.462 53.286
87166 93.785 -78.448 53.262
87167 93.787 -78.434 53.238
87168 93.791 -78.420 53.215
87169 93.818 -78.452 53.192
87170 93.843 -78.476 53.171
87171 93.883 -78.458 53.150
87172 93.926 -78.441 53.129
87173 93.970 -78.423 53.113
87174 94.014 -78.406 53.097
87175 94.019 -78.413 53.081
87176 94.018 -78.426 53.063
87177 94.014 -78.438 53.046
87178 94.010 -78.451 53.028
87179 94.019 -78.458 53.006
87180 94.033 -78.458 52.981
87181 94.049 -78.455 52.958
87182 94.064 -78.452 52.935
87183 94.080 -78.449 52.913
87184 94.096 -78.446 52.892
87185 94.111 -78.443 52.870
87186 94.127 -78.440 52.847
87187 94.142 -78.437 52.830
87188 94.158 -78.434 52.814
87189 94.174 -78.431 52.798
87190 94.189 -78.428 52.782
87191 94.196 -78.433 52.764
87192 94.197 -78.445 52.745
87193 94.199 -78.438 52.725
87194 94.202 -78.432 52.706
87195 94.204 -78.425 52.689
87196 94.206 -78.432 52.673
87197 94.208 -78.495 52.655
87198 94.214 -78.537 52.639
87199 94.231 -78.542 52.623
87200 94.248 -78.548 52.605
87201 94.264 -78.554 52.587
87202 94.280 -78.560 52.570
87203 94.296 -78.565 52.552
87204 94.313 -78.571 52.535
87205 94.329 -78.577 52.518
87206 94.345 -78.582 52.498
87207 94.365 -78.581 52.477
87208 94.388 -78.574 52.456
87209 94.411 -78.567 52.434
87210 94.434 -78.561 52.412
87211 94.457 -78.554 52.386
87212 94.478 -78.545 52.359
87213 94.487 -78.545 52.330
87214 94.496 -78.545 52.297
87215 94.506 -78.546 52.263
87216 94.515 -78.546 52.229
87217 94.525 -78.546 52.194
87218 94.534 -78.546 52.162
87219 94.543 -78.547 52.129
87220 94.552 -78.547 52.096
87221 94.562 -78.547 52.060
87222 94.578 -78.551 52.025
87223 94.594 -78.556 51.990
87224 94.602 -78.558 51.954
87225 94.613 -78.557 51.919
87226 94.623 -78.555 51.882
87227 94.633 -78.554 51.845
87228 94.643 -78.553 51.803
87229 94.654 -78.551 51.758
87230 94.666 -78.552 51.710
87231 94.679 -78.555 51.663
87232 94.692 -78.559 51.612
87233 94.705 -78.563 51.560
87234 94.719 -78.567 51.509
87235 94.733 -78.570 51.458
87236 94.747 -78.572 51.407
87237 94.765 -78.576 51.355
87238 94.786 -78.579 51.304
87239 94.807 -78.582 51.254
87240 94.828 -78.586 51.204
87241 94.849 -78.589 51.155
87242 94.870 -78.592 51.107
87243 94.879 -78.596 51.061
87244 94.882 -78.601 51.015
87245 94.885 -78.604 50.970
87246 94.888 -78.609 50.926
87247 94.880 -78.612 50.883
87248 94.875 -78.624 50.841
87249 94.871 -78.636 50.799
87250 94.874 -78.640 50.757
87251 94.888 -78.622 50.716
87252 94.903 -78.604 50.676
87253 94.917 -78.586 50.634
87254 94.931 -78.569 50.591
87255 94.946 -78.551 50.549
87256 94.960 -78.532 50.505
87257 94.974 -78.515 50.463
87258 94.989 -78.497 50.419
87259 95.003 -78.479 50.375
87260 95.018 -78.461 50.331
87261 95.032 -78.445 50.286
87262 95.053 -78.450 50.242
87263 95.073 -78.454 50.198
87264 95.094 -78.460 50.155
87265 95.114 -78.465 50.111
87266 95.134 -78.470 50.068
87267 95.155 -78.476 50.025
87268 95.175 -78.480 49.983
87269 95.188 -78.485 49.942
87270 95.199 -78.489 49.904
87271 95.209 -78.492 49.865
87272 95.220 -78.495 49.827
87273 95.231 -78.499 49.788
87274 95.243 -78.504 49.750
87275 95.276 -78.520 49.713
87276 95.309 -78.538 49.676
87277 95.342 -78.554 49.639
87278 95.375 -78.571 49.602
87279 95.408 -78.588 49.564
87280 95.441 -78.605 49.528
87281 95.474 -78.622 49.492
87282 95.507 -78.639 49.456
87283 95.538 -78.655 49.419
87284 95.549 -78.674 49.384
87285 95.562 -78.695 49.350
87286 95.565 -78.704 49.316
87287 95.560 -78.704 49.284
87288 95.555 -78.703 49.252
87289 95.549 -78.702 49.218
87290 95.544 -78.702 49.183
87291 95.539 -78.702 49.143
87292 95.534 -78.701 49.103
87293 95.526 -78.699 49.061
87294 95.509 -78.695 49.018
87295 95.492 -78.690 48.973
87296 95.475 -78.685 48.928
87297 95.458 -78.680 48.883
87298 95.441 -78.675 48.840
87299 95.421 -78.672 48.798
87300 95.442 -78.673 48.757
87301 95.467 -78.673 48.717
87302 95.491 -78.673 48.678
87303 95.517 -78.674 48.638
87304 95.541 -78.675 48.597
87305 95.566 -78.675 48.558
87306 95.591 -78.676 48.520
87307 95.616 -78.677 48.481
87308 95.633 -78.680 48.443
87309 95.649 -78.684 48.406
87310 95.663 -78.689 48.370
87311 95.676 -78.693 48.335
87312 95.688 -78.697 48.299
87313 95.700 -78.702 48.261
87314 95.713 -78.706 48.220
87315 95.742 -78.717 48.180
87316 95.771 -78.717 48.139
87317 95.800 -78.710 48.099
87318 95.830 -78.704 48.058
87319 95.859 -78.697 48.019
87320 95.877 -78.699 47.979
87321 95.892 -78.704 47.937
87322 95.907 -78.709 47.895
87323 95.922 -78.714 47.854
87324 95.934 -78.720 47.812
87325 95.942 -78.726 47.772
87326 95.950 -78.732 47.731
87327 95.958 -78.737 47.691
87328 95.982 -78.734 47.651
87329 96.007 -78.731 47.610
87330 96.031 -78.727 47.570
87331 96.055 -78.724 47.530
87332 96.079 -78.721 47.489
87333 96.098 -78.736 47.449
87334 96.115 -78.754 47.409
87335 96.133 -78.771 47.370
87336 96.151 -78.789 47.331
87337 96.168 -78.806 47.293
87338 96.186 -78.819 47.254
87339 96.202 -78.806 47.218
87340 96.219 -78.794 47.180
87341 96.235 -78.782 47.143
87342 96.252 -78.769 47.107
87343 96.268 -78.757 47.073
87344 96.280 -78.759 47.041
87345 96.290 -78.767 47.009
87346 96.300 -78.776 46.977
87347 96.310 -78.783 46.944
87348 96.320 -78.793 46.913
87349 96.331 -78.810 46.882
87350 96.342 -78.827 46.851
87351 96.354 -78.845 46.821
87352 96.365 -78.863 46.791
87353 96.379 -78.865 46.761
87354 96.425 -78.865 46.734
87355 96.468 -78.863 46.707
87356 96.508 -78.858 46.680
87357 96.549 -78.853 46.654
87358 96.589 -78.849 46.627
87359 96.629 -78.844 46.599
87360 96.670 -78.839 46.573
87361 96.711 -78.834 46.549
87362 96.751 -78.829 46.527
87363 96.792 -78.824 46.505
87364 96.832 -78.820 46.484
87365 96.844 -78.817 46.464
87366 96.855 -78.814 46.443
87367 96.868 -78.815 46.421
87368 96.888 -78.826 46.402
87369 96.908 -78.836 46.383
87370 96.929 -78.843 46.364
87371 96.946 -78.842 46.343
87372 96.947 -78.880 46.324
87373 96.965 -78.918 46.305
87374 96.969 -78.924 46.289
87375 96.958 -78.940 46.274
87376 96.944 -78.957 46.258
87377 96.950 -78.968 46.242
87378 96.958 -78.977 46.226
87379 96.962 -78.968 46.210
87380 96.956 -78.983 46.194
87381 96.976 -79.027 46.178
87382 96.996 -79.021 46.164
87383 97.016 -79.016 46.149
87384 97.036 -79.012 46.133
87385 97.059 -79.012 46.116
87386 97.083 -79.017 46.098
87387 97.106 -79.021 46.079
87388 97.130 -79.026 46.062
87389 97.154 -79.031 46.049
87390 97.177 -79.035 46.035
87391 97.201 -79.040 46.022
87392 97.221 -79.047 46.007
87393 97.237 -79.059 45.992
87394 97.252 -79.069 45.976
87395 97.268 -79.081 45.957
87396 97.275 -79.084 45.936
87397 97.280 -79.087 45.916
87398 97.295 -79.096 45.896
87399 97.310 -79.104 45.878
87400 97.325 -79.113 45.859
87401 97.340 -79.122 45.839
87402 97.355 -79.131 45.821
87403 97.367 -79.131 45.803
87404 97.378 -79.131 45.786
87405 97.389 -79.129 45.769
87406 97.401 -79.128 45.752
87407 97.412 -79.128 45.735
87408 97.424 -79.127 45.719
87409 97.435 -79.126 45.700
87410 97.446 -79.125 45.684
87411 97.457 -79.125 45.669
87412 97.478 -79.134 45.653
87413 97.500 -79.144 45.638
87414 97.522 -79.156 45.623
87415 97.544 -79.166 45.608
87416 97.567 -79.178 45.593
87417 97.588 -79.190 45.576
87418 97.610 -79.206 45.559
87419 97.612 -79.211 45.543
87420 97.607 -79.210 45.529
87421 97.605 -79.209 45.512
87422 97.605 -79.209 45.494
87423 97.604 -79.209 45.478
87424 97.604 -79.208 45.465
87425 97.604 -79.208 45.451
87426 97.604 -79.207 45.437
87427 97.604 -79.207 45.420
87428 97.604 -79.206 45.403
87429 97.604 -79.206 45.387
87430 97.610 -79.213 45.373
87431 97.629 -79.234 45.360
87432 97.647 -79.254 45.348
87433 97.665 -79.274 45.337
87434 97.683 -79.294 45.326
87435 97.703 -79.315 45.314
87436 97.743 -79.339 45.303
87437 97.783 -79.363 45.291
87438 97.806 -79.375 45.280
87439 97.823 -79.384 45.270
87440 97.840 -79.392 45.260
87441 97.856 -79.400 45.251
87442 97.871 -79.412 45.242
87443 97.883 -79.428 45.232
87444 97.895 -79.443 45.223
87445 97.908 -79.458 45.215
87446 97.920 -79.473 45.206
87447 97.933 -79.488 45.197
87448 97.945 -79.504 45.189
87449 97.966 -79.514 45.181
87450 97.990 -79.524 45.173
87451 98.014 -79.532 45.166
87452 98.038 -79.540 45.158
87453 98.063 -79.547 45.151
87454 98.087 -79.554 45.143
87455 98.112 -79.561 45.135
87456 98.137 -79.569 45.126
87457 98.179 -79.564 45.117
87458 98.249 -79.545 45.108
87459 98.314 -79.528 45.099
87460 98.340 -79.551 45.089
87461 98.369 -79.567 45.080
87462 98.402 -79.576 45.071
87463 98.435 -79.585 45.062
87464 98.468 -79.594 45.053
87465 98.502 -79.602 45.044
87466 98.535 -79.611 45.036
87467 98.563 -79.620 45.029
87468 98.589 -79.629 45.022
87469 98.614 -79.639 45.016
87470 98.640 -79.648 45.009
87471 98.665 -79.657 45.003
87472 98.691 -79.666 44.997
87473 98.694 -79.688 44.992
87474 98.690 -79.709 44.987
87475 98.707 -79.701 44.984
87476 98.724 -79.692 44.980
87477 98.741 -79.683 44.976
87478 98.758 -79.675 44.973
87479 98.775 -79.666 44.968
87480 98.791 -79.663 44.963
87481 98.807 -79.662 44.958
87482 98.826 -79.673 44.950
87483 98.845 -79.684 44.943
87484 98.864 -79.695 44.935
87485 98.884 -79.708 44.928
87486 98.903 -79.719 44.919
87487 98.925 -79.729 44.910
87488 98.966 -79.730 44.902
87489 99.007 -79.731 44.892
87490 99.048 -79.732 44.879
87491 99.089 -79.733 44.867
87492 99.131 -79.734 44.855
87493 99.155 -79.747 44.843
87494 99.180 -79.761 44.832
87495 99.185 -79.775 44.822
87496 99.189 -79.789 44.810
87497 99.192 -79.804 44.799
87498 99.211 -79.812 44.789
87499 99.235 -79.818 44.780
87500 99.259 -79.824 44.770
87501 99.282 -79.831 44.761
87502 99.306 -79.837 44.754
87503 99.330 -79.843 44.747
87504 99.354 -79.849 44.739
87505 99.377 -79.856 44.730
87506 99.401 -79.862 44.721
87507 99.425 -79.868 44.712
87508 99.448 -79.874 44.704
87509 99.472 -79.881 44.695
87510 99.494 -79.885 44.687
87511 99.509 -79.879 44.678
87512 99.524 -79.874 44.670
87513 99.540 -79.867 44.662
87514 99.555 -79.861 44.653
87515 99.561 -79.875 44.645
87516 99.559 -79.903 44.636
87517 99.558 -79.931 44.628
87518 99.557 -79.959 44.619
87519 99.556 -79.987 44.611
87520 99.554 -80.015 44.603
87521 99.553 -80.043 44.595
87522 99.552 -80.071 44.588
87523 99.551 -80.099 44.581
87524 99.549 -80.127 44.573
87525 99.548 -80.155 44.566
87526 99.547 -80.183 44.559
87527 99.548 -80.211 44.552
87528 99.567 -80.235 44.545
87529 99.587 -80.260 44.537
87530 99.613 -80.290 44.529
87531 99.651 -80.331 44.520
87532 99.690 -80.369 44.511
87533 99.751 -80.349 44.501
87534 99.802 -80.362 44.490
87535 99.851 -80.376 44.480
87536 99.896 -80.392 44.468
87537 99.940 -80.409 44.456
87538 99.985 -80.425 44.444
87539 100.029 -80.441 44.432
87540 100.074 -80.458 44.419
87541 100.118 -80.474 44.407
87542 100.163 -80.491 44.394
87543 100.207 -80.507 44.380
87544 100.252 -80.523 44.366
87838 93.567 -78.714 53.638
87839 93.577 -78.711 53.628
87840 93.587 -78.708 53.617
87841 93.596 -78.706 53.607
87842 93.606 -78.704 53.598
87843 93.616 -78.701 53.590
87844 93.617 -78.693 53.582
87845 93.616 -78.684 53.574
87846 93.615 -78.675 53.567
87847 93.613 -78.666 53.561
87848 93.612 -78.656 53.557
87849 93.611 -78.647 53.556
87850 93.624 -78.636 53.552
87851 93.644 -78.626 53.547
87852 93.663 -78.615 53.539
87853 93.683 -78.604 53.528
87854 93.703 -78.595 53.514
87855 93.722 -78.584 53.498
87856 93.741 -78.575 53.481
87857 93.755 -78.570 53.460
87858 93.770 -78.564 53.439
87859 93.784 -78.559 53.416
87860 93.786 -78.545 53.393
87861 93.784 -78.527 53.371
87862 93.783 -78.510 53.350
87863 93.781 -78.493 53.329
87864 93.779 -78.476 53.310
87865 93.778 -78.458 53.292
87866 93.776 -78.442 53.273
87867 93.779 -78.427 53.252
87868 93.781 -78.413 53.231
87869 93.784 -78.398 53.209
87870 93.806 -78.420 53.187
87871 93.844 -78.444 53.166
87872 93.891 -78.432 53.145
87873 93.935 -78.418 53.126
87874 93.981 -78.404 53.111
87875 94.024 -78.391 53.095
87876 94.024 -78.404 53.079
87877 94.020 -78.417 53.059
87878 94.016 -78.430 53.041
87879 94.012 -78.442 53.021
87880 94.021 -78.450 52.999
87881 94.033 -78.455 52.974
87882 94.048 -78.452 52.951
87883 94.064 -78.449 52.929
87884 94.079 -78.446 52.907
87885 94.095 -78.443 52.886
87886 94.111 -78.440 52.867
87887 94.126 -78.438 52.846
87888 94.141 -78.435 52.831
87889 94.157 -78.432 52.816
87890 94.172 -78.429 52.801
87891 94.185 -78.428 52.786
87892 94.190 -78.435 52.769
87893 94.190 -78.447 52.750
87894 94.191 -78.459 52.729
87895 94.191 -78.470 52.712
87896 94.194 -78.466 52.695
87897 94.196 -78.459 52.679
87898 94.198 -78.518 52.662
87899 94.214 -78.523 52.646
87900 94.231 -78.529 52.630
87901 94.248 -78.534 52.612
87902 94.264 -78.539 52.594
87903 94.280 -78.545 52.576
87904 94.296 -78.551 52.559
87905 94.313 -78.557 52.542
87906 94.329 -78.563 52.525
87907 94.346 -78.568 52.506
87908 94.368 -78.561 52.485
87909 94.391 -78.554 52.463
87910 94.414 -78.548 52.441
87911 94.437 -78.542 52.418
87912 94.460 -78.535 52.393
87913 94.470 -78.548 52.367
87914 94.484 -78.542 52.338
87915 94.497 -78.538 52.305
87916 94.509 -78.540 52.272
87917 94.522 -78.543 52.238
87918 94.535 -78.545 52.203
87919 94.545 -78.546 52.171
87920 94.555 -78.547 52.140
87921 94.564 -78.547 52.107
87922 94.573 -78.547 52.072
87923 94.582 -78.547 52.037
87924 94.596 -78.550 52.002
87925 94.608 -78.553 51.967
87926 94.618 -78.551 51.931
87927 94.628 -78.550 51.894
87928 94.639 -78.549 51.856
87929 94.649 -78.548 51.815
87930 94.660 -78.546 51.769
87931 94.670 -78.545 51.722
87932 94.681 -78.545 51.674
87933 94.695 -78.549 51.622
87934 94.708 -78.553 51.570
87935 94.722 -78.555 51.518
87936 94.737 -78.558 51.466
87937 94.751 -78.561 51.414
87938 94.767 -78.564 51.361
87939 94.788 -78.567 51.308
87940 94.809 -78.570 51.258
87941 94.830 -78.574 51.208
87942 94.851 -78.577 51.158
87943 94.865 -78.581 51.109
87944 94.868 -78.586 51.061
87945 94.871 -78.589 51.015
87946 94.874 -78.594 50.970
87947 94.877 -78.598 50.926
87948 94.879 -78.602 50.883
87949 94.872 -78.604 50.841
87950 94.869 -78.618 50.799
87951 94.881 -78.608 50.757
87952 94.893 -78.591 50.716
87953 94.908 -78.573 50.676
87954 94.922 -78.555 50.633
87955 94.936 -78.538 50.590
87956 94.951 -78.520 50.547
87957 94.965 -78.501 50.505
87958 94.979 -78.484 50.461
87959 94.994 -78.466 50.417
87960 95.008 -78.448 50.373
87961 95.023 -78.430 50.330
87962 95.040 -78.424 50.287
87963 95.061 -78.429 50.244
87964 95.081 -78.435 50.200
87965 95.102 -78.440 50.157
87966 95.122 -78.445 50.113
87967 95.142 -78.450 50.069
87968 95.163 -78.455 50.027
87969 95.183 -78.460 49.986
87970 95.202 -78.465 49.944
87971 95.212 -78.469 49.906
87972 95.224 -78.473 49.867
87973 95.234 -78.476 49.829
87974 95.245 -78.480 49.790
87975 95.269 -78.492 49.752
87976 95.303 -78.509 49.714
87977 95.335 -78.526 49.677
87978 95.369 -78.542 49.642
87979 95.401 -78.560 49.604
87980 95.435 -78.576 49.567
87981 95.467 -78.593 49.532
87982 95.500 -78.610 49.496
87983 95.533 -78.627 49.459
87984 95.566 -78.644 49.423
87985 95.592 -78.661 49.389
87986 95.599 -78.673 49.354
87987 95.593 -78.673 49.320
87988 95.588 -78.673 49.288
87989 95.583 -78.672 49.256
87990 95.578 -78.672 49.222
87991 95.572 -78.671 49.187
87992 95.567 -78.670 49.148
87993 95.561 -78.670 49.108
87994 95.556 -78.670 49.068
87995 95.546 -78.667 49.024
87996 95.529 -78.662 48.979
87997 95.512 -78.658 48.934
87998 95.495 -78.653 48.890
87999 95.478 -78.648 48.847
88000 95.454 -78.646 48.806
88001 95.445 -78.646 48.766
88002 95.470 -78.647 48.726
88003 95.495 -78.648 48.688
88004 95.520 -78.648 48.648
88005 95.544 -78.648 48.606
88006 95.569 -78.649 48.567
88007 95.594 -78.650 48.528
88008 95.619 -78.650 48.489
88009 95.635 -78.654 48.451
88010 95.650 -78.658 48.414
88011 95.665 -78.663 48.378
88012 95.680 -78.667 48.343
88013 95.694 -78.672 48.308
88014 95.708 -78.677 48.269
88015 95.720 -78.680 48.228
88016 95.748 -78.685 48.187
88017 95.777 -78.678 48.147
88018 95.806 -78.673 48.106
88019 95.834 -78.669 48.066
88020 95.862 -78.667 48.027
88021 95.883 -78.666 47.986
88022 95.903 -78.665 47.944
88023 95.918 -78.670 47.903
88024 95.933 -78.676 47.862
88025 95.944 -78.681 47.820
88026 95.952 -78.687 47.779
88027 95.960 -78.693 47.738
88028 95.967 -78.699 47.698
88029 95.985 -78.699 47.659
88030 96.010 -78.696 47.619
88031 96.034 -78.693 47.580
88032 96.058 -78.689 47.541
88033 96.079 -78.696 47.500
88034 96.097 -78.714 47.462
88035 96.114 -78.732 47.422
88036 96.132 -78.749 47.383
88037 96.150 -78.767 47.344
88038 96.167 -78.784 47.305
88039 96.185 -78.802 47.267
88040 96.202 -78.796 47.231
88041 96.218 -78.783 47.193
88042 96.235 -78.771 47.156
88043 96.251 -78.758 47.119
88044 96.264 -78.759 47.085
88045 96.273 -78.767 47.053
88046 96.284 -78.775 47.022
88047 96.294 -78.783 46.992
88048 96.304 -78.792 46.960
88049 96.313 -78.800 46.929
88050 96.323 -78.808 46.899
88051 96.333 -78.817 46.868
88052 96.344 -78.833 46.838
88053 96.356 -78.850 46.808
88054 96.374 -78.858 46.779
88055 96.417 -78.855 46.752
88056 96.458 -78.851 46.724
88057 96.498 -78.846 46.695
88058 96.538 -78.841 46.670
88059 96.579 -78.836 46.642
88060 96.619 -78.831 46.614
88061 96.660 -78.827 46.589
88062 96.700 -78.822 46.565
88063 96.741 -78.817 46.543
88064 96.782 -78.812 46.522
88065 96.822 -78.808 46.501
88066 96.841 -78.804 46.481
88067 96.852 -78.802 46.460
88068 96.868 -78.807 46.438
88069 96.888 -78.817 46.419
88070 96.908 -78.827 46.399
88071 96.928 -78.838 46.379
88072 96.948 -78.835 46.359
88073 96.952 -78.876 46.340
88074 96.964 -78.899 46.322
88075 96.968 -78.906 46.305
88076 96.958 -78.921 46.290
88077 96.944 -78.938 46.275
88078 96.939 -78.953 46.259
88079 96.947 -78.964 46.243
88080 96.955 -78.968 46.227
88081 96.953 -78.975 46.210
88082 96.973 -78.997 46.195
88083 96.994 -78.992 46.181
88084 97.016 -78.993 46.165
88085 97.039 -78.997 46.148
88086 97.063 -79.002 46.131
88087 97.087 -79.006 46.113
88088 97.110 -79.011 46.094
88089 97.134 -79.015 46.076
88090 97.158 -79.020 46.061
88091 97.181 -79.024 46.047
88092 97.205 -79.029 46.033
88093 97.223 -79.038 46.021
88094 97.239 -79.049 46.007
88095 97.254 -79.060 45.991
88096 97.271 -79.071 45.973
88097 97.283 -79.080 45.953
88098 97.289 -79.083 45.933
88099 97.302 -79.090 45.911
88100 97.317 -79.100 45.893
88101 97.332 -79.108 45.874
88102 97.348 -79.116 45.855
88103 97.359 -79.115 45.837
88104 97.370 -79.115 45.820
88105 97.381 -79.114 45.803
88106 97.392 -79.113 45.787
88107 97.404 -79.112 45.770
88108 97.415 -79.112 45.754
88109 97.427 -79.111 45.737
88110 97.438 -79.109 45.720
88111 97.449 -79.109 45.704
88112 97.460 -79.108 45.690
88113 97.478 -79.115 45.675
88114 97.501 -79.125 45.660
88115 97.523 -79.137 45.645
88116 97.544 -79.150 45.630
88117 97.566 -79.162 45.613
88118 97.587 -79.176 45.594
88119 97.611 -79.194 45.574
88120 97.610 -79.197 45.555
88121 97.605 -79.197 45.540
88122 97.600 -79.196 45.523
88123 97.596 -79.195 45.506
88124 97.596 -79.195 45.490
88125 97.596 -79.194 45.474
88126 97.595 -79.194 45.459
88127 97.595 -79.194 45.444
88128 97.595 -79.193 45.429
88129 97.595 -79.193 45.414
88130 97.595 -79.192 45.399
88131 97.602 -79.200 45.383
88132 97.620 -79.221 45.370
88133 97.639 -79.241 45.358
88134 97.657 -79.261 45.347
88135 97.675 -79.281 45.336
88136 97.714 -79.305 45.325
88137 97.754 -79.328 45.314
88138 97.789 -79.349 45.302
88139 97.806 -79.358 45.292
88140 97.823 -79.366 45.283
88141 97.840 -79.375 45.273
88142 97.856 -79.383 45.264
88143 97.872 -79.393 45.254
88144 97.884 -79.409 45.245
88145 97.896 -79.424 45.236
88146 97.909 -79.439 45.228
88147 97.921 -79.454 45.220
88148 97.934 -79.469 45.213
88149 97.951 -79.482 45.205
88150 97.973 -79.492 45.197
88151 97.997 -79.502 45.190
88152 98.021 -79.510 45.183
88153 98.045 -79.518 45.175
88154 98.070 -79.525 45.167
88155 98.095 -79.532 45.158
88156 98.119 -79.539 45.150
88157 98.144 -79.547 45.141
88158 98.183 -79.543 45.131
88159 98.238 -79.538 45.120
88160 98.293 -79.533 45.110
88161 98.332 -79.539 45.100
88162 98.365 -79.548 45.092
88163 98.398 -79.557 45.083
88164 98.432 -79.566 45.074
88165 98.465 -79.574 45.065
88166 98.499 -79.583 45.056
88167 98.532 -79.592 45.049
88168 98.565 -79.601 45.042
88169 98.594 -79.610 45.036
88170 98.620 -79.619 45.029
88171 98.645 -79.628 45.023
88172 98.670 -79.637 45.018
88173 98.696 -79.647 45.011
88174 98.695 -79.670 45.006
88175 98.687 -79.699 45.000
88176 98.701 -79.694 44.995
88177 98.718 -79.686 44.991
88178 98.735 -79.676 44.988
88179 98.752 -79.668 44.984
88180 98.769 -79.662 44.978
88181 98.785 -79.661 44.971
88182 98.800 -79.661 44.965
88183 98.819 -79.669 44.957
88184 98.838 -79.680 44.947
88185 98.857 -79.692 44.938
88186 98.876 -79.703 44.929
88187 98.902 -79.712 44.920
88188 98.942 -79.713 44.911
88189 98.984 -79.714 44.902
88190 99.025 -79.715 44.893
88191 99.066 -79.716 44.881
88192 99.107 -79.717 44.868
88193 99.148 -79.718 44.857
88194 99.174 -79.730 44.845
88195 99.192 -79.745 44.835
88196 99.196 -79.759 44.825
88197 99.199 -79.773 44.814
88198 99.203 -79.788 44.804
88199 99.215 -79.798 44.794
88200 99.239 -79.805 44.784
88201 99.263 -79.811 44.774
88202 99.286 -79.817 44.764
88203 99.310 -79.823 44.756
88204 99.334 -79.830 44.749
88205 99.357 -79.836 44.741
88206 99.381 -79.842 44.731
88207 99.405 -79.849 44.723
88208 99.429 -79.855 44.715
88209 99.452 -79.861 44.707
88210 99.476 -79.867 44.698
88211 99.491 -79.861 44.689
88212 99.506 -79.855 44.680
88213 99.521 -79.850 44.672
88214 99.537 -79.844 44.664
88215 99.552 -79.838 44.655
88216 99.556 -79.855 44.647
88217 99.555 -79.883 44.640
88218 99.553 -79.911 44.632
88219 99.552 -79.939 44.623
88220 99.551 -79.967 44.614
88221 99.550 -79.995 44.606
88222 99.548 -80.023 44.599
88223 99.547 -80.051 44.592
88224 99.546 -80.079 44.585
88225 99.545 -80.107 44.577
88226 99.543 -80.135 44.570
88227 99.550 -80.162 44.563
88228 99.569 -80.186 44.556
88229 99.589 -80.211 44.549
88230 99.609 -80.235 44.542
88231 99.636 -80.267 44.534
88232 99.674 -80.308 44.525
88233 99.712 -80.349 44.516
88234 99.747 -80.375 44.506
88235 99.809 -80.354 44.496
88236 99.853 -80.370 44.485
88237 99.898 -80.387 44.474
88238 99.942 -80.403 44.462
88239 99.987 -80.419 44.450
88240 100.031 -80.435 44.437
88241 100.076 -80.452 44.425
88242 100.121 -80.469 44.412
88243 100.165 -80.485 44.399
88244 100.210 -80.501 44.386
88245 100.255 -80.517 44.371
88246 100.299 -80.534 44.357
88247 100.344 -80.550 44.344
88248 100.388 -80.566 44.330
88539 93.568 -78.702 53.644
88540 93.577 -78.700 53.633
88541 93.587 -78.698 53.623
88542 93.597 -78.695 53.612
88543 93.607 -78.692 53.602
88544 93.615 -78.689 53.593
88545 93.614 -78.680 53.584
88546 93.612 -78.671 53.575
88547 93.611 -78.661 53.567
88548 93.610 -78.652 53.560
88549 93.609 -78.643 53.555
88550 93.608 -78.633 53.552
88551 93.631 -78.618 53.547
88552 93.653 -78.604 53.540
88553 93.673 -78.594 53.530
88554 93.692 -78.583 53.518
88555 93.712 -78.573 53.502
88556 93.729 -78.565 53.484
88557 93.744 -78.560 53.464
88558 93.758 -78.554 53.442
88559 93.773 -78.549 53.419
88560 93.787 -78.544 53.397
88561 93.798 -78.535 53.374
88562 93.796 -78.519 53.353
88563 93.795 -78.501 53.332
88564 93.793 -78.484 53.313
88565 93.791 -78.467 53.294
88566 93.790 -78.450 53.277
88567 93.788 -78.433 53.259
88568 93.787 -78.416 53.241
88569 93.785 -78.398 53.222
88570 93.783 -78.382 53.201
88571 93.799 -78.346 53.180
88572 93.847 -78.399 53.159
88573 93.895 -78.395 53.139
88574 93.940 -78.381 53.121
88575 93.980 -78.373 53.105
88576 94.007 -78.385 53.089
88577 94.026 -78.396 53.073
88578 94.022 -78.409 53.054
88579 94.018 -78.422 53.035
88580 94.014 -78.435 53.014
88581 94.023 -78.442 52.992
88582 94.035 -78.447 52.969
88583 94.048 -78.450 52.947
88584 94.063 -78.447 52.924
88585 94.078 -78.444 52.904
88586 94.094 -78.441 52.886
88587 94.110 -78.438 52.868
88588 94.125 -78.435 52.850
88589 94.141 -78.432 52.836
88590 94.155 -78.429 52.821
88591 94.168 -78.429 52.805
88592 94.181 -78.427 52.790
88593 94.184 -78.437 52.773
88594 94.184 -78.449 52.755
88595 94.185 -78.460 52.735
88596 94.185 -78.473 52.718
88597 94.186 -78.484 52.702
88598 94.186 -78.496 52.687
88599 94.198 -78.504 52.670
88600 94.214 -78.509 52.653
88601 94.231 -78.514 52.637
88602 94.248 -78.520 52.619
88603 94.264 -78.525 52.601
88604 94.280 -78.531 52.584
88605 94.297 -78.537 52.567
88606 94.313 -78.542 52.550
88607 94.329 -78.548 52.533
88608 94.348 -78.548 52.514
88609 94.371 -78.542 52.493
88610 94.394 -78.535 52.470
88611 94.417 -78.529 52.447
88612 94.439 -78.523 52.425
88613 94.449 -78.537 52.400
88614 94.459 -78.550 52.374
88615 94.472 -78.545 52.347
88616 94.486 -78.539 52.313
88617 94.500 -78.532 52.281
88618 94.513 -78.529 52.247
88619 94.526 -78.531 52.213
88620 94.538 -78.533 52.181
88621 94.551 -78.535 52.150
88622 94.563 -78.539 52.120
88623 94.576 -78.541 52.086
88624 94.589 -78.543 52.052
88625 94.601 -78.545 52.017
88626 94.613 -78.548 51.982
88627 94.623 -78.546 51.945
88628 94.634 -78.545 51.907
88629 94.644 -78.543 51.869
88630 94.654 -78.542 51.827
88631 94.665 -78.540 51.781
88632 94.675 -78.539 51.734
88633 94.685 -78.537 51.685
88634 94.698 -78.539 51.632
88635 94.712 -78.542 51.578
88636 94.726 -78.544 51.525
88637 94.740 -78.547 51.472
88638 94.754 -78.550 51.420
88639 94.769 -78.552 51.367
88640 94.789 -78.556 51.313
88641 94.811 -78.559 51.262
88642 94.832 -78.563 51.211
88643 94.851 -78.566 51.160
88644 94.854 -78.570 51.111
88645 94.857 -78.574 51.063
88646 94.860 -78.579 51.017
88647 94.862 -78.582 50.972
88648 94.865 -78.587 50.929
88649 94.868 -78.591 50.886
88650 94.871 -78.595 50.845
88651 94.876 -78.594 50.803
88652 94.887 -78.576 50.761
88653 94.899 -78.560 50.719
88654 94.913 -78.542 50.677
88655 94.927 -78.524 50.634
88656 94.942 -78.506 50.591
88657 94.956 -78.489 50.547
88658 94.970 -78.470 50.504
88659 94.985 -78.453 50.461
88660 94.999 -78.435 50.416
88661 95.014 -78.416 50.373
88662 95.028 -78.399 50.329
88663 95.048 -78.404 50.285
88664 95.068 -78.410 50.242
88665 95.089 -78.414 50.201
88666 95.110 -78.420 50.159
88667 95.130 -78.425 50.115
88668 95.150 -78.430 50.071
88669 95.171 -78.435 50.029
88670 95.191 -78.440 49.988
88671 95.212 -78.445 49.946
88672 95.227 -78.450 49.908
88673 95.237 -78.454 49.869
88674 95.248 -78.457 49.831
88675 95.263 -78.463 49.795
88676 95.296 -78.480 49.756
88677 95.329 -78.498 49.719
88678 95.362 -78.514 49.682
88679 95.395 -78.531 49.645
88680 95.428 -78.548 49.607
88681 95.461 -78.565 49.570
88682 95.494 -78.582 49.535
88683 95.527 -78.598 49.499
88684 95.560 -78.615 49.464
88685 95.589 -78.630 49.429
88686 95.612 -78.642 49.394
88687 95.620 -78.643 49.359
88688 95.621 -78.642 49.324
88689 95.616 -78.642 49.291
88690 95.611 -78.642 49.258
88691 95.605 -78.641 49.223
88692 95.600 -78.640 49.188
88693 95.595 -78.640 49.150
88694 95.590 -78.639 49.112
88695 95.584 -78.639 49.072
88696 95.579 -78.639 49.029
88697 95.566 -78.635 48.985
88698 95.549 -78.630 48.941
88699 95.532 -78.625 48.898
88700 95.514 -78.620 48.856
88701 95.487 -78.620 48.816
88702 95.461 -78.620 48.776
88703 95.473 -78.620 48.737
88704 95.498 -78.621 48.697
88705 95.523 -78.622 48.657
88706 95.548 -78.623 48.615
88707 95.572 -78.623 48.575
88708 95.597 -78.623 48.535
88709 95.621 -78.624 48.496
88710 95.636 -78.628 48.458
88711 95.652 -78.632 48.421
88712 95.666 -78.636 48.384
88713 95.681 -78.641 48.349
88714 95.696 -78.646 48.314
88715 95.711 -78.651 48.277
88716 95.726 -78.655 48.237
88717 95.752 -78.653 48.195
88718 95.780 -78.649 48.154
88719 95.808 -78.646 48.113
88720 95.837 -78.642 48.073
88721 95.859 -78.642 48.034
88722 95.881 -78.644 47.994
88723 95.902 -78.641 47.953
88724 95.923 -78.639 47.913
88725 95.944 -78.637 47.873
88726 95.954 -78.642 47.831
88727 95.962 -78.648 47.787
88728 95.970 -78.655 47.747
88729 95.977 -78.660 47.706
88730 95.988 -78.664 47.666
88731 96.012 -78.661 47.626
88732 96.036 -78.658 47.586
88733 96.060 -78.657 47.546
88734 96.078 -78.674 47.507
88735 96.096 -78.692 47.470
88736 96.113 -78.709 47.431
88737 96.131 -78.727 47.392
88738 96.149 -78.744 47.354
88739 96.166 -78.761 47.316
88740 96.184 -78.779 47.279
88741 96.201 -78.786 47.242
88742 96.218 -78.773 47.205
88743 96.234 -78.761 47.168
88744 96.247 -78.758 47.132
88745 96.257 -78.767 47.097
88746 96.267 -78.775 47.066
88747 96.277 -78.783 47.037
88748 96.287 -78.792 47.007
88749 96.297 -78.800 46.978
88750 96.307 -78.808 46.948
88751 96.317 -78.817 46.918
88752 96.327 -78.824 46.888
88753 96.337 -78.833 46.857
88754 96.346 -78.841 46.827
88755 96.367 -78.849 46.797
88756 96.407 -78.843 46.768
88757 96.448 -78.839 46.739
88758 96.488 -78.834 46.710
88759 96.528 -78.829 46.683
88760 96.569 -78.824 46.656
88761 96.609 -78.819 46.629
88762 96.650 -78.814 46.605
88763 96.690 -78.810 46.583
88764 96.731 -78.805 46.562
88765 96.771 -78.800 46.540
88766 96.812 -78.796 46.519
88767 96.838 -78.791 46.498
88768 96.848 -78.789 46.477
88769 96.867 -78.798 46.455
88770 96.887 -78.808 46.436
88771 96.907 -78.818 46.417
88772 96.928 -78.829 46.398
88773 96.948 -78.830 46.378
88774 96.956 -78.871 46.357
88775 96.963 -78.882 46.338
88776 96.967 -78.888 46.319
88777 96.958 -78.902 46.304
88778 96.944 -78.920 46.289
88779 96.931 -78.937 46.273
88780 96.936 -78.949 46.257
88781 96.944 -78.959 46.241
88782 96.950 -78.969 46.225
88783 96.972 -78.973 46.208
88784 96.996 -78.977 46.192
88785 97.020 -78.982 46.176
88786 97.043 -78.987 46.159
88787 97.067 -78.991 46.142
88788 97.091 -78.996 46.125
88789 97.114 -79.000 46.107
88790 97.138 -79.005 46.090
88791 97.162 -79.009 46.075
88792 97.185 -79.014 46.061
88793 97.209 -79.018 46.046
88794 97.226 -79.028 46.034
88795 97.241 -79.040 46.020
88796 97.257 -79.050 46.005
88797 97.273 -79.062 45.988
88798 97.289 -79.072 45.971
88799 97.298 -79.078 45.952
88800 97.310 -79.085 45.932
88801 97.325 -79.094 45.911
88802 97.339 -79.101 45.890
88803 97.351 -79.100 45.871
88804 97.362 -79.100 45.854
88805 97.373 -79.099 45.838
88806 97.384 -79.097 45.822
88807 97.396 -79.096 45.806
88808 97.407 -79.096 45.788
88809 97.418 -79.095 45.771
88810 97.430 -79.094 45.753
88811 97.441 -79.093 45.736
88812 97.452 -79.093 45.721
88813 97.463 -79.092 45.708
88814 97.479 -79.096 45.695
88815 97.501 -79.108 45.679
88816 97.522 -79.121 45.663
88817 97.543 -79.134 45.645
88818 97.565 -79.147 45.627
88819 97.586 -79.161 45.607
88820 97.612 -79.184 45.587
88821 97.607 -79.183 45.568
88822 97.603 -79.182 45.552
88823 97.598 -79.182 45.536
88824 97.593 -79.181 45.519
88825 97.588 -79.181 45.502
88826 97.587 -79.181 45.485
88827 97.587 -79.180 45.469
88828 97.587 -79.179 45.453
88829 97.586 -79.179 45.439
88830 97.586 -79.178 45.425
88831 97.586 -79.178 45.409
88832 97.595 -79.187 45.394
88833 97.613 -79.208 45.381
88834 97.631 -79.228 45.369
88835 97.650 -79.248 45.358
88836 97.686 -79.272 45.347
88837 97.726 -79.295 45.336
88838 97.766 -79.319 45.325
88839 97.789 -79.332 45.315
88840 97.806 -79.341 45.305
88841 97.823 -79.349 45.296
88842 97.840 -79.357 45.287
88843 97.856 -79.366 45.277
88844 97.873 -79.374 45.267
88845 97.885 -79.389 45.258
88846 97.897 -79.405 45.250
88847 97.910 -79.420 45.242
88848 97.922 -79.435 45.235
88849 97.935 -79.451 45.228
88850 97.958 -79.460 45.220
88851 97.981 -79.470 45.213
88852 98.004 -79.480 45.206
88853 98.028 -79.488 45.199
88854 98.053 -79.496 45.191
88855 98.077 -79.503 45.183
88856 98.102 -79.510 45.174
88857 98.127 -79.517 45.165
88858 98.151 -79.519 45.155
88859 98.195 -79.514 45.144
88860 98.250 -79.509 45.132
88861 98.301 -79.505 45.121
88862 98.330 -79.512 45.111
88863 98.363 -79.523 45.103
88864 98.395 -79.535 45.094
88865 98.428 -79.547 45.086
88866 98.462 -79.555 45.078
88867 98.495 -79.564 45.070
88868 98.528 -79.573 45.064
88869 98.562 -79.582 45.057
88870 98.595 -79.591 45.050
88871 98.625 -79.600 45.044
88872 98.650 -79.609 45.038
88873 98.675 -79.618 45.032
88874 98.701 -79.627 45.026
88875 98.697 -79.653 45.020
88876 98.688 -79.682 45.014
88877 98.696 -79.687 45.008
88878 98.713 -79.679 45.002
88879 98.730 -79.670 44.997
88880 98.747 -79.661 44.993
88881 98.763 -79.661 44.986
88882 98.778 -79.660 44.979
88883 98.794 -79.659 44.972
88884 98.812 -79.665 44.964
88885 98.831 -79.676 44.953
88886 98.850 -79.688 44.942
88887 98.878 -79.695 44.933
88888 98.919 -79.696 44.923
88889 98.960 -79.697 44.912
88890 99.001 -79.698 44.902
88891 99.042 -79.699 44.893
88892 99.083 -79.700 44.882
88893 99.124 -79.701 44.870
88894 99.165 -79.702 44.860
88895 99.192 -79.714 44.849
88896 99.203 -79.728 44.838
88897 99.206 -79.742 44.827
88898 99.210 -79.757 44.818
88899 99.213 -79.771 44.808
88900 99.219 -79.785 44.798
88901 99.242 -79.791 44.790
88902 99.267 -79.797 44.781
88903 99.290 -79.803 44.771
88904 99.314 -79.810 44.761
88905 99.338 -79.816 44.752
88906 99.361 -79.822 44.743
88907 99.385 -79.828 44.733
88908 99.409 -79.835 44.725
88909 99.432 -79.841 44.717
88910 99.456 -79.847 44.708
88911 99.473 -79.844 44.700
88912 99.488 -79.838 44.691
88913 99.503 -79.832 44.683
88914 99.518 -79.827 44.675
88915 99.534 -79.820 44.666
88916 99.549 -79.814 44.658
88917 99.551 -79.834 44.651
88918 99.550 -79.862 44.643
88919 99.549 -79.890 44.635
88920 99.548 -79.918 44.627
88921 99.546 -79.946 44.619
88922 99.545 -79.974 44.610
88923 99.544 -80.002 44.604
88924 99.543 -80.030 44.596
88925 99.541 -80.059 44.589
88926 99.540 -80.087 44.582
88927 99.552 -80.112 44.574
88928 99.571 -80.137 44.567
88929 99.591 -80.162 44.561
88930 99.611 -80.186 44.554
88931 99.630 -80.210 44.547
88932 99.659 -80.243 44.539
88933 99.697 -80.284 44.531
88934 99.735 -80.325 44.521
88935 99.765 -80.362 44.511
88936 99.815 -80.346 44.502
88937 99.859 -80.359 44.490
88938 99.900 -80.381 44.479
88939 99.945 -80.397 44.468
88940 99.989 -80.413 44.456
88941 100.034 -80.430 44.444
88942 100.078 -80.446 44.431
88943 100.123 -80.463 44.417
88944 100.168 -80.478 44.404
88945 100.212 -80.495 44.391
88946 100.257 -80.512 44.377
88947 100.301 -80.528 44.362
88948 100.346 -80.544 44.348
88949 100.391 -80.560 44.334
88950 100.435 -80.577 44.321
88951 100.480 -80.593 44.307
89241 93.578 -78.689 53.639
89242 93.587 -78.686 53.628
89243 93.597 -78.684 53.617
89244 93.607 -78.682 53.606
89245 93.612 -78.677 53.596
89246 93.610 -78.667 53.586
89247 93.609 -78.658 53.575
89248 93.608 -78.648 53.566
89249 93.606 -78.639 53.558
89250 93.605 -78.630 53.550
89251 93.614 -78.618 53.545
89252 93.636 -78.603 53.538
89253 93.658 -78.588 53.530
89254 93.681 -78.573 53.519
89255 93.701 -78.561 53.504
89256 93.717 -78.555 53.487
89257 93.731 -78.550 53.467
89258 93.747 -78.545 53.446
89259 93.761 -78.539 53.425
89260 93.776 -78.534 53.404
89261 93.790 -78.529 53.382
89262 93.805 -78.523 53.360
89263 93.808 -78.510 53.340
89264 93.807 -78.493 53.319
89265 93.805 -78.476 53.300
89266 93.803 -78.459 53.282
89267 93.802 -78.442 53.264
89268 93.800 -78.424 53.246
89269 93.799 -78.407 53.229
89270 93.797 -78.390 53.210
89271 93.795 -78.373 53.191
89272 93.815 -78.358 53.171
89273 93.849 -78.354 53.151
89274 93.900 -78.358 53.132
89275 93.936 -78.355 53.115
89276 93.964 -78.366 53.098
89277 93.991 -78.377 53.082
89278 94.018 -78.388 53.066
89279 94.024 -78.401 53.048
89280 94.020 -78.413 53.029
89281 94.016 -78.426 53.009
89282 94.025 -78.433 52.987
89283 94.036 -78.438 52.966
89284 94.048 -78.444 52.943
89285 94.062 -78.444 52.921
89286 94.078 -78.441 52.903
89287 94.094 -78.438 52.887
89288 94.109 -78.435 52.871
89289 94.125 -78.432 52.855
89290 94.138 -78.430 52.840
89291 94.151 -78.429 52.826
89292 94.165 -78.428 52.811
89293 94.177 -78.428 52.795
89294 94.178 -78.439 52.779
89295 94.178 -78.451 52.762
89296 94.179 -78.463 52.743
89297 94.179 -78.475 52.727
89298 94.180 -78.486 52.711
89299 94.183 -78.492 52.696
89300 94.197 -78.492 52.679
89301 94.215 -78.495 52.663
89302 94.231 -78.501 52.647
89303 94.248 -78.506 52.629
89304 94.264 -78.511 52.611
89305 94.281 -78.517 52.593
89306 94.297 -78.523 52.577
89307 94.313 -78.529 52.559
89308 94.330 -78.534 52.541
89309 94.351 -78.529 52.523
89310 94.374 -78.523 52.502
89311 94.397 -78.517 52.480
89312 94.418 -78.512 52.457
89313 94.428 -78.526 52.435
89314 94.438 -78.539 52.411
89315 94.448 -78.552 52.385
89316 94.461 -78.548 52.357
89317 94.475 -78.542 52.324
89318 94.488 -78.535 52.291
89319 94.502 -78.529 52.258
89320 94.516 -78.523 52.225
89321 94.529 -78.519 52.193
89322 94.542 -78.521 52.163
89323 94.554 -78.523 52.134
89324 94.567 -78.526 52.102
89325 94.579 -78.529 52.069
89326 94.593 -78.531 52.034
89327 94.609 -78.532 51.999
89328 94.625 -78.534 51.961
89329 94.635 -78.532 51.922
89330 94.645 -78.531 51.882
89331 94.655 -78.529 51.839
89332 94.666 -78.528 51.793
89333 94.676 -78.526 51.745
89334 94.687 -78.525 51.696
89335 94.700 -78.526 51.641
89336 94.715 -78.529 51.585
89337 94.730 -78.532 51.530
89338 94.744 -78.535 51.477
89339 94.758 -78.539 51.425
89340 94.772 -78.541 51.372
89341 94.791 -78.545 51.318
89342 94.812 -78.548 51.266
89343 94.833 -78.551 51.214
89344 94.840 -78.555 51.163
89345 94.843 -78.559 51.114
89346 94.846 -78.564 51.066
89347 94.849 -78.567 51.020
89348 94.852 -78.572 50.976
89349 94.854 -78.576 50.933
89350 94.863 -78.578 50.891
89351 94.873 -78.578 50.850
89352 94.883 -78.562 50.808
89353 94.894 -78.545 50.766
89354 94.905 -78.528 50.723
89355 94.918 -78.510 50.680
89356 94.932 -78.493 50.636
89357 94.947 -78.475 50.592
89358 94.961 -78.457 50.547
89359 94.975 -78.439 50.504
89360 94.990 -78.422 50.460
89361 95.004 -78.404 50.416
89362 95.019 -78.385 50.372
89363 95.036 -78.379 50.328
89364 95.056 -78.384 50.284
89365 95.076 -78.389 50.241
89366 95.097 -78.395 50.200
89367 95.117 -78.400 50.158
89368 95.138 -78.405 50.115
89369 95.159 -78.410 50.072
89370 95.179 -78.415 50.031
89371 95.199 -78.420 49.990
89372 95.220 -78.426 49.950
89373 95.240 -78.431 49.912
89374 95.251 -78.434 49.874
89375 95.262 -78.438 49.837
89376 95.290 -78.452 49.800
89377 95.323 -78.469 49.762
89378 95.356 -78.485 49.724
89379 95.389 -78.503 49.686
89380 95.422 -78.520 49.649
89381 95.455 -78.536 49.611
89382 95.483 -78.551 49.574
89383 95.508 -78.563 49.539
89384 95.532 -78.575 49.503
89385 95.555 -78.586 49.467
89386 95.580 -78.598 49.433
89387 95.603 -78.611 49.397
89388 95.611 -78.612 49.362
89389 95.616 -78.611 49.328
89390 95.621 -78.611 49.294
89391 95.626 -78.611 49.260
89392 95.631 -78.610 49.225
89393 95.628 -78.610 49.190
89394 95.623 -78.609 49.154
89395 95.617 -78.608 49.117
89396 95.612 -78.608 49.078
89397 95.607 -78.608 49.036
89398 95.602 -78.607 48.993
89399 95.586 -78.603 48.950
89400 95.568 -78.598 48.908
89401 95.547 -78.595 48.867
89402 95.521 -78.595 48.827
89403 95.494 -78.595 48.788
89404 95.476 -78.595 48.749
89405 95.501 -78.595 48.709
89406 95.526 -78.595 48.668
89407 95.551 -78.596 48.626
89408 95.576 -78.597 48.585
89409 95.601 -78.598 48.544
89410 95.623 -78.599 48.505
89411 95.638 -78.602 48.467
89412 95.653 -78.606 48.429
89413 95.668 -78.611 48.392
89414 95.683 -78.615 48.355
89415 95.697 -78.620 48.319
89416 95.712 -78.624 48.281
89417 95.730 -78.627 48.241
89418 95.754 -78.626 48.200
89419 95.782 -78.623 48.158
89420 95.811 -78.619 48.118
89421 95.835 -78.618 48.079
89422 95.856 -78.619 48.040
89423 95.878 -78.620 48.000
89424 95.900 -78.619 47.959
89425 95.921 -78.616 47.918
89426 95.941 -78.614 47.878
89427 95.957 -78.613 47.838
89428 95.972 -78.610 47.797
89429 95.980 -78.615 47.757
89430 95.987 -78.621 47.717
89431 95.995 -78.627 47.677
89432 96.015 -78.626 47.636
89433 96.039 -78.623 47.596
89434 96.059 -78.634 47.556
89435 96.077 -78.651 47.516
89436 96.094 -78.669 47.478
89437 96.112 -78.686 47.439
89438 96.130 -78.704 47.400
89439 96.148 -78.722 47.362
89440 96.165 -78.739 47.324
89441 96.183 -78.757 47.286
89442 96.200 -78.774 47.250
89443 96.217 -78.762 47.214
89444 96.231 -78.758 47.178
89445 96.241 -78.767 47.143
89446 96.251 -78.775 47.109
89447 96.261 -78.783 47.078
89448 96.270 -78.792 47.048
89449 96.281 -78.800 47.019
89450 96.291 -78.808 46.988
89451 96.303 -78.812 46.958
89452 96.315 -78.816 46.929
89453 96.328 -78.819 46.899
89454 96.341 -78.823 46.869
89455 96.354 -78.826 46.839
89456 96.352 -78.815 46.810
89457 96.393 -78.817 46.781
89458 96.436 -78.818 46.752
89459 96.477 -78.820 46.725
89460 96.518 -78.817 46.698
89461 96.559 -78.812 46.672
89462 96.599 -78.807 46.648
89463 96.640 -78.802 46.625
89464 96.680 -78.798 46.603
89465 96.721 -78.793 46.581
89466 96.761 -78.788 46.561
89467 96.802 -78.783 46.539
89468 96.835 -78.779 46.518
89469 96.847 -78.778 46.496
89470 96.867 -78.789 46.475
89471 96.887 -78.799 46.455
89472 96.907 -78.810 46.435
89473 96.927 -78.820 46.415
89474 96.947 -78.824 46.395
89475 96.957 -78.858 46.374
89476 96.962 -78.864 46.354
89477 96.966 -78.871 46.337
89478 96.958 -78.884 46.321
89479 96.945 -78.901 46.305
89480 96.931 -78.918 46.288
89481 96.925 -78.933 46.273
89482 96.933 -78.944 46.257
89483 96.949 -78.955 46.240
89484 96.974 -78.961 46.221
89485 97.000 -78.966 46.204
89486 97.024 -78.971 46.186
89487 97.047 -78.975 46.169
89488 97.071 -78.980 46.153
89489 97.095 -78.984 46.136
89490 97.118 -78.989 46.119
89491 97.142 -78.993 46.102
89492 97.166 -78.998 46.088
89493 97.189 -79.002 46.074
89494 97.212 -79.008 46.060
89495 97.228 -79.019 46.048
89496 97.243 -79.030 46.034
89497 97.259 -79.041 46.019
89498 97.275 -79.052 46.003
89499 97.291 -79.063 45.988
89500 97.306 -79.074 45.971
89501 97.317 -79.080 45.952
89502 97.331 -79.085 45.929
89503 97.342 -79.084 45.907
89504 97.354 -79.084 45.886
89505 97.365 -79.083 45.869
89506 97.376 -79.082 45.854
89507 97.387 -79.081 45.839
89508 97.399 -79.081 45.823
89509 97.410 -79.080 45.805
89510 97.422 -79.079 45.787
89511 97.436 -79.078 45.768
89512 97.449 -79.078 45.750
89513 97.462 -79.078 45.736
89514 97.471 -79.075 45.724
89515 97.479 -79.079 45.712
89516 97.500 -79.093 45.696
89517 97.521 -79.106 45.677
89518 97.542 -79.119 45.657
89519 97.564 -79.132 45.638
89520 97.585 -79.146 45.618
89521 97.610 -79.169 45.599
89522 97.605 -79.169 45.581
89523 97.600 -79.169 45.565
89524 97.596 -79.168 45.549
89525 97.591 -79.168 45.532
89526 97.586 -79.167 45.514
89527 97.581 -79.166 45.497
89528 97.578 -79.165 45.480
89529 97.578 -79.165 45.464
89530 97.578 -79.165 45.449
89531 97.578 -79.165 45.434
89532 97.578 -79.164 45.419
89533 97.587 -79.175 45.405
89534 97.605 -79.195 45.392
89535 97.624 -79.216 45.380
89536 97.658 -79.238 45.369
89537 97.698 -79.262 45.359
89538 97.738 -79.285 45.348
89539 97.773 -79.306 45.337
89540 97.789 -79.314 45.327
89541 97.806 -79.322 45.317
89542 97.823 -79.332 45.308
89543 97.840 -79.340 45.299
89544 97.856 -79.348 45.289
89545 97.873 -79.357 45.279
89546 97.886 -79.370 45.271
89547 97.898 -79.385 45.264
89548 97.911 -79.401 45.256
89549 97.923 -79.416 45.249
89550 97.942 -79.429 45.242
89551 97.965 -79.438 45.234
89552 97.988 -79.448 45.227
89553 98.011 -79.458 45.219
89554 98.035 -79.466 45.210
89555 98.060 -79.473 45.202
89556 98.084 -79.481 45.193
89557 98.109 -79.488 45.184
89558 98.134 -79.494 45.174
89559 98.158 -79.490 45.164
89560 98.207 -79.485 45.154
89561 98.262 -79.480 45.144
89562 98.301 -79.479 45.133
89563 98.328 -79.484 45.123
89564 98.361 -79.496 45.116
89565 98.394 -79.507 45.107
89566 98.427 -79.520 45.099
89567 98.459 -79.531 45.092
89568 98.492 -79.543 45.085
89569 98.525 -79.554 45.078
89570 98.558 -79.563 45.072
89571 98.591 -79.572 45.066
89572 98.625 -79.581 45.060
89573 98.655 -79.589 45.053
89574 98.680 -79.598 45.047
89575 98.706 -79.607 45.041
89576 98.699 -79.636 45.035
89577 98.690 -79.665 45.029
89578 98.690 -79.681 45.022
89579 98.707 -79.672 45.014
89580 98.724 -79.664 45.007
89581 98.740 -79.660 45.000
89582 98.756 -79.659 44.994
89583 98.772 -79.658 44.987
89584 98.788 -79.657 44.980
89585 98.804 -79.661 44.971
89586 98.823 -79.672 44.960
89587 98.854 -79.678 44.950
89588 98.895 -79.679 44.940
89589 98.936 -79.680 44.928
89590 98.978 -79.681 44.917
89591 99.018 -79.683 44.906
89592 99.060 -79.683 44.896
89593 99.101 -79.684 44.884
89594 99.142 -79.685 44.873
89595 99.183 -79.686 44.863
89596 99.209 -79.698 44.853
89597 99.213 -79.712 44.842
89598 99.216 -79.726 44.832
89599 99.220 -79.740 44.823
89600 99.223 -79.755 44.814
89601 99.227 -79.769 44.805
89602 99.246 -79.777 44.796
89603 99.270 -79.783 44.786
89604 99.294 -79.789 44.776
89605 99.317 -79.796 44.766
89606 99.342 -79.802 44.756
89607 99.365 -79.808 44.746
89608 99.389 -79.814 44.737
89609 99.413 -79.821 44.728
89610 99.436 -79.827 44.720
89611 99.454 -79.826 44.711
89612 99.470 -79.820 44.702
89613 99.485 -79.814 44.694
89614 99.500 -79.808 44.685
89615 99.515 -79.803 44.677
89616 99.530 -79.797 44.669
89617 99.545 -79.791 44.662
89618 99.547 -79.814 44.655
89619 99.545 -79.842 44.647
89620 99.544 -79.870 44.639
89621 99.543 -79.898 44.632
89622 99.542 -79.926 44.624
89623 99.540 -79.954 44.615
89624 99.539 -79.982 44.608
89625 99.538 -80.010 44.601
89626 99.537 -80.038 44.593
89627 99.554 -80.063 44.587
89628 99.573 -80.087 44.579
89629 99.593 -80.112 44.572
89630 99.613 -80.137 44.566
89631 99.632 -80.161 44.558
89632 99.652 -80.186 44.551
89633 99.682 -80.220 44.543
89634 99.720 -80.261 44.534
89635 99.758 -80.302 44.525
89636 99.783 -80.336 44.516
89637 99.807 -80.356 44.506
89638 99.868 -80.350 44.495
89639 99.913 -80.361 44.484
89640 99.947 -80.391 44.473
89641 99.992 -80.407 44.461
89642 100.036 -80.424 44.448
89643 100.081 -80.441 44.436
89644 100.125 -80.457 44.423
89645 100.170 -80.473 44.410
89646 100.214 -80.489 44.396
89647 100.259 -80.506 44.382
89648 100.303 -80.522 44.367
89649 100.348 -80.538 44.353
89650 100.393 -80.555 44.339
89651 100.437 -80.571 44.325
89652 100.482 -80.588 44.311
89653 100.527 -80.604 44.296
89654 100.571 -80.620 44.282
89655 100.616 -80.636 44.267
89942 93.578 -78.678 53.644
89943 93.588 -78.676 53.632
89944 93.597 -78.673 53.620
89945 93.607 -78.670 53.609
89946 93.608 -78.663 53.598
89947 93.607 -78.654 53.587
89948 93.606 -78.645 53.575
89949 93.604 -78.636 53.565
89950 93.603 -78.626 53.555
89951 93.602 -78.617 53.546
89952 93.619 -78.602 53.538
89953 93.641 -78.587 53.528
89954 93.664 -78.572 53.518
89955 93.687 -78.557 53.505
89956 93.705 -78.545 53.490
89957 93.720 -78.539 53.471
89958 93.734 -78.535 53.451
89959 93.749 -78.529 53.431
89960 93.764 -78.524 53.412
89961 93.778 -78.519 53.392
89962 93.793 -78.514 53.371
89963 93.807 -78.508 53.350
89964 93.821 -78.501 53.330
89965 93.819 -78.485 53.310
89966 93.817 -78.467 53.291
89967 93.816 -78.451 53.272
89968 93.814 -78.433 53.254
89969 93.813 -78.416 53.235
89970 93.811 -78.399 53.218
89971 93.809 -78.382 53.201
89972 93.807 -78.364 53.182
89973 93.830 -78.350 53.163
89974 93.859 -78.336 53.144
89975 93.893 -78.337 53.125
89976 93.920 -78.348 53.108
89977 93.947 -78.359 53.092
89978 93.975 -78.370 53.076
89979 94.002 -78.381 53.060
89980 94.026 -78.392 53.043
89981 94.022 -78.405 53.025
89982 94.018 -78.418 53.005
89983 94.026 -78.425 52.985
89984 94.038 -78.430 52.963
89985 94.050 -78.435 52.942
89986 94.062 -78.441 52.921
89987 94.077 -78.438 52.905
89988 94.093 -78.435 52.890
89989 94.109 -78.432 52.875
89990 94.122 -78.431 52.860
89991 94.135 -78.430 52.845
89992 94.148 -78.429 52.831
89993 94.161 -78.428 52.817
89994 94.171 -78.429 52.802
89995 94.171 -78.442 52.786
89996 94.172 -78.453 52.770
89997 94.172 -78.465 52.753
89998 94.173 -78.477 52.736
89999 94.176 -78.482 52.721
90000 94.180 -78.486 52.706
90001 94.191 -78.488 52.690
90002 94.213 -78.483 52.674
90003 94.231 -78.486 52.658
90004 94.248 -78.492 52.641
90005 94.265 -78.497 52.623
90006 94.281 -78.503 52.605
90007 94.297 -78.508 52.589
90008 94.314 -78.514 52.570
90009 94.332 -78.517 52.553
90010 94.354 -78.510 52.535
90011 94.377 -78.504 52.514
90012 94.397 -78.501 52.492
90013 94.407 -78.514 52.470
90014 94.417 -78.528 52.447
90015 94.427 -78.542 52.422
90016 94.436 -78.554 52.396
90017 94.450 -78.552 52.369
90018 94.463 -78.545 52.336
90019 94.477 -78.539 52.303
90020 94.491 -78.532 52.270
90021 94.504 -78.526 52.238
90022 94.518 -78.520 52.206
90023 94.532 -78.513 52.175
90024 94.545 -78.509 52.145
90025 94.558 -78.512 52.114
90026 94.571 -78.514 52.081
90027 94.588 -78.516 52.046
90028 94.604 -78.517 52.011
90029 94.621 -78.519 51.973
90030 94.637 -78.520 51.934
90031 94.647 -78.518 51.893
90032 94.657 -78.517 51.850
90033 94.668 -78.515 51.802
90034 94.678 -78.514 51.752
90035 94.688 -78.512 51.702
90036 94.701 -78.514 51.646
90037 94.716 -78.517 51.591
90038 94.732 -78.520 51.535
90039 94.748 -78.524 51.481
90040 94.762 -78.527 51.429
90041 94.776 -78.529 51.376
90042 94.793 -78.532 51.323
90043 94.814 -78.536 51.270
90044 94.826 -78.540 51.219
90045 94.829 -78.544 51.168
90046 94.832 -78.548 51.120
90047 94.835 -78.552 51.071
90048 94.841 -78.555 51.024
90049 94.851 -78.557 50.981
90050 94.861 -78.558 50.939
90051 94.871 -78.560 50.899
90052 94.881 -78.546 50.858
90053 94.890 -78.529 50.816
90054 94.900 -78.514 50.773
90055 94.911 -78.497 50.729
90056 94.923 -78.479 50.685
90057 94.937 -78.462 50.640
90058 94.952 -78.444 50.594
90059 94.966 -78.426 50.548
90060 94.980 -78.408 50.504
90061 94.995 -78.390 50.460
90062 95.009 -78.373 50.415
90063 95.024 -78.354 50.371
90064 95.044 -78.359 50.327
90065 95.064 -78.364 50.284
90066 95.085 -78.369 50.240
90067 95.105 -78.374 50.198
90068 95.125 -78.379 50.156
90069 95.146 -78.385 50.114
90070 95.167 -78.390 50.073
90071 95.187 -78.395 50.033
90072 95.207 -78.400 49.993
90073 95.228 -78.405 49.953
90074 95.248 -78.410 49.916
90075 95.265 -78.415 49.879
90076 95.283 -78.423 49.842
90077 95.317 -78.440 49.804
90078 95.349 -78.457 49.766
90079 95.379 -78.472 49.729
90080 95.402 -78.484 49.691
90081 95.426 -78.495 49.653
90082 95.451 -78.507 49.616
90083 95.474 -78.520 49.579
90084 95.498 -78.532 49.543
90085 95.523 -78.543 49.506
90086 95.546 -78.555 49.471
90087 95.570 -78.567 49.436
90088 95.594 -78.579 49.400
90089 95.602 -78.580 49.366
90090 95.607 -78.580 49.332
90091 95.612 -78.579 49.298
90092 95.617 -78.579 49.263
90093 95.622 -78.579 49.228
90094 95.628 -78.579 49.194
90095 95.633 -78.578 49.159
90096 95.638 -78.578 49.123
90097 95.640 -78.577 49.085
90098 95.635 -78.576 49.045
90099 95.629 -78.576 49.003
90100 95.623 -78.575 48.961
90101 95.606 -78.570 48.919
90102 95.581 -78.570 48.878
90103 95.554 -78.569 48.839
90104 95.527 -78.569 48.800
90105 95.501 -78.569 48.762
90106 95.504 -78.569 48.721
90107 95.529 -78.570 48.680
90108 95.554 -78.570 48.637
90109 95.579 -78.571 48.595
90110 95.604 -78.571 48.554
90111 95.624 -78.573 48.515
90112 95.639 -78.577 48.476
90113 95.655 -78.581 48.438
90114 95.670 -78.585 48.400
90115 95.684 -78.589 48.362
90116 95.699 -78.594 48.325
90117 95.714 -78.598 48.286
90118 95.735 -78.599 48.246
90119 95.758 -78.598 48.206
90120 95.784 -78.596 48.165
90121 95.811 -78.594 48.124
90122 95.832 -78.595 48.085
90123 95.854 -78.595 48.046
90124 95.876 -78.596 48.006
90125 95.897 -78.597 47.965
90126 95.917 -78.595 47.925
90127 95.934 -78.594 47.885
90128 95.951 -78.593 47.845
90129 95.966 -78.590 47.805
90130 95.981 -78.587 47.766
90131 95.997 -78.583 47.727
90132 96.006 -78.589 47.687
90133 96.018 -78.592 47.647
90134 96.040 -78.594 47.607
90135 96.058 -78.611 47.567
90136 96.076 -78.629 47.528
90137 96.093 -78.647 47.488
90138 96.111 -78.664 47.449
90139 96.128 -78.682 47.410
90140 96.147 -78.699 47.372
90141 96.164 -78.717 47.334
90142 96.182 -78.734 47.296
90143 96.199 -78.752 47.260
90144 96.215 -78.758 47.224
90145 96.225 -78.766 47.189
90146 96.238 -78.769 47.155
90147 96.250 -78.773 47.123
90148 96.263 -78.776 47.092
90149 96.276 -78.779 47.061
90150 96.289 -78.783 47.030
90151 96.302 -78.786 47.000
90152 96.315 -78.789 46.970
90153 96.327 -78.793 46.940
90154 96.340 -78.796 46.911
90155 96.353 -78.799 46.882
90156 96.366 -78.802 46.852
90157 96.357 -78.788 46.823
90158 96.377 -78.777 46.795
90159 96.419 -78.779 46.767
90160 96.461 -78.780 46.741
90161 96.503 -78.782 46.714
90162 96.545 -78.784 46.690
90163 96.587 -78.786 46.667
90164 96.629 -78.787 46.645
90165 96.670 -78.786 46.622
90166 96.711 -78.780 46.601
90167 96.751 -78.776 46.581
90168 96.792 -78.771 46.561
90169 96.831 -78.766 46.538
90170 96.846 -78.769 46.516
90171 96.866 -78.780 46.494
90172 96.886 -78.790 46.474
90173 96.906 -78.801 46.453
90174 96.927 -78.811 46.432
90175 96.947 -78.820 46.412
90176 96.956 -78.840 46.391
90177 96.960 -78.846 46.370
90178 96.965 -78.852 46.353
90179 96.958 -78.865 46.337
90180 96.945 -78.883 46.320
90181 96.931 -78.900 46.304
90182 96.917 -78.918 46.289
90183 96.931 -78.931 46.272
90184 96.947 -78.943 46.253
90185 96.973 -78.948 46.234
90186 96.999 -78.953 46.215
90187 97.024 -78.959 46.197
90188 97.050 -78.964 46.179
90189 97.075 -78.969 46.163
90190 97.099 -78.974 46.147
90191 97.122 -78.978 46.131
90192 97.146 -78.983 46.114
90193 97.170 -78.987 46.100
90194 97.193 -78.992 46.087
90195 97.214 -78.998 46.075
90196 97.230 -79.009 46.062
90197 97.246 -79.020 46.047
90198 97.261 -79.031 46.032
90199 97.277 -79.042 46.018
90200 97.293 -79.053 46.003
90201 97.309 -79.065 45.987
90202 97.325 -79.071 45.970
90203 97.338 -79.070 45.947
90204 97.352 -79.070 45.924
90205 97.365 -79.069 45.901
90206 97.379 -79.069 45.882
90207 97.392 -79.069 45.867
90208 97.405 -79.068 45.852
90209 97.419 -79.068 45.836
90210 97.433 -79.068 45.819
90211 97.445 -79.067 45.802
90212 97.453 -79.065 45.783
90213 97.462 -79.062 45.765
90214 97.470 -79.060 45.750
90215 97.478 -79.058 45.736
90216 97.487 -79.056 45.723
90217 97.499 -79.078 45.706
90218 97.520 -79.091 45.688
90219 97.541 -79.104 45.668
90220 97.562 -79.118 45.651
90221 97.586 -79.135 45.632
90222 97.608 -79.156 45.614
90223 97.603 -79.155 45.597
90224 97.598 -79.155 45.579
90225 97.593 -79.154 45.561
90226 97.589 -79.153 45.542
90227 97.584 -79.153 45.524
90228 97.579 -79.153 45.508
90229 97.574 -79.152 45.492
90230 97.570 -79.151 45.476
90231 97.569 -79.151 45.461
90232 97.569 -79.150 45.445
90233 97.569 -79.150 45.430
90234 97.579 -79.162 45.417
90235 97.597 -79.182 45.404
90236 97.630 -79.204 45.392
90237 97.669 -79.228 45.380
90238 97.709 -79.252 45.371
90239 97.749 -79.275 45.360
90240 97.773 -79.288 45.350
90241 97.790 -79.297 45.339
90242 97.806 -79.305 45.329
90243 97.823 -79.313 45.320
90244 97.840 -79.322 45.310
90245 97.856 -79.331 45.301
90246 97.873 -79.339 45.292
90247 97.887 -79.351 45.284
90248 97.899 -79.366 45.276
90249 97.912 -79.382 45.269
90250 97.926 -79.396 45.261
90251 97.949 -79.406 45.255
90252 97.972 -79.416 45.247
90253 97.996 -79.426 45.239
90254 98.019 -79.435 45.231
90255 98.042 -79.444 45.222
90256 98.067 -79.451 45.212
90257 98.092 -79.459 45.203
90258 98.116 -79.466 45.194
90259 98.141 -79.465 45.184
90260 98.165 -79.461 45.174
90261 98.218 -79.456 45.164
90262 98.274 -79.451 45.155
90263 98.301 -79.454 45.145
90264 98.327 -79.457 45.136
90265 98.359 -79.468 45.130
90266 98.392 -79.479 45.121
90267 98.425 -79.491 45.114
90268 98.457 -79.503 45.107
90269 98.490 -79.515 45.100
90270 98.523 -79.527 45.094
90271 98.556 -79.538 45.088
90272 98.588 -79.551 45.083
90273 98.622 -79.561 45.077
90274 98.655 -79.570 45.070
90275 98.686 -79.579 45.063
90276 98.709 -79.590 45.057
90277 98.700 -79.619 45.050
90278 98.692 -79.648 45.044
90279 98.685 -79.674 45.037
90280 98.701 -79.666 45.028
90281 98.718 -79.659 45.020
90282 98.734 -79.658 45.014
90283 98.749 -79.657 45.007
90284 98.765 -79.657 45.000
90285 98.781 -79.655 44.991
90286 98.797 -79.657 44.982
90287 98.831 -79.661 44.971
90288 98.872 -79.663 44.960
90289 98.913 -79.664 44.949
90290 98.954 -79.664 44.936
90291 98.995 -79.665 44.924
90292 99.036 -79.667 44.913
90293 99.077 -79.667 44.903
90294 99.118 -79.668 44.891
90295 99.159 -79.670 44.880
90296 99.200 -79.670 44.870
90297 99.220 -79.681 44.861
90298 99.223 -79.695 44.850
90299 99.227 -79.710 44.840
90300 99.230 -79.724 44.830
90301 99.234 -79.739 44.821
90302 99.237 -79.753 44.812
90303 99.251 -79.764 44.802
90304 99.274 -79.770 44.792
90305 99.298 -79.776 44.782
90306 99.322 -79.783 44.772
90307 99.345 -79.789 44.761
90308 99.369 -79.795 44.751
90309 99.393 -79.801 44.742
90310 99.417 -79.808 44.733
90311 99.436 -79.808 44.724
90312 99.451 -79.802 44.714
90313 99.467 -79.796 44.705
90314 99.482 -79.791 44.697
90315 99.497 -79.785 44.688
90316 99.512 -79.779 44.680
90317 99.527 -79.773 44.674
90318 99.542 -79.767 44.667
90319 99.542 -79.793 44.661
90320 99.541 -79.821 44.654
90321 99.540 -79.849 44.647
90322 99.538 -79.877 44.639
90323 99.537 -79.905 44.629
90324 99.536 -79.933 44.620
90325 99.535 -79.961 44.612
90326 99.536 -79.989 44.604
90327 99.555 -80.014 44.597
90328 99.575 -80.038 44.591
90329 99.594 -80.062 44.584
90330 99.614 -80.087 44.577
90331 99.634 -80.112 44.571
90332 99.654 -80.137 44.563
90333 99.674 -80.161 44.556
90334 99.705 -80.196 44.548
90335 99.743 -80.237 44.539
90336 99.778 -80.277 44.530
90337 99.801 -80.309 44.521
90338 99.825 -80.342 44.511
90339 99.869 -80.347 44.500
90340 99.921 -80.354 44.490
90341 99.967 -80.362 44.478
90342 99.994 -80.402 44.466
90343 100.039 -80.418 44.454
90344 100.083 -80.435 44.441
90345 100.128 -80.450 44.429
90346 100.172 -80.467 44.415
90347 100.217 -80.484 44.401
90348 100.261 -80.500 44.386
90349 100.306 -80.516 44.372
90350 100.350 -80.532 44.358
90351 100.395 -80.549 44.343
90352 100.439 -80.565 44.329
90353 100.484 -80.582 44.314
90354 100.528 -80.598 44.300
90355 100.573 -80.614 44.285
90356 100.618 -80.631 44.270
90357 100.662 -80.647 44.256
90358 100.707 -80.663 44.241
90359 100.752 -80.679 44.227
90643 93.578 -78.667 53.649
90644 93.588 -78.665 53.636
90645 93.597 -78.662 53.624
90646 93.606 -78.659 53.611
90647 93.605 -78.650 53.599
90648 93.604 -78.641 53.587
90649 93.602 -78.632 53.575
90650 93.601 -78.622 53.564
90651 93.600 -78.613 53.553
90652 93.602 -78.601 53.541
90653 93.624 -78.586 53.530
90654 93.647 -78.572 53.517
90655 93.670 -78.557 53.504
90656 93.692 -78.542 53.490
90657 93.707 -78.532 53.474
90658 93.722 -78.526 53.456
90659 93.737 -78.519 53.438
90660 93.752 -78.514 53.419
90661 93.767 -78.509 53.400
90662 93.781 -78.504 53.381
90663 93.796 -78.498 53.362
90664 93.811 -78.493 53.341
90665 93.825 -78.488 53.321
90666 93.831 -78.476 53.301
90667 93.829 -78.459 53.283
90668 93.828 -78.442 53.264
90669 93.826 -78.425 53.245
90670 93.825 -78.407 53.226
90671 93.823 -78.391 53.209
90672 93.821 -78.373 53.191
90673 93.820 -78.356 53.173
90674 93.844 -78.341 53.155
90675 93.873 -78.328 53.137
90676 93.897 -78.334 53.119
90677 93.911 -78.342 53.101
90678 93.931 -78.352 53.086
90679 93.958 -78.363 53.071
90680 93.985 -78.374 53.056
90681 94.012 -78.385 53.039
90682 94.024 -78.397 53.021
90683 94.020 -78.410 53.003
90684 94.028 -78.416 52.982
90685 94.040 -78.422 52.961
90686 94.052 -78.427 52.941
90687 94.064 -78.432 52.922
90688 94.077 -78.435 52.908
90689 94.092 -78.433 52.894
90690 94.105 -78.432 52.880
90691 94.118 -78.431 52.865
90692 94.131 -78.429 52.850
90693 94.144 -78.429 52.836
90694 94.157 -78.427 52.825
90695 94.165 -78.432 52.809
90696 94.165 -78.444 52.794
90697 94.166 -78.455 52.779
90698 94.166 -78.467 52.762
90699 94.170 -78.472 52.747
90700 94.174 -78.476 52.732
90701 94.178 -78.481 52.718
90702 94.186 -78.483 52.702
90703 94.208 -78.479 52.687
90704 94.230 -78.475 52.672
90705 94.248 -78.478 52.654
90706 94.265 -78.483 52.637
90707 94.281 -78.489 52.620
90708 94.297 -78.495 52.604
90709 94.314 -78.500 52.586
90710 94.334 -78.498 52.568
90711 94.357 -78.491 52.550
90712 94.377 -78.490 52.529
90713 94.386 -78.504 52.507
90714 94.396 -78.517 52.484
90715 94.406 -78.530 52.460
90716 94.416 -78.544 52.434
90717 94.425 -78.557 52.408
90718 94.438 -78.555 52.380
90719 94.452 -78.549 52.348
90720 94.466 -78.542 52.316
90721 94.479 -78.535 52.283
90722 94.493 -78.529 52.250
90723 94.506 -78.523 52.219
90724 94.521 -78.517 52.188
90725 94.534 -78.510 52.156
90726 94.552 -78.499 52.124
90727 94.566 -78.498 52.090
90728 94.582 -78.500 52.056
90729 94.599 -78.501 52.021
90730 94.616 -78.503 51.983
90731 94.632 -78.504 51.943
90732 94.648 -78.506 51.902
90733 94.659 -78.504 51.857
90734 94.669 -78.503 51.808
90735 94.679 -78.501 51.757
90736 94.690 -78.500 51.706
90737 94.702 -78.501 51.651
90738 94.718 -78.504 51.596
90739 94.734 -78.508 51.540
90740 94.749 -78.511 51.486
90741 94.765 -78.515 51.433
90742 94.779 -78.519 51.381
90743 94.794 -78.521 51.328
90744 94.813 -78.525 51.276
90745 94.816 -78.529 51.224
90746 94.820 -78.532 51.174
90747 94.829 -78.534 51.126
90748 94.839 -78.535 51.079
90749 94.849 -78.537 51.033
90750 94.859 -78.539 50.990
90751 94.868 -78.540 50.948
90752 94.878 -78.530 50.908
90753 94.888 -78.514 50.868
90754 94.898 -78.498 50.825
90755 94.908 -78.482 50.782
90756 94.918 -78.465 50.737
90757 94.929 -78.448 50.691
90758 94.943 -78.431 50.644
90759 94.957 -78.413 50.597
90760 94.971 -78.395 50.549
90761 94.985 -78.377 50.504
90762 95.000 -78.359 50.459
90763 95.014 -78.341 50.414
90764 95.031 -78.334 50.370
90765 95.052 -78.339 50.326
90766 95.072 -78.344 50.283
90767 95.093 -78.349 50.241
90768 95.113 -78.354 50.198
90769 95.133 -78.360 50.156
90770 95.154 -78.365 50.115
90771 95.174 -78.369 50.076
90772 95.195 -78.375 50.037
90773 95.215 -78.380 49.997
90774 95.236 -78.385 49.957
90775 95.256 -78.391 49.919
90776 95.278 -78.398 49.882
90777 95.298 -78.404 49.844
90778 95.322 -78.416 49.807
90779 95.345 -78.429 49.768
90780 95.370 -78.441 49.730
90781 95.393 -78.452 49.693
90782 95.417 -78.464 49.657
90783 95.442 -78.476 49.620
90784 95.465 -78.489 49.583
90785 95.489 -78.500 49.547
90786 95.513 -78.512 49.512
90787 95.537 -78.524 49.476
90788 95.561 -78.535 49.440
90789 95.585 -78.548 49.405
90790 95.593 -78.549 49.370
90791 95.598 -78.549 49.336
90792 95.603 -78.548 49.303
90793 95.609 -78.548 49.268
90794 95.614 -78.548 49.234
90795 95.619 -78.548 49.201
90796 95.624 -78.547 49.167
90797 95.629 -78.547 49.132
90798 95.634 -78.546 49.095
90799 95.639 -78.545 49.055
90800 95.644 -78.545 49.014
90801 95.649 -78.545 48.972
90802 95.640 -78.544 48.931
90803 95.614 -78.544 48.893
90804 95.587 -78.544 48.853
90805 95.561 -78.543 48.812
90806 95.534 -78.543 48.775
90807 95.508 -78.543 48.735
90808 95.532 -78.543 48.692
90809 95.557 -78.544 48.649
90810 95.582 -78.545 48.606
90811 95.607 -78.545 48.565
90812 95.626 -78.548 48.526
90813 95.641 -78.551 48.486
90814 95.656 -78.555 48.448
90815 95.671 -78.559 48.410
90816 95.686 -78.564 48.372
90817 95.701 -78.568 48.333
90818 95.716 -78.572 48.294
90819 95.740 -78.571 48.254
90820 95.763 -78.570 48.214
90821 95.787 -78.569 48.174
90822 95.809 -78.570 48.133
90823 95.830 -78.570 48.092
90824 95.852 -78.572 48.052
90825 95.874 -78.573 48.012
90826 95.896 -78.573 47.972
90827 95.911 -78.573 47.933
90828 95.927 -78.573 47.892
90829 95.944 -78.573 47.851
90830 95.961 -78.571 47.810
90831 95.976 -78.568 47.773
90832 95.991 -78.564 47.735
90833 96.007 -78.561 47.697
90834 96.022 -78.558 47.659
90835 96.039 -78.571 47.620
90836 96.056 -78.588 47.581
90837 96.073 -78.604 47.542
90838 96.090 -78.622 47.502
90839 96.108 -78.639 47.463
90840 96.125 -78.656 47.424
90841 96.142 -78.673 47.384
90842 96.159 -78.689 47.346
90843 96.177 -78.707 47.308
90844 96.190 -78.720 47.272
90845 96.208 -78.738 47.236
90846 96.235 -78.742 47.202
90847 96.250 -78.745 47.169
90848 96.262 -78.749 47.138
90849 96.275 -78.752 47.107
90850 96.288 -78.756 47.076
90851 96.301 -78.759 47.044
90852 96.314 -78.762 47.014
90853 96.327 -78.766 46.985
90854 96.339 -78.769 46.955
90855 96.353 -78.772 46.925
90856 96.365 -78.776 46.897
90857 96.378 -78.779 46.868
90858 96.369 -78.765 46.839
90859 96.361 -78.738 46.811
90860 96.402 -78.739 46.784
90861 96.445 -78.742 46.758
90862 96.486 -78.743 46.732
90863 96.528 -78.745 46.707
90864 96.571 -78.746 46.684
90865 96.612 -78.748 46.662
90866 96.655 -78.749 46.642
90867 96.696 -78.751 46.622
90868 96.738 -78.752 46.602
90869 96.781 -78.754 46.581
90870 96.822 -78.754 46.558
90871 96.846 -78.761 46.535
90872 96.866 -78.771 46.513
90873 96.886 -78.781 46.492
90874 96.906 -78.792 46.472
90875 96.926 -78.802 46.451
90876 96.946 -78.812 46.429
90877 96.955 -78.822 46.406
90878 96.959 -78.829 46.383
90879 96.964 -78.835 46.363
90880 96.958 -78.847 46.347
90881 96.936 -78.869 46.332
90882 96.923 -78.890 46.317
90883 96.919 -78.905 46.301
90884 96.929 -78.919 46.283
90885 96.946 -78.930 46.264
90886 96.971 -78.935 46.244
90887 96.997 -78.940 46.226
90888 97.023 -78.946 46.208
90889 97.048 -78.951 46.190
90890 97.074 -78.956 46.174
90891 97.100 -78.962 46.157
90892 97.125 -78.967 46.142
90893 97.150 -78.971 46.127
90894 97.174 -78.976 46.114
90895 97.197 -78.981 46.101
90896 97.217 -78.989 46.089
90897 97.232 -78.999 46.074
90898 97.248 -79.011 46.060
90899 97.265 -79.021 46.046
90900 97.287 -79.032 46.031
90901 97.309 -79.043 46.016
90902 97.330 -79.053 46.000
90903 97.349 -79.059 45.981
90904 97.362 -79.059 45.959
90905 97.375 -79.059 45.937
90906 97.389 -79.058 45.915
90907 97.402 -79.058 45.895
90908 97.416 -79.058 45.879
90909 97.428 -79.056 45.864
90910 97.436 -79.054 45.849
90911 97.444 -79.052 45.834
90912 97.453 -79.049 45.817
90913 97.461 -79.047 45.800
90914 97.469 -79.045 45.783
90915 97.477 -79.043 45.768
90916 97.486 -79.040 45.752
90917 97.494 -79.038 45.736
90918 97.499 -79.060 45.720
90919 97.519 -79.076 45.703
90920 97.540 -79.089 45.685
90921 97.561 -79.103 45.669
90922 97.587 -79.124 45.651
90923 97.605 -79.142 45.632
90924 97.600 -79.141 45.613
90925 97.596 -79.140 45.593
90926 97.591 -79.140 45.573
90927 97.586 -79.140 45.554
90928 97.582 -79.139 45.536
90929 97.577 -79.139 45.520
90930 97.572 -79.138 45.504
90931 97.567 -79.137 45.489
90932 97.563 -79.137 45.473
90933 97.560 -79.137 45.457
90934 97.560 -79.136 45.442
90935 97.572 -79.149 45.429
90936 97.601 -79.171 45.415
90937 97.641 -79.194 45.403
90938 97.681 -79.218 45.391
90939 97.721 -79.242 45.382
90940 97.756 -79.263 45.372
90941 97.773 -79.271 45.361
90942 97.790 -79.279 45.352
90943 97.806 -79.288 45.341
90944 97.823 -79.296 45.331
90945 97.840 -79.305 45.321
90946 97.856 -79.313 45.312
90947 97.873 -79.322 45.305
90948 97.888 -79.332 45.297
90949 97.900 -79.347 45.288
90950 97.913 -79.363 45.280
90951 97.934 -79.374 45.273
90952 97.957 -79.384 45.266
90953 97.980 -79.394 45.258
90954 98.003 -79.404 45.251
90955 98.026 -79.413 45.242
90956 98.050 -79.422 45.233
90957 98.074 -79.429 45.224
90958 98.099 -79.437 45.214
90959 98.124 -79.440 45.204
90960 98.148 -79.436 45.195
90961 98.175 -79.432 45.185
90962 98.230 -79.427 45.175
90963 98.275 -79.425 45.166
90964 98.301 -79.428 45.157
90965 98.327 -79.432 45.149
90966 98.357 -79.440 45.142
90967 98.390 -79.452 45.134
90968 98.423 -79.463 45.127
90969 98.456 -79.476 45.121
90970 98.489 -79.487 45.115
90971 98.521 -79.499 45.109
90972 98.554 -79.510 45.103
90973 98.587 -79.523 45.098
90974 98.620 -79.534 45.092
90975 98.652 -79.546 45.086
90976 98.685 -79.558 45.080
90977 98.711 -79.573 45.073
90978 98.702 -79.601 45.066
90979 98.693 -79.630 45.059
90980 98.685 -79.659 45.052
90981 98.696 -79.659 45.043
90982 98.712 -79.657 45.035
90983 98.727 -79.657 45.026
90984 98.743 -79.655 45.019
90985 98.762 -79.645 45.011
90986 98.784 -79.629 45.002
90987 98.805 -79.643 44.992
90988 98.845 -79.645 44.981
90989 98.886 -79.646 44.969
90990 98.928 -79.647 44.957
90991 98.969 -79.648 44.945
90992 99.010 -79.649 44.933
90993 99.051 -79.650 44.922
90994 99.092 -79.651 44.911
90995 99.134 -79.652 44.900
90996 99.175 -79.653 44.890
90997 99.216 -79.654 44.880
90998 99.230 -79.664 44.871
90999 99.234 -79.679 44.860
91000 99.237 -79.693 44.849
91001 99.241 -79.708 44.839
91002 99.244 -79.722 44.829
91003 99.248 -79.736 44.819
91004 99.254 -79.750 44.809
91005 99.278 -79.756 44.799
91006 99.302 -79.762 44.788
91007 99.326 -79.768 44.778
91008 99.349 -79.775 44.769
91009 99.373 -79.781 44.759
91010 99.397 -79.787 44.749
91011 99.418 -79.790 44.739
91012 99.433 -79.785 44.730
91013 99.448 -79.779 44.720
91014 99.464 -79.773 44.710
91015 99.479 -79.767 44.702
91016 99.494 -79.761 44.694
91017 99.509 -79.755 44.685
91018 99.524 -79.750 44.678
91019 99.539 -79.745 44.671
91020 99.537 -79.773 44.665
91021 99.536 -79.801 44.658
91022 99.535 -79.829 44.651
91023 99.534 -79.857 44.642
91024 99.533 -79.885 44.634
91025 99.531 -79.913 44.625
91026 99.538 -79.939 44.617
91027 99.557 -79.965 44.610
91028 99.577 -79.989 44.603
91029 99.596 -80.013 44.597
91030 99.616 -80.038 44.589
91031 99.636 -80.062 44.583
91032 99.655 -80.087 44.576
91033 99.675 -80.112 44.569
91034 99.695 -80.137 44.562
91035 99.728 -80.173 44.553
91036 99.766 -80.214 44.545
91037 99.796 -80.250 44.535
91038 99.820 -80.284 44.526
91039 99.843 -80.316 44.517
91040 99.870 -80.343 44.506
91041 99.922 -80.350 44.495
91042 99.975 -80.358 44.484
91043 100.021 -80.363 44.472
91044 100.042 -80.411 44.459
91045 100.085 -80.428 44.447
91046 100.130 -80.445 44.434
91047 100.174 -80.461 44.420
91048 100.219 -80.478 44.406
91049 100.264 -80.494 44.391
91050 100.308 -80.510 44.377
91051 100.353 -80.526 44.362
91052 100.397 -80.543 44.348
91053 100.442 -80.560 44.333
91054 100.486 -80.576 44.317
91055 100.531 -80.592 44.302
91056 100.575 -80.608 44.287
91057 100.620 -80.625 44.272
91058 100.665 -80.641 44.258
91059 100.709 -80.657 44.243
91060 100.754 -80.674 44.229
91061 100.798 -80.690 44.215
91062 100.843 -80.707 44.201
91063 100.888 -80.723 44.187
91345 93.588 -78.654 53.641
91346 93.598 -78.651 53.628
91347 93.603 -78.646 53.614
91348 93.602 -78.637 53.601
91349 93.600 -78.627 53.588
91350 93.599 -78.618 53.575
91351 93.597 -78.609 53.563
91352 93.596 -78.600 53.550
91353 93.608 -78.586 53.537
91354 93.630 -78.571 53.524
91355 93.652 -78.556 53.511
91356 93.675 -78.541 53.496
91357 93.694 -78.528 53.481
91358 93.709 -78.520 53.464
91359 93.724 -78.513 53.446
91360 93.740 -78.505 53.428
91361 93.755 -78.498 53.410
91362 93.769 -78.494 53.392
91363 93.784 -78.489 53.373
91364 93.798 -78.483 53.353
91365 93.813 -78.478 53.333
91366 93.828 -78.473 53.314
91367 93.843 -78.467 53.295
91368 93.841 -78.451 53.276
91369 93.840 -78.433 53.257
91370 93.838 -78.416 53.237
91371 93.837 -78.399 53.219
91372 93.835 -78.382 53.200
91373 93.833 -78.365 53.183
91374 93.832 -78.347 53.165
91375 93.858 -78.334 53.148
91376 93.894 -78.322 53.130
91377 93.907 -78.331 53.112
91378 93.920 -78.339 53.095
91379 93.934 -78.348 53.080
91380 93.947 -78.357 53.066
91381 93.969 -78.366 53.051
91382 93.996 -78.378 53.034
91383 94.024 -78.388 53.017
91384 94.022 -78.401 52.999
91385 94.030 -78.408 52.978
91386 94.042 -78.413 52.959
91387 94.054 -78.419 52.940
91388 94.065 -78.424 52.922
91389 94.077 -78.429 52.909
91390 94.088 -78.432 52.896
91391 94.101 -78.432 52.882
91392 94.114 -78.430 52.868
91393 94.127 -78.429 52.855
91394 94.140 -78.428 52.842
91395 94.153 -78.427 52.831
91396 94.158 -78.434 52.817
91397 94.159 -78.446 52.803
91398 94.159 -78.457 52.790
91399 94.164 -78.462 52.773
91400 94.168 -78.467 52.758
91401 94.172 -78.470 52.743
91402 94.176 -78.475 52.729
91403 94.181 -78.479 52.714
91404 94.202 -78.475 52.699
91405 94.224 -78.470 52.684
91406 94.247 -78.467 52.667
91407 94.265 -78.469 52.650
91408 94.281 -78.474 52.634
91409 94.297 -78.480 52.618
91410 94.315 -78.485 52.601
91411 94.337 -78.478 52.584
91412 94.355 -78.479 52.567
91413 94.365 -78.492 52.545
91414 94.375 -78.506 52.521
91415 94.385 -78.519 52.497
91416 94.395 -78.532 52.472
91417 94.405 -78.546 52.445
91418 94.414 -78.559 52.417
91419 94.427 -78.559 52.388
91420 94.440 -78.552 52.358
91421 94.454 -78.546 52.327
91422 94.468 -78.539 52.295
91423 94.482 -78.533 52.262
91424 94.495 -78.526 52.230
91425 94.509 -78.520 52.198
91426 94.533 -78.504 52.166
91427 94.557 -78.488 52.132
91428 94.569 -78.482 52.098
91429 94.581 -78.484 52.063
91430 94.595 -78.486 52.028
91431 94.611 -78.488 51.990
91432 94.627 -78.489 51.950
91433 94.644 -78.491 51.907
91434 94.661 -78.492 51.860
91435 94.671 -78.491 51.811
91436 94.681 -78.489 51.761
91437 94.691 -78.488 51.709
91438 94.703 -78.488 51.655
91439 94.719 -78.492 51.600
91440 94.735 -78.495 51.545
91441 94.750 -78.499 51.490
91442 94.766 -78.502 51.437
91443 94.781 -78.506 51.385
91444 94.797 -78.510 51.334
91445 94.808 -78.511 51.282
91446 94.818 -78.513 51.231
91447 94.827 -78.514 51.181
91448 94.837 -78.516 51.133
91449 94.847 -78.517 51.087
91450 94.856 -78.519 51.042
91451 94.866 -78.520 51.000
91452 94.876 -78.514 50.958
91453 94.886 -78.498 50.918
91454 94.896 -78.482 50.878
91455 94.906 -78.466 50.835
91456 94.915 -78.449 50.790
91457 94.925 -78.433 50.744
91458 94.935 -78.416 50.696
91459 94.948 -78.400 50.648
91460 94.962 -78.382 50.599
91461 94.976 -78.363 50.550
91462 94.990 -78.346 50.503
91463 95.005 -78.328 50.457
91464 95.019 -78.310 50.413
91465 95.039 -78.313 50.369
91466 95.060 -78.319 50.325
91467 95.080 -78.324 50.282
91468 95.101 -78.329 50.240
91469 95.121 -78.335 50.198
91470 95.141 -78.339 50.157
91471 95.162 -78.344 50.117
91472 95.182 -78.350 50.079
91473 95.203 -78.355 50.040
91474 95.223 -78.360 50.000
91475 95.260 -78.390 49.959
91476 95.274 -78.383 49.921
91477 95.284 -78.374 49.883
91478 95.294 -78.376 49.845
91479 95.313 -78.385 49.808
91480 95.336 -78.398 49.771
91481 95.360 -78.409 49.734
91482 95.384 -78.421 49.697
91483 95.408 -78.433 49.662
91484 95.432 -78.445 49.625
91485 95.456 -78.457 49.588
91486 95.480 -78.469 49.552
91487 95.504 -78.481 49.516
91488 95.528 -78.493 49.480
91489 95.552 -78.504 49.444
91490 95.576 -78.517 49.409
91491 95.584 -78.518 49.375
91492 95.589 -78.517 49.341
91493 95.595 -78.517 49.308
91494 95.600 -78.517 49.275
91495 95.605 -78.517 49.242
91496 95.610 -78.516 49.208
91497 95.615 -78.516 49.175
91498 95.620 -78.515 49.140
91499 95.625 -78.515 49.103
91500 95.630 -78.514 49.064
91501 95.635 -78.514 49.024
91502 95.641 -78.515 48.984
91503 95.650 -78.522 48.944
91504 95.645 -78.518 48.905
91505 95.620 -78.518 48.866
91506 95.594 -78.517 48.827
91507 95.567 -78.517 48.788
91508 95.541 -78.517 48.747
91509 95.535 -78.517 48.705
91510 95.560 -78.518 48.661
91511 95.585 -78.519 48.619
91512 95.610 -78.519 48.578
91513 95.627 -78.522 48.538
91514 95.642 -78.526 48.498
91515 95.658 -78.529 48.459
91516 95.673 -78.533 48.421
91517 95.687 -78.538 48.382
91518 95.702 -78.542 48.344
91519 95.721 -78.544 48.304
91520 95.744 -78.543 48.265
91521 95.767 -78.543 48.226
91522 95.789 -78.547 48.186
91523 95.810 -78.546 48.145
91524 95.832 -78.547 48.103
91525 95.853 -78.548 48.061
91526 95.876 -78.548 48.020
91527 95.895 -78.549 47.980
91528 95.907 -78.548 47.940
91529 95.923 -78.549 47.899
91530 95.939 -78.549 47.858
91531 95.955 -78.549 47.818
91532 95.971 -78.551 47.781
91533 95.987 -78.548 47.743
91534 96.004 -78.546 47.704
91535 96.020 -78.544 47.666
91536 96.032 -78.543 47.627
91537 96.049 -78.557 47.588
91538 96.066 -78.575 47.550
91539 96.084 -78.592 47.511
91540 96.101 -78.609 47.472
91541 96.118 -78.626 47.433
91542 96.135 -78.642 47.393
91543 96.152 -78.659 47.355
91544 96.164 -78.670 47.319
91545 96.174 -78.695 47.283
91546 96.192 -78.713 47.249
91547 96.219 -78.717 47.217
91548 96.245 -78.721 47.185
91549 96.272 -78.726 47.156
91550 96.287 -78.729 47.125
91551 96.300 -78.733 47.094
91552 96.313 -78.736 47.061
91553 96.326 -78.739 47.032
91554 96.338 -78.742 47.003
91555 96.352 -78.745 46.974
91556 96.364 -78.749 46.945
91557 96.377 -78.752 46.916
91558 96.390 -78.755 46.887
91559 96.382 -78.742 46.858
91560 96.344 -78.698 46.829
91561 96.386 -78.701 46.802
91562 96.428 -78.702 46.776
91563 96.470 -78.704 46.750
91564 96.512 -78.705 46.724
91565 96.554 -78.707 46.701
91566 96.596 -78.708 46.679
91567 96.638 -78.710 46.658
91568 96.680 -78.711 46.638
91569 96.726 -78.717 46.618
91570 96.774 -78.727 46.596
91571 96.822 -78.737 46.574
91572 96.834 -78.748 46.552
91573 96.855 -78.758 46.529
91574 96.875 -78.769 46.509
91575 96.895 -78.780 46.489
91576 96.916 -78.790 46.468
91577 96.936 -78.801 46.445
91578 96.938 -78.819 46.421
91579 96.938 -78.837 46.397
91580 96.944 -78.843 46.377
91581 96.933 -78.845 46.360
91582 96.928 -78.861 46.344
91583 96.924 -78.877 46.329
91584 96.920 -78.893 46.311
91585 96.926 -78.906 46.293
91586 96.944 -78.917 46.274
91587 96.969 -78.922 46.255
91588 96.995 -78.927 46.237
91589 97.021 -78.933 46.220
91590 97.046 -78.938 46.202
91591 97.072 -78.943 46.186
91592 97.098 -78.949 46.171
91593 97.123 -78.954 46.156
91594 97.149 -78.959 46.141
91595 97.175 -78.965 46.128
91596 97.201 -78.970 46.116
91597 97.224 -78.979 46.103
91598 97.246 -78.990 46.088
91599 97.267 -78.999 46.073
91600 97.289 -79.010 46.058
91601 97.311 -79.021 46.043
91602 97.332 -79.031 46.028
91603 97.354 -79.042 46.012
91604 97.372 -79.048 45.993
91605 97.386 -79.047 45.971
91606 97.399 -79.047 45.950
91607 97.410 -79.046 45.929
91608 97.419 -79.043 45.908
91609 97.427 -79.041 45.891
91610 97.435 -79.039 45.876
91611 97.443 -79.037 45.862
91612 97.452 -79.034 45.848
91613 97.460 -79.032 45.832
91614 97.468 -79.030 45.816
91615 97.476 -79.028 45.800
91616 97.485 -79.025 45.786
91617 97.493 -79.023 45.770
91618 97.501 -79.020 45.752
91619 97.509 -79.024 45.737
91620 97.518 -79.061 45.721
91621 97.539 -79.075 45.705
91622 97.560 -79.087 45.690
91623 97.589 -79.113 45.671
91624 97.603 -79.128 45.651
91625 97.598 -79.128 45.629
91626 97.593 -79.127 45.606
91627 97.589 -79.126 45.586
91628 97.584 -79.126 45.567
91629 97.579 -79.125 45.549
91630 97.574 -79.125 45.533
91631 97.570 -79.124 45.517
91632 97.565 -79.124 45.500
91633 97.560 -79.123 45.484
91634 97.556 -79.122 45.469
91635 97.551 -79.122 45.454
91636 97.573 -79.137 45.440
91637 97.613 -79.161 45.427
91638 97.653 -79.184 45.414
91639 97.692 -79.208 45.402
91640 97.732 -79.232 45.394
91641 97.756 -79.245 45.383
91642 97.773 -79.253 45.372
91643 97.790 -79.262 45.360
91644 97.806 -79.270 45.350
91645 97.823 -79.278 45.340
91646 97.840 -79.287 45.330
91647 97.856 -79.296 45.321
91648 97.873 -79.304 45.313
91649 97.889 -79.313 45.305
91650 97.901 -79.328 45.296
91651 97.918 -79.342 45.289
91652 97.941 -79.352 45.282
91653 97.964 -79.362 45.275
91654 97.987 -79.372 45.269
91655 98.010 -79.382 45.262
91656 98.033 -79.391 45.254
91657 98.057 -79.400 45.246
91658 98.082 -79.407 45.237
91659 98.106 -79.415 45.227
91660 98.131 -79.411 45.217
91661 98.155 -79.407 45.208
91662 98.187 -79.403 45.198
91663 98.242 -79.397 45.188
91664 98.275 -79.399 45.178
91665 98.301 -79.402 45.170
91666 98.327 -79.406 45.162
91667 98.356 -79.412 45.155
91668 98.388 -79.424 45.147
91669 98.421 -79.435 45.140
91670 98.454 -79.447 45.134
91671 98.487 -79.459 45.128
91672 98.519 -79.471 45.123
91673 98.552 -79.483 45.118
91674 98.585 -79.494 45.113
91675 98.618 -79.507 45.105
91676 98.650 -79.518 45.099
91677 98.684 -79.530 45.094
91678 98.720 -79.542 45.087
91679 98.726 -79.546 45.080
91680 98.727 -79.548 45.073
91681 98.699 -79.616 45.067
91682 98.693 -79.645 45.058
91683 98.715 -79.629 45.048
91684 98.736 -79.612 45.038
91685 98.758 -79.595 45.030
91686 98.779 -79.591 45.021
91687 98.801 -79.607 45.012
91688 98.823 -79.622 45.002
91689 98.858 -79.628 44.989
91690 98.899 -79.629 44.978
91691 98.940 -79.630 44.966
91692 98.982 -79.631 44.953
91693 99.023 -79.632 44.943
91694 99.064 -79.633 44.932
91695 99.105 -79.634 44.921
91696 99.146 -79.635 44.911
91697 99.188 -79.636 44.902
91698 99.215 -79.636 44.891
91699 99.229 -79.648 44.880
91700 99.243 -79.663 44.869
91701 99.248 -79.677 44.859
91702 99.251 -79.692 44.849
91703 99.255 -79.706 44.838
91704 99.258 -79.720 44.828
91705 99.262 -79.735 44.818
91706 99.282 -79.742 44.807
91707 99.305 -79.748 44.796
91708 99.329 -79.755 44.786
91709 99.353 -79.761 44.777
91710 99.377 -79.767 44.766
91711 99.400 -79.773 44.756
91712 99.415 -79.767 44.746
91713 99.430 -79.761 44.736
91714 99.445 -79.755 44.727
91715 99.461 -79.749 44.718
91716 99.476 -79.744 44.709
91717 99.491 -79.738 44.700
91718 99.506 -79.732 44.691
91719 99.521 -79.726 44.682
91720 99.534 -79.724 44.677
91721 99.533 -79.752 44.669
91722 99.532 -79.780 44.661
91723 99.530 -79.808 44.655
91724 99.529 -79.836 44.646
91725 99.527 -79.864 44.638
91726 99.540 -79.890 44.630
91727 99.559 -79.915 44.624
91728 99.579 -79.939 44.617
91729 99.598 -79.964 44.610
91730 99.618 -79.989 44.603
91731 99.638 -80.013 44.595
91732 99.657 -80.038 44.588
91733 99.677 -80.062 44.582
91734 99.697 -80.087 44.575
91735 99.716 -80.112 44.568
91736 99.751 -80.149 44.560
91737 99.789 -80.190 44.551
91738 99.815 -80.225 44.541
91739 99.838 -80.257 44.532
91740 99.861 -80.290 44.523
91741 99.887 -80.321 44.512
91742 99.922 -80.347 44.501
91743 99.976 -80.355 44.490
91744 100.028 -80.363 44.478
91745 100.075 -80.364 44.465
91746 100.096 -80.412 44.453
91747 100.132 -80.439 44.439
91748 100.177 -80.455 44.425
91749 100.221 -80.472 44.411
91750 100.266 -80.488 44.397
91751 100.311 -80.504 44.382
91752 100.355 -80.521 44.367
91753 100.400 -80.537 44.352
91754 100.444 -80.554 44.337
91755 100.489 -80.569 44.321
91756 100.533 -80.586 44.305
91757 100.578 -80.603 44.290
91758 100.622 -80.619 44.275
91759 100.667 -80.635 44.260
91760 100.711 -80.651 44.245
91761 100.756 -80.668 44.231
91762 100.800 -80.685 44.217
91763 100.845 -80.701 44.202
91764 100.890 -80.717 44.188
91765 100.933 -80.735 44.175
91766 100.974 -80.757 44.161
91767 101.014 -80.779 44.147
92046 93.588 -78.643 53.644
92047 93.598 -78.640 53.631
92048 93.600 -78.633 53.617
92049 93.598 -78.623 53.603
92050 93.596 -78.614 53.590
92051 93.595 -78.605 53.575
92052 93.594 -78.596 53.562
92053 93.595 -78.586 53.548
92054 93.613 -78.570 53.534
92055 93.635 -78.555 53.520
92056 93.658 -78.540 53.505
92057 93.680 -78.525 53.490
92058 93.696 -78.514 53.474
92059 93.711 -78.507 53.456
92060 93.726 -78.500 53.439
92061 93.742 -78.492 53.421
92062 93.757 -78.485 53.403
92063 93.772 -78.478 53.385
92064 93.787 -78.473 53.366
92065 93.801 -78.468 53.347
92066 93.816 -78.463 53.327
92067 93.831 -78.457 53.308
92068 93.845 -78.452 53.288
92069 93.853 -78.442 53.269
92070 93.852 -78.425 53.250
92071 93.850 -78.407 53.232
92072 93.849 -78.391 53.212
92073 93.847 -78.373 53.193
92074 93.845 -78.357 53.175
92075 93.844 -78.339 53.158
92076 93.873 -78.326 53.140
92077 93.903 -78.319 53.123
92078 93.917 -78.328 53.106
92079 93.930 -78.337 53.089
92080 93.944 -78.345 53.073
92081 93.957 -78.354 53.059
92082 93.971 -78.363 53.045
92083 93.984 -78.371 53.028
92084 94.007 -78.381 53.011
92085 94.024 -78.393 52.994
92086 94.032 -78.400 52.974
92087 94.044 -78.405 52.956
92088 94.055 -78.410 52.938
92089 94.067 -78.415 52.923
92090 94.079 -78.420 52.909
92091 94.089 -78.425 52.896
92092 94.099 -78.429 52.883
92093 94.111 -78.430 52.871
92094 94.124 -78.429 52.859
92095 94.136 -78.428 52.848
92096 94.149 -78.426 52.837
92097 94.152 -78.436 52.825
92098 94.153 -78.448 52.813
92099 94.157 -78.452 52.800
92100 94.161 -78.456 52.784
92101 94.165 -78.460 52.769
92102 94.169 -78.464 52.754
92103 94.173 -78.469 52.739
92104 94.178 -78.473 52.725
92105 94.197 -78.470 52.710
92106 94.219 -78.466 52.695
92107 94.241 -78.462 52.679
92108 94.263 -78.458 52.664
92109 94.281 -78.460 52.649
92110 94.298 -78.466 52.633
92111 94.318 -78.466 52.616
92112 94.334 -78.468 52.599
92113 94.344 -78.482 52.581
92114 94.354 -78.495 52.558
92115 94.364 -78.508 52.534
92116 94.373 -78.521 52.508
92117 94.384 -78.535 52.481
92118 94.394 -78.548 52.453
92119 94.403 -78.561 52.425
92120 94.415 -78.562 52.397
92121 94.429 -78.556 52.368
92122 94.443 -78.549 52.336
92123 94.457 -78.543 52.303
92124 94.470 -78.536 52.268
92125 94.488 -78.526 52.236
92126 94.512 -78.510 52.203
92127 94.537 -78.493 52.171
92128 94.561 -78.476 52.137
92129 94.574 -78.469 52.101
92130 94.585 -78.468 52.066
92131 94.598 -78.470 52.030
92132 94.610 -78.472 51.992
92133 94.623 -78.474 51.952
92134 94.639 -78.476 51.909
92135 94.655 -78.477 51.862
92136 94.672 -78.479 51.814
92137 94.683 -78.477 51.764
92138 94.693 -78.476 51.713
92139 94.704 -78.475 51.659
92140 94.720 -78.479 51.603
92141 94.736 -78.482 51.548
92142 94.751 -78.486 51.494
92143 94.767 -78.490 51.442
92144 94.784 -78.489 51.390
92145 94.802 -78.483 51.339
92146 94.816 -78.493 51.288
92147 94.825 -78.495 51.237
92148 94.835 -78.496 51.188
92149 94.845 -78.498 51.140
92150 94.854 -78.499 51.095
92151 94.864 -78.501 51.052
92152 94.874 -78.498 51.011
92153 94.884 -78.482 50.970
92154 94.894 -78.466 50.929
92155 94.903 -78.450 50.889
92156 94.913 -78.433 50.844
92157 94.923 -78.417 50.798
92158 94.933 -78.401 50.750
92159 94.943 -78.385 50.701
92160 94.954 -78.368 50.650
92161 94.967 -78.351 50.600
92162 94.981 -78.332 50.550
92163 94.995 -78.315 50.502
92164 95.010 -78.297 50.456
92165 95.027 -78.288 50.412
92166 95.047 -78.294 50.367
92167 95.067 -78.299 50.324
92168 95.088 -78.304 50.281
92169 95.109 -78.309 50.239
92170 95.129 -78.314 50.197
92171 95.149 -78.319 50.158
92172 95.170 -78.325 50.119
92173 95.191 -78.331 50.082
92174 95.239 -78.376 50.042
92175 95.259 -78.380 50.001
92176 95.269 -78.369 49.960
92177 95.280 -78.357 49.921
92178 95.290 -78.352 49.883
92179 95.300 -78.354 49.846
92180 95.309 -78.357 49.808
92181 95.327 -78.366 49.773
92182 95.351 -78.378 49.738
92183 95.375 -78.390 49.703
92184 95.399 -78.402 49.667
92185 95.423 -78.413 49.630
92186 95.447 -78.426 49.592
92187 95.471 -78.438 49.556
92188 95.495 -78.450 49.520
92189 95.519 -78.461 49.484
92190 95.543 -78.473 49.448
92191 95.567 -78.485 49.414
92192 95.576 -78.487 49.379
92193 95.581 -78.486 49.345
92194 95.586 -78.486 49.313
92195 95.591 -78.485 49.282
92196 95.596 -78.485 49.250
92197 95.601 -78.485 49.216
92198 95.606 -78.485 49.182
92199 95.611 -78.484 49.147
92200 95.616 -78.484 49.110
92201 95.621 -78.483 49.072
92202 95.627 -78.484 49.034
92203 95.636 -78.492 48.996
92204 95.645 -78.498 48.956
92205 95.644 -78.498 48.918
92206 95.640 -78.494 48.880
92207 95.627 -78.492 48.841
92208 95.601 -78.492 48.802
92209 95.574 -78.492 48.761
92210 95.547 -78.492 48.718
92211 95.563 -78.492 48.675
92212 95.588 -78.492 48.633
92213 95.613 -78.493 48.592
92214 95.629 -78.497 48.552
92215 95.644 -78.500 48.511
92216 95.659 -78.504 48.472
92217 95.674 -78.507 48.433
92218 95.689 -78.511 48.394
92219 95.704 -78.516 48.356
92220 95.726 -78.516 48.317
92221 95.749 -78.517 48.278
92222 95.770 -78.521 48.239
92223 95.792 -78.525 48.198
92224 95.812 -78.523 48.157
92225 95.833 -78.523 48.115
92226 95.854 -78.523 48.072
92227 95.877 -78.524 48.031
92228 95.893 -78.525 47.990
92229 95.903 -78.524 47.949
92230 95.919 -78.524 47.908
92231 95.935 -78.524 47.868
92232 95.950 -78.526 47.828
92233 95.967 -78.529 47.789
92234 95.985 -78.533 47.750
92235 96.002 -78.532 47.711
92236 96.018 -78.529 47.672
92237 96.031 -78.529 47.633
92238 96.042 -78.529 47.595
92239 96.059 -78.545 47.558
92240 96.077 -78.562 47.519
92241 96.094 -78.579 47.481
92242 96.111 -78.595 47.442
92243 96.126 -78.609 47.404
92244 96.137 -78.620 47.368
92245 96.149 -78.636 47.332
92246 96.158 -78.670 47.296
92247 96.176 -78.688 47.264
92248 96.202 -78.692 47.233
92249 96.229 -78.696 47.203
92250 96.256 -78.701 47.174
92251 96.282 -78.705 47.144
92252 96.309 -78.709 47.112
92253 96.325 -78.712 47.081
92254 96.338 -78.716 47.052
92255 96.350 -78.719 47.023
92256 96.364 -78.722 46.994
92257 96.376 -78.726 46.965
92258 96.389 -78.729 46.936
92259 96.402 -78.733 46.907
92260 96.394 -78.718 46.877
92261 96.335 -78.664 46.849
92262 96.370 -78.661 46.822
92263 96.411 -78.663 46.795
92264 96.453 -78.664 46.768
92265 96.496 -78.666 46.742
92266 96.537 -78.667 46.717
92267 96.584 -78.675 46.695
92268 96.633 -78.685 46.673
92269 96.682 -78.695 46.652
92270 96.732 -78.705 46.632
92271 96.781 -78.714 46.610
92272 96.812 -78.724 46.589
92273 96.821 -78.736 46.568
92274 96.841 -78.746 46.545
92275 96.862 -78.757 46.525
92276 96.882 -78.767 46.505
92277 96.903 -78.777 46.484
92278 96.918 -78.790 46.460
92279 96.918 -78.808 46.436
92280 96.919 -78.827 46.412
92281 96.919 -78.846 46.391
92282 96.930 -78.855 46.374
92283 96.930 -78.847 46.357
92284 96.926 -78.863 46.340
92285 96.921 -78.879 46.322
92286 96.924 -78.894 46.303
92287 96.942 -78.904 46.284
92288 96.968 -78.909 46.266
92289 96.994 -78.915 46.249
92290 97.019 -78.920 46.232
92291 97.045 -78.925 46.216
92292 97.071 -78.930 46.200
92293 97.096 -78.936 46.186
92294 97.122 -78.941 46.170
92295 97.148 -78.946 46.154
92296 97.176 -78.956 46.142
92297 97.205 -78.967 46.130
92298 97.239 -78.982 46.117
92299 97.262 -78.990 46.101
92300 97.285 -78.999 46.086
92301 97.308 -79.008 46.070
92302 97.331 -79.016 46.054
92303 97.354 -79.024 46.039
92304 97.376 -79.033 46.023
92305 97.393 -79.035 46.004
92306 97.401 -79.033 45.981
92307 97.409 -79.031 45.961
92308 97.418 -79.028 45.941
92309 97.426 -79.026 45.921
92310 97.434 -79.024 45.905
92311 97.442 -79.021 45.890
92312 97.451 -79.019 45.875
92313 97.459 -79.017 45.862
92314 97.467 -79.015 45.846
92315 97.475 -79.012 45.830
92316 97.484 -79.009 45.814
92317 97.492 -79.007 45.802
92318 97.500 -79.005 45.787
92319 97.509 -79.002 45.771
92320 97.517 -79.000 45.756
92321 97.519 -79.040 45.742
92322 97.538 -79.059 45.726
92323 97.561 -79.075 45.708
92324 97.590 -79.103 45.690
92325 97.601 -79.114 45.669
92326 97.596 -79.113 45.644
92327 97.591 -79.113 45.621
92328 97.586 -79.112 45.601
92329 97.582 -79.112 45.581
92330 97.577 -79.112 45.563
92331 97.572 -79.111 45.547
92332 97.567 -79.110 45.531
92333 97.563 -79.109 45.515
92334 97.558 -79.109 45.497
92335 97.560 -79.106 45.481
92336 97.565 -79.106 45.466
92337 97.601 -79.129 45.452
92338 97.637 -79.153 45.438
92339 97.672 -79.175 45.425
92340 97.708 -79.199 45.414
92341 97.739 -79.219 45.404
92342 97.756 -79.227 45.393
92343 97.773 -79.236 45.382
92344 97.790 -79.244 45.369
92345 97.806 -79.253 45.359
92346 97.823 -79.261 45.349
92347 97.840 -79.269 45.339
92348 97.857 -79.278 45.330
92349 97.873 -79.287 45.322
92350 97.890 -79.295 45.314
92351 97.902 -79.310 45.305
92352 97.925 -79.319 45.299
92353 97.948 -79.330 45.292
92354 97.971 -79.340 45.285
92355 97.995 -79.350 45.279
92356 98.017 -79.360 45.272
92357 98.040 -79.369 45.265
92358 98.064 -79.379 45.258
92359 98.089 -79.385 45.249
92360 98.113 -79.387 45.240
92361 98.138 -79.382 45.230
92362 98.162 -79.378 45.220
92363 98.199 -79.373 45.210
92364 98.249 -79.370 45.200
92365 98.275 -79.373 45.191
92366 98.301 -79.376 45.184
92367 98.327 -79.380 45.176
92368 98.354 -79.384 45.168
92369 98.386 -79.396 45.160
92370 98.420 -79.407 45.153
92371 98.452 -79.419 45.146
92372 98.485 -79.432 45.141
92373 98.518 -79.443 45.136
92374 98.551 -79.455 45.132
92375 98.583 -79.466 45.127
92376 98.616 -79.479 45.118
92377 98.649 -79.490 45.111
92378 98.685 -79.503 45.105
92379 98.722 -79.515 45.098
92380 98.740 -79.522 45.090
92381 98.740 -79.524 45.082
92382 98.741 -79.526 45.076
92383 98.741 -79.529 45.068
92384 98.731 -79.562 45.058
92385 98.753 -79.545 45.047
92386 98.775 -79.554 45.036
92387 98.797 -79.570 45.026
92388 98.819 -79.585 45.017
92389 98.841 -79.601 45.009
92390 98.870 -79.611 44.998
92391 98.912 -79.612 44.987
92392 98.953 -79.613 44.976
92393 98.994 -79.614 44.963
92394 99.035 -79.615 44.953
92395 99.076 -79.616 44.942
92396 99.118 -79.617 44.931
92397 99.159 -79.618 44.922
92398 99.201 -79.614 44.912
92399 99.211 -79.620 44.901
92400 99.224 -79.632 44.890
92401 99.239 -79.646 44.879
92402 99.254 -79.661 44.868
92403 99.262 -79.675 44.858
92404 99.265 -79.689 44.848
92405 99.269 -79.704 44.837
92406 99.272 -79.718 44.827
92407 99.286 -79.729 44.816
92408 99.309 -79.735 44.804
92409 99.333 -79.741 44.794
92410 99.357 -79.747 44.784
92411 99.380 -79.754 44.773
92412 99.397 -79.749 44.763
92413 99.412 -79.743 44.754
92414 99.427 -79.738 44.744
92415 99.442 -79.732 44.735
92416 99.457 -79.726 44.726
92417 99.472 -79.720 44.717
92418 99.487 -79.714 44.706
92419 99.502 -79.708 44.697
92420 99.517 -79.703 44.688
92421 99.529 -79.704 44.682
92422 99.528 -79.732 44.674
92423 99.527 -79.760 44.666
92424 99.525 -79.788 44.660
92425 99.524 -79.816 44.652
92426 99.542 -79.841 44.644
92427 99.561 -79.865 44.637
92428 99.581 -79.890 44.631
92429 99.600 -79.914 44.624
92430 99.620 -79.939 44.616
92431 99.640 -79.964 44.609
92432 99.659 -79.989 44.602
92433 99.679 -80.013 44.596
92434 99.699 -80.037 44.589
92435 99.718 -80.062 44.582
92436 99.738 -80.087 44.574
92437 99.774 -80.126 44.566
92438 99.810 -80.165 44.558
92439 99.833 -80.198 44.548
92440 99.856 -80.231 44.539
92441 99.880 -80.263 44.530
92442 99.908 -80.292 44.519
92443 99.931 -80.320 44.508
92444 99.967 -80.345 44.496
92445 100.030 -80.359 44.484
92446 100.082 -80.366 44.472
92447 100.129 -80.365 44.459
92448 100.150 -80.413 44.445
92449 100.179 -80.450 44.431
92450 100.224 -80.466 44.417
92451 100.268 -80.482 44.403
92452 100.313 -80.498 44.387
92453 100.357 -80.515 44.372
92454 100.402 -80.532 44.356
92455 100.446 -80.547 44.341
92456 100.491 -80.564 44.326
92457 100.536 -80.580 44.310
92458 100.580 -80.597 44.293
92459 100.625 -80.613 44.278
92460 100.670 -80.629 44.263
92461 100.714 -80.646 44.247
92462 100.759 -80.662 44.233
92463 100.803 -80.679 44.219
92464 100.848 -80.695 44.204
92465 100.889 -80.715 44.190
92466 100.929 -80.737 44.176
92467 100.970 -80.759 44.163
92468 101.010 -80.781 44.150
92469 101.050 -80.803 44.137
92470 101.090 -80.826 44.123
92748 93.597 -78.629 53.633
92749 93.596 -78.620 53.620
92750 93.594 -78.611 53.605
92751 93.593 -78.601 53.591
92752 93.592 -78.592 53.575
92753 93.591 -78.582 53.561
92754 93.596 -78.570 53.546
92755 93.618 -78.554 53.531
92756 93.641 -78.540 53.516
92757 93.663 -78.525 53.501
92758 93.685 -78.510 53.485
92759 93.697 -78.502 53.468
92760 93.713 -78.495 53.451
92761 93.728 -78.487 53.433
92762 93.744 -78.479 53.416
92763 93.759 -78.473 53.397
92764 93.774 -78.465 53.378
92765 93.790 -78.457 53.359
92766 93.804 -78.453 53.341
92767 93.819 -78.448 53.321
92768 93.834 -78.442 53.302
92769 93.848 -78.437 53.283
92770 93.863 -78.432 53.263
92771 93.864 -78.416 53.245
92772 93.862 -78.399 53.226
92773 93.861 -78.382 53.206
92774 93.859 -78.365 53.187
92775 93.857 -78.347 53.169
92776 93.858 -78.331 53.151
92777 93.888 -78.317 53.134
92778 93.913 -78.316 53.117
92779 93.927 -78.326 53.100
92780 93.940 -78.334 53.083
92781 93.954 -78.342 53.067
92782 93.967 -78.351 53.052
92783 93.981 -78.360 53.037
92784 93.994 -78.368 53.020
92785 94.007 -78.377 53.004
92786 94.021 -78.385 52.987
92787 94.034 -78.391 52.969
92788 94.045 -78.396 52.952
92789 94.057 -78.401 52.936
92790 94.069 -78.406 52.922
92791 94.081 -78.410 52.909
92792 94.090 -78.415 52.898
92793 94.101 -78.420 52.887
92794 94.110 -78.424 52.875
92795 94.120 -78.428 52.866
92796 94.133 -78.427 52.856
92797 94.145 -78.426 52.846
92798 94.146 -78.438 52.834
92799 94.150 -78.442 52.823
92800 94.154 -78.446 52.811
92801 94.158 -78.450 52.795
92802 94.163 -78.454 52.779
92803 94.167 -78.458 52.763
92804 94.171 -78.463 52.749
92805 94.175 -78.467 52.735
92806 94.192 -78.466 52.721
92807 94.214 -78.461 52.706
92808 94.236 -78.457 52.691
92809 94.258 -78.454 52.677
92810 94.280 -78.449 52.664
92811 94.298 -78.451 52.649
92812 94.314 -78.457 52.632
92813 94.323 -78.470 52.613
92814 94.333 -78.484 52.593
92815 94.343 -78.497 52.569
92816 94.353 -78.510 52.544
92817 94.362 -78.523 52.517
92818 94.372 -78.537 52.489
92819 94.382 -78.550 52.460
92820 94.392 -78.564 52.432
92821 94.404 -78.566 52.405
92822 94.418 -78.560 52.375
92823 94.431 -78.553 52.342
92824 94.445 -78.546 52.307
92825 94.469 -78.531 52.272
92826 94.493 -78.514 52.239
92827 94.518 -78.498 52.207
92828 94.542 -78.482 52.174
92829 94.566 -78.465 52.139
92830 94.579 -78.457 52.103
92831 94.590 -78.451 52.066
92832 94.602 -78.454 52.030
92833 94.614 -78.456 51.992
92834 94.627 -78.457 51.952
92835 94.639 -78.460 51.910
92836 94.651 -78.461 51.863
92837 94.667 -78.463 51.814
92838 94.684 -78.464 51.764
92839 94.694 -78.463 51.714
92840 94.705 -78.463 51.660
92841 94.721 -78.466 51.605
92842 94.737 -78.470 51.549
92843 94.752 -78.473 51.495
92844 94.771 -78.465 51.443
92845 94.790 -78.457 51.391
92846 94.807 -78.458 51.341
92847 94.821 -78.471 51.290
92848 94.833 -78.476 51.241
92849 94.843 -78.478 51.194
92850 94.852 -78.479 51.148
92851 94.862 -78.481 51.105
92852 94.872 -78.482 51.063
92853 94.882 -78.466 51.022
92854 94.892 -78.450 50.981
92855 94.901 -78.434 50.938
92856 94.911 -78.417 50.894
92857 94.921 -78.401 50.848
92858 94.931 -78.385 50.800
92859 94.941 -78.369 50.751
92860 94.950 -78.353 50.700
92861 94.960 -78.336 50.649
92862 94.972 -78.319 50.599
92863 94.986 -78.301 50.549
92864 95.000 -78.284 50.501
92865 95.015 -78.266 50.455
92866 95.035 -78.269 50.410
92867 95.055 -78.273 50.366
92868 95.075 -78.279 50.323
92869 95.096 -78.284 50.281
92870 95.116 -78.289 50.239
92871 95.137 -78.294 50.198
92872 95.170 -78.317 50.157
92873 95.217 -78.363 50.118
92874 95.244 -78.376 50.079
92875 95.255 -78.366 50.040
92876 95.265 -78.354 50.000
92877 95.276 -78.343 49.959
92878 95.286 -78.332 49.921
92879 95.296 -78.330 49.883
92880 95.306 -78.333 49.846
92881 95.315 -78.335 49.810
92882 95.324 -78.338 49.776
92883 95.342 -78.347 49.742
92884 95.366 -78.359 49.708
92885 95.390 -78.370 49.672
92886 95.414 -78.382 49.634
92887 95.438 -78.395 49.597
92888 95.462 -78.407 49.560
92889 95.486 -78.418 49.523
92890 95.510 -78.430 49.487
92891 95.534 -78.442 49.452
92892 95.558 -78.454 49.418
92893 95.566 -78.456 49.384
92894 95.571 -78.455 49.351
92895 95.577 -78.455 49.319
92896 95.582 -78.454 49.289
92897 95.587 -78.454 49.258
92898 95.592 -78.454 49.225
92899 95.597 -78.454 49.190
92900 95.602 -78.453 49.155
92901 95.607 -78.452 49.118
92902 95.613 -78.454 49.081
92903 95.622 -78.460 49.044
92904 95.631 -78.467 49.007
92905 95.640 -78.475 48.969
92906 95.643 -78.477 48.931
92907 95.639 -78.473 48.893
92908 95.635 -78.470 48.856
92909 95.631 -78.467 48.817
92910 95.607 -78.466 48.775
92911 95.581 -78.466 48.732
92912 95.566 -78.466 48.689
92913 95.592 -78.467 48.648
92914 95.615 -78.467 48.607
92915 95.630 -78.471 48.566
92916 95.645 -78.475 48.526
92917 95.661 -78.479 48.487
92918 95.676 -78.482 48.448
92919 95.691 -78.485 48.409
92920 95.707 -78.489 48.371
92921 95.730 -78.491 48.332
92922 95.752 -78.495 48.292
92923 95.773 -78.500 48.252
92924 95.795 -78.503 48.211
92925 95.815 -78.501 48.169
92926 95.834 -78.499 48.127
92927 95.856 -78.499 48.085
92928 95.878 -78.500 48.043
92929 95.891 -78.500 48.002
92930 95.900 -78.499 47.960
92931 95.915 -78.499 47.920
92932 95.931 -78.500 47.879
92933 95.945 -78.502 47.837
92934 95.961 -78.506 47.797
92935 95.980 -78.510 47.758
92936 95.999 -78.514 47.719
92937 96.016 -78.515 47.680
92938 96.029 -78.515 47.641
92939 96.041 -78.515 47.602
92940 96.052 -78.515 47.565
92941 96.070 -78.532 47.527
92942 96.087 -78.548 47.490
92943 96.099 -78.560 47.453
92944 96.111 -78.570 47.417
92945 96.123 -78.580 47.383
92946 96.132 -78.611 47.349
92947 96.141 -78.646 47.313
92948 96.160 -78.663 47.281
92949 96.186 -78.667 47.252
92950 96.213 -78.671 47.223
92951 96.240 -78.676 47.192
92952 96.266 -78.680 47.162
92953 96.293 -78.684 47.132
92954 96.319 -78.688 47.101
92955 96.345 -78.692 47.074
92956 96.363 -78.695 47.044
92957 96.376 -78.698 47.014
92958 96.388 -78.702 46.987
92959 96.401 -78.705 46.958
92960 96.414 -78.709 46.928
92961 96.406 -78.695 46.897
92962 96.347 -78.641 46.870
92963 96.353 -78.622 46.843
92964 96.395 -78.623 46.815
92965 96.443 -78.632 46.787
92966 96.492 -78.642 46.760
92967 96.541 -78.651 46.734
92968 96.590 -78.661 46.710
92969 96.640 -78.671 46.688
92970 96.688 -78.681 46.666
92971 96.738 -78.691 46.645
92972 96.787 -78.701 46.624
92973 96.803 -78.712 46.604
92974 96.808 -78.723 46.582
92975 96.828 -78.733 46.560
92976 96.849 -78.744 46.539
92977 96.869 -78.755 46.518
92978 96.889 -78.765 46.496
92979 96.898 -78.780 46.473
92980 96.898 -78.799 46.450
92981 96.899 -78.817 46.427
92982 96.907 -78.833 46.405
92983 96.917 -78.843 46.387
92984 96.927 -78.854 46.369
92985 96.927 -78.850 46.351
92986 96.923 -78.866 46.333
92987 96.922 -78.881 46.315
92988 96.941 -78.891 46.297
92989 96.966 -78.896 46.278
92990 96.992 -78.902 46.261
92991 97.018 -78.907 46.245
92992 97.043 -78.912 46.229
92993 97.069 -78.918 46.213
92994 97.095 -78.923 46.197
92995 97.123 -78.933 46.181
92996 97.152 -78.944 46.165
92997 97.181 -78.955 46.153
92998 97.210 -78.965 46.141
92999 97.241 -78.978 46.128
93000 97.277 -78.995 46.114
93001 97.300 -79.003 46.098
93002 97.322 -79.012 46.082
93003 97.345 -79.021 46.066
93004 97.368 -79.029 46.051
93005 97.386 -79.031 46.034
93006 97.400 -79.029 46.016
93007 97.408 -79.024 45.997
93008 97.417 -79.018 45.977
93009 97.425 -79.012 45.956
93010 97.433 -79.009 45.936
93011 97.442 -79.006 45.920
93012 97.450 -79.003 45.904
93013 97.458 -79.001 45.888
93014 97.466 -78.999 45.874
93015 97.474 -78.996 45.858
93016 97.483 -78.994 45.843
93017 97.491 -78.992 45.828
93018 97.500 -78.990 45.815
93019 97.508 -78.987 45.801
93020 97.516 -78.985 45.786
93021 97.524 -78.983 45.772
93022 97.529 -79.005 45.758
93023 97.537 -79.044 45.741
93024 97.562 -79.065 45.723
93025 97.592 -79.092 45.704
93026 97.598 -79.100 45.682
93027 97.594 -79.100 45.658
93028 97.589 -79.099 45.636
93029 97.584 -79.099 45.615
93030 97.579 -79.098 45.596
93031 97.575 -79.097 45.577
93032 97.570 -79.096 45.561
93033 97.573 -79.094 45.545
93034 97.575 -79.091 45.528
93035 97.578 -79.088 45.510
93036 97.580 -79.086 45.494
93037 97.605 -79.100 45.479
93038 97.641 -79.123 45.464
93039 97.676 -79.147 45.450
93040 97.712 -79.169 45.437
93041 97.742 -79.189 45.425
93042 97.749 -79.194 45.414
93043 97.754 -79.206 45.403
93044 97.760 -79.219 45.392
93045 97.779 -79.227 45.379
93046 97.798 -79.235 45.368
93047 97.817 -79.244 45.358
93048 97.836 -79.252 45.349
93049 97.855 -79.260 45.340
93050 97.873 -79.269 45.332
93051 97.890 -79.278 45.324
93052 97.910 -79.288 45.316
93053 97.932 -79.297 45.310
93054 97.955 -79.307 45.303
93055 97.979 -79.317 45.296
93056 98.002 -79.327 45.289
93057 98.025 -79.337 45.282
93058 98.048 -79.347 45.276
93059 98.071 -79.356 45.269
93060 98.096 -79.362 45.260
93061 98.120 -79.357 45.251
93062 98.145 -79.353 45.241
93063 98.169 -79.349 45.230
93064 98.210 -79.344 45.220
93065 98.249 -79.344 45.210
93066 98.275 -79.347 45.203
93067 98.301 -79.350 45.196
93068 98.327 -79.354 45.188
93069 98.353 -79.357 45.180
93070 98.385 -79.368 45.173
93071 98.418 -79.380 45.166
93072 98.450 -79.391 45.159
93073 98.483 -79.404 45.153
93074 98.516 -79.415 45.147
93075 98.549 -79.427 45.141
93076 98.581 -79.439 45.134
93077 98.614 -79.451 45.124
93078 98.650 -79.463 45.115
93079 98.687 -79.475 45.108
93080 98.723 -79.487 45.102
93081 98.754 -79.498 45.095
93082 98.754 -79.500 45.086
93083 98.755 -79.502 45.078
93084 98.755 -79.504 45.071
93085 98.756 -79.507 45.061
93086 98.771 -79.519 45.051
93087 98.792 -79.534 45.039
93088 98.814 -79.549 45.030
93089 98.837 -79.565 45.021
93090 98.858 -79.580 45.014
93091 98.883 -79.594 45.004
93092 98.924 -79.595 44.994
93093 98.966 -79.596 44.983
93094 99.007 -79.597 44.972
93095 99.048 -79.598 44.962
93096 99.089 -79.599 44.951
93097 99.130 -79.600 44.941
93098 99.173 -79.599 44.931
93099 99.196 -79.598 44.920
93100 99.206 -79.604 44.908
93101 99.219 -79.615 44.899
93102 99.234 -79.629 44.888
93103 99.249 -79.644 44.877
93104 99.264 -79.658 44.868
93105 99.276 -79.673 44.857
93106 99.279 -79.688 44.847
93107 99.283 -79.701 44.836
93108 99.290 -79.715 44.824
93109 99.313 -79.721 44.812
93110 99.337 -79.727 44.801
93111 99.361 -79.733 44.790
93112 99.378 -79.732 44.781
93113 99.394 -79.726 44.773
93114 99.409 -79.720 44.764
93115 99.424 -79.714 44.754
93116 99.439 -79.708 44.745
93117 99.454 -79.702 44.735
93118 99.469 -79.697 44.726
93119 99.484 -79.691 44.715
93120 99.499 -79.685 44.706
93121 99.514 -79.679 44.698
93122 99.524 -79.683 44.692
93123 99.523 -79.711 44.684
93124 99.522 -79.739 44.675
93125 99.524 -79.767 44.667
93126 99.544 -79.792 44.659
93127 99.563 -79.816 44.652
93128 99.583 -79.841 44.645
93129 99.603 -79.865 44.640
93130 99.622 -79.890 44.632
93131 99.642 -79.914 44.624
93132 99.661 -79.939 44.617
93133 99.681 -79.964 44.611
93134 99.701 -79.988 44.604
93135 99.720 -80.013 44.596
93136 99.740 -80.037 44.590
93137 99.760 -80.062 44.582
93138 99.797 -80.102 44.574
93139 99.828 -80.140 44.565
93140 99.851 -80.172 44.556
93141 99.874 -80.205 44.547
93142 99.903 -80.234 44.538
93143 99.930 -80.262 44.527
93144 99.952 -80.290 44.516
93145 99.976 -80.319 44.504
93146 100.011 -80.344 44.492
93147 100.080 -80.362 44.479
93148 100.135 -80.371 44.466
93149 100.184 -80.368 44.453
93150 100.204 -80.414 44.439
93151 100.226 -80.460 44.424
93152 100.271 -80.476 44.409
93153 100.315 -80.493 44.394
93154 100.360 -80.509 44.378
93155 100.404 -80.526 44.363
93156 100.449 -80.541 44.346
93157 100.493 -80.558 44.331
93158 100.538 -80.575 44.315
93159 100.582 -80.591 44.299
93160 100.627 -80.607 44.283
93161 100.671 -80.623 44.267
93162 100.716 -80.640 44.252
93163 100.761 -80.656 44.237
93164 100.805 -80.673 44.223
93165 100.845 -80.695 44.208
93166 100.885 -80.717 44.193
93167 100.925 -80.739 44.179
93168 100.966 -80.761 44.165
93169 101.006 -80.784 44.152
93170 101.046 -80.806 44.139
93171 101.087 -80.828 44.125
93449 93.594 -78.616 53.636
93450 93.592 -78.607 53.621
93451 93.591 -78.598 53.606
93452 93.590 -78.588 53.591
93453 93.588 -78.579 53.576
93454 93.587 -78.570 53.560
93455 93.602 -78.554 53.545
93456 93.624 -78.539 53.529
93457 93.646 -78.524 53.513
93458 93.669 -78.509 53.497
93459 93.687 -78.497 53.481
93460 93.699 -78.489 53.464
93461 93.714 -78.482 53.446
93462 93.730 -78.474 53.428
93463 93.746 -78.467 53.411
93464 93.761 -78.460 53.392
93465 93.776 -78.452 53.373
93466 93.791 -78.445 53.354
93467 93.807 -78.438 53.335
93468 93.822 -78.432 53.315
93469 93.836 -78.427 53.296
93470 93.851 -78.422 53.278
93471 93.865 -78.416 53.259
93472 93.876 -78.408 53.240
93473 93.874 -78.391 53.221
93474 93.872 -78.374 53.200
93475 93.871 -78.357 53.182
93476 93.869 -78.339 53.164
93477 93.872 -78.322 53.146
93478 93.902 -78.309 53.129
93479 93.923 -78.314 53.112
93480 93.937 -78.322 53.095
93481 93.950 -78.331 53.078
93482 93.964 -78.340 53.062
93483 93.977 -78.348 53.046
93484 93.991 -78.357 53.030
93485 94.004 -78.366 53.013
93486 94.017 -78.374 52.998
93487 94.026 -78.378 52.982
93488 94.036 -78.382 52.965
93489 94.047 -78.387 52.949
93490 94.059 -78.391 52.935
93491 94.071 -78.397 52.922
93492 94.082 -78.401 52.911
93493 94.092 -78.406 52.901
93494 94.102 -78.410 52.892
93495 94.112 -78.414 52.881
93496 94.122 -78.419 52.871
93497 94.131 -78.423 52.862
93498 94.140 -78.427 52.853
93499 94.144 -78.432 52.842
93500 94.148 -78.435 52.831
93501 94.152 -78.440 52.819
93502 94.156 -78.444 52.802
93503 94.160 -78.448 52.786
93504 94.165 -78.453 52.771
93505 94.169 -78.457 52.759
93506 94.173 -78.461 52.745
93507 94.186 -78.461 52.730
93508 94.208 -78.457 52.716
93509 94.230 -78.453 52.702
93510 94.253 -78.449 52.689
93511 94.274 -78.445 52.675
93512 94.295 -78.438 52.660
93513 94.304 -78.454 52.642
93514 94.313 -78.471 52.623
93515 94.322 -78.486 52.602
93516 94.332 -78.499 52.577
93517 94.342 -78.513 52.550
93518 94.351 -78.526 52.522
93519 94.361 -78.539 52.493
93520 94.371 -78.552 52.464
93521 94.381 -78.566 52.435
93522 94.393 -78.570 52.408
93523 94.406 -78.563 52.376
93524 94.424 -78.553 52.343
93525 94.449 -78.536 52.308
93526 94.473 -78.520 52.272
93527 94.497 -78.503 52.239
93528 94.522 -78.487 52.207
93529 94.546 -78.470 52.174
93530 94.571 -78.454 52.138
93531 94.584 -78.445 52.102
93532 94.595 -78.439 52.065
93533 94.607 -78.438 52.027
93534 94.619 -78.439 51.990
93535 94.631 -78.442 51.950
93536 94.643 -78.443 51.908
93537 94.655 -78.445 51.861
93538 94.668 -78.447 51.812
93539 94.680 -78.449 51.762
93540 94.695 -78.451 51.712
93541 94.706 -78.450 51.659
93542 94.722 -78.454 51.604
93543 94.740 -78.451 51.548
93544 94.759 -78.442 51.494
93545 94.777 -78.434 51.442
93546 94.796 -78.426 51.391
93547 94.812 -78.434 51.341
93548 94.826 -78.446 51.291
93549 94.840 -78.458 51.244
93550 94.850 -78.460 51.198
93551 94.860 -78.461 51.155
93552 94.870 -78.463 51.113
93553 94.880 -78.451 51.072
93554 94.890 -78.434 51.030
93555 94.899 -78.418 50.987
93556 94.909 -78.401 50.943
93557 94.919 -78.385 50.896
93558 94.928 -78.369 50.847
93559 94.938 -78.353 50.798
93560 94.948 -78.337 50.748
93561 94.958 -78.320 50.697
93562 94.968 -78.304 50.646
93563 94.978 -78.288 50.595
93564 94.991 -78.270 50.546
93565 95.005 -78.252 50.498
93566 95.022 -78.243 50.452
93567 95.043 -78.248 50.407
93568 95.063 -78.253 50.364
93569 95.084 -78.259 50.322
93570 95.104 -78.263 50.280
93571 95.148 -78.304 50.238
93572 95.196 -78.350 50.197
93573 95.229 -78.373 50.156
93574 95.240 -78.363 50.116
93575 95.250 -78.351 50.076
93576 95.261 -78.340 50.037
93577 95.271 -78.329 49.998
93578 95.282 -78.318 49.958
93579 95.293 -78.307 49.920
93580 95.302 -78.308 49.883
93581 95.312 -78.311 49.846
93582 95.321 -78.313 49.811
93583 95.330 -78.316 49.777
93584 95.339 -78.319 49.743
93585 95.357 -78.327 49.709
93586 95.381 -78.339 49.672
93587 95.405 -78.351 49.635
93588 95.429 -78.363 49.598
93589 95.453 -78.375 49.561
93590 95.477 -78.387 49.526
93591 95.501 -78.399 49.492
93592 95.525 -78.410 49.456
93593 95.549 -78.423 49.423
93594 95.557 -78.425 49.390
93595 95.562 -78.424 49.357
93596 95.567 -78.423 49.326
93597 95.572 -78.423 49.295
93598 95.578 -78.423 49.264
93599 95.583 -78.423 49.233
93600 95.588 -78.422 49.199
93601 95.593 -78.422 49.163
93602 95.599 -78.423 49.128
93603 95.608 -78.430 49.092
93604 95.617 -78.437 49.057
93605 95.626 -78.444 49.021
93606 95.635 -78.451 48.984
93607 95.642 -78.457 48.946
93608 95.638 -78.453 48.909
93609 95.634 -78.449 48.872
93610 95.630 -78.446 48.833
93611 95.626 -78.442 48.791
93612 95.614 -78.440 48.749
93613 95.588 -78.440 48.706
93614 95.595 -78.440 48.665
93615 95.617 -78.442 48.623
93616 95.632 -78.445 48.581
93617 95.647 -78.449 48.542
93618 95.662 -78.453 48.502
93619 95.677 -78.457 48.463
93620 95.692 -78.460 48.424
93621 95.711 -78.465 48.386
93622 95.733 -78.470 48.346
93623 95.755 -78.473 48.304
93624 95.777 -78.478 48.262
93625 95.798 -78.481 48.220
93626 95.818 -78.479 48.179
93627 95.837 -78.477 48.137
93628 95.857 -78.476 48.095
93629 95.879 -78.476 48.055
93630 95.890 -78.476 48.015
93631 95.898 -78.475 47.973
93632 95.911 -78.475 47.932
93633 95.926 -78.476 47.892
93634 95.940 -78.479 47.850
93635 95.956 -78.482 47.811
93636 95.975 -78.487 47.771
93637 95.993 -78.491 47.732
93638 96.011 -78.496 47.695
93639 96.023 -78.498 47.657
93640 96.036 -78.499 47.619
93641 96.049 -78.500 47.582
93642 96.061 -78.502 47.544
93643 96.073 -78.510 47.507
93644 96.085 -78.520 47.469
93645 96.097 -78.530 47.434
93646 96.107 -78.552 47.400
93647 96.116 -78.587 47.365
93648 96.124 -78.622 47.330
93649 96.144 -78.638 47.300
93650 96.170 -78.642 47.269
93651 96.197 -78.646 47.239
93652 96.224 -78.651 47.210
93653 96.250 -78.655 47.181
93654 96.276 -78.658 47.151
93655 96.303 -78.663 47.120
93656 96.329 -78.667 47.093
93657 96.356 -78.671 47.065
93658 96.382 -78.675 47.036
93659 96.400 -78.679 47.007
93660 96.413 -78.682 46.978
93661 96.426 -78.686 46.949
93662 96.418 -78.672 46.919
93663 96.360 -78.617 46.891
93664 96.350 -78.599 46.862
93665 96.400 -78.609 46.833
93666 96.449 -78.619 46.806
93667 96.498 -78.629 46.779
93668 96.547 -78.639 46.753
93669 96.596 -78.648 46.728
93670 96.645 -78.658 46.706
93671 96.694 -78.668 46.683
93672 96.744 -78.678 46.661
93673 96.793 -78.688 46.639
93674 96.793 -78.699 46.618
93675 96.795 -78.710 46.596
93676 96.815 -78.720 46.574
93677 96.835 -78.731 46.553
93678 96.856 -78.742 46.532
93679 96.876 -78.752 46.510
93680 96.879 -78.770 46.487
93681 96.879 -78.788 46.464
93682 96.883 -78.805 46.441
93683 96.894 -78.819 46.420
93684 96.904 -78.830 46.401
93685 96.915 -78.840 46.382
93686 96.925 -78.851 46.364
93687 96.924 -78.853 46.345
93688 96.920 -78.869 46.328
93689 96.939 -78.878 46.310
93690 96.965 -78.883 46.291
93691 96.991 -78.889 46.273
93692 97.016 -78.894 46.257
93693 97.042 -78.899 46.242
93694 97.071 -78.910 46.226
93695 97.099 -78.921 46.210
93696 97.128 -78.931 46.195
93697 97.157 -78.943 46.180
93698 97.186 -78.953 46.167
93699 97.215 -78.964 46.154
93700 97.244 -78.975 46.141
93701 97.288 -78.997 46.127
93702 97.314 -79.008 46.111
93703 97.337 -79.017 46.095
93704 97.360 -79.025 46.078
93705 97.376 -79.025 46.062
93706 97.392 -79.024 46.045
93707 97.407 -79.024 46.028
93708 97.416 -79.018 46.010
93709 97.425 -79.012 45.991
93710 97.433 -79.006 45.970
93711 97.441 -79.000 45.949
93712 97.449 -78.994 45.933
93713 97.457 -78.988 45.917
93714 97.465 -78.984 45.901
93715 97.474 -78.981 45.885
93716 97.482 -78.979 45.870
93717 97.490 -78.977 45.855
93718 97.499 -78.974 45.841
93719 97.507 -78.972 45.828
93720 97.515 -78.970 45.815
93721 97.523 -78.968 45.801
93722 97.532 -78.965 45.787
93723 97.539 -78.969 45.771
93724 97.540 -79.021 45.754
93725 97.564 -79.053 45.736
93726 97.593 -79.081 45.715
93727 97.596 -79.086 45.693
93728 97.591 -79.086 45.671
93729 97.587 -79.085 45.650
93730 97.584 -79.084 45.629
93731 97.586 -79.081 45.610
93732 97.589 -79.078 45.592
93733 97.591 -79.076 45.575
93734 97.594 -79.073 45.558
93735 97.596 -79.070 45.541
93736 97.599 -79.068 45.523
93737 97.609 -79.071 45.507
93738 97.645 -79.093 45.492
93739 97.680 -79.117 45.478
93740 97.716 -79.140 45.463
93741 97.747 -79.160 45.450
93742 97.754 -79.165 45.438
93743 97.761 -79.170 45.425
93744 97.768 -79.178 45.413
93745 97.772 -79.192 45.401
93746 97.776 -79.206 45.389
93747 97.781 -79.219 45.380
93748 97.800 -79.227 45.370
93749 97.819 -79.235 45.360
93750 97.838 -79.244 45.351
93751 97.857 -79.252 45.342
93752 97.875 -79.247 45.333
93753 97.893 -79.240 45.326
93754 97.911 -79.232 45.320
93755 97.946 -79.259 45.314
93756 97.981 -79.287 45.308
93757 98.009 -79.305 45.300
93758 98.032 -79.315 45.293
93759 98.055 -79.325 45.285
93760 98.079 -79.334 45.277
93761 98.103 -79.333 45.269
93762 98.128 -79.328 45.260
93763 98.152 -79.324 45.250
93764 98.176 -79.320 45.241
93765 98.222 -79.316 45.230
93766 98.249 -79.319 45.221
93767 98.275 -79.322 45.214
93768 98.301 -79.325 45.207
93769 98.327 -79.328 45.200
93770 98.353 -79.332 45.192
93771 98.383 -79.340 45.184
93772 98.416 -79.352 45.176
93773 98.449 -79.363 45.169
93774 98.482 -79.375 45.162
93775 98.514 -79.388 45.154
93776 98.547 -79.399 45.146
93777 98.580 -79.411 45.137
93778 98.615 -79.423 45.128
93779 98.652 -79.435 45.118
93780 98.688 -79.447 45.109
93781 98.725 -79.460 45.102
93782 98.762 -79.472 45.095
93783 98.768 -79.476 45.087
93784 98.768 -79.478 45.077
93785 98.769 -79.480 45.070
93786 98.769 -79.482 45.061
93787 98.787 -79.496 45.051
93788 98.810 -79.513 45.041
93789 98.832 -79.529 45.031
93790 98.854 -79.545 45.023
93791 98.876 -79.560 45.017
93792 98.898 -79.575 45.009
93793 98.937 -79.578 44.999
93794 98.978 -79.579 44.988
93795 99.019 -79.580 44.979
93796 99.060 -79.581 44.969
93797 99.101 -79.582 44.959
93798 99.143 -79.583 44.950
93799 99.182 -79.576 44.937
93800 99.192 -79.582 44.924
93801 99.201 -79.587 44.913
93802 99.214 -79.598 44.903
93803 99.229 -79.613 44.893
93804 99.244 -79.627 44.883
93805 99.259 -79.642 44.874
93806 99.274 -79.657 44.864
93807 99.289 -79.671 44.853
93808 99.293 -79.686 44.842
93809 99.296 -79.700 44.832
93810 99.317 -79.708 44.820
93811 99.341 -79.714 44.809
93812 99.360 -79.714 44.799
93813 99.375 -79.708 44.790
93814 99.391 -79.702 44.781
93815 99.406 -79.696 44.772
93816 99.421 -79.691 44.762
93817 99.436 -79.685 44.752
93818 99.451 -79.679 44.742
93819 99.466 -79.673 44.732
93820 99.481 -79.667 44.723
93821 99.496 -79.661 44.714
93822 99.511 -79.656 44.707
93823 99.520 -79.663 44.700
93824 99.519 -79.691 44.692
93825 99.525 -79.718 44.683
93826 99.545 -79.742 44.675
93827 99.565 -79.767 44.667
93828 99.584 -79.792 44.660
93829 99.604 -79.816 44.653
93830 99.624 -79.841 44.648
93831 99.644 -79.865 44.641
93832 99.663 -79.889 44.633
93833 99.683 -79.914 44.625
93834 99.703 -79.939 44.620
93835 99.722 -79.964 44.613
93836 99.742 -79.988 44.605
93837 99.762 -80.013 44.598
93838 99.782 -80.038 44.590
93839 99.820 -80.079 44.582
93840 99.846 -80.113 44.574
93841 99.869 -80.146 44.565
93842 99.896 -80.176 44.556
93843 99.924 -80.205 44.547
93844 99.951 -80.234 44.537
93845 99.974 -80.262 44.526
93846 99.997 -80.289 44.514
93847 100.020 -80.317 44.501
93848 100.055 -80.343 44.489
93849 100.125 -80.360 44.475
93850 100.189 -80.375 44.462
93851 100.238 -80.374 44.447
93852 100.258 -80.416 44.432
93853 100.278 -80.463 44.417
93854 100.318 -80.487 44.402
93855 100.362 -80.504 44.386
93856 100.407 -80.519 44.369
93857 100.451 -80.536 44.353
93858 100.496 -80.552 44.336
93859 100.540 -80.569 44.320
93860 100.585 -80.585 44.304
93861 100.629 -80.601 44.288
93862 100.674 -80.618 44.272
93863 100.718 -80.634 44.257
93864 100.760 -80.654 44.242
93865 100.800 -80.676 44.227
93866 100.841 -80.698 44.212
93867 100.881 -80.720 44.197
93868 100.921 -80.742 44.183
93869 100.962 -80.764 44.169
93870 101.002 -80.786 44.155
93871 101.042 -80.808 44.142
93872 101.082 -80.830 44.128
93873 101.122 -80.852 44.114
94143 93.531 -78.625 53.747
94144 93.541 -78.623 53.732
94145 93.551 -78.620 53.717
94146 93.560 -78.617 53.702
94147 93.570 -78.615 53.686
94148 93.579 -78.613 53.671
94149 93.589 -78.611 53.655
94150 93.590 -78.602 53.638
94151 93.589 -78.593 53.623
94152 93.588 -78.584 53.607
94153 93.586 -78.575 53.591
94154 93.585 -78.566 53.576
94155 93.589 -78.554 53.559
94156 93.607 -78.539 53.543
94157 93.629 -78.523 53.527
94158 93.652 -78.508 53.510
94159 93.674 -78.494 53.494
94160 93.688 -78.484 53.477
94161 93.701 -78.476 53.459
94162 93.716 -78.469 53.441
94163 93.732 -78.461 53.423
94164 93.747 -78.454 53.405
94165 93.763 -78.447 53.386
94166 93.778 -78.439 53.368
94167 93.793 -78.432 53.349
94168 93.809 -78.425 53.330
94169 93.824 -78.417 53.311
94170 93.839 -78.412 53.292
94171 93.854 -78.407 53.273
94172 93.868 -78.401 53.254
94173 93.883 -78.396 53.235
94174 93.887 -78.382 53.216
94175 93.885 -78.365 53.197
94176 93.883 -78.348 53.178
94177 93.882 -78.331 53.159
94178 93.887 -78.314 53.142
94179 93.920 -78.303 53.125
94180 93.933 -78.311 53.108
94181 93.946 -78.319 53.092
94182 93.960 -78.329 53.074
94183 93.974 -78.337 53.057
94184 93.987 -78.345 53.041
94185 94.000 -78.354 53.024
94186 94.013 -78.362 53.009
94187 94.021 -78.366 52.994
94188 94.029 -78.369 52.979
94189 94.037 -78.373 52.963
94190 94.049 -78.378 52.948
94191 94.061 -78.382 52.934
94192 94.073 -78.388 52.924
94193 94.084 -78.392 52.914
94194 94.094 -78.396 52.905
94195 94.104 -78.401 52.896
94196 94.113 -78.405 52.886
94197 94.123 -78.410 52.876
94198 94.132 -78.414 52.867
94199 94.136 -78.423 52.859
94200 94.141 -78.428 52.849
94201 94.146 -78.430 52.838
94202 94.150 -78.434 52.825
94203 94.154 -78.438 52.808
94204 94.158 -78.442 52.793
94205 94.162 -78.447 52.779
94206 94.166 -78.451 52.767
94207 94.171 -78.455 52.754
94208 94.181 -78.457 52.741
94209 94.203 -78.452 52.727
94210 94.225 -78.448 52.713
94211 94.247 -78.445 52.699
94212 94.266 -78.433 52.684
94213 94.286 -78.425 52.668
94214 94.300 -78.435 52.649
94215 94.309 -78.452 52.628
94216 94.317 -78.470 52.606
94217 94.326 -78.487 52.580
94218 94.334 -78.504 52.552
94219 94.343 -78.521 52.523
94220 94.351 -78.539 52.493
94221 94.360 -78.554 52.464
94222 94.370 -78.568 52.435
94223 94.382 -78.573 52.405
94224 94.405 -78.557 52.372
94225 94.429 -78.542 52.337
94226 94.454 -78.525 52.301
94227 94.478 -78.508 52.266
94228 94.502 -78.492 52.234
94229 94.527 -78.476 52.202
94230 94.551 -78.459 52.169
94231 94.575 -78.442 52.132
94232 94.590 -78.433 52.095
94233 94.601 -78.428 52.057
94234 94.611 -78.422 52.020
94235 94.623 -78.423 51.981
94236 94.635 -78.425 51.941
94237 94.647 -78.427 51.897
94238 94.660 -78.429 51.851
94239 94.672 -78.431 51.803
94240 94.684 -78.433 51.755
94241 94.696 -78.435 51.706
94242 94.708 -78.437 51.653
94243 94.726 -78.430 51.599
94244 94.745 -78.422 51.544
94245 94.764 -78.414 51.490
94246 94.782 -78.407 51.437
94247 94.801 -78.399 51.386
94248 94.817 -78.409 51.337
94249 94.831 -78.421 51.290
94250 94.845 -78.433 51.244
94251 94.858 -78.442 51.201
94252 94.868 -78.443 51.160
94253 94.878 -78.435 51.119
94254 94.888 -78.418 51.077
94255 94.897 -78.402 51.032
94256 94.907 -78.385 50.987
94257 94.917 -78.369 50.941
94258 94.926 -78.353 50.892
94259 94.936 -78.337 50.842
94260 94.946 -78.321 50.792
94261 94.956 -78.304 50.741
94262 94.966 -78.288 50.690
94263 94.975 -78.272 50.639
94264 94.985 -78.256 50.589
94265 94.996 -78.239 50.541
94266 95.011 -78.221 50.494
94267 95.030 -78.223 50.448
94268 95.051 -78.228 50.403
94269 95.079 -78.244 50.361
94270 95.126 -78.291 50.320
94271 95.174 -78.336 50.278
94272 95.214 -78.370 50.236
94273 95.225 -78.359 50.195
94274 95.235 -78.348 50.155
94275 95.246 -78.337 50.114
94276 95.257 -78.326 50.073
94277 95.267 -78.314 50.034
94278 95.278 -78.304 49.995
94279 95.288 -78.292 49.957
94280 95.299 -78.283 49.920
94281 95.308 -78.286 49.883
94282 95.317 -78.289 49.847
94283 95.327 -78.291 49.812
94284 95.336 -78.294 49.778
94285 95.345 -78.297 49.743
94286 95.355 -78.300 49.709
94287 95.372 -78.308 49.672
94288 95.396 -78.320 49.635
94289 95.420 -78.332 49.599
94290 95.444 -78.344 49.563
94291 95.468 -78.356 49.530
94292 95.492 -78.367 49.497
94293 95.516 -78.379 49.462
94294 95.540 -78.391 49.429
94295 95.548 -78.393 49.397
94296 95.553 -78.393 49.365
94297 95.558 -78.392 49.333
94298 95.563 -78.392 49.302
94299 95.569 -78.391 49.271
94300 95.574 -78.391 49.240
94301 95.579 -78.391 49.207
94302 95.585 -78.392 49.173
94303 95.594 -78.399 49.138
94304 95.603 -78.406 49.104
94305 95.612 -78.413 49.070
94306 95.621 -78.420 49.034
94307 95.631 -78.427 48.997
94308 95.640 -78.435 48.961
94309 95.637 -78.432 48.924
94310 95.633 -78.429 48.888
94311 95.629 -78.425 48.847
94312 95.625 -78.422 48.807
94313 95.621 -78.418 48.765
94314 95.617 -78.415 48.724
94315 95.598 -78.414 48.682
94316 95.618 -78.416 48.640
94317 95.633 -78.420 48.598
94318 95.649 -78.424 48.559
94319 95.665 -78.430 48.519
94320 95.681 -78.435 48.480
94321 95.697 -78.442 48.440
94322 95.712 -78.444 48.400
94323 95.736 -78.448 48.359
94324 95.758 -78.452 48.317
94325 95.780 -78.456 48.275
94326 95.801 -78.459 48.232
94327 95.821 -78.457 48.190
94328 95.840 -78.455 48.149
94329 95.860 -78.454 48.107
94330 95.879 -78.452 48.067
94331 95.888 -78.451 48.027
94332 95.897 -78.451 47.986
94333 95.907 -78.450 47.946
94334 95.920 -78.453 47.906
94335 95.935 -78.455 47.865
94336 95.951 -78.459 47.826
94337 95.969 -78.463 47.786
94338 95.985 -78.470 47.748
94339 95.999 -78.476 47.711
94340 96.012 -78.479 47.673
94341 96.025 -78.480 47.636
94342 96.037 -78.483 47.598
94343 96.050 -78.492 47.561
94344 96.062 -78.497 47.523
94345 96.073 -78.498 47.486
94346 96.083 -78.499 47.452
94347 96.091 -78.528 47.417
94348 96.099 -78.562 47.382
94349 96.108 -78.597 47.348
94350 96.128 -78.613 47.317
94351 96.154 -78.617 47.286
94352 96.181 -78.621 47.256
94353 96.207 -78.626 47.228
94354 96.234 -78.630 47.200
94355 96.260 -78.633 47.171
94356 96.287 -78.638 47.140
94357 96.313 -78.642 47.112
94358 96.340 -78.646 47.084
94359 96.366 -78.650 47.056
94360 96.393 -78.655 47.027
94361 96.419 -78.658 46.998
94362 96.438 -78.662 46.969
94363 96.431 -78.649 46.940
94364 96.413 -78.624 46.910
94365 96.409 -78.610 46.880
94366 96.405 -78.595 46.851
94367 96.455 -78.605 46.824
94368 96.504 -78.615 46.798
94369 96.553 -78.626 46.772
94370 96.602 -78.636 46.746
94371 96.651 -78.645 46.724
94372 96.700 -78.655 46.701
94373 96.749 -78.665 46.678
94374 96.783 -78.675 46.656
94375 96.784 -78.686 46.633
94376 96.784 -78.697 46.611
94377 96.802 -78.708 46.589
94378 96.822 -78.718 46.568
94379 96.842 -78.729 46.547
94380 96.859 -78.741 46.526
94381 96.859 -78.760 46.502
94382 96.860 -78.778 46.479
94383 96.871 -78.793 46.457
94384 96.882 -78.805 46.436
94385 96.892 -78.816 46.416
94386 96.902 -78.827 46.397
94387 96.912 -78.837 46.378
94388 96.923 -78.849 46.360
94389 96.921 -78.856 46.341
94390 96.938 -78.865 46.323
94391 96.963 -78.871 46.305
94392 96.989 -78.876 46.287
94393 97.018 -78.886 46.270
94394 97.046 -78.897 46.255
94395 97.075 -78.908 46.240
94396 97.104 -78.919 46.225
94397 97.133 -78.930 46.211
94398 97.162 -78.941 46.196
94399 97.191 -78.952 46.182
94400 97.220 -78.962 46.169
94401 97.249 -78.974 46.156
94402 97.287 -78.991 46.140
94403 97.328 -79.012 46.125
94404 97.351 -79.021 46.109
94405 97.367 -79.020 46.091
94406 97.382 -79.018 46.074
94407 97.398 -79.018 46.058
94408 97.414 -79.017 46.041
94409 97.424 -79.012 46.021
94410 97.432 -79.007 46.003
94411 97.440 -79.001 45.983
94412 97.448 -78.995 45.962
94413 97.457 -78.989 45.944
94414 97.465 -78.983 45.928
94415 97.473 -78.977 45.913
94416 97.481 -78.971 45.897
94417 97.490 -78.965 45.881
94418 97.498 -78.959 45.867
94419 97.506 -78.956 45.854
94420 97.514 -78.954 45.842
94421 97.522 -78.952 45.830
94422 97.531 -78.949 45.816
94423 97.539 -78.947 45.800
94424 97.547 -78.945 45.784
94425 97.550 -78.985 45.766
94426 97.571 -79.029 45.747
94427 97.595 -79.070 45.726
94428 97.597 -79.071 45.704
94429 97.599 -79.068 45.683
94430 97.602 -79.066 45.663
94431 97.604 -79.063 45.643
94432 97.607 -79.060 45.623
94433 97.609 -79.058 45.604
94434 97.612 -79.055 45.587
94435 97.614 -79.052 45.569
94436 97.617 -79.049 45.553
94437 97.619 -79.046 45.536
94438 97.649 -79.065 45.521
94439 97.684 -79.087 45.506
94440 97.720 -79.111 45.493
94441 97.752 -79.131 45.478
94442 97.759 -79.137 45.465
94443 97.767 -79.141 45.452
94444 97.774 -79.147 45.439
94445 97.781 -79.152 45.426
94446 97.785 -79.165 45.414
94447 97.789 -79.178 45.402
94448 97.793 -79.192 45.392
94449 97.797 -79.206 45.382
94450 97.802 -79.219 45.372
94451 97.821 -79.224 45.363
94452 97.839 -79.216 45.353
94453 97.857 -79.208 45.344
94454 97.874 -79.200 45.336
94455 97.892 -79.191 45.329
94456 97.910 -79.183 45.322
94457 97.928 -79.175 45.316
94458 97.962 -79.202 45.308
94459 97.997 -79.229 45.300
94460 98.032 -79.256 45.292
94461 98.084 -79.300 45.283
94462 98.110 -79.303 45.275
94463 98.135 -79.300 45.266
94464 98.159 -79.295 45.257
94465 98.183 -79.291 45.250
94466 98.223 -79.289 45.240
94467 98.249 -79.293 45.231
94468 98.275 -79.296 45.224
94469 98.301 -79.300 45.217
94470 98.327 -79.303 45.210
94471 98.353 -79.306 45.203
94472 98.381 -79.313 45.193
94473 98.414 -79.324 45.184
94474 98.447 -79.336 45.175
94475 98.480 -79.347 45.167
94476 98.512 -79.360 45.158
94477 98.545 -79.371 45.148
94478 98.580 -79.383 45.138
94479 98.617 -79.395 45.129
94480 98.653 -79.407 45.118
94481 98.690 -79.420 45.107
94482 98.726 -79.432 45.099
94483 98.763 -79.444 45.091
94484 98.781 -79.451 45.083
94485 98.782 -79.454 45.074
94486 98.782 -79.456 45.067
94487 98.783 -79.458 45.058
94488 98.803 -79.473 45.049
94489 98.827 -79.491 45.039
94490 98.850 -79.508 45.030
94491 98.872 -79.523 45.023
94492 98.894 -79.539 45.018
94493 98.916 -79.554 45.011
94494 98.949 -79.561 45.002
94495 98.991 -79.562 44.991
94496 99.031 -79.563 44.980
94497 99.073 -79.564 44.971
94498 99.114 -79.565 44.961
94499 99.162 -79.557 44.952
94500 99.177 -79.560 44.940
94501 99.187 -79.565 44.928
94502 99.197 -79.570 44.917
94503 99.209 -79.582 44.906
94504 99.224 -79.596 44.896
94505 99.239 -79.610 44.886
94506 99.254 -79.626 44.877
94507 99.269 -79.640 44.867
94508 99.284 -79.654 44.857
94509 99.299 -79.669 44.848
94510 99.307 -79.683 44.838
94511 99.321 -79.694 44.827
94512 99.342 -79.696 44.817
94513 99.357 -79.690 44.807
94514 99.372 -79.685 44.798
94515 99.387 -79.679 44.789
94516 99.403 -79.673 44.779
94517 99.418 -79.667 44.770
94518 99.433 -79.661 44.759
94519 99.448 -79.655 44.748
94520 99.463 -79.650 44.738
94521 99.478 -79.644 44.729
94522 99.493 -79.638 44.721
94523 99.508 -79.632 44.714
94524 99.515 -79.642 44.706
94525 99.527 -79.668 44.698
94526 99.547 -79.693 44.690
94527 99.567 -79.717 44.681
94528 99.586 -79.742 44.674
94529 99.606 -79.767 44.668
94530 99.626 -79.792 44.661
94531 99.645 -79.816 44.656
94532 99.665 -79.840 44.650
94533 99.685 -79.865 44.643
94534 99.704 -79.889 44.636
94535 99.724 -79.914 44.630
94536 99.744 -79.939 44.623
94537 99.763 -79.964 44.616
94538 99.783 -79.988 44.608
94539 99.804 -80.015 44.600
94540 99.841 -80.055 44.591
94541 99.864 -80.087 44.582
94542 99.890 -80.118 44.574
94543 99.918 -80.147 44.565
94544 99.946 -80.176 44.556
94545 99.973 -80.204 44.547
94546 99.996 -80.232 44.536
94547 100.018 -80.260 44.525
94548 100.042 -80.288 44.512
94549 100.065 -80.316 44.499
94550 100.100 -80.341 44.486
94551 100.169 -80.359 44.472
94552 100.239 -80.377 44.457
94553 100.292 -80.380 44.442
94554 100.312 -80.416 44.426
94555 100.332 -80.465 44.410
94556 100.364 -80.497 44.393
94557 100.409 -80.513 44.376
94558 100.454 -80.530 44.359
94559 100.498 -80.546 44.343
94560 100.543 -80.563 44.326
94561 100.587 -80.579 44.309
94562 100.632 -80.595 44.293
94563 100.676 -80.612 44.277
94564 100.716 -80.634 44.262
94565 100.757 -80.656 44.248
94566 100.796 -80.678 44.232
94567 100.837 -80.701 44.218
94568 100.877 -80.723 44.203
94569 100.917 -80.745 44.189
94570 100.958 -80.767 44.174
94571 100.998 -80.789 44.159
94572 101.038 -80.811 44.145
94573 101.078 -80.833 44.131
94574 101.119 -80.855 44.118
94575 101.159 -80.877 44.105
94843 93.522 -78.617 53.766
94844 93.532 -78.614 53.752
94845 93.541 -78.611 53.737
94846 93.551 -78.609 53.721
94847 93.561 -78.607 53.706
94848 93.570 -78.604 53.690
94849 93.580 -78.601 53.674
94850 93.588 -78.598 53.658
94851 93.587 -78.589 53.641
94852 93.586 -78.580 53.625
94853 93.584 -78.571 53.609
94854 93.583 -78.562 53.592
94855 93.582 -78.553 53.575
94856 93.590 -78.538 53.557
94857 93.612 -78.523 53.541
94858 93.635 -78.508 53.525
94859 93.657 -78.493 53.508
94860 93.677 -78.479 53.491
94861 93.689 -78.471 53.473
94862 93.703 -78.463 53.455
94863 93.718 -78.456 53.437
94864 93.734 -78.448 53.418
94865 93.749 -78.442 53.399
94866 93.765 -78.434 53.381
94867 93.780 -78.426 53.363
94868 93.795 -78.420 53.344
94869 93.811 -78.412 53.326
94870 93.826 -78.404 53.307
94871 93.842 -78.398 53.288
94872 93.856 -78.391 53.268
94873 93.871 -78.386 53.249
94874 93.886 -78.381 53.230
94875 93.898 -78.374 53.211
94876 93.897 -78.357 53.193
94877 93.895 -78.340 53.174
94878 93.894 -78.322 53.155
94879 93.901 -78.307 53.138
94880 93.929 -78.300 53.121
94881 93.943 -78.308 53.104
94882 93.957 -78.317 53.089
94883 93.970 -78.326 53.072
94884 93.983 -78.334 53.055
94885 93.997 -78.343 53.039
94886 94.008 -78.349 53.023
94887 94.016 -78.353 53.009
94888 94.025 -78.357 52.995
94889 94.033 -78.360 52.981
94890 94.041 -78.364 52.967
94891 94.051 -78.368 52.953
94892 94.063 -78.373 52.938
94893 94.075 -78.378 52.927
94894 94.085 -78.382 52.917
94895 94.096 -78.387 52.907
94896 94.105 -78.391 52.897
94897 94.115 -78.395 52.887
94898 94.124 -78.401 52.877
94899 94.128 -78.410 52.869
94900 94.133 -78.419 52.861
94901 94.138 -78.426 52.852
94902 94.142 -78.429 52.842
94903 94.147 -78.431 52.831
94904 94.151 -78.433 52.814
94905 94.156 -78.436 52.799
94906 94.160 -78.441 52.786
94907 94.164 -78.445 52.775
94908 94.168 -78.449 52.764
94909 94.176 -78.452 52.752
94910 94.197 -78.448 52.738
94911 94.219 -78.444 52.724
94912 94.239 -78.432 52.708
94913 94.257 -78.418 52.691
94914 94.277 -78.412 52.672
94915 94.296 -78.416 52.652
94916 94.305 -78.433 52.630
94917 94.313 -78.451 52.607
94918 94.322 -78.468 52.579
94919 94.330 -78.485 52.550
94920 94.339 -78.502 52.520
94921 94.347 -78.520 52.490
94922 94.356 -78.537 52.461
94923 94.364 -78.554 52.431
94924 94.384 -78.561 52.400
94925 94.409 -78.546 52.364
94926 94.433 -78.530 52.328
94927 94.458 -78.514 52.292
94928 94.482 -78.497 52.257
94929 94.507 -78.480 52.226
94930 94.531 -78.464 52.194
94931 94.556 -78.448 52.161
94932 94.580 -78.431 52.124
94933 94.595 -78.422 52.084
94934 94.606 -78.416 52.045
94935 94.616 -78.410 52.007
94936 94.627 -78.407 51.967
94937 94.639 -78.409 51.925
94938 94.651 -78.410 51.880
94939 94.664 -78.413 51.833
94940 94.676 -78.415 51.786
94941 94.688 -78.416 51.738
94942 94.700 -78.419 51.690
94943 94.712 -78.419 51.639
94944 94.725 -78.420 51.586
94945 94.744 -78.411 51.532
94946 94.763 -78.404 51.478
94947 94.781 -78.396 51.427
94948 94.800 -78.388 51.377
94949 94.814 -78.393 51.330
94950 94.831 -78.402 51.286
94951 94.848 -78.410 51.243
94952 94.864 -78.420 51.202
94953 94.876 -78.419 51.162
94954 94.886 -78.402 51.119
94955 94.895 -78.386 51.073
94956 94.905 -78.369 51.025
94957 94.915 -78.354 50.977
94958 94.924 -78.337 50.929
94959 94.934 -78.321 50.880
94960 94.944 -78.305 50.829
94961 94.954 -78.288 50.778
94962 94.964 -78.272 50.727
94963 94.973 -78.256 50.677
94964 94.983 -78.240 50.628
94965 94.993 -78.224 50.580
94966 95.003 -78.207 50.533
94967 95.018 -78.197 50.488
94968 95.057 -78.232 50.443
94969 95.105 -78.277 50.399
94970 95.153 -78.322 50.356
94971 95.199 -78.367 50.315
94972 95.210 -78.356 50.274
94973 95.220 -78.344 50.232
94974 95.231 -78.334 50.192
94975 95.242 -78.322 50.152
94976 95.252 -78.311 50.113
94977 95.263 -78.300 50.073
94978 95.273 -78.289 50.034
94979 95.284 -78.278 49.996
94980 95.295 -78.266 49.957
94981 95.305 -78.261 49.920
94982 95.314 -78.264 49.885
94983 95.323 -78.267 49.849
94984 95.333 -78.270 49.814
94985 95.342 -78.272 49.779
94986 95.351 -78.275 49.744
94987 95.361 -78.279 49.708
94988 95.370 -78.282 49.672
94989 95.387 -78.288 49.636
94990 95.411 -78.300 49.602
94991 95.435 -78.313 49.567
94992 95.459 -78.324 49.535
94993 95.483 -78.336 49.503
94994 95.507 -78.348 49.469
94995 95.531 -78.360 49.436
94996 95.539 -78.362 49.405
94997 95.544 -78.362 49.374
94998 95.550 -78.361 49.342
94999 95.555 -78.361 49.311
95000 95.560 -78.360 49.279
95001 95.565 -78.360 49.247
95002 95.571 -78.361 49.215
95003 95.580 -78.368 49.183
95004 95.589 -78.376 49.150
95005 95.598 -78.382 49.116
95006 95.608 -78.390 49.082
95007 95.617 -78.397 49.047
95008 95.626 -78.404 49.010
95009 95.635 -78.411 48.974
95010 95.636 -78.411 48.938
95011 95.632 -78.408 48.902
95012 95.628 -78.404 48.862
95013 95.624 -78.401 48.822
95014 95.620 -78.398 48.781
95015 95.615 -78.394 48.740
95016 95.611 -78.391 48.699
95017 95.621 -78.393 48.659
95018 95.637 -78.399 48.618
95019 95.654 -78.405 48.578
95020 95.670 -78.411 48.538
95021 95.686 -78.417 48.497
95022 95.701 -78.422 48.455
95023 95.713 -78.423 48.414
95024 95.737 -78.427 48.372
95025 95.760 -78.431 48.331
95026 95.783 -78.435 48.289
95027 95.804 -78.437 48.246
95028 95.824 -78.435 48.204
95029 95.842 -78.434 48.161
95030 95.859 -78.433 48.120
95031 95.874 -78.432 48.080
95032 95.885 -78.432 48.041
95033 95.895 -78.431 48.001
95034 95.905 -78.430 47.961
95035 95.917 -78.431 47.921
95036 95.930 -78.432 47.881
95037 95.945 -78.435 47.842
95038 95.959 -78.442 47.803
95039 95.973 -78.449 47.765
95040 95.987 -78.456 47.729
95041 96.001 -78.460 47.691
95042 96.013 -78.464 47.653
95043 96.026 -78.472 47.614
95044 96.038 -78.480 47.576
95045 96.051 -78.488 47.539
95046 96.062 -78.492 47.505
95047 96.070 -78.500 47.470
95048 96.075 -78.511 47.435
95049 96.083 -78.538 47.400
95050 96.092 -78.573 47.366
95051 96.112 -78.588 47.334
95052 96.138 -78.592 47.303
95053 96.165 -78.596 47.274
95054 96.191 -78.600 47.246
95055 96.218 -78.604 47.218
95056 96.244 -78.608 47.190
95057 96.270 -78.613 47.161
95058 96.297 -78.617 47.131
95059 96.324 -78.621 47.102
95060 96.350 -78.625 47.074
95061 96.377 -78.630 47.048
95062 96.405 -78.634 47.018
95063 96.434 -78.640 46.988
95064 96.463 -78.644 46.958
95065 96.472 -78.636 46.927
95066 96.468 -78.621 46.898
95067 96.464 -78.607 46.870
95068 96.461 -78.592 46.843
95069 96.510 -78.602 46.818
95070 96.559 -78.612 46.791
95071 96.608 -78.622 46.764
95072 96.657 -78.632 46.743
95073 96.707 -78.642 46.721
95074 96.755 -78.651 46.698
95075 96.773 -78.662 46.675
95076 96.774 -78.673 46.653
95077 96.774 -78.684 46.630
95078 96.789 -78.695 46.607
95079 96.809 -78.705 46.584
95080 96.829 -78.716 46.563
95081 96.839 -78.731 46.542
95082 96.839 -78.749 46.519
95083 96.847 -78.766 46.497
95084 96.859 -78.780 46.475
95085 96.870 -78.792 46.452
95086 96.880 -78.802 46.433
95087 96.890 -78.813 46.414
95088 96.900 -78.824 46.396
95089 96.910 -78.834 46.378
95090 96.920 -78.846 46.359
95091 96.936 -78.852 46.339
95092 96.965 -78.863 46.320
95093 96.994 -78.874 46.302
95094 97.023 -78.885 46.286
95095 97.051 -78.896 46.270
95096 97.080 -78.907 46.255
95097 97.109 -78.918 46.240
95098 97.138 -78.928 46.224
95099 97.167 -78.940 46.209
95100 97.196 -78.950 46.197
95101 97.225 -78.961 46.184
95102 97.254 -78.972 46.170
95103 97.286 -78.985 46.154
95104 97.333 -79.010 46.139
95105 97.357 -79.014 46.122
95106 97.373 -79.013 46.103
95107 97.388 -79.012 46.085
95108 97.404 -79.012 46.068
95109 97.420 -79.011 46.050
95110 97.432 -79.007 46.031
95111 97.440 -79.001 46.013
95112 97.448 -78.996 45.993
95113 97.456 -78.990 45.972
95114 97.464 -78.984 45.956
95115 97.472 -78.977 45.940
95116 97.480 -78.971 45.924
95117 97.489 -78.966 45.908
95118 97.497 -78.960 45.893
95119 97.505 -78.954 45.879
95120 97.513 -78.948 45.866
95121 97.522 -78.942 45.855
95122 97.530 -78.936 45.842
95123 97.538 -78.932 45.829
95124 97.546 -78.930 45.814
95125 97.555 -78.927 45.798
95126 97.560 -78.949 45.779
95127 97.583 -78.993 45.759
95128 97.604 -79.031 45.737
95129 97.617 -79.050 45.715
95130 97.620 -79.048 45.694
95131 97.623 -79.045 45.674
95132 97.626 -79.042 45.655
95133 97.628 -79.040 45.636
95134 97.631 -79.037 45.616
95135 97.633 -79.034 45.598
95136 97.636 -79.031 45.581
95137 97.638 -79.028 45.565
95138 97.652 -79.035 45.549
95139 97.688 -79.058 45.534
95140 97.724 -79.081 45.521
95141 97.756 -79.103 45.507
95142 97.764 -79.107 45.493
95143 97.771 -79.112 45.479
95144 97.778 -79.118 45.466
95145 97.786 -79.123 45.454
95146 97.793 -79.128 45.441
95147 97.798 -79.137 45.428
95148 97.802 -79.151 45.416
95149 97.806 -79.165 45.405
95150 97.810 -79.178 45.395
95151 97.814 -79.192 45.385
95152 97.826 -79.200 45.374
95153 97.842 -79.193 45.364
95154 97.857 -79.182 45.355
95155 97.872 -79.172 45.346
95156 97.887 -79.161 45.337
95157 97.901 -79.151 45.329
95158 97.917 -79.140 45.321
95159 97.932 -79.130 45.313
95160 97.947 -79.120 45.305
95161 98.008 -79.172 45.297
95162 98.069 -79.225 45.287
95163 98.114 -79.258 45.279
95164 98.141 -79.267 45.271
95165 98.166 -79.266 45.263
95166 98.191 -79.262 45.257
95167 98.223 -79.263 45.249
95168 98.249 -79.267 45.240
95169 98.275 -79.270 45.232
95170 98.301 -79.274 45.225
95171 98.327 -79.277 45.217
95172 98.353 -79.280 45.208
95173 98.379 -79.285 45.197
95174 98.412 -79.296 45.186
95175 98.445 -79.308 45.176
95176 98.478 -79.319 45.166
95177 98.511 -79.332 45.156
95178 98.545 -79.344 45.146
95179 98.582 -79.356 45.136
95180 98.618 -79.368 45.126
95181 98.655 -79.380 45.116
95182 98.692 -79.392 45.106
95183 98.728 -79.404 45.098
95184 98.765 -79.417 45.088
95185 98.795 -79.427 45.079
95186 98.795 -79.429 45.071
95187 98.796 -79.432 45.064
95188 98.796 -79.434 45.056
95189 98.819 -79.451 45.047
95190 98.843 -79.468 45.038
95191 98.866 -79.485 45.030
95192 98.889 -79.503 45.024
95193 98.912 -79.518 45.018
95194 98.933 -79.534 45.012
95195 98.962 -79.545 45.004
95196 99.003 -79.545 44.993
95197 99.044 -79.546 44.982
95198 99.085 -79.548 44.972
95199 99.131 -79.542 44.962
95200 99.162 -79.537 44.953
95201 99.172 -79.543 44.942
95202 99.182 -79.548 44.931
95203 99.192 -79.554 44.919
95204 99.205 -79.565 44.907
95205 99.219 -79.579 44.898
95206 99.234 -79.594 44.888
95207 99.250 -79.609 44.879
95208 99.264 -79.623 44.868
95209 99.279 -79.638 44.860
95210 99.294 -79.652 44.852
95211 99.309 -79.667 44.842
95212 99.324 -79.679 44.833
95213 99.339 -79.673 44.824
95214 99.354 -79.667 44.814
95215 99.369 -79.661 44.805
95216 99.384 -79.656 44.795
95217 99.400 -79.650 44.785
95218 99.415 -79.645 44.775
95219 99.431 -79.639 44.765
95220 99.446 -79.634 44.755
95221 99.462 -79.629 44.745
95222 99.477 -79.623 44.735
95223 99.492 -79.617 44.728
95224 99.507 -79.612 44.720
95225 99.531 -79.619 44.712
95226 99.549 -79.644 44.704
95227 99.569 -79.668 44.696
95228 99.588 -79.693 44.688
95229 99.608 -79.717 44.681
95230 99.628 -79.742 44.676
95231 99.647 -79.767 44.670
95232 99.667 -79.791 44.665
95233 99.687 -79.816 44.659
95234 99.706 -79.840 44.653
95235 99.726 -79.865 44.646
95236 99.746 -79.889 44.640
95237 99.765 -79.914 44.633
95238 99.785 -79.939 44.626
95239 99.804 -79.963 44.618
95240 99.828 -79.991 44.610
95241 99.859 -80.028 44.601
95242 99.883 -80.061 44.592
95243 99.912 -80.089 44.584
95244 99.940 -80.118 44.576
95245 99.969 -80.146 44.568
95246 99.994 -80.175 44.558
95247 100.017 -80.203 44.548
95248 100.040 -80.231 44.537
95249 100.063 -80.259 44.525
95250 100.086 -80.287 44.511
95251 100.114 -80.313 44.497
95252 100.150 -80.338 44.483
95253 100.213 -80.358 44.468
95254 100.283 -80.375 44.453
95255 100.346 -80.386 44.437
95256 100.366 -80.418 44.419
95257 100.386 -80.466 44.402
95258 100.411 -80.508 44.385
95259 100.456 -80.524 44.368
95260 100.500 -80.541 44.351
95261 100.545 -80.557 44.333
95262 100.590 -80.573 44.317
95263 100.632 -80.592 44.301
95264 100.672 -80.614 44.285
95265 100.712 -80.636 44.269
95266 100.753 -80.659 44.254
95267 100.792 -80.681 44.238
95268 100.833 -80.703 44.224
95269 100.873 -80.725 44.209
95270 100.913 -80.747 44.194
95271 100.954 -80.769 44.179
95272 100.994 -80.791 44.164
95273 101.034 -80.813 44.149
95274 101.074 -80.835 44.135
95275 101.115 -80.857 44.122
95276 101.159 -80.872 44.108
95277 101.206 -80.885 44.094
95543 93.512 -78.608 53.784
95544 93.522 -78.606 53.770
95545 93.532 -78.603 53.757
95546 93.542 -78.601 53.741
95547 93.551 -78.598 53.725
95548 93.561 -78.596 53.710
95549 93.570 -78.593 53.694
95550 93.580 -78.591 53.677
95551 93.585 -78.586 53.661
95552 93.584 -78.576 53.643
95553 93.582 -78.567 53.626
95554 93.581 -78.557 53.609
95555 93.580 -78.548 53.592
95556 93.581 -78.538 53.574
95557 93.595 -78.523 53.556
95558 93.618 -78.507 53.539
95559 93.640 -78.492 53.522
95560 93.662 -78.477 53.504
95561 93.679 -78.467 53.487
95562 93.690 -78.458 53.469
95563 93.705 -78.451 53.451
95564 93.720 -78.443 53.433
95565 93.735 -78.436 53.415
95566 93.751 -78.429 53.395
95567 93.767 -78.421 53.377
95568 93.782 -78.414 53.358
95569 93.797 -78.407 53.339
95570 93.813 -78.399 53.321
95571 93.828 -78.391 53.302
95572 93.844 -78.385 53.283
95573 93.859 -78.377 53.264
95574 93.874 -78.371 53.245
95575 93.889 -78.366 53.226
95576 93.903 -78.360 53.207
95577 93.909 -78.348 53.189
95578 93.907 -78.331 53.170
95579 93.906 -78.314 53.152
95580 93.916 -78.298 53.135
95581 93.939 -78.297 53.118
95582 93.953 -78.306 53.102
95583 93.967 -78.314 53.087
95584 93.980 -78.322 53.070
95585 93.993 -78.332 53.054
95586 94.003 -78.337 53.039
95587 94.011 -78.341 53.025
95588 94.019 -78.344 53.011
95589 94.028 -78.347 52.998
95590 94.036 -78.351 52.984
95591 94.044 -78.355 52.972
95592 94.053 -78.359 52.958
95593 94.065 -78.364 52.944
95594 94.077 -78.369 52.931
95595 94.087 -78.373 52.921
95596 94.097 -78.378 52.910
95597 94.107 -78.382 52.899
95598 94.116 -78.387 52.887
95599 94.120 -78.396 52.877
95600 94.125 -78.404 52.869
95601 94.129 -78.413 52.861
95602 94.134 -78.423 52.854
95603 94.139 -78.428 52.845
95604 94.143 -78.430 52.835
95605 94.148 -78.432 52.819
95606 94.152 -78.435 52.805
95607 94.157 -78.437 52.792
95608 94.161 -78.439 52.782
95609 94.166 -78.443 52.772
95610 94.170 -78.448 52.762
95611 94.192 -78.443 52.749
95612 94.211 -78.431 52.733
95613 94.229 -78.417 52.716
95614 94.248 -78.404 52.696
95615 94.268 -78.400 52.674
95616 94.289 -78.401 52.651
95617 94.301 -78.414 52.627
95618 94.310 -78.432 52.603
95619 94.318 -78.449 52.574
95620 94.326 -78.467 52.543
95621 94.335 -78.483 52.513
95622 94.343 -78.501 52.483
95623 94.352 -78.518 52.453
95624 94.360 -78.535 52.421
95625 94.381 -78.529 52.389
95626 94.414 -78.535 52.353
95627 94.438 -78.519 52.316
95628 94.463 -78.502 52.281
95629 94.487 -78.485 52.246
95630 94.511 -78.469 52.215
95631 94.536 -78.453 52.184
95632 94.560 -78.436 52.151
95633 94.585 -78.420 52.111
95634 94.601 -78.410 52.070
95635 94.611 -78.404 52.029
95636 94.622 -78.398 51.988
95637 94.632 -78.392 51.946
95638 94.644 -78.392 51.903
95639 94.656 -78.395 51.857
95640 94.668 -78.396 51.809
95641 94.680 -78.398 51.762
95642 94.692 -78.401 51.715
95643 94.705 -78.402 51.667
95644 94.717 -78.401 51.618
95645 94.729 -78.402 51.567
95646 94.743 -78.401 51.515
95647 94.762 -78.393 51.463
95648 94.780 -78.385 51.413
95649 94.799 -78.378 51.365
95650 94.812 -78.381 51.322
95651 94.827 -78.388 51.280
95652 94.844 -78.397 51.240
95653 94.861 -78.405 51.201
95654 94.886 -78.407 51.159
95655 94.897 -78.408 51.113
95656 94.907 -78.410 51.065
95657 94.918 -78.411 51.014
95658 94.929 -78.410 50.965
95659 94.938 -78.393 50.915
95660 94.948 -78.376 50.866
95661 94.958 -78.359 50.815
95662 94.968 -78.342 50.764
95663 94.977 -78.325 50.714
95664 94.987 -78.308 50.666
95665 94.997 -78.291 50.618
95666 95.006 -78.274 50.571
95667 95.016 -78.257 50.525
95668 95.075 -78.271 50.481
95669 95.131 -78.310 50.438
95670 95.179 -78.355 50.395
95671 95.195 -78.353 50.352
95672 95.205 -78.341 50.311
95673 95.216 -78.330 50.270
95674 95.227 -78.319 50.230
95675 95.237 -78.308 50.190
95676 95.248 -78.297 50.150
95677 95.258 -78.286 50.112
95678 95.269 -78.275 50.073
95679 95.280 -78.263 50.035
95680 95.290 -78.252 49.997
95681 95.301 -78.241 49.959
95682 95.311 -78.239 49.922
95683 95.320 -78.242 49.886
95684 95.329 -78.245 49.851
95685 95.339 -78.248 49.815
95686 95.348 -78.250 49.780
95687 95.357 -78.253 49.745
95688 95.367 -78.257 49.709
95689 95.376 -78.260 49.674
95690 95.386 -78.262 49.639
95691 95.402 -78.269 49.606
95692 95.426 -78.281 49.573
95693 95.450 -78.293 49.542
95694 95.474 -78.305 49.510
95695 95.497 -78.317 49.477
95696 95.522 -78.329 49.446
95697 95.531 -78.331 49.415
95698 95.536 -78.331 49.385
95699 95.541 -78.330 49.353
95700 95.546 -78.329 49.322
95701 95.551 -78.329 49.291
95702 95.557 -78.330 49.260
95703 95.566 -78.338 49.229
95704 95.575 -78.344 49.197
95705 95.584 -78.352 49.164
95706 95.594 -78.359 49.129
95707 95.603 -78.366 49.094
95708 95.612 -78.373 49.058
95709 95.621 -78.380 49.022
95710 95.630 -78.387 48.986
95711 95.635 -78.391 48.951
95712 95.631 -78.388 48.916
95713 95.627 -78.384 48.878
95714 95.623 -78.380 48.837
95715 95.618 -78.377 48.796
95716 95.614 -78.373 48.754
95717 95.610 -78.370 48.714
95718 95.625 -78.379 48.674
95719 95.642 -78.381 48.633
95720 95.659 -78.387 48.591
95721 95.675 -78.393 48.551
95722 95.692 -78.398 48.509
95723 95.705 -78.401 48.467
95724 95.717 -78.403 48.426
95725 95.738 -78.407 48.385
95726 95.761 -78.410 48.343
95727 95.784 -78.413 48.302
95728 95.804 -78.416 48.260
95729 95.821 -78.416 48.219
95730 95.838 -78.415 48.177
95731 95.855 -78.414 48.137
95732 95.869 -78.413 48.097
95733 95.880 -78.413 48.057
95734 95.891 -78.412 48.016
95735 95.905 -78.413 47.976
95736 95.917 -78.413 47.936
95737 95.929 -78.414 47.896
95738 95.943 -78.417 47.856
95739 95.955 -78.424 47.818
95740 95.967 -78.430 47.780
95741 95.979 -78.437 47.741
95742 95.992 -78.444 47.704
95743 96.005 -78.452 47.666
95744 96.019 -78.460 47.628
95745 96.032 -78.467 47.590
95746 96.045 -78.475 47.555
95747 96.059 -78.485 47.522
95748 96.059 -78.499 47.487
95749 96.062 -78.512 47.452
95750 96.067 -78.523 47.418
95751 96.075 -78.548 47.385
95752 96.096 -78.563 47.354
95753 96.122 -78.567 47.322
95754 96.149 -78.571 47.291
95755 96.175 -78.575 47.262
95756 96.201 -78.579 47.234
95757 96.230 -78.585 47.206
95758 96.259 -78.590 47.178
95759 96.289 -78.595 47.150
95760 96.318 -78.601 47.121
95761 96.347 -78.607 47.093
95762 96.377 -78.613 47.065
95763 96.406 -78.618 47.035
95764 96.435 -78.623 47.004
95765 96.464 -78.625 46.974
95766 96.493 -78.628 46.945
95767 96.523 -78.630 46.917
95768 96.523 -78.617 46.889
95769 96.519 -78.603 46.862
95770 96.516 -78.589 46.836
95771 96.565 -78.599 46.809
95772 96.614 -78.609 46.782
95773 96.663 -78.619 46.760
95774 96.713 -78.629 46.738
95775 96.761 -78.639 46.716
95776 96.764 -78.649 46.693
95777 96.765 -78.661 46.670
95778 96.765 -78.671 46.647
95779 96.775 -78.683 46.623
95780 96.796 -78.693 46.601
95781 96.816 -78.704 46.579
95782 96.819 -78.720 46.557
95783 96.822 -78.739 46.536
95784 96.835 -78.753 46.513
95785 96.847 -78.767 46.491
95786 96.857 -78.777 46.469
95787 96.867 -78.789 46.449
95788 96.877 -78.799 46.430
95789 96.887 -78.810 46.412
95790 96.897 -78.821 46.394
95791 96.911 -78.836 46.375
95792 96.932 -78.854 46.355
95793 96.962 -78.865 46.335
95794 96.992 -78.875 46.318
95795 97.022 -78.886 46.302
95796 97.051 -78.896 46.286
95797 97.081 -78.907 46.270
95798 97.111 -78.918 46.255
95799 97.141 -78.928 46.239
95800 97.170 -78.939 46.224
95801 97.200 -78.949 46.212
95802 97.230 -78.960 46.198
95803 97.259 -78.971 46.184
95804 97.288 -78.981 46.168
95805 97.325 -78.995 46.152
95806 97.363 -79.007 46.134
95807 97.378 -79.006 46.114
95808 97.394 -79.006 46.096
95809 97.409 -79.005 46.078
95810 97.426 -79.004 46.060
95811 97.439 -79.002 46.042
95812 97.447 -78.996 46.023
95813 97.456 -78.990 46.003
95814 97.464 -78.984 45.983
95815 97.472 -78.978 45.967
95816 97.480 -78.972 45.950
95817 97.489 -78.966 45.934
95818 97.497 -78.960 45.918
95819 97.505 -78.955 45.903
95820 97.513 -78.949 45.890
95821 97.521 -78.943 45.877
95822 97.529 -78.937 45.866
95823 97.537 -78.930 45.854
95824 97.545 -78.925 45.840
95825 97.554 -78.919 45.827
95826 97.562 -78.913 45.811
95827 97.572 -78.912 45.792
95828 97.593 -78.949 45.770
95829 97.614 -78.987 45.748
95830 97.636 -79.024 45.726
95831 97.641 -79.027 45.706
95832 97.644 -79.024 45.685
95833 97.646 -79.021 45.666
95834 97.649 -79.018 45.647
95835 97.651 -79.015 45.628
95836 97.654 -79.013 45.610
95837 97.656 -79.010 45.593
95838 97.659 -79.007 45.578
95839 97.692 -79.028 45.562
95840 97.728 -79.052 45.548
95841 97.761 -79.073 45.534
95842 97.769 -79.078 45.519
95843 97.776 -79.084 45.506
95844 97.783 -79.089 45.494
95845 97.790 -79.093 45.481
95846 97.798 -79.099 45.469
95847 97.805 -79.104 45.455
95848 97.812 -79.110 45.442
95849 97.816 -79.124 45.429
95850 97.819 -79.137 45.417
95851 97.823 -79.151 45.406
95852 97.827 -79.165 45.396
95853 97.842 -79.172 45.385
95854 97.859 -79.179 45.374
95855 97.874 -79.169 45.365
95856 97.889 -79.159 45.357
95857 97.903 -79.149 45.347
95858 97.919 -79.138 45.338
95859 97.934 -79.128 45.329
95860 97.949 -79.118 45.319
95861 97.963 -79.107 45.309
95862 97.997 -79.111 45.300
95863 98.055 -79.149 45.290
95864 98.116 -79.202 45.281
95865 98.144 -79.214 45.273
95866 98.171 -79.223 45.266
95867 98.197 -79.233 45.259
95868 98.223 -79.238 45.252
95869 98.249 -79.241 45.245
95870 98.275 -79.244 45.237
95871 98.301 -79.248 45.230
95872 98.327 -79.251 45.221
95873 98.353 -79.255 45.211
95874 98.379 -79.258 45.200
95875 98.411 -79.269 45.190
95876 98.443 -79.280 45.179
95877 98.476 -79.292 45.167
95878 98.510 -79.303 45.157
95879 98.547 -79.316 45.147
95880 98.583 -79.328 45.137
95881 98.620 -79.341 45.126
95882 98.656 -79.353 45.116
95883 98.693 -79.365 45.106
95884 98.730 -79.377 45.097
95885 98.766 -79.389 45.086
95886 98.803 -79.401 45.077
95887 98.809 -79.405 45.069
95888 98.809 -79.407 45.062
95889 98.811 -79.411 45.054
95890 98.835 -79.429 45.045
95891 98.859 -79.446 45.036
95892 98.882 -79.463 45.030
95893 98.906 -79.480 45.024
95894 98.929 -79.498 45.018
95895 98.951 -79.513 45.012
95896 98.975 -79.527 45.005
95897 99.016 -79.529 44.996
95898 99.057 -79.529 44.983
95899 99.099 -79.529 44.973
95900 99.148 -79.515 44.962
95901 99.158 -79.521 44.952
95902 99.167 -79.526 44.942
95903 99.178 -79.532 44.931
95904 99.187 -79.538 44.920
95905 99.200 -79.548 44.908
95906 99.215 -79.563 44.898
95907 99.229 -79.577 44.889
95908 99.244 -79.592 44.880
95909 99.260 -79.607 44.870
95910 99.274 -79.621 44.863
95911 99.289 -79.636 44.855
95912 99.304 -79.651 44.846
95913 99.326 -79.659 44.838
95914 99.345 -79.661 44.828
95915 99.360 -79.656 44.818
95916 99.375 -79.651 44.808
95917 99.391 -79.645 44.798
95918 99.406 -79.639 44.788
95919 99.421 -79.634 44.779
95920 99.437 -79.629 44.769
95921 99.452 -79.623 44.760
95922 99.468 -79.617 44.750
95923 99.483 -79.612 44.741
95924 99.498 -79.607 44.733
95925 99.517 -79.604 44.726
95926 99.547 -79.607 44.718
95927 99.579 -79.617 44.710
95928 99.597 -79.642 44.702
95929 99.615 -79.667 44.695
95930 99.632 -79.692 44.689
95931 99.650 -79.717 44.684
95932 99.669 -79.742 44.679
95933 99.689 -79.767 44.674
95934 99.708 -79.791 44.669
95935 99.728 -79.816 44.663
95936 99.748 -79.840 44.657
95937 99.767 -79.864 44.650
95938 99.787 -79.889 44.643
95939 99.806 -79.914 44.636
95940 99.826 -79.939 44.629
95941 99.850 -79.968 44.620
95942 99.877 -80.002 44.612
95943 99.905 -80.031 44.604
95944 99.934 -80.060 44.598
95945 99.963 -80.089 44.590
95946 99.993 -80.118 44.582
95947 100.033 -80.143 44.574
95948 100.067 -80.167 44.564
95949 100.096 -80.193 44.552
95950 100.124 -80.220 44.540
95951 100.153 -80.247 44.526
95952 100.182 -80.273 44.512
95953 100.207 -80.300 44.497
95954 100.242 -80.326 44.482
95955 100.303 -80.346 44.467
95956 100.360 -80.361 44.451
95957 100.384 -80.368 44.433
95958 100.414 -80.418 44.414
95959 100.440 -80.467 44.396
95960 100.460 -80.515 44.378
95961 100.502 -80.535 44.360
95962 100.547 -80.551 44.342
95963 100.587 -80.573 44.325
95964 100.628 -80.594 44.308
95965 100.668 -80.617 44.292
95966 100.708 -80.639 44.276
95967 100.749 -80.661 44.260
95968 100.789 -80.683 44.245
95969 100.829 -80.705 44.230
95970 100.869 -80.727 44.215
95971 100.909 -80.749 44.200
95972 100.950 -80.771 44.185
95973 100.990 -80.793 44.170
95974 101.030 -80.815 44.154
95975 101.071 -80.836 44.139
95976 101.117 -80.849 44.126
95977 101.164 -80.862 44.112
95978 101.210 -80.875 44.099
95979 101.257 -80.888 44.085
96244 93.512 -78.597 53.788
96245 93.522 -78.595 53.774
96246 93.532 -78.592 53.760
96247 93.542 -78.590 53.745
96248 93.551 -78.587 53.729
96249 93.561 -78.585 53.714
96250 93.571 -78.582 53.697
96251 93.580 -78.580 53.680
96252 93.582 -78.573 53.663
96253 93.580 -78.563 53.645
96254 93.579 -78.554 53.627
96255 93.578 -78.545 53.609
96256 93.576 -78.535 53.591
96257 93.583 -78.522 53.573
96258 93.601 -78.507 53.555
96259 93.623 -78.492 53.537
96260 93.645 -78.476 53.518
96261 93.668 -78.462 53.500
96262 93.680 -78.454 53.483
96263 93.691 -78.445 53.465
96264 93.707 -78.438 53.446
96265 93.722 -78.430 53.428
96266 93.737 -78.423 53.410
96267 93.753 -78.416 53.391
96268 93.768 -78.408 53.372
96269 93.784 -78.401 53.353
96270 93.799 -78.394 53.335
96271 93.815 -78.386 53.315
96272 93.830 -78.379 53.296
96273 93.845 -78.372 53.277
96274 93.861 -78.364 53.259
96275 93.876 -78.357 53.240
96276 93.892 -78.350 53.221
96277 93.906 -78.345 53.203
96278 93.921 -78.340 53.184
96279 93.919 -78.322 53.165
96280 93.918 -78.306 53.148
96281 93.930 -78.290 53.132
96282 93.949 -78.294 53.117
96283 93.963 -78.303 53.102
96284 93.976 -78.311 53.087
96285 93.990 -78.320 53.071
96286 93.998 -78.324 53.056
96287 94.006 -78.328 53.041
96288 94.015 -78.332 53.028
96289 94.023 -78.335 53.016
96290 94.031 -78.339 53.003
96291 94.039 -78.342 52.989
96292 94.047 -78.346 52.977
96293 94.056 -78.350 52.964
96294 94.067 -78.354 52.950
96295 94.079 -78.360 52.935
96296 94.088 -78.363 52.924
96297 94.099 -78.368 52.913
96298 94.107 -78.373 52.901
96299 94.112 -78.382 52.888
96300 94.117 -78.391 52.877
96301 94.121 -78.400 52.868
96302 94.126 -78.409 52.861
96303 94.130 -78.417 52.855
96304 94.135 -78.426 52.847
96305 94.140 -78.429 52.837
96306 94.144 -78.432 52.822
96307 94.149 -78.433 52.809
96308 94.153 -78.435 52.798
96309 94.158 -78.438 52.788
96310 94.163 -78.440 52.779
96311 94.167 -78.442 52.769
96312 94.183 -78.429 52.756
96313 94.201 -78.416 52.738
96314 94.220 -78.402 52.718
96315 94.238 -78.388 52.696
96316 94.259 -78.387 52.672
96317 94.280 -78.389 52.647
96318 94.297 -78.395 52.621
96319 94.306 -78.413 52.596
96320 94.314 -78.430 52.566
96321 94.323 -78.448 52.535
96322 94.331 -78.464 52.503
96323 94.339 -78.482 52.473
96324 94.348 -78.499 52.441
96325 94.359 -78.510 52.408
96326 94.381 -78.502 52.375
96327 94.411 -78.507 52.338
96328 94.443 -78.507 52.302
96329 94.467 -78.491 52.266
96330 94.492 -78.474 52.232
96331 94.516 -78.457 52.199
96332 94.541 -78.442 52.165
96333 94.565 -78.425 52.131
96334 94.590 -78.408 52.090
96335 94.606 -78.398 52.047
96336 94.616 -78.392 52.003
96337 94.627 -78.386 51.960
96338 94.637 -78.380 51.917
96339 94.648 -78.376 51.872
96340 94.661 -78.378 51.825
96341 94.673 -78.380 51.776
96342 94.685 -78.382 51.729
96343 94.697 -78.384 51.683
96344 94.709 -78.386 51.637
96345 94.721 -78.384 51.589
96346 94.734 -78.382 51.541
96347 94.745 -78.386 51.492
96348 94.761 -78.382 51.442
96349 94.779 -78.375 51.395
96350 94.798 -78.367 51.351
96351 94.812 -78.369 51.310
96352 94.824 -78.373 51.272
96353 94.841 -78.382 51.234
96354 94.862 -78.387 51.195
96355 94.887 -78.388 51.150
96356 94.907 -78.389 51.101
96357 94.918 -78.391 51.050
96358 94.928 -78.392 51.000
96359 94.939 -78.394 50.950
96360 94.950 -78.395 50.900
96361 94.961 -78.397 50.850
96362 94.972 -78.398 50.799
96363 94.982 -78.400 50.750
96364 94.993 -78.401 50.702
96365 95.003 -78.397 50.655
96366 95.013 -78.379 50.608
96367 95.023 -78.363 50.563
96368 95.047 -78.354 50.518
96369 95.119 -78.373 50.475
96370 95.165 -78.327 50.432
96371 95.191 -78.338 50.390
96372 95.201 -78.327 50.348
96373 95.212 -78.316 50.307
96374 95.222 -78.305 50.267
96375 95.233 -78.294 50.228
96376 95.244 -78.282 50.189
96377 95.254 -78.272 50.150
96378 95.265 -78.260 50.112
96379 95.275 -78.249 50.074
96380 95.286 -78.238 50.036
96381 95.297 -78.227 49.999
96382 95.307 -78.216 49.962
96383 95.317 -78.217 49.924
96384 95.326 -78.220 49.888
96385 95.335 -78.223 49.852
96386 95.345 -78.226 49.817
96387 95.354 -78.228 49.782
96388 95.364 -78.232 49.746
96389 95.373 -78.235 49.712
96390 95.382 -78.238 49.678
96391 95.392 -78.241 49.644
96392 95.401 -78.243 49.612
96393 95.417 -78.250 49.581
96394 95.441 -78.262 49.550
96395 95.465 -78.274 49.519
96396 95.488 -78.285 49.488
96397 95.513 -78.297 49.458
96398 95.522 -78.300 49.428
96399 95.527 -78.299 49.398
96400 95.532 -78.299 49.367
96401 95.537 -78.298 49.335
96402 95.543 -78.300 49.304
96403 95.552 -78.307 49.273
96404 95.561 -78.314 49.243
96405 95.570 -78.321 49.211
96406 95.580 -78.328 49.178
96407 95.589 -78.335 49.144
96408 95.598 -78.342 49.107
96409 95.607 -78.350 49.071
96410 95.616 -78.357 49.034
96411 95.625 -78.363 48.998
96412 95.634 -78.370 48.963
96413 95.630 -78.366 48.928
96414 95.626 -78.363 48.891
96415 95.621 -78.360 48.852
96416 95.617 -78.357 48.811
96417 95.613 -78.353 48.769
96418 95.615 -78.353 48.728
96419 95.629 -78.362 48.687
96420 95.646 -78.366 48.646
96421 95.664 -78.368 48.604
96422 95.681 -78.374 48.562
96423 95.697 -78.380 48.520
96424 95.708 -78.382 48.478
96425 95.721 -78.383 48.438
96426 95.739 -78.385 48.397
96427 95.762 -78.389 48.355
96428 95.785 -78.396 48.314
96429 95.801 -78.397 48.273
96430 95.817 -78.396 48.232
96431 95.834 -78.396 48.190
96432 95.850 -78.395 48.150
96433 95.863 -78.395 48.109
96434 95.874 -78.394 48.069
96435 95.891 -78.395 48.027
96436 95.907 -78.398 47.986
96437 95.919 -78.398 47.946
96438 95.931 -78.399 47.906
96439 95.946 -78.403 47.867
96440 95.959 -78.407 47.829
96441 95.971 -78.414 47.791
96442 95.983 -78.421 47.752
96443 95.996 -78.428 47.715
96444 96.010 -78.435 47.678
96445 96.023 -78.444 47.640
96446 96.036 -78.451 47.605
96447 96.050 -78.463 47.571
96448 96.064 -78.475 47.539
96449 96.068 -78.486 47.504
96450 96.067 -78.501 47.469
96451 96.067 -78.517 47.435
96452 96.066 -78.532 47.404
96453 96.084 -78.541 47.373
96454 96.114 -78.546 47.341
96455 96.143 -78.551 47.309
96456 96.172 -78.557 47.280
96457 96.202 -78.563 47.251
96458 96.231 -78.569 47.222
96459 96.260 -78.574 47.194
96460 96.290 -78.580 47.168
96461 96.319 -78.586 47.140
96462 96.348 -78.591 47.111
96463 96.378 -78.597 47.083
96464 96.407 -78.601 47.052
96465 96.436 -78.604 47.021
96466 96.465 -78.607 46.992
96467 96.495 -78.609 46.963
96468 96.524 -78.612 46.935
96469 96.553 -78.614 46.908
96470 96.578 -78.614 46.881
96471 96.574 -78.600 46.854
96472 96.571 -78.586 46.828
96473 96.620 -78.595 46.801
96474 96.669 -78.605 46.777
96475 96.718 -78.615 46.754
96476 96.754 -78.626 46.732
96477 96.755 -78.636 46.709
96478 96.755 -78.648 46.686
96479 96.756 -78.659 46.663
96480 96.762 -78.670 46.639
96481 96.783 -78.680 46.617
96482 96.799 -78.692 46.594
96483 96.800 -78.711 46.572
96484 96.810 -78.726 46.551
96485 96.823 -78.740 46.529
96486 96.834 -78.753 46.506
96487 96.844 -78.764 46.485
96488 96.855 -78.774 46.465
96489 96.865 -78.786 46.445
96490 96.875 -78.796 46.426
96491 96.889 -78.812 46.408
96492 96.908 -78.836 46.390
96493 96.928 -78.859 46.370
96494 96.955 -78.868 46.350
96495 96.984 -78.879 46.334
96496 97.014 -78.890 46.318
96497 97.044 -78.900 46.302
96498 97.074 -78.910 46.285
96499 97.103 -78.921 46.270
96500 97.133 -78.931 46.255
96501 97.163 -78.942 46.240
96502 97.192 -78.952 46.226
96503 97.222 -78.963 46.213
96504 97.252 -78.974 46.198
96505 97.283 -78.981 46.182
96506 97.318 -78.981 46.164
96507 97.356 -78.993 46.145
96508 97.384 -79.000 46.125
96509 97.400 -78.999 46.107
96510 97.416 -78.998 46.089
96511 97.431 -78.997 46.070
96512 97.447 -78.996 46.052
96513 97.455 -78.990 46.034
96514 97.463 -78.984 46.014
96515 97.471 -78.979 45.993
96516 97.479 -78.973 45.977
96517 97.488 -78.967 45.960
96518 97.496 -78.961 45.943
96519 97.504 -78.955 45.927
96520 97.512 -78.949 45.913
96521 97.521 -78.943 45.900
96522 97.529 -78.937 45.886
96523 97.537 -78.931 45.876
96524 97.545 -78.925 45.864
96525 97.554 -78.919 45.850
96526 97.562 -78.914 45.838
96527 97.572 -78.908 45.821
96528 97.590 -78.905 45.801
96529 97.603 -78.905 45.780
96530 97.625 -78.943 45.756
96531 97.646 -78.980 45.735
96532 97.662 -79.006 45.715
96533 97.664 -79.003 45.696
96534 97.667 -79.000 45.676
96535 97.670 -78.997 45.657
96536 97.672 -78.995 45.639
96537 97.675 -78.992 45.623
96538 97.677 -78.989 45.606
96539 97.696 -78.999 45.590
96540 97.732 -79.022 45.575
96541 97.766 -79.044 45.561
96542 97.773 -79.049 45.546
96543 97.781 -79.055 45.532
96544 97.788 -79.059 45.518
96545 97.795 -79.065 45.505
96546 97.802 -79.070 45.493
96547 97.809 -79.075 45.480
96548 97.817 -79.080 45.467
96549 97.824 -79.085 45.454
96550 97.829 -79.096 45.442
96551 97.833 -79.110 45.429
96552 97.837 -79.124 45.417
96553 97.841 -79.137 45.406
96554 97.858 -79.143 45.395
96555 97.874 -79.150 45.385
96556 97.891 -79.157 45.375
96557 97.906 -79.147 45.366
96558 97.921 -79.136 45.356
96559 97.936 -79.126 45.345
96560 97.951 -79.115 45.335
96561 97.965 -79.105 45.324
96562 97.981 -79.095 45.313
96563 97.996 -79.099 45.303
96564 98.046 -79.101 45.293
96565 98.101 -79.126 45.284
96566 98.148 -79.161 45.276
96567 98.174 -79.170 45.268
96568 98.200 -79.182 45.260
96569 98.224 -79.196 45.253
96570 98.249 -79.209 45.247
96571 98.275 -79.219 45.241
96572 98.301 -79.222 45.233
96573 98.327 -79.225 45.224
96574 98.353 -79.229 45.214
96575 98.379 -79.232 45.203
96576 98.409 -79.241 45.194
96577 98.442 -79.252 45.183
96578 98.475 -79.264 45.170
96579 98.512 -79.276 45.161
96580 98.548 -79.288 45.151
96581 98.585 -79.300 45.140
96582 98.622 -79.313 45.130
96583 98.658 -79.325 45.118
96584 98.695 -79.338 45.106
96585 98.731 -79.350 45.096
96586 98.768 -79.362 45.086
96587 98.804 -79.374 45.076
96588 98.822 -79.381 45.068
96589 98.823 -79.383 45.061
96590 98.828 -79.388 45.051
96591 98.851 -79.406 45.043
96592 98.875 -79.423 45.036
96593 98.898 -79.441 45.031
96594 98.922 -79.458 45.025
96595 98.945 -79.475 45.019
96596 98.969 -79.492 45.013
96597 98.991 -79.507 45.005
96598 99.028 -79.511 44.996
96599 99.069 -79.513 44.986
96600 99.119 -79.500 44.975
96601 99.143 -79.499 44.965
96602 99.153 -79.504 44.954
96603 99.163 -79.510 44.942
96604 99.172 -79.516 44.931
96605 99.183 -79.521 44.921
96606 99.195 -79.532 44.910
96607 99.210 -79.546 44.899
96608 99.225 -79.561 44.890
96609 99.239 -79.576 44.880
96610 99.255 -79.590 44.872
96611 99.270 -79.604 44.864
96612 99.284 -79.619 44.857
96613 99.305 -79.629 44.849
96614 99.327 -79.637 44.840
96615 99.348 -79.645 44.831
96616 99.366 -79.645 44.821
96617 99.381 -79.639 44.810
96618 99.397 -79.634 44.800
96619 99.412 -79.629 44.791
96620 99.427 -79.623 44.782
96621 99.442 -79.617 44.773
96622 99.458 -79.612 44.764
96623 99.473 -79.607 44.755
96624 99.489 -79.601 44.747
96625 99.504 -79.595 44.739
96626 99.532 -79.598 44.732
96627 99.561 -79.601 44.725
96628 99.591 -79.604 44.717
96629 99.623 -79.612 44.710
96630 99.646 -79.640 44.704
96631 99.663 -79.665 44.699
96632 99.681 -79.690 44.694
96633 99.699 -79.715 44.689
96634 99.716 -79.740 44.684
96635 99.733 -79.765 44.678
96636 99.751 -79.790 44.671
96637 99.769 -79.815 44.664
96638 99.789 -79.840 44.658
96639 99.808 -79.864 44.651
96640 99.828 -79.889 44.644
96641 99.848 -79.914 44.638
96642 99.872 -79.943 44.631
96643 99.899 -79.974 44.623
96644 99.927 -80.002 44.616
96645 99.960 -80.033 44.609
96646 99.992 -80.064 44.601
96647 100.024 -80.096 44.593
96648 100.063 -80.120 44.585
96649 100.109 -80.139 44.575
96650 100.154 -80.157 44.564
96651 100.192 -80.180 44.552
96652 100.221 -80.206 44.539
96653 100.250 -80.233 44.526
96654 100.276 -80.260 44.512
96655 100.294 -80.290 44.497
96656 100.334 -80.314 44.481
96657 100.384 -80.331 44.464
96658 100.434 -80.365 44.447
96659 100.462 -80.410 44.428
96660 100.482 -80.450 44.409
96661 100.502 -80.491 44.390
96662 100.555 -80.554 44.371
96663 100.543 -80.553 44.352
96664 100.583 -80.576 44.333
96665 100.624 -80.598 44.316
96666 100.664 -80.620 44.300
96667 100.704 -80.641 44.284
96668 100.745 -80.663 44.268
96669 100.785 -80.685 44.253
96670 100.825 -80.707 44.238
96671 100.865 -80.729 44.222
96672 100.905 -80.751 44.206
96673 100.946 -80.773 44.190
96674 100.986 -80.795 44.175
96675 101.029 -80.814 44.159
96676 101.075 -80.827 44.144
96677 101.121 -80.840 44.130
96678 101.168 -80.853 44.116
96679 101.214 -80.866 44.103
96680 101.261 -80.879 44.089
96681 101.307 -80.892 44.075
96944 93.503 -78.589 53.807
96945 93.513 -78.586 53.793
96946 93.522 -78.584 53.779
96947 93.532 -78.582 53.764
96948 93.542 -78.579 53.749
96949 93.552 -78.576 53.733
96950 93.561 -78.574 53.716
96951 93.571 -78.572 53.699
96952 93.580 -78.568 53.681
96953 93.578 -78.559 53.664
96954 93.577 -78.550 53.646
96955 93.576 -78.541 53.627
96956 93.574 -78.532 53.608
96957 93.574 -78.522 53.590
96958 93.584 -78.507 53.571
96959 93.606 -78.492 53.552
96960 93.629 -78.476 53.534
96961 93.651 -78.461 53.515
96962 93.669 -78.449 53.497
96963 93.681 -78.441 53.479
96964 93.693 -78.432 53.460
96969 93.770 -78.395 53.367
96970 93.786 -78.388 53.348
96971 93.801 -78.381 53.330
96972 93.817 -78.373 53.309
96973 93.832 -78.366 53.290
96974 93.847 -78.359 53.271
96975 93.863 -78.351 53.253
96976 93.878 -78.344 53.234
96977 93.894 -78.337 53.215
96978 93.909 -78.329 53.198
96979 93.923 -78.325 53.179
96980 93.931 -78.314 53.161
96981 93.930 -78.297 53.144
96982 93.946 -78.283 53.129
96983 93.959 -78.291 53.116
96984 93.973 -78.300 53.102
96985 93.985 -78.308 53.088
96986 93.994 -78.312 53.074
96987 94.002 -78.316 53.059
96988 94.010 -78.319 53.044
96989 94.018 -78.322 53.032
96990 94.026 -78.326 53.020
96991 94.035 -78.330 53.007
96992 94.043 -78.334 52.994
96993 94.051 -78.338 52.981
96994 94.059 -78.341 52.967
96995 94.069 -78.345 52.953
96996 94.080 -78.350 52.940
96997 94.090 -78.354 52.928
96998 94.099 -78.360 52.915
96999 94.104 -78.369 52.902
97000 94.108 -78.377 52.889
97001 94.113 -78.386 52.877
97002 94.118 -78.395 52.868
97003 94.122 -78.404 52.859
97004 94.127 -78.413 52.853
97005 94.132 -78.422 52.846
97006 94.136 -78.428 52.837
97007 94.141 -78.430 52.824
97008 94.145 -78.432 52.813
97009 94.150 -78.435 52.803
97010 94.154 -78.437 52.793
97011 94.159 -78.439 52.783
97012 94.163 -78.432 52.772
97013 94.174 -78.414 52.757
97014 94.192 -78.401 52.737
97015 94.210 -78.387 52.715
97016 94.228 -78.373 52.691
97017 94.250 -78.374 52.666
97018 94.271 -78.376 52.640
97019 94.291 -78.379 52.613
97020 94.301 -78.394 52.586
97021 94.310 -78.411 52.555
97022 94.319 -78.429 52.524
97023 94.327 -78.446 52.492
97024 94.336 -78.463 52.461
97025 94.344 -78.480 52.428
97026 94.359 -78.483 52.394
97027 94.381 -78.475 52.360
97028 94.406 -78.473 52.323
97029 94.444 -78.487 52.286
97030 94.472 -78.479 52.249
97031 94.496 -78.463 52.214
97032 94.521 -78.446 52.178
97033 94.545 -78.430 52.143
97034 94.570 -78.413 52.107
97035 94.594 -78.397 52.064
97036 94.611 -78.386 52.021
97037 94.622 -78.380 51.976
97038 94.632 -78.375 51.932
97039 94.642 -78.369 51.887
97040 94.653 -78.363 51.840
97041 94.665 -78.362 51.791
97042 94.677 -78.364 51.743
97043 94.689 -78.366 51.696
97044 94.701 -78.368 51.651
97045 94.713 -78.369 51.605
97046 94.726 -78.366 51.558
97047 94.739 -78.363 51.512
97048 94.750 -78.367 51.466
97049 94.760 -78.371 51.420
97050 94.778 -78.364 51.377
97051 94.797 -78.357 51.337
97052 94.811 -78.358 51.300
97053 94.823 -78.362 51.264
97054 94.838 -78.366 51.225
97055 94.863 -78.368 51.184
97056 94.888 -78.369 51.136
97057 94.913 -78.371 51.086
97058 94.928 -78.373 51.034
97059 94.939 -78.374 50.983
97060 94.950 -78.376 50.934
97061 94.960 -78.377 50.885
97062 94.971 -78.379 50.836
97063 94.982 -78.380 50.787
97064 94.992 -78.382 50.739
97065 95.003 -78.383 50.692
97066 95.014 -78.385 50.645
97067 95.025 -78.385 50.599
97068 95.034 -78.382 50.555
97069 95.051 -78.340 50.510
97070 95.093 -78.288 50.467
97071 95.126 -78.244 50.426
97072 95.162 -78.266 50.385
97073 95.198 -78.289 50.345
97074 95.218 -78.291 50.305
97075 95.229 -78.279 50.265
97076 95.239 -78.268 50.227
97077 95.250 -78.257 50.189
97078 95.260 -78.246 50.152
97079 95.271 -78.235 50.116
97080 95.281 -78.224 50.079
97081 95.293 -78.213 50.040
97082 95.303 -78.201 50.002
97083 95.313 -78.193 49.965
97084 95.323 -78.195 49.928
97085 95.332 -78.198 49.890
97086 95.341 -78.201 49.854
97087 95.351 -78.204 49.819
97088 95.360 -78.207 49.784
97089 95.370 -78.210 49.749
97090 95.379 -78.213 49.715
97091 95.388 -78.216 49.684
97092 95.398 -78.219 49.652
97093 95.407 -78.221 49.621
97094 95.416 -78.224 49.590
97095 95.431 -78.231 49.561
97096 95.456 -78.242 49.532
97097 95.479 -78.254 49.502
97098 95.503 -78.266 49.472
97099 95.513 -78.269 49.444
97100 95.518 -78.268 49.415
97101 95.523 -78.268 49.384
97102 95.529 -78.269 49.351
97103 95.538 -78.276 49.318
97104 95.547 -78.283 49.287
97105 95.556 -78.290 49.256
97106 95.565 -78.297 49.225
97107 95.574 -78.304 49.193
97108 95.584 -78.312 49.160
97109 95.593 -78.319 49.123
97110 95.602 -78.326 49.085
97111 95.611 -78.333 49.047
97112 95.620 -78.340 49.009
97113 95.630 -78.347 48.973
97114 95.629 -78.346 48.938
97115 95.625 -78.343 48.903
97116 95.620 -78.339 48.865
97117 95.616 -78.335 48.825
97118 95.612 -78.332 48.784
97119 95.620 -78.336 48.742
97120 95.634 -78.344 48.700
97121 95.650 -78.353 48.657
97122 95.668 -78.354 48.615
97123 95.686 -78.356 48.572
97124 95.700 -78.360 48.530
97125 95.712 -78.361 48.489
97126 95.725 -78.363 48.449
97127 95.740 -78.366 48.408
97128 95.762 -78.373 48.367
97129 95.783 -78.376 48.325
97130 95.798 -78.377 48.284
97131 95.814 -78.377 48.242
97132 95.830 -78.376 48.202
97133 95.846 -78.376 48.160
97134 95.858 -78.376 48.118
97135 95.875 -78.378 48.077
97136 95.892 -78.379 48.035
97137 95.908 -78.382 47.995
97138 95.920 -78.382 47.955
97139 95.934 -78.385 47.915
97140 95.948 -78.388 47.876
97141 95.963 -78.392 47.836
97142 95.975 -78.398 47.798
97143 95.987 -78.404 47.761
97144 96.001 -78.412 47.724
97145 96.014 -78.420 47.688
97146 96.027 -78.429 47.652
97147 96.042 -78.441 47.620
97148 96.056 -78.453 47.588
97149 96.071 -78.465 47.555
97150 96.077 -78.473 47.520
97151 96.077 -78.489 47.486
97152 96.076 -78.504 47.453
97153 96.075 -78.520 47.422
97154 96.090 -78.525 47.391
97155 96.115 -78.530 47.359
97156 96.144 -78.536 47.328
97157 96.174 -78.542 47.299
97158 96.203 -78.547 47.269
97159 96.232 -78.553 47.239
97160 96.261 -78.558 47.211
97161 96.291 -78.564 47.184
97162 96.320 -78.570 47.157
97163 96.349 -78.576 47.130
97164 96.379 -78.579 47.101
97165 96.408 -78.582 47.070
97166 96.437 -78.586 47.039
97167 96.466 -78.589 47.010
97168 96.496 -78.591 46.981
97169 96.525 -78.593 46.953
97170 96.554 -78.596 46.926
97171 96.584 -78.598 46.901
97172 96.613 -78.601 46.874
97173 96.629 -78.597 46.847
97174 96.626 -78.582 46.820
97175 96.675 -78.592 46.793
97176 96.724 -78.602 46.769
97177 96.745 -78.613 46.747
97178 96.745 -78.624 46.725
97179 96.746 -78.635 46.702
97180 96.746 -78.646 46.678
97181 96.749 -78.657 46.654
97182 96.769 -78.667 46.630
97183 96.780 -78.682 46.608
97184 96.786 -78.698 46.586
97185 96.799 -78.713 46.565
97186 96.811 -78.727 46.543
97187 96.822 -78.739 46.521
97188 96.832 -78.750 46.499
97189 96.842 -78.761 46.480
97190 96.853 -78.772 46.461
97191 96.868 -78.789 46.441
97192 96.887 -78.812 46.422
97193 96.906 -78.836 46.403
97194 96.926 -78.859 46.385
97195 96.947 -78.872 46.366
97196 96.977 -78.882 46.349
97197 97.007 -78.893 46.333
97198 97.036 -78.903 46.319
97199 97.066 -78.914 46.303
97200 97.096 -78.924 46.286
97201 97.125 -78.935 46.271
97202 97.155 -78.946 46.257
97203 97.185 -78.956 46.243
97204 97.215 -78.966 46.228
97205 97.244 -78.977 46.213
97206 97.277 -78.979 46.196
97207 97.312 -78.975 46.177
97208 97.348 -78.977 46.157
97209 97.385 -78.990 46.137
97210 97.406 -78.993 46.118
97211 97.422 -78.992 46.099
97212 97.437 -78.991 46.080
97213 97.453 -78.990 46.063
97214 97.463 -78.985 46.043
97215 97.471 -78.979 46.023
97216 97.479 -78.973 46.003
97217 97.487 -78.968 45.986
97218 97.496 -78.962 45.970
97219 97.504 -78.955 45.953
97220 97.512 -78.949 45.937
97221 97.520 -78.943 45.922
97222 97.528 -78.938 45.908
97223 97.536 -78.932 45.894
97224 97.544 -78.926 45.882
97225 97.553 -78.920 45.869
97226 97.561 -78.914 45.856
97227 97.572 -78.908 45.842
97228 97.590 -78.905 45.825
97229 97.608 -78.902 45.807
97230 97.621 -78.899 45.786
97231 97.635 -78.899 45.765
97232 97.656 -78.936 45.745
97233 97.678 -78.974 45.726
97234 97.685 -78.982 45.707
97235 97.688 -78.979 45.689
97236 97.690 -78.977 45.671
97237 97.693 -78.974 45.653
97238 97.696 -78.971 45.637
97239 97.700 -78.970 45.621
97240 97.736 -78.993 45.605
97241 97.771 -79.015 45.589
97242 97.778 -79.021 45.573
97243 97.785 -79.026 45.558
97244 97.792 -79.031 45.544
97245 97.800 -79.036 45.531
97246 97.807 -79.041 45.518
97247 97.814 -79.046 45.505
97248 97.821 -79.051 45.492
97249 97.829 -79.056 45.478
97250 97.836 -79.062 45.466
97251 97.843 -79.069 45.454
97252 97.846 -79.083 45.442
97253 97.850 -79.096 45.428
97254 97.857 -79.109 45.416
97255 97.873 -79.115 45.405
97256 97.890 -79.122 45.395
97257 97.907 -79.129 45.385
97258 97.923 -79.134 45.374
97259 97.938 -79.124 45.364
97260 97.953 -79.113 45.352
97261 97.967 -79.103 45.340
97262 97.983 -79.093 45.328
97263 97.998 -79.082 45.317
97264 98.004 -79.087 45.307
97265 98.045 -79.089 45.297
97266 98.096 -79.091 45.287
97267 98.148 -79.103 45.278
97268 98.177 -79.118 45.270
97269 98.202 -79.132 45.262
97270 98.227 -79.145 45.254
97271 98.252 -79.159 45.247
97272 98.277 -79.172 45.242
97273 98.302 -79.185 45.234
97274 98.327 -79.198 45.226
97275 98.353 -79.203 45.217
97276 98.379 -79.206 45.207
97277 98.407 -79.212 45.197
97278 98.440 -79.225 45.187
97279 98.477 -79.237 45.175
97280 98.513 -79.249 45.166
97281 98.550 -79.261 45.155
97282 98.586 -79.273 45.143
97283 98.623 -79.285 45.132
97284 98.660 -79.297 45.121
97285 98.696 -79.310 45.110
97286 98.733 -79.322 45.098
97287 98.770 -79.335 45.089
97288 98.806 -79.347 45.079
97289 98.836 -79.357 45.070
97290 98.836 -79.359 45.061
97291 98.844 -79.366 45.053
97292 98.867 -79.383 45.045
97293 98.890 -79.400 45.038
97294 98.914 -79.418 45.032
97295 98.938 -79.435 45.026
97296 98.961 -79.453 45.020
97297 98.985 -79.470 45.014
97298 99.008 -79.487 45.006
97299 99.041 -79.494 44.998
97300 99.088 -79.486 44.988
97301 99.129 -79.476 44.977
97302 99.139 -79.482 44.967
97303 99.148 -79.488 44.956
97304 99.158 -79.494 44.944
97305 99.168 -79.499 44.933
97306 99.178 -79.505 44.922
97307 99.190 -79.515 44.913
97308 99.205 -79.529 44.901
97309 99.220 -79.545 44.890
97310 99.235 -79.559 44.881
97311 99.250 -79.573 44.873
97312 99.265 -79.588 44.866
97313 99.284 -79.598 44.859
97314 99.306 -79.607 44.851
97315 99.328 -79.615 44.842
97316 99.349 -79.623 44.833
97317 99.371 -79.632 44.824
97318 99.387 -79.629 44.813
97319 99.403 -79.623 44.803
97320 99.418 -79.617 44.793
97321 99.433 -79.612 44.784
97322 99.448 -79.607 44.775
97323 99.464 -79.601 44.768
97324 99.479 -79.595 44.760
97325 99.495 -79.590 44.752
97326 99.515 -79.588 44.745
97327 99.546 -79.592 44.738
97328 99.576 -79.595 44.730
97329 99.606 -79.598 44.723
97330 99.636 -79.601 44.717
97331 99.666 -79.607 44.712
97332 99.695 -79.638 44.706
97333 99.712 -79.663 44.701
97334 99.730 -79.688 44.696
97335 99.748 -79.713 44.691
97336 99.765 -79.739 44.685
97337 99.782 -79.764 44.679
97338 99.800 -79.789 44.673
97339 99.818 -79.814 44.666
97340 99.835 -79.839 44.659
97341 99.852 -79.864 44.653
97342 99.870 -79.889 44.648
97343 99.894 -79.916 44.640
97344 99.926 -79.947 44.633
97345 99.958 -79.978 44.626
97346 99.990 -80.009 44.618
97347 100.023 -80.041 44.611
97348 100.055 -80.072 44.603
97349 100.093 -80.098 44.594
97350 100.139 -80.116 44.584
97351 100.185 -80.134 44.574
97352 100.230 -80.153 44.563
97353 100.276 -80.171 44.552
97354 100.318 -80.192 44.540
97355 100.345 -80.219 44.526
97356 100.364 -80.252 44.512
97357 100.387 -80.295 44.495
97358 100.430 -80.329 44.478
97359 100.482 -80.362 44.461
97360 100.517 -80.411 44.443
97361 100.543 -80.455 44.425
97362 100.628 -80.530 44.407
97363 100.607 -80.585 44.388
97364 100.545 -80.563 44.367
97365 100.579 -80.578 44.347
97366 100.620 -80.600 44.328
97367 100.660 -80.622 44.312
97368 100.700 -80.644 44.295
97369 100.741 -80.666 44.279
97370 100.781 -80.688 44.263
97371 100.821 -80.710 44.246
97372 100.861 -80.732 44.230
97373 100.901 -80.754 44.213
97374 100.941 -80.776 44.197
97375 100.986 -80.792 44.181
97376 101.033 -80.804 44.165
97377 101.079 -80.817 44.150
97378 101.125 -80.830 44.136
97379 101.172 -80.843 44.121
97380 101.218 -80.856 44.107
97381 101.265 -80.869 44.093
97382 101.312 -80.882 44.079
97383 101.358 -80.895 44.065
97645 93.503 -78.578 53.812
97646 93.513 -78.576 53.798
97647 93.523 -78.573 53.783
97648 93.533 -78.570 53.768
97649 93.542 -78.568 53.753
97650 93.552 -78.566 53.736
97651 93.562 -78.563 53.718
97652 93.571 -78.561 53.700
97653 93.576 -78.555 53.682
97654 93.575 -78.546 53.664
97655 93.574 -78.537 53.645
97656 93.572 -78.528 53.626
97657 93.571 -78.518 53.607
97658 93.575 -78.506 53.588
97659 93.589 -78.491 53.569
97660 93.612 -78.476 53.549
97661 93.634 -78.460 53.530
97662 93.656 -78.445 53.511
97663 93.671 -78.435 53.493
97664 93.682 -78.428 53.474
97673 93.819 -78.360 53.304
97674 93.834 -78.354 53.285
97675 93.849 -78.346 53.266
97676 93.864 -78.338 53.246
97677 93.880 -78.332 53.227
97678 93.896 -78.324 53.209
97679 93.911 -78.316 53.193
97680 93.926 -78.310 53.174
97681 93.941 -78.304 53.157
97682 93.942 -78.288 53.140
97683 93.956 -78.280 53.125
97684 93.969 -78.289 53.113
97685 93.981 -78.295 53.102
97686 93.989 -78.299 53.089
97687 93.997 -78.303 53.076
97688 94.005 -78.307 53.061
97689 94.013 -78.310 53.047
97690 94.022 -78.314 53.035
97691 94.030 -78.317 53.023
97692 94.038 -78.321 53.011
97693 94.046 -78.325 52.998
97694 94.054 -78.329 52.985
97695 94.062 -78.332 52.971
97696 94.071 -78.336 52.957
97697 94.082 -78.341 52.944
97698 94.091 -78.346 52.931
97699 94.096 -78.355 52.918
97700 94.100 -78.363 52.903
97701 94.105 -78.373 52.890
97702 94.110 -78.382 52.878
97703 94.114 -78.390 52.868
97704 94.119 -78.399 52.857
97705 94.123 -78.408 52.850
97706 94.128 -78.416 52.843
97707 94.133 -78.426 52.834
97708 94.137 -78.429 52.822
97709 94.142 -78.432 52.811
97710 94.146 -78.434 52.801
97711 94.151 -78.436 52.792
97712 94.155 -78.438 52.780
97713 94.158 -78.420 52.766
97714 94.164 -78.399 52.750
97715 94.183 -78.385 52.728
97716 94.201 -78.372 52.705
97717 94.219 -78.360 52.681
97718 94.241 -78.362 52.656
97719 94.262 -78.363 52.629
97720 94.282 -78.366 52.601
97721 94.297 -78.375 52.574
97722 94.306 -78.392 52.542
97723 94.315 -78.410 52.510
97724 94.323 -78.427 52.478
97725 94.332 -78.444 52.446
97726 94.340 -78.461 52.413
97727 94.359 -78.456 52.379
97728 94.381 -78.448 52.345
97729 94.402 -78.439 52.307
97730 94.439 -78.453 52.268
97731 94.476 -78.467 52.230
97732 94.501 -78.451 52.191
97733 94.526 -78.435 52.153
97734 94.550 -78.419 52.115
97735 94.574 -78.402 52.077
97736 94.599 -78.385 52.035
97737 94.616 -78.374 51.992
97738 94.627 -78.369 51.948
97739 94.637 -78.363 51.903
97740 94.648 -78.357 51.856
97741 94.659 -78.351 51.807
97742 94.669 -78.346 51.757
97743 94.681 -78.347 51.710
97744 94.693 -78.350 51.664
97745 94.705 -78.351 51.618
97746 94.718 -78.352 51.572
97747 94.731 -78.349 51.527
97748 94.744 -78.346 51.483
97749 94.755 -78.348 51.439
97750 94.765 -78.352 51.398
97751 94.778 -78.354 51.359
97752 94.796 -78.346 51.323
97753 94.810 -78.347 51.289
97754 94.823 -78.351 51.252
97755 94.844 -78.351 51.213
97756 94.866 -78.351 51.170
97757 94.889 -78.351 51.121
97758 94.913 -78.352 51.070
97759 94.938 -78.354 51.018
97760 94.949 -78.355 50.967
97761 94.960 -78.357 50.918
97762 94.971 -78.358 50.870
97763 94.981 -78.360 50.823
97764 94.992 -78.361 50.775
97765 95.002 -78.361 50.728
97766 95.011 -78.357 50.681
97767 95.020 -78.354 50.635
97768 95.029 -78.349 50.591
97769 95.039 -78.338 50.547
97770 95.061 -78.278 50.503
97771 95.083 -78.219 50.461
97772 95.111 -78.183 50.421
97773 95.126 -78.184 50.381
97774 95.162 -78.207 50.342
97775 95.198 -78.229 50.302
97776 95.234 -78.252 50.264
97777 95.246 -78.243 50.227
97778 95.256 -78.232 50.190
97779 95.267 -78.220 50.153
97780 95.277 -78.210 50.117
97781 95.288 -78.198 50.079
97782 95.299 -78.187 50.041
97783 95.309 -78.175 50.004
97784 95.319 -78.171 49.967
97785 95.329 -78.174 49.929
97786 95.338 -78.176 49.893
97787 95.347 -78.179 49.857
97788 95.356 -78.182 49.823
97789 95.366 -78.185 49.790
97790 95.376 -78.188 49.757
97791 95.385 -78.191 49.726
97792 95.394 -78.194 49.693
97793 95.404 -78.197 49.661
97794 95.413 -78.199 49.629
97795 95.422 -78.202 49.600
97796 95.431 -78.205 49.572
97797 95.447 -78.211 49.543
97798 95.470 -78.223 49.514
97799 95.494 -78.235 49.484
97800 95.503 -78.238 49.455
97801 95.509 -78.237 49.427
97802 95.515 -78.238 49.397
97803 95.524 -78.245 49.365
97804 95.533 -78.253 49.333
97805 95.542 -78.260 49.303
97806 95.551 -78.266 49.272
97807 95.560 -78.274 49.240
97808 95.570 -78.281 49.208
97809 95.579 -78.288 49.174
97810 95.588 -78.295 49.138
97811 95.597 -78.302 49.100
97812 95.606 -78.310 49.060
97813 95.616 -78.316 49.022
97814 95.625 -78.323 48.984
97815 95.628 -78.326 48.946
97816 95.623 -78.322 48.911
97817 95.619 -78.319 48.874
97818 95.615 -78.315 48.835
97819 95.611 -78.312 48.795
97820 95.625 -78.319 48.753
97821 95.639 -78.328 48.712
97822 95.654 -78.336 48.670
97823 95.671 -78.341 48.629
97824 95.689 -78.342 48.586
97825 95.703 -78.343 48.543
97826 95.716 -78.344 48.500
97827 95.728 -78.347 48.459
97828 95.740 -78.349 48.419
97829 95.762 -78.355 48.377
97830 95.780 -78.357 48.335
97831 95.796 -78.357 48.293
97832 95.811 -78.357 48.251
97833 95.826 -78.357 48.209
97834 95.842 -78.358 48.168
97835 95.860 -78.360 48.126
97836 95.876 -78.362 48.085
97837 95.893 -78.363 48.044
97838 95.910 -78.366 48.003
97839 95.922 -78.366 47.963
97840 95.937 -78.370 47.925
97841 95.951 -78.374 47.888
97842 95.965 -78.378 47.850
97843 95.979 -78.382 47.813
97844 95.991 -78.388 47.776
97845 96.005 -78.396 47.740
97846 96.019 -78.406 47.704
97847 96.033 -78.418 47.667
97848 96.047 -78.430 47.635
97849 96.062 -78.442 47.602
97850 96.076 -78.454 47.569
97851 96.086 -78.461 47.535
97852 96.085 -78.476 47.501
97853 96.084 -78.492 47.468
97854 96.085 -78.506 47.437
97855 96.102 -78.510 47.407
97856 96.118 -78.514 47.376
97857 96.146 -78.520 47.346
97858 96.175 -78.526 47.317
97859 96.204 -78.531 47.287
97860 96.234 -78.537 47.257
97861 96.263 -78.542 47.229
97862 96.292 -78.548 47.201
97863 96.321 -78.554 47.174
97864 96.350 -78.558 47.147
97865 96.380 -78.561 47.118
97866 96.409 -78.564 47.087
97867 96.438 -78.568 47.057
97868 96.467 -78.570 47.026
97869 96.497 -78.573 46.998
97870 96.526 -78.575 46.971
97871 96.555 -78.578 46.944
97872 96.585 -78.580 46.918
97873 96.614 -78.582 46.891
97874 96.643 -78.585 46.864
97875 96.672 -78.587 46.837
97876 96.681 -78.579 46.810
97877 96.730 -78.589 46.785
97878 96.735 -78.600 46.762
97879 96.736 -78.611 46.739
97880 96.736 -78.622 46.716
97881 96.737 -78.633 46.692
97882 96.737 -78.644 46.669
97883 96.756 -78.655 46.645
97884 96.761 -78.671 46.623
97885 96.774 -78.686 46.601
97886 96.787 -78.700 46.579
97887 96.799 -78.715 46.557
97888 96.809 -78.726 46.535
97889 96.819 -78.736 46.513
97890 96.830 -78.747 46.494
97891 96.846 -78.766 46.474
97892 96.865 -78.789 46.455
97893 96.885 -78.812 46.436
97894 96.904 -78.836 46.418
97895 96.923 -78.859 46.399
97896 96.943 -78.880 46.381
97897 96.969 -78.886 46.364
97898 96.999 -78.896 46.348
97899 97.029 -78.906 46.333
97900 97.058 -78.917 46.317
97901 97.088 -78.927 46.301
97902 97.118 -78.938 46.286
97903 97.148 -78.949 46.271
97904 97.177 -78.959 46.256
97905 97.207 -78.970 46.241
97906 97.237 -78.980 46.226
97907 97.272 -78.976 46.208
97908 97.306 -78.973 46.189
97909 97.341 -78.970 46.169
97910 97.377 -78.975 46.148
97911 97.412 -78.986 46.129
97912 97.428 -78.985 46.110
97913 97.443 -78.984 46.092
97914 97.459 -78.984 46.074
97915 97.470 -78.980 46.055
97916 97.478 -78.974 46.035
97917 97.487 -78.968 46.015
97918 97.495 -78.962 45.997
97919 97.503 -78.956 45.980
97920 97.511 -78.950 45.963
97921 97.519 -78.944 45.946
97922 97.528 -78.938 45.930
97923 97.536 -78.932 45.915
97924 97.544 -78.927 45.901
97925 97.552 -78.921 45.886
97926 97.561 -78.915 45.873
97927 97.573 -78.910 45.860
97928 97.590 -78.906 45.845
97929 97.608 -78.903 45.829
97930 97.626 -78.899 45.811
97931 97.640 -78.897 45.792
97932 97.653 -78.895 45.774
97933 97.666 -78.893 45.755
97934 97.688 -78.930 45.737
97935 97.706 -78.961 45.719
97936 97.709 -78.959 45.702
97937 97.711 -78.955 45.685
97938 97.714 -78.952 45.667
97939 97.717 -78.950 45.653
97940 97.740 -78.964 45.637
97941 97.775 -78.987 45.621
97942 97.783 -78.992 45.603
97943 97.790 -78.996 45.586
97944 97.797 -79.002 45.571
97945 97.804 -79.007 45.556
97946 97.812 -79.012 45.544
97947 97.819 -79.017 45.531
97948 97.826 -79.022 45.518
97949 97.833 -79.028 45.505
97950 97.841 -79.033 45.492
97951 97.848 -79.037 45.479
97952 97.855 -79.043 45.466
97953 97.860 -79.056 45.453
97954 97.864 -79.069 45.439
97955 97.872 -79.080 45.427
97956 97.889 -79.087 45.416
97957 97.906 -79.093 45.405
97958 97.922 -79.100 45.394
97959 97.938 -79.107 45.383
97960 97.955 -79.111 45.373
97961 97.969 -79.101 45.361
97962 97.985 -79.090 45.348
97963 98.000 -79.080 45.336
97964 98.014 -79.070 45.323
97965 98.021 -79.074 45.311
97966 98.045 -79.078 45.300
97967 98.095 -79.080 45.291
97968 98.145 -79.081 45.282
97969 98.177 -79.096 45.273
97970 98.203 -79.105 45.265
97971 98.228 -79.115 45.256
97972 98.253 -79.124 45.249
97973 98.279 -79.134 45.242
97974 98.304 -79.143 45.235
97975 98.329 -79.153 45.227
97976 98.355 -79.162 45.218
97977 98.379 -79.174 45.209
97978 98.405 -79.184 45.200
97979 98.441 -79.197 45.190
97980 98.478 -79.209 45.179
97981 98.515 -79.221 45.169
97982 98.552 -79.234 45.158
97983 98.588 -79.246 45.147
97984 98.625 -79.258 45.136
97985 98.661 -79.270 45.125
97986 98.698 -79.282 45.113
97987 98.734 -79.294 45.102
97988 98.771 -79.306 45.093
97989 98.808 -79.319 45.083
97990 98.844 -79.332 45.073
97991 98.850 -79.335 45.063
97992 98.860 -79.343 45.055
97993 98.883 -79.360 45.048
97994 98.907 -79.378 45.041
97995 98.930 -79.395 45.034
97996 98.954 -79.413 45.027
97997 98.978 -79.430 45.021
97998 99.001 -79.447 45.015
97999 99.025 -79.465 45.007
98000 99.057 -79.473 44.999
98001 99.108 -79.457 44.989
98002 99.124 -79.460 44.979
98003 99.134 -79.466 44.969
98004 99.144 -79.472 44.959
98005 99.153 -79.477 44.949
98006 99.163 -79.483 44.938
98007 99.173 -79.488 44.926
98008 99.185 -79.498 44.915
98009 99.200 -79.513 44.903
98010 99.215 -79.528 44.892
98011 99.230 -79.542 44.882
98012 99.245 -79.557 44.874
98013 99.264 -79.568 44.866
98014 99.285 -79.576 44.858
98015 99.307 -79.585 44.850
98016 99.329 -79.593 44.841
98017 99.350 -79.602 44.832
98018 99.372 -79.610 44.823
98019 99.393 -79.617 44.814
98020 99.408 -79.612 44.804
98021 99.424 -79.607 44.795
98022 99.439 -79.601 44.786
98023 99.454 -79.595 44.778
98024 99.470 -79.590 44.771
98025 99.485 -79.585 44.764
98026 99.500 -79.579 44.756
98027 99.530 -79.582 44.749
98028 99.560 -79.585 44.743
98029 99.590 -79.588 44.736
98030 99.620 -79.592 44.729
98031 99.650 -79.595 44.724
98032 99.680 -79.598 44.719
98033 99.710 -79.604 44.713
98034 99.744 -79.636 44.708
98035 99.761 -79.661 44.703
98036 99.779 -79.686 44.698
98037 99.796 -79.711 44.692
98038 99.814 -79.736 44.687
98039 99.831 -79.761 44.681
98040 99.849 -79.786 44.674
98041 99.866 -79.811 44.668
98042 99.884 -79.836 44.661
98043 99.900 -79.862 44.656
98044 99.931 -79.899 44.649
98045 99.957 -79.924 44.641
98046 99.989 -79.955 44.633
98047 100.021 -79.987 44.626
98048 100.054 -80.018 44.618
98049 100.086 -80.049 44.610
98050 100.123 -80.075 44.603
98051 100.169 -80.093 44.594
98052 100.215 -80.112 44.584
98053 100.261 -80.131 44.574
98054 100.307 -80.149 44.563
98055 100.352 -80.168 44.552
98056 100.392 -80.201 44.539
98057 100.431 -80.239 44.525
98058 100.464 -80.279 44.511
98059 100.487 -80.322 44.495
98060 100.529 -80.359 44.478
98061 100.564 -80.408 44.460
98062 100.563 -80.458 44.442
98063 100.536 -80.509 44.424
98064 100.532 -80.554 44.406
98065 100.554 -80.582 44.385
98066 100.583 -80.591 44.365
98067 100.616 -80.602 44.346
98068 100.656 -80.624 44.327
98069 100.696 -80.646 44.309
98070 100.737 -80.668 44.291
98071 100.777 -80.690 44.273
98072 100.817 -80.712 44.256
98073 100.857 -80.734 44.239
98074 100.897 -80.756 44.222
98075 100.944 -80.770 44.204
98076 100.990 -80.782 44.188
98077 101.037 -80.795 44.172
98078 101.083 -80.808 44.157
98079 101.130 -80.821 44.142
98080 101.176 -80.834 44.127
98081 101.222 -80.847 44.113
98082 101.269 -80.860 44.098
98083 101.316 -80.873 44.084
98084 101.362 -80.886 44.070
98085 101.408 -80.898 44.055
98345 93.494 -78.570 53.829
98346 93.504 -78.567 53.816
98347 93.513 -78.564 53.802
98348 93.523 -78.562 53.787
98349 93.533 -78.560 53.771
98350 93.543 -78.557 53.755
98351 93.552 -78.554 53.738
98352 93.562 -78.552 53.720
98353 93.571 -78.550 53.701
98354 93.573 -78.542 53.682
98355 93.571 -78.533 53.663
98356 93.570 -78.523 53.644
98357 93.569 -78.514 53.624
98358 93.568 -78.505 53.605
98359 93.577 -78.490 53.586
98360 93.594 -78.475 53.566
98361 93.617 -78.460 53.546
98362 93.639 -78.445 53.526
98363 93.660 -78.431 53.507
98364 93.672 -78.423 53.488
98365 93.683 -78.414 53.469
98375 93.836 -78.341 53.280
98376 93.851 -78.333 53.260
98377 93.866 -78.326 53.241
98378 93.882 -78.319 53.222
98379 93.897 -78.311 53.204
98380 93.913 -78.304 53.187
98381 93.928 -78.297 53.169
98382 93.943 -78.289 53.152
98383 93.954 -78.280 53.135
98384 93.966 -78.278 53.121
98385 93.976 -78.283 53.109
98386 93.984 -78.287 53.098
98387 93.992 -78.290 53.088
98388 94.000 -78.294 53.076
98389 94.008 -78.297 53.063
98390 94.016 -78.301 53.049
98391 94.025 -78.305 53.036
98392 94.033 -78.309 53.025
98393 94.041 -78.313 53.013
98394 94.049 -78.316 53.001
98395 94.057 -78.319 52.988
98396 94.066 -78.323 52.975
98397 94.074 -78.327 52.961
98398 94.082 -78.332 52.948
98399 94.087 -78.341 52.934
98400 94.092 -78.350 52.920
98401 94.097 -78.359 52.904
98402 94.101 -78.367 52.890
98403 94.106 -78.376 52.878
98404 94.111 -78.385 52.866
98405 94.115 -78.394 52.854
98406 94.120 -78.403 52.846
98407 94.124 -78.412 52.838
98408 94.129 -78.420 52.828
98409 94.134 -78.429 52.816
98410 94.138 -78.431 52.805
98411 94.143 -78.432 52.795
98412 94.147 -78.435 52.786
98413 94.151 -78.429 52.772
98414 94.153 -78.407 52.757
98415 94.156 -78.385 52.739
98416 94.173 -78.371 52.717
98417 94.191 -78.357 52.694
98418 94.210 -78.347 52.669
98419 94.231 -78.349 52.643
98420 94.253 -78.351 52.615
98421 94.273 -78.353 52.588
98422 94.294 -78.356 52.560
98423 94.302 -78.373 52.528
98424 94.311 -78.391 52.496
98425 94.319 -78.408 52.464
98426 94.328 -78.425 52.431
98427 94.339 -78.438 52.398
98428 94.360 -78.429 52.364
98429 94.381 -78.421 52.328
98430 94.402 -78.413 52.289
98431 94.434 -78.420 52.248
98432 94.472 -78.433 52.207
98433 94.506 -78.440 52.164
98434 94.530 -78.423 52.123
98435 94.555 -78.407 52.083
98436 94.579 -78.391 52.044
98437 94.604 -78.374 52.002
98438 94.622 -78.363 51.960
98439 94.632 -78.357 51.918
98440 94.642 -78.351 51.874
98441 94.653 -78.345 51.826
98442 94.664 -78.339 51.776
98443 94.674 -78.333 51.725
98444 94.686 -78.332 51.676
98445 94.698 -78.333 51.630
98446 94.710 -78.335 51.585
98447 94.722 -78.335 51.540
98448 94.735 -78.332 51.497
98449 94.748 -78.329 51.456
98450 94.760 -78.329 51.414
98451 94.770 -78.332 51.375
98452 94.781 -78.336 51.340
98453 94.795 -78.335 51.308
98454 94.810 -78.335 51.274
98455 94.829 -78.336 51.237
98456 94.851 -78.336 51.196
98457 94.873 -78.336 51.152
98458 94.895 -78.335 51.102
98459 94.917 -78.335 51.051
98460 94.939 -78.335 51.001
98461 94.960 -78.336 50.952
98462 94.970 -78.338 50.903
98463 94.980 -78.337 50.856
98464 94.989 -78.333 50.810
98465 94.997 -78.329 50.763
98466 95.006 -78.325 50.717
98467 95.015 -78.321 50.670
98468 95.024 -78.317 50.625
98469 95.032 -78.313 50.582
98470 95.049 -78.276 50.539
98471 95.071 -78.216 50.497
98472 95.100 -78.175 50.456
98473 95.142 -78.169 50.416
98474 95.151 -78.164 50.376
98475 95.161 -78.160 50.339
98476 95.170 -78.156 50.300
98477 95.197 -78.170 50.263
98478 95.233 -78.193 50.227
98479 95.262 -78.206 50.190
98480 95.273 -78.195 50.153
98481 95.283 -78.184 50.116
98482 95.294 -78.172 50.078
98483 95.305 -78.162 50.041
98484 95.316 -78.150 50.005
98485 95.325 -78.149 49.968
98486 95.334 -78.152 49.932
98487 95.344 -78.154 49.896
98488 95.353 -78.157 49.862
98489 95.362 -78.160 49.829
98490 95.372 -78.163 49.797
98491 95.382 -78.166 49.766
98492 95.391 -78.169 49.735
98493 95.400 -78.172 49.702
98494 95.409 -78.175 49.670
98495 95.419 -78.178 49.638
98496 95.428 -78.180 49.611
98497 95.438 -78.183 49.583
98498 95.447 -78.186 49.555
98499 95.461 -78.192 49.526
98500 95.485 -78.203 49.497
98501 95.494 -78.206 49.469
98502 95.500 -78.207 49.441
98503 95.510 -78.215 49.412
98504 95.519 -78.222 49.382
98505 95.528 -78.228 49.351
98506 95.537 -78.236 49.321
98507 95.546 -78.243 49.290
98508 95.556 -78.250 49.257
98509 95.565 -78.257 49.223
98510 95.574 -78.264 49.189
98511 95.583 -78.272 49.153
98512 95.593 -78.279 49.115
98513 95.602 -78.285 49.074
98514 95.611 -78.293 49.034
98515 95.620 -78.300 48.994
98516 95.626 -78.305 48.955
98517 95.622 -78.301 48.918
98518 95.618 -78.298 48.881
98519 95.614 -78.294 48.843
98520 95.615 -78.294 48.803
98521 95.629 -78.303 48.763
98522 95.644 -78.311 48.722
98523 95.659 -78.319 48.681
98524 95.674 -78.328 48.640
98525 95.691 -78.329 48.596
98526 95.704 -78.329 48.552
98527 95.718 -78.330 48.509
98528 95.731 -78.331 48.468
98529 95.743 -78.332 48.427
98530 95.762 -78.337 48.386
98531 95.777 -78.337 48.344
98532 95.793 -78.337 48.303
98533 95.810 -78.333 48.260
98534 95.827 -78.332 48.217
98535 95.845 -78.340 48.176
98536 95.862 -78.343 48.135
98537 95.878 -78.345 48.094
98538 95.894 -78.347 48.054
98539 95.911 -78.350 48.015
98540 95.925 -78.352 47.976
98541 95.939 -78.356 47.938
98542 95.954 -78.360 47.901
98543 95.968 -78.363 47.864
98544 95.982 -78.366 47.828
98545 95.995 -78.373 47.791
98546 96.010 -78.384 47.755
98547 96.024 -78.396 47.719
98548 96.039 -78.408 47.681
98549 96.053 -78.420 47.649
98550 96.067 -78.432 47.616
98551 96.082 -78.445 47.583
98552 96.092 -78.451 47.549
98553 96.094 -78.463 47.516
98554 96.093 -78.479 47.483
98555 96.097 -78.491 47.451
98556 96.114 -78.495 47.422
98557 96.130 -78.500 47.392
98558 96.147 -78.504 47.363
98559 96.176 -78.510 47.334
98560 96.205 -78.516 47.304
98561 96.235 -78.521 47.275
98562 96.264 -78.526 47.245
98563 96.293 -78.532 47.218
98564 96.322 -78.536 47.190
98565 96.352 -78.540 47.163
98566 96.381 -78.543 47.134
98567 96.410 -78.547 47.105
98568 96.439 -78.550 47.074
98569 96.468 -78.552 47.044
98570 96.498 -78.554 47.016
98571 96.527 -78.557 46.989
98572 96.556 -78.559 46.963
98573 96.585 -78.562 46.936
98574 96.615 -78.564 46.909
98575 96.644 -78.567 46.881
98576 96.673 -78.569 46.853
98577 96.702 -78.571 46.828
98578 96.724 -78.576 46.802
98579 96.726 -78.587 46.777
98580 96.726 -78.598 46.754
98581 96.727 -78.609 46.730
98582 96.727 -78.620 46.706
98583 96.728 -78.631 46.683
98584 96.740 -78.643 46.660
98585 96.750 -78.658 46.638
98586 96.762 -78.673 46.616
98587 96.775 -78.688 46.593
98588 96.787 -78.701 46.571
98589 96.797 -78.712 46.549
98590 96.807 -78.723 46.528
98591 96.824 -78.742 46.508
98592 96.843 -78.766 46.488
98593 96.863 -78.789 46.470
98594 96.882 -78.812 46.451
98595 96.901 -78.836 46.432
98596 96.921 -78.859 46.414
98597 96.940 -78.882 46.395
98598 96.963 -78.891 46.379
98599 96.992 -78.899 46.362
98600 97.021 -78.910 46.345
98601 97.051 -78.921 46.329
98602 97.081 -78.931 46.314
98603 97.110 -78.942 46.299
98604 97.140 -78.952 46.283
98605 97.170 -78.962 46.268
98606 97.199 -78.973 46.253
98607 97.232 -78.977 46.237
98608 97.266 -78.974 46.219
98609 97.301 -78.971 46.200
98610 97.335 -78.967 46.181
98611 97.370 -78.964 46.159
98612 97.407 -78.973 46.141
98613 97.434 -78.978 46.123
98614 97.449 -78.977 46.105
98615 97.465 -78.977 46.087
98616 97.478 -78.974 46.068
98617 97.486 -78.968 46.048
98618 97.494 -78.962 46.028
98619 97.503 -78.956 46.010
98620 97.511 -78.951 45.991
98621 97.519 -78.945 45.973
98622 97.527 -78.939 45.956
98623 97.535 -78.933 45.939
98624 97.543 -78.927 45.923
98625 97.551 -78.921 45.906
98626 97.560 -78.915 45.891
98627 97.573 -78.911 45.877
98628 97.591 -78.907 45.862
98629 97.609 -78.904 45.847
98630 97.627 -78.900 45.832
98631 97.645 -78.897 45.816
98632 97.659 -78.894 45.797
98633 97.672 -78.892 45.781
98634 97.685 -78.890 45.764
98635 97.698 -78.887 45.747
98636 97.719 -78.924 45.729
98637 97.730 -78.937 45.713
98638 97.732 -78.934 45.698
98639 97.735 -78.932 45.683
98640 97.744 -78.934 45.668
98641 97.780 -78.957 45.652
98642 97.787 -78.962 45.636
98643 97.795 -78.968 45.617
98644 97.802 -78.973 45.600
98645 97.809 -78.978 45.584
98646 97.816 -78.983 45.569
98647 97.824 -78.988 45.557
98648 97.831 -78.993 45.544
98649 97.838 -78.999 45.532
98650 97.846 -79.003 45.520
98651 97.853 -79.009 45.506
98652 97.860 -79.014 45.493
98653 97.867 -79.019 45.480
98654 97.873 -79.028 45.466
98655 97.877 -79.042 45.452
98656 97.888 -79.051 45.439
98657 97.904 -79.058 45.428
98658 97.921 -79.065 45.416
98659 97.938 -79.071 45.404
98660 97.954 -79.078 45.391
98661 97.970 -79.085 45.380
98662 97.987 -79.088 45.368
98663 98.002 -79.078 45.356
98664 98.017 -79.068 45.343
98665 98.031 -79.057 45.329
98666 98.039 -79.060 45.317
98667 98.044 -79.065 45.305
98668 98.095 -79.068 45.296
98669 98.142 -79.077 45.286
98670 98.176 -79.095 45.276
98671 98.201 -79.104 45.268
98672 98.227 -79.114 45.259
98673 98.252 -79.123 45.250
98674 98.277 -79.133 45.243
98675 98.302 -79.142 45.236
98676 98.327 -79.152 45.227
98677 98.354 -79.160 45.219
98678 98.384 -79.164 45.210
98679 98.416 -79.169 45.202
98680 98.450 -79.178 45.193
98681 98.484 -79.187 45.182
98682 98.518 -79.196 45.172
98683 98.553 -79.206 45.161
98684 98.590 -79.218 45.151
98685 98.626 -79.231 45.140
98686 98.663 -79.243 45.128
98687 98.700 -79.255 45.117
98688 98.736 -79.267 45.107
98689 98.773 -79.279 45.098
98690 98.809 -79.291 45.087
98691 98.846 -79.303 45.076
98692 98.863 -79.311 45.066
98693 98.876 -79.321 45.059
98694 98.899 -79.338 45.053
98695 98.923 -79.355 45.046
98696 98.946 -79.372 45.038
98697 98.970 -79.390 45.031
98698 98.994 -79.407 45.023
98699 99.022 -79.419 45.017
98700 99.052 -79.429 45.009
98701 99.081 -79.438 45.000
98702 99.110 -79.438 44.991
98703 99.119 -79.444 44.981
98704 99.129 -79.450 44.971
98705 99.139 -79.455 44.961
98706 99.149 -79.460 44.951
98707 99.158 -79.466 44.940
98708 99.168 -79.472 44.928
98709 99.181 -79.482 44.916
98710 99.195 -79.496 44.904
98711 99.210 -79.511 44.893
98712 99.225 -79.526 44.882
98713 99.242 -79.538 44.873
98714 99.264 -79.546 44.865
98715 99.286 -79.554 44.857
98716 99.308 -79.563 44.849
98717 99.330 -79.572 44.840
98718 99.351 -79.580 44.831
98719 99.373 -79.588 44.822
98720 99.395 -79.597 44.814
98721 99.414 -79.601 44.805
98722 99.430 -79.595 44.796
98723 99.445 -79.590 44.788
98724 99.461 -79.585 44.781
98725 99.476 -79.579 44.774
98726 99.491 -79.574 44.767
98727 99.514 -79.573 44.760
98728 99.544 -79.576 44.753
98729 99.574 -79.579 44.747
98730 99.604 -79.582 44.741
98731 99.634 -79.585 44.735
98732 99.664 -79.588 44.730
98733 99.694 -79.592 44.725
98734 99.724 -79.595 44.720
98735 99.754 -79.599 44.715
98736 99.792 -79.634 44.710
98737 99.810 -79.659 44.705
98738 99.828 -79.684 44.699
98739 99.845 -79.709 44.694
98740 99.862 -79.734 44.688
98741 99.880 -79.759 44.681
98742 99.898 -79.784 44.675
98743 99.915 -79.810 44.669
98744 99.902 -79.846 44.664
98745 99.973 -79.886 44.657
98746 99.997 -79.909 44.650
98747 100.021 -79.933 44.644
98748 100.053 -79.963 44.636
98749 100.085 -79.994 44.628
98750 100.117 -80.025 44.620
98751 100.153 -80.052 44.612
98752 100.199 -80.071 44.604
98753 100.245 -80.090 44.595
98754 100.291 -80.108 44.586
98755 100.335 -80.129 44.575
98756 100.374 -80.167 44.565
98757 100.412 -80.206 44.553
98758 100.451 -80.243 44.539
98759 100.489 -80.281 44.526
98760 100.527 -80.316 44.510
98761 100.559 -80.344 44.494
98762 100.555 -80.390 44.477
98763 100.552 -80.435 44.459
98764 100.548 -80.479 44.441
98765 100.545 -80.524 44.424
98766 100.570 -80.559 44.405
98767 100.595 -80.593 44.387
98768 100.622 -80.618 44.368
98769 100.652 -80.626 44.347
98770 100.692 -80.648 44.326
98771 100.733 -80.670 44.306
98772 100.773 -80.692 44.285
98773 100.813 -80.714 44.267
98774 100.855 -80.734 44.249
98775 100.901 -80.747 44.231
98776 100.948 -80.760 44.213
98777 100.994 -80.773 44.196
98778 101.041 -80.786 44.180
98779 101.088 -80.798 44.164
98780 101.134 -80.811 44.148
98781 101.180 -80.824 44.133
98782 101.227 -80.837 44.118
98783 101.273 -80.850 44.104
98784 101.320 -80.863 44.089
98785 101.366 -80.876 44.075
98786 101.413 -80.889 44.060
98787 101.459 -80.902 44.047
99046 93.494 -78.558 53.833
99047 93.504 -78.556 53.820
99048 93.514 -78.554 53.806
99049 93.523 -78.551 53.791
99050 93.533 -78.548 53.775
99051 93.543 -78.546 53.758
99052 93.552 -78.544 53.740
99053 93.562 -78.542 53.722
99054 93.571 -78.539 53.703
99055 93.569 -78.529 53.683
99056 93.568 -78.520 53.663
99057 93.567 -78.510 53.643
99058 93.566 -78.501 53.623
99059 93.568 -78.490 53.603
99060 93.578 -78.475 53.584
99061 93.600 -78.460 53.563
99062 93.622 -78.445 53.542
99063 93.645 -78.429 53.522
99064 93.661 -78.418 53.502
99065 93.673 -78.410 53.483
99066 93.685 -78.401 53.464
99076 93.838 -78.328 53.274
99077 93.853 -78.320 53.255
99078 93.868 -78.313 53.235
99079 93.884 -78.306 53.216
99080 93.899 -78.298 53.198
99081 93.915 -78.291 53.180
99082 93.930 -78.284 53.163
99083 93.945 -78.276 53.146
99084 93.961 -78.269 53.130
99085 93.969 -78.272 53.116
99086 93.977 -78.276 53.104
99087 93.984 -78.280 53.093
99088 93.992 -78.285 53.086
99089 93.999 -78.288 53.075
99090 94.007 -78.293 53.063
99091 94.014 -78.297 53.050
99092 94.022 -78.301 53.037
99093 94.030 -78.305 53.025
99094 94.037 -78.310 53.014
99095 94.045 -78.313 53.003
99096 94.052 -78.318 52.990
99097 94.060 -78.322 52.975
99098 94.067 -78.326 52.961
99099 94.075 -78.331 52.947
99100 94.081 -78.338 52.932
99101 94.088 -78.345 52.916
99102 94.093 -78.354 52.900
99103 94.098 -78.363 52.885
99104 94.102 -78.372 52.872
99105 94.107 -78.380 52.861
99106 94.112 -78.389 52.849
99107 94.116 -78.398 52.839
99108 94.121 -78.407 52.830
99109 94.126 -78.416 52.820
99110 94.130 -78.425 52.808
99111 94.135 -78.429 52.796
99112 94.139 -78.432 52.785
99113 94.144 -78.434 52.775
99114 94.146 -78.416 52.760
99115 94.149 -78.395 52.744
99116 94.151 -78.373 52.726
99117 94.164 -78.356 52.704
99118 94.182 -78.342 52.680
99119 94.201 -78.334 52.654
99120 94.222 -78.336 52.627
99121 94.244 -78.338 52.600
99122 94.265 -78.340 52.572
99123 94.285 -78.342 52.544
99124 94.298 -78.354 52.512
99125 94.307 -78.372 52.480
99126 94.315 -78.389 52.448
99127 94.324 -78.406 52.415
99128 94.339 -78.410 52.381
99129 94.360 -78.402 52.347
99130 94.381 -78.394 52.310
99131 94.402 -78.385 52.269
99132 94.429 -78.385 52.227
99133 94.467 -78.399 52.183
99134 94.504 -78.413 52.137
99135 94.535 -78.412 52.093
99136 94.559 -78.396 52.050
99137 94.584 -78.379 52.009
99138 94.608 -78.363 51.967
99139 94.627 -78.351 51.925
99140 94.637 -78.344 51.883
99141 94.648 -78.339 51.839
99142 94.659 -78.333 51.791
99143 94.669 -78.327 51.741
99144 94.680 -78.322 51.691
99145 94.690 -78.316 51.643
99146 94.702 -78.317 51.598
99147 94.714 -78.319 51.554
99148 94.727 -78.316 51.510
99149 94.740 -78.314 51.469
99150 94.753 -78.311 51.428
99151 94.765 -78.310 51.389
99152 94.775 -78.313 51.354
99153 94.786 -78.317 51.321
99154 94.797 -78.321 51.289
99155 94.815 -78.322 51.255
99156 94.835 -78.322 51.215
99157 94.856 -78.322 51.172
99158 94.876 -78.322 51.128
99159 94.897 -78.322 51.079
99160 94.917 -78.322 51.030
99161 94.937 -78.322 50.982
99162 94.958 -78.320 50.935
99163 94.975 -78.304 50.888
99164 94.984 -78.300 50.841
99165 94.992 -78.297 50.796
99166 95.001 -78.293 50.751
99167 95.010 -78.289 50.705
99168 95.019 -78.285 50.660
99169 95.028 -78.281 50.617
99170 95.037 -78.274 50.575
99171 95.059 -78.215 50.533
99172 95.085 -78.166 50.492
99173 95.128 -78.160 50.452
99174 95.172 -78.154 50.413
99175 95.182 -78.150 50.375
99176 95.191 -78.145 50.338
99177 95.200 -78.141 50.300
99178 95.210 -78.137 50.262
99179 95.219 -78.133 50.225
99180 95.233 -78.133 50.189
99181 95.269 -78.156 50.151
99182 95.290 -78.158 50.114
99183 95.301 -78.147 50.077
99184 95.311 -78.136 50.042
99185 95.322 -78.125 50.007
99186 95.331 -78.127 49.971
99187 95.340 -78.130 49.936
99188 95.350 -78.132 49.902
99189 95.359 -78.135 49.869
99190 95.369 -78.138 49.836
99191 95.378 -78.141 49.804
99192 95.387 -78.144 49.772
99193 95.397 -78.147 49.741
99194 95.406 -78.150 49.710
99195 95.415 -78.153 49.679
99196 95.425 -78.156 49.649
99197 95.435 -78.158 49.620
99198 95.444 -78.161 49.592
99199 95.453 -78.164 49.565
99200 95.462 -78.167 49.538
99201 95.476 -78.172 49.512
99202 95.486 -78.177 49.486
99203 95.496 -78.184 49.459
99204 95.505 -78.191 49.430
99205 95.514 -78.198 49.402
99206 95.523 -78.205 49.373
99207 95.533 -78.213 49.342
99208 95.542 -78.219 49.309
99209 95.551 -78.226 49.275
99210 95.560 -78.234 49.240
99211 95.569 -78.241 49.205
99212 95.579 -78.247 49.169
99213 95.588 -78.255 49.130
99214 95.597 -78.262 49.088
99215 95.606 -78.269 49.045
99216 95.615 -78.276 49.003
99217 95.624 -78.283 48.963
99218 95.621 -78.281 48.924
99219 95.617 -78.277 48.887
99220 95.613 -78.274 48.849
99221 95.620 -78.277 48.809
99222 95.634 -78.285 48.771
99223 95.649 -78.294 48.731
99224 95.664 -78.303 48.690
99225 95.678 -78.311 48.649
99226 95.692 -78.315 48.604
99227 95.705 -78.316 48.560
99228 95.719 -78.316 48.517
99229 95.732 -78.317 48.476
99230 95.745 -78.318 48.435
99231 95.760 -78.315 48.394
99232 95.777 -78.311 48.352
99233 95.795 -78.307 48.311
99234 95.812 -78.305 48.269
99235 95.830 -78.313 48.226
99236 95.848 -78.321 48.185
99237 95.865 -78.326 48.144
99238 95.881 -78.328 48.104
99239 95.897 -78.330 48.066
99240 95.913 -78.333 48.029
99241 95.928 -78.337 47.991
99242 95.943 -78.341 47.952
99243 95.958 -78.344 47.915
99244 95.972 -78.347 47.879
99245 95.987 -78.351 47.842
99246 96.002 -78.360 47.806
99247 96.016 -78.374 47.770
99248 96.030 -78.386 47.733
99249 96.045 -78.398 47.695
99250 96.059 -78.410 47.662
99251 96.074 -78.423 47.630
99252 96.088 -78.435 47.597
99253 96.098 -78.440 47.564
99254 96.103 -78.451 47.531
99255 96.102 -78.467 47.498
99256 96.109 -78.476 47.466
99257 96.126 -78.481 47.436
99258 96.143 -78.485 47.407
99259 96.159 -78.489 47.379
99260 96.177 -78.494 47.351
99261 96.206 -78.500 47.321
99262 96.236 -78.505 47.291
99263 96.265 -78.511 47.261
99264 96.294 -78.515 47.233
99265 96.323 -78.518 47.206
99266 96.353 -78.522 47.179
99267 96.382 -78.525 47.151
99268 96.410 -78.529 47.122
99269 96.440 -78.532 47.093
99270 96.469 -78.534 47.064
99271 96.499 -78.536 47.035
99272 96.528 -78.539 47.008
99273 96.557 -78.541 46.982
99274 96.586 -78.543 46.954
99275 96.613 -78.546 46.927
99276 96.640 -78.549 46.899
99277 96.665 -78.552 46.871
99278 96.691 -78.555 46.846
99279 96.703 -78.564 46.820
99280 96.707 -78.574 46.793
99281 96.710 -78.586 46.769
99282 96.713 -78.596 46.745
99283 96.716 -78.608 46.721
99284 96.718 -78.619 46.698
99285 96.725 -78.631 46.675
99286 96.738 -78.646 46.653
99287 96.750 -78.661 46.632
99288 96.763 -78.675 46.609
99289 96.774 -78.687 46.586
99290 96.785 -78.698 46.565
99291 96.802 -78.720 46.544
99292 96.822 -78.742 46.524
99293 96.841 -78.766 46.505
99294 96.861 -78.789 46.485
99295 96.880 -78.812 46.466
99296 96.899 -78.836 46.447
99297 96.919 -78.859 46.428
99298 96.938 -78.882 46.409
99299 96.959 -78.900 46.393
99300 96.984 -78.902 46.375
99301 97.014 -78.913 46.358
99302 97.043 -78.924 46.342
99303 97.073 -78.934 46.326
99304 97.103 -78.945 46.310
99305 97.133 -78.955 46.295
99306 97.162 -78.966 46.279
99307 97.192 -78.977 46.264
99308 97.226 -78.974 46.247
99309 97.261 -78.971 46.230
99310 97.296 -78.968 46.211
99311 97.330 -78.965 46.192
99312 97.365 -78.962 46.173
99313 97.399 -78.959 46.154
99314 97.437 -78.971 46.134
99315 97.455 -78.971 46.115
99316 97.471 -78.970 46.097
99317 97.486 -78.969 46.078
99318 97.494 -78.963 46.059
99319 97.502 -78.957 46.040
99320 97.510 -78.951 46.023
99321 97.518 -78.945 46.004
99322 97.526 -78.940 45.984
99323 97.535 -78.933 45.967
99324 97.543 -78.927 45.950
99325 97.551 -78.921 45.932
99326 97.560 -78.915 45.914
99327 97.574 -78.912 45.898
99328 97.591 -78.908 45.883
99329 97.609 -78.905 45.867
99330 97.627 -78.901 45.852
99331 97.645 -78.898 45.836
99332 97.663 -78.894 45.820
99333 97.677 -78.892 45.803
99334 97.690 -78.890 45.787
99335 97.704 -78.886 45.771
99336 97.717 -78.884 45.755
99337 97.730 -78.882 45.738
99338 97.750 -78.916 45.722
99339 97.753 -78.914 45.708
99340 97.755 -78.911 45.695
99341 97.784 -78.927 45.679
99342 97.792 -78.933 45.664
99343 97.799 -78.939 45.648
99344 97.807 -78.944 45.629
99345 97.814 -78.949 45.612
99346 97.821 -78.954 45.597
99347 97.828 -78.959 45.582
99348 97.836 -78.965 45.571
99349 97.843 -78.969 45.559
99350 97.850 -78.974 45.547
99351 97.858 -78.980 45.535
99352 97.865 -78.985 45.521
99353 97.872 -78.990 45.507
99354 97.879 -78.995 45.494
99355 97.886 -79.001 45.479
99356 97.890 -79.015 45.466
99357 97.903 -79.023 45.453
99358 97.920 -79.030 45.440
99359 97.937 -79.037 45.427
99360 97.953 -79.043 45.413
99361 97.969 -79.050 45.399
99362 97.986 -79.057 45.387
99363 98.003 -79.064 45.375
99364 98.019 -79.065 45.363
99365 98.033 -79.055 45.350
99366 98.049 -79.045 45.336
99367 98.056 -79.047 45.323
99368 98.062 -79.053 45.311
99369 98.094 -79.056 45.301
99370 98.138 -79.074 45.290
99371 98.174 -79.093 45.280
99372 98.200 -79.103 45.271
99373 98.225 -79.112 45.262
99374 98.250 -79.122 45.253
99375 98.276 -79.132 45.244
99376 98.305 -79.137 45.237
99377 98.335 -79.141 45.229
99378 98.366 -79.146 45.220
99379 98.396 -79.150 45.210
99380 98.428 -79.156 45.201
99381 98.462 -79.165 45.192
99382 98.496 -79.174 45.183
99383 98.530 -79.182 45.174
99384 98.564 -79.191 45.165
99385 98.598 -79.200 45.155
99386 98.632 -79.209 45.143
99387 98.666 -79.218 45.132
99388 98.701 -79.228 45.120
99389 98.738 -79.240 45.109
99390 98.774 -79.252 45.098
99391 98.811 -79.264 45.088
99392 98.848 -79.276 45.077
99393 98.877 -79.287 45.067
99394 98.891 -79.298 45.060
99395 98.915 -79.316 45.053
99396 98.939 -79.333 45.046
99397 98.965 -79.347 45.039
99398 98.994 -79.357 45.031
99399 99.024 -79.367 45.023
99400 99.047 -79.379 45.017
99401 99.066 -79.391 45.008
99402 99.086 -79.404 45.000
99403 99.101 -79.416 44.991
99404 99.115 -79.427 44.982
99405 99.124 -79.433 44.971
99406 99.134 -79.438 44.961
99407 99.144 -79.444 44.951
99408 99.154 -79.450 44.939
99409 99.163 -79.456 44.927
99410 99.175 -79.465 44.915
99411 99.191 -79.480 44.903
99412 99.205 -79.494 44.892
99413 99.222 -79.507 44.881
99414 99.243 -79.516 44.871
99415 99.265 -79.524 44.863
99416 99.287 -79.532 44.855
99417 99.309 -79.541 44.847
99418 99.331 -79.550 44.839
99419 99.352 -79.558 44.830
99420 99.374 -79.567 44.821
99421 99.396 -79.575 44.813
99422 99.417 -79.583 44.804
99423 99.435 -79.585 44.797
99424 99.451 -79.579 44.790
99425 99.466 -79.574 44.783
99426 99.482 -79.568 44.776
99427 99.498 -79.563 44.769
99428 99.528 -79.567 44.763
99429 99.558 -79.570 44.757
99430 99.588 -79.573 44.751
99431 99.618 -79.576 44.745
99432 99.648 -79.579 44.740
99433 99.678 -79.582 44.735
99434 99.708 -79.586 44.731
99435 99.738 -79.589 44.726
99436 99.768 -79.592 44.722
99437 99.798 -79.595 44.717
99438 99.837 -79.629 44.712
99439 99.859 -79.657 44.706
99440 99.876 -79.682 44.701
99441 99.894 -79.707 44.696
99442 99.911 -79.733 44.689
99443 99.929 -79.758 44.683
99444 99.930 -79.789 44.677
99445 99.930 -79.833 44.671
99446 100.015 -79.874 44.665
99447 100.039 -79.896 44.659
99448 100.063 -79.920 44.653
99449 100.087 -79.943 44.645
99450 100.116 -79.971 44.638
99451 100.148 -80.002 44.630
99452 100.184 -80.030 44.622
99453 100.229 -80.048 44.614
99454 100.275 -80.067 44.606
99455 100.317 -80.096 44.598
99456 100.355 -80.134 44.588
99457 100.394 -80.171 44.577
99458 100.431 -80.206 44.565
99459 100.466 -80.233 44.553
99460 100.500 -80.261 44.539
99461 100.535 -80.288 44.523
99462 100.569 -80.316 44.507
99463 100.567 -80.360 44.491
99464 100.563 -80.405 44.473
99465 100.560 -80.450 44.455
99466 100.562 -80.493 44.436
99467 100.587 -80.527 44.418
99468 100.612 -80.562 44.400
99469 100.637 -80.597 44.381
99470 100.662 -80.632 44.362
99471 100.690 -80.653 44.342
99472 100.728 -80.673 44.320
99473 100.769 -80.695 44.298
99474 100.813 -80.711 44.278
99475 100.859 -80.724 44.259
99476 100.906 -80.737 44.241
99477 100.952 -80.750 44.223
99478 100.998 -80.763 44.205
99479 101.045 -80.776 44.189
99480 101.092 -80.789 44.172
99481 101.138 -80.801 44.156
99482 101.184 -80.814 44.140
99483 101.231 -80.827 44.125
99484 101.277 -80.840 44.110
99485 101.324 -80.853 44.095
99486 101.370 -80.866 44.080
99487 101.417 -80.879 44.066
99488 101.463 -80.892 44.052
99489 101.510 -80.905 44.037
99747 93.495 -78.548 53.838
99748 93.504 -78.545 53.824
99749 93.514 -78.542 53.809
99750 93.523 -78.540 53.794
99751 93.534 -78.538 53.777
99752 93.543 -78.535 53.760
99753 93.553 -78.532 53.741
99754 93.562 -78.530 53.723
99755 93.567 -78.525 53.703
99756 93.566 -78.516 53.683
99757 93.565 -78.507 53.663
99758 93.563 -78.498 53.642
99759 93.562 -78.489 53.620
99760 93.569 -78.474 53.600
99761 93.583 -78.459 53.580
99762 93.606 -78.444 53.559
99763 93.628 -78.429 53.538
99764 93.650 -78.413 53.518
99765 93.662 -78.405 53.498
99766 93.674 -78.397 53.478
99767 93.686 -78.388 53.459
99777 93.840 -78.315 53.267
99778 93.855 -78.307 53.249
99779 93.870 -78.300 53.229
99780 93.886 -78.293 53.210
99781 93.901 -78.285 53.191
99782 93.917 -78.279 53.173
99783 93.932 -78.271 53.155
99784 93.947 -78.263 53.138
99785 93.962 -78.257 53.124
99786 93.967 -78.262 53.110
99787 93.970 -78.272 53.097
99788 93.973 -78.282 53.087
99789 93.979 -78.288 53.082
99790 93.987 -78.293 53.073
99791 93.994 -78.297 53.062
99792 94.002 -78.301 53.049
99793 94.010 -78.305 53.036
99794 94.017 -78.310 53.024
99795 94.025 -78.313 53.013
99796 94.033 -78.318 53.001
99797 94.040 -78.322 52.987
99798 94.048 -78.326 52.972
99799 94.055 -78.330 52.956
99800 94.063 -78.335 52.942
99801 94.070 -78.340 52.926
99802 94.076 -78.347 52.908
99803 94.083 -78.354 52.891
99804 94.089 -78.362 52.875
99805 94.096 -78.369 52.861
99806 94.103 -78.376 52.850
99807 94.108 -78.385 52.839
99808 94.113 -78.393 52.828
99809 94.117 -78.402 52.818
99810 94.122 -78.411 52.807
99811 94.127 -78.420 52.794
99812 94.131 -78.429 52.782
99813 94.136 -78.431 52.770
99814 94.140 -78.425 52.757
99815 94.142 -78.403 52.743
99816 94.144 -78.382 52.726
99817 94.147 -78.360 52.707
99818 94.154 -78.341 52.685
99819 94.173 -78.327 52.660
99820 94.192 -78.322 52.635
99821 94.213 -78.323 52.608
99822 94.235 -78.326 52.580
99823 94.256 -78.328 52.552
99824 94.276 -78.329 52.524
99825 94.294 -78.335 52.492
99826 94.303 -78.353 52.461
99827 94.312 -78.370 52.429
99828 94.320 -78.387 52.395
99829 94.339 -78.383 52.361
99830 94.360 -78.375 52.325
99831 94.381 -78.366 52.288
99832 94.402 -78.358 52.246
99833 94.424 -78.351 52.202
99834 94.462 -78.366 52.157
99835 94.499 -78.379 52.109
99836 94.537 -78.394 52.062
99837 94.564 -78.385 52.016
99838 94.589 -78.368 51.971
99839 94.613 -78.351 51.929
99840 94.632 -78.339 51.887
99841 94.642 -78.333 51.845
99842 94.653 -78.327 51.801
99843 94.664 -78.321 51.753
99844 94.674 -78.316 51.705
99845 94.685 -78.310 51.657
99846 94.695 -78.304 51.611
99847 94.706 -78.300 51.566
99848 94.718 -78.302 51.523
99849 94.732 -78.299 51.481
99850 94.744 -78.296 51.441
99851 94.757 -78.294 51.401
99852 94.770 -78.291 51.364
99853 94.780 -78.294 51.331
99854 94.791 -78.299 51.299
99855 94.801 -78.304 51.267
99856 94.812 -78.308 51.231
99857 94.825 -78.312 51.190
99858 94.845 -78.312 51.147
99859 94.865 -78.312 51.103
99860 94.886 -78.312 51.057
99861 94.906 -78.312 51.010
99862 94.927 -78.313 50.964
99863 94.947 -78.305 50.918
99864 94.968 -78.284 50.873
99865 94.988 -78.264 50.828
99866 94.996 -78.260 50.783
99867 95.005 -78.257 50.739
99868 95.014 -78.253 50.695
99869 95.023 -78.249 50.651
99870 95.031 -78.244 50.609
99871 95.047 -78.213 50.568
99872 95.070 -78.157 50.527
99873 95.114 -78.151 50.488
99874 95.157 -78.146 50.448
99875 95.201 -78.140 50.410
99876 95.213 -78.135 50.372
99877 95.222 -78.131 50.335
99878 95.232 -78.127 50.297
99879 95.241 -78.122 50.260
99880 95.250 -78.118 50.224
99881 95.259 -78.114 50.187
99882 95.269 -78.109 50.151
99883 95.278 -78.106 50.115
99884 95.305 -78.119 50.079
99885 95.317 -78.110 50.044
99886 95.328 -78.102 50.010
99887 95.337 -78.105 49.975
99888 95.346 -78.108 49.942
99889 95.356 -78.111 49.909
99890 95.365 -78.113 49.876
99891 95.375 -78.116 49.844
99892 95.384 -78.119 49.812
99893 95.393 -78.122 49.781
99894 95.403 -78.125 49.751
99895 95.412 -78.128 49.722
99896 95.421 -78.131 49.692
99897 95.431 -78.134 49.663
99898 95.442 -78.138 49.634
99899 95.452 -78.142 49.607
99900 95.462 -78.147 49.581
99901 95.471 -78.150 49.556
99902 95.477 -78.154 49.531
99903 95.483 -78.155 49.506
99904 95.491 -78.160 49.480
99905 95.500 -78.167 49.453
99906 95.510 -78.175 49.424
99907 95.519 -78.181 49.394
99908 95.528 -78.188 49.362
99909 95.537 -78.196 49.328
99910 95.546 -78.203 49.293
99911 95.555 -78.210 49.257
99912 95.564 -78.217 49.220
99913 95.573 -78.224 49.181
99914 95.583 -78.232 49.141
99915 95.592 -78.238 49.099
99916 95.601 -78.245 49.057
99917 95.610 -78.253 49.015
99918 95.619 -78.260 48.972
99919 95.620 -78.260 48.931
99920 95.616 -78.257 48.891
99921 95.612 -78.253 48.852
99922 95.624 -78.260 48.813
99923 95.639 -78.269 48.775
99924 95.654 -78.277 48.736
99925 95.669 -78.286 48.696
99926 95.686 -78.298 48.655
99927 95.703 -78.304 48.613
99928 95.706 -78.302 48.570
99929 95.719 -78.303 48.526
99930 95.732 -78.304 48.484
99931 95.745 -78.304 48.442
99932 95.762 -78.285 48.400
99933 95.779 -78.282 48.359
99934 95.797 -78.278 48.317
99935 95.815 -78.285 48.275
99936 95.833 -78.294 48.234
99937 95.851 -78.302 48.195
99938 95.868 -78.308 48.156
99939 95.884 -78.310 48.117
99940 95.900 -78.313 48.079
99941 95.917 -78.316 48.042
99942 95.934 -78.320 48.005
99943 95.948 -78.324 47.966
99944 95.963 -78.327 47.929
99945 95.979 -78.330 47.893
99946 95.995 -78.332 47.856
99947 96.010 -78.344 47.820
99948 96.022 -78.362 47.783
99949 96.036 -78.376 47.746
99950 96.050 -78.388 47.709
99951 96.065 -78.401 47.676
99952 96.079 -78.413 47.643
99953 96.094 -78.425 47.611
99954 96.104 -78.430 47.578
99955 96.112 -78.438 47.545
99956 96.111 -78.454 47.512
99957 96.121 -78.462 47.481
99958 96.138 -78.466 47.451
99959 96.155 -78.470 47.422
99960 96.171 -78.475 47.394
99961 96.187 -78.479 47.367
99962 96.207 -78.484 47.337
99963 96.237 -78.489 47.307
99964 96.266 -78.493 47.278
99965 96.295 -78.497 47.249
99966 96.324 -78.500 47.221
99967 96.354 -78.504 47.194
99968 96.383 -78.507 47.167
99969 96.411 -78.510 47.140
99970 96.441 -78.513 47.111
99971 96.470 -78.516 47.082
99972 96.498 -78.518 47.054
99973 96.523 -78.521 47.027
99974 96.549 -78.524 47.000
99975 96.576 -78.527 46.972
99976 96.601 -78.530 46.944
99977 96.627 -78.533 46.916
99978 96.654 -78.536 46.889
99979 96.680 -78.540 46.863
99980 96.683 -78.551 46.838
99981 96.686 -78.562 46.813
99982 96.689 -78.573 46.789
99983 96.692 -78.584 46.763
99984 96.700 -78.598 46.738
99985 96.708 -78.608 46.714
99986 96.719 -78.616 46.691
99987 96.726 -78.633 46.669
99988 96.739 -78.648 46.648
99989 96.751 -78.662 46.625
99990 96.762 -78.673 46.603
99991 96.781 -78.696 46.582
99992 96.800 -78.720 46.562
99993 96.819 -78.742 46.542
99994 96.839 -78.766 46.522
99995 96.858 -78.789 46.503
99996 96.878 -78.812 46.484
99997 96.897 -78.836 46.464
99998 96.916 -78.859 46.445
99999 96.936 -78.882 46.426
100000 96.955 -78.905 46.409
100001 96.979 -78.911 46.391
100002 97.006 -78.917 46.374
100003 97.036 -78.927 46.357
100004 97.066 -78.938 46.339
100005 97.095 -78.949 46.323
100006 97.125 -78.959 46.307
100007 97.155 -78.969 46.291
100008 97.186 -78.975 46.275
100009 97.221 -78.972 46.258
100010 97.255 -78.968 46.241
100011 97.290 -78.965 46.223
100012 97.325 -78.962 46.204
100013 97.360 -78.959 46.186
100014 97.394 -78.955 46.168
100015 97.430 -78.955 46.148
100016 97.461 -78.965 46.128
100017 97.476 -78.964 46.109
100018 97.492 -78.963 46.090
100019 97.502 -78.958 46.070
100020 97.510 -78.952 46.051
100021 97.518 -78.946 46.034
100022 97.526 -78.940 46.015
100023 97.534 -78.933 45.996
100024 97.542 -78.928 45.979
100025 97.550 -78.922 45.961
100026 97.559 -78.916 45.942
100027 97.574 -78.912 45.923
100028 97.592 -78.908 45.906
100029 97.609 -78.905 45.890
100030 97.628 -78.902 45.873
100031 97.645 -78.899 45.857
100032 97.663 -78.895 45.842
100033 97.681 -78.892 45.826
100034 97.696 -78.889 45.809
100035 97.709 -78.886 45.794
100036 97.722 -78.884 45.777
100037 97.735 -78.882 45.761
100038 97.749 -78.880 45.746
100039 97.762 -78.877 45.731
100040 97.774 -78.893 45.717
100041 97.788 -78.899 45.704
100042 97.797 -78.905 45.689
100043 97.804 -78.910 45.673
100044 97.811 -78.915 45.658
100045 97.818 -78.920 45.640
100046 97.826 -78.925 45.624
100047 97.833 -78.930 45.610
100048 97.841 -78.936 45.597
100049 97.848 -78.940 45.586
100050 97.855 -78.946 45.575
100051 97.862 -78.951 45.563
100052 97.869 -78.956 45.550
100053 97.877 -78.961 45.536
100054 97.884 -78.966 45.523
100055 97.891 -78.971 45.509
100056 97.898 -78.977 45.494
100057 97.903 -78.987 45.480
100058 97.919 -78.994 45.466
100059 97.936 -79.001 45.452
100060 97.952 -79.008 45.438
100061 97.968 -79.015 45.423
100062 97.985 -79.021 45.408
100063 98.002 -79.028 45.394
100064 98.018 -79.035 45.381
100065 98.034 -79.042 45.368
100066 98.051 -79.043 45.356
100067 98.066 -79.032 45.342
100068 98.073 -79.034 45.329
100069 98.079 -79.040 45.317
100070 98.093 -79.046 45.306
100071 98.134 -79.071 45.295
100072 98.173 -79.093 45.285
100073 98.199 -79.103 45.274
100074 98.225 -79.110 45.265
100075 98.256 -79.114 45.256
100076 98.286 -79.118 45.246
100077 98.316 -79.122 45.238
100078 98.347 -79.127 45.229
100079 98.377 -79.131 45.220
100080 98.407 -79.135 45.210
100081 98.440 -79.143 45.201
100082 98.474 -79.152 45.192
100083 98.508 -79.160 45.183
100084 98.543 -79.169 45.176
100085 98.576 -79.178 45.167
100086 98.610 -79.187 45.157
100087 98.645 -79.196 45.147
100088 98.678 -79.204 45.135
100089 98.713 -79.213 45.123
100090 98.746 -79.222 45.112
100091 98.781 -79.231 45.100
100092 98.815 -79.240 45.090
100093 98.849 -79.249 45.080
100094 98.886 -79.261 45.069
100095 98.908 -79.275 45.060
100096 98.934 -79.287 45.052
100097 98.954 -79.299 45.044
100098 98.975 -79.311 45.037
100099 98.995 -79.323 45.030
100100 99.015 -79.335 45.022
100101 99.034 -79.347 45.015
100102 99.055 -79.360 45.007
100103 99.071 -79.375 44.998
100104 99.083 -79.380 44.990
100105 99.100 -79.397 44.981
100106 99.118 -79.414 44.970
100107 99.130 -79.422 44.959
100108 99.139 -79.428 44.949
100109 99.149 -79.433 44.936
100110 99.159 -79.439 44.924
100111 99.171 -79.448 44.912
100112 99.186 -79.463 44.900
100113 99.201 -79.477 44.890
100114 99.223 -79.485 44.880
100115 99.244 -79.494 44.870
100116 99.266 -79.503 44.861
100117 99.288 -79.511 44.852
100118 99.309 -79.520 44.845
100119 99.331 -79.528 44.836
100120 99.353 -79.536 44.828
100121 99.375