#!/usr/bin/env perl
use strict;
use warnings;
use 5.010;

use IO::All;
use Data::Dumper;
use YAML::PP::Loader;
use File::Basename qw/ basename /;
use HTML::Entities qw/ encode_entities /;
use YAML::PP::Highlight;
use Encode;

my @dirs = grep { m{/[0-9A-Z]{4}$} } map { "$_" } io->dir('data')->all;
my @valid = grep { not -f "$_/error" } @dirs;
my @invalid = grep { -f "$_/error" } @dirs;

my %tags;
for my $tagdir (io->dir('data/tags')->all) {
    for my $id (io->dir($tagdir)->all) {
        my $tag = basename $tagdir;
        push @{ $tags{ basename $id } }, basename $tag;
    }
}

my $html = <<"EOM";
<html>
<head>
<title>YAML Test Suite Highlighted</title>
<link rel="stylesheet" type="text/css" href="css/yaml.css">
<style>
body {
    font-family: Arial;
}
table.highlight {
    border: 1px solid #bbb;
    border-collapse: collapse;
    box-shadow: 2px 2px 4px 1px grey;
}
table.highlight tr th, table.highlight tr td {
    border: 1px solid #bbb;
    background-color: white;
    padding: 2px 3px 2px 3px;

}
td.error {
    background-color: #ff7777;
}
td.diff {
    background-color: #ffff77;
}
td.ok {
    background-color: #7777ff;
}
span.anchor { color: green; }
span.indent { background-color: #e8e8e8; }
span.dash { font-weight: bold; color: magenta; }
span.colon { font-weight: bold; color: magenta; }
span.question { font-weight: bold; color: magenta; }
span.yaml_directive { color: cyan; }
span.tag_directive { color: cyan; }
span.tag { color: blue; }
span.comment { color: grey; }
span.alias { color: green; }
span.singlequote { font-weight: bold; color: green; }
span.doublequote { font-weight: bold; color: green; }
span.singlequoted { color: green; }
span.doublequoted { color: green; }
span.literal { font-weight: bold; color: magenta; }
span.folded { font-weight: bold; color: magenta; }
span.doc_start { font-weight: bold; }
span.doc_end { font-weight: bold; }
span.block_scalar_content { color: #aa7700; }
span.tab { background-color: lightblue; }
span.error { background-color: #ff8888; }

pre {
    background-color: white;
    padding: 2px;
}
pre.error {
    border: 1px solid red;
}
pre.diff {
    border: 1px solid #ff9900;
}
</style>
</head>
<body>
<a href="#valid">Valid</a><br>
<a href="#invalid">Invalid</a><br>
EOM

my $ypp = YAML::PP::Loader->new;
my $table;
for my $dir (sort @valid) {
    my $test = highlight_test($dir);
    $table .= $test;
}
$html .= <<"EOM";
<h1><a name="valid">Valid</a></h1>
<table class="highlight">
$table
</table>
EOM
$table = '';

for my $dir (sort @invalid) {
    my $test = highlight_test($dir);
    $table .= $test;
}
$html .= <<"EOM";
<h1><a name="invalid">Invalid</a></h1>
<table class="highlight">
$table
</table>
EOM

sub highlight_test {
    my ($dir) = @_;
    my $html;
    my $file = "$dir/in.yaml";
    my $id = basename $dir;
    my $title = io->file("$dir/===")->slurp;
    my $yaml;

    $yaml = do { open my $fh, '<', $file or die $!; local $/; <$fh> };

    my $class = "ok";
    eval {
        my @docs = $ypp->load($yaml);
    };
    my $error = $@ || '';
    my $tokens = $ypp->parser->tokens;
    my $remaining = '';
    if ($error) {
        $class = "error";
        $remaining = $ypp->parser->yaml;
        $remaining = $$remaining;
        push @$tokens, ['ERROR', $remaining];
        my $out = join '', map { $_->[1] } @$tokens;
        if ($out ne $yaml) {
            warn "$id error diff";
        }
    }
    else {
        my $out = join '', map { $_->[1] } @$tokens;
        if ($out ne $yaml) {
            $class = "diff";
            say "$id diff";
        }
    }
    $title = decode_utf8($title);
    $title = encode_entities($title);
    my $taglist = join ', ', @{ $tags{ $id } || [] };
    $html .= <<"EOM";
<tr><td style="width: 30em;" valign="top" ><b>$id - $title</b><br>
Tags: $taglist<br>
<a href="https://github.com/yaml/yaml-test-suite/blob/master/test/$id.tml">View source</a><br>
</td>
EOM
    my $high = YAML::PP::Highlight->htmlcolored($tokens);
    $html .= <<"EOM";
<td valign="top"><pre class="$class">$high</pre>
$error
</td></tr>
EOM
#    $yaml = decode_utf8($yaml);
#    $yaml = encode_entities($yaml);
#    $html .= qq{<tr><td valign="top"><pre>$yaml</pre></td>\n};
    return $html;

}

$html .= <<"EOM";
</body></html>
EOM

binmode STDOUT, ":encoding(utf-8)";
say $html;
