package IPC::Semaphore::Set::Resource;
use strict;
use warnings;

use 5.008;
use IPC::SysV qw(SEM_UNDO IPC_NOWAIT);

our $VERSION = 0.01;

############
## Public ##
############

sub new
{
	my $class = shift;
	my $args  = ref($_[0]) ? $_[0] : {@_};
	if (ref($args->{set}) ne 'IPC::Semaphore::Set') {
		die "'set' is a required field and must be a IPC::Semaphore::Set";
	}
	if (!defined($args->{number})) {
		die "'number' is a required field, you must provide the resource number you want in the set";
	}
	my $self = bless($args, $class);
	if (!defined($self->available)) {
		my $total = () = $self->set->sem->getall;
		die $self->number . ' is not a valid resource for semaphore [' . $self->set->key
			. "] which only has [$total] total resources. The resources start at 0.";
	}
	return bless($args, $class);
}

sub lock
{
	my $self = shift;
	my $lock = $self->set->sem->op($self->number, -1, SEM_UNDO | IPC_NOWAIT);
	if ($lock) {
		$self->{_locks}++;
		return 1;
	} else {
		return 0;
	}
}

sub lockWait
{
	my $self = shift;
	my $lock = $self->set->sem->op($self->number, -1, SEM_UNDO);
	if ($lock) {
		$self->{_locks}++;
		return 1;
	} else {
		return 0;
	}
}

sub lockOrDie
{
	my $self = shift;
	if ($self->set->sem->op($self->number, -1, SEM_UNDO | IPC_NOWAIT)) {
		$self->{_locks}++;
		return 1;
	} else {
		die 'could not lock on semaphore [' . $self->set->key . '] resource number [' . $self->number . "]\n";
	}
}

sub unlock
{
	my $self   = shift;
	die 'tried to unlock resource [' . $self->number . '] on semaphore [' . $self->set->key . '] too many times'
		if ($self->{_locks} <= 0);
	my $unlock = $self->set->sem->op($self->number, 1, SEM_UNDO |IPC_NOWAIT);
	if ($unlock) {
		$self->{_locks}--;
		return 1;
	} else {
		die 'could not unlock on semaphore [' . $self->set->key . '] resource number [' . $self->number . ']';
	}
}

############
## Helper ##
############

sub available {
	my $self = shift;
	return $self->set->sem->getval($self->number);
}

sub number {return shift->{number}}
sub set    {return shift->{set}}
sub value  {return shift->{value}}

#############
## Private ##
#############

1;

__END__

=head1 NAME

IPC::Semaphore::Set::Resource;

=head1 DESCRIPTION

A simple interface to resources available in a semaphore set.

It assumes to use SEM_UNDO in all cases, so when the program
exits it should give up its locks.

=head1 SYNOPSIS

Get a resource from a IPC::Semaphore::Set object:

	my $semset   = IPC::Semaphore::Set->new;
	my $resource = $semset->resource($number);

Attempt to get a lock on the resource:

	if ($resource->lock) {
		# ... got a lock
	} else {
		# ... couldn't lock resource
	}

=head1 METHODS

=over

=item new

Return a new IPC::Semaphore::Set::Resource, but you probably want to be calling
'->resource' on an IPC::Semaphore::Set object to get this.

=item available

Returns the value of the resource (current availability in terms of greater than 0)

=item lock

Returns boolean as to whether a lock was possible.

=item lockWait

Returns boolean, but waits for the resource to become available.

=item lockOrDie

Returns 1 or dies if lock can not be made.

=item number

Returns the resources's number in the semaphore set.

=item set

Returns the IPC::Semaphore::Set object the resource belongs to.

=item unlock

Returns 1 or dies if we for some reason can't unlock on the resource.

=item value

Same thing as available, but uses the traditional nomenclature.

=back

=cut

