package IPC::Semaphore::Set;
use strict;
use warnings;

use 5.008;
use IPC::SysV qw(IPC_PRIVATE IPC_CREAT IPC_NOWAIT SEM_UNDO S_IRUSR S_IWUSR);
use IPC::Semaphore;
use IPC::Semaphore::Set::Resource;

our $VERSION = 0.01;

############
## Public ##
############

sub new
{
	my $class = shift;
	my $args  = ref($_[0]) ? $_[0] : {@_};
	$args->{available} = $args->{available} ? $args->{available} : 1;
	$args->{resources} = $args->{resources} ? $args->{resources} : 1;
	$args->{flags}     = $args->{flags}     ? $args->{flags}     : S_IRUSR | S_IWUSR | IPC_CREAT | SEM_UNDO;
	my $self = bless($args, $class);
	if (my $key = $self->{key}) {
		die "key [$key] was not a number" if ($key =~ m/[^0-9]/);
		$self->{_pre_exist} = semget($key, 0, IPC_NOWAIT);
		$self->{_key}       = $key;
		$self->{_semaphore} = IPC::Semaphore->new($key, $self->{resources}, $self->{flags});
	} elsif (my $key_name = $self->{key_name}) {
		$self->{_key_name}  = $key_name;
		$self->{_key}       = $self->_name_to_key($key_name);
		$self->{_pre_exist} = semget($self->{_key}, 0, IPC_NOWAIT);
		$self->{_semaphore} = IPC::Semaphore->new($self->key, $self->{resources}, $self->{flags});
	} else {
		$self->{_semaphore} = IPC::Semaphore->new(IPC_PRIVATE, $self->{resources}, $self->{flags});
	}
	die "could not get semaphore: $!" if (ref($self->sem) ne 'IPC::Semaphore');
	unless ($self->{_pre_exist}) {
		$self->sem->setall(($self->{available}) x $self->{resources});
	}
	return $self;
}

sub resource
{
	my $self   = shift;
	my $number = shift;
	if (!$self->{_current_resource} || (defined($number) && $self->{_current_resource}->number != $number)) {
		$self->{_current_resource} = $self->_resource($number || 0);
	}
	return $self->{_current_resource};
}

sub resources
{
	my $self  = shift;
	my $total = () = $self->sem->getall;
	my @resources;
	for (my $count = 0; $count <= $total - 1; $count++) {
		push(@resources, $self->_resource($count));
	}
	return wantarray ? @resources : \@resources;
}

############
## Helper ##
############

sub id      {return shift->sem->id}
sub key     {return shift->{_key}}
sub keyName {return shift->{_key_name}}
sub remove  {return shift->sem->remove ? 1 : 0}
sub sem     {return shift->{_semaphore}}

#############
## Private ##
#############

sub _name_to_key
{
	my $self     = shift;
	my $key_name = shift;
	if ((my $length = length($key_name)) > 50) {
		die "key_name length was [$length], maximum is 50";
	}
	$key_name .= 'IPC::Semaphore::Set';
	my @codes  = map {ord} split('', $key_name);
	my $key    = eval join('+', @codes);
	return $key;
}

sub _resource
{
	my $self   = shift;
	my $number = shift;
	return IPC::Semaphore::Set::Resource->new(number => $number, set => $self);
}

1;

__END__

=head1 NAME

IPC::Semaphore::Set

=head1 DESCRIPTION

An abstract interface to semaphores sets and their resources

A semaphore is a tool to help control access to common resources.
Generally when interfacing with semaphores in Perl, you refer to
things with numbers and use old IPC calls like semop, semget, e.t.c.
The point of this module is to let you think of semaphores in terms
of the key objects, and the resource objects those keys have.

This module also tries to "Do The Right Thing". It assumes a lot about
what you're looking for if you're using it, and basically assumes that
what you want is to have a semaphore with at least one resource that has
at least an availability of 1. If this assumption is wrong for your
purposes, pay close attention to the options for '->new'.

=head1 SYNOPSIS

To check for resource availability:

	my $semset = IPC::Semaphore::Set->new;
	if ($semset->resource->lock) {
		# ... can use resource!
	} else {
		# ... can't use resource!
	}

To wait for resource availability:

	my $semset = IPC::Semaphore::Set->new;
	$semset->resource->lockWait;
	# ... resource is now available

To die if we can't get the resource:

	my $semset = IPC::Semaphore::Set->new;
	$semset->resource->lockOrDie;
	# ... if we're here we have a lock

You can provide arguments to new to use a "word" as the key for the
semaphore, and to select how many resources the set has, and the
total availability for those resources:

	my $semset = IPC::Semaphore::Set->new(
		key_name     => "my_key",
		resources    => 5,
		availability => 2, # If you set this, it wont be overwritten
				   # until ->remove is called, or if you override
				   # it explicitly from the $self->sem object using
				   # ->setall
	);

Now you can get the first resource (resource 0):

	my $resource = $semset->resource;

Or you can select the resource explicitly:

	my $resource = $semset->resource(4);

But note that with 5 resources total, 4 is our last resource because
the scalar is 0..X

=head1 METHODS

=over

=item new

Get a new IPC::Semaphore::Set object. If 'key' is provided, get
or create a semaphore with that key. if 'key_name' is provided,
generate a key based off of the ascii character codes of that name.
If neither is provided, a new 'private' semaphore set will be created
(note that 'private' is how SysV refers to it, but this is something
of a misnomer).

New uses the following flags by default:

	S_IRUSR | S_IWUSR | IPC_CREAT | SEM_UNDO

Which means it creates it if it doesn't exist, keeps track of ownership,
and will clean up it's changes after exit.

=item resource

Returns a IPC::Semaphore::Set::Resource object given the resource number,
or the last Resource accessed by number.

If a number isn't passed to it, and it hasn't yet encountered a resource,
it assumes resource 0 (the first resource in the set) is what you wanted
and tries to get that.

=item resources

Returns a list or arrayref of all the IPC::Semaphore::Set::Resource objects
available for this semaphore set.

=item id

Returns the numeric system ID of the semaphore set.

=item key

Returns the numeric key if available.

=item keyName

Returns the 'word' key if used.

=item remove

Remove the semaphore set entirely from the system.

=item sem

Returns the internal 'IPC::Semaphore' object.

=back

=cut

