package HTML::Template::Compiled::Plugin::I18N::DefaultTranslator;

use strict;
use warnings;

use Carp qw(croak);
use HTML::Template::Compiled::Plugin::I18N;

our $VERSION = '1.00';

my $escape_ref = sub {
    my $string = shift;

    defined $string
        and return $string;

    return 'undef';
};

sub set_escape {
    my (undef, $code_ref) = @_;

    ref $code_ref eq 'CODE'
        or croak 'Coderef expected';
    $escape_ref = $code_ref;

    return;
}

sub get_escape {
    return $escape_ref;
}

sub translate {
    my (undef, $attr_ref) = @_;

    my $escape
        = exists $attr_ref->{escape}
        ? delete $attr_ref->{escape}
        : ();
    if ( defined $escape ) {
        ESCAPE:
        for ( qw(text plural) ) {
            exists $attr_ref->{$_}
                or next ESCAPE;
            $attr_ref->{$_} = HTML::Template::Compiled::Plugin::I18N->escape(
                $attr_ref->{$_},
                $escape,
            );
        }
        ESCAPE:
        for ( qw(maketext) ) {
            exists $attr_ref->{$_}
                or next ESCAPE;
            for my $value ( @{ $attr_ref->{$_} } ) {
                $value = HTML::Template::Compiled::Plugin::I18N->escape(
                    $value,
                    $escape,
                );
            }
        }
        ESCAPE:
        for ( qw(gettext) ) {
            exists $attr_ref->{$_}
                or next ESCAPE;
            for my $value ( values %{ $attr_ref->{$_} } ) {
                $value = HTML::Template::Compiled::Plugin::I18N->escape(
                    $value,
                    $escape,
                );
            }
        }
    }

    return join q{;}, map {
        exists $attr_ref->{$_}
        ? (
            "$_="
            . join q{,}, map {
                get_escape()->($_);
            } (
                ref $attr_ref->{$_} eq 'ARRAY'
                ? @{ $attr_ref->{$_} }
                : ref $attr_ref->{$_} eq 'HASH'
                ? do {
                    my $key = $_;
                    map {
                        ( $_, $attr_ref->{$key}->{$_} );
                    } sort keys %{ $attr_ref->{$key} };
                }
                : $attr_ref->{$_}
            )
        )
        : ();
    } qw(
        context text plural count maketext gettext formatter unescaped
    );
}

1;

__END__

=pod

=head1 NAME

HTML::Template::Compiled::Plugin::I18N::DefaultTranslator
- an extremly simple translater class for the HTC plugin I18N

$Id: DefaultTranslator.pm 150 2009-11-18 20:33:13Z steffenw $

$HeadURL: https://htc-plugin-i18n.svn.sourceforge.net/svnroot/htc-plugin-i18n/trunk/lib/HTML/Template/Compiled/Plugin/I18N/DefaultTranslator.pm $

=head1 VERSION

1.00

=head1 SYNOPSIS

=head1 DESCRIPTION

This module is very useful to run the application
before the translator module has finished.

The output string is human readable
except the escape callback sub.

=head1 SUBROUTINES/METHODS

=head2 class method set_escape

Set a escape code reference to escape all the values.
The example describes the default to have no undefined values.

    HTML::Template::Compiled::Plugin::I18N::DefaultTranslator->set_escape(
        sub {
            my $string = shift;

            defined $string
                and return $string;

            return 'undef';
        },
    );

=head2 class method get_escape

Get back the current escape code reference.

   $code_ref
       = HTML::Template::Compiled::Plugin::I18N::DefaultTranslator->get_escape();

=head2 class method translate

Possible hash keys are
context, text, plural, maketext, count, gettext, formatter and escape_code.
The last one is to escape text and plural after translation.

    $string
        = HTML::Template::Compiled::Plugin::I18N::DefaultTranslator->translate({
            text => 'text',
            ...
        });

=head1 DIAGNOSTICS

none

=head1 CONFIGURATION AND ENVIRONMENT

none

=head1 DEPENDENCIES

Carp

=head1 INCOMPATIBILITIES

The output is not readable by a parser
but very good during the application development.

=head1 BUGS AND LIMITATIONS

not known

=head1 AUTHOR

Steffen Winkler

=head1 LICENSE AND COPYRIGHT

Copyright (c) 2009,
Steffen Winkler
C<< <steffenw at cpan.org> >>.
All rights reserved.

This module is free software;
you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut
