use 5.008;

use strict;
use warnings;

use ExtUtils::MakeMaker;
use Config;

my $libdirs = $ENV{JS_LIBS} || '';
my $incdirs = $ENV{JS_INC} || '';
my $defines = $ENV{JS_DEFINES} || '';
my $CC = $Config{cc};
my $myextlib = '';
my $needtest = 1;

if($^O eq 'MSWin32') {
    # Search in PATH for xulrunner SDK
    my $xulsdk = $ENV{XUL_PATH};
    unless($xulsdk) {
	my @dirs = split ';', $ENV{PATH};
	for(@dirs) {
	    if(-x "$_/xulrunner.exe" && -e "$_/../include/jsapi.h") {
		$xulsdk = $_;
		last;
	    }
	}
	unless($xulsdk) {
	    warn "Can't found your XulRunner SDK directory in PATH\n";
	    exit 0;
	}
	$xulsdk =~ s/bin$//;
    }
    warn "XulRunner SDK found in '$xulsdk'\n";

    my($dll) = glob "$xulsdk\\bin\\js*.dll";
    my($libname) = $dll =~ /(js\d+)\.dll/;
    $libdirs = "-L${xulsdk}lib -l$libname -lnspr4";
    $incdirs = "-I${xulsdk}include";
    $defines = "-DXP_WIN=1 -D_CRT_SECURE_NO_WARNINGS -TP";
    open(my $fd, ">JS_Env.h") or die "Can't create JS_Env.h: $!\n";
    print $fd <<EOF;
/* Autogenerated, don't edit */
#define JS_THREADSAFE	1
EOF

    warn "Will use\n\tJS_LIBS=$libdirs\n\tJS_INC=$incdirs\n\tJS_DEFINES=$defines\n";
    $needtest=0;
}

if($ENV{JS_SRC} && $CC eq 'gcc') {
    # For developers only, as its uses the static spidermonkey library.
    $defines = '-Wno-unused-variable -DXP_UNIX';
    $libdirs = "-lm -lnspr4";
    ($myextlib) = glob "$ENV{JS_SRC}/libjs*.a";
    $incdirs = "-I$ENV{JS_SRC}/.. -I$ENV{JS_SRC}";
    my($vhead) = grep -f $_, glob "$ENV{JS_SRC}/../js{version,config}.h";

    die "Your SpiderMonkey doesn't seems to be compiled yet.\nI was looking for $myextlib\n"
	unless(-f $myextlib);

    my $ver = (split(' ',`grep "#define JS_VERSION " $vhead`))[2];
    if($ver >= 185) {
	$CC = 'c++';
    }

    warn "Found SpiderMonkey $ver (static build)\n";
    warn "Will use\n\tJS_LIBS=$libdirs\n\tJS_INC=$incdirs\n\tJS_DEFINES=$defines\n\tEXTLIB=$myextlib\n";
}

my $pkgn = '';
unless($libdirs && $incdirs) {
PKGCONF: {
    # In the common case we depend on a pkg-config file.
    #warn "Will try with pkg-config files.\n";
    my @build_reqs = (
	'mozilla-js',
	'libjs'
    );
    unshift @build_reqs, $ARGV[0] if $ARGV[0];

    for(@build_reqs) {
	#warn "Testing for $_...\n";
	my $res = system 'pkg-config', '--exists',  "$_ >= 1.7";
	if($res == -1) {
	    warn "Can't run pkg-config\n";
	    last PKGCONF;
	}
	if(!$res) {
	    $pkgn = $_;
	    last;
	}
    }

    unless($pkgn) {
	warn "Sad, I can't find any pkg-config file for SpiderMonkey\n";
	last;
    }
    my $vers = `pkg-config --modversion $pkgn`;
    chomp $vers;
    warn "Found $pkgn ($vers)\n";
    $libdirs = `pkg-config --libs $pkgn`;
    chomp $libdirs;
    $defines = `pkg-config --cflags-only-other $pkgn`;
    chomp $defines;
    $incdirs = `pkg-config --cflags-only-I $pkgn`;
    chomp $incdirs;

    $defines .= ' -DXP_UNIX' unless $defines =~ /-DXP/; # Just in case.

    warn qq{Will use\n\tJS_LIBS="$libdirs"\n\tJS_INC="$incdirs"\n\tJS_DEFINES="$defines"\n};
}}

unless($incdirs && $libdirs) {
    warn <<EOF;
SpiderMonkey headers and libraries not found.

Try manually defining JS_LIBS, JS_INC and JS_DEFINES environment variables like:
export JS_LIBS="-L/path/to/spidermonkey/library -l<libname>"
export JS_INC="-I/path/to/spidermonkey/include"
export JS_DEFINE="-DXP_UNIX"
EOF
    exit 0;
}

# Try a small compile to determine if we can find libs and headers and
# to generate the dynamic header file
if($needtest) {
    require File::Temp;
    my $exe = File::Temp::tmpnam();
    my $cc = join(" ", $CC, $defines, $incdirs, $libdirs, "-ldl -o", $exe);
    #warn "$cc testjs.c\n";
    qx($cc utils/testjs.c $myextlib);
    if($?) {
	warn "Test compile failed, check JS_INC and JS_DEFINE for errors.\n";
	warn "The package-config file '$pkgn.pc' can be broken!\n" if $pkgn;
	exit 0;
    }

    my $res = qx($exe);
    if(!$res) {
	warn "Unusable library, please make sure it's available in your LD_LIBRARY_PATH.\n";
	exit 0;
    }
    open(my $fd, ">JS_Env.h") or die "Can't create JS_Env.h: $!\n";
    print $fd "/* Autogenerated, don't edit */\n$res\n";
    unlink($exe);
}

# Write makefile
WriteMakefile(
    NAME            => "JSP",
    VERSION_FROM    => "lib/JSP.pm",
    PREREQ_PM       => {
        "Test::More"      => 0,
        "Test::Exception" => 0,
    },
    ABSTRACT_FROM   => "lib/JSP.pm", # retrieve abstract from module
    AUTHOR          => "Salvador Ortiz <sortiz\@cpan.org>",
    DEFINE          => "$defines",
    LIBS            => "$libdirs",
    INC             => $incdirs,
    LICENSE         => "perl",
    EXE_FILES	    => [ 'bin/jsp' ],
    OBJECT          => q/$(O_FILES)/,
    CC              => $CC,
    MYEXTLIB        => $myextlib,
    ($CC eq 'c++' ? (XSOPT => '-C++', LD => $CC) : ()),
);

package MY;

use File::Spec;

sub post_initialize {
    my($self) = shift;

    my @headers = <*.h>;
    for (@headers, 'typemap' ) {
        $self->{PM}->{$_} = File::Spec->catfile($self->{INST_ARCHAUTODIR}, $_);
    }
    return '';
}
