package Bio::Phylo::Dictionary;
use strict;
use Bio::Phylo::Listable;
use Bio::Phylo::Util::CONSTANT qw(_DICTIONARY_ _NONE_);
use Bio::Phylo::Util::Exceptions 'throw';
use vars '@ISA';
@ISA=qw(Bio::Phylo::Listable);
{    

=head1 NAME

Bio::Phylo::Dictionary - An object to annotate other objects with.

=head1 SYNOPSIS

 use Bio::Phylo::Factory;
 my $fac = Bio::Phylo::Factory->new;
 # there are several data types: string, integer, float, uri, any, etc.
 # refer to nexml.org for a complete list
 my $dic = $fac->create_dictionary( -tag => 'string' );
 my $ann = $fac->create_annotation( -value => 'This is a description' );
 $dic->insert( $ann );
 print $dic->to_xml;

 # prints: <dict><string id="string2">This is a description</string></dict>

=head1 DESCRIPTION

To comply with the NeXML standard (L<http://www.nexml.org>), Bio::Phylo implements
dictionary annotations, which consist conceptually of key/value pairs where the
key is an xml id (which is either autogenerated or can be set using the set_xml_id
method) and the value is the containing element (whose name specifies the data
type of its text contents, i.e. string, integer, float and so on).

This class implements a container for a set of key/value pairs. The container
can also hold other containers, thereby creating a recursive data structure. For
the syntax of each individual key/value pair, refer to L<Bio::Phylo::Annotation>. 

=head1 METHODS

=head2 CONSTRUCTOR

=over

=item new()

Factory constructor.

 Type    : Constructor
 Title   : new
 Usage   : my $dict = Bio::Phylo::Dictionary->new;
 Function: Initializes a Bio::Phylo::Dictionary object.
 Returns : A Bio::Phylo::Dictionary object.
 Args    : optional constructor arguments are key/value
 		   pairs where the key corresponds with any of
 		   the methods that starts with set_ (i.e. mutators) 
 		   and the value is the permitted argument for such 
 		   a method. The method name is changed such that,
 		   in order to access the set_value($val) method
 		   in the constructor, you would pass -value => $val

=cut

    sub new { 
        return shift->SUPER::new( '-tag' => 'dict', @_ ); 
    }

=back

=head2 MUTATORS

=over

=item add_dictionary()

This method is here as an override of the method by the same name in
L<Bio::Phylo::Util::XMLWritable>. This is here to ensure that dictionaries
are L<not> attached to other dictionaries. They can be inserted, recursively,
into other dictionaries (but that's not the same thing! User the insert() method
for that). If you call add_dictionary to an dictionary an exception of type
Bio::Phylo::Util::Exceptions::BadArgs will be thrown (see exceptions: 
L<Bio::Phylo::Util::Exceptions>).

=cut

    sub add_dictionary {
        throw 'BadArgs' => "Can't attach dictionaries to other dictionaries"
    }

=back

=head1 SEE ALSO

=over

=item L<Bio::Phylo::Annotation>

An annotation object is a key/value pair inside a dictionary.

=item L<Bio::Phylo::Listable>

This object inherits from L<Bio::Phylo::Listable>, so methods
defined there are also applicable here.

=item L<Bio::Phylo::Manual>

Also see the manual: L<Bio::Phylo::Manual> and L<http://rutgervos.blogspot.com>.

=back

=head1 REVISION

 $Id: Annotation.pm 604 2008-09-05 17:32:28Z rvos $

=cut

    my $TYPE_CONSTANT      = _DICTIONARY_;
    my $CONTAINER_CONSTANT = _DICTIONARY_;
    sub _type      { $TYPE_CONSTANT }
    sub _container { $CONTAINER_CONSTANT }     
}
1;