package DBICx::LookupManager;

use strict;
use warnings;

=head1 NAME

DBICx::LookupManager - A lazzy dbic component caching queries.

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

use Moose;
use Data::Dumper;
use Smart::Comments -ENV;

my %CACHE; # main class variable containing all cached objects






=head1 SYNOPSIS

	use DBICx::LookupManager;
	
	DBICx::LookupManager->FETCH_ID_BY_NAME(  $schema, 'Permission', 'name', 'Admisitrator' );




=head1 DESCRIPTION

This module does DBIx::Class queries by means of arguments you passed by and stores the result in a nested hashing data structure (cache).
It does a DBIx::Class query if and only if that one is not yet stored in the cache (lazziness).

This module only supports tables having one signle primary key.

The module closely works with L<DBICx::LookupRelation> package which provides convenient methods for accessing data by a client table pointing on a lookup table. 
What's meant as lookup table is a table containing some terms definition, such as Permission ( 1, 'Administrator'; 2, 'User'; 3, 'Reader' ) which 
is for instance associated with a client table such as User ( id, first_name, last_name, permission_id ).
Though the module could also be used in an independently way. 



=head1 EXPORT

	FETCH_ID_BY_NAME
	FETCH_NAME_BY_ID
	RESET_CACHE
	RESET_CACHE_LOOKUP_TABLE

=head1 SUBROUTINES/METHODS




=head1 METHODS

=head2 FETCH_ID_BY_NAME

=over4

=item Arguments : $schema, $lookup_table, $field_name, $name

=item Return Value: id in the pointed table (lookup table)

item Example : 

	FETCH_ID_BY_NAME( $schema, 'Permission', name, 'Administrator' ) returns the permission id stored in the pointed table.

=back




=head2 FETCH_NAME_BY_ID

=over4

=item Arguments : $schema, $lookup_table, $field_name, $id

=item Return Value: the value of the $field_name in the pointed table (lookup table)

item Example : 

	FETCH_NAME_BY_ID( $schema, 'Permission', name, 1 ) returns the permission stored in the pointed table with id 1, e.g. 'Administrator'.

=back




=head2 RESET_CACHE

=over4

=item Arguments : no argument

=item Return Value: no returned value

item Description :
 	
 	Reset the whole nested hashing data structure.
 	
=backs




=head2 RESET_CACHE_LOOKUP_TABLE

=over4

=item Arguments : name of the table whose data are stored in the cache

=item Return Value: no returned value

item Description :
 	
 	Reset the hashing whose key is the table's name argument.
 	
=backs

 


=head2 ENSURE_LOOKUP_IS_CACHED

=over4

item Description :
 	
 	Carrying about doing a query if and only if that one not yet sotred in the cache. For internally use.
 	
=backs
 


=cut

sub FETCH_ID_BY_NAME {
    my ( $class, $schema, $lookup_table, $field_name, $name ) = @_;
	confess "Bad args" unless defined $name;
    my $cache	= $class->ENSURE_LOOKUP_IS_CACHED(  $schema, $lookup_table, $field_name );
    my $id		= $cache->{name2id}{$name} or confess "name [$name] does not exist in (cached) Lookup table [$lookup_table]";
    return $id;
}




sub FETCH_NAME_BY_ID {
    my ( $class, $schema, $lookup_table, $field_name, $id ) = @_;
	confess "Bad args" unless defined $id;
    my $cache	= $class->ENSURE_LOOKUP_IS_CACHED( $schema, $lookup_table, $field_name );
    my $name	= $cache->{id2name}{$id} or confess "Bad type_name [$id] in Lookup table [$lookup_table]";
    return $name;
}




sub ENSURE_LOOKUP_IS_CACHED {
    my ( $class, $schema, $lookup_table, $field_name ) = @_;
	
	# check the table and field names
	my $source_table = $schema->source( $lookup_table ) or confess "unknown table called $lookup_table";
    confess "the $field_name as field name does not exist in the $lookup_table lookup table" 
    	unless $source_table->has_column( $field_name );
		
    #### ENSURE_LOOKUP_IS_CACHED: $lookup_table, $field_name
 
    unless ( $CACHE{$lookup_table} ) {
		$CACHE{$lookup_table} = {};
		
		# get primary key name         
        my @primary_columns = $schema->source( $lookup_table )->primary_columns;
        confess "Error, no primary defined in lookup table $lookup_table" unless @primary_columns;
        confess "we only support lookup table with ONE primary key for table $lookup_table" if @primary_columns > 1; 
        my $primary_key = shift @primary_columns;
        
       	# query for feching all (id, name) rows from lookup table
        my $rs = $schema->resultset($lookup_table)->search( undef, { select=>[$primary_key, $field_name] });
		
		my ($id, $name);
		my $id2name = $CACHE{$lookup_table}{id2name} ||= {};
		my $name2id = $CACHE{$lookup_table}{name2id} ||= {};
		my $cursor =  $rs->cursor;
		# fetch all and fill the cache
		while ( ($id, $name) = $cursor->next ){
			$id2name->{$id} = $name;
			$name2id->{$name} = $id;
		}
    }
    return $CACHE{$lookup_table};
}




sub RESET_CACHE {
    %CACHE = ();
}




sub RESET_CACHE_LOOKUP_TABLE {
	my ( $lookup_table ) = @_;
    $CACHE{$lookup_table} = ();
}




=head1 AUTHOR

"Karl Forner", C<< <"karl.forner@gmail.com"> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-dbicx-vocabularies at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=DBICx-Vocabularies>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc DBICx::LookupManager


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=DBICx-Vocabularies>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/DBICx-Vocabularies>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/DBICx-Vocabularies>

=item * Search CPAN

L<http://search.cpan.org/dist/DBICx-Vocabularies/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2012 "".

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.


=cut

1; # End of DBICx::LookupManager
