package Locale::Country::SubCountry::Base;

use strict;
use warnings;

use Data::Section::Simple 'get_data_section';

use Hash::FieldHash ':all';

use Locale::Country::SubCountry::Util::Config;

use Text::CSV_XS;

fieldhash my %all_countries     => 'all_countries';
fieldhash my %all_sub_countries => 'all_sub_countries';
fieldhash my %config            => 'config';
fieldhash my %data              => 'data';
fieldhash my %verbose           => 'verbose';

our $VERSION = '1.03';

# -----------------------------------------------

sub _build_countries
{
	my($self, $data) = @_;
	my($csv)   = Text::CSV_XS -> new;
	my(@data)  = split(/\n/, $data);
	my($count) = 0;

	# Discard heading.

	shift @data;

	my(@country);
	my(@field);

	for my $line (@data)
	{
		if ($csv -> parse($line) )
		{
			@field = $csv -> fields;

			push @country,
			{
				address_format => $field[3],
				code2          => $field[1],
				code3          => $field[2],
				id             => ++$count,
				name           => $field[0],
			}
		}
		else
		{
			die "Unable to parse <$line>";
		}
	}

	$self -> all_countries(\@country);

} # End of _build_countries.

# -----------------------------------------------

sub _build_sub_countries
{
	my($self, $data) = @_;
	my($csv)  = Text::CSV_XS -> new({binary => 1});
	my(@data) = split(/\n/, $data);
	my($count) = 0;

	# Discard heading.

	shift @data;

	my(@field);
	my(@sub_country);

	for my $line (@data)
	{
		if ($csv -> parse($line) )
		{
			$count++;

			@field = $csv -> fields;

			push @sub_country,
			{
				country_id => $field[0],
				code       => $field[1],
				id         => $count,
				name       => $field[2],
			}
		}
		else
		{
			die "Unable to parse <$line>";
		}
	}

	$self -> all_sub_countries(\@sub_country);

} # End of build_sub_countries.

# -----------------------------------------------

sub _init
{
	my($self, $arg) = @_;
	$$arg{config}   = Locale::Country::SubCountry::Util::Config -> new -> config;
	$$arg{verbose}  ||= 0;
	$$arg{data}     = get_data_section;
	$self           = from_hash($self, $arg);

	$self -> _build_countries($$arg{data}{countries});
	$self -> _build_sub_countries($$arg{data}{sub_countries});

	return $self;

} # End of _init.

# -----------------------------------------------

sub new
{
	my($class, %arg) = @_;
    my($self)        = bless {}, $class;

	return $self -> _init(\%arg);

} # End of new.

# -----------------------------------------------

1;

=pod

=head1 NAME

L<Locale::Country::SubCountry::Base> - Country names in English, and subcountry names in native scripts

=head1 Synopsis

See L<Locale::Country::SubCountry>.

=head1 Description

L<Locale::Country::SubCountry> provides subcountry names in their native scripts.

This module serves as a parent for various other modules.

Use L<Locale::Country::SubCountry::Database::Base> for apps.

=head1 Methods

=head2 all_countries()

Returns an arrayref, 1 element per country, where each element is a hashref with these keys:

=over 4

=item o id

Unique identifier for the country. Countries are numbered from 1 up.

=item o address_format

A string showing how to format an address in this country. Only set for some countries.

Split the string on '#' characters, to produce a format of N lines.

Replace the tokens ':X' with the corresponding data (if available).

=item o code2

The ISO3166-1 2-letter code for the country.

=item o code3

The ISO3166-1 3-letter code for the country.

=item o name

The name of the country, in English.

=back

Note: A copy of the file data/countries.csv is stored below (under __DATA__), so that that file is not needed at run-time.

=head2 all_subcountries()

Returns an arrayref, 1 element per subcountry, where each element is a hashref with these keys:

=over 4

=item o id

Unique identifier for the subcountry. Subcountries are numbered from 1 up.

=item o country_id

The id of the country (starts from 1).

=item o code

The ISO3166-2 1 .. 5-letter code for the country.

=item o name

The name of the subcountry, in the country's native script.

=back

Note: A copy of the file data/sub_countries.csv is stored below (under __DATA__), so that that file is not needed at run-time.

=head2 _init()

For use by subclasses.

Sets default values for object attributes.

=head2 new()

For use by subclasses.

=head1 Support

Email the author, or log a bug on RT:

L<https://rt.cpan.org/Public/Dist/Display.html?Name=Locale::Country::SubCountry>.

=head1 Author

L<Locale::Country::SubCountry> was written by Ron Savage I<E<lt>ron@savage.net.auE<gt>> in 2011.

Home page: L<http://savage.net.au/index.html>.

=head1 Copyright

Australian copyright (c) 2011, Ron Savage.

	All Programs of mine are 'OSI Certified Open Source Software';
	you can redistribute them and/or modify them under the terms of
	The Artistic License, a copy of which is available at:
	http://www.opensource.org/licenses/index.html

=cut

__DATA__

@@ countries
"name","code2","code3","address_format"
"Afghanistan","AF","AFG",""
"Albania","AL","ALB",""
"Algeria","DZ","DZA",""
"American Samoa","AS","ASM",""
"Andorra","AD","AND",""
"Angola","AO","AGO",""
"Anguilla","AI","AIA",""
"Antarctica","AQ","ATA",""
"Antigua and Barbuda","AG","ATG",""
"Argentina","AR","ARG",":name#:street_address#:postcode :city#:country"
"Armenia","AM","ARM",""
"Aruba","AW","ABW",""
"Australia","AU","AUS",":name#:street_address#:suburb :state_code :postcode#:country"
"Austria","AT","AUT",":name#:street_address#A-:postcode :city#:country"
"Azerbaijan","AZ","AZE",""
"Bahamas","BS","BHS",""
"Bahrain","BH","BHR",""
"Bangladesh","BD","BGD",""
"Barbados","BB","BRB",""
"Belarus","BY","BLR",""
"Belgium","BE","BEL",":name#:street_address#B-:postcode :city#:country"
"Belize","BZ","BLZ",""
"Benin","BJ","BEN",""
"Bermuda","BM","BMU",""
"Bhutan","BT","BTN",""
"Bolivia","BO","BOL",""
"Bosnia and Herzegowina","BA","BIH",""
"Botswana","BW","BWA",""
"Bouvet Island","BV","BVT",""
"Brazil","BR","BRA",":name#:street_address#:state#:postcode#:country"
"British Indian Ocean Territory","IO","IOT",""
"Brunei Darussalam","BN","BRN",""
"Bulgaria","BG","BGR",""
"Burkina Faso","BF","BFA",""
"Burundi","BI","BDI",""
"Cambodia","KH","KHM",""
"Cameroon","CM","CMR",""
"Canada","CA","CAN",":name#:street_address#:city :state_code :postcode#:country"
"Cape Verde","CV","CPV",""
"Cayman Islands","KY","CYM",""
"Central African Republic","CF","CAF",""
"Chad","TD","TCD",""
"Chile","CL","CHL",":name#:street_address#:city#:country"
"China","CN","CHN",":name#:street_address#:postcode :city#:country"
"Christmas Island","CX","CXR",""
"Cocos (Keeling) Islands","CC","CCK",""
"Colombia","CO","COL",""
"Comoros","KM","COM",""
"Congo","CG","COG",""
"Cook Islands","CK","COK",""
"Costa Rica","CR","CRI",""
"Cote D'Ivoire","CI","CIV",""
"Croatia","HR","HRV",""
"Cuba","CU","CUB",""
"Cyprus","CY","CYP",""
"Czech Republic","CZ","CZE",""
"Denmark","DK","DNK",":name#:street_address#DK-:postcode :city#:country"
"Djibouti","DJ","DJI",""
"Dominica","DM","DMA",""
"Dominican Republic","DO","DOM",""
"East Timor","TP","TMP",""
"Ecuador","EC","ECU",""
"Egypt","EG","EGY",""
"El Salvador","SV","SLV",""
"Equatorial Guinea","GQ","GNQ",""
"Eritrea","ER","ERI",""
"Estonia","EE","EST",""
"Ethiopia","ET","ETH",""
"Falkland Islands (Malvinas)","FK","FLK",""
"Faroe Islands","FO","FRO",""
"Fiji","FJ","FJI",""
"Finland","FI","FIN",":name#:street_address#FIN-:postcode :city#:country"
"France","FR","FRA",":name#:street_address#:postcode :city#:country"
"France, Metropolitan","FX","FXX",":name#:street_address#:postcode :city#:country"
"French Guiana","GF","GUF",":name#:street_address#:postcode :city#:country"
"French Polynesia","PF","PYF",":name#:street_address#:postcode :city#:country"
"French Southern Territories","TF","ATF",":name#:street_address#:postcode :city#:country"
"Gabon","GA","GAB",""
"Gambia","GM","GMB",""
"Georgia","GE","GEO",""
"Germany","DE","DEU",":name#:street_address#D-:postcode :city#:country"
"Ghana","GH","GHA",""
"Gibraltar","GI","GIB",""
"Greece","GR","GRC",""
"Greenland","GL","GRL",":name#:street_address#DK-:postcode :city#:country"
"Grenada","GD","GRD",""
"Guadeloupe","GP","GLP",""
"Guam","GU","GUM",""
"Guatemala","GT","GTM",""
"Guinea","GN","GIN",""
"Guinea-Bissau","GW","GNB",""
"Guyana","GY","GUY",""
"Haiti","HT","HTI",""
"Heard and McDonald Islands","HM","HMD",""
"Honduras","HN","HND",""
"Hong Kong","HK","HKG",":name#:street_address#:city#:country"
"Hungary","HU","HUN",""
"Iceland","IS","ISL",":name#:street_address#IS:postcode :city#:country"
"India","IN","IND",":name#:street_address#:city-:postcode#:country"
"Indonesia","ID","IDN",":name#:street_address#:city :postcode#:country"
"Iran","IR","IRN",""
"Iraq","IQ","IRQ",""
"Ireland","IE","IRL",":name#:street_address#IE-:city#:country"
"Israel","IL","ISR",":name#:street_address#:postcode :city#:country"
"Italy","IT","ITA",":name#:street_address#:postcode-:city :state_code#:country"
"Jamaica","JM","JAM",""
"Japan","JP","JPN",":name#:street_address, :suburb#:city :postcode#:country"
"Jordan","JO","JOR",""
"Kazakhstan","KZ","KAZ",""
"Kenya","KE","KEN",""
"Kiribati","KI","KIR",""
"Korea, North","KP","PRK",""
"Korea, South","KR","KOR",""
"Kuwait","KW","KWT",""
"Kyrgyzstan","KG","KGZ",""
"Laos","LA","LAO",""
"Latvia","LV","LVA",""
"Lebanon","LB","LBN",""
"Lesotho","LS","LSO",""
"Liberia","LR","LBR",""
"Libyan Arab Jamahiriya","LY","LBY",""
"Liechtenstein","LI","LIE",""
"Lithuania","LT","LTU",""
"Luxembourg","LU","LUX",":name#:street_address#L-:postcode :city#:country"
"Macau","MO","MAC",""
"Macedonia","MK","MKD",""
"Madagascar","MG","MDG",""
"Malawi","MW","MWI",""
"Malaysia","MY","MYS",""
"Maldives","MV","MDV",""
"Mali","ML","MLI",""
"Malta","MT","MLT",""
"Marshall Islands","MH","MHL",""
"Martinique","MQ","MTQ",""
"Mauritania","MR","MRT",""
"Mauritius","MU","MUS",""
"Mayotte","YT","MYT",""
"Mexico","MX","MEX",":name#:street_address#:postcode :city, :state_code#:country"
"Micronesia","FM","FSM",""
"Moldova","MD","MDA",""
"Monaco","MC","MCO",""
"Mongolia","MN","MNG",""
"Montserrat","MS","MSR",""
"Morocco","MA","MAR",""
"Mozambique","MZ","MOZ",""
"Myanmar","MM","MMR",""
"Namibia","NA","NAM",""
"Nauru","NR","NRU",""
"Nepal","NP","NPL",""
"Netherlands","NL","NLD",":name#:street_address#:postcode :city#:country"
"Netherlands Antilles","AN","ANT",":name#:street_address#:postcode :city#:country"
"New Caledonia","NC","NCL",""
"New Zealand","NZ","NZL",":name#:street_address#:suburb#:city :postcode#:country"
"Nicaragua","NI","NIC",""
"Niger","NE","NER",""
"Nigeria","NG","NGA",""
"Niue","NU","NIU",""
"Norfolk Island","NF","NFK",""
"Northern Mariana Islands","MP","MNP",""
"Norway","NO","NOR",":name#:street_address#NO-:postcode :city#:country"
"Oman","OM","OMN",""
"Pakistan","PK","PAK",""
"Palau","PW","PLW",""
"Panama","PA","PAN",""
"Papua New Guinea","PG","PNG",""
"Paraguay","PY","PRY",""
"Peru","PE","PER",""
"Philippines","PH","PHL",""
"Pitcairn","PN","PCN",""
"Poland","PL","POL",":name#:street_address#:postcode :city#:country"
"Portugal","PT","PRT",":name#:street_address#:postcode :city#:country"
"Puerto Rico","PR","PRI",""
"Qatar","QA","QAT",""
"Reunion","RE","REU",""
"Romania","RO","ROM",""
"Russia","RU","RUS",":name#:street_address#:postcode :city#:country"
"Rwanda","RW","RWA",""
"Saint Kitts and Nevis","KN","KNA",""
"Saint Lucia","LC","LCA",""
"Saint Vincent and the Grenadines","VC","VCT",""
"Samoa","WS","WSM",""
"San Marino","SM","SMR",""
"Sao Tome and Principe","ST","STP",""
"Saudi Arabia","SA","SAU",""
"Senegal","SN","SEN",""
"Seychelles","SC","SYC",""
"Sierra Leone","SL","SLE",""
"Singapore","SG","SGP",":name#:street_address#:city :postcode#:country"
"Slovakia","SK","SVK",""
"Slovenia","SI","SVN",""
"Solomon Islands","SB","SLB",""
"Somalia","SO","SOM",""
"South Africa","ZA","ZAF",":name#:street_address#:suburb#:city#:postcode :country"
"South Georgia and the South Sandwich Islands","GS","SGS",""
"South Sudan","SS","SSN",""
"Spain","ES","ESP",":name#:street_address#:postcode :city#:country"
"Sri Lanka","LK","LKA",""
"St. Helena","SH","SHN",""
"St. Pierre and Miquelon","PM","SPM",""
"Sudan","SD","SDN",""
"Suriname","SR","SUR",""
"Svalbard and Jan Mayen Islands","SJ","SJM",""
"Swaziland","SZ","SWZ",""
"Sweden","SE","SWE",":name#:street_address#:postcode :city#:country"
"Switzerland","CH","CHE",":name#:street_address#:postcode :city#:country"
"Syrian Arab Republic","SY","SYR",""
"Taiwan","TW","TWN",":name#:street_address#:city :postcode#:country"
"Tajikistan","TJ","TJK",""
"Tanzania","TZ","TZA",""
"Thailand","TH","THA",""
"Togo","TG","TGO",""
"Tokelau","TK","TKL",""
"Tonga","TO","TON",""
"Trinidad and Tobago","TT","TTO",""
"Tunisia","TN","TUN",""
"Turkey","TR","TUR",""
"Turkmenistan","TM","TKM",""
"Turks and Caicos Islands","TC","TCA",""
"Tuvalu","TV","TUV",""
"Uganda","UG","UGA",""
"Ukraine","UA","UKR",""
"United Arab Emirates","AE","ARE",""
"United Kingdom","GB","GBR",":name#:street_address#:city#:postcode#:country"
"United States of America","US","USA",":name#:street_address#:city :state_code :postcode#:country"
"United States Minor Outlying Islands","UM","UMI",""
"Uruguay","UY","URY",""
"Uzbekistan","UZ","UZB",""
"Vanuatu","VU","VUT",""
"Vatican City State (Holy See)","VA","VAT",""
"Venezuela","VE","VEN",""
"Vietnam","VN","VNM",""
"Virgin Islands (British)","VG","VGB",""
"Virgin Islands (U.S.)","VI","VIR",""
"Wallis and Futuna Islands","WF","WLF",""
"Western Sahara","EH","ESH",""
"Yemen","YE","YEM",""
"Yugoslavia","YU","YUG",""
"Zaire","ZR","ZAR",""
"Zambia","ZM","ZMB",""
"Zimbabwe","ZW","ZWE",""

@@ sub_countries
"country_id","code","name"
"1","BDS","بد خشان"
"1","BDG","بادغیس"
"1","BGL","بغلان"
"1","BAL","بلخ"
"1","BAM","بامیان"
"1","DAY","دایکندی"
"1","FRA","فراه"
"1","FYB","فارياب"
"1","GHA","غزنى"
"1","GHO","غور"
"1","HEL","هلمند"
"1","HER","هرات"
"1","JOW","جوزجان"
"1","KAB","کابل"
"1","KAN","قندھار"
"1","KAP","کاپيسا"
"1","KHO","خوست"
"1","KNR","کُنَر"
"1","KDZ","كندوز"
"1","LAG","لغمان"
"1","LOW","لوګر"
"1","NAN","ننگرهار"
"1","NIM","نیمروز"
"1","NUR","نورستان"
"1","ORU","ؤروزگان"
"1","PIA","پکتیا"
"1","PKA","پکتيکا"
"1","PAN","پنج شیر"
"1","PAR","پروان"
"1","SAM","سمنگان"
"1","SAR","سر پل"
"1","TAK","تخار"
"1","WAR","وردک"
"1","ZAB","زابل"
"2","BR","Beratit"
"2","BU","Bulqizës"
"2","DI","Dibrës"
"2","DL","Delvinës"
"2","DR","Durrësit"
"2","DV","Devollit"
"2","EL","Elbasanit"
"2","ER","Kolonjës"
"2","FR","Fierit"
"2","GJ","Gjirokastrës"
"2","GR","Gramshit"
"2","HA","Hasit"
"2","KA","Kavajës"
"2","KB","Kurbinit"
"2","KC","Kuçovës"
"2","KO","Korçës"
"2","KR","Krujës"
"2","KU","Kukësit"
"2","LB","Librazhdit"
"2","LE","Lezhës"
"2","LU","Lushnjës"
"2","MK","Mallakastrës"
"2","MM","Malësisë së Madhe"
"2","MR","Mirditës"
"2","MT","Matit"
"2","PG","Pogradecit"
"2","PQ","Peqinit"
"2","PR","Përmetit"
"2","PU","Pukës"
"2","SH","Shkodrës"
"2","SK","Skraparit"
"2","SR","Sarandës"
"2","TE","Tepelenës"
"2","TP","Tropojës"
"2","TR","Tiranës"
"2","VL","Vlorës"
"3","01","ولاية أدرار"
"3","02","ولاية الشلف"
"3","03","ولاية الأغواط"
"3","04","ولاية أم البواقي"
"3","05","ولاية باتنة"
"3","06","ولاية بجاية"
"3","07","ولاية بسكرة"
"3","08","ولاية بشار"
"3","09","البليدة‎"
"3","10","ولاية البويرة"
"3","11","ولاية تمنراست"
"3","12","ولاية تبسة"
"3","13","تلمسان"
"3","14","ولاية تيارت"
"3","15","تيزي وزو"
"3","16","ولاية الجزائر"
"3","17","ولاية عين الدفلى"
"3","18","ولاية جيجل"
"3","19","ولاية سطيف"
"3","20","ولاية سعيدة"
"3","21","السكيكدة"
"3","22","ولاية سيدي بلعباس"
"3","23","ولاية عنابة"
"3","24","ولاية قالمة"
"3","25","قسنطينة"
"3","26","ولاية المدية"
"3","27","ولاية مستغانم"
"3","28","ولاية المسيلة"
"3","29","ولاية معسكر"
"3","30","ورقلة"
"3","31","وهران"
"3","32","ولاية البيض"
"3","33","ولاية اليزي"
"3","34","ولاية برج بوعريريج"
"3","35","ولاية بومرداس"
"3","36","ولاية الطارف"
"3","37","تندوف"
"3","38","ولاية تسمسيلت"
"3","39","ولاية الوادي"
"3","40","ولاية خنشلة"
"3","41","ولاية سوق أهراس"
"3","42","ولاية تيبازة"
"3","43","ولاية ميلة"
"3","44","ولاية عين الدفلى"
"3","45","ولاية النعامة"
"3","46","ولاية عين تموشنت"
"3","47","ولاية غرداية"
"3","48","ولاية غليزان"
"4","EA","Eastern"
"4","MA","Manu'a"
"4","RI","Rose Island"
"4","SI","Swains Island"
"4","WE","Western"
"5","AN","Andorra la Vella"
"5","CA","Canillo"
"5","EN","Encamp"
"5","LE","Escaldes-Engordany"
"5","LM","La Massana"
"5","OR","Ordino"
"5","SJ","Sant Juliá de Lória"
"6","BGO","Bengo"
"6","BGU","Benguela"
"6","BIE","Bié"
"6","CAB","Cabinda"
"6","CCU","Cuando Cubango"
"6","CNO","Cuanza Norte"
"6","CUS","Cuanza Sul"
"6","CNN","Cunene"
"6","HUA","Huambo"
"6","HUI","Huíla"
"6","LUA","Luanda"
"6","LNO","Lunda Norte"
"6","LSU","Lunda Sul"
"6","MAL","Malanje"
"6","MOX","Moxico"
"6","NAM","Namibe"
"6","UIG","Uíge"
"6","ZAI","Zaire"
"9","BAR","Barbuda"
"9","SGE","Saint George"
"9","SJO","Saint John"
"9","SMA","Saint Mary"
"9","SPA","Saint Paul"
"9","SPE","Saint Peter"
"9","SPH","Saint Philip"
"10","A","Salta"
"10","B","Buenos Aires Province"
"10","C","Capital Federal"
"10","D","San Luis"
"10","E","Entre Ríos"
"10","F","La Rioja"
"10","G","Santiago del Estero"
"10","H","Chaco"
"10","J","San Juan"
"10","K","Catamarca"
"10","L","La Pampa"
"10","M","Mendoza"
"10","N","Misiones"
"10","P","Formosa"
"10","Q","Neuquén"
"10","R","Río Negro"
"10","S","Santa Fe"
"10","T","Tucumán"
"10","U","Chubut"
"10","V","Tierra del Fuego"
"10","W","Corrientes"
"10","X","Córdoba"
"10","Y","Jujuy"
"10","Z","Santa Cruz"
"11","AG","Արագածոտն"
"11","AR","Արարատ"
"11","AV","Արմավիր"
"11","ER","Երևան"
"11","GR","Գեղարքունիք"
"11","KT","Կոտայք"
"11","LO","Լոռի"
"11","SH","Շիրակ"
"11","SU","Սյունիք"
"11","TV","Տավուշ"
"11","VD","Վայոց Ձոր"
"13","ACT","Australian Capital Territory"
"13","NSW","New South Wales"
"13","NT","Northern Territory"
"13","QLD","Queensland"
"13","SA","South Australia"
"13","TAS","Tasmania"
"13","VIC","Victoria"
"13","WA","Western Australia"
"14","1","Burgenland"
"14","2","Kärnten"
"14","3","Niederösterreich"
"14","4","Oberösterreich"
"14","5","Salzburg"
"14","6","Steiermark"
"14","7","Tirol"
"14","8","Voralberg"
"14","9","Wien"
"15","AB","Əli Bayramlı"
"15","ABS","Abşeron"
"15","AGC","Ağcabədi"
"15","AGM","Ağdam"
"15","AGS","Ağdaş"
"15","AGA","Ağstafa"
"15","AGU","Ağsu"
"15","AST","Astara"
"15","BA","Bakı"
"15","BAB","Babək"
"15","BAL","Balakən"
"15","BAR","Bərdə"
"15","BEY","Beyləqan"
"15","BIL","Biləsuvar"
"15","CAB","Cəbrayıl"
"15","CAL","Cəlilabab"
"15","CUL","Julfa"
"15","DAS","Daşkəsən"
"15","DAV","Dəvəçi"
"15","FUZ","Füzuli"
"15","GA","Gəncə"
"15","GAD","Gədəbəy"
"15","GOR","Goranboy"
"15","GOY","Göyçay"
"15","HAC","Hacıqabul"
"15","IMI","İmişli"
"15","ISM","İsmayıllı"
"15","KAL","Kəlbəcər"
"15","KUR","Kürdəmir"
"15","LA","Lənkəran"
"15","LAC","Laçın"
"15","LAN","Lənkəran"
"15","LER","Lerik"
"15","MAS","Masallı"
"15","MI","Mingəçevir"
"15","NA","Naftalan"
"15","NEF","Neftçala"
"15","OGU","Oğuz"
"15","ORD","Ordubad"
"15","QAB","Qəbələ"
"15","QAX","Qax"
"15","QAZ","Qazax"
"15","QOB","Qobustan"
"15","QBA","Quba"
"15","QBI","Qubadlı"
"15","QUS","Qusar"
"15","SA","Şəki"
"15","SAT","Saatlı"
"15","SAB","Sabirabad"
"15","SAD","Sədərək"
"15","SAH","Şahbuz"
"15","SAK","Şəki"
"15","SAL","Salyan"
"15","SM","Sumqayıt"
"15","SMI","Şamaxı"
"15","SKR","Şəmkir"
"15","SMX","Samux"
"15","SAR","Şərur"
"15","SIY","Siyəzən"
"15","SS","Şuşa (City)"
"15","SUS","Şuşa"
"15","TAR","Tərtər"
"15","TOV","Tovuz"
"15","UCA","Ucar"
"15","XA","Xankəndi"
"15","XAC","Xaçmaz"
"15","XAN","Xanlar"
"15","XIZ","Xızı"
"15","XCI","Xocalı"
"15","XVD","Xocavənd"
"15","YAR","Yardımlı"
"15","YE","Yevlax (City)"
"15","YEV","Yevlax"
"15","ZAN","Zəngilan"
"15","ZAQ","Zaqatala"
"15","ZAR","Zərdab"
"15","NX","Nakhichevan"
"16","AC","Acklins and Crooked Islands"
"16","BI","Bimini"
"16","CI","Cat Island"
"16","EX","Exuma"
"16","FR","Freeport"
"16","FC","Fresh Creek"
"16","GH","Governor's Harbour"
"16","GT","Green Turtle Cay"
"16","HI","Harbour Island"
"16","HR","High Rock"
"16","IN","Inagua"
"16","KB","Kemps Bay"
"16","LI","Long Island"
"16","MH","Marsh Harbour"
"16","MA","Mayaguana"
"16","NP","New Providence"
"16","NT","Nicholls Town and Berry Islands"
"16","RI","Ragged Island"
"16","RS","Rock Sound"
"16","SS","San Salvador and Rum Cay"
"16","SP","Sandy Point"
"17","01","الحد"
"17","02","المحرق"
"17","03","المنامة"
"17","04","جد حفص"
"17","05","المنطقة الشمالية"
"17","06","سترة"
"17","07","المنطقة الوسطى"
"17","08","مدينة عيسى"
"17","09","الرفاع والمنطقة الجنوبية"
"17","10","المنطقة الغربية"
"17","11","جزر حوار"
"17","12","مدينة حمد"
"18","01","Bandarban"
"18","02","Barguna"
"18","03","Bogra"
"18","04","Brahmanbaria"
"18","05","Bagerhat"
"18","06","Barisal"
"18","07","Bhola"
"18","08","Comilla"
"18","09","Chandpur"
"18","10","Chittagong"
"18","11","Cox's Bazar"
"18","12","Chuadanga"
"18","13","Dhaka"
"18","14","Dinajpur"
"18","15","Faridpur"
"18","16","Feni"
"18","17","Gopalganj"
"18","18","Gazipur"
"18","19","Gaibandha"
"18","20","Habiganj"
"18","21","Jamalpur"
"18","22","Jessore"
"18","23","Jhenaidah"
"18","24","Jaipurhat"
"18","25","Jhalakati"
"18","26","Kishoreganj"
"18","27","Khulna"
"18","28","Kurigram"
"18","29","Khagrachari"
"18","30","Kushtia"
"18","31","Lakshmipur"
"18","32","Lalmonirhat"
"18","33","Manikganj"
"18","34","Mymensingh"
"18","35","Munshiganj"
"18","36","Madaripur"
"18","37","Magura"
"18","38","Moulvibazar"
"18","39","Meherpur"
"18","40","Narayanganj"
"18","41","Netrakona"
"18","42","Narsingdi"
"18","43","Narail"
"18","44","Natore"
"18","45","Nawabganj"
"18","46","Nilphamari"
"18","47","Noakhali"
"18","48","Naogaon"
"18","49","Pabna"
"18","50","Pirojpur"
"18","51","Patuakhali"
"18","52","Panchagarh"
"18","53","Rajbari"
"18","54","Rajshahi"
"18","55","Rangpur"
"18","56","Rangamati"
"18","57","Sherpur"
"18","58","Satkhira"
"18","59","Sirajganj"
"18","60","Sylhet"
"18","61","Sunamganj"
"18","62","Shariatpur"
"18","63","Tangail"
"18","64","Thakurgaon"
"19","A","Saint Andrew"
"19","C","Christ Church"
"19","E","Saint Peter"
"19","G","Saint George"
"19","J","Saint John"
"19","L","Saint Lucy"
"19","M","Saint Michael"
"19","O","Saint Joseph"
"19","P","Saint Philip"
"19","S","Saint James"
"19","T","Saint Thomas"
"20","BR","Брэ́сцкая во́бласць"
"20","HO","Го́мельская во́бласць"
"20","HR","Гро́дзенская во́бласць"
"20","MA","Магілёўская во́бласць"
"20","MI","Мі́нская во́бласць"
"20","VI","Ві́цебская во́бласць"
"21","BRU","Brussel"
"21","VAN","Antwerpen"
"21","VBR","Vlaams-Brabant"
"21","VLI","Limburg"
"21","VOV","Oost-Vlaanderen"
"21","VWV","West-Vlaanderen"
"21","WBR","Brabant Wallon"
"21","WHT","Hainaut"
"21","WLG","Liège/Lüttich"
"21","WLX","Luxembourg"
"21","WNA","Namur"
"22","BZ","Belize District"
"22","CY","Cayo District"
"22","CZL","Corozal District"
"22","OW","Orange Walk District"
"22","SC","Stann Creek District"
"22","TOL","Toledo District"
"23","AL","Alibori"
"23","AK","Atakora"
"23","AQ","Atlantique"
"23","BO","Borgou"
"23","CO","Collines"
"23","DO","Donga"
"23","KO","Kouffo"
"23","LI","Littoral"
"23","MO","Mono"
"23","OU","Ouémé"
"23","PL","Plateau"
"23","ZO","Zou"
"24","DEV","Devonshire"
"24","HA","Hamilton City"
"24","HAM","Hamilton"
"24","PAG","Paget"
"24","PEM","Pembroke"
"24","SAN","Sandys"
"24","SG","Saint George City"
"24","SGE","Saint George's"
"24","SMI","Smiths"
"24","SOU","Southampton"
"24","WAR","Warwick"
"25","11","Paro"
"25","12","Chukha"
"25","13","Haa"
"25","14","Samtse"
"25","15","Thimphu"
"25","21","Tsirang"
"25","22","Dagana"
"25","23","Punakha"
"25","24","Wangdue Phodrang"
"25","31","Sarpang"
"25","32","Trongsa"
"25","33","Bumthang"
"25","34","Zhemgang"
"25","41","Trashigang"
"25","42","Mongar"
"25","43","Pemagatshel"
"25","44","Luentse"
"25","45","Samdrup Jongkhar"
"25","GA","Gasa"
"25","TY","Trashiyangse"
"26","B","El Beni"
"26","C","Cochabamba"
"26","H","Chuquisaca"
"26","L","La Paz"
"26","N","Pando"
"26","O","Oruro"
"26","P","Potosí"
"26","S","Santa Cruz"
"26","T","Tarija"
"28","CE","Central"
"28","GH","Ghanzi"
"28","KG","Kgalagadi"
"28","KL","Kgatleng"
"28","KW","Kweneng"
"28","NE","North-East"
"28","NW","North-West"
"28","SE","South-East"
"28","SO","Southern"
"30","AC","Acre"
"30","AL","Alagoas"
"30","AM","Amazônia"
"30","AP","Amapá"
"30","BA","Bahia"
"30","CE","Ceará"
"30","DF","Distrito Federal"
"30","ES","Espírito Santo"
"30","GO","Goiás"
"30","MA","Maranhão"
"30","MG","Minas Gerais"
"30","MS","Mato Grosso do Sul"
"30","MT","Mato Grosso"
"30","PA","Pará"
"30","PB","Paraíba"
"30","PE","Pernambuco"
"30","PI","Piauí"
"30","PR","Paraná"
"30","RJ","Rio de Janeiro"
"30","RN","Rio Grande do Norte"
"30","RO","Rondônia"
"30","RR","Roraima"
"30","RS","Rio Grande do Sul"
"30","SC","Santa Catarina"
"30","SE","Sergipe"
"30","SP","São Paulo"
"30","TO","Tocantins"
"31","PB","Peros Banhos"
"31","SI","Salomon Islands"
"31","NI","Nelsons Island"
"31","TB","Three Brothers"
"31","EA","Eagle Islands"
"31","DI","Danger Island"
"31","EG","Egmont Islands"
"31","DG","Diego Garcia"
"32","BE","Belait"
"32","BM","Brunei-Muara"
"32","TE","Temburong"
"32","TU","Tutong"
"33","01","Blagoevgrad"
"33","02","Burgas"
"33","03","Varna"
"33","04","Veliko Tarnovo"
"33","05","Vidin"
"33","06","Vratsa"
"33","07","Gabrovo"
"33","08","Dobrich"
"33","09","Kardzhali"
"33","10","Kyustendil"
"33","11","Lovech"
"33","12","Montana"
"33","13","Pazardzhik"
"33","14","Pernik"
"33","15","Pleven"
"33","16","Plovdiv"
"33","17","Razgrad"
"33","18","Ruse"
"33","19","Silistra"
"33","20","Sliven"
"33","21","Smolyan"
"33","23","Sofia"
"33","22","Sofia Province"
"33","24","Stara Zagora"
"33","25","Targovishte"
"33","26","Haskovo"
"33","27","Shumen"
"33","28","Yambol"
"34","BAL","Balé"
"34","BAM","Bam"
"34","BAN","Banwa"
"34","BAZ","Bazèga"
"34","BGR","Bougouriba"
"34","BLG","Boulgou"
"34","BLK","Boulkiemdé"
"34","COM","Komoé"
"34","GAN","Ganzourgou"
"34","GNA","Gnagna"
"34","GOU","Gourma"
"34","HOU","Houet"
"34","IOB","Ioba"
"34","KAD","Kadiogo"
"34","KEN","Kénédougou"
"34","KMD","Komondjari"
"34","KMP","Kompienga"
"34","KOP","Koulpélogo"
"34","KOS","Kossi"
"34","KOT","Kouritenga"
"34","KOW","Kourwéogo"
"34","LER","Léraba"
"34","LOR","Loroum"
"34","MOU","Mouhoun"
"34","NAM","Namentenga"
"34","NAO","Naouri"
"34","NAY","Nayala"
"34","NOU","Noumbiel"
"34","OUB","Oubritenga"
"34","OUD","Oudalan"
"34","PAS","Passoré"
"34","PON","Poni"
"34","SEN","Séno"
"34","SIS","Sissili"
"34","SMT","Sanmatenga"
"34","SNG","Sanguié"
"34","SOM","Soum"
"34","SOR","Sourou"
"34","TAP","Tapoa"
"34","TUI","Tui"
"34","YAG","Yagha"
"34","YAT","Yatenga"
"34","ZIR","Ziro"
"34","ZON","Zondoma"
"34","ZOU","Zoundwéogo"
"35","BB","Bubanza"
"35","BJ","Bujumbura Mairie"
"35","BR","Bururi"
"35","CA","Cankuzo"
"35","CI","Cibitoke"
"35","GI","Gitega"
"35","KR","Karuzi"
"35","KY","Kayanza"
"35","KI","Kirundo"
"35","MA","Makamba"
"35","MU","Muramvya"
"35","MY","Muyinga"
"35","MW","Mwaro"
"35","NG","Ngozi"
"35","RT","Rutana"
"35","RY","Ruyigi"
"37","AD","Adamaoua"
"37","CE","Centre"
"37","EN","Extrême-Nord"
"37","ES","Est"
"37","LT","Littoral"
"37","NO","Nord"
"37","NW","Nord-Ouest"
"37","OU","Ouest"
"37","SU","Sud"
"37","SW","Sud-Ouest"
"38","AB","Alberta"
"38","BC","British Columbia"
"38","MB","Manitoba"
"38","NB","New Brunswick"
"38","NL","Newfoundland and Labrador"
"38","NS","Nova Scotia"
"38","NT","Northwest Territories"
"38","NU","Nunavut"
"38","ON","Ontario"
"38","PE","Prince Edward Island"
"38","QC","Quebec"
"38","SK","Saskatchewan"
"38","YT","Yukon Territory"
"39","BR","Brava"
"39","BV","Boa Vista"
"39","CA","Santa Catarina"
"39","CR","Santa Cruz"
"39","CS","Calheta de São Miguel"
"39","MA","Maio"
"39","MO","Mosteiros"
"39","PA","Paúl"
"39","PN","Porto Novo"
"39","PR","Praia"
"39","RG","Ribeira Grande"
"39","SD","São Domingos"
"39","SF","São Filipe"
"39","SL","Sal"
"39","SN","São Nicolau"
"39","SV","São Vicente"
"39","TA","Tarrafal"
"40","CR","Creek"
"40","EA","Eastern"
"40","MI","Midland"
"40","SO","South Town"
"40","SP","Spot Bay"
"40","ST","Stake Bay"
"40","WD","West End"
"40","WN","Western"
"41","AC ","Ouham"
"41","BB ","Bamingui-Bangoran"
"41","BGF","Bangui"
"41","BK ","Basse-Kotto"
"41","HK ","Haute-Kotto"
"41","HM ","Haut-Mbomou"
"41","HS ","Mambéré-Kadéï"
"41","KB ","Nana-Grébizi"
"41","KG ","Kémo"
"41","LB ","Lobaye"
"41","MB ","Mbomou"
"41","MP ","Ombella-M'Poko"
"41","NM ","Nana-Mambéré"
"41","OP ","Ouham-Pendé"
"41","SE ","Sangha-Mbaéré"
"41","UK ","Ouaka"
"41","VR ","Vakaga"
"42","BA ","Batha"
"42","BET","Borkou-Ennedi-Tibesti"
"42","BI ","Biltine"
"42","CB ","Chari-Baguirmi"
"42","GR ","Guéra"
"42","KA ","Kanem"
"42","LC ","Lac"
"42","LR ","Logone-Oriental"
"42","LO ","Logone-Occidental"
"42","MC ","Moyen-Chari"
"42","MK ","Mayo-Kébbi"
"42","OD ","Ouaddaï"
"42","SA ","Salamat"
"42","TA ","Tandjilé"
"43","AI","Aisén del General Carlos Ibañez"
"43","AN","Antofagasta"
"43","AR","La Araucanía"
"43","AT","Atacama"
"43","BI","Biobío"
"43","CO","Coquimbo"
"43","LI","Libertador Bernardo O'Higgins"
"43","LL","Los Lagos"
"43","MA","Magallanes y de la Antartica"
"43","ML","Maule"
"43","RM","Metropolitana de Santiago"
"43","TA","Tarapacá"
"43","VS","Valparaíso"
"44","11","北京"
"44","12","天津"
"44","13","河北"
"44","14","山西"
"44","15","内蒙古自治区"
"44","21","辽宁"
"44","22","吉林"
"44","23","黑龙江省"
"44","31","上海"
"44","32","江苏"
"44","33","浙江"
"44","34","安徽"
"44","35","福建"
"44","36","江西"
"44","37","山东"
"44","41","河南"
"44","42","湖北"
"44","43","湖南"
"44","44","广东"
"44","45","广西壮族自治区"
"44","46","海南"
"44","50","重庆"
"44","51","四川"
"44","52","贵州"
"44","53","云南"
"44","54","西藏自治区"
"44","61","陕西"
"44","62","甘肃"
"44","63","青海"
"44","64","宁夏"
"44","65","新疆"
"44","71","臺灣"
"44","91","香港"
"44","92","澳門"
"46","D","Direction Island"
"46","H","Home Island"
"46","O","Horsburgh Island"
"46","S","South Island"
"46","W","West Island"
"47","AMA","Amazonas"
"47","ANT","Antioquia"
"47","ARA","Arauca"
"47","ATL","Atlántico"
"47","BOL","Bolívar"
"47","BOY","Boyacá"
"47","CAL","Caldas"
"47","CAQ","Caquetá"
"47","CAS","Casanare"
"47","CAU","Cauca"
"47","CES","Cesar"
"47","CHO","Chocó"
"47","COR","Córdoba"
"47","CUN","Cundinamarca"
"47","DC","Bogotá Distrito Capital"
"47","GUA","Guainía"
"47","GUV","Guaviare"
"47","HUI","Huila"
"47","LAG","La Guajira"
"47","MAG","Magdalena"
"47","MET","Meta"
"47","NAR","Nariño"
"47","NSA","Norte de Santander"
"47","PUT","Putumayo"
"47","QUI","Quindío"
"47","RIS","Risaralda"
"47","SAN","Santander"
"47","SAP","San Andrés y Providencia"
"47","SUC","Sucre"
"47","TOL","Tolima"
"47","VAC","Valle del Cauca"
"47","VAU","Vaupés"
"47","VID","Vichada"
"48","A","Anjouan"
"48","G","Grande Comore"
"48","M","Mohéli"
"49","BC","Congo-Central"
"49","BN","Bandundu"
"49","EQ","Équateur"
"49","KA","Katanga"
"49","KE","Kasai-Oriental"
"49","KN","Kinshasa"
"49","KW","Kasai-Occidental"
"49","MA","Maniema"
"49","NK","Nord-Kivu"
"49","OR","Orientale"
"49","SK","Sud-Kivu"
"50","PU","Pukapuka"
"50","RK","Rakahanga"
"50","MK","Manihiki"
"50","PE","Penrhyn"
"50","NI","Nassau Island"
"50","SU","Surwarrow"
"50","PA","Palmerston"
"50","AI","Aitutaki"
"50","MA","Manuae"
"50","TA","Takutea"
"50","MT","Mitiaro"
"50","AT","Atiu"
"50","MU","Mauke"
"50","RR","Rarotonga"
"50","MG","Mangaia"
"51","A","Alajuela"
"51","C","Cartago"
"51","G","Guanacaste"
"51","H","Heredia"
"51","L","Limón"
"51","P","Puntarenas"
"51","SJ","San José"
"52","01","Lagunes"
"52","02","Haut-Sassandra"
"52","03","Savanes"
"52","04","Vallée du Bandama"
"52","05","Moyen-Comoé"
"52","06","Dix-Huit"
"52","07","Lacs"
"52","08","Zanzan"
"52","09","Bas-Sassandra"
"52","10","Denguélé"
"52","11","N'zi-Comoé"
"52","12","Marahoué"
"52","13","Sud-Comoé"
"52","14","Worodouqou"
"52","15","Sud-Bandama"
"52","16","Agnébi"
"52","17","Bafing"
"52","18","Fromager"
"52","19","Moyen-Cavally"
"53","01","Zagrebačka županija"
"53","02","Krapinsko-zagorska županija"
"53","03","Sisačko-moslavačka županija"
"53","04","Karlovačka županija"
"53","05","Varaždinska županija"
"53","06","Koprivničko-križevačka županija"
"53","07","Bjelovarsko-bilogorska županija"
"53","08","Primorsko-goranska županija"
"53","09","Ličko-senjska županija"
"53","10","Virovitičko-podravska županija"
"53","11","Požeško-slavonska županija"
"53","12","Brodsko-posavska županija"
"53","13","Zadarska županija"
"53","14","Osječko-baranjska županija"
"53","15","Šibensko-kninska županija"
"53","16","Vukovarsko-srijemska županija"
"53","17","Splitsko-dalmatinska županija"
"53","18","Istarska županija"
"53","19","Dubrovačko-neretvanska županija"
"53","20","Međimurska županija"
"53","21","Zagreb"
"54","01","Pinar del Río"
"54","02","La Habana"
"54","03","Ciudad de La Habana"
"54","04","Matanzas"
"54","05","Villa Clara"
"54","06","Cienfuegos"
"54","07","Sancti Spíritus"
"54","08","Ciego de Ávila"
"54","09","Camagüey"
"54","10","Las Tunas"
"54","11","Holguín"
"54","12","Granma"
"54","13","Santiago de Cuba"
"54","14","Guantánamo"
"54","99","Isla de la Juventud"
"55","01","Κερύvεια"
"55","02","Λευκωσία"
"55","03","Αμμόχωστος"
"55","04","Λάρνακα"
"55","05","Λεμεσός"
"55","06","Πάφος"
"56","JC","Jihočeský kraj"
"56","JM","Jihomoravský kraj"
"56","KA","Karlovarský kraj"
"56","VY","Vysočina kraj"
"56","KR","Královéhradecký kraj"
"56","LI","Liberecký kraj"
"56","MO","Moravskoslezský kraj"
"56","OL","Olomoucký kraj"
"56","PA","Pardubický kraj"
"56","PL","Plzeňský kraj"
"56","PR","Hlavní město Praha"
"56","ST","Středočeský kraj"
"56","US","Ústecký kraj"
"56","ZL","Zlínský kraj"
"57","040","Bornholms Regionskommune"
"57","101","København"
"57","147","Frederiksberg"
"57","070","Århus Amt"
"57","015","Københavns Amt"
"57","020","Frederiksborg Amt"
"57","042","Fyns Amt"
"57","080","Nordjyllands Amt"
"57","055","Ribe Amt"
"57","065","Ringkjøbing Amt"
"57","025","Roskilde Amt"
"57","050","Sønderjyllands Amt"
"57","035","Storstrøms Amt"
"57","060","Vejle Amt"
"57","030","Vestsjællands Amt"
"57","076","Viborg Amt"
"58","AS","Region d'Ali Sabieh"
"58","AR","Region d'Arta"
"58","DI","Region de Dikhil"
"58","DJ","Ville de Djibouti"
"58","OB","Region d'Obock"
"58","TA","Region de Tadjourah"
"59","AND","Saint Andrew Parish"
"59","DAV","Saint David Parish"
"59","GEO","Saint George Parish"
"59","JOH","Saint John Parish"
"59","JOS","Saint Joseph Parish"
"59","LUK","Saint Luke Parish"
"59","MAR","Saint Mark Parish"
"59","PAT","Saint Patrick Parish"
"59","PAU","Saint Paul Parish"
"59","PET","Saint Peter Parish"
"60","01","Distrito Nacional"
"60","02","Ázua"
"60","03","Baoruco"
"60","04","Barahona"
"60","05","Dajabón"
"60","06","Duarte"
"60","07","Elías Piña"
"60","08","El Seibo"
"60","09","Espaillat"
"60","10","Independencia"
"60","11","La Altagracia"
"60","12","La Romana"
"60","13","La Vega"
"60","14","María Trinidad Sánchez"
"60","15","Monte Cristi"
"60","16","Pedernales"
"60","17","Peravia"
"60","18","Puerto Plata"
"60","19","Salcedo"
"60","20","Samaná"
"60","21","San Cristóbal"
"60","22","San Juan"
"60","23","San Pedro de Macorís"
"60","24","Sánchez Ramírez"
"60","25","Santiago"
"60","26","Santiago Rodríguez"
"60","27","Valverde"
"60","28","Monseñor Nouel"
"60","29","Monte Plata"
"60","30","Hato Mayor"
"61","AL","Aileu"
"61","AN","Ainaro"
"61","BA","Baucau"
"61","BO","Bobonaro"
"61","CO","Cova-Lima"
"61","DI","Dili"
"61","ER","Ermera"
"61","LA","Lautem"
"61","LI","Liquiçá"
"61","MF","Manufahi"
"61","MT","Manatuto"
"61","OE","Oecussi"
"61","VI","Viqueque"
"62","A","Azuay"
"62","B","Bolívar"
"62","C","Carchi"
"62","D","Orellana"
"62","E","Esmeraldas"
"62","F","Cañar"
"62","G","Guayas"
"62","H","Chimborazo"
"62","I","Imbabura"
"62","L","Loja"
"62","M","Manabí"
"62","N","Napo"
"62","O","El Oro"
"62","P","Pichincha"
"62","R","Los Ríos"
"62","S","Morona-Santiago"
"62","T","Tungurahua"
"62","U","Sucumbíos"
"62","W","Galápagos"
"62","X","Cotopaxi"
"62","Y","Pastaza"
"62","Z","Zamora-Chinchipe"
"63","ALX","الإسكندرية"
"63","ASN","أسوان"
"63","AST","أسيوط"
"63","BA","البحر الأحمر"
"63","BH","البحيرة"
"63","BNS","بني سويف"
"63","C","القاهرة"
"63","DK","الدقهلية"
"63","DT","دمياط"
"63","FYM","الفيوم"
"63","GH","الغربية"
"63","GZ","الجيزة"
"63","IS","الإسماعيلية"
"63","JS","جنوب سيناء"
"63","KB","القليوبية"
"63","KFS","كفر الشيخ"
"63","KN","قنا"
"63","MN","محافظة المنيا"
"63","MNF","المنوفية"
"63","MT","مطروح"
"63","PTS","محافظة بور سعيد"
"63","SHG","محافظة سوهاج"
"63","SHR","المحافظة الشرقيّة"
"63","SIN","شمال سيناء"
"63","SUZ","السويس"
"63","WAD","الوادى الجديد"
"64","AH","Ahuachapán"
"64","CA","Cabañas"
"64","CH","Chalatenango"
"64","CU","Cuscatlán"
"64","LI","La Libertad"
"64","MO","Morazán"
"64","PA","La Paz"
"64","SA","Santa Ana"
"64","SM","San Miguel"
"64","SO","Sonsonate"
"64","SS","San Salvador"
"64","SV","San Vicente"
"64","UN","La Unión"
"64","US","Usulután"
"65","AN","Annobón"
"65","BN","Bioko Norte"
"65","BS","Bioko Sur"
"65","CS","Centro Sur"
"65","KN","Kié-Ntem"
"65","LI","Litoral"
"65","WN","Wele-Nzas"
"66","AN","Zoba Anseba"
"66","DK","Zoba Debubawi Keyih Bahri"
"66","DU","Zoba Debub"
"66","GB","Zoba Gash-Barka"
"66","MA","Zoba Ma'akel"
"66","SK","Zoba Semienawi Keyih Bahri"
"67","37","Harju maakond"
"67","39","Hiiu maakond"
"67","44","Ida-Viru maakond"
"67","49","Jõgeva maakond"
"67","51","Järva maakond"
"67","57","Lääne maakond"
"67","59","Lääne-Viru maakond"
"67","65","Põlva maakond"
"67","67","Pärnu maakond"
"67","70","Rapla maakond"
"67","74","Saare maakond"
"67","78","Tartu maakond"
"67","82","Valga maakond"
"67","84","Viljandi maakond"
"67","86","Võru maakond"
"68","AA","አዲስ አበባ"
"68","AF","አፋር"
"68","AH","አማራ"
"68","BG","ቤንሻንጉል-ጉምዝ"
"68","DD","ድሬዳዋ"
"68","GB","ጋምቤላ ሕዝቦች"
"68","HR","ሀረሪ ሕዝብ"
"68","OR","ኦሮሚያ"
"68","SM","ሶማሌ"
"68","SN","ደቡብ ብሔሮች ብሔረሰቦችና ሕዝቦች"
"68","TG","ትግራይ"
"71","C","Central"
"71","E","Northern"
"71","N","Eastern"
"71","R","Rotuma"
"71","W","Western"
"72","AL","Ahvenanmaan maakunta"
"72","ES","Etelä-Suomen lääni"
"72","IS","Itä-Suomen lääni"
"72","LL","Lapin lääni"
"72","LS","Länsi-Suomen lääni"
"72","OL","Oulun lääni"
"73","01","Ain"
"73","02","Aisne"
"73","03","Allier"
"73","04","Alpes-de-Haute-Provence"
"73","05","Hautes-Alpes"
"73","06","Alpes-Maritimes"
"73","07","Ardèche"
"73","08","Ardennes"
"73","09","Ariège"
"73","10","Aube"
"73","11","Aude"
"73","12","Aveyron"
"73","13","Bouches-du-Rhône"
"73","14","Calvados"
"73","15","Cantal"
"73","16","Charente"
"73","17","Charente-Maritime"
"73","18","Cher"
"73","19","Corrèze"
"73","21","Côte-d'Or"
"73","22","Côtes-d'Armor"
"73","23","Creuse"
"73","24","Dordogne"
"73","25","Doubs"
"73","26","Drôme"
"73","27","Eure"
"73","28","Eure-et-Loir"
"73","29","Finistère"
"73","2A","Corse-du-Sud"
"73","2B","Haute-Corse"
"73","30","Gard"
"73","31","Haute-Garonne"
"73","32","Gers"
"73","33","Gironde"
"73","34","Hérault"
"73","35","Ille-et-Vilaine"
"73","36","Indre"
"73","37","Indre-et-Loire"
"73","38","Isère"
"73","39","Jura"
"73","40","Landes"
"73","41","Loir-et-Cher"
"73","42","Loire"
"73","43","Haute-Loire"
"73","44","Loire-Atlantique"
"73","45","Loiret"
"73","46","Lot"
"73","47","Lot-et-Garonne"
"73","48","Lozère"
"73","49","Maine-et-Loire"
"73","50","Manche"
"73","51","Marne"
"73","52","Haute-Marne"
"73","53","Mayenne"
"73","54","Meurthe-et-Moselle"
"73","55","Meuse"
"73","56","Morbihan"
"73","57","Moselle"
"73","58","Nièvre"
"73","59","Nord"
"73","60","Oise"
"73","61","Orne"
"73","62","Pas-de-Calais"
"73","63","Puy-de-Dôme"
"73","64","Pyrénées-Atlantiques"
"73","65","Hautes-Pyrénées"
"73","66","Pyrénées-Orientales"
"73","67","Bas-Rhin"
"73","68","Haut-Rhin"
"73","69","Rhône"
"73","70","Haute-Saône"
"73","71","Saône-et-Loire"
"73","72","Sarthe"
"73","73","Savoie"
"73","74","Haute-Savoie"
"73","75","Paris"
"73","76","Seine-Maritime"
"73","77","Seine-et-Marne"
"73","78","Yvelines"
"73","79","Deux-Sèvres"
"73","80","Somme"
"73","81","Tarn"
"73","82","Tarn-et-Garonne"
"73","83","Var"
"73","84","Vaucluse"
"73","85","Vendée"
"73","86","Vienne"
"73","87","Haute-Vienne"
"73","88","Vosges"
"73","89","Yonne"
"73","90","Territoire de Belfort"
"73","91","Essonne"
"73","92","Hauts-de-Seine"
"73","93","Seine-Saint-Denis"
"73","94","Val-de-Marne"
"73","95","Val-d'Oise"
"73","NC","Territoire des Nouvelle-Calédonie et Dependances"
"73","PF","Polynésie Française"
"73","PM","Saint-Pierre et Miquelon"
"73","TF","Terres australes et antarctiques françaises"
"73","YT","Mayotte"
"73","WF","Territoire des îles Wallis et Futuna"
"76","M","Archipel des Marquises"
"76","T","Archipel des Tuamotu"
"76","I","Archipel des Tubuai"
"76","V","Iles du Vent"
"76","S","Iles Sous-le-Vent "
"77","C","Iles Crozet"
"77","K","Iles Kerguelen"
"77","A","Ile Amsterdam"
"77","P","Ile Saint-Paul"
"77","D","Adelie Land"
"78","ES","Estuaire"
"78","HO","Haut-Ogooue"
"78","MO","Moyen-Ogooue"
"78","NG","Ngounie"
"78","NY","Nyanga"
"78","OI","Ogooue-Ivindo"
"78","OL","Ogooue-Lolo"
"78","OM","Ogooue-Maritime"
"78","WN","Woleu-Ntem"
"79","AH","Ashanti"
"79","BA","Brong-Ahafo"
"79","CP","Central"
"79","EP","Eastern"
"79","AA","Greater Accra"
"79","NP","Northern"
"79","UE","Upper East"
"79","UW","Upper West"
"79","TV","Volta"
"79","WP","Western"
"80","AB","აფხაზეთი"
"80","AJ","აჭარა"
"80","GU","გურია"
"80","IM","იმერეთი"
"80","KA","კახეთი"
"80","KK","ქვემო ქართლი"
"80","MM","მცხეთა-მთიანეთი"
"80","RL","რაჭა-ლეჩხუმი და ქვემო სვანეთი"
"80","SJ","სამცხე-ჯავახეთი"
"80","SK","შიდა ქართლი"
"80","SZ","სამეგრელო-ზემო სვანეთი"
"80","TB","თბილისი"
"81","BE","Berlin"
"81","BR","Brandenburg"
"81","BW","Baden-Württemberg"
"81","BY","Bayern"
"81","HB","Bremen"
"81","HE","Hessen"
"81","HH","Hamburg"
"81","MV","Mecklenburg-Vorpommern"
"81","NI","Niedersachsen"
"81","NW","Nordrhein-Westfalen"
"81","RP","Rheinland-Pfalz"
"81","SH","Schleswig-Holstein"
"81","SL","Saarland"
"81","SN","Sachsen"
"81","ST","Sachsen-Anhalt"
"81","TH","Thüringen"
"82","AA","Greater Accra"
"82","AH","Ashanti"
"82","BA","Brong-Ahafo"
"82","CP","Central"
"82","EP","Eastern"
"82","NP","Northern"
"82","TV","Volta"
"82","UE","Upper East"
"82","UW","Upper West"
"82","WP","Western"
"84","01","Αιτωλοακαρνανία"
"84","03","Βοιωτία"
"84","04","Εύβοια"
"84","05","Ευρυτανία"
"84","06","Φθιώτιδα"
"84","07","Φωκίδα"
"84","11","Αργολίδα"
"84","12","Αρκαδία"
"84","13","Ἀχαΐα"
"84","14","Ηλεία"
"84","15","Κορινθία"
"84","16","Λακωνία"
"84","17","Μεσσηνία"
"84","21","Ζάκυνθος"
"84","22","Κέρκυρα"
"84","23","Κεφαλλονιά"
"84","24","Λευκάδα"
"84","31","Άρτα"
"84","32","Θεσπρωτία"
"84","33","Ιωάννινα"
"84","34","Πρεβεζα"
"84","41","Καρδίτσα"
"84","42","Λάρισα"
"84","43","Μαγνησία"
"84","44","Τρίκαλα"
"84","51","Γρεβενά"
"84","52","Δράμα"
"84","53","Ημαθία"
"84","54","Θεσσαλονίκη"
"84","55","Καβάλα"
"84","56","Καστοριά"
"84","57","Κιλκίς"
"84","58","Κοζάνη"
"84","59","Πέλλα"
"84","61","Πιερία"
"84","62","Σερρών"
"84","63","Φλώρινα"
"84","64","Χαλκιδική"
"84","69","Όρος Άθως"
"84","71","Έβρος"
"84","72","Ξάνθη"
"84","73","Ροδόπη"
"84","81","Δωδεκάνησα"
"84","82","Κυκλάδες"
"84","83","Λέσβου"
"84","84","Σάμος"
"84","85","Χίος"
"84","91","Ηράκλειο"
"84","92","Λασίθι"
"84","93","Ρεθύμνο"
"84","94","Χανίων"
"84","A1","Αττική"
"85","A","Avannaa"
"85","T","Tunu "
"85","K","Kitaa"
"86","A","Saint Andrew"
"86","D","Saint David"
"86","G","Saint George"
"86","J","Saint John"
"86","M","Saint Mark"
"86","P","Saint Patrick"
"89","AV","Alta Verapaz"
"89","BV","Baja Verapaz"
"89","CM","Chimaltenango"
"89","CQ","Chiquimula"
"89","ES","Escuintla"
"89","GU","Guatemala"
"89","HU","Huehuetenango"
"89","IZ","Izabal"
"89","JA","Jalapa"
"89","JU","Jutiapa"
"89","PE","El Petén"
"89","PR","El Progreso"
"89","QC","El Quiché"
"89","QZ","Quetzaltenango"
"89","RE","Retalhuleu"
"89","SA","Sacatepéquez"
"89","SM","San Marcos"
"89","SO","Sololá"
"89","SR","Santa Rosa"
"89","SU","Suchitepéquez"
"89","TO","Totonicapán"
"89","ZA","Zacapa"
"90","BE","Beyla"
"90","BF","Boffa"
"90","BK","Boké"
"90","CO","Coyah"
"90","DB","Dabola"
"90","DI","Dinguiraye"
"90","DL","Dalaba"
"90","DU","Dubréka"
"90","FA","Faranah"
"90","FO","Forécariah"
"90","FR","Fria"
"90","GA","Gaoual"
"90","GU","Guékédou"
"90","KA","Kankan"
"90","KB","Koubia"
"90","KD","Kindia"
"90","KE","Kérouané"
"90","KN","Koundara"
"90","KO","Kouroussa"
"90","KS","Kissidougou"
"90","LA","Labé"
"90","LE","Lélouma"
"90","LO","Lola"
"90","MC","Macenta"
"90","MD","Mandiana"
"90","ML","Mali"
"90","MM","Mamou"
"90","NZ","Nzérékoré"
"90","PI","Pita"
"90","SI","Siguiri"
"90","TE","Télimélé"
"90","TO","Tougué"
"90","YO","Yomou"
"91","BF","Bafata"
"91","BB","Biombo"
"91","BS","Bissau"
"91","BL","Bolama"
"91","CA","Cacheu"
"91","GA","Gabu"
"91","OI","Oio"
"91","QU","Quinara"
"91","TO","Tombali"
"92","BA","Barima-Waini"
"92","CU","Cuyuni-Mazaruni"
"92","DE","Demerara-Mahaica"
"92","EB","East Berbice-Corentyne"
"92","ES","Essequibo Islands-West Demerara"
"92","MA","Mahaica-Berbice"
"92","PM","Pomeroon-Supenaam"
"92","PT","Potaro-Siparuni"
"92","UD","Upper Demerara-Berbice"
"92","UT","Upper Takutu-Upper Essequibo"
"93","AR","Artibonite"
"93","CE","Centre"
"93","GA","Grand'Anse"
"93","NI","Nippes"
"93","ND","Nord"
"93","NE","Nord-Est"
"93","NO","Nord-Ouest"
"93","OU","Ouest"
"93","SD","Sud"
"93","SE","Sud-Est"
"94","F","Flat Island"
"94","M","McDonald Island"
"94","S","Shag Island"
"94","H","Heard Island"
"95","AT","Atlántida"
"95","CH","Choluteca"
"95","CL","Colón"
"95","CM","Comayagua"
"95","CP","Copán"
"95","CR","Cortés"
"95","EP","El Paraíso"
"95","FM","Francisco Morazán"
"95","GD","Gracias a Dios"
"95","IB","Islas de la Bahía"
"95","IN","Intibucá"
"95","LE","Lempira"
"95","LP","La Paz"
"95","OC","Ocotepeque"
"95","OL","Olancho"
"95","SB","Santa Bárbara"
"95","VA","Valle"
"95","YO","Yoro"
"96","HCW","中西區"
"96","HEA","東區"
"96","HSO","南區"
"96","HWC","灣仔區"
"96","KKC","九龍城區"
"96","KKT","觀塘區"
"96","KSS","深水埗區"
"96","KWT","黃大仙區"
"96","KYT","油尖旺區"
"96","NIS","離島區"
"96","NKT","葵青區"
"96","NNO","北區"
"96","NSK","西貢區"
"96","NST","沙田區"
"96","NTP","大埔區"
"96","NTW","荃灣區"
"96","NTM","屯門區"
"96","NYL","元朗區"
"97","BA","Baranja megye"
"97","BC","Békéscsaba"
"97","BE","Békés megye"
"97","BK","Bács-Kiskun megye"
"97","BU","Budapest"
"97","BZ","Borsod-Abaúj-Zemplén megye"
"97","CS","Csongrád megye"
"97","DE","Debrecen"
"97","DU","Dunaújváros"
"97","EG","Eger"
"97","FE","Fejér megye"
"97","GS","Győr-Moson-Sopron megye"
"97","GY","Győr"
"97","HB","Hajdú-Bihar megye"
"97","HE","Heves megye"
"97","HV","Hódmezővásárhely"
"97","JN","Jász-Nagykun-Szolnok megye"
"97","KE","Komárom-Esztergom megye"
"97","KM","Kecskemét"
"97","KV","Kaposvár"
"97","MI","Miskolc"
"97","NK","Nagykanizsa"
"97","NO","Nógrád megye"
"97","NY","Nyíregyháza"
"97","PE","Pest megye"
"97","PS","Pécs"
"97","SD","Szeged"
"97","SF","Székesfehérvár"
"97","SH","Szombathely"
"97","SK","Szolnok"
"97","SN","Sopron"
"97","SO","Somogy megye"
"97","SS","Szekszárd"
"97","ST","Salgótarján"
"97","SZ","Szabolcs-Szatmár-Bereg megye"
"97","TB","Tatabánya"
"97","TO","Tolna megye"
"97","VA","Vas megye"
"97","VE","Veszprém megye"
"97","VM","Veszprém"
"97","ZA","Zala megye"
"97","ZE","Zalaegerszeg"
"98","1","Höfuðborgarsvæðið"
"98","2","Suðurnes"
"98","3","Vesturland"
"98","4","Vestfirðir"
"98","5","Norðurland vestra"
"98","6","Norðurland eystra"
"98","7","Austfirðir"
"98","8","Suðurland"
"99","IN-AN","अंडमान और निकोबार द्वीप"
"99","IN-AP","ఆంధ్ర ప్రదేశ్"
"99","IN-AR","अरुणाचल प्रदेश"
"99","IN-AS","অসম"
"99","IN-BR","बिहार"
"99","IN-CH","चंडीगढ़"
"99","IN-CT","छत्तीसगढ़"
"99","IN-DD","દમણ અને દિવ"
"99","IN-DL","दिल्ली"
"99","IN-DN","દાદરા અને નગર હવેલી"
"99","IN-GA","गोंय"
"99","IN-GJ","ગુજરાત"
"99","IN-HP","हिमाचल प्रदेश"
"99","IN-HR","हरियाणा"
"99","IN-JH","झारखंड"
"99","IN-JK","जम्मू और कश्मीर"
"99","IN-KA","ಕನಾ೯ಟಕ"
"99","IN-KL","കേരളം"
"99","IN-LD","ലക്ഷദ്വീപ്"
"99","IN-ML","मेघालय"
"99","IN-MH","महाराष्ट्र"
"99","IN-MN","मणिपुर"
"99","IN-MP","मध्य प्रदेश"
"99","IN-MZ","मिज़ोरम"
"99","IN-NL","नागालैंड"
"99","IN-OR","उड़ीसा"
"99","IN-PB","ਪੰਜਾਬ"
"99","IN-PY","புதுச்சேரி"
"99","IN-RJ","राजस्थान"
"99","IN-SK","सिक्किम"
"99","IN-TN","தமிழ் நாடு"
"99","IN-TR","ত্রিপুরা"
"99","IN-UL","उत्तरांचल"
"99","IN-UP","उत्तर प्रदेश"
"99","IN-WB","পশ্চিমবঙ্গ"
"100","AC","Aceh"
"100","BA","Bali"
"100","BB","Bangka-Belitung"
"100","BE","Bengkulu"
"100","BT","Banten"
"100","GO","Gorontalo"
"100","IJ","Papua"
"100","JA","Jambi"
"100","JI","Jawa Timur"
"100","JK","Jakarta Raya"
"100","JR","Jawa Barat"
"100","JT","Jawa Tengah"
"100","KB","Kalimantan Barat"
"100","KI","Kalimantan Timur"
"100","KS","Kalimantan Selatan"
"100","KT","Kalimantan Tengah"
"100","LA","Lampung"
"100","MA","Maluku"
"100","MU","Maluku Utara"
"100","NB","Nusa Tenggara Barat"
"100","NT","Nusa Tenggara Timur"
"100","RI","Riau"
"100","SB","Sumatera Barat"
"100","SG","Sulawesi Tenggara"
"100","SL","Sumatera Selatan"
"100","SN","Sulawesi Selatan"
"100","ST","Sulawesi Tengah"
"100","SW","Sulawesi Utara"
"100","SU","Sumatera Utara"
"100","YO","Yogyakarta"
"101","01","محافظة آذربایجان شرقي"
"101","02","محافظة آذربایجان غربي"
"101","03","محافظة اردبیل"
"101","04","محافظة اصفهان"
"101","05","محافظة ایلام"
"101","06","محافظة بوشهر"
"101","07","محافظة طهران"
"101","08","محافظة چهارمحل و بختیاري"
"101","09","محافظة خراسان رضوي"
"101","10","محافظة خوزستان"
"101","11","محافظة زنجان"
"101","12","محافظة سمنان"
"101","13","محافظة سيستان وبلوتشستان"
"101","14","محافظة فارس"
"101","15","محافظة کرمان"
"101","16","محافظة کردستان"
"101","17","محافظة کرمانشاه"
"101","18","محافظة کهکیلویه و بویر أحمد"
"101","19","محافظة گیلان"
"101","20","محافظة لرستان"
"101","21","محافظة مازندران"
"101","22","محافظة مرکزي"
"101","23","محافظة هرمزگان"
"101","24","محافظة همدان"
"101","25","محافظة یزد"
"101","26","محافظة قم"
"101","27","محافظة گلستان"
"101","28","محافظة قزوين"
"102","AN","محافظة الأنبار"
"102","AR","أربيل"
"102","BA","محافظة البصرة"
"102","BB","بابل"
"102","BG","محافظة بغداد"
"102","DA","دهوك"
"102","DI","ديالى"
"102","DQ","ذي قار"
"102","KA","كربلاء"
"102","MA","ميسان"
"102","MU","المثنى"
"102","NA","النجف"
"102","NI","نینوى"
"102","QA","القادسية"
"102","SD","صلاح الدين"
"102","SW","محافظة السليمانية"
"102","TS","التأمیم"
"102","WA","واسط"
"103","C","Corcaigh"
"103","CE","Contae an Chláir"
"103","CN","An Cabhán"
"103","CW","Ceatharlach"
"103","D","Baile Átha Cliath"
"103","DL","Dún na nGall"
"103","G","Gaillimh"
"103","KE","Cill Dara"
"103","KK","Cill Chainnigh"
"103","KY","Contae Chiarraí"
"103","LD","An Longfort"
"103","LH","Contae Lú"
"103","LK","Luimneach"
"103","LM","Contae Liatroma"
"103","LS","Contae Laoise"
"103","MH","Contae na Mí"
"103","MN","Muineachán"
"103","MO","Contae Mhaigh Eo"
"103","OY","Contae Uíbh Fhailí"
"103","RN","Ros Comáin"
"103","SO","Sligeach"
"103","TA","Tiobraid Árann"
"103","WD","Port Lairge"
"103","WH","Contae na hIarmhí"
"103","WW","Cill Mhantáin"
"103","WX","Loch Garman"
"104","D ","מחוז הדרום"
"104","HA","מחוז חיפה"
"104","JM","ירושלים"
"104","M ","מחוז המרכז"
"104","TA","תל אביב-יפו"
"104","Z ","מחוז הצפון"
"105","AG","Agrigento"
"105","AL","Alessandria"
"105","AN","Ancona"
"105","AO","Valle d'Aosta"
"105","AP","Ascoli Piceno"
"105","AQ","L'Aquila"
"105","AR","Arezzo"
"105","AT","Asti"
"105","AV","Avellino"
"105","BA","Bari"
"105","BG","Bergamo"
"105","BI","Biella"
"105","BL","Belluno"
"105","BN","Benevento"
"105","BO","Bologna"
"105","BR","Brindisi"
"105","BS","Brescia"
"105","BT","Barletta-Andria-Trani"
"105","BZ","Alto Adige"
"105","CA","Cagliari"
"105","CB","Campobasso"
"105","CE","Caserta"
"105","CH","Chieti"
"105","CI","Carbonia-Iglesias"
"105","CL","Caltanissetta"
"105","CN","Cuneo"
"105","CO","Como"
"105","CR","Cremona"
"105","CS","Cosenza"
"105","CT","Catania"
"105","CZ","Catanzaro"
"105","EN","Enna"
"105","FE","Ferrara"
"105","FG","Foggia"
"105","FI","Firenze"
"105","FM","Fermo"
"105","FO","Forlì-Cesena"
"105","FR","Frosinone"
"105","GE","Genova"
"105","GO","Gorizia"
"105","GR","Grosseto"
"105","IM","Imperia"
"105","IS","Isernia"
"105","KR","Crotone"
"105","LC","Lecco"
"105","LE","Lecce"
"105","LI","Livorno"
"105","LO","Lodi"
"105","LT","Latina"
"105","LU","Lucca"
"105","MC","Macerata"
"105","MD","Medio Campidano"
"105","ME","Messina"
"105","MI","Milano"
"105","MN","Mantova"
"105","MO","Modena"
"105","MS","Massa-Carrara"
"105","MT","Matera"
"105","MZ","Monza e Brianza"
"105","NA","Napoli"
"105","NO","Novara"
"105","NU","Nuoro"
"105","OG","Ogliastra"
"105","OR","Oristano"
"105","OT","Olbia-Tempio"
"105","PA","Palermo"
"105","PC","Piacenza"
"105","PD","Padova"
"105","PE","Pescara"
"105","PG","Perugia"
"105","PI","Pisa"
"105","PN","Pordenone"
"105","PO","Prato"
"105","PR","Parma"
"105","PS","Pesaro e Urbino"
"105","PT","Pistoia"
"105","PV","Pavia"
"105","PZ","Potenza"
"105","RA","Ravenna"
"105","RC","Reggio Calabria"
"105","RE","Reggio Emilia"
"105","RG","Ragusa"
"105","RI","Rieti"
"105","RM","Roma"
"105","RN","Rimini"
"105","RO","Rovigo"
"105","SA","Salerno"
"105","SI","Siena"
"105","SO","Sondrio"
"105","SP","La Spezia"
"105","SR","Siracusa"
"105","SS","Sassari"
"105","SV","Savona"
"105","TA","Taranto"
"105","TE","Teramo"
"105","TN","Trento"
"105","TO","Torino"
"105","TP","Trapani"
"105","TR","Terni"
"105","TS","Trieste"
"105","TV","Treviso"
"105","UD","Udine"
"105","VA","Varese"
"105","VB","Verbano-Cusio-Ossola"
"105","VC","Vercelli"
"105","VE","Venezia"
"105","VI","Vicenza"
"105","VR","Verona"
"105","VT","Viterbo"
"105","VV","Vibo Valentia"
"106","01","Kingston"
"106","02","Half Way Tree"
"106","03","Morant Bay"
"106","04","Port Antonio"
"106","05","Port Maria"
"106","06","Saint Ann's Bay"
"106","07","Falmouth"
"106","08","Montego Bay"
"106","09","Lucea"
"106","10","Savanna-la-Mar"
"106","11","Black River"
"106","12","Mandeville"
"106","13","May Pen"
"106","14","Spanish Town"
"107","01","北海道"
"107","02","青森"
"107","03","岩手"
"107","04","宮城"
"107","05","秋田"
"107","06","山形"
"107","07","福島"
"107","08","茨城"
"107","09","栃木"
"107","10","群馬"
"107","11","埼玉"
"107","12","千葉"
"107","13","東京"
"107","14","神奈川"
"107","15","新潟"
"107","16","富山"
"107","17","石川"
"107","18","福井"
"107","19","山梨"
"107","20","長野"
"107","21","岐阜"
"107","22","静岡"
"107","23","愛知"
"107","24","三重"
"107","25","滋賀"
"107","26","京都"
"107","27","大阪"
"107","28","兵庫"
"107","29","奈良"
"107","30","和歌山"
"107","31","鳥取"
"107","32","島根"
"107","33","岡山"
"107","34","広島"
"107","35","山口"
"107","36","徳島"
"107","37","香川"
"107","38","愛媛"
"107","39","高知"
"107","40","福岡"
"107","41","佐賀"
"107","42","長崎"
"107","43","熊本"
"107","44","大分"
"107","45","宮崎"
"107","46","鹿児島"
"107","47","沖縄"
"108","AJ","محافظة عجلون"
"108","AM","محافظة العاصمة"
"108","AQ","محافظة العقبة"
"108","AT","محافظة الطفيلة"
"108","AZ","محافظة الزرقاء"
"108","BA","محافظة البلقاء"
"108","JA","محافظة جرش"
"108","JR","محافظة إربد"
"108","KA","محافظة الكرك"
"108","MA","محافظة المفرق"
"108","MD","محافظة مادبا"
"108","MN","محافظة معان"
"109","AL","Алматы"
"109","AC","Almaty City"
"109","AM","Ақмола"
"109","AQ","Ақтөбе"
"109","AS","Астана"
"109","AT","Атырау"
"109","BA","Батыс Қазақстан"
"109","BY","Байқоңыр"
"109","MA","Маңғыстау"
"109","ON","Оңтүстік Қазақстан"
"109","PA","Павлодар"
"109","QA","Қарағанды"
"109","QO","Қостанай"
"109","QY","Қызылорда"
"109","SH","Шығыс Қазақстан"
"109","SO","Солтүстік Қазақстан"
"109","ZH","Жамбыл"
"110","110","Nairobi"
"110","200","Central"
"110","300","Mombasa"
"110","400","Eastern"
"110","500","North Eastern"
"110","600","Nyanza"
"110","700","Rift Valley"
"110","900","Western"
"111","G","Gilbert Islands"
"111","L","Line Islands"
"111","P","Phoenix Islands"
"112","CHA","자강도"
"112","HAB","함경 북도"
"112","HAN","함경 남도"
"112","HWB","황해 북도"
"112","HWN","황해 남도"
"112","KAN","강원도"
"112","KAE","개성시"
"112","NAJ","라선 직할시"
"112","NAM","남포 특급시"
"112","PYB","평안 북도"
"112","PYN","평안 남도"
"112","PYO","평양 직할시"
"112","YAN","량강도"
"113","11","서울특별시"
"113","26","부산 광역시"
"113","27","대구 광역시"
"113","28","인천광역시"
"113","29","광주 광역시"
"113","30","대전 광역시"
"113","31","울산 광역시"
"113","41","경기도"
"113","42","강원도"
"113","43","충청 북도"
"113","44","충청 남도"
"113","45","전라 북도"
"113","46","전라 남도"
"113","47","경상 북도"
"113","48","경상 남도"
"113","49","제주특별자치도"
"114","AH","الاحمدي"
"114","FA","الفروانية"
"114","JA","الجهراء"
"114","KU","ألعاصمه"
"114","HW","حولي"
"114","MU","مبارك الكبير"
"115","B","Баткен областы"
"115","C","Чүй областы"
"115","GB","Бишкек"
"115","J","Жалал-Абад областы"
"115","N","Нарын областы"
"115","O","Ош областы"
"115","T","Талас областы"
"115","Y","Ысык-Көл областы"
"116","AT","ອັດຕະປື"
"116","BK","ບໍ່ແກ້ວ"
"116","BL","ບໍລິຄໍາໄຊ"
"116","CH","ຈໍາປາສັກ"
"116","HO","ຫົວພັນ"
"116","KH","ຄໍາມ່ວນ"
"116","LM","ຫລວງນໍ້າທາ"
"116","LP","ຫລວງພະບາງ"
"116","OU","ອຸດົມໄຊ"
"116","PH","ຜົງສາລີ"
"116","SL","ສາລະວັນ"
"116","SV","ສະຫວັນນະເຂດ"
"116","VI","ວຽງຈັນ"
"116","VT","ວຽງຈັນ"
"116","XA","ໄຊຍະບູລີ"
"116","XE","ເຊກອງ"
"116","XI","ຊຽງຂວາງ"
"116","XN","ໄຊສົມບູນ"
"117","AI","Aizkraukles rajons"
"117","AL","Alūksnes rajons"
"117","BL","Balvu rajons"
"117","BU","Bauskas rajons"
"117","CE","Cēsu rajons"
"117","DA","Daugavpils rajons"
"117","DGV","Daugpilis"
"117","DO","Dobeles rajons"
"117","GU","Gulbenes rajons"
"117","JEL","Jelgava"
"117","JK","Jēkabpils rajons"
"117","JL","Jelgavas rajons"
"117","JUR","Jūrmala"
"117","KR","Krāslavas rajons"
"117","KU","Kuldīgas rajons"
"117","LE","Liepājas rajons"
"117","LM","Limbažu rajons"
"117","LPX","Liepoja"
"117","LU","Ludzas rajons"
"117","MA","Madonas rajons"
"117","OG","Ogres rajons"
"117","PR","Preiļu rajons"
"117","RE","Rēzeknes rajons"
"117","REZ","Rēzekne"
"117","RI","Rīgas rajons"
"117","RIX","Rīga"
"117","SA","Saldus rajons"
"117","TA","Talsu rajons"
"117","TU","Tukuma rajons"
"117","VE","Ventspils rajons"
"117","VEN","Ventspils"
"117","VK","Valkas rajons"
"117","VM","Valmieras rajons"
"119","A","Maseru"
"119","B","Butha-Buthe"
"119","C","Leribe"
"119","D","Berea"
"119","E","Mafeteng"
"119","F","Mohale's Hoek"
"119","G","Quthing"
"119","H","Qacha's Nek"
"119","J","Mokhotlong"
"119","K","Thaba-Tseka"
"120","BG","Bong"
"120","BM","Bomi"
"120","CM","Grand Cape Mount"
"120","GB","Grand Bassa"
"120","GG","Grand Gedeh"
"120","GK","Grand Kru"
"120","GP","Gbarpolu"
"120","LO","Lofa"
"120","MG","Margibi"
"120","MO","Montserrado"
"120","MY","Maryland"
"120","NI","Nimba"
"120","RG","River Gee"
"120","RI","Rivercess"
"120","SI","Sinoe"
"121","AJ","Ajdābiyā"
"121","BA","Banghāzī"
"121","BU","Al Buţnān"
"121","BW","Banī Walīd"
"121","DR","Darnah"
"121","GD","Ghadāmis"
"121","GR","Gharyān"
"121","GT","Ghāt"
"121","HZ","Al Ḩizām al Akhḑar"
"121","JA","Al Jabal al Akhḑar"
"121","JB","Jaghbūb"
"121","JI","Al Jifārah"
"121","JU","Al Jufrah"
"121","KF","Al Kufrah"
"121","MB","Al Marqab"
"121","MI","Mişrātah"
"121","MJ","Al Marj"
"121","MQ","Murzuq"
"121","MZ","Mizdah"
"121","NL","Nālūt"
"121","NQ","An Nuqaţ al Khams"
"121","QB","Al Qubbah"
"121","QT","Al Qaţrūn"
"121","SB","Sabhā"
"121","SH","Ash Shāţi"
"121","SR","Surt"
"121","SS","Şabrātah Şurmān"
"121","TB","Ţarābulus"
"121","TM","Tarhūnah-Masallātah"
"121","TN","Tājūrā wa an Nawāḩī al Arbāʻ"
"121","WA","Al Wāḩah"
"121","WD","Wādī al Ḩayāt"
"121","YJ","Yafran-Jādū"
"121","ZA","Az Zāwiyah"
"122","B","Balzers"
"122","E","Eschen"
"122","G","Gamprin"
"122","M","Mauren"
"122","P","Planken"
"122","R","Ruggell"
"122","A","Schaan"
"122","L","Schellenberg"
"122","N","Triesen"
"122","T","Triesenberg"
"122","V","Vaduz"
"123","AL","Alytaus Apskritis"
"123","KL","Klaipėdos Apskritis"
"123","KU","Kauno Apskritis"
"123","MR","Marijampolės Apskritis"
"123","PN","Panevėžio Apskritis"
"123","SA","Šiaulių Apskritis"
"123","TA","Tauragės Apskritis"
"123","TE","Telšių Apskritis"
"123","UT","Utenos Apskritis"
"123","VL","Vilniaus Apskritis"
"124","D","Diekirch"
"124","G","Grevenmacher"
"124","L","Luxemburg"
"125","I","海島市"
"125","M","澳門市"
"126","BR","Berovo"
"126","CH","Чешиново-Облешево"
"126","DL","Делчево"
"126","KB","Карбинци"
"126","OC","Кочани"
"126","LO","Лозово"
"126","MK","Македонска каменица"
"126","PH","Пехчево"
"126","PT","Пробиштип"
"126","ST","Штип"
"126","SL","Свети Николе"
"126","NI","Виница"
"126","ZR","Зрновци"
"126","KY","Кратово"
"126","KZ","Крива Паланка"
"126","UM","Куманово"
"126","LI","Липково"
"126","RN","Ранковце"
"126","NA","Старо Нагоричане"
"126","TL","Битола"
"126","DM","Демир Хисар"
"126","DE","Долнени"
"126","KG","Кривогаштани"
"126","KS","Крушево"
"126","MG","Могила"
"126","NV","Новаци"
"126","PP","Прилеп"
"126","RE","Ресен"
"126","VJ","Боговиње"
"126","BN","Брвеница"
"126","GT","Гостивар"
"126","JG","Јегуновце"
"126","MR","Маврово и Ростуша"
"126","TR","Теарце"
"126","ET","Тетово"
"126","VH","Врапчиште"
"126","ZE","Желино"
"126","AD","Аеродром"
"126","AR","Арачиново"
"126","BU","Бутел"
"126","CI","Чаир"
"126","CE","Центар"
"126","CS","Чучер Сандево"
"126","GB","Гази Баба"
"126","GP","Ѓорче Петров"
"126","IL","Илинден"
"126","KX","Карпош"
"126","VD","Кисела Вода"
"126","PE","Петровец"
"126","AJ","Сарај"
"126","SS","Сопиште"
"126","SU","Студеничани"
"126","SO","Шуто Оризари"
"126","ZK","Зелениково"
"126","BG","Богданци"
"126","BS","Босилово"
"126","GV","Гевгелија"
"126","KN","Конче"
"126","NS","Ново Село"
"126","RV","Радовиш"
"126","SD","Стар Дојран"
"126","RU","Струмица"
"126","VA","Валандово"
"126","VL","Василево"
"126","CZ","Центар Жупа"
"126","DB","Дебар"
"126","DA","Дебарца"
"126","DR","Другово"
"126","KH","Кичево"
"126","MD","Македонски Брод"
"126","OD","Охрид"
"126","OS","Осломеј"
"126","PN","Пласница"
"126","UG","Струга"
"126","VV","Вевчани"
"126","VC","Вранештица"
"126","ZA","Зајас"
"126","CA","Чашка"
"126","DK","Демир Капија"
"126","GR","Градско"
"126","AV","Кавадарци"
"126","NG","Неготино"
"126","RM","Росоман"
"126","VE","Велес"
"127","A","Toamasina"
"127","D","Antsiranana"
"127","F","Fianarantsoa"
"127","M","Mahajanga"
"127","T","Antananarivo"
"127","U","Toliara"
"128","BA","Balaka"
"128","BL","Blantyre"
"128","C","Central"
"128","CK","Chikwawa"
"128","CR","Chiradzulu"
"128","CT","Chitipa"
"128","DE","Dedza"
"128","DO","Dowa"
"128","KR","Karonga"
"128","KS","Kasungu"
"128","LK","Likoma Island"
"128","LI","Lilongwe"
"128","MH","Machinga"
"128","MG","Mangochi"
"128","MC","Mchinji"
"128","MU","Mulanje"
"128","MW","Mwanza"
"128","MZ","Mzimba"
"128","N","Northern"
"128","NB","Nkhata"
"128","NK","Nkhotakota"
"128","NS","Nsanje"
"128","NU","Ntcheu"
"128","NI","Ntchisi"
"128","PH","Phalombe"
"128","RU","Rumphi"
"128","S","Southern"
"128","SA","Salima"
"128","TH","Thyolo"
"128","ZO","Zomba"
"129","01","Johor Darul Takzim"
"129","02","Kedah Darul Aman"
"129","03","Kelantan Darul Naim"
"129","04","Melaka Negeri Bersejarah"
"129","05","Negeri Sembilan Darul Khusus"
"129","06","Pahang Darul Makmur"
"129","07","Pulau Pinang"
"129","08","Perak Darul Ridzuan"
"129","09","Perlis Indera Kayangan"
"129","10","Selangor Darul Ehsan"
"129","11","Terengganu Darul Iman"
"129","12","Sabah Negeri Di Bawah Bayu"
"129","13","Sarawak Bumi Kenyalang"
"129","14","Wilayah Persekutuan Kuala Lumpur"
"129","15","Wilayah Persekutuan Labuan"
"129","16","Wilayah Persekutuan Putrajaya"
"130","THU","Thiladhunmathi Uthuru"
"130","THD","Thiladhunmathi Dhekunu"
"130","MLU","Miladhunmadulu Uthuru"
"130","MLD","Miladhunmadulu Dhekunu"
"130","MAU","Maalhosmadulu Uthuru"
"130","MAD","Maalhosmadulu Dhekunu"
"130","FAA","Faadhippolhu"
"130","MAA","Male Atoll"
"130","AAU","Ari Atoll Uthuru"
"130","AAD","Ari Atoll Dheknu"
"130","FEA","Felidhe Atoll"
"130","MUA","Mulaku Atoll"
"130","NAU","Nilandhe Atoll Uthuru"
"130","NAD","Nilandhe Atoll Dhekunu"
"130","KLH","Kolhumadulu"
"130","HDH","Hadhdhunmathi"
"130","HAU","Huvadhu Atoll Uthuru"
"130","HAD","Huvadhu Atoll Dhekunu"
"130","FMU","Fua Mulaku"
"130","ADD","Addu"
"131","1","Kayes"
"131","2","Koulikoro"
"131","3","Sikasso"
"131","4","Ségou"
"131","5","Mopti"
"131","6","Tombouctou"
"131","7","Gao"
"131","8","Kidal"
"131","BK0","Bamako"
"132","ATT","Attard"
"132","BAL","Balzan"
"132","BGU","Birgu"
"132","BKK","Birkirkara"
"132","BRZ","Birzebbuga"
"132","BOR","Bormla"
"132","DIN","Dingli"
"132","FGU","Fgura"
"132","FLO","Floriana"
"132","GDJ","Gudja"
"132","GZR","Gzira"
"132","GRG","Gargur"
"132","GXQ","Gaxaq"
"132","HMR","Hamrun"
"132","IKL","Iklin"
"132","ISL","Isla"
"132","KLK","Kalkara"
"132","KRK","Kirkop"
"132","LIJ","Lija"
"132","LUQ","Luqa"
"132","MRS","Marsa"
"132","MKL","Marsaskala"
"132","MXL","Marsaxlokk"
"132","MDN","Mdina"
"132","MEL","Melliea"
"132","MGR","Mgarr"
"132","MST","Mosta"
"132","MQA","Mqabba"
"132","MSI","Msida"
"132","MTF","Mtarfa"
"132","NAX","Naxxar"
"132","PAO","Paola"
"132","PEM","Pembroke"
"132","PIE","Pieta"
"132","QOR","Qormi"
"132","QRE","Qrendi"
"132","RAB","Rabat"
"132","SAF","Safi"
"132","SGI","San Giljan"
"132","SLU","Santa Lucija"
"132","SPB","San Pawl il-Bahar"
"132","SGW","San Gwann"
"132","SVE","Santa Venera"
"132","SIG","Siggiewi"
"132","SLM","Sliema"
"132","SWQ","Swieqi"
"132","TXB","Ta Xbiex"
"132","TRX","Tarxien"
"132","VLT","Valletta"
"132","XGJ","Xgajra"
"132","ZBR","Zabbar"
"132","ZBG","Zebbug"
"132","ZJT","Zejtun"
"132","ZRQ","Zurrieq"
"132","FNT","Fontana"
"132","GHJ","Ghajnsielem"
"132","GHR","Gharb"
"132","GHS","Ghasri"
"132","KRC","Kercem"
"132","MUN","Munxar"
"132","NAD","Nadur"
"132","QAL","Qala"
"132","VIC","Victoria"
"132","SLA","San Lawrenz"
"132","SNT","Sannat"
"132","ZAG","Xagra"
"132","XEW","Xewkija"
"132","ZEB","Zebbug"
"133","ALK","Ailuk"
"133","ALL","Ailinglapalap"
"133","ARN","Arno"
"133","AUR","Aur"
"133","EBO","Ebon"
"133","ENI","Eniwetok"
"133","JAB","Jabat"
"133","JAL","Jaluit"
"133","KIL","Kili"
"133","KWA","Kwajalein"
"133","LAE","Lae"
"133","LIB","Lib"
"133","LIK","Likiep"
"133","MAJ","Majuro"
"133","MAL","Maloelap"
"133","MEJ","Mejit"
"133","MIL","Mili"
"133","NMK","Namorik"
"133","NMU","Namu"
"133","RON","Rongelap"
"133","UJA","Ujae"
"133","UJL","Ujelang"
"133","UTI","Utirik"
"133","WTJ","Wotje"
"133","WTN","Wotho"
"135","01","ولاية الحوض الشرقي"
"135","02","ولاية الحوض الغربي"
"135","03","ولاية العصابة"
"135","04","ولاية كركول"
"135","05","ولاية البراكنة"
"135","06","ولاية الترارزة"
"135","07","ولاية آدرار"
"135","08","ولاية داخلت نواذيبو"
"135","09","ولاية تكانت"
"135","10","ولاية كيدي ماغة"
"135","11","ولاية تيرس زمور"
"135","12","ولاية إينشيري"
"135","NKC","نواكشوط"
"136","AG","Agalega Islands"
"136","BL","Black River"
"136","BR","Beau Bassin-Rose Hill"
"136","CC","Cargados Carajos Shoals"
"136","CU","Curepipe"
"136","FL","Flacq"
"136","GP","Grand Port"
"136","MO","Moka"
"136","PA","Pamplemousses"
"136","PL","Port Louis"
"136","PU","Port Louis City"
"136","PW","Plaines Wilhems"
"136","QB","Quatre Bornes"
"136","RO","Rodrigues"
"136","RR","Riviere du Rempart"
"136","SA","Savanne"
"136","VP","Vacoas-Phoenix"
"138","AGU","Aguascalientes"
"138","BCN","Baja California"
"138","BCS","Baja California Sur"
"138","CAM","Campeche"
"138","CHH","Chihuahua"
"138","CHP","Chiapas"
"138","COA","Coahuila"
"138","COL","Colima"
"138","DIF","Distrito Federal"
"138","DUR","Durango"
"138","GRO","Guerrero"
"138","GUA","Guanajuato"
"138","HID","Hidalgo"
"138","JAL","Jalisco"
"138","MEX","Mexico"
"138","MIC","Michoacán"
"138","MOR","Morelos"
"138","NAY","Nayarit"
"138","NLE","Nuevo León"
"138","OAX","Oaxaca"
"138","PUE","Puebla"
"138","QUE","Querétaro"
"138","ROO","Quintana Roo"
"138","SIN","Sinaloa"
"138","SLP","San Luis Potosí"
"138","SON","Sonora"
"138","TAB","Tabasco"
"138","TAM","Tamaulipas"
"138","TLA","Tlaxcala"
"138","VER","Veracruz"
"138","YUC","Yucatan"
"138","ZAC","Zacatecas"
"139","KSA","Kosrae"
"139","PNI","Pohnpei"
"139","TRK","Chuuk"
"139","YAP","Yap"
"140","BA","Bălţi"
"140","CA","Cahul"
"140","CU","Chişinău"
"140","ED","Edineţ"
"140","GA","Găgăuzia"
"140","LA","Lăpuşna"
"140","OR","Orhei"
"140","SN","Stânga Nistrului"
"140","SO","Soroca"
"140","TI","Tighina"
"140","UN","Ungheni"
"141","MC","Monte Carlo"
"141","LR","La Rousse"
"141","LA","Larvotto"
"141","MV","Monaco Ville"
"141","SM","Saint Michel"
"141","CO","Condamine"
"141","LC","La Colle"
"141","RE","Les Révoires"
"141","MO","Moneghetti"
"141","FV","Fontvieille"
"142","1","Улаанбаатар"
"142","035","Орхон аймаг"
"142","037","Дархан-Уул аймаг"
"142","039","Хэнтий аймаг"
"142","041","Хөвсгөл аймаг"
"142","043","Ховд аймаг"
"142","046","Увс аймаг"
"142","047","Төв аймаг"
"142","049","Сэлэнгэ аймаг"
"142","051","Сүхбаатар аймаг"
"142","053","Өмнөговь аймаг"
"142","055","Өвөрхангай аймаг"
"142","057","Завхан аймаг"
"142","059","Дундговь аймаг"
"142","061","Дорнод аймаг"
"142","063","Дорноговь аймаг"
"142","064","Говьсүмбэр аймаг"
"142","065","Говь-Алтай аймаг"
"142","067","Булган аймаг"
"142","069","Баянхонгор аймаг"
"142","071","Баян Өлгий аймаг"
"142","073","Архангай аймаг"
"143","A","Saint Anthony"
"143","G","Saint Georges"
"143","P","Saint Peter"
"145","A","Niassa"
"145","B","Manica"
"145","G","Gaza"
"145","I","Inhambane"
"145","L","Maputo"
"145","MPM","Maputo cidade"
"145","N","Nampula"
"145","P","Cabo Delgado"
"145","Q","Zambézia"
"145","S","Sofala"
"145","T","Tete"
"146","AY","ဧရာ၀တီတိုင္‌း"
"146","BG","ပဲခူးတုိင္‌း"
"146","MG","မကေ္ဝးတိုင္‌း"
"146","MD","မန္တလေးတုိင္‌း"
"146","SG","စစ္‌ကုိင္‌း‌တုိင္‌း"
"146","TN","တနင္သာရိတုိင္‌း"
"146","YG","ရန္‌ကုန္‌တုိင္‌း"
"146","CH","ခ္ယင္‌းပ္ရည္‌နယ္‌"
"146","KC","ကခ္ယင္‌ပ္ရည္‌နယ္‌"
"146","KH","ကယား‌ပ္ရည္‌နယ္‌"
"146","KN","ကရင္‌‌ပ္ရည္‌နယ္‌"
"146","MN","မ္ဝန္‌ပ္ရည္‌နယ္‌"
"146","RK","ရခုိင္‌ပ္ရည္‌နယ္‌"
"146","SH","ရုမ္‌းပ္ရည္‌နယ္‌"
"147","CA","Caprivi"
"147","ER","Erongo"
"147","HA","Hardap"
"147","KA","Karas"
"147","KH","Khomas"
"147","KU","Kunene"
"147","OD","Otjozondjupa"
"147","OH","Omaheke"
"147","OK","Okavango"
"147","ON","Oshana"
"147","OS","Omusati"
"147","OT","Oshikoto"
"147","OW","Ohangwena"
"148","AO","Aiwo"
"148","AA","Anabar"
"148","AT","Anetan"
"148","AI","Anibare"
"148","BA","Baiti"
"148","BO","Boe"
"148","BU","Buada"
"148","DE","Denigomodu"
"148","EW","Ewa"
"148","IJ","Ijuw"
"148","ME","Meneng"
"148","NI","Nibok"
"148","UA","Uaboe"
"148","YA","Yaren"
"149","BA","Bagmati"
"149","BH","Bheri"
"149","DH","Dhawalagiri"
"149","GA","Gandaki"
"149","JA","Janakpur"
"149","KA","Karnali"
"149","KO","Kosi"
"149","LU","Lumbini"
"149","MA","Mahakali"
"149","ME","Mechi"
"149","NA","Narayani"
"149","RA","Rapti"
"149","SA","Sagarmatha"
"149","SE","Seti"
"150","DR","Drenthe"
"150","FL","Flevoland"
"150","FR","Friesland"
"150","GE","Gelderland"
"150","GR","Groningen"
"150","LI","Limburg"
"150","NB","Noord-Brabant"
"150","NH","Noord-Holland"
"150","OV","Overijssel"
"150","UT","Utrecht"
"150","ZE","Zeeland"
"150","ZH","Zuid-Holland"
"152","L","Province des Îles"
"152","N","Province Nord"
"152","S","Province Sud"
"153","AUK","Auckland"
"153","BOP","Bay of Plenty"
"153","CAN","Canterbury"
"153","GIS","Gisborne"
"153","HKB","Hawke's Bay"
"153","MBH","Marlborough"
"153","MWT","Manawatu-Wanganui"
"153","NSN","Nelson"
"153","NTL","Northland"
"153","OTA","Otago"
"153","STL","Southland"
"153","TAS","Tasman"
"153","TKI","Taranaki"
"153","WGN","Wellington"
"153","WKO","Waikato"
"153","WTC","West Coast"
"154","AN","Atlántico Norte"
"154","AS","Atlántico Sur"
"154","BO","Boaco"
"154","CA","Carazo"
"154","CI","Chinandega"
"154","CO","Chontales"
"154","ES","Estelí"
"154","GR","Granada"
"154","JI","Jinotega"
"154","LE","León"
"154","MD","Madriz"
"154","MN","Managua"
"154","MS","Masaya"
"154","MT","Matagalpa"
"154","NS","Nueva Segovia"
"154","RI","Rivas"
"154","SJ","Río San Juan"
"155","1","Agadez"
"155","2","Daffa"
"155","3","Dosso"
"155","4","Maradi"
"155","5","Tahoua"
"155","6","Tillabéry"
"155","7","Zinder"
"155","8","Niamey"
"156","AB","Abia State"
"156","AD","Adamawa State"
"156","AK","Akwa Ibom State"
"156","AN","Anambra State"
"156","BA","Bauchi State"
"156","BE","Benue State"
"156","BO","Borno State"
"156","BY","Bayelsa State"
"156","CR","Cross River State"
"156","DE","Delta State"
"156","EB","Ebonyi State"
"156","ED","Edo State"
"156","EK","Ekiti State"
"156","EN","Enugu State"
"156","GO","Gombe State"
"156","IM","Imo State"
"156","JI","Jigawa State"
"156","KB","Kebbi State"
"156","KD","Kaduna State"
"156","KN","Kano State"
"156","KO","Kogi State"
"156","KT","Katsina State"
"156","KW","Kwara State"
"156","LA","Lagos State"
"156","NA","Nassarawa State"
"156","NI","Niger State"
"156","OG","Ogun State"
"156","ON","Ondo State"
"156","OS","Osun State"
"156","OY","Oyo State"
"156","PL","Plateau State"
"156","RI","Rivers State"
"156","SO","Sokoto State"
"156","TA","Taraba State"
"156","ZA","Zamfara State"
"159","N","Northern Islands"
"159","R","Rota"
"159","S","Saipan"
"159","T","Tinian"
"160","01","Østfold fylke"
"160","02","Akershus fylke"
"160","03","Oslo fylke"
"160","04","Hedmark fylke"
"160","05","Oppland fylke"
"160","06","Buskerud fylke"
"160","07","Vestfold fylke"
"160","08","Telemark fylke"
"160","09","Aust-Agder fylke"
"160","10","Vest-Agder fylke"
"160","11","Rogaland fylke"
"160","12","Hordaland fylke"
"160","14","Sogn og Fjordane fylke"
"160","15","Møre og Romsdal fylke"
"160","16","Sør-Trøndelag fylke"
"160","17","Nord-Trøndelag fylke"
"160","18","Nordland fylke"
"160","19","Troms fylke"
"160","20","Finnmark fylke"
"161","BA","الباطنة"
"161","DA","الداخلية"
"161","DH","ظفار"
"161","MA","مسقط"
"161","MU","مسندم"
"161","SH","الشرقية"
"161","WU","الوسطى"
"161","ZA","الظاهرة"
"162","BA","بلوچستان"
"162","IS","وفاقی دارالحکومت"
"162","JK","آزاد کشمیر"
"162","NA","شمالی علاقہ جات"
"162","NW","شمال مغربی سرحدی صوبہ"
"162","PB","پنجاب"
"162","SD","سندھ"
"162","TA","وفاقی قبائلی علاقہ جات"
"163","AM","Aimeliik"
"163","AR","Airai"
"163","AN","Angaur"
"163","HA","Hatohobei"
"163","KA","Kayangel"
"163","KO","Koror"
"163","ME","Melekeok"
"163","NA","Ngaraard"
"163","NG","Ngarchelong"
"163","ND","Ngardmau"
"163","NT","Ngatpang"
"163","NC","Ngchesar"
"163","NR","Ngeremlengui"
"163","NW","Ngiwal"
"163","PE","Peleliu"
"163","SO","Sonsorol"
"164","1","Bocas del Toro"
"164","2","Coclé"
"164","3","Colón"
"164","4","Chiriquí"
"164","5","Darién"
"164","6","Herrera"
"164","7","Los Santos"
"164","8","Panamá"
"164","9","Veraguas"
"164","Q","Kuna Yala"
"165","CPK","Chimbu"
"165","CPM","Central"
"165","EBR","East New Britain"
"165","EHG","Eastern Highlands"
"165","EPW","Enga"
"165","ESW","East Sepik"
"165","GPK","Gulf"
"165","MBA","Milne Bay"
"165","MPL","Morobe"
"165","MPM","Madang"
"165","MRL","Manus"
"165","NCD","National Capital District"
"165","NIK","New Ireland"
"165","NPP","Northern"
"165","NSA","North Solomons"
"165","SAN","Sandaun"
"165","SHM","Southern Highlands"
"165","WBK","West New Britain"
"165","WHM","Western Highlands"
"165","WPD","Western"
"166","1","Concepción"
"166","2","San Pedro"
"166","3","Cordillera"
"166","4","Guairá"
"166","5","Caaguazú"
"166","6","Caazapá"
"166","7","Itapúa"
"166","8","Misiones"
"166","9","Paraguarí"
"166","10","Alto Paraná"
"166","11","Central"
"166","12","Ñeembucú"
"166","13","Amambay"
"166","14","Canindeyú"
"166","15","Presidente Hayes"
"166","16","Alto Paraguay"
"166","19","Boquerón"
"166","ASU","Asunción"
"167","AMA","Amazonas"
"167","ANC","Ancash"
"167","APU","Apurímac"
"167","ARE","Arequipa"
"167","AYA","Ayacucho"
"167","CAJ","Cajamarca"
"167","CAL","Callao"
"167","CUS","Cuzco"
"167","HUC","Huánuco"
"167","HUV","Huancavelica"
"167","ICA","Ica"
"167","JUN","Junín"
"167","LAL","La Libertad"
"167","LAM","Lambayeque"
"167","LIM","Lima"
"167","LOR","Loreto"
"167","MDD","Madre de Dios"
"167","MOQ","Moquegua"
"167","PAS","Pasco"
"167","PIU","Piura"
"167","PUN","Puno"
"167","SAM","San Martín"
"167","TAC","Tacna"
"167","TUM","Tumbes"
"167","UCA","Ucayali"
"168","ABR","Abra"
"168","AGN","Agusan del Norte"
"168","AGS","Agusan del Sur"
"168","AKL","Aklan"
"168","ALB","Albay"
"168","ANT","Antique"
"168","APA","Apayao"
"168","AUR","Aurora"
"168","BAN","Bataan"
"168","BAS","Basilan"
"168","BEN","Benguet"
"168","BIL","Biliran"
"168","BOH","Bohol"
"168","BTG","Batangas"
"168","BTN","Batanes"
"168","BUK","Bukidnon"
"168","BUL","Bulacan"
"168","CAG","Cagayan"
"168","CAM","Camiguin"
"168","CAN","Camarines Norte"
"168","CAP","Capiz"
"168","CAS","Camarines Sur"
"168","CAT","Catanduanes"
"168","CAV","Cavite"
"168","CEB","Cebu"
"168","COM","Compostela Valley"
"168","DAO","Davao Oriental"
"168","DAS","Davao del Sur"
"168","DAV","Davao del Norte"
"168","EAS","Eastern Samar"
"168","GUI","Guimaras"
"168","IFU","Ifugao"
"168","ILI","Iloilo"
"168","ILN","Ilocos Norte"
"168","ILS","Ilocos Sur"
"168","ISA","Isabela"
"168","KAL","Kalinga"
"168","LAG","Laguna"
"168","LAN","Lanao del Norte"
"168","LAS","Lanao del Sur"
"168","LEY","Leyte"
"168","LUN","La Union"
"168","MAD","Marinduque"
"168","MAG","Maguindanao"
"168","MAS","Masbate"
"168","MDC","Mindoro Occidental"
"168","MDR","Mindoro Oriental"
"168","MOU","Mountain Province"
"168","MSC","Misamis Occidental"
"168","MSR","Misamis Oriental"
"168","NCO","Cotabato"
"168","NSA","Northern Samar"
"168","NEC","Negros Occidental"
"168","NER","Negros Oriental"
"168","NUE","Nueva Ecija"
"168","NUV","Nueva Vizcaya"
"168","PAM","Pampanga"
"168","PAN","Pangasinan"
"168","PLW","Palawan"
"168","QUE","Quezon"
"168","QUI","Quirino"
"168","RIZ","Rizal"
"168","ROM","Romblon"
"168","SAR","Sarangani"
"168","SCO","South Cotabato"
"168","SIG","Siquijor"
"168","SLE","Southern Leyte"
"168","SLU","Sulu"
"168","SOR","Sorsogon"
"168","SUK","Sultan Kudarat"
"168","SUN","Surigao del Norte"
"168","SUR","Surigao del Sur"
"168","TAR","Tarlac"
"168","TAW","Tawi-Tawi"
"168","WSA","Samar"
"168","ZAN","Zamboanga del Norte"
"168","ZAS","Zamboanga del Sur"
"168","ZMB","Zambales"
"168","ZSI","Zamboanga Sibugay"
"170","DS","Dolnośląskie"
"170","KP","Kujawsko-Pomorskie"
"170","LU","Lubelskie"
"170","LB","Lubuskie"
"170","LD","Łódzkie"
"170","MA","Małopolskie"
"170","MZ","Mazowieckie"
"170","OP","Opolskie"
"170","PK","Podkarpackie"
"170","PD","Podlaskie"
"170","PM","Pomorskie"
"170","SL","Śląskie"
"170","SK","Świętokrzyskie"
"170","WN","Warmińsko-Mazurskie"
"170","WP","Wielkopolskie"
"170","ZP","Zachodniopomorskie"
"171","01","Aveiro"
"171","02","Beja"
"171","03","Braga"
"171","04","Bragança"
"171","05","Castelo Branco"
"171","06","Coimbra"
"171","07","Évora"
"171","08","Faro"
"171","09","Guarda"
"171","10","Leiria"
"171","11","Lisboa"
"171","12","Portalegre"
"171","13","Porto"
"171","14","Santarém"
"171","15","Setúbal"
"171","16","Viana do Castelo"
"171","17","Vila Real"
"171","18","Viseu"
"171","20","Região Autónoma dos Açores"
"171","30","Região Autónoma da Madeira"
"173","DA","الدوحة"
"173","GH","الغويرية"
"173","JB","جريان الباطنة"
"173","JU","الجميلية"
"173","KH","الخور"
"173","ME","مسيعيد"
"173","MS","الشمال"
"173","RA","الريان"
"173","US","أم صلال"
"173","WA","الوكرة"
"175","AB","Alba"
"175","AG","Argeş"
"175","AR","Arad"
"175","B","Bucureşti"
"175","BC","Bacău"
"175","BH","Bihor"
"175","BN","Bistriţa-Năsăud"
"175","BR","Brăila"
"175","BT","Botoşani"
"175","BV","Braşov"
"175","BZ","Buzău"
"175","CJ","Cluj"
"175","CL","Călăraşi"
"175","CS","Caraş-Severin"
"175","CT","Constanţa"
"175","CV","Covasna"
"175","DB","Dâmboviţa"
"175","DJ","Dolj"
"175","GJ","Gorj"
"175","GL","Galaţi"
"175","GR","Giurgiu"
"175","HD","Hunedoara"
"175","HG","Harghita"
"175","IF","Ilfov"
"175","IL","Ialomiţa"
"175","IS","Iaşi"
"175","MH","Mehedinţi"
"175","MM","Maramureş"
"175","MS","Mureş"
"175","NT","Neamţ"
"175","OT","Olt"
"175","PH","Prahova"
"175","SB","Sibiu"
"175","SJ","Sălaj"
"175","SM","Satu Mare"
"175","SV","Suceava"
"175","TL","Tulcea"
"175","TM","Timiş"
"175","TR","Teleorman"
"175","VL","Vâlcea"
"175","VN","Vrancea"
"175","VS","Vaslui"
"176","AD","Адыге́я Респу́блика"
"176","AGB","Аги́нский-Буря́тский автоно́мный о́круг"
"176","AL","Алта́й Респу́блика"
"176","ALT","Алта́йский край"
"176","AMU","Аму́рская о́бласть"
"176","ARK","Арха́нгельская о́бласть"
"176","AST","Астраха́нская о́бласть"
"176","BA","Башкортоста́н Респу́блика"
"176","BEL","Белгоро́дская о́бласть"
"176","BRY","Бря́нская о́бласть"
"176","BU","Буря́тия Респу́блика"
"176","CE","Чече́нская Респу́блика"
"176","CHE","Челя́бинская о́бласть"
"176","CHI","Чити́нская о́бласть"
"176","CHU","Чуко́тский автоно́мный о́круг"
"176","CU","Чува́шская Респу́блика"
"176","DA","Дагеста́н Респу́блика"
"176","EVE","Эвенки́йский автоно́мный о́круг"
"176","IN","Ингуше́тия Респу́блика"
"176","IRK","Ирку́тская о́бласть"
"176","IVA","Ива́новская о́бласть"
"176","KAM","Камча́тская о́бласть"
"176","KB","Кабарди́но-Балка́рская Респу́блика"
"176","KC","Карача́ево-Черке́сская Респу́блика"
"176","KDA","Краснода́рский край"
"176","KEM","Ке́меровская о́бласть"
"176","KGD","Калинингра́дская о́бласть"
"176","KGN","Курга́нская о́бласть"
"176","KHA","Хаба́ровский край"
"176","KHM","Ха́нты-Манси́йский автоно́мный о́круг—Югра́"
"176","KIA","Красноя́рский край"
"176","KIR","Ки́ровская о́бласть"
"176","KK","Хака́сия"
"176","KL","Калмы́кия Респу́блика"
"176","KLU","Калу́жская о́бласть"
"176","KO","Ко́ми Респу́блика"
"176","KOR","Коря́кский автоно́мный о́круг"
"176","KOS","Костромска́я о́бласть"
"176","KR","Каре́лия Респу́блика"
"176","KRS","Ку́рская о́бласть"
"176","LEN","Ленингра́дская о́бласть"
"176","LIP","Ли́пецкая о́бласть"
"176","MAG","Магада́нская о́бласть"
"176","ME","Мари́й Эл Респу́блика"
"176","MO","Мордо́вия Респу́блика"
"176","MOS","Моско́вская о́бласть"
"176","MOW","Москва́"
"176","MUR","Му́рманская о́бласть"
"176","NEN","Нене́цкий автоно́мный о́круг"
"176","NGR","Новгоро́дская о́бласть"
"176","NIZ","Нижегоро́дская о́бласть"
"176","NVS","Новосиби́рская о́бласть"
"176","OMS","О́мская о́бласть"
"176","ORE","Оренбу́ргская о́бласть"
"176","ORL","Орло́вская о́бласть"
"176","PNZ","Пе́нзенская о́бласть"
"176","PRI","Примо́рский край"
"176","PSK","Пско́вская о́бласть"
"176","ROS","Росто́вская о́бласть"
"176","RYA","Ряза́нская о́бласть"
"176","SA","Саха́ (Яку́тия) Респу́блика"
"176","SAK","Сахали́нская о́бласть"
"176","SAM","Сама́рская о́бласть"
"176","SAR","Сара́товская о́бласть"
"176","SE","Се́верная Осе́тия–Ала́ния Респу́блика"
"176","SMO","Смол́енская о́бласть"
"176","SPE","Санкт-Петербу́рг"
"176","STA","Ставропо́льский край"
"176","SVE","Свердло́вская о́бласть"
"176","TA","Респу́блика Татарста́н"
"176","TAM","Тамбо́вская о́бласть"
"176","TAY","Таймы́рский автоно́мный о́круг"
"176","TOM","То́мская о́бласть"
"176","TUL","Ту́льская о́бласть"
"176","TVE","Тверска́я о́бласть"
"176","TY","Тыва́ Респу́блика"
"176","TYU","Тюме́нская о́бласть"
"176","UD","Удму́ртская Респу́блика"
"176","ULY","Улья́новская о́бласть"
"176","UOB","Усть-Орды́нский Буря́тский автоно́мный о́круг"
"176","VGG","Волгогра́дская о́бласть"
"176","VLA","Влади́мирская о́бласть"
"176","VLG","Волого́дская о́бласть"
"176","VOR","Воро́нежская о́бласть"
"176","XXX","Пе́рмский край"
"176","YAN","Яма́ло-Нене́цкий автоно́мный о́круг"
"176","YAR","Яросла́вская о́бласть"
"176","YEV","Евре́йская автоно́мная о́бласть"
"177","N","Nord"
"177","E","Est"
"177","S","Sud"
"177","O","Ouest"
"177","K","Kigali"
"178","K","Saint Kitts"
"178","N","Nevis"
"179","AR","Anse-la-Raye"
"179","CA","Castries"
"179","CH","Choiseul"
"179","DA","Dauphin"
"179","DE","Dennery"
"179","GI","Gros-Islet"
"179","LA","Laborie"
"179","MI","Micoud"
"179","PR","Praslin"
"179","SO","Soufriere"
"179","VF","Vieux-Fort"
"180","C","Charlotte"
"180","R","Grenadines"
"180","A","Saint Andrew"
"180","D","Saint David"
"180","G","Saint George"
"180","P","Saint Patrick"
"181","AA","A'ana"
"181","AL","Aiga-i-le-Tai"
"181","AT","Atua"
"181","FA","Fa'asaleleaga"
"181","GE","Gaga'emauga"
"181","GI","Gaga'ifomauga"
"181","PA","Palauli"
"181","SA","Satupa'itea"
"181","TU","Tuamasaga"
"181","VF","Va'a-o-Fonoti"
"181","VS","Vaisigano"
"182","AC","Acquaviva"
"182","BM","Borgo Maggiore"
"182","CH","Chiesanuova"
"182","DO","Domagnano"
"182","FA","Faetano"
"182","FI","Fiorentino"
"182","MO","Montegiardino"
"182","SM","Citta di San Marino"
"182","SE","Serravalle"
"183","P","Príncipe"
"183","S","São Tomé"
"184","01","الرياض"
"184","02","مكة المكرمة"
"184","03","المدينه"
"184","04","الشرقية"
"184","05","القصيم"
"184","06","حائل"
"184","07","تبوك"
"184","08","الحدود الشمالية"
"184","09","جيزان"
"184","10","نجران"
"184","11","الباحة"
"184","12","الجوف"
"184","14","عسير"
"185","DA","Dakar"
"185","DI","Diourbel"
"185","FA","Fatick"
"185","KA","Kaolack"
"185","KO","Kolda"
"185","LO","Louga"
"185","MA","Matam"
"185","SL","Saint-Louis"
"185","TA","Tambacounda"
"185","TH","Thies "
"185","ZI","Ziguinchor"
"186","AP","Anse aux Pins"
"186","AB","Anse Boileau"
"186","AE","Anse Etoile"
"186","AL","Anse Louis"
"186","AR","Anse Royale"
"186","BL","Baie Lazare"
"186","BS","Baie Sainte Anne"
"186","BV","Beau Vallon"
"186","BA","Bel Air"
"186","BO","Bel Ombre"
"186","CA","Cascade"
"186","GL","Glacis"
"186","GM","Grand' Anse (on Mahe)"
"186","GP","Grand' Anse (on Praslin)"
"186","DG","La Digue"
"186","RA","La Riviere Anglaise"
"186","MB","Mont Buxton"
"186","MF","Mont Fleuri"
"186","PL","Plaisance"
"186","PR","Pointe La Rue"
"186","PG","Port Glaud"
"186","SL","Saint Louis"
"186","TA","Takamaka"
"187","E","Eastern"
"187","N","Northern"
"187","S","Southern"
"187","W","Western"
"189","BC","Banskobystrický kraj"
"189","BL","Bratislavský kraj"
"189","KI","Košický kraj"
"189","NJ","Nitrianský kraj"
"189","PV","Prešovský kraj"
"189","TA","Trnavský kraj"
"189","TC","Trenčianský kraj"
"189","ZI","Žilinský kraj"
"190","001","Ajdovščina"
"190","002","Beltinci"
"190","003","Bled"
"190","004","Bohinj"
"190","005","Borovnica"
"190","006","Bovec"
"190","007","Brda"
"190","008","Brezovica"
"190","009","Brežice"
"190","010","Tišina"
"190","011","Celje"
"190","012","Cerklje na Gorenjskem"
"190","013","Cerknica"
"190","014","Cerkno"
"190","015","Črenšovci"
"190","016","Črna na Koroškem"
"190","017","Črnomelj"
"190","018","Destrnik"
"190","019","Divača"
"190","020","Dobrepolje"
"190","021","Dobrova-Polhov Gradec"
"190","022","Dol pri Ljubljani"
"190","023","Domžale"
"190","024","Dornava"
"190","025","Dravograd"
"190","026","Duplek"
"190","027","Gorenja vas-Poljane"
"190","028","Gorišnica"
"190","029","Gornja Radgona"
"190","030","Gornji Grad"
"190","031","Gornji Petrovci"
"190","032","Grosuplje"
"190","033","Šalovci"
"190","034","Hrastnik"
"190","035","Hrpelje-Kozina"
"190","036","Idrija"
"190","037","Ig"
"190","038","Ilirska Bistrica"
"190","039","Ivančna Gorica"
"190","040","Izola"
"190","041","Jesenice"
"190","042","Juršinci"
"190","043","Kamnik"
"190","044","Kanal ob Soči"
"190","045","Kidričevo"
"190","046","Kobarid"
"190","047","Kobilje"
"190","048","Kočevje"
"190","049","Komen"
"190","050","Koper"
"190","051","Kozje"
"190","052","Kranj"
"190","053","Kranjska Gora"
"190","054","Krško"
"190","055","Kungota"
"190","056","Kuzma"
"190","057","Laško"
"190","058","Lenart"
"190","059","Lendava"
"190","060","Litija"
"190","061","Ljubljana"
"190","062","Ljubno"
"190","063","Ljutomer"
"190","064","Logatec"
"190","065","Loška Dolina"
"190","066","Loški Potok"
"190","067","Luče"
"190","068","Lukovica"
"190","069","Majšperk"
"190","070","Maribor"
"190","071","Medvode"
"190","072","Mengeš"
"190","073","Metlika"
"190","074","Mežica"
"190","075","Miren-Kostanjevica"
"190","076","Mislinja"
"190","077","Moravče"
"190","078","Moravske Toplice"
"190","079","Mozirje"
"190","080","Murska Sobota"
"190","081","Muta"
"190","082","Naklo"
"190","083","Nazarje"
"190","084","Nova Gorica"
"190","085","Novo mesto"
"190","086","Odranci"
"190","087","Ormož"
"190","088","Osilnica"
"190","089","Pesnica"
"190","090","Piran"
"190","091","Pivka"
"190","092","Podčetrtek"
"190","093","Podvelka"
"190","094","Postojna"
"190","095","Preddvor"
"190","096","Ptuj"
"190","097","Puconci"
"190","098","Rače-Fram"
"190","099","Radeče"
"190","100","Radenci"
"190","101","Radlje ob Dravi"
"190","102","Radovljica"
"190","103","Ravne na Koroškem"
"190","104","Ribnica"
"190","106","Rogaška Slatina"
"190","105","Rogašovci"
"190","107","Rogatec"
"190","108","Ruše"
"190","109","Semič"
"190","110","Sevnica"
"190","111","Sežana"
"190","112","Slovenj Gradec"
"190","113","Slovenska Bistrica"
"190","114","Slovenske Konjice"
"190","115","Starše"
"190","116","Sveti Jurij"
"190","117","Šenčur"
"190","118","Šentilj"
"190","119","Šentjernej"
"190","120","Šentjur pri Celju"
"190","121","Škocjan"
"190","122","Škofja Loka"
"190","123","Škofljica"
"190","124","Šmarje pri Jelšah"
"190","125","Šmartno ob Paki"
"190","126","Šoštanj"
"190","127","Štore"
"190","128","Tolmin"
"190","129","Trbovlje"
"190","130","Trebnje"
"190","131","Tržič"
"190","132","Turnišče"
"190","133","Velenje"
"190","134","Velike Lašče"
"190","135","Videm"
"190","136","Vipava"
"190","137","Vitanje"
"190","138","Vodice"
"190","139","Vojnik"
"190","140","Vrhnika"
"190","141","Vuzenica"
"190","142","Zagorje ob Savi"
"190","143","Zavrč"
"190","144","Zreče"
"190","146","Železniki"
"190","147","Žiri"
"190","148","Benedikt"
"190","149","Bistrica ob Sotli"
"190","150","Bloke"
"190","151","Braslovče"
"190","152","Cankova"
"190","153","Cerkvenjak"
"190","154","Dobje"
"190","155","Dobrna"
"190","156","Dobrovnik"
"190","157","Dolenjske Toplice"
"190","158","Grad"
"190","159","Hajdina"
"190","160","Hoče-Slivnica"
"190","161","Hodoš"
"190","162","Horjul"
"190","163","Jezersko"
"190","164","Komenda"
"190","165","Kostel"
"190","166","Križevci"
"190","167","Lovrenc na Pohorju"
"190","168","Markovci"
"190","169","Miklavž na Dravskem polju"
"190","170","Mirna Peč"
"190","171","Oplotnica"
"190","172","Podlehnik"
"190","173","Polzela"
"190","174","Prebold"
"190","175","Prevalje"
"190","176","Razkrižje"
"190","177","Ribnica na Pohorju"
"190","178","Selnica ob Dravi"
"190","179","Sodražica"
"190","180","Solčava"
"190","181","Sveta Ana"
"190","182","Sveti Andraž v Slovenskih goricah"
"190","183","Šempeter-Vrtojba"
"190","184","Tabor"
"190","185","Trnovska vas"
"190","186","Trzin"
"190","187","Velika Polana"
"190","188","Veržej"
"190","189","Vransko"
"190","190","Žalec"
"190","191","Žetale"
"190","192","Žirovnica"
"190","193","Žužemberk"
"190","194","Šmartno pri Litiji"
"191","CE","Central"
"191","CH","Choiseul"
"191","GC","Guadalcanal"
"191","HO","Honiara"
"191","IS","Isabel"
"191","MK","Makira"
"191","ML","Malaita"
"191","RB","Rennell and Bellona"
"191","TM","Temotu"
"191","WE","Western"
"192","AD","Awdal"
"192","BK","Bakool"
"192","BN","Banaadir"
"192","BR","Bari"
"192","BY","Bay"
"192","GD","Gedo"
"192","GG","Galguduud"
"192","HR","Hiiraan"
"192","JD","Jubbada Dhexe"
"192","JH","Jubbada Hoose"
"192","MD","Mudug"
"192","NG","Nugaal"
"192","SD","Shabeellaha Dhexe"
"192","SG","Sanaag"
"192","SH","Shabeellaha Hoose"
"192","SL","Sool"
"192","TG","Togdheer"
"192","WG","Woqooyi Galbeed"
"193","EC","Eastern Cape"
"193","FS","Free State"
"193","GT","Gauteng"
"193","LP","Limpopo"
"193","MP","Mpumalanga"
"193","NC","Northern Cape"
"193","NL","KwaZulu-Natal"
"193","NW","North-West"
"193","WC","Western Cape"
"196","AN","Andalucía"
"196","AR","Aragón"
"196","A","Alicante"
"196","AB","Albacete"
"196","AL","Almería"
"196","AN","Andalucía"
"196","AV","Ávila"
"196","B","Barcelona"
"196","BA","Badajoz"
"196","BI","Vizcaya"
"196","BU","Burgos"
"196","C","A Coruña"
"196","CA","Cádiz"
"196","CC","Cáceres"
"196","CE","Ceuta"
"196","CL","Castilla y León"
"196","CM","Castilla-La Mancha"
"196","CN","Islas Canarias"
"196","CO","Córdoba"
"196","CR","Ciudad Real"
"196","CS","Castellón"
"196","CT","Catalonia"
"196","CU","Cuenca"
"196","EX","Extremadura"
"196","GA","Galicia"
"196","GC","Las Palmas"
"196","GI","Girona"
"196","GR","Granada"
"196","GU","Guadalajara"
"196","H","Huelva"
"196","HU","Huesca"
"196","IB","Islas Baleares"
"196","J","Jaén"
"196","L","Lleida"
"196","LE","León"
"196","LO","La Rioja"
"196","LU","Lugo"
"196","M","Madrid"
"196","MA","Málaga"
"196","ML","Melilla"
"196","MU","Murcia"
"196","NA","Navarre"
"196","O","Asturias"
"196","OR","Ourense"
"196","P","Palencia"
"196","PM","Baleares"
"196","PO","Pontevedra"
"196","PV","Basque Euskadi"
"196","S","Cantabria"
"196","SA","Salamanca"
"196","SE","Seville"
"196","SG","Segovia"
"196","SO","Soria"
"196","SS","Guipúzcoa"
"196","T","Tarragona"
"196","TE","Teruel"
"196","TF","Santa Cruz De Tenerife"
"196","TO","Toledo"
"196","V","Valencia"
"196","VA","Valladolid"
"196","VI","Álava"
"196","Z","Zaragoza"
"196","ZA","Zamora"
"197","CE","Central"
"197","NC","North Central"
"197","NO","North"
"197","EA","Eastern"
"197","NW","North Western"
"197","SO","Southern"
"197","UV","Uva"
"197","SA","Sabaragamuwa"
"197","WE","Western"
"200","ANL","أعالي النيل"
"200","BAM","البحر الأحمر"
"200","BRT","البحيرات"
"200","JZR","ولاية الجزيرة"
"200","KRT","الخرطوم"
"200","QDR","القضارف"
"200","WDH","الوحدة"
"200","ANB","النيل الأبيض"
"200","ANZ","النيل الأزرق"
"200","ASH","الشمالية"
"200","BJA","الاستوائية الوسطى"
"200","GIS","غرب الاستوائية"
"200","GBG","غرب بحر الغزال"
"200","GDA","غرب دارفور"
"200","GKU","غرب كردفان"
"200","JDA","جنوب دارفور"
"200","JKU","جنوب كردفان"
"200","JQL","جونقلي"
"200","KSL","كسلا"
"200","NNL","ولاية نهر النيل"
"200","SBG","شمال بحر الغزال"
"200","SDA","شمال دارفور"
"200","SKU","شمال كردفان"
"200","SIS","شرق الاستوائية"
"200","SNR","سنار"
"200","WRB","واراب"
"201","BR","Brokopondo"
"201","CM","Commewijne"
"201","CR","Coronie"
"201","MA","Marowijne"
"201","NI","Nickerie"
"201","PM","Paramaribo"
"201","PR","Para"
"201","SA","Saramacca"
"201","SI","Sipaliwini"
"201","WA","Wanica"
"203","HH","Hhohho"
"203","LU","Lubombo"
"203","MA","Manzini"
"203","SH","Shiselweni"
"204","AB","Stockholms län"
"204","C","Uppsala län"
"204","D","Södermanlands län"
"204","E","Östergötlands län"
"204","F","Jönköpings län"
"204","G","Kronobergs län"
"204","H","Kalmar län"
"204","I","Gotlands län"
"204","K","Blekinge län"
"204","M","Skåne län"
"204","N","Hallands län"
"204","O","Västra Götalands län"
"204","S","Värmlands län"
"204","T","Örebro län"
"204","U","Västmanlands län"
"204","W","Dalarnas län"
"204","X","Gävleborgs län"
"204","Y","Västernorrlands län"
"204","Z","Jämtlands län"
"204","AC","Västerbottens län"
"204","BD","Norrbottens län"
"205","ZH","Zürich"
"205","BE","Bern"
"205","LU","Luzern"
"205","UR","Uri"
"205","SZ","Schwyz"
"205","OW","Obwalden"
"205","NW","Nidwalden"
"205","GL","Glasrus"
"205","ZG","Zug"
"205","FR","Fribourg"
"205","SO","Solothurn"
"205","BS","Basel-Stadt"
"205","BL","Basel-Landschaft"
"205","SH","Schaffhausen"
"205","AR","Appenzell Ausserrhoden"
"205","AI","Appenzell Innerrhoden"
"205","SG","Saint Gallen"
"205","GR","Graubünden"
"205","AG","Aargau"
"205","TG","Thurgau"
"205","TI","Ticino"
"205","VD","Vaud"
"205","VS","Valais"
"205","NE","Nuechâtel"
"205","GE","Genève"
"205","JU","Jura"
"206","DI","دمشق"
"206","DR","درعا"
"206","DZ","دير الزور"
"206","HA","الحسكة"
"206","HI","حمص"
"206","HL","حلب"
"206","HM","حماه"
"206","ID","ادلب"
"206","LA","اللاذقية"
"206","QU","القنيطرة"
"206","RA","الرقة"
"206","RD","ریف دمشق"
"206","SU","السويداء"
"206","TA","طرطوس"
"207","CHA","彰化縣"
"207","CYI","嘉義市"
"207","CYQ","嘉義縣"
"207","HSQ","新竹縣"
"207","HSZ","新竹市"
"207","HUA","花蓮縣"
"207","ILA","宜蘭縣"
"207","KEE","基隆市"
"207","KHH","高雄市"
"207","KHQ","高雄縣"
"207","MIA","苗栗縣"
"207","NAN","南投縣"
"207","PEN","澎湖縣"
"207","PIF","屏東縣"
"207","TAO","桃源县"
"207","TNN","台南市"
"207","TNQ","台南縣"
"207","TPE","臺北市"
"207","TPQ","臺北縣"
"207","TTT","台東縣"
"207","TXG","台中市"
"207","TXQ","台中縣"
"207","YUN","雲林縣"
"208","GB","کوهستان بدخشان"
"208","KT","ختلان"
"208","SU","سغد"
"209","01","Arusha"
"209","02","Dar es Salaam"
"209","03","Dodoma"
"209","04","Iringa"
"209","05","Kagera"
"209","06","Pemba Sever"
"209","07","Zanzibar Sever"
"209","08","Kigoma"
"209","09","Kilimanjaro"
"209","10","Pemba Jih"
"209","11","Zanzibar Jih"
"209","12","Lindi"
"209","13","Mara"
"209","14","Mbeya"
"209","15","Zanzibar Západ"
"209","16","Morogoro"
"209","17","Mtwara"
"209","18","Mwanza"
"209","19","Pwani"
"209","20","Rukwa"
"209","21","Ruvuma"
"209","22","Shinyanga"
"209","23","Singida"
"209","24","Tabora"
"209","25","Tanga"
"209","26","Manyara"
"210","TH-10","กรุงเทพมหานคร"
"210","TH-11","สมุทรปราการ"
"210","TH-12","นนทบุรี"
"210","TH-13","ปทุมธานี"
"210","TH-14","พระนครศรีอยุธยา"
"210","TH-15","อ่างทอง"
"210","TH-16","ลพบุรี"
"210","TH-17","สิงห์บุรี"
"210","TH-18","ชัยนาท"
"210","TH-19","สระบุรี"
"210","TH-20","ชลบุรี"
"210","TH-21","ระยอง"
"210","TH-22","จันทบุรี"
"210","TH-23","ตราด"
"210","TH-24","ฉะเชิงเทรา"
"210","TH-25","ปราจีนบุรี"
"210","TH-26","นครนายก"
"210","TH-27","สระแก้ว"
"210","TH-30","นครราชสีมา"
"210","TH-31","บุรีรัมย์"
"210","TH-32","สุรินทร์"
"210","TH-33","ศรีสะเกษ"
"210","TH-34","อุบลราชธานี"
"210","TH-35","ยโสธร"
"210","TH-36","ชัยภูมิ"
"210","TH-37","อำนาจเจริญ"
"210","TH-39","หนองบัวลำภู"
"210","TH-40","ขอนแก่น"
"210","TH-41","อุดรธานี"
"210","TH-42","เลย"
"210","TH-43","หนองคาย"
"210","TH-44","มหาสารคาม"
"210","TH-45","ร้อยเอ็ด"
"210","TH-46","กาฬสินธุ์"
"210","TH-47","สกลนคร"
"210","TH-48","นครพนม"
"210","TH-49","มุกดาหาร"
"210","TH-50","เชียงใหม่"
"210","TH-51","ลำพูน"
"210","TH-52","ลำปาง"
"210","TH-53","อุตรดิตถ์"
"210","TH-55","น่าน"
"210","TH-56","พะเยา"
"210","TH-57","เชียงราย"
"210","TH-58","แม่ฮ่องสอน"
"210","TH-60","นครสวรรค์"
"210","TH-61","อุทัยธานี"
"210","TH-62","กำแพงเพชร"
"210","TH-63","ตาก"
"210","TH-64","สุโขทัย"
"210","TH-66","ชุมพร"
"210","TH-67","พิจิตร"
"210","TH-70","ราชบุรี"
"210","TH-71","กาญจนบุรี"
"210","TH-72","สุพรรณบุรี"
"210","TH-73","นครปฐม"
"210","TH-74","สมุทรสาคร"
"210","TH-75","สมุทรสงคราม"
"210","TH-76","เพชรบุรี"
"210","TH-77","ประจวบคีรีขันธ์"
"210","TH-80","นครศรีธรรมราช"
"210","TH-81","กระบี่"
"210","TH-82","พังงา"
"210","TH-83","ภูเก็ต"
"210","TH-84","สุราษฎร์ธานี"
"210","TH-85","ระนอง"
"210","TH-86","ชุมพร"
"210","TH-90","สงขลา"
"210","TH-91","สตูล"
"210","TH-92","ตรัง"
"210","TH-93","พัทลุง"
"210","TH-94","ปัตตานี"
"210","TH-95","ยะลา"
"210","TH-96","นราธิวาส"
"211","C","Centrale"
"211","K","Kara"
"211","M","Maritime"
"211","P","Plateaux"
"211","S","Savanes"
"212","A","Atafu"
"212","F","Fakaofo"
"212","N","Nukunonu"
"213","H","Ha'apai"
"213","T","Tongatapu"
"213","V","Vava'u"
"214","ARI","Arima"
"214","CHA","Chaguanas"
"214","CTT","Couva-Tabaquite-Talparo"
"214","DMN","Diego Martin"
"214","ETO","Eastern Tobago"
"214","RCM","Rio Claro-Mayaro"
"214","PED","Penal-Debe"
"214","PTF","Point Fortin"
"214","POS","Port of Spain"
"214","PRT","Princes Town"
"214","SFO","San Fernando"
"214","SGE","Sangre Grande"
"214","SJL","San Juan-Laventille"
"214","SIP","Siparia"
"214","TUP","Tunapuna-Piarco"
"214","WTO","Western Tobago"
"215","11","ولاية تونس"
"215","12","ولاية أريانة"
"215","13","ولاية بن عروس"
"215","14","ولاية منوبة"
"215","21","ولاية نابل"
"215","22","ولاية زغوان"
"215","23","ولاية بنزرت"
"215","31","ولاية باجة"
"215","32","ولاية جندوبة"
"215","33","ولاية الكاف"
"215","34","ولاية سليانة"
"215","41","ولاية القيروان"
"215","42","ولاية القصرين"
"215","43","ولاية سيدي بوزيد"
"215","51","ولاية سوسة"
"215","52","ولاية المنستير"
"215","53","ولاية المهدية"
"215","61","ولاية صفاقس"
"215","71","ولاية قفصة"
"215","72","ولاية توزر"
"215","73","ولاية قبلي"
"215","81","ولاية قابس"
"215","82","ولاية مدنين"
"215","83","ولاية تطاوين"
"216","01","Adana"
"216","02","Adıyaman"
"216","03","Afyonkarahisar"
"216","04","Ağrı"
"216","05","Amasya"
"216","06","Ankara"
"216","07","Antalya"
"216","08","Artvin"
"216","09","Aydın"
"216","10","Balıkesir"
"216","11","Bilecik"
"216","12","Bingöl"
"216","13","Bitlis"
"216","14","Bolu"
"216","15","Burdur"
"216","16","Bursa"
"216","17","Çanakkale"
"216","18","Çankırı"
"216","19","Çorum"
"216","20","Denizli"
"216","21","Diyarbakır"
"216","22","Edirne"
"216","23","Elazığ"
"216","24","Erzincan"
"216","25","Erzurum"
"216","26","Eskişehir"
"216","27","Gaziantep"
"216","28","Giresun"
"216","29","Gümüşhane"
"216","30","Hakkari"
"216","31","Hatay"
"216","32","Isparta"
"216","33","Mersin"
"216","34","İstanbul"
"216","35","İzmir"
"216","36","Kars"
"216","37","Kastamonu"
"216","38","Kayseri"
"216","39","Kırklareli"
"216","40","Kırşehir"
"216","41","Kocaeli"
"216","42","Konya"
"216","43","Kütahya"
"216","44","Malatya"
"216","45","Manisa"
"216","46","Kahramanmaraş"
"216","47","Mardin"
"216","48","Muğla"
"216","49","Muş"
"216","50","Nevşehir"
"216","51","Niğde"
"216","52","Ordu"
"216","53","Rize"
"216","54","Sakarya"
"216","55","Samsun"
"216","56","Siirt"
"216","57","Sinop"
"216","58","Sivas"
"216","59","Tekirdağ"
"216","60","Tokat"
"216","61","Trabzon"
"216","62","Tunceli"
"216","63","Şanlıurfa"
"216","64","Uşak"
"216","65","Van"
"216","66","Yozgat"
"216","67","Zonguldak"
"216","68","Aksaray"
"216","69","Bayburt"
"216","70","Karaman"
"216","71","Kırıkkale"
"216","72","Batman"
"216","73","Şırnak"
"216","74","Bartın"
"216","75","Ardahan"
"216","76","Iğdır"
"216","77","Yalova"
"216","78","Karabük"
"216","79","Kilis"
"216","80","Osmaniye"
"216","81","Düzce"
"217","A","Ahal welaýaty"
"217","B","Balkan welaýaty"
"217","D","Daşoguz welaýaty"
"217","L","Lebap welaýaty"
"217","M","Mary welaýaty"
"218","AC","Ambergris Cays"
"218","DC","Dellis Cay"
"218","FC","French Cay"
"218","LW","Little Water Cay"
"218","RC","Parrot Cay"
"218","PN","Pine Cay"
"218","SL","Salt Cay"
"218","GT","Grand Turk"
"218","SC","South Caicos"
"218","EC","East Caicos"
"218","MC","Middle Caicos"
"218","NC","North Caicos"
"218","PR","Providenciales"
"218","WC","West Caicos"
"219","FUN","Funafuti"
"219","NMA","Nanumea"
"219","NMG","Nanumanga"
"219","NIT","Niutao"
"219","NIU","Nui"
"219","NKF","Nukufetau"
"219","NKL","Nukulaelae"
"219","VAI","Vaitupu"
"220","101","Kalangala"
"220","102","Kampala"
"220","103","Kiboga"
"220","104","Luwero"
"220","105","Masaka"
"220","106","Mpigi"
"220","107","Mubende"
"220","108","Mukono"
"220","109","Nakasongola"
"220","110","Rakai"
"220","111","Sembabule"
"220","112","Kayunga"
"220","113","Wakiso"
"220","201","Bugiri"
"220","202","Busia"
"220","203","Iganga"
"220","204","Jinja"
"220","205","Kamuli"
"220","206","Kapchorwa"
"220","207","Katakwi"
"220","208","Kumi"
"220","209","Mbale"
"220","210","Pallisa"
"220","211","Soroti"
"220","212","Tororo"
"220","213","Kaberamaido"
"220","214","Mayuge"
"220","215","Sironko"
"220","301","Adjumani"
"220","302","Apac"
"220","303","Arua"
"220","304","Gulu"
"220","305","Kitgum"
"220","306","Kotido"
"220","307","Lira"
"220","308","Moroto"
"220","309","Moyo"
"220","310","Nebbi"
"220","311","Nakapiripirit"
"220","312","Pader"
"220","313","Yumbe"
"220","401","Bundibugyo"
"220","402","Bushenyi"
"220","403","Hoima"
"220","404","Kabale"
"220","405","Kabarole"
"220","406","Kasese"
"220","407","Kibale"
"220","408","Kisoro"
"220","409","Masindi"
"220","410","Mbarara"
"220","411","Ntungamo"
"220","412","Rukungiri"
"220","413","Kamwenge"
"220","414","Kanungu"
"220","415","Kyenjojo"
"221","05","Вінницька область"
"221","07","Волинська область"
"221","09","Луганська область"
"221","12","Дніпропетровська область"
"221","14","Донецька область"
"221","18","Житомирська область"
"221","19","Рівненська область"
"221","21","Закарпатська область"
"221","23","Запорізька область"
"221","26","Івано-Франківська область"
"221","30","Київ"
"221","32","Київська область"
"221","35","Кіровоградська область"
"221","40","Севастополь"
"221","43","Автономная Республика Крым"
"221","46","Львівська область"
"221","48","Миколаївська область"
"221","51","Одеська область"
"221","53","Полтавська область"
"221","59","Сумська область"
"221","61","Тернопільська область"
"221","63","Харківська область"
"221","65","Херсонська область"
"221","68","Хмельницька область"
"221","71","Черкаська область"
"221","74","Чернігівська область"
"221","77","Чернівецька область"
"223","ABD","Aberdeenshire"
"223","ABE","Aberdeen"
"223","AGB","Argyll and Bute"
"223","AGY","Isle of Anglesey"
"223","ANS","Angus"
"223","ANT","Antrim"
"223","ARD","Ards"
"223","ARM","Armagh"
"223","BAS","Bath and North East Somerset"
"223","BBD","Blackburn with Darwen"
"223","BDF","Bedfordshire"
"223","BDG","Barking and Dagenham"
"223","BEN","Brent"
"223","BEX","Bexley"
"223","BFS","Belfast"
"223","BGE","Bridgend"
"223","BGW","Blaenau Gwent"
"223","BIR","Birmingham"
"223","BKM","Buckinghamshire"
"223","BLA","Ballymena"
"223","BLY","Ballymoney"
"223","BMH","Bournemouth"
"223","BNB","Banbridge"
"223","BNE","Barnet"
"223","BNH","Brighton and Hove"
"223","BNS","Barnsley"
"223","BOL","Bolton"
"223","BPL","Blackpool"
"223","BRC","Bracknell"
"223","BRD","Bradford"
"223","BRY","Bromley"
"223","BST","Bristol"
"223","BUR","Bury"
"223","CAM","Cambridgeshire"
"223","CAY","Caerphilly"
"223","CGN","Ceredigion"
"223","CGV","Craigavon"
"223","CHS","Cheshire"
"223","CKF","Carrickfergus"
"223","CKT","Cookstown"
"223","CLD","Calderdale"
"223","CLK","Clackmannanshire"
"223","CLR","Coleraine"
"223","CMA","Cumbria"
"223","CMD","Camden"
"223","CMN","Carmarthenshire"
"223","CON","Cornwall"
"223","COV","Coventry"
"223","CRF","Cardiff"
"223","CRY","Croydon"
"223","CSR","Castlereagh"
"223","CWY","Conwy"
"223","DAL","Darlington"
"223","DBY","Derbyshire"
"223","DEN","Denbighshire"
"223","DER","Derby"
"223","DEV","Devon"
"223","DGN","Dungannon and South Tyrone"
"223","DGY","Dumfries and Galloway"
"223","DNC","Doncaster"
"223","DND","Dundee"
"223","DOR","Dorset"
"223","DOW","Down"
"223","DRY","Derry"
"223","DUD","Dudley"
"223","DUR","Durham"
"223","EAL","Ealing"
"223","EAY","East Ayrshire"
"223","EDH","Edinburgh"
"223","EDU","East Dunbartonshire"
"223","ELN","East Lothian"
"223","ELS","Eilean Siar"
"223","ENF","Enfield"
"223","ERW","East Renfrewshire"
"223","ERY","East Riding of Yorkshire"
"223","ESS","Essex"
"223","ESX","East Sussex"
"223","FAL","Falkirk"
"223","FER","Fermanagh"
"223","FIF","Fife"
"223","FLN","Flintshire"
"223","GAT","Gateshead"
"223","GLG","Glasgow"
"223","GLS","Gloucestershire"
"223","GRE","Greenwich"
"223","GSY","Guernsey"
"223","GWN","Gwynedd"
"223","HAL","Halton"
"223","HAM","Hampshire"
"223","HAV","Havering"
"223","HCK","Hackney"
"223","HEF","Herefordshire"
"223","HIL","Hillingdon"
"223","HLD","Highland"
"223","HMF","Hammersmith and Fulham"
"223","HNS","Hounslow"
"223","HPL","Hartlepool"
"223","HRT","Hertfordshire"
"223","HRW","Harrow"
"223","HRY","Haringey"
"223","IOS","Isles of Scilly"
"223","IOW","Isle of Wight"
"223","ISL","Islington"
"223","IVC","Inverclyde"
"223","JSY","Jersey"
"223","KEC","Kensington and Chelsea"
"223","KEN","Kent"
"223","KHL","Kingston upon Hull"
"223","KIR","Kirklees"
"223","KTT","Kingston upon Thames"
"223","KWL","Knowsley"
"223","LAN","Lancashire"
"223","LBH","Lambeth"
"223","LCE","Leicester"
"223","LDS","Leeds"
"223","LEC","Leicestershire"
"223","LEW","Lewisham"
"223","LIN","Lincolnshire"
"223","LIV","Liverpool"
"223","LMV","Limavady"
"223","LND","London"
"223","LRN","Larne"
"223","LSB","Lisburn"
"223","LUT","Luton"
"223","MAN","Manchester"
"223","MDB","Middlesbrough"
"223","MDW","Medway"
"223","MFT","Magherafelt"
"223","MIK","Milton Keynes"
"223","MLN","Midlothian"
"223","MON","Monmouthshire"
"223","MRT","Merton"
"223","MRY","Moray"
"223","MTY","Merthyr Tydfil"
"223","MYL","Moyle"
"223","NAY","North Ayrshire"
"223","NBL","Northumberland"
"223","NDN","North Down"
"223","NEL","North East Lincolnshire"
"223","NET","Newcastle upon Tyne"
"223","NFK","Norfolk"
"223","NGM","Nottingham"
"223","NLK","North Lanarkshire"
"223","NLN","North Lincolnshire"
"223","NSM","North Somerset"
"223","NTA","Newtownabbey"
"223","NTH","Northamptonshire"
"223","NTL","Neath Port Talbot"
"223","NTT","Nottinghamshire"
"223","NTY","North Tyneside"
"223","NWM","Newham"
"223","NWP","Newport"
"223","NYK","North Yorkshire"
"223","NYM","Newry and Mourne"
"223","OLD","Oldham"
"223","OMH","Omagh"
"223","ORK","Orkney Islands"
"223","OXF","Oxfordshire"
"223","PEM","Pembrokeshire"
"223","PKN","Perth and Kinross"
"223","PLY","Plymouth"
"223","POL","Poole"
"223","POR","Portsmouth"
"223","POW","Powys"
"223","PTE","Peterborough"
"223","RCC","Redcar and Cleveland"
"223","RCH","Rochdale"
"223","RCT","Rhondda Cynon Taf"
"223","RDB","Redbridge"
"223","RDG","Reading"
"223","RFW","Renfrewshire"
"223","RIC","Richmond upon Thames"
"223","ROT","Rotherham"
"223","RUT","Rutland"
"223","SAW","Sandwell"
"223","SAY","South Ayrshire"
"223","SCB","Scottish Borders"
"223","SFK","Suffolk"
"223","SFT","Sefton"
"223","SGC","South Gloucestershire"
"223","SHF","Sheffield"
"223","SHN","Saint Helens"
"223","SHR","Shropshire"
"223","SKP","Stockport"
"223","SLF","Salford"
"223","SLG","Slough"
"223","SLK","South Lanarkshire"
"223","SND","Sunderland"
"223","SOL","Solihull"
"223","SOM","Somerset"
"223","SOS","Southend-on-Sea"
"223","SRY","Surrey"
"223","STB","Strabane"
"223","STE","Stoke-on-Trent"
"223","STG","Stirling"
"223","STH","Southampton"
"223","STN","Sutton"
"223","STS","Staffordshire"
"223","STT","Stockton-on-Tees"
"223","STY","South Tyneside"
"223","SWA","Swansea"
"223","SWD","Swindon"
"223","SWK","Southwark"
"223","TAM","Tameside"
"223","TFW","Telford and Wrekin"
"223","THR","Thurrock"
"223","TOB","Torbay"
"223","TOF","Torfaen"
"223","TRF","Trafford"
"223","TWH","Tower Hamlets"
"223","VGL","Vale of Glamorgan"
"223","WAR","Warwickshire"
"223","WBK","West Berkshire"
"223","WDU","West Dunbartonshire"
"223","WFT","Waltham Forest"
"223","WGN","Wigan"
"223","WIL","Wiltshire"
"223","WKF","Wakefield"
"223","WLL","Walsall"
"223","WLN","West Lothian"
"223","WLV","Wolverhampton"
"223","WNM","Windsor and Maidenhead"
"223","WOK","Wokingham"
"223","WOR","Worcestershire"
"223","WRL","Wirral"
"223","WRT","Warrington"
"223","WRX","Wrexham"
"223","WSM","Westminster"
"223","WSX","West Sussex"
"223","YOR","York"
"223","ZET","Shetland Islands"
"224","AK","Alaska"
"224","AL","Alabama"
"224","AS","American Samoa"
"224","AR","Arkansas"
"224","AZ","Arizona"
"224","CA","California"
"224","CO","Colorado"
"224","CT","Connecticut"
"224","DC","District of Columbia"
"224","DE","Delaware"
"224","FL","Florida"
"224","GA","Georgia"
"224","GU","Guam"
"224","HI","Hawaii"
"224","IA","Iowa"
"224","ID","Idaho"
"224","IL","Illinois"
"224","IN","Indiana"
"224","KS","Kansas"
"224","KY","Kentucky"
"224","LA","Louisiana"
"224","MA","Massachusetts"
"224","MD","Maryland"
"224","ME","Maine"
"224","MI","Michigan"
"224","MN","Minnesota"
"224","MO","Missouri"
"224","MS","Mississippi"
"224","MT","Montana"
"224","NC","North Carolina"
"224","ND","North Dakota"
"224","NE","Nebraska"
"224","NH","New Hampshire"
"224","NJ","New Jersey"
"224","NM","New Mexico"
"224","NV","Nevada"
"224","NY","New York"
"224","MP","Northern Mariana Islands"
"224","OH","Ohio"
"224","OK","Oklahoma"
"224","OR","Oregon"
"224","PA","Pennsylvania"
"224","PR","Puerto Rico"
"224","RI","Rhode Island"
"224","SC","South Carolina"
"224","SD","South Dakota"
"224","TN","Tennessee"
"224","TX","Texas"
"224","UM","U.S. Minor Outlying Islands"
"224","UT","Utah"
"224","VA","Virginia"
"224","VI","Virgin Islands of the U.S."
"224","VT","Vermont"
"224","WA","Washington"
"224","WI","Wisconsin"
"224","WV","West Virginia"
"224","WY","Wyoming"
"225","BI","Baker Island"
"225","HI","Howland Island"
"225","JI","Jarvis Island"
"225","JA","Johnston Atoll"
"225","KR","Kingman Reef"
"225","MA","Midway Atoll"
"225","NI","Navassa Island"
"225","PA","Palmyra Atoll"
"225","WI","Wake Island"
"226","AR","Artigas"
"226","CA","Canelones"
"226","CL","Cerro Largo"
"226","CO","Colonia"
"226","DU","Durazno"
"226","FD","Florida"
"226","FS","Flores"
"226","LA","Lavalleja"
"226","MA","Maldonado"
"226","MO","Montevideo"
"226","PA","Paysandu"
"226","RN","Río Negro"
"226","RO","Rocha"
"226","RV","Rivera"
"226","SA","Salto"
"226","SJ","San José"
"226","SO","Soriano"
"226","TA","Tacuarembó"
"226","TT","Treinta y Tres"
"227","AN","Andijon viloyati"
"227","BU","Buxoro viloyati"
"227","FA","Farg'ona viloyati"
"227","JI","Jizzax viloyati"
"227","NG","Namangan viloyati"
"227","NW","Navoiy viloyati"
"227","QA","Qashqadaryo viloyati"
"227","QR","Qoraqalpog'iston Respublikasi"
"227","SA","Samarqand viloyati"
"227","SI","Sirdaryo viloyati"
"227","SU","Surxondaryo viloyati"
"227","TK","Toshkent"
"227","TO","Toshkent viloyati"
"227","XO","Xorazm viloyati"
"228","MAP","Malampa"
"228","PAM","Pénama"
"228","SAM","Sanma"
"228","SEE","Shéfa"
"228","TAE","Taféa"
"228","TOB","Torba"
"230","A","Distrito Capital"
"230","B","Anzoátegui"
"230","C","Apure"
"230","D","Aragua"
"230","E","Barinas"
"230","F","Bolívar"
"230","G","Carabobo"
"230","H","Cojedes"
"230","I","Falcón"
"230","J","Guárico"
"230","K","Lara"
"230","L","Mérida"
"230","M","Miranda"
"230","N","Monagas"
"230","O","Nueva Esparta"
"230","P","Portuguesa"
"230","R","Sucre"
"230","S","Tachira"
"230","T","Trujillo"
"230","U","Yaracuy"
"230","V","Zulia"
"230","W","Capital Dependencia"
"230","X","Vargas"
"230","Y","Delta Amacuro"
"230","Z","Amazonas"
"231","01","Lai Châu"
"231","02","Lào Cai"
"231","03","Hà Giang"
"231","04","Cao Bằng"
"231","05","Sơn La"
"231","06","Yên Bái"
"231","07","Tuyên Quang"
"231","09","Lạng Sơn"
"231","13","Quảng Ninh"
"231","14","Hòa Bình"
"231","15","Hà Tây"
"231","18","Ninh Bình"
"231","20","Thái Bình"
"231","21","Thanh Hóa"
"231","22","Nghệ An"
"231","23","Hà Tĩnh"
"231","24","Quảng Bình"
"231","25","Quảng Trị"
"231","26","Thừa Thiên-Huế"
"231","27","Quảng Nam"
"231","28","Kon Tum"
"231","29","Quảng Ngãi"
"231","30","Gia Lai"
"231","31","Bình Định"
"231","32","Phú Yên"
"231","33","Đắk Lắk"
"231","34","Khánh Hòa"
"231","35","Lâm Đồng"
"231","36","Ninh Thuận"
"231","37","Tây Ninh"
"231","39","Đồng Nai"
"231","40","Bình Thuận"
"231","41","Long An"
"231","43","Bà Rịa-Vũng Tàu"
"231","44","An Giang"
"231","45","Đồng Tháp"
"231","46","Tiền Giang"
"231","47","Kiên Giang"
"231","48","Cần Thơ"
"231","49","Vĩnh Long"
"231","50","Bến Tre"
"231","51","Trà Vinh"
"231","52","Sóc Trăng"
"231","53","Bắc Kạn"
"231","54","Bắc Giang"
"231","55","Bạc Liêu"
"231","56","Bắc Ninh"
"231","57","Bình Dương"
"231","58","Bình Phước"
"231","59","Cà Mau"
"231","60","Đà Nẵng"
"231","61","Hải Dương"
"231","62","Hải Phòng"
"231","63","Hà Nam"
"231","64","Hà Nội"
"231","65","Sài Gòn"
"231","66","Hưng Yên"
"231","67","Nam Định"
"231","68","Phú Thọ"
"231","69","Thái Nguyên"
"231","70","Vĩnh Phúc"
"231","71","Điện Biên"
"231","72","Đắk Nông"
"231","73","Hậu Giang"
"233","C","Saint Croix"
"233","J","Saint John"
"233","T","Saint Thomas"
"234","A","Alo"
"234","S","Sigave"
"234","W","Wallis"
"236","AB","أبين‎"
"236","AD","عدن"
"236","AM","عمران"
"236","BA","البيضاء"
"236","DA","الضالع"
"236","DH","ذمار"
"236","HD","حضرموت"
"236","HJ","حجة"
"236","HU","الحديدة"
"236","IB","إب"
"236","JA","الجوف"
"236","LA","لحج"
"236","MA","مأرب"
"236","MR","المهرة"
"236","MW","المحويت"
"236","SD","صعدة"
"236","SN","صنعاء"
"236","SH","شبوة"
"236","TA","تعز"
"239","01","Western"
"239","02","Central"
"239","03","Eastern"
"239","04","Luapula"
"239","05","Northern"
"239","06","North-Western"
"239","07","Southern"
"239","08","Copperbelt"
"239","09","Lusaka"
"240","MA","Manicaland"
"240","MC","Mashonaland Central"
"240","ME","Mashonaland East"
"240","MI","Midlands"
"240","MN","Matabeleland North"
"240","MS","Matabeleland South"
"240","MV","Masvingo"
"240","MW","Mashonaland West"
