package Statocles::App::Blog;
# ABSTRACT: A blog application
$Statocles::App::Blog::VERSION = '0.037';
use Statocles::Base 'Class';
use Getopt::Long qw( GetOptionsFromArray );
use Statocles::Store::File;
use Statocles::Page::Document;
use Statocles::Page::List;
use Statocles::Page::Feed;

extends 'Statocles::App';


has store => (
    is => 'ro',
    isa => Store,
    coerce => Store->coercion,
    required => 1,
);


has url_root => (
    is => 'ro',
    isa => Str,
    required => 1,
);


has page_size => (
    is => 'ro',
    isa => Int,
    default => sub { 5 },
);


has index_tags => (
    is => 'ro',
    isa => ArrayRef[Str],
    default => sub { [] },
);

# A cache of the last set of post pages we have
# XXX: We need to allow apps to have a "clear" the way that Store and Theme do
has _post_pages => (
    is => 'rw',
    isa => ArrayRef,
    default => sub { [] },
);


our $default_post = {
    tags => undef,
    content => <<'ENDCONTENT',
Markdown content goes here.
ENDCONTENT
};

my $USAGE_INFO = <<'ENDHELP';
Usage:
    $name help -- This help file
    $name post [--date YYYY-MM-DD] <title> -- Create a new blog post with the given title
ENDHELP

sub command {
    my ( $self, $name, @argv ) = @_;

    if ( !$argv[0] ) {
        say STDERR "ERROR: Missing command";
        say STDERR eval "qq{$USAGE_INFO}";
        return 1;
    }

    if ( $argv[0] eq 'help' ) {
        say eval "qq{$USAGE_INFO}";
    }
    elsif ( $argv[0] eq 'post' ) {
        my %opt;
        GetOptionsFromArray( \@argv, \%opt,
            'date:s',
        );

        my $title = join " ", @argv[1..$#argv];
        if ( !$ENV{EDITOR} && !$title ) {
            say STDERR <<"ENDHELP";
Title is required when \$EDITOR is not set.

Usage: $name post <title>
ENDHELP
            return 1;
        }

        my ( $year, $mon, $day );
        if ( $opt{ date } ) {
            ( $year, $mon, $day ) = split /-/, $opt{date};
        }
        else {
            ( undef, undef, undef, $day, $mon, $year ) = localtime;
            $year += 1900;
            $mon += 1;
        }

        my @date_parts = (
            sprintf( '%04i', $year ),
            sprintf( '%02i', $mon ),
            sprintf( '%02i', $day ),
        );

        my %doc = (
            %$default_post,
            title => $title,
            last_modified => Time::Piece->new,
        );

        # Read post content on STDIN
        if ( !-t *STDIN ) {
            $doc{content} = do { local $/; <STDIN> };
            # Re-open STDIN as the TTY so that the editor (vim) can use it
            # XXX Is this also a problem on Windows?
            if ( -e '/dev/tty' ) {
                close STDIN;
                open STDIN, '/dev/tty';
            }
        }

        if ( $ENV{EDITOR} ) {
            # I can see no good way to test this automatically
            my $slug = lc $title || "new post";
            $slug =~ s/\s+/-/g;
            my $path = Path::Tiny->new( @date_parts, "$slug.markdown" );
            my $tmp_path = $self->store->write_document( $path => \%doc );
            system $ENV{EDITOR}, $tmp_path;
            %doc = %{ $self->store->read_document( $path ) };
            $self->store->path->child( $path )->remove;
            $title = $doc{title};
        }

        my $slug = lc $title;
        $slug =~ s/\s+/-/g;
        my $path = Path::Tiny->new( @date_parts, "$slug.markdown" );
        my $full_path = $self->store->write_document( $path => \%doc );
        say "New post at: $full_path";

    }
    else {
        say STDERR qq{ERROR: Unknown command "$argv[0]"};
        say STDERR eval "qq{$USAGE_INFO}";
        return 1;
    }

    return 0;
}


sub post_pages {
    my ( $self ) = @_;
    my $today = Time::Piece->new->ymd;
    my @pages;
    for my $doc ( @{ $self->store->documents } ) {
        my $path = join "/", $self->url_root, $doc->path;
        $path =~ s{/{2,}}{/}g;
        $path =~ s{[.]\w+$}{.html};

        my @date_parts = $path =~ m{/(\d{4})/(\d{2})/(\d{2})/[^/]+$};
        next unless @date_parts;
        my $date = join "-", @date_parts;

        next if $date gt $today;

        my @tags;
        for my $tag ( @{ $doc->tags } ) {
            push @tags, Statocles::Link->new(
                text => $tag,
                href => $self->_tag_url( $tag ),
            );
        }

        push @pages, Statocles::Page::Document->new(
            app => $self,
            layout => $self->site->theme->template( site => 'layout.html' ),
            template => $self->site->theme->template( blog => 'post.html' ),
            document => $doc,
            path => $path,
            last_modified => $doc->has_last_modified ? $doc->last_modified : Time::Piece->strptime( $date, '%Y-%m-%d' ),
            tags => \@tags,
        );
    }

    $self->_post_pages( [ @pages ] );

    return @pages;
}


my %FEEDS = (
    rss => {
        text => 'RSS',
        type => 'application/rss+xml',
        template => 'index.rss',
    },
    atom => {
        text => 'Atom',
        type => 'application/atom+xml',
        template => 'index.atom',
    },
);

sub index {
    my ( $self, @all_post_pages ) = @_;

    # Filter the index_tags
    my @index_post_pages;
    PAGE: for my $page ( @all_post_pages ) {
        my $add = 1;
        for my $tag_spec ( @{ $self->index_tags } ) {
            my $flag = substr $tag_spec, 0, 1;
            my $tag = substr $tag_spec, 1;
            if ( grep { $_ eq $tag } @{ $page->document->tags } ) {
                $add = $flag eq '-' ? 0 : 1;
            }
        }
        push @index_post_pages, $page if $add;
    }

    my @pages = Statocles::Page::List->paginate(
        after => $self->page_size,
        path => join( "/", $self->url_root, 'page/%i/index.html' ),
        index => join( "/", $self->url_root, 'index.html' ),
        # Sorting by path just happens to also sort by date
        pages => [ sort { $b->path cmp $a->path } @index_post_pages ],
        app => $self,
        template => $self->site->theme->template( blog => 'index.html' ),
        layout => $self->site->theme->template( site => 'layout.html' ),
    );

    my $index = $pages[0];
    my @feed_pages;
    my @feed_links;
    for my $feed ( sort keys %FEEDS ) {
        my $page = Statocles::Page::Feed->new(
            app => $self,
            type => $FEEDS{ $feed }{ type },
            page => $index,
            path => join( "/", $self->url_root, 'index.' . $feed ),
            template => $self->site->theme->template( blog => $FEEDS{$feed}{template} ),
        );
        push @feed_pages, $page;
        push @feed_links, Statocles::Link->new(
            text => $FEEDS{ $feed }{ text },
            href => $page->path->stringify,
            type => $page->type,
        );
    }

    # Add the feeds to all the pages
    for my $page ( @pages ) {
        $page->_links->{feed} = \@feed_links;
    }

    return ( @pages, @feed_pages );
}


sub tag_pages {
    my ( $self, @post_pages ) = @_;

    my %tagged_docs = $self->_tag_docs( @post_pages );

    my @pages;
    for my $tag ( keys %tagged_docs ) {
        my @tag_pages = Statocles::Page::List->paginate(
            after => $self->page_size,
            path => join( "/", $self->url_root, 'tag', $tag, 'page/%i/index.html' ),
            index => join( "/", $self->_tag_url( $tag ), 'index.html' ),
            # Sorting by path just happens to also sort by date
            pages => [ sort { $b->path cmp $a->path } @{ $tagged_docs{ $tag } } ],
            app => $self,
            template => $self->site->theme->template( blog => 'index.html' ),
            layout => $self->site->theme->template( site => 'layout.html' ),
        );

        my $index = $tag_pages[0];
        my @feed_pages;
        my @feed_links;
        for my $feed ( sort keys %FEEDS ) {
            my $tag_file = $tag . '.' . $feed;
            $tag_file =~ s/\s+/-/g;

            my $page = Statocles::Page::Feed->new(
                type => $FEEDS{ $feed }{ type },
                app => $self,
                page => $index,
                path => join( "/", $self->url_root, 'tag', $tag_file ),
                template => $self->site->theme->template( blog => $FEEDS{$feed}{template} ),
            );
            push @feed_pages, $page;
            push @feed_links, Statocles::Link->new(
                text => $FEEDS{ $feed }{ text },
                href => $page->path->stringify,
                type => $page->type,
            );
        }

        # Add the feeds to all the pages
        for my $page ( @tag_pages ) {
            $page->_links->{feed} = \@feed_links;
        }

        push @pages, @tag_pages, @feed_pages;
    }

    return @pages;
}


sub pages {
    my ( $self ) = @_;
    my @post_pages = $self->post_pages;
    return (
        ( map { $self->$_( @post_pages ) } qw( index tag_pages ) ),
        @post_pages,
    );
}


sub tags {
    my ( $self ) = @_;
    my %tagged_docs = $self->_tag_docs( @{ $self->_post_pages } );
    return map {; Statocles::Link->new( text => $_, href => $self->_tag_url( $_ ) ) }
        sort keys %tagged_docs
}

sub _tag_docs {
    my ( $self, @post_pages ) = @_;
    my %tagged_docs;
    for my $page ( @post_pages ) {
        for my $tag ( @{ $page->document->tags } ) {
            push @{ $tagged_docs{ $tag } }, $page;
        }
    }
    return %tagged_docs;
}

sub _tag_url {
    my ( $self, $tag ) = @_;
    $tag =~ s/\s+/-/g;
    return join "/", $self->url_root, "tag", $tag;
}

1;

__END__

=pod

=head1 NAME

Statocles::App::Blog - A blog application

=head1 VERSION

version 0.037

=head1 DESCRIPTION

This is a simple blog application for Statocles.

=head2 FEATURES

=over

=item *

Content dividers. By dividing your main content with "---", you create
sections. Only the first section will show up on the index page or in RSS
feeds.

=item *

RSS and Atom syndication feeds.

=item *

Tags to organize blog posts. Tags have their own custom feeds so users can
subscribe to only those posts they care about.

=item *

Crosspost links to redirect users to a syndicated blog. Useful when you
participate in many blogs and want to drive traffic to them.

=item *

Post-dated blog posts to appear automatically when the date is passed. If a
blog post is set in the future, it will not be added to the site when running
C<build> or C<deploy>.

In order to ensure that post-dated blogs get added, you may want to run
C<deploy> in a nightly cron job.

=back

=head1 ATTRIBUTES

=head2 store

The L<store|Statocles::Store> to read for documents.

=head2 url_root

The URL root of this application. All pages from this app will be under this
root. Use this to ensure two apps do not try to write the same path.

=head2 page_size

The number of posts to put in a page (the main page and the tag pages). Defaults
to 5.

=head2 index_tags

Filter the tags shown in the index page. An array of tags prefixed with either
a + or a -. By prefixing the tag with a "-", it will be removed from the index,
unless a later tag prefixed with a "+" also matches.

By default, all tags are shown on the index page.

So, given a document with tags "foo", and "bar":

    index_tags => [ ];                  # document will be included
    index_tags => [ '-foo' ];           # document will not be included
    index_tags => [ '-foo', '+bar' ];   # document will be included

=head1 METHODS

=head2 command( app_name, args )

Run a command on this app. The app name is used to build the help, so
users get exactly what they need to run.

=head2 post_pages()

Get the individual post Statocles::Page objects.

=head2 index()

Get the index page (a L<list page|Statocles::Page::List>) for this application.
This includes all the relevant L<feed pages|Statocles::Page::Feed>.

=head2 tag_pages()

Get L<pages|Statocles::Page> for the tags in the blog post documents.

=head2 pages()

Get all the L<pages|Statocles::Page> for this application.

=head2 tags()

Get a set of L<link objects|Statocles::Link> suitable for creating a list of
tag links. The common attributes are:

    text => 'The tag text'
    href => 'The URL to the tag page'

=head1 THEME

=over

=item blog => index

The index page template. Gets the following template variables:

=over

=item site

The L<Statocles::Site> object.

=item pages

An array reference containing all the blog post pages. Each page is a hash reference with the following keys:

=over

=item content

The post content

=item title

The post title

=item author

The post author

=back

=item blog => post

The main post page template. Gets the following template variables:

=over

=item site

The L<Statocles::Site> object

=item content

The post content

=item title

The post title

=item author

The post author

=back

=back

=back

=head1 SEE ALSO

=over 4

=item L<Statocles::App>

=back

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
