package CPAN::Testers::API::Controller::Report;
our $VERSION = '0.007';
# ABSTRACT: Work with raw test reports

#pod =head1 DESCRIPTION
#pod
#pod This API allows working directly with the JSON report documents
#pod submitted by the army of testers of CPAN.
#pod
#pod =head1 SEE ALSO
#pod
#pod L<CPAN::Testers::Schema::Result::TestReport>, L<Mojolicious::Controller>
#pod
#pod =cut

use Mojo::Base 'Mojolicious::Controller';
use CPAN::Testers::API::Base;
use Mojo::JSON qw( encode_json );
use Data::UUID;

#pod =method report_post
#pod
#pod     ### Requests:
#pod     POST /v3/report
#pod     { ... }
#pod
#pod     ### Response:
#pod     201 Created
#pod     { "id": "..." }
#pod
#pod Submit a new CPAN Testers report. This is used by testers when they're
#pod finished running a test.
#pod
#pod =cut

sub report_post( $c ) {
    $c->openapi->valid_input or return;
    my $report = $c->validation->param( 'report' );
    $report->{id} = Data::UUID->new->create_str();
    $c->schema->resultset( 'TestReport' )->create( {
        id => $report->{id},
        report => encode_json( $report ),
    } );
    return $c->render(
        status => 201,
        openapi => {
            id => $report->{id},
        },
    );
}

1;

__END__

=pod

=head1 NAME

CPAN::Testers::API::Controller::Report - Work with raw test reports

=head1 VERSION

version 0.007

=head1 DESCRIPTION

This API allows working directly with the JSON report documents
submitted by the army of testers of CPAN.

=head1 METHODS

=head2 report_post

    ### Requests:
    POST /v3/report
    { ... }

    ### Response:
    201 Created
    { "id": "..." }

Submit a new CPAN Testers report. This is used by testers when they're
finished running a test.

=head1 SEE ALSO

L<CPAN::Testers::Schema::Result::TestReport>, L<Mojolicious::Controller>

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
