## Domain Registry Interface, CentralNic DNS TTL EPP extension
## (http://labs.centralnic.com/epp/ext/ttl.php)
##
## Copyright (c) 2007 Patrick Mevzek <netdri@dotandco.com>. All rights reserved.
##
## This file is part of Net::DRI
##
## Net::DRI is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## See the LICENSE file that comes with this distribution for more details.
#
# 
#
####################################################################################################

package Net::DRI::Protocol::EPP::Extensions::CentralNic::TTL;

use strict;

use DateTime::Duration;

our $VERSION=do { my @r=(q$Revision: 1.1 $=~/\d+/g); sprintf("%d".".%02d" x $#r, @r); };

=pod

=head1 NAME

Net::DRI::Protocol::EPP::Extensions::CentralNic::TTL - EPP DNS TTL CentralNic extension commands for Net::DRI

=head1 DESCRIPTION

Please see the README file for details.

=head1 SUPPORT

For now, support questions should be sent to:

E<lt>netdri@dotandco.comE<gt>

Please also see the SUPPORT file in the distribution.

=head1 SEE ALSO

E<lt>http://www.dotandco.com/services/software/Net-DRI/E<gt>

=head1 AUTHOR

Patrick Mevzek, E<lt>netdri@dotandco.comE<gt>

=head1 COPYRIGHT

Copyright (c) 2007 Patrick Mevzek <netdri@dotandco.com>.
All rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

See the LICENSE file that comes with this distribution for more details.

=cut

####################################################################################################

sub register_commands
{
 my ($class,$version)=@_;
 my %tmp=( create => [ \&create ],
            update => [ \&update ],
            info   => [ undef, \&info_parse ],
          );
 return { 'domain' => \%tmp };
}

sub capabilities_add
{
 return { 'domain_update' => { 'ttl' => ['set'] } };
}

our @NS=qw/urn:centralnic:params:xml:ns:ttl-1.0 ttl-1.0.xsd/;

####################################################################################################
########### Query commands

sub info_parse
{
 my ($po,$otype,$oaction,$oname,$rinfo)=@_;
 my $mes=$po->message();
 return unless $mes->is_success();

 my $infdata=$mes->get_content('infData',$NS[0],1);
 return unless $infdata;

 my @secs=$infdata->getElementsByTagNameNS($NS[0],'secs');
 return unless @secs;

 $rinfo->{domain}->{$oname}->{ttl}=DateTime::Duration->new(seconds => $secs[0]->getFirstChild()->getData());
}

############ Transform commands

sub create
{
 my ($epp,$domain,$rd)=@_;
 my $mes=$epp->message();

 return unless (exists($rd->{ttl}) && ((ref($rd->{ttl}) && UNIVERSAL::isa($rd->{ttl},'DateTime::Duration')) || $rd->{ttl}=~m/^\d+$/));

 my $eid=$mes->command_extension_register('ttl:create',sprintf('xmlns:ttl="%s" xsi:schemaLocation="%s %s"',$NS[0],@NS));
 my @n=(['ttl:secs',ref($rd->{ttl})? $rd->{ttl}->in_units('seconds') : $rd->{ttl}]);
 $mes->command_extension($eid,\@n);
}

sub update
{
 my ($epp,$domain,$todo)=@_;
 my $mes=$epp->message();

 my $toset=$todo->set('ttl');
 return unless (defined($toset) && ((ref($toset) && UNIVERSAL::isa($toset,'DateTime::Duration')) || $toset=~m/^\d+$/));

 my $eid=$mes->command_extension_register('ttl:update',sprintf('xmlns:ttl="%s" xsi:schemaLocation="%s %s"',$NS[0],@NS));
 my @n=(['ttl:secs',ref($toset)? $toset->in_units('seconds') : $toset]);
 $mes->command_extension($eid,\@n);
}

####################################################################################################
1;
