/*  You may distribute under the terms of either the GNU General Public License
 *  or the Artistic License (the same terms as Perl itself)
 *
 *  (C) Paul Evans, 2019-2020 -- leonerd@leonerd.org.uk
 */

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "XSParseSublike.h"

#define HAVE_PERL_VERSION(R, V, S) \
    (PERL_REVISION > (R) || (PERL_REVISION == (R) && (PERL_VERSION > (V) || (PERL_VERSION == (V) && (PERL_SUBVERSION >= (S))))))

#if HAVE_PERL_VERSION(5, 31, 3)
#  define HAVE_PARSE_SUBSIGNATURE
#elif HAVE_PERL_VERSION(5, 26, 0)
#  include "parse_subsignature.c.inc"
#  define HAVE_PARSE_SUBSIGNATURE
#endif

#if !HAVE_PERL_VERSION(5, 22, 0)
#  include "block_start.c.inc"
#  include "block_end.c.inc"
#endif

#ifndef wrap_keyword_plugin
#  include "wrap_keyword_plugin.c.inc"
#endif

#include "lexer-additions.c.inc"

/* Support two sets of hooks so we can handle xs_parse_sublike_any() with one
 * set which then finds a custom keyword which provides a second
 * Either or both may be NULL
 */
static int parse2(pTHX_ const struct XSParseSublikeHooks *hooksA, const struct XSParseSublikeHooks *hooksB, OP **op_ptr)
{
  SV *name = lex_scan_ident();
  lex_read_space(0);

  ENTER_with_name("parse_block");
  /* From here onwards any `return` must be prefixed by LEAVE_with_name() */

  I32 floor_ix = start_subparse(FALSE, name ? 0 : CVf_ANON);
  SAVEFREESV(PL_compcv);

  OP *attrs = NULL;
  if(lex_peek_unichar(0) == ':') {
    lex_read_unichar(0);

    attrs = lex_scan_attrs(PL_compcv);
  }

  PL_hints |= HINT_LOCALIZE_HH;
  I32 save_ix = block_start(TRUE);

  if(hooksA && hooksA->post_blockstart)
    (*hooksA->post_blockstart)(aTHX);
  if(hooksB && hooksB->post_blockstart)
    (*hooksB->post_blockstart)(aTHX);

#ifdef HAVE_PARSE_SUBSIGNATURE
  OP *sigop = NULL;
  if(lex_peek_unichar(0) == '(') {
    lex_read_unichar(0);

    sigop = parse_subsignature(0);
    lex_read_space(0);

    if(PL_parser->error_count) {
      LEAVE_with_name("parse_block");
      return 0;
    }

    if(lex_peek_unichar(0) != ')')
      croak("Expected ')'");
    lex_read_unichar(0);
    lex_read_space(0);
  }
#endif

  OP *body = parse_block(0);
  SvREFCNT_inc(PL_compcv);

#ifdef HAVE_PARSE_SUBSIGNATURE
  if(sigop)
    body = op_append_list(OP_LINESEQ, sigop, body);
#endif

  if(PL_parser->error_count) {
    /* parse_block() still sometimes returns a valid body even if a parse
     * error happens.
     * We need to destroy this partial body before returning a valid(ish)
     * state to the keyword hook mechanism, so it will find the error count
     * correctly
     *   See https://rt.cpan.org/Ticket/Display.html?id=130417
     */
    op_free(body);
    *op_ptr = newOP(OP_NULL, 0);
    if(name) {
      SvREFCNT_dec(name);
      LEAVE_with_name("parse_block");
      return KEYWORD_PLUGIN_STMT;
    }
    else {
      LEAVE_with_name("parse_block");
      return KEYWORD_PLUGIN_EXPR;
    }
  }

  if(hooksB && hooksB->pre_blockend)
    body = (*hooksB->pre_blockend)(aTHX_ body);
  if(hooksA && hooksA->pre_blockend)
    body = (*hooksA->pre_blockend)(aTHX_ body);

  body = block_end(save_ix, body);

  CV *cv = newATTRSUB(floor_ix,
    name ? newSVOP(OP_CONST, 0, SvREFCNT_inc(name)) : NULL,
    NULL,
    attrs,
    body);

  if(hooksA && hooksA->post_newcv)
    (*hooksA->post_newcv)(aTHX_ cv);
  if(hooksB && hooksB->post_newcv)
    (*hooksB->post_newcv)(aTHX_ cv);

  LEAVE_with_name("parse_block");

  if(name) {
    *op_ptr = newOP(OP_NULL, 0);

    SvREFCNT_dec(name);
    return KEYWORD_PLUGIN_STMT;
  }
  else {
    *op_ptr = newUNOP(OP_REFGEN, 0,
      newSVOP(OP_ANONCODE, 0, (SV *)cv));

    return KEYWORD_PLUGIN_EXPR;
  }
}

static int IMPL_xs_parse_sublike(pTHX_ const struct XSParseSublikeHooks *hooks, OP **op_ptr)
{
  return parse2(aTHX_ hooks, NULL, op_ptr);
}

static HV *registered_hooks;

static void IMPL_register_xs_parse_sublike(pTHX_ const char *kw, const struct XSParseSublikeHooks *hooks)
{
  if(!registered_hooks)
    registered_hooks = newHV();

  /* TODO: Check for clashes */
  hv_store(registered_hooks, kw, strlen(kw), newSVuv(PTR2UV(hooks)), 0);
}

static int IMPL_xs_parse_sublike_any(pTHX_ const struct XSParseSublikeHooks *hooksA, OP **op_ptr)
{
  SV *kwsv = lex_scan_ident();
  if(!kwsv || !SvCUR(kwsv))
    croak("Expected a keyword to introduce a sub or sub-like construction");

  const char *kw = SvPV_nolen(kwsv);
  STRLEN kwlen = SvCUR(kwsv);

  lex_read_space(0);

  const struct XSParseSublikeHooks *hooksB = NULL;
  /* We permit 'sub' as a NULL set of hooks; anything else should be a registered keyword */
  if(kwlen != 3 || !strEQ(kw, "sub")) {
    SV **svp = registered_hooks ? hv_fetch(registered_hooks, kw, kwlen, 0) : NULL;
    if(!svp)
      croak("Expected a keyword to introduce a sub or sub-like construction, found \"%.*s\"",
        kwlen, kw);

    hooksB = INT2PTR(const struct XSParseSublikeHooks *, SvUV(*svp));

    if(hooksB->permit && !(*hooksB->permit)(aTHX))
      croak("sub-like keyword \"%.*s\" is not permitted",
        kwlen, kw);
  }

  SvREFCNT_dec(kwsv);

  return parse2(aTHX_ hooksA, hooksB, op_ptr);
}

static int (*next_keyword_plugin)(pTHX_ char *, STRLEN, OP **);

static int my_keyword_plugin(pTHX_ char *kw, STRLEN kwlen, OP **op_ptr)
{
  SV **svp;
  const struct XSParseSublikeHooks *hooks;

  if(!registered_hooks ||
     !(svp = hv_fetch(registered_hooks, kw, kwlen, 0)))
    return (*next_keyword_plugin)(aTHX_ kw, kwlen, op_ptr);

  hooks = INT2PTR(const struct XSParseSublikeHooks *, SvUV(*svp));

  if(hooks->permit && !(*hooks->permit)(aTHX))
    return (*next_keyword_plugin)(aTHX_ kw, kwlen, op_ptr);

  lex_read_space(0);

  return parse2(aTHX_ NULL, hooks, op_ptr);
}

MODULE = XS::Parse::Sublike    PACKAGE = XS::Parse::Sublike

BOOT:
  sv_setiv(get_sv("XS::Parse::Sublike::ABIVERSION", GV_ADD), XSPARSESUBLIKE_ABI_VERSION);
  sv_setuv(get_sv("XS::Parse::Sublike::PARSE",      GV_ADD), PTR2UV(&IMPL_xs_parse_sublike));
  sv_setuv(get_sv("XS::Parse::Sublike::REGISTER",   GV_ADD), PTR2UV(&IMPL_register_xs_parse_sublike));
  sv_setuv(get_sv("XS::Parse::Sublike::PARSEANY",   GV_ADD), PTR2UV(&IMPL_xs_parse_sublike_any));

  wrap_keyword_plugin(&my_keyword_plugin, &next_keyword_plugin);
