#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2023 -- leonerd@leonerd.org.uk

package XS::Parse::Keyword::FromPerl 0.01;

use v5.14;
use warnings;

require XSLoader;
XSLoader::load( __PACKAGE__, our $VERSION );

=head1 NAME

C<XS::Parse::Keyword::FromPerl> - drive C<XS::Parse::Keyword> directly from Perl

=head1 DESCRIPTION

This module provides a Perl-visible API wrapping (some of) the functionality
provided by L<XS::Parse::Keyword>, allowing extension keywords to be added to
the Perl language by writing code in Perl itself.

It provides a thin wrapping later over the XS functions provided by XPK
itself, and additionally provides some extra perl-visible functions for things
like optree management, which would normally be written directly in C code. No
real attempt is made here to provide further abstractions on top of the API
already provided by Perl and XPK, so users will have to be familiar with the
overall concepts there as well.

This module is currently experimental, on top of the already-experimental
nature of C<XS::Parse::Keyword> itself.

=cut

require B; # for the B::OP classes

use Exporter 'import';
push our @EXPORT_OK, qw(
   opcode
   newOP
   newUNOP
   newBINOP
   newSVOP
   newGVOP

   register_xs_parse_keyword
);

=head1 UTILITY FUNCTIONS

The following helper functions are provided to allow Perl code to get access
to various parts of the C-level API that would be useful when building optrees
for keywords. They are not part of the C<XS::Parse::Keyword> API.

=head2 opcode

   $type = opcode( $opname );

Returns an opcode integer corresponding to the given op name, which should be
lowercase and without the leading C<OP_...> prefix. As this involves a linear
search across the entire C<PL_op_name> array you may wish to perform this just 
once and store the result, perhaps using C<use constant> for convenience.

   use constant OP_CONST => opcode("const");

=head2 new*OP

This family of functions return a new OP of the given class, for the type,
flags, and other arguments specified.

A suitable C<$type> can be obtained by using the L</opcode> function.

C<$flags> contains the opflags; a bitmask of the following constants.

   OPf_WANT OPf_WANT_VOID OPf_WANT_SCALAR OPf_WANT_LIST
   OPf_KIDS
   OPf_PARENS
   OPf_REF
   OPf_MOD
   OPf_STACKED
   OPf_SPECIAL

The op is returned as a C<B::OP> instance or a subclass thereof.

These functions can only be called during the C<build> phase of a keyword
hook, because they depend on having the correct context set by the
currently-compiling function.

=head3 newOP

   $op = newOP( $type, $flags );

Returns a new base OP for the given type and flags.

=head3 newUNOP

   $op = newUNOP( $type, $flags, $first );

Returns a new UNOP for the given type, flags, and first OP child.

=head3 newBINOP

   $op = newBINOP( $type, $flags, $first, $last );

Returns a new BINOP for the given type, flags, and first and last OP child.

=head3 newSVOP

   $op = newSVOP( $type, $flags, $sv );

Returns a new SVOP for the given type, flags, and SV. A copy of the given
scalar will be stored in the SVOP itself.

=head3 newGVOP

   $op = newGVOP( $type, $flags, $gvref );

Returns a new SVOP for the given type, flags, and GV given by a GLOB
reference. The referred-to GLOB will be stored in the SVOP itself.

=cut

=head1 XPK FUNCTIONS

=head2 register_xs_parse_keyword

   register_xs_parse_keyword "name" => %args;

Registers a new extension keyword into the C<XS::Parse::Keyword> registry,
defined using the given name and arguments.

Takes the following named arguments:

=over 4

=item flags => INT

Optional. If present, a bitmask of the following flag constants:

=over 4

=item XPK_FLAG_EXPR

The build phase is expected to return C<KEYWORD_PLUGIN_EXPR>.

=item XPK_FLAG_STMT

The build phase is expected to return C<KEYWORD_PLUGIN_STMT>.

=item XPK_FLAG_AUTOSEMI

The syntax forms a complete statement, which should be followed by C<;>.

=back

=item pieces => ARRAY

Optional. If present, contains definitions for the syntax pieces to be parsed
for the syntax of this keyword. At present this is not supported, so it must
be empty.

=item permit_hintkey => STRING

Optional. A string value to use for the "permit_hintkey".

=item permit => CODE

Optional. Callback function for the "permit" phase of keyword parsing.

   $ok = $permit->( $hookdata );

When invoked, it is passed a single arugment containing the (optional)
hookdata value, and its result should be a boolean scalar.

At least one of C<permit_hintkey> or C<permit> must be provided.

=item check => CODE

Optional. Callback function for the "check" phase of keyword parsing.

   $check->( $hookdata );

When invoked, it is passsed a single argument containing the (optional)
bookdata value.

=item build => CODE

Callback function for the "build" phase of keyword parsing.

   $ret = $build->( \$out, \@args, $hookdata );

When invoked, it is passed a SCALAR ref to store the output optree into, an
ARRAY reference containing the parsed arguments, and the (optional) hookdata
value.

At present, argument pieces are not supported, so the C<@args> array will
always be empty.

The callback function should be build an optree as a C<B::OP> fragment,
possibly by calling the various C<new*OP()> functions defined above, and store
the eventual result into the scalar referred to by the first argument.

The callback should return one of C<KEYWORD_PLUGIN_EXPR> or
C<KEYWORD_PLUGIN_STMT> to indicate how its syntax should be interpreted by the
perl parser.

=item hookdata => SCALAR

Optional. If present, this scalar value is stored by the keyword definition
and passed into each of the phase callbacks when invoked. If not present then
C<undef> will be passed to the callbacks instead.

=back

=cut

=head1 TODO

=over 4

=item *

C<pieces> for parsing.

=item *

More C<new*OP()> wrapper functions. Support LISTOPs - possibly by just passing
a list of child ops and doing the OP_LIST + op_convert_list dance so users
don't have to.

=item *

More optree-mangling functions. At least, some way to set the TARG might be
handy; or else some wrappers like hax/newPADxVOP.

=back

=cut

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
