#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2009-2016 -- leonerd@leonerd.org.uk

package Tickit::Event;

use strict;
use warnings;

our $VERSION = '0.58';

use Carp;

=head1 NAME

C<Tickit::Event> - event information structures

=head1 DESCRIPTION

When event handlers bound to L<Tickit::Term> instances are invoked they
receive an object instance to contain information about the event. Details of
the event can be accessed as via accessor methods on these instances.

For back-compatibility, each event type also provides a hash reference
overload method that yields a hash containing keys named after the accessors.
This code will be removed in a later version as per deprecation policy.

=head1 ACCESSORS

The following methods are shared between C<Tickit::Event::Key> and
C<Tickit::Event::Mouse> instances.

=head2 mod_is_alt

=head2 mod_is_ctrl

=head2 mod_is_shift

Convenient shortcuts to tests on the C<mod> bitmask to test if each of the
modifiers is set.

=cut

sub mod_is_alt   { shift->mod & Tickit::Term::MOD_ALT }
sub mod_is_ctrl  { shift->mod & Tickit::Term::MOD_CTRL }
sub mod_is_shift { shift->mod & Tickit::Term::MOD_SHIFT }

package
   Tickit::Event::Key;
our @ISA = qw( Tickit::Event );

use overload '%{}' => sub {
   my $self = shift;
   return +{ map { $_ => $self->$_ } qw( type str mod ) };
};

=head1 Tickit::Event::Key

=head2 type

A dualvar giving the key event type as an integer or string event name (C<text> or C<key>).

=head2 str

A string containing the key event string.

=head2 mod

An integer bitmask indicating the modifier state.

=cut

package
   Tickit::Event::Mouse;
our @ISA = qw( Tickit::Event );

use overload '%{}' => sub {
   my $self = shift;
   return +{ map { $_ => $self->$_ } qw( type button line col mod ) };
};

=head1 Tickit::Event::Mouse

=head2 type

A dualvar giving the mouse event type as an integer or string event name (C<press>, C<drag>, C<release> or C<wheel>).

=head2 button

An integer for non-wheel events or a dualvar for wheel events giving the
wheel direction (C<up> or C<down>).

=head2 line

=head2 col

Integers giving the mouse position.

=head2 mod

An integer bitmask indicating the modifier state.

=cut

package
   Tickit::Event::Resize;
our @ISA = qw( Tickit::Event );

use overload '%{}' => sub {
   my $self = shift;
   return +{ map { $_ => $self->$_ } qw( lines cols ) };
};

=head1 Tickit::Event::Resize

=head2 lines

=head2 cols

Integers giving the new size.

=cut

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
