#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2016 -- leonerd@leonerd.org.uk

package Net::Prometheus::Metric;

use strict;
use warnings;

our $VERSION = '0.02';

use Carp;
our @CARP_NOT = qw( Net::Prometheus );

use Net::Prometheus::Types qw( Sample MetricSamples );

=head1 NAME

C<Net::Prometheus::Metric> - the base class for observed metrics

=head1 DESCRIPTION

This class provides the basic methods shared by the concrete subclasses,

=over 2

=item *

L<Net::Prometheus::Gauge>

=item *

L<Net::Prometheus::Counter>

=item *

L<Net::Prometheus::Summary>

=back

=cut

sub make_childlabel_method
{
   my $class = shift;
   my ( $method ) = @_;

   no strict 'refs';
   *{"${class}::${method}"} = sub {
      my $self = shift;
      $self->labels( splice @_, 0, $self->labelcount )->$method( @_ );
   };
}

=head1 CONSTRUCTOR

=cut

=head2 new

   $metric = Net::Prometheus::Metric->new(
      name => $name,
      help => $help,
   )

The constructor is not normally used directly by instrumented code. Instead it
is more common to use one of the C<new_*> methods on the containing
L<Net::Prometheus> client instance.

   $metric = $prometheus->new_counter(
      name => $name,
      help => $help,
   )

In either case, it returns a newly-constructed metric.

Takes the following named arguments:

=over

=item namespace => STR

=item subsystem => STR

Optional strings giving the namespace and subsystem name parts of the variable
name.

=item name => STR

The basename of the exported variable.

=item help => STR

Descriptive help text for the variable.

=item labels => ARRAY of STR

Optional ARRAY reference giving the names of labels for the metric.

=back

=cut

sub new
{
   my $class = shift;
   my %args = @_;

   defined $args{name} or
      croak "Required 'name' argument missing";
   defined $args{help} or
      croak "Required 'help' argument missing";

   my $fullname = join "_", grep { defined } $args{namespace}, $args{subsystem}, $args{name};

   my $labellist = $args{labels} || [];

   # See
   #   https://prometheus.io/docs/concepts/data_model/#metric-names-and-labels
   $fullname =~ m/^[a-zA-Z_:][a-zA-Z0-9_:]*$/ or
      croak "Invalid metric name '$fullname'";

   $_ =~ m/^[a-zA-Z_][a-zA-Z0-9_]*$/ or
      croak "Invalid label name '$_'" for @$labellist;
   $_ =~ m/^__/ and
      croak "Label name '$_' is reserved" for @$labellist;

   return bless {
      fullname => $fullname,
      help => $args{help},
      labels => $labellist,
      labelvalues => {},
   }, $class;
}

=head1 METHODS

=cut

=head2 fullname

   $fullname = $metric->fullname

Returns the full name for the metric. This is formed by joining any of the
defined values for C<namespace>, C<subsystem> and C<name> with C<'_'>.

=cut

sub fullname
{
   my $self = shift;
   return $self->{fullname};
}

=head2 labelcount

   $labels = $metric->labelcount

Returns the number of labels defined for this metric.

=cut

sub labelcount
{
   my $self = shift;
   return scalar @{ $self->{labels} };
}

=head2 labels

   $child = $metric->labels( @values )

Returns a child metric to represent the general one with the given set of
labels. This object may be cached for efficiency.

The child instance supports the same methods to control the value of the
reported metric as the parent metric object, except that any label values are
already provided.

=cut

sub labels
{
   my $self = shift;
   my @values = @_;

   my $labelcount = $self->labelcount;
   @values >= $labelcount or
      croak "Insufficient values given for labels";
   @values == $labelcount or
      croak "Too many values given for labels";

   length $values[$_] or
      croak "Value for $self->{labels}[$_] may not empty" for 0 .. $#values;

   my $labelkey = join "\x00", map {
      # Encode \x00 or \x01 as \x{01}0 or \x{01}1 in order to escape the \x00
      # but preserve full leixcal ordering
      my $value = $_;
      $value =~ s/\x01/\x011/g;
      $value =~ s/\x00/\x010/g;
      $value;
   } @values;

   $self->{labelvalues}{$labelkey} = \@values;

   return Net::Prometheus::Metric::_Child->new(
      $self, $labelkey
   );
}

=head2 make_sample

   $sample = $metric->make_sample( $suffix, $labelkey, $value )

Returns a new L<Net::Prometheus::Types/Sample> structure to represent the
given value, by expanding the opaque C<$labelkey> value into its actual label
names and values and appending the given suffix (which may be an empty string)
to the metric's fullname. If provided, the suffix will be separated by an
underscore C<'_'>.

=cut

sub make_sample
{
   my $self = shift;
   my ( $suffix, $labelkey, $value ) = @_;

   my $labelnames  = $self->{labels};
   my $labelvalues = $self->{labelvalues}{$labelkey};

   return Sample(
      ( $suffix ? $self->fullname . "_$suffix" : $self->fullname ),
      [ map { $labelnames->[$_], $labelvalues->[$_] } 0 .. $#$labelnames ],
      $value,
   );
}

sub collect
{
   my $self = shift;

   return MetricSamples(
      $self->fullname, $self->_type, $self->{help},
      [ $self->samples ],
   );
}

=head2 samples

   @samples = $metric->samples

An abstract method in this class, this method is intended to be overriden by
subclasses.

Called during the value collection process, this method should return a list
of L<Net::Prometheus::Types/Sample> instances containing the values to report
from this metric.

=cut

sub samples
{
   croak "Abstract Net::Prometheus::Metric->samples invoked directly";
}

package
   Net::Prometheus::Metric::_Child;

use constant {
   METRIC   => 0,
   LABELKEY => 1,
};

sub new
{
   my $class = shift;
   my ( $metric, $labelkey ) = @_;
   return bless [ $metric, $labelkey ], $class;
}

sub metric   { shift->[METRIC] }
sub labelkey { shift->[LABELKEY] }

foreach my $method (qw( set inc dec observe )) {
   my $childmethod = "_${method}_child";

   no strict 'refs';
   *$method = sub {
      my $self = shift;
      $self->[METRIC]->$childmethod( $self->[LABELKEY], @_ );
   };
}

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
