package Bencher::Scenario::TreeObject::Build;

our $DATE = '2016-04-14'; # DATE
our $VERSION = '0.04'; # VERSION

use 5.010001;
use strict;
use warnings;

use Bencher::ScenarioUtil::TreeObject;
use Tree::Create::Size;

my $classes = \%Bencher::ScenarioUtil::TreeObject::classes;

our $scenario = {
    summary => 'Benchmark tree building using Tree::Create::Size',
    include_result_size => 1,
    participants => [
        map {
            my $spec = $classes->{$_};
            +{
                module => $_,
                code_template => "Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => '".($spec->{use_my_class} ? "My::$_":$_)."')",
            }
        } sort keys %$classes,
    ],
    datasets => \@Bencher::ScenarioUtil::TreeObject::trees_datasets,
};

1;
# ABSTRACT: Benchmark tree building using Tree::Create::Size

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::TreeObject::Build - Benchmark tree building using Tree::Create::Size

=head1 VERSION

This document describes version 0.04 of Bencher::Scenario::TreeObject::Build (from Perl distribution Bencher-Scenarios-TreeObject), released on 2016-04-14.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m TreeObject::Build

To run module startup overhead benchmark:

 % bencher --module-startup -m TreeObject::Build

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Tree::Object::Array> 0.07

L<Tree::Object::Array::Glob> 0.07

L<Tree::Object::Hash> 0.07

L<Tree::Object::Hash::ChildrenAsList> 0.07

L<Tree::Object::InsideOut> 0.07

L<Tree::ObjectXS::Array> 0.02

L<Tree::ObjectXS::Hash> 0.02

=head1 BENCHMARK PARTICIPANTS

=over

=item * Tree::Object::Array (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'My::Tree::Object::Array')



=item * Tree::Object::Array::Glob (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'My::Tree::Object::Array::Glob')



=item * Tree::Object::Hash (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'Tree::Object::Hash')



=item * Tree::Object::Hash::ChildrenAsList (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'Tree::Object::Hash::ChildrenAsList')



=item * Tree::Object::InsideOut (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'Tree::Object::InsideOut')



=item * Tree::ObjectXS::Array (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'My::Tree::ObjectXS::Array')



=item * Tree::ObjectXS::Hash (perl_code)

Code template:

 Tree::Create::Size::create_tree(height => <height>, num_children => <num_children>, class => 'Tree::ObjectXS::Hash')



=back

=head1 BENCHMARK DATASETS

=over

=item * tiny1 (3 nodes)

A tree with height=1 and 2 children per non-leaf nodes, nodes=1 + 2 = 3

=item * small1 (31 nodes)

A tree with height=4 and 2 children per non-leaf nodes, nodes=1 + 2 + 4 + 8 + 16 = 31

=item * small2 (364 nodes)

A tree with height=5 and 3 children per non-leaf nodes, nodes=1 + 3 + 9 + 27 + 81 + 243 = 364

=item * small3 (1365 nodes)

A tree with height=5 and 4 children per non-leaf nodes, nodes=1 + 4 + 16 + 64 + 256 + 1024 = 1365

=item * medium1 (19531 nodes)

A tree with height=6 and 5 children per non-leaf nodes, nodes=1 + 5 + 25 + 125 + 625 + 3125 + 15625 = 19531

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with C<< bencher -m TreeObject::Build --include-datasets 'small1 (31 nodes)' >>:

 +------------------------------------+-----------+-----------+------------+------------------+---------+---------+
 | participant                        | rate (/s) | time (μs) | vs_slowest | result_size (kB) | errors  | samples |
 +------------------------------------+-----------+-----------+------------+------------------+---------+---------+
 | Tree::Object::InsideOut            | 1.1e+04   | 93        | 1          | 0.14             | 3.2e-07 | 20      |
 | Tree::Object::Hash::ChildrenAsList | 1.6e+04   | 62        | 1.5        | 9.7              | 1e-07   | 21      |
 | Tree::Object::Hash                 | 1.7e+04   | 60        | 1.5        | 9.7              | 8e-08   | 20      |
 | Tree::Object::Array::Glob          | 1.8e+04   | 56        | 1.7        | 3.8              | 1.1e-07 | 20      |
 | Tree::Object::Array                | 2e+04     | 50        | 1.9        | 6.1              | 1.6e-07 | 21      |
 | Tree::ObjectXS::Hash               | 2.4e+04   | 42        | 2.2        | 7.9              | 4.9e-08 | 24      |
 | Tree::ObjectXS::Array              | 2.4e+04   | 41        | 2.3        | 6.1              | 5.3e-08 | 20      |
 +------------------------------------+-----------+-----------+------------+------------------+---------+---------+


Benchmark with C<< bencher -m TreeObject::Build --include-datasets 'medium1 (19531 nodes)' >>:

 +------------------------------------+-----------+-----------+------------+------------------+---------+---------+
 | participant                        | rate (/s) | time (ms) | vs_slowest | result_size (MB) | errors  | samples |
 +------------------------------------+-----------+-----------+------------+------------------+---------+---------+
 | Tree::Object::InsideOut            | 18        | 57        | 1          | 0.00014          | 0.00012 | 20      |
 | Tree::Object::Hash::ChildrenAsList | 30        | 33        | 1.7        | 5.8              | 3.7e-05 | 20      |
 | Tree::Object::Hash                 | 30.9      | 32.4      | 1.76       | 5.81             | 8.7e-06 | 21      |
 | Tree::Object::Array::Glob          | 35.1      | 28.5      | 2          | 2.38             | 1.7e-05 | 21      |
 | Tree::Object::Array                | 39.7      | 25.2      | 2.26       | 3.16             | 7.2e-06 | 21      |
 | Tree::ObjectXS::Hash               | 46.7      | 21.4      | 2.66       | 4.14             | 9.9e-06 | 20      |
 | Tree::ObjectXS::Array              | 48        | 20.8      | 2.73       | 3.16             | 4.8e-06 | 21      |
 +------------------------------------+-----------+-----------+------------+------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-TreeObject>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-TreeObject>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-TreeObject>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
