package
    main;

use List::Util 'first';

package Bencher::Scenario::PERLANCAR::In;

our $DATE = '2016-04-13'; # DATE
our $VERSION = '0.04'; # VERSION

use 5.010001;
use strict;
use warnings;

@main::ary_100  = (1..100);
@main::ary_10k  = (1..10_000);
@main::ary_1mil = (1..1000_000);

our $scenario = {
    summary => 'Benchmark the task of checking whether an item is in an array',
    participants => [
        {
            name => 'grep',
            code_template => 'grep { $_ <op:raw> <needle> } @main::<haystack:raw>',
        },
        {
            name => 'first',
            code_template => 'first { $_ <op:raw> <needle> } @main::<haystack:raw>',
        },
        {
            name => 'first (array)',
            fcall_template => 'Array::AllUtils::first(sub { $_ <op:raw> <needle> }, \\@main::<haystack:raw>)',
        },
        {
            name => 'smartmatch',
            code_template => 'use experimental "smartmatch"; <needle> ~~ @main::<haystack:raw>',
        },
    ],
    datasets => [
        {name => '100 items' , args => {'haystack'=>'ary_100' , op => '==', 'needle@' => [1, 50, 100]}},
        {name => '10k' , args => {'haystack'=>'ary_10k' , op => '==', 'needle@' => [1, 5000, 10_000]}},
        {name => '1mil', args => {'haystack'=>'ary_1mil', op => '==', 'needle@' => [1, 500_000, 1000_000]}},
    ],
};

1;
# ABSTRACT:

__END__

=pod

=encoding UTF-8

=head1 NAME

main

=head1 VERSION

This document describes version 0.04 of main (from Perl distribution Bencher-Scenarios-PERLANCAR), released on 2016-04-13.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m PERLANCAR::In

To run module startup overhead benchmark:

 % bencher --module-startup -m PERLANCAR::In

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Array::AllUtils> 0.001

=head1 BENCHMARK PARTICIPANTS

=over

=item * grep (perl_code)

Code template:

 grep { $_ <op:raw> <needle> } @main::<haystack:raw>



=item * first (perl_code)

Code template:

 first { $_ <op:raw> <needle> } @main::<haystack:raw>



=item * first (array) (perl_code)

Function call template:

 Array::AllUtils::first(sub { $_ <op:raw> <needle> }, \@main::<haystack:raw>)



=item * smartmatch (perl_code)

Code template:

 use experimental "smartmatch"; <needle> ~~ @main::<haystack:raw>



=back

=head1 BENCHMARK DATASETS

=over

=item * 100 items

=item * 10k

=item * 1mil

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m PERLANCAR::In >>):

 +---------------+-----------+------------+------------+-----------+------------+---------+---------+
 | participant   | dataset   | arg_needle | rate (/s)  | time (ms) | vs_slowest | errors  | samples |
 +---------------+-----------+------------+------------+-----------+------------+---------+---------+
 | first (array) | 1mil      | 1000000    | 6          | 167       | 1          | 7.8e-05 | 20      |
 | first (array) | 1mil      | 500000     | 12         | 83.3      | 2          | 2.9e-05 | 20      |
 | grep          | 1mil      | 1          | 42.1       | 23.7      | 7.03       | 1.5e-05 | 20      |
 | grep          | 1mil      | 500000     | 42.3       | 23.6      | 7.06       | 5.3e-06 | 20      |
 | grep          | 1mil      | 1000000    | 42.4       | 23.6      | 7.06       | 4.7e-06 | 20      |
 | first         | 1mil      | 1000000    | 48.3       | 20.7      | 8.05       | 1.3e-05 | 20      |
 | smartmatch    | 1mil      | 1000000    | 64.2       | 15.6      | 10.7       | 2.9e-06 | 20      |
 | first         | 1mil      | 500000     | 83         | 12        | 14         | 1.6e-05 | 20      |
 | smartmatch    | 1mil      | 500000     | 129        | 7.78      | 21.4       | 1.1e-06 | 20      |
 | first         | 1mil      | 1          | 3.1e+02    | 3.3       | 51         | 5e-06   | 21      |
 | first (array) | 10k       | 10000      | 603        | 1.66      | 101        | 4.8e-07 | 20      |
 | first (array) | 10k       | 5000       | 1.2e+03    | 0.84      | 2e+02      | 8.5e-07 | 20      |
 | grep          | 10k       | 1          | 4349.97    | 0.229887  | 725.296    | 1.9e-10 | 20      |
 | grep          | 10k       | 5000       | 4388.386   | 0.2278742 | 731.7022   | 1.1e-11 | 20      |
 | grep          | 10k       | 10000      | 4398.049   | 0.2273735 | 733.3134   | 1.1e-11 | 20      |
 | first         | 10k       | 10000      | 5345.281   | 0.1870809 | 891.2512   | 1.2e-11 | 20      |
 | smartmatch    | 10k       | 10000      | 6412.612   | 0.1559427 | 1069.214   | 1e-11   | 23      |
 | first         | 10k       | 5000       | 9.97e+03   | 0.1       | 1.66e+03   | 2.7e-08 | 20      |
 | smartmatch    | 10k       | 5000       | 12838      | 0.077896  | 2140.5     | 9.3e-11 | 20      |
 | first (array) | 100 items | 100        | 59091.5    | 0.0169229 | 9852.69    | 1e-11   | 20      |
 | first         | 10k       | 1          | 7.5e+04    | 0.013     | 1.2e+04    | 2.4e-08 | 25      |
 | first (array) | 100 items | 50         | 1.16e+05   | 0.00859   | 1.94e+04   | 3.3e-09 | 20      |
 | grep          | 100 items | 50         | 4.189e+05  | 0.002387  | 6.985e+04  | 9.3e-11 | 20      |
 | grep          | 100 items | 100        | 4.2112e+05 | 0.0023746 | 70216      | 1.1e-11 | 20      |
 | grep          | 100 items | 1          | 4.214e+05  | 0.002373  | 7.026e+04  | 4.7e-11 | 20      |
 | first         | 100 items | 100        | 4.96e+05   | 0.00202   | 8.26e+04   | 6.9e-10 | 29      |
 | smartmatch    | 100 items | 100        | 6.04e+05   | 0.00166   | 1.01e+05   | 8.3e-10 | 20      |
 | first         | 100 items | 50         | 8.735e+05  | 0.001145  | 1.456e+05  | 1.2e-11 | 20      |
 | smartmatch    | 100 items | 50         | 1.168e+06  | 0.0008562 | 1.947e+05  | 3.5e-11 | 20      |
 | first (array) | 10k       | 1          | 2.365e+06  | 0.0004229 | 3.942e+05  | 1.2e-11 | 20      |
 | first (array) | 1mil      | 1          | 2.37e+06   | 0.000422  | 3.95e+05   | 4.6e-11 | 20      |
 | first (array) | 100 items | 1          | 2.368e+06  | 0.0004222 | 3.949e+05  | 3.4e-11 | 20      |
 | first         | 100 items | 1          | 3.592e+06  | 0.0002784 | 5.99e+05   | 1.1e-11 | 20      |
 | smartmatch    | 1mil      | 1          | 1.39e+07   | 7.17e-05  | 2.32e+06   | 3.5e-11 | 22      |
 | smartmatch    | 100 items | 1          | 1.39e+07   | 7.17e-05  | 2.33e+06   | 4.7e-11 | 26      |
 | smartmatch    | 10k       | 1          | 1.43e+07   | 6.99e-05  | 2.38e+06   | 1.1e-11 | 20      |
 +---------------+-----------+------------+------------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m PERLANCAR::In --module-startup >>):

 +---------------------+-----------+------------------------+------------+---------+---------+
 | participant         | time (ms) | mod_overhead_time (ms) | vs_slowest | errors  | samples |
 +---------------------+-----------+------------------------+------------+---------+---------+
 | Array::AllUtils     | 6.9       | 2.9                    | 1          | 6.7e-05 | 20      |
 | perl -e1 (baseline) | 4         | 0                      | 1.7        | 7.6e-06 | 20      |
 +---------------------+-----------+------------------------+------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-PERLANCAR>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-PERLANCAR>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-PERLANCAR>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
