package Bencher::Scenario::LogGer::Overhead;

our $DATE = '2021-01-31'; # DATE
our $VERSION = '0.017'; # VERSION

use 5.010001;
use strict;
use warnings;

use File::Temp qw(tempfile);

my ($fh, $fname) = tempfile();

our $scenario = {
    summary => 'Measure startup overhead of various codes',
    modules => {
        'Log::Any' => {},
        'Log::ger' => {version=>'0.019'},
        'Log::ger::App' => {version=>'0.002'},
        'Log::ger::Output' => {version=>'0.005'},
        'Log::ger::Layout::Pattern' => {version=>'0'},
        'Log::Contextual' => {version=>'0'},
        'Log::Dispatch' => {version=>'0'},
        'Log::Dispatch::Null' => {version=>'0'},
        'Log::Log4perl' => {version=>'0'},
        'Log::Log4perl::Tiny' => {version=>'0'},
        'Log::Dispatchouli' => {version=>'0'},
        'Mojo::Log' => {version=>'0'},
    },
    code_startup => 1,
    participants => [
        # a benchmark for Log::ger: strict/warnings
        {code_template=>'use strict;'},
        {code_template=>'use warnings;'},
        {code_template=>'use strict; use warnings;'},

        {code_template=>'use Log::ger ();'},
        {code_template=>'use Log::ger;'},
        {code_template=>'use Log::ger; Log::ger->get_logger;'},
        {code_template=>'use Log::ger::App;'},
        {code_template=>'use Log::ger::App; use Log::ger;'},
        {code_template=>'use Log::ger::Plugin::OptAway; use Log::ger;'},
        {code_template=>'use Log::ger::Like::LogAny;'},
        {code_template=>'use Log::ger::Like::Log4perl;'},
        {code_template=>'use Log::ger::App;'},

        {code_template=>'use Log::Any;'},
        {code_template=>'use Log::Any q($log);'},

        {code_template=>'use Log::Contextual qw(:log);'},

        {code_template=>'use Log::Log4perl;'},

        {code_template=>'use Log::Log4perl::Tiny;'},

        {code_template=>'use Log::Dispatch;'},
        {code_template=>'use Log::Dispatch; my $null = Log::Dispatch->new(outputs=>[ ["Null", min_level=>"warn"] ])', tags=>['output']},

        {code_template=>'use Log::Dispatchouli;'},

        {code_template=>'use Log::ger::Output::Screen;', tags=>['output']},
        {code_template=>'use Log::ger::Output::Composite;', tags=>['output']},

        {code_template=>'use Mojo::Log;'},
        {code_template=>'use Mojo::Log; my $log=Mojo::Log->new(level=>"warn")'},

        # TODO: Lg + Composite (2 outputs)
        # TODO: Lg + Composite (2 outputs + pattern layouts)
        # TODO: Log::Any + Screen
        # TODO: Log::Log4perl + easy_init
    ],
};

1;
# ABSTRACT: Measure startup overhead of various codes

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LogGer::Overhead - Measure startup overhead of various codes

=head1 VERSION

This document describes version 0.017 of Bencher::Scenario::LogGer::Overhead (from Perl distribution Bencher-Scenarios-LogGer), released on 2021-01-31.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LogGer::Overhead

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Log::Any> 1.708

L<Log::Contextual> 0.008001

L<Log::Dispatch> 2.70

L<Log::Dispatch::Null> 2.70

L<Log::Dispatchouli> 2.022

L<Log::Log4perl> 1.53

L<Log::Log4perl::Tiny> 1.4.0

L<Log::ger> 0.038

L<Log::ger::App> 0.018

L<Log::ger::Layout::Pattern> 0.007

L<Log::ger::Output> 0.038

L<Mojo::Log>

=head1 BENCHMARK PARTICIPANTS

=over

=item * use strict; (perl_code)

Code template:

 use strict;



=item * use warnings; (perl_code)

Code template:

 use warnings;



=item * use strict; use warnings; (perl_code)

Code template:

 use strict; use warnings;



=item * use Log::ger (); (perl_code)

Code template:

 use Log::ger ();



=item * use Log::ger; (perl_code)

Code template:

 use Log::ger;



=item * use Log::ger; Log::ger->get_logger; (perl_code)

Code template:

 use Log::ger; Log::ger->get_logger;



=item * use Log::ger::App; (perl_code)

Code template:

 use Log::ger::App;



=item * use Log::ger::App; use Log::ger; (perl_code)

Code template:

 use Log::ger::App; use Log::ger;



=item * use Log::ger::Plugin::OptAway; use Log::ger; (perl_code)

Code template:

 use Log::ger::Plugin::OptAway; use Log::ger;



=item * use Log::ger::Like::LogAny; (perl_code)

Code template:

 use Log::ger::Like::LogAny;



=item * use Log::ger::Like::Log4perl; (perl_code)

Code template:

 use Log::ger::Like::Log4perl;



=item * use Log::ger::App; (perl_code)

Code template:

 use Log::ger::App;



=item * use Log::Any; (perl_code)

Code template:

 use Log::Any;



=item * use Log::Any q($log); (perl_code)

Code template:

 use Log::Any q($log);



=item * use Log::Contextual qw(:log); (perl_code)

Code template:

 use Log::Contextual qw(:log);



=item * use Log::Log4perl; (perl_code)

Code template:

 use Log::Log4perl;



=item * use Log::Log4perl::Tiny; (perl_code)

Code template:

 use Log::Log4perl::Tiny;



=item * use Log::Dispatch; (perl_code)

Code template:

 use Log::Dispatch;



=item * use Log::Dispatch; my $null = Log::Dispatch->new(outputs=>[ ["Nu (perl_code) [output]

Code template:

 use Log::Dispatch; my $null = Log::Dispatch->new(outputs=>[ ["Null", min_level=>"warn"] ])



=item * use Log::Dispatchouli; (perl_code)

Code template:

 use Log::Dispatchouli;



=item * use Log::ger::Output::Screen; (perl_code) [output]

Code template:

 use Log::ger::Output::Screen;



=item * use Log::ger::Output::Composite; (perl_code) [output]

Code template:

 use Log::ger::Output::Composite;



=item * use Mojo::Log; (perl_code)

Code template:

 use Mojo::Log;



=item * use Mojo::Log; my $log=Mojo::Log->new(level=>"warn") (perl_code)

Code template:

 use Mojo::Log; my $log=Mojo::Log->new(level=>"warn")



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 5.3.0-68-generic >>.

Benchmark with default options (C<< bencher -m LogGer::Overhead >>):

 #table1#
 +------------------------------------------------------------------+-----------+---------------------+-----------------------+-----------------------+-----------+---------+
 | participant                                                      | time (ms) |  code_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +------------------------------------------------------------------+-----------+---------------------+-----------------------+-----------------------+-----------+---------+
 | use Mojo::Log; my $log=Mojo::Log->new(level=>"warn")             |      85   | 80                  |                 0.00% |              1719.57% |   0.00017 |      24 |
 | use Mojo::Log;                                                   |      84   | 79                  |                 1.00% |              1701.58% |   0.00019 |      20 |
 | use Log::Dispatchouli;                                           |      67   | 62                  |                27.26% |              1329.84% |   0.00017 |      21 |
 | use Log::Dispatch; my $null = Log::Dispatch->new(outputs=>[ ["Nu |      64   | 59                  |                32.79% |              1270.24% |   0.00011 |      20 |
 | use Log::Dispatch;                                               |      60   | 55                  |                41.57% |              1185.27% |   0.00011 |      20 |
 | use Log::Contextual qw(:log);                                    |      57   | 52                  |                49.72% |              1115.33% | 9.9e-05   |      20 |
 | use Log::Log4perl;                                               |      29   | 24                  |               192.80% |               521.43% | 9.2e-05   |      20 |
 | use Log::ger::App; use Log::ger;                                 |      23   | 18                  |               263.20% |               400.98% | 6.9e-05   |      21 |
 | use Log::ger::App;                                               |      19   | 14                  |               348.71% |               305.51% | 3.8e-05   |      21 |
 | use Log::Log4perl::Tiny;                                         |      20   | 15                  |               446.58% |               232.90% |   0.00017 |      20 |
 | use Log::ger::Like::Log4perl;                                    |      13   |  8                  |               539.05% |               184.73% | 3.5e-05   |      21 |
 | use Log::Any q($log);                                            |      10   |  5                  |               641.14% |               145.51% |   0.00025 |      20 |
 | use Log::Any;                                                    |      11   |  6                  |               688.89% |               130.65% | 6.1e-05   |      22 |
 | use Log::ger::Output::Composite;                                 |      10   |  5                  |               724.33% |               120.73% |   0.00027 |      20 |
 | use Log::ger::Output::Screen;                                    |      10   |  5                  |               733.20% |               118.38% |   0.00021 |      20 |
 | use warnings;                                                    |       8   |  3                  |              1019.12% |                62.59% |   0.00012 |      20 |
 | use Log::ger::Plugin::OptAway; use Log::ger;                     |       7.2 |  2.2                |              1083.39% |                53.76% | 2.8e-05   |      23 |
 | use strict;                                                      |       7   |  2                  |              1208.49% |                39.06% | 9.5e-05   |      20 |
 | use strict; use warnings;                                        |       6   |  1                  |              1322.64% |                27.90% | 2.6e-05   |      20 |
 | use Log::ger::Like::LogAny;                                      |       5.4 |  0.4                |              1471.83% |                15.76% | 4.5e-05   |      21 |
 | use Log::ger;                                                    |       5   |  0                  |              1588.19% |                 7.78% | 7.6e-05   |      20 |
 | use Log::ger ();                                                 |       4.9 | -0.0999999999999996 |              1650.50% |                 3.95% | 2.7e-05   |      20 |
 | perl -e1 (baseline)                                              |       5   |  0                  |              1656.78% |                 3.57% | 7.1e-05   |      20 |
 | use Log::ger; Log::ger->get_logger;                              |       4.7 | -0.3                |              1719.57% |                 0.00% | 1.7e-05   |      21 |
 +------------------------------------------------------------------+-----------+---------------------+-----------------------+-----------------------+-----------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-LogGer>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-LogGer>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-LogGer>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2020, 2018, 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
