package Bencher::Scenario::Serializers;

our $DATE = '2016-01-19'; # DATE
our $VERSION = '0.07'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 bytes long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-bytes-long Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-byte long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers - Benchmark Perl data serialization modules

=head1 VERSION

This document describes version 0.07 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2016-01-19.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<JSON::PP>

L<JSON::XS>

L<Cpanel::JSON::XS>

L<JSON::MaybeXS>

L<JSON::Decode::Regexp> 1

L<JSON::Decode::Marpa> 1

L<Pegex::JSON> 1

L<JSON::Create>

L<JSON::Parse> 1

L<YAML::Old>

L<YAML::Syck>

L<YAML::XS>

L<Storable>

L<Sereal>

L<Data::MessagePack>

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::XS::encode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code)

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code)

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * Pegex::JSON (perl_code)

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * JSON::Create::create_json (perl_code)

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Parse::parse_json (perl_code)

Function call template:

 JSON::Parse::parse_json(<data>)



=item * YAML::Old::Dump (perl_code)

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Load (perl_code)

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Syck::Dump (perl_code)

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Load (perl_code)

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::XS::Dump (perl_code)

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Load (perl_code)

Function call template:

 YAML::XS::Load(<data>)



=item * Storable::freeze (perl_code)

Function call template:

 Storable::freeze(<data>)



=item * Storable::thaw (perl_code)

Function call template:

 Storable::thaw(<data>)



=item * Sereal::encode_sereal (perl_code)

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code)

Function call template:

 Sereal::decode_sereal(<data>)



=item * Data::MessagePack::pack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::unpack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=back

=head1 BENCHMARK DATASETS

=over

=item * undef

undef

=item * num

A single number (-1.23)

=item * str1k

A non-Unicode string 1024 bytes long

=item * str1k

A Unicode string 1024 bytes long

=item * array_int_10

A 10-element array containing ints

=item * array_int_100

A 100-element array containing ints

=item * array_int_1000

A 1000-element array containing ints

=item * array_str1k_10

A 10-element array containing 1024-bytes-long non-Unicode strings

=item * array_ustr1k_10

A 10-element array containing 1024-bytes-long Unicode strings

=item * hash_int_10

A 10-key hash {1=>0, ..., 10=>0}

=item * hash_int_100

A 100-key hash {1=>0, ..., 100=>0}

=item * hash_int_1000

A 1000-key hash {1=>0, ..., 1000=>0}

=item * json:null

null

=item * json:num

A single number (-1.23)

=item * json:str1k

A non-Unicode (ASCII) string 1024-byte long

=item * json:array_int_10

A 10-element array containing ints

=item * json:array_int_100

A 10-element array containing ints

=item * json:array_int_1000

A 1000-element array containing ints

=item * json:array_str1k_10

A 10-element array containing 1024-bytes-long non-Unicode strings

=item * json:hash_int_10

A 10-key hash {"1":0, ..., "10":0}

=item * json:hash_int_100

A 100-key hash {"1":0, ..., "100":0}

=item * json:hash_int_1000

A 1000-key hash {"1":0, ..., "1000":0}

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark serializing (C<< bencher --include-participant-tags-json '["serialize"]' >>):

 +-------------------------------+-----------------+------------+------------+---------+---------+
 | participant                   | dataset         | rate (/s)  | time (ms)  | errors  | samples |
 +-------------------------------+-----------------+------------+------------+---------+---------+
 | YAML::Old::Dump               | hash_int_1000   | 24.2       | 41.4       | 4.7e-05 | 20      |
 | YAML::Old::Dump               | array_int_1000  | 32         | 31.3       | 3.2e-05 | 20      |
 | JSON::PP::encode_json         | hash_int_1000   | 224        | 4.47       | 1.1e-05 | 20      |
 | YAML::Old::Dump               | hash_int_100    | 242        | 4.13       | 2.2e-06 | 20      |
 | YAML::Old::Dump               | array_int_100   | 314        | 3.19       | 4.9e-06 | 20      |
 | JSON::PP::encode_json         | array_int_1000  | 568        | 1.76       | 1.8e-06 | 20      |
 | YAML::Syck::Dump              | hash_int_1000   | 689        | 1.45       | 6.8e-07 | 21      |
 | YAML::XS::Dump                | hash_int_1000   | 698        | 1.43       | 3.8e-06 | 20      |
 | YAML::Old::Dump               | array_ustr1k_10 | 867        | 1.15       | 2.2e-06 | 20      |
 | YAML::Syck::Dump              | array_int_1000  | 1.6e+03    | 0.62       | 4.1e-06 | 25      |
 | YAML::XS::Dump                | array_int_1000  | 1756       | 0.5695     | 5.3e-08 | 20      |
 | JSON::PP::encode_json         | array_ustr1k_10 | 1.9e+03    | 0.527      | 2e-07   | 23      |
 | YAML::Old::Dump               | hash_int_10     | 2.02e+03   | 0.495      | 9.1e-07 | 20      |
 | YAML::Old::Dump               | array_str1k_10  | 2.04e+03   | 0.491      | 4.2e-07 | 21      |
 | JSON::PP::encode_json         | hash_int_100    | 2.16e+03   | 0.464      | 1.1e-06 | 20      |
 | YAML::Old::Dump               | array_int_10    | 2.59e+03   | 0.386      | 4.3e-07 | 20      |
 | YAML::XS::Dump                | array_ustr1k_10 | 3057       | 0.3271     | 5.2e-08 | 21      |
 | JSON::MaybeXS::encode_json    | array_ustr1k_10 | 3.34e+03   | 0.3        | 3.7e-07 | 33      |
 | Cpanel::JSON::XS::encode_json | array_ustr1k_10 | 3.58e+03   | 0.28       | 2.7e-07 | 20      |
 | JSON::XS::encode_json         | array_ustr1k_10 | 3.6e+03    | 0.278      | 2.1e-07 | 20      |
 | Sereal::encode_sereal         | hash_int_1000   | 3.72e+03   | 0.269      | 2.1e-07 | 20      |
 | Storable::freeze              | hash_int_1000   | 4.29e+03   | 0.233      | 5.5e-07 | 27      |
 | Data::MessagePack::pack       | hash_int_1000   | 4.32e+03   | 0.231      | 2.1e-07 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_1000   | 4.86e+03   | 0.206      | 2.1e-07 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_1000   | 4870       | 0.2054     | 5.3e-08 | 20      |
 | JSON::XS::encode_json         | hash_int_1000   | 4.87e+03   | 0.205      | 2.1e-07 | 20      |
 | JSON::Create::create_json     | hash_int_1000   | 5.04e+03   | 0.198      | 2.1e-07 | 20      |
 | JSON::PP::encode_json         | array_int_100   | 5.56e+03   | 0.18       | 2e-07   | 23      |
 | YAML::Old::Dump               | str1k           | 6.76e+03   | 0.148      | 2e-07   | 23      |
 | YAML::Syck::Dump              | hash_int_100    | 6.89e+03   | 0.145      | 2.5e-07 | 23      |
 | YAML::Syck::Dump              | array_ustr1k_10 | 7.04e+03   | 0.142      | 2.1e-07 | 20      |
 | YAML::XS::Dump                | array_str1k_10  | 7.24e+03   | 0.138      | 1.5e-07 | 22      |
 | YAML::XS::Dump                | hash_int_100    | 7.29e+03   | 0.137      | 2.1e-07 | 20      |
 | JSON::PP::encode_json         | array_str1k_10  | 1.02e+04   | 0.098      | 2.1e-07 | 20      |
 | YAML::Old::Dump               | str1k           | 1.23e+04   | 0.0813     | 1.1e-07 | 20      |
 | YAML::Old::Dump               | num             | 1.32e+04   | 0.0758     | 1.1e-07 | 20      |
 | Storable::freeze              | array_int_1000  | 1.39e+04   | 0.0721     | 1.1e-07 | 20      |
 | JSON::Create::create_json     | array_ustr1k_10 | 1.39e+04   | 0.072      | 1.1e-07 | 20      |
 | YAML::Old::Dump               | undef           | 1.53e+04   | 0.0653     | 1.2e-07 | 23      |
 | YAML::Syck::Dump              | array_int_100   | 1.58e+04   | 0.0634     | 2.5e-08 | 22      |
 | YAML::XS::Dump                | array_int_100   | 1.65e+04   | 0.0607     | 8e-08   | 20      |
 | JSON::PP::encode_json         | str1k           | 1.83e+04   | 0.0547     | 1.1e-07 | 20      |
 | JSON::PP::encode_json         | hash_int_10     | 1.88e+04   | 0.0532     | 8e-08   | 20      |
 | YAML::Syck::Dump              | array_str1k_10  | 1.95e+04   | 0.0512     | 1.1e-07 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_1000  | 2.08e+04   | 0.0481     | 5.3e-08 | 20      |
 | JSON::XS::encode_json         | array_int_1000  | 2.08e+04   | 0.048      | 4.9e-08 | 24      |
 | JSON::MaybeXS::encode_json    | array_int_1000  | 2.087e+04  | 0.04791    | 1.3e-08 | 20      |
 | Cpanel::JSON::XS::encode_json | str1k           | 2.13e+04   | 0.047      | 4.4e-08 | 29      |
 | JSON::MaybeXS::encode_json    | str1k           | 2.2e+04    | 0.046      | 2.1e-07 | 24      |
 | JSON::XS::encode_json         | str1k           | 2.2e+04    | 0.0455     | 9.5e-08 | 25      |
 | JSON::Create::create_json     | array_int_1000  | 2.33e+04   | 0.0428     | 1.2e-07 | 21      |
 | Data::MessagePack::pack       | array_int_1000  | 26897.6    | 0.03717803 | 1.1e-11 | 20      |
 | YAML::XS::Dump                | str1k           | 2.774e+04  | 0.03605    | 1.1e-08 | 28      |
 | Sereal::encode_sereal         | array_int_1000  | 2.83e+04   | 0.0353     | 1.2e-08 | 24      |
 | JSON::Create::create_json     | array_str1k_10  | 4.027e+04  | 0.02483    | 6.1e-09 | 24      |
 | JSON::PP::encode_json         | array_int_10    | 4.1e+04    | 0.024      | 8.3e-08 | 22      |
 | Storable::freeze              | hash_int_100    | 4.32e+04   | 0.0232     | 2.7e-08 | 20      |
 | Sereal::encode_sereal         | hash_int_100    | 4.4e+04    | 0.023      | 8e-08   | 20      |
 | JSON::MaybeXS::encode_json    | array_str1k_10  | 4.48e+04   | 0.0223     | 2.7e-08 | 20      |
 | Cpanel::JSON::XS::encode_json | array_str1k_10  | 4.493e+04  | 0.02225    | 6.7e-09 | 20      |
 | JSON::XS::encode_json         | array_str1k_10  | 4.619e+04  | 0.02165    | 6.4e-09 | 22      |
 | Data::MessagePack::pack       | hash_int_100    | 4.69e+04   | 0.0213     | 3.3e-08 | 20      |
 | YAML::Syck::Dump              | hash_int_10     | 5.08e+04   | 0.0197     | 4.7e-08 | 26      |
 | YAML::Syck::Dump              | str1k           | 5.42e+04   | 0.0185     | 6.7e-09 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_100    | 5.48e+04   | 0.0183     | 3.3e-08 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_100    | 5.66e+04   | 0.0177     | 2.7e-08 | 20      |
 | JSON::XS::encode_json         | hash_int_100    | 5.71e+04   | 0.0175     | 2e-08   | 20      |
 | JSON::Create::create_json     | hash_int_100    | 5.86e+04   | 0.0171     | 2.5e-08 | 22      |
 | YAML::XS::Dump                | hash_int_10     | 5.94e+04   | 0.0168     | 2.7e-08 | 20      |
 | YAML::XS::Dump                | str1k           | 6.1e+04    | 0.016      | 8e-08   | 20      |
 | YAML::Syck::Dump              | array_int_10    | 84900.1    | 0.0117786  | 2.3e-11 | 20      |
 | JSON::PP::encode_json         | str1k           | 8.882e+04  | 0.01126    | 3.3e-09 | 20      |
 | YAML::Syck::Dump              | str1k           | 1.03e+05   | 0.00971    | 1.3e-08 | 20      |
 | Storable::freeze              | array_int_100   | 1.03e+05   | 0.00971    | 1.5e-08 | 24      |
 | YAML::XS::Dump                | array_int_10    | 1.06e+05   | 0.00939    | 1e-08   | 20      |
 | Storable::freeze              | array_ustr1k_10 | 1.26e+05   | 0.00794    | 3.2e-09 | 22      |
 | YAML::Syck::Dump              | num             | 1.35e+05   | 0.0074     | 1.3e-08 | 20      |
 | Storable::freeze              | hash_int_10     | 1.42e+05   | 0.00705    | 3.2e-09 | 22      |
 | JSON::Create::create_json     | str1k           | 141921     | 0.00704619 | 1.1e-11 | 29      |
 | YAML::Syck::Dump              | undef           | 1.76e+05   | 0.00569    | 6.4e-09 | 22      |
 | Storable::freeze              | array_str1k_10  | 1.77e+05   | 0.00565    | 5e-09   | 20      |
 | JSON::PP::encode_json         | num             | 1.82e+05   | 0.0055     | 8.3e-09 | 20      |
 | JSON::MaybeXS::encode_json    | array_int_100   | 1.84e+05   | 0.00543    | 8.3e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_100   | 1.85e+05   | 0.005407   | 1.7e-09 | 20      |
 | JSON::XS::encode_json         | array_int_100   | 1.9e+05    | 0.00526    | 5e-09   | 20      |
 | YAML::XS::Dump                | num             | 2.16e+05   | 0.00464    | 1.7e-09 | 20      |
 | Storable::freeze              | array_int_10    | 2.34e+05   | 0.00428    | 6.7e-09 | 20      |
 | JSON::Create::create_json     | array_int_100   | 2.39e+05   | 0.00418    | 1.5e-09 | 24      |
 | Data::MessagePack::pack       | array_int_100   | 2.43e+05   | 0.00412    | 1.5e-09 | 26      |
 | Sereal::encode_sereal         | array_int_100   | 2.5e+05    | 0.00401    | 6.7e-09 | 20      |
 | Data::MessagePack::pack       | array_ustr1k_10 | 2.84e+05   | 0.00352    | 5e-09   | 20      |
 | Sereal::encode_sereal         | array_ustr1k_10 | 2.87e+05   | 0.00348    | 6.2e-09 | 23      |
 | JSON::PP::encode_json         | undef           | 3.06e+05   | 0.00327    | 6.2e-09 | 23      |
 | Data::MessagePack::pack       | array_str1k_10  | 3.465e+05  | 0.002886   | 9.3e-11 | 20      |
 | Sereal::encode_sereal         | array_str1k_10  | 3.59e+05   | 0.00278    | 3.3e-09 | 21      |
 | JSON::Create::create_json     | str1k           | 3.6785e+05 | 0.0027185  | 5.7e-11 | 22      |
 | YAML::XS::Dump                | undef           | 3.93e+05   | 0.00255    | 3.3e-09 | 20      |
 | JSON::MaybeXS::encode_json    | str1k           | 4.08e+05   | 0.00245    | 8.3e-10 | 20      |
 | Cpanel::JSON::XS::encode_json | str1k           | 4.08e+05   | 0.00245    | 3.3e-09 | 21      |
 | Sereal::encode_sereal         | hash_int_10     | 4.1e+05    | 0.00244    | 7.5e-09 | 20      |
 | Data::MessagePack::pack       | hash_int_10     | 4.28e+05   | 0.00234    | 6.7e-09 | 20      |
 | JSON::XS::encode_json         | str1k           | 4.36e+05   | 0.00229    | 3.3e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | hash_int_10     | 4.6118e+05 | 0.0021683  | 1.2e-11 | 20      |
 | JSON::MaybeXS::encode_json    | hash_int_10     | 4.9e+05    | 0.0021     | 2.1e-08 | 20      |
 | JSON::Create::create_json     | hash_int_10     | 5e+05      | 0.002      | 6.7e-09 | 20      |
 | JSON::XS::encode_json         | hash_int_10     | 5.01e+05   | 0.002      | 3.3e-09 | 20      |
 | Sereal::encode_sereal         | array_int_10    | 1.1e+06    | 0.000906   | 2e-09   | 22      |
 | JSON::MaybeXS::encode_json    | array_int_10    | 1.24e+06   | 0.000808   | 2.1e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | array_int_10    | 1.25e+06   | 0.000797   | 1.7e-09 | 20      |
 | JSON::XS::encode_json         | array_int_10    | 1.27e+06   | 0.00079    | 1.5e-09 | 24      |
 | Data::MessagePack::pack       | array_int_10    | 1.43e+06   | 0.000699   | 3.8e-10 | 25      |
 | JSON::Create::create_json     | array_int_10    | 1.44e+06   | 0.000692   | 1.2e-09 | 20      |
 | Cpanel::JSON::XS::encode_json | num             | 1.5e+06    | 0.000667   | 8.3e-10 | 20      |
 | JSON::MaybeXS::encode_json    | num             | 1.5e+06    | 0.000665   | 8.3e-10 | 20      |
 | JSON::XS::encode_json         | num             | 1.69e+06   | 0.000591   | 2.1e-10 | 20      |
 | Sereal::encode_sereal         | str1k           | 1.7e+06    | 0.00058    | 3.7e-09 | 20      |
 | Sereal::encode_sereal         | str1k           | 2.01e+06   | 0.000498   | 2.3e-10 | 20      |
 | JSON::Create::create_json     | num             | 2.2e+06    | 0.000454   | 1.7e-10 | 29      |
 | Data::MessagePack::pack       | str1k           | 2.337e+06  | 0.0004278  | 2.3e-11 | 20      |
 | Sereal::encode_sereal         | num             | 2.5e+06    | 0.00041    | 2.6e-09 | 21      |
 | Data::MessagePack::pack       | str1k           | 2.68e+06   | 0.000373   | 2e-10   | 22      |
 | Sereal::encode_sereal         | undef           | 2.82e+06   | 0.000355   | 1e-09   | 20      |
 | Data::MessagePack::pack       | num             | 4.36e+06   | 0.00023    | 1e-10   | 22      |
 | JSON::MaybeXS::encode_json    | undef           | 4.8e+06    | 0.00021    | 1.7e-09 | 20      |
 | JSON::XS::encode_json         | undef           | 4.969e+06  | 0.0002013  | 1.2e-11 | 20      |
 | Data::MessagePack::pack       | undef           | 5.03e+06   | 0.000199   | 1e-10   | 20      |
 | Cpanel::JSON::XS::encode_json | undef           | 5e+06      | 0.0002     | 9.4e-10 | 20      |
 | JSON::Create::create_json     | undef           | 5.7e+06    | 0.00018    | 1.2e-09 | 23      |
 +-------------------------------+-----------------+------------+------------+---------+---------+


Benchmark deserializing (C<< bencher --include-participant-tags-json '["deserialize"]' >>):

 +---------------------------------+---------------------+------------+-----------+---------+---------+
 | participant                     | dataset             | rate (/s)  | time (ms) | errors  | samples |
 +---------------------------------+---------------------+------------+-----------+---------+---------+
 | JSON::Decode::Regexp::from_json | json:hash_int_1000  | 6.5        | 1.5e+02   | 0.0013  | 20      |
 | Pegex::JSON                     | json:hash_int_1000  | 15         | 66.7      | 4.9e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_1000  | 17.2       | 58        | 6.8e-05 | 20      |
 | Pegex::JSON                     | json:array_int_1000 | 26.5       | 37.8      | 3.1e-05 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_1000 | 36         | 28        | 0.00027 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_1000 | 38.1       | 26.2      | 3e-05   | 20      |
 | JSON::PP::decode_json           | json:hash_int_1000  | 86.8       | 11.5      | 1.3e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_str1k_10 | 89         | 11        | 5.2e-05 | 20      |
 | JSON::PP::decode_json           | json:array_str1k_10 | 131        | 7.62      | 2.1e-05 | 20      |
 | Pegex::JSON                     | json:hash_int_100   | 1.4e+02    | 7         | 2.7e-05 | 20      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_100   | 153        | 6.54      | 2e-05   | 20      |
 | JSON::PP::decode_json           | json:array_int_1000 | 172        | 5.82      | 8.5e-06 | 20      |
 | Pegex::JSON                     | json:array_int_100  | 251        | 3.98      | 3.8e-06 | 20      |
 | JSON::Decode::Regexp::from_json | json:hash_int_100   | 311        | 3.22      | 6.7e-06 | 20      |
 | JSON::Decode::Marpa::from_json  | json:array_int_100  | 315        | 3.17      | 6.7e-06 | 20      |
 | Pegex::JSON                     | json:array_str1k_10 | 599        | 1.67      | 4.2e-06 | 22      |
 | JSON::Decode::Marpa::from_json  | json:hash_int_10    | 727        | 1.38      | 2.6e-06 | 21      |
 | JSON::Decode::Regexp::from_json | json:array_int_100  | 876        | 1.14      | 9.1e-07 | 20      |
 | Pegex::JSON                     | json:hash_int_10    | 903        | 1.11      | 9.1e-07 | 20      |
 | JSON::PP::decode_json           | json:hash_int_100   | 938        | 1.07      | 8.3e-07 | 21      |
 | JSON::Decode::Marpa::from_json  | json:array_int_10   | 976        | 1.02      | 1.4e-06 | 21      |
 | Pegex::JSON                     | json:array_int_10   | 1.22e+03   | 0.821     | 4.8e-07 | 20      |
 | JSON::PP::decode_json           | json:str1k          | 1.3e+03    | 0.77      | 2.8e-06 | 21      |
 | JSON::PP::decode_json           | json:array_int_100  | 1.87e+03   | 0.535     | 2.1e-07 | 20      |
 | Pegex::JSON                     | json:str1k          | 1.95e+03   | 0.514     | 2.1e-07 | 20      |
 | Pegex::JSON                     | json:null           | 2.16e+03   | 0.463     | 4.3e-07 | 20      |
 | Pegex::JSON                     | json:num            | 2.2e+03    | 0.45      | 1.8e-06 | 20      |
 | JSON::Decode::Regexp::from_json | json:hash_int_10    | 3.42e+03   | 0.292     | 6.4e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_str1k_10 | 3.42e+03   | 0.292     | 4.3e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:array_int_10   | 5.7e+03    | 0.175     | 2.1e-07 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_1000  | 6.67e+03   | 0.15      | 4.3e-07 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_1000  | 6.8e+03    | 0.15      | 9.1e-07 | 20      |
 | JSON::XS::decode_json           | json:hash_int_1000  | 6.84e+03   | 0.146     | 2.1e-07 | 20      |
 | JSON::MaybeXS::decode_json      | json:hash_int_1000  | 7.06e+03   | 0.142     | 2.1e-07 | 21      |
 | JSON::PP::decode_json           | json:hash_int_10    | 9.2e+03    | 0.109     | 2e-07   | 23      |
 | JSON::Decode::Regexp::from_json | json:str1k          | 1.12e+04   | 0.0894    | 2.1e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:num            | 1.2e+04    | 0.084     | 3.2e-07 | 20      |
 | JSON::Decode::Regexp::from_json | json:null           | 1.5e+04    | 0.065     | 2.4e-07 | 20      |
 | JSON::PP::decode_json           | json:array_int_10   | 1.67e+04   | 0.0599    | 8e-08   | 20      |
 | JSON::Parse::parse_json         | json:array_int_1000 | 2.58e+04   | 0.0387    | 1.3e-08 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_1000 | 2.9e+04    | 0.0344    | 5.3e-08 | 20      |
 | JSON::XS::decode_json           | json:array_int_1000 | 2.98e+04   | 0.0336    | 1.3e-08 | 20      |
 | JSON::MaybeXS::decode_json      | json:array_int_1000 | 2.99e+04   | 0.0334    | 6.7e-08 | 20      |
 | JSON::Parse::parse_json         | json:array_str1k_10 | 3.58e+04   | 0.0279    | 1e-08   | 33      |
 | Cpanel::JSON::XS::decode_json   | json:array_str1k_10 | 4.92e+04   | 0.0203    | 6.7e-09 | 20      |
 | JSON::MaybeXS::decode_json      | json:array_str1k_10 | 5e+04      | 0.02      | 3.1e-08 | 23      |
 | JSON::XS::decode_json           | json:array_str1k_10 | 65486.7    | 0.0152703 | 1.2e-11 | 20      |
 | JSON::PP::decode_json           | json:num            | 70923      | 0.0141    | 5.8e-11 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_100   | 8.781e+04  | 0.01139   | 3.3e-09 | 20      |
 | JSON::MaybeXS::decode_json      | json:hash_int_100   | 89321.6    | 0.0111955 | 1.2e-11 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_100   | 8.96e+04   | 0.0112    | 1.3e-08 | 20      |
 | JSON::XS::decode_json           | json:hash_int_100   | 9.29e+04   | 0.0108    | 1e-08   | 20      |
 | JSON::PP::decode_json           | json:null           | 1.3e+05    | 0.0079    | 4e-08   | 20      |
 | JSON::Parse::parse_json         | json:array_int_100  | 2.38e+05   | 0.0042    | 1.7e-09 | 20      |
 | JSON::XS::decode_json           | json:array_int_100  | 2.39e+05   | 0.00418   | 1.4e-09 | 30      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_100  | 2.41e+05   | 0.00415   | 6.5e-09 | 21      |
 | JSON::MaybeXS::decode_json      | json:array_int_100  | 2.43e+05   | 0.00412   | 5e-09   | 20      |
 | JSON::Parse::parse_json         | json:str1k          | 3.5715e+05 | 0.0028    | 4.4e-11 | 28      |
 | Cpanel::JSON::XS::decode_json   | json:str1k          | 5e+05      | 0.002     | 7.5e-09 | 20      |
 | JSON::MaybeXS::decode_json      | json:str1k          | 5.08e+05   | 0.00197   | 7.6e-10 | 24      |
 | JSON::XS::decode_json           | json:str1k          | 6.14e+05   | 0.00163   | 8.3e-10 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:hash_int_10    | 8.39e+05   | 0.00119   | 1.5e-09 | 24      |
 | JSON::MaybeXS::decode_json      | json:hash_int_10    | 8.49e+05   | 0.00118   | 3.3e-09 | 20      |
 | JSON::Parse::parse_json         | json:hash_int_10    | 8.6e+05    | 0.0012    | 3.8e-09 | 20      |
 | JSON::XS::decode_json           | json:hash_int_10    | 9.02e+05   | 0.00111   | 2e-09   | 21      |
 | JSON::XS::decode_json           | json:num            | 9.3e+05    | 0.0011    | 8.7e-09 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:num            | 9.8e+05    | 0.00102   | 1.7e-09 | 20      |
 | JSON::MaybeXS::decode_json      | json:num            | 9.8e+05    | 0.001     | 3.3e-09 | 21      |
 | JSON::Parse::parse_json         | json:num            | 1.07e+06   | 0.000933  | 1.3e-09 | 20      |
 | JSON::XS::decode_json           | json:array_int_10   | 1.402e+06  | 0.000713  | 3.5e-11 | 20      |
 | JSON::Parse::parse_json         | json:array_int_10   | 1.4e+06    | 0.00071   | 6e-09   | 20      |
 | Cpanel::JSON::XS::decode_json   | json:array_int_10   | 1.47e+06   | 0.00068   | 7e-10   | 28      |
 | JSON::MaybeXS::decode_json      | json:array_int_10   | 1.53e+06   | 0.000654  | 8.4e-10 | 20      |
 | Cpanel::JSON::XS::decode_json   | json:null           | 6.91e+06   | 0.000145  | 7e-11   | 21      |
 | JSON::MaybeXS::decode_json      | json:null           | 6.93e+06   | 0.000144  | 5.8e-11 | 20      |
 | JSON::XS::decode_json           | json:null           | 7.105e+06  | 0.0001407 | 2.3e-11 | 20      |
 | JSON::Parse::parse_json         | json:null           | 7.77e+06   | 0.000129  | 1.1e-10 | 20      |
 +---------------------------------+---------------------+------------+-----------+---------+---------+


Benchmark module startup overhead:

 +----------------------+-----------+------------------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time (ms) | errors  | samples |
 +----------------------+-----------+------------------------+---------+---------+
 | JSON::Decode::Marpa  | 1.3e+02   | 124.3                  | 0.00047 | 20      |
 | YAML::XS             | 52.4      | 46.7                   | 8.5e-05 | 20      |
 | JSON::PP             | 25.7      | 20                     | 2.6e-05 | 20      |
 | Pegex::JSON          | 22.8      | 17.1                   | 5.9e-05 | 20      |
 | JSON::MaybeXS        | 18.1      | 12.4                   | 5.6e-05 | 20      |
 | Storable             | 16        | 10.3                   | 9.1e-05 | 20      |
 | Sereal               | 16        | 10.3                   | 5.2e-05 | 20      |
 | YAML::Old            | 15.1      | 9.4                    | 3.4e-05 | 20      |
 | JSON::XS             | 13        | 7.3                    | 5e-05   | 20      |
 | JSON::Parse          | 13        | 7.3                    | 8.5e-05 | 20      |
 | YAML::Syck           | 13        | 7.3                    | 8.7e-05 | 20      |
 | Cpanel::JSON::XS     | 12.6      | 6.9                    | 3.5e-05 | 20      |
 | JSON::Create         | 10        | 4.3                    | 6.1e-05 | 20      |
 | Data::MessagePack    | 9.7       | 4                      | 6.9e-05 | 20      |
 | JSON::Decode::Regexp | 9.3       | 3.6                    | 3.1e-05 | 20      |
 | perl -e1 (baseline)  | 5.7       | 0                      | 2.8e-05 | 20      |
 +----------------------+-----------+------------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
