package App::perlmv::scriptlet::add_prefix_datestamp;

use 5.010001;
use strict;
use warnings;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2023-02-27'; # DATE
our $DIST = 'App-perlmv-scriptlet-add_prefix_datestamp'; # DIST
our $VERSION = '0.001'; # VERSION

our $SCRIPTLET = {
    summary => 'Add datestamp prefix (YYYYMMDD-) to filenames, using files\' modification time as date',
    args => {
        avoid_duplicate_prefix => {
            summary => 'Avoid adding prefix when filename already has prefix that looks like datestamp (1xxxxxxx- to 2xxxxxxx)',
            schema => 'bool*',
        },
    },
    code => sub {
        package
            App::perlmv::code;

        require POSIX;

        use vars qw($ARGS);

        if ($ARGS->{avoid_duplicate_prefix} && /\A[12][0-9]{3}(0[1-9]|10|11|12)([0-2][0-9]|30|31)-/) {
            return $_;
        }
        my @stat = stat($_);
        my $prefix = POSIX::strftime("%Y%m%d-", localtime($stat[9]));

        "$prefix$_";
    },
};

1;

# ABSTRACT: Add datestamp prefix (YYYYMMDD-) to filenames, using files' modification time as date

__END__

=pod

=encoding UTF-8

=head1 NAME

App::perlmv::scriptlet::add_prefix_datestamp - Add datestamp prefix (YYYYMMDD-) to filenames, using files' modification time as date

=head1 VERSION

This document describes version 0.001 of App::perlmv::scriptlet::add_prefix_datestamp (from Perl distribution App-perlmv-scriptlet-add_prefix_datestamp), released on 2023-02-27.

=head1 SYNOPSIS

With filenames:

 foo.txt
 new-bar.txt

This command:

 % perlmv add-prefix -a prefix=new- *

will rename the files as follow:

 foo.txt -> new-foo.txt
 new-bar.txt -> new-new-bar.txt

This command:

 % perlmv add-prefix -a prefix=new- -a avoid_duplicate_prefix=1 *

will rename the files as follow:

 foo.txt -> new-foo.txt

=head1 SCRIPTLET ARGUMENTS

Arguments can be passed using the C<-a> (C<--arg>) L<perlmv> option, e.g. C<< -a name=val >>.

=head2 avoid_duplicate_prefix

Avoid adding prefix when filename already has prefix that looks like datestamp (1xxxxxxx- to 2xxxxxxx). 

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-perlmv-scriptlet-add_prefix_datestamp>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-perlmv-scriptlet-add_prefix_datestamp>.

=head1 SEE ALSO

L<App::perlmv::scriptlet::add_suffix>

The C<remove-common-prefix> scriptlet

L<perlmv> (from L<App::perlmv>)

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>,
L<Pod::Weaver::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla- and/or Pod::Weaver plugins. Any additional steps required beyond
that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2023 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-perlmv-scriptlet-add_prefix_datestamp>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
