package App::metasyn;

our $DATE = '2017-02-17'; # DATE
our $VERSION = '0.002'; # VERSION

use 5.010001;
use strict;
use warnings;

our %SPEC;

$SPEC{metasyn} = {
    v => 1.1,
    summary => 'Alternative front-end to Acme::MetaSyntactic',
    description => <<'_',

This script is an alternative front-end to <pm:Acme::MetaSyntactic>. Compared to
the official CLI <prog:meta>, this CLI is more oriented towards listing names
instead of giving you one or several random names.

_
    args => {
        action => {
            schema => ['str*', in=>[qw/list-themes list-names/]],
            default => 'list-names',
            cmdline_aliases => {
                l => { summary => 'List installed themes', is_flag => 1, code => sub { $_[0]{action} = 'list-themes' } },
            },
        },
        theme => {
            schema => 'str*',
            pos => 0,
            completion => sub {
                require Complete::Acme::MetaSyntactic;
                Complete::Acme::MetaSyntactic::complete_meta_themes_and_categories(@_);
            },
        },
        shuffle => {
            schema => ['bool*', is=>1],
        },
        categories => {
            schema => ['bool*', is=>1],
            cmdline_aliases => {c=>{}},
        },
    },
    examples => [
        {
            summary => 'List all installed themes',
            argv => [qw/-l/],
        },
        {
            summary => 'List all names from a theme',
            argv => [qw/foo/],
        },
        {
            summary => 'List all names from a theme in random order',
            argv => [qw(christmas/elf --shuffle)],
        },
        {
            summary => 'List all categories from a theme',
            argv => [qw(christmas --categories)],
        },
    ],
    links => [
        {url=>'prog:meta'},
    ],
};
sub metasyn {
    no strict 'refs';
    require Acme::MetaSyntactic;

    my %args = @_;

    my $action = $args{action};

    return [200, "OK", [Acme::MetaSyntactic->new->themes]]
        if $action eq 'list-themes';

    my $theme = $args{theme};
    return [400, "Please specify theme"] unless $theme;
    my $cat = $theme =~ s{/(.+)\z}{} ? $1 : undef;

    my $pkg = "Acme::MetaSyntactic::$theme";
    (my $pkg_pm = "$pkg.pm") =~ s!::!/!g;
    return [500, "Can't load $pkg: $@"] unless (eval { require $pkg_pm; 1 });

    if ($args{categories}) {
        my @res;
        eval { @res = $pkg->categories };
        #warn if $@;
        return [200, "OK", \@res];
    }
    #my $meta = Acme::MetaSyntactic->new($theme);
    my @names;
    if (defined $cat) {
        @names = @{ ${"$pkg\::MultiList"}{$cat} // [] };
    } else {
        @names = @{"$pkg\::List"};
        unless (@names) {
            @names = map { @{ ${"$pkg\::MultiList"}{$_} } }
                sort keys %{"$pkg\::MultiList"};
        }
    }
    if ($args{shuffle}) {
        require List::Util;
        @names = List::Util::shuffle(@names);
    }
    return [200, "OK", \@names];
}

1;
# ABSTRACT: Alternative front-end to Acme::MetaSyntactic

__END__

=pod

=encoding UTF-8

=head1 NAME

App::metasyn - Alternative front-end to Acme::MetaSyntactic

=head1 VERSION

This document describes version 0.002 of App::metasyn (from Perl distribution App-metasyn), released on 2017-02-17.

=head1 SYNOPSIS

Use the included script L<metasyn>.

=head1 FUNCTIONS


=head2 metasyn

Usage:

 metasyn(%args) -> [status, msg, result, meta]

Alternative front-end to Acme::MetaSyntactic.

Examples:

=over

=item * List all installed themes:

 metasyn( action => "list-themes");

Result:

 [
   200,
   "OK",
   [
     "abba",
     "afke",
     "alice",
     "alphabet",
     "amber",
     "antlers",
     "any",
     "asterix",
     "barbapapa",
     "barbarella",
     "batman",
     "ben_and_jerry",
     "bible",
     "booze",
     "bottles",
     "browser",
     "buffy",
     "calvin",
     "camelidae",
     "care_bears",
     "chess",
     "chinese_zodiac",
     "christmas",
     "colors",
     "colours",
     "constellations",
     "contrade",
     "contributors",
     "counting_rhyme",
     "counting_to_one",
     "countries",
     "crypto",
     "currency",
     "dancers",
     "dangdut",
     "debian",
     "dilbert",
     "discworld",
     "display_resolution",
     "doctor_who",
     "donmartin",
     "dwarves",
     "elements",
     "evangelion",
     "fabeltjeskrant",
     "facecards",
     "fawlty_towers",
     "flintstones",
     "foo",
     "frasier",
     "french_presidents",
     "garbage",
     "garfield",
     "gems",
     "good_omens",
     "groo",
     "haddock",
     "hhgg",
     "iata",
     "icao",
     "id_names",
     "invasions",
     "jabberwocky",
     "jamesbond",
     "jerkcity",
     "linux",
     "loremipsum",
     "lotr",
     "lucky_luke",
     "magic8ball",
     "magicroundabout",
     "magma",
     "mars",
     "metro",
     "monty_spam",
     "muses",
     "nis",
     "nobel_prize",
     "norse_mythology",
     "octothorpe",
     "olympics",
     "opcodes",
     "oulipo",
     "pantagruel",
     "pasta",
     "pause_id",
     "peanuts",
     "pgpfone",
     "phonetic",
     "pie",
     "planets",
     "pm_groups",
     "pokemon",
     "pooh",
     "pop2",
     "pop3",
     "pornstars",
     "pumpkings",
     "punctuation",
     "pynchon",
     "python",
     "quantum",
     "regions",
     "reindeer",
     "renault",
     "robin",
     "roman",
     "scooby_doo",
     "screw_drives",
     "seinfeld",
     "services",
     "shadok",
     "simpsons",
     "sins",
     "smtp",
     "smurfs",
     "space_missions",
     "sql",
     "stars",
     "state_flowers",
     "summerwine",
     "swords",
     "tarot",
     "teletubbies",
     "thunderbirds",
     "tld",
     "tmnt",
     "tokipona",
     "tour_de_france",
     "trigan",
     "unicode",
     "us_presidents",
     "userfriendly",
     "vcs",
     "viclones",
     "wales_towns",
     "weekdays",
     "yapc",
     "zodiac",
   ],
   {},
 ]

=item * List all names from a theme:

 metasyn( theme => "foo");

Result:

 [
   200,
   "OK",
   [
     "foo",
     "bar",
     "baz",
     "foobar",
     "fubar",
     "qux",
     "quux",
     "corge",
     "grault",
     "garply",
     "waldo",
     "fred",
     "plugh",
     "xyzzy",
     "thud",
     "toto",
     "titi",
     "tata",
     "tutu",
     "pipo",
     "bidon",
     "test1",
     "test2",
     "test3",
     "truc",
     "chose",
     "machin",
     "chouette",
     "bidule",
     "aap",
     "noot",
     "mies",
     "wim",
     "zus",
     "jet",
     "teun",
     "vuur",
     "gijs",
     "lam",
     "kees",
     "bok",
     "weide",
     "does",
     "hok",
     "duif",
     "schapen",
   ],
   {},
 ]

=item * List all names from a theme in random order:

 metasyn( theme => "christmas/elf", shuffle => 1);

Result:

 [
   200,
   "OK",
   [
     "minstix",
     "bushy",
     "snowball",
     "upatree",
     "alabaster",
     "opneslae",
     "mary",
     "sugarplum",
     "shinny",
     "wunorse",
     "evergreen",
     "pepper",
   ],
   {},
 ]

=item * List all categories from a theme:

 metasyn( theme => "christmas", categories => 1);

Result:

 [200, "OK", ["santa", "snowman", "elf", "reindeer"], {}]

=back

This script is an alternative front-end to L<Acme::MetaSyntactic>. Compared to
the official CLI L<meta>, this CLI is more oriented towards listing names
instead of giving you one or several random names.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<action> => I<str> (default: "list-names")

=item * B<categories> => I<bool>

=item * B<shuffle> => I<bool>

=item * B<theme> => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-metasyn>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-metasyn>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-metasyn>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO


L<meta>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
