package App::imgsize;

our $DATE = '2016-10-06'; # DATE
our $VERSION = '0.001'; # VERSION

use strict;
use warnings;

our %SPEC;
require Exporter;
our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(imgsize);

$SPEC{imgsize} = {
    v => 1.1,
    summary =>
        'Show dimensions of image files',
    args => {
        filenames => {
            'x.name.is_plural' => 1,
            schema => ['array*' => {of => 'filename*'}],
            req => 1,
            pos => 0,
            greedy => 1,
        },
    },
};
sub imgsize {
    require Image::Size;

    my %args = @_;

    my @res;
    for my $filename (@{ $args{filenames} }) {
        unless (-f $filename) {
            warn "No such file or not a file: $filename, skipped\n";
            next;
        }

        my ($x, $y) = Image::Size::imgsize($filename);

        push @res, {
            filename => $filename,
            filesize => (-s $filename),
            x => $x,
            y => $y,
        };
    }

    [200, "OK", \@res,
     {'table.fields' => ['filename', 'x', 'y', 'filesize']}];
}

1;
# ABSTRACT: Show dimensions of image files

__END__

=pod

=encoding UTF-8

=head1 NAME

App::imgsize - Show dimensions of image files

=head1 VERSION

This document describes version 0.001 of App::imgsize (from Perl distribution App-imgsize), released on 2016-10-06.

=head1 SYNOPSIS

 # Use via imgsize CLI script

=head1 FUNCTIONS


=head2 imgsize(%args) -> [status, msg, result, meta]

Show dimensions of image files.

This function is not exported by default, but exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<filenames>* => I<array[filename]>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-imgsize>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-imgsize>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-imgsize>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
