package WebService::PayPal::PaymentsAdvanced::Error::HostedForm;
$WebService::PayPal::PaymentsAdvanced::Error::HostedForm::VERSION = '0.000007';
use Moo;

use Types::Standard qw( InstanceOf );

extends 'Throwable::Error';

has http_response => (
    is       => 'ro',
    isa      => InstanceOf ['HTTP::Response'],
    required => 1,
);

1;

# ABSTRACT: An error message which has been parsed out of a hosted form

__END__

=pod

=encoding UTF-8

=head1 NAME

WebService::PayPal::PaymentsAdvanced::Error::HostedForm - An error message which has been parsed out of a hosted form

=head1 VERSION

version 0.000007

=head1 SYNOPSIS

    use Try::Tiny;
    use WebService::PayPal::PaymentsAdvanced;

    my $payments = WebService::PayPal::PaymentsAdvanced->new(...);
    my $response = $payments->create_secure_token(...);

    my $uri;
    try {
        $uri = $payments->hosted_form_uri($response);
    }
    catch {
        die $_ unless blessed $_;
        if (
            $_->isa(
                'WebService::PayPal::PaymentsAdvanced::Error::HostedForm')
            ) {
            log_hosted_form_error(
                message          => $_->message,
                response_content => $_->http_response->content,
            );
        }

        # handle other exceptions
    };

=head1 DESCRIPTION

This class represents an error which is embedded into the HTML of a hosted
form.   It will only be thrown if you have enabled
L<WebService::PayPal::PaymentsAdvanced/validate_hosted_form_uri>.

It extends L<Throwable::Error> and adds one attribute of its own.  The message
attribute (inherited from L<Throwable::Error>) will contain the error message
which was parsed out of the content of the HTML.

=head1 METHODS

The C<< $error->message() >>, and C<< $error->stack_trace() >> methods are
inherited from L<Throwable::Error>.

=head2 http_response

Returns the L<HTTP::Response> object which was returned when attempting to GET
the hosted form.

=head1 AUTHOR

Olaf Alders <olaf@wundercounter.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by MaxMind, Inc..

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut
