##-*- Mode: CPerl -*-

## File: DDC::Query.pm
## Author: Bryan Jurish <moocow@cpan.org>
##======================================================================

package DDC::Query;
use DDC::Utils qw(:escape);
use strict;

##======================================================================
## Globals etc.
our @ISA = qw();

##======================================================================
## $q = $CLASS_OR_OBJ->new(%args)
##  + %args, %$q:
##    (
##     class   => $class,     ##-- subclass
##     dtrs    => \@args,     ##-- array of sub-queries (e.g. DDC::Query objects)
##     #...
##    )
sub new {
  my ($that,%args) = @_;
  my $class = $args{class};
  my ($new);
  if ($class && defined($new=UNIVERSAL::can((__PACKAGE__ . "::$class"), 'new'))) {
    delete($args{class});
    return $new->((__PACKAGE__ . "::$class"),%args) if ($new ne __PACKAGE__->can('new'));
  }
  return bless(\%args,
	       ($class ? (__PACKAGE__ . "::$class") : (ref($that)||$that))
	      );
}

## $q = $q->negate()
sub negate {
  $_[0]{negated} = !$_[0]{negated};
  return $_[0];
}

## $bool = $q->negatable()
sub negatable { return 0; }

## $bool = $q->negated()
sub negated { return $_[0]->negatable ? $_[0]{negated} : 0; }

## @items = $q->expandItems()
##  + returns list of source items to be expanded (e.g. by CAB); empty list for none
sub expandItems { return qw(); }

## @dtrs = $q->dtrs()
sub dtrs {
  return $_[0]{dtrs} ? @{$_[0]{dtrs}} : qw();
}

## $str = $q->toString()
##  + stringification operator
sub toString {
  my $q = shift;
  confess(ref($q)."::toString(): abstract method called");
}

## \@nodes = $q->dfs()
##  + returns query nodes in depth-first search order
sub dfs {
  my $q = shift;
  my @nodes = ($q);
  foreach ($q->dtrs) {
    push(@nodes,@{$_->dfs});
  }
  return \@nodes;
}

##======================================================================
## Root: root queries (with filters)
package DDC::Query::Root;
our @ISA = qw(DDC::Query);

## $q = $CLASS_OR_OBJ->new(%args)
##  + %args, %$q:
##    (
##     root => $q,            ##-- query root
##     filters => \@filters,  ##-- array of DDC::Query::Filter objects
##    )
sub new {
  my $that = shift;
  return $that->SUPER::new(
			   root=>$that->SUPER::new(),
			   filters=>[],
			   @_
			  );
}

sub dtrs {
  return $_[0]{root};
}

## $str = $q->toString()
##  + stringification operator
sub toString {
  my $q = shift;
  return join(' ', map {$_->toString} $q->{root}, @{$q->{filters}});
}

##======================================================================
## Negatable: negatable queries
package DDC::Query::Negatable;
our @ISA = qw(DDC::Query);

## $bool = $q->negatable()
sub negatable { return 1; }

## $str = $q->negString($str)
##  + stringification operator
sub negString {
  my ($q,$str) = @_;
  return $q->negated ? ('!'.$str) : $str;
}

##======================================================================
## BinOp
package DDC::Query::BinOp;
our @ISA = qw(DDC::Query::Negatable);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>undef,dtrs=>undef,@_);
}

sub toString {
  my $q = shift;
  return $q->negString('('.join(" $q->{op} ", map {$_->toString} @{$q->{dtrs}}).')');
}

##--------------------------------------------------------------
## Q1 && Q2
package DDC::Query::And;
our @ISA = qw(DDC::Query::BinOp);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>'&&',@_);
}

##--------------------------------------------------------------
## Q1 || Q2
package DDC::Query::Or;
our @ISA = qw(DDC::Query::BinOp);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>'||',@_);
}

##--------------------------------------------------------------
## Q1 with Q2
package DDC::Query::With;
our @ISA = qw(DDC::Query::BinOp);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>'WITH',@_);
}
## $bool = $q->negatable()
sub negatable { return 0; }

##======================================================================
## Atomic: sequenceable queries
package DDC::Query::Atomic;
our @ISA = qw(DDC::Query::Negatable);

##======================================================================
## Token: single-token queries

##--------------------------------------------------------------
package DDC::Query::Token;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Atomic);
sub new {
  my $that = shift;
  return $that->SUPER::new(
			   index=>undef,
			   #value=>undef,
			   #expand=>[]
			   @_);
}

sub toString {
  my $q = shift;
  return $q->negString(
		       $q->indexString . $q->valueString . $q->expandString
		      );
}

## $str = $q->indexString()       ##-- uses $q->{index}
## $str = $q->indexString($index)
sub indexString {
  my $q     = shift;
  my $index = @_ ? shift : $q->{index};
  return '' if (!defined($index) || $index eq '');
  return '$'.escapeq($index).'=';
}

## $str = $q->valueString() ##-- uses $q->{value}
## $str = $q->valueString($val)
sub valueString {
  my $q = shift;
  my $value = @_ ? shift : $q->{value};
  $value = '' if (!defined($value));
  return escapeq($value);
}

## $str = $q->expandString() ##-- uses $q->{expand}
## $str = $q->expandString(\@expand)
sub expandString {
  my $q      = shift;
  my $expand = @_ ? shift : $q->{expand};
  return '' if (!$expand || !@$expand);
  return join('', map {'|'.escapeq($_)} @$expand);
}

##--------------------------------------------------------------
## $INDEX=@WORD
package DDC::Query::TokExact;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '@'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## $INDEX=WORD
package DDC::Query::TokExpand;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);

##--------------------------------------------------------------
## $INDEX=WORD*
package DDC::Query::TokPrefix;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return $q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## $INDEX=*WORD
package DDC::Query::TokSuffix;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## $INDEX=*WORD*
package DDC::Query::TokInfix;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## $INDEX=/REGEX/
##  + 'value' argument should contain surrounding slashes and all relevant escapes!
package DDC::Query::TokRegex;
#use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return @_ ? shift : $q->{value};
}

##--------------------------------------------------------------
## $INDEX=*
package DDC::Query::Any;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  return '*';
}

##--------------------------------------------------------------
## TokSet: $INDEX={W1,W2,...} : abstract
package DDC::Query::TokSet;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);
sub new {
  my $that = shift;
  return $that->SUPER::new(values=>[],@_);
}

## $str = $q->valueString() ##-- uses $q->{values}
## $str = $q->valueString(\@values)
##   + returned string includes brackets
sub valueString {
  my $q      = shift;
  my $values = @_ ? shift : $q->{values};
  return '{'.join(',', map {escapeq($_)} @{$values||[]}).'}';
}

##--------------------------------------------------------------
## TokSetExpand: $INDEX={W1,W2,...}
package DDC::Query::TokSetExpand;
our @ISA = qw(DDC::Query::TokSet);
sub new {
  my $that = shift;
  return $that->SUPER::new(expand=>[],@_);
}

## @items = $q->expandItems()
sub expandItems { return @{$_[0]{expand}||[]}; }

##--------------------------------------------------------------
## TokSetExact: $INDEX=@{W1,W2,...}
package DDC::Query::TokSetExact;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return '@'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokSetPrefix: $INDEX={W1,W2,...}*
package DDC::Query::TokSetPrefix;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return $q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## TokSetSuffix: $INDEX=*{W1,W2,...}
package DDC::Query::TokSetSuffix;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokSetInfix: $INDEX=*{W1,W2,...}*
package DDC::Query::TokSetInfix;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## TokThes: $INDEX={THES1:THES2:...}
package DDC::Query::TokThes;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return ':{'.$q->SUPER::valueString(@_).'}';
}

##--------------------------------------------------------------
## TokMorph: $INDEX=[M1,M2,..]
package DDC::Query::TokMorph;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q     = shift;
  my $items = @_ ? shift : $q->{items};
  return '['.join(',', map {escapeq($_)} @{$items||[]}).']';
}

##--------------------------------------------------------------
## TokLemma: $INDEX=%LEMMA
package DDC::Query::TokLemma;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '%'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokChunk: $INDEX=^CHUNK
package DDC::Query::TokChunk;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '^'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokAncor: $.BREAK=POSITION
package DDC::Query::TokAnchor;
our @ISA = qw(DDC::Query::Token);
sub indexString {
  my $q     = shift;
  my $index = @_ ? shift : $q->{index};
  return '$.'.(defined($index) && $index ne '' ? escapeq($index) : '').'=';
}

##--------------------------------------------------------------
## <FILE
package DDC::Query::TokFile;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '<'.$q->SUPER::valueString(@_);
}

##======================================================================
## Multi-token queries

##--------------------------------------------------------------
## Near :
##     NEAR(W1,W2,   N)
##   | NEAR(W1,W2,W3,N)
package DDC::Query::Near;
our @ISA = qw(DDC::Query::Negatable);
sub new {
  my $that = shift;
  return $that->SUPER::new(dtrs=>[],dist=>1,@_);
}
sub toString {
  my $q = shift;
  return $q->negString('NEAR('
		       .join(',',
			     (map {$_->toString} @{$q->{dtrs}}),
			     $q->{dist}
			    )
		       .')'
		      );
}


##--------------------------------------------------------------
## Phrase: "W1 #D1 W2 ... WN"
package DDC::Query::Phrase;
our @ISA = qw(DDC::Query::Atomic);
sub new {
  my $that = shift;
  return $that->SUPER::new(args=>[],@_); ##-- $args=[[$w1,?$d1,?$op1],...,[$wN]]
}
sub dtrs {
  my $q = shift;
  return map {$_->[0]} @{$q->{args}};
}
sub toString {
  my $q = shift;
  return $q->negString('"'
		       .join(' ',
			     map {
			       ($_->[0]->toString . ($_->[1] ? " #$_->[1]" : ''))
			     } @{$q->{args}})
		       .'"'
		      );
}

package DDC::Query::Seq;
our @ISA = qw(DDC::Query::Phrase);

1; ##-- be happy

__END__

##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl

##========================================================================
## NAME
=pod

=head1 NAME

DDC::Query - pure-perl wrapper for parsed DDC queries [DEPRECATED]

=cut

##========================================================================
## PACKAGES
=pod

=head1 PACKAGES

=over 4

=item DDC::Query

=item DDC::Query::Root

=item DDC::Query::Negatable

=item DDC::Query::BinOp

=item DDC::Query::And

=item DDC::Query::Or

=item DDC::Query::With

=item DDC::Query::Atomic

=item DDC::Query::Token

=item DDC::Query::TokExact

=item DDC::Query::TokExpand

=item DDC::Query::TokPrefix

=item DDC::Query::TokSuffix

=item DDC::Query::TokInfix

=item DDC::Query::TokRegex

=item DDC::Query::Any

=item DDC::Query::TokSet

=item DDC::Query::TokSetExpand

=item DDC::Query::TokSetExact

=item DDC::Query::TokSetPrefix

=item DDC::Query::TokSetSuffix

=item DDC::Query::TokSetInfix

=item DDC::Query::TokThes

=item DDC::Query::TokMorph

=item DDC::Query::TokLemma

=item DDC::Query::TokChunk

=item DDC::Query::TokAnchor

=item DDC::Query::TokFile

=item DDC::Query::Near

=item DDC::Query::Phrase

=item DDC::Query::Seq

=back

=cut

##========================================================================
## SYNOPSIS
=pod

=head1 SYNOPSIS

 ##========================================================================
 ## PRELIMINARIES
 
 use DDC::Query;
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query
=pod

=head2 DDC::Query Synopsis

 ##========================================================================
 ## Constructors etc.
 
 $q = $CLASS_OR_OBJ->new(%args);
 $q = $q->negate();
 $bool = $q->negatable();
 $bool = $q->negated();
 @items = $q->expandItems();
 @dtrs = $q->dtrs();
 $str = $q->toString();
 \@nodes = $q->dfs();
 
 ##========================================================================
 ## Root: root queries (with filters)
 
 $q = $CLASS_OR_OBJ->new(%args);
 $str = $q->toString();
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Root
=pod

=head2 DDC::Query::Root Synopsis

 ##========================================================================
 ## Negatable: negatable queries
 
 $bool = $q->negatable();
 $str = $q->negString($str);
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Negatable
=pod

=head2 DDC::Query::Negatable Synopsis

 ##========================================================================
 ## BinOp
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::BinOp
=pod

=head2 DDC::Query::BinOp Synopsis

 ##========================================================================
 ## Q1 && Q2
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::And
=pod

=head2 DDC::Query::And Synopsis

 ##========================================================================
 ## Q1 || Q2
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Or
=pod

=head2 DDC::Query::Or Synopsis

 ##========================================================================
 ## Q1 with Q2
 
 $bool = $q->negatable();
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::With
=pod

=head2 DDC::Query::With Synopsis


=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Atomic
=pod

=head2 DDC::Query::Atomic Synopsis

 ##========================================================================
 ## Token: single-token queries
 
 $str = $q->indexString()       ##-- uses $q->{index};
 $str = $q->valueString() ##-- uses $q->{value};
 $str = $q->expandString() ##-- uses $q->{expand};
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Token
=pod

=head2 DDC::Query::Token Synopsis

 ##========================================================================
 ## $INDEX=@WORD
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokExact
=pod

=head2 DDC::Query::TokExact Synopsis


=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokExpand
=pod

=head2 DDC::Query::TokExpand Synopsis

 ##========================================================================
 ## $INDEX=WORD*
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokPrefix
=pod

=head2 DDC::Query::TokPrefix Synopsis

 ##========================================================================
 ## $INDEX=*WORD
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSuffix
=pod

=head2 DDC::Query::TokSuffix Synopsis

 ##========================================================================
 ## $INDEX=*WORD*
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokInfix
=pod

=head2 DDC::Query::TokInfix Synopsis

 ##========================================================================
 ## $INDEX=/REGEX/
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokRegex
=pod

=head2 DDC::Query::TokRegex Synopsis

 ##========================================================================
 ## $INDEX=*
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Any
=pod

=head2 DDC::Query::Any Synopsis

 ##========================================================================
 ## TokSet: $INDEX={W1,W2,...} : abstract
 
 $str = $q->valueString() ##-- uses $q->{values};
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSet
=pod

=head2 DDC::Query::TokSet Synopsis

 ##========================================================================
 ## TokSetExpand: $INDEX={W1,W2,...}
 
 @items = $q->expandItems();
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSetExpand
=pod

=head2 DDC::Query::TokSetExpand Synopsis

 ##========================================================================
 ## TokSetExact: $INDEX=@{W1,W2,...}
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSetExact
=pod

=head2 DDC::Query::TokSetExact Synopsis

 ##========================================================================
 ## TokSetPrefix: $INDEX={W1,W2,...}*
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSetPrefix
=pod

=head2 DDC::Query::TokSetPrefix Synopsis

 ##========================================================================
 ## TokSetSuffix: $INDEX=*{W1,W2,...}
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSetSuffix
=pod

=head2 DDC::Query::TokSetSuffix Synopsis

 ##========================================================================
 ## TokSetInfix: $INDEX=*{W1,W2,...}*
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokSetInfix
=pod

=head2 DDC::Query::TokSetInfix Synopsis

 ##========================================================================
 ## TokThes: $INDEX={THES1:THES2:...}
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokThes
=pod

=head2 DDC::Query::TokThes Synopsis

 ##========================================================================
 ## TokMorph: $INDEX=[M1,M2,..]
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokMorph
=pod

=head2 DDC::Query::TokMorph Synopsis

 ##========================================================================
 ## TokLemma: $INDEX=%LEMMA
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokLemma
=pod

=head2 DDC::Query::TokLemma Synopsis

 ##========================================================================
 ## TokChunk: $INDEX=^CHUNK
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokChunk
=pod

=head2 DDC::Query::TokChunk Synopsis

 ##========================================================================
 ## TokAncor: $.BREAK=POSITION
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokAnchor
=pod

=head2 DDC::Query::TokAnchor Synopsis

 ##========================================================================
 ## <FILE
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::TokFile
=pod

=head2 DDC::Query::TokFile Synopsis

 ##========================================================================
 ## Near :
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Near
=pod

=head2 DDC::Query::Near Synopsis

 ##========================================================================
 ## Phrase: "W1 #D1 W2 ... WN"
 
 

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Phrase
=pod

=head2 DDC::Query::Phrase Synopsis

(no documentation found for package DDC::Query::Phrase)

=cut

##------------------------------------------------------------------------
## SYNOPSIS: DDC::Query::Seq
=pod

=head2 DDC::Query::Seq Synopsis

(no documentation found for package DDC::Query::Seq)

=cut

##========================================================================
## DESCRIPTION
=pod

=head1 DESCRIPTION

The DDC::Query hierarchy provides
pure-perl wrapper classes for parsed DDC queries.
Its use is deprecated in favor of the L<DDC::XS::CQuery|DDC::XS::CQuery>
class hierarchy providing direct access to the underlying C++ libraries.

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query
=pod

=head2 DDC::Query Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query: Globals etc.
=pod

=head3 Globals etc.

=over 4

=item Variable: @ISA

(undocumented)

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query: Constructors etc.
=pod

=head3 Constructors etc.

=over 4

=item new

 $q = $CLASS_OR_OBJ->new(%args);

%args, %$q:

    (
     class   => $class,     ##-- subclass
     dtrs    => \@args,     ##-- array of sub-queries (e.g. DDC::Query objects)
     #...
    )

=item negate

 $q = $q->negate();

(undocumented)

=item negatable

 $bool = $q->negatable();

(undocumented)

=item negated

 $bool = $q->negated();

(undocumented)

=item expandItems

 @items = $q->expandItems();

returns list of source items to be expanded (e.g. by CAB); empty list for none

=item dtrs

 @dtrs = $q->dtrs();

(undocumented)

=item toString

 $str = $q->toString();

stringification operator

=item dfs

 \@nodes = $q->dfs();

returns query nodes in depth-first search order

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query: Root: root queries (with filters)
=pod

=head3 Root: root queries (with filters)

=over 4

=item Variable: @ISA

(undocumented)

=item new

 $q = $CLASS_OR_OBJ->new(%args);

%args, %$q:

    (
     root => $q,            ##-- query root
     filters => \@filters,  ##-- array of DDC::Query::Filter objects
    )

=item dtrs

(undocumented)

=item toString

 $str = $q->toString();

stringification operator

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Root
=pod

=head2 DDC::Query::Root Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Root: Negatable: negatable queries
=pod

=head3 Negatable: negatable queries

=over 4

=item Variable: @ISA

(undocumented)

=item negatable

 $bool = $q->negatable();

(undocumented)

=item negString

 $str = $q->negString($str);

stringification operator

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Negatable
=pod

=head2 DDC::Query::Negatable Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Negatable: BinOp
=pod

=head3 BinOp

=over 4

=item Variable: @ISA

(undocumented)

=item new

(undocumented)

=item toString

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::BinOp
=pod

=head2 DDC::Query::BinOp Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::BinOp: Q1 && Q2
=pod

=head3 Q1 && Q2

=over 4

=item Variable: @ISA

(undocumented)

=item new

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::And
=pod

=head2 DDC::Query::And Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::And: Q1 || Q2
=pod

=head3 Q1 || Q2

=over 4

=item Variable: @ISA

(undocumented)

=item new

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Or
=pod

=head2 DDC::Query::Or Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Or: Q1 with Q2
=pod

=head3 Q1 with Q2

=over 4

=item Variable: @ISA

(undocumented)

=item new

(undocumented)

=item negatable

 $bool = $q->negatable();

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::With
=pod

=head2 DDC::Query::With Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::With: Atomic: sequenceable queries
=pod

=head3 Atomic: sequenceable queries

=over 4

=item Variable: @ISA

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Atomic
=pod

=head2 DDC::Query::Atomic Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Atomic: Token: single-token queries
=pod

=head3 Token: single-token queries

=over 4

=item Variable: @ISA

(undocumented)

=item new

(undocumented)

=item toString

(undocumented)

=item indexString

 $str = $q->indexString()       ##-- uses $q->{index};

$str = $q-E<gt>indexString($index)

=item valueString

 $str = $q->valueString() ##-- uses $q->{value};

$str = $q-E<gt>valueString($val)

=item expandString

 $str = $q->expandString() ##-- uses $q->{expand};

$str = $q-E<gt>expandString(\@expand)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Token
=pod

=head2 DDC::Query::Token Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Token: $INDEX=@WORD
=pod

=head3 $INDEX=@WORD

=over 4

=item Variable: @ISA

(undocumented)

=item valueString

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokExact
=pod

=head2 DDC::Query::TokExact Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokExact: $INDEX=WORD
=pod

=head3 $INDEX=WORD

=over 4

=item Variable: @ISA

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokExpand
=pod

=head2 DDC::Query::TokExpand Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokExpand: $INDEX=WORD*
=pod

=head3 $INDEX=WORD*

=over 4

=item Variable: @ISA

(undocumented)

=item valueString

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokPrefix
=pod

=head2 DDC::Query::TokPrefix Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokPrefix: $INDEX=*WORD
=pod

=head3 $INDEX=*WORD

=over 4

=item Variable: @ISA

(undocumented)

=item valueString

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSuffix
=pod

=head2 DDC::Query::TokSuffix Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSuffix: $INDEX=*WORD*
=pod

=head3 $INDEX=*WORD*

=over 4

=item Variable: @ISA

(undocumented)

=item valueString

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokInfix
=pod

=head2 DDC::Query::TokInfix Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokInfix: $INDEX=/REGEX/
=pod

=head3 $INDEX=/REGEX/

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokRegex
=pod

=head2 DDC::Query::TokRegex Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokRegex: $INDEX=*
=pod

=head3 $INDEX=*

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Any
=pod

=head2 DDC::Query::Any Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Any: TokSet: $INDEX={W1,W2,...} : abstract
=pod

=head3 TokSet: $INDEX={W1,W2,...} : abstract

=over 4

=item Variable: @ISA

(undocumented)

=item new



(undocumented)

=item valueString

 $str = $q->valueString() ##-- uses $q->{values};

$str = $q-E<gt>valueString(\@values)
returned string includes brackets

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSet
=pod

=head2 DDC::Query::TokSet Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSet: TokSetExpand: $INDEX={W1,W2,...}
=pod

=head3 TokSetExpand: $INDEX={W1,W2,...}

=over 4

=item Variable: @ISA

(undocumented)

=item new



(undocumented)

=item expandItems

 @items = $q->expandItems();

(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetExpand
=pod

=head2 DDC::Query::TokSetExpand Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetExpand: TokSetExact: $INDEX=@{W1,W2,...}
=pod

=head3 TokSetExact: $INDEX=@{W1,W2,...}

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetExact
=pod

=head2 DDC::Query::TokSetExact Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetExact: TokSetPrefix: $INDEX={W1,W2,...}*
=pod

=head3 TokSetPrefix: $INDEX={W1,W2,...}*

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetPrefix
=pod

=head2 DDC::Query::TokSetPrefix Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetPrefix: TokSetSuffix: $INDEX=*{W1,W2,...}
=pod

=head3 TokSetSuffix: $INDEX=*{W1,W2,...}

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetSuffix
=pod

=head2 DDC::Query::TokSetSuffix Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetSuffix: TokSetInfix: $INDEX=*{W1,W2,...}*
=pod

=head3 TokSetInfix: $INDEX=*{W1,W2,...}*

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetInfix
=pod

=head2 DDC::Query::TokSetInfix Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokSetInfix: TokThes: $INDEX={THES1:THES2:...}
=pod

=head3 TokThes: $INDEX={THES1:THES2:...}

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokThes
=pod

=head2 DDC::Query::TokThes Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokThes: TokMorph: $INDEX=[M1,M2,..]
=pod

=head3 TokMorph: $INDEX=[M1,M2,..]

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokMorph
=pod

=head2 DDC::Query::TokMorph Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokMorph: TokLemma: $INDEX=%LEMMA
=pod

=head3 TokLemma: $INDEX=%LEMMA

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokLemma
=pod

=head2 DDC::Query::TokLemma Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokLemma: TokChunk: $INDEX=^CHUNK
=pod

=head3 TokChunk: $INDEX=^CHUNK

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokChunk
=pod

=head2 DDC::Query::TokChunk Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokChunk: TokAncor: $.BREAK=POSITION
=pod

=head3 TokAncor: $.BREAK=POSITION

=over 4

=item Variable: @ISA

(undocumented)

=item indexString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokAnchor
=pod

=head2 DDC::Query::TokAnchor Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokAnchor: <FILE
=pod

=head3 <FILE

=over 4

=item Variable: @ISA

(undocumented)

=item valueString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokFile
=pod

=head2 DDC::Query::TokFile Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::TokFile: Near :
=pod

=head3 Near :

=over 4

=item Variable: @ISA

(undocumented)

=item new



(undocumented)

=item toString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Near
=pod

=head2 DDC::Query::Near Description

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Query::Near: Phrase: "W1 #D1 W2 ... WN"
=pod

=head3 Phrase: "W1 #D1 W2 ... WN"

=over 4

=item Variable: @ISA

(undocumented)

=item Variable: @ISA

(undocumented)

=item new



(undocumented)

=item dtrs



(undocumented)

=item toString



(undocumented)

=back

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Phrase
=pod

=head2 DDC::Query::Phrase Description

(no documentation found for package DDC::Query::Phrase)

=cut

##------------------------------------------------------------------------
## DESCRIPTION: DDC::Query::Seq
=pod

=head2 DDC::Query::Seq Description

(no documentation found for package DDC::Query::Seq)

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl

##======================================================================
## Footer
=pod

=head1 ACKNOWLEDGEMENTS

perl by Larry Wall.

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=head1 COPYRIGHT

Copyright (c) 2011-2015, Bryan Jurish.  All rights reserved.

This package is free software.  You may redistribute it
and/or modify it under the same terms as Perl itself.

=head1 SEE ALSO

perl(1),
DDC::XS(3perl).

=cut
