#-*- Mode: CPerl -*-

## File: DDC::Format.pm
## Author: Bryan Jurish <moocow@cpan.org>
## Description:
##  + DDC Query utilities: output formatting
##======================================================================

package DDC::Format;
use IO::File;
use Carp;
use strict;

##======================================================================
## Globals

##======================================================================
## Constructors, etc.

## $fmt = $CLASS_OR_OBJ->new(%args)
##  + %args:
sub new {
  my $that = shift;
  return bless { @_ }, ref($that)||$that;
}

## $fmt = $fmt->reset()
##  + reset counters, etc.
sub reset { return $_[0]; }

##======================================================================
## API

## $str = $fmt->toString($hitList)
sub toString {
  my ($fmt,$hits) = @_;
  return "$hits";
}

## undef = $fmt->toFile($hitList,$file)
##  + default implementation calls $fmt->toFh()
sub toFile {
  my ($fmt,$hits,$file) = @_;
  my $fh = ref($file) ? $file : IO::File->new(">$file");
  confess(ref($fmt)."::toFile(): open failed for '$file': $!") if (!$fh);
  my $rc = $fmt->toFh($hits,$fh);
  $fh->close() if (!ref($file));
  return $rc;
}

## undef = $fmt->toFh($hitList,$fh)
##  + default implementation calls $fmt->toString()
sub toFh {
  my ($fmt,$hits,$fh) = @_;
  $fh->print($fmt->toString($hits));
}


1; ##-- be happy

##======================================================================
## Docs
=pod

=head1 NAME

DDC::Format - Abstract output formatting API for DDC hits

=head1 SYNOPSIS

 use DDC::Concordance;

 $hitList = DDC::Client::Distributed->new()->query('foo&&bar'); ##-- get some hits

 $fmt  = DDC::Format->new(%some_args); ##-- new format
 $str = $fmt->toString($hitList);      ##-- conversion to string
 $fmt->toFile($hitList,$filename);     ##-- output to file
 $fmt->toFh($hitList,$fh);             ##-- output to filehandle

=cut

##======================================================================
## Description
=pod

=head1 DESCRIPTION

Abstract class for hit formatting.

=cut

##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl

##----------------------------------------------------------------
## DESCRIPTION: DDC::Format: Constructors, etc.
=pod

=head2 Constructors, etc.

=over 4

=item new

 $fmt = $CLASS_OR_OBJ->new(%args);

Derived classes may accept keyword arguments in %args.
The abstract implementation interprets them as literal keys in
the underlying object structure (a HASH-ref).


=item reset

 $fmt = $fmt->reset();

Reset counters, etc.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Format: API
=pod

=head2 API

=over 4

=item toString

 $str = $fmt->toString($hitList);

Should format a DDC::HitList object $hitList to
a string representation.

Dummy implementation.

=item toFile

 undef = $fmt->toFile($hitList,$file);

Should format a DDC::HitList objects $hitList to
a named file $file.

Default implementation calls $fmt-E<gt>toFh().


=item toFh

 undef = $fmt->toFh($hitList,$fh);

Should format a DDC::HitList object $hitList to
a filehandle $fh opened for output.

Default implementation calls $fmt-E<gt>toString()

=back

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl


##======================================================================
## Footer
##======================================================================
=pod

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006-2011 by Bryan Jurish

This package is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.10.0 or,
at your option, any later version of Perl 5 you may have available.

=cut
