package POE::Component::Server::Twirc;
use MooseX::POE;

use LWP::UserAgent::POE;
use POE qw(Component::Server::IRC);
use Net::Twitter;
use Email::Valid;
use Text::Truncate;
use POE::Component::Server::Twirc::LogAppender;
use POE::Component::Server::Twirc::State;

with 'MooseX::Log::Log4perl';

# Net::Twitter returns text with encoded HTML entities.  I *think* decoding
# properly belongs in Net::Twitter.  So, if it gets added, there:
# TODO: remove HTML::Entities and decode_entities calls.
use HTML::Entities;

our $VERSION = '0.04';

=head1 NAME

POE::Component::Server::Twirc - Twitter/IRC gateway

=head1 SYNOPSIS

    use POE::Component::Server::Twirc;

    POE::Component::Server::Twirc->new(
        irc_nickname        => $my_irc_nickname,
        twitter_username    => $my_twitter_username,
        twitter_password    => $my_twitter_password,
        twitter_screen_name => $my_twitter_screen_name,
    );

    POE::Kernel->run;

=head1 DESCRIPTION

C<POE::Component::Server::Twirc> provides an IRC/Twitter gateway.  Twitter friends are
added to a channel and messages they post on twitter appear as channel
messages in IRC.  The IRC interface supports several Twitter features,
including posting status updates, following and un-following Twitter feeds,
enabling and disabling device notifications, sending direct messages, and
querying information about specific Twitter users.

Friends who are also followers are given "voice" as a visual clue in IRC.

=head1 METHODS

=head2 new

Spawns a POE component encapsulating the Twitter/IRC gateway.

Arguments:

=over 4

=item irc_nickname

(Required) The irc nickname used by the owning user.

=cut

has irc_nickname        => ( isa => 'Str', is => 'ro', required => 1 );

=item twitter_username

(Required) The username (email address) used to authenticate with Twitter.

=cut

has twitter_username    => ( isa => 'Str', is => 'ro', required => 1 );

=item twitter_password

(Required) The password used to authenticate with Twitter.

=cut

has twitter_password    => ( isa => 'Str', is => 'ro', required => 1 );

=item twitter_screen_name

(Required) The user's Twitter screen name.

=cut

has twitter_screen_name => ( isa => 'Str', is => 'ro', required => 1 );


=item irc_server_name

(Optional) The name of the IRC server. Defaults to C<twitter.irc>.

=cut

has irc_server_name     => ( isa => 'Str', is => 'ro', default => 'twitter.irc' );

=item irc_server_port

(Optional) The port number the IRC server binds to. Defaults to 6667.

=cut

has irc_server_port     => ( isa => 'Int', is => 'ro', default => 6667 );

=item irc_server_bindaddr

(Optional) The local address to bind to. Defaults to all interfaces.

=cut

# will be defaulted to INADDR_ANY by POE::Wheel::SocketFactory
has irc_server_bindaddr => ( isa => 'Str', is => 'ro', default => undef );

=item irc_mask

(Optional) The IRC user/host mask used to restrict connecting users.  Defaults to C<*@127.0.0.1>.

=cut

has irc_mask            => ( isa => 'Str', is => 'ro', default => '*@127.0.0.1' );


=item irc_password

(Optional) Password used to authenticate to the IRC server.

=cut

has irc_password        => ( isa => 'Str', is => 'ro' );


=item irc_botname

(Optional) The name of the channel operator bot.  Defaults to C<tweeter>.  Select a name
that does not conflict with friends, followers, or your own IRC nick.

=cut

has irc_botname         => ( isa => 'Str', is => 'ro', default => 'tweeter' );


=item irc_botircname

(Optional) Text to be used as the channel operator bot's IRC full name.

=cut

has irc_botircname      => ( isa => 'Str', is => 'ro', default => 'Your friendly Twitter Agent' );


=item irc_channel

(Optional) The name of the channel to use.  Defaults to C<&twitter>.

=cut

has irc_channel         => ( isa => 'Str', is => 'ro', default => '&twitter' );


=item twitter_retry

(Optional) The number of seconds between polls for new status updates.  Defaults to 300
(5 minutes).  Twitter imposes a rate limit of 100 API calls per hour.  By default,
after initial start up, twirc makes a single API call every C<twitter_retry>
seconds.  Adding L</"check_replies"> and L</"check_direct_messages"> each
add an additional API call.  Setting C<twitter_retry> too low can cause twirc
to exceed the rate limit and delay receipt of messages.

Use the L</"rate_limit_status"> command to check your available API calls.

=cut

has twitter_retry       => ( isa => 'Int', is => 'ro', default => 300 );


=item twitter_retry_on_error

(Optional) The number of seconds to wait before retrying a failed poll for friends,
followers, or status updates.  Defaults to 60 (1 minute).

=cut

has twitter_retry_on_error => ( isa => 'Int', is => 'ro', default => 60 );


=item twitter_alias

(Optional) An alias to use for displaying incoming status updates from the owning user.
This is necessary if the user's IRC nickname and Twitter screen name are the
same.  Defaults to C<me>.

=cut

has twitter_alias       => ( isa => 'Str', is => 'ro', default => 'me' );

=item echo_posts

(Optional) If false, posts sent by L<POE::Component::Server::Twirc> will not be redisplayed when received
is the friends_timeline.  Defaults to false.

Set C<echo_posts(1)> to see your own tweets in chronological order with the others.

=cut

has echo_posts => ( isa => 'Bool', is => 'rw', default => 0 );

=item favorites_count

(Optional) How many favorites candidates to display for selection. Defaults to 3.

=cut

has favorites_count => ( isa => 'Int', is => 'ro', default => 3 );

=item truncate_to

(Optional) When displaying tweets for selection, they will be truncated to this length.
Defaults to 60.

=cut

has truncate_to         => ( isa => 'Int', is => 'ro', default => 60 );

=item check_replies

(Optional) If true, checks for @replies when polling for friends' timeline updates
and merges them with normal status updates.  Normally, only replies from
friends are displayed.  This provides the display of @replies from
users not followed.

C<check_replies> adds an API call, counted against Twitter's rate limit
every L</"twitter_retry"> seconds.

This also has the effect of adding senders of @replies to the channel,
even though they are not followed.

=cut

has check_replies => ( isa => 'Bool', is => 'rw', default => 0 );

=item check_direct_messages

(Optional) If true, checks for direct messages in each timeline polling cycle.


C<check_direct_messages> adds an API call, counted against Twitter's rate limit
every L</"twitter_retry"> seconds.

=cut

has check_direct_messages => ( isa => 'Bool', is => 'rw', default => 0 );

=item log_channel

(Optional) If specified, twirc will post log messages to this channel.

=cut

has log_channel => ( isa => 'Str', is => 'ro' );

=item state_file

(Optional) File used to store state information between sessions, including last message read for
replies, direct messages, and timelines.

=cut

has state_file => ( isa => 'Str', is => 'ro' );

=back

=cut

has _ircd => (
       accessor => 'ircd', isa => 'POE::Component::Server::IRC', is => 'rw', weak_ref => 1 );
has _twitter => (
       accessor => 'twitter',  isa => 'Net::Twitter', is => 'rw' );
has _users => (
       accessor => 'users', isa => 'HashRef[Str]', is => 'rw', lazy => 1, default => sub { {} } );
has _joined => (
       accessor => 'joined', isa => 'Bool', is => 'rw', default => 0 );
has _tweet_stack => (
       accessor => 'tweet_stack', isa => 'ArrayRef[HashRef]', is => 'rw', default => sub { [] } );
has _dm_stack => (
       accessor => 'dm_stack', isa => 'ArrayRef[HashRef]', is => 'rw', default => sub { [] } );
has _stash => (
       accessor => 'stash', isa => 'Maybe[HashRef]', is => 'rw' );
has _state => (
       accessor => 'state', isa => 'POE::Component::Server::Twirc::State', is => 'rw',
       default => sub { POE::Component::Server::Twirc::State->new } );

sub post_ircd {
    my $self = shift;
    $self->ircd->yield(@_);
}

sub bot_says  {
    my ($self, $channel, $text) = @_;

    $self->post_ircd('daemon_cmd_privmsg', $self->irc_botname, $channel, $text);
};

sub bot_notice {
    my ($self, $channel, $text) = @_;

    $self->post_ircd(daemon_cmd_notice => $self->irc_botname, $channel, $text);
}


sub twitter_error {
    my ($self, $text) = @_;

    $self->bot_notice($self->irc_channel, "Twitter error: $text");
};

# set topic from status, iff newest status
sub set_topic {
    my ($self, $status) = @_;

    $self->post_ircd(daemon_cmd_topic => $self->irc_botname, $self->irc_channel,
           decode_entities($status->{text}));
};

# match any nick
sub nicks_alternation {
    my $self = shift;

    return join '|', map quotemeta, keys %{$self->users};
}

sub START {
    my ($self) = @_;

    $self->ircd(
        POE::Component::Server::IRC->spawn(
            config => {
                servername => $self->irc_server_name,
                nicklen    => 15,
                network    => 'SimpleNET'
            },
            inline_states => {
                _stop  => sub { $self->log->debug('[ircd:stop]') },
            },
        )
    );

    # register ircd to receive events
    $self->post_ircd('register' );
    $self->ircd->add_auth(
        mask     => $self->irc_mask,
        password => $self->irc_password,
    );
    $self->post_ircd('add_listener', port     => $self->irc_server_port,
                                     bindaddr => $self->irc_server_bindaddr);

    # add super user
    $self->post_ircd(
        add_spoofed_nick =>
        { nick => $self->irc_botname, ircname => $self->irc_botircname }
    );
    $self->post_ircd(daemon_cmd_join => $self->irc_botname, $self->irc_channel);

    # logging
    if ( $self->log_channel ) {
        $self->post_ircd(daemon_cmd_join => $self->irc_botname, $self->log_channel);
        my $logger = Log::Log4perl->get_logger('');
        my $appender = Log::Log4perl::Appender->new(
            'POE::Component::Server::Twirc::LogAppender',
            name        => 'twirc-logger',
            ircd        => $self->ircd,
            irc_botname => $self->irc_botname,
            irc_channel => $self->log_channel,
        );
        $logger->add_appender($appender);
    }

    $self->yield('friends');
    $self->yield('user_timeline'); # for topic setting
    $self->yield('delay_friends_timeline');

    $self->twitter(Net::Twitter->new(
        useragent_class => 'LWP::UserAgent::POE',
        username  => $self->twitter_username,
        password  => $self->twitter_password,
        useragent => "twirc/$VERSION",
        source    => 'twircgw',
    ));

    if ( $self->state_file && -r $self->state_file ) {
        eval {
            $self->state(POE::Component::Server::Twirc::State->load($self->state_file))
        };
        if ( $@ ) {
            $@ =~ s/ at .*//s;
            $self->log->error($@);
        }
    }

    return $self;
}

# Without detaching the ircd child session, the application will not
# shut down.  Bug in PoCo::Server::IRC?
event _child => sub {
    my ($self, $kernel, $event, $child) = @_[OBJECT, KERNEL, ARG0, ARG1];

    $self->log->debug("[_child] $event $child");
    $kernel->detach_child($child) if $event eq 'create';
};

event poco_shutdown => sub {
    my ($self) = @_;

    $self->log->debug("[poco_shutdown]");
    $_[KERNEL]->alarm_remove_all();
    $self->post_ircd('unregister');
    $self->post_ircd('shutdown');
    if ( $self->state_file ) {
        eval { $self->state->store($self->state_file) };
        if ( $@ ) {
            $@ =~ s/ at .*//s;
            $self->log->error($@);
        }
    }
};

########################################################################
# IRC events
########################################################################

event ircd_daemon_nick => sub {
    my ($self, $sender, $nick, $new_nick, $host) = @_[OBJECT, SENDER, ARG0, ARG1, ARG5];

    $self->log->debug("[ircd_daemon_nick] $nick, $new_nick, $host");

    return if $nick eq $self->irc_botname;

    $self->log->debug("    nick = $nick");

    # Abuse!  Calling the private implementation of ircd to force-join the connecting
    # user to the twitter channel. ircd set's it's heap to $self: see ircd's perldoc.
    $sender->get_heap()->_daemon_cmd_join($nick, $self->irc_channel);
};

event ircd_daemon_join => sub {
    my($self, $sender, $user, $ch) = @_[OBJECT, SENDER, ARG0, ARG1];

    $self->log->debug("[ircd_daemon_join] $user, $ch");
    return unless my($nick) = $user =~ /^([^!]+)!/;
    return if $self->users->{$nick};
    return if $nick eq $self->irc_botname;

    if ( $ch eq $self->irc_channel ) {
        $self->joined(1);
        $self->log->debug("    joined!");
        $self->yield('display_direct_messages');
        $self->yield('throttle_messages');
        return;
    }
    elsif ( $self->log_channel && $ch eq $self->log_channel ) {
        my $appender = Log::Log4perl->appender_by_name('twirc-logger');
        $appender->dump_history;
    }
    else {
        $self->log->debug("    ** part **");
        # only one channel allowed
        $sender->get_heap()->_daemon_cmd_part($nick, $ch);
    }
};

event ircd_daemon_part => sub {
    my($self, $user, $ch) = @_[OBJECT, ARG0, ARG1];

    return unless my($nick) = $user =~ /^([^!]+)!/;
    return if $self->users->{$nick};
    return if $nick eq $self->irc_botname;

    $self->joined(0) if $ch eq $self->irc_channel;
};

event ircd_daemon_quit => sub {
    my($self, $user) = @_[OBJECT, ARG0];

    $self->log->debug("[ircd_daemon_quit]");
    return unless my($nick) = $user =~ /^([^!]+)!/;
    return if $self->users->{$nick};
    return if $nick eq $self->irc_botname;

    $self->joined(0);
    $self->yield('poco_shutdown');
};

event ircd_daemon_public => sub {
    my ($self, $user, $channel, $text) = @_[OBJECT, ARG0, ARG1, ARG2];

    return unless $channel eq $self->irc_channel;

    my $nick = ( $user =~ m/^(.*)!/)[0];
    $self->log->debug("[ircd_daemon_public] $nick: $text");
    return unless $nick eq $self->irc_nickname;

    # give any command handler a shot
    if ( $self->stash ) {
        $self->log->debug("stash exists...");
        my $handler = delete $self->stash->{handler};
        if ( $handler ) {
            return if $self->$handler($channel, $text); # handled
        }
        else {
            $self->log->error("stash exsits with no handler");
        }
        # the user ignored a command completion request, kill it
        $self->stash(undef);
    }

    # treat "nick: ..." as "post @nick ..."
    my $nick_alternation = $self->nicks_alternation;
    if ( $text =~ s/^($nick_alternation):\s+/\@$1 /i ) {
        $self->yield(cmd_post => $channel, $text);
        return;
    }

    my ($command, $argstr) = split /\s+/, $text, 2;
    if ( $command =~ /^\w+$/ ) {
        my $event = "cmd_$command";
        if ( $self->can($event) ) {
            $self->yield($event, $channel, $argstr);
        }
        else {
            $self->bot_says($channel, qq/I don't understand "$command". Try "help"./)
        }
    }
    else {
        $self->bot_says($channel, qq/That doesn't look like a command. Try "help"./);
    }
};

event ircd_daemon_privmsg => sub {
    my ($self, $user, $target_nick, $text) = @_[OBJECT, ARG0..ARG2];

    # owning user is the only one allowed to send direct messages
    my $me = $self->irc_nickname;
    return unless $user =~ /^\Q$me\E!/;

    unless ( $self->users->{$target_nick} ) {
        # TODO: handle the error the way IRC would?? (What channel?)
        $self->bot_says($self->irc_channel, qq/You don't appear to be following $target_nick; message not sent./);
        return;
    }

    unless ( eval { $self->twitter->new_direct_message({ user => $target_nick, text => $text }) } ) {
        # TODO what channel?
        $self->bot_says($self->irc_channel, "new_direct_message failed.");
    }
};

########################################################################
# Twitter events
########################################################################

# This is the main loop; check for updates every twitter_retry seconds.
event delay_friends_timeline => sub {
    my ($self) = @_;

    $self->yield('direct_messages') if $self->check_direct_messages;
    $self->yield('friends_timeline');
    $_[KERNEL]->delay(delay_friends_timeline => $self->twitter_retry);
};

event throttle_messages => sub {
    my ($self) = @_;

    $self->log->debug("[throttle_messages] ", scalar @{$self->tweet_stack}, " messages");

    for my $entry ( @{$self->tweet_stack} ) {
        my @lines = split /\r?\n/, $entry->{text};
        $self->post_ircd(daemon_cmd_privmsg => $entry->{name}, $self->irc_channel, $_)
            for @lines;
    }

    $self->tweet_stack([]);
};

# Add friends to the channel
event friends => sub {
    my ($self, $page ) = @_[OBJECT, ARG0];

    my $retry = $self->twitter_retry_on_error;

    $self->log->debug("[twitter:friends] calling...");
    $page ||= 1;
    for (;;) {
        my $friends = eval { $self->twitter->friends({page => $page}) }; 
        unless ( $friends ) {
            $self->twitter_error("request for friends failed; retrying in $retry seconds");
            $_[KERNEL]->delay(friends => $retry, $page);
            return;
        }
        $self->log->debug("    friends returned ", scalar @$friends, " friends");

        ++$page;

        # Current API gets 100 friends per page.  If we have exactly 100 friends
        # we have to try again with page=2 and we should get (I'm assuming, here)
        # an empty arrayref.  What if the API changes to 200, etc.?  Might as well
        # just loop until we get an empty arrayref.  That will handle either case.
        last unless @$friends;

        for my $friend ( @$friends ) {
            my ($nick, $name) = @{$friend}{qw/screen_name name/};

            next if $self->users->{$nick};
            $self->post_ircd(add_spoofed_nick => { nick => $nick, ircname => $name });
            $self->post_ircd(daemon_cmd_join => $nick, $self->irc_channel);
            $self->users->{$nick} = $friend;
        }
    }
    $self->yield('followers');
};

# Give friends who are also followers voice; it's just a visual hint to the user.
event followers => sub {
    my ($self, $page ) = @_[OBJECT, ARG0];

    my $retry = $self->twitter_retry_on_error;

    $self->log->debug("[twitter:followers] calling...");
    $page ||= 1;
    while ( my $followers = eval { $self->twitter->followers({page => $page}) } ) {
        $self->log->debug("    page: $page");
        unless ( $followers ) {
            $self->twitter_error("request for followers failed; retrying in $retry seconds");
            $_[KERNEL]->delay(followers => $retry, $page);
            return;
        }
        ++$page;

        $self->log->debug("    followers returned ", scalar @$followers, " followers");

        # see comments for event friends
        last unless @$followers;

        for my $follower ( @$followers ) {
            my $nick = $follower->{screen_name};
            if ( $self->users->{$nick} ) {
                $self->post_ircd(daemon_cmd_mode =>
                    $self->irc_botname, $self->irc_channel, '+v', $nick);
            }
        }
    }
};

event direct_messages => sub {
    my ($self) = @_;

    # We don't want to flood the user with DMs, so if this is the first time,
    # i.e., no DM id in saved state, just set the high water mark and return.
    unless ( $self->state->direct_message_id ) {
        my $high_water = $self->twitter->direct_messages;
        if ( $high_water ) {
            $self->state->direct_message_id($high_water->[0]{id}) if @$high_water;
        }
        else {
            $self->twitter_error('direct_messages failed');
        }
        return;
    }

    my $messages = $self->twitter->direct_messages({ since_id => $self->state->direct_message_id });
    unless ( $messages ) {
        $self->twitter_error('direct_messages failed');
        return;
    }

    if ( @$messages ) {
        $self->state->direct_message_id($messages->[0]{id});

        for my $msg ( reverse @$messages ) {
            my ($nick, $ircname) = @{$msg->{sender}}{qw/screen_name name/};
            unless ( $self->users->{$nick} ) {
                $self->log->warn("Joining $nick from a direct message; expected $nick already joined.");
                $self->post_ircd(add_spoofed_nick => { nick => $nick, ircname => $ircname });
                $self->post_ircd(daemon_cmd_join => $nick, $self->irc_channel);
                $self->users->{$nick} = {}; # don't have a status to store
            }

            push @{$self->dm_stack}, { name => $nick, text => $msg->{text} };
        }
        $self->yield('display_direct_messages') if $self->joined;
    }
};

event display_direct_messages => sub {
    my ($self) = @_;

    while ( my $msg = shift @{$self->dm_stack} ) {
        my @lines = split /\r?\n/, $msg->{text};
        $self->post_ircd(daemon_cmd_privmsg => $msg->{name}, $self->irc_nickname, $_)
            for @lines;
    }
};

event friends_timeline => sub {
    my ($self) = @_;

    $self->log->debug("[friends_timeline]");

    my $statuses = eval {
        $self->twitter->friends_timeline({
            since_id => $self->state->friends_timeline_id
        });
    };

    unless ( $statuses ) {
        $self->twitter_error('friends_timeline request failed');
        return;
    }

    $self->log->debug("    friends_timeline returned ", scalar @$statuses, " statuses");
    $self->state->friends_timeline_id($statuses->[0]{id}) if @$statuses;

    $statuses = $self->merge_replies($statuses);

    my $channel = $self->irc_channel;
    my $new_topic;
    for my $status (reverse @{ $statuses }) {
        my ($name, $ircname) = @{$status->{user}}{qw/screen_name name/};
        my $text = decode_entities($status->{text});

        # alias our twitter_name if configured
        # (to avoid a collision in case our twitter screen name and irc nick are the same)
        $self->log->debug("    \$name = $name, \$twitter_name = "), $self->twitter_screen_name;

        # message from self
        if ( $name eq $self->twitter_screen_name ) {
            $new_topic = $status unless $status =~ /^\s*\@/;

            # TODO: is this even necessary? Can we just send a privmsg from a real user?
            $name = $self->twitter_alias if $self->twitter_alias;
            next if !$self->echo_posts && $status->{id} <= ($self->state->user_timeline_id || 0);

            $self->state->user_timeline_id($status->{id})
                if $status->{id} > ($self->state->user_timeline_id || 0);
        }

        unless ( $self->users->{$name} ) {
            $self->post_ircd(add_spoofed_nick => { nick => $name, ircname => $ircname });
            $self->post_ircd(daemon_cmd_join => $name, $channel);
        }
        $self->users->{$name} = $status->{user};

        $self->log->debug("    { $name, $text }");
        push @{ $self->tweet_stack }, { name => $name, text => $text }
    }

    unless (@$statuses) {
      $self->bot_notice($channel, "That refresh didn't get any new tweets.");
    }

    $self->set_topic($new_topic) if $new_topic;
    $self->yield('throttle_messages') if $self->joined;

    # periodically store state
    if ( $self->state_file ) {
        eval { $self->state->store($self->state_file) };
        if ( $@ ) {
            $@ =~ s/ at .*//s;
            $self->log->error($@);
        }
    }
};

sub merge_replies {
    my ($self, $statuses) = @_;
    return $statuses unless $self->check_replies;

    # TODO: find a better way to initialize this??
    unless ( $self->state->reply_id ) {
        $self->state->reply_id(
            @$statuses ? $statuses->[-1]{id} : $self->state->user_timeline_id
         );
    }

    my $replies = eval {$self->twitter->replies({ since_id => $self->state->reply_id }) };
    if ( $replies ) {
        if ( @$replies ) {
            $self->log->debug("[merge_replies] ", scalar @$replies, " replies");

            $self->state->reply_id($replies->[0]{id});

            # TODO: clarification needed: I'm assuming we get replies
            # from friends in *both* friends_timeline and replies,
            # so, we need to weed them.
            my %seen = map { ($_->{id}, $_) } @{$statuses}, @{$replies};

            $statuses = [ sort { $b->{id} <=> $a->{id} } values %seen ];
        }
    }
    else {
        $self->twitter_error('replies failed');
    }
    return $statuses;
}

event user_timeline => sub {
    my ($self) = @_;

    $self->log->debug("[user_timetline] calling...");
    my $statuses = eval { $self->twitter->user_timeline };
    unless ( $statuses ) {
        $self->twitter_error($self->irc_channel, 'user_timeline request failed; retrying in 60 seconds');
        $_[KERNEL]->delay(user_timeline => 60);
    }
    $self->log->debug("    urser_timeline returned");

    return unless @$statuses;

    $self->state->user_timeline_id($statuses->[0]{id});
    for my $status ( @$statuses ) {
        # skip @replies
        unless ( $status->{text} =~ /^\s*\@/ ) {
            $self->set_topic($status);
            return;
        }
    }

    #couldn't find an non-@reply status, punt
    $self->set_topic($statuses->[0]);
};

########################################################################
# Commands
########################################################################

=head2 COMMANDS

Commands are entered as public messages in the IRC channel in the form:

    command arg1 arg2 ... argn

Where the arguments, if any, depend upon the command.

=over 4

=item post I<status>

Post a status update.  E.g.,

    post Now cooking tweets with twirc!

=cut

event cmd_post => sub {
    my ($self, $channel, $text) = @_[OBJECT, ARG0, ARG1];

    $self->log->debug("[cmd_post_status]");

    if ( (my $n = length($text) - 140) > 0 ) {
        $self->bot_says($channel, "Message not sent; $n characters too long. Limit is 140 characters.");
        return;
    }

    my $status = eval { $self->twitter->update($text) };
    unless ( $status ) {
        $self->bot_says($channel, 'status update failed; try again later');
        return;
    }

    $self->log->debug("    update returned $status");

    $self->set_topic($status);
    $self->state->user_timeline_id($status->{id});
};

=item follow I<id>

Follow a new Twitter user, I<id>.  In Twitter parlance, this creates a friendship.

=cut

event cmd_follow => sub {
    my ($self, $channel, $id) = @_[OBJECT, ARG0, ARG1];

    if ( $self->users->{$id} ) {
        $self->bot_says($channel, qq/You're already following $id./);
        return;
    }
    elsif ( $id !~ /^\w+$/ ) {
        $self->bot_says($channel, qq/"$id" doesn't look like a user ID to me./);
        return;
    }

    my $friend = eval { $self->twitter->create_friend($id) };
    unless ( $friend ) {
        $self->bot_says($channel, 'create_friend failed');
        return;
    }

    my ($nick, $name) = @{$friend}{qw/screen_name name/};
    $self->post_ircd('add_spoofed_nick', { nick => $nick, ircname => $name });
    $self->post_ircd(daemon_cmd_join => $name, $self->irc_channel);
    $self->users->{$nick} = $friend;

    if ( eval { $self->twitter->relationship_exists($nick, $self->twitter_screen_name) } ) {
        $self->post_ircd(daemon_cmd_mode =>
            $self->irc_botname, $self->irc_channel, '+v', $nick);
        $self->bot_notice($channel, qq/Now following $id./);
    }
};

=item unfollow I<id>

Stop following Twitter user I<id>.  In Twitter, parlance, this destroys a
friendship.

=cut

event cmd_unfollow => sub {
    my ($self, $channel, $id) = @_[OBJECT, ARG0, ARG1];

    if ( !$self->users->{$id} ) {
        $self->bot_says($channel, qq/You don't appear to be following $id./);
        return;
    }

    my $friend = eval { $self->twitter->destroy_friend($id) };
    unless ( $friend ) {
        $self->bot_says($channel, 'destroy_friend failed');
        return;
    }

    $self->post_ircd(daemon_cmd_part => $id, $self->irc_channel);
    $self->post_ircd(del_spooked_nick => $id);
    delete $self->users->{$id};
    $self->bot_notice($channel, qq/No longer following $id./);
};

=item block I<id>

Block Twitter user I<id>.

=cut

event cmd_block => sub {
    my ($self, $channel, $id) = @_[OBJECT, ARG0, ARG1];

    if ( $id !~ /^\w+$/ ) {
        $self->bot_says($channel, qq/"$id" doesn't look like a user ID to me./);
        return;
    }

    unless ( eval { $self->twitter->create_block($id) } ) {
        $self->bot_says($channel, 'create_block failed');
        return;
    }

    if ( $self->users->{$id} ) {
        $self->post_ircd(daemon_cmd_mode =>
            $self->irc_botname, $self->irc_channel, '-v', $id);
        $self->bot_notice($channel, qq/Blocked $id./);
    }
};

=item unblock I<id>

Stop blocking Twitter user I<id>.

=cut

event cmd_unblock => sub {
    my ($self, $channel, $id) = @_[OBJECT, ARG0, ARG1];

    if ( $id !~ /^\w+$/ ) {
        $self->bot_says($channel, qq/"$id" doesn't look like a user ID to me./);
        return;
    }

    unless ( eval { $self->twitter->destroy_block($id) } ) {
        $self->bot_says($channel, 'destroy_block failed');
        return;
    }

    if ( $self->users->{id} ) {
        $self->post_ircd(daemon_cmd_mode =>
            $self->irc_botname, $self->irc_channel, '+v', $id);
        $self->bot_notice($channel, qq/Unblocked $id./);
    }
};

=item whois I<id>

Displays information about Twitter user I<id>, including name, location, and
description.

=cut

event cmd_whois => sub {
    my ($self, $channel, $id) = @_[OBJECT, ARG0, ARG1];

    $self->log->debug("[cmd_whois] $id");

    my $user = $self->users->{$id};
    unless ( $user ) {
        $self->log->debug("     $id not in users; fetching");
        my $arg = Email::Valid->address($id) ? { email => $id } : { id => $id };
        $user = eval { $self->twitter->show_user($arg) };
    }
    if ( $user ) {
        $self->bot_says($channel, "$user->{screen_name} [$user->{id}]: $user->{name}, $user->{location}");
        for ( @{$user}{qw/description url/} ) {
            $self->bot_says($channel, $_) if $_;
        }
    }
    else {
        $self->bot_says($channel, "I don't know $id.");
    }
};

=item notify I<on|off> I<id ...>

Turns device notifications on or off for the list of Twitter IDs.

=cut

event cmd_notify => sub {
    my ($self, $channel, $argstr) = @_[OBJECT, ARG0, ARG1];

    my @nicks = split /\s+/, $argstr;
    my $onoff = shift @nicks;

    unless ( $onoff && $onoff =~ /^on|off$/ ) {
        $self->bot_says($channel, "Usage: notify [on|off] nick[ nick [...]]");
        return;
    }

    my $method = $onoff eq 'on' ? 'enable_notifications' : 'disable_notifications';
    for my $nick ( @nicks ) {
        unless ( eval { $self->twitter->$method({ id => $nick }) } ) {
            $self->bot_says($channel, "notify $onoff failed for $nick");
        }
    }
};

=item favorite I<friend> [I<count>]

Mark I<friend>'s tweet as a favorite.  Optionally, specify the number of tweets
to display for selection with I<count> (Defaults to 3.)

=cut

event cmd_favorite => sub {
    my ($self, $channel, $args) = @_[OBJECT, ARG0, ARG1];

    my ($nick, $count) = split /\s+/, $args;
    $count ||= $self->favorites_count;

    $self->log->debug("[cmd_favorite] $nick");

    unless ( $self->users->{$nick} ) {
        $self->bot_says($channel, "You're not following $nick.");
        return;
    }

    my $recent = eval { $self->twitter->user_timeline({ id => $nick, count => $count }) };
    unless ( $recent ) {
        $self->bot_says($channel, 'user_timeline failed');
        return;
    }
    if ( @$recent == 0 ) {
        $self->bot_says($channel, "$nick has no recent tweets");
        return;
    }

    $self->stash({
        favorite_candidates => [ map $_->{id}, @$recent ],
        handler => 'handle_favorite',
    });

    $self->bot_says($channel, 'Which tweet?');
    for ( 1..@$recent ) {
        $self->bot_says($channel, "[$_] " . truncstr($recent->[$_ - 1]{text}, $self->truncate_to));
    }
};

sub handle_favorite {
    my ($self, $channel, $index) = @_;

    $self->log->debug("[handle_favorite] $index");

    my @favorite_candidates = @{$self->stash->{favorite_candidates} || []};
    if ( $index =~ /^\d+$/ && 0 < $index && $index <= @favorite_candidates ) {
        if ( eval { $self->twitter->create_favorite({
                    id => $favorite_candidates[$index - 1]
                }) } ) {
            $self->bot_notice($channel, 'favorite added');
        }
        else {
            $self->bot_says($channel, 'create_favorite failed');
        }
        $self->stash(undef);
        return 1; # handled
    }
    return 0; # unhandled
};

=item check_replies I<on|off>

Turns reply checking on or off.  See L</"check_replies"> in configuration.

=cut

event cmd_check_replies => sub {
    my ($self, $channel, $onoff) = @_[OBJECT, ARG0, ARG1];

    unless ( $onoff && $onoff =~ /^on|off$/ ) {
        $self->bot_says($channel, "Usage: check_replies [on|off]");
        return;
    }
    $self->check_replies($onoff eq 'on' ? 1 : 0);
};

=item check_direct_messages I<on|off>

Turns direct message checking on or off.  See L</"check_direct_messages"> in configuration.

=cut

event cmd_check_direct_messages => sub {
    my ($self, $channel, $onoff) = @_[OBJECT, ARG0, ARG1];

    unless ( $onoff && $onoff =~ /^on|off$/ ) {
        $self->bot_says($channel, "Usage: check_replies [on|off]");
        return;
    }
    $self->check_direct_messages($onoff eq 'on' ? 1 : 0);
};

=item rate_limit_status

Displays the remaining number of API requests available in the current hour.

=cut

event cmd_rate_limit_status => sub {
    my ($self, $channel) = @_[OBJECT, ARG0];

    if ( defined(my $r = $self->twitter->rate_limit_status) ) {
        my $reset_time = sprintf "%02d:%02d:%02d", (localtime $r->{reset_time_in_seconds})[2,1,0];
        my $seconds_remaning = $r->{reset_time_in_seconds} - time;
        my $time_remaning = sprintf "%d:%02d", int($seconds_remaning / 60), $seconds_remaning % 60;
        $self->bot_says($channel, <<"");
$r->{remaining_hits} API calls remaining for the next $time_remaning (until $reset_time), hourly limit is $r->{hourly_limit}

    }
    else {
        $self->bot_says($channel, "rate_limit_status failed");
    }
};

=item help

Display a simple help message

=cut

event cmd_help => sub {
    my ($self, $channel, $argstr)=@_[OBJECT, ARG0, ARG1];
    $self->bot_says($channel, "Available commands:");
    $self->bot_says($channel, join ' ' => sort qw/
        post follow unfollow block unblock whois notify refresh favorite
        check_replies rate_limit_status
    /);
    $self->bot_says($channel, '/msg nick for a direct message.')
};

event cmd_refresh => sub {
    my ($self) = @_;

    $self->yield('delay_friends_timeline');
};

1;

__END__

=item /msg I<id> I<text>

Sends a direct message to Twitter user I<id> using an IRC private message.

=back

=head1 SEE ALSO

L<App::Twirc>

=head1 AUTHOR

Marc Mims <marc@questright.com>

=head1 LICENSE

Copyright (c) 2008 Marc Mims

You may distribute this code and/or modify it under the same terms as Perl itself.
