use v6;

use TMap::Topic;
use TMap::TopicMapConstruct;
use TMap::Association;
use TMap::Reifiable;

package TMap:ver<0.2.0> {

  role TopicMap {

    has @.topicmap-documents;
    has TMap::Topic %.topics;


    #---------------------------------------------------------------------------
    # Initialize topicmap
    #
    method initialize ( Str $topicmap-fn ) {

      # Process this topicmap to merge all other topicmaps pointed by <mergeMap>
      #
      self.merge-topicmaps($topicmap-fn);

      # Process the topic map topic ids
      #
      self.process-topic-ids;

      # Process the topic map items
      #
      self.process-topicmap-items;

      # Combine and remove any duplicates in the topicmap
      #
      self.merge-any-duplicates;
    }

    #---------------------------------------------------------------------------
    # Load a topicmap semi-xml document and return a topicmap object
    #
    method load-topicmap ( Str $topicmap-fn --> TMap::TopicMap ) {

      my Semi-xml $x .= new(:init);

      # Split filename in its parts
      #
      my @path-spec = $*SPEC.splitpath($topicmap-fn);
      $x.configuration<output><filepath> = @path-spec[1];
      $x.configuration<output><filename> = @path-spec[2];

      # Drop extension
      #
      $x.configuration<output><filename> ~~ s/\.<-[\.]>+$//;

      my $topicmap;
      if $topicmap-fn.IO ~~ :r {
say "TM: $topicmap-fn";
        $x.parse-file(:filename($topicmap-fn));

        # Get the root element and promote it to Topicmap
        #
        $topicmap = $x.root-element.cloneNode;
        $x.root-element.remove;

        # Check topicmap attributes
        #
        if ?$topicmap.attribs<version> and $topicmap.attribs<version> ne '2.0' {
          die "Bad topicmap version";
        }

        # Here we promote to TMap::TopicMapConstruct, TMap::Reifiable
        # and  TMap::TopicMap beforehand
        #
        $topicmap does TMap::TopicMapConstruct;
        $topicmap does TMap::Reifiable;
        $topicmap does TMap::TopicMap;
        $topicmap.set-filename($topicmap-fn);
        $topicmap.check-item-identifiers;
      }

      else {
        die "Failed to read $topicmap-fn";
      }

      return $topicmap;
    }

    #---------------------------------------------------------------------------
    # Merge topic maps using the mergeMap element and replacing this with the
    # loaded map. All references of id's in any map must be made absolute or
    # relative to the starting map.
    #
    method merge-topicmaps ( Str $topicmap-fn ) {

      # Store filename
      #
      self.set-filename($topicmap-fn);
      self.check-item-identifiers;

      # Process the topic map items
      #
      self.set-topicmap-items-construct-role( $topicmap-fn, self);

      # Store document name
      #
      self.add-topicmap-doc($*SPEC.rel2abs($topicmap-fn));
      my Bool $have-merged = True;

      # Merge topicmaps
      #
#      while $have-merged {
#        $have-merged = False;

#        for self.nodes -> $node {
        my $node-count = 0;
        while $node-count < self.nodes.elems {
          my $node = self.nodes[$node-count++];

          if $node ~~ XML::Element and $node.name eq 'mergeMap' {
            my $mm-topicmap-fn = $node.attribs<href>;

            # Store document name. Check if it has been seen before. If so
            # skip rest of the loop.
            #
            if self.add-topicmap-doc($mm-topicmap-fn) {
              note "File $mm-topicmap-fn already loaded";
              $node.remove;
              next;
            }

            # Load the topicmap and remove <mergeMap>.
            #
            my $mm-topicmap = self.load-topicmap($mm-topicmap-fn);
            $node.remove;

            # Process the topic map items to add the TMap::TopicMapConstruct
            # role
            #
            self.set-topicmap-items-construct-role( $mm-topicmap-fn, $mm-topicmap);

            # Copy all xml nodes into the main topicmap all new mergemaps
            # included. $have-merged will be set which will cause another round
            # through the nodes.
            #
            for $mm-topicmap.nodes -> $mm-node {
              self.append($mm-node);
            }


#            $have-merged = True;
          }
        }
      }
#    }

    #---------------------------------------------------------------------------
    # Process any elements in the topic map.to apply the basic roles
    # TMap::TopicMapConstruct and TMap::Reifier. With this a filename can be
    # stored to construct a proper reference to topics. This is done for every
    # element except for topicmaps which is set when loading the map. See
    # load-topicmap(). The reifiable role is applied to all but the topic
    # object.
    #
    method set-topicmap-items-construct-role (
      Str $topicmap-fn,
      $merged-node where $merged-node ~~ any(XML::Element|TMap::TopicMap)
    ) {

      for $merged-node.nodes -> $node {
        if $node ~~ XML::Element {
          next unless
            $node.name ~~ any < topic name occurrence type scope value variant
                                association role topicRef
                              >;

          $node does TMap::TopicMapConstruct;
          $node does TMap::Reifiable unless $node.name eq 'topic';
          $node.set-filename($*SPEC.rel2abs($topicmap-fn));

          self.set-topicmap-items-construct-role( $topicmap-fn, $node)
            if ?$node.nodes;
        }
      }
    }

    #-------------------------------------------------------------------------------
    # Process any topicmap items found on any item in the topic map.
    #
    method process-topicmap-items (  ) {

say "xmlns of {self.name}: ", self.attribs<xmlns>;
      self.set-reifier;

      # First get all topics and then the associations
      #
      for self.nodes -> $node {

        if $node ~~ XML::Element {
          given $node.name {

            # Process topic nodes
            #
            when 'topic' {
              # Some work already done. See process-topic-ids. E.g. roles were
              # added.
              #
              my TMap::Topic $topic := $node;
              $topic.initialize;
            }
            
            when 'association' {
              $node does TMap::Association;
              my TMap::Association $association := $node;
              $association.initialize(self);
            }

            # Ignored elements, will be removed later
            #
            when 'mergeMap' { }
            when 'itemIdentity' { }

            default {
              say "Unknown element name: $_";
            }
          }
        }
      }
    }

    #---------------------------------------------------------------------------
    # Process all topics and register the topics id in the topicmap along with
    # the topic object. When only one topicmap is given the second is set to the
    # first. The second map is processed and modified. The topicmap node is used
    # to make notes.
    #
    method process-topic-ids (  ) {

      for self.nodes -> $node {
        if $node ~~ XML::Element and $node.name eq 'topic' {

          # Promote to TMap::Topic, then set topic parent to that of the
          # topicmap and add this node to the topicmap.
          #
          $node does TMap::Topic;      
          my TMap::Topic $topic := $node;
          $topic.set-parent(self);

          # Convert id to an absolute IRI and then store it. If the IRI has a
          # '#' in the id then no conversion takes place and the topic must be
          # referenced exactly as specified.
          #
          my Str $topic-id = $topic.attribs<id>;
          if $topic-id !~~ m/ '#' / {
            $topic-id = [~] $topic.get('filename'), '#', $topic-id;
          }

          self.add-topic( $topic-id, $topic);
        }
      }
    }

    #---------------------------------------------------------------------------
    # Add topicmap document. Return True if already added before.
    #
    method add-topicmap-doc ( Str $topicmap-fn --> Bool ) {
      my Bool $found = False;
      
      if any(@!topicmap-documents) ~~ $topicmap-fn {
        $found = True;
      }

      else {
        @!topicmap-documents.push($topicmap-fn);
      }

      return $found;
    }

    #---------------------------------------------------------------------------
    #
    method add-topic ( Str $topic-id, TMap::Topic $topic ) {

      die "No topic id" unless ?$topic-id;

      # Convert id to an absolute IRI
      #
      # ...
my $tid = $topic-id;
$tid ~~ s/^ .**30/.../ unless $topic-id.chars < 70;
say "Topic id: $tid";

      # Check existence
      #
      if ?%!topics{$topic-id} {
        note "Topic with id $topic-id already stored, should be merged?";
      }

      else {
        %!topics{$topic-id} = $topic;
      }
    }

    #---------------------------------------------------------------------------
    #
    method get-topic-from-id ( Str $id --> TMap::Topic ) {
      return self.get( 'topic', $id);
    }

    #---------------------------------------------------------------------------
    # Exporting file to $xml-fn
    #
    method export-xml ( Str $xml-fn ) {

      my Semi-xml $x .= new(:init);
#      $x does sxml-role;

      # Split filename in its parts
      #
      my @path-spec = $*SPEC.splitpath('topicmap.xml');
      $x.configuration<output><filepath> = @path-spec[1];
      $x.configuration<output><filename> = @path-spec[2];
      $x.configuration<output><program><xml> =
       "| xmllint --format - > $xml-fn";

      my XML::Document $doc .= new(self);

      $x.save( :run-code('xml'), :other-document($doc));
    }

    #---------------------------------------------------------------------------
    #
    method merge-any-duplicates (  ) {

    }

    #---------------------------------------------------------------------------
    # Some getters
    #
    multi method get ( 'topic', Str $key --> TMap::Topic ) {

      my TMap::Topic $topic;
      if ?%!topics{$key} {
        $topic = %!topics{$key};
      }
      
      else {
        note "No topic found for key '$key', search externally?";
      }
      
      return $topic;
    }
  }
}
