########################################################################
# housekeeping
########################################################################

use v6.c;

my $Script;
my $bin_from;
my $real_bin    = False;

########################################################################
# module makes FindBin::Bin & ::Script available as late calls in 
# in the code available.
########################################################################

module FindBin:ver<0.1.0>:auth<CPAN:lembark>
{
    INIT given $*PROGRAM-NAME
    {
        when '-e' | '-' | 'interactive'
        {
            $Script     = $_.Str;
            $bin_from   = $*CWD;
        }

        default
        {
            $Script     = $*PROGRAM.basename;
            $bin_from   = $*PROGRAM;
        }
    }

    our sub Script  
    {
        $Script
    }
    our sub Bin ( $use_real = $real_bin --> Str )
    {
        $use_real
        ?? $bin_from.resolve.IO.dirname
        !! $bin_from.dirname.IO.absolute
    }
}

########################################################################
# push $Bin & Bin into the caller as requested, set the default 
# to RealBin.
########################################################################

our sub EXPORT( *@argz )
{
    my %varz;

    $real_bin   = ? first { $_ eq 'RealBin' }, @argz;

    for @argz
    {
        when '$Bin' { %varz< $Bin > =  FindBin::Bin() }
        when 'Bin'  { %varz< &Bin > = &FindBin::Bin   }
    }

    %varz
}

=finish

=begin pod

=head1 NAME

FindBin - Simple replacement of Perl5's FindBin for Perl6.

=head1 SYNOPSIS

    use FindBin <$Bin>;   

    my $dir_from    = $Bin;
    my $script_base = $Script;

    # export the Bin path after resolving any
    # symlinks by default (see argument to Bin,
    # below).

    use FindBin <$Bin RealBin>;

    my $dir_from    = $Bin;
    my $script_base = $Script;

    # export &Bin (vs. $Bin).
    # default to not resolving relative paths.
    
    use FindBin <Bin>;

    my $dir_from    = Bin;

    # default Bin to returning the 
    # resolved path without exporting
    # anything at all.

    use FindBin <RealBin>;

    my $path    = FindBin::Bin();

    # default to using the directory as-is,
    # without resolving any symlinks; override
    # default with argument to Bin call.

    use FindBin <Bin>;

    my $resolved    = Bin( True );
    my $symlinked   = Bin;

    # as above with default of resolving the
    # symlinks.

    use FindBin <Bin RealBin>;

    my $resolved    = Bin;
    my $symlinked   = Bin( False );

    # determine if the current executable is running
    # from a symlinked path. this will be true if the
    # resoled and -unersolved.


    use FindBin <Bin>;

    Bin( False ) ne Bin( True )
    and say "There is a symlink in '$Bin'";

=head1 DESCRIPTION

This module exports $Script and $Bin. If perl6 is running in 
interactive mode, with '-e', '-', or input redirected from stdin
then $Script is 'interactive', '-e', or '-' and $Bin is the 
current working directory from $*CWD.

=head2 Notes

    One difference between and the original FindBin is that "RealBin" 
    is not (yet) available, what you get is *$CWD.

=head1 SEE ALSO

Perl variables $*PROGRAM, $*PROGRAM-NAME:

    https://docs.perl6.org/language/variables#index-entry-%24%2APROGRAM

Class which implements the dirname, basename, and absolute methods 
use to determine the absolute path of $Bin.

    https://docs.perl6.org/type/IO::Path
    
=head1 AUTHOR

Steven Lembark <lembark@wrkhors.com>

=head1 COPYRIGHT AND LICENSE

Copyright 2018 lembark

This library is free software; you can redistribute it and/or modify 
it under the Artistic License 2.0.

=end pod

