########################################################################
# housekeeping
########################################################################

package Module::FromPerlVer::Git;
use 5.006;
use strict;
use version;
use parent  qw( Module::FromPerlVer::Extract );

use Archive::Tar;

use Cwd             qw( getcwd          );
use File::Basename  qw( basename        );
use FindBin         qw( $Bin            );
use List::Util      qw( first           );
use Symbol          qw( qualify_to_ref  );

########################################################################
# package variables & sanity checks
########################################################################

our $VERSION    = version->parse( 'v0.1.1' )->numify;

my $nil         = sub{};

# there is no consistent way to determine the location of git
# due to Windows not having a 'which' commnad. solution for
# now is ignoring the issue, letting the path do its work, and
# leaving 'git' without an absolute path.

my @checkout    = qw( git checkout                      );
my @clone       = qw( git clone     --no-checkout       );
my @restore     = qw( git checkout  --theirs            );

########################################################################
# utility subs
########################################################################

my $locate_tarball
= sub
{
    # note that $Bin may be ./t, in which case the
    # extract may to move up a level.

    my $extract = shift;
    my $base    = shift;
    my $root    = $Bin;

    print "# Locate tarball: '$base' ($root).";

    for(;;)
    {
        '/' eq $root
        and return;

        print "#\t$root";

        -e and return $_
        for "$root/$base";

        $root       = dirname $root;
    }

    return
};

my $extract_tarball
= sub
{

    my $extract = shift;
    my $tarball = shift
    or die "Bogus extract_tarball: false tarball basename.\n";

    my $path    = $extract->$locate_tarball( $tarball )
    or die "Missing: '$tarball' ($Bin)";

    print "# Extract repo from: '$path'";

    for my $tar ( Archive::Tar->new )
    {
        $tar->extract_archive( $path )
        and last;

        my $error   = $tar->error;

        die "Failed extract: '$path'.\n$error\n"
    };

    # at this point the .git tarball should have been
    # extracted.

    -e '.git'
    or die "Botched extract_tarball: './.git' in '$path'.\n";

    '.'
};

my $clone_url
= sub
{
    my $extract = shift;
    my $url     = $extract->value( 'git_repo_url' )
    or die "Botched clone_url: no 'git_repo_url'.\n";
    my $cmd     = join ' ' => @clone, $url;
    my $source  = basename $url, '.git';

    print STDERR "\n# Cloning from: '$url'\n";

    my $output  = qx{ $cmd };

    if( my $status  = $? )
    {
        die <<END

Non-zero:   '$status'.
Executing:  '$cmd'.
Cloning:    '$url'."
Output:

$output

END
    }
    else
    {
        my $base    = basename $url, '.git';

        print STDERR "\n# Clone complete: '$base'\n";

        $base
    }

    # caller gets back the git repository path or death.
};

my $copy_source
= sub
{
    my $extract = shift;
};

########################################################################
# methods
########################################################################

sub source_prefix
{
    my $extract = shift;

    $extract->value( 'git_prefix' )
}

sub module_sources
{
    my $extract = shift;
    my $dir     = getcwd;

    if( -e '.git' )
    {
        # nothing more do to
    }
    elsif( my $tb   = $extract->value( 'git_tarball' ) )
    {
        $extract->$extract_tarball( $tb )
    }
    elsif( my $url  = $extract->value( 'git_url' ) )
    {
        $extract->$clone_url( $url )
    }
    else
    {
        die
        "Botched module_sources: no git_tarball, git_url, or ./git.\n";
    }

    # at this point we need a ./.git directory.

    -e '.git'
    or die "Botched module_sources: no '$dir/.git'.\n";

    my $prefix  = $extract->value( 'git_prefix' )
    or die "Botched module_sources: no 'git_prefix' available.\n";
    
    # force a list context for qx to get tags as array.

    chomp
    (
        my @tagz = qx{ git tag --list '$prefix*' }
    )
    or
    die "No tags like '$prefix*' found.\n";

    print join "\n#\t" => "# Git tags: '$prefix*'", @tagz;

    @tagz
}

sub source_files
{
    # avoid returning true in scalar context.

    return
}

sub get_files
{
    my $extract = shift;
    my $tag     = $extract->value( 'module_source' );

    # deal with parsing this iff cleanup is called.

    chomp( my $curr = ( qx{ git branch } )[0] );

    print "# Saving: '$curr' for cleanup.";

    $extract->value( restore_branch => $curr );

    # no telling what the tag might look like.
    # quotes protect it from the shell.

    my @status
    = do
    {
        local $,    = ' ';

        qx{ @checkout '$tag' }
    };

    if( $? )
    {
        local $, = "\n@\t";

        chomp @status;

        print 'Checkout result:', @status;

        warn "Non-zero exit: $? from @checkout '$tag'";
    }
    else
    {
        print "# Checkout '$tag' complete.";
    }
}

sub cleanup
{
    my $extract = shift;
    my $branch  = $extract->value( 'restore_branch' );

    system @restore, "'$branch'"
    and
    do
    {
        local $, = ' ';
        warn "Non-zero exit: $? from @restore '$branch'";
    }
}

# keep require happy
1
__END__
