package SMS::API::QuickTelecom;

use 5.006;
use strict;
use warnings;
use Carp qw( croak ) ;
use LWP::UserAgent;
use POSIX qw(strftime);

=head1 NAME

SMS::API::QuickTelecom - QuickTelecom SMS services

=head1 VERSION

Version 0.02

=cut

our $VERSION = '0.02';

my @fields = qw(
    user pass host path ssl test
    CLIENTADR comment HTTP_ACCEPT_LANGUAGE sender );

sub _set {
    my $self = shift;
    my $k = shift;
    $self->{$k} = shift;
}

sub _get {
    my $self = shift;
    $self->{shift()};
}

sub _get_post_request {
    my $class = shift;
    my %arg = @_;

    my $post = {
            user => $class->{user},
            pass => $class->{pass},
            gzip => $class->{gzip} || '',
    };
    map { $post->{$_}=$arg{$_} } keys %arg;

    my $url = 'http'.($class->{ssl} ? 's':'').'://'.$class->{host}.':'.($class->{port} ? $class->{port} : ($class->{ssl} ? 443 : 80)).($class->{path}||'').'/';

    my $ua = LWP::UserAgent->new( agent => 'Mozilla/5.0 (X11; Linux i586; rv:41.0) Gecko/20120101 Firefox/41.0' );

    my $res = $ua->post( $url, $post, 'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8', );

    if ($res->is_success) {
        $res = $res->content;
    } else {
        $res = '<?xml version="1.0" encoding="UTF-8"?><output><RECEIVER AGT_ID="';
        $res .= $class->{user};
        $res .= '" DATE_REPORT="';
        $res .= (strftime "%d.%m.%Y %T", localtime);
        $res .= '"><xml_result res="';
        $res .= $res->code();
        $res .= '" description="';
        $res .= $res->message();
        $res .= '"/></output>';
    }

    $res;
}

=head1 SYNOPSIS

Usage

    use SMS::API::QuickTelecom;

    my $qtsms = SMS::API::QuickTelecom->new(
        user => 'account-login',
        pass => 'account-password',
        [host => 'hostname-to-query',]
        [path => 'path-on-hostname-to-query',]
        [ssl  => '1',]
        [gzip  => 'none',]
        [comment  => '',]
        [HTTP_ACCEPT_LANGUAGE  => '',]
        [CLIENTADR  => '',]
        [sender  => '',]
    );


=head2 Overview

A quick perl port of few classes from official QTSMS.class for SMS sending.

=head1 METHODS

=cut

=head2 new

Instantiate and initialise object with the following:

=over 4

=item * user, pass

Account username you receive during your registration. Mandatory.

=item * pass

Account password you receive during your registration. Mandatory.

=item * host

Optional.

Host name to perform POST/GET requests to. When using SSL (by default) it's default to L<go.qtelecom.ru>.

If you are willing to use insecure communication via plain HTTP then host must be set to L<service.qtelecom.ru> and ssl=0.

=item * path

Optional.

Path on server to perform requests. Default is L</public/http>.

=item * ssl

Optional.

Flag to use SSL, default is L<1> (on). Optional.

=item * gzip

Optional.

Flag to enable|disable gzip-encoding of data, possible values L<none> or L<on>, default is L<none> (do not use).

Optional

=item * comment

Optional.

Connection description.

=item * HTTP_ACCEPT_LANGUAGE

Optional.

Langage to use for the returned data content.

=item * CLIENTADR

Optional.

IP-address of the sender. Set to L<127.0.0.1> if not specified explicitly.

=item * sender

Optional.

Name of the sender, one of registered in system on https://go.qtelecom.ru.

=back

=cut

sub new {
    my $class = shift;
    my $self = {
        'version' => $VERSION,
        'path'    => '/public/http',
        'host'    => 'go.qtelecom.ru',
        'ssl'     => 1,
        'gzip'    => 'none',
    };
    my %arg   = @_;

    bless $self, $class;

    foreach my $field (@fields) {
        $self->_set($field, $arg{$field}) if exists $arg{$field};
    }

    foreach my $field (qw/ user pass /) {
        croak "new() $field is mandatory" unless defined $self->_get($field);
    }

    $self;
}

=head2 balance

Returns an XML desribing account balance in the national currency.

Takes no parameters.

=cut

sub balance {
    shift->_get_post_request( action => 'balance' );
}

=head2 send_sms

Sends a text SMS to one or more recipients.

=over 4

=item * message

Mandatory.

Message text to send. Enconding must be ASCII or UTF-8.

=item * target

Mandatory.

list of the recipients to send, comma delimited if there are more then one recipient:

"+70010001212, 80009990000"

Mutualy exclusive with L<phl_codename>

=item * phl_codename

Mutualy exclusive with L<target>, one or other is mandatory.

Codename of a recipient via contatc-list created on https://go.qtelecom.ru.

=item * sender

Optional.

Sender's nickname, one of the registered on https://go.qtelecom.ru.

=item * period

Optional.

Time-to-Live for a message, units are second. Message will be discarded if system fails to deliver message over this period of time.

Caution: this service may not work for some routes, like a CDMA phones.

=item * time_period

Optional.

Timeperiod during which message shall be delivered to the recipients (like 10:00-21:00).
Use of this option allows to skip delivery of messages during night hours.
For this service to work correctly one may specify L<time_local> timezone.

=item * time_local

Optional.

Selection of time zone for L<time_period> option:

1 - means L<time_period> is local time for the recipient,
0 - means L<time_period> was spcified according to the sender settings.

=item * autotrimtext

Optional.

Automatically trim leading and trailign spaces from a message.

=back

=cut

sub send_sms {
    my $class = shift;
    my %arg = @_;

    $arg{action}='post_sms';
    $arg{sms_type}='';

    $arg{post_id}='1458868327'.'td'.$$ unless $arg{post_id};
    $arg{CLIENTADR}='127.0.0.1' unless $arg{CLIENTADR};
    $arg{HTTP_ACCEPT_LANGUAGE}='ru-ru,ru;q=0.8,en-us;q=0.5,en;q=0.3' unless $arg{HTTP_ACCEPT_LANGUAGE};

    $class->_get_post_request( %arg );
}

=head1 AUTHOR

Pasichnichenko Konstantin, C<< <pasichnichenko.k at gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-sms-api-quicktelecom at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=SMS-API-QuickTelecom>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc SMS::API::QuickTelecom


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=SMS-API-QuickTelecom>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/SMS-API-QuickTelecom>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/SMS-API-QuickTelecom>

=item * Search CPAN

L<http://search.cpan.org/dist/SMS-API-QuickTelecom/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2016 Pasichnichenko Konstantin.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of SMS::API::QuickTelecom
