#!/usr/bin/perl
use strict;
use warnings;
use Test::More tests => 32;

use_ok( 'RDF::Query' );

SKIP: {
	eval "use RDF::Query::Model::Redland;";
	skip "Failed to load RDF::Redland", 31 if $@;
	
	my @data	= map { RDF::Redland::URI->new( 'file://' . File::Spec->rel2abs( "data/$_" ) ) } qw(about.xrdf foaf.xrdf);
	my $storage	= new RDF::Redland::Storage("hashes", "test", "new='yes',hash-type='memory'");
	my $model	= new RDF::Redland::Model($storage, "");
	my $parser	= new RDF::Redland::Parser("rdfxml");
	$parser->parse_into_model($_, $_, $model) for (@data);
	
	{
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			SELECT	?person ?nick
			WHERE	{
						?person foaf:name "Lauren Bradford" .
						OPTIONAL { ?person foaf:nick ?nick }
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		my $row		= $stream->current;
		isa_ok( $row, "ARRAY" );
		my ($p,$n)	= @{ $row || [] };
		ok( $query->bridge->isa_node( $p ), 'isa_node' );
		is( $n, undef, 'missing nick' );
	}
	
	{
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			SELECT	?person ?nick
			WHERE	{
						?person foaf:name "Gregory Todd Williams" .
						OPTIONAL { ?person foaf:nick ?nick }
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		while ($stream and not $stream->finished) {
			my $row		= $stream->current;
			isa_ok( $row, "ARRAY" );
			my ($p,$n)	= @{ $row };
			ok( $query->bridge->isa_node( $p ), 'isa_node' );
			ok( $query->bridge->isa_literal( $n ), 'isa_literal(nick)' );
			like( ($n and $n->getLabel), qr/kasei|The Samo Fool/, ($n and $n->getLabel) );
			last;
		} continue { $stream->next }
	}
	
	{
		print "# optional with trailing triples\n";
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			SELECT	?person ?nick ?page
			WHERE	{
						?person foaf:name "Gregory Todd Williams" .
						OPTIONAL { ?person foaf:nick ?nick } .
						?person foaf:homepage ?page
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		while ($stream and not $stream->finished) {
			my $row		= $stream->current;
			isa_ok( $row, "ARRAY" );
			my ($p,$n,$h)	= @{ $row };
			ok( $query->bridge->isa_node( $p ), 'isa_node' );
			ok( $query->bridge->isa_literal( $n ), 'isa_literal(nick)' );
			ok( $query->bridge->isa_resource( $h ), 'isa_resource(homepage)' );
			is( $query->bridge->uri_value( $h ), 'http://kasei.us/' );
			like( ($n and $n->getLabel), qr/kasei|The Samo Fool/, ($n and $n->getLabel) );
			last;
		} continue { $stream->next }
	}
	
	{
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			PREFIX	dc: <http://purl.org/dc/elements/1.1/>
			SELECT	?person ?h
			WHERE	{
						?person foaf:name "Gregory Todd Williams" .
						OPTIONAL {
							?person foaf:homepage ?h .
						}
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		my $row		= $stream->current;
		isa_ok( $row, "ARRAY" );
		my ($p,$h)	= @{ $row || [] };
		ok( $query->bridge->isa_node( $p ), 'isa_node(person)' );
		ok( $query->bridge->isa_node( $h ), 'isa_node(homepage)' );
	}
	
	{
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			PREFIX	dc: <http://purl.org/dc/elements/1.1/>
			SELECT	?person ?h ?title
			WHERE	{
						?person foaf:name "Gregory Todd Williams" .
						OPTIONAL {
							?person foaf:homepage ?h .
							?h dc:title ?title
						}
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		my $row		= $stream->current;
		isa_ok( $row, "ARRAY" );
		my ($p,$h,$t)	= @{ $row || [] };
		ok( $query->bridge->isa_node( $p ), 'isa_node' );
		is( $h, undef, 'no homepage' );
		is( $t, undef, 'no homepage title' );
	}
	
	{
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			SELECT	?person ?nick
			WHERE	{
						?person foaf:name "Lauren Bradford" .
						OPTIONAL { ?person foaf:nick ?nick } .
						FILTER BOUND(?nick) .
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		my $row		= $stream->current;
		ok( not($row), 'no results: successful BOUND() filter' );
	}
	
	{
		my $query	= new RDF::Query ( <<"END", undef, undef, 'sparql' );
			PREFIX	foaf: <http://xmlns.com/foaf/0.1/>
			SELECT	?school
			WHERE	{
						?person a foaf:Person ; foaf:nick "kasei" .
						OPTIONAL {
							?person foaf:schoolHomepage ?school .
						} .
					}
END
		my $stream	= $query->execute( $model );
		isa_ok( $stream, 'RDF::Query::Stream' );
		my $count	= 0;
		while ($stream and not $stream->finished) {
			my $row		= $stream->current;
			my $school	= $row->[0];
			my $str		= $query->bridge->as_string( $school );
			like( $str, qr<(smmusd|wheatonma)>, "exected school: $str" );
		} continue { $stream->next; $count++ }
		is( $count, 2, 'expected result count' );
	}
	
}
