#! /bin/false

# Copyright (C) 2019 Guido Flohr <guido.flohr@cantanea.com>,
# all rights reserved.

# This program is free software. It comes without any warranty, to
# the extent permitted by applicable law. You can redistribute it
# and/or modify it under the terms of the Do What the Fuck You Want
# to Public License, Version 2, as published by Sam Hocevar. See
# http://www.wtfpl.net/ for more details.

# Make Dist::Zilla happy.
# ABSTRACT: Read chess opening books in polyglot format

package Chess::Opening::ECO;
$Chess::Opening::ECO::VERSION = '0.1';
use common::sense;

use Locale::TextDomain 'com.cantanea.Chess-Opening';

sub positions {
	return {
		'1rbq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/2R2RK1 b - - 3 12' => {
			eco => 'A39r',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6 11. Bd2 Rb8 12. Rac1

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6 11. Bd2 Rb8 12.Rac1'),
			moves => {
			},
		},
		'1rbq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/R4RK1 w - - 2 12' => {
			eco => 'A39q',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6 11. Bd2 Rb8

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6 11. Bd2 Rb8'),
			moves => {
				'a1c1' => '1rbq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/2R2RK1 b - - 3 12',
			},
		},
		'1rbq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/1PN3P1/P1QNPPBP/R1B2RK1 b - - 0 11' => {
			eco => 'E66j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2 Rb8 11. b3

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... a6 10. Qc2 Rb8 11. b3'),
			moves => {
				'b7b5' => '1rbq1rk1/4ppbp/p2p1np1/nppP4/2P5/1PN3P1/P1QNPPBP/R1B2RK1 w - b6 0 12',
			},
		},
		'1rbq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/1RBQ1RK1 w - - 2 11' => {
			eco => 'E66g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Rb1 Rb8

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... a6 10. Rb1 Rb8'),
			moves => {
			},
		},
		'1rbq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PPQNPPBP/R1B2RK1 w - - 2 11' => {
			eco => 'E66i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2 Rb8

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... a6 10. Qc2 Rb8'),
			moves => {
				'b2b3' => '1rbq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/1PN3P1/P1QNPPBP/R1B2RK1 b - - 0 11',
			},
		},
		'1rbq1rk1/1pp1ppb1/p1np1np1/7p/2PPP2P/2N1BP2/PP1QN1P1/2KR1B1R b - - 1 10' => {
			eco => 'E84n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. h4 h5 10. O-O-O

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. h4 h5 10.O-O-O'),
			moves => {
				'b7b5' => '1rbq1rk1/2p1ppb1/p1np1np1/1p5p/2PPP2P/2N1BP2/PP1QN1P1/2KR1B1R w - b6 0 11',
			},
		},
		'1rbq1rk1/1pp1ppb1/p1np1np1/7p/2PPP2P/2N1BP2/PP1QN1P1/R3KB1R w KQ h6 0 10' => {
			eco => 'E84m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. h4 h5

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. h4 h5'),
			moves => {
				'e1c1' => '1rbq1rk1/1pp1ppb1/p1np1np1/7p/2PPP2P/2N1BP2/PP1QN1P1/2KR1B1R b - - 1 10',
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/3N4/2PP4/1P3NP1/P3PPBP/R1BQ1RK1 b - - 2 9' => {
			eco => 'E63g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. b3 Rb8 9. Nd5

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. b3 Rb8 9. Nd5'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E63e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. b3 Rb8

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. b3 Rb8'),
			moves => {
				'c1b2' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 b - - 2 9',
				'c3d5' => '1rbq1rk1/1pp1ppbp/p1np1np1/3N4/2PP4/1P3NP1/P3PPBP/R1BQ1RK1 b - - 2 9',
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 b - - 2 9' => {
			eco => 'E63f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. b3 Rb8 9. Bb2

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. b3 Rb8 9. Bb2'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N1BNPP/PP2PPB1/R2Q1RK1 b - - 2 9' => {
			eco => 'E63j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. h3 Rb8 9. Be3

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. h3 Rb8 9. Be3'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 w - - 1 9' => {
			eco => 'E63i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. h3 Rb8

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. h3 Rb8'),
			moves => {
				'c1e3' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N1BNPP/PP2PPB1/R2Q1RK1 b - - 2 9',
				'e2e4' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 b - e3 0 9',
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP1P1/2N1BP2/PP1QN2P/R3KB1R b KQ g3 0 9' => {
			eco => 'E84d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. g4

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. g4'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP2P/2N1BP2/PP1QN1P1/R3KB1R b KQ h3 0 9' => {
			eco => 'E84k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. h4

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. h4'),
			moves => {
				'b7b5' => '1rbq1rk1/2p1ppbp/p1np1np1/1p6/2PPP2P/2N1BP2/PP1QN1P1/R3KB1R w KQ b6 0 10',
				'h7h5' => '1rbq1rk1/1pp1ppb1/p1np1np1/7p/2PPP2P/2N1BP2/PP1QN1P1/R3KB1R w KQ h6 0 10',
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1Q2PP/R1N1KB1R b KQ - 3 9' => {
			eco => 'E84g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Nc1

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Nc1'),
			moves => {
				'e7e5' => '1rbq1rk1/1pp2pbp/p1np1np1/4p3/2PPP3/2N1BP2/PP1Q2PP/R1N1KB1R w KQ e6 0 10',
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/1R2KB1R b K - 3 9' => {
			eco => 'E84e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Rb1

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Rb1'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/2KR1B1R b - - 3 9' => {
			eco => 'E84c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. O-O-O

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9.O-O-O'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 2 9' => {
			eco => 'E84a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8

			variation => N__('King\'s Indian: Sämisch, Panno Main Line'),
			moves => {
				'a1b1' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/1R2KB1R b K - 3 9',
				'a2a3' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/P1N1BP2/1P1QN1PP/R3KB1R b KQ - 0 9',
				'e1c1' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/2KR1B1R b - - 3 9',
				'e2c1' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1Q2PP/R1N1KB1R b KQ - 3 9',
				'e3h6' => '1rbq1rk1/1pp1ppbp/p1np1npB/8/2PPP3/2N2P2/PP1QN1PP/R3KB1R b KQ - 3 9',
				'g2g4' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP1P1/2N1BP2/PP1QN2P/R3KB1R b KQ g3 0 9',
				'h2h4' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP2P/2N1BP2/PP1QN1P1/R3KB1R b KQ h3 0 9',
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 b - e3 0 9' => {
			eco => 'E63k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. h3 Rb8 9. e4

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. h3 Rb8 9. e4'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/P1N1BP2/1P1QN1PP/R3KB1R b KQ - 0 9' => {
			eco => 'E84b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. a3

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. a3'),
			moves => {
			},
		},
		'1rbq1rk1/1pp1ppbp/p1np1npB/8/2PPP3/2N2P2/PP1QN1PP/R3KB1R b KQ - 3 9' => {
			eco => 'E84f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Bh6

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Bh6'),
			moves => {
			},
		},
		'1rbq1rk1/1pp2pbp/p1np1np1/3Pp3/2P1P3/2N1BP2/PP1Q2PP/R1N1KB1R b KQ - 0 10' => {
			eco => 'E84i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Nc1 e5 10. d5

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Nc1 e5 10. d5'),
			moves => {
				'c6d4' => '1rbq1rk1/1pp2pbp/p2p1np1/3Pp3/2PnP3/2N1BP2/PP1Q2PP/R1N1KB1R w KQ - 1 11',
			},
		},
		'1rbq1rk1/1pp2pbp/p1np1np1/4p3/2PPP3/1NN1BP2/PP1Q2PP/R3KB1R b KQ - 1 10' => {
			eco => 'E84h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Nc1 e5 10. Nb3

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Nc1 e5 10. Nb3'),
			moves => {
			},
		},
		'1rbq1rk1/1pp2pbp/p1np1np1/4p3/2PPP3/2N1BP2/PP1Q2PP/R1N1KB1R w KQ e6 0 10' => {
			eco => 'E84g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Nc1 e5

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Nc1 e5'),
			moves => {
				'c1b3' => '1rbq1rk1/1pp2pbp/p1np1np1/4p3/2PPP3/1NN1BP2/PP1Q2PP/R3KB1R b KQ - 1 10',
				'd4d5' => '1rbq1rk1/1pp2pbp/p1np1np1/3Pp3/2P1P3/2N1BP2/PP1Q2PP/R1N1KB1R b KQ - 0 10',
			},
		},
		'1rbq1rk1/1pp2pbp/p2p1np1/3Pp3/2PnP3/1NN1BP2/PP1Q2PP/R3KB1R b KQ - 2 11' => {
			eco => 'E84j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. Nc1 e5 10. d5 Nd4 11. Nb3

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Nc1 e5 10. d5 Nd4 11. Nb3'),
			moves => {
			},
		},
		'1rbq1rk1/1pp2pbp/p2p1np1/3Pp3/2PnP3/2N1BP2/PP1Q2PP/R1N1KB1R w KQ - 1 11' => {
			eco => 'E84i',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. Nc1 e5 10. d5'),
			moves => {
				'c1b3' => '1rbq1rk1/1pp2pbp/p2p1np1/3Pp3/2PnP3/1NN1BP2/PP1Q2PP/R3KB1R b KQ - 2 11',
			},
		},
		'1rbq1rk1/2p1ppb1/p1np1np1/1p5p/2PPP2P/2N1BP2/PP1QN1P1/2KR1B1R w - b6 0 11' => {
			eco => 'E84n',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. h4 h5 10.O-O-O'),
			moves => {
				'e3h6' => '1rbq1rk1/2p1ppb1/p1np1npB/1p5p/2PPP2P/2N2P2/PP1QN1P1/2KR1B1R b - - 1 11',
			},
		},
		'1rbq1rk1/2p1ppb1/p1np1npB/1p5p/2PPP2P/2N2P2/PP1QN1P1/2KR1B1R b - - 1 11' => {
			eco => 'E84o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. h4 h5 10. O-O-O b5 11. Bh6

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. h4 h5 10.O-O-O b5 11. Bh6'),
			moves => {
			},
		},
		'1rbq1rk1/2p1ppbp/p1np1np1/1p6/2PPP2P/2N1BP2/PP1QN1P1/R3KB1R w KQ b6 0 10' => {
			eco => 'E84l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2 Rb8 9. h4 b5

			variation => N__('King\'s Indian: Sämisch, Panno Main Line, 9. h4 b5'),
			moves => {
			},
		},
		'1rbq1rk1/4ppbp/p2p1np1/n1pP4/2p5/1PN3P1/PBQNPPBP/R4RK1 w - - 0 13' => {
			eco => 'E66n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2 Rb8 11. b3 b5 12. Bb2 bxc4

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, Main Line, 12. Bb2 bxc4'),
			moves => {
			},
		},
		'1rbq1rk1/4ppbp/p2p1np1/nppP4/2P5/1PN3P1/P1QNPPBP/R1B2RK1 w - b6 0 12' => {
			eco => 'E66k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2 Rb8 11. b3 b5

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, Main Line'),
			moves => {
				'c1b2' => '1rbq1rk1/4ppbp/p2p1np1/nppP4/2P5/1PN3P1/PBQNPPBP/R4RK1 b - - 1 12',
			},
		},
		'1rbq1rk1/4ppbp/p2p1np1/nppP4/2P5/1PN3P1/PBQNPPBP/R4RK1 b - - 1 12' => {
			eco => 'E66l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2 Rb8 11. b3 b5 12. Bb2

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, Main Line, 12. Bb2'),
			moves => {
				'b5c4' => '1rbq1rk1/4ppbp/p2p1np1/n1pP4/2p5/1PN3P1/PBQNPPBP/R4RK1 w - - 0 13',
				'e7e5' => '1rbq1rk1/5pbp/p2p1np1/nppPp3/2P5/1PN3P1/PBQNPPBP/R4RK1 w - e6 0 13',
			},
		},
		'1rbq1rk1/5pbp/p2p1np1/nppPp3/2P5/1PN3P1/PBQNPPBP/R4RK1 w - e6 0 13' => {
			eco => 'E66m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2 Rb8 11. b3 b5 12. Bb2 e5

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, Main Line, 12. Bb2 e5'),
			moves => {
			},
		},
		'1rbq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 6 10' => {
			eco => 'B25r',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Rb8

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Rb8'),
			moves => {
			},
		},
		'1rbq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A36x',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3 Rb8

			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3 Rb8'),
			moves => {
			},
		},
		'1rbq1rk1/pp2ppbp/3p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w - - 2 11' => {
			eco => 'A39o',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 Rb8

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 Rb8'),
			moves => {
			},
		},
		'1rbq1rk1/ppp1ppbp/2np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 8' => {
			eco => 'E62o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O Rb8

			variation => N__('King\'s Indian: Fianchetto, 6... Nc6 7.O-O Rb8'),
			moves => {
			},
		},
		'1rbq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 6 8' => {
			eco => 'E83e',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Sämisch, Ruban Variation'),
			moves => {
				'f8e8' => '1rbqr1k1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 7 9',
			},
		},
		'1rbq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 5 8' => {
			eco => 'E83e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 Rb8

			variation => N__('King\'s Indian: Sämisch, Ruban Variation'),
			moves => {
				'd1d2' => '1rbq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 6 8',
			},
		},
		'1rbqk1nr/p3ppbp/2np2p1/1pp5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQk b6 0 8' => {
			eco => 'B26i',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8 7. Qd2 b5

			variation => N__('Sicilian: Closed, 6. Be3 Rb8 7. Qd2 b5'),
			moves => {
				'g1e2' => '1rbqk1nr/p3ppbp/2np2p1/1pp5/4P3/2NPB1P1/PPPQNPBP/R3K2R b KQk - 1 8',
			},
		},
		'1rbqk1nr/p3ppbp/2np2p1/1pp5/4P3/2NPB1P1/PPPQNPBP/R3K2R b KQk - 1 8' => {
			eco => 'B26j',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8 7. Qd2 b5 8. Nge2

			variation => N__('Sicilian: Closed, 6. Be3 Rb8 7. Qd2 b5 8.Nge2'),
			moves => {
			},
		},
		'1rbqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQk - 2 7' => {
			eco => 'B26g',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8

			variation => N__('Sicilian: Closed, 6. Be3 Rb8'),
			moves => {
				'd1d2' => '1rbqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQk - 3 7',
			},
		},
		'1rbqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQk - 3 7' => {
			eco => 'B26h',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8 7. Qd2

			variation => N__('Sicilian: Closed, 6. Be3 Rb8 7. Qd2'),
			moves => {
				'b7b5' => '1rbqk1nr/p3ppbp/2np2p1/1pp5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQk b6 0 8',
			},
		},
		'1rbqk1nr/pp2ppbp/2np2p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQk - 1 7' => {
			eco => 'B25f',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 Rb8

			variation => N__('Sicilian: Closed, 6. f4 Rb8'),
			moves => {
			},
		},
		'1rbqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQk - 2 8' => {
			eco => 'B26p',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Rb8

			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Rb8'),
			moves => {
				'g1e2' => '1rbqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQNPBP/R3K2R b KQk - 3 8',
			},
		},
		'1rbqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQNPBP/R3K2R b KQk - 3 8' => {
			eco => 'B26q',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Rb8 8. Nge2

			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Rb8 8.Nge2'),
			moves => {
			},
		},
		'1rbqk2r/2pp1ppp/p1n2n2/1pb1p3/P3P3/1B3N2/1PPP1PPP/RNBQ1RK1 w k - 1 8' => {
			eco => 'C78m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. a4 Rb8

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7. a4 Rb8'),
			moves => {
			},
		},
		'1rbqkb1r/p4ppp/2p2n2/nB2p1N1/8/5Q2/PPPP1PPP/RNB1K2R w KQk - 2 9' => {
			eco => 'C58g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Qf3 Rb8

			variation => N__('Two Knights: Morphy, Colman Variation'),
			moves => {
			},
		},
		'1rbqkb1r/ppp2ppp/2n1pn2/8/2pP4/2N2NP1/PP2PPBP/R1BQ1RK1 b k - 5 7' => {
			eco => 'E04u',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 dxc4 5. Nf3 Nc6 6. O-O Rb8 7. Nc3

			variation => N__('Catalan: Open, 5. Nf3 Nc6 6.O-O Rb8 7. Nc3'),
			moves => {
			},
		},
		'1rbqkb1r/ppp2ppp/2n1pn2/8/2pP4/5NP1/PP2PPBP/RNBQ1RK1 w k - 4 7' => {
			eco => 'E04t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 dxc4 5. Nf3 Nc6 6. O-O Rb8

			variation => N__('Catalan: Open, 5. Nf3 Nc6 6.O-O Rb8'),
			moves => {
				'b1c3' => '1rbqkb1r/ppp2ppp/2n1pn2/8/2pP4/2N2NP1/PP2PPBP/R1BQ1RK1 b k - 5 7',
			},
		},
		'1rbqr1k1/1p1n1pbp/p2p1np1/2pP4/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 b - - 2 13' => {
			eco => 'A64',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8 13. Nc4

			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8 13. Nc4'),
			moves => {
				'd7b6' => '1rbqr1k1/1p3pbp/pn1p1np1/2pP4/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 w - - 3 14',
				'd7e5' => '1rbqr1k1/1p3pbp/p2p1np1/2pPn3/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 w - - 3 14',
			},
		},
		'1rbqr1k1/1p1n1pbp/p2p1np1/2pP4/P7/2N3PP/1P1NPPB1/R1BQ1RK1 w - - 1 13' => {
			eco => 'A64',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8

			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8'),
			moves => {
				'd2c4' => '1rbqr1k1/1p1n1pbp/p2p1np1/2pP4/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 b - - 2 13',
			},
		},
		'1rbqr1k1/1p3pbp/p2p1np1/2pPn3/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 w - - 3 14' => {
			eco => 'A64',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8 13. Nc4 Ne5

			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8 13. Nc4 Ne5'),
			moves => {
			},
		},
		'1rbqr1k1/1p3pbp/pn1p1np1/2pP4/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 w - - 3 14' => {
			eco => 'A64',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8 13. Nc4 Nb6

			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8 13. Nc4 Nb6'),
			moves => {
			},
		},
		'1rbqr1k1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 7 9' => {
			eco => 'E83e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 Rb8 8. Qd2 Re8

			variation => N__('King\'s Indian: Sämisch, Ruban, 8. Qd2 Re8'),
			moves => {
			},
		},
		'2bqkbnr/r4ppp/p1npp3/1p6/4P3/1BN2N2/PP2QPPP/R1B2RK1 w k - 2 10' => {
			eco => 'B21y',
                        # TRANSLATORS: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O a6 8. Qe2 b5 9. Bb3 Ra7

			variation => N__('Sicilian: Smith-Morra Gambit, Chicago Defence'),
			moves => {
			},
		},
		'2kr1b1r/ppp2ppp/2n1N3/q7/3p4/5N2/PPP2PPP/R1BQR1K1 b - - 0 11' => {
			eco => 'C56j',
                        # TRANSLATORS:

			variation => N__('Two Knights: Classical, Moller, 10.Neg5'),
			moves => {
				'f7e6' => '2kr1b1r/ppp3pp/2n1p3/q7/3p4/5N2/PPP2PPP/R1BQR1K1 w - - 0 12',
			},
		},
		'2kr1b1r/ppp2ppp/2n1b3/q5N1/3p4/5N2/PPP2PPP/R1BQR1K1 w - - 3 11' => {
			eco => 'C56j',
                        # TRANSLATORS:

			variation => N__('Two Knights: Classical, Moller, 10.Neg5'),
			moves => {
				'g5e6' => '2kr1b1r/ppp2ppp/2n1N3/q7/3p4/5N2/PPP2PPP/R1BQR1K1 b - - 0 11',
			},
		},
		'2kr1b1r/ppp3pp/2n1R3/q7/3p4/5N2/PPP2PPP/R1BQ2K1 b - - 0 12' => {
			eco => 'C56j',
                        # TRANSLATORS:

			variation => N__('Two Knights: Classical, Moller, 10.Neg5'),
			moves => {
				'f8d6' => '2kr3r/ppp3pp/2nbR3/q7/3p4/5N2/PPP2PPP/R1BQ2K1 w - - 1 13',
			},
		},
		'2kr1b1r/ppp3pp/2n1p3/q7/3p4/5N2/PPP2PPP/R1BQR1K1 w - - 0 12' => {
			eco => 'C56j',
                        # TRANSLATORS:

			variation => N__('Two Knights: Classical, Moller, 10.Neg5'),
			moves => {
				'e1e6' => '2kr1b1r/ppp3pp/2n1R3/q7/3p4/5N2/PPP2PPP/R1BQ2K1 b - - 0 12',
			},
		},
		'2kr1b1r/ppqn1pp1/2p1p2p/7P/3Pn3/3Q1NP1/PPPB1P2/2KR3R w - - 0 15' => {
			eco => 'B19t',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6 13. Ne4 O-O-O 14. g3 Nxe4

			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 14. g3 Nxe4'),
			moves => {
			},
		},
		'2kr1b1r/ppqn1pp1/2p1pn1p/7P/3PN3/3Q1N2/PPPB1PP1/2KR3R w - - 4 14' => {
			eco => 'B19s',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6 13. Ne4 O-O-O

			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 13. Ne4 O-O-O'),
			moves => {
				'g2g3' => '2kr1b1r/ppqn1pp1/2p1pn1p/7P/3PN3/3Q1NP1/PPPB1P2/2KR3R b - - 0 14',
			},
		},
		'2kr1b1r/ppqn1pp1/2p1pn1p/7P/3PN3/3Q1NP1/PPPB1P2/2KR3R b - - 0 14' => {
			eco => 'B19s',
                        # TRANSLATORS:

			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 13. Ne4 O-O-O'),
			moves => {
				'f6e4' => '2kr1b1r/ppqn1pp1/2p1p2p/7P/3Pn3/3Q1NP1/PPPB1P2/2KR3R w - - 0 15',
			},
		},
		'2kr1bnr/pppq1ppp/2n1b3/4P3/2Pp4/5NP1/PP1NPPBP/R1BQK2R w KQ - 5 8' => {
			eco => 'D09',
                        # TRANSLATORS: 1. d4 d5 2. c4 e5 3. dxe5 d4 4. Nf3 Nc6 5. g3 Be6 6. Nbd2 Qd7 7. Bg2 O-O-O

			variation => N__('QGD: Albin, 5. g3 Be6 6.Nbd2 Qd7 7. Bg2 O-O-O'),
			moves => {
			},
		},
		'2kr1bnr/pppq1ppp/2n5/4P3/2Pp2b1/5NP1/PP2PPBP/RNBQ1RK1 w - - 5 8' => {
			eco => 'D09',
                        # TRANSLATORS: 1. d4 d5 2. c4 e5 3. dxe5 d4 4. Nf3 Nc6 5. g3 Bg4 6. Bg2 Qd7 7. O-O O-O-O

			variation => N__('QGD: Albin, 5. g3 Bg4 6. Bg2 Qd7 7.O-O O-O-O'),
			moves => {
			},
		},
		'2kr3r/pp1bqppp/2n2n2/2b5/8/2N1PN2/PP2BPPP/R1BQK2R w KQ - 3 12' => {
			eco => 'D32i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 cxd4 5. Qa4+ Bd7 6. Qxd4 exd5 7. Qxd5 Nc6 8. Nf3 Nf6 9. Qd1 Bc5 10. e3 Qe7 11. Be2 O-O-O

			variation => N__('QGD Tarrasch: von Hennig-Schara Gambit, Main Line (11. Be2 O-O-O)'),
			moves => {
			},
		},
		'2kr3r/pp3ppp/n1p1p3/3q4/P1N5/8/1PK1Q1PP/R4B1R w - - 1 16' => {
			eco => 'D17t',
                        # TRANSLATORS: 1. d4 d5 2. c4 c6 3. Nf3 Nf6 4. Nc3 dxc4 5. a4 Bf5 6. Ne5 e6 7. f3 Bb4 8. e4 Bxe4 9. fxe4 Nxe4 10. Bd2 Qxd4 11. Nxe4 Qxe4+ 12. Qe2 Bxd2+ 13. Kxd2 Qd5+ 14. Kc2 Na6 15. Nxc4 O-O-O

			variation => N__('Slav: Central, Main Line, 15.Nxc4 O-O-O'),
			moves => {
			},
		},
		'2kr3r/ppp2ppp/1bn1bPq1/6N1/2ppNPP1/8/PPP4P/R1BQR1K1 w - - 1 14' => {
			eco => 'C55x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Bc5 6. e5 d5 7. exf6 dxc4 8. Re1+ Be6 9. Ng5 Qd5 10. Nc3 Qf5 11. g4 Qg6 12. Nce4 Bb6 13. f4 O-O-O

			variation => N__('Two Knights: Max Lange, Berger Variation'),
			moves => {
			},
		},
		'2kr3r/ppp3pp/2nbR3/q7/3p4/5N2/PPP2PPP/R1BQ2K1 w - - 1 13' => {
			eco => 'C56k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Nxe4 6. Re1 d5 7. Bxd5 Qxd5 8. Nc3 Qa5 9. Nxe4 Be6 10. Neg5 O-O-O 11. Nxe6 fxe6 12. Rxe6 Bd6

			variation => N__('Two Knights: Classical, Moller, 10.Neg5 O-O-O 11.Nxe6 fxe6 12.Rxe6 Bd6'),
			moves => {
			},
		},
		'2r1kb1r/pp1bpppp/3p1n2/q5B1/3QP3/2N5/PPP2PPP/2KR1B1R w k - 1 10' => {
			eco => 'B61g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Bd7 7. Qd2 Rc8 8. O-O-O Nxd4 9. Qxd4 Qa5

			variation => N__('Sicilian: Richter-Rauzer, Larsen, Main Line'),
			moves => {
				'f2f4' => '2r1kb1r/pp1bpppp/3p1n2/q5B1/3QPP2/2N5/PPP3PP/2KR1B1R b k f3 0 10',
				'g5d2' => '2r1kb1r/pp1bpppp/3p1n2/q7/3QP3/2N5/PPPB1PPP/2KR1B1R b k - 2 10',
			},
		},
		'2r1kb1r/pp1bpppp/3p1n2/q5B1/3QPP2/2N5/PPP3PP/2KR1B1R b k f3 0 10' => {
			eco => 'B61i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Bd7 7. Qd2 Rc8 8. O-O-O Nxd4 9. Qxd4 Qa5 10. f4

			variation => N__('Sicilian: Richter-Rauzer, Larsen, Main Line, 10. f4'),
			moves => {
			},
		},
		'2r1kb1r/pp1bpppp/3p1n2/q7/3QP3/2N5/PPPB1PPP/2KR1B1R b k - 2 10' => {
			eco => 'B61h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Bd7 7. Qd2 Rc8 8. O-O-O Nxd4 9. Qxd4 Qa5 10. Bd2

			variation => N__('Sicilian: Richter-Rauzer, Larsen, Main Line, 10. Bd2'),
			moves => {
			},
		},
		'2r2rk1/1bq1bppp/p2p1n2/np1Pp3/4P3/5N1P/PPBN1PP1/R1BQR1K1 w - - 1 15' => {
			eco => 'C99h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Bb7 14. d5 Rac8

			variation => N__('Spanish: Closed, Chigorin, 13... Bb7 14. d5 Rac8'),
			moves => {
			},
		},
		'2r2rk1/1bq1bppp/p2p1n2/np2p3/3PP3/5N1P/PPB1RPP1/R1BQ1NK1 b - - 4 15' => {
			eco => 'C99f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Bb7 14. Nf1 Rac8 15. Re2

			variation => N__('Spanish: Closed, Chigorin, 13... Bb7 14. Nf1 Rac8 15. Re2'),
			moves => {
			},
		},
		'2r2rk1/1bq1bppp/p2p1n2/np2p3/3PP3/5N1P/PPB2PP1/R1BQRNK1 w - - 3 15' => {
			eco => 'C99e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Bb7 14. Nf1 Rac8

			variation => N__('Spanish: Closed, Chigorin, 13... Bb7 14. Nf1 Rac8'),
			moves => {
				'e1e2' => '2r2rk1/1bq1bppp/p2p1n2/np2p3/3PP3/5N1P/PPB1RPP1/R1BQ1NK1 b - - 4 15',
			},
		},
		'2r2rk1/2qbbppp/p2p1n2/np2p3/3PP3/4NN1P/PPB2PP1/R1BQR1K1 b - - 4 15' => {
			eco => 'C99d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Bd7 14. Nf1 Rac8 15. Ne3

			variation => N__('Spanish: Closed, Chigorin, 13... Bd7 14. Nf1 Rac8 15. Ne3'),
			moves => {
			},
		},
		'2r2rk1/2qbbppp/p2p1n2/np2p3/3PP3/5N1P/PPB2PP1/R1BQRNK1 w - - 3 15' => {
			eco => 'C99c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Chigorin, 13... Bd7 14. Nf1'),
			moves => {
				'f1e3' => '2r2rk1/2qbbppp/p2p1n2/np2p3/3PP3/4NN1P/PPB2PP1/R1BQR1K1 b - - 4 15',
			},
		},
		'2r2rk1/pbqnbppp/1pp1pn2/3p4/2PPP3/1P3NP1/PBQN1PBP/R4RK1 w - - 3 12' => {
			eco => 'E09p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. e4 Bb7 10. b3 Rc8 11. Bb2 Qc7

			variation => N__('Catalan: Closed, Main Line, 9. e4 Bb7 10. b3 Rc8 11. Bb2 Qc7'),
			moves => {
			},
		},
		'2r2rk1/pp1bppbp/2np1np1/q7/3NP3/1BN1BP2/PPPQ2PP/2KR3R w - - 9 12' => {
			eco => 'B78s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Qa5 11. Bb3 Rac8

			variation => N__('Sicilian: Dragon, Yugoslav, Old Main Line, 11. Bb3 Rac8'),
			moves => {
			},
		},
		'2rq1rk1/pb1nbppp/1p2pn2/2pp4/2PPP3/1P3NP1/PBQN1PBP/R4RK1 w - - 0 12' => {
			eco => 'E09q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. e4 Bb7 10. b3 Rc8 11. Bb2 c5

			variation => N__('Catalan: Closed, Main Line, 9. e4 Bb7 10. b3 Rc8 11. Bb2 c5'),
			moves => {
			},
		},
		'2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PP4/1P3NP1/PBQNPPBP/R4RK1 w - - 3 11' => {
			eco => 'E09g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. b3 Bb7 10. Bb2 Rc8

			variation => N__('Catalan: Closed, Main Line, 9. b3 Bb7 10. Bb2 Rc8'),
			moves => {
			},
		},
		'2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PP4/1PN2NP1/P1Q1PPBP/R1BR2K1 w - - 4 11' => {
			eco => 'E08q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3 b6 9. Rd1 Bb7 10. Nc3 Rc8

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3 b6 9. Rd1 Bb7 10. Nc3 Rc8'),
			moves => {
				'e2e4' => '2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PPP3/1PN2NP1/P1Q2PBP/R1BR2K1 b - e3 0 11',
			},
		},
		'2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PPP3/1P3NP1/P1QN1PBP/R1B2RK1 w - - 1 11' => {
			eco => 'E09n',
                        # TRANSLATORS:

			variation => N__('Catalan: Closed, Main Line, 9. e4 Bb7 10. b3'),
			moves => {
				'c1b2' => '2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PPP3/1P3NP1/PBQN1PBP/R4RK1 b - - 2 11',
			},
		},
		'2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PPP3/1P3NP1/PBQN1PBP/R4RK1 b - - 2 11' => {
			eco => 'E09o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. e4 Bb7 10. b3 Rc8 11. Bb2

			variation => N__('Catalan: Closed, Main Line, 9. e4 Bb7 10. b3 Rc8 11. Bb2'),
			moves => {
				'c6c5' => '2rq1rk1/pb1nbppp/1p2pn2/2pp4/2PPP3/1P3NP1/PBQN1PBP/R4RK1 w - - 0 12',
				'd8c7' => '2r2rk1/pbqnbppp/1pp1pn2/3p4/2PPP3/1P3NP1/PBQN1PBP/R4RK1 w - - 3 12',
			},
		},
		'2rq1rk1/pb1nbppp/1pp1pn2/3p4/2PPP3/1PN2NP1/P1Q2PBP/R1BR2K1 b - e3 0 11' => {
			eco => 'E08q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3 b6 9. Rd1 Bb7 10. Nc3 Rc8 11. e4

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3 b6 9. Rd1 Bb7 10. Nc3 Rc8 11. e4'),
			moves => {
				'd5e4' => '2rq1rk1/pb1nbppp/1pp1pn2/8/2PPp3/1PN2NP1/P1Q2PBP/R1BR2K1 w - - 0 12',
			},
		},
		'2rq1rk1/pb1nbppp/1pp1pn2/8/2PPp3/1PN2NP1/P1Q2PBP/R1BR2K1 w - - 0 12' => {
			eco => 'E08r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3 b6 9. Rd1 Bb7 10. Nc3 Rc8 11. e4 dxe4

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3 b6 9. Rd1 Bb7 10. Nc3 Rc8 11. e4 dxe4'),
			moves => {
			},
		},
		'2rq1rk1/pp1bpp2/3p1npb/4n2p/3NP2P/1BN2P2/PPPQ2P1/2KR3R w - - 0 14' => {
			eco => 'B78m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Bh6 Bxh6

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Bh6 Bxh6'),
			moves => {
			},
		},
		'2rq1rk1/pp1bppb1/3p1np1/4n1Bp/3NP2P/1BN2P2/PPPQ2P1/2KR3R b - - 1 13' => {
			eco => 'B78n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Bg5

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Bg5'),
			moves => {
				'c8c5' => '3q1rk1/pp1bppb1/3p1np1/2r1n1Bp/3NP2P/1BN2P2/PPPQ2P1/2KR3R w - - 2 14',
			},
		},
		'2rq1rk1/pp1bppb1/3p1np1/4n2p/3NP2P/1BN1BP2/PPPQ2P1/1K1R3R b - - 1 13' => {
			eco => 'B78k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Kb1

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Kb1'),
			moves => {
			},
		},
		'2rq1rk1/pp1bppb1/3p1np1/4n2p/3NP2P/1BN1BP2/PPPQ2P1/2KR3R w - h6 0 13' => {
			eco => 'B78j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5'),
			moves => {
				'c1b1' => '2rq1rk1/pp1bppb1/3p1np1/4n2p/3NP2P/1BN1BP2/PPPQ2P1/1K1R3R b - - 1 13',
				'e3g5' => '2rq1rk1/pp1bppb1/3p1np1/4n1Bp/3NP2P/1BN2P2/PPPQ2P1/2KR3R b - - 1 13',
				'e3h6' => '2rq1rk1/pp1bppb1/3p1npB/4n2p/3NP2P/1BN2P2/PPPQ2P1/2KR3R b - - 1 13',
			},
		},
		'2rq1rk1/pp1bppb1/3p1npB/4n2p/3NP2P/1BN2P2/PPPQ2P1/2KR3R b - - 1 13' => {
			eco => 'B78l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Bh6

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Bh6'),
			moves => {
				'g7h6' => '2rq1rk1/pp1bpp2/3p1npb/4n2p/3NP2P/1BN2P2/PPPQ2P1/2KR3R w - - 0 14',
			},
		},
		'2rq1rk1/pp1bppbp/2np1np1/8/2BNP2P/2N1BP2/PPPQ2P1/R3K2R w KQ - 1 11' => {
			eco => 'B77j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. h4 Rc8

			variation => N__('Sicilian: Dragon, Yugoslav, 9. Bc4 Bd7 10. h4 Rc8'),
			moves => {
			},
		},
		'2rq1rk1/pp1bppbp/2np1np1/8/2BNP3/2N1BP2/PPPQ2PP/2KR3R w - - 7 11' => {
			eco => 'B78c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8

			variation => N__('Sicilian: Dragon, Yugoslav, 10.O-O-O Rc8'),
			moves => {
				'c4b3' => '2rq1rk1/pp1bppbp/2np1np1/8/3NP3/1BN1BP2/PPPQ2PP/2KR3R b - - 8 11',
			},
		},
		'2rq1rk1/pp1bppbp/2np1np1/8/3NP3/1BN1BP2/PPPQ2PP/2KR3R b - - 8 11' => {
			eco => 'B78c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3

			variation => N__('Sicilian: Dragon, Yugoslav, 10.O-O-O Rc8 11. Bb3'),
			moves => {
				'c6e5' => '2rq1rk1/pp1bppbp/3p1np1/4n3/3NP3/1BN1BP2/PPPQ2PP/2KR3R w - - 9 12',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/4n3/3NP2P/1BN1BP2/PPPQ2P1/2KR3R b - h3 0 12' => {
			eco => 'B78g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4'),
			moves => {
				'e5c4' => '2rq1rk1/pp1bppbp/3p1np1/8/2nNP2P/1BN1BP2/PPPQ2P1/2KR3R w - - 1 13',
				'h7h5' => '2rq1rk1/pp1bppb1/3p1np1/4n2p/3NP2P/1BN1BP2/PPPQ2P1/2KR3R w - h6 0 13',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/4n3/3NP3/1BN1BP2/PPPQ2PP/1K1R3R b - - 10 12' => {
			eco => 'B78e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. Kb1

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. Kb1'),
			moves => {
				'e5c4' => '2rq1rk1/pp1bppbp/3p1np1/8/2nNP3/1BN1BP2/PPPQ2PP/1K1R3R w - - 11 13',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/4n3/3NP3/1BN1BP2/PPPQ2PP/2KR3R w - - 9 12' => {
			eco => 'B78d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5

			variation => N__('Sicilian: Dragon, Yugoslav, Modern Main Line'),
			moves => {
				'c1b1' => '2rq1rk1/pp1bppbp/3p1np1/4n3/3NP3/1BN1BP2/PPPQ2PP/1K1R3R b - - 10 12',
				'h2h4' => '2rq1rk1/pp1bppbp/3p1np1/4n3/3NP2P/1BN1BP2/PPPQ2P1/2KR3R b - h3 0 12',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/8/2BNP2P/2N1BP2/PPPQ2P1/2KR3R b - - 0 13' => {
			eco => 'B78g',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 Nc4'),
			moves => {
				'c8c4' => '3q1rk1/pp1bppbp/3p1np1/8/2rNP2P/2N1BP2/PPPQ2P1/2KR3R w - - 0 14',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/8/2BNP3/2N1BP2/PPPQ2PP/1K1R3R b - - 0 13' => {
			eco => 'B78e',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. Kb1'),
			moves => {
				'c8c4' => '3q1rk1/pp1bppbp/3p1np1/8/2rNP3/2N1BP2/PPPQ2PP/1K1R3R w - - 0 14',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/8/2nNP2P/1BN1BP2/PPPQ2P1/2KR3R w - - 1 13' => {
			eco => 'B78g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 Nc4

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 Nc4'),
			moves => {
				'b3c4' => '2rq1rk1/pp1bppbp/3p1np1/8/2BNP2P/2N1BP2/PPPQ2P1/2KR3R b - - 0 13',
			},
		},
		'2rq1rk1/pp1bppbp/3p1np1/8/2nNP3/1BN1BP2/PPPQ2PP/1K1R3R w - - 11 13' => {
			eco => 'B78e',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. Kb1'),
			moves => {
				'b3c4' => '2rq1rk1/pp1bppbp/3p1np1/8/2BNP3/2N1BP2/PPPQ2PP/1K1R3R b - - 0 13',
			},
		},
		'2rq1rk1/pp2ppbp/2npbnp1/8/4PP2/1NN1B3/PPP1B1PP/R2Q1RK1 w - - 1 11' => {
			eco => 'B74',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Nb3 Be6 10. f4 Rc8

			variation => N__('Sicilian: Dragon, Classical, 9. Nb3 Be6 10. f4 Rc8'),
			moves => {
			},
		},
		'2rqk2r/pp1bppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/2KR1B1R w k - 5 10' => {
			eco => 'B75',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 Nc6 8. Qd2 Bd7 9. O-O-O Rc8

			variation => N__('Sicilian: Dragon, Yugoslav, 7... Nc6 8. Qd2 Bd7 9.O-O-O Rc8'),
			moves => {
			},
		},
		'2rqkb1r/1p1b1ppp/p1nppn2/6B1/3NP3/2N2P2/PPPQ2PP/2KR1B1R w k - 1 10' => {
			eco => 'B67d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O Bd7 9. f3 Rc8

			variation => N__('Sicilian: Richter-Rauzer, 7... a6, 9. f3 Rc8'),
			moves => {
			},
		},
		'2rqkb1r/pp1bpppp/2np1n2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R b k - 8 8' => {
			eco => 'B61f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Bd7 7. Qd2 Rc8 8. O-O-O

			variation => N__('Sicilian: Richter-Rauzer, Larsen, 7. Qd2 Rc8 8.O-O-O'),
			moves => {
				'c6d4' => '2rqkb1r/pp1bpppp/3p1n2/6B1/3nP3/2N5/PPPQ1PPP/2KR1B1R w k - 0 9',
			},
		},
		'2rqkb1r/pp1bpppp/2np1n2/6B1/3NP3/2N5/PPPQ1PPP/R3KB1R w KQk - 7 8' => {
			eco => 'B61d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Bd7 7. Qd2 Rc8

			variation => N__('Sicilian: Richter-Rauzer, Larsen, 7. Qd2 Rc8'),
			moves => {
				'e1c1' => '2rqkb1r/pp1bpppp/2np1n2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R b k - 8 8',
				'f2f4' => '2rqkb1r/pp1bpppp/2np1n2/6B1/3NPP2/2N5/PPPQ2PP/R3KB1R b KQk f3 0 8',
			},
		},
		'2rqkb1r/pp1bpppp/2np1n2/6B1/3NPP2/2N5/PPPQ2PP/R3KB1R b KQk f3 0 8' => {
			eco => 'B61e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Bd7 7. Qd2 Rc8 8. f4

			variation => N__('Sicilian: Richter-Rauzer, Larsen, 7. Qd2 Rc8 8. f4'),
			moves => {
			},
		},
		'2rqkb1r/pp1bpppp/3p1n2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R b k - 0 9' => {
			eco => 'B61f',
                        # TRANSLATORS:

			variation => N__('Sicilian: Richter-Rauzer, Larsen, 7. Qd2 Rc8 8.O-O-O'),
			moves => {
				'd8a5' => '2r1kb1r/pp1bpppp/3p1n2/q5B1/3QP3/2N5/PPP2PPP/2KR1B1R w k - 1 10',
			},
		},
		'2rqkb1r/pp1bpppp/3p1n2/6B1/3nP3/2N5/PPPQ1PPP/2KR1B1R w k - 0 9' => {
			eco => 'B61f',
                        # TRANSLATORS:

			variation => N__('Sicilian: Richter-Rauzer, Larsen, 7. Qd2 Rc8 8.O-O-O'),
			moves => {
				'd2d4' => '2rqkb1r/pp1bpppp/3p1n2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R b k - 0 9',
			},
		},
		'3q1rk1/3nppbp/r2p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQ3R w - - 1 13' => {
			eco => 'A59o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Ra6

			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Ra6'),
			moves => {
				'h1e1' => '3q1rk1/3nppbp/r2p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQR3 b - - 2 13',
			},
		},
		'3q1rk1/3nppbp/r2p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQR3 b - - 2 13' => {
			eco => 'A59o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Ra6 13. Re1

			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Ra6 13. Re1'),
			moves => {
			},
		},
		'3q1rk1/p2bppbp/3p1np1/1p6/2rNP1P1/2N1BP2/PPPQ3P/1K1R3R w - b6 0 15' => {
			eco => 'B78f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. Kb1 Nc4 13. Bxc4 Rxc4 14. g4 b5

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. Kb1: 14. g4 b5'),
			moves => {
			},
		},
		'3q1rk1/pp1bppb1/3p1np1/2r1n1Bp/3NP1PP/1BN2P2/PPPQ4/2KR3R b - g3 0 14' => {
			eco => 'B78p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Bg5 Rc5 14. g4

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Bg5 Rc5 14. g4'),
			moves => {
			},
		},
		'3q1rk1/pp1bppb1/3p1np1/2r1n1Bp/3NP2P/1BN2P2/PPPQ2P1/1K1R3R b - - 3 14' => {
			eco => 'B78q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Bg5 Rc5 14. Kb1

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Bg5 Rc5 14. Kb1'),
			moves => {
			},
		},
		'3q1rk1/pp1bppb1/3p1np1/2r1n1Bp/3NP2P/1BN2P2/PPPQ2P1/2KR3R w - - 2 14' => {
			eco => 'B78o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 h5 13. Bg5 Rc5

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 h5 13. Bg5 Rc5'),
			moves => {
				'c1b1' => '3q1rk1/pp1bppb1/3p1np1/2r1n1Bp/3NP2P/1BN2P2/PPPQ2P1/1K1R3R b - - 3 14',
				'g2g4' => '3q1rk1/pp1bppb1/3p1np1/2r1n1Bp/3NP1PP/1BN2P2/PPPQ4/2KR3R b - g3 0 14',
			},
		},
		'3q1rk1/pp1bppbp/3p1np1/7P/2rNP3/2N1BP2/PPPQ2P1/2KR3R b - - 0 14' => {
			eco => 'B78i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 Nc4 13. Bxc4 Rxc4 14. h5

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 Nc4: 14. h5'),
			moves => {
			},
		},
		'3q1rk1/pp1bppbp/3p1np1/8/2rNP1P1/2N1BP2/PPPQ3P/1K1R3R b - g3 0 14' => {
			eco => 'B78e',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. Kb1'),
			moves => {
				'b7b5' => '3q1rk1/p2bppbp/3p1np1/1p6/2rNP1P1/2N1BP2/PPPQ3P/1K1R3R w - b6 0 15',
			},
		},
		'3q1rk1/pp1bppbp/3p1np1/8/2rNP1PP/2N1BP2/PPPQ4/2KR3R b - g3 0 14' => {
			eco => 'B78h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. h4 Nc4 13. Bxc4 Rxc4 14. g4

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 Nc4: 14. g4'),
			moves => {
			},
		},
		'3q1rk1/pp1bppbp/3p1np1/8/2rNP2P/2N1BP2/PPPQ2P1/2KR3R w - - 0 14' => {
			eco => 'B78g',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. h4 Nc4'),
			moves => {
				'g2g4' => '3q1rk1/pp1bppbp/3p1np1/8/2rNP1PP/2N1BP2/PPPQ4/2KR3R b - g3 0 14',
				'h4h5' => '3q1rk1/pp1bppbp/3p1np1/7P/2rNP3/2N1BP2/PPPQ2P1/2KR3R b - - 0 14',
			},
		},
		'3q1rk1/pp1bppbp/3p1np1/8/2rNP3/2N1BP2/PPPQ2PP/1K1R3R w - - 0 14' => {
			eco => 'B78e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Bd7 10. O-O-O Rc8 11. Bb3 Ne5 12. Kb1 Nc4 13. Bxc4 Rxc4

			variation => N__('Sicilian: Dragon, Yugoslav, Main Line, 12. Kb1'),
			moves => {
				'g2g4' => '3q1rk1/pp1bppbp/3p1np1/8/2rNP1P1/2N1BP2/PPPQ3P/1K1R3R b - g3 0 14',
			},
		},
		'4rrk1/5ppp/p1pb4/1p1n4/3P2b1/1BPQB1Pq/PP1N1P1P/R3R1K1 b - - 6 17' => {
			eco => 'C89q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2

			variation => N__('Spanish: Marshall, Main Line, 16. Qd3 Rae8 17. Nd2'),
			moves => {
				'e8e6' => '5rk1/5ppp/p1pbr3/1p1n4/3P2b1/1BPQB1Pq/PP1N1P1P/R3R1K1 w - - 7 18',
				'f7f5' => '4rrk1/6pp/p1pb4/1p1n1p2/3P2b1/1BPQB1Pq/PP1N1P1P/R3R1K1 w - f6 0 18',
			},
		},
		'4rrk1/5ppp/p1pb4/1p1n4/3P2b1/1BPQB1Pq/PP3P1P/RN2R1K1 w - - 5 17' => {
			eco => 'C89q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8

			variation => N__('Spanish: Marshall, Main Line, 15. Be3 Bg4 16. Qd3 Rae8'),
			moves => {
				'b1d2' => '4rrk1/5ppp/p1pb4/1p1n4/3P2b1/1BPQB1Pq/PP1N1P1P/R3R1K1 b - - 6 17',
			},
		},
		'4rrk1/6pp/p1pb4/1p1n1p2/3P2b1/1BPQB1Pq/PP1N1P1P/R3R1K1 w - f6 0 18' => {
			eco => 'C89r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 f5

			variation => N__('Spanish: Marshall, Main Line, Pawn Push Variation'),
			moves => {
			},
		},
		'5rk1/5ppp/p1pbr3/1p1B4/3P2b1/2PQB1Pq/PP1N1P1P/R3R1K1 b - - 0 18' => {
			eco => 'C89u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6 18. Bxd5

			variation => N__('Spanish: Marshall, Main Line, Classical, 18.Bxd5'),
			moves => {
			},
		},
		'5rk1/5ppp/p1pbr3/1p1n3q/P2P2b1/1BPQB1P1/1P1N1P1P/R3R1K1 w - - 1 19' => {
			eco => 'C89x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6 18. a4 Qh5

			variation => N__('Spanish: Marshall, Main Line, Classical, Spassky Variation'),
			moves => {
			},
		},
		'5rk1/5ppp/p1pbr3/1p1n4/2PP2b1/1B1QB1Pq/PP1N1P1P/R3R1K1 b - - 0 18' => {
			eco => 'C89t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6 18. c4

			variation => N__('Spanish: Marshall, Main Line, Classical, 18. c4'),
			moves => {
			},
		},
		'5rk1/5ppp/p1pbr3/1p1n4/3P2b1/1BP1B1Pq/PP1N1P1P/R3RQK1 b - - 8 18' => {
			eco => 'C89v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6 18. Qf1

			variation => N__('Spanish: Marshall, Main Line, Classical, 18. Qf1'),
			moves => {
			},
		},
		'5rk1/5ppp/p1pbr3/1p1n4/3P2b1/1BPQB1Pq/PP1N1P1P/R3R1K1 w - - 7 18' => {
			eco => 'C89s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6

			variation => N__('Spanish: Marshall, Main Line, Classical Variation'),
			moves => {
				'a2a4' => '5rk1/5ppp/p1pbr3/1p1n4/P2P2b1/1BPQB1Pq/1P1N1P1P/R3R1K1 b - a3 0 18',
				'b3d5' => '5rk1/5ppp/p1pbr3/1p1B4/3P2b1/2PQB1Pq/PP1N1P1P/R3R1K1 b - - 0 18',
				'c3c4' => '5rk1/5ppp/p1pbr3/1p1n4/2PP2b1/1B1QB1Pq/PP1N1P1P/R3R1K1 b - - 0 18',
				'd3f1' => '5rk1/5ppp/p1pbr3/1p1n4/3P2b1/1BP1B1Pq/PP1N1P1P/R3RQK1 b - - 8 18',
			},
		},
		'5rk1/5ppp/p1pbr3/1p1n4/P2P2b1/1BPQB1Pq/1P1N1P1P/R3R1K1 b - a3 0 18' => {
			eco => 'C89w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6 18. a4

			variation => N__('Spanish: Marshall, Main Line, Classical, 18. a4'),
			moves => {
				'f7f5' => '5rk1/6pp/p1pbr3/1p1n1p2/P2P2b1/1BPQB1Pq/1P1N1P1P/R3R1K1 w - f6 0 19',
				'h3h5' => '5rk1/5ppp/p1pbr3/1p1n3q/P2P2b1/1BPQB1P1/1P1N1P1P/R3R1K1 w - - 1 19',
			},
		},
		'5rk1/6pp/p1pbr3/1p1n1p2/P2P2b1/1BPQB1Pq/1P1N1P1P/R3R1K1 w - f6 0 19' => {
			eco => 'C89y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3 Bg4 16. Qd3 Rae8 17. Nd2 Re6 18. a4 f5

			variation => N__('Spanish: Marshall, Main Line, Classical, 18. a4 f5'),
			moves => {
			},
		},
		'N1bk1b1r/p2pq2p/1pnn2p1/3Qpp2/8/1B1P4/PPP2PPP/R1B1K1NR b KQ - 0 11' => {
			eco => 'C27j',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nf6 3. Bc4 Nxe4 4. Qh5 Nd6 5. Bb3 Nc6 6. Nb5 g6 7. Qf3 f5 8. Qd5 Qe7 9. Nxc7+ Kd8 10. Nxa8 b6 11. d3

			variation => N__('Vienna: Frankenstein-Dracula, 11. d3'),
			moves => {
			},
		},
		'N1bk1b1r/p2pq2p/1pnn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR w KQ - 0 11' => {
			eco => 'C27i',
                        # TRANSLATORS:

			variation => N__('Vienna: Frankenstein-Dracula Variation'),
			moves => {
				'd2d3' => 'N1bk1b1r/p2pq2p/1pnn2p1/3Qpp2/8/1B1P4/PPP2PPP/R1B1K1NR b KQ - 0 11',
			},
		},
		'N1bk1b1r/pp1pq2p/2nn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR b KQ - 0 10' => {
			eco => 'C27i',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nf6 3. Bc4 Nxe4 4. Qh5 Nd6 5. Bb3 Nc6 6. Nb5 g6 7. Qf3 f5 8. Qd5 Qe7 9. Nxc7+ Kd8 10. Nxa8

			variation => N__('Vienna: Frankenstein-Dracula Variation'),
			moves => {
				'b7b6' => 'N1bk1b1r/p2pq2p/1pnn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR w KQ - 0 11',
			},
		},
		'N1bk2nr/pp1p1ppp/2n5/8/1b6/5B1q/PPPN1P1P/R1BQK2R b KQ - 0 10' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'g8f6' => 'N1bk3r/pp1p1ppp/2n2n2/8/1b6/5B1q/PPPN1P1P/R1BQK2R w KQ - 1 11',
			},
		},
		'N1bk3r/pp1p1ppp/2n2n2/8/1b6/5B1q/PPPN1P1P/R1BQK2R w KQ - 1 11' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'a2a3' => 'N1bk3r/pp1p1ppp/2n2n2/8/1b6/P4B1q/1PPN1P1P/R1BQK2R b KQ - 0 11',
			},
		},
		'N1bk3r/pp1p1ppp/2n2n2/8/1b6/P4B1q/1PPN1P1P/R1BQK2R b KQ - 0 11' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Bb4+ 6. Nd2 Qxe4+ 7. Be2 Qxg2 8. Bf3 Qh3 9. Nxc7+ Kd8 10. Nxa8 Nf6 11. a3

			variation => N__('Scotch: Steinitz, Berger Variation'),
			moves => {
			},
		},
		'r1b1k1nr/1p1p1ppp/p1n2Q2/4p3/4P3/8/PPP2PPP/RNB1KB1R b KQkq - 0 8' => {
			eco => 'B32l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6 7. Qxd6 Qf6 8. Qxf6

			variation => N__('Sicilian: Lowenthal, 5... a6, Queenswap'),
			moves => {
			},
		},
		'r1b1k1nr/1p1p1ppp/p1n2q2/4p3/4P3/8/PPP2PPP/RNBQKB1R b KQkq - 2 8' => {
			eco => 'B32m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6 7. Qxd6 Qf6 8. Qd1

			variation => N__('Sicilian: Lowenthal, 5... a6, 8. Qd1'),
			moves => {
			},
		},
		'r1b1k1nr/1p1p1ppp/p1nQ1q2/4p3/4P3/8/PPP2PPP/RNB1KB1R w KQkq - 1 8' => {
			eco => 'B32k',
                        # TRANSLATORS:

			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'd6d1' => 'r1b1k1nr/1p1p1ppp/p1n2q2/4p3/4P3/8/PPP2PPP/RNBQKB1R b KQkq - 2 8',
				'd6f6' => 'r1b1k1nr/1p1p1ppp/p1n2Q2/4p3/4P3/8/PPP2PPP/RNB1KB1R b KQkq - 0 8',
			},
		},
		'r1b1k1nr/1ppp1ppp/pbn2q2/8/P3P3/1NN5/1PP2PPP/R1BQKB1R w KQkq - 2 8' => {
			eco => 'C45q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6 7. Nc3 Qf6

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6 7. Nc3 Qf6'),
			moves => {
			},
		},
		'r1b1k1nr/pp3pbp/2npp1p1/q1p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8' => {
			eco => 'B26m',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Qa5

			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Qa5'),
			moves => {
			},
		},
		'r1b1k1nr/pp3ppp/n1p1p3/8/1BP1q3/8/PP2BPPP/R2QK1NR w KQkq - 2 9' => {
			eco => 'D31z',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c6 4. e4 dxe4 5. Nxe4 Bb4+ 6. Bd2 Qxd4 7. Bxb4 Qxe4+ 8. Be2 Na6

			variation => N__('Semi-Slav: Marshall Gambit, 8. Be2 Na6'),
			moves => {
			},
		},
		'r1b1k1nr/ppNp1ppp/2n5/8/1b6/5B1q/PPPN1P1P/R1BQK2R b KQkq - 0 9' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'e8d8' => 'r1bk2nr/ppNp1ppp/2n5/8/1b6/5B1q/PPPN1P1P/R1BQK2R w KQ - 1 10',
			},
		},
		'r1b1k1nr/ppp2ppp/2n5/3qp3/1b1P4/2N1P3/PP1B1PPP/R2QKBNR b KQkq - 3 6' => {
			eco => 'D07c',
                        # TRANSLATORS:

			variation => N__('QGD: Chigorin, 3.cxd5'),
			moves => {
				'b4c3' => 'r1b1k1nr/ppp2ppp/2n5/3qp3/3P4/2b1P3/PP1B1PPP/R2QKBNR w KQkq - 0 7',
			},
		},
		'r1b1k1nr/ppp2ppp/2n5/3qp3/1b1P4/2N1P3/PP3PPP/R1BQKBNR w KQkq - 2 6' => {
			eco => 'D07c',
                        # TRANSLATORS:

			variation => N__('QGD: Chigorin, 3.cxd5'),
			moves => {
				'c1d2' => 'r1b1k1nr/ppp2ppp/2n5/3qp3/1b1P4/2N1P3/PP1B1PPP/R2QKBNR b KQkq - 3 6',
			},
		},
		'r1b1k1nr/ppp2ppp/2n5/3qp3/3P4/2B1P3/PP3PPP/R2QKBNR b KQkq - 0 7' => {
			eco => 'D07e',
                        # TRANSLATORS: 1. d4 d5 2. c4 Nc6 3. cxd5 Qxd5 4. e3 e5 5. Nc3 Bb4 6. Bd2 Bxc3 7. Bxc3

			variation => N__('QGD: Chigorin, 3.cxd5 Main Line, 7.Bxc3'),
			moves => {
			},
		},
		'r1b1k1nr/ppp2ppp/2n5/3qp3/3P4/2P1P3/P2B1PPP/R2QKBNR b KQkq - 0 7' => {
			eco => 'D07d',
                        # TRANSLATORS: 1. d4 d5 2. c4 Nc6 3. cxd5 Qxd5 4. e3 e5 5. Nc3 Bb4 6. Bd2 Bxc3 7. bxc3

			variation => N__('QGD: Chigorin, 3.cxd5, Main Line, 7.bxc3'),
			moves => {
			},
		},
		'r1b1k1nr/ppp2ppp/2n5/3qp3/3P4/2b1P3/PP1B1PPP/R2QKBNR w KQkq - 0 7' => {
			eco => 'D07c',
                        # TRANSLATORS: 1. d4 d5 2. c4 Nc6 3. cxd5 Qxd5 4. e3 e5 5. Nc3 Bb4 6. Bd2 Bxc3

			variation => N__('QGD: Chigorin, 3.cxd5, Main Line'),
			moves => {
				'b2c3' => 'r1b1k1nr/ppp2ppp/2n5/3qp3/3P4/2P1P3/P2B1PPP/R2QKBNR b KQkq - 0 7',
				'd2c3' => 'r1b1k1nr/ppp2ppp/2n5/3qp3/3P4/2B1P3/PP3PPP/R2QKBNR b KQkq - 0 7',
			},
		},
		'r1b1k1nr/ppp2ppp/2p2q2/2b5/4P3/2N5/PPPQ1PPP/R1B1KB1R b KQkq - 1 7' => {
			eco => 'C45t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nxc6 Qf6 6. Qd2 dxc6 7. Nc3

			variation => N__('Scotch: 4.Nxd4 Bc5 5.Nxc6 Qf6 6. Qd2 dxc6 7. Nc3'),
			moves => {
			},
		},
		'r1b1k1nr/ppp2ppp/2p2q2/2b5/4P3/8/PPPQ1PPP/RNB1KB1R w KQkq - 0 7' => {
			eco => 'C45s',
                        # TRANSLATORS:

			variation => N__('Scotch: 4.Nxd4 Bc5 5.Nxc6 Qf6 6. Qd2'),
			moves => {
				'b1c3' => 'r1b1k1nr/ppp2ppp/2p2q2/2b5/4P3/2N5/PPPQ1PPP/R1B1KB1R b KQkq - 1 7',
			},
		},
		'r1b1k1nr/pppp1Bpp/1b6/nP2N1q1/4P3/8/P1PP1PPP/RNBQK2R b KQkq - 0 7' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'e8e7' => 'r1b3nr/ppppkBpp/1b6/nP2N1q1/4P3/8/P1PP1PPP/RNBQK2R w KQ - 1 8',
			},
		},
		'r1b1k1nr/pppp1Qpp/1b6/nP2q3/2B1P3/8/P1PP1PPP/RNB1K2R b KQkq - 0 8' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'e8d8' => 'r1bk2nr/pppp1Qpp/1b6/nP2q3/2B1P3/8/P1PP1PPP/RNB1K2R w KQ - 1 9',
			},
		},
		'r1b1k1nr/pppp1ppp/1b6/nP2N1q1/2B1P3/5Q2/P1PP1PPP/RNB1K2R b KQkq - 2 7' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'g5e5' => 'r1b1k1nr/pppp1ppp/1b6/nP2q3/2B1P3/5Q2/P1PP1PPP/RNB1K2R w KQkq - 0 8',
			},
		},
		'r1b1k1nr/pppp1ppp/1b6/nP2N1q1/2B1P3/8/P1PP1PPP/RNBQK2R w KQkq - 1 7' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. b5 Na5 6. Nxe5 Qg5

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'c4f7' => 'r1b1k1nr/pppp1Bpp/1b6/nP2N1q1/4P3/8/P1PP1PPP/RNBQK2R b KQkq - 0 7',
				'd1f3' => 'r1b1k1nr/pppp1ppp/1b6/nP2N1q1/2B1P3/5Q2/P1PP1PPP/RNB1K2R b KQkq - 2 7',
			},
		},
		'r1b1k1nr/pppp1ppp/1b6/nP2q3/2B1P3/5Q2/P1PP1PPP/RNB1K2R w KQkq - 0 8' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'f3f7' => 'r1b1k1nr/pppp1Qpp/1b6/nP2q3/2B1P3/8/P1PP1PPP/RNB1K2R b KQkq - 0 8',
			},
		},
		'r1b1k1nr/pppp1ppp/2N2q2/2b5/4P3/8/PPP2PPP/RNBQKB1R w KQkq - 1 6' => {
			eco => 'C45r',
                        # TRANSLATORS:

			variation => N__('Scotch: 4.Nxd4 Bc5 5.Nxc6'),
			moves => {
				'd1d2' => 'r1b1k1nr/pppp1ppp/2N2q2/2b5/4P3/8/PPPQ1PPP/RNB1KB1R b KQkq - 2 6',
			},
		},
		'r1b1k1nr/pppp1ppp/2N2q2/2b5/4P3/8/PPPQ1PPP/RNB1KB1R b KQkq - 2 6' => {
			eco => 'C45s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nxc6 Qf6 6. Qd2

			variation => N__('Scotch: 4.Nxd4 Bc5 5.Nxc6 Qf6 6. Qd2'),
			moves => {
				'd7c6' => 'r1b1k1nr/ppp2ppp/2p2q2/2b5/4P3/8/PPPQ1PPP/RNB1KB1R w KQkq - 0 7',
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5' => {
			eco => 'C64e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. c3 Qf6

			variation => N__('Spanish: Classical, 4. c3 Qf6'),
			moves => {
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 6 5' => {
			eco => 'C64m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Qf6

			variation => N__('Spanish: Classical, 4.O-O Qf6'),
			moves => {
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/1Nb5/4P3/4B3/PPP2PPP/RN1QKB1R b KQkq - 4 6' => {
			eco => 'C45v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. Nb5

			variation => N__('Scotch: Blumenfeld Attack'),
			moves => {
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/2b5/3NP3/2P1B3/PP3PPP/RN1QKB1R b KQkq - 0 6' => {
			eco => 'C45v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Be3 Qf6 6. c3'),
			moves => {
				'g8e7' => 'r1b1k2r/ppppnppp/2n2q2/2b5/3NP3/2P1B3/PP3PPP/RN1QKB1R w KQkq - 1 7',
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/2b5/3NP3/4B3/PPP2PPP/RN1QKB1R w KQkq - 3 6' => {
			eco => 'C45u',
                        # TRANSLATORS:

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Be3'),
			moves => {
				'c2c3' => 'r1b1k1nr/pppp1ppp/2n2q2/2b5/3NP3/2P1B3/PP3PPP/RN1QKB1R b KQkq - 0 6',
				'd4b5' => 'r1b1k1nr/pppp1ppp/2n2q2/1Nb5/4P3/4B3/PPP2PPP/RN1QKB1R b KQkq - 4 6',
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/b3P3/2B5/1Qp2N2/P4PPP/RNB2RK1 b kq - 0 9' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'f6g6' => 'r1b1k1nr/pppp1ppp/2n3q1/b3P3/2B5/1Qp2N2/P4PPP/RNB2RK1 w kq - 1 10',
			},
		},
		'r1b1k1nr/pppp1ppp/2n2q2/b7/2B1P3/1Qp2N2/P4PPP/RNB2RK1 w kq - 2 9' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'e4e5' => 'r1b1k1nr/pppp1ppp/2n2q2/b3P3/2B5/1Qp2N2/P4PPP/RNB2RK1 b kq - 0 9',
			},
		},
		'r1b1k1nr/pppp1ppp/2n3q1/b3P3/2B5/1QN2N2/P4PPP/R1B2RK1 b kq - 0 10' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'g8e7' => 'r1b1k2r/ppppnppp/2n3q1/b3P3/2B5/1QN2N2/P4PPP/R1B2RK1 w kq - 1 11',
			},
		},
		'r1b1k1nr/pppp1ppp/2n3q1/b3P3/2B5/1Qp2N2/P4PPP/RNB2RK1 w kq - 1 10' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'b1c3' => 'r1b1k1nr/pppp1ppp/2n3q1/b3P3/2B5/1QN2N2/P4PPP/R1B2RK1 b kq - 0 10',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2P2q/8/PPP2PPP/RNBQKB1R w KQkq - 3 6' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Bb4+

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'b1d2' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2P2q/8/PPPN1PPP/R1BQKB1R b KQkq - 4 6',
				'c1d2' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2P2q/8/PPPB1PPP/RN1QKB1R b KQkq - 4 6',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2P2q/8/PPPB1PPP/RN1QKB1R b KQkq - 4 6' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Bb4+ 6. Bd2

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'h4e4' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPB1PPP/RN1QKB1R w KQkq - 0 7',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2P2q/8/PPPN1PPP/R1BQKB1R b KQkq - 4 6' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'h4e4' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPN1PPP/R1BQKB1R w KQkq - 0 7',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPB1PPP/RN1QKB1R w KQkq - 0 7' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'f1e2' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPBBPPP/RN1QK2R b KQkq - 1 7',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPBBPPP/RN1QK2R b KQkq - 1 7' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'e8d8' => 'r1bk2nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPBBPPP/RN1QK2R w KQ - 2 8',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPN1PPP/R1BQKB1R w KQkq - 0 7' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'f1e2' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPNBPPP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPNBPPP/R1BQK2R b KQkq - 1 7' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'e4g2' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b6/8/PPPNBPqP/R1BQK2R w KQkq - 0 8',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b6/5B1q/PPPN1P1P/R1BQK2R w KQkq - 2 9' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'b5c7' => 'r1b1k1nr/ppNp1ppp/2n5/8/1b6/5B1q/PPPN1P1P/R1BQK2R b KQkq - 0 9',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b6/5B2/PPPN1PqP/R1BQK2R b KQkq - 1 8' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'g2h3' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b6/5B1q/PPPN1P1P/R1BQK2R w KQkq - 2 9',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1N6/1b6/8/PPPNBPqP/R1BQK2R w KQkq - 0 8' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'e2f3' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b6/5B2/PPPN1PqP/R1BQK2R b KQkq - 1 8',
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1Nb5/4P2q/5Q2/PPP2PPP/RNB1KB1R b KQkq - 4 6' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Bc5 6. Qf3

			variation => N__('Scotch: Steinitz, 5. Nb5 Bc5 6. Qf3'),
			moves => {
			},
		},
		'r1b1k1nr/pppp1ppp/2n5/1Nb5/4P2q/8/PPP2PPP/RNBQKB1R w KQkq - 3 6' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Bc5

			variation => N__('Scotch: Steinitz, 5. Nb5 Bc5'),
			moves => {
				'd1f3' => 'r1b1k1nr/pppp1ppp/2n5/1Nb5/4P2q/5Q2/PPP2PPP/RNB1KB1R b KQkq - 4 6',
			},
		},
		'r1b1k1nr/ppppqppp/1bn5/3Pp3/2B1P3/2P2N2/PP3PPP/RNBQK2R b KQkq - 0 6' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'c6b8' => 'rnb1k1nr/ppppqppp/1b6/3Pp3/2B1P3/2P2N2/PP3PPP/RNBQK2R w KQkq - 1 7',
			},
		},
		'r1b1k1nr/ppppqppp/1bn5/4p1B1/2BPP3/2P2N2/PP3PPP/RN1QK2R b KQkq - 2 6' => {
			eco => 'C53',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Qe7 5. d4 Bb6 6. Bg5

			variation => N__('Giuoco Piano: Mestel Variation'),
			moves => {
			},
		},
		'r1b1k1nr/ppppqppp/1bn5/4p3/2BPP3/2P2N2/PP3PPP/RNBQ1RK1 b kq - 2 6' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'g8f6' => 'r1b1k2r/ppppqppp/1bn2n2/4p3/2BPP3/2P2N2/PP3PPP/RNBQ1RK1 w kq - 3 7',
			},
		},
		'r1b1k1nr/ppppqppp/1bn5/4p3/2BPP3/2P2N2/PP3PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C53',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Qe7 5. d4 Bb6

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'c1g5' => 'r1b1k1nr/ppppqppp/1bn5/4p1B1/2BPP3/2P2N2/PP3PPP/RN1QK2R b KQkq - 2 6',
				'd4d5' => 'r1b1k1nr/ppppqppp/1bn5/3Pp3/2B1P3/2P2N2/PP3PPP/RNBQK2R b KQkq - 0 6',
				'e1g1' => 'r1b1k1nr/ppppqppp/1bn5/4p3/2BPP3/2P2N2/PP3PPP/RNBQ1RK1 b kq - 2 6',
			},
		},
		'r1b1k1nr/ppppqppp/2n5/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5' => {
			eco => 'C64b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. c3 Qe7

			variation => N__('Spanish: Classical, Boden Variation'),
			moves => {
			},
		},
		'r1b1k1nr/ppppqppp/2n5/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5' => {
			eco => 'C53',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Qe7

			variation => N__('Giuoco Piano: Close Variation'),
			moves => {
				'd2d4' => 'r1b1k1nr/ppppqppp/2n5/2b1p3/2BPP3/2P2N2/PP3PPP/RNBQK2R b KQkq d3 0 5',
			},
		},
		'r1b1k1nr/ppppqppp/2n5/2b1p3/2BPP3/2P2N2/PP3PPP/RNBQK2R b KQkq d3 0 5' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Close Variation'),
			moves => {
				'c5b6' => 'r1b1k1nr/ppppqppp/1bn5/4p3/2BPP3/2P2N2/PP3PPP/RNBQK2R w KQkq - 1 6',
			},
		},
		'r1b1k1nr/pppq1ppp/1b6/n3N3/2B1P3/BQP5/P4PPP/RN3RK1 b kq - 0 11' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6 7. Qb3 Qd7 8. dxe5 dxe5 9. O-O Bb6 10. Ba3 Na5 11. Nxe5

			variation => N__('Evans Gambit: Tartakower, Levenfish Variation'),
			moves => {
			},
		},
		'r1b1k1nr/pppq1ppp/1b6/n3p3/2B1P3/BQP2N2/P4PPP/RN3RK1 w kq - 4 11' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Tartakower Attack, 8.dxe5 dxe5'),
			moves => {
				'f3e5' => 'r1b1k1nr/pppq1ppp/1b6/n3N3/2B1P3/BQP5/P4PPP/RN3RK1 b kq - 0 11',
			},
		},
		'r1b1k1nr/pppq1ppp/1bn5/4p3/2B1P3/1QP2N2/P4PPP/RNB2RK1 w kq - 2 10' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Tartakower Attack, 8.dxe5 dxe5'),
			moves => {
				'c1a3' => 'r1b1k1nr/pppq1ppp/1bn5/4p3/2B1P3/BQP2N2/P4PPP/RN3RK1 b kq - 3 10',
			},
		},
		'r1b1k1nr/pppq1ppp/1bn5/4p3/2B1P3/BQP2N2/P4PPP/RN3RK1 b kq - 3 10' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Tartakower Attack, 8.dxe5 dxe5'),
			moves => {
				'c6a5' => 'r1b1k1nr/pppq1ppp/1b6/n3p3/2B1P3/BQP2N2/P4PPP/RN3RK1 w kq - 4 11',
			},
		},
		'r1b1k1nr/pppq1ppp/1bnp4/4P3/2B1P3/1QP2N2/P4PPP/RNB1K2R w KQkq - 1 9' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6 7. Qb3 Qd7 8. dxe5 Bb6

			variation => N__('Evans Gambit: Tartakower Attack, 8.dxe5 Bb6'),
			moves => {
			},
		},
		'r1b1k1nr/pppq1ppp/2n5/b3p3/2B1P3/1QP2N2/P4PPP/RNB1K2R w KQkq - 0 9' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6 7. Qb3 Qd7 8. dxe5 dxe5

			variation => N__('Evans Gambit: Tartakower Attack, 8.dxe5 dxe5'),
			moves => {
				'e1g1' => 'r1b1k1nr/pppq1ppp/2n5/b3p3/2B1P3/1QP2N2/P4PPP/RNB2RK1 b kq - 1 9',
			},
		},
		'r1b1k1nr/pppq1ppp/2n5/b3p3/2B1P3/1QP2N2/P4PPP/RNB2RK1 b kq - 1 9' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Tartakower Attack, 8.dxe5 dxe5'),
			moves => {
				'a5b6' => 'r1b1k1nr/pppq1ppp/1bn5/4p3/2B1P3/1QP2N2/P4PPP/RNB2RK1 w kq - 2 10',
			},
		},
		'r1b1k1nr/pppq1ppp/2np4/b3P3/2B1P3/1QP2N2/P4PPP/RNB1K2R b KQkq - 0 8' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Tartakower Attack, 7... Qd7'),
			moves => {
				'a5b6' => 'r1b1k1nr/pppq1ppp/1bnp4/4P3/2B1P3/1QP2N2/P4PPP/RNB1K2R w KQkq - 1 9',
				'd6e5' => 'r1b1k1nr/pppq1ppp/2n5/b3p3/2B1P3/1QP2N2/P4PPP/RNB1K2R w KQkq - 0 9',
			},
		},
		'r1b1k1nr/pppq1ppp/2np4/b3p3/2BPP3/1QP2N2/P4PPP/RNB1K2R w KQkq - 2 8' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6 7. Qb3 Qd7

			variation => N__('Evans Gambit: Tartakower Attack, 7... Qd7'),
			moves => {
				'd4e5' => 'r1b1k1nr/pppq1ppp/2np4/b3P3/2B1P3/1QP2N2/P4PPP/RNB1K2R b KQkq - 0 8',
			},
		},
		'r1b1k1r1/ppq1np1Q/2n1p3/3pP3/3p1P2/P1P5/2P1N1PP/R1B1KB1R b KQq f3 0 11' => {
			eco => 'C18t',
                        # TRANSLATORS:

			variation => N__('French: Winawer, Poisoned Pawn, Konstantinopolsky Variation'),
			moves => {
				'c8d7' => 'r3k1r1/ppqbnp1Q/2n1p3/3pP3/3p1P2/P1P5/2P1N1PP/R1B1KB1R w KQq - 1 12',
			},
		},
		'r1b1k1r1/ppq1np1Q/2n1p3/3pP3/3p4/P1P5/2P1NPPP/R1B1KB1R w KQq - 2 11' => {
			eco => 'C18t',
                        # TRANSLATORS:

			variation => N__('French: Winawer, Poisoned Pawn, Konstantinopolsky Variation'),
			moves => {
				'f2f4' => 'r1b1k1r1/ppq1np1Q/2n1p3/3pP3/3p1P2/P1P5/2P1N1PP/R1B1KB1R b KQq f3 0 11',
			},
		},
		'r1b1k2N/ppp1q1pp/2n2n2/3Pp3/2B5/8/PPPP1bPP/RNBQ1K1R b q - 0 8' => {
			eco => 'C57c',
                        # TRANSLATORS:

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1'),
			moves => {
				'c6d4' => 'r1b1k2N/ppp1q1pp/5n2/3Pp3/2Bn4/8/PPPP1bPP/RNBQ1K1R w q - 1 9',
				'c8g4' => 'r3k2N/ppp1q1pp/2n2n2/3Pp3/2B3b1/8/PPPP1bPP/RNBQ1K1R w q - 1 9',
			},
		},
		'r1b1k2N/ppp1q1pp/2n2n2/3pp3/2B1P3/8/PPPP1bPP/RNBQ1K1R w q d6 0 8' => {
			eco => 'C57c',
                        # TRANSLATORS:

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1'),
			moves => {
				'e4d5' => 'r1b1k2N/ppp1q1pp/2n2n2/3Pp3/2B5/8/PPPP1bPP/RNBQ1K1R b q - 0 8',
			},
		},
		'r1b1k2N/ppp1q1pp/3P1n2/4p3/2Bn4/8/PPPP1bPP/RNBQ1K1R b q - 0 9' => {
			eco => 'C57c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 Bc5 5. Nxf7 Bxf2+ 6. Kf1 Qe7 7. Nxh8 d5 8. exd5 Nd4 9. d6

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1, Menovsky Variation'),
			moves => {
			},
		},
		'r1b1k2N/ppp1q1pp/5n2/3Pp3/2Bn4/7P/PPPP1bP1/RNBQ1K1R b q - 0 9' => {
			eco => 'C57c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 Bc5 5. Nxf7 Bxf2+ 6. Kf1 Qe7 7. Nxh8 d5 8. exd5 Nd4 9. h3

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1, Palkin Variation'),
			moves => {
			},
		},
		'r1b1k2N/ppp1q1pp/5n2/3Pp3/2Bn4/8/PPPP1bPP/RNBQ1K1R w q - 1 9' => {
			eco => 'C57c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 Bc5 5. Nxf7 Bxf2+ 6. Kf1 Qe7 7. Nxh8 d5 8. exd5 Nd4

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1, 8... Nd4'),
			moves => {
				'd5d6' => 'r1b1k2N/ppp1q1pp/3P1n2/4p3/2Bn4/8/PPPP1bPP/RNBQ1K1R b q - 0 9',
				'h2h3' => 'r1b1k2N/ppp1q1pp/5n2/3Pp3/2Bn4/7P/PPPP1bP1/RNBQ1K1R b q - 0 9',
			},
		},
		'r1b1k2N/ppppq1pp/2n2n2/4p3/2B1P3/8/PPPP1bPP/RNBQ1K1R b q - 0 7' => {
			eco => 'C57c',
                        # TRANSLATORS:

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1'),
			moves => {
				'd7d5' => 'r1b1k2N/ppp1q1pp/2n2n2/3pp3/2B1P3/8/PPPP1bPP/RNBQ1K1R w q d6 0 8',
			},
		},
		'r1b1k2r/1p1nbppp/p2ppn2/6B1/3NPP2/q1N5/P1PQB1PP/1R3RK1 w kq - 6 12' => {
			eco => 'B97n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. Be2 Be7 11. O-O Nbd7

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 10. Be2 Be7 11.O-O Nbd7'),
			moves => {
			},
		},
		'r1b1k2r/1p1nbppp/pq1ppn2/6B1/4PP2/1NN2Q2/PPP3PP/R3KB1R w KQkq - 5 10' => {
			eco => 'B97h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Nb3 Nbd7 9. Qf3 Be7

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 8. Nb3 Nbd7 9. Qf3 Be7'),
			moves => {
			},
		},
		'r1b1k2r/1p2bppp/pqnppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w kq - 2 10' => {
			eco => 'B63g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Qb6 8. Nb3 a6 9. O-O-O Be7

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Qb6 8. Nb3 a6 9.O-O-O Be7'),
			moves => {
			},
		},
		'r1b1k2r/1pp1qppp/2np1n2/p2Pp3/2P1P3/5NP1/PP1N1PBP/R2Q1RK1 b kq - 0 10' => {
			eco => 'E11w',
                        # TRANSLATORS:

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 Main Line, 8... a5 9. e4'),
			moves => {
				'c6b8' => 'rnb1k2r/1pp1qppp/3p1n2/p2Pp3/2P1P3/5NP1/PP1N1PBP/R2Q1RK1 w kq - 1 11',
			},
		},
		'r1b1k2r/1pp1qppp/2np1n2/p3p3/2PPP3/5NP1/PP1N1PBP/R2Q1RK1 w kq - 0 10' => {
			eco => 'E11w',
                        # TRANSLATORS:

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 Main Line, 8... a5 9. e4'),
			moves => {
				'd4d5' => 'r1b1k2r/1pp1qppp/2np1n2/p2Pp3/2P1P3/5NP1/PP1N1PBP/R2Q1RK1 b kq - 0 10',
			},
		},
		'r1b1k2r/1pp1qppp/2nppn2/p7/2PP4/5NP1/PP1NPPBP/R2Q1RK1 w kq a6 0 9' => {
			eco => 'E11w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2 Bxd2+ 7. Nbxd2 d6 8. O-O a5

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 Main Line, 8... a5'),
			moves => {
				'e2e4' => 'r1b1k2r/1pp1qppp/2nppn2/p7/2PPP3/5NP1/PP1N1PBP/R2Q1RK1 b kq e3 0 9',
			},
		},
		'r1b1k2r/1pp1qppp/2nppn2/p7/2PPP3/5NP1/PP1N1PBP/R2Q1RK1 b kq e3 0 9' => {
			eco => 'E11w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2 Bxd2+ 7. Nbxd2 d6 8. O-O a5 9. e4

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 Main Line, 8... a5 9. e4'),
			moves => {
				'e6e5' => 'r1b1k2r/1pp1qppp/2np1n2/p3p3/2PPP3/5NP1/PP1N1PBP/R2Q1RK1 w kq - 0 10',
			},
		},
		'r1b1k2r/1pp1qppp/pbnp1n2/4p3/P1BPP3/2P2N1P/1P3PP1/RNBQR1K1 b kq - 0 9' => {
			eco => 'C53',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Qe7 5. d4 Bb6 6. O-O Nf6 7. a4 a6 8. Re1 d6 9. h3

			variation => N__('Giuoco Piano: Tarrasch Variation'),
			moves => {
			},
		},
		'r1b1k2r/1pp1qppp/pbnp1n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQR1K1 w kq - 0 9' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'h2h3' => 'r1b1k2r/1pp1qppp/pbnp1n2/4p3/P1BPP3/2P2N1P/1P3PP1/RNBQR1K1 b kq - 0 9',
			},
		},
		'r1b1k2r/1pppqppp/pbn2n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'f1e1' => 'r1b1k2r/1pppqppp/pbn2n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQR1K1 b kq - 1 8',
			},
		},
		'r1b1k2r/1pppqppp/pbn2n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQR1K1 b kq - 1 8' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'd7d6' => 'r1b1k2r/1pp1qppp/pbnp1n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQR1K1 w kq - 0 9',
			},
		},
		'r1b1k2r/1pq1bppp/p1Nppn2/6B1/4PP2/2N2Q2/PPP3PP/2KR1B1R b kq - 0 10' => {
			eco => 'B98q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nc6 10. Nxc6

			variation => N__('Sicilian: Najdorf, 8... Qc7 9.O-O-O Nc6 10.Nxc6'),
			moves => {
			},
		},
		'r1b1k2r/1pq1bppp/p1nppn2/6B1/3NPP2/2N2Q2/PPP3PP/2KR1B1R w kq - 5 10' => {
			eco => 'B98p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nc6

			variation => N__('Sicilian: Najdorf, 8... Qc7 9.O-O-O Nc6'),
			moves => {
				'd4c6' => 'r1b1k2r/1pq1bppp/p1Nppn2/6B1/4PP2/2N2Q2/PPP3PP/2KR1B1R b kq - 0 10',
			},
		},
		'r1b1k2r/1pq1bppp/p1nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQR1K1 w kq - 2 10' => {
			eco => 'B80u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. g3 Nc6 7. Bg2 Qc7 8. O-O a6 9. Re1 Be7

			variation => N__('Sicilian: Scheveningen, Fianchetto Variation'),
			moves => {
			},
		},
		'r1b1k2r/1pq1bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w kq - 3 10' => {
			eco => 'B84',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 Qc7 9. Be3 Nc6

			variation => N__('Sicilian: Scheveningen, Classical, 7.O-O Be7 8. f4 Qc7 9. Be3 Nc6'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbp2/p2ppn1p/6P1/3NP2B/2NB1Q2/PPP3PP/2KR3R b kq - 0 12' => {
			eco => 'B99i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3 h6 11. Bh4 g5 12. fxg5

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 h6 11. Bh4 g5 12.fxg5'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbp2/p2ppn1p/6p1/3NPP1B/2NB1Q2/PPP3PP/2KR3R w kq g6 0 12' => {
			eco => 'B99h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3 h6 11. Bh4 g5

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 h6 11. Bh4 g5'),
			moves => {
				'f4g5' => 'r1b1k2r/1pqnbp2/p2ppn1p/6P1/3NP2B/2NB1Q2/PPP3PP/2KR3R b kq - 0 12',
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/6B1/3NPP2/2NB1Q2/PPP3PP/2KR3R w kq - 0 11' => {
			eco => 'B99f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3 h6

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 h6'),
			moves => {
				'f3h3' => 'r1b1k2r/1pqnbpp1/p2ppn1p/6B1/3NPP2/2NB3Q/PPP3PP/2KR3R b kq - 1 11',
				'g5h4' => 'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2NB1Q2/PPP3PP/2KR3R b kq - 1 11',
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/6B1/3NPP2/2NB3Q/PPP3PP/2KR3R b kq - 1 11' => {
			eco => 'B99g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3 h6 11. Qh3

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 h6 11. Qh3'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/6B1/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq - 0 11' => {
			eco => 'B99m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 h6

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 h6'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2N2Q2/PPP1B1PP/2KR3R b kq - 5 11' => {
			eco => 'B98j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 h6 9. Bh4 Qc7 10. O-O-O Nbd7 11. Be2

			variation => N__('Sicilian: Najdorf, Browne, 10.O-O-O Nbd7 11. Be2'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2N2Q2/PPP3PP/2KR1B1R w kq - 4 11' => {
			eco => 'B98h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 h6 9. Bh4 Qc7 10. O-O-O Nbd7

			variation => N__('Sicilian: Najdorf, Browne, 10.O-O-O Nbd7'),
			moves => {
				'f1e2' => 'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2N2Q2/PPP1B1PP/2KR3R b kq - 5 11',
				'f3g3' => 'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2N3Q1/PPP3PP/2KR1B1R b kq - 5 11',
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2N3Q1/PPP3PP/2KR1B1R b kq - 5 11' => {
			eco => 'B98i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 h6 9. Bh4 Qc7 10. O-O-O Nbd7 11. Qg3

			variation => N__('Sicilian: Najdorf, Browne, 10.O-O-O Nbd7 11. Qg3'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbpp1/p2ppn1p/8/3NPP1B/2NB1Q2/PPP3PP/2KR3R b kq - 1 11' => {
			eco => 'B99h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3 h6 11. Bh4

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 h6 11. Bh4'),
			moves => {
				'g7g5' => 'r1b1k2r/1pqnbp2/p2ppn1p/6p1/3NPP1B/2NB1Q2/PPP3PP/2KR3R w kq g6 0 12',
			},
		},
		'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2N2Q2/PPP1B1PP/2KR3R b kq - 6 10' => {
			eco => 'B99c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Be2

			variation => N__('Sicilian: Najdorf, Main Line, Keres Variation'),
			moves => {
				'b7b5' => 'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPP2/2N2Q2/PPP1B1PP/2KR3R w kq b6 0 11',
			},
		},
		'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2N2Q2/PPP3PP/2KR1B1R w kq - 5 10' => {
			eco => 'B99a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7

			variation => N__('Sicilian: Najdorf, Main Line'),
			moves => {
				'f1d3' => 'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2NB1Q2/PPP3PP/2KR3R b kq - 6 10',
				'f1e2' => 'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2N2Q2/PPP1B1PP/2KR3R b kq - 6 10',
				'f3g3' => 'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2N3Q1/PPP3PP/2KR1B1R b kq - 6 10',
				'g2g4' => 'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPPP1/2N2Q2/PPP4P/2KR1B1R b kq g3 0 10',
			},
		},
		'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2N3Q1/PPP3PP/2KR1B1R b kq - 6 10' => {
			eco => 'B99b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Qg3

			variation => N__('Sicilian: Najdorf, Main Line, 10. Qg3'),
			moves => {
			},
		},
		'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPP2/2NB1Q2/PPP3PP/2KR3R b kq - 6 10' => {
			eco => 'B99f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3'),
			moves => {
				'b7b5' => 'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KR3R w kq b6 0 11',
				'h7h6' => 'r1b1k2r/1pqnbpp1/p2ppn1p/6B1/3NPP2/2NB1Q2/PPP3PP/2KR3R w kq - 0 11',
			},
		},
		'r1b1k2r/1pqnbppp/p2ppn2/6B1/3NPPP1/2N2Q2/PPP4P/2KR1B1R b kq g3 0 10' => {
			eco => 'B99m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4'),
			moves => {
				'b7b5' => 'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq b6 0 11',
				'h7h6' => 'r1b1k2r/1pqnbpp1/p2ppn1p/6B1/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq - 0 11',
			},
		},
		'r1b1k2r/1pqp1ppp/p1N1pn2/8/1b2P3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 0 9' => {
			eco => 'B49i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Nxc6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9.Nxc6'),
			moves => {
			},
		},
		'r1b1k2r/1pqp1ppp/p1n1pn2/2b5/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 4 9' => {
			eco => 'B47n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 Nf6 8. O-O Bc5

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 Nf6 8.O-O Bc5'),
			moves => {
			},
		},
		'r1b1k2r/1pqp1ppp/p1n1pn2/8/1b1NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq - 4 9' => {
			eco => 'B49h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4'),
			moves => {
				'c3a4' => 'r1b1k2r/1pqp1ppp/p1n1pn2/8/Nb1NP3/4B3/PPP1BPPP/R2Q1RK1 b kq - 5 9',
				'd4c6' => 'r1b1k2r/1pqp1ppp/p1N1pn2/8/1b2P3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 0 9',
			},
		},
		'r1b1k2r/1pqp1ppp/p1n1pn2/8/Nb1NP3/4B3/PPP1BPPP/R2Q1RK1 b kq - 5 9' => {
			eco => 'B49j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Na4

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9. Na4'),
			moves => {
				'b4e7' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/N2NP3/4B3/PPP1BPPP/R2Q1RK1 w kq - 6 10',
				'e8g8' => 'r1b2rk1/1pqp1ppp/p1n1pn2/8/Nb1NP3/4B3/PPP1BPPP/R2Q1RK1 w - - 6 10',
			},
		},
		'r1b1k2r/1pqp1ppp/p1nbpn2/8/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq - 4 9' => {
			eco => 'B48r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O Bd6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O Bd6'),
			moves => {
			},
		},
		'r1b1k2r/1pqp1ppp/p3pn2/2b1n3/3NP3/2NBB2P/PPP2PP1/R2Q1R1K b kq - 2 10' => {
			eco => 'B48v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O Ne5 9. h3 Bc5 10. Kh1

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O Ne5 9. h3 Bc5 10. Kh1'),
			moves => {
			},
		},
		'r1b1k2r/1pqp1ppp/p3pn2/2b1n3/3NP3/2NBB2P/PPP2PP1/R2Q1RK1 w kq - 1 10' => {
			eco => 'B48u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O Ne5 9. h3 Bc5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O Ne5 9. h3 Bc5'),
			moves => {
				'g1h1' => 'r1b1k2r/1pqp1ppp/p3pn2/2b1n3/3NP3/2NBB2P/PPP2PP1/R2Q1R1K b kq - 2 10',
			},
		},
		'r1b1k2r/1pqp1ppp/p3pn2/2b5/3QP3/2N5/PPP1BPPP/R1B2R1K w kq - 1 10' => {
			eco => 'B47y',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Nxd4'),
			moves => {
				'd4d3' => 'r1b1k2r/1pqp1ppp/p3pn2/2b5/4P3/2NQ4/PPP1BPPP/R1B2R1K b kq - 2 10',
			},
		},
		'r1b1k2r/1pqp1ppp/p3pn2/2b5/4P3/2NQ4/PPP1BPPP/R1B2R1K b kq - 2 10' => {
			eco => 'B47y',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. Kh1 Nxd4 9. Qxd4 Bc5 10. Qd3

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Nxd4, 10. Qd3'),
			moves => {
				'b7b5' => 'r1b1k2r/2qp1ppp/p3pn2/1pb5/4P3/2NQ4/PPP1BPPP/R1B2R1K w kq b6 0 11',
			},
		},
		'r1b1k2r/1pqpbppp/p1N1pn2/8/N3P3/4B3/PPP1BPPP/R2Q1RK1 b kq - 0 10' => {
			eco => 'B49m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Na4 Be7 10. Nxc6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9. Na4 Be7 10.Nxc6'),
			moves => {
				'b7c6' => 'r1b1k2r/2qpbppp/p1p1pn2/8/N3P3/4B3/PPP1BPPP/R2Q1RK1 w kq - 0 11',
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq - 4 9' => {
			eco => 'B49g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Be7

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Be7'),
			moves => {
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 4 9' => {
			eco => 'B47o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 Nf6 8. O-O Be7

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 Nf6 8.O-O Be7'),
			moves => {
				'f1e1' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQR1K1 b kq - 5 9',
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQR1K1 b kq - 5 9' => {
			eco => 'B47p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 Nf6 8. O-O Be7 9. Re1

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 Nf6 8.O-O Be7 9. Re1'),
			moves => {
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQ1R1K w kq - 4 9' => {
			eco => 'B47x',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. Kh1 Be7

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Be7'),
			moves => {
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/P1N1B3/1PP1BPPP/R2Q1RK1 b kq - 2 9' => {
			eco => 'B49e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. a3 Be7 9. O-O

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8. a3 Be7 9.O-O'),
			moves => {
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/P1N1B3/1PP1BPPP/R2QK2R w KQkq - 1 9' => {
			eco => 'B49e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. a3 Be7

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8. a3 Be7'),
			moves => {
				'e1g1' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/P1N1B3/1PP1BPPP/R2Q1RK1 b kq - 2 9',
			},
		},
		'r1b1k2r/1pqpbppp/p1n1pn2/8/N2NP3/4B3/PPP1BPPP/R2Q1RK1 w kq - 6 10' => {
			eco => 'B49l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Na4 Be7

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9. Na4 Be7'),
			moves => {
				'd4c6' => 'r1b1k2r/1pqpbppp/p1N1pn2/8/N3P3/4B3/PPP1BPPP/R2Q1RK1 b kq - 0 10',
			},
		},
		'r1b1k2r/2q1bp1p/p2ppP2/1pn5/3NP3/2N2Q2/PPP4P/2KR1B1R b kq - 0 15' => {
			eco => 'B99x',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5 14. f6'),
			moves => {
				'e7f8' => 'r1b1kb1r/2q2p1p/p2ppP2/1pn5/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 1 16',
			},
		},
		'r1b1k2r/2q1bp1p/p2ppp2/1pn3P1/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 0 15' => {
			eco => 'B99x',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5 14. f6'),
			moves => {
				'g5f6' => 'r1b1k2r/2q1bp1p/p2ppP2/1pn5/3NP3/2N2Q2/PPP4P/2KR1B1R b kq - 0 15',
			},
		},
		'r1b1k2r/2q1bppp/p1np1n2/1pp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w kq - 1 12' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'd4d5' => 'r1b1k2r/2q1bppp/p1np1n2/1ppPp3/4P3/2P2N1P/PPB2PP1/RNBQR1K1 b kq - 0 12',
			},
		},
		'r1b1k2r/2q1bppp/p1np1n2/1ppPp3/4P3/2P2N1P/PPB2PP1/RNBQR1K1 b kq - 0 12' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'c6b8' => 'rnb1k2r/2q1bppp/p2p1n2/1ppPp3/4P3/2P2N1P/PPB2PP1/RNBQR1K1 w kq - 1 13',
			},
		},
		'r1b1k2r/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 b kq - 0 11' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'a5c6' => 'r1b1k2r/2q1bppp/p1np1n2/1pp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w kq - 1 12',
			},
		},
		'r1b1k2r/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N2/PPB2PPP/RNBQR1K1 w kq - 1 11' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'a2a4' => 'r1b1k2r/2q1bppp/p2p1n2/npp1p3/P2PP3/2P2N2/1PB2PPP/RNBQR1K1 b kq a3 0 11',
				'h2h3' => 'r1b1k2r/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 b kq - 0 11',
			},
		},
		'r1b1k2r/2q1bppp/p2p1n2/npp1p3/P2PP3/2P2N2/1PB2PPP/RNBQR1K1 b kq a3 0 11' => {
			eco => 'C88e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 d6 8. c3 Na5 9. Bc2 c5 10. d4 Qc7 11. a4

			variation => N__('Spanish: Closed, Balla Variation'),
			moves => {
			},
		},
		'r1b1k2r/2q1bppp/p2pp3/1pn2PP1/3NP2P/2N2Q2/PPP5/2KR1B1R b kq h3 0 14' => {
			eco => 'B99w',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5 Nc5 14. h4

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5 14. h4'),
			moves => {
			},
		},
		'r1b1k2r/2q1bppp/p2pp3/1pn2PP1/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 1 14' => {
			eco => 'B99v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5 Nc5

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5'),
			moves => {
				'f5f6' => 'r1b1k2r/2q1bppp/p2ppP2/1pn3P1/3NP3/2N2Q2/PPP4P/2KR1B1R b kq - 0 14',
				'h2h4' => 'r1b1k2r/2q1bppp/p2pp3/1pn2PP1/3NP2P/2N2Q2/PPP5/2KR1B1R b kq h3 0 14',
			},
		},
		'r1b1k2r/2q1bppp/p2ppP2/1pn3P1/3NP3/2N2Q2/PPP4P/2KR1B1R b kq - 0 14' => {
			eco => 'B99x',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5 Nc5 14. f6

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5 14. f6'),
			moves => {
				'g7f6' => 'r1b1k2r/2q1bp1p/p2ppp2/1pn3P1/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 0 15',
			},
		},
		'r1b1k2r/2q1bppp/p2ppn2/1p2P3/3N1P2/2N2Q2/PPP1B1PP/2KR3R b kq - 0 12' => {
			eco => 'B99c',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Main Line, Keres Variation'),
			moves => {
				'c8b7' => 'r3k2r/1bq1bppp/p2ppn2/1p2P3/3N1P2/2N2Q2/PPP1B1PP/2KR3R w kq - 1 13',
			},
		},
		'r1b1k2r/2q1bppp/p2ppn2/1p4P1/3NPP2/2N2Q2/PPP4P/2KR1B1R b kq - 0 12' => {
			eco => 'B99r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11.Bxf6 Nxf6 12. g5'),
			moves => {
				'f6d7' => 'r1b1k2r/2qnbppp/p2pp3/1p4P1/3NPP2/2N2Q2/PPP4P/2KR1B1R w kq - 1 13',
			},
		},
		'r1b1k2r/2q1bppp/p2ppn2/1p6/3NPP2/2N2Q2/PPP1B1PP/2KR3R w kq - 0 12' => {
			eco => 'B99c',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Main Line, Keres Variation'),
			moves => {
				'e4e5' => 'r1b1k2r/2q1bppp/p2ppn2/1p2P3/3N1P2/2N2Q2/PPP1B1PP/2KR3R b kq - 0 12',
			},
		},
		'r1b1k2r/2q1bppp/p2ppn2/1p6/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq - 0 12' => {
			eco => 'B99r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11.Bxf6 Nxf6'),
			moves => {
				'g4g5' => 'r1b1k2r/2q1bppp/p2ppn2/1p4P1/3NPP2/2N2Q2/PPP4P/2KR1B1R b kq - 0 12',
			},
		},
		'r1b1k2r/2q2ppp/p2pp3/1p2nPb1/3NP3/2N2Q2/PPP4P/1K1R1B1R w kq - 2 15' => {
			eco => 'B99u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5 Bxg5+ 14. Kb1 Ne5

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Bxg5+ 14. Kb1 Ne5'),
			moves => {
			},
		},
		'r1b1k2r/2qn1ppp/p2pp3/1p3Pb1/3NP3/2N2Q2/PPP4P/1K1R1B1R b kq - 1 14' => {
			eco => 'B99t',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Bxg5+'),
			moves => {
				'd7e5' => 'r1b1k2r/2q2ppp/p2pp3/1p2nPb1/3NP3/2N2Q2/PPP4P/1K1R1B1R w kq - 2 15',
			},
		},
		'r1b1k2r/2qn1ppp/p2pp3/1p3Pb1/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 0 14' => {
			eco => 'B99t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5 Bxg5+

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Bxg5+'),
			moves => {
				'c1b1' => 'r1b1k2r/2qn1ppp/p2pp3/1p3Pb1/3NP3/2N2Q2/PPP4P/1K1R1B1R b kq - 1 14',
			},
		},
		'r1b1k2r/2qnbp1p/p2ppp2/1p6/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq - 0 12' => {
			eco => 'B99q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 gxf6

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11.Bxf6 gxf6'),
			moves => {
			},
		},
		'r1b1k2r/2qnbppp/p2pp3/1p3PP1/3NP3/2N2Q2/PPP4P/2KR1B1R b kq - 0 13' => {
			eco => 'B99s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5'),
			moves => {
				'd7c5' => 'r1b1k2r/2q1bppp/p2pp3/1pn2PP1/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 1 14',
				'e7g5' => 'r1b1k2r/2qn1ppp/p2pp3/1p3Pb1/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 0 14',
			},
		},
		'r1b1k2r/2qnbppp/p2pp3/1p4P1/3NPP2/2N2Q2/PPP4P/2KR1B1R w kq - 1 13' => {
			eco => 'B99r',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11.Bxf6 Nxf6 12. g5'),
			moves => {
				'a2a3' => 'r1b1k2r/2qnbppp/p2pp3/1p4P1/3NPP2/P1N2Q2/1PP4P/2KR1B1R b kq - 0 13',
				'f4f5' => 'r1b1k2r/2qnbppp/p2pp3/1p3PP1/3NP3/2N2Q2/PPP4P/2KR1B1R b kq - 0 13',
			},
		},
		'r1b1k2r/2qnbppp/p2pp3/1p4P1/3NPP2/P1N2Q2/1PP4P/2KR1B1R b kq - 0 13' => {
			eco => 'B99r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. a3

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11.Bxf6 Nxf6 12. g5 Nd7 13. a3'),
			moves => {
			},
		},
		'r1b1k2r/2qnbppp/p2ppB2/1p6/3NPP2/2N2Q2/PPP1B1PP/2KR3R b kq - 0 11' => {
			eco => 'B99c',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Main Line, Keres Variation'),
			moves => {
				'd7f6' => 'r1b1k2r/2q1bppp/p2ppn2/1p6/3NPP2/2N2Q2/PPP1B1PP/2KR3R w kq - 0 12',
			},
		},
		'r1b1k2r/2qnbppp/p2ppB2/1p6/3NPPP1/2N2Q2/PPP4P/2KR1B1R b kq - 0 11' => {
			eco => 'B99p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11.Bxf6'),
			moves => {
				'd7f6' => 'r1b1k2r/2q1bppp/p2ppn2/1p6/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq - 0 12',
				'g7f6' => 'r1b1k2r/2qnbp1p/p2ppp2/1p6/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq - 0 12',
			},
		},
		'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPP2/2N2Q2/PPP1B1PP/2KR3R w kq b6 0 11' => {
			eco => 'B99c',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Main Line, Keres Variation'),
			moves => {
				'g5f6' => 'r1b1k2r/2qnbppp/p2ppB2/1p6/3NPP2/2N2Q2/PPP1B1PP/2KR3R b kq - 0 11',
			},
		},
		'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KR3R w kq b6 0 11' => {
			eco => 'B99j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. Bd3 b5

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 b5'),
			moves => {
				'h1e1' => 'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KRR3 b kq - 1 11',
			},
		},
		'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KRR3 b kq - 1 11' => {
			eco => 'B99j',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Main Line, 10. Bd3 b5'),
			moves => {
				'c8b7' => 'r3k2r/1bqnbppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KRR3 w kq - 2 12',
			},
		},
		'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPPP1/2N2Q2/PPP4P/2KR1B1R w kq b6 0 11' => {
			eco => 'B99n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5'),
			moves => {
				'a2a3' => 'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPPP1/P1N2Q2/1PP4P/2KR1B1R b kq - 0 11',
				'f1d3' => 'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPPP1/2NB1Q2/PPP4P/2KR3R b kq - 1 11',
				'g5f6' => 'r1b1k2r/2qnbppp/p2ppB2/1p6/3NPPP1/2N2Q2/PPP4P/2KR1B1R b kq - 0 11',
			},
		},
		'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPPP1/2NB1Q2/PPP4P/2KR3R b kq - 1 11' => {
			eco => 'B99n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bd3

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11. Bd3'),
			moves => {
				'c8b7' => 'r3k2r/1bqnbppp/p2ppn2/1p4B1/3NPPP1/2NB1Q2/PPP4P/2KR3R w kq - 2 12',
			},
		},
		'r1b1k2r/2qnbppp/p2ppn2/1p4B1/3NPPP1/P1N2Q2/1PP4P/2KR1B1R b kq - 0 11' => {
			eco => 'B99o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. a3

			variation => N__('Sicilian: Najdorf, Main Line, 10. g4 b5 11. a3'),
			moves => {
			},
		},
		'r1b1k2r/2qp1ppp/p3pn2/1pb5/4P3/2NQ4/PPP1BPPP/R1B2R1K w kq b6 0 11' => {
			eco => 'B47y',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. Kh1 Nxd4 9. Qxd4 Bc5 10. Qd3 b5

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Nxd4, 10. Qd3 b5'),
			moves => {
				'f2f4' => 'r1b1k2r/2qp1ppp/p3pn2/1pb5/4PP2/2NQ4/PPP1B1PP/R1B2R1K b kq f3 0 11',
			},
		},
		'r1b1k2r/2qp1ppp/p3pn2/1pb5/4PP2/2NQ4/PPP1B1PP/R1B2R1K b kq f3 0 11' => {
			eco => 'B47z',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. Kh1 Nxd4 9. Qxd4 Bc5 10. Qd3 b5 11. f4

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Nxd4, 10. Qd3 b5 11. f4'),
			moves => {
			},
		},
		'r1b1k2r/2qpbppp/p1p1pn2/8/N3P3/4B3/PPP1BPPP/R2Q1RK1 w kq - 0 11' => {
			eco => 'B49m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Na4 Be7 10. Nxc6 bxc6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9. Na4 Be7 10.Nxc6 bxc6'),
			moves => {
				'a4b6' => 'r1b1k2r/2qpbppp/pNp1pn2/8/4P3/4B3/PPP1BPPP/R2Q1RK1 b kq - 1 11',
			},
		},
		'r1b1k2r/2qpbppp/pNp1pn2/8/4P3/4B3/PPP1BPPP/R2Q1RK1 b kq - 1 11' => {
			eco => 'B49n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Na4 Be7 10. Nxc6 bxc6 11. Nb6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9. Na4 Be7 10.Nxc6 bxc6 11. Nb6'),
			moves => {
			},
		},
		'r1b1k2r/4b2p/p1p1pp2/4p3/4N3/q7/P1PQ2PP/1R2KB1R w Kkq - 2 16' => {
			eco => 'B97w',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 dxe5 14. Bxf6 gxf6 15. Ne4 Be7

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4 Be7'),
			moves => {
				'f1e2' => 'r1b1k2r/4b2p/p1p1pp2/4p3/4N3/q7/P1PQB1PP/1R2K2R b Kkq - 3 16',
			},
		},
		'r1b1k2r/4b2p/p1p1pp2/4p3/4N3/q7/P1PQB1PP/1R2K2R b Kkq - 3 16' => {
			eco => 'B97w',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4 Be7'),
			moves => {
				'h7h5' => 'r1b1k2r/4b3/p1p1pp2/4p2p/4N3/q7/P1PQB1PP/1R2K2R w Kkq h6 0 17',
			},
		},
		'r1b1k2r/4b3/p1p1pN2/4p2p/q7/1R6/P1PQB1PP/4K2R b Kkq - 0 18' => {
			eco => 'B97y',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 dxe5 14. Bxf6 gxf6 15. Ne4 Be7 16. Be2 h5 17. Rb3 Qa4 18. Nxf6+

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5 with 18.Nxf6+'),
			moves => {
			},
		},
		'r1b1k2r/4b3/p1p1pp2/4p2p/4N3/q7/P1PQB1PP/1R2K2R w Kkq h6 0 17' => {
			eco => 'B97w',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4 Be7'),
			moves => {
				'b1b3' => 'r1b1k2r/4b3/p1p1pp2/4p2p/4N3/qR6/P1PQB1PP/4K2R b Kkq - 1 17',
			},
		},
		'r1b1k2r/4b3/p1p1pp2/4p2p/4N3/qR6/P1PQB1PP/4K2R b Kkq - 1 17' => {
			eco => 'B97w',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4 Be7'),
			moves => {
				'a3a4' => 'r1b1k2r/4b3/p1p1pp2/4p2p/q3N3/1R6/P1PQB1PP/4K2R w Kkq - 2 18',
			},
		},
		'r1b1k2r/4b3/p1p1pp2/4p2p/q1P1N3/1R6/P2QB1PP/4K2R b Kkq c3 0 18' => {
			eco => 'B97x',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 dxe5 14. Bxf6 gxf6 15. Ne4 Be7 16. Be2 h5 17. Rb3 Qa4 18. c4

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5 with 18. c4'),
			moves => {
			},
		},
		'r1b1k2r/4b3/p1p1pp2/4p2p/q3N3/1R6/P1PQB1PP/4K2R w Kkq - 2 18' => {
			eco => 'B97w',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4 Be7'),
			moves => {
				'c2c4' => 'r1b1k2r/4b3/p1p1pp2/4p2p/q1P1N3/1R6/P2QB1PP/4K2R b Kkq c3 0 18',
				'e4f6' => 'r1b1k2r/4b3/p1p1pN2/4p2p/q7/1R6/P1PQB1PP/4K2R b Kkq - 0 18',
			},
		},
		'r1b1k2r/p1q2pp1/2pb1n1p/n3N3/3Pp3/8/PPP1BPPP/RNBQK2R w KQkq - 1 12' => {
			eco => 'C59j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. d4 Qc7

			variation => N__('Two Knights: Main Line, 11. d4 Qc7'),
			moves => {
				'c1d2' => 'r1b1k2r/p1q2pp1/2pb1n1p/n3N3/3Pp3/8/PPPBBPPP/RN1QK2R b KQkq - 2 12',
			},
		},
		'r1b1k2r/p1q2pp1/2pb1n1p/n3N3/3Pp3/8/PPPBBPPP/RN1QK2R b KQkq - 2 12' => {
			eco => 'C59j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. d4 Qc7 12. Bd2

			variation => N__('Two Knights: Main Line, Knorre Variation'),
			moves => {
			},
		},
		'r1b1k2r/p1q2pp1/2pb1n1p/n7/8/1P1N4/P1P1BPPP/RNBQK2R b KQkq - 0 13' => {
			eco => 'C59l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. d4 exd3 12. Nxd3 Qc7 13. b3

			variation => N__('Two Knights: Main Line, 11. d4, Honfi Variation'),
			moves => {
			},
		},
		'r1b1k2r/p1q2pp1/2pb1n1p/n7/8/3N4/PPP1BPPP/RNBQK2R w KQkq - 1 13' => {
			eco => 'C59k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. d4 exd3 12. Nxd3 Qc7

			variation => N__('Two Knights: Main Line, 11. d4 exd3 12.Nxd3 Qc7'),
			moves => {
				'b2b3' => 'r1b1k2r/p1q2pp1/2pb1n1p/n7/8/1P1N4/P1P1BPPP/RNBQK2R b KQkq - 0 13',
			},
		},
		'r1b1k2r/pp1n1pb1/2p1pqpp/8/1PBP4/2N1PN2/P4PPP/R2Q1RK1 b kq b3 0 11' => {
			eco => 'D43w',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3 dxc4 9. Bxc4 g6 10. O-O Bg7 11. b4

			variation => N__('Semi-Slav: Moscow, Main Line, 11. b4'),
			moves => {
			},
		},
		'r1b1k2r/pp1n1pb1/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/2RQ1RK1 b kq - 3 11' => {
			eco => 'D43x',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3 dxc4 9. Bxc4 g6 10. O-O Bg7 11. Rc1

			variation => N__('Semi-Slav: Moscow, Main Line, 11. Rc1'),
			moves => {
			},
		},
		'r1b1k2r/pp1n1pb1/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/R2Q1RK1 w kq - 2 11' => {
			eco => 'D43v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3 dxc4 9. Bxc4 g6 10. O-O Bg7

			variation => N__('Semi-Slav: Moscow, Main Line'),
			moves => {
				'a1c1' => 'r1b1k2r/pp1n1pb1/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/2RQ1RK1 b kq - 3 11',
				'b2b4' => 'r1b1k2r/pp1n1pb1/2p1pqpp/8/1PBP4/2N1PN2/P4PPP/R2Q1RK1 b kq b3 0 11',
			},
		},
		'r1b1k2r/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PP1N1PPP/R2QKB1R w KQkq - 4 8' => {
			eco => 'D52i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4

			variation => N__('QGD: Cambridge Springs, Bogoljubow Variation'),
			moves => {
				'd1c2' => 'r1b1k2r/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PPQN1PPP/R3KB1R b KQkq - 5 8',
			},
		},
		'r1b1k2r/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PPQN1PPP/R3KB1R b KQkq - 5 8' => {
			eco => 'D52j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4 8. Qc2

			variation => N__('QGD: Cambridge Springs, Bogoljubow, 8. Qc2'),
			moves => {
				'd5c4' => 'r1b1k2r/pp1n1ppp/2p1pn2/q5B1/1bpP4/2N1P3/PPQN1PPP/R3KB1R w KQkq - 0 9',
				'e8g8' => 'r1b2rk1/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PPQN1PPP/R3KB1R w KQ - 6 9',
			},
		},
		'r1b1k2r/pp1n1ppp/2p1pn2/q5B1/1bpP4/2N1P3/PPQN1PPP/R3KB1R w KQkq - 0 9' => {
			eco => 'D52k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4 8. Qc2 dxc4

			variation => N__('QGD: Cambridge Springs, 8. Qc2 dxc4'),
			moves => {
			},
		},
		'r1b1k2r/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 b kq - 7 8' => {
			eco => 'B35b',
                        # TRANSLATORS:

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5'),
			moves => {
				'e8g8' => 'r1b2rk1/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 w - - 8 9',
			},
		},
		'r1b1k2r/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQkq - 6 8' => {
			eco => 'B35b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5'),
			moves => {
				'e1g1' => 'r1b1k2r/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 b kq - 7 8',
			},
		},
		'r1b1k2r/pp1pppbp/4n1p1/q7/2P1P3/2N1B3/PP1Q1PPP/2R1KB1R b Kkq - 5 11' => {
			eco => 'B39l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 Ne6 10. Rc1 Qa5 11. Qd2

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 Ne6 10. Rc1 Qa5 11. Qd2'),
			moves => {
			},
		},
		'r1b1k2r/pp1pppbp/4n1p1/q7/2P1P3/2N1B3/PP3PPP/2RQKB1R w Kkq - 4 11' => {
			eco => 'B39k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 Ne6 10. Rc1 Qa5

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 Ne6 10. Rc1 Qa5'),
			moves => {
				'd1d2' => 'r1b1k2r/pp1pppbp/4n1p1/q7/2P1P3/2N1B3/PP1Q1PPP/2R1KB1R b Kkq - 5 11',
			},
		},
		'r1b1k2r/pp2bppp/1qnppn2/8/2B1P3/1NN1B3/PPP2PPP/R2Q1RK1 b kq - 3 9' => {
			eco => 'B57o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3 e6 8. O-O Be7 9. Be3

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3 e6 8.O-O Be7 9. Be3'),
			moves => {
			},
		},
		'r1b1k2r/pp2bppp/1qnppn2/8/2B1P3/1NN5/PPP2PPP/R1BQ1RK1 w kq - 2 9' => {
			eco => 'B57n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3 e6 8. O-O Be7

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3 e6 8.O-O Be7'),
			moves => {
				'c1e3' => 'r1b1k2r/pp2bppp/1qnppn2/8/2B1P3/1NN1B3/PPP2PPP/R2Q1RK1 b kq - 3 9',
			},
		},
		'r1b1k2r/pp2nppp/2n1p3/q1ppP3/P2P4/2P2N2/2P2PPP/R1BQKB1R w KQkq - 1 9' => {
			eco => 'C19n',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Nf3 Nbc6 8. a4 Qa5

			variation => N__('French: Winawer, 6... Ne7 7. Nf3 Nbc6 8. a4 Qa5'),
			moves => {
				'c1d2' => 'r1b1k2r/pp2nppp/2n1p3/q1ppP3/P2P4/2P2N2/2PB1PPP/R2QKB1R b KQkq - 2 9',
				'd1d2' => 'r1b1k2r/pp2nppp/2n1p3/q1ppP3/P2P4/2P2N2/2PQ1PPP/R1B1KB1R b KQkq - 2 9',
			},
		},
		'r1b1k2r/pp2nppp/2n1p3/q1ppP3/P2P4/2P2N2/2PB1PPP/R2QKB1R b KQkq - 2 9' => {
			eco => 'C19p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Nf3 Nbc6 8. a4 Qa5 9. Bd2

			variation => N__('French: Winawer, 6... Ne7 7. Nf3 Nbc6 8. a4 Qa5+ 9. Bd2'),
			moves => {
				'c8d7' => 'r3k2r/pp1bnppp/2n1p3/q1ppP3/P2P4/2P2N2/2PB1PPP/R2QKB1R w KQkq - 3 10',
			},
		},
		'r1b1k2r/pp2nppp/2n1p3/q1ppP3/P2P4/2P2N2/2PQ1PPP/R1B1KB1R b KQkq - 2 9' => {
			eco => 'C19n',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Nf3 Nbc6 8. a4 Qa5 9. Qd2

			variation => N__('French: Winawer, 6... Ne7 7. Nf3 Nbc6 8. a4 Qa5 9. Qd2'),
			moves => {
				'c8d7' => 'r3k2r/pp1bnppp/2n1p3/q1ppP3/P2P4/2P2N2/2PQ1PPP/R1B1KB1R w KQkq - 3 10',
			},
		},
		'r1b1k2r/pp2ppbp/2n3p1/q1p5/3PP3/2P1BN2/P2Q1PPP/2R1KB1R b Kkq - 5 10' => {
			eco => 'D85v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Nf3 c5 8. Be3 Qa5 9. Qd2 Nc6 10. Rc1

			variation => N__('Grünfeld: Modern Exchange, 8. Be3 Qa5 9. Qd2 Nc6 10. Rc1'),
			moves => {
			},
		},
		'r1b1k2r/pp2ppbp/2n3p1/q1p5/3PP3/2P1BN2/P2Q1PPP/R3KB1R w KQkq - 4 10' => {
			eco => 'D85u',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Modern Exchange, 8. Be3 Qa5 9. Qd2'),
			moves => {
				'a1c1' => 'r1b1k2r/pp2ppbp/2n3p1/q1p5/3PP3/2P1BN2/P2Q1PPP/2R1KB1R b Kkq - 5 10',
			},
		},
		'r1b1k2r/pp3ppp/2nqpn2/3p4/3P4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 0 9' => {
			eco => 'D13t',
                        # TRANSLATORS: 1. d4 d5 2. c4 c6 3. Nf3 Nf6 4. cxd5 cxd5 5. Nc3 Nc6 6. Bf4 e6 7. e3 Bd6 8. Bxd6 Qxd6

			variation => N__('Slav: Exchange, 6. Bf4 e6 7. e3 Bd6 8.Bxd6 Qxd6'),
			moves => {
			},
		},
		'r1b1k2r/pp4pp/1qnbpn2/3p4/3P4/2NB1N2/PP3PPP/R1BQ1RK1 b kq - 5 12' => {
			eco => 'C06p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 f6 9. exf6 Nxf6 10. O-O Bd6 11. Nf3 Qb6 12. Nc3

			variation => N__('French: Tarrasch, Closed, 8... f6 9.exf6, 11... Qb6 12. Nc3'),
			moves => {
			},
		},
		'r1b1k2r/pp4pp/1qnbpn2/3p4/3P4/3B1N2/PP2NPPP/R1BQ1RK1 w kq - 4 12' => {
			eco => 'C06o',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 f6 9. exf6 Nxf6 10. O-O Bd6 11. Nf3 Qb6

			variation => N__('French: Tarrasch, Closed, 8... f6 9.exf6, 11... Qb6'),
			moves => {
				'e2c3' => 'r1b1k2r/pp4pp/1qnbpn2/3p4/3P4/2NB1N2/PP3PPP/R1BQ1RK1 b kq - 5 12',
			},
		},
		'r1b1k2r/ppp1qppp/2nppn2/8/2PP4/5NP1/PP1NPPBP/R2Q1RK1 b kq - 1 8' => {
			eco => 'E11v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2 Bxd2+ 7. Nbxd2 d6 8. O-O

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 Main Line'),
			moves => {
				'a7a5' => 'r1b1k2r/1pp1qppp/2nppn2/p7/2PP4/5NP1/PP1NPPBP/R2Q1RK1 w kq a6 0 9',
				'e8g8' => 'r1b2rk1/ppp1qppp/2nppn2/8/2PP4/5NP1/PP1NPPBP/R2Q1RK1 w - - 2 9',
			},
		},
		'r1b1k2r/ppp1qppp/2nppn2/8/2PP4/5NP1/PP1NPPBP/R2QK2R w KQkq - 0 8' => {
			eco => 'E11u',
                        # TRANSLATORS:

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Bg2 Bxd2+ 7.Nbxd2'),
			moves => {
				'e1g1' => 'r1b1k2r/ppp1qppp/2nppn2/8/2PP4/5NP1/PP1NPPBP/R2Q1RK1 b kq - 1 8',
			},
		},
		'r1b1k2r/ppp1qppp/3p1n2/4P1B1/2BnP3/8/PPP3PP/RN1Q1RK1 b kq - 0 9' => {
			eco => 'C55b',
                        # TRANSLATORS:

			variation => N__('Two Knights: Deutz Gambit'),
			moves => {
				'd6e5' => 'r1b1k2r/ppp1qppp/5n2/4p1B1/2BnP3/8/PPP3PP/RN1Q1RK1 w kq - 0 10',
			},
		},
		'r1b1k2r/ppp1qppp/3p1n2/4p1B1/2BnPP2/8/PPP3PP/RN1Q1RK1 w kq - 1 9' => {
			eco => 'C55b',
                        # TRANSLATORS:

			variation => N__('Two Knights: Deutz Gambit'),
			moves => {
				'f4e5' => 'r1b1k2r/ppp1qppp/3p1n2/4P1B1/2BnP3/8/PPP3PP/RN1Q1RK1 b kq - 0 9',
			},
		},
		'r1b1k2r/ppp1qppp/5n2/4p1B1/2BnP3/2N5/PPP3PP/R2Q1RK1 b kq - 1 10' => {
			eco => 'C55b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. O-O Bc5 5. d4 Bxd4 6. Nxd4 Nxd4 7. Bg5 d6 8. f4 Qe7 9. fxe5 dxe5 10. Nc3

			variation => N__('Two Knights: 4.O-O Gambit, Holzhausen Attack'),
			moves => {
			},
		},
		'r1b1k2r/ppp1qppp/5n2/4p1B1/2BnP3/8/PPP3PP/RN1Q1RK1 w kq - 0 10' => {
			eco => 'C55b',
                        # TRANSLATORS:

			variation => N__('Two Knights: Deutz Gambit'),
			moves => {
				'b1c3' => 'r1b1k2r/ppp1qppp/5n2/4p1B1/2BnP3/2N5/PPP3PP/R2Q1RK1 b kq - 1 10',
			},
		},
		'r1b1k2r/pppp1ppp/5q2/1Bb5/8/2P5/PPP2PPP/R1BQK2R w KQkq - 1 9' => {
			eco => 'C48l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Nd4 5. Nxd4 exd4 6. e5 dxc3 7. exf6 Qxf6 8. dxc3 Bc5

			variation => N__('Four Knights: Rubinstein, Exchange Variation, 8... Bc5'),
			moves => {
			},
		},
		'r1b1k2r/ppppnp1p/2n4b/4q3/2B2p2/2NP1Q2/PPPB2PP/4RRK1 b kq - 6 11' => {
			eco => 'C37',
                        # TRANSLATORS: 1. e4 e5 2. f4 exf4 3. Nf3 g5 4. Bc4 g4 5. O-O gxf3 6. Qxf3 Qf6 7. e5 Qxe5 8. d3 Bh6 9. Nc3 Ne7 10. Bd2 Nbc6 11. Rae1

			variation => N__('KGA: Muzio Gambit, Paulsen Variation'),
			moves => {
			},
		},
		'r1b1k2r/ppppnp1p/2n4b/4q3/2B2p2/2NP1Q2/PPPB2PP/R4RK1 w kq - 5 11' => {
			eco => 'C37',
                        # TRANSLATORS:

			variation => N__('KGA: Muzio Gambit'),
			moves => {
				'a1e1' => 'r1b1k2r/ppppnp1p/2n4b/4q3/2B2p2/2NP1Q2/PPPB2PP/4RRK1 b kq - 6 11',
			},
		},
		'r1b1k2r/ppppnppp/2n2q2/1Bb5/3NP3/2P1B3/PP3PPP/RN1QK2R b KQkq - 2 7' => {
			eco => 'C45w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Bb5

			variation => N__('Scotch: Paulsen Attack'),
			moves => {
				'c6d8' => 'r1bnk2r/ppppnppp/5q2/1Bb5/3NP3/2P1B3/PP3PPP/RN1QK2R w KQkq - 3 8',
			},
		},
		'r1b1k2r/ppppnppp/2n2q2/2b5/2BNP3/2P1B3/PP3PPP/RN1QK2R b KQkq - 2 7' => {
			eco => 'C45x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Bc4

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Bc4'),
			moves => {
				'c6e5' => 'r1b1k2r/ppppnppp/5q2/2b1n3/2BNP3/2P1B3/PP3PPP/RN1QK2R w KQkq - 3 8',
			},
		},
		'r1b1k2r/ppppnppp/2n2q2/2b5/3NP3/2P1B3/PP1Q1PPP/RN2KB1R b KQkq - 2 7' => {
			eco => 'C45w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Qd2

			variation => N__('Scotch: Blackburne Attack'),
			moves => {
			},
		},
		'r1b1k2r/ppppnppp/2n2q2/2b5/3NP3/2P1B3/PP3PPP/RN1QKB1R w KQkq - 1 7' => {
			eco => 'C45w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7'),
			moves => {
				'd1d2' => 'r1b1k2r/ppppnppp/2n2q2/2b5/3NP3/2P1B3/PP1Q1PPP/RN2KB1R b KQkq - 2 7',
				'd4c2' => 'r1b1k2r/ppppnppp/2n2q2/2b5/4P3/2P1B3/PPN2PPP/RN1QKB1R b KQkq - 2 7',
				'f1b5' => 'r1b1k2r/ppppnppp/2n2q2/1Bb5/3NP3/2P1B3/PP3PPP/RN1QK2R b KQkq - 2 7',
				'f1c4' => 'r1b1k2r/ppppnppp/2n2q2/2b5/2BNP3/2P1B3/PP3PPP/RN1QK2R b KQkq - 2 7',
			},
		},
		'r1b1k2r/ppppnppp/2n2q2/2b5/4P3/2P1B3/PPN2PPP/RN1QKB1R b KQkq - 2 7' => {
			eco => 'C45w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Nc2

			variation => N__('Scotch: Meitner Variation'),
			moves => {
			},
		},
		'r1b1k2r/ppppnppp/2n3q1/b3P3/2B5/1QN2N2/P4PPP/R1B2RK1 w kq - 1 11' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'c1a3' => 'r1b1k2r/ppppnppp/2n3q1/b3P3/2B5/BQN2N2/P4PPP/R4RK1 b kq - 2 11',
				'f1d1' => 'r1b1k2r/ppppnppp/2n3q1/b3P3/2B5/1QN2N2/P4PPP/R1BR2K1 b kq - 2 11',
			},
		},
		'r1b1k2r/ppppnppp/2n3q1/b3P3/2B5/1QN2N2/P4PPP/R1BR2K1 b kq - 2 11' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 exd4 7. O-O dxc3 8. Qb3 Qf6 9. e5 Qg6 10. Nxc3 Nge7 11. Rd1

			variation => N__('Evans Gambit: Compromised Defence, Potter Variation'),
			moves => {
			},
		},
		'r1b1k2r/ppppnppp/2n3q1/b3P3/2B5/BQN2N2/P4PPP/R4RK1 b kq - 2 11' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 exd4 7. O-O dxc3 8. Qb3 Qf6 9. e5 Qg6 10. Nxc3 Nge7 11. Ba3

			variation => N__('Evans Gambit: Compromised Defence, Paulsen Variation'),
			moves => {
			},
		},
		'r1b1k2r/ppppnppp/5q2/2b1n3/2BNP3/2P1B3/PP3PPP/RN1QK2R w KQkq - 3 8' => {
			eco => 'C45y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Bc4 Ne5

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Bc4 Ne5'),
			moves => {
			},
		},
		'r1b1k2r/ppppqNpp/2n2n2/4p3/2B1P3/8/PPPP1bPP/RNBQ1K1R w kq - 2 7' => {
			eco => 'C57c',
                        # TRANSLATORS:

			variation => N__('Two Knights: Traxler, 5.Nxf7 & 6. Kf1'),
			moves => {
				'f7h8' => 'r1b1k2N/ppppq1pp/2n2n2/4p3/2B1P3/8/PPPP1bPP/RNBQ1K1R b q - 0 7',
			},
		},
		'r1b1k2r/ppppqppp/1bn2n2/4p3/2BPP3/2P2N2/PP3PPP/RNBQ1RK1 w kq - 3 7' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'a2a4' => 'r1b1k2r/ppppqppp/1bn2n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQ1RK1 b kq a3 0 7',
			},
		},
		'r1b1k2r/ppppqppp/1bn2n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQ1RK1 b kq a3 0 7' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Centre-Holding Variation'),
			moves => {
				'a7a6' => 'r1b1k2r/1pppqppp/pbn2n2/4p3/P1BPP3/2P2N2/1P3PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1b1k2r/ppppqppp/2n1p3/8/2PPn3/2B2NP1/PP2PP1P/2RQKB1R b Kkq - 2 8' => {
			eco => 'E11s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Nc3 Bxc3 7. Bxc3 Ne4 8. Rc1

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Nc3 Bxc3 7.Bxc3 Ne4 8. Rc1'),
			moves => {
			},
		},
		'r1b1k2r/ppppqppp/2n1p3/8/2PPn3/2B2NP1/PP2PP1P/R2QKB1R w KQkq - 1 8' => {
			eco => 'E11s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Nc3 Bxc3 7. Bxc3 Ne4

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Nc3 Bxc3 7.Bxc3 Ne4'),
			moves => {
				'a1c1' => 'r1b1k2r/ppppqppp/2n1p3/8/2PPn3/2B2NP1/PP2PP1P/2RQKB1R b Kkq - 2 8',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/1bPP4/2N2NP1/PP1BPP1P/R2QKB1R b KQkq - 2 6' => {
			eco => 'E11r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Nc3

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Nc3'),
			moves => {
				'b4c3' => 'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/2b2NP1/PP1BPP1P/R2QKB1R w KQkq - 0 7',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/1bPP4/5NP1/PP1BPP1P/RN1QKB1R w KQkq - 1 6' => {
			eco => 'E11r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6'),
			moves => {
				'b1c3' => 'r1b1k2r/ppppqppp/2n1pn2/8/1bPP4/2N2NP1/PP1BPP1P/R2QKB1R b KQkq - 2 6',
				'f1g2' => 'r1b1k2r/ppppqppp/2n1pn2/8/1bPP4/5NP1/PP1BPPBP/RN1QK2R b KQkq - 2 6',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/1bPP4/5NP1/PP1BPPBP/RN1QK2R b KQkq - 2 6' => {
			eco => 'E11t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Bg2'),
			moves => {
				'b4d2' => 'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/5NP1/PP1bPPBP/RN1QK2R w KQkq - 0 7',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/2B2NP1/PP2PP1P/R2QKB1R b KQkq - 0 7' => {
			eco => 'E11s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Nc3 Bxc3 7. Bxc3

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Nc3 Bxc3 7.Bxc3'),
			moves => {
				'f6e4' => 'r1b1k2r/ppppqppp/2n1p3/8/2PPn3/2B2NP1/PP2PP1P/R2QKB1R w KQkq - 1 8',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/2b2NP1/PP1BPP1P/R2QKB1R w KQkq - 0 7' => {
			eco => 'E11s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Nc3 Bxc3

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Nc3 Bxc3'),
			moves => {
				'd2c3' => 'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/2B2NP1/PP2PP1P/R2QKB1R b KQkq - 0 7',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/5NP1/PP1NPPBP/R2QK2R b KQkq - 0 7' => {
			eco => 'E11u',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2 Bxd2+ 7. Nbxd2

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Bg2 Bxd2+ 7.Nbxd2'),
			moves => {
				'd7d6' => 'r1b1k2r/ppp1qppp/2nppn2/8/2PP4/5NP1/PP1NPPBP/R2QK2R w KQkq - 0 8',
			},
		},
		'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/5NP1/PP1bPPBP/RN1QK2R w KQkq - 0 7' => {
			eco => 'E11t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2 Bxd2+

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 6. Bg2 Bxd2+'),
			moves => {
				'b1d2' => 'r1b1k2r/ppppqppp/2n1pn2/8/2PP4/5NP1/PP1NPPBP/R2QK2R b KQkq - 0 7',
			},
		},
		'r1b1k2r/ppppqppp/2n5/4P3/1bP2Bn1/4PN2/PP1N1PPP/R2QKB1R b KQkq - 0 7' => {
			eco => 'A52m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6 5. Nf3 Bb4+ 6. Nbd2 Qe7 7. e3

			variation => N__('Budapest: Rubinstein, Main Line, 6.Nbd2 Qe7 7. e3'),
			moves => {
			},
		},
		'r1b1k2r/ppppqppp/2n5/4P3/1bP2Bn1/5N2/PP1NPPPP/R2QKB1R w KQkq - 7 7' => {
			eco => 'A52l',
                        # TRANSLATORS:

			variation => N__('Budapest: Rubinstein, Main Line, 6.Nbd2'),
			moves => {
				'e2e3' => 'r1b1k2r/ppppqppp/2n5/4P3/1bP2Bn1/4PN2/PP1N1PPP/R2QKB1R b KQkq - 0 7',
			},
		},
		'r1b1k2r/ppq1bppp/2nppn2/8/3NPP2/2N1BQ2/PPP3PP/R3KB1R w KQkq - 5 9' => {
			eco => 'B82',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. f4 Nc6 7. Be3 Be7 8. Qf3 Qc7

			variation => N__('Sicilian: Scheveningen, Tal, 8... Qc7'),
			moves => {
			},
		},
		'r1b1k2r/ppq3pp/2nbpn2/3p2B1/3P4/3B1N2/PP2NPPP/R2Q1RK1 b kq - 5 12' => {
			eco => 'C06r',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 f6 9. exf6 Nxf6 10. O-O Bd6 11. Nf3 Qc7 12. Bg5

			variation => N__('French: Tarrasch, Closed, 8... f6 9.exf6, 11... Qc7 12. Bg5'),
			moves => {
			},
		},
		'r1b1k2r/ppq3pp/2nbpn2/3p4/3P4/3B1N2/PP2NPPP/R1BQ1RK1 w kq - 4 12' => {
			eco => 'C06q',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 f6 9. exf6 Nxf6 10. O-O Bd6 11. Nf3 Qc7

			variation => N__('French: Tarrasch, Closed, 8... f6 9.exf6, 11... Qc7'),
			moves => {
				'c1g5' => 'r1b1k2r/ppq3pp/2nbpn2/3p2B1/3P4/3B1N2/PP2NPPP/R2Q1RK1 b kq - 5 12',
			},
		},
		'r1b1k2r/ppqn1pp1/2pbp2p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R w KQkq - 1 11' => {
			eco => 'B17n',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4 10. Qxe4 Qc7

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3, Main Line 10.Qxe4 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/1p1n1ppp/p2ppn2/q5B1/2BNP3/2N5/PPPQ1PPP/2KR3R b kq - 1 9' => {
			eco => 'B94k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 Nbd7 7. Bc4 Qa5 8. Qd2 e6 9. O-O-O

			variation => N__('Sicilian: Najdorf, 6. Bg5 Nbd7 7. Bc4 Qa5 8. Qd2 e6 9.O-O-O'),
			moves => {
			},
		},
		'r1b1kb1r/1p1n1ppp/p2ppn2/q5B1/2BNP3/2N5/PPPQ1PPP/R3K2R w KQkq - 0 9' => {
			eco => 'B94j',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, 6. Bg5 Nbd7 7. Bc4 Qa5'),
			moves => {
				'e1c1' => 'r1b1kb1r/1p1n1ppp/p2ppn2/q5B1/2BNP3/2N5/PPPQ1PPP/2KR3R b kq - 1 9',
			},
		},
		'r1b1kb1r/1p1n1ppp/pq1ppn2/6B1/4PP2/1NN2Q2/PPP3PP/R3KB1R b KQkq - 4 9' => {
			eco => 'B97g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Nb3 Nbd7 9. Qf3

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 8. Nb3 Nbd7 9. Qf3'),
			moves => {
				'f8e7' => 'r1b1k2r/1p1nbppp/pq1ppn2/6B1/4PP2/1NN2Q2/PPP3PP/R3KB1R w KQkq - 5 10',
			},
		},
		'r1b1kb1r/1p1n1ppp/pq1ppn2/6B1/4PP2/1NN5/PPP3PP/R2QKB1R w KQkq - 3 9' => {
			eco => 'B97e',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 8. Nb3'),
			moves => {
				'd1f3' => 'r1b1kb1r/1p1n1ppp/pq1ppn2/6B1/4PP2/1NN2Q2/PPP3PP/R3KB1R b KQkq - 4 9',
			},
		},
		'r1b1kb1r/1p1npppp/p2p1n2/q5B1/2BNP3/2N5/PPP2PPP/R2QK2R w KQkq - 4 8' => {
			eco => 'B94j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 Nbd7 7. Bc4 Qa5

			variation => N__('Sicilian: Najdorf, 6. Bg5 Nbd7 7. Bc4 Qa5'),
			moves => {
				'd1d2' => 'r1b1kb1r/1p1npppp/p2p1n2/q5B1/2BNP3/2N5/PPPQ1PPP/R3K2R b KQkq - 5 8',
			},
		},
		'r1b1kb1r/1p1npppp/p2p1n2/q5B1/2BNP3/2N5/PPPQ1PPP/R3K2R b KQkq - 5 8' => {
			eco => 'B94j',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, 6. Bg5 Nbd7 7. Bc4 Qa5'),
			moves => {
				'e7e6' => 'r1b1kb1r/1p1n1ppp/p2ppn2/q5B1/2BNP3/2N5/PPPQ1PPP/R3K2R w KQkq - 0 9',
			},
		},
		'r1b1kb1r/1p3p1p/p1nppp2/8/3NPP2/q1N5/P1PQB1PP/1R2K2R w Kkq - 2 12' => {
			eco => 'B97o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. Bxf6 gxf6 11. Be2 Nc6

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 10.Bxf6 gxf6 11. Be2 Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/1p3ppp/p1npPn2/6B1/3NP3/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 11' => {
			eco => 'B97q',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 10. f5 Nc6'),
			moves => {
				'f7e6' => 'r1b1kb1r/1p4pp/p1nppn2/6B1/3NP3/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 12',
			},
		},
		'r1b1kb1r/1p3ppp/p1nppn2/5PB1/3NP3/q1N5/P1PQ2PP/1R2KB1R w Kkq - 1 11' => {
			eco => 'B97q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 10. f5 Nc6'),
			moves => {
				'f5e6' => 'r1b1kb1r/1p3ppp/p1npPn2/6B1/3NP3/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 11',
			},
		},
		'r1b1kb1r/1p3ppp/p1nppn2/6B1/4PP2/1NN5/PqPQ2PP/R3KB1R w KQkq - 2 10' => {
			eco => 'B97j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Nb3 Nc6

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 9. Nb3 Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/1p3ppp/p2qpn2/8/2BN4/2P5/PP3PPP/R1BQ1RK1 b kq - 0 11' => {
			eco => 'C07s',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 Qxd5 5. Ngf3 cxd4 6. Bc4 Qd6 7. O-O Nf6 8. Nb3 Nc6 9. Nbxd4 Nxd4 10. Nxd4 a6 11. c3

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4 a6 11. c3'),
			moves => {
			},
		},
		'r1b1kb1r/1p3ppp/p2qpn2/8/2BN4/8/PPP2PPP/R1BQ1RK1 w kq - 0 11' => {
			eco => 'C07r',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 Qxd5 5. Ngf3 cxd4 6. Bc4 Qd6 7. O-O Nf6 8. Nb3 Nc6 9. Nbxd4 Nxd4 10. Nxd4 a6

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4 a6'),
			moves => {
				'c2c3' => 'r1b1kb1r/1p3ppp/p2qpn2/8/2BN4/2P5/PP3PPP/R1BQ1RK1 b kq - 0 11',
				'f1e1' => 'r1b1kb1r/1p3ppp/p2qpn2/8/2BN4/8/PPP2PPP/R1BQR1K1 b kq - 1 11',
			},
		},
		'r1b1kb1r/1p3ppp/p2qpn2/8/2BN4/8/PPP2PPP/R1BQR1K1 b kq - 1 11' => {
			eco => 'C07t',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 Qxd5 5. Ngf3 cxd4 6. Bc4 Qd6 7. O-O Nf6 8. Nb3 Nc6 9. Nbxd4 Nxd4 10. Nxd4 a6 11. Re1

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4 a6 11. Re1'),
			moves => {
			},
		},
		'r1b1kb1r/1p3ppp/pqnppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R b kq - 1 9' => {
			eco => 'B63f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Qb6 8. Nb3 a6 9. O-O-O

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Qb6 8. Nb3 a6 9.O-O-O'),
			moves => {
				'f8e7' => 'r1b1k2r/1p2bppp/pqnppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w kq - 2 10',
			},
		},
		'r1b1kb1r/1p3ppp/pqnppn2/6B1/4P3/1NN5/PPPQ1PPP/R3KB1R w KQkq - 0 9' => {
			eco => 'B63e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Qb6 8. Nb3 a6

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Qb6 8. Nb3 a6'),
			moves => {
				'e1c1' => 'r1b1kb1r/1p3ppp/pqnppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R b kq - 1 9',
			},
		},
		'r1b1kb1r/1p4pp/p1Nppn2/6B1/4P3/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 12' => {
			eco => 'B97q',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 10. f5 Nc6'),
			moves => {
				'b7c6' => 'r1b1kb1r/6pp/p1pppn2/6B1/4P3/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 13',
			},
		},
		'r1b1kb1r/1p4pp/p1nppn2/6B1/3NP3/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 12' => {
			eco => 'B97q',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, 10. f5 Nc6'),
			moves => {
				'd4c6' => 'r1b1kb1r/1p4pp/p1Nppn2/6B1/4P3/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 12',
			},
		},
		'r1b1kb1r/1pp2ppp/2p5/p3Pn2/8/2N2N1P/PPP2PP1/R1B2RK1 w - a6 0 11' => {
			eco => 'C67x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+ Kxd8 9. Nc3 Ke8 10. h3 a5

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap, 9. Nc3 Ke8 10. h3 a5'),
			moves => {
			},
		},
		'r1b1kb1r/1pp2ppp/p1p5/4N3/3qN3/8/PPPP1PPP/R1BQ1RK1 b kq - 2 8' => {
			eco => 'C48a',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Ranken Variation'),
			moves => {
				'd4e5' => 'r1b1kb1r/1pp2ppp/p1p5/4q3/4N3/8/PPPP1PPP/R1BQ1RK1 w kq - 0 9',
			},
		},
		'r1b1kb1r/1pp2ppp/p1p5/4N3/3qN3/8/PPPP1PPP/R1BQK2R w KQkq - 1 8' => {
			eco => 'C48a',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Ranken Variation'),
			moves => {
				'e1g1' => 'r1b1kb1r/1pp2ppp/p1p5/4N3/3qN3/8/PPPP1PPP/R1BQ1RK1 b kq - 2 8',
			},
		},
		'r1b1kb1r/1pp2ppp/p1p5/4q3/4N3/8/PPPP1PPP/R1BQ1RK1 w kq - 0 9' => {
			eco => 'C48a',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Ranken Variation'),
			moves => {
				'f1e1' => 'r1b1kb1r/1pp2ppp/p1p5/4q3/4N3/8/PPPP1PPP/R1BQR1K1 b kq - 1 9',
			},
		},
		'r1b1kb1r/1pp2ppp/p1p5/4q3/4N3/8/PPPP1PPP/R1BQR1K1 b kq - 1 9' => {
			eco => 'C48a',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Ranken Variation'),
			moves => {
				'c8e6' => 'r3kb1r/1pp2ppp/p1p1b3/4q3/4N3/8/PPPP1PPP/R1BQR1K1 w kq - 2 10',
			},
		},
		'r1b1kb1r/1pq2pp1/p1nppn1p/8/3NP3/2N1B3/PPPQ1PPP/2KR1B1R w kq - 2 10' => {
			eco => 'B66j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Qc7

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/1pq2ppp/p1n1pn2/2p5/P1BP4/4PN2/1P2QPPP/RNB2RK1 w kq - 3 9' => {
			eco => 'D27u',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. O-O a6 7. a4 Nc6 8. Qe2 Qc7

			variation => N__('QGA: Classical, Rubinstein, 8. Qe2 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/1pq2ppp/p1nppn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq - 4 9' => {
			eco => 'B84',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Qc7 8. Be3 Nc6

			variation => N__('Sicilian: Scheveningen, Classical, 7.O-O Qc7 8. Be3 Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/1pq2ppp/p1nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 0 9' => {
			eco => 'B80t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. g3 Nc6 7. Bg2 Qc7 8. O-O a6

			variation => N__('Sicilian: Scheveningen, Fianchetto Variation'),
			moves => {
				'f1e1' => 'r1b1kb1r/1pq2ppp/p1nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQR1K1 b kq - 1 9',
			},
		},
		'r1b1kb1r/1pq2ppp/p1nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQR1K1 b kq - 1 9' => {
			eco => 'B80t',
                        # TRANSLATORS:

			variation => N__('Sicilian: Scheveningen, Fianchetto Variation'),
			moves => {
				'f8e7' => 'r1b1k2r/1pq1bppp/p1nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQR1K1 w kq - 2 10',
			},
		},
		'r1b1kb1r/1pq2ppp/p1nppn2/8/3NP3/2N5/PPP1BPPP/R1BQ1R1K w kq - 4 9' => {
			eco => 'B84',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Qc7 8. Kh1 Nc6

			variation => N__('Sicilian: Scheveningen, Classical, 7.O-O Qc7 8. Kh1 Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/1pq2ppp/p1nppn2/8/3NPP2/2N5/PPP1B1PP/R1BQ1RK1 w kq - 1 9' => {
			eco => 'B84',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Qc7 8. f4 Nc6

			variation => N__('Sicilian: Scheveningen, Classical, 7.O-O Qc7 8. f4 Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/1pq2ppp/p1nppn2/8/P2NP3/2N5/1PP1BPPP/R1BQ1RK1 w kq - 1 9' => {
			eco => 'B84',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Qc7 8. a4 Nc6

			variation => N__('Sicilian: Scheveningen, Classical, 7.O-O Qc7 8. a4 Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/1pqn1ppp/p2p1n2/4p3/P3PP2/2N2N2/1PP3PP/R1BQKB1R w KQkq - 1 9' => {
			eco => 'B93q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 e5 7. Nf3 Nbd7 8. a4 Qc7

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/1pqn1ppp/p2ppn2/6B1/3NPP2/2N2Q2/PPP3PP/2KR1B1R b kq - 4 9' => {
			eco => 'B96j',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, 7. f4 Nbd7 8. Qf3 Qc7'),
			moves => {
				'b7b5' => 'r1b1kb1r/2qn1ppp/p2ppn2/1p4B1/3NPP2/2N2Q2/PPP3PP/2KR1B1R w kq b6 0 10',
			},
		},
		'r1b1kb1r/1pqn1ppp/p2ppn2/6B1/3NPP2/2N2Q2/PPP3PP/R3KB1R w KQkq - 3 9' => {
			eco => 'B96j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Nbd7 8. Qf3 Qc7

			variation => N__('Sicilian: Najdorf, 7. f4 Nbd7 8. Qf3 Qc7'),
			moves => {
				'e1c1' => 'r1b1kb1r/1pqn1ppp/p2ppn2/6B1/3NPP2/2N2Q2/PPP3PP/2KR1B1R b kq - 4 9',
			},
		},
		'r1b1kb1r/1pqn1ppp/p2ppn2/6B1/3NPP2/2N5/PPP1Q1PP/R3KB1R w KQkq - 3 9' => {
			eco => 'B96i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Nbd7 8. Qe2 Qc7

			variation => N__('Sicilian: Najdorf, 7. f4 Nbd7 8. Qe2 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/1pqnpppp/p2p1n2/8/4PP2/2N2N2/PPP3PP/R1BQKB1R w KQkq - 3 8' => {
			eco => 'B93i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 Qc7 7. Nf3 Nbd7

			variation => N__('Sicilian: Najdorf, 6. f4 Qc7 7. Nf3 Nbd7'),
			moves => {
				'f1d3' => 'r1b1kb1r/1pqnpppp/p2p1n2/8/4PP2/2NB1N2/PPP3PP/R1BQK2R b KQkq - 4 8',
			},
		},
		'r1b1kb1r/1pqnpppp/p2p1n2/8/4PP2/2NB1N2/PPP3PP/R1BQK2R b KQkq - 4 8' => {
			eco => 'B93j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 Qc7 7. Nf3 Nbd7 8. Bd3

			variation => N__('Sicilian: Najdorf, 6. f4 Qc7 7. Nf3 Nbd7 8. Bd3'),
			moves => {
			},
		},
		'r1b1kb1r/1pqp1pp1/p1n1pn2/7p/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq h6 0 9' => {
			eco => 'B48p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O h5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O h5'),
			moves => {
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 3 8' => {
			eco => 'B49f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O'),
			moves => {
				'b7b5' => 'r1b1kb1r/2qp1ppp/p1n1pn2/1p6/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq b6 0 9',
				'f8b4' => 'r1b1k2r/1pqp1ppp/p1n1pn2/8/1b1NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq - 4 9',
				'f8e7' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq - 4 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq - 2 8' => {
			eco => 'B49c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6'),
			moves => {
				'a2a3' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/P1N1B3/1PP1BPPP/R2QK2R b KQkq - 0 8',
				'e1g1' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 3 8',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R w KQkq - 2 8' => {
			eco => 'B48g',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Qd2'),
			moves => {
				'f2f3' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R b KQkq - 0 8',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R b KQkq - 0 8' => {
			eco => 'B48h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Qd2 Nf6 8. f3

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Qd2 Nf6 8. f3'),
			moves => {
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 b kq - 3 8' => {
			eco => 'B47m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 Nf6 8. O-O

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 Nf6 8.O-O'),
			moves => {
				'f8c5' => 'r1b1k2r/1pqp1ppp/p1n1pn2/2b5/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 4 9',
				'f8e7' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 4 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQK2R w KQkq - 2 8' => {
			eco => 'B47l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 Nf6

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 Nf6'),
			moves => {
				'e1g1' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 b kq - 3 8',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQ1R1K b kq - 3 8' => {
			eco => 'B47w',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. Kh1

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1'),
			moves => {
				'c6d4' => 'r1b1kb1r/1pqp1ppp/p3pn2/8/3nP3/2N5/PPP1BPPP/R1BQ1R1K w kq - 0 9',
				'f8e7' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQ1R1K w kq - 4 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQ1RK1 w kq - 2 8' => {
			eco => 'B47v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6'),
			moves => {
				'a2a3' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/P1N5/1PP1BPPP/R1BQ1RK1 b kq - 0 8',
				'g1h1' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQ1R1K b kq - 3 8',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2NBB3/PPP2PPP/R2Q1RK1 b kq - 3 8' => {
			eco => 'B48n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O'),
			moves => {
				'b7b5' => 'r1b1kb1r/2qp1ppp/p1n1pn2/1p6/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq b6 0 9',
				'c6d4' => 'r1b1kb1r/1pqp1ppp/p3pn2/8/3nP3/2NBB3/PPP2PPP/R2Q1RK1 w kq - 0 9',
				'c6e5' => 'r1b1kb1r/1pqp1ppp/p3pn2/4n3/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq - 4 9',
				'f8d6' => 'r1b1k2r/1pqp1ppp/p1nbpn2/8/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq - 4 9',
				'h7h5' => 'r1b1kb1r/1pqp1pp1/p1n1pn2/7p/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq h6 0 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2NBB3/PPP2PPP/R2QK2R w KQkq - 2 8' => {
			eco => 'B48m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6'),
			moves => {
				'e1g1' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2NBB3/PPP2PPP/R2Q1RK1 b kq - 3 8',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/P1N1B3/1PP1BPPP/R2QK2R b KQkq - 0 8' => {
			eco => 'B49d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. a3

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8. a3'),
			moves => {
				'b7b5' => 'r1b1kb1r/2qp1ppp/p1n1pn2/1p6/3NP3/P1N1B3/1PP1BPPP/R2QK2R w KQkq b6 0 9',
				'f8e7' => 'r1b1k2r/1pqpbppp/p1n1pn2/8/3NP3/P1N1B3/1PP1BPPP/R2QK2R w KQkq - 1 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/P1N5/1PP1BPPP/R1BQ1RK1 b kq - 0 8' => {
			eco => 'B47v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. a3

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. a3'),
			moves => {
			},
		},
		'r1b1kb1r/1pqp1ppp/p3pn2/4n3/3NP3/2NBB2P/PPP2PP1/R2Q1RK1 b kq - 0 9' => {
			eco => 'B48t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O Ne5 9. h3

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O Ne5 9. h3'),
			moves => {
				'f8c5' => 'r1b1k2r/1pqp1ppp/p3pn2/2b1n3/3NP3/2NBB2P/PPP2PP1/R2Q1RK1 w kq - 1 10',
			},
		},
		'r1b1kb1r/1pqp1ppp/p3pn2/4n3/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq - 4 9' => {
			eco => 'B48s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O Ne5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O Ne5'),
			moves => {
				'h2h3' => 'r1b1kb1r/1pqp1ppp/p3pn2/4n3/3NP3/2NBB2P/PPP2PP1/R2Q1RK1 b kq - 0 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p3pn2/8/3QP3/2N5/PPP1BPPP/R1B2R1K b kq - 0 9' => {
			eco => 'B47y',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Nxd4'),
			moves => {
				'f8c5' => 'r1b1k2r/1pqp1ppp/p3pn2/2b5/3QP3/2N5/PPP1BPPP/R1B2R1K w kq - 1 10',
			},
		},
		'r1b1kb1r/1pqp1ppp/p3pn2/8/3nP3/2N5/PPP1BPPP/R1BQ1R1K w kq - 0 9' => {
			eco => 'B47y',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O Nf6 8. Kh1 Nxd4

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O Nf6 8. Kh1 Nxd4'),
			moves => {
				'd1d4' => 'r1b1kb1r/1pqp1ppp/p3pn2/8/3QP3/2N5/PPP1BPPP/R1B2R1K b kq - 0 9',
			},
		},
		'r1b1kb1r/1pqp1ppp/p3pn2/8/3nP3/2NBB3/PPP2PPP/R2Q1RK1 w kq - 0 9' => {
			eco => 'B48o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O Nxd4

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O Nxd4'),
			moves => {
			},
		},
		'r1b1kb1r/1pqpnppp/p1n1p3/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq - 2 8' => {
			eco => 'B49a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nge7

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nge7'),
			moves => {
			},
		},
		'r1b1kb1r/2q2p1p/p2ppP2/1pn5/3NP3/2N2Q2/PPP4P/2KR1B1R w kq - 1 16' => {
			eco => 'B99x',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5 14. f6'),
			moves => {
				'h1g1' => 'r1b1kb1r/2q2p1p/p2ppP2/1pn5/3NP3/2N2Q2/PPP4P/2KR1BR1 b kq - 2 16',
			},
		},
		'r1b1kb1r/2q2p1p/p2ppP2/1pn5/3NP3/2N2Q2/PPP4P/2KR1BR1 b kq - 2 16' => {
			eco => 'B99y',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Be7 8. Qf3 Qc7 9. O-O-O Nbd7 10. g4 b5 11. Bxf6 Nxf6 12. g5 Nd7 13. f5 Nc5 14. f6 gxf6 15. gxf6 Bf8 16. Rg1

			variation => N__('Sicilian: Najdorf, Modern Main Line 13. f5 Nc5 14. f6: Perenyi\'s 16. Rg1'),
			moves => {
			},
		},
		'r1b1kb1r/2qn1ppp/p1n1p3/1p2P1B1/3N4/2N5/PPP1Q1PP/2KR1B1R w kq - 5 12' => {
			eco => 'B96u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 b5 8. e5 dxe5 9. fxe5 Qc7 10. Qe2 Nfd7 11. O-O-O Nc6

			variation => N__('Sicilian: Najdorf, Polugaevsky, 10. Qe2 Nfd7 11.O-O-O Nc6'),
			moves => {
			},
		},
		'r1b1kb1r/2qn1ppp/p2ppn2/1p2P1B1/3N1P2/2N2Q2/PPP3PP/2KR1B1R b kq - 0 10' => {
			eco => 'B96m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Nbd7 8. Qf3 Qc7 9. O-O-O b5 10. e5

			variation => N__('Sicilian: Najdorf, 7. f4 Nbd7 8. Qf3 Qc7 9.O-O-O b5 10. e5'),
			moves => {
			},
		},
		'r1b1kb1r/2qn1ppp/p2ppn2/1p4B1/3NPP2/2N2Q2/PPP3PP/2KR1B1R w kq b6 0 10' => {
			eco => 'B96k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Nbd7 8. Qf3 Qc7 9. O-O-O b5

			variation => N__('Sicilian: Najdorf, 7. f4 Nbd7 8. Qf3 Qc7 9.O-O-O b5'),
			moves => {
				'e4e5' => 'r1b1kb1r/2qn1ppp/p2ppn2/1p2P1B1/3N1P2/2N2Q2/PPP3PP/2KR1B1R b kq - 0 10',
				'f1d3' => 'r1b1kb1r/2qn1ppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KR3R b kq - 1 10',
			},
		},
		'r1b1kb1r/2qn1ppp/p2ppn2/1p4B1/3NPP2/2NB1Q2/PPP3PP/2KR3R b kq - 1 10' => {
			eco => 'B96l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Nbd7 8. Qf3 Qc7 9. O-O-O b5 10. Bd3

			variation => N__('Sicilian: Najdorf, 7. f4 Nbd7 8. Qf3 Qc7 9.O-O-O b5 10. Bd3'),
			moves => {
			},
		},
		'r1b1kb1r/2qn1ppp/p3pn2/1ppP4/4P3/2NB1N2/PP3PPP/R1BQK2R w KQkq - 1 11' => {
			eco => 'D48n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 a6 9. e4 c5 10. d5 Qc7

			variation => N__('Semi-Slav: Meran, Reynolds, 10... Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/2qn2pp/p3pn2/1p6/2p1P3/2N2N2/PPB2PPP/R1BQ1RK1 b kq - 3 13' => {
			eco => 'D48p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 a6 9. e4 c5 10. d5 c4 11. dxe6 fxe6 12. Bc2 Qc7 13. O-O

			variation => N__('Semi-Slav: Meran, Reynolds, 10... c4 11.dxe6 fxe6 12. Bc2 Qc7 13.O-O'),
			moves => {
			},
		},
		'r1b1kb1r/2qn2pp/p3pn2/1p6/2p1P3/2N2N2/PPB2PPP/R1BQK2R w KQkq - 2 13' => {
			eco => 'D48p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 a6 9. e4 c5 10. d5 c4 11. dxe6 fxe6 12. Bc2 Qc7

			variation => N__('Semi-Slav: Meran, Reynolds, 10... c4 11.dxe6 fxe6 12. Bc2 Qc7'),
			moves => {
				'e1g1' => 'r1b1kb1r/2qn2pp/p3pn2/1p6/2p1P3/2N2N2/PPB2PPP/R1BQ1RK1 b kq - 3 13',
			},
		},
		'r1b1kb1r/2qp1ppp/p1n1pn2/1p6/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w kq b6 0 9' => {
			eco => 'B49f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O b5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O b5'),
			moves => {
			},
		},
		'r1b1kb1r/2qp1ppp/p1n1pn2/1p6/3NP3/2NBB3/PPP2PPP/R2Q1RK1 w kq b6 0 9' => {
			eco => 'B48q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 Nf6 8. O-O b5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 Nf6 8.O-O b5'),
			moves => {
			},
		},
		'r1b1kb1r/2qp1ppp/p1n1pn2/1p6/3NP3/P1N1B3/1PP1BPPP/R2QK2R w KQkq b6 0 9' => {
			eco => 'B49d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. a3 b5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8. a3 b5'),
			moves => {
			},
		},
		'r1b1kb1r/3n1ppp/1q2pP2/1p6/3p4/3B1N2/PP3PPP/R1BQK2R w KQkq - 1 13' => {
			eco => 'D49j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 a6 9. e4 c5 10. e5 cxd4 11. Nxb5 axb5 12. exf6 Qb6

			variation => N__('Semi-Slav: Meran, Old Main Line, Botvinnik Variation'),
			moves => {
			},
		},
		'r1b1kb1r/6pp/p1p1pB2/4p3/8/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 14' => {
			eco => 'B97u',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5'),
			moves => {
				'g7f6' => 'r1b1kb1r/7p/p1p1pp2/4p3/8/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 15',
			},
		},
		'r1b1kb1r/6pp/p1p1pn2/4p1B1/8/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 14' => {
			eco => 'B97u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 dxe5

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5'),
			moves => {
				'g5f6' => 'r1b1kb1r/6pp/p1p1pB2/4p3/8/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 14',
			},
		},
		'r1b1kb1r/6pp/p1ppp3/3nP1B1/8/q1N5/P1PQ2PP/1R2KB1R w Kkq - 1 14' => {
			eco => 'B97t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 Nd5

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 Nd5'),
			moves => {
			},
		},
		'r1b1kb1r/6pp/p1pppn2/4P1B1/8/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 13' => {
			eco => 'B97t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5'),
			moves => {
				'd6e5' => 'r1b1kb1r/6pp/p1p1pn2/4p1B1/8/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 14',
				'f6d5' => 'r1b1kb1r/6pp/p1ppp3/3nP1B1/8/q1N5/P1PQ2PP/1R2KB1R w Kkq - 1 14',
			},
		},
		'r1b1kb1r/6pp/p1pppn2/6B1/4P3/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 13' => {
			eco => 'B97r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line'),
			moves => {
				'e4e5' => 'r1b1kb1r/6pp/p1pppn2/4P1B1/8/q1N5/P1PQ2PP/1R2KB1R b Kkq - 0 13',
				'f1e2' => 'r1b1kb1r/6pp/p1pppn2/6B1/4P3/q1N5/P1PQB1PP/1R2K2R b Kkq - 1 13',
			},
		},
		'r1b1kb1r/6pp/p1pppn2/6B1/4P3/q1N5/P1PQB1PP/1R2K2R b Kkq - 1 13' => {
			eco => 'B97s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. Be2

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, Timman\'s 13. Be2'),
			moves => {
			},
		},
		'r1b1kb1r/7p/p1p1pp2/4p3/4N3/8/q1PQ2PP/1R2KB1R w Kkq - 0 16' => {
			eco => 'B97v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 dxe5 14. Bxf6 gxf6 15. Ne4 Qxa2

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4 Qxa2'),
			moves => {
			},
		},
		'r1b1kb1r/7p/p1p1pp2/4p3/4N3/q7/P1PQ2PP/1R2KB1R b Kkq - 1 15' => {
			eco => 'B97u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. f4 Qb6 8. Qd2 Qxb2 9. Rb1 Qa3 10. f5 Nc6 11. fxe6 fxe6 12. Nxc6 bxc6 13. e5 dxe5 14. Bxf6 gxf6 15. Ne4

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5: 15. Ne4'),
			moves => {
				'a3a2' => 'r1b1kb1r/7p/p1p1pp2/4p3/4N3/8/q1PQ2PP/1R2KB1R w Kkq - 0 16',
				'f8e7' => 'r1b1k2r/4b2p/p1p1pp2/4p3/4N3/q7/P1PQ2PP/1R2KB1R w Kkq - 2 16',
			},
		},
		'r1b1kb1r/7p/p1p1pp2/4p3/8/q1N5/P1PQ2PP/1R2KB1R w Kkq - 0 15' => {
			eco => 'B97u',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, Poisoned Pawn, Main Line, 13. e5 dxe5'),
			moves => {
				'c3e4' => 'r1b1kb1r/7p/p1p1pp2/4p3/4N3/q7/P1PQ2PP/1R2KB1R b Kkq - 1 15',
			},
		},
		'r1b1kb1r/p1p2ppp/8/1p1np3/3n3q/2P3N1/PP1P1PPP/RNBQKB1R b KQkq - 3 9' => {
			eco => 'C57k',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fritz, Gruber Variation (8. Ne4)'),
			moves => {
				'c8b7' => 'r3kb1r/pbp2ppp/8/1p1np3/3n3q/2P3N1/PP1P1PPP/RNBQKB1R w KQkq - 4 10',
				'c8g4' => 'r3kb1r/p1p2ppp/8/1p1np3/3n2bq/2P3N1/PP1P1PPP/RNBQKB1R w KQkq - 4 10',
			},
		},
		'r1b1kb1r/p1p2ppp/8/1p1np3/3nN2q/2P5/PP1P1PPP/RNBQKB1R w KQkq - 2 9' => {
			eco => 'C57k',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fritz, Gruber Variation (8. Ne4)'),
			moves => {
				'e4g3' => 'r1b1kb1r/p1p2ppp/8/1p1np3/3n3q/2P3N1/PP1P1PPP/RNBQKB1R b KQkq - 3 9',
			},
		},
		'r1b1kb1r/p1ppqppp/2p2n2/4P3/8/8/PPP1QPPP/RNB1KB1R b KQkq - 2 7' => {
			eco => 'C45j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Nf6 5. Nxc6 bxc6 6. e5 Qe7 7. Qe2

			variation => N__('Scotch: Mieses, 6... Qe7 7. Qe2'),
			moves => {
				'f6d5' => 'r1b1kb1r/p1ppqppp/2p5/3nP3/8/8/PPP1QPPP/RNB1KB1R w KQkq - 3 8',
			},
		},
		'r1b1kb1r/p1ppqppp/2p2n2/4P3/8/8/PPP2PPP/RNBQKB1R w KQkq - 1 7' => {
			eco => 'C45j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Nf6 5. Nxc6 bxc6 6. e5 Qe7

			variation => N__('Scotch: Mieses, 6... Qe7'),
			moves => {
				'd1e2' => 'r1b1kb1r/p1ppqppp/2p2n2/4P3/8/8/PPP1QPPP/RNB1KB1R b KQkq - 2 7',
			},
		},
		'r1b1kb1r/p1ppqppp/2p5/3nP3/2P5/8/PP2QPPP/RNB1KB1R b KQkq c3 0 8' => {
			eco => 'C45k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Nf6 5. Nxc6 bxc6 6. e5 Qe7 7. Qe2 Nd5 8. c4

			variation => N__('Scotch: Mieses, 8. c4'),
			moves => {
				'c8a6' => 'r3kb1r/p1ppqppp/b1p5/3nP3/2P5/8/PP2QPPP/RNB1KB1R w KQkq - 1 9',
			},
		},
		'r1b1kb1r/p1ppqppp/2p5/3nP3/8/8/PPP1QPPP/RNB1KB1R w KQkq - 3 8' => {
			eco => 'C45j',
                        # TRANSLATORS:

			variation => N__('Scotch: Mieses, 6... Qe7 7. Qe2'),
			moves => {
				'c2c4' => 'r1b1kb1r/p1ppqppp/2p5/3nP3/2P5/8/PP2QPPP/RNB1KB1R b KQkq c3 0 8',
			},
		},
		'r1b1kb1r/p1q2pp1/2p2n1p/n3N3/4p3/8/PPPPBPPP/RNBQK2R w KQkq - 2 11' => {
			eco => 'C59e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Qc7

			variation => N__('Two Knights: Morphy, Göring Variation'),
			moves => {
			},
		},
		'r1b1kb1r/p1q2ppp/2p2n2/n3p1N1/8/3B1Q2/PPPP1PPP/RNB1K2R b KQkq - 3 9' => {
			eco => 'C58f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Qf3 Qc7 9. Bd3

			variation => N__('Two Knights: Morphy, Paoli Variation'),
			moves => {
			},
		},
		'r1b1kb1r/p1q2ppp/2p2n2/nB2p1N1/8/5Q2/PPPP1PPP/RNB1K2R w KQkq - 2 9' => {
			eco => 'C58f',
                        # TRANSLATORS:

			variation => N__('Two Knights: Morphy, Bogoljubow Variation'),
			moves => {
				'b5d3' => 'r1b1kb1r/p1q2ppp/2p2n2/n3p1N1/8/3B1Q2/PPPP1PPP/RNB1K2R b KQkq - 3 9',
			},
		},
		'r1b1kb1r/p1qp1ppp/2p1p3/3nP3/4N3/8/PPP2PPP/R1BQKB1R w KQkq - 3 9' => {
			eco => 'B45n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Nxc6 bxc6 7. e5 Nd5 8. Ne4 Qc7

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Nxc6 bxc6 7. e5 Nd5 8. Ne4 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/p2n1p2/2p1pn2/qp2P1B1/2pP4/2N3P1/PP3P1P/R2QKB1R w KQkq - 1 12' => {
			eco => 'D44l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 dxc4 6. e4 b5 7. e5 h6 8. Bh4 g5 9. Nxg5 hxg5 10. Bxg5 Nbd7 11. g3 Qa5

			variation => N__('Semi-Slav: Botvinnik, Lilienthal, 11... Qa5'),
			moves => {
			},
		},
		'r1b1kb1r/p3pp1p/1qpp1np1/8/2B1P3/2N5/PPP2PPP/R1BQ1RK1 w kq - 0 9' => {
			eco => 'B57h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nxc6 bxc6 8. O-O g6

			variation => N__('Sicilian: Sozin, Benko, 7.Nxc6 bxc6 8.O-O g6'),
			moves => {
			},
		},
		'r1b1kb1r/p3pppp/1qpp1n2/8/2B1P3/2N5/PPP2PPP/R1BQ1RK1 b kq - 1 8' => {
			eco => 'B57g',
                        # TRANSLATORS:

			variation => N__('Sicilian: Sozin, Benko, 7.Nxc6'),
			moves => {
				'g7g6' => 'r1b1kb1r/p3pp1p/1qpp1np1/8/2B1P3/2N5/PPP2PPP/R1BQ1RK1 w kq - 0 9',
			},
		},
		'r1b1kb1r/p3pppp/1qpp1n2/8/2B1P3/2N5/PPP2PPP/R1BQK2R w KQkq - 0 8' => {
			eco => 'B57g',
                        # TRANSLATORS:

			variation => N__('Sicilian: Sozin, Benko, 7.Nxc6'),
			moves => {
				'e1g1' => 'r1b1kb1r/p3pppp/1qpp1n2/8/2B1P3/2N5/PPP2PPP/R1BQ1RK1 b kq - 1 8',
			},
		},
		'r1b1kb1r/p4pp1/2p2n1p/n3N3/3qp3/8/PPPPBPPP/RNBQK2R w KQkq - 2 11' => {
			eco => 'C59f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Qd4

			variation => N__('Two Knights: Morphy, Geller Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1p2/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/R2Q1RK1 b kq - 1 10' => {
			eco => 'D43v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3 dxc4 9. Bxc4 g6 10. O-O

			variation => N__('Semi-Slav: Moscow, Main Line'),
			moves => {
				'f8g7' => 'r1b1k2r/pp1n1pb1/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/R2Q1RK1 w kq - 2 11',
			},
		},
		'r1b1kb1r/pp1n1p2/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/R2QK2R w KQkq - 0 10' => {
			eco => 'D43v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3 dxc4 9. Bxc4 g6

			variation => N__('Semi-Slav: Moscow, Main Line'),
			moves => {
				'e1g1' => 'r1b1kb1r/pp1n1p2/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/R2Q1RK1 b kq - 1 10',
			},
		},
		'r1b1kb1r/pp1n1pp1/2p1pq1p/3p4/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 1 8' => {
			eco => 'D43t',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7

			variation => N__('Semi-Slav: Moscow, 7. e3 Nd7'),
			moves => {
				'f1d3' => 'r1b1kb1r/pp1n1pp1/2p1pq1p/3p4/2PP4/2NBPN2/PP3PPP/R2QK2R b KQkq - 2 8',
			},
		},
		'r1b1kb1r/pp1n1pp1/2p1pq1p/3p4/2PP4/2NBPN2/PP3PPP/R2QK2R b KQkq - 2 8' => {
			eco => 'D43u',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3

			variation => N__('Semi-Slav: Moscow, 7. e3 Nd7 8. Bd3'),
			moves => {
				'd5c4' => 'r1b1kb1r/pp1n1pp1/2p1pq1p/8/2pP4/2NBPN2/PP3PPP/R2QK2R w KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1pp1/2p1pq1p/8/2BP4/2N1PN2/PP3PPP/R2QK2R b KQkq - 0 9' => {
			eco => 'D43v',
                        # TRANSLATORS:

			variation => N__('Semi-Slav: Moscow, 7. e3 Nd7 8. Bd3 dxc4'),
			moves => {
				'g7g6' => 'r1b1kb1r/pp1n1p2/2p1pqpp/8/2BP4/2N1PN2/PP3PPP/R2QK2R w KQkq - 0 10',
			},
		},
		'r1b1kb1r/pp1n1pp1/2p1pq1p/8/2pP4/2NBPN2/PP3PPP/R2QK2R w KQkq - 0 9' => {
			eco => 'D43v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 h6 6. Bxf6 Qxf6 7. e3 Nd7 8. Bd3 dxc4

			variation => N__('Semi-Slav: Moscow, 7. e3 Nd7 8. Bd3 dxc4'),
			moves => {
				'd3c4' => 'r1b1kb1r/pp1n1pp1/2p1pq1p/8/2BP4/2N1PN2/PP3PPP/R2QK2R b KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P1P/2P2N2/PP4P1/R1BQKBNR b KQkq h3 0 8' => {
			eco => 'C05j',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6 8. h4

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. h4'),
			moves => {
				'c5d4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3p1P1P/2P2N2/PP4P1/R1BQKBNR w KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P2/2P2N2/PP2N1PP/R1BQKB1R b KQkq - 4 8' => {
			eco => 'C05k',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6 8. Ne2

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. Ne2'),
			moves => {
				'c5d4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3p1P2/2P2N2/PP2N1PP/R1BQKB1R w KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P2/2P2N2/PP4PP/R1BQKBNR w KQkq - 3 8' => {
			eco => 'C05i',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6'),
			moves => {
				'g1e2' => 'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P2/2P2N2/PP2N1PP/R1BQKB1R b KQkq - 4 8',
				'g2g3' => 'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P2/2P2NP1/PP5P/R1BQKBNR b KQkq - 0 8',
				'h2h4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P1P/2P2N2/PP4P1/R1BQKBNR b KQkq h3 0 8',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P1P2/2P2NP1/PP5P/R1BQKBNR b KQkq - 0 8' => {
			eco => 'C05l',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6 8. g3

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. g3'),
			moves => {
				'c5d4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3p1P2/2P2NP1/PP5P/R1BQKBNR w KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P4/2PB1N2/PP1N1PPP/R1BQ1RK1 b kq - 4 8' => {
			eco => 'C05u',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ngf3 Qb6 8. O-O

			variation => N__('French: Tarrasch, Closed, Nunn-Korchnoi Gambit'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P4/2PB1N2/PP1N1PPP/R1BQK2R w KQkq - 3 8' => {
			eco => 'C05r',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Closed, 7.Ngf3'),
			moves => {
				'e1g1' => 'r1b1kb1r/pp1n1ppp/1qn1p3/2ppP3/3P4/2PB1N2/PP1N1PPP/R1BQ1RK1 b kq - 4 8',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3N1P2/2N1B3/PPP3PP/R2QKB1R w KQkq - 1 9' => {
			eco => 'C11m',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. e5 Nfd7 5. f4 c5 6. Nf3 Nc6 7. Be3 cxd4 8. Nxd4 Qb6

			variation => N__('French: Steinitz, Boleslavsky, 7...cxd4 8.Nxd4 Qb6'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P1P1P/5N2/PP4P1/R1BQKBNR b KQkq - 0 9' => {
			eco => 'C05j',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6 8. h4 cxd4 9. cxd4

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. h4'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P1P2/5N2/PP2N1PP/R1BQKB1R b KQkq - 0 9' => {
			eco => 'C05k',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6 8. Ne2 cxd4 9. cxd4

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. Ne2'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P1P2/5NP1/PP5P/R1BQKBNR b KQkq - 0 9' => {
			eco => 'C05l',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. f4 c5 6. c3 Nc6 7. Ndf3 Qb6 8. g3 cxd4 9. cxd4

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. g3'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P4/3B1N2/PP2NPPP/R1BQK2R b KQkq - 2 9' => {
			eco => 'C06i',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 Qb6 9. Nf3

			variation => N__('French: Tarrasch, Closed, 8... Qb6 9. Nf3'),
			moves => {
				'f7f6' => 'r1b1kb1r/pp1n2pp/1qn1pp2/3pP3/3P4/3B1N2/PP2NPPP/R1BQK2R w KQkq - 0 10',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P4/3B4/PP1NNPPP/R1BQ1RK1 b kq - 2 9' => {
			eco => 'C06h',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 Qb6 9. O-O

			variation => N__('French: Tarrasch, Closed, 8... Qb6 9.O-O'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P4/3B4/PP1NNPPP/R1BQK2R w KQkq - 1 9' => {
			eco => 'C06g',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 Qb6

			variation => N__('French: Tarrasch, Closed, 8... Qb6'),
			moves => {
				'd2f3' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P4/3B1N2/PP2NPPP/R1BQK2R b KQkq - 2 9',
				'e1g1' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P4/3B4/PP1NNPPP/R1BQ1RK1 b kq - 2 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3p1P1P/2P2N2/PP4P1/R1BQKBNR w KQkq - 0 9' => {
			eco => 'C05j',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. h4'),
			moves => {
				'c3d4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P1P1P/5N2/PP4P1/R1BQKBNR b KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3p1P2/2P2N2/PP2N1PP/R1BQKB1R w KQkq - 0 9' => {
			eco => 'C05k',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. Ne2'),
			moves => {
				'c3d4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P1P2/5N2/PP2N1PP/R1BQKB1R b KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3p1P2/2P2NP1/PP5P/R1BQKBNR w KQkq - 0 9' => {
			eco => 'C05l',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Closed, 5. f4 c5 6. c3 Nc6 7.Ndf3 Qb6 8. g3'),
			moves => {
				'c3d4' => 'r1b1kb1r/pp1n1ppp/1qn1p3/3pP3/3P1P2/5NP1/PP5P/R1BQKBNR b KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp1n1ppp/2n1p3/q1ppP3/3P4/2PB1N2/PP3PPP/R1BQK1NR w KQkq - 3 8' => {
			eco => 'C05q',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ndf3 Qa5

			variation => N__('French: Tarrasch, Closed, 7.Ndf3 Qa5'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1p3/q2n2B1/3P4/2N1PN2/PP1Q1PPP/R3KB1R b KQkq - 1 8' => {
			eco => 'D52f',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. cxd5 Nxd5 8. Qd2

			variation => N__('QGD: Cambridge Springs, Yugoslav, 8. Qd2'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1p3/q2n2B1/3P4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 0 8' => {
			eco => 'D52e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. cxd5 Nxd5

			variation => N__('QGD: Cambridge Springs, Yugoslav Variation'),
			moves => {
				'd1d2' => 'r1b1kb1r/pp1n1ppp/2p1p3/q2n2B1/3P4/2N1PN2/PP1Q1PPP/R3KB1R b KQkq - 1 8',
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pB2/q2p4/2PP4/2N1PN2/PP3PPP/R2QKB1R b KQkq - 0 7' => {
			eco => 'D52c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Bxf6

			variation => N__('QGD: Cambridge Springs, Capablanca Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pn2/q2P2B1/3P4/2N1PN2/PP3PPP/R2QKB1R b KQkq - 0 7' => {
			eco => 'D52d',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. cxd5

			variation => N__('QGD: Cambridge Springs, 7.cxd5'),
			moves => {
				'f6d5' => 'r1b1kb1r/pp1n1ppp/2p1p3/q2n2B1/3P4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 0 8',
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pn2/q2p2B1/2PP4/2N1P3/PP1N1PPP/R2QKB1R b KQkq - 3 7' => {
			eco => 'D52g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2

			variation => N__('QGD: Cambridge Springs, 7. Nd2'),
			moves => {
				'd5c4' => 'r1b1kb1r/pp1n1ppp/2p1pn2/q5B1/2pP4/2N1P3/PP1N1PPP/R2QKB1R w KQkq - 0 8',
				'f8b4' => 'r1b1k2r/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PP1N1PPP/R2QKB1R w KQkq - 4 8',
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pn2/q2p2B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 2 7' => {
			eco => 'D52b',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5

			variation => N__('QGD: Cambridge Springs Defence'),
			moves => {
				'c4d5' => 'r1b1kb1r/pp1n1ppp/2p1pn2/q2P2B1/3P4/2N1PN2/PP3PPP/R2QKB1R b KQkq - 0 7',
				'f3d2' => 'r1b1kb1r/pp1n1ppp/2p1pn2/q2p2B1/2PP4/2N1P3/PP1N1PPP/R2QKB1R b KQkq - 3 7',
				'g5f6' => 'r1b1kb1r/pp1n1ppp/2p1pB2/q2p4/2PP4/2N1PN2/PP3PPP/R2QKB1R b KQkq - 0 7',
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pn2/q2p2B1/2PP4/2N2N2/PP2PPPP/2RQKB1R w Kkq - 2 7' => {
			eco => 'D51',
                        # TRANSLATORS:

			variation => N__('QGD: 4. Bg5 Nbd7 5. Nf3 c6'),
			moves => {
				'g5d2' => 'r1b1kb1r/pp1n1ppp/2p1pn2/q2p4/2PP4/2N2N2/PP1BPPPP/2RQKB1R b Kkq - 3 7',
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pn2/q2p4/2PP4/2N2N2/PP1BPPPP/2RQKB1R b Kkq - 3 7' => {
			eco => 'D51',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. Nf3 c6 6. Rc1 Qa5 7. Bd2

			variation => N__('QGD: Rochlin Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n1ppp/2p1pn2/q5B1/2pP4/2N1P3/PP1N1PPP/R2QKB1R w KQkq - 0 8' => {
			eco => 'D52h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 dxc4

			variation => N__('QGD: Cambridge Springs, Rubinstein Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp1n2pp/1qn1pp2/3pP3/3P4/3B1N2/PP2NPPP/R1BQK2R w KQkq - 0 10' => {
			eco => 'C06j',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 Qb6 9. Nf3 f6

			variation => N__('French: Tarrasch, Closed, 8... Qb6 9. Nf3 f6'),
			moves => {
			},
		},
		'r1b1kb1r/pp1p1ppp/1q2pn2/4n3/2P5/1NN3P1/PP2PP1P/R1BQKB1R w KQkq - 3 8' => {
			eco => 'A33q',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Qb6 7. Nb3 Ne5

			variation => N__('English: Symmetrical, Geller, 7. Nb3 Ne5'),
			moves => {
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/2P5/1NN3P1/PP2PP1P/R1BQKB1R b KQkq - 2 7' => {
			eco => 'A33p',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Qb6 7. Nb3

			variation => N__('English: Symmetrical, Geller, 7. Nb3'),
			moves => {
				'c6e5' => 'r1b1kb1r/pp1p1ppp/1q2pn2/4n3/2P5/1NN3P1/PP2PP1P/R1BQKB1R w KQkq - 3 8',
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/2PN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq - 1 7' => {
			eco => 'A33o',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Qb6

			variation => N__('English: Symmetrical, Geller Variation'),
			moves => {
				'd4b3' => 'r1b1kb1r/pp1p1ppp/1qn1pn2/8/2P5/1NN3P1/PP2PP1P/R1BQKB1R b KQkq - 2 7',
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/4P3/1NN5/PPP2PPP/R1BQKB1R w KQkq - 0 7' => {
			eco => 'B33b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6 6. Nb3 e6

			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6 6. Nb3 e6'),
			moves => {
				'f1d3' => 'r1b1kb1r/pp1p1ppp/1qn1pn2/8/4P3/1NNB4/PPP2PPP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/4P3/1NNB4/PPP2PPP/R1BQK2R b KQkq - 1 7' => {
			eco => 'B33c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6 6. Nb3 e6 7. Bd3

			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6 6. Nb3 e6 7. Bd3'),
			moves => {
			},
		},
		'r1b1kb1r/pp1ppppp/1qn2n2/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq - 3 6' => {
			eco => 'B33b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6

			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6'),
			moves => {
				'd4b3' => 'r1b1kb1r/pp1ppppp/1qn2n2/8/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq - 4 6',
			},
		},
		'r1b1kb1r/pp1ppppp/1qn2n2/8/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq - 4 6' => {
			eco => 'B33b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6 6. Nb3

			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6 6. Nb3'),
			moves => {
				'e7e6' => 'r1b1kb1r/pp1p1ppp/1qn1pn2/8/4P3/1NN5/PPP2PPP/R1BQKB1R w KQkq - 0 7',
			},
		},
		'r1b1kb1r/pp2pppp/1qNp1n2/8/2B1P3/2N5/PPP2PPP/R1BQK2R b KQkq - 0 7' => {
			eco => 'B57g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nxc6

			variation => N__('Sicilian: Sozin, Benko, 7.Nxc6'),
			moves => {
				'b7c6' => 'r1b1kb1r/p3pppp/1qpp1n2/8/2B1P3/2N5/PPP2PPP/R1BQK2R w KQkq - 0 8',
			},
		},
		'r1b1kb1r/pp2pppp/1qn2n2/3p2B1/2PP4/2N5/PP3PPP/R2QKBNR w KQkq - 5 7' => {
			eco => 'B13v',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 Qb6

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Spielmann Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/1qnp1n2/1N6/2B1P3/2N5/PPP2PPP/R1BQK2R b KQkq - 6 7' => {
			eco => 'B57f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Ndb5

			variation => N__('Sicilian: Sozin, Benko, 7.Ndb5'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/1qnp1n2/6B1/3NP3/2N5/PPP2PPP/R2QKB1R w KQkq - 5 7' => {
			eco => 'B60d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Qb6

			variation => N__('Sicilian: Richter-Rauzer, 6... Qb6'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/1qnp1n2/8/2B1P3/1NN5/PPP2PPP/R1BQK2R b KQkq - 6 7' => {
			eco => 'B57i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3'),
			moves => {
				'e7e6' => 'r1b1kb1r/pp3ppp/1qnppn2/8/2B1P3/1NN5/PPP2PPP/R1BQK2R w KQkq - 0 8',
			},
		},
		'r1b1kb1r/pp2pppp/1qnp1n2/8/2B1P3/2N5/PPP1NPPP/R1BQK2R b KQkq - 6 7' => {
			eco => 'B57e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nde2

			variation => N__('Sicilian: Sozin, Benko, 7.Nde2'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/1qnp1n2/8/2BNP3/2N5/PPP2PPP/R1BQK2R w KQkq - 5 7' => {
			eco => 'B57d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6

			variation => N__('Sicilian: Sozin, Benko Variation'),
			moves => {
				'd4b3' => 'r1b1kb1r/pp2pppp/1qnp1n2/8/2B1P3/1NN5/PPP2PPP/R1BQK2R b KQkq - 6 7',
				'd4b5' => 'r1b1kb1r/pp2pppp/1qnp1n2/1N6/2B1P3/2N5/PPP2PPP/R1BQK2R b KQkq - 6 7',
				'd4c6' => 'r1b1kb1r/pp2pppp/1qNp1n2/8/2B1P3/2N5/PPP2PPP/R1BQK2R b KQkq - 0 7',
				'd4e2' => 'r1b1kb1r/pp2pppp/1qnp1n2/8/2B1P3/2N5/PPP1NPPP/R1BQK2R b KQkq - 6 7',
			},
		},
		'r1b1kb1r/pp2pppp/1qnp1n2/8/2PNP3/5P2/PP4PP/RNBQKB1R w KQkq - 1 7' => {
			eco => 'B54k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. f3 Nc6 6. c4 Qb6

			variation => N__('Sicilian: Prins (Moscow), 5... Nc6 6. c4 Qb6'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/2n2n2/q2p2B1/2PP4/2N5/PP3PPP/R2QKBNR w KQkq - 5 7' => {
			eco => 'B13w',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 Qa5

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Czerniak Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/2np1n2/q5B1/3NP3/2N5/PPP2PPP/R2QKB1R w KQkq - 5 7' => {
			eco => 'B60c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 Qa5

			variation => N__('Sicilian: Richter-Rauzer, 6... Qa5'),
			moves => {
			},
		},
		'r1b1kb1r/pp3ppp/1qn1p2n/2ppP3/3P4/P1P2N2/1P3PPP/RNBQKB1R w KQkq - 1 7' => {
			eco => 'C02x',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. a3 Nh6

			variation => N__('French: Advance, 5. Nf3 Qb6 6. a3 Nh6'),
			moves => {
			},
		},
		'r1b1kb1r/pp3ppp/1qnppn2/6B1/3NP3/2N5/PPPQ1PPP/R3KB1R w KQkq - 2 8' => {
			eco => 'B63c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Qb6

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Qb6'),
			moves => {
				'd4b3' => 'r1b1kb1r/pp3ppp/1qnppn2/6B1/4P3/1NN5/PPPQ1PPP/R3KB1R b KQkq - 3 8',
			},
		},
		'r1b1kb1r/pp3ppp/1qnppn2/6B1/4P3/1NN5/PPPQ1PPP/R3KB1R b KQkq - 3 8' => {
			eco => 'B63d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Qb6 8. Nb3

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Qb6 8. Nb3'),
			moves => {
				'a7a6' => 'r1b1kb1r/1p3ppp/pqnppn2/6B1/4P3/1NN5/PPPQ1PPP/R3KB1R w KQkq - 0 9',
			},
		},
		'r1b1kb1r/pp3ppp/1qnppn2/8/2B1P3/1NN1B3/PPP2PPP/R2QK2R b KQkq - 1 8' => {
			eco => 'B57k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3 e6 8. Be3

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3 e6 8. Be3'),
			moves => {
			},
		},
		'r1b1kb1r/pp3ppp/1qnppn2/8/2B1P3/1NN5/PPP2PPP/R1BQ1RK1 b kq - 1 8' => {
			eco => 'B57m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3 e6 8. O-O

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3 e6 8.O-O'),
			moves => {
				'f8e7' => 'r1b1k2r/pp2bppp/1qnppn2/8/2B1P3/1NN5/PPP2PPP/R1BQ1RK1 w kq - 2 9',
			},
		},
		'r1b1kb1r/pp3ppp/1qnppn2/8/2B1P3/1NN5/PPP2PPP/R1BQK2R w KQkq - 0 8' => {
			eco => 'B57j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3 e6

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3 e6'),
			moves => {
				'c1e3' => 'r1b1kb1r/pp3ppp/1qnppn2/8/2B1P3/1NN1B3/PPP2PPP/R2QK2R b KQkq - 1 8',
				'c1f4' => 'r1b1kb1r/pp3ppp/1qnppn2/8/2B1PB2/1NN5/PPP2PPP/R2QK2R b KQkq - 1 8',
				'e1g1' => 'r1b1kb1r/pp3ppp/1qnppn2/8/2B1P3/1NN5/PPP2PPP/R1BQ1RK1 b kq - 1 8',
			},
		},
		'r1b1kb1r/pp3ppp/1qnppn2/8/2B1PB2/1NN5/PPP2PPP/R2QK2R b KQkq - 1 8' => {
			eco => 'B57l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bc4 Qb6 7. Nb3 e6 8. Bf4

			variation => N__('Sicilian: Sozin, Benko, 7. Nb3 e6 8. Bf4'),
			moves => {
			},
		},
		'r1b1kb1r/pp3ppp/2nqpn2/8/2BN4/5N2/PPP2PPP/R1BQ1RK1 b kq - 0 9' => {
			eco => 'C07o',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3 cxd4 6. Bc4 Qd6'),
			moves => {
				'c6d4' => 'r1b1kb1r/pp3ppp/3qpn2/8/2Bn4/5N2/PPP2PPP/R1BQ1RK1 w kq - 0 10',
			},
		},
		'r1b1kb1r/pp3ppp/2nqpn2/8/2Bp4/1N3N2/PPP2PPP/R1BQ1RK1 w kq - 6 9' => {
			eco => 'C07o',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3 cxd4 6. Bc4 Qd6'),
			moves => {
				'b3d4' => 'r1b1kb1r/pp3ppp/2nqpn2/8/2BN4/5N2/PPP2PPP/R1BQ1RK1 b kq - 0 9',
			},
		},
		'r1b1kb1r/pp3ppp/3qpn2/8/2BN4/8/PPP2PPP/R1BQ1RK1 b kq - 0 10' => {
			eco => 'C07q',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 Qxd5 5. Ngf3 cxd4 6. Bc4 Qd6 7. O-O Nf6 8. Nb3 Nc6 9. Nbxd4 Nxd4 10. Nxd4

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4'),
			moves => {
				'a7a6' => 'r1b1kb1r/1p3ppp/p2qpn2/8/2BN4/8/PPP2PPP/R1BQ1RK1 w kq - 0 11',
			},
		},
		'r1b1kb1r/pp3ppp/3qpn2/8/2BQ4/5N2/PPP2PPP/R1B2RK1 b kq - 0 10' => {
			eco => 'C07p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 Qxd5 5. Ngf3 cxd4 6. Bc4 Qd6 7. O-O Nf6 8. Nb3 Nc6 9. Nbxd4 Nxd4 10. Qxd4

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Qxd4'),
			moves => {
				'd6d4' => 'r1b1kb1r/pp3ppp/4pn2/8/2Bq4/5N2/PPP2PPP/R1B2RK1 w kq - 0 11',
			},
		},
		'r1b1kb1r/pp3ppp/3qpn2/8/2Bn4/5N2/PPP2PPP/R1BQ1RK1 w kq - 0 10' => {
			eco => 'C07o',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3 cxd4 6. Bc4 Qd6'),
			moves => {
				'd1d4' => 'r1b1kb1r/pp3ppp/3qpn2/8/2BQ4/5N2/PPP2PPP/R1B2RK1 b kq - 0 10',
				'f3d4' => 'r1b1kb1r/pp3ppp/3qpn2/8/2BN4/8/PPP2PPP/R1BQ1RK1 b kq - 0 10',
			},
		},
		'r1b1kb1r/pp3ppp/4pn2/8/2Bq4/5N2/PPP2PPP/R1B2RK1 w kq - 0 11' => {
			eco => 'C07p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 Qxd5 5. Ngf3 cxd4 6. Bc4 Qd6 7. O-O Nf6 8. Nb3 Nc6 9. Nbxd4 Nxd4 10. Qxd4 Qxd4

			variation => N__('French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Qxd4 Qxd4'),
			moves => {
			},
		},
		'r1b1kb1r/ppNpq2p/2nn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR b KQkq - 0 9' => {
			eco => 'C27h',
                        # TRANSLATORS:

			variation => N__('Vienna: 3. Bc4 Nxe4 4. Qh5 Nd6 5. Bb3 Nc6'),
			moves => {
				'e8d8' => 'r1bk1b1r/ppNpq2p/2nn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR w KQ - 1 10',
			},
		},
		'r1b1kb1r/ppp2ppp/2n5/3q4/3pn3/2N2N2/PPP2PPP/R1BQR1K1 b kq - 1 8' => {
			eco => 'C56d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Nxe4 6. Re1 d5 7. Bxd5 Qxd5 8. Nc3

			variation => N__('Two Knights: Classical, 8. Nc3'),
			moves => {
				'd5a5' => 'r1b1kb1r/ppp2ppp/2n5/q7/3pn3/2N2N2/PPP2PPP/R1BQR1K1 w kq - 2 9',
				'd5d8' => 'r1bqkb1r/ppp2ppp/2n5/8/3pn3/2N2N2/PPP2PPP/R1BQR1K1 w kq - 2 9',
				'd5h5' => 'r1b1kb1r/ppp2ppp/2n5/7q/3pn3/2N2N2/PPP2PPP/R1BQR1K1 w kq - 2 9',
			},
		},
		'r1b1kb1r/ppp2ppp/2n5/3q4/3pn3/5N2/PPP2PPP/RNBQR1K1 w kq - 0 8' => {
			eco => 'C56d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Nxe4 6. Re1 d5 7. Bxd5 Qxd5

			variation => N__('Two Knights: Classical, 7.Bxd5 Qxd5'),
			moves => {
				'b1c3' => 'r1b1kb1r/ppp2ppp/2n5/3q4/3pn3/2N2N2/PPP2PPP/R1BQR1K1 b kq - 1 8',
			},
		},
		'r1b1kb1r/ppp2ppp/2n5/7q/3pN3/5N2/PPP2PPP/R1BQR1K1 b kq - 0 9' => {
			eco => 'C56f',
                        # TRANSLATORS:

			variation => N__('Two Knights: Classical, 8. Nc3 Qh5'),
			moves => {
				'c8e6' => 'r3kb1r/ppp2ppp/2n1b3/7q/3pN3/5N2/PPP2PPP/R1BQR1K1 w kq - 1 10',
			},
		},
		'r1b1kb1r/ppp2ppp/2n5/7q/3pn3/2N2N2/PPP2PPP/R1BQR1K1 w kq - 2 9' => {
			eco => 'C56f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Nxe4 6. Re1 d5 7. Bxd5 Qxd5 8. Nc3 Qh5

			variation => N__('Two Knights: Classical, 8. Nc3 Qh5'),
			moves => {
				'c3e4' => 'r1b1kb1r/ppp2ppp/2n5/7q/3pN3/5N2/PPP2PPP/R1BQR1K1 b kq - 0 9',
			},
		},
		'r1b1kb1r/ppp2ppp/2n5/q7/3pN3/5N2/PPP2PPP/R1BQR1K1 b kq - 0 9' => {
			eco => 'C56h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Nxe4 6. Re1 d5 7. Bxd5 Qxd5 8. Nc3 Qa5 9. Nxe4

			variation => N__('Two Knights: Classical, Moller, 9.Nxe4'),
			moves => {
				'c8e6' => 'r3kb1r/ppp2ppp/2n1b3/q7/3pN3/5N2/PPP2PPP/R1BQR1K1 w kq - 1 10',
			},
		},
		'r1b1kb1r/ppp2ppp/2n5/q7/3pn3/2N2N2/PPP2PPP/R1BQR1K1 w kq - 2 9' => {
			eco => 'C56h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d4 exd4 5. O-O Nxe4 6. Re1 d5 7. Bxd5 Qxd5 8. Nc3 Qa5

			variation => N__('Two Knights: Classical, Moller Variation'),
			moves => {
				'c3e4' => 'r1b1kb1r/ppp2ppp/2n5/q7/3pN3/5N2/PPP2PPP/R1BQR1K1 b kq - 0 9',
			},
		},
		'r1b1kb1r/ppp2ppp/2p5/4Pn2/8/2N2N1P/PPP2PP1/R1B2RK1 b - - 0 10' => {
			eco => 'C67w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+ Kxd8 9. Nc3 Ke8 10. h3

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap, 9. Nc3 Ke8 10. h3'),
			moves => {
				'a7a5' => 'r1b1kb1r/1pp2ppp/2p5/p3Pn2/8/2N2N1P/PPP2PP1/R1B2RK1 w - a6 0 11',
			},
		},
		'r1b1kb1r/ppp2ppp/2p5/4Pn2/8/2N2N2/PPP2PPP/R1B2RK1 w - - 2 10' => {
			eco => 'C67u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+ Kxd8 9. Nc3 Ke8

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap, 9. Nc3 Ke8'),
			moves => {
				'f1d1' => 'r1b1kb1r/ppp2ppp/2p5/4Pn2/8/2N2N2/PPP2PPP/R1BR2K1 b - - 3 10',
				'h2h3' => 'r1b1kb1r/ppp2ppp/2p5/4Pn2/8/2N2N1P/PPP2PP1/R1B2RK1 b - - 0 10',
			},
		},
		'r1b1kb1r/ppp2ppp/2p5/4Pn2/8/2N2N2/PPP2PPP/R1BR2K1 b - - 3 10' => {
			eco => 'C67v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+ Kxd8 9. Nc3 Ke8 10. Rd1

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap, 9. Nc3 Ke8 10. Rd1'),
			moves => {
			},
		},
		'r1b1kb1r/pppn1pp1/3ppq1p/8/3PP3/2N2N2/PPP2PPP/R2QKB1R w KQkq - 2 7' => {
			eco => 'A46z',
                        # TRANSLATORS: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7

			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7'),
			moves => {
				'd1d2' => 'r1b1kb1r/pppn1pp1/3ppq1p/8/3PP3/2N2N2/PPPQ1PPP/R3KB1R b KQkq - 3 7',
			},
		},
		'r1b1kb1r/pppn1pp1/3ppq1p/8/3PP3/2N2N2/PPPQ1PPP/R3KB1R b KQkq - 3 7' => {
			eco => 'A46z',
                        # TRANSLATORS: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7 7. Qd2

			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7 7. Qd2'),
			moves => {
			},
		},
		'r1b1kb1r/pppnqppp/3p1n2/6B1/8/3P1N2/PPP1QPPP/RN2KB1R w KQkq - 3 8' => {
			eco => 'C42n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nf6 3. Nxe5 d6 4. Nf3 Nxe4 5. Qe2 Qe7 6. d3 Nf6 7. Bg5 Nbd7

			variation => N__('Russian Game: Milner-Barry Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pppp1ppp/5q2/1B6/8/2P5/PPP2PPP/R1BQK2R b KQkq - 0 8' => {
			eco => 'C48k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Rubinstein, Exchange Variation'),
			moves => {
				'f6e5' => 'r1b1kb1r/pppp1ppp/8/1B2q3/8/2P5/PPP2PPP/R1BQK2R w KQkq - 1 9',
				'f8c5' => 'r1b1k2r/pppp1ppp/5q2/1Bb5/8/2P5/PPP2PPP/R1BQK2R w KQkq - 1 9',
			},
		},
		'r1b1kb1r/pppp1ppp/5q2/1B6/8/2p5/PPPP1PPP/R1BQK2R w KQkq - 0 8' => {
			eco => 'C48k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Rubinstein, Exchange Variation'),
			moves => {
				'd2c3' => 'r1b1kb1r/pppp1ppp/5q2/1B6/8/2P5/PPP2PPP/R1BQK2R b KQkq - 0 8',
			},
		},
		'r1b1kb1r/pppp1ppp/8/1B2q3/8/2P5/PPP2PPP/R1BQK2R w KQkq - 1 9' => {
			eco => 'C48m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Nd4 5. Nxd4 exd4 6. e5 dxc3 7. exf6 Qxf6 8. dxc3 Qe5+

			variation => N__('Four Knights: Rubinstein, Exchange Variation, 8... Qe5+'),
			moves => {
			},
		},
		'r1b1kb1r/ppppq2p/2nn2p1/1N1Qpp2/8/1B6/PPPP1PPP/R1B1K1NR w KQkq - 2 9' => {
			eco => 'C27h',
                        # TRANSLATORS:

			variation => N__('Vienna: 3. Bc4 Nxe4 4. Qh5 Nd6 5. Bb3 Nc6'),
			moves => {
				'b5c7' => 'r1b1kb1r/ppNpq2p/2nn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR b KQkq - 0 9',
			},
		},
		'r1b1kb1r/ppppqppp/5n2/1B2N3/3nP3/2N5/PPPP1PPP/R1BQK2R w KQkq - 1 6' => {
			eco => 'C48h',
                        # TRANSLATORS:

			variation => N__('Four Knights: Rubinstein, 5.Nxe5'),
			moves => {
				'f2f4' => 'r1b1kb1r/ppppqppp/5n2/1B2N3/3nPP2/2N5/PPPP2PP/R1BQK2R b KQkq f3 0 6',
			},
		},
		'r1b1kb1r/ppppqppp/5n2/1B2N3/3nPP2/2N5/PPPP2PP/R1BQK2R b KQkq f3 0 6' => {
			eco => 'C48h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Nd4 5. Nxe5 Qe7 6. f4

			variation => N__('Four Knights: Rubinstein, Bogoljubow Variation'),
			moves => {
			},
		},
		'r1b1kb1r/ppq2ppp/2nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 b kq - 4 8' => {
			eco => 'B80s',
                        # TRANSLATORS:

			variation => N__('Sicilian: Scheveningen, Fianchetto Variation'),
			moves => {
				'a7a6' => 'r1b1kb1r/1pq2ppp/p1nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 0 9',
			},
		},
		'r1b1kb1r/ppq2ppp/2nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQK2R w KQkq - 3 8' => {
			eco => 'B80s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. g3 Nc6 7. Bg2 Qc7

			variation => N__('Sicilian: Scheveningen, Fianchetto Variation'),
			moves => {
				'e1g1' => 'r1b1kb1r/ppq2ppp/2nppn2/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 b kq - 4 8',
			},
		},
		'r1b1kb1r/ppq2ppp/2nppn2/8/3NPP2/2N1B3/PPP3PP/R2QKB1R w KQkq - 3 8' => {
			eco => 'B82',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. f4 Nc6 7. Be3 Qc7

			variation => N__('Sicilian: Scheveningen, 6. f4 Nc6 7. Be3 Qc7'),
			moves => {
			},
		},
		'r1b1kb1r/ppqp1ppp/2n1pn2/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 5 7' => {
			eco => 'B48b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 Nf6

			variation => N__('Sicilian: Taimanov, 6. Be3 Nf6'),
			moves => {
			},
		},
		'r1b1kb1r/ppqp1ppp/2n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQK2R w KQkq - 5 7' => {
			eco => 'B47q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 Nf6

			variation => N__('Sicilian: Taimanov, 6. Be2 Nf6'),
			moves => {
			},
		},
		'r1b1kbnr/1pp2ppp/p1p5/8/3NP3/8/PPP2PPP/RNB1K2R b KQkq - 0 7' => {
			eco => 'C68e',
                        # TRANSLATORS:

			variation => N__('Spanish: Exchange, Lasker Variation'),
			moves => {
				'c8d7' => 'r3kbnr/1ppb1ppp/p1p5/8/3NP3/8/PPP2PPP/RNB1K2R w KQkq - 1 8',
			},
		},
		'r1b1kbnr/1pp2ppp/p1p5/8/3qP3/5N2/PPP2PPP/RNB1K2R w KQkq - 0 7' => {
			eco => 'C68e',
                        # TRANSLATORS:

			variation => N__('Spanish: Exchange, Lasker Variation'),
			moves => {
				'f3d4' => 'r1b1kbnr/1pp2ppp/p1p5/8/3NP3/8/PPP2PPP/RNB1K2R b KQkq - 0 7',
			},
		},
		'r1b1kbnr/1pp2ppp/p1pq4/4p3/4P3/3P1N2/PPP2PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C68r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O Qd6 6. d3

			variation => N__('Spanish: Exchange, Bronstein, 6. d3'),
			moves => {
			},
		},
		'r1b1kbnr/1pp2ppp/p1pq4/4p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 2 6' => {
			eco => 'C68q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O Qd6

			variation => N__('Spanish: Exchange, Bronstein Variation'),
			moves => {
				'b1a3' => 'r1b1kbnr/1pp2ppp/p1pq4/4p3/4P3/N4N2/PPPP1PPP/R1BQ1RK1 b kq - 3 6',
				'd2d3' => 'r1b1kbnr/1pp2ppp/p1pq4/4p3/4P3/3P1N2/PPP2PPP/RNBQ1RK1 b kq - 0 6',
			},
		},
		'r1b1kbnr/1pp2ppp/p1pq4/4p3/4P3/N4N2/PPPP1PPP/R1BQ1RK1 b kq - 3 6' => {
			eco => 'C68s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O Qd6 6. Na3

			variation => N__('Spanish: Exchange, Bronstein, 6. Na3'),
			moves => {
				'b7b5' => 'r1b1kbnr/2p2ppp/p1pq4/1p2p3/4P3/N4N2/PPPP1PPP/R1BQ1RK1 w kq b6 0 7',
				'c8e6' => 'r3kbnr/1pp2ppp/p1pqb3/4p3/4P3/N4N2/PPPP1PPP/R1BQ1RK1 w kq - 4 7',
			},
		},
		'r1b1kbnr/1pp3pp/p4p2/2p5/4P3/1N6/PPP2PPP/RNBR2K1 b kq - 0 9' => {
			eco => 'C69k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O f6 6. d4 exd4 7. Nxd4 c5 8. Nb3 Qxd1 9. Rxd1

			variation => N__('Spanish: Exchange, Gligoric, 8. Nb3 Queenswap'),
			moves => {
				'c8d7' => 'r3kbnr/1ppb2pp/p4p2/2p5/4P3/1N6/PPP2PPP/RNBR2K1 w kq - 1 10',
				'c8g4' => 'r3kbnr/1pp3pp/p4p2/2p5/4P1b1/1N6/PPP2PPP/RNBR2K1 w kq - 1 10',
			},
		},
		'r1b1kbnr/1pp3pp/p4p2/2p5/4P3/1N6/PPP2PPP/RNBq1RK1 w kq - 0 9' => {
			eco => 'C69k',
                        # TRANSLATORS:

			variation => N__('Spanish: Exchange, Gligoric, 8. Nb3'),
			moves => {
				'f1d1' => 'r1b1kbnr/1pp3pp/p4p2/2p5/4P3/1N6/PPP2PPP/RNBR2K1 b kq - 0 9',
			},
		},
		'r1b1kbnr/1pq2ppp/p1npp3/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 b kq - 1 8' => {
			eco => 'B47i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 d6 8. O-O

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 d6 8.O-O'),
			moves => {
				'c8d7' => 'r3kbnr/1pqb1ppp/p1npp3/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 w kq - 2 9',
			},
		},
		'r1b1kbnr/1pq2ppp/p1npp3/8/3NP3/2N3P1/PPP2PBP/R1BQK2R w KQkq - 0 8' => {
			eco => 'B47i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2 d6

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2 d6'),
			moves => {
				'e1g1' => 'r1b1kbnr/1pq2ppp/p1npp3/8/3NP3/2N3P1/PPP2PBP/R1BQ1RK1 b kq - 1 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1N1p3/8/4PP2/2N5/PPP3PP/R1BQKB1R b KQkq - 0 7' => {
			eco => 'B47f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. f4 a6 7. Nxc6

			variation => N__('Sicilian: Taimanov, 6. f4 a6 7.Nxc6'),
			moves => {
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N1B3/PPP1BPPP/R2QK2R b KQkq - 1 7' => {
			eco => 'B49a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2'),
			moves => {
				'b7b5' => 'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq b6 0 8',
				'g8e7' => 'r1b1kb1r/1pqpnppp/p1n1p3/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq - 2 8',
				'g8f6' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq - 2 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 0 7' => {
			eco => 'B48c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6'),
			moves => {
				'a2a3' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/P1N1B3/1PP2PPP/R2QKB1R b KQkq - 0 7',
				'd1d2' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R b KQkq - 1 7',
				'f1d3' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2NBB3/PPP2PPP/R2QK2R b KQkq - 1 7',
				'f1e2' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N1B3/PPP1BPPP/R2QK2R b KQkq - 1 7',
				'f2f4' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NPP2/2N1B3/PPP3PP/R2QKB1R b KQkq f3 0 7',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R b KQkq - 1 7' => {
			eco => 'B48g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Qd2

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Qd2'),
			moves => {
				'g8f6' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R w KQkq - 2 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N3P1/PPP2P1P/R1BQKB1R w KQkq - 0 7' => {
			eco => 'B47h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6

			variation => N__('Sicilian: Taimanov, 6. g3 a6'),
			moves => {
				'f1g2' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N3P1/PPP2PBP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N3P1/PPP2PBP/R1BQK2R b KQkq - 1 7' => {
			eco => 'B47h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3 a6 7. Bg2

			variation => N__('Sicilian: Taimanov, 6. g3 a6 7. Bg2'),
			moves => {
				'd7d6' => 'r1b1kbnr/1pq2ppp/p1npp3/8/3NP3/2N3P1/PPP2PBP/R1BQK2R w KQkq - 0 8',
				'g8f6' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N3P1/PPP2PBP/R1BQK2R w KQkq - 2 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N5/PPP1BPPP/R1BQ1RK1 b kq - 1 7' => {
			eco => 'B47t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O'),
			moves => {
				'b7b5' => 'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2N5/PPP1BPPP/R1BQ1RK1 w kq b6 0 8',
				'g8f6' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQ1RK1 w kq - 2 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N5/PPP1BPPP/R1BQK2R w KQkq - 0 7' => {
			eco => 'B47r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6

			variation => N__('Sicilian: Taimanov, 6. Be2 a6'),
			moves => {
				'e1g1' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N5/PPP1BPPP/R1BQ1RK1 b kq - 1 7',
				'f2f4' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NPP2/2N5/PPP1B1PP/R1BQK2R b KQkq f3 0 7',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2NBB3/PPP2PPP/R2QK2R b KQkq - 1 7' => {
			eco => 'B48i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3'),
			moves => {
				'b7b5' => 'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2NBB3/PPP2PPP/R2QK2R w KQkq b6 0 8',
				'g8f6' => 'r1b1kb1r/1pqp1ppp/p1n1pn2/8/3NP3/2NBB3/PPP2PPP/R2QK2R w KQkq - 2 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/P1N1B3/1PP2PPP/R2QKB1R b KQkq - 0 7' => {
			eco => 'B48d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. a3

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. a3'),
			moves => {
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NPP2/2N1B3/PPP3PP/R2QKB1R b KQkq f3 0 7' => {
			eco => 'B48e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. f4

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. f4'),
			moves => {
				'b7b5' => 'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NPP2/2N1B3/PPP3PP/R2QKB1R w KQkq b6 0 8',
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NPP2/2N5/PPP1B1PP/R1BQK2R b KQkq f3 0 7' => {
			eco => 'B47s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. f4

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7. f4'),
			moves => {
			},
		},
		'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NPP2/2N5/PPP3PP/R1BQKB1R w KQkq - 0 7' => {
			eco => 'B47e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. f4 a6

			variation => N__('Sicilian: Taimanov, 6. f4 a6'),
			moves => {
				'd4c6' => 'r1b1kbnr/1pqp1ppp/p1N1p3/8/4PP2/2N5/PPP3PP/R1BQKB1R b KQkq - 0 7',
			},
		},
		'r1b1kbnr/2p2ppp/p1pq4/1p2p3/4P3/N4N2/PPPP1PPP/R1BQ1RK1 w kq b6 0 7' => {
			eco => 'C68t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O Qd6 6. Na3 b5

			variation => N__('Spanish: Exchange, Bronstein, 6. Na3 b5'),
			moves => {
			},
		},
		'r1b1kbnr/2qp1ppp/p1N1p3/1p6/4P3/2N1B3/PPP1BPPP/R2QK2R b KQkq - 0 8' => {
			eco => 'B49b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 b5 8. Nxc6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 b5 8.Nxc6'),
			moves => {
			},
		},
		'r1b1kbnr/2qp1ppp/p1N1p3/1p6/4P3/2NBB3/PPP2PPP/R2QK2R b KQkq - 0 8' => {
			eco => 'B48k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 b5 8. Nxc6

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 b5 8.Nxc6'),
			moves => {
			},
		},
		'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq b6 0 8' => {
			eco => 'B49b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 b5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 b5'),
			moves => {
				'd4c6' => 'r1b1kbnr/2qp1ppp/p1N1p3/1p6/4P3/2N1B3/PPP1BPPP/R2QK2R b KQkq - 0 8',
			},
		},
		'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2N5/PPP1BPPP/R1BQ1RK1 w kq b6 0 8' => {
			eco => 'B47u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2 a6 7. O-O b5

			variation => N__('Sicilian: Taimanov, 6. Be2 a6 7.O-O b5'),
			moves => {
			},
		},
		'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2NBB3/PPP2PPP/R2Q1RK1 b kq - 1 8' => {
			eco => 'B48l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 b5 8. O-O

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 b5 8.O-O'),
			moves => {
			},
		},
		'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2NBB3/PPP2PPP/R2QK2R w KQkq b6 0 8' => {
			eco => 'B48j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Bd3 b5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Bd3 b5'),
			moves => {
				'd4c6' => 'r1b1kbnr/2qp1ppp/p1N1p3/1p6/4P3/2NBB3/PPP2PPP/R2QK2R b KQkq - 0 8',
				'e1g1' => 'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NP3/2NBB3/PPP2PPP/R2Q1RK1 b kq - 1 8',
			},
		},
		'r1b1kbnr/2qp1ppp/p1n1p3/1p6/3NPP2/2N1B3/PPP3PP/R2QKB1R w KQkq b6 0 8' => {
			eco => 'B48f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. f4 b5

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. f4 b5'),
			moves => {
			},
		},
		'r1b1kbnr/p1pp1ppp/1pn5/8/3PPp1q/2N5/PPP1K1PP/R1BQ1BNR w kq - 0 6' => {
			eco => 'C25n',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. f4 exf4 4. d4 Qh4+ 5. Ke2 b6

			variation => N__('Vienna: Steinitz Gambit, Fraser-Minckwitz Variation'),
			moves => {
			},
		},
		'r1b1kbnr/pp1ppppp/1qn5/1Bp5/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'B30l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 Qb6

			variation => N__('Sicilian: Rossolimo, 3... Qb6'),
			moves => {
			},
		},
		'r1b1kbnr/pp1ppppp/1qn5/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq - 1 5' => {
			eco => 'B32e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Qb6

			variation => N__('Sicilian: Open, 2... Nc6, 4... Qb6'),
			moves => {
			},
		},
		'r1b1kbnr/pp1qpppp/2np4/1B6/3QP3/5N2/PPP2PPP/RNB1K2R w KQkq - 3 6' => {
			eco => 'B53m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Qxd4 Nc6 5. Bb5 Qd7

			variation => N__('Sicilian: Chekhover, Zaitsev Variation'),
			moves => {
			},
		},
		'r1b1kbnr/pp2pppp/2n5/2pq4/3P4/2P2N2/PP3PPP/RNBQKB1R b KQkq - 2 5' => {
			eco => 'B22h',
                        # TRANSLATORS: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3

			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3'),
			moves => {
				'c5d4' => 'r1b1kbnr/pp2pppp/2n5/3q4/3p4/2P2N2/PP3PPP/RNBQKB1R w KQkq - 0 6',
				'c8g4' => 'r3kbnr/pp2pppp/2n5/2pq4/3P2b1/2P2N2/PP3PPP/RNBQKB1R w KQkq - 3 6',
			},
		},
		'r1b1kbnr/pp2pppp/2n5/2pq4/3P4/2P5/PP3PPP/RNBQKBNR w KQkq - 1 5' => {
			eco => 'B22h',
                        # TRANSLATORS: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6

			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6'),
			moves => {
				'g1f3' => 'r1b1kbnr/pp2pppp/2n5/2pq4/3P4/2P2N2/PP3PPP/RNBQKB1R b KQkq - 2 5',
			},
		},
		'r1b1kbnr/pp2pppp/2n5/3q4/3P4/5N2/PP3PPP/RNBQKB1R b KQkq - 0 6' => {
			eco => 'B22h',
                        # TRANSLATORS: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4 6. cxd4

			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4 6.cxd4'),
			moves => {
			},
		},
		'r1b1kbnr/pp2pppp/2n5/3q4/3p4/2P2N2/PP3PPP/RNBQKB1R w KQkq - 0 6' => {
			eco => 'B22h',
                        # TRANSLATORS: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4

			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4'),
			moves => {
				'c3d4' => 'r1b1kbnr/pp2pppp/2n5/3q4/3P4/5N2/PP3PPP/RNBQKB1R b KQkq - 0 6',
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/2P2N2/PP2BPPP/RNBQK2R b KQkq - 4 6' => {
			eco => 'C02t',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. Be2

			variation => N__('French: Advance, 5. Nf3 Qb6 6. Be2'),
			moves => {
				'c5d4' => 'r1b1kbnr/pp3ppp/1qn1p3/3pP3/3p4/2P2N2/PP2BPPP/RNBQK2R w KQkq - 0 7',
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/2P2N2/PP3PPP/RNBQKB1R w KQkq - 3 6' => {
			eco => 'C02r',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6

			variation => N__('French: Advance, 5. Nf3 Qb6'),
			moves => {
				'a2a3' => 'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/P1P2N2/1P3PPP/RNBQKB1R b KQkq - 0 6',
				'f1d3' => 'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/2PB1N2/PP3PPP/RNBQK2R b KQkq - 4 6',
				'f1e2' => 'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/2P2N2/PP2BPPP/RNBQK2R b KQkq - 4 6',
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/2PB1N2/PP3PPP/RNBQK2R b KQkq - 4 6' => {
			eco => 'C02s',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. Bd3

			variation => N__('French: Advance, Milner-Barry Gambit'),
			moves => {
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/2ppP3/3P4/P1P2N2/1P3PPP/RNBQKB1R b KQkq - 0 6' => {
			eco => 'C02v',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. a3

			variation => N__('French: Advance, 5. Nf3 Qb6 6. a3'),
			moves => {
				'c5c4' => 'r1b1kbnr/pp3ppp/1qn1p3/3pP3/2pP4/P1P2N2/1P3PPP/RNBQKB1R w KQkq - 0 7',
				'c8d7' => 'r3kbnr/pp1b1ppp/1qn1p3/2ppP3/3P4/P1P2N2/1P3PPP/RNBQKB1R w KQkq - 1 7',
				'g8h6' => 'r1b1kb1r/pp3ppp/1qn1p2n/2ppP3/3P4/P1P2N2/1P3PPP/RNBQKB1R w KQkq - 1 7',
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/3pP3/2pP4/P1P2N2/1P1N1PPP/R1BQKB1R b KQkq - 1 7' => {
			eco => 'C02z',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. a3 c4 7. Nbd2

			variation => N__('French: Advance, 5. Nf3 Qb6 6. a3 c4 7.Nbd2'),
			moves => {
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/3pP3/2pP4/P1P2N2/1P3PPP/RNBQKB1R w KQkq - 0 7' => {
			eco => 'C02y',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. a3 c4

			variation => N__('French: Advance, 5. Nf3 Qb6 6. a3 c4'),
			moves => {
				'b1d2' => 'r1b1kbnr/pp3ppp/1qn1p3/3pP3/2pP4/P1P2N2/1P1N1PPP/R1BQKB1R b KQkq - 1 7',
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/3pP3/3P4/5N2/PP2BPPP/RNBQK2R b KQkq - 0 7' => {
			eco => 'C02u',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. e5 c5 4. c3 Nc6 5. Nf3 Qb6 6. Be2 cxd4 7. cxd4

			variation => N__('French: Advance, 5. Nf3 Qb6 6. Be2 cxd4 7.cxd4'),
			moves => {
			},
		},
		'r1b1kbnr/pp3ppp/1qn1p3/3pP3/3p4/2P2N2/PP2BPPP/RNBQK2R w KQkq - 0 7' => {
			eco => 'C02t',
                        # TRANSLATORS:

			variation => N__('French: Advance, 5. Nf3 Qb6 6. Be2'),
			moves => {
				'c3d4' => 'r1b1kbnr/pp3ppp/1qn1p3/3pP3/3P4/5N2/PP2BPPP/RNBQK2R b KQkq - 0 7',
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq - 1 4' => {
			eco => 'B00v',
                        # TRANSLATORS: 1. e4 Nc6 2. d4 d5 3. exd5 Qxd5 4. Nc3

			variation => N__('Nimzowitsch Defence: Marshall Gambit'),
			moves => {
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/4P3/PP3PPP/RNBQKBNR b KQkq - 0 4' => {
			eco => 'D07c',
                        # TRANSLATORS:

			variation => N__('QGD: Chigorin, 3.cxd5'),
			moves => {
				'e7e5' => 'r1b1kbnr/ppp2ppp/2n5/3qp3/3P4/4P3/PP3PPP/RNBQKBNR w KQkq e6 0 5',
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq - 1 4' => {
			eco => 'B00v',
                        # TRANSLATORS: 1. e4 Nc6 2. d4 d5 3. exd5 Qxd5 4. Nf3

			variation => N__('Nimzowitsch Defence: 2. d4 d5 3.exd5 Qxd5 4. Nf3'),
			moves => {
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/8/PP2PPPP/RNBQKBNR w KQkq - 0 4' => {
			eco => 'D07c',
                        # TRANSLATORS:

			variation => N__('QGD: Chigorin, 3.cxd5'),
			moves => {
				'e2e3' => 'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/4P3/PP3PPP/RNBQKBNR b KQkq - 0 4',
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/8/PPP2PPP/RNBQKBNR w KQkq - 0 4' => {
			eco => 'B00v',
                        # TRANSLATORS: 1. e4 Nc6 2. d4 d5 3. exd5 Qxd5

			variation => N__('Nimzowitsch Defence: 2. d4 d5 3.exd5 Qxd5'),
			moves => {
				'b1c3' => 'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq - 1 4',
				'g1f3' => 'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq - 1 4',
			},
		},
		'r1b1kbnr/ppp1qppp/2n5/4P3/2Pp4/5N2/PP1NPPPP/R1BQKB1R w KQkq - 4 6' => {
			eco => 'D08',
                        # TRANSLATORS: 1. d4 d5 2. c4 e5 3. dxe5 d4 4. Nf3 Nc6 5. Nbd2 Qe7

			variation => N__('QGD: Albin, Balogh Variation'),
			moves => {
			},
		},
		'r1b1kbnr/ppp2ppp/2n5/3p4/3PPp1q/2N5/PPP1K1PP/R1BQ1BNR w kq d6 0 6' => {
			eco => 'C25o',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. f4 exf4 4. d4 Qh4+ 5. Ke2 d5

			variation => N__('Vienna: Steinitz Gambit, Zukertort Defence'),
			moves => {
			},
		},
		'r1b1kbnr/ppp2ppp/2n5/3qp3/3P4/2N1P3/PP3PPP/R1BQKBNR b KQkq - 1 5' => {
			eco => 'D07c',
                        # TRANSLATORS:

			variation => N__('QGD: Chigorin, 3.cxd5'),
			moves => {
				'f8b4' => 'r1b1k1nr/ppp2ppp/2n5/3qp3/1b1P4/2N1P3/PP3PPP/R1BQKBNR w KQkq - 2 6',
			},
		},
		'r1b1kbnr/ppp2ppp/2n5/3qp3/3P4/4P3/PP3PPP/RNBQKBNR w KQkq e6 0 5' => {
			eco => 'D07c',
                        # TRANSLATORS:

			variation => N__('QGD: Chigorin, 3.cxd5'),
			moves => {
				'b1c3' => 'r1b1kbnr/ppp2ppp/2n5/3qp3/3P4/2N1P3/PP3PPP/R1BQKBNR b KQkq - 1 5',
			},
		},
		'r1b1kbnr/ppp3pp/2N5/1B1q4/4p3/8/PPPP1PPP/R1BQK2R w KQkq - 1 8' => {
			eco => 'C63t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 f5 4. Nc3 fxe4 5. Nxe4 d5 6. Nxe5 dxe4 7. Nxc6 Qd5

			variation => N__('Spanish: Schliemann, 4. Nc3 fxe4 5.Nxe4 d5 6.Nxe5 dxe4 7.Nxc6 Qd5'),
			moves => {
			},
		},
		'r1b1kbnr/ppp3pp/2N5/1B4q1/4p3/8/PPPP1PPP/R1BQK2R w KQkq - 1 8' => {
			eco => 'C63u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 f5 4. Nc3 fxe4 5. Nxe4 d5 6. Nxe5 dxe4 7. Nxc6 Qg5

			variation => N__('Spanish: Schliemann, 4. Nc3 fxe4 5.Nxe4 d5 6.Nxe5 dxe4 7.Nxc6 Qg5'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1p1p/2n5/4N3/2B1Pppq/8/PPPP2PP/RNBQ1K1R w kq - 4 7' => {
			eco => 'C37',
                        # TRANSLATORS: 1. e4 e5 2. f4 exf4 3. Nf3 g5 4. Bc4 g4 5. Ne5 Qh4+ 6. Kf1 Nc6

			variation => N__('KGA: Salvio, Herzfeld Gambit'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1p1p/5q2/4n3/2B1PQ2/2N5/PPPP2PP/R1B2RK1 w kq - 1 9' => {
			eco => 'C25p',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. f4 exf4 4. Nf3 g5 5. Bc4 g4 6. O-O gxf3 7. Qxf3 Ne5 8. Qxf4 Qf6

			variation => N__('Vienna: Hamppe-Muzio, Dubois Variation'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1ppp/2n2q2/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq - 1 5' => {
			eco => 'C45d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qf6

			variation => N__('Scotch: 4.Nxd4 Qf6'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/1N6/4P2q/8/PPP2PPP/RNBQKB1R b KQkq - 2 5' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5

			variation => N__('Scotch: Steinitz, 5. Nb5'),
			moves => {
				'f8b4' => 'r1b1k1nr/pppp1ppp/2n5/1N6/1b2P2q/8/PPP2PPP/RNBQKB1R w KQkq - 3 6',
				'f8c5' => 'r1b1k1nr/pppp1ppp/2n5/1Nb5/4P2q/8/PPP2PPP/RNBQKB1R w KQkq - 3 6',
				'h4e4' => 'r1b1kbnr/pppp1ppp/2n5/1N6/4q3/8/PPP2PPP/RNBQKB1R w KQkq - 0 6',
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/1N6/4q3/8/PPP2PPP/RNBQKB1R w KQkq - 0 6' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Qxe4+

			variation => N__('Scotch: Steinitz, 5. Nb5 Qxe4+'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/2B1Pp1q/8/PPPP2PP/RNBQ1KNR w kq - 4 5' => {
			eco => 'C33',
                        # TRANSLATORS: 1. e4 e5 2. f4 exf4 3. Bc4 Qh4+ 4. Kf1 Nc6

			variation => N__('KGA: Bishop\'s Gambit, Boden Defence'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/3NP2q/2N5/PPP2PPP/R1BQKB1R b KQkq - 2 5' => {
			eco => 'C45f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nc3

			variation => N__('Scotch: Steinitz, 5. Nc3'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/3NP2q/3Q4/PPP2PPP/RNB1KB1R b KQkq - 2 5' => {
			eco => 'C45e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Qd3

			variation => N__('Scotch: Steinitz, 5. Qd3'),
			moves => {
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/3NP2q/8/PPP2PPP/RNBQKB1R w KQkq - 1 5' => {
			eco => 'C45e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4

			variation => N__('Scotch: Steinitz Variation'),
			moves => {
				'b1c3' => 'r1b1kbnr/pppp1ppp/2n5/8/3NP2q/2N5/PPP2PPP/R1BQKB1R b KQkq - 2 5',
				'd1d3' => 'r1b1kbnr/pppp1ppp/2n5/8/3NP2q/3Q4/PPP2PPP/RNB1KB1R b KQkq - 2 5',
				'd4b5' => 'r1b1kbnr/pppp1ppp/2n5/1N6/4P2q/8/PPP2PPP/RNBQKB1R b KQkq - 2 5',
				'd4f3' => 'r1b1kbnr/pppp1ppp/2n5/8/4P2q/5N2/PPP2PPP/RNBQKB1R b KQkq - 2 5',
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/3PPp1q/2N5/PPP1K1PP/R1BQ1BNR b kq - 2 5' => {
			eco => 'C25n',
                        # TRANSLATORS:

			variation => N__('Vienna: Steinitz Gambit'),
			moves => {
				'b7b6' => 'r1b1kbnr/p1pp1ppp/1pn5/8/3PPp1q/2N5/PPP1K1PP/R1BQ1BNR w kq - 0 6',
				'd7d5' => 'r1b1kbnr/ppp2ppp/2n5/3p4/3PPp1q/2N5/PPP1K1PP/R1BQ1BNR w kq d6 0 6',
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/3PPp1q/2N5/PPP3PP/R1BQKBNR w KQkq - 1 5' => {
			eco => 'C25n',
                        # TRANSLATORS:

			variation => N__('Vienna: Steinitz Gambit'),
			moves => {
				'e1e2' => 'r1b1kbnr/pppp1ppp/2n5/8/3PPp1q/2N5/PPP1K1PP/R1BQ1BNR b kq - 2 5',
			},
		},
		'r1b1kbnr/pppp1ppp/2n5/8/4P2q/5N2/PPP2PPP/RNBQKB1R b KQkq - 2 5' => {
			eco => 'C45e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nf3

			variation => N__('Scotch: Steinitz, Fraser Attack'),
			moves => {
			},
		},
		'r1b1kbnr/ppppqppp/2n5/1B2p3/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'C60f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Qe7

			variation => N__('Spanish: Vinogradov Variation'),
			moves => {
			},
		},
		'r1b1kbnr/ppppqppp/2n5/4P3/8/5N2/PPP1PPPP/RNBQKB1R w KQkq - 3 4' => {
			eco => 'A40i',
                        # TRANSLATORS: 1. d4 e5 2. dxe5 Nc6 3. Nf3 Qe7

			variation => N__('Englund Gambit: 2.dxe5 Nc6 3. Nf3 Qe7'),
			moves => {
			},
		},
		'r1b1kbnr/ppqp1ppp/2N1p3/8/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq - 0 6' => {
			eco => 'B47b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Nxc6

			variation => N__('Sicilian: Taimanov, 6.Nxc6'),
			moves => {
			},
		},
		'r1b1kbnr/ppqp1ppp/2n1p3/1N6/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq - 4 6' => {
			eco => 'B47c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Ndb5

			variation => N__('Sicilian: Taimanov, 6.Ndb5'),
			moves => {
			},
		},
		'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 4 6' => {
			eco => 'B48a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3

			variation => N__('Sicilian: Taimanov, 6. Be3'),
			moves => {
				'a7a6' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 0 7',
				'g8f6' => 'r1b1kb1r/ppqp1ppp/2n1pn2/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 5 7',
			},
		},
		'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N3P1/PPP2P1P/R1BQKB1R b KQkq - 0 6' => {
			eco => 'B47g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. g3

			variation => N__('Sicilian: Taimanov, 6. g3'),
			moves => {
				'a7a6' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N3P1/PPP2P1P/R1BQKB1R w KQkq - 0 7',
			},
		},
		'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N5/PPP1BPPP/R1BQK2R b KQkq - 4 6' => {
			eco => 'B47q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be2

			variation => N__('Sicilian: Taimanov, 6. Be2'),
			moves => {
				'a7a6' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NP3/2N5/PPP1BPPP/R1BQK2R w KQkq - 0 7',
				'g8f6' => 'r1b1kb1r/ppqp1ppp/2n1pn2/8/3NP3/2N5/PPP1BPPP/R1BQK2R w KQkq - 5 7',
			},
		},
		'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq - 3 6' => {
			eco => 'B47a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7

			variation => N__('Sicilian: Taimanov, Bastrikov Variation'),
			moves => {
				'c1e3' => 'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 4 6',
				'd4b5' => 'r1b1kbnr/ppqp1ppp/2n1p3/1N6/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq - 4 6',
				'd4c6' => 'r1b1kbnr/ppqp1ppp/2N1p3/8/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq - 0 6',
				'f1e2' => 'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N5/PPP1BPPP/R1BQK2R b KQkq - 4 6',
				'f2f4' => 'r1b1kbnr/ppqp1ppp/2n1p3/8/3NPP2/2N5/PPP3PP/R1BQKB1R b KQkq f3 0 6',
				'g2g3' => 'r1b1kbnr/ppqp1ppp/2n1p3/8/3NP3/2N3P1/PPP2P1P/R1BQKB1R b KQkq - 0 6',
			},
		},
		'r1b1kbnr/ppqp1ppp/2n1p3/8/3NPP2/2N5/PPP3PP/R1BQKB1R b KQkq f3 0 6' => {
			eco => 'B47d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. f4

			variation => N__('Sicilian: Taimanov, 6. f4'),
			moves => {
				'a7a6' => 'r1b1kbnr/1pqp1ppp/p1n1p3/8/3NPP2/2N5/PPP3PP/R1BQKB1R w KQkq - 0 7',
			},
		},
		'r1b1kbnr/ppqppppp/2n5/1N6/4P3/8/PPP2PPP/RNBQKB1R b KQkq - 2 5' => {
			eco => 'B32g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Qc7 5. Nb5

			variation => N__('Sicilian: Flohr, 5. Nb5'),
			moves => {
			},
		},
		'r1b1kbnr/ppqppppp/2n5/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq - 1 5' => {
			eco => 'B32f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Qc7

			variation => N__('Sicilian: Flohr Variation'),
			moves => {
				'd4b5' => 'r1b1kbnr/ppqppppp/2n5/1N6/4P3/8/PPP2PPP/RNBQKB1R b KQkq - 2 5',
			},
		},
		'r1b1qrk1/1pp1b1pp/n2ppn2/p4p2/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 w - - 2 10' => {
			eco => 'A99',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Bb2 Na6

			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Bb2 Na6'),
			moves => {
			},
		},
		'r1b1qrk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2Q1RK1 b - - 5 11' => {
			eco => 'E92w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 a5 8. Bg5 h6 9. Bh4 Na6 10. Nd2 Qe8 11. O-O

			variation => N__('King\'s Indian: Petrosian, Stein, Main Line, 10... Qe8 11.O-O'),
			moves => {
				'c8d7' => 'r3qrk1/1ppb1pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2Q1RK1 w - - 6 12',
				'f6h7' => 'r1b1qrk1/1pp2pbn/n2p2pp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2Q1RK1 w - - 6 12',
			},
		},
		'r1b1qrk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2QK2R w KQ - 4 11' => {
			eco => 'E92v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 a5 8. Bg5 h6 9. Bh4 Na6 10. Nd2 Qe8

			variation => N__('King\'s Indian: Petrosian, Stein, Main Line, 10... Qe8'),
			moves => {
				'e1g1' => 'r1b1qrk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2Q1RK1 b - - 5 11',
			},
		},
		'r1b1qrk1/1pp2pbn/n2p2pp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2Q1RK1 w - - 6 12' => {
			eco => 'E92x',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 a5 8. Bg5 h6 9. Bh4 Na6 10. Nd2 Qe8 11. O-O Nh7

			variation => N__('King\'s Indian: Petrosian, Stein, Main Line, 10... Qe8 11.O-O Nh7'),
			moves => {
			},
		},
		'r1b1qrk1/ppp1b1pp/2nppn2/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A99',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 Nc6

			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 Nc6'),
			moves => {
			},
		},
		'r1b1qrk1/ppp1b1pp/2nppn2/5p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 w - - 4 9' => {
			eco => 'A98',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Qc2 Nc6

			variation => N__('Dutch: Ilyin-Zhenevsky, 8. Qc2 Nc6'),
			moves => {
			},
		},
		'r1b1qrk1/ppp1p1bp/n2p1np1/3P1p2/2P5/2N2NP1/PP2PPBP/1RBQ1RK1 b - - 2 9' => {
			eco => 'A87',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. d5 Na6 9. Rb1

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. d5 Na6 9. Rb1'),
			moves => {
			},
		},
		'r1b1qrk1/ppp1p1bp/n2p1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A87',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. d5 Na6

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. d5 Na6'),
			moves => {
				'a1b1' => 'r1b1qrk1/ppp1p1bp/n2p1np1/3P1p2/2P5/2N2NP1/PP2PPBP/1RBQ1RK1 b - - 2 9',
			},
		},
		'r1b1qrk1/ppp2pb1/n5pp/4p1B1/2P1P1n1/2N2N1P/PP2BPP1/R2Q1RK1 w - - 0 12' => {
			eco => 'E94o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Be3 Ng4 9. Bg5 Qe8 10. dxe5 dxe5 11. h3 h6

			variation => N__('King\'s Indian: Glek, Main Line, 11. h3 h6'),
			moves => {
			},
		},
		'r1b1qrk1/ppp2pbp/n2p1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR w KQ - 1 9' => {
			eco => 'E73u',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6 7. Qd2 e5 8. d5 Qe8

			variation => N__('King\'s Indian: Averbakh, 6... Na6 7. Qd2 e5 8. d5 Qe8'),
			moves => {
			},
		},
		'r1b1qrk1/ppp2pbp/n2p2p1/4P1B1/2P1P1n1/2N2N2/PP2BPPP/R2Q1RK1 b - - 0 10' => {
			eco => 'E94n',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Glek, 8. Be3 Ng4 9. Ng5 Qe8'),
			moves => {
				'd6e5' => 'r1b1qrk1/ppp2pbp/n5p1/4p1B1/2P1P1n1/2N2N2/PP2BPPP/R2Q1RK1 w - - 0 11',
			},
		},
		'r1b1qrk1/ppp2pbp/n2p2p1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 w - - 6 10' => {
			eco => 'E94n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Be3 Ng4 9. Bg5 Qe8

			variation => N__('King\'s Indian: Glek, 8. Be3 Ng4 9. Ng5 Qe8'),
			moves => {
				'd4e5' => 'r1b1qrk1/ppp2pbp/n2p2p1/4P1B1/2P1P1n1/2N2N2/PP2BPPP/R2Q1RK1 b - - 0 10',
			},
		},
		'r1b1qrk1/ppp2pbp/n5p1/4p1B1/2P1P1n1/2N2N1P/PP2BPP1/R2Q1RK1 b - - 0 11' => {
			eco => 'E94o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Be3 Ng4 9. Bg5 Qe8 10. dxe5 dxe5 11. h3

			variation => N__('King\'s Indian: Glek, Main Line, 11. h3'),
			moves => {
				'h7h6' => 'r1b1qrk1/ppp2pb1/n5pp/4p1B1/2P1P1n1/2N2N1P/PP2BPP1/R2Q1RK1 w - - 0 12',
			},
		},
		'r1b1qrk1/ppp2pbp/n5p1/4p1B1/2P1P1n1/2N2N2/PP2BPPP/R2Q1RK1 w - - 0 11' => {
			eco => 'E94o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Be3 Ng4 9. Bg5 Qe8 10. dxe5 dxe5

			variation => N__('King\'s Indian: Glek, Main Line'),
			moves => {
				'h2h3' => 'r1b1qrk1/ppp2pbp/n5p1/4p1B1/2P1P1n1/2N2N1P/PP2BPP1/R2Q1RK1 b - - 0 11',
			},
		},
		'r1b1qrk1/pppnb1pp/3ppn2/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A99',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 Nbd7

			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 Nbd7'),
			moves => {
			},
		},
		'r1b1r1k1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13' => {
			eco => 'C97f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Re8

			variation => N__('Spanish: Closed, Chigorin, 12... Re8'),
			moves => {
			},
		},
		'r1b1r1k1/pp1n1pbp/1qpp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 3 11' => {
			eco => 'E69l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. Re1 Re8

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. Re1 Re8'),
			moves => {
			},
		},
		'r1b1r1k1/pp1n1pbp/1qpp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQR1K1 w - - 1 12' => {
			eco => 'E69n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. Re1 exd4 11. Nxd4 Re8

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. Re1 exd4 11.Nxd4 Re8'),
			moves => {
			},
		},
		'r1b1r1k1/ppq2ppp/2n2n2/2p1P3/8/P1PBPN2/2Q2PPP/R1B2RK1 b - - 0 13' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Bd3 e5 12. Qc2 Re8 13. dxe5

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bd3 e5 12. Qc2 Re8 13.de5'),
			moves => {
			},
		},
		'r1b1r1k1/ppq2ppp/2n2n2/2p1p3/3P4/P1PBPN2/2Q2PPP/R1B2RK1 w - - 2 13' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Bd3 e5 12. Qc2 Re8

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bd3 e5 12. Qc2 Re8'),
			moves => {
				'd4e5' => 'r1b1r1k1/ppq2ppp/2n2n2/2p1P3/8/P1PBPN2/2Q2PPP/R1B2RK1 b - - 0 13',
			},
		},
		'r1b1r1k1/ppq2ppp/2n2n2/2pp4/P2P4/2PBPN2/5PPP/R1BQ1RK1 w - - 1 12' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. cxd5 exd5 11. a4 Re8

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10.cxd5 exd5 11. a4 Re8'),
			moves => {
			},
		},
		'r1b1r1k1/ppqn1pbp/2pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 3 11' => {
			eco => 'E69t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Re8 10. Re1 Qc7

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Re8 10. Re1 Qc7'),
			moves => {
			},
		},
		'r1b2rk1/1p1nqpbp/p2p1np1/2pP4/P3PB2/2N2N2/1P2BPPP/R2Q1RK1 w - - 3 12' => {
			eco => 'A74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Bf4 Qe7

			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Bf4 Qe7'),
			moves => {
			},
		},
		'r1b2rk1/1pq1bppp/p1nppn2/8/3NP3/1BN1B3/PPP1QPPP/2KR3R w - - 0 11' => {
			eco => 'B89',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Be3 Be7 8. Qe2 O-O 9. O-O-O Qc7 10. Bb3 a6

			variation => N__('Sicilian: Velimirovic, 9.O-O-O Qc7 10. Bb3 a6'),
			moves => {
			},
		},
		'r1b2rk1/1pq1bppp/p1nppn2/8/3NPP2/2N1B1Q1/PPP1B1PP/R4RK1 b - - 6 11' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. Qe1 Qc7 11. Qg3

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. Qe1 Qc7 11. Qg3'),
			moves => {
			},
		},
		'r1b2rk1/1pq1bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1R1K w - - 5 11' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. Kh1 Qc7

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. Kh1 Qc7'),
			moves => {
			},
		},
		'r1b2rk1/1pq1bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R3QRK1 w - - 5 11' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. Qe1 Qc7

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. Qe1 Qc7'),
			moves => {
				'e1g3' => 'r1b2rk1/1pq1bppp/p1nppn2/8/3NPP2/2N1B1Q1/PPP1B1PP/R4RK1 b - - 6 11',
			},
		},
		'r1b2rk1/1pq1bppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1R1K b - - 2 11' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. a4 Qc7 11. Kh1

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. a4 Qc7'),
			moves => {
			},
		},
		'r1b2rk1/1pq1bppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1RK1 w - - 1 11' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. a4 Qc7

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. a4 Qc7'),
			moves => {
				'g1h1' => 'r1b2rk1/1pq1bppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1R1K b - - 2 11',
			},
		},
		'r1b2rk1/1pqp1ppp/p1n1pn2/8/Nb1NP3/4B3/PPP1BPPP/R2Q1RK1 w - - 6 10' => {
			eco => 'B49k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Qc7 6. Be3 a6 7. Be2 Nf6 8. O-O Bb4 9. Na4 O-O

			variation => N__('Sicilian: Taimanov, 6. Be3 a6 7. Be2 Nf6 8.O-O Bb4 9. Na4 O-O'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/2np1n2/1p2p3/p2PP3/1N2BN1P/PPB2PP1/R2QR1K1 w - - 0 16' => {
			eco => 'C99m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. Nb3 a5 15. Be3 a4

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nb3 a5 15. Be3 a4'),
			moves => {
				'b3d2' => 'r1b2rk1/2q1bppp/2np1n2/1p2p3/p2PP3/4BN1P/PPBN1PP1/R2QR1K1 b - - 1 16',
			},
		},
		'r1b2rk1/2q1bppp/2np1n2/1p2p3/p2PP3/4BN1P/PPBN1PP1/R2QR1K1 b - - 1 16' => {
			eco => 'C99m',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nb3 a5 15. Be3 a4'),
			moves => {
				'c6b4' => 'r1b2rk1/2q1bppp/3p1n2/1p2p3/pn1PP3/4BN1P/PPBN1PP1/R2QR1K1 w - - 2 17',
				'c8d7' => 'r4rk1/2qbbppp/2np1n2/1p2p3/p2PP3/4BN1P/PPBN1PP1/R2QR1K1 w - - 2 17',
			},
		},
		'r1b2rk1/2q1bppp/2np1n2/pp2p3/3PP3/1N2BN1P/PPB2PP1/R2QR1K1 b - - 1 15' => {
			eco => 'C99l',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nb3 a5'),
			moves => {
				'a5a4' => 'r1b2rk1/2q1bppp/2np1n2/1p2p3/p2PP3/1N2BN1P/PPB2PP1/R2QR1K1 w - - 0 16',
			},
		},
		'r1b2rk1/2q1bppp/2np1n2/pp2p3/3PP3/1N3N1P/PPB2PP1/R1BQR1K1 w - - 0 15' => {
			eco => 'C99l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. Nb3 a5

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nb3 a5'),
			moves => {
				'c1e3' => 'r1b2rk1/2q1bppp/2np1n2/pp2p3/3PP3/1N2BN1P/PPB2PP1/R2QR1K1 b - - 1 15',
			},
		},
		'r1b2rk1/2q1bppp/3p1n2/1p2p3/pn1PP3/4BN1P/PPBN1PP1/R2QR1K1 w - - 2 17' => {
			eco => 'C99n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. Nb3 a5 15. Be3 a4 16. Nbd2 Nb4

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nb3 a5 15. Be3 a4 16.Nbd2 Nb4'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p1n2n2/1pp1p3/4P3/2P2N1P/PPB2PP1/R1BQRNK1 b - - 1 14' => {
			eco => 'C98c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. dxc5 dxc5 14. Nf1

			variation => N__('Spanish: Closed, Chigorin, Rauzer, 14. Nf1'),
			moves => {
				'c8e6' => 'r4rk1/2q1bppp/p1n1bn2/1pp1p3/4P3/2P2N1P/PPB2PP1/R1BQRNK1 w - - 2 15',
			},
		},
		'r1b2rk1/2q1bppp/p1n2n2/1pp1p3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 0 14' => {
			eco => 'C98b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. dxc5 dxc5

			variation => N__('Spanish: Closed, Chigorin, Rauzer Attack'),
			moves => {
				'd2f1' => 'r1b2rk1/2q1bppp/p1n2n2/1pp1p3/4P3/2P2N1P/PPB2PP1/R1BQRNK1 b - - 1 14',
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1p1Pp3/4P3/5N1P/PPBN1PP1/R1BQR1K1 b - - 0 14' => {
			eco => 'C99k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. d5

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. d5'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/1N3N1P/PPB2PP1/R1BQR1K1 b - - 2 14' => {
			eco => 'C99l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. Nb3

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nb3'),
			moves => {
				'a6a5' => 'r1b2rk1/2q1bppp/2np1n2/pp2p3/3PP3/1N3N1P/PPB2PP1/R1BQR1K1 w - - 0 15',
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/5N1P/PPB2PP1/R1BQRNK1 b - - 2 14' => {
			eco => 'C99j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. Nf1

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. Nf1'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 w - - 1 14' => {
			eco => 'C99i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6'),
			moves => {
				'a2a3' => 'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/P4N1P/1PBN1PP1/R1BQR1K1 b - - 0 14',
				'd2b3' => 'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/1N3N1P/PPB2PP1/R1BQR1K1 b - - 2 14',
				'd2f1' => 'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/5N1P/PPB2PP1/R1BQRNK1 b - - 2 14',
				'd4d5' => 'r1b2rk1/2q1bppp/p1np1n2/1p1Pp3/4P3/5N1P/PPBN1PP1/R1BQR1K1 b - - 0 14',
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/P4N1P/1PBN1PP1/R1BQR1K1 b - - 0 14' => {
			eco => 'C99i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4 Nc6 14. a3

			variation => N__('Spanish: Closed, Chigorin, 13... Nc6 14. a3'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1pP1p3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 0 13' => {
			eco => 'C98b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. dxc5

			variation => N__('Spanish: Closed, Chigorin, Rauzer Attack'),
			moves => {
				'd6c5' => 'r1b2rk1/2q1bppp/p1n2n2/1pp1p3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 0 14',
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1pp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13' => {
			eco => 'C98a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6

			variation => N__('Spanish: Closed, Chigorin, 12... Nc6'),
			moves => {
				'd4c5' => 'r1b2rk1/2q1bppp/p1np1n2/1pP1p3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 0 13',
				'd4d5' => 'r1b2rk1/2q1bppp/p1np1n2/1ppPp3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 0 13',
			},
		},
		'r1b2rk1/2q1bppp/p1np1n2/1ppPp3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 0 13' => {
			eco => 'C98e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. d5

			variation => N__('Spanish: Closed, Chigorin, 12... Nc6 13. d5'),
			moves => {
				'c6a5' => 'r1b2rk1/2q1bppp/p2p1n2/nppPp3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 1 14',
				'c6d8' => 'r1bn1rk1/2q1bppp/p2p1n2/1ppPp3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 1 14',
			},
		},
		'r1b2rk1/2q1bppp/p1pp1n2/np2p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12' => {
			eco => 'C96d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c6 11. d4 Qc7

			variation => N__('Spanish: Closed, Chigorin, Rossolimo Variation'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/np1Pp3/2p1P3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 0 13' => {
			eco => 'C97c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. d5 c4

			variation => N__('Spanish: Closed, Chigorin, 11. d4 Qc7 12. d5 c4'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/np2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 b - - 0 13' => {
			eco => 'C99a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4 13. cxd4

			variation => N__('Spanish: Closed, Chigorin, 12...cxd4 13.cxd4'),
			moves => {
				'a5c6' => 'r1b2rk1/2q1bppp/p1np1n2/1p2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 w - - 1 14',
				'c8b7' => 'r4rk1/1bq1bppp/p2p1n2/np2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 w - - 1 14',
				'c8d7' => 'r4rk1/2qbbppp/p2p1n2/np2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 w - - 1 14',
				'f8d8' => 'r1br2k1/2q1bppp/p2p1n2/np2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 w - - 1 14',
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/np2p3/3pP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 0 13' => {
			eco => 'C99a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 cxd4

			variation => N__('Spanish: Closed, Chigorin, 12...cxd4'),
			moves => {
				'c3d4' => 'r1b2rk1/2q1bppp/p2p1n2/np2p3/3PP3/5N1P/PPBN1PP1/R1BQR1K1 b - - 0 13',
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12' => {
			eco => 'C97a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7

			variation => N__('Spanish: Closed, Chigorin, 11. d4 Qc7'),
			moves => {
				'b1d2' => 'r1b2rk1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 2 12',
				'd4d5' => 'r1b2rk1/2q1bppp/p2p1n2/nppPp3/4P3/2P2N1P/PPB2PP1/RNBQR1K1 b - - 0 12',
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 2 12' => {
			eco => 'C97d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2

			variation => N__('Spanish: Closed, Chigorin, 11. d4 Qc7 12.Nbd2'),
			moves => {
				'a5c6' => 'r1b2rk1/2q1bppp/p1np1n2/1pp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13',
				'c5d4' => 'r1b2rk1/2q1bppp/p2p1n2/np2p3/3pP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 0 13',
				'c8b7' => 'r4rk1/1bq1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13',
				'c8d7' => 'r4rk1/2qbbppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13',
				'f8d8' => 'r1br2k1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13',
				'f8e8' => 'r1b1r1k1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 3 13',
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 w - - 2 12' => {
			eco => 'C90i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 Na5 10. Bc2 c5 11. Nbd2 Qc7

			variation => N__('Spanish: Closed, Pilnik, 11.Nbd2 Qc7'),
			moves => {
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/nppPp3/4P3/2P2N1P/PPB2PP1/RNBQR1K1 b - - 0 12' => {
			eco => 'C97b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. d5

			variation => N__('Spanish: Closed, Chigorin, 11. d4 Qc7 12. d5'),
			moves => {
				'c5c4' => 'r1b2rk1/2q1bppp/p2p1n2/np1Pp3/2p1P3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 0 13',
			},
		},
		'r1b2rk1/2q1bppp/p2p1n2/nppPp3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 1 14' => {
			eco => 'C98f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. d5 Na5

			variation => N__('Spanish: Closed, Chigorin, 12... Nc6 13. d5 Na5'),
			moves => {
			},
		},
		'r1b2rk1/5ppp/p1pb4/1p1n4/3P3q/1BP3P1/PP3P1P/RNBQR1K1 b - - 0 14' => {
			eco => 'C89m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3

			variation => N__('Spanish: Marshall, Main Line, 13. Re1 Qh4 14. g3'),
			moves => {
				'h4h3' => 'r1b2rk1/5ppp/p1pb4/1p1n4/3P4/1BP3Pq/PP3P1P/RNBQR1K1 w - - 1 15',
			},
		},
		'r1b2rk1/5ppp/p1pb4/1p1n4/3P3q/1BP5/PP3PPP/RNBQR1K1 w - - 3 14' => {
			eco => 'C89m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4

			variation => N__('Spanish: Marshall, Main Line, 13. Re1 Qh4'),
			moves => {
				'g2g3' => 'r1b2rk1/5ppp/p1pb4/1p1n4/3P3q/1BP3P1/PP3P1P/RNBQR1K1 b - - 0 14',
			},
		},
		'r1b2rk1/5ppp/p1pb4/1p1n4/3P4/1BP1B1Pq/PP3P1P/RN1QR1K1 b - - 2 15' => {
			eco => 'C89p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Be3

			variation => N__('Spanish: Marshall, Main Line, 15. Be3'),
			moves => {
				'c8g4' => 'r4rk1/5ppp/p1pb4/1p1n4/3P2b1/1BP1B1Pq/PP3P1P/RN1QR1K1 w - - 3 16',
			},
		},
		'r1b2rk1/5ppp/p1pb4/1p1n4/3P4/1BP3Pq/PP3P1P/RNBQR1K1 w - - 1 15' => {
			eco => 'C89n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3

			variation => N__('Spanish: Marshall, Main Line, 14. g3 Qh3'),
			moves => {
				'c1e3' => 'r1b2rk1/5ppp/p1pb4/1p1n4/3P4/1BP1B1Pq/PP3P1P/RN1QR1K1 b - - 2 15',
				'e1e4' => 'r1b2rk1/5ppp/p1pb4/1p1n4/3PR3/1BP3Pq/PP3P1P/RNBQ2K1 b - - 2 15',
			},
		},
		'r1b2rk1/5ppp/p1pb4/1p1n4/3PR3/1BP3Pq/PP3P1P/RNBQ2K1 b - - 2 15' => {
			eco => 'C89o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1 Qh4 14. g3 Qh3 15. Re4

			variation => N__('Spanish: Marshall, Main Line, 15. Re4'),
			moves => {
			},
		},
		'r1b2rk1/5ppp/p1pb4/1p1n4/7q/1BPP4/PP3PPP/RNBQR1K1 w - - 3 14' => {
			eco => 'C89j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d3 Bd6 13. Re1 Qh4

			variation => N__('Spanish: Marshall, 12. d3 Bd6 13. Re1 Qh4'),
			moves => {
			},
		},
		'r1b2rk1/pp1n1pbp/1qpp1np1/2P1p3/3PP3/2N2NPP/PP3PB1/R1BQ1RK1 b - - 0 10' => {
			eco => 'E69j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. c5

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. c5'),
			moves => {
			},
		},
		'r1b2rk1/pp1n1pbp/1qpp1np1/3Pp3/2P1P3/2N2NPP/PP3PB1/R1BQ1RK1 b - - 0 10' => {
			eco => 'E69i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. d5

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. d5'),
			moves => {
			},
		},
		'r1b2rk1/pp1n1pbp/1qpp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10' => {
			eco => 'E69i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6'),
			moves => {
				'c4c5' => 'r1b2rk1/pp1n1pbp/1qpp1np1/2P1p3/3PP3/2N2NPP/PP3PB1/R1BQ1RK1 b - - 0 10',
				'd4d5' => 'r1b2rk1/pp1n1pbp/1qpp1np1/3Pp3/2P1P3/2N2NPP/PP3PB1/R1BQ1RK1 b - - 0 10',
				'f1e1' => 'r1b2rk1/pp1n1pbp/1qpp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 b - - 2 10',
			},
		},
		'r1b2rk1/pp1n1pbp/1qpp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 b - - 2 10' => {
			eco => 'E69k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. Re1

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. Re1'),
			moves => {
				'e5d4' => 'r1b2rk1/pp1n1pbp/1qpp1np1/8/2PpP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 0 11',
				'f8e8' => 'r1b1r1k1/pp1n1pbp/1qpp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 3 11',
			},
		},
		'r1b2rk1/pp1n1pbp/1qpp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQR1K1 b - - 0 11' => {
			eco => 'E69m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. Re1 exd4 11. Nxd4

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. Re1 exd4'),
			moves => {
				'f8e8' => 'r1b1r1k1/pp1n1pbp/1qpp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQR1K1 w - - 1 12',
			},
		},
		'r1b2rk1/pp1n1pbp/1qpp1np1/8/2PpP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 0 11' => {
			eco => 'E69m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qb6 10. Re1 exd4

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qb6 10. Re1 exd4'),
			moves => {
				'f3d4' => 'r1b2rk1/pp1n1pbp/1qpp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQR1K1 b - - 0 11',
			},
		},
		'r1b2rk1/pp1n1pbp/2pp1np1/q3p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10' => {
			eco => 'E69g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qa5

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qa5'),
			moves => {
				'f1e1' => 'r1b2rk1/pp1n1pbp/2pp1np1/q3p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 b - - 2 10',
			},
		},
		'r1b2rk1/pp1n1pbp/2pp1np1/q3p3/2PPP3/2N2NPP/PP3PB1/R1BQR1K1 b - - 2 10' => {
			eco => 'E69h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qa5 10. Re1

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qa5 10. Re1'),
			moves => {
				'e5d4' => 'r1b2rk1/pp1n1pbp/2pp1np1/q7/2PpP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 0 11',
			},
		},
		'r1b2rk1/pp1n1pbp/2pp1np1/q7/2PNP3/2N3PP/PP3PB1/R1BQR1K1 b - - 0 11' => {
			eco => 'E69h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qa5 10. Re1 exd4 11. Nxd4

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qa5 10. Re1 exd4'),
			moves => {
			},
		},
		'r1b2rk1/pp1n1pbp/2pp1np1/q7/2PpP3/2N2NPP/PP3PB1/R1BQR1K1 w - - 0 11' => {
			eco => 'E69h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qa5 10. Re1 exd4

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qa5 10. Re1 exd4'),
			moves => {
				'f3d4' => 'r1b2rk1/pp1n1pbp/2pp1np1/q7/2PNP3/2N3PP/PP3PB1/R1BQR1K1 b - - 0 11',
			},
		},
		'r1b2rk1/pp1n1ppp/2n1p3/2qpP3/5P2/2N2N2/PPPQ2PP/R3KB1R w KQ - 0 11' => {
			eco => 'C14o',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. Bg5 Be7 5. e5 Nfd7 6. Bxe7 Qxe7 7. f4 O-O 8. Nf3 c5 9. Qd2 Nc6 10. dxc5 Qxc5

			variation => N__('French: Classical, Steinitz, 9. Qd2 Nc6 10.dxc5 Qxc5'),
			moves => {
			},
		},
		'r1b2rk1/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PPQN1PPP/R3KB1R w KQ - 6 9' => {
			eco => 'D52l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4 8. Qc2 O-O

			variation => N__('QGD: Cambridge Springs, 8. Qc2 O-O'),
			moves => {
				'f1e2' => 'r1b2rk1/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PPQNBPPP/R3K2R b KQ - 7 9',
				'g5h4' => 'r1b2rk1/pp1n1ppp/2p1pn2/q2p4/1bPP3B/2N1P3/PPQN1PPP/R3KB1R b KQ - 7 9',
			},
		},
		'r1b2rk1/pp1n1ppp/2p1pn2/q2p2B1/1bPP4/2N1P3/PPQNBPPP/R3K2R b KQ - 7 9' => {
			eco => 'D52n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4 8. Qc2 O-O 9. Be2

			variation => N__('QGD: Cambridge Springs, 8. Qc2 O-O 9. Be2'),
			moves => {
				'e6e5' => 'r1b2rk1/pp1n1ppp/2p2n2/q2pp1B1/1bPP4/2N1P3/PPQNBPPP/R3K2R w KQ - 0 10',
			},
		},
		'r1b2rk1/pp1n1ppp/2p1pn2/q2p4/1bPP3B/2N1P3/PPQN1PPP/R3KB1R b KQ - 7 9' => {
			eco => 'D52m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4 8. Qc2 O-O 9. Bh4

			variation => N__('QGD: Cambridge Springs, Argentine Variation'),
			moves => {
			},
		},
		'r1b2rk1/pp1n1ppp/2p2n2/q2pp1B1/1bPP4/2N1P3/PPQNBPPP/R3K2R w KQ - 0 10' => {
			eco => 'D52o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Nbd7 5. e3 c6 6. Nf3 Qa5 7. Nd2 Bb4 8. Qc2 O-O 9. Be2 e5

			variation => N__('QGD: Cambridge Springs, 8. Qc2 O-O 9. Be2 e5'),
			moves => {
			},
		},
		'r1b2rk1/pp1nppbp/6p1/q1p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11' => {
			eco => 'D87c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. Be3 Qa5 10. O-O Nd7

			variation => N__('Grünfeld: Classical Exchange, 8... c5 9. Be3 Qa5'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqp1p/6p1/2pp4/3P4/3BPN2/PPQ2PPP/R3K2R w KQ - 0 12' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 Nxd5 9. Bxe7 Qxe7 10. Nxd5 exd5 11. Bd3 g6

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5, Main Line, 11. Bd3 g6'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqpbp/2pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10' => {
			eco => 'E69b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 Qe7

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 Qe7'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2n1p3/2PpP3/5P2/2N2N2/PPPQ2PP/R3KB1R b KQ - 0 10' => {
			eco => 'C14n',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. Bg5 Be7 5. e5 Nfd7 6. Bxe7 Qxe7 7. f4 O-O 8. Nf3 c5 9. Qd2 Nc6 10. dxc5

			variation => N__('French: Classical, Steinitz, 9. Qd2 Nc6 10.dxc5'),
			moves => {
				'e7c5' => 'r1b2rk1/pp1n1ppp/2n1p3/2qpP3/5P2/2N2N2/PPPQ2PP/R3KB1R w KQ - 0 11',
			},
		},
		'r1b2rk1/pp1nqppp/2n1p3/2ppP3/3P1P2/2N2N2/PPPQ2PP/2KR1B1R b - - 3 10' => {
			eco => 'C14m',
                        # TRANSLATORS:

			variation => N__('French: Classical, Steinitz, 9. Qd2'),
			moves => {
				'c5c4' => 'r1b2rk1/pp1nqppp/2n1p3/3pP3/2pP1P2/2N2N2/PPPQ2PP/2KR1B1R w - - 0 11',
			},
		},
		'r1b2rk1/pp1nqppp/2n1p3/2ppP3/3P1P2/2N2N2/PPPQ2PP/R3KB1R w KQ - 2 10' => {
			eco => 'C14m',
                        # TRANSLATORS:

			variation => N__('French: Classical, Steinitz, 9. Qd2'),
			moves => {
				'd4c5' => 'r1b2rk1/pp1nqppp/2n1p3/2PpP3/5P2/2N2N2/PPPQ2PP/R3KB1R b KQ - 0 10',
				'e1c1' => 'r1b2rk1/pp1nqppp/2n1p3/2ppP3/3P1P2/2N2N2/PPPQ2PP/2KR1B1R b - - 3 10',
			},
		},
		'r1b2rk1/pp1nqppp/2n1p3/3pP3/2pP1P2/2N2N2/PPPQ2PP/2KR1B1R w - - 0 11' => {
			eco => 'C14m',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. Bg5 Be7 5. e5 Nfd7 6. Bxe7 Qxe7 7. f4 O-O 8. Nf3 c5 9. Qd2 Nc6 10. O-O-O c4

			variation => N__('French: Classical, Stahlberg Variation'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2p1p3/3n4/2BP4/2N1PN2/PP3PPP/2RQ1RK1 b - - 1 11' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O

			variation => N__('QGD: Orthodox, Main Line, Capablanca, 11.O-O'),
			moves => {
				'd5c3' => 'r1b2rk1/pp1nqppp/2p1p3/8/2BP4/2n1PN2/PP3PPP/2RQ1RK1 w - - 0 12',
			},
		},
		'r1b2rk1/pp1nqppp/2p1p3/3n4/2BP4/2N1PN2/PP3PPP/2RQK2R w K - 0 11' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7

			variation => N__('QGD: Orthodox, Main Line, Capablanca Variation'),
			moves => {
				'c3e4' => 'r1b2rk1/pp1nqppp/2p1p3/3n4/2BPN3/4PN2/PP3PPP/2RQK2R b K - 1 11',
				'e1g1' => 'r1b2rk1/pp1nqppp/2p1p3/3n4/2BP4/2N1PN2/PP3PPP/2RQ1RK1 b - - 1 11',
			},
		},
		'r1b2rk1/pp1nqppp/2p1p3/3n4/2BPN3/4PN2/PP3PPP/2RQK2R b K - 1 11' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. Ne4

			variation => N__('QGD: Orthodox, Main Line, Alekhine Variation'),
			moves => {
				'd5f6' => 'r1b2rk1/pp1nqppp/2p1pn2/8/2BPN3/4PN2/PP3PPP/2RQK2R w K - 2 12',
				'e6e5' => 'r1b2rk1/pp1nqppp/2p5/3np3/2BPN3/4PN2/PP3PPP/2RQK2R w K - 0 12',
			},
		},
		'r1b2rk1/pp1nqppp/2p1p3/8/2BP4/2R1PN2/PP3PPP/3Q1RK1 b - - 0 12' => {
			eco => 'D67',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Main Line, Capablanca, 11.O-O Nxc3'),
			moves => {
				'e6e5' => 'r1b2rk1/pp1nqppp/2p5/4p3/2BP4/2R1PN2/PP3PPP/3Q1RK1 w - - 0 13',
			},
		},
		'r1b2rk1/pp1nqppp/2p1p3/8/2BP4/2n1PN2/PP3PPP/2RQ1RK1 w - - 0 12' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3

			variation => N__('QGD: Orthodox, Main Line, Capablanca, 11.O-O Nxc3'),
			moves => {
				'c1c3' => 'r1b2rk1/pp1nqppp/2p1p3/8/2BP4/2R1PN2/PP3PPP/3Q1RK1 b - - 0 12',
			},
		},
		'r1b2rk1/pp1nqppp/2p1pn2/8/2BPN3/4PN2/PP3PPP/2RQK2R w K - 2 12' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. Ne4 N5f6

			variation => N__('QGD: Orthodox, Main Line, Alekhine, 11...N5f6'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2p5/3np3/2BPN3/4PN2/PP3PPP/2RQK2R w K - 0 12' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. Ne4 e5

			variation => N__('QGD: Orthodox, Main Line, Alekhine, 11... e5 (Lasker)'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2p5/4P3/2B5/2R1PN2/PP3PPP/3Q1RK1 b - - 0 13' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5

			variation => N__('QGD: Orthodox, Classical, 13.dxe5'),
			moves => {
				'd7e5' => 'r1b2rk1/pp2qppp/2p5/4n3/2B5/2R1PN2/PP3PPP/3Q1RK1 w - - 0 14',
			},
		},
		'r1b2rk1/pp1nqppp/2p5/4p3/2BP4/2R1PN2/PP3PPP/1Q3RK1 b - - 1 13' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Qb1

			variation => N__('QGD: Orthodox, Classical, Maroczy Variation'),
			moves => {
				'e5d4' => 'r1b2rk1/pp1nqppp/2p5/8/2Bp4/2R1PN2/PP3PPP/1Q3RK1 w - - 0 14',
			},
		},
		'r1b2rk1/pp1nqppp/2p5/4p3/2BP4/2R1PN2/PP3PPP/3Q1RK1 w - - 0 13' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5

			variation => N__('QGD: Orthodox, Classical Variation'),
			moves => {
				'c4b3' => 'r1b2rk1/pp1nqppp/2p5/4p3/3P4/1BR1PN2/PP3PPP/3Q1RK1 b - - 1 13',
				'd1b1' => 'r1b2rk1/pp1nqppp/2p5/4p3/2BP4/2R1PN2/PP3PPP/1Q3RK1 b - - 1 13',
				'd1c2' => 'r1b2rk1/pp1nqppp/2p5/4p3/2BP4/2R1PN2/PPQ2PPP/5RK1 b - - 1 13',
				'd4e5' => 'r1b2rk1/pp1nqppp/2p5/4P3/2B5/2R1PN2/PP3PPP/3Q1RK1 b - - 0 13',
			},
		},
		'r1b2rk1/pp1nqppp/2p5/4p3/2BP4/2R1PN2/PPQ2PPP/5RK1 b - - 1 13' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Qc2

			variation => N__('QGD: Orthodox, Classical, Vidmar Variation'),
			moves => {
				'e5d4' => 'r1b2rk1/pp1nqppp/2p5/8/2Bp4/2R1PN2/PPQ2PPP/5RK1 w - - 0 14',
				'e5e4' => 'r1b2rk1/pp1nqppp/2p5/8/2BPp3/2R1PN2/PPQ2PPP/5RK1 w - - 0 14',
			},
		},
		'r1b2rk1/pp1nqppp/2p5/4p3/3P4/1BR1PN2/PP3PPP/3Q1RK1 b - - 1 13' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Bb3

			variation => N__('QGD: Orthodox, Classical, 13. Bb3'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2p5/8/2BP4/2R2N2/PPQ2PPP/5RK1 b - - 0 14' => {
			eco => 'D68',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Classical, Vidmar, 13...exd4'),
			moves => {
				'd7b6' => 'r1b2rk1/pp2qppp/1np5/8/2BP4/2R2N2/PPQ2PPP/5RK1 w - - 1 15',
			},
		},
		'r1b2rk1/pp1nqppp/2p5/8/2BPp3/2R1PN2/PPQ2PPP/5RK1 w - - 0 14' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Qc2 e4

			variation => N__('QGD: Orthodox, Classical, Vidmar, 13... e4'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2p5/8/2Bp4/2R1PN2/PP3PPP/1Q3RK1 w - - 0 14' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Qb1 exd4

			variation => N__('QGD: Orthodox, Classical, Maroczy, 13...exd4'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2p5/8/2Bp4/2R1PN2/PPQ2PPP/5RK1 w - - 0 14' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Qc2 exd4

			variation => N__('QGD: Orthodox, Classical, Vidmar, 13...exd4'),
			moves => {
				'e3d4' => 'r1b2rk1/pp1nqppp/2p5/8/2BP4/2R2N2/PPQ2PPP/5RK1 b - - 0 14',
			},
		},
		'r1b2rk1/pp1nqppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 w - - 7 9' => {
			eco => 'D45x',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. Be2 O-O 8. O-O Qe7

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. Be2 O-O 8.O-O Qe7'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/2N1PN1P/PPQ2PP1/R1B2RK1 b - - 0 10' => {
			eco => 'D46j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 Qe7 10. h3

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... Qe7 10. h3'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - - 1 10' => {
			eco => 'D46j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 Qe7

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... Qe7'),
			moves => {
				'a2a3' => 'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/P1N1PN2/1PQ2PPP/R1B2RK1 b - - 0 10',
				'c1d2' => 'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/2N1PN2/PPQB1PPP/R4RK1 b - - 2 10',
				'h2h3' => 'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/2N1PN1P/PPQ2PP1/R1B2RK1 b - - 0 10',
			},
		},
		'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/2N1PN2/PPQB1PPP/R4RK1 b - - 2 10' => {
			eco => 'D46j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 Qe7 10. Bd2

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... Qe7 10. Bd2'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/P1N1PN2/1PQ2PPP/R1B2RK1 b - - 0 10' => {
			eco => 'D46j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 Qe7 10. a3

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... Qe7 10. a3'),
			moves => {
			},
		},
		'r1b2rk1/pp1nqppp/4p3/2pN4/3P4/4PN2/PPQ2PPP/R3KB1R b KQ - 0 10' => {
			eco => 'D62',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7'),
			moves => {
				'e6d5' => 'r1b2rk1/pp1nqppp/8/2pp4/3P4/4PN2/PPQ2PPP/R3KB1R w KQ - 0 11',
			},
		},
		'r1b2rk1/pp1nqppp/4p3/2pn4/3P4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 10' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 Nxd5 9. Bxe7 Qxe7

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7'),
			moves => {
				'c3d5' => 'r1b2rk1/pp1nqppp/4p3/2pN4/3P4/4PN2/PPQ2PPP/R3KB1R b KQ - 0 10',
			},
		},
		'r1b2rk1/pp1nqppp/8/2pp4/3P4/3BPN2/PPQ2PPP/R3K2R b KQ - 1 11' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 Nxd5 9. Bxe7 Qxe7 10. Nxd5 exd5 11. Bd3

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5, Main Line, 11. Bd3'),
			moves => {
				'g7g6' => 'r1b2rk1/pp1nqp1p/6p1/2pp4/3P4/3BPN2/PPQ2PPP/R3K2R w KQ - 0 12',
			},
		},
		'r1b2rk1/pp1nqppp/8/2pp4/3P4/4PN2/PPQ2PPP/R3KB1R w KQ - 0 11' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 Nxd5 9. Bxe7 Qxe7 10. Nxd5 exd5

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5, Main Line'),
			moves => {
				'f1d3' => 'r1b2rk1/pp1nqppp/8/2pp4/3P4/3BPN2/PPQ2PPP/R3K2R b KQ - 1 11',
			},
		},
		'r1b2rk1/pp1pppbp/1qn2np1/8/2P5/2N3P1/PPN1PPBP/R1BQ1RK1 b - - 2 9' => {
			eco => 'A39e',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Qb6 9. Nc2

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Qb6'),
			moves => {
			},
		},
		'r1b2rk1/pp1pppbp/1qn2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A39e',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Qb6

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Qb6'),
			moves => {
				'd4c2' => 'r1b2rk1/pp1pppbp/1qn2np1/8/2P5/2N3P1/PPN1PPBP/R1BQ1RK1 b - - 2 9',
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 w - - 8 9' => {
			eco => 'B35c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O'),
			moves => {
				'c4b3' => 'r1b2rk1/pp1pppbp/2n2np1/q7/3NP3/1BN1B3/PPP2PPP/R2Q1RK1 b - - 9 9',
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A39d',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Qa5

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Qa5'),
			moves => {
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/3NP3/1BN1B3/PPP2PPP/R2Q1RK1 b - - 9 9' => {
			eco => 'B35d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O 9. Bb3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'd7d6' => 'r1b2rk1/pp2ppbp/2np1np1/q7/3NP3/1BN1B3/PPP2PPP/R2Q1RK1 w - - 0 10',
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ - 8 9' => {
			eco => 'B35k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 Qa5

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 Qa5'),
			moves => {
			},
		},
		'r1b2rk1/pp2bppp/1qnppn2/6B1/4P3/1NN2P2/PPPQ2PP/2KR1B1R b - - 0 10' => {
			eco => 'B63u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. Nb3 Qb6 10. f3

			variation => N__('Sicilian: Richter-Rauzer, Podebrad, 10. f3'),
			moves => {
				'f8d8' => 'r1br2k1/pp2bppp/1qnppn2/6B1/4P3/1NN2P2/PPPQ2PP/2KR1B1R w - - 1 11',
			},
		},
		'r1b2rk1/pp2bppp/1qnppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w - - 6 10' => {
			eco => 'B63t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. Nb3 Qb6

			variation => N__('Sicilian: Richter-Rauzer, Podebrad Variation'),
			moves => {
				'f2f3' => 'r1b2rk1/pp2bppp/1qnppn2/6B1/4P3/1NN2P2/PPPQ2PP/2KR1B1R b - - 0 10',
			},
		},
		'r1b2rk1/pp2bppp/3ppn2/q5B1/2BQPP2/2N5/PPP3PP/2KR3R b - - 2 11' => {
			eco => 'B65e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 Nxd4 10. Qxd4 Qa5 11. Bc4

			variation => N__('Sicilian: Richter-Rauzer, 7... Be7, 9. f4 Nxd4 10.Qxd4 Qa5 11. Bc4'),
			moves => {
				'c8d7' => 'r4rk1/pp1bbppp/3ppn2/q5B1/2BQPP2/2N5/PPP3PP/2KR3R w - - 3 12',
			},
		},
		'r1b2rk1/pp2bppp/3ppn2/q5B1/3QPP2/2N5/PPP3PP/1K1R1B1R b - - 2 11' => {
			eco => 'B65d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 Nxd4 10. Qxd4 Qa5 11. Kb1

			variation => N__('Sicilian: Richter-Rauzer, 7... Be7, 9. f4 Nxd4 10.Qxd4 Qa5 11. Kb1'),
			moves => {
			},
		},
		'r1b2rk1/pp2bppp/3ppn2/q5B1/3QPP2/2N5/PPP3PP/2KR1B1R w - - 1 11' => {
			eco => 'B65c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 Nxd4 10. Qxd4 Qa5

			variation => N__('Sicilian: Richter-Rauzer, 7... Be7, 9. f4 Nxd4 10.Qxd4 Qa5'),
			moves => {
				'c1b1' => 'r1b2rk1/pp2bppp/3ppn2/q5B1/3QPP2/2N5/PPP3PP/1K1R1B1R b - - 2 11',
				'f1c4' => 'r1b2rk1/pp2bppp/3ppn2/q5B1/2BQPP2/2N5/PPP3PP/2KR3R b - - 2 11',
			},
		},
		'r1b2rk1/pp2nppp/2pq1n2/3pp1B1/1b2P2N/1BNP4/PPP2PPP/R2Q1RK1 w - - 2 11' => {
			eco => 'C49k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Ne7 8. Nh4 c6 9. Bc4 d5 10. Bb3 Qd6

			variation => N__('Four Knights: Symmetrical, Blake Variation'),
			moves => {
			},
		},
		'r1b2rk1/pp2ppbp/1qnp1np1/4P3/3N1P2/2N1B3/PPP1B1PP/R2Q1RK1 b - - 0 10' => {
			eco => 'B73',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. f4 Qb6 10. e5

			variation => N__('Sicilian: Dragon, Classical, Zollner Gambit'),
			moves => {
			},
		},
		'r1b2rk1/pp2ppbp/1qnp1np1/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w - - 1 10' => {
			eco => 'B73',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Classical, 9. f4'),
			moves => {
				'e4e5' => 'r1b2rk1/pp2ppbp/1qnp1np1/4P3/3N1P2/2N1B3/PPP1B1PP/R2Q1RK1 b - - 0 10',
			},
		},
		'r1b2rk1/pp2ppbp/2n3p1/q1p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11' => {
			eco => 'D87h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 Qa5

			variation => N__('Grünfeld: Classical Exchange, 10. Be3 Qa5'),
			moves => {
			},
		},
		'r1b2rk1/pp2ppbp/2np1np1/2q5/2P1PP2/2NB1N2/PP2Q1PP/R1B1K2R w KQ - 2 10' => {
			eco => 'E76m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Nf3 c5 7. dxc5 Qa5 8. Bd3 Qxc5 9. Qe2 Nc6

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Nf3 c5 7.dxc5 Qa5 8. Bd3 Qxc5 9. Qe2 Nc6'),
			moves => {
			},
		},
		'r1b2rk1/pp2ppbp/2np1np1/q7/3NP3/1BN1B2P/PPP2PP1/R2Q1RK1 b - - 0 10' => {
			eco => 'B35d',
                        # TRANSLATORS:

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'c8d7' => 'r4rk1/pp1bppbp/2np1np1/q7/3NP3/1BN1B2P/PPP2PP1/R2Q1RK1 w - - 1 11',
			},
		},
		'r1b2rk1/pp2ppbp/2np1np1/q7/3NP3/1BN1B3/PPP2PPP/R2Q1RK1 w - - 0 10' => {
			eco => 'B35d',
                        # TRANSLATORS:

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'h2h3' => 'r1b2rk1/pp2ppbp/2np1np1/q7/3NP3/1BN1B2P/PPP2PP1/R2Q1RK1 b - - 0 10',
			},
		},
		'r1b2rk1/pp2qppp/1np5/8/2BP4/2R2N2/PPQ2PPP/5RK1 w - - 1 15' => {
			eco => 'D68',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. Qc2 exd4 14. exd4 Nb6

			variation => N__('QGD: Orthodox, Classical, Vidmar, 13...exd4 14.exd4 Nb6'),
			moves => {
			},
		},
		'r1b2rk1/pp2qppp/2p5/4N3/2B5/2R1P3/PP3PPP/3Q1RK1 b - - 0 14' => {
			eco => 'D69',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Classical, 13.dxe5'),
			moves => {
				'e7e5' => 'r1b2rk1/pp3ppp/2p5/4q3/2B5/2R1P3/PP3PPP/3Q1RK1 w - - 0 15',
			},
		},
		'r1b2rk1/pp2qppp/2p5/4n3/2B5/2R1PN2/PP3PPP/3Q1RK1 w - - 0 14' => {
			eco => 'D69',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Classical, 13.dxe5'),
			moves => {
				'f3e5' => 'r1b2rk1/pp2qppp/2p5/4N3/2B5/2R1P3/PP3PPP/3Q1RK1 b - - 0 14',
			},
		},
		'r1b2rk1/pp2qppp/3pnn2/1Bp1p3/3PP3/2P2N2/P1P2PPP/R1BQR1K1 w - c6 0 12' => {
			eco => 'C49r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3 8. bxc3 Qe7 9. Re1 Nd8 10. d4 Ne6 11. Bc1 c5

			variation => N__('Four Knights: Symmetrical, Metger, 10. d4 Ne6 11. Bc1 c5'),
			moves => {
			},
		},
		'r1b2rk1/pp3ppp/2n1pn2/q1bp4/2P2B2/P1N1PN2/1PQ2PPP/2KR1B1R b - - 2 10' => {
			eco => 'D37v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 Be7 5. Bf4 O-O 6. e3 c5 7. dxc5 Bxc5 8. Qc2 Nc6 9. a3 Qa5 10. O-O-O

			variation => N__('QGD: Classical, 5...O-O 6. e3 c5 7.dxc5 Bxc5 8. Qc2 Nc6 9. a3 Qa5 10.O-O-O'),
			moves => {
			},
		},
		'r1b2rk1/pp3ppp/2n1pn2/q1bp4/2P2B2/P1N1PN2/1PQ2PPP/3RKB1R b K - 2 10' => {
			eco => 'D37u',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 Be7 5. Bf4 O-O 6. e3 c5 7. dxc5 Bxc5 8. Qc2 Nc6 9. a3 Qa5 10. Rd1

			variation => N__('QGD: Classical, 5...O-O 6. e3 c5 7.dxc5 Bxc5 8. Qc2 Nc6 9. a3 Qa5 10. Rd1'),
			moves => {
			},
		},
		'r1b2rk1/pp3ppp/2n1pn2/q1bp4/2P2B2/P1N1PN2/1PQ2PPP/R3KB1R w KQ - 1 10' => {
			eco => 'D37t',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 Be7 5. Bf4 O-O 6. e3 c5 7. dxc5 Bxc5 8. Qc2 Nc6 9. a3 Qa5

			variation => N__('QGD: Classical, 5...O-O 6. e3 c5 7.dxc5 Bxc5 8. Qc2 Nc6 9. a3 Qa5'),
			moves => {
				'a1d1' => 'r1b2rk1/pp3ppp/2n1pn2/q1bp4/2P2B2/P1N1PN2/1PQ2PPP/3RKB1R b K - 2 10',
				'e1c1' => 'r1b2rk1/pp3ppp/2n1pn2/q1bp4/2P2B2/P1N1PN2/1PQ2PPP/2KR1B1R b - - 2 10',
			},
		},
		'r1b2rk1/pp3ppp/2n2q2/2b5/3p4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 12' => {
			eco => 'D34l',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5 d4'),
			moves => {
				'c3d5' => 'r1b2rk1/pp3ppp/2n2q2/2bN4/3p4/5NP1/PP2PPBP/R2Q1RK1 b - - 1 12',
			},
		},
		'r1b2rk1/pp3ppp/2n2q2/2bN4/3p4/5NP1/PP2PPBP/R2Q1RK1 b - - 1 12' => {
			eco => 'D34m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5 Bxc5 10. Bg5 d4 11. Bxf6 Qxf6 12. Nd5

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5 d4 11.Bxf6 Qxf6 12. Nd5'),
			moves => {
				'f6d8' => 'r1bq1rk1/pp3ppp/2n5/2bN4/3p4/5NP1/PP2PPBP/R2Q1RK1 w - - 2 13',
			},
		},
		'r1b2rk1/pp3ppp/2p2q2/5P2/2B5/2R1P3/PP4PP/3Q1RK1 b - - 0 16' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5 Nxe5 14. Nxe5 Qxe5 15. f4 Qf6 16. f5

			variation => N__('QGD: Orthodox, Classical, 13.dxe5: 15. f4 Qf6 16. f5'),
			moves => {
			},
		},
		'r1b2rk1/pp3ppp/2p2q2/8/2B1PP2/2R5/PP4PP/3Q1RK1 b - - 0 16' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5 Nxe5 14. Nxe5 Qxe5 15. f4 Qf6 16. e4

			variation => N__('QGD: Orthodox, Classical, 13.dxe5: 15. f4 Qf6 16. e4'),
			moves => {
			},
		},
		'r1b2rk1/pp3ppp/2p2q2/8/2B2P2/2R1P3/PP4PP/3Q1RK1 w - - 1 16' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5 Nxe5 14. Nxe5 Qxe5 15. f4 Qf6

			variation => N__('QGD: Orthodox, Classical, 13.dxe5: 15. f4 Qf6'),
			moves => {
				'e3e4' => 'r1b2rk1/pp3ppp/2p2q2/8/2B1PP2/2R5/PP4PP/3Q1RK1 b - - 0 16',
				'f4f5' => 'r1b2rk1/pp3ppp/2p2q2/5P2/2B5/2R1P3/PP4PP/3Q1RK1 b - - 0 16',
			},
		},
		'r1b2rk1/pp3ppp/2p5/4q3/2B2P2/2R1P3/PP4PP/3Q1RK1 b - f3 0 15' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5 Nxe5 14. Nxe5 Qxe5 15. f4

			variation => N__('QGD: Orthodox, Classical, 13.dxe5: 15. f4'),
			moves => {
				'e5e4' => 'r1b2rk1/pp3ppp/2p5/8/2B1qP2/2R1P3/PP4PP/3Q1RK1 w - - 1 16',
				'e5f6' => 'r1b2rk1/pp3ppp/2p2q2/8/2B2P2/2R1P3/PP4PP/3Q1RK1 w - - 1 16',
			},
		},
		'r1b2rk1/pp3ppp/2p5/4q3/2B5/2R1P3/PP3PPP/3Q1RK1 w - - 0 15' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5 Nxe5 14. Nxe5 Qxe5

			variation => N__('QGD: Orthodox, Classical, 13.dxe5'),
			moves => {
				'f2f4' => 'r1b2rk1/pp3ppp/2p5/4q3/2B2P2/2R1P3/PP4PP/3Q1RK1 b - f3 0 15',
			},
		},
		'r1b2rk1/pp3ppp/2p5/8/2B1qP2/2R1P3/PP4PP/3Q1RK1 w - - 1 16' => {
			eco => 'D69',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. Bxe7 Qxe7 11. O-O Nxc3 12. Rxc3 e5 13. dxe5 Nxe5 14. Nxe5 Qxe5 15. f4 Qe4

			variation => N__('QGD: Orthodox, Classical, 13.dxe5: 15. f4 Qe4'),
			moves => {
			},
		},
		'r1b2rk1/ppB1ppbp/2n3p1/3q4/3P4/4P3/PP2NPPP/R2QKB1R b KQ - 2 9' => {
			eco => 'D84',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Bf4 Bg7 5. e3 O-O 6. cxd5 Nxd5 7. Nxd5 Qxd5 8. Bxc7 Nc6 9. Ne2

			variation => N__('Grünfeld: Grünfeld Gambit Accepted, 8... Nc6 9. Ne2'),
			moves => {
				'c8g4' => 'r4rk1/ppB1ppbp/2n3p1/3q4/3P2b1/4P3/PP2NPPP/R2QKB1R w KQ - 3 10',
			},
		},
		'r1b2rk1/ppB1ppbp/2n3p1/3q4/3P4/4P3/PP3PPP/R2QKBNR w KQ - 1 9' => {
			eco => 'D84',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Bf4 Bg7 5. e3 O-O 6. cxd5 Nxd5 7. Nxd5 Qxd5 8. Bxc7 Nc6

			variation => N__('Grünfeld: Grünfeld Gambit Accepted, 8... Nc6'),
			moves => {
				'g1e2' => 'r1b2rk1/ppB1ppbp/2n3p1/3q4/3P4/4P3/PP2NPPP/R2QKB1R b KQ - 2 9',
			},
		},
		'r1b2rk1/ppB1ppbp/B5p1/3q4/3P4/4P3/PP3PPP/R2QK1NR b KQ - 0 9' => {
			eco => 'D84',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Grünfeld Gambit Accepted, 8... Na6'),
			moves => {
				'd5g2' => 'r1b2rk1/ppB1ppbp/B5p1/8/3P4/4P3/PP3PqP/R2QK1NR w KQ - 0 10',
			},
		},
		'r1b2rk1/ppB1ppbp/B5p1/8/3P4/4P3/PP3PqP/R2QK1NR w KQ - 0 10' => {
			eco => 'D84',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Bf4 Bg7 5. e3 O-O 6. cxd5 Nxd5 7. Nxd5 Qxd5 8. Bxc7 Na6 9. Bxa6 Qxg2

			variation => N__('Grünfeld: Grünfeld Gambit Accepted, 8... Na6 9.Bxa6 Qxg2'),
			moves => {
			},
		},
		'r1b2rk1/ppB1ppbp/n5p1/3q4/3P4/4P3/PP3PPP/R2QKBNR w KQ - 1 9' => {
			eco => 'D84',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Bf4 Bg7 5. e3 O-O 6. cxd5 Nxd5 7. Nxd5 Qxd5 8. Bxc7 Na6

			variation => N__('Grünfeld: Grünfeld Gambit Accepted, 8... Na6'),
			moves => {
				'f1a6' => 'r1b2rk1/ppB1ppbp/B5p1/3q4/3P4/4P3/PP3PPP/R2QK1NR b KQ - 0 9',
			},
		},
		'r1b2rk1/ppp1qppp/2np1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2Q1RK1 w - - 1 9' => {
			eco => 'C49n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3 8. bxc3 Qe7

			variation => N__('Four Knights: Symmetrical, Metger unpin'),
			moves => {
				'f1e1' => 'r1b2rk1/ppp1qppp/2np1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2QR1K1 b - - 2 9',
			},
		},
		'r1b2rk1/ppp1qppp/2np1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2QR1K1 b - - 2 9' => {
			eco => 'C49n',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Metger unpin'),
			moves => {
				'c6d8' => 'r1bn1rk1/ppp1qppp/3p1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2QR1K1 w - - 3 10',
			},
		},
		'r1b2rk1/ppp1qppp/2nppn2/8/2PP4/5NP1/PP1NPPBP/R2Q1RK1 w - - 2 9' => {
			eco => 'E11v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nf3 Bb4+ 4. Bd2 Qe7 5. g3 Nc6 6. Bg2 Bxd2+ 7. Nbxd2 d6 8. O-O O-O

			variation => N__('Bogo-Indian: Nimzowitsch, 5. g3 Nc6 Main Line, 8...O-O'),
			moves => {
			},
		},
		'r1b2rk1/ppp1qppp/3pnn2/1B2p1B1/3PP3/2P2N2/P1P2PPP/R2QR1K1 w - - 1 11' => {
			eco => 'C49p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3 8. bxc3 Qe7 9. Re1 Nd8 10. d4 Ne6

			variation => N__('Four Knights: Symmetrical, Metger, 10. d4 Ne6'),
			moves => {
				'g5c1' => 'r1b2rk1/ppp1qppp/3pnn2/1B2p3/3PP3/2P2N2/P1P2PPP/R1BQR1K1 b - - 2 11',
			},
		},
		'r1b2rk1/ppp1qppp/3pnn2/1B2p3/3PP3/2P2N2/P1P2PPP/R1BQR1K1 b - - 2 11' => {
			eco => 'C49p',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Metger, 10. d4 Ne6'),
			moves => {
				'c7c5' => 'r1b2rk1/pp2qppp/3pnn2/1Bp1p3/3PP3/2P2N2/P1P2PPP/R1BQR1K1 w - c6 0 12',
				'f8d8' => 'r1br2k1/ppp1qppp/3pnn2/1B2p3/3PP3/2P2N2/P1P2PPP/R1BQR1K1 w - - 3 12',
			},
		},
		'r1b2rk1/ppq1bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/2KR3R w - - 8 10' => {
			eco => 'B89',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Be3 Be7 8. Qe2 O-O 9. O-O-O Qc7

			variation => N__('Sicilian: Velimirovic, 9.O-O-O Qc7'),
			moves => {
				'c4b3' => 'r1b2rk1/ppq1bppp/2nppn2/8/3NP3/1BN1B3/PPP1QPPP/2KR3R b - - 9 10',
			},
		},
		'r1b2rk1/ppq1bppp/2nppn2/8/3NP3/1BN1B3/PPP1QPPP/2KR3R b - - 9 10' => {
			eco => 'B89',
                        # TRANSLATORS:

			variation => N__('Sicilian: Velimirovic, 9.O-O-O Qc7'),
			moves => {
				'a7a6' => 'r1b2rk1/1pq1bppp/p1nppn2/8/3NP3/1BN1B3/PPP1QPPP/2KR3R w - - 0 11',
			},
		},
		'r1b2rk1/ppq1ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/2RQ1RK1 b - - 5 11' => {
			eco => 'D87j',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Classical Exchange, 10. Be3 Qc7'),
			moves => {
				'f8d8' => 'r1br2k1/ppq1ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/2RQ1RK1 w - - 6 12',
			},
		},
		'r1b2rk1/ppq1ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11' => {
			eco => 'D87j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 Qc7

			variation => N__('Grünfeld: Classical Exchange, 10. Be3 Qc7'),
			moves => {
				'a1c1' => 'r1b2rk1/ppq1ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/2RQ1RK1 b - - 5 11',
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/1Bp5/3P4/P1P1PN2/5PPP/R1BQ1RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Bb5

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bb5'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/1B3PPP/R2Q1RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Bb2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bb2'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/2Q2PPP/R1B2RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Qc2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Qc2'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/4QPPP/R1B2RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Qe2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Qe2'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/5PPP/R1BQ1RK1 w - - 1 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7'),
			moves => {
				'a3a4' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/P1BP4/2P1PN2/5PPP/R1BQ1RK1 b - - 0 11',
				'c1b2' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/1B3PPP/R2Q1RK1 b - - 2 11',
				'c4a2' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/3P4/P1P1PN2/B4PPP/R1BQ1RK1 b - - 2 11',
				'c4b5' => 'r1b2rk1/ppq2ppp/2n1pn2/1Bp5/3P4/P1P1PN2/5PPP/R1BQ1RK1 b - - 2 11',
				'c4d3' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/3P4/P1PBPN2/5PPP/R1BQ1RK1 b - - 2 11',
				'c4e2' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/3P4/P1P1PN2/4BPPP/R1BQ1RK1 b - - 2 11',
				'd1c2' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/2Q2PPP/R1B2RK1 b - - 2 11',
				'd1e2' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/4QPPP/R1B2RK1 b - - 2 11',
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/3P4/P1P1PN2/4BPPP/R1BQ1RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Be2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Be2'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/3P4/P1P1PN2/B4PPP/R1BQ1RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Ba2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Ba2'),
			moves => {
				'e6e5' => 'r1b2rk1/ppq2ppp/2n2n2/2p1p3/3P4/P1P1PN2/B4PPP/R1BQ1RK1 w - - 0 12',
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/3P4/P1PBPN2/5PPP/R1BQ1RK1 b - - 2 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Bd3

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bd3'),
			moves => {
				'e6e5' => 'r1b2rk1/ppq2ppp/2n2n2/2p1p3/3P4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 12',
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2p5/P1BP4/2P1PN2/5PPP/R1BQ1RK1 b - - 0 11' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. a4

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. a4'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2pP4/3P4/P1PBPN2/5PPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. cxd5

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10.cxd5'),
			moves => {
				'e6d5' => 'r1b2rk1/ppq2ppp/2n2n2/2pp4/3P4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 11',
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN1P/5PP1/R1BQ1RK1 b - - 0 10' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. h3

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10. h3'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN2/1B3PPP/R2Q1RK1 b - - 2 10' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. Bb2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10. Bb2'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN2/2Q2PPP/R1B2RK1 b - - 2 10' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. Qc2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10. Qc2'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN2/5PPP/R1BQ1RK1 w - - 1 10' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7'),
			moves => {
				'c1b2' => 'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN2/1B3PPP/R2Q1RK1 b - - 2 10',
				'c4d5' => 'r1b2rk1/ppq2ppp/2n1pn2/2pP4/3P4/P1PBPN2/5PPP/R1BQ1RK1 b - - 0 10',
				'd1c2' => 'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN2/2Q2PPP/R1B2RK1 b - - 2 10',
				'h2h3' => 'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN1P/5PP1/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1b2rk1/ppq2ppp/2n2n2/2p1p3/3P4/P1P1PN2/B4PPP/R1BQ1RK1 w - - 0 12' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Ba2 e5

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Ba2 e5'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n2n2/2p1p3/3P4/P1PBPN2/2Q2PPP/R1B2RK1 b - - 1 12' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4 Qc7 11. Bd3 e5 12. Qc2

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bd3 e5 12. Qc2'),
			moves => {
				'f8e8' => 'r1b1r1k1/ppq2ppp/2n2n2/2p1p3/3P4/P1PBPN2/2Q2PPP/R1B2RK1 w - - 2 13',
			},
		},
		'r1b2rk1/ppq2ppp/2n2n2/2p1p3/3P4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 12' => {
			eco => 'E59',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4 Qc7 11. Bd3'),
			moves => {
				'd1c2' => 'r1b2rk1/ppq2ppp/2n2n2/2p1p3/3P4/P1PBPN2/2Q2PPP/R1B2RK1 b - - 1 12',
			},
		},
		'r1b2rk1/ppq2ppp/2n2n2/2pp4/3P3N/P1PBP3/5PPP/R1BQ1RK1 b - - 1 11' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. cxd5 exd5 11. Nh4

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10.cxd5 exd5 11. Nh4'),
			moves => {
			},
		},
		'r1b2rk1/ppq2ppp/2n2n2/2pp4/3P4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 11' => {
			eco => 'E58',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10.cxd5'),
			moves => {
				'a3a4' => 'r1b2rk1/ppq2ppp/2n2n2/2pp4/P2P4/2PBPN2/5PPP/R1BQ1RK1 b - - 0 11',
				'f3h4' => 'r1b2rk1/ppq2ppp/2n2n2/2pp4/3P3N/P1PBP3/5PPP/R1BQ1RK1 b - - 1 11',
			},
		},
		'r1b2rk1/ppq2ppp/2n2n2/2pp4/P2P4/2PBPN2/5PPP/R1BQ1RK1 b - - 0 11' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 Qc7 10. cxd5 exd5 11. a4

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 Qc7 10.cxd5 exd5 11. a4'),
			moves => {
				'f8e8' => 'r1b1r1k1/ppq2ppp/2n2n2/2pp4/P2P4/2PBPN2/5PPP/R1BQ1RK1 w - - 1 12',
			},
		},
		'r1b2rk1/ppqnbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 4 9' => {
			eco => 'A55o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2 Qc7

			variation => N__('Old Indian: Main Line, 8. Qc2 Qc7'),
			moves => {
			},
		},
		'r1b2rk1/ppqnbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 4 9' => {
			eco => 'A55e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O Qc7

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O Qc7'),
			moves => {
			},
		},
		'r1b3nr/ppppkBpp/1b6/nP2N1q1/4P3/8/P1PP1PPP/RNBQK2R w KQ - 1 8' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'd1h5' => 'r1b3nr/ppppkBpp/1b6/nP2N1qQ/4P3/8/P1PP1PPP/RNB1K2R b KQ - 2 8',
			},
		},
		'r1b3nr/ppppkBpp/1b6/nP2N1qQ/4P3/8/P1PP1PPP/RNB1K2R b KQ - 2 8' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. b5 Na5 6. Nxe5 Qg5 7. Bxf7+ Ke7 8. Qh5

			variation => N__('Evans Gambit Declined, Vasquez Variation'),
			moves => {
			},
		},
		'r1b3r1/ppp2k1p/1b6/nP2B1q1/3PP3/8/P1P2PPP/RN1QK2R w KQ - 1 12' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Lange Variation'),
			moves => {
				'b1d2' => 'r1b3r1/ppp2k1p/1b6/nP2B1q1/3PP3/8/P1PN1PPP/R2QK2R b KQ - 2 12',
			},
		},
		'r1b3r1/ppp2k1p/1b6/nP2B1q1/3PP3/8/P1PN1PPP/R2QK2R b KQ - 2 12' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. b5 Na5 6. Nxe5 Nh6 7. d4 d6 8. Bxh6 dxe5 9. Bxg7 Rg8 10. Bxf7+ Kxf7 11. Bxe5 Qg5 12. Nd2

			variation => N__('Evans Gambit Declined, Pavlov Variation'),
			moves => {
			},
		},
		'r1bQkb1r/p4ppp/2p2n2/1N6/5p2/2N5/PP2PPPP/R3KB1R b KQkq - 0 10' => {
			eco => 'A33i',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'e8d8' => 'r1bk1b1r/p4ppp/2p2n2/1N6/5p2/2N5/PP2PPPP/R3KB1R w KQ - 0 11',
			},
		},
		'r1bQkb1r/ppp2ppp/2p5/4Pn2/8/5N2/PPP2PPP/RNB2RK1 b kq - 0 8' => {
			eco => 'C67r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap'),
			moves => {
				'e8d8' => 'r1bk1b1r/ppp2ppp/2p5/4Pn2/8/5N2/PPP2PPP/RNB2RK1 w - - 0 9',
			},
		},
		'r1bk1b1r/p4ppp/2p2n2/1N6/5p2/2N5/PP2PPPP/R3KB1R w KQ - 0 11' => {
			eco => 'A33j',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5 exf4 9. dxc6 bxc6 10. Qxd8+ Kxd8

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5 Queenswap'),
			moves => {
			},
		},
		'r1bk1b1r/ppNpq2p/2nn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR w KQ - 1 10' => {
			eco => 'C27h',
                        # TRANSLATORS:

			variation => N__('Vienna: 3. Bc4 Nxe4 4. Qh5 Nd6 5. Bb3 Nc6'),
			moves => {
				'c7a8' => 'N1bk1b1r/pp1pq2p/2nn2p1/3Qpp2/8/1B6/PPPP1PPP/R1B1K1NR b KQ - 0 10',
			},
		},
		'r1bk1b1r/ppp2pp1/2p4p/4Pn2/8/2N2N2/PPP2PPP/R1B2RK1 w - - 0 10' => {
			eco => 'C67t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+ Kxd8 9. Nc3 h6

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap, 9. Nc3 h6'),
			moves => {
			},
		},
		'r1bk1b1r/ppp2ppp/2p5/4Pn2/8/2N2N2/PPP2PPP/R1B2RK1 b - - 1 9' => {
			eco => 'C67s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Nd6 6. Bxc6 dxc6 7. dxe5 Nf5 8. Qxd8+ Kxd8 9. Nc3

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap, 9. Nc3'),
			moves => {
				'd8e8' => 'r1b1kb1r/ppp2ppp/2p5/4Pn2/8/2N2N2/PPP2PPP/R1B2RK1 w - - 2 10',
				'h7h6' => 'r1bk1b1r/ppp2pp1/2p4p/4Pn2/8/2N2N2/PPP2PPP/R1B2RK1 w - - 0 10',
			},
		},
		'r1bk1b1r/ppp2ppp/2p5/4Pn2/8/5N2/PPP2PPP/RNB2RK1 w - - 0 9' => {
			eco => 'C67r',
                        # TRANSLATORS:

			variation => N__('Spanish: Open Berlin, 5. d4 Nd6 Queenswap'),
			moves => {
				'b1c3' => 'r1bk1b1r/ppp2ppp/2p5/4Pn2/8/2N2N2/PPP2PPP/R1B2RK1 b - - 1 9',
			},
		},
		'r1bk2nr/ppNp1ppp/2n5/8/1b6/5B1q/PPPN1P1P/R1BQK2R w KQ - 1 10' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+'),
			moves => {
				'c7a8' => 'N1bk2nr/pp1p1ppp/2n5/8/1b6/5B1q/PPPN1P1P/R1BQK2R b KQ - 0 10',
			},
		},
		'r1bk2nr/pppp1Qpp/1b6/nP2q3/2B1P3/8/P1PP1PPP/RNB1K2R w KQ - 1 9' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Hirschbach Variation'),
			moves => {
				'c1b2' => 'r1bk2nr/pppp1Qpp/1b6/nP2q3/2B1P3/8/PBPP1PPP/RN2K2R b KQ - 2 9',
			},
		},
		'r1bk2nr/pppp1Qpp/1b6/nP2q3/2B1P3/8/PBPP1PPP/RN2K2R b KQ - 2 9' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. b5 Na5 6. Nxe5 Qg5 7. Qf3 Qxe5 8. Qxf7+ Kd8 9. Bb2

			variation => N__('Evans Gambit Declined, Hicken Variation'),
			moves => {
			},
		},
		'r1bk2nr/pppp1ppp/2n3q1/1N6/8/8/PPPNBPPP/R2Q1RK1 w - - 1 10' => {
			eco => 'C45g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Qh4 5. Nb5 Bb4+ 6. Bd2 Qxe4+ 7. Be2 Kd8 8. O-O Bxd2 9. Nxd2 Qg6

			variation => N__('Scotch: Steinitz, Rosenthal Variation'),
			moves => {
			},
		},
		'r1bk2nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPBBPPP/RN1Q1RK1 b - - 3 8' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'b4d2' => 'r1bk2nr/pppp1ppp/2n5/1N6/4q3/8/PPPbBPPP/RN1Q1RK1 w - - 0 9',
			},
		},
		'r1bk2nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPBBPPP/RN1QK2R w KQ - 2 8' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'e1g1' => 'r1bk2nr/pppp1ppp/2n5/1N6/1b2q3/8/PPPBBPPP/RN1Q1RK1 b - - 3 8',
			},
		},
		'r1bk2nr/pppp1ppp/2n5/1N6/4q3/8/PPPNBPPP/R2Q1RK1 b - - 0 9' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'e4g6' => 'r1bk2nr/pppp1ppp/2n3q1/1N6/8/8/PPPNBPPP/R2Q1RK1 w - - 1 10',
			},
		},
		'r1bk2nr/pppp1ppp/2n5/1N6/4q3/8/PPPbBPPP/RN1Q1RK1 w - - 0 9' => {
			eco => 'C45g',
                        # TRANSLATORS:

			variation => N__('Scotch: Steinitz, 5. Nb5 Bb4+ 6. Bd2'),
			moves => {
				'b1d2' => 'r1bk2nr/pppp1ppp/2n5/1N6/4q3/8/PPPNBPPP/R2Q1RK1 b - - 0 9',
			},
		},
		'r1bn1rk1/2q1bppp/p2p1n2/1ppPp3/4P3/2P2N1P/PPB2PP1/R1BQRNK1 b - - 2 14' => {
			eco => 'C98h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. d5 Nd8 14. Nf1

			variation => N__('Spanish: Closed, Chigorin, 12... Nc6 13. d5 Nd8 14. Nf1'),
			moves => {
			},
		},
		'r1bn1rk1/2q1bppp/p2p1n2/1ppPp3/4P3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 1 14' => {
			eco => 'C98g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. d5 Nd8

			variation => N__('Spanish: Closed, Chigorin, 12... Nc6 13. d5 Nd8'),
			moves => {
				'a2a4' => 'r1bn1rk1/2q1bppp/p2p1n2/1ppPp3/P3P3/2P2N1P/1PBN1PP1/R1BQR1K1 b - a3 0 14',
				'd2f1' => 'r1bn1rk1/2q1bppp/p2p1n2/1ppPp3/4P3/2P2N1P/PPB2PP1/R1BQRNK1 b - - 2 14',
			},
		},
		'r1bn1rk1/2q1bppp/p2p1n2/1ppPp3/P3P3/2P2N1P/1PBN1PP1/R1BQR1K1 b - a3 0 14' => {
			eco => 'C98i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Qc7 12. Nbd2 Nc6 13. d5 Nd8 14. a4

			variation => N__('Spanish: Closed, Chigorin, 12... Nc6 13. d5 Nd8 14. a4'),
			moves => {
			},
		},
		'r1bn1rk1/ppp1qppp/3p1n2/1B2p1B1/3PP3/2P2N2/P1P2PPP/R2QR1K1 b - - 0 10' => {
			eco => 'C49n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3 8. bxc3 Qe7 9. Re1 Nd8 10. d4

			variation => N__('Four Knights: Symmetrical, Metger, 10. d4'),
			moves => {
				'c8g4' => 'r2n1rk1/ppp1qppp/3p1n2/1B2p1B1/3PP1b1/2P2N2/P1P2PPP/R2QR1K1 w - - 1 11',
				'd8e6' => 'r1b2rk1/ppp1qppp/3pnn2/1B2p1B1/3PP3/2P2N2/P1P2PPP/R2QR1K1 w - - 1 11',
			},
		},
		'r1bn1rk1/ppp1qppp/3p1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2QR1K1 w - - 3 10' => {
			eco => 'C49n',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Metger unpin'),
			moves => {
				'd3d4' => 'r1bn1rk1/ppp1qppp/3p1n2/1B2p1B1/3PP3/2P2N2/P1P2PPP/R2QR1K1 b - - 0 10',
			},
		},
		'r1bnk2r/ppppnppp/5q2/1Bb5/3NP3/2P1B3/PP3PPP/RN1QK2R w KQkq - 3 8' => {
			eco => 'C45w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3 Qf6 6. c3 Nge7 7. Bb5 Nd8

			variation => N__('Scotch: Paulsen, Gunsberg Defence'),
			moves => {
			},
		},
		'r1bq1b1r/pp4pp/2p1k3/3np3/1nB1Q3/2N5/PPPP1PPP/R1B1K2R w KQ - 0 10' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'a2a3' => 'r1bq1b1r/pp4pp/2p1k3/3np3/1nB1Q3/P1N5/1PPP1PPP/R1B1K2R b KQ - 0 10',
			},
		},
		'r1bq1b1r/pp4pp/2p1k3/3np3/1nB1Q3/P1N5/1PPP1PPP/R1B1K2R b KQ - 0 10' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'b4a6' => 'r1bq1b1r/pp4pp/n1p1k3/3np3/2B1Q3/P1N5/1PPP1PPP/R1B1K2R w KQ - 1 11',
			},
		},
		'r1bq1b1r/pp4pp/n1p1k3/3np3/2B1Q3/P1N5/1PPP1PPP/R1B1K2R w KQ - 1 11' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'd2d4' => 'r1bq1b1r/pp4pp/n1p1k3/3np3/2BPQ3/P1N5/1PP2PPP/R1B1K2R b KQ d3 0 11',
			},
		},
		'r1bq1b1r/pp4pp/n1p1k3/3np3/2BPQ3/P1N5/1PP2PPP/R1B1K2R b KQ d3 0 11' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'a6c7' => 'r1bq1b1r/ppn3pp/2p1k3/3np3/2BPQ3/P1N5/1PP2PPP/R1B1K2R w KQ - 1 12',
			},
		},
		'r1bq1b1r/ppn3pp/2p1k3/3np3/2BPQ3/P1N5/1PP2PPP/R1B1K2R w KQ - 1 12' => {
			eco => 'C57g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Nxd5 6. Nxf7 Kxf7 7. Qf3+ Ke6 8. Nc3 Nb4 9. Qe4 c6 10. a3 Na6 11. d4 Nc7

			variation => N__('Two Knights: Fegatello, Leonhardt Variation'),
			moves => {
			},
		},
		'r1bq1b1r/ppp1n1pp/4k3/3np3/2B5/2N2Q2/PPPP1PPP/R1B1K2R w KQ - 4 9' => {
			eco => 'C57g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Nxd5 6. Nxf7 Kxf7 7. Qf3+ Ke6 8. Nc3 Ne7

			variation => N__('Two Knights: Fegatello, Polerio Defence'),
			moves => {
			},
		},
		'r1bq1b1r/ppp2kpp/2n5/3np3/2B5/5Q2/PPPP1PPP/RNB1K2R b KQ - 1 7' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'f7e6' => 'r1bq1b1r/ppp3pp/2n1k3/3np3/2B5/5Q2/PPPP1PPP/RNB1K2R w KQ - 2 8',
			},
		},
		'r1bq1b1r/ppp2kpp/2n5/3np3/2B5/8/PPPP1PPP/RNBQK2R w KQ - 0 7' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'd1f3' => 'r1bq1b1r/ppp2kpp/2n5/3np3/2B5/5Q2/PPPP1PPP/RNB1K2R b KQ - 1 7',
			},
		},
		'r1bq1b1r/ppp3pp/2n1k3/3np3/2B5/2N2Q2/PPPP1PPP/R1B1K2R b KQ - 3 8' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'c6b4' => 'r1bq1b1r/ppp3pp/4k3/3np3/1nB5/2N2Q2/PPPP1PPP/R1B1K2R w KQ - 4 9',
				'c6e7' => 'r1bq1b1r/ppp1n1pp/4k3/3np3/2B5/2N2Q2/PPPP1PPP/R1B1K2R w KQ - 4 9',
			},
		},
		'r1bq1b1r/ppp3pp/2n1k3/3np3/2B5/5Q2/PPPP1PPP/RNB1K2R w KQ - 2 8' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'b1c3' => 'r1bq1b1r/ppp3pp/2n1k3/3np3/2B5/2N2Q2/PPPP1PPP/R1B1K2R b KQ - 3 8',
			},
		},
		'r1bq1b1r/ppp3pp/4k3/3np3/1nB1Q3/2N5/PPPP1PPP/R1B1K2R b KQ - 5 9' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'c7c6' => 'r1bq1b1r/pp4pp/2p1k3/3np3/1nB1Q3/2N5/PPPP1PPP/R1B1K2R w KQ - 0 10',
			},
		},
		'r1bq1b1r/ppp3pp/4k3/3np3/1nB5/2N2Q2/PPPP1PPP/R1B1K2R w KQ - 4 9' => {
			eco => 'C57g',
                        # TRANSLATORS:

			variation => N__('Two Knights: Fegatello (Fried Liver) Attack'),
			moves => {
				'f3e4' => 'r1bq1b1r/ppp3pp/4k3/3np3/1nB1Q3/2N5/PPPP1PPP/R1B1K2R b KQ - 5 9',
			},
		},
		'r1bq1k1r/2p2ppp/p4n2/1pbPR1N1/3n4/1B6/PPPP1PPP/RNBQ2K1 w - - 1 11' => {
			eco => 'C78f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 d6 7. Ng5 d5 8. exd5 Nd4 9. Re1 Bc5 10. Rxe5+ Kf8

			variation => N__('Spanish: Rabinovich Variation'),
			moves => {
			},
		},
		'r1bq1r1k/ppp1npbp/3p1np1/3Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - - 1 10' => {
			eco => 'E97g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Kh8

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... Kh8'),
			moves => {
			},
		},
		'r1bq1r1k/pppnn1bp/3p2p1/3Ppp2/2P1P1P1/2N2P2/PP2B2P/R1BQNRK1 w - - 1 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. g4 Kh8

			variation => N__('King\'s Indian: Mar del Plata, Benko Attack, 11... Kh8'),
			moves => {
			},
		},
		'r1bq1r1k/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 w - - 2 12' => {
			eco => 'E98p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Nd3 f5 11. Bd2 Kh8

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Nd3 f5 11. Bd2 Kh8'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pb1/p2p1np1/3Pp2p/4P1P1/2N1BP1P/PP1QN3/R3KB1R b KQ - 0 12' => {
			eco => 'E89i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 c6 8. Nge2 cxd5 9. cxd5 Nbd7 10. Qd2 a6 11. g4 h5 12. h3

			variation => N__('King\'s Indian: Sämisch, Orthodox Main Line, 10. Qd2 a6 11. g4 h5 12. h3'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pb1/p2p1np1/3Pp2p/4P1P1/2N1BP2/PP1QN2P/R3KB1R w KQ h6 0 12' => {
			eco => 'E89h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 c6 8. Nge2 cxd5 9. cxd5 Nbd7 10. Qd2 a6 11. g4 h5

			variation => N__('King\'s Indian: Sämisch, Orthodox Main Line, 10. Qd2 a6 11. g4 h5'),
			moves => {
				'h2h3' => 'r1bq1rk1/1p1n1pb1/p2p1np1/3Pp2p/4P1P1/2N1BP1P/PP1QN3/R3KB1R b KQ - 0 12',
			},
		},
		'r1bq1rk1/1p1n1pb1/p2p1npp/2pP4/P3P3/2N1BP2/1P1Q2PP/R3KBNR w KQ - 1 12' => {
			eco => 'A65x',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Nbd7

			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/2p2np1/p3p3/2P1P3/2N2N2/PP3PPP/R1BQRBK1 w - - 0 11' => {
			eco => 'E96',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 a5 10. dxe5 dxe5

			variation => N__('King\'s Indian: 7.O-O Nbd7, Old Main Line, 10.dxe5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/2pp1np1/p3P3/2P1P3/2N2N2/PP3PPP/R1BQRBK1 b - - 0 10' => {
			eco => 'E96',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 a5 10. dxe5

			variation => N__('King\'s Indian: 7.O-O Nbd7, Old Main Line, 10.dxe5'),
			moves => {
				'd6e5' => 'r1bq1rk1/1p1n1pbp/2p2np1/p3p3/2P1P3/2N2N2/PP3PPP/R1BQRBK1 w - - 0 11',
			},
		},
		'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N1P/PP3PP1/R1BQRBK1 b - - 0 10' => {
			eco => 'E96',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 a5 10. h3

			variation => N__('King\'s Indian: 7.O-O Nbd7, Old Main Line, 10. h3'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N2/PP3PPP/1RBQRBK1 b - - 1 10' => {
			eco => 'E96',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 a5 10. Rb1

			variation => N__('King\'s Indian: 7.O-O Nbd7, Old Main Line, 10. Rb1'),
			moves => {
				'f8e8' => 'r1bqr1k1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N2/PP3PPP/1RBQRBK1 w - - 2 11',
			},
		},
		'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 w - a6 0 10' => {
			eco => 'E96',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 a5

			variation => N__('King\'s Indian: 7.O-O Nbd7, Old Main Line'),
			moves => {
				'a1b1' => 'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N2/PP3PPP/1RBQRBK1 b - - 1 10',
				'd4e5' => 'r1bq1rk1/1p1n1pbp/2pp1np1/p3P3/2P1P3/2N2N2/PP3PPP/R1BQRBK1 b - - 0 10',
				'h2h3' => 'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N1P/PP3PP1/R1BQRBK1 b - - 0 10',
			},
		},
		'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - a6 0 10' => {
			eco => 'E69d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 a5

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p1pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/1K1R1B1R b - - 1 10' => {
			eco => 'E86i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 c6 8. Qd2 Nbd7 9. O-O-O a6 10. Kb1

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 c6 8. Qd2 Nbd7 9.O-O-O a6 10. Kb1'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p1pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/2KR1B1R w - - 0 10' => {
			eco => 'E86h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 c6 8. Qd2 Nbd7 9. O-O-O a6

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 c6 8. Qd2 Nbd7 9.O-O-O a6'),
			moves => {
				'c1b1' => 'r1bq1rk1/1p1n1pbp/p1pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/1K1R1B1R b - - 1 10',
			},
		},
		'r1bq1rk1/1p1n1pbp/p1pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 0 10' => {
			eco => 'E69c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 a6

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP2B1/P3P3/2N2N2/1P2BPPP/R2Q1RK1 b - - 2 11' => {
			eco => 'A74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Bg5

			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 0 11' => {
			eco => 'A63',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6

			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2 a6'),
			moves => {
				'a2a4' => 'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P7/2N3P1/1P1NPPBP/R1BQ1RK1 b - a3 0 11',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N2N2/1P2BPPP/R1BQ1RK1 w - - 1 11' => {
			eco => 'A74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7

			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7'),
			moves => {
				'c1f4' => 'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3PB2/2N2N2/1P2BPPP/R2Q1RK1 b - - 2 11',
				'c1g5' => 'r1bq1rk1/1p1n1pbp/p2p1np1/2pP2B1/P3P3/2N2N2/1P2BPPP/R2Q1RK1 b - - 2 11',
				'f3d2' => 'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N5/1P1NBPPP/R1BQ1RK1 b - - 2 11',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N5/1P1NBPPP/R1BQ1RK1 b - - 2 11' => {
			eco => 'A74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Nd2

			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Nd2'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQ1RK1 b - - 2 11' => {
			eco => 'A70s',
                        # TRANSLATORS:

			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7'),
			moves => {
				'f8e8' => 'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQ1RK1 w - - 3 12',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQK2R w KQ - 1 11' => {
			eco => 'A70s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7

			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7'),
			moves => {
				'e1g1' => 'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQ1RK1 b - - 2 11',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3PB2/2N2N2/1P2BPPP/R2Q1RK1 b - - 2 11' => {
			eco => 'A74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Bf4

			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Bf4'),
			moves => {
				'd8e7' => 'r1b2rk1/1p1nqpbp/p2p1np1/2pP4/P3PB2/2N2N2/1P2BPPP/R2Q1RK1 w - - 3 12',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P7/2N2NP1/1P2PPBP/R1BQ1RK1 w - - 1 11' => {
			eco => 'A63',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O a6 10. a4 Nbd7

			variation => N__('Benoni: Fianchetto, 9... a6 10. a4 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P7/2N3P1/1P1NPPBP/R1BQ1RK1 b - a3 0 11' => {
			eco => 'A63',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4

			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2 a6 11. a4'),
			moves => {
				'f8e8' => 'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P7/2N3P1/1P1NPPBP/R1BQ1RK1 w - - 1 12',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/3Pp3/4P1P1/2N1BP2/PP1QN2P/R3KB1R b KQ g3 0 11' => {
			eco => 'E89g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 c6 8. Nge2 cxd5 9. cxd5 Nbd7 10. Qd2 a6 11. g4

			variation => N__('King\'s Indian: Sämisch, Orthodox Main Line, 10. Qd2 a6 11. g4'),
			moves => {
				'h7h5' => 'r1bq1rk1/1p1n1pb1/p2p1np1/3Pp2p/4P1P1/2N1BP2/PP1QN2P/R3KB1R w KQ h6 0 12',
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/3Pp3/4P3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 0 11' => {
			eco => 'E89f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 c6 8. Nge2 cxd5 9. cxd5 Nbd7 10. Qd2 a6

			variation => N__('King\'s Indian: Sämisch, Orthodox Main Line, 10. Qd2 a6'),
			moves => {
				'g2g4' => 'r1bq1rk1/1p1n1pbp/p2p1np1/3Pp3/4P1P1/2N1BP2/PP1QN2P/R3KB1R b KQ g3 0 11',
			},
		},
		'r1bq1rk1/1p1n1ppp/p1pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - - 0 10' => {
			eco => 'D46k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 a6

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... a6'),
			moves => {
				'f1d1' => 'r1bq1rk1/1p1n1ppp/p1pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1BR2K1 b - - 1 10',
			},
		},
		'r1bq1rk1/1p1n1ppp/p1pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1BR2K1 b - - 1 10' => {
			eco => 'D46k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 a6 10. Rd1

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... a6 10. Rd1'),
			moves => {
				'b7b5' => 'r1bq1rk1/3n1ppp/p1pbpn2/1p6/2BP4/2N1PN2/PPQ2PPP/R1BR2K1 w - b6 0 11',
			},
		},
		'r1bq1rk1/1p1n1ppp/p3pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1B2RK1 w - - 0 10' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. Qe2 a6

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. Qe2 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbpp1/p1p1pn1p/3p2B1/2PP4/P1N1PN2/1PQ2PPP/2R1KB1R w K - 0 10' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. a3 h6

			variation => N__('QGD: Orthodox, Rubinstein Attack, Grünfeld Variation, 9... h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/1p1nbpp1/p1p1pn1p/3p4/2PP3B/P1N1PN2/1PQ2PPP/2R1KB1R b K - 1 10',
			},
		},
		'r1bq1rk1/1p1nbpp1/p1p1pn1p/3p4/2PP3B/P1N1PN2/1PQ2PPP/2R1KB1R b K - 1 10' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. a3 h6 10. Bh4

			variation => N__('QGD: Orthodox, Rubinstein Attack, Grünfeld Variation, 9... h6 10. Bh4'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbpp1/p1p2n1p/3p2B1/3P4/2NBPN2/PPQ2PPP/2R1K2R w K - 0 11' => {
			eco => 'D65',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. cxd5 exd5 10. Bd3 h6

			variation => N__('QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10. Bd3 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/1p1nbpp1/p1p2n1p/3p4/3P3B/2NBPN2/PPQ2PPP/2R1K2R b K - 1 11',
			},
		},
		'r1bq1rk1/1p1nbpp1/p1p2n1p/3p4/3P3B/2NBPN2/PPQ2PPP/2R1K2R b K - 1 11' => {
			eco => 'D65',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. cxd5 exd5 10. Bd3 h6 11. Bh4

			variation => N__('QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10. Bd3 h6 11. Bh4'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/2pp1n2/p3p3/P1BPP3/2N2N2/1PP2PPP/R1BQR1K1 w - a6 0 9' => {
			eco => 'C41y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. Re1 c6 8. a4 a5

			variation => N__('Philidor: Improved Hanham, 7. Re1 c6 8. a4 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1p1p3/3n2B1/3P4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 10' => {
			eco => 'D65',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. cxd5 Nxd5

			variation => N__('QGD: Orthodox, Rubinstein Attack, 9.cxd5 Nxd5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1p1pn2/2Pp2B1/3P4/2N1PN2/PP3PPP/2RQKB1R w K - 0 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. c5 c6

			variation => N__('QGD: Orthodox, Swiss, 8. c5 c6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1p1pn2/3P2B1/3P4/2N1PN2/PPQ2PPP/2R1KB1R b K - 0 9' => {
			eco => 'D65',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. cxd5

			variation => N__('QGD: Orthodox, Rubinstein Attack, 9.cxd5'),
			moves => {
				'e6d5' => 'r1bq1rk1/1p1nbppp/p1p2n2/3p2B1/3P4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 10',
				'f6d5' => 'r1bq1rk1/1p1nbppp/p1p1p3/3n2B1/3P4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 10',
			},
		},
		'r1bq1rk1/1p1nbppp/p1p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 9' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6

			variation => N__('QGD: Orthodox, Rubinstein Attack, Carlsbad Variation'),
			moves => {
				'a2a3' => 'r1bq1rk1/1p1nbppp/p1p1pn2/3p2B1/2PP4/P1N1PN2/1PQ2PPP/2R1KB1R b K - 0 9',
				'c4d5' => 'r1bq1rk1/1p1nbppp/p1p1pn2/3P2B1/3P4/2N1PN2/PPQ2PPP/2R1KB1R b K - 0 9',
			},
		},
		'r1bq1rk1/1p1nbppp/p1p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R w K - 0 9' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 a6

			variation => N__('QGD: Orthodox, Main Line, 8... a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1p1pn2/3p2B1/2PP4/P1N1PN2/1PQ2PPP/2R1KB1R b K - 0 9' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. a3

			variation => N__('QGD: Orthodox, Rubinstein Attack, Grünfeld Variation'),
			moves => {
				'h7h6' => 'r1bq1rk1/1p1nbpp1/p1p1pn1p/3p2B1/2PP4/P1N1PN2/1PQ2PPP/2R1KB1R w K - 0 10',
			},
		},
		'r1bq1rk1/1p1nbppp/p1p2n2/3p2B1/3P4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 10' => {
			eco => 'D65',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 a6 9. cxd5 exd5

			variation => N__('QGD: Orthodox, Rubinstein Attack, 9.cxd5 exd5'),
			moves => {
				'f1d3' => 'r1bq1rk1/1p1nbppp/p1p2n2/3p2B1/3P4/2NBPN2/PPQ2PPP/2R1K2R b K - 1 10',
			},
		},
		'r1bq1rk1/1p1nbppp/p1p2n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R w K - 0 10' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. cxd5 exd5 9. Bd3 c6

			variation => N__('QGD: Orthodox, Swiss, Karlsbad, 9. Bd3 c6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1p2n2/3p2B1/3P4/2NBPN2/PPQ2PPP/2R1K2R b K - 1 10' => {
			eco => 'D65',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Rubinstein Attack, 9.cxd5 exd5'),
			moves => {
				'f8e8' => 'r1bqr1k1/1p1nbppp/p1p2n2/3p2B1/3P4/2NBPN2/PPQ2PPP/2R1K2R w K - 2 11',
				'h7h6' => 'r1bq1rk1/1p1nbpp1/p1p2n1p/3p2B1/3P4/2NBPN2/PPQ2PPP/2R1K2R w K - 0 11',
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 0 9' => {
			eco => 'A55n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Be3 a6

			variation => N__('Old Indian: Main Line, 8. Be3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9' => {
			eco => 'A55s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1 a6

			variation => N__('Old Indian: Main Line, 8. Re1 a6'),
			moves => {
				'e2f1' => 'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b - - 1 9',
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b - - 1 9' => {
			eco => 'A55t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1 a6 9. Bf1

			variation => N__('Old Indian: Main Line, 8. Re1 a6 9. Bf1'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 0 9' => {
			eco => 'A55p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2 a6

			variation => N__('Old Indian: Main Line, 8. Qc2 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A55g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O a6

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O a6'),
			moves => {
				'a2a4' => 'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/P1PPP3/2N2NP1/1P3PBP/R1BQ1RK1 b - a3 0 9',
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/P1PPP3/2N2NP1/1P3PBP/R1BQ1RK1 b - a3 0 9' => {
			eco => 'A55h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O a6 9. a4

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O a6 9. a4'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p2p1n2/4p3/P3PP2/2NB1N2/1PP3PP/R1BQ1RK1 b - - 4 10' => {
			eco => 'B93s',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3'),
			moves => {
				'd7c5' => 'r1bq1rk1/1p2bppp/p2p1n2/2n1p3/P3PP2/2NB1N2/1PP3PP/R1BQ1RK1 w - - 5 11',
				'e5f4' => 'r1bq1rk1/1p1nbppp/p2p1n2/8/P3Pp2/2NB1N2/1PP3PP/R1BQ1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/1p1nbppp/p2p1n2/4p3/P3PP2/2NB1N2/1PP3PP/R1BQK2R w KQ - 3 10' => {
			eco => 'B93s',
                        # TRANSLATORS:

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3'),
			moves => {
				'e1g1' => 'r1bq1rk1/1p1nbppp/p2p1n2/4p3/P3PP2/2NB1N2/1PP3PP/R1BQ1RK1 b - - 4 10',
			},
		},
		'r1bq1rk1/1p1nbppp/p2p1n2/8/P3Pp2/2NB1N2/1PP3PP/R1BQ1RK1 w - - 0 11' => {
			eco => 'B93u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3 O-O 10. O-O exf4

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3 O-O 10.O-O exf4'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p3pn2/2p3B1/2BP4/2N1PN2/PP3PPP/R2Q1RK1 w - - 0 10' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 dxc4 8. Bxc4 c5 9. O-O a6

			variation => N__('QGD: Orthodox, Botvinnik, 8... c5 9.O-O a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nppbp/2np2p1/p7/2P1P3/2N5/PPNBBPPP/R2Q1RK1 w - a6 0 11' => {
			eco => 'B37o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9. O-O Nd7 10. Bd2 a5

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9.O-O Nd7 10. Bd2 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nppbp/p2p1np1/2p5/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 0 9' => {
			eco => 'E81t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nbd7 7. Qd2 c5 8. Nge2 a6

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 Nbd7 7. Qd2 c5 8.Nge2 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/2n2np1/p7/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ a6 0 9' => {
			eco => 'B35l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 a5

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 a5'),
			moves => {
				'a2a4' => 'r1bq1rk1/1p1pppbp/2n2np1/p7/P2NP3/1BN1B3/1PP2PPP/R2QK2R b KQ a3 0 9',
				'f2f3' => 'r1bq1rk1/1p1pppbp/2n2np1/p7/3NP3/1BN1BP2/PPP3PP/R2QK2R b KQ - 0 9',
			},
		},
		'r1bq1rk1/1p1pppbp/2n2np1/p7/3NP3/1BN1BP2/PPP3PP/R2QK2R b KQ - 0 9' => {
			eco => 'B35n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 a5 9. f3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 a5 9. f3'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/2n2np1/p7/P2NP3/1BN1B3/1PP2PPP/R2QK2R b KQ a3 0 9' => {
			eco => 'B35m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 a5 9. a4

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 a5 9. a4'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/p1n2np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'A38l',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 a6

			variation => N__('English: Symmetrical, Main Line, 7. d3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/p1n2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A39f',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 a6

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bpp1/p1np1n1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQR1K1 w - - 0 11' => {
			eco => 'B58o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nf3 h6 8. O-O Be7 9. Re1 O-O 10. h3 a6

			variation => N__('Sicilian: Boleslavsky, 7. Nf3, 9. Re1 O-O 10. h3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/2np1n2/p3p3/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - a6 0 10' => {
			eco => 'B59n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. Be3 a5

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O 9. Be3 a5'),
			moves => {
				'a2a4' => 'r1bq1rk1/1p2bppp/2np1n2/p3p3/P3P3/1NN1B3/1PP1BPPP/R2Q1RK1 b - a3 0 10',
			},
		},
		'r1bq1rk1/1p2bppp/2np1n2/p3p3/P3P3/1NN1B3/1PP1BPPP/R2Q1RK1 b - a3 0 10' => {
			eco => 'B59o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. Be3 a5 10. a4

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O 9. Be3 a5 10. a4'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/2nppn2/p5B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w - a6 0 10' => {
			eco => 'B63r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. Nb3 a5

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. Nb3 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/p1n1p3/3n4/P1BP4/2N2N2/1P2QPPP/R1BR2K1 w - - 3 12' => {
			eco => 'D27w',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. O-O a6 7. a4 Nc6 8. Qe2 cxd4 9. Rd1 Be7 10. exd4 O-O 11. Nc3 Nd5

			variation => N__('QGA: Classical, Rubinstein, 8. Qe2 cxd4, 11. Nc3 Nd5'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/p1n1pn2/8/2BP4/P1N2N2/1P3PPP/R1BQR1K1 w - - 0 12' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7 11. Re1 a6

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7 11. Re1 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/p1n1pn2/8/P1BP4/2N2N2/1P2QPPP/R1BR2K1 b - - 2 11' => {
			eco => 'D27v',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. O-O a6 7. a4 Nc6 8. Qe2 cxd4 9. Rd1 Be7 10. exd4 O-O 11. Nc3

			variation => N__('QGA: Classical, Rubinstein, 8. Qe2 cxd4, 11. Nc3'),
			moves => {
				'f6d5' => 'r1bq1rk1/1p2bppp/p1n1p3/3n4/P1BP4/2N2N2/1P2QPPP/R1BR2K1 w - - 3 12',
			},
		},
		'r1bq1rk1/1p2bppp/p1n1pn2/8/P1BP4/5N2/1P2QPPP/RNBR2K1 w - - 1 11' => {
			eco => 'D27v',
                        # TRANSLATORS:

			variation => N__('QGA: Classical, Rubinstein, 8. Qe2 cxd4'),
			moves => {
				'b1c3' => 'r1bq1rk1/1p2bppp/p1n1pn2/8/P1BP4/2N2N2/1P2QPPP/R1BR2K1 b - - 2 11',
			},
		},
		'r1bq1rk1/1p2bppp/p1np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1R1K w - - 6 10' => {
			eco => 'B92q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. Kh1 Nc6

			variation => N__('Sicilian: Najdorf, 6. Be2 e5 7. Nb3 Be7 8.O-O O-O 9. Kh1 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w - - 0 10' => {
			eco => 'B63s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. Nb3 a6

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. Nb3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP2BPPP/R1BQ1RK1 b - - 5 10' => {
			eco => 'B44v',
                        # TRANSLATORS:

			variation => N__('Sicilian, Taimanov, Szen, 7.N1c3 a6 8. Na3 Be7 9. Be2 O-O'),
			moves => {
				'b7b6' => 'r1bq1rk1/4bppp/ppnppn2/8/2P1P3/N1N5/PP2BPPP/R1BQ1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP2BPPP/R1BQK2R w KQ - 4 10' => {
			eco => 'B44v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7 9. Be2 O-O

			variation => N__('Sicilian, Taimanov, Szen, 7.N1c3 a6 8. Na3 Be7 9. Be2 O-O'),
			moves => {
				'e1g1' => 'r1bq1rk1/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP2BPPP/R1BQ1RK1 b - - 5 10',
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1R1K b - - 4 10' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. Kh1

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. Kh1'),
			moves => {
				'c8d7' => 'r2q1rk1/1p1bbppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1R1K w - - 5 11',
				'd8c7' => 'r1b2rk1/1pq1bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1R1K w - - 5 11',
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w - - 3 10' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6'),
			moves => {
				'a2a4' => 'r1bq1rk1/1p2bppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1RK1 b - a3 0 10',
				'd1e1' => 'r1bq1rk1/1p2bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R3QRK1 b - - 4 10',
				'g1h1' => 'r1bq1rk1/1p2bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1R1K b - - 4 10',
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R3QRK1 b - - 4 10' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. Qe1

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. Qe1'),
			moves => {
				'd8c7' => 'r1b2rk1/1pq1bppp/p1nppn2/8/3NPP2/2N1B3/PPP1B1PP/R3QRK1 w - - 5 11',
			},
		},
		'r1bq1rk1/1p2bppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1RK1 b - a3 0 10' => {
			eco => 'B85',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 a6 7. O-O Be7 8. f4 O-O 9. Be3 Nc6 10. a4

			variation => N__('Sicilian: Scheveningen, Classical, Main Line, 9... Nc6 10. a4'),
			moves => {
				'c8d7' => 'r2q1rk1/1p1bbppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1RK1 w - - 1 11',
				'd8c7' => 'r1b2rk1/1pq1bppp/p1nppn2/8/P2NPP2/2N1B3/1PP1B1PP/R2Q1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/1p2bppp/p2p1n2/2n1p3/P3PP2/2NB1N2/1PP3PP/R1BQ1RK1 w - - 5 11' => {
			eco => 'B93t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3 O-O 10. O-O Nc5

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3 O-O 10.O-O Nc5'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/2np1np1/p7/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ a6 0 10' => {
			eco => 'B77b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 a5

			variation => N__('Sicilian: Dragon, Yugoslav, Byrne Variation'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/2np1np1/p7/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - a6 0 10' => {
			eco => 'B74',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Nb3 a5

			variation => N__('Sicilian: Dragon, Classical, Alekhine Variation'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A36z',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3 a6

			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/2p5/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A38r',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. a3 a6

			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. a3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/6B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'B70u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O O-O 8. Bg5 Nc6 9. Nb3 a6

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O O-O 8. Bg5 Nc6 9. Nb3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'B38p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 d6 9. O-O a6

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 d6 9.O-O a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'B74',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Nb3 a6

			variation => N__('Sicilian: Dragon, Classical, 9. Nb3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/R4RK1 b - - 1 11' => {
			eco => 'A39q',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6 11. Bd2

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6 11. Bd2'),
			moves => {
				'a8b8' => '1rbq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/R4RK1 w - - 2 12',
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w - - 0 11' => {
			eco => 'A39p',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6'),
			moves => {
				'c1d2' => 'r1bq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/R4RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/1RBQ1RK1 b - - 1 10' => {
			eco => 'E66f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Rb1

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... a6 10. Rb1'),
			moves => {
				'a8b8' => '1rbq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/1RBQ1RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'E66e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... a6'),
			moves => {
				'a1b1' => 'r1bq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/1RBQ1RK1 b - - 1 10',
				'd1c2' => 'r1bq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PPQNPPBP/R1B2RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PPQNPPBP/R1B2RK1 b - - 1 10' => {
			eco => 'E66h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 a6 10. Qc2

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... a6 10. Qc2'),
			moves => {
				'a8b8' => '1rbq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PPQNPPBP/R1B2RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/1pp1b1pp/p1p2p2/2n1p3/2N1P2N/3P4/PPP2PPP/R1BQ1RK1 w - - 2 11' => {
			eco => 'C85q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7 8. Nbd2 O-O 9. Nc4 f6 10. Nh4 Nc5

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2 O-O 9. Nc4 f6 10. Nh4 Nc5'),
			moves => {
				'h4f5' => 'r1bq1rk1/1pp1b1pp/p1p2p2/2n1pN2/2N1P3/3P4/PPP2PPP/R1BQ1RK1 b - - 3 11',
			},
		},
		'r1bq1rk1/1pp1b1pp/p1p2p2/2n1pN2/2N1P3/3P4/PPP2PPP/R1BQ1RK1 b - - 3 11' => {
			eco => 'C85r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7 8. Nbd2 O-O 9. Nc4 f6 10. Nh4 Nc5 11. Nf5

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2 O-O 9. Nc4 f6 10. Nh4 Nc5 11. Nf5'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1bppp/p1np1n2/4p3/B2PP3/2P2N2/PP3PPP/RNBQR1K1 b - d3 0 8' => {
			eco => 'C87j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. c3 O-O 8. d4

			variation => N__('Spanish: Closed, Averbakh, 7. c3 O-O 8. d4'),
			moves => {
				'c8d7' => 'r2q1rk1/1ppbbppp/p1np1n2/4p3/B2PP3/2P2N2/PP3PPP/RNBQR1K1 w - - 1 9',
			},
		},
		'r1bq1rk1/1pp1bppp/p1np1n2/4p3/B3P3/2P2N1P/PP1P1PP1/RNBQR1K1 b - - 0 8' => {
			eco => 'C87n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. c3 O-O 8. h3

			variation => N__('Spanish: Closed, Averbakh, 7. c3 O-O 8. h3'),
			moves => {
				'c8d7' => 'r2q1rk1/1ppbbppp/p1np1n2/4p3/B3P3/2P2N1P/PP1P1PP1/RNBQR1K1 w - - 1 9',
			},
		},
		'r1bq1rk1/1pp1bppp/p1np1n2/4p3/B3P3/2P2N2/PP1P1PPP/RNBQR1K1 w - - 1 8' => {
			eco => 'C87i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. c3 O-O

			variation => N__('Spanish: Closed, Averbakh, 7. c3 O-O'),
			moves => {
				'd2d4' => 'r1bq1rk1/1pp1bppp/p1np1n2/4p3/B2PP3/2P2N2/PP3PPP/RNBQR1K1 b - d3 0 8',
				'h2h3' => 'r1bq1rk1/1pp1bppp/p1np1n2/4p3/B3P3/2P2N1P/PP1P1PP1/RNBQR1K1 b - - 0 8',
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/3Pp3/1BP1P3/2N2N2/P3BPPP/R2Q1RK1 b - - 0 11' => {
			eco => 'E97k',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... a5 10. Ba3 axb4'),
			moves => {
				'f6d7' => 'r1bq1rk1/1ppnnpbp/3p2p1/3Pp3/1BP1P3/2N2N2/P3BPPP/R2Q1RK1 w - - 1 12',
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/3Pp3/1pP1P3/B1N2N2/P3BPPP/R2Q1RK1 w - - 0 11' => {
			eco => 'E97k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 a5 10. Ba3 axb4

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... a5 10. Ba3 axb4'),
			moves => {
				'a3b4' => 'r1bq1rk1/1pp1npbp/3p1np1/3Pp3/1BP1P3/2N2N2/P3BPPP/R2Q1RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/P2Pp3/2P1P3/2N2N2/P3BPPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E97i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 a5 10. bxa5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... a5 10.bxa5'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - a6 0 10' => {
			eco => 'E97i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 a5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... a5'),
			moves => {
				'b4a5' => 'r1bq1rk1/1pp1npbp/3p1np1/P2Pp3/2P1P3/2N2N2/P3BPPP/R1BQ1RK1 b - - 0 10',
				'c1a3' => 'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/1PP1P3/B1N2N2/P3BPPP/R2Q1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/1PP1P3/B1N2N2/P3BPPP/R2Q1RK1 b - - 1 10' => {
			eco => 'E97j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 a5 10. Ba3

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... a5 10. Ba3'),
			moves => {
				'a5b4' => 'r1bq1rk1/1pp1npbp/3p1np1/3Pp3/1pP1P3/B1N2N2/P3BPPP/R2Q1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/2P1P3/2N5/PP1NBPPP/1RBQ1RK1 b - - 1 10' => {
			eco => 'E97x',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 a5 10. Rb1

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5 10. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - a6 0 10' => {
			eco => 'E97x',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 a5

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5'),
			moves => {
				'a1b1' => 'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/2P1P3/2N5/PP1NBPPP/1RBQ1RK1 b - - 1 10',
				'a2a3' => 'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/2P1P3/P1N5/1P1NBPPP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/2P1P3/P1N5/1P1NBPPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E97y',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 a5 10. a3

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5 10. a3'),
			moves => {
				'c8d7' => 'r2q1rk1/1ppbnpbp/3p1np1/p2Pp3/2P1P3/P1N5/1P1NBPPP/R1BQ1RK1 w - - 1 11',
				'f6d7' => 'r1bq1rk1/1ppnnpbp/3p2p1/p2Pp3/2P1P3/P1N5/1P1NBPPP/R1BQ1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/1pp1npbp/3p2p1/P2Pp3/2P1Pn2/2N2N2/P4PPP/R1BQRBK1 b - - 0 12' => {
			eco => 'E97q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 Nf4 11. Bf1 a5 12. bxa5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 Nf4 11. Bf1 a5 12.bxa5'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1npbp/3p2p1/p2Pp2n/1PP1P3/2N2N2/P3BPPP/R1BQR1K1 w - a6 0 11' => {
			eco => 'E97p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 a5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1npbp/3p2p1/p2Pp3/1PP1Pn2/2N2N2/P4PPP/R1BQRBK1 w - a6 0 12' => {
			eco => 'E97q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 Nf4 11. Bf1 a5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 Nf4 11. Bf1 a5'),
			moves => {
				'b4a5' => 'r1bq1rk1/1pp1npbp/3p2p1/P2Pp3/2P1Pn2/2N2N2/P4PPP/R1BQRBK1 b - - 0 12',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/3P4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E63l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. d5

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. d5'),
			moves => {
				'c6a5' => 'r1bq1rk1/1pp1ppbp/p2p1np1/n2P4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E63d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. b3

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. b3'),
			moves => {
				'a8b8' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E63a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6

			variation => N__('King\'s Indian: Fianchetto, Panno Variation'),
			moves => {
				'b2b3' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8',
				'd4d5' => 'r1bq1rk1/1pp1ppbp/p1np1np1/3P4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
				'e2e4' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 8',
				'f1e1' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 b - - 1 8',
				'h2h3' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 b - - 1 8' => {
			eco => 'E63b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. Re1

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. Re1'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 b - - 0 8' => {
			eco => 'E63h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. h3

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. h3'),
			moves => {
				'a8b8' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1Q2PP/2KR1BNR b - - 1 8' => {
			eco => 'E83c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Qd2 a6 8. O-O-O

			variation => N__('King\'s Indian: Sämisch, 6... Nc6 7. Qd2 a6 8.O-O-O'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1Q2PP/R3KBNR w KQ - 0 8' => {
			eco => 'E83c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Qd2 a6

			variation => N__('King\'s Indian: Sämisch, 6... Nc6 7. Qd2 a6'),
			moves => {
				'e1c1' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1Q2PP/2KR1BNR b - - 1 8',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 1 8' => {
			eco => 'E83i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Qd2

			variation => N__('King\'s Indian: Sämisch, Panno, 8. Qd2'),
			moves => {
				'a8b8' => '1rbq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 2 9',
				'f8e8' => 'r1bqr1k1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 2 9',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 0 8' => {
			eco => 'E83f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6

			variation => N__('King\'s Indian: Sämisch, Panno'),
			moves => {
				'a2a3' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/P1N1BP2/1P2N1PP/R2QKB1R b KQ - 0 8',
				'd1d2' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 1 8',
				'e2c1' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP4PP/R1NQKB1R b KQ - 1 8',
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP4PP/R1NQKB1R b KQ - 1 8' => {
			eco => 'E83h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. Nc1

			variation => N__('King\'s Indian: Sämisch, Panno, 8. Nc1'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 8' => {
			eco => 'E63c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. e4

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. e4'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/P1N1BP2/1P2N1PP/R2QKB1R b KQ - 0 8' => {
			eco => 'E83g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2 a6 8. a3

			variation => N__('King\'s Indian: Sämisch, Panno, 8. a3'),
			moves => {
			},
		},
		'r1bq1rk1/1pp1ppbp/p2p1np1/n2P4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E63m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O a6 8. d5 Na5

			variation => N__('King\'s Indian: Fianchetto, Panno, 8. d5 Na5'),
			moves => {
			},
		},
		'r1bq1rk1/1pp2pb1/2np1npp/p3p3/2P5/P1NP1NP1/1P2PPBP/1RBQ1RK1 w - - 0 10' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1 a5 9. a3 h6

			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1 a5 9. a3 h6'),
			moves => {
			},
		},
		'r1bq1rk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ - 2 10' => {
			eco => 'E92t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 a5 8. Bg5 h6 9. Bh4 Na6

			variation => N__('King\'s Indian: Petrosian, Stein, 8. Bg5 h6 9. Bh4 Na6'),
			moves => {
				'f3d2' => 'r1bq1rk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2QK2R b KQ - 3 10',
			},
		},
		'r1bq1rk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2QK2R b KQ - 3 10' => {
			eco => 'E92u',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 a5 8. Bg5 h6 9. Bh4 Na6 10. Nd2

			variation => N__('King\'s Indian: Petrosian, Stein, Main Line'),
			moves => {
				'd8e8' => 'r1b1qrk1/1pp2pb1/n2p1npp/p2Pp3/2P1P2B/2N5/PP1NBPPP/R2QK2R w KQ - 4 11',
			},
		},
		'r1bq1rk1/1pp2pbp/2np1np1/p3p3/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 w - a6 0 9' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1 a5

			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1 a5'),
			moves => {
				'a2a3' => 'r1bq1rk1/1pp2pbp/2np1np1/p3p3/2P5/P1NP1NP1/1P2PPBP/1RBQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/1pp2pbp/2np1np1/p3p3/2P5/P1NP1NP1/1P2PPBP/1RBQ1RK1 b - - 0 9' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1 a5 9. a3

			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1 a5 9. a3'),
			moves => {
				'h7h6' => 'r1bq1rk1/1pp2pb1/2np1npp/p3p3/2P5/P1NP1NP1/1P2PPBP/1RBQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/1pp2pbp/3p1np1/p1nPp3/2P1P3/2N2N2/PPQ1BPPP/R1B2RK1 w - a6 0 10' => {
			eco => 'E94r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. d5 Nc5 9. Qc2 a5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. d5 Nc5 9. Qc2 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1pp2ppp/p1n1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E51',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6 7. O-O a6

			variation => N__('Nimzo-Indian: 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6 7.O-O a6'),
			moves => {
			},
		},
		'r1bq1rk1/1ppn1pb1/3p1npp/p2Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ a6 0 10' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5 h6 9. Bh4 a5

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6 9. Bh4 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1ppn1pb1/p2p1npp/3Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ - 0 10' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5 h6 9. Bh4 a6

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6 9. Bh4 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1ppn1pbp/3p1np1/p3p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - a6 0 9' => {
			eco => 'E95b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 a5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 a5'),
			moves => {
			},
		},
		'r1bq1rk1/1ppn1pbp/p2p1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E68i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 a6

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnb1pp/p1p2p2/4p3/2N1P2N/3P4/PPP2PPP/R1BQ1RK1 b - - 1 10' => {
			eco => 'C85p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7 8. Nbd2 O-O 9. Nc4 f6 10. Nh4

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2 O-O 9. Nc4 f6 10. Nh4'),
			moves => {
				'd7c5' => 'r1bq1rk1/1pp1b1pp/p1p2p2/2n1p3/2N1P2N/3P4/PPP2PPP/R1BQ1RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/1ppnb1pp/p1p2p2/4p3/2N1P3/3P1N2/PPP2PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'C85o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7 8. Nbd2 O-O 9. Nc4 f6

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2 O-O 9. Nc4 f6'),
			moves => {
				'f3h4' => 'r1bq1rk1/1ppnb1pp/p1p2p2/4p3/2N1P2N/3P4/PPP2PPP/R1BQ1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/1ppnbppp/p1p5/4p3/2N1P3/3P1N2/PPP2PPP/R1BQ1RK1 b - - 4 9' => {
			eco => 'C85n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7 8. Nbd2 O-O 9. Nc4

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2 O-O 9. Nc4'),
			moves => {
				'f7f6' => 'r1bq1rk1/1ppnb1pp/p1p2p2/4p3/2N1P3/3P1N2/PPP2PPP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/1ppnbppp/p1p5/4p3/4P3/3P1N2/PPPN1PPP/R1BQ1RK1 w - - 3 9' => {
			eco => 'C85m',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2'),
			moves => {
				'd2c4' => 'r1bq1rk1/1ppnbppp/p1p5/4p3/2N1P3/3P1N2/PPP2PPP/R1BQ1RK1 b - - 4 9',
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/2Pp2B1/3P4/2N1PN2/PP3PPP/2RQKB1R b K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. c5

			variation => N__('QGD: Orthodox, Swiss, 8. c5'),
			moves => {
				'c7c6' => 'r1bq1rk1/1p1nbppp/p1p1pn2/2Pp2B1/3P4/2N1PN2/PP3PPP/2RQKB1R w K - 0 9',
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/3P2B1/3P4/2N1PN2/PP3PPP/2RQKB1R b K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. cxd5

			variation => N__('QGD: Orthodox, Swiss, Karlsbad Variation'),
			moves => {
				'e6d5' => 'r1bq1rk1/1ppnbppp/p4n2/3p2B1/3P4/2N1PN2/PP3PPP/2RQKB1R w K - 0 9',
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6

			variation => N__('QGD: Orthodox, Swiss Variation'),
			moves => {
				'a2a3' => 'r1bq1rk1/1ppnbppp/p3pn2/3p2B1/2PP4/P1N1PN2/1P3PPP/2RQKB1R b K - 0 8',
				'c4c5' => 'r1bq1rk1/1ppnbppp/p3pn2/2Pp2B1/3P4/2N1PN2/PP3PPP/2RQKB1R b K - 0 8',
				'c4d5' => 'r1bq1rk1/1ppnbppp/p3pn2/3P2B1/3P4/2N1PN2/PP3PPP/2RQKB1R b K - 0 8',
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 a6

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/3p2B1/2PP4/P1N1PN2/1P3PPP/2RQKB1R b K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. a3

			variation => N__('QGD: Orthodox, Swiss, 8. a3'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/6B1/2BP4/2N1PN2/PP3PPP/2RQK2R w K - 0 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 dxc4 8. Bxc4 a6

			variation => N__('QGD: Orthodox, 7. Rc1 dxc4 8.Bxc4 a6'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnbppp/p3pn2/6B1/2BP4/2N1PN2/PP3PPP/R2QK2R w KQ - 0 9' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 dxc4 8. Bxc4 a6

			variation => N__('QGD: Orthodox, Botvinnik, 8... a6'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnbppp/p4n2/3p2B1/3P4/2N1PN2/PP3PPP/2RQKB1R w K - 0 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. cxd5 exd5

			variation => N__('QGD: Orthodox, Swiss, Karlsbad Variation'),
			moves => {
				'f1d3' => 'r1bq1rk1/1ppnbppp/p4n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R b K - 1 9',
			},
		},
		'r1bq1rk1/1ppnbppp/p4n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R b K - 1 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 a6 8. cxd5 exd5 9. Bd3

			variation => N__('QGD: Orthodox, Swiss, Karlsbad, 9. Bd3'),
			moves => {
				'c7c6' => 'r1bq1rk1/1p1nbppp/p1p2n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R w K - 0 10',
				'f8e8' => 'r1bqr1k1/1ppnbppp/p4n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R w K - 2 10',
			},
		},
		'r1bq1rk1/1ppnn1bp/3p2p1/p2Ppp2/1PP1P3/P1N5/3NBPPP/1RBQ1RK1 b - b3 0 12' => {
			eco => 'E97z',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 a5 10. a3 Nd7 11. Rb1 f5 12. b4

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5 10. a3 Nd7 11. Rb1 f5 12. b4'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnn1bp/3p2p1/p2Ppp2/2P1P3/P1N5/1P1NBPPP/1RBQ1RK1 w - f6 0 12' => {
			eco => 'E97z',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5 10. a3 Nd7 11. Rb1'),
			moves => {
				'b2b4' => 'r1bq1rk1/1ppnn1bp/3p2p1/p2Ppp2/1PP1P3/P1N5/3NBPPP/1RBQ1RK1 b - b3 0 12',
			},
		},
		'r1bq1rk1/1ppnnpbp/3p2p1/3Pp3/1BP1P3/2N2N2/P3BPPP/R2Q1RK1 w - - 1 12' => {
			eco => 'E97k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 a5 10. Ba3 axb4 11. Bxb4 Nd7

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... a5 10. Ba3 axb4 11.Bxb4 Nd7'),
			moves => {
			},
		},
		'r1bq1rk1/1ppnnpbp/3p2p1/p2Pp3/2P1P3/P1N5/1P1NBPPP/1RBQ1RK1 b - - 2 11' => {
			eco => 'E97z',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 a5 10. a3 Nd7 11. Rb1

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5 10. a3 Nd7 11. Rb1'),
			moves => {
				'f7f5' => 'r1bq1rk1/1ppnn1bp/3p2p1/p2Ppp2/2P1P3/P1N5/1P1NBPPP/1RBQ1RK1 w - f6 0 12',
			},
		},
		'r1bq1rk1/1ppnnpbp/3p2p1/p2Pp3/2P1P3/P1N5/1P1NBPPP/R1BQ1RK1 w - - 1 11' => {
			eco => 'E97z',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 a5 10. a3 Nd7

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 a5 10. a3 Nd7'),
			moves => {
				'a1b1' => 'r1bq1rk1/1ppnnpbp/3p2p1/p2Pp3/2P1P3/P1N5/1P1NBPPP/1RBQ1RK1 b - - 2 11',
			},
		},
		'r1bq1rk1/1pppbppp/p1n2n2/4P3/B2p4/5N2/PPP2PPP/RNBQR1K1 b - - 0 8' => {
			eco => 'C84o',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1 O-O'),
			moves => {
				'f6e8' => 'r1bqnrk1/1pppbppp/p1n5/4P3/B2p4/5N2/PPP2PPP/RNBQR1K1 w - - 1 9',
			},
		},
		'r1bq1rk1/1pppbppp/p1n2n2/8/B2pP3/5N2/PPP2PPP/RNBQR1K1 w - - 2 8' => {
			eco => 'C84o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. Re1 O-O

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1 O-O'),
			moves => {
				'e4e5' => 'r1bq1rk1/1pppbppp/p1n2n2/4P3/B2p4/5N2/PPP2PPP/RNBQR1K1 b - - 0 8',
			},
		},
		'r1bq1rk1/1pppbppp/p1n5/4P3/B2Nn3/8/PPP2PPP/RNBQ1RK1 w - - 1 9' => {
			eco => 'C84k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4 8. Nxd4 O-O

			variation => N__('Spanish: Closed, Centre Attack, 7. e5 Ne4 8.Nxd4 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 b - d3 0 10' => {
			eco => 'C93',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 h6 10. d4

			variation => N__('Spanish: Closed, Smyslov, 10. d4'),
			moves => {
				'f8e8' => 'r1bqr1k1/2p1bpp1/p1np1n1p/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 w - - 1 11',
			},
		},
		'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 0 10' => {
			eco => 'C93',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 h6

			variation => N__('Spanish: Closed, Smyslov Defence'),
			moves => {
				'd2d3' => 'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/4P3/1BPP1N1P/PP3PP1/RNBQR1K1 b - - 0 10',
				'd2d4' => 'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 b - d3 0 10',
			},
		},
		'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/4P3/1BPP1N1P/PP3PP1/RNBQR1K1 b - - 0 10' => {
			eco => 'C93',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 h6 10. d3

			variation => N__('Spanish: Closed, Smyslov, 10. d3'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/4P3/1BPP1N2/PP3PPP/RNBQR1K1 w - - 0 10' => {
			eco => 'C90f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 h6

			variation => N__('Spanish: Closed, Pilnik, 9... h6'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/2np1n2/pp2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 0 10' => {
			eco => 'C92b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 a5

			variation => N__('Spanish: Closed, Keres Variation'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1n2n2/1p1P4/4p3/1BP2N2/PP1P1PPP/RNBQR1K1 w - - 0 10' => {
			eco => 'C89b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 e4

			variation => N__('Spanish: Marshall, Herman Steiner Variation'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1n2n2/1p1Pp3/8/1BP2N2/PP1P1PPP/RNBQR1K1 b - - 0 9' => {
			eco => 'C89b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5

			variation => N__('Spanish: Marshall, 9.exd5'),
			moves => {
				'e5e4' => 'r1bq1rk1/2p1bppp/p1n2n2/1p1P4/4p3/1BP2N2/PP1P1PPP/RNBQR1K1 w - - 0 10',
				'f6d5' => 'r1bq1rk1/2p1bppp/p1n5/1p1np3/8/1BP2N2/PP1P1PPP/RNBQR1K1 w - - 0 10',
			},
		},
		'r1bq1rk1/2p1bppp/p1n2n2/1p1Pp3/8/1BP2N2/PP1PQPPP/RNB2RK1 b - - 0 9' => {
			eco => 'C86j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3 d5 9. exd5

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3 d5 9.exd5'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1n2n2/1p1pp3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 w - d6 0 9' => {
			eco => 'C89a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5

			variation => N__('Spanish: Marshall Counterattack'),
			moves => {
				'e4d5' => 'r1bq1rk1/2p1bppp/p1n2n2/1p1Pp3/8/1BP2N2/PP1P1PPP/RNBQR1K1 b - - 0 9',
			},
		},
		'r1bq1rk1/2p1bppp/p1n2n2/1p1pp3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 w - d6 0 9' => {
			eco => 'C86j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3 d5

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3 d5'),
			moves => {
				'd2d3' => 'r1bq1rk1/2p1bppp/p1n2n2/1p1pp3/4P3/1BPP1N2/PP2QPPP/RNB2RK1 b - - 0 9',
				'e4d5' => 'r1bq1rk1/2p1bppp/p1n2n2/1p1Pp3/8/1BP2N2/PP1PQPPP/RNB2RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/2p1bppp/p1n2n2/1p1pp3/4P3/1BPP1N2/PP2QPPP/RNB2RK1 b - - 0 9' => {
			eco => 'C86k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3 d5 9. d3

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3 d5 9. d3'),
			moves => {
				'c8b7' => 'r2q1rk1/1bp1bppp/p1n2n2/1p1pp3/4P3/1BPP1N2/PP2QPPP/RNB2RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/2p1bppp/p1n5/1p1nN3/8/1BP5/PP1P1PPP/RNBQR1K1 b - - 0 10' => {
			eco => 'C89d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5

			variation => N__('Spanish: Marshall, 9.exd5 Nxd5 10.Nxe5'),
			moves => {
				'c6e5' => 'r1bq1rk1/2p1bppp/p7/1p1nn3/8/1BP5/PP1P1PPP/RNBQR1K1 w - - 0 11',
			},
		},
		'r1bq1rk1/2p1bppp/p1n5/1p1np3/8/1BP2N2/PP1P1PPP/RNBQR1K1 w - - 0 10' => {
			eco => 'C89c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5

			variation => N__('Spanish: Marshall, 9.exd5 Nxd5'),
			moves => {
				'f3e5' => 'r1bq1rk1/2p1bppp/p1n5/1p1nN3/8/1BP5/PP1P1PPP/RNBQR1K1 b - - 0 10',
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/3PP3/1BP2N2/PP2QPPP/RNB2RK1 b - d3 0 9' => {
			eco => 'C86h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3 d6 9. d4

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3 d6 9. d4'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/3PP3/1BP2N2/PP3PPP/RNBQR1K1 b - d3 0 9' => {
			eco => 'C91a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d4

			variation => N__('Spanish: Closed, 9. d4'),
			moves => {
				'c8g4' => 'r2q1rk1/2p1bppp/p1np1n2/1p2p3/3PP1b1/1BP2N2/PP3PPP/RNBQR1K1 w - - 1 10',
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1B1P1N2/PPP2PPP/RNBQR1K1 w - - 0 9' => {
			eco => 'C88g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. d3 d6

			variation => N__('Spanish: Closed, 8. d3 d6'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 b - - 0 9' => {
			eco => 'C92a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3

			variation => N__('Spanish: Closed, 9. h3'),
			moves => {
				'a6a5' => 'r1bq1rk1/2p1bppp/2np1n2/pp2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 0 10',
				'c6a5' => 'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10',
				'c6b8' => 'rnbq1rk1/2p1bppp/p2p1n2/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10',
				'c8b7' => 'r2q1rk1/1bp1bppp/p1np1n2/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10',
				'c8e6' => 'r2q1rk1/2p1bppp/p1npbn2/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10',
				'f6d7' => 'r1bq1rk1/2pnbppp/p1np4/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10',
				'h7h6' => 'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 0 10',
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 w - - 0 9' => {
			eco => 'C90a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6

			variation => N__('Spanish: Closed, 8... d6'),
			moves => {
				'a2a3' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/PBP2N2/1P1P1PPP/RNBQR1K1 b - - 0 9',
				'a2a4' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/P3P3/1BP2N2/1P1P1PPP/RNBQR1K1 b - a3 0 9',
				'b3c2' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/2P2N2/PPBP1PPP/RNBQR1K1 b - - 1 9',
				'd2d3' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BPP1N2/PP3PPP/RNBQR1K1 b - - 0 9',
				'd2d4' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/3PP3/1BP2N2/PP3PPP/RNBQR1K1 b - d3 0 9',
				'h2h3' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 b - - 0 9',
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 w - - 0 9' => {
			eco => 'C86g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3 d6

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3 d6'),
			moves => {
				'd2d4' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/3PP3/1BP2N2/PP2QPPP/RNB2RK1 b - d3 0 9',
				'f1d1' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNBR2K1 b - - 1 9',
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNBR2K1 b - - 1 9' => {
			eco => 'C86i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3 d6 9. Rd1

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3 d6 9. Rd1'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BPP1N2/PP3PPP/RNBQR1K1 b - - 0 9' => {
			eco => 'C90e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3

			variation => N__('Spanish: Closed, Pilnik Variation'),
			moves => {
				'c6a5' => 'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/1BPP1N2/PP3PPP/RNBQR1K1 w - - 1 10',
				'h7h6' => 'r1bq1rk1/2p1bpp1/p1np1n1p/1p2p3/4P3/1BPP1N2/PP3PPP/RNBQR1K1 w - - 0 10',
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/2P2N2/PPBP1PPP/RNBQR1K1 b - - 1 9' => {
			eco => 'C90b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. Bc2

			variation => N__('Spanish: Closed, Lutikov Variation'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/PBP2N2/1P1P1PPP/RNBQR1K1 b - - 0 9' => {
			eco => 'C90c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. a3

			variation => N__('Spanish: Closed, Suetin Variation'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/P3P3/1BP2N2/1P1P1PPP/RNBQR1K1 b - a3 0 9' => {
			eco => 'C90d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. a4

			variation => N__('Spanish: Closed, 8... d6 9. a4'),
			moves => {
				'c8g4' => 'r2q1rk1/2p1bppp/p1np1n2/1p2p3/P3P1b1/1BP2N2/1P1P1PPP/RNBQR1K1 w - - 1 10',
			},
		},
		'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10' => {
			eco => 'C96a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5

			variation => N__('Spanish: Closed, Chigorin'),
			moves => {
				'b3c2' => 'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 b - - 2 10',
			},
		},
		'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/1BPP1N2/PP3PPP/RNBQR1K1 w - - 1 10' => {
			eco => 'C90g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 Na5

			variation => N__('Spanish: Closed, Pilnik, 9... Na5'),
			moves => {
				'b3c2' => 'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/2PP1N2/PPB2PPP/RNBQR1K1 b - - 2 10',
			},
		},
		'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 b - - 2 10' => {
			eco => 'C96b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2

			variation => N__('Spanish: Closed, Chigorin, 10. Bc2'),
			moves => {
				'c7c5' => 'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 w - c6 0 11',
				'c7c6' => 'r1bq1rk1/4bppp/p1pp1n2/np2p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 w - - 0 11',
				'c8b7' => 'r2q1rk1/1bp1bppp/p2p1n2/np2p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 w - - 3 11',
			},
		},
		'r1bq1rk1/2p1bppp/p2p1n2/np2p3/4P3/2PP1N2/PPB2PPP/RNBQR1K1 b - - 2 10' => {
			eco => 'C90g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 Na5 10. Bc2

			variation => N__('Spanish: Closed, Pilnik, 9... Na5'),
			moves => {
				'c7c5' => 'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPB2PPP/RNBQR1K1 w - c6 0 11',
			},
		},
		'r1bq1rk1/2p1bppp/p4n2/1p2R3/8/1BP5/PP1P1PPP/RNBQ2K1 w - - 1 12' => {
			eco => 'C89e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 Nf6

			variation => N__('Spanish: Marshall, 11.Rxe5 Nf6'),
			moves => {
			},
		},
		'r1bq1rk1/2p1bppp/p7/1p1nR3/8/1BP5/PP1P1PPP/RNBQ2K1 b - - 0 11' => {
			eco => 'C89d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5

			variation => N__('Spanish: Marshall, 11.Rxe5'),
			moves => {
				'c7c6' => 'r1bq1rk1/4bppp/p1p5/1p1nR3/8/1BP5/PP1P1PPP/RNBQ2K1 w - - 0 12',
				'd5f6' => 'r1bq1rk1/2p1bppp/p4n2/1p2R3/8/1BP5/PP1P1PPP/RNBQ2K1 w - - 1 12',
			},
		},
		'r1bq1rk1/2p1bppp/p7/1p1nn3/8/1BP5/PP1P1PPP/RNBQR1K1 w - - 0 11' => {
			eco => 'C89d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5

			variation => N__('Spanish: Marshall, 9.exd5 Nxd5 10.Nxe5 Nxe5'),
			moves => {
				'e1e5' => 'r1bq1rk1/2p1bppp/p7/1p1nR3/8/1BP5/PP1P1PPP/RNBQ2K1 b - - 0 11',
			},
		},
		'r1bq1rk1/2pn1ppp/p1np1b2/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 w - - 1 11' => {
			eco => 'C92f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nd7 10. d4 Bf6

			variation => N__('Spanish: Closed, Karpov Variation, 10. d4 Bf6'),
			moves => {
				'a2a4' => 'r1bq1rk1/2pn1ppp/p1np1b2/1p2p3/P2PP3/1BP2N1P/1P3PP1/RNBQR1K1 b - a3 0 11',
			},
		},
		'r1bq1rk1/2pn1ppp/p1np1b2/1p2p3/P2PP3/1BP2N1P/1P3PP1/RNBQR1K1 b - a3 0 11' => {
			eco => 'C92g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nd7 10. d4 Bf6 11. a4

			variation => N__('Spanish: Closed, Karpov Variation, 10. d4 Bf6 11. a4'),
			moves => {
				'c8b7' => 'r2q1rk1/1bpn1ppp/p1np1b2/1p2p3/P2PP3/1BP2N1P/1P3PP1/RNBQR1K1 w - - 1 12',
			},
		},
		'r1bq1rk1/2pnbppp/p1np4/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 b - d3 0 10' => {
			eco => 'C92e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Karpov Variation'),
			moves => {
				'e7f6' => 'r1bq1rk1/2pn1ppp/p1np1b2/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 w - - 1 11',
			},
		},
		'r1bq1rk1/2pnbppp/p1np4/1p2p3/4P3/1BP2N1P/PP1P1PP1/RNBQR1K1 w - - 1 10' => {
			eco => 'C92e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nd7

			variation => N__('Spanish: Closed, Karpov Variation'),
			moves => {
				'd2d4' => 'r1bq1rk1/2pnbppp/p1np4/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 b - d3 0 10',
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p1B1/3PP3/1BP2N1P/PP3PP1/RN1QR1K1 b - - 2 11' => {
			eco => 'C95d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d4 Nbd7 11. Bg5

			variation => N__('Spanish: Closed, Breyer, 10. d4 Nbd7 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/2PPP3/1B3N1P/PP3PP1/RNBQR1K1 b - - 0 11' => {
			eco => 'C95e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d4 Nbd7 11. c4

			variation => N__('Spanish: Closed, Breyer, Arseniev Variation'),
			moves => {
				'c7c6' => 'r1bq1rk1/3nbppp/p1pp1n2/1p2p3/2PPP3/1B3N1P/PP3PP1/RNBQR1K1 w - - 0 12',
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/3PP2N/1BP4P/PP3PP1/RNBQR1K1 b - - 2 11' => {
			eco => 'C95c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d4 Nbd7 11. Nh4

			variation => N__('Spanish: Closed, Breyer, Simagin Variation'),
			moves => {
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/3PP3/1BP2N1P/PP1N1PP1/R1BQR1K1 b - - 2 11' => {
			eco => 'C95f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d4 Nbd7 11. Nbd2

			variation => N__('Spanish: Closed, Breyer, 10. d4 Nbd7 11.Nbd2'),
			moves => {
				'c8b7' => 'r2q1rk1/1bpnbppp/p2p1n2/1p2p3/3PP3/1BP2N1P/PP1N1PP1/R1BQR1K1 w - - 3 12',
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/3PP3/1BP2N1P/PP3PP1/RNBQR1K1 w - - 1 11' => {
			eco => 'C95b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d4 Nbd7

			variation => N__('Spanish: Closed, Breyer, 10. d4 Nbd7'),
			moves => {
				'b1d2' => 'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/3PP3/1BP2N1P/PP1N1PP1/R1BQR1K1 b - - 2 11',
				'c1g5' => 'r1bq1rk1/2pnbppp/p2p1n2/1p2p1B1/3PP3/1BP2N1P/PP3PP1/RN1QR1K1 b - - 2 11',
				'c3c4' => 'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/2PPP3/1B3N1P/PP3PP1/RNBQR1K1 b - - 0 11',
				'f3h4' => 'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/3PP2N/1BP4P/PP3PP1/RNBQR1K1 b - - 2 11',
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/4P3/1BPP1N1P/PP1N1PP1/R1BQR1K1 b - - 2 11' => {
			eco => 'C94e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d3 Nbd7 11. Nbd2

			variation => N__('Spanish: Closed, Breyer, 10. d3 Nbd7 11.Nbd2'),
			moves => {
				'c8b7' => 'r2q1rk1/1bpnbppp/p2p1n2/1p2p3/4P3/1BPP1N1P/PP1N1PP1/R1BQR1K1 w - - 3 12',
			},
		},
		'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/4P3/1BPP1N1P/PP3PP1/RNBQR1K1 w - - 1 11' => {
			eco => 'C94d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d3 Nbd7

			variation => N__('Spanish: Closed, Breyer, 10. d3 Nbd7'),
			moves => {
				'b1d2' => 'r1bq1rk1/2pnbppp/p2p1n2/1p2p3/4P3/1BPP1N1P/PP1N1PP1/R1BQR1K1 b - - 2 11',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B1P1N2/PPP2PPP/RNBQR1K1 b - - 0 8' => {
			eco => 'C88g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. d3

			variation => N__('Spanish: Closed, 8. d3'),
			moves => {
				'd7d6' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1B1P1N2/PPP2PPP/RNBQR1K1 w - - 0 9',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B3N1P/PPPP1PP1/RNBQR1K1 b - - 0 8' => {
			eco => 'C88h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. h3

			variation => N__('Spanish: Closed, 8. h3'),
			moves => {
				'c8b7' => 'r2q1rk1/1bppbppp/p1n2n2/1p2p3/4P3/1B3N1P/PPPP1PP1/RNBQR1K1 w - - 1 9',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 w - - 2 8' => {
			eco => 'C88f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O

			variation => N__('Spanish: Closed, 7...O-O'),
			moves => {
				'a2a4' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/P3P3/1B3N2/1PPP1PPP/RNBQR1K1 b - a3 0 8',
				'c2c3' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 b - - 0 8',
				'd2d3' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B1P1N2/PPP2PPP/RNBQR1K1 b - - 0 8',
				'h2h3' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B3N1P/PPPP1PP1/RNBQR1K1 b - - 0 8',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB2RK1 w - - 2 8' => {
			eco => 'C86e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O

			variation => N__('Spanish: Worrall Attack, 7...O-O'),
			moves => {
				'a2a4' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/P3P3/1B3N2/1PPPQPPP/RNB2RK1 b - a3 0 8',
				'c2c3' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 b - - 0 8' => {
			eco => 'C88q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3

			variation => N__('Spanish: Closed, 8. c3'),
			moves => {
				'd7d5' => 'r1bq1rk1/2p1bppp/p1n2n2/1p1pp3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 w - d6 0 9',
				'd7d6' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 w - - 0 9',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 b - - 0 8' => {
			eco => 'C86f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. c3

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. c3'),
			moves => {
				'd7d5' => 'r1bq1rk1/2p1bppp/p1n2n2/1p1pp3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 w - d6 0 9',
				'd7d6' => 'r1bq1rk1/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/P3P3/1B3N2/1PPP1PPP/RNBQR1K1 b - a3 0 8' => {
			eco => 'C88j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. a4

			variation => N__('Spanish: Closed, Anti-Marshall 8. a4'),
			moves => {
				'b5b4' => 'r1bq1rk1/2ppbppp/p1n2n2/4p3/Pp2P3/1B3N2/1PPP1PPP/RNBQR1K1 w - - 0 9',
				'c8b7' => 'r2q1rk1/1bppbppp/p1n2n2/1p2p3/P3P3/1B3N2/1PPP1PPP/RNBQR1K1 w - - 1 9',
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/P3P3/1B3N2/1PPPQPPP/RNB2RK1 b - a3 0 8' => {
			eco => 'C86e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 O-O 8. a4

			variation => N__('Spanish: Worrall Attack, 7...O-O 8. a4'),
			moves => {
			},
		},
		'r1bq1rk1/2ppbppp/p1n2n2/4p3/Pp2P3/1B3N2/1PPP1PPP/RNBQR1K1 w - - 0 9' => {
			eco => 'C88k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. a4 b4

			variation => N__('Spanish: Closed, Anti-Marshall 8. a4 b4'),
			moves => {
			},
		},
		'r1bq1rk1/3n1ppp/p1pbpn2/1p6/2BP4/2N1PN2/PPQ2PPP/R1BR2K1 w - b6 0 11' => {
			eco => 'D46k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 a6 10. Rd1 b5

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... a6 10. Rd1 b5'),
			moves => {
			},
		},
		'r1bq1rk1/3nbppp/1pp1pn2/p2p4/2PP4/1P3NP1/P1QNPPBP/R1B2RK1 w - a6 0 10' => {
			eco => 'E09e',
                        # TRANSLATORS:

			variation => N__('Catalan: Closed, Main Line, 9. b3'),
			moves => {
				'c1b2' => 'r1bq1rk1/3nbppp/1pp1pn2/p2p4/2PP4/1P3NP1/PBQNPPBP/R4RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/3nbppp/1pp1pn2/p2p4/2PP4/1P3NP1/PBQNPPBP/R4RK1 b - - 1 10' => {
			eco => 'E09e',
                        # TRANSLATORS:

			variation => N__('Catalan: Closed, Main Line, 9. b3'),
			moves => {
				'c8a6' => 'r2q1rk1/3nbppp/bpp1pn2/p2p4/2PP4/1P3NP1/PBQNPPBP/R4RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/3nbppp/p1p1pn2/1p4B1/3P4/2NBPN2/PP3PPP/2RQ1RK1 b - - 1 11' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 b5 10. Bd3 a6 11. O-O

			variation => N__('QGD: Orthodox, Main Line, Fianchetto, 11.O-O'),
			moves => {
			},
		},
		'r1bq1rk1/3nbppp/p1p1pn2/1p4B1/3P4/2NBPN2/PP3PPP/2RQK2R w K - 0 11' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 b5 10. Bd3 a6

			variation => N__('QGD: Orthodox, Main Line, Fianchetto Variation'),
			moves => {
				'e1g1' => 'r1bq1rk1/3nbppp/p1p1pn2/1p4B1/3P4/2NBPN2/PP3PPP/2RQ1RK1 b - - 1 11',
				'e3e4' => 'r1bq1rk1/3nbppp/p1p1pn2/1p4B1/3PP3/2NB1N2/PP3PPP/2RQK2R b K - 0 11',
			},
		},
		'r1bq1rk1/3nbppp/p1p1pn2/1p4B1/3PP3/2NB1N2/PP3PPP/2RQK2R b K - 0 11' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 b5 10. Bd3 a6 11. e4

			variation => N__('QGD: Orthodox, Main Line, Fianchetto, 11. e4'),
			moves => {
			},
		},
		'r1bq1rk1/3nbppp/p1pp1n2/1p2p3/2PPP3/1B3N1P/PP3PP1/RNBQR1K1 w - - 0 12' => {
			eco => 'C95e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Nb8 10. d4 Nbd7 11. c4 c6

			variation => N__('Spanish: Closed, Breyer, Arseniev, 11... c6'),
			moves => {
			},
		},
		'r1bq1rk1/3nbppp/p2p4/np2p3/3pP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 0 13' => {
			eco => 'C96o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Nd7 12. Nbd2 cxd4

			variation => N__('Spanish: Closed, Chigorin, Keres, 11.Nbd2 cxd4'),
			moves => {
			},
		},
		'r1bq1rk1/3nbppp/p2p4/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12' => {
			eco => 'C96n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Nd7

			variation => N__('Spanish: Closed, Chigorin, Keres Variation'),
			moves => {
				'b1d2' => 'r1bq1rk1/3nbppp/p2p4/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 2 12',
			},
		},
		'r1bq1rk1/3nbppp/p2p4/npp1p3/3PP3/2P2N1P/PPBN1PP1/R1BQR1K1 b - - 2 12' => {
			eco => 'C96n',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Chigorin, Keres Variation'),
			moves => {
				'c5d4' => 'r1bq1rk1/3nbppp/p2p4/np2p3/3pP3/2P2N1P/PPBN1PP1/R1BQR1K1 w - - 0 13',
			},
		},
		'r1bq1rk1/4bppp/p1np1n2/1pp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12' => {
			eco => 'C96m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 Nc6

			variation => N__('Spanish: Closed, Chigorin, Borisenko Variation'),
			moves => {
			},
		},
		'r1bq1rk1/4bppp/p1np1n2/1pp1p3/4P3/2PP1N1P/PPB2PP1/RNBQR1K1 w - - 1 12' => {
			eco => 'C96h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d3 Nc6

			variation => N__('Spanish: Closed, Chigorin, 11. d3 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/4bppp/p1np1n2/1pp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 w - - 2 12' => {
			eco => 'C90j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 Na5 10. Bc2 c5 11. Nbd2 Nc6

			variation => N__('Spanish: Closed, Pilnik, 11.Nbd2 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/4bppp/p1p5/1p1BR3/8/2P5/PP1P1PPP/RNBQ2K1 b - - 0 12' => {
			eco => 'C89g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. Bxd5

			variation => N__('Spanish: Marshall, 12.Bxd5'),
			moves => {
				'c6d5' => 'r1bq1rk1/4bppp/p7/1p1pR3/8/2P5/PP1P1PPP/RNBQ2K1 w - - 0 13',
			},
		},
		'r1bq1rk1/4bppp/p1p5/1p1nR3/3P4/1BP5/PP3PPP/RNBQ2K1 b - d3 0 12' => {
			eco => 'C89k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4

			variation => N__('Spanish: Marshall, Main Line (12. d4)'),
			moves => {
				'e7d6' => 'r1bq1rk1/5ppp/p1pb4/1p1nR3/3P4/1BP5/PP3PPP/RNBQ2K1 w - - 1 13',
			},
		},
		'r1bq1rk1/4bppp/p1p5/1p1nR3/8/1BP5/PP1P1PPP/RNBQ2K1 w - - 0 12' => {
			eco => 'C89f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6

			variation => N__('Spanish: Marshall, 11.Rxe5 c6'),
			moves => {
				'b3d5' => 'r1bq1rk1/4bppp/p1p5/1p1BR3/8/2P5/PP1P1PPP/RNBQ2K1 b - - 0 12',
				'd2d3' => 'r1bq1rk1/4bppp/p1p5/1p1nR3/8/1BPP4/PP3PPP/RNBQ2K1 b - - 0 12',
				'd2d4' => 'r1bq1rk1/4bppp/p1p5/1p1nR3/3P4/1BP5/PP3PPP/RNBQ2K1 b - d3 0 12',
			},
		},
		'r1bq1rk1/4bppp/p1p5/1p1nR3/8/1BPP4/PP3PPP/RNBQ2K1 b - - 0 12' => {
			eco => 'C89i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d3

			variation => N__('Spanish: Marshall, 12. d3'),
			moves => {
				'e7d6' => 'r1bq1rk1/5ppp/p1pb4/1p1nR3/8/1BPP4/PP3PPP/RNBQ2K1 w - - 1 13',
			},
		},
		'r1bq1rk1/4bppp/p1pp1n2/np1Pp3/4P3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 13' => {
			eco => 'C91i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d4 Bg4 10. d5 Na5 11. Bc2 c6 12. h3 Bc8

			variation => N__('Spanish: Closed, Bogoljubow, 10. d5 Na5 11. Bc2 c6 12. h3 Bc8'),
			moves => {
			},
		},
		'r1bq1rk1/4bppp/p1pp1n2/np2p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 b - d3 0 11' => {
			eco => 'C96c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Chigorin, 10... c6'),
			moves => {
				'd8c7' => 'r1b2rk1/2q1bppp/p1pp1n2/np2p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12',
			},
		},
		'r1bq1rk1/4bppp/p1pp1n2/np2p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 w - - 0 11' => {
			eco => 'C96c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c6

			variation => N__('Spanish: Closed, Chigorin, 10... c6'),
			moves => {
				'd2d4' => 'r1bq1rk1/4bppp/p1pp1n2/np2p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 b - d3 0 11',
			},
		},
		'r1bq1rk1/4bppp/p2p1n2/np2p3/3pP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 0 12' => {
			eco => 'C96j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4 cxd4

			variation => N__('Spanish: Closed, Chigorin, 11. d4 cxd4'),
			moves => {
			},
		},
		'r1bq1rk1/4bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 b - d3 0 11' => {
			eco => 'C96i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d4

			variation => N__('Spanish: Closed, Chigorin, 11. d4'),
			moves => {
				'a5c6' => 'r1bq1rk1/4bppp/p1np1n2/1pp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12',
				'c5d4' => 'r1bq1rk1/4bppp/p2p1n2/np2p3/3pP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 0 12',
				'c8b7' => 'r2q1rk1/1b2bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12',
				'd8c7' => 'r1b2rk1/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12',
				'f6d7' => 'r1bq1rk1/3nbppp/p2p4/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 w - - 1 12',
			},
		},
		'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2P2N1P/PPBP1PP1/RNBQR1K1 w - c6 0 11' => {
			eco => 'C96f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5

			variation => N__('Spanish: Closed, Chigorin, 10... c5'),
			moves => {
				'd2d3' => 'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2PP1N1P/PPB2PP1/RNBQR1K1 b - - 0 11',
				'd2d4' => 'r1bq1rk1/4bppp/p2p1n2/npp1p3/3PP3/2P2N1P/PPB2PP1/RNBQR1K1 b - d3 0 11',
			},
		},
		'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2PP1N1P/PPB2PP1/RNBQR1K1 b - - 0 11' => {
			eco => 'C96g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. h3 Na5 10. Bc2 c5 11. d3

			variation => N__('Spanish: Closed, Chigorin, 11. d3'),
			moves => {
				'a5c6' => 'r1bq1rk1/4bppp/p1np1n2/1pp1p3/4P3/2PP1N1P/PPB2PP1/RNBQR1K1 w - - 1 12',
			},
		},
		'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPB2PPP/RNBQR1K1 w - c6 0 11' => {
			eco => 'C90g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 Na5 10. Bc2 c5

			variation => N__('Spanish: Closed, Pilnik, 9... Na5'),
			moves => {
				'b1d2' => 'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 b - - 1 11',
			},
		},
		'r1bq1rk1/4bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 b - - 1 11' => {
			eco => 'C90h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d6 9. d3 Na5 10. Bc2 c5 11. Nbd2

			variation => N__('Spanish: Closed, Pilnik, 11.Nbd2'),
			moves => {
				'a5c6' => 'r1bq1rk1/4bppp/p1np1n2/1pp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 w - - 2 12',
				'd8c7' => 'r1b2rk1/2q1bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 w - - 2 12',
				'f8e8' => 'r1bqr1k1/4bppp/p2p1n2/npp1p3/4P3/2PP1N2/PPBN1PPP/R1BQR1K1 w - - 2 12',
			},
		},
		'r1bq1rk1/4bppp/p7/1p1pR3/3P4/2P5/PP3PPP/RNBQ2K1 b - d3 0 13' => {
			eco => 'C89g',
                        # TRANSLATORS:

			variation => N__('Spanish: Marshall, 12.Bxd5'),
			moves => {
				'e7d6' => 'r1bq1rk1/5ppp/p2b4/1p1pR3/3P4/2P5/PP3PPP/RNBQ2K1 w - - 1 14',
			},
		},
		'r1bq1rk1/4bppp/p7/1p1pR3/8/2P5/PP1P1PPP/RNBQ2K1 w - - 0 13' => {
			eco => 'C89g',
                        # TRANSLATORS:

			variation => N__('Spanish: Marshall, 12.Bxd5'),
			moves => {
				'd2d4' => 'r1bq1rk1/4bppp/p7/1p1pR3/3P4/2P5/PP3PPP/RNBQ2K1 b - d3 0 13',
			},
		},
		'r1bq1rk1/4bppp/pp1ppn2/4n3/2P1P3/N1N1B3/PP2BPPP/R2Q1RK1 w - - 2 12' => {
			eco => 'B44x',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7 9. Be2 O-O 10. O-O b6 11. Be3 Ne5

			variation => N__('Sicilian: Taimanov, Szen, Hedgehog, 11. Be3 Ne5'),
			moves => {
			},
		},
		'r1bq1rk1/4bppp/ppnppn2/8/2P1P3/N1N1B3/PP2BPPP/R2Q1RK1 b - - 1 11' => {
			eco => 'B44x',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7 9. Be2 O-O 10. O-O b6 11. Be3

			variation => N__('Sicilian: Taimanov, Szen, Hedgehog, 11. Be3'),
			moves => {
				'c6e5' => 'r1bq1rk1/4bppp/pp1ppn2/4n3/2P1P3/N1N1B3/PP2BPPP/R2Q1RK1 w - - 2 12',
				'c8b7' => 'r2q1rk1/1b2bppp/ppnppn2/8/2P1P3/N1N1B3/PP2BPPP/R2Q1RK1 w - - 2 12',
				'c8d7' => 'r2q1rk1/3bbppp/ppnppn2/8/2P1P3/N1N1B3/PP2BPPP/R2Q1RK1 w - - 2 12',
			},
		},
		'r1bq1rk1/4bppp/ppnppn2/8/2P1P3/N1N5/PP2BPPP/R1BQ1RK1 w - - 0 11' => {
			eco => 'B44w',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7 9. Be2 O-O 10. O-O b6

			variation => N__('Sicilian: Taimanov, Szen, Hedgehog Variation'),
			moves => {
				'c1e3' => 'r1bq1rk1/4bppp/ppnppn2/8/2P1P3/N1N1B3/PP2BPPP/R2Q1RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/5ppp/p1np1b2/1p1Np3/4P3/2P5/PPN2PPP/R2QKB1R b KQ - 2 12' => {
			eco => 'B33o',
                        # TRANSLATORS:

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O'),
			moves => {
				'f6g5' => 'r1bq1rk1/5ppp/p1np4/1p1Np1b1/4P3/2P5/PPN2PPP/R2QKB1R w KQ - 3 13',
			},
		},
		'r1bq1rk1/5ppp/p1np1b2/1p1Np3/4P3/N1P5/PP3PPP/R2QKB1R w KQ - 1 12' => {
			eco => 'B33o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O'),
			moves => {
				'a3c2' => 'r1bq1rk1/5ppp/p1np1b2/1p1Np3/4P3/2P5/PPN2PPP/R2QKB1R b KQ - 2 12',
			},
		},
		'r1bq1rk1/5ppp/p1np4/1p1Np1b1/4P3/2P5/PPN2PPP/R2QKB1R w KQ - 3 13' => {
			eco => 'B33q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O 12. Nc2 Bg5

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O 12. Nc2 Bg5'),
			moves => {
				'a2a4' => 'r1bq1rk1/5ppp/p1np4/1p1Np1b1/P3P3/2P5/1PN2PPP/R2QKB1R b KQ a3 0 13',
			},
		},
		'r1bq1rk1/5ppp/p1np4/1p1Np1b1/P3P3/2P5/1PN2PPP/R2QKB1R b KQ a3 0 13' => {
			eco => 'B33r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O 12. Nc2 Bg5 13. a4

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O 12. Nc2 Bg5 13. a4'),
			moves => {
			},
		},
		'r1bq1rk1/5ppp/p1pb4/1p1n4/3P4/1BP5/PP2RPPP/RNBQ2K1 b - - 2 13' => {
			eco => 'C89l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re2

			variation => N__('Spanish: Marshall, Main Line, 13. Re2'),
			moves => {
			},
		},
		'r1bq1rk1/5ppp/p1pb4/1p1n4/3P4/1BP5/PP3PPP/RNBQR1K1 b - - 2 13' => {
			eco => 'C89m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6 13. Re1

			variation => N__('Spanish: Marshall, Main Line, 13. Re1'),
			moves => {
				'd8h4' => 'r1b2rk1/5ppp/p1pb4/1p1n4/3P3q/1BP5/PP3PPP/RNBQR1K1 w - - 3 14',
			},
		},
		'r1bq1rk1/5ppp/p1pb4/1p1n4/8/1BPP4/PP3PPP/RNBQR1K1 b - - 2 13' => {
			eco => 'C89i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d3 Bd6 13. Re1

			variation => N__('Spanish: Marshall, 12. d3 Bd6 13. Re1'),
			moves => {
				'd8h4' => 'r1b2rk1/5ppp/p1pb4/1p1n4/7q/1BPP4/PP3PPP/RNBQR1K1 w - - 3 14',
			},
		},
		'r1bq1rk1/5ppp/p1pb4/1p1nR3/3P4/1BP5/PP3PPP/RNBQ2K1 w - - 1 13' => {
			eco => 'C89k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. d4 Bd6

			variation => N__('Spanish: Marshall, Main Line (12. d4 Bd6)'),
			moves => {
				'e5e1' => 'r1bq1rk1/5ppp/p1pb4/1p1n4/3P4/1BP5/PP3PPP/RNBQR1K1 b - - 2 13',
				'e5e2' => 'r1bq1rk1/5ppp/p1pb4/1p1n4/3P4/1BP5/PP2RPPP/RNBQ2K1 b - - 2 13',
			},
		},
		'r1bq1rk1/5ppp/p1pb4/1p1nR3/8/1BPP4/PP3PPP/RNBQ2K1 w - - 1 13' => {
			eco => 'C89i',
                        # TRANSLATORS:

			variation => N__('Spanish: Marshall, 12. d3'),
			moves => {
				'e5e1' => 'r1bq1rk1/5ppp/p1pb4/1p1n4/8/1BPP4/PP3PPP/RNBQR1K1 b - - 2 13',
			},
		},
		'r1bq1rk1/5ppp/p2b4/1p1p4/3P4/2P1R3/PP3PPP/RNBQ2K1 b - - 2 14' => {
			eco => 'C89h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 O-O 8. c3 d5 9. exd5 Nxd5 10. Nxe5 Nxe5 11. Rxe5 c6 12. Bxd5 cxd5 13. d4 Bd6 14. Re3

			variation => N__('Spanish: Marshall, Kevitz Variation'),
			moves => {
			},
		},
		'r1bq1rk1/5ppp/p2b4/1p1pR3/3P4/2P5/PP3PPP/RNBQ2K1 w - - 1 14' => {
			eco => 'C89g',
                        # TRANSLATORS:

			variation => N__('Spanish: Marshall, 12.Bxd5'),
			moves => {
				'e5e3' => 'r1bq1rk1/5ppp/p2b4/1p1p4/3P4/2P1R3/PP3PPP/RNBQ2K1 b - - 2 14',
			},
		},
		'r1bq1rk1/p1p1ppbp/1pn3p1/8/2BPP3/2P5/P3NPPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D86',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 Nc6 9. O-O b6

			variation => N__('Grünfeld: Classical Exchange, Simagin Improved, 9.O-O b6'),
			moves => {
			},
		},
		'r1bq1rk1/p1p2ppp/5n2/3p2B1/1b6/2NB4/PPP2PPP/R2Q1RK1 b - - 3 10' => {
			eco => 'C47u',
                        # TRANSLATORS:

			variation => N__('Four Knights: Scotch, Main Line, 8.exd5 cxd5 9.O-O O-O'),
			moves => {
				'c7c6' => 'r1bq1rk1/p4ppp/2p2n2/3p2B1/1b6/2NB4/PPP2PPP/R2Q1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/p1p2ppp/5n2/3p4/1b6/2NB4/PPP2PPP/R1BQ1RK1 w - - 2 10' => {
			eco => 'C47u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5 8. exd5 cxd5 9. O-O O-O

			variation => N__('Four Knights: Scotch, Main Line, 8.exd5 cxd5 9.O-O O-O'),
			moves => {
				'c1g5' => 'r1bq1rk1/p1p2ppp/5n2/3p2B1/1b6/2NB4/PPP2PPP/R2Q1RK1 b - - 3 10',
			},
		},
		'r1bq1rk1/p1pnbppp/1p2pn2/3P2B1/3P4/2N1PN2/PP3PPP/2RQKB1R b K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, 7. Rc1 b6'),
			moves => {
				'e6d5' => 'r1bq1rk1/p1pnbppp/1p3n2/3p2B1/3P4/2N1PN2/PP3PPP/2RQKB1R w K - 0 9',
			},
		},
		'r1bq1rk1/p1pnbppp/1p2pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 b6

			variation => N__('QGD: Orthodox, 7. Rc1 b6'),
			moves => {
				'c4d5' => 'r1bq1rk1/p1pnbppp/1p2pn2/3P2B1/3P4/2N1PN2/PP3PPP/2RQKB1R b K - 0 8',
			},
		},
		'r1bq1rk1/p1pnbppp/1p2pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R w KQ - 0 8' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 b6

			variation => N__('QGD: Orthodox, Botvinnik, 7... b6'),
			moves => {
			},
		},
		'r1bq1rk1/p1pnbppp/1p2pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - - 0 8' => {
			eco => 'E08c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 b6

			variation => N__('Catalan: Closed, 7. Qc2 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p1pnbppp/1p3n2/1B1p2B1/3P4/2N1PN2/PP3PPP/2RQK2R b K - 1 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 b6 8. cxd5 exd5 9. Bb5

			variation => N__('QGD: Orthodox, 7. Rc1 b6, Capablanca Variation'),
			moves => {
			},
		},
		'r1bq1rk1/p1pnbppp/1p3n2/3p2B1/3P4/2N1PN2/PP3PPP/2RQKB1R w K - 0 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 b6 8. cxd5 exd5

			variation => N__('QGD: Orthodox, 7. Rc1 b6'),
			moves => {
				'd1a4' => 'r1bq1rk1/p1pnbppp/1p3n2/3p2B1/Q2P4/2N1PN2/PP3PPP/2R1KB1R b K - 1 9',
				'f1b5' => 'r1bq1rk1/p1pnbppp/1p3n2/1B1p2B1/3P4/2N1PN2/PP3PPP/2RQK2R b K - 1 9',
				'f1d3' => 'r1bq1rk1/p1pnbppp/1p3n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R b K - 1 9',
			},
		},
		'r1bq1rk1/p1pnbppp/1p3n2/3p2B1/3P4/2NBPN2/PP3PPP/2RQK2R b K - 1 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 b6 8. cxd5 exd5 9. Bd3

			variation => N__('QGD: Orthodox, 7. Rc1 b6, Pillsbury Attack'),
			moves => {
			},
		},
		'r1bq1rk1/p1pnbppp/1p3n2/3p2B1/Q2P4/2N1PN2/PP3PPP/2R1KB1R b K - 1 9' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 b6 8. cxd5 exd5 9. Qa4

			variation => N__('QGD: Orthodox, 7. Rc1 b6, Duras Variation'),
			moves => {
			},
		},
		'r1bq1rk1/p2n1ppp/1p2pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1B2RK1 w - - 0 10' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. Qe2 b6

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. Qe2 b6'),
			moves => {
				'f1d1' => 'r1bq1rk1/p2n1ppp/1p2pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1BR2K1 b - - 1 10',
			},
		},
		'r1bq1rk1/p2n1ppp/1p2pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1BR2K1 b - - 1 10' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. Qe2 b6 10. Rd1

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. Qe2 b6 10. Rd1'),
			moves => {
			},
		},
		'r1bq1rk1/p2n1ppp/1ppbpn2/3p4/2PP4/1PN1PN2/P1Q1BPPP/R1B1K2R w KQ - 0 9' => {
			eco => 'D45s',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. b3 O-O 8. Be2 b6

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. b3 O-O 8. Be2 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p2n1ppp/2pbpn2/1p6/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - b6 0 10' => {
			eco => 'D46l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 b5

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... b5'),
			moves => {
				'c4e2' => 'r1bq1rk1/p2n1ppp/2pbpn2/1p6/3P4/2N1PN2/PPQ1BPPP/R1B2RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/p2n1ppp/2pbpn2/1p6/3P4/2N1PN2/PPQ1BPPP/R1B2RK1 b - - 1 10' => {
			eco => 'D46m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 b5 10. Be2

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... b5 10. Be2'),
			moves => {
				'c8b7' => 'r2q1rk1/pb1n1ppp/2pbpn2/1p6/3P4/2N1PN2/PPQ1BPPP/R1B2RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/p2nbpp1/2p1pn1p/1p6/2BP3B/2N1PN2/PP3PPP/2RQK2R w K b6 0 11' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 h6 9. Bh4 dxc4 10. Bxc4 b5

			variation => N__('QGD: Orthodox, Main Line, 8... h6 9. Bh4 dxc4 10.Bxc4 b5'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/1pp1p3/8/2PPQ3/5NP1/PP3PBP/R1B2RK1 b - - 0 11' => {
			eco => 'E09k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. e4 dxe4 10. Nxe4 Nxe4 11. Qxe4

			variation => N__('Catalan: Closed, Main Line, 9. e4 dxe4 10.Nxe4 Nxe4 11.Qxe4'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/1pp1p3/8/2PPn3/5NP1/PPQ2PBP/R1B2RK1 w - - 0 11' => {
			eco => 'E09j',
                        # TRANSLATORS:

			variation => N__('Catalan: Closed, Main Line, 9. e4 dxe4'),
			moves => {
				'c2e4' => 'r1bq1rk1/p2nbppp/1pp1p3/8/2PPQ3/5NP1/PP3PBP/R1B2RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP1B2/5NP1/PPQ1PPBP/RN3RK1 w - - 0 9' => {
			eco => 'E08j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Bf4 b6

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. Bf4 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 w - - 0 9' => {
			eco => 'E08m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3 b6

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3 b6'),
			moves => {
				'f1d1' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNBR2K1 b - - 1 9',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNBR2K1 b - - 1 9' => {
			eco => 'E08n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3 b6 9. Rd1

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3 b6 9. Rd1'),
			moves => {
				'c8a6' => 'r2q1rk1/p2nbppp/bpp1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNBR2K1 w - - 2 10',
				'c8b7' => 'r2q1rk1/pb1nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNBR2K1 w - - 2 10',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1QNPPBP/R1B2RK1 b - - 0 9' => {
			eco => 'E09e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. b3

			variation => N__('Catalan: Closed, Main Line, 9. b3'),
			moves => {
				'a7a5' => 'r1bq1rk1/3nbppp/1pp1pn2/p2p4/2PP4/1P3NP1/P1QNPPBP/R1B2RK1 w - a6 0 10',
				'c8b7' => 'r2q1rk1/pb1nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1QNPPBP/R1B2RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNBR2K1 w - - 0 9' => {
			eco => 'E08f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Rd1 b6

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. Rd1 b6'),
			moves => {
				'a2a4' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/P1PP4/5NP1/1PQ1PPBP/RNBR2K1 b - a3 0 9',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 w - - 0 9' => {
			eco => 'E09d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6

			variation => N__('Catalan: Closed, Main Line, 8... b6'),
			moves => {
				'b2b3' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1QNPPBP/R1B2RK1 b - - 0 9',
				'e2e4' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PPP3/5NP1/PPQN1PBP/R1B2RK1 b - e3 0 9',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PPP3/5NP1/PPQN1PBP/R1B2RK1 b - e3 0 9' => {
			eco => 'E09h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. e4

			variation => N__('Catalan: Closed, Main Line, 9. e4'),
			moves => {
				'c8a6' => 'r2q1rk1/p2nbppp/bpp1pn2/3p4/2PPP3/5NP1/PPQN1PBP/R1B2RK1 w - - 1 10',
				'c8b7' => 'r2q1rk1/pb1nbppp/1pp1pn2/3p4/2PPP3/5NP1/PPQN1PBP/R1B2RK1 w - - 1 10',
				'd5e4' => 'r1bq1rk1/p2nbppp/1pp1pn2/8/2PPp3/5NP1/PPQN1PBP/R1B2RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/3p4/P1PP4/5NP1/1PQ1PPBP/RNBR2K1 b - a3 0 9' => {
			eco => 'E08f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Rd1 b6 9. a4

			variation => N__('Catalan: Closed, Zagoryansky Variation'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/8/2PPN3/5NP1/PPQ2PBP/R1B2RK1 b - - 0 10' => {
			eco => 'E09j',
                        # TRANSLATORS:

			variation => N__('Catalan: Closed, Main Line, 9. e4 dxe4'),
			moves => {
				'f6e4' => 'r1bq1rk1/p2nbppp/1pp1p3/8/2PPn3/5NP1/PPQ2PBP/R1B2RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/p2nbppp/1pp1pn2/8/2PPp3/5NP1/PPQN1PBP/R1B2RK1 w - - 0 10' => {
			eco => 'E09j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b6 9. e4 dxe4

			variation => N__('Catalan: Closed, Main Line, 9. e4 dxe4'),
			moves => {
				'd2e4' => 'r1bq1rk1/p2nbppp/1pp1pn2/8/2PPN3/5NP1/PPQ2PBP/R1B2RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/p2nbppp/1ppp1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQR1K1 w - - 0 9' => {
			eco => 'C41z',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. Re1 c6 8. a4 b6

			variation => N__('Philidor: Improved Hanham, 7. Re1 c6 8. a4 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/2p1pn2/1p1p4/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 w - b6 0 9' => {
			eco => 'E08l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3 b5

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3 b5'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/2p1pn2/1p1p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 w - b6 0 9' => {
			eco => 'E09b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2 b5

			variation => N__('Catalan: Closed, Main Line, 8... b5'),
			moves => {
			},
		},
		'r1bq1rk1/p2nbppp/2p1pn2/1p4B1/2BP4/2N1PN2/PP3PPP/2RQK2R w K b6 0 10' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 b5

			variation => N__('QGD: Orthodox, Main Line, Fianchetto Variation'),
			moves => {
				'c4d3' => 'r1bq1rk1/p2nbppp/2p1pn2/1p4B1/3P4/2NBPN2/PP3PPP/2RQK2R b K - 1 10',
			},
		},
		'r1bq1rk1/p2nbppp/2p1pn2/1p4B1/3P4/2NBPN2/PP3PPP/2RQK2R b K - 1 10' => {
			eco => 'D66',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Main Line, Fianchetto Variation'),
			moves => {
				'a7a6' => 'r1bq1rk1/3nbppp/p1p1pn2/1p4B1/3P4/2NBPN2/PP3PPP/2RQK2R w K - 0 11',
			},
		},
		'r1bq1rk1/p2p1ppp/1pn1pn2/2p5/2PP4/P1PBP3/4NPPP/R1BQK2R w KQ - 0 9' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 b6

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 b6'),
			moves => {
				'e3e4' => 'r1bq1rk1/p2p1ppp/1pn1pn2/2p5/2PPP3/P1PB4/4NPPP/R1BQK2R b KQ - 0 9',
			},
		},
		'r1bq1rk1/p2p1ppp/1pn1pn2/2p5/2PPP3/P1PB4/4NPPP/R1BQK2R b KQ - 0 9' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 b6 9. e4

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 b6 9. e4'),
			moves => {
				'f6e8' => 'r1bqnrk1/p2p1ppp/1pn1p3/2p5/2PPP3/P1PB4/4NPPP/R1BQK2R w KQ - 1 10',
			},
		},
		'r1bq1rk1/p2pppbp/1pn2np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 b - - 1 9' => {
			eco => 'B38i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 b6 9. O-O

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 b6 9.O-O'),
			moves => {
				'c8b7' => 'r2q1rk1/pb1pppbp/1pn2np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 2 10',
			},
		},
		'r1bq1rk1/p2pppbp/1pn2np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R w KQ - 0 9' => {
			eco => 'B38h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 b6

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 b6'),
			moves => {
				'e1g1' => 'r1bq1rk1/p2pppbp/1pn2np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 b - - 1 9',
			},
		},
		'r1bq1rk1/p3bppp/1pn1p3/8/2BP4/2P2N2/P4PPP/R1BQR1K1 w - - 0 12' => {
			eco => 'D41t',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 7. Bc4 Main Line'),
			moves => {
				'c4d3' => 'r1bq1rk1/p3bppp/1pn1p3/8/3P4/2PB1N2/P4PPP/R1BQR1K1 b - - 1 12',
			},
		},
		'r1bq1rk1/p3bppp/1pn1p3/8/3P4/2PB1N2/P4PPP/R1BQR1K1 b - - 1 12' => {
			eco => 'D41t',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 7. Bc4 Main Line'),
			moves => {
				'c8b7' => 'r2q1rk1/pb2bppp/1pn1p3/8/3P4/2PB1N2/P4PPP/R1BQR1K1 w - - 2 13',
			},
		},
		'r1bq1rk1/p3bppp/1pn1pn2/6B1/3P4/P1NB1N2/1P3PPP/R2QR1K1 b - - 1 12' => {
			eco => 'D42n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6 11. a3 b6 12. Bg5

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6 11. a3 b6 12. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/p3bppp/1pn1pn2/8/3P4/P1N2N2/1PB2PPP/R1BQR1K1 b - - 1 12' => {
			eco => 'D42o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6 11. a3 b6 12. Bc2

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6 11. a3 b6 12. Bc2'),
			moves => {
			},
		},
		'r1bq1rk1/p3bppp/1pn1pn2/8/3P4/P1NB1N2/1P3PPP/R1BQR1K1 w - - 0 12' => {
			eco => 'D42m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6 11. a3 b6

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6 11. a3 b6'),
			moves => {
				'c1e3' => 'r1bq1rk1/p3bppp/1pn1pn2/8/3P4/P1NBBN2/1P3PPP/R2QR1K1 b - - 1 12',
				'c1g5' => 'r1bq1rk1/p3bppp/1pn1pn2/6B1/3P4/P1NB1N2/1P3PPP/R2QR1K1 b - - 1 12',
				'd3c2' => 'r1bq1rk1/p3bppp/1pn1pn2/8/3P4/P1N2N2/1PB2PPP/R1BQR1K1 b - - 1 12',
			},
		},
		'r1bq1rk1/p3bppp/1pn1pn2/8/3P4/P1NBBN2/1P3PPP/R2QR1K1 b - - 1 12' => {
			eco => 'D42m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6 11. a3 b6 12. Be3

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6 11. a3 b6 12. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/p3bppp/2p2n2/3p2B1/8/2NB1Q2/PPP2PPP/R4RK1 w - - 2 12' => {
			eco => 'C47y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5 8. exd5 cxd5 9. O-O O-O 10. Bg5 c6 11. Qf3 Be7

			variation => N__('Four Knights: Scotch, Main Line, 10. Bg5 c6 11. Qf3 Be7'),
			moves => {
			},
		},
		'r1bq1rk1/p3npbp/1pnpp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 0 10' => {
			eco => 'B25s',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 b6

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p3ppbp/1pnp1np1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'B74',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Nb3 b6

			variation => N__('Sicilian: Dragon, Classical, 9. Nb3 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p3ppbp/2p3p1/3N4/8/4BP2/PPPQ2PP/2KR1B1R b - - 0 12' => {
			eco => 'B76r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5 10. exd5 Nxd5 11. Nxc6 bxc6 12. Nxd5

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12.Nxd5'),
			moves => {
			},
		},
		'r1bq1rk1/p3ppbp/2p3p1/3n4/3B4/2N2P2/PPPQ2PP/2KR1B1R b - - 1 12' => {
			eco => 'B76s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5 10. exd5 Nxd5 11. Nxc6 bxc6 12. Bd4

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12. Bd4'),
			moves => {
				'e7e5' => 'r1bq1rk1/p4pbp/2p3p1/3np3/3B4/2N2P2/PPPQ2PP/2KR1B1R w - e6 0 13',
			},
		},
		'r1bq1rk1/p3ppbp/2p3p1/3n4/8/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 12' => {
			eco => 'B76q',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6'),
			moves => {
				'c3d5' => 'r1bq1rk1/p3ppbp/2p3p1/3N4/8/4BP2/PPPQ2PP/2KR1B1R b - - 0 12',
				'e3d4' => 'r1bq1rk1/p3ppbp/2p3p1/3n4/3B4/2N2P2/PPPQ2PP/2KR1B1R b - - 1 12',
			},
		},
		'r1bq1rk1/p4pbp/2p3p1/2Bnp3/8/2N2P2/PPPQ2PP/2KR1B1R b - - 1 13' => {
			eco => 'B76t',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12. Bd4 e5'),
			moves => {
				'c8e6' => 'r2q1rk1/p4pbp/2p1b1p1/2Bnp3/8/2N2P2/PPPQ2PP/2KR1B1R w - - 2 14',
			},
		},
		'r1bq1rk1/p4pbp/2p3p1/3np3/3B4/2N2P2/PPPQ2PP/2KR1B1R w - e6 0 13' => {
			eco => 'B76t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5 10. exd5 Nxd5 11. Nxc6 bxc6 12. Bd4 e5

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12. Bd4 e5'),
			moves => {
				'd4c5' => 'r1bq1rk1/p4pbp/2p3p1/2Bnp3/8/2N2P2/PPPQ2PP/2KR1B1R b - - 1 13',
			},
		},
		'r1bq1rk1/p4ppp/1pn1p3/8/2BPP3/5N2/P2Q1PPP/3R1RK1 b - - 1 13' => {
			eco => 'D41o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e4 Nxc3 7. bxc3 cxd4 8. cxd4 Bb4+ 9. Bd2 Bxd2+ 10. Qxd2 O-O 11. Bc4 Nc6 12. O-O b6 13. Rad1

			variation => N__('QGD: Semi-Tarrasch, 6. e4, Main Line, 12... b6'),
			moves => {
			},
		},
		'r1bq1rk1/p4ppp/1pn1p3/8/2BPP3/5N2/P2Q1PPP/R4RK1 w - - 0 13' => {
			eco => 'D41o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e4 Nxc3 7. bxc3 cxd4 8. cxd4 Bb4+ 9. Bd2 Bxd2+ 10. Qxd2 O-O 11. Bc4 Nc6 12. O-O b6

			variation => N__('QGD: Semi-Tarrasch, 6. e4, Main Line, 12... b6'),
			moves => {
				'a1d1' => 'r1bq1rk1/p4ppp/1pn1p3/8/2BPP3/5N2/P2Q1PPP/3R1RK1 b - - 1 13',
			},
		},
		'r1bq1rk1/p4ppp/1pn1pn2/2pp4/2PP4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 b6

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 b6'),
			moves => {
			},
		},
		'r1bq1rk1/p4ppp/2p2n2/3p2B1/1b6/2NB1Q2/PPP2PPP/R4RK1 b - - 1 11' => {
			eco => 'C47x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5 8. exd5 cxd5 9. O-O O-O 10. Bg5 c6 11. Qf3

			variation => N__('Four Knights: Scotch, Main Line, 10. Bg5 c6 11. Qf3'),
			moves => {
				'b4e7' => 'r1bq1rk1/p3bppp/2p2n2/3p2B1/8/2NB1Q2/PPP2PPP/R4RK1 w - - 2 12',
			},
		},
		'r1bq1rk1/p4ppp/2p2n2/3p2B1/1b6/2NB4/PPP2PPP/R2Q1RK1 w - - 0 11' => {
			eco => 'C47v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5 8. exd5 cxd5 9. O-O O-O 10. Bg5 c6

			variation => N__('Four Knights: Scotch, Main Line, 9.O-O O-O 10. Bg5 c6'),
			moves => {
				'c3a4' => 'r1bq1rk1/p4ppp/2p2n2/3p2B1/Nb6/3B4/PPP2PPP/R2Q1RK1 b - - 1 11',
				'd1f3' => 'r1bq1rk1/p4ppp/2p2n2/3p2B1/1b6/2NB1Q2/PPP2PPP/R4RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/p4ppp/2p2n2/3p2B1/Nb6/3B4/PPP2PPP/R2Q1RK1 b - - 1 11' => {
			eco => 'C47w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5 8. exd5 cxd5 9. O-O O-O 10. Bg5 c6 11. Na4

			variation => N__('Four Knights: Scotch, Main Line, 10. Bg5 c6 11. Na4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2n1p1p1/3pN3/3P1P2/2N1B1P1/PP2P1BP/R2Q1RK1 b - - 2 11' => {
			eco => 'D79m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Ne5 e6 9. Nc3 Nfd7 10. f4 Nc6 11. Be3

			variation => N__('Neo-Grünfeld, Main Line, 10. f4 Nc6 11. Be3'),
			moves => {
				'd7b6' => 'r1bq1rk1/pp3pbp/1nn1p1p1/3pN3/3P1P2/2N1B1P1/PP2P1BP/R2Q1RK1 w - - 3 12',
				'f7f6' => 'r1bq1rk1/pp1n2bp/2n1ppp1/3pN3/3P1P2/2N1B1P1/PP2P1BP/R2Q1RK1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp1n1pbp/2n1p1p1/3pN3/3P1P2/2N3P1/PP2P1BP/R1BQ1RK1 w - - 1 11' => {
			eco => 'D79l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Ne5 e6 9. Nc3 Nfd7 10. f4 Nc6

			variation => N__('Neo-Grünfeld, Main Line, 10. f4 Nc6'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp1n1pbp/2n1p1p1/3pN3/3P1P2/2N1B1P1/PP2P1BP/R2Q1RK1 b - - 2 11',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/3Pp3/2P1P3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 0 9' => {
			eco => 'E94y',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Be3 c6 9. d5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Be3 c6 9. d5'),
			moves => {
				'c6c5' => 'r1bq1rk1/pp1n1pbp/3p1np1/2pPp3/2P1P3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/3Pp3/2P1P3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 0 9' => {
			eco => 'E86f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 c6 8. Qd2 Nbd7 9. d5

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 c6 8. Qd2 Nbd7 9. d5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 w - - 0 9' => {
			eco => 'E67i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. Qc2 c6

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. Qc2 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 w - - 0 9' => {
			eco => 'E67l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. h3 c6

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. h3 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/1PN2NP1/P4PBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E68l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. b3

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 c6 9. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 0 9' => {
			eco => 'E94x',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Be3 c6

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Be3 c6'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp1n1pbp/2pp1np1/3Pp3/2P1P3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/2KR1B1R b - - 3 9' => {
			eco => 'E86g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 c6 8. Qd2 Nbd7 9. O-O-O

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 c6 8. Qd2 Nbd7 9.O-O-O'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1n1pbp/p1pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/2KR1B1R w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 2 9' => {
			eco => 'E86e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 c6 8. Qd2 Nbd7

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 c6 8. Qd2 Nbd7'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp1n1pbp/2pp1np1/3Pp3/2P1P3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 0 9',
				'e1c1' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N1BP2/PP1QN1PP/2KR1B1R b - - 3 9',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP2BPPP/1RBQR1K1 b - - 1 9' => {
			eco => 'E95i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Rb1

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 c6 9. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9' => {
			eco => 'E95h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 c6'),
			moves => {
				'a1b1' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP2BPPP/1RBQR1K1 b - - 1 9',
				'e2f1' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b - - 1 9',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b - - 1 9' => {
			eco => 'E95j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 c6 9. Bf1'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 w - a6 0 10',
				'e5d4' => 'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PpP3/2N2N2/PP3PPP/R1BQRBK1 w - - 0 10',
				'f8e8' => 'r1bqr1k1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 w - - 2 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 0 9' => {
			eco => 'E94t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Qc2 c6

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Qc2 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NP1/PP3PBP/1RBQ1RK1 b - - 1 9' => {
			eco => 'E68k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. Rb1

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 c6 9. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E68j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 c6'),
			moves => {
				'a1b1' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NP1/PP3PBP/1RBQ1RK1 b - - 1 9',
				'b2b3' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/1PN2NP1/P4PBP/R1BQ1RK1 b - - 0 9',
				'h2h3' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 b - - 0 9' => {
			eco => 'E69a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p1n1pbp/2pp1np1/p3p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - a6 0 10',
				'a7a6' => 'r1bq1rk1/1p1n1pbp/p1pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 0 10',
				'd8a5' => 'r1b2rk1/pp1n1pbp/2pp1np1/q3p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10',
				'd8b6' => 'r1b2rk1/pp1n1pbp/1qpp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10',
				'd8e7' => 'r1b2rk1/pp1nqpbp/2pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10',
				'e5d4' => 'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PpP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 0 10',
				'f8e8' => 'r1bqr1k1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQ1RK1 b - - 0 10' => {
			eco => 'E69e',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 exd4'),
			moves => {
				'f8e8' => 'r1bqr1k1/pp1n1pbp/2pp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQ1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PNP3/2N5/PP3PPP/R1BQRBK1 b - - 0 10' => {
			eco => 'E95l',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 c6 9. Bf1 exd4'),
			moves => {
				'f6g4' => 'r1bq1rk1/pp1n1pbp/2pp2p1/8/2PNP1n1/2N5/PP3PPP/R1BQRBK1 w - - 1 11',
				'f8e8' => 'r1bqr1k1/pp1n1pbp/2pp1np1/8/2PNP3/2N5/PP3PPP/R1BQRBK1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PpP3/2N2N2/PP3PPP/R1BQRBK1 w - - 0 10' => {
			eco => 'E95l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 exd4

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 c6 9. Bf1 exd4'),
			moves => {
				'f3d4' => 'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PNP3/2N5/PP3PPP/R1BQRBK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PpP3/2N2NPP/PP3PB1/R1BQ1RK1 w - - 0 10' => {
			eco => 'E69e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 c6 9. h3 exd4

			variation => N__('King\'s Indian: Fianchetto, Classical, 9. h3 exd4'),
			moves => {
				'f3d4' => 'r1bq1rk1/pp1n1pbp/2pp1np1/8/2PNP3/2N3PP/PP3PB1/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/2pp2p1/8/2PNP1n1/2N5/PP3PPP/R1BQRBK1 w - - 1 11' => {
			eco => 'E95m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 c6 9. Bf1 exd4 10. Nxd4 Ng4

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 c6 9. Bf1 exd4 10.Nxd4 Ng4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 6 10' => {
			eco => 'A73',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Nbd7

			variation => N__('Benoni: Classical, 9.O-O Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 10' => {
			eco => 'A63',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7

			variation => N__('Benoni: Fianchetto, 9...Nbd7'),
			moves => {
				'f3d2' => 'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 6 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 6 10' => {
			eco => 'A63',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2

			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 0 11',
				'f8e8' => 'r1bqr1k1/pp1n1pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 7 11',
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pPp3/2P1P3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'E94y',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Be3 c6 9. d5 c5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Be3 c6 9. d5 c5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/3Pp3/4P3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 2 10' => {
			eco => 'E89e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 c6 8. Nge2 cxd5 9. cxd5 Nbd7 10. Qd2

			variation => N__('King\'s Indian: Sämisch, Orthodox Main Line, 10. Qd2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1n1pbp/p2p1np1/3Pp3/4P3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 0 11',
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/3Pp3/4P3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 1 10' => {
			eco => 'E89d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 c6 8. Nge2 cxd5 9. cxd5 Nbd7

			variation => N__('King\'s Indian: Sämisch, Orthodox Main Line, 9...Nbd7'),
			moves => {
				'd1d2' => 'r1bq1rk1/pp1n1pbp/3p1np1/3Pp3/4P3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 2 10',
			},
		},
		'r1bq1rk1/pp1n1pbp/n2p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQ1RK1 b - - 4 11' => {
			eco => 'A67r',
                        # TRANSLATORS:

			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3 Na6'),
			moves => {
				'a6c7' => 'r1bq1rk1/ppnn1pbp/3p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQ1RK1 w - - 5 12',
			},
		},
		'r1bq1rk1/pp1n1pbp/n2p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQK2R w KQ - 3 11' => {
			eco => 'A67r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O 10. Nf3 Na6

			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3 Na6'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1n1pbp/n2p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQ1RK1 b - - 4 11',
			},
		},
		'r1bq1rk1/pp1n1pbp/n2p2p1/2pP4/4PP2/2NB1N2/PP4PP/R1BQK2R w KQ - 7 11' => {
			eco => 'A67j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3 O-O 10. Nf3 Na6

			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3 O-O 10. Nf3 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pp1/2p1pb1p/3p4/2PP4/2NBPN2/PP3PPP/2RQ1RK1 b - - 3 10' => {
			eco => 'D55j',
                        # TRANSLATORS:

			variation => N__('QGD: Neo-Orthodox, Gligoric Variation'),
			moves => {
				'd5c4' => 'r1bq1rk1/pp1n1pp1/2p1pb1p/8/2pP4/2NBPN2/PP3PPP/2RQ1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp1n1pp1/2p1pb1p/3p4/2PP4/2NBPN2/PP3PPP/2RQK2R w K - 2 10' => {
			eco => 'D55j',
                        # TRANSLATORS:

			variation => N__('QGD: Neo-Orthodox, Gligoric Variation'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1n1pp1/2p1pb1p/3p4/2PP4/2NBPN2/PP3PPP/2RQ1RK1 b - - 3 10',
			},
		},
		'r1bq1rk1/pp1n1pp1/2p1pb1p/8/2BP4/2N1PN2/PP3PPP/2RQ1RK1 b - - 0 11' => {
			eco => 'D55k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 h6 7. Bxf6 Bxf6 8. Rc1 c6 9. Bd3 Nd7 10. O-O dxc4 11. Bxc4

			variation => N__('QGD: Neo-Orthodox, Petrosian Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pp1/2p1pb1p/8/2pP4/2NBPN2/PP3PPP/2RQ1RK1 w - - 0 11' => {
			eco => 'D55j',
                        # TRANSLATORS:

			variation => N__('QGD: Neo-Orthodox, Gligoric Variation'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp1n1pp1/2p1pb1p/8/2BP4/2N1PN2/PP3PPP/2RQ1RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/pp1n1pp1/2pb3p/4p3/2PP4/5N2/PPB2PPP/R1BQ1RK1 w - - 0 12' => {
			eco => 'D46r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxe4 9. Nxe4 Nxe4 10. Bxe4 h6 11. Bc2 e5

			variation => N__('Semi-Slav: Chigorin, Main Line, 10... h6 11. Bc2 e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pp1/2pbp2p/8/2PP4/5N2/PPB2PPP/R1BQ1RK1 b - - 1 11' => {
			eco => 'D46r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxe4 9. Nxe4 Nxe4 10. Bxe4 h6 11. Bc2

			variation => N__('Semi-Slav: Chigorin, Main Line, 10... h6 11. Bc2'),
			moves => {
				'e6e5' => 'r1bq1rk1/pp1n1pp1/2pb3p/4p3/2PP4/5N2/PPB2PPP/R1BQ1RK1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp1n1pp1/2pbp2p/8/2PPB3/5N2/PP3PPP/R1BQ1RK1 w - - 0 11' => {
			eco => 'D46r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxe4 9. Nxe4 Nxe4 10. Bxe4 h6

			variation => N__('Semi-Slav: Chigorin, Main Line, 10... h6'),
			moves => {
				'e4c2' => 'r1bq1rk1/pp1n1pp1/2pbp2p/8/2PP4/5N2/PPB2PPP/R1BQ1RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/pp1n1ppp/2n1p3/2bpP3/5PQ1/P1N2N2/1PP3PP/R1B1KB1R b KQ - 3 9' => {
			eco => 'C11g',
                        # TRANSLATORS:

			variation => N__('French: Steinitz, 6.dxc5 Nc6'),
			moves => {
				'f7f6' => 'r1bq1rk1/pp1n2pp/2n1pp2/2bpP3/5PQ1/P1N2N2/1PP3PP/R1B1KB1R w KQ - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/2n1p3/2bpP3/5PQ1/P1N5/1PP3PP/R1B1KBNR w KQ - 2 9' => {
			eco => 'C11g',
                        # TRANSLATORS:

			variation => N__('French: Steinitz, 6.dxc5 Nc6'),
			moves => {
				'g1f3' => 'r1bq1rk1/pp1n1ppp/2n1p3/2bpP3/5PQ1/P1N2N2/1PP3PP/R1B1KB1R b KQ - 3 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2p1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8' => {
			eco => 'D46b',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bb4 7. O-O O-O

			variation => N__('Semi-Slav: Romih, 7.O-O O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/2pb1n2/3pp3/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 w - - 0 9' => {
			eco => 'D45w',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. Be2 O-O 8. O-O e5

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. Be2 O-O 8.O-O e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/2pb1n2/4p3/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - - 0 10' => {
			eco => 'D46i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4 e5

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line, 9... e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/2pb1n2/4p3/2BPP3/2N2N2/PP3PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D46p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxc4 9. Bxc4 e5

			variation => N__('Semi-Slav: Chigorin, 7.O-O O-O 8. e4 dxc4 9.Bxc4 e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbp3/8/2PPB3/5N2/PP3PPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'D46q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxe4 9. Nxe4 Nxe4 10. Bxe4

			variation => N__('Semi-Slav: Chigorin, Main Line'),
			moves => {
				'h7h6' => 'r1bq1rk1/pp1n1pp1/2pbp2p/8/2PPB3/5N2/PP3PPP/R1BQ1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbp3/8/2PPn3/3B1N2/PP3PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D46q',
                        # TRANSLATORS:

			variation => N__('Semi-Slav: Chigorin, Main Line'),
			moves => {
				'd3e4' => 'r1bq1rk1/pp1n1ppp/2pbp3/8/2PPB3/5N2/PP3PPP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q1BPPP/R1B1K2R b KQ - 2 8' => {
			eco => 'D45r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. b3 O-O 8. Be2

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. b3 O-O 8. Be2'),
			moves => {
				'b7b6' => 'r1bq1rk1/p2n1ppp/1ppbpn2/3p4/2PP4/1PN1PN2/P1Q1BPPP/R1B1K2R w KQ - 0 9',
				'f8e8' => 'r1bqr1k1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q1BPPP/R1B1K2R w KQ - 3 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q2PPP/R1B1KB1R w KQ - 1 8' => {
			eco => 'D45p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. b3 O-O

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. b3 O-O'),
			moves => {
				'c1b2' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/PBQ2PPP/R3KB1R b KQ - 2 8',
				'f1e2' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q1BPPP/R1B1K2R b KQ - 2 8',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/PBQ2PPP/R3KB1R b KQ - 2 8' => {
			eco => 'D45q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. b3 O-O 8. Bb2

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. b3 O-O 8. Bb2'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B1K2R w KQ - 5 8' => {
			eco => 'D45v',
                        # TRANSLATORS:

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. Be2'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 b - - 6 8',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 b - - 6 8' => {
			eco => 'D45v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. Be2 O-O 8. O-O

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. Be2 O-O 8.O-O'),
			moves => {
				'd8e7' => 'r1b2rk1/pp1nqppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 w - - 7 9',
				'e6e5' => 'r1bq1rk1/pp1n1ppp/2pb1n2/3pp3/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B2RK1 w - - 7 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8' => {
			eco => 'D46o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O

			variation => N__('Semi-Slav: Chigorin, 7.O-O O-O'),
			moves => {
				'e3e4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PPP3/2NB1N2/PP3PPP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PPQ2PPP/R1B1K2R w KQ - 5 8' => {
			eco => 'D46h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 O-O'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PPQ2PPP/R1B2RK1 b - - 6 8',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PPQ2PPP/R1B2RK1 b - - 6 8' => {
			eco => 'D46h',
                        # TRANSLATORS:

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 O-O'),
			moves => {
				'd5c4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/8/2pP4/2NBPN2/PPQ2PPP/R1B2RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PPP3/2NB1N2/PP3PPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'D46p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4

			variation => N__('Semi-Slav: Chigorin, 7.O-O O-O 8. e4'),
			moves => {
				'd5c4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/8/2pPP3/2NB1N2/PP3PPP/R1BQ1RK1 w - - 0 9',
				'd5e4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/8/2PPp3/2NB1N2/PP3PPP/R1BQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 b - - 0 9' => {
			eco => 'D46i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 O-O 8. O-O dxc4 9. Bxc4

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 Main Line'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1n1ppp/p1pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - - 0 10',
				'b7b5' => 'r1bq1rk1/p2n1ppp/2pbpn2/1p6/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - b6 0 10',
				'd8e7' => 'r1b2rk1/pp1nqppp/2pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - - 1 10',
				'e6e5' => 'r1bq1rk1/pp1n1ppp/2pb1n2/4p3/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/8/2BPP3/2N2N2/PP3PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D46p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxc4 9. Bxc4

			variation => N__('Semi-Slav: Chigorin, 7.O-O O-O 8. e4 dxc4 9.Bxc4'),
			moves => {
				'e6e5' => 'r1bq1rk1/pp1n1ppp/2pb1n2/4p3/2BPP3/2N2N2/PP3PPP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/8/2PPN3/3B1N2/PP3PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D46q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxe4 9. Nxe4

			variation => N__('Semi-Slav: Chigorin, Main Line'),
			moves => {
				'f6e4' => 'r1bq1rk1/pp1n1ppp/2pbp3/8/2PPn3/3B1N2/PP3PPP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/8/2PPp3/2NB1N2/PP3PPP/R1BQ1RK1 w - - 0 9' => {
			eco => 'D46q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxe4

			variation => N__('Semi-Slav: Chigorin, Main Line'),
			moves => {
				'c3e4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/8/2PPN3/3B1N2/PP3PPP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/8/2pP4/2NBPN2/PPQ2PPP/R1B2RK1 w - - 0 9' => {
			eco => 'D46h',
                        # TRANSLATORS:

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 O-O'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B2RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/2pbpn2/8/2pPP3/2NB1N2/PP3PPP/R1BQ1RK1 w - - 0 9' => {
			eco => 'D46p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O O-O 8. e4 dxc4

			variation => N__('Semi-Slav: Chigorin, 7.O-O O-O 8. e4 dxc4'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp1n1ppp/2pbpn2/8/2BPP3/2N2N2/PP3PPP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1B2RK1 b - - 2 9' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. Qe2

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. Qe2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1n1ppp/p3pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1B2RK1 w - - 0 10',
				'b7b6' => 'r1bq1rk1/p2n1ppp/1p2pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1B2RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/2N1PN2/PP3PPP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7

			variation => N__('Nimzo-Indian: Main Line, Bronstein Variation'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/P1N1PN2/1P3PPP/R1BQ1RK1 b - - 0 9',
				'c3e2' => 'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/4PN2/PP2NPPP/R1BQ1RK1 b - - 2 9',
				'd1e2' => 'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/2N1PN2/PP2QPPP/R1B2RK1 b - - 2 9',
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/4PN2/PP2NPPP/R1BQ1RK1 b - - 2 9' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. Ne2

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. Ne2'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/2p5/1bBP4/P1N1PN2/1P3PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. a3

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. a3'),
			moves => {
				'c5d4' => 'r1bq1rk1/pp1n1ppp/4pn2/8/1bBp4/P1N1PN2/1P3PPP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/2pp4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E53',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nbd7

			variation => N__('Nimzo-Indian: Main Line, 7...Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/8/1bBP4/P1N2N2/1P3PPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E55',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nbd7 9. a3 cxd4 10. exd4

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. a3 cd4 10.ed4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1ppp/4pn2/8/1bBp4/P1N1PN2/1P3PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'E55',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Main Line, Bronstein, 9. a3'),
			moves => {
				'e3d4' => 'r1bq1rk1/pp1n1ppp/4pn2/8/1bBP4/P1N2N2/1P3PPP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp1n2bp/2n1ppp1/3pN3/3P1P2/2N1B1P1/PP2P1BP/R2Q1RK1 w - - 0 12' => {
			eco => 'D79m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Ne5 e6 9. Nc3 Nfd7 10. f4 Nc6 11. Be3 f6

			variation => N__('Neo-Grünfeld, Main Line, 10. f4 Nc6 11. Be3 f6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n2pp/2n1pp2/2bpP3/5PQ1/P1N2N2/1PP3PP/R1B1KB1R w KQ - 0 10' => {
			eco => 'C11g',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. e5 Nfd7 5. f4 c5 6. dxc5 Nc6 7. a3 Bxc5 8. Qg4 O-O 9. Nf3 f6

			variation => N__('French: Steinitz, Brodsky-Jones Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n2pp/2p1pn2/3p1p2/2PP4/NP3NP1/P3PPBP/R2Q1RK1 w - - 1 10' => {
			eco => 'A94',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bxa3 9. Nxa3 Nbd7

			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bxa3 9.Nxa3 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nB1pp/2p1pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RN1Q1RK1 b - - 0 9' => {
			eco => 'A94',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Nbd7 9. Bxe7

			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Nbd7 9.Bxe7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nBppp/2p1p3/3n4/2BP4/2N1PN2/PP3PPP/2RQK2R b K - 0 10' => {
			eco => 'D67',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Main Line, Capablanca Freeing Manoevure'),
			moves => {
				'd8e7' => 'r1b2rk1/pp1nqppp/2p1p3/3n4/2BP4/2N1PN2/PP3PPP/2RQK2R w K - 0 11',
			},
		},
		'r1bq1rk1/pp1nBppp/4p3/2pn4/3P4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 0 9' => {
			eco => 'D62',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 Nxd5'),
			moves => {
				'd8e7' => 'r1b2rk1/pp1nqppp/4p3/2pn4/3P4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 10',
			},
		},
		'r1bq1rk1/pp1nb1pp/2p1pn2/3p1p2/2PP4/BP3NP1/P3PPBP/RN1Q1RK1 w - - 2 9' => {
			eco => 'A94',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Nbd7

			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Nbd7'),
			moves => {
				'a3e7' => 'r1bq1rk1/pp1nB1pp/2p1pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RN1Q1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p1pn1p/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 9' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 h6

			variation => N__('QGD: Orthodox, Rubinstein Attack, 8... h6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbpp1/2p1pn1p/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R w K - 0 9' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 h6

			variation => N__('QGD: Orthodox, Main Line, 8... h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pp1nbpp1/2p1pn1p/3p4/2PP3B/2NBPN2/PP3PPP/2RQK2R b K - 1 9',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p1pn1p/3p4/2PP3B/2NBPN2/PP3PPP/2RQK2R b K - 1 9' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 h6 9. Bh4

			variation => N__('QGD: Orthodox, Main Line, 8... h6 9. Bh4'),
			moves => {
				'd5c4' => 'r1bq1rk1/pp1nbpp1/2p1pn1p/8/2pP3B/2NBPN2/PP3PPP/2RQK2R w K - 0 10',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p1pn1p/8/2BP3B/2N1PN2/PP3PPP/2RQK2R b K - 0 10' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 h6 9. Bh4 dxc4 10. Bxc4

			variation => N__('QGD: Orthodox, Main Line, 8... h6 9. Bh4 dxc4 10.Bxc4'),
			moves => {
				'b7b5' => 'r1bq1rk1/p2nbpp1/2p1pn1p/1p6/2BP3B/2N1PN2/PP3PPP/2RQK2R w K b6 0 11',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p1pn1p/8/2pP3B/2NBPN2/PP3PPP/2RQK2R w K - 0 10' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 h6 9. Bh4 dxc4

			variation => N__('QGD: Orthodox, Main Line, 8... h6 9. Bh4 dxc4'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp1nbpp1/2p1pn1p/8/2BP3B/2N1PN2/PP3PPP/2RQK2R b K - 0 10',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p2n1p/3p2B1/3P4/2NBP3/PPQ1NPPP/R3K2R w KQ - 0 10' => {
			eco => 'D36k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O 9. Nge2 h6

			variation => N__('QGD: Exchange, Main Line, 9.Nge2 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pp1nbpp1/2p2n1p/3p4/3P3B/2NBP3/PPQ1NPPP/R3K2R b KQ - 1 10',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p2n1p/3p2B1/3P4/2NBPN2/PPQ2PPP/R3K2R w KQ - 0 10' => {
			eco => 'D36o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O 9. Nf3 h6

			variation => N__('QGD: Exchange, Main Line, 9. Nf3 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pp1nbpp1/2p2n1p/3p4/3P3B/2NBPN2/PPQ2PPP/R3K2R b KQ - 1 10',
			},
		},
		'r1bq1rk1/pp1nbpp1/2p2n1p/3p4/3P3B/2NBP3/PPQ1NPPP/R3K2R b KQ - 1 10' => {
			eco => 'D36k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O 9. Nge2 h6 10. Bh4

			variation => N__('QGD: Exchange, Main Line, 9.Nge2 h6 10. Bh4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbpp1/2p2n1p/3p4/3P3B/2NBPN2/PPQ2PPP/R3K2R b KQ - 1 10' => {
			eco => 'D36o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O 9. Nf3 h6 10. Bh4

			variation => N__('QGD: Exchange, Main Line, 9. Nf3 h6 10. Bh4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbpp1/4pn1p/2pp4/2PP3B/2N1PN2/PPQ2PPP/R3KB1R w KQ c6 0 9' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 h6 8. Bh4 c5

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 h6 8. Bh4 c5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1p3/3n2B1/2BP3P/2N1PN2/PP3PP1/2RQK2R b K h3 0 10' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5 10. h4

			variation => N__('QGD: Orthodox, Main Line, Janowski Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1p3/3n2B1/2BP4/2N1PN2/PP3PPP/2RQK2R w K - 1 10' => {
			eco => 'D67',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4 Nd5

			variation => N__('QGD: Orthodox, Main Line, Capablanca Freeing Manoevure'),
			moves => {
				'g5e7' => 'r1bq1rk1/pp1nBppp/2p1p3/3n4/2BP4/2N1PN2/PP3PPP/2RQK2R b K - 0 10',
				'h2h4' => 'r1bq1rk1/pp1nbppp/2p1p3/3n2B1/2BP3P/2N1PN2/PP3PP1/2RQK2R b K h3 0 10',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1p3/3p2B1/2PPn3/2N1PN2/PPQ2PPP/2R1KB1R w K - 2 9' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 Ne4

			variation => N__('QGD: Orthodox, Rubinstein Attack, Wolf Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1p3/3p3n/2PP1B2/5NP1/PPQ1PPBP/RN3RK1 w - - 2 9' => {
			eco => 'E08h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Bf4 Nh5

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. Bf4 Nh5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1p3/3p4/2PPnB2/5NP1/PPQ1PPBP/RN3RK1 w - - 2 9' => {
			eco => 'E08i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Bf4 Ne4

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. Bf4 Ne4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/1QN1PN2/PP3PPP/R3KB1R w KQ - 0 8' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qb3 c6

			variation => N__('QGD: Orthodox, Rauzer Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6

			variation => N__('QGD: Orthodox, 7. Rc1 c6'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/P1N1PN2/1P3PPP/2RQKB1R b K - 0 8',
				'd1c2' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R b K - 1 8',
				'f1d3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R b K - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R b K - 1 8' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2

			variation => N__('QGD: Orthodox, Rubinstein Attack'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p1p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 9',
				'd5c4' => 'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2pP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 9',
				'f6e4' => 'r1bq1rk1/pp1nbppp/2p1p3/3p2B1/2PPn3/2N1PN2/PPQ2PPP/2R1KB1R w K - 2 9',
				'f8e8' => 'r1bqr1k1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 2 9',
				'h7h6' => 'r1bq1rk1/pp1nbpp1/2p1pn1p/3p2B1/2PP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/3RKB1R b K - 1 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c6 8. Rd1

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c6 8. Rd1'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c6

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c6'),
			moves => {
				'a1d1' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/3RKB1R b K - 1 8',
				'a2a3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/P1N1PN2/1PQ2PPP/R3KB1R b KQ - 0 8',
				'f1d3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PPQ2PPP/R3K2R b KQ - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R b K - 1 8' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3

			variation => N__('QGD: Orthodox, Main Line'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p1p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R w K - 0 9',
				'd5c4' => 'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2pP4/2NBPN2/PP3PPP/2RQK2R w K - 0 9',
				'f8e8' => 'r1bqr1k1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R w K - 2 9',
				'h7h6' => 'r1bq1rk1/pp1nbpp1/2p1pn1p/3p2B1/2PP4/2NBPN2/PP3PPP/2RQK2R w K - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R w KQ - 0 8' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 c6

			variation => N__('QGD: Orthodox, Botvinnik, 7... c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PPQ2PPP/R3K2R b KQ - 1 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c6 8. Bd3

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c6 8. Bd3'),
			moves => {
				'd5c4' => 'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2pP4/2NBPN2/PPQ2PPP/R3K2R w KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/P1N1PN2/1P3PPP/2RQKB1R b K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. a3

			variation => N__('QGD: Orthodox, 7. Rc1 c6 8. a3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/P1N1PN2/1PQ2PPP/R3KB1R b KQ - 0 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c6 8. a3

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c6 8. a3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP1B2/5NP1/PPQ1PPBP/RN3RK1 b - - 1 8' => {
			eco => 'E08g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Bf4

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. Bf4'),
			moves => {
				'b7b6' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP1B2/5NP1/PPQ1PPBP/RN3RK1 w - - 0 9',
				'f6e4' => 'r1bq1rk1/pp1nbppp/2p1p3/3p4/2PPnB2/5NP1/PPQ1PPBP/RN3RK1 w - - 2 9',
				'f6h5' => 'r1bq1rk1/pp1nbppp/2p1p3/3p3n/2PP1B2/5NP1/PPQ1PPBP/RN3RK1 w - - 2 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 b - - 0 8' => {
			eco => 'E08k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. b3

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. b3'),
			moves => {
				'b7b5' => 'r1bq1rk1/p2nbppp/2p1pn2/1p1p4/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 w - b6 0 9',
				'b7b6' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w - - 0 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. b3 c6

			variation => N__('Catalan: Closed, 6...Nbd7 7. b3'),
			moves => {
				'c1a3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/BP3NP1/P3PPBP/RN1Q1RK1 b - - 1 8',
				'c1b2' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b - - 1 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. b3 c6 8. Bb2

			variation => N__('Catalan: Closed, 6...Nbd7 7. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3 c6 8. b3

			variation => N__('Catalan: Closed, 6...Nbd7 7. Nc3 c6 8. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1PNBPN2/P4PPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'D46d',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Be7 7. O-O O-O 8. b3

			variation => N__('Semi-Slav: Bogoljubow, 7.O-O O-O 8. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1QN2NP1/PP2PPBP/R1B2RK1 b - - 1 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3 c6 8. Qb3

			variation => N__('Catalan: Closed, 6...Nbd7 7. Nc3 c6 8. Qb3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3 c6

			variation => N__('Catalan: Closed, 6...Nbd7 7. Nc3 c6'),
			moves => {
				'b2b3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8',
				'd1b3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1QN2NP1/PP2PPBP/R1B2RK1 b - - 1 8',
				'd1d3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/2NQ1NP1/PP2PPBP/R1B2RK1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8' => {
			eco => 'D46c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Be7 7. O-O O-O

			variation => N__('Semi-Slav: Bogoljubow, 7.O-O O-O'),
			moves => {
				'b2b3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1PNBPN2/P4PPP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/2NQ1NP1/PP2PPBP/R1B2RK1 b - - 1 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3 c6 8. Qd3

			variation => N__('Catalan: Closed, Botvinnik Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - - 0 8' => {
			eco => 'E08d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6

			variation => N__('Catalan: Closed, 7. Qc2 c6'),
			moves => {
				'b1d2' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 b - - 1 8',
				'b2b3' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 b - - 0 8',
				'c1f4' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP1B2/5NP1/PPQ1PPBP/RN3RK1 b - - 1 8',
				'f1d1' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNBR2K1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNBR2K1 b - - 1 8' => {
			eco => 'E08e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Rd1

			variation => N__('Catalan: Closed, 7. Qc2 c6 8. Rd1'),
			moves => {
				'b7b6' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNBR2K1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 b - - 1 8' => {
			eco => 'E09a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c6 8. Nbd2

			variation => N__('Catalan: Closed, Main Line'),
			moves => {
				'b7b5' => 'r1bq1rk1/p2nbppp/2p1pn2/1p1p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 w - b6 0 9',
				'b7b6' => 'r1bq1rk1/p2nbppp/1pp1pn2/3p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQNPPBP/R1B2RK1 w - - 2 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/BP3NP1/P3PPBP/RN1Q1RK1 b - - 1 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. b3 c6 8. Ba3

			variation => N__('Catalan: Closed, 6...Nbd7 7. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2BP4/2N1PN2/PP3PPP/2RQK2R b K - 0 9' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4 9. Bxc4

			variation => N__('QGD: Orthodox, Main Line, 8...dxc4'),
			moves => {
				'b7b5' => 'r1bq1rk1/p2nbppp/2p1pn2/1p4B1/2BP4/2N1PN2/PP3PPP/2RQK2R w K b6 0 10',
				'f6d5' => 'r1bq1rk1/pp1nbppp/2p1p3/3n2B1/2BP4/2N1PN2/PP3PPP/2RQK2R w K - 1 10',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2BP4/2N1PN2/PPQ2PPP/R3K2R b KQ - 0 9' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c6 8. Bd3 dxc4 9. Bxc4

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c6 8. Bd3 dxc4 9.Bxc4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2pP4/2N1PN2/PPQ2PPP/2R1KB1R w K - 0 9' => {
			eco => 'D64',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Qc2 dxc4

			variation => N__('QGD: Orthodox, Rubinstein Attack, 8...dxc4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2pP4/2NBPN2/PP3PPP/2RQK2R w K - 0 9' => {
			eco => 'D66',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 c6 8. Bd3 dxc4

			variation => N__('QGD: Orthodox, Main Line, 8...dxc4'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2BP4/2N1PN2/PP3PPP/2RQK2R b K - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2pP4/2NBPN2/PPQ2PPP/R3K2R w KQ - 0 9' => {
			eco => 'D61',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c6 8. Bd3'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp1nbppp/2p1pn2/6B1/2BP4/2N1PN2/PPQ2PPP/R3K2R b KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ1NPPP/R3K2R b KQ - 4 9' => {
			eco => 'D36k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O 9. Nge2

			variation => N__('QGD: Exchange, Main Line, 9.Nge2'),
			moves => {
				'f8e8' => 'r1bqr1k1/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ1NPPP/R3K2R w KQ - 5 10',
				'h7h6' => 'r1bq1rk1/pp1nbpp1/2p2n1p/3p2B1/3P4/2NBP3/PPQ1NPPP/R3K2R w KQ - 0 10',
			},
		},
		'r1bq1rk1/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ2PPP/R3K1NR w KQ - 3 9' => {
			eco => 'D36k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O

			variation => N__('QGD: Exchange, Main Line'),
			moves => {
				'g1e2' => 'r1bq1rk1/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ1NPPP/R3K2R b KQ - 4 9',
				'g1f3' => 'r1bq1rk1/pp1nbppp/2p2n2/3p2B1/3P4/2NBPN2/PPQ2PPP/R3K2R b KQ - 4 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2p2n2/3p2B1/3P4/2NBPN2/PPQ2PPP/R3K2R b KQ - 4 9' => {
			eco => 'D36o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3 O-O 9. Nf3

			variation => N__('QGD: Exchange, Main Line, 9. Nf3'),
			moves => {
				'f8e8' => 'r1bqr1k1/pp1nbppp/2p2n2/3p2B1/3P4/2NBPN2/PPQ2PPP/R3K2R w KQ - 5 10',
				'h7h6' => 'r1bq1rk1/pp1nbpp1/2p2n1p/3p2B1/3P4/2NBPN2/PPQ2PPP/R3K2R w KQ - 0 10',
			},
		},
		'r1bq1rk1/pp1nbppp/2p4n/3pp1N1/2BPPP2/2P5/PP4PP/RNBQ1RK1 w - - 0 9' => {
			eco => 'C41g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nd7 4. Bc4 c6 5. Ng5 Nh6 6. f4 Be7 7. O-O O-O 8. c3 d5

			variation => N__('Philidor: Hanham, Berger Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p1B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 O-O 7. Qc2 c6

			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 O-O 7. Qc2 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2BPP3/2N2N2/PPP1QPPP/R1B2RK1 w - - 0 8' => {
			eco => 'C41w',
                        # TRANSLATORS:

			variation => N__('Philidor: Improved Hanham, 7. Qe2'),
			moves => {
				'a2a4' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP1QPPP/R1B2RK1 b - a3 0 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2BPP3/2N2N2/PPP2PPP/R1BQR1K1 w - - 0 8' => {
			eco => 'C41x',
                        # TRANSLATORS:

			variation => N__('Philidor: Improved Hanham, 7. Re1'),
			moves => {
				'a2a4' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQR1K1 b - a3 0 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O 7. O-O c6

			variation => N__('Old Indian: 5. g3 Be7, Main Line'),
			moves => {
				'd1c2' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b - - 1 8' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O 7. O-O c6 8. Qc2

			variation => N__('Old Indian: 5. g3 Be7, Main Line, 8. Qc2'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8' => {
			eco => 'A55m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Be3

			variation => N__('Old Indian: Main Line, 8. Be3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N1P/PP2BPP1/R1BQ1RK1 b - - 0 8' => {
			eco => 'A55l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. h3

			variation => N__('Old Indian: Main Line, 8. h3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'A55k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O

			variation => N__('Old Indian: Main Line'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8',
				'd1c2' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 b - - 3 8',
				'f1e1' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b - - 3 8',
				'h2h3' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N1P/PP2BPP1/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b - - 3 8' => {
			eco => 'A55q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1

			variation => N__('Old Indian: Main Line, 8. Re1'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 4 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 b - - 3 8' => {
			eco => 'A55o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2

			variation => N__('Old Indian: Main Line, 8. Qc2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 0 9',
				'd8c7' => 'r1b2rk1/ppqnbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 4 9',
				'f8e8' => 'r1bqr1k1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 4 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - - 3 8' => {
			eco => 'A55e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9',
				'd8c7' => 'r1b2rk1/ppqnbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 4 9',
				'f8e8' => 'r1bqr1k1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 4 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQK2R w KQ - 2 8' => {
			eco => 'A55e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - - 3 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP1QPPP/R1B2RK1 b - - 1 8' => {
			eco => 'C41w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. a4 c6 8. Qe2

			variation => N__('Philidor: Improved Hanham, 7. Qe2 c6 8. a4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP1QPPP/R1B2RK1 b - a3 0 8' => {
			eco => 'C41w',
                        # TRANSLATORS:

			variation => N__('Philidor: Improved Hanham, 7. Qe2'),
			moves => {
				'e5d4' => 'r1bq1rk1/pp1nbppp/2pp1n2/8/P1BpP3/2N2N2/1PP1QPPP/R1B2RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'C41v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. a4 c6

			variation => N__('Philidor: Improved Hanham, 7. a4 c6'),
			moves => {
				'd1e2' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP1QPPP/R1B2RK1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQR1K1 b - a3 0 8' => {
			eco => 'C41x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. Re1 c6 8. a4

			variation => N__('Philidor: Improved Hanham, 7. Re1 c6 8. a4'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p1nbppp/2pp1n2/p3p3/P1BPP3/2N2N2/1PP2PPP/R1BQR1K1 w - a6 0 9',
				'b7b6' => 'r1bq1rk1/p2nbppp/1ppp1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQR1K1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/8/P1BpP3/2N2N2/1PP1QPPP/R1B2RK1 w - - 0 9' => {
			eco => 'C41w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. Qe2 c6 8. a4 exd4

			variation => N__('Philidor: Improved Hanham, Sozin Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp3n/4p1N1/2BPPP2/2P5/PP4PP/RNBQ1RK1 b - - 0 8' => {
			eco => 'C41g',
                        # TRANSLATORS:

			variation => N__('Philidor: Hanham, Kmoch Variation'),
			moves => {
				'd6d5' => 'r1bq1rk1/pp1nbppp/2p4n/3pp1N1/2BPPP2/2P5/PP4PP/RNBQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/2pp3n/4p1N1/2BPPP2/8/PPP3PP/RNBQ1RK1 w - - 3 8' => {
			eco => 'C41g',
                        # TRANSLATORS:

			variation => N__('Philidor: Hanham, Kmoch Variation'),
			moves => {
				'c2c3' => 'r1bq1rk1/pp1nbppp/2pp3n/4p1N1/2BPPP2/2P5/PP4PP/RNBQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1nbppp/4p3/2pn2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 9' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 Nxd5

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 Nxd5'),
			moves => {
				'g5e7' => 'r1bq1rk1/pp1nBppp/4p3/2pn4/3P4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2p3B1/2BP4/2N1PN2/PP3PPP/R2Q1RK1 b - - 1 9' => {
			eco => 'D60',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, Botvinnik, 8... c5'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nbppp/p3pn2/2p3B1/2BP4/2N1PN2/PP3PPP/R2Q1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2p3B1/2BP4/2N1PN2/PP3PPP/R2QK2R w KQ c6 0 9' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 dxc4 8. Bxc4 c5

			variation => N__('QGD: Orthodox, Botvinnik, 8... c5'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1nbppp/4pn2/2p3B1/2BP4/2N1PN2/PP3PPP/R2Q1RK1 b - - 1 9',
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2pP2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 0 8' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5'),
			moves => {
				'c5d4' => 'r1bq1rk1/pp1nbppp/4pn2/3P2B1/3p4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 9',
				'e6d5' => 'r1bq1rk1/pp1nbppp/5n2/2pp2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 9',
				'f6d5' => 'r1bq1rk1/pp1nbppp/4p3/2pn2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2pp2B1/2PP4/2N1PN2/PPQ2PPP/2KR1B1R b - - 1 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. O-O-O

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.O-O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2pp2B1/2PP4/2N1PN2/PPQ2PPP/3RKB1R b K - 1 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. Rd1

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8. Rd1'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2pp2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ c6 0 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5'),
			moves => {
				'a1d1' => 'r1bq1rk1/pp1nbppp/4pn2/2pp2B1/2PP4/2N1PN2/PPQ2PPP/3RKB1R b K - 1 8',
				'c4d5' => 'r1bq1rk1/pp1nbppp/4pn2/2pP2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 0 8',
				'e1c1' => 'r1bq1rk1/pp1nbppp/4pn2/2pp2B1/2PP4/2N1PN2/PPQ2PPP/2KR1B1R b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/2pp4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - c6 0 8' => {
			eco => 'E08b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2 c5

			variation => N__('Catalan: Closed, 7. Qc2 c5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/4pn2/3P2B1/3p4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 9' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 cxd4

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 cxd4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/5n2/2pp2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 9' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 exd5

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 exd5'),
			moves => {
				'f1d3' => 'r1bq1rk1/pp1nbppp/5n2/2pp2B1/3P4/2NBPN2/PPQ2PPP/R3K2R b KQ - 1 9',
			},
		},
		'r1bq1rk1/pp1nbppp/5n2/2pp2B1/3P4/2NBPN2/PPQ2PPP/R3K2R b KQ - 1 9' => {
			eco => 'D62',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 c5 8. cxd5 exd5 9. Bd3

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 c5 8.cxd5 exd5 9. Bd3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nn1bp/3p2p1/2pPpp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 w - c6 0 12' => {
			eco => 'E98o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Nd3 f5 11. Bd2 c5

			variation => N__('King\'s Indian: Mar del Plata, Fischer Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2np2p1/8/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 5 10' => {
			eco => 'B77d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Nd7

			variation => N__('Sicilian: Dragon, Yugoslav, 9. Bc4 Nd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2np2p1/8/2P1P3/2N5/PPN1BPPP/R1BQ1RK1 w - - 4 10' => {
			eco => 'B37n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9. O-O Nd7

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9.O-O Nd7'),
			moves => {
				'c1d2' => 'r1bq1rk1/pp1nppbp/2np2p1/8/2P1P3/2N5/PPNBBPPP/R2Q1RK1 b - - 5 10',
			},
		},
		'r1bq1rk1/pp1nppbp/2np2p1/8/2P1P3/2N5/PPNBBPPP/R2Q1RK1 b - - 5 10' => {
			eco => 'B37o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9. O-O Nd7 10. Bd2

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9.O-O Nd7 10. Bd2'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p1nppbp/2np2p1/p7/2P1P3/2N5/PPNBBPPP/R2Q1RK1 w - a6 0 11',
				'd7c5' => 'r1bq1rk1/pp2ppbp/2np2p1/2n5/2P1P3/2N5/PPNBBPPP/R2Q1RK1 w - - 6 11',
			},
		},
		'r1bq1rk1/pp1nppbp/2np2p1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 2 10' => {
			eco => 'B38o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 d6 9. O-O Nd7

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 d6 9.O-O Nd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2p2np1/3p4/2PP4/5NP1/PP1NPPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'D78l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. Nbd2 Nbd7

			variation => N__('Neo-Grünfeld, 6.O-O c6 7.Nbd2 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/6B1/2PP4/2N1PN2/PP2BPPP/R2Q1RK1 b - - 1 8' => {
			eco => 'E61z',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 O-O 5. Bg5 d6 6. e3 Nbd7 7. Be2 c6 8. O-O

			variation => N__('King\'s Indian: Smyslov System, 5...O-O 6. e3 Nbd7 7. Be2 c6 8.O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/6B1/2PP4/2N1PN2/PP2BPPP/R2QK2R w KQ - 0 8' => {
			eco => 'E61z',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 O-O 5. Bg5 d6 6. e3 Nbd7 7. Be2 c6

			variation => N__('King\'s Indian: Smyslov System, 5...O-O 6. e3 Nbd7 7. Be2 c6'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1nppbp/2pp1np1/6B1/2PP4/2N1PN2/PP2BPPP/R2Q1RK1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/6B1/2PPP3/2N5/PP1QBPPP/R3K1NR w KQ - 2 8' => {
			eco => 'E73l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 c6 7. Qd2 Nbd7

			variation => N__('King\'s Indian: Averbakh, 6... c6 7. Qd2 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E67c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O c6

			variation => N__('King\'s Indian: Fianchetto with 6... Nd7 7.O-O c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E91t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nbd7 7. O-O c6

			variation => N__('King\'s Indian: 6. Be2 Nbd7 7.O-O c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/8/P2PP3/2N2N2/1PP1BPPP/R1BQ1RK1 w - - 1 8' => {
			eco => 'B08s',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O c6 7. a4 Nbd7

			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O c6 7. a4 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/3p1np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E65c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nbd7

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, 7.O-O Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/3p1np1/2p5/2PPP3/2N1BP2/PP1Q2PP/R3KBNR w KQ c6 0 8' => {
			eco => 'E81s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nbd7 7. Qd2 c5

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 Nbd7 7. Qd2 c5'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp1nppbp/3p1np1/2pP4/2P1P3/2N1BP2/PP1Q2PP/R3KBNR b KQ - 0 8',
				'g1e2' => 'r1bq1rk1/pp1nppbp/3p1np1/2p5/2PPP3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 1 8',
			},
		},
		'r1bq1rk1/pp1nppbp/3p1np1/2p5/2PPP3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 1 8' => {
			eco => 'E81t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nbd7 7. Qd2 c5 8. Nge2

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 Nbd7 7. Qd2 c5 8.Nge2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1nppbp/p2p1np1/2p5/2PPP3/2N1BP2/PP1QN1PP/R3KB1R w KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1nppbp/3p1np1/2pP4/2P1P3/2N1BP2/PP1Q2PP/R3KBNR b KQ - 0 8' => {
			eco => 'E81s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nbd7 7. Qd2 c5 8. d5

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 Nbd7 7. Qd2 c5 8. d5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/5np1/2pp2B1/3P4/2PBPN2/PP1N1PPP/R2QK2R w KQ - 1 8' => {
			eco => 'D03u',
                        # TRANSLATORS: 1. d4 d5 2. Nf3 Nf6 3. Bg5 g6 4. Nbd2 Bg7 5. e3 O-O 6. Bd3 c5 7. c3 Nbd7

			variation => N__('Torre Attack: 3... g6 4. e3 Bg7 5.Nbd2 O-O 6. Bd3 c5 7. c3 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nppbp/5np1/3P4/3P4/1QN5/PP2BPPP/R1B1K1NR w KQ - 3 9' => {
			eco => 'B14h',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3 Bg7 7. cxd5 O-O 8. Be2 Nbd7

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6, Main Line, 8. Be2 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/2n1pn2/2b5/2P5/P1N2N2/1PQ1PPPP/R1B1KB1R w KQ - 2 8' => {
			eco => 'E39k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 O-O 6. a3 Bxc5 7. Nf3 Nc6

			variation => N__('Nimzo-Indian: Classical, Pirc, 6. a3 Bxc5 7. Nf3 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 b - - 3 7' => {
			eco => 'E50',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 c5 6. Bd3 Nc6 7. O-O

			variation => N__('Nimzo-Indian: 4. e3 O-O 5. Nf3 c5 6. Bd3 Nc6 7.O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBPN2/PP3PPP/R1BQK2R w KQ - 2 7' => {
			eco => 'E50',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 c5 6. Bd3 Nc6

			variation => N__('Nimzo-Indian: 4. e3 O-O 5. Nf3 c5 6. Bd3 Nc6'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 b - - 3 7',
			},
		},
		'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBP3/4NPPP/R1BQK2R b KQ - 3 8' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2'),
			moves => {
				'b7b6' => 'r1bq1rk1/p2p1ppp/1pn1pn2/2p5/2PP4/P1PBP3/4NPPP/R1BQK2R w KQ - 0 9',
				'd7d6' => 'r1bq1rk1/pp3ppp/2nppn2/2p5/2PP4/P1PBP3/4NPPP/R1BQK2R w KQ - 0 9',
				'e6e5' => 'r1bq1rk1/pp1p1ppp/2n2n2/2p1p3/2PP4/P1PBP3/4NPPP/R1BQK2R w KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBP3/5PPP/R1BQK1NR w KQ - 2 8' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6'),
			moves => {
				'g1e2' => 'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBP3/4NPPP/R1BQK2R b KQ - 3 8',
				'g1f3' => 'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBPN2/5PPP/R1BQK2R b KQ - 3 8',
			},
		},
		'r1bq1rk1/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBPN2/5PPP/R1BQK2R b KQ - 3 8' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6 8. Nf3

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6 8. Nf3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/2n2n2/2p1p3/2PP4/P1PBP3/4NPPP/R1BQK2R w KQ - 0 9' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 e5

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/2N2N2/PPQ1PPPP/R1B1KB1R w KQ - 3 7' => {
			eco => 'E39e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 O-O 6. Nf3 Na6

			variation => N__('Nimzo-Indian: Classical, Pirc, 6. Nf3 Na6'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/P1N2N2/1PQ1PPPP/R1B1KB1R b KQ - 0 7',
				'c1d2' => 'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/2N2N2/PPQBPPPP/R3KB1R b KQ - 4 7',
				'g2g3' => 'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/2N2NP1/PPQ1PP1P/R1B1KB1R b KQ - 0 7',
			},
		},
		'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/2N2N2/PPQBPPPP/R3KB1R b KQ - 4 7' => {
			eco => 'E39f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 O-O 6. Nf3 Na6 7. Bd2

			variation => N__('Nimzo-Indian: Classical, Pirc, 6. Nf3 Na6 7. Bd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/2N2NP1/PPQ1PP1P/R1B1KB1R b KQ - 0 7' => {
			eco => 'E39h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 O-O 6. Nf3 Na6 7. g3

			variation => N__('Nimzo-Indian: Classical, Pirc, 6. Nf3 Na6 7. g3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1p1ppp/n3pn2/2P5/1bP5/P1N2N2/1PQ1PPPP/R1B1KB1R b KQ - 0 7' => {
			eco => 'E39g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 O-O 6. Nf3 Na6 7. a3

			variation => N__('Nimzo-Indian: Classical, Pirc, 6. Nf3 Na6 7. a3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 w - - 4 8' => {
			eco => 'A36p',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'd2d3' => 'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A36q',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7 7. O-O O-O 8. d3

			variation => N__('English: Symmetrical, Botvinnik, 5... e6, 8. d3'),
			moves => {
				'd7d6' => 'r1bq1rk1/pp2npbp/2npp1p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP1BPPBP/R2Q1RK1 b - - 2 8' => {
			eco => 'A37p',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. d3 O-O 8. Bd2

			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. d3 O-O 8. Bd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 1 8' => {
			eco => 'A37o',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. d3 O-O

			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. d3 O-O'),
			moves => {
				'c1d2' => 'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP1BPPBP/R2Q1RK1 b - - 2 8',
			},
		},
		'r1bq1rk1/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 1 8' => {
			eco => 'A37y',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7 7. d3 O-O

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7 7. d3 O-O'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp1pnpbp/2n3p1/2p1p3/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1pnpbp/2n3p1/2p1p3/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A37y',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7 7. d3 O-O 8. a3

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7 7. d3 O-O 8. a3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/3PP3/2P2N2/PP3PPP/RNBQR1K1 b - d3 0 7' => {
			eco => 'B31n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. d4

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. d4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/4P3/2P2N1P/PP1P1PP1/RNBQR1K1 b - - 0 7' => {
			eco => 'B31m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. h3

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. h3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQR1K1 w - - 1 7' => {
			eco => 'B31l',
                        # TRANSLATORS:

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3'),
			moves => {
				'd2d4' => 'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/3PP3/2P2N2/PP3PPP/RNBQR1K1 b - d3 0 7',
				'h2h3' => 'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/4P3/2P2N1P/PP1P1PP1/RNBQR1K1 b - - 0 7',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/1PN2NP1/P2PPPBP/R1BQ1RK1 b - - 0 7' => {
			eco => 'A38h',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. b3

			variation => N__('English: Symmetrical, Main Line, 7. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/1RBQ1RK1 b - - 7 7' => {
			eco => 'A38g',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. Rb1

			variation => N__('English: Symmetrical, Main Line, 7. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w - - 6 7' => {
			eco => 'A38f',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O

			variation => N__('English: Symmetrical, Main Line, 6.O-O O-O'),
			moves => {
				'a1b1' => 'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/1RBQ1RK1 b - - 7 7',
				'a2a3' => 'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/P1N2NP1/1P1PPPBP/R1BQ1RK1 b - - 0 7',
				'b2b3' => 'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/1PN2NP1/P2PPPBP/R1BQ1RK1 b - - 0 7',
				'd2d3' => 'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b - - 0 7',
				'd2d4' => 'r1bq1rk1/pp1pppbp/2n2np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - d3 0 7',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b - - 0 7' => {
			eco => 'A38k',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3

			variation => N__('English: Symmetrical, Main Line, 7. d3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1pppbp/p1n2np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
				'd7d5' => 'r1bq1rk1/pp2ppbp/2n2np1/2pp4/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - d6 0 8',
				'd7d6' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/P1N2NP1/1P1PPPBP/R1BQ1RK1 b - - 0 7' => {
			eco => 'A38i',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. a3

			variation => N__('English: Symmetrical, Main Line, 7. a3'),
			moves => {
				'd7d6' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/P1N2NP1/1P1PPPBP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - d3 0 7' => {
			eco => 'A39a',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4

			variation => N__('English: Symmetrical, Main Line 7. d4'),
			moves => {
				'c5d4' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2Pp4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 b - - 7 8' => {
			eco => 'B35h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. O-O

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8.O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQ - 6 8' => {
			eco => 'B35f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O'),
			moves => {
				'c4b3' => 'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/1BN1B3/PPP2PPP/R2QK2R b KQ - 7 8',
				'e1g1' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 b - - 7 8',
				'f2f3' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1BP2/PPP3PP/R2QK2R b KQ - 0 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1BP2/PPP3PP/R2QK2R b KQ - 0 8' => {
			eco => 'B35g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. f3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. f3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2P1P3/2N5/PPN2PPP/R1BQKB1R w KQ - 5 8' => {
			eco => 'B37h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 O-O

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A39c',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4

			variation => N__('English: Symmetrical, Main Line 7. d4'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p1pppbp/p1n2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 9',
				'c6d4' => 'r1bq1rk1/pp1pppbp/5np1/8/2Pn4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 9',
				'd7d6' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 9',
				'd8a5' => 'r1b2rk1/pp1pppbp/2n2np1/q7/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 1 9',
				'd8b6' => 'r1b2rk1/pp1pppbp/1qn2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 1 9',
				'f6g4' => 'r1bq1rk1/pp1pppbp/2n3p1/8/2PN2n1/2N3P1/PP2PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R b KQ - 6 8' => {
			eco => 'B38g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2'),
			moves => {
				'b7b6' => 'r1bq1rk1/p2pppbp/1pn2np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R w KQ - 0 9',
				'd7d6' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R w KQ - 0 9',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R w KQ - 5 8' => {
			eco => 'B38g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O'),
			moves => {
				'f1e2' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R b KQ - 6 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2Pp4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'A39b',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4

			variation => N__('English: Symmetrical, Main Line 7. d4'),
			moves => {
				'f3d4' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/1BN1B3/PPP2PPP/R2QK2R b KQ - 7 8' => {
			eco => 'B35i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p1pppbp/2n2np1/p7/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ a6 0 9',
				'd7d6' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ - 0 9',
				'd8a5' => 'r1b2rk1/pp1pppbp/2n2np1/q7/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ - 8 9',
				'f6g4' => 'r1bq1rk1/pp1pppbp/2n3p1/8/3NP1n1/1BN1B3/PPP2PPP/R2QK2R w KQ - 8 9',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b - - 7 8' => {
			eco => 'B34v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O 8. O-O

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O 8.O-O'),
			moves => {
				'd7d5' => 'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w - d6 0 9',
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQ - 6 8' => {
			eco => 'B34u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b - - 7 8',
			},
		},
		'r1bq1rk1/pp1pppbp/2n3p1/8/2PN2n1/2N1P1P1/PP3PBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'A39h',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Ng4'),
			moves => {
				'd7d6' => 'r1bq1rk1/pp2ppbp/2np2p1/8/2PN2n1/2N1P1P1/PP3PBP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1pppbp/2n3p1/8/2PN2n1/2N3P1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A39h',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Ng4

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Ng4'),
			moves => {
				'e2e3' => 'r1bq1rk1/pp1pppbp/2n3p1/8/2PN2n1/2N1P1P1/PP3PBP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp1pppbp/2n3p1/8/3NP1n1/1BN1B3/PPP2PPP/R2QK2R w KQ - 8 9' => {
			eco => 'B35j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 Ng4

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 Ng4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/5np1/8/2PQ4/2N3P1/PP2PPBP/R1B2RK1 b - - 0 9' => {
			eco => 'A39j',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4

			variation => N__('English: Symmetrical, Main Line 7. d4, 9.Qxd4'),
			moves => {
				'd7d6' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2PQ4/2N3P1/PP2PPBP/R1B2RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp1pppbp/5np1/8/2Pn4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A39j',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4

			variation => N__('English: Symmetrical, Main Line 7. d4, 8...Nxd4'),
			moves => {
				'd1d4' => 'r1bq1rk1/pp1pppbp/5np1/8/2PQ4/2N3P1/PP2PPBP/R1B2RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp2bpp1/2n2n1p/3p2B1/3N4/2N3P1/PP2PPBP/R2Q1RK1 w - - 0 11' => {
			eco => 'D34s',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bg5 cxd4 10. Nxd4 h6

			variation => N__('QGD Tarrasch: 9. Bg5 cxd4 10.Nxd4 h6'),
			moves => {
				'g5e3' => 'r1bq1rk1/pp2bpp1/2n2n1p/3p4/3N4/2N1B1P1/PP2PPBP/R2Q1RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/pp2bpp1/2n2n1p/3p4/3N4/2N1B1P1/PP2PPBP/R2Q1RK1 b - - 1 11' => {
			eco => 'D34t',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bg5 cxd4 10. Nxd4 h6 11. Be3

			variation => N__('QGD Tarrasch: 9. Bg5 cxd4 10.Nxd4 h6 11. Be3'),
			moves => {
				'f8e8' => 'r1bqr1k1/pp2bpp1/2n2n1p/3p4/3N4/2N1B1P1/PP2PPBP/R2Q1RK1 w - - 2 12',
			},
		},
		'r1bq1rk1/pp2bpp1/2np1n1p/4p3/3NPP1B/2N5/PPPQ2PP/2KR1B1R w - - 0 11' => {
			eco => 'B64f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 h6 10. Bh4 e5

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. f4 h6 10. Bh4 e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bpp1/2np1n1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQR1K1 b - - 0 10' => {
			eco => 'B58n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nf3 h6 8. O-O Be7 9. Re1 O-O 10. h3

			variation => N__('Sicilian: Boleslavsky, 7. Nf3, 9. Re1 O-O 10. h3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2bpp1/p1np1n1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQR1K1 w - - 0 11',
				'c8e6' => 'r2q1rk1/pp2bpp1/2npbn1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQR1K1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp2bpp1/2np1n1p/4p3/4P3/2N2N2/PPP1BPPP/R1BQR1K1 w - - 4 10' => {
			eco => 'B58m',
                        # TRANSLATORS:

			variation => N__('Sicilian: Boleslavsky, 7. Nf3 h6 8.O-O Be7 9. Re1'),
			moves => {
				'h2h3' => 'r1bq1rk1/pp2bpp1/2np1n1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQR1K1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp2bpp1/2nppn1p/6B1/3NPP2/2N5/PPPQ2PP/2KR1B1R w - - 0 10' => {
			eco => 'B64c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 h6

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. f4 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pp2bpp1/2nppn1p/8/3NPP1B/2N5/PPPQ2PP/2KR1B1R b - - 1 10',
			},
		},
		'r1bq1rk1/pp2bpp1/2nppn1p/8/3NPP1B/2N5/PPPQ2PP/2KR1B1R b - - 1 10' => {
			eco => 'B64d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 h6 10. Bh4

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. f4 h6 10. Bh4'),
			moves => {
				'c8d7' => 'r2q1rk1/pp1bbpp1/2nppn1p/8/3NPP1B/2N5/PPPQ2PP/2KR1B1R w - - 2 11',
				'e6e5' => 'r1bq1rk1/pp2bpp1/2np1n1p/4p3/3NPP1B/2N5/PPPQ2PP/2KR1B1R w - - 0 11',
			},
		},
		'r1bq1rk1/pp2bpp1/3ppn1p/6B1/3QPP2/2N5/PPP3PP/2KR1B1R w - - 0 11' => {
			eco => 'B65b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 Nxd4 10. Qxd4 h6

			variation => N__('Sicilian: Richter-Rauzer, 7... Be7, 9. f4 Nxd4 10.Qxd4 h6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/1nn1p3/2p5/3PP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 1 10' => {
			eco => 'D41f',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O 9. e4 Nb6

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9. e4 Nb6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/2p5/1n1PP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 1 10' => {
			eco => 'D41g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O 9. e4 Ndb4

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9. e4 Ndb4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/2pN4/3P4/5NP1/PP2PPBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D41h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O 9. Nxd5

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9.Nxd5'),
			moves => {
				'e6d5' => 'r1bq1rk1/pp2bppp/2n5/2pp4/3P4/5NP1/PP2PPBP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/2pn4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 9' => {
			eco => 'D41d',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line'),
			moves => {
				'c3d5' => 'r1bq1rk1/pp2bppp/2n1p3/2pN4/3P4/5NP1/PP2PPBP/R1BQ1RK1 b - - 0 9',
				'e2e4' => 'r1bq1rk1/pp2bppp/2n1p3/2pn4/3PP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 9',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/2pn4/3P4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 9' => {
			eco => 'D42c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 Be7 8. O-O O-O

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bd3 Be7 8.O-O O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/2pn4/3PP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 9' => {
			eco => 'D41e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O 9. e4

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9. e4'),
			moves => {
				'd5b4' => 'r1bq1rk1/pp2bppp/2n1p3/2p5/1n1PP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 1 10',
				'd5b6' => 'r1bq1rk1/pp2bppp/1nn1p3/2p5/3PP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQ1RK1 w - - 3 10' => {
			eco => 'D41s',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bc4 cxd4 8. exd4 Be7 9. O-O O-O

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bc4'),
			moves => {
				'f1e1' => 'r1bq1rk1/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQR1K1 b - - 4 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQR1K1 b - - 4 10' => {
			eco => 'D41s',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bc4'),
			moves => {
				'd5c3' => 'r1bq1rk1/pp2bppp/2n1p3/8/2BP4/2n2N2/PP3PPP/R1BQR1K1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQ1RK1 w - - 3 10' => {
			eco => 'D42f',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 9.O-O O-O'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp2bppp/2n1p3/3n4/3P4/P1NB1N2/1P3PPP/R1BQ1RK1 b - - 0 10',
				'f1e1' => 'r1bq1rk1/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQR1K1 b - - 4 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQR1K1 b - - 4 10' => {
			eco => 'D42j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1'),
			moves => {
				'c6b4' => 'r1bq1rk1/pp2bppp/4p3/3n4/1n1P4/2NB1N2/PP3PPP/R1BQR1K1 w - - 5 11',
				'd5f6' => 'r1bq1rk1/pp2bppp/2n1pn2/8/3P4/2NB1N2/PP3PPP/R1BQR1K1 w - - 5 11',
				'e7f6' => 'r1bq1rk1/pp3ppp/2n1pb2/3n4/3P4/2NB1N2/PP3PPP/R1BQR1K1 w - - 5 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/3n4/3P4/P1NB1N2/1P3PPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'D42g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. a3

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. a3'),
			moves => {
				'd5f6' => 'r1bq1rk1/pp2bppp/2n1pn2/8/3P4/P1NB1N2/1P3PPP/R1BQ1RK1 w - - 1 11',
				'e7f6' => 'r1bq1rk1/pp3ppp/2n1pb2/3n4/3P4/P1NB1N2/1P3PPP/R1BQ1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/8/2BP4/2P2N2/P4PPP/R1BQR1K1 b - - 0 11' => {
			eco => 'D41t',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bc4 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nxc3 11. bxc3

			variation => N__('QGD: Semi-Tarrasch, 7. Bc4 Main Line'),
			moves => {
				'b7b6' => 'r1bq1rk1/p3bppp/1pn1p3/8/2BP4/2P2N2/P4PPP/R1BQR1K1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp2bppp/2n1p3/8/2BP4/2n2N2/PP3PPP/R1BQR1K1 w - - 0 11' => {
			eco => 'D41s',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bc4'),
			moves => {
				'b2c3' => 'r1bq1rk1/pp2bppp/2n1p3/8/2BP4/2P2N2/P4PPP/R1BQR1K1 b - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/1BPp2B1/3P4/2N2N2/PP3PPP/R2QK2R b KQ - 2 9' => {
			eco => 'B13z',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7 8. c5 O-O 9. Bb5

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation, 8. c5 O-O 9. Bb5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2Pp2B1/3P4/2N2N2/PP3PPP/R2QKB1R w KQ - 1 9' => {
			eco => 'B13z',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7 8. c5 O-O

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation, 8. c5 O-O'),
			moves => {
				'f1b5' => 'r1bq1rk1/pp2bppp/2n1pn2/1BPp2B1/3P4/2N2N2/PP3PPP/R2QK2R b KQ - 2 9',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/2P5/1P2PNP1/PB1P1PBP/RN1Q1RK1 b - - 0 8' => {
			eco => 'A14',
                        # TRANSLATORS: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 c5 7. Bb2 Nc6 8. e3

			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 c5 7. Bb2 Nc6 8. e3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 w - - 2 8' => {
			eco => 'A14',
                        # TRANSLATORS: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 c5 7. Bb2 Nc6

			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 c5 7. Bb2 Nc6'),
			moves => {
				'e2e3' => 'r1bq1rk1/pp2bppp/2n1pn2/2pp4/2P5/1P2PNP1/PB1P1PBP/RN1Q1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w - - 1 8' => {
			eco => 'A08',
                        # TRANSLATORS:

			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'f1e1' => 'r1bq1rk1/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQR1K1 b - - 2 8',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQR1K1 b - - 2 8' => {
			eco => 'A08',
                        # TRANSLATORS: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 Be7 7. e4 O-O 8. Re1

			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/6B1/2BP4/P1N2N2/1P3PPP/R2Q1RK1 b - - 2 11' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7 11. Bg5

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/6B1/3P4/2NB1N2/PP3PPP/R2QR1K1 b - - 6 11' => {
			eco => 'D42l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6 11. Bg5

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/2BP1B2/P1N2N2/1P3PPP/R2Q1RK1 b - - 2 11' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7 11. Bf4

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7 11. Bf4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1N1BN2/1P3PPP/R2Q1RK1 b - - 2 11' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7 11. Be3

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7 11. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1N2N2/1P3PPP/R1BQ1RK1 w - - 1 11' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1N1BN2/1P3PPP/R2Q1RK1 b - - 2 11',
				'c1f4' => 'r1bq1rk1/pp2bppp/2n1pn2/8/2BP1B2/P1N2N2/1P3PPP/R2Q1RK1 b - - 2 11',
				'c1g5' => 'r1bq1rk1/pp2bppp/2n1pn2/6B1/2BP4/P1N2N2/1P3PPP/R2Q1RK1 b - - 2 11',
				'd1d3' => 'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1NQ1N2/1P3PPP/R1B2RK1 b - - 2 11',
				'f1e1' => 'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1N2N2/1P3PPP/R1BQR1K1 b - - 2 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1N2N2/1P3PPP/R1BQR1K1 b - - 2 11' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7 11. Re1

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7 11. Re1'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2bppp/p1n1pn2/8/2BP4/P1N2N2/1P3PPP/R1BQR1K1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1NQ1N2/1P3PPP/R1B2RK1 b - - 2 11' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4 Be7 11. Qd3

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4 Be7 11. Qd3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/3P4/2NB1N2/PP3PPP/R1BQR1K1 w - - 5 11' => {
			eco => 'D42l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp2bppp/2n1pn2/8/3P4/P1NB1N2/1P3PPP/R1BQR1K1 b - - 0 11',
				'c1g5' => 'r1bq1rk1/pp2bppp/2n1pn2/6B1/3P4/2NB1N2/PP3PPP/R2QR1K1 b - - 6 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/3P4/P1NB1N2/1P3PPP/R1BQ1RK1 w - - 1 11' => {
			eco => 'D42h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. a3 Nf6

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. a3 Nf6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/8/3P4/P1NB1N2/1P3PPP/R1BQR1K1 b - - 0 11' => {
			eco => 'D42m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Nf6 11. a3

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Nf6 11. a3'),
			moves => {
				'b7b6' => 'r1bq1rk1/p3bppp/1pn1pn2/8/3P4/P1NB1N2/1P3PPP/R1BQR1K1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2P5/3p4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D34h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5 d4

			variation => N__('QGD Tarrasch: Tarrasch\'s Gambit'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2Pp4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D34h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5

			variation => N__('QGD Tarrasch: 9.dxc5'),
			moves => {
				'd5d4' => 'r1bq1rk1/pp2bppp/2n2n2/2P5/3p4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 10',
				'e7c5' => 'r1bq1rk1/pp3ppp/2n2n2/2bp4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2pp2B1/3P4/2N2NP1/PP2PPBP/R2Q1RK1 b - - 6 9' => {
			eco => 'D34o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bg5

			variation => N__('QGD Tarrasch: 9. Bg5'),
			moves => {
				'c5c4' => 'r1bq1rk1/pp2bppp/2n2n2/3p2B1/2pP4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 10',
				'c5d4' => 'r1bq1rk1/pp2bppp/2n2n2/3p2B1/3p4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 10',
				'c8e6' => 'r2q1rk1/pp2bppp/2n1bn2/2pp2B1/3P4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 7 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P1B2/2N2NP1/PP2PPBP/R2Q1RK1 b - - 6 9' => {
			eco => 'D34f',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bf4

			variation => N__('QGD Tarrasch: 9. Bf4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D34d',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. b3

			variation => N__('QGD Tarrasch: 9. b3'),
			moves => {
				'f6e4' => 'r1bq1rk1/pp2bppp/2n5/2pp4/3Pn3/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P4/2N1BNP1/PP2PPBP/R2Q1RK1 b - - 6 9' => {
			eco => 'D34g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Be3

			variation => N__('QGD Tarrasch: 9. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 9' => {
			eco => 'D34c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O

			variation => N__('QGD Tarrasch: 8.O-O O-O (Main Line)'),
			moves => {
				'b2b3' => 'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 9',
				'c1e3' => 'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P4/2N1BNP1/PP2PPBP/R2Q1RK1 b - - 6 9',
				'c1f4' => 'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P1B2/2N2NP1/PP2PPBP/R2Q1RK1 b - - 6 9',
				'c1g5' => 'r1bq1rk1/pp2bppp/2n2n2/2pp2B1/3P4/2N2NP1/PP2PPBP/R2Q1RK1 b - - 6 9',
				'd4c5' => 'r1bq1rk1/pp2bppp/2n2n2/2Pp4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/3p2B1/2pP4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 10' => {
			eco => 'D34p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bg5 c4

			variation => N__('QGD Tarrasch: 9. Bg5 c4'),
			moves => {
				'f3e5' => 'r1bq1rk1/pp2bppp/2n2n2/3pN1B1/2pP4/2N3P1/PP2PPBP/R2Q1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/3p2B1/3N4/2N3P1/PP2PPBP/R2Q1RK1 b - - 0 10' => {
			eco => 'D34r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bg5 cxd4 10. Nxd4

			variation => N__('QGD Tarrasch: 9. Bg5 cxd4'),
			moves => {
				'h7h6' => 'r1bq1rk1/pp2bpp1/2n2n1p/3p2B1/3N4/2N3P1/PP2PPBP/R2Q1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/3p2B1/3p4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 10' => {
			eco => 'D34r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. Bg5 cxd4

			variation => N__('QGD Tarrasch: 9. Bg5 cxd4'),
			moves => {
				'f3d4' => 'r1bq1rk1/pp2bppp/2n2n2/3p2B1/3N4/2N3P1/PP2PPBP/R2Q1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/3p4/3N4/2N1B1P1/PP2PPBP/R2Q1RK1 b - - 4 10' => {
			eco => 'D33q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 cxd4 8. Nxd4 Be7 9. O-O O-O 10. Be3

			variation => N__('QGD Tarrasch: 7. Bg2 cxd4 8.Nxd4 Be7 9.O-O O-O 10. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/3p4/3N4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 3 10' => {
			eco => 'D33p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 cxd4 8. Nxd4 Be7 9. O-O O-O

			variation => N__('QGD Tarrasch: 7. Bg2 cxd4 8.Nxd4 Be7'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2bppp/2n2n2/3p4/3N4/2N1B1P1/PP2PPBP/R2Q1RK1 b - - 4 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n2n2/3pN1B1/2pP4/2N3P1/PP2PPBP/R2Q1RK1 b - - 1 10' => {
			eco => 'D34p',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: 9. Bg5 c4'),
			moves => {
				'c8e6' => 'r2q1rk1/pp2bppp/2n1bn2/3pN1B1/2pP4/2N3P1/PP2PPBP/R2Q1RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n5/2Pp4/8/5NP1/PP2PPBP/R1BQ1RK1 b - - 0 10' => {
			eco => 'D41h',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9.Nxd5'),
			moves => {
				'e7c5' => 'r1bq1rk1/pp3ppp/2n5/2bp4/8/5NP1/PP2PPBP/R1BQ1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/2n5/2pp4/3P4/5NP1/PP2PPBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D41h',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9.Nxd5'),
			moves => {
				'd4c5' => 'r1bq1rk1/pp2bppp/2n5/2Pp4/8/5NP1/PP2PPBP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/2n5/2pp4/3Pn3/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 10' => {
			eco => 'D34e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. b3 Ne4

			variation => N__('QGD Tarrasch: 9. b3 Ne4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/2p3B1/4P3/2N2N2/PPP1BPPP/R2Q1RK1 b - - 3 9' => {
			eco => 'C41m',
                        # TRANSLATORS:

			variation => N__('Philidor: Antoshin Variation'),
			moves => {
				'c8e6' => 'r2q1rk1/pp2bppp/2npbn2/2p3B1/4P3/2N2N2/PPP1BPPP/R2Q1RK1 w - - 4 10',
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/2p5/4P3/2N2N2/PPP1BPPP/R1BQ1RK1 w - - 2 9' => {
			eco => 'C41m',
                        # TRANSLATORS:

			variation => N__('Philidor: Antoshin Variation'),
			moves => {
				'c1g5' => 'r1bq1rk1/pp2bppp/2np1n2/2p3B1/4P3/2N2N2/PPP1BPPP/R2Q1RK1 b - - 3 9',
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p1B1/3NPP2/2N5/PPPQ2PP/2KR1B1R w - - 0 10' => {
			eco => 'B64b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 e5

			variation => N__('Sicilian: Richter-Rauzer, Geller Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p1B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 b - - 5 9' => {
			eco => 'B59h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. Bg5

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O 9. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p3/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w - - 0 10' => {
			eco => 'B83',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 Nc6 7. O-O Be7 8. Be3 O-O 9. f4 e5

			variation => N__('Sicilian: Scheveningen, Modern, 9. f4 e5'),
			moves => {
				'd4b3' => 'r1bq1rk1/pp2bppp/2np1n2/4p3/4PP2/1NN1B3/PPP1B1PP/R2Q1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p3/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 b - - 5 9' => {
			eco => 'B59k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. Be3

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O 9. Be3'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p2bppp/2np1n2/p3p3/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - a6 0 10',
				'c8e6' => 'r2q1rk1/pp2bppp/2npbn2/4p3/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - - 6 10',
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1R1K b - - 5 9' => {
			eco => 'B59i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. Kh1

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O 9. Kh1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1RK1 w - - 4 9' => {
			eco => 'B59g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2bppp/2np1n2/4p3/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 b - - 5 9',
				'c1g5' => 'r1bq1rk1/pp2bppp/2np1n2/4p1B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 b - - 5 9',
				'f2f4' => 'r1bq1rk1/pp2bppp/2np1n2/4p3/4PP2/1NN5/PPP1B1PP/R1BQ1RK1 b - f3 0 9',
				'g1h1' => 'r1bq1rk1/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1R1K b - - 5 9',
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p3/4PP2/1NN1B3/PPP1B1PP/R2Q1RK1 b - - 1 10' => {
			eco => 'B83',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 Nc6 7. O-O Be7 8. Be3 O-O 9. f4 e5 10. Nb3

			variation => N__('Sicilian: Scheveningen, Modern, 9. f4 e5 10. Nb3'),
			moves => {
				'e5f4' => 'r1bq1rk1/pp2bppp/2np1n2/8/4Pp2/1NN1B3/PPP1B1PP/R2Q1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/4p3/4PP2/1NN5/PPP1B1PP/R1BQ1RK1 b - f3 0 9' => {
			eco => 'B59j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O O-O 9. f4

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O O-O 9. f4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/8/4PB2/1NN5/PPP1B1PP/R2Q1RK1 b - - 0 11' => {
			eco => 'B83',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 Nc6 7. O-O Be7 8. Be3 O-O 9. f4 e5 10. Nb3 exf4 11. Bxf4

			variation => N__('Sicilian: Scheveningen, Modern, 9. f4 e5 10. Nb3 exf4 11.Bxf4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2np1n2/8/4Pp2/1NN1B3/PPP1B1PP/R2Q1RK1 w - - 0 11' => {
			eco => 'B83',
                        # TRANSLATORS:

			variation => N__('Sicilian: Scheveningen, Modern, 9. f4 e5 10. Nb3'),
			moves => {
				'e3f4' => 'r1bq1rk1/pp2bppp/2np1n2/8/4PB2/1NN5/PPP1B1PP/R2Q1RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/6B1/3NP3/2N2P2/PPPQ2PP/2KR1B1R b - - 0 9' => {
			eco => 'B63p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f3

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. f3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R w - - 4 9' => {
			eco => 'B63o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7 8.O-O-O O-O'),
			moves => {
				'd4b3' => 'r1bq1rk1/pp2bppp/2nppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R b - - 5 9',
				'f2f3' => 'r1bq1rk1/pp2bppp/2nppn2/6B1/3NP3/2N2P2/PPPQ2PP/2KR1B1R b - - 0 9',
				'f2f4' => 'r1bq1rk1/pp2bppp/2nppn2/6B1/3NPP2/2N5/PPPQ2PP/2KR1B1R b - f3 0 9',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/6B1/3NPP2/2N5/PPPQ2PP/2KR1B1R b - f3 0 9' => {
			eco => 'B64a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. f4'),
			moves => {
				'c6d4' => 'r1bq1rk1/pp2bppp/3ppn2/6B1/3nPP2/2N5/PPPQ2PP/2KR1B1R w - - 0 10',
				'e6e5' => 'r1bq1rk1/pp2bppp/2np1n2/4p1B1/3NPP2/2N5/PPPQ2PP/2KR1B1R w - - 0 10',
				'h7h6' => 'r1bq1rk1/pp2bpp1/2nppn1p/6B1/3NPP2/2N5/PPPQ2PP/2KR1B1R w - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R b - - 5 9' => {
			eco => 'B63q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. Nb3

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7, 9. Nb3'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p2bppp/2nppn2/p5B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w - a6 0 10',
				'a7a6' => 'r1bq1rk1/1p2bppp/p1nppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w - - 0 10',
				'd8b6' => 'r1b2rk1/pp2bppp/1qnppn2/6B1/4P3/1NN5/PPPQ1PPP/2KR1B1R w - - 6 10',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/2KR3R b - - 7 9' => {
			eco => 'B89',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Be3 Be7 8. Qe2 O-O 9. O-O-O

			variation => N__('Sicilian: Velimirovic, 9.O-O-O'),
			moves => {
				'd8c7' => 'r1b2rk1/ppq1bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/2KR3R w - - 8 10',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/R3K2R w KQ - 6 9' => {
			eco => 'B89',
                        # TRANSLATORS:

			variation => N__('Sicilian: Velimirovic'),
			moves => {
				'e1c1' => 'r1bq1rk1/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/2KR3R b - - 7 9',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/8/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ - 6 9' => {
			eco => 'B88',
                        # TRANSLATORS:

			variation => N__('Sicilian: Sozin-Scheveningen, 7. Bb3'),
			moves => {
				'f2f4' => 'r1bq1rk1/pp2bppp/2nppn2/8/3NPP2/1BN1B3/PPP3PP/R2QK2R b KQ f3 0 9',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w - - 6 9' => {
			eco => 'B83',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 Nc6 7. O-O Be7 8. Be3 O-O

			variation => N__('Sicilian: Scheveningen, Modern'),
			moves => {
				'f2f4' => 'r1bq1rk1/pp2bppp/2nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 b - f3 0 9',
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/8/3NPP2/1BN1B3/PPP3PP/R2QK2R b KQ f3 0 9' => {
			eco => 'B88',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Bb3 Be7 8. Be3 O-O 9. f4

			variation => N__('Sicilian: Sozin, Fischer Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/2nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 b - f3 0 9' => {
			eco => 'B83',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 Nc6 7. O-O Be7 8. Be3 O-O 9. f4

			variation => N__('Sicilian: Scheveningen, Modern, 9. f4'),
			moves => {
				'c8d7' => 'r2q1rk1/pp1bbppp/2nppn2/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w - - 1 10',
				'e6e5' => 'r1bq1rk1/pp2bppp/2np1n2/4p3/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/3ppn2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R w - - 1 10' => {
			eco => 'B63n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O Nxd4 9. Qxd4 O-O

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7 8.O-O-O Nxd4 9.Qxd4 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp2bppp/3ppn2/6B1/3QPP2/2N5/PPP3PP/2KR1B1R b - - 0 10' => {
			eco => 'B65a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 Nxd4 10. Qxd4

			variation => N__('Sicilian: Richter-Rauzer, 7... Be7, 9. f4 Nxd4'),
			moves => {
				'd8a5' => 'r1b2rk1/pp2bppp/3ppn2/q5B1/3QPP2/2N5/PPP3PP/2KR1B1R w - - 1 11',
				'h7h6' => 'r1bq1rk1/pp2bpp1/3ppn1p/6B1/3QPP2/2N5/PPP3PP/2KR1B1R w - - 0 11',
			},
		},
		'r1bq1rk1/pp2bppp/3ppn2/6B1/3nPP2/2N5/PPPQ2PP/2KR1B1R w - - 0 10' => {
			eco => 'B65a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O O-O 9. f4 Nxd4

			variation => N__('Sicilian: Richter-Rauzer, 7... Be7, 9. f4 Nxd4'),
			moves => {
				'd2d4' => 'r1bq1rk1/pp2bppp/3ppn2/6B1/3QPP2/2N5/PPP3PP/2KR1B1R b - - 0 10',
			},
		},
		'r1bq1rk1/pp2bppp/4p3/3n4/1n1P4/2NB1N2/PP3PPP/R1BQR1K1 w - - 5 11' => {
			eco => 'D42k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Ncb4

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Ncb4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2n1bp/2pp1np1/3PppN1/1PP1P3/2N2B2/P4PPP/R1BQR1K1 w - - 0 13' => {
			eco => 'E97s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 f5 11. Ng5 Nf6 12. Bf3 c6

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 f5 11. Ng5 Nf6 12. Bf3 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/2npp1p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A36r',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7 7. O-O O-O 8. d3 d6

			variation => N__('English: Symmetrical, Botvinnik, 5... e6, 8. d3 d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 w - - 4 9' => {
			eco => 'B25p',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O

			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 b - - 5 9',
			},
		},
		'r1bq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 b - - 5 9' => {
			eco => 'B25q',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3'),
			moves => {
				'a8b8' => '1rbq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 6 10',
				'b7b6' => 'r1bq1rk1/p3npbp/1pnpp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 0 10',
				'c6d4' => 'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 6 10',
			},
		},
		'r1bq1rk1/pp2npbp/3p1np1/2pPp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - c6 0 10' => {
			eco => 'E97w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 c5

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 c5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/3p1np1/2pPp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 w - c6 0 10' => {
			eco => 'E98b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 c5

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 c5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p1P3/3n1P2/2NPBNP1/PPP3BP/R2Q1RK1 b - - 0 10' => {
			eco => 'B25w',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. e5

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. e5'),
			moves => {
				'e7f5' => 'r1bq1rk1/pp3pbp/3pp1p1/2p1Pn2/3n1P2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPP3BP/1R1Q1RK1 b - - 7 10' => {
			eco => 'B25u',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. Rb1

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 6 10' => {
			eco => 'B25t',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4'),
			moves => {
				'a1b1' => 'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPP3BP/1R1Q1RK1 b - - 7 10',
				'd1d2' => 'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPPQ2BP/R4RK1 b - - 7 10',
				'e4e5' => 'r1bq1rk1/pp2npbp/3pp1p1/2p1P3/3n1P2/2NPBNP1/PPP3BP/R2Q1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPPQ2BP/R4RK1 b - - 7 10' => {
			eco => 'B25v',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. Qd2

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. Qd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2nppp/1bn5/1B1p4/8/1NP2N2/PP3PPP/R1BQ1RK1 w - - 3 11' => {
			eco => 'C09i',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. c3 O-O 10. Nb3 Bb6

			variation => N__('French: Tarrasch, Open, Main Line, 9. c3 O-O 10. Nb3 Bb6'),
			moves => {
				'f1e1' => 'r1bq1rk1/pp2nppp/1bn5/1B1p4/8/1NP2N2/PP3PPP/R1BQR1K1 b - - 4 11',
			},
		},
		'r1bq1rk1/pp2nppp/1bn5/1B1p4/8/1NP2N2/PP3PPP/R1BQR1K1 b - - 4 11' => {
			eco => 'C09j',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. c3 O-O 10. Nb3 Bb6 11. Re1

			variation => N__('French: Tarrasch, Open, Main Line, 9. c3 O-O 10. Nb3 Bb6 11. Re1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2nppp/2n1p3/2ppP3/3P2Q1/P1P2N2/2P2PPP/R1B1KB1R w KQ - 5 9' => {
			eco => 'C18q',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Qg4 O-O 8. Nf3 Nbc6

			variation => N__('French: Winawer, 6... Ne7 7. Qg4 O-O 8. Nf3 Nbc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2nppp/2n1p3/2ppP3/3P2Q1/P1PB4/2P2PPP/R1B1K1NR w KQ - 5 9' => {
			eco => 'C18p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Qg4 O-O 8. Bd3 Nbc6

			variation => N__('French: Winawer, 6... Ne7 7. Qg4 O-O 8. Bd3 Nbc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2nppp/2n5/1Bbp4/8/1NP2N2/PP3PPP/R1BQ1RK1 b - - 2 10' => {
			eco => 'C09g',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, Main Line, 9. c3'),
			moves => {
				'c5b6' => 'r1bq1rk1/pp2nppp/1bn5/1B1p4/8/1NP2N2/PP3PPP/R1BQ1RK1 w - - 3 11',
				'c5d6' => 'r1bq1rk1/pp2nppp/2nb4/1B1p4/8/1NP2N2/PP3PPP/R1BQ1RK1 w - - 3 11',
			},
		},
		'r1bq1rk1/pp2nppp/2n5/1Bbp4/8/2P2N2/PP1N1PPP/R1BQ1RK1 w - - 1 10' => {
			eco => 'C09g',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, Main Line, 9. c3'),
			moves => {
				'd2b3' => 'r1bq1rk1/pp2nppp/2n5/1Bbp4/8/1NP2N2/PP3PPP/R1BQ1RK1 b - - 2 10',
			},
		},
		'r1bq1rk1/pp2nppp/2nb4/1B1p2B1/8/1N3N2/PPP2PPP/R2QR1K1 b - - 5 11' => {
			eco => 'C09p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3 Bd6 10. Re1 O-O 11. Bg5

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bd6 10. Re1 O-O 11. Bg5'),
			moves => {
				'c8g4' => 'r2q1rk1/pp2nppp/2nb4/1B1p2B1/6b1/1N3N2/PPP2PPP/R2QR1K1 w - - 6 12',
			},
		},
		'r1bq1rk1/pp2nppp/2nb4/1B1p4/8/1N3N2/PPP2PPP/R1BQR1K1 w - - 4 11' => {
			eco => 'C09o',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bd6 10. Re1'),
			moves => {
				'c1g5' => 'r1bq1rk1/pp2nppp/2nb4/1B1p2B1/8/1N3N2/PPP2PPP/R2QR1K1 b - - 5 11',
			},
		},
		'r1bq1rk1/pp2nppp/2nb4/1B1p4/8/1NP2N2/PP3PPP/R1BQ1RK1 w - - 3 11' => {
			eco => 'C09h',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. c3 O-O 10. Nb3 Bd6

			variation => N__('French: Tarrasch, Open, Main Line, 9. c3 O-O 10. Nb3 Bd6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w - - 1 8' => {
			eco => 'A08',
                        # TRANSLATORS:

			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'f1e1' => 'r1bq1rk1/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQR1K1 b - - 2 8',
			},
		},
		'r1bq1rk1/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQR1K1 b - - 2 8' => {
			eco => 'A08',
                        # TRANSLATORS: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2 Nge7 7. e4 O-O 8. Re1

			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp2nppp/2p2n2/3pp1B1/1b2P2N/1BNP4/PPP2PPP/R2Q1RK1 b - - 1 10' => {
			eco => 'C49k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Pillsbury Variation'),
			moves => {
				'd8d6' => 'r1b2rk1/pp2nppp/2pq1n2/3pp1B1/1b2P2N/1BNP4/PPP2PPP/R2Q1RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/pp2nppp/2p2n2/3pp1B1/1bB1P2N/2NP4/PPP2PPP/R2Q1RK1 w - - 0 10' => {
			eco => 'C49k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Pillsbury Variation'),
			moves => {
				'c4b3' => 'r1bq1rk1/pp2nppp/2p2n2/3pp1B1/1b2P2N/1BNP4/PPP2PPP/R2Q1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/pp2nppp/2pp1n2/1B2p1B1/1b2P2N/2NP4/PPP2PPP/R2Q1RK1 w - - 0 9' => {
			eco => 'C49k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Pillsbury Variation'),
			moves => {
				'b5c4' => 'r1bq1rk1/pp2nppp/2pp1n2/4p1B1/1bB1P2N/2NP4/PPP2PPP/R2Q1RK1 b - - 1 9',
			},
		},
		'r1bq1rk1/pp2nppp/2pp1n2/4p1B1/1bB1P2N/2NP4/PPP2PPP/R2Q1RK1 b - - 1 9' => {
			eco => 'C49k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Pillsbury Variation'),
			moves => {
				'd6d5' => 'r1bq1rk1/pp2nppp/2p2n2/3pp1B1/1bB1P2N/2NP4/PPP2PPP/R2Q1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp2nppp/4pb2/3n4/3PB3/2N2N2/PP3PPP/R1BQR1K1 w - - 7 12' => {
			eco => 'D42q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Bf6 11. Be4 Nce7

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Bf6 11. Be4 Nce7'),
			moves => {
				'f3e5' => 'r1bq1rk1/pp2nppp/4pb2/3nN3/3PB3/2N5/PP3PPP/R1BQR1K1 b - - 8 12',
			},
		},
		'r1bq1rk1/pp2nppp/4pb2/3nN3/3PB3/2N5/PP3PPP/R1BQR1K1 b - - 8 12' => {
			eco => 'D42r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Bf6 11. Be4 Nce7 12. Ne5

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Bf6 11. Be4 Nce7 12. Ne5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2p1bp/n1pp1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A88',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. b3 Na6

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. b3 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2pp1p/6p1/2pPb3/4P3/2P5/P2QBPPP/1RB1K2R b K - 1 12' => {
			eco => 'D85p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Nf3 c5 8. Rb1 O-O 9. Be2 Nc6 10. d5 Ne5 11. Nxe5 Bxe5 12. Qd2

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6, Main Line'),
			moves => {
				'e7e6' => 'r1bq1rk1/pp3p1p/4p1p1/2pPb3/4P3/2P5/P2QBPPP/1RB1K2R w K - 0 13',
			},
		},
		'r1bq1rk1/pp2pp1p/6p1/2pPb3/4P3/2P5/P3BPPP/1RBQK2R w K - 0 12' => {
			eco => 'D85o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6'),
			moves => {
				'd1d2' => 'r1bq1rk1/pp2pp1p/6p1/2pPb3/4P3/2P5/P2QBPPP/1RB1K2R b K - 1 12',
			},
		},
		'r1bq1rk1/pp2ppbp/1np3p1/n2P2B1/4P3/2N2NP1/PP3PBP/R2Q1RK1 b - - 1 11' => {
			eco => 'D76l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. d5 Na5 10. e4 c6 11. Bg5

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. d5 Na5 10. e4 c6 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/1np3p1/n2P4/4P3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 11' => {
			eco => 'D76k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. d5 Na5 10. e4 c6

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. d5 Na5 10. e4 c6'),
			moves => {
				'c1g5' => 'r1bq1rk1/pp2ppbp/1np3p1/n2P2B1/4P3/2N2NP1/PP3PBP/R2Q1RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2N3p1/3n4/8/2N1BP2/PPPQ2PP/2KR1B1R b - - 0 11' => {
			eco => 'B76q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5 10. exd5 Nxd5 11. Nxc6

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6'),
			moves => {
				'b7c6' => 'r1bq1rk1/p3ppbp/2p3p1/3n4/8/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 12',
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/2p5/2P2B2/2N2NP1/PP2PPBP/R2Q1RK1 b - - 1 9' => {
			eco => 'E65h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. dxc5 dxc5 9. Bf4

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, Exchange, 9. Bf4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/2p5/2P5/2N1BNP1/PP2PPBP/R2Q1RK1 b - - 1 9' => {
			eco => 'E65g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. dxc5 dxc5 9. Be3

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, Exchange, 9. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/2p5/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E65f',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, Exchange'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2ppbp/2n2np1/2p5/2P5/2N1BNP1/PP2PPBP/R2Q1RK1 b - - 1 9',
				'c1f4' => 'r1bq1rk1/pp2ppbp/2n2np1/2p5/2P2B2/2N2NP1/PP2PPBP/R2Q1RK1 b - - 1 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/2pp4/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - d6 0 8' => {
			eco => 'A38m',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d5

			variation => N__('English: Symmetrical, Main Line, 7. d3 d5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3P4/3N4/2N1B3/PPP1BPPP/R2Q1RK1 b - - 0 9' => {
			eco => 'B34w',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O 8. O-O d5 9. exd5

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O 8.O-O d5 9.exd5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3P4/3N4/2N1BP2/PPPQ2PP/2KR1B1R b - - 0 10' => {
			eco => 'B76q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5 10. exd5

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.exd5'),
			moves => {
				'f6d5' => 'r1bq1rk1/pp2ppbp/2n3p1/3n4/3N4/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w - d6 0 9' => {
			eco => 'B34v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O 8. O-O d5

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O 8.O-O d5'),
			moves => {
				'e4d5' => 'r1bq1rk1/pp2ppbp/2n2np1/3P4/3N4/2N1B3/PPP1BPPP/R2Q1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1BP2/PPP3PP/2KRQB1R b - - 1 10' => {
			eco => 'B76p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5 10. Qe1

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10. Qe1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 10' => {
			eco => 'B76o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O d5

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5'),
			moves => {
				'd2e1' => 'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1BP2/PPP3PP/2KRQB1R b - - 1 10',
				'e4d5' => 'r1bq1rk1/pp2ppbp/2n2np1/3P4/3N4/2N1BP2/PPPQ2PP/2KR1B1R b - - 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3p4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 2 9' => {
			eco => 'D79f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Nc3 Nc6

			variation => N__('Neo-Grünfeld, 6.O-O c6 7.cxd5 cxd5 8. Nc3 Nc6'),
			moves => {
				'f3e5' => 'r1bq1rk1/pp2ppbp/2n2np1/3pN3/3P4/2N3P1/PP2PPBP/R1BQ1RK1 b - - 3 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3pN3/3P4/2N3P1/PP2PPBP/R1BQ1RK1 b - - 3 9' => {
			eco => 'D79g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Nc3 Nc6 9. Ne5

			variation => N__('Neo-Grünfeld, 6.O-O c6 7.cxd5 cxd5 8. Nc3 Nc6 9. Ne5'),
			moves => {
				'e7e6' => 'r1bq1rk1/pp3pbp/2n1pnp1/3pN3/3P4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 b - - 3 10' => {
			eco => 'D87g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3

			variation => N__('Grünfeld: Classical Exchange, 9.O-O Nc6 10. Be3'),
			moves => {
				'c5d4' => 'r1bq1rk1/pp2ppbp/2n3p1/8/2BpP3/2P1B3/P3NPPP/R2Q1RK1 w - - 0 11',
				'c6a5' => 'r1bq1rk1/pp2ppbp/6p1/n1p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11',
				'c8g4' => 'r2q1rk1/pp2ppbp/2n3p1/2p5/2BPP1b1/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11',
				'd8a5' => 'r1b2rk1/pp2ppbp/2n3p1/q1p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11',
				'd8c7' => 'r1b2rk1/ppq1ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/R2QK2R w KQ - 2 10' => {
			eco => 'D87d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. Be3 Nc6

			variation => N__('Grünfeld: Classical Exchange, 8... c5 9. Be3 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/2p5/2BPP3/2P5/P3NPPP/R1BQ1RK1 w - - 2 10' => {
			eco => 'D87f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6

			variation => N__('Grünfeld: Classical Exchange, 9.O-O Nc6'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2ppbp/2n3p1/2p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 b - - 3 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/2p5/3PP3/2P2N2/P3BPPP/1RBQK2R w K - 4 10' => {
			eco => 'D85o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Nf3 c5 8. Rb1 O-O 9. Be2 Nc6

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp2ppbp/2n3p1/2pP4/4P3/2P2N2/P3BPPP/1RBQK2R b K - 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/2p5/3PP3/2P2N2/P3BPPP/R1BQ1RK1 w - - 4 10' => {
			eco => 'D85l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Nf3 c5 8. Be2 O-O 9. O-O Nc6

			variation => N__('Grünfeld: Modern Exchange, 8. Be2 O-O 9.O-O Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/2pP4/4P3/2P2N2/P3BPPP/1RBQK2R b K - 0 10' => {
			eco => 'D85o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6'),
			moves => {
				'c6e5' => 'r1bq1rk1/pp2ppbp/6p1/2pPn3/4P3/2P2N2/P3BPPP/1RBQK2R w K - 1 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/3n4/3N4/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 11' => {
			eco => 'B76q',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.exd5'),
			moves => {
				'd4c6' => 'r1bq1rk1/pp2ppbp/2N3p1/3n4/8/2N1BP2/PPPQ2PP/2KR1B1R b - - 0 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/8/2BPP3/4B3/P3NPPP/R2Q1RK1 b - - 0 11' => {
			eco => 'D88',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 cxd4 11. cxd4

			variation => N__('Grünfeld: Classical Exchange, Main Line, 11.cxd4'),
			moves => {
				'c6a5' => 'r1bq1rk1/pp2ppbp/6p1/n7/2BPP3/4B3/P3NPPP/R2Q1RK1 w - - 1 12',
				'c8g4' => 'r2q1rk1/pp2ppbp/2n3p1/8/2BPP1b1/4B3/P3NPPP/R2Q1RK1 w - - 1 12',
			},
		},
		'r1bq1rk1/pp2ppbp/2n3p1/8/2BpP3/2P1B3/P3NPPP/R2Q1RK1 w - - 0 11' => {
			eco => 'D88',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 cxd4

			variation => N__('Grünfeld: Classical Exchange, Main Line'),
			moves => {
				'c3d4' => 'r1bq1rk1/pp2ppbp/2n3p1/8/2BPP3/4B3/P3NPPP/R2Q1RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2P5/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E65f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. dxc5

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, Exchange'),
			moves => {
				'd6c5' => 'r1bq1rk1/pp2ppbp/2n2np1/2p5/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 w - - 4 8' => {
			eco => 'A36v',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O

			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 7.O-O O-O'),
			moves => {
				'd2d3' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A36w',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3

			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2ppbp/p1np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 0 9',
				'a8b8' => '1rbq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 1 9',
				'f6e8' => 'r1bqnrk1/pp2ppbp/2np2p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP1BPPBP/R2Q1RK1 b - - 1 8' => {
			eco => 'A38p',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. Bd2

			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. Bd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 b - - 1 8' => {
			eco => 'A38o',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. Rb1

			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'A38n',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6

			variation => N__('English: Symmetrical, Main Line, 7. d3 d6'),
			moves => {
				'a1b1' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 b - - 1 8',
				'a2a3' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b - - 0 8',
				'c1d2' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP1BPPBP/R2Q1RK1 b - - 1 8',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/P1N2NP1/1P1PPPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'A38j',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. a3 d6

			variation => N__('English: Symmetrical, Main Line, 7. a3 d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A38q',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. a3

			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. a3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2ppbp/p1np1np1/2p5/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b - - 3 8' => {
			eco => 'E60i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nf3 Bg7 4. b3 d6 5. Bb2 O-O 6. g3 c5 7. Bg2 Nc6 8. O-O

			variation => N__('King\'s Indian: b3+g3 System'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PP4/1P3NP1/PB2PPBP/RN1QK2R w KQ - 2 8' => {
			eco => 'E60i',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: b3+g3 System'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b - - 3 8',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E65d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, 7.O-O Nc6'),
			moves => {
				'd4c5' => 'r1bq1rk1/pp2ppbp/2np1np1/2P5/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
				'd4d5' => 'r1bq1rk1/pp2ppbp/2np1np1/2pP4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
				'h2h3' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 b - - 0 8' => {
			eco => 'E65e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. h3

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, 7.O-O Nc6 8. h3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 2 8' => {
			eco => 'E81o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 c5 7. Nge2 Nc6

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 c5 7.Nge2 Nc6'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp2ppbp/2np1np1/2pP4/2P1P3/2N1BP2/PP2N1PP/R2QKB1R b KQ - 0 8',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E91k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 c5 7. O-O Nc6

			variation => N__('King\'s Indian: 6. Be2 c5 7.O-O Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2pP4/2P1P3/2N1BP2/PP2N1PP/R2QKB1R b KQ - 0 8' => {
			eco => 'E81p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 c5 7. Nge2 Nc6 8. d5

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 c5 7.Nge2 Nc6 8. d5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2pP4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E66a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno'),
			moves => {
				'c6a5' => 'r1bq1rk1/pp2ppbp/3p1np1/n1pP4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/6B1/3NP3/2N5/PPP1BPPP/R2Q1RK1 w - - 6 9' => {
			eco => 'B70s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O O-O 8. Bg5 Nc6

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O O-O 8. Bg5'),
			moves => {
				'd4b3' => 'r1bq1rk1/pp2ppbp/2np1np1/6B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 b - - 7 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/6B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 b - - 7 9' => {
			eco => 'B70t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O O-O 8. Bg5 Nc6 9. Nb3

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O O-O 8. Bg5 Nc6 9. Nb3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2ppbp/p1np1np1/6B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 w - - 0 10',
				'c8e6' => 'r2q1rk1/pp2ppbp/2npbnp1/6B1/4P3/1NN5/PPP1BPPP/R2Q1RK1 w - - 8 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2BNP3/2N1BP2/PPPQ2PP/R3K2R b KQ - 4 9' => {
			eco => 'B77a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4

			variation => N__('Sicilian: Dragon, Yugoslav, 9. Bc4'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p2ppbp/2np1np1/p7/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ a6 0 10',
				'c6d4' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2BnP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 0 10',
				'c6e5' => 'r1bq1rk1/pp2ppbp/3p1np1/4n3/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 5 10',
				'c8d7' => 'r2q1rk1/pp1bppbp/2np1np1/8/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 5 10',
				'f6d7' => 'r1bq1rk1/pp1nppbp/2np2p1/8/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 5 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN1BPPP/R1BQ1RK1 b - - 3 9' => {
			eco => 'B37l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9. O-O

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9.O-O'),
			moves => {
				'c8e6' => 'r2q1rk1/pp2ppbp/2npbnp1/8/2P1P3/2N5/PPN1BPPP/R1BQ1RK1 w - - 4 10',
				'f6d7' => 'r1bq1rk1/pp1nppbp/2np2p1/8/2P1P3/2N5/PPN1BPPP/R1BQ1RK1 w - - 4 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN1BPPP/R1BQK2R w KQ - 2 9' => {
			eco => 'B37k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN1BPPP/R1BQ1RK1 b - - 3 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'A39g',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 d6

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 b - - 1 9' => {
			eco => 'B38n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 d6 9. O-O

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 d6 9.O-O'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2ppbp/p1np1np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 0 10',
				'c6d4' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2PnP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 0 10',
				'c8d7' => 'r2q1rk1/pp1bppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 2 10',
				'f6d7' => 'r1bq1rk1/pp1nppbp/2np2p1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 2 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R w KQ - 0 9' => {
			eco => 'B38l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 d6

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 d6'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2Q1RK1 b - - 1 9',
				'f2f3' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2PNP3/2N1BP2/PP2B1PP/R2QK2R b KQ - 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PNP3/2N1BP2/PP2B1PP/R2QK2R b KQ - 0 9' => {
			eco => 'B38m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 d6 9. f3

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 d6 9. f3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PNPP2/2N1B3/PP2B1PP/R2QK2R b KQ - 2 9' => {
			eco => 'E79a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Be2 c5 7. Nf3 cxd4 8. Nxd4 Nc6 9. Be3

			variation => N__('King\'s Indian: Four Pawns Attack, Main Line'),
			moves => {
				'c6d4' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2PnPP2/2N1B3/PP2B1PP/R2QK2R w KQ - 0 10',
				'c8d7' => 'r2q1rk1/pp1bppbp/2np1np1/8/2PNPP2/2N1B3/PP2B1PP/R2QK2R w KQ - 3 10',
				'c8g4' => 'r2q1rk1/pp2ppbp/2np1np1/8/2PNPPb1/2N1B3/PP2B1PP/R2QK2R w KQ - 3 10',
				'f6g4' => 'r1bq1rk1/pp2ppbp/2np2p1/8/2PNPPn1/2N1B3/PP2B1PP/R2QK2R w KQ - 3 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PNPP2/2N5/PP2B1PP/R1BQK2R w KQ - 1 9' => {
			eco => 'E78',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Four Pawns Attack, 7. Nf3 cxd4'),
			moves => {
				'c1e3' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2PNPP2/2N1B3/PP2B1PP/R2QK2R b KQ - 2 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP1P1/2N1BP2/PPPQ3P/R3KB1R b KQ g3 0 9' => {
			eco => 'B76d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. g4

			variation => N__('Sicilian: Dragon, Yugoslav, 9. g4'),
			moves => {
				'c6d4' => 'r1bq1rk1/pp2ppbp/3p1np1/8/3nP1P1/2N1BP2/PPPQ3P/R3KB1R w KQ - 0 10',
				'c8e6' => 'r2q1rk1/pp2ppbp/2npbnp1/8/3NP1P1/2N1BP2/PPPQ3P/R3KB1R w KQ - 1 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1B2P/PPP2PP1/R2QK2R b KQ - 0 9' => {
			eco => 'B35o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6 9. h3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6 9. h3'),
			moves => {
				'c8d7' => 'r2q1rk1/pp1bppbp/2np1np1/8/3NP3/1BN1B2P/PPP2PP1/R2QK2R w KQ - 1 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ - 0 9' => {
			eco => 'B35o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6'),
			moves => {
				'f2f3' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1BP2/PPP3PP/R2QK2R b KQ - 0 9',
				'h2h3' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1B2P/PPP2PP1/R2QK2R b KQ - 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1BP2/PPP3PP/R2QK2R b KQ - 0 9' => {
			eco => 'B35q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6 9. f3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6 9. f3'),
			moves => {
				'c8d7' => 'r2q1rk1/pp1bppbp/2np1np1/8/3NP3/1BN1BP2/PPP3PP/R2QK2R w KQ - 1 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1R1K b - - 7 9' => {
			eco => 'B73',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Kh1

			variation => N__('Sicilian: Dragon, Classical, 9. Kh1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w - - 6 9' => {
			eco => 'B73',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O

			variation => N__('Sicilian: Dragon, Classical, 8.O-O O-O'),
			moves => {
				'd1d2' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/R4RK1 b - - 7 9',
				'd4b3' => 'r1bq1rk1/pp2ppbp/2np1np1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 b - - 7 9',
				'f2f4' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 b - f3 0 9',
				'g1h1' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1R1K b - - 7 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/2KR3R b - - 7 9' => {
			eco => 'B72',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. Qd2 O-O 9. O-O-O

			variation => N__('Sicilian: Dragon, Classical, Grigoriev Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/R3K2R w KQ - 6 9' => {
			eco => 'B72',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Classical, Amsterdam Variation'),
			moves => {
				'e1c1' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/2KR3R b - - 7 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/R4RK1 b - - 7 9' => {
			eco => 'B73',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Qd2

			variation => N__('Sicilian: Dragon, Classical, Richter Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/2KR1B1R b - - 4 9' => {
			eco => 'B76g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O'),
			moves => {
				'c6d4' => 'r1bq1rk1/pp2ppbp/3p1np1/8/3nP3/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 10',
				'c8d7' => 'r2q1rk1/pp1bppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/2KR1B1R w - - 5 10',
				'd6d5' => 'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R w KQ - 3 9' => {
			eco => 'B76c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6

			variation => N__('Sicilian: Dragon, Yugoslav, 8. Qd2 Nc6'),
			moves => {
				'e1c1' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/2KR1B1R b - - 4 9',
				'f1c4' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2BNP3/2N1BP2/PPPQ2PP/R3K2R b KQ - 4 9',
				'g2g4' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP1P1/2N1BP2/PPPQ3P/R3KB1R b KQ g3 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N5/PPP1BPPP/R1BQR1K1 w - - 6 9' => {
			eco => 'B70p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O O-O 8. Re1 Nc6

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O O-O 8. Re1 Nc6'),
			moves => {
				'd4b3' => 'r1bq1rk1/pp2ppbp/2np1np1/8/4P3/1NN5/PPP1BPPP/R1BQR1K1 b - - 7 9',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 b - f3 0 9' => {
			eco => 'B73',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. f4

			variation => N__('Sicilian: Dragon, Classical, 9. f4'),
			moves => {
				'd8b6' => 'r1b2rk1/pp2ppbp/1qnp1np1/8/3NPP2/2N1B3/PPP1B1PP/R2Q1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 b - - 7 9' => {
			eco => 'B74',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O O-O 9. Nb3

			variation => N__('Sicilian: Dragon, Classical, 9. Nb3'),
			moves => {
				'a7a5' => 'r1bq1rk1/1p2ppbp/2np1np1/p7/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - a6 0 10',
				'a7a6' => 'r1bq1rk1/1p2ppbp/p1np1np1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - - 0 10',
				'b7b6' => 'r1bq1rk1/p3ppbp/1pnp1np1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - - 0 10',
				'c8e6' => 'r2q1rk1/pp2ppbp/2npbnp1/8/4P3/1NN1B3/PPP1BPPP/R2Q1RK1 w - - 8 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/4P3/1NN5/PPP1BPPP/R1BQ1RK1 w - - 6 9' => {
			eco => 'B70n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O Nc6 8. Nb3 O-O

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O Nc6 8. Nb3 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/4P3/1NN5/PPP1BPPP/R1BQR1K1 b - - 7 9' => {
			eco => 'B70q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O O-O 8. Re1 Nc6 9. Nb3

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O O-O 8. Re1 Nc6 9. Nb3'),
			moves => {
				'c8e6' => 'r2q1rk1/pp2ppbp/2npbnp1/8/4P3/1NN5/PPP1BPPP/R1BQR1K1 w - - 8 10',
			},
		},
		'r1bq1rk1/pp2ppbp/2np2p1/2n5/1PP1P3/2N5/P1NBBPPP/R2Q1RK1 b - b3 0 11' => {
			eco => 'B37q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9. O-O Nd7 10. Bd2 Nc5 11. b4

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9.O-O Nd7 10. Bd2 Nc5 11. b4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np2p1/2n5/2P1P3/2N5/PPNBBPPP/R2Q1RK1 w - - 6 11' => {
			eco => 'B37p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9. O-O Nd7 10. Bd2 Nc5

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 O-O 9.O-O Nd7 10. Bd2 Nc5'),
			moves => {
				'b2b4' => 'r1bq1rk1/pp2ppbp/2np2p1/2n5/1PP1P3/2N5/P1NBBPPP/R2Q1RK1 b - b3 0 11',
			},
		},
		'r1bq1rk1/pp2ppbp/2np2p1/8/2PN2n1/2N1P1P1/PP3PBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'A39i',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Ng4 9. e3 d6

			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Ng4 9. e3 d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np2p1/8/2PNPPn1/2N1B3/PP2B1PP/R2QK2R w KQ - 3 10' => {
			eco => 'E79c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Be2 c5 7. Nf3 cxd4 8. Nxd4 Nc6 9. Be3 Ng4

			variation => N__('King\'s Indian: Four Pawns Attack, Main Line, 9... Ng4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/4n3/2BNP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 5 10' => {
			eco => 'B77c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Ne5

			variation => N__('Sicilian: Dragon, Yugoslav, 9. Bc4 Ne5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/6B1/2P1P3/2N5/PP1QBPPP/R3K2R b KQ - 4 10' => {
			eco => 'B36m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. Bg5 O-O 10. Qd2

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9. Bg5 O-O 10. Qd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/6B1/2PQP3/2N5/PP2BPPP/R3K2R w KQ - 3 10' => {
			eco => 'B36l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. Bg5 O-O

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9. Bg5 O-O'),
			moves => {
				'd4d2' => 'r1bq1rk1/pp2ppbp/3p1np1/6B1/2P1P3/2N5/PP1QBPPP/R3K2R b KQ - 4 10',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2BnP3/2N1BP2/PPPQ2PP/R3K2R w KQ - 0 10' => {
			eco => 'B77e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. Bc4 Nxd4

			variation => N__('Sicilian: Dragon, Yugoslav, 9. Bc4 Nxd4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2P1P3/2N1B3/PP1QBPPP/R3K2R b KQ - 4 10' => {
			eco => 'B36o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. Be3 O-O 10. Qd2

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9. Be3 O-O 10. Qd2'),
			moves => {
				'c8e6' => 'r2q1rk1/pp2ppbp/3pbnp1/8/2P1P3/2N1B3/PP1QBPPP/R3K2R w KQ - 5 11',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2P5/2N3P1/PP1QPPBP/R1B2RK1 b - - 1 10' => {
			eco => 'A39l',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd2

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 b - - 1 10' => {
			eco => 'A39m',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3

			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w - - 0 11',
				'a8b8' => '1rbq1rk1/pp2ppbp/3p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w - - 2 11',
				'c8f5' => 'r2q1rk1/pp2ppbp/3p1np1/5b2/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w - - 2 11',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2PQ4/2N3P1/PP2PPBP/R1B2RK1 w - - 0 10' => {
			eco => 'A39k',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6

			variation => N__('English: Symmetrical, Main Line 7. d4, 9... d6'),
			moves => {
				'd4d2' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2P5/2N3P1/PP1QPPBP/R1B2RK1 b - - 1 10',
				'd4d3' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2PQP3/2N1B3/PP2BPPP/R3K2R w KQ - 3 10' => {
			eco => 'B36n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. Be3 O-O

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9. Be3 O-O'),
			moves => {
				'd4d2' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2P1P3/2N1B3/PP1QBPPP/R3K2R b KQ - 4 10',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2PnP3/2N1B3/PP2BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'B38q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 O-O 8. Be2 d6 9. O-O Nxd4

			variation => N__('Sicilian: Maroczy Bind, 7. Nc3 O-O 8. Be2 d6 9.O-O Nxd4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2PnPP2/2N1B3/PP2B1PP/R2QK2R w KQ - 0 10' => {
			eco => 'E79b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Be2 c5 7. Nf3 cxd4 8. Nxd4 Nc6 9. Be3 Nxd4

			variation => N__('King\'s Indian: Four Pawns Attack, Main Line, 9...Nxd4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/3BP3/2N2P2/PPPQ2PP/2KR1B1R b - - 0 10' => {
			eco => 'B76j',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4'),
			moves => {
				'c8e6' => 'r2q1rk1/pp2ppbp/3pbnp1/8/3BP3/2N2P2/PPPQ2PP/2KR1B1R w - - 1 11',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/3nP1P1/2N1BP2/PPPQ3P/R3KB1R w KQ - 0 10' => {
			eco => 'B76f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. g4 Nxd4

			variation => N__('Sicilian: Dragon, Yugoslav, 9. g4 Nxd4'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/3nP3/2N1BP2/PPPQ2PP/2KR1B1R w - - 0 10' => {
			eco => 'B76j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 O-O 8. Qd2 Nc6 9. O-O-O Nxd4

			variation => N__('Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4'),
			moves => {
				'e3d4' => 'r1bq1rk1/pp2ppbp/3p1np1/8/3BP3/2N2P2/PPPQ2PP/2KR1B1R b - - 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/n1pP4/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E66c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno'),
			moves => {
				'f3d2' => 'r1bq1rk1/pp2ppbp/3p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 2 9',
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 2 9' => {
			eco => 'E66b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno'),
			moves => {
				'a7a6' => 'r1bq1rk1/1p2ppbp/p2p1np1/n1pP4/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 0 10',
				'e7e5' => 'r1bq1rk1/pp3pbp/3p1np1/n1pPp3/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 w - e6 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/6p1/2pPN3/4P3/2P5/P3BPPP/1RBQK2R b K - 0 11' => {
			eco => 'D85o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6'),
			moves => {
				'g7e5' => 'r1bq1rk1/pp2pp1p/6p1/2pPb3/4P3/2P5/P3BPPP/1RBQK2R w K - 0 12',
			},
		},
		'r1bq1rk1/pp2ppbp/6p1/2pPn3/4P3/2P2N2/P3BPPP/1RBQK2R w K - 1 11' => {
			eco => 'D85o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6'),
			moves => {
				'f3e5' => 'r1bq1rk1/pp2ppbp/6p1/2pPN3/4P3/2P5/P3BPPP/1RBQK2R b K - 0 11',
			},
		},
		'r1bq1rk1/pp2ppbp/6p1/n1p5/2BPP3/2P1B3/P3NPPP/R2Q1RK1 w - - 4 11' => {
			eco => 'D87i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 Na5

			variation => N__('Grünfeld: Classical Exchange, 10. Be3 Na5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/6p1/n7/2BPP3/4B3/P3NPPP/R2Q1RK1 w - - 1 12' => {
			eco => 'D88',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 cxd4 11. cxd4 Na5

			variation => N__('Grünfeld: Classical Exchange, Main Line, 11.cxd4 Na5'),
			moves => {
				'c4d3' => 'r1bq1rk1/pp2ppbp/6p1/n7/3PP3/3BB3/P3NPPP/R2Q1RK1 b - - 2 12',
			},
		},
		'r1bq1rk1/pp2ppbp/6p1/n7/3PP3/3BB3/P3NPPP/R2Q1RK1 b - - 2 12' => {
			eco => 'D88',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 c5 9. O-O Nc6 10. Be3 cxd4 11. cxd4 Na5 12. Bd3

			variation => N__('Grünfeld: Classical Exchange, Main Line, 11.cxd4 Na5 12. Bd3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/n1pp1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E91n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Na6 7. O-O c6

			variation => N__('King\'s Indian: Kazakh Variation, 7.O-O c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2p5/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - c6 0 8' => {
			eco => 'E91m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Na6 7. O-O c5

			variation => N__('King\'s Indian: Kazakh Variation, 7.O-O c5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2p5/3PPP2/2NB1N2/PPP3PP/R1BQ1RK1 w - c6 0 8' => {
			eco => 'B09m',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6 7. O-O c5

			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6 7.O-O c5'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4PP2/2NB1N2/PPP3PP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/2P5/2N2NP1/PP2PPBP/R1BQK2R w KQ - 1 8' => {
			eco => 'E64e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. d5 Na6

			variation => N__('King\'s Indian: Fianchetto, Yugoslav, 7. d5 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4P3/2N2N1P/PPP1BPP1/R1BQ1RK1 b - - 0 8' => {
			eco => 'A43z',
                        # TRANSLATORS: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O 7. O-O Na6 8. h3

			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O 7.O-O Na6 8. h3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4P3/2N2N2/PPP1BPPP/R1BQ1RK1 w - - 6 8' => {
			eco => 'A43y',
                        # TRANSLATORS: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O 7. O-O Na6

			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O 7.O-O Na6'),
			moves => {
				'h2h3' => 'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4P3/2N2N1P/PPP1BPP1/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4PP2/2NB1N2/PPP3PP/R1BQ1RK1 b - - 0 8' => {
			eco => 'B09m',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6 7. O-O c5 8. d5

			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6 7.O-O c5 8. d5'),
			moves => {
				'c8g4' => 'r2q1rk1/pp2ppbp/n2p1np1/2pP4/4PPb1/2NB1N2/PPP3PP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pp2ppbp/n4np1/2p5/2QPP3/2N2N2/PP2BPPP/R1B1K2R w KQ c6 0 9' => {
			eco => 'D97o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'd4d5' => 'r1bq1rk1/pp2ppbp/n4np1/2pP4/2Q1P3/2N2N2/PP2BPPP/R1B1K2R b KQ - 0 9',
			},
		},
		'r1bq1rk1/pp2ppbp/n4np1/2pP4/2Q1P3/2N2N2/PP2BPPP/R1B1K2R b KQ - 0 9' => {
			eco => 'D97o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'e7e6' => 'r1bq1rk1/pp3pbp/n3pnp1/2pP4/2Q1P3/2N2N2/PP2BPPP/R1B1K2R w KQ - 0 10',
			},
		},
		'r1bq1rk1/pp2ppbp/n5p1/2Pn4/8/5NP1/PP2PPBP/RNBQ1RK1 w - - 1 9' => {
			eco => 'D75',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O c5 8. dxc5 Na6

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O c5 8.dxc5 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3p1p/4p1p1/2pPb3/4P3/2P5/P2QBPPP/1RB1K2R w K - 0 13' => {
			eco => 'D85p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Nf3 c5 8. Rb1 O-O 9. Be2 Nc6 10. d5 Ne5 11. Nxe5 Bxe5 12. Qd2 e6

			variation => N__('Grünfeld: Modern Exchange, 8. Rb1 O-O 9. Be2 Nc6, Main Line, 12... e6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/1nn1p1p1/3pN3/3P1P2/2N1B1P1/PP2P1BP/R2Q1RK1 w - - 3 12' => {
			eco => 'D79n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Ne5 e6 9. Nc3 Nfd7 10. f4 Nc6 11. Be3 Nb6

			variation => N__('Neo-Grünfeld, Main Line, 10. f4 Nc6 11. Be3 Nb6'),
			moves => {
				'e3f2' => 'r1bq1rk1/pp3pbp/1nn1p1p1/3pN3/3P1P2/2N3P1/PP2PBBP/R2Q1RK1 b - - 4 12',
			},
		},
		'r1bq1rk1/pp3pbp/1nn1p1p1/3pN3/3P1P2/2N3P1/PP2PBBP/R2Q1RK1 b - - 4 12' => {
			eco => 'D79n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Ne5 e6 9. Nc3 Nfd7 10. f4 Nc6 11. Be3 Nb6 12. Bf2

			variation => N__('Neo-Grünfeld, Main Line, 10. f4 Nc6 11. Be3 Nb6 12. Bf2'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/2n1pnp1/3pN3/3P4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D79h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6 7. cxd5 cxd5 8. Nc3 Nc6 9. Ne5 e6

			variation => N__('Neo-Grünfeld, 6.O-O c6 7.cxd5 cxd5 8. Nc3 Nc6 9. Ne5 e6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/3p1np1/n1pPp3/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 w - e6 0 10' => {
			eco => 'E66d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 c5 7. O-O Nc6 8. d5 Na5 9. Nd2 e5

			variation => N__('King\'s Indian: Fianchetto, Yugoslav Panno, 9... e5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/3pp1p1/2p1Pn2/3n1P2/2NPBNP1/PPP3BP/R2Q1RK1 w - - 1 11' => {
			eco => 'B25x',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. e5 Nef5

			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. e5 Nef5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n1pp1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR w KQ - 0 9' => {
			eco => 'E73t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6 7. Qd2 e5 8. d5 c6

			variation => N__('King\'s Indian: Averbakh, 6... Na6 7. Qd2 e5 8. d5 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n1pp1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9' => {
			eco => 'E94k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Re1 c6

			variation => N__('King\'s Indian: Glek, 8. Re1 c6'),
			moves => {
				'e2f1' => 'r1bq1rk1/pp3pbp/n1pp1np1/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b - - 1 9',
			},
		},
		'r1bq1rk1/pp3pbp/n1pp1np1/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b - - 1 9' => {
			eco => 'E94k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Re1 c6 9. Bf1

			variation => N__('King\'s Indian: Glek, 8. Re1 c6 9. Bf1'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 6 10' => {
			eco => 'A73',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Na6

			variation => N__('Benoni: Classical, 9.O-O Na6'),
			moves => {
				'f3d2' => 'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 b - - 7 10',
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 b - - 7 10' => {
			eco => 'A73',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Na6 10. Nd2

			variation => N__('Benoni: Classical, 9.O-O Na6 10. Nd2'),
			moves => {
				'a6c7' => 'r1bq1rk1/ppn2pbp/3p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 w - - 8 11',
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQK2R w KQ - 3 10' => {
			eco => 'A70m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 Na6

			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQ1RK1 w - - 6 10' => {
			eco => 'A65i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2 O-O 9. O-O Na6

			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2 O-O 9.O-O Na6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 10' => {
			eco => 'A62',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Na6

			variation => N__('Benoni: Fianchetto, 9.O-O Na6'),
			moves => {
				'f3d2' => 'r1bq1rk1/pp3pbp/n2p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 6 10',
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 6 10' => {
			eco => 'A62',
                        # TRANSLATORS:

			variation => N__('Benoni: Fianchetto, 9.O-O Na6'),
			moves => {
				'a6c7' => 'r1bq1rk1/ppn2pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 7 11',
			},
		},
		'r1bq1rk1/pp3pbp/n2ppnp1/2pP4/2P1P3/2NB1N1P/PP3PP1/R1BQ1RK1 b - - 3 9' => {
			eco => 'E90w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 c5 7. d5 e6 8. Bd3 Na6 9. O-O

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 c5 7. d5 e6 8. Bd3 Na6 9.O-O'),
			moves => {
				'a6c7' => 'r1bq1rk1/ppn2pbp/3ppnp1/2pP4/2P1P3/2NB1N1P/PP3PP1/R1BQ1RK1 w - - 4 10',
			},
		},
		'r1bq1rk1/pp3pbp/n2ppnp1/2pP4/2P1P3/2NB1N1P/PP3PP1/R1BQK2R w KQ - 2 9' => {
			eco => 'E90w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 c5 7. d5 e6 8. Bd3 Na6

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 c5 7. d5 e6 8. Bd3 Na6'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp3pbp/n2ppnp1/2pP4/2P1P3/2NB1N1P/PP3PP1/R1BQ1RK1 b - - 3 9',
			},
		},
		'r1bq1rk1/pp3pbp/n3pnp1/2pP4/2Q1P3/2N2N2/PP2BPPP/R1B1K2R w KQ - 0 10' => {
			eco => 'D97o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp3pbp/n3pnp1/2pP4/2Q1P3/2N2N2/PP2BPPP/R1B2RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/pp3pbp/n3pnp1/2pP4/2Q1P3/2N2N2/PP2BPPP/R1B2RK1 b - - 1 10' => {
			eco => 'D97o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'e6d5' => 'r1bq1rk1/pp3pbp/n4np1/2pp4/2Q1P3/2N2N2/PP2BPPP/R1B2RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp3pbp/n4np1/2pP4/2Q5/2N2N2/PP2BPPP/R1B2RK1 b - - 0 11' => {
			eco => 'D97o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'c8f5' => 'r2q1rk1/pp3pbp/n4np1/2pP1b2/2Q5/2N2N2/PP2BPPP/R1B2RK1 w - - 1 12',
			},
		},
		'r1bq1rk1/pp3pbp/n4np1/2pp4/2Q1P3/2N2N2/PP2BPPP/R1B2RK1 w - - 0 11' => {
			eco => 'D97o',
                        # TRANSLATORS:

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'e4d5' => 'r1bq1rk1/pp3pbp/n4np1/2pP4/2Q5/2N2N2/PP2BPPP/R1B2RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PP1B1PPP/R2QK2R w KQ - 4 10' => {
			eco => 'B14x',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Bd2 Nc6 9. Bd3 O-O

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1p3/8/2BPP3/5N2/P2Q1PPP/R3K2R w KQ - 3 12' => {
			eco => 'D41n',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. e4, Main Line'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp3ppp/2n1p3/8/2BPP3/5N2/P2Q1PPP/R4RK1 b - - 4 12',
			},
		},
		'r1bq1rk1/pp3ppp/2n1p3/8/2BPP3/5N2/P2Q1PPP/R4RK1 b - - 4 12' => {
			eco => 'D41n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e4 Nxc3 7. bxc3 cxd4 8. cxd4 Bb4+ 9. Bd2 Bxd2+ 10. Qxd2 O-O 11. Bc4 Nc6 12. O-O

			variation => N__('QGD: Semi-Tarrasch, 6. e4, Main Line'),
			moves => {
				'b7b6' => 'r1bq1rk1/p4ppp/1pn1p3/8/2BPP3/5N2/P2Q1PPP/R4RK1 w - - 0 13',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pb2/3n4/3P4/2NB1N2/PP3PPP/R1BQR1K1 w - - 5 11' => {
			eco => 'D42p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Bf6

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Bf6'),
			moves => {
				'd3e4' => 'r1bq1rk1/pp3ppp/2n1pb2/3n4/3PB3/2N2N2/PP3PPP/R1BQR1K1 b - - 6 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pb2/3n4/3P4/P1NB1N2/1P3PPP/R1BQ1RK1 w - - 1 11' => {
			eco => 'D42i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. a3 Bf6

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. a3 Bf6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pb2/3n4/3PB3/2N2N2/PP3PPP/R1BQR1K1 b - - 6 11' => {
			eco => 'D42p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7 9. O-O O-O 10. Re1 Bf6 11. Be4

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line, 10. Re1 Bf6 11. Be4'),
			moves => {
				'c6e7' => 'r1bq1rk1/pp2nppp/4pb2/3n4/3PB3/2N2N2/PP3PPP/R1BQR1K1 w - - 7 12',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2bp4/2P2B2/2N1PN2/PPQ2PPP/R3KB1R w KQ - 2 9' => {
			eco => 'D37r',
                        # TRANSLATORS:

			variation => N__('QGD: Classical, 5...O-O 6. e3 c5 7.dxc5 Bxc5 8. Qc2'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp3ppp/2n1pn2/2bp4/2P2B2/P1N1PN2/1PQ2PPP/R3KB1R b KQ - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2bp4/2P2B2/P1N1PN2/1PQ2PPP/R3KB1R b KQ - 0 9' => {
			eco => 'D37s',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 Be7 5. Bf4 O-O 6. e3 c5 7. dxc5 Bxc5 8. Qc2 Nc6 9. a3

			variation => N__('QGD: Classical, 5...O-O 6. e3 c5 7.dxc5 Bxc5 8. Qc2 Nc6 9. a3'),
			moves => {
				'd8a5' => 'r1b2rk1/pp3ppp/2n1pn2/q1bp4/2P2B2/P1N1PN2/1PQ2PPP/R3KB1R w KQ - 1 10',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2bp4/8/2PBPN2/PP1N1PPP/R1BQ1RK1 w - - 0 9' => {
			eco => 'D05n',
                        # TRANSLATORS: 1. d4 d5 2. Nf3 Nf6 3. e3 e6 4. Bd3 c5 5. c3 Nc6 6. Nbd2 Bd6 7. O-O O-O 8. dxc5 Bxc5

			variation => N__('Colle: 5. c3 Nc6 6.Nbd2 Bd6 7.O-O O-O 8.dxc5 Bxc5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2p5/1bBP4/2N1PN2/PP3PPP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O dxc4 8. Bxc4 Nc6

			variation => N__('Nimzo-Indian: Main Line, 7...dxc4 8.Bxc4 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2p5/1bBP4/P1N1PN2/1P3PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 dxc4 9.Bxc4'),
			moves => {
				'b4a5' => 'r1bq1rk1/pp3ppp/2n1pn2/b1p5/2BP4/P1N1PN2/1P3PPP/R1BQ1RK1 w - - 1 10',
				'c5d4' => 'r1bq1rk1/pp3ppp/2n1pn2/8/1bBp4/P1N1PN2/1P3PPP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2p5/1bpP4/P1NBPN2/1P3PPP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 dxc4'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp3ppp/2n1pn2/2p5/1bBP4/P1N1PN2/1P3PPP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2p5/2BP4/P1P1PN2/5PPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4 10. Bxc4

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4 10.Bxc4'),
			moves => {
				'd8c7' => 'r1b2rk1/ppq2ppp/2n1pn2/2p5/2BP4/P1P1PN2/5PPP/R1BQ1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2p5/2pP4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'E59',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3 dxc4

			variation => N__('Nimzo-Indian: Main Line, 9.bxc3 dxc4'),
			moves => {
				'd3c4' => 'r1bq1rk1/pp3ppp/2n1pn2/2p5/2BP4/P1P1PN2/5PPP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2pP4/1b1P4/2NBPN2/PP3PPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. cxd5

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8.cxd5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2pp4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6'),
			moves => {
				'a2a3' => 'r1bq1rk1/pp3ppp/2n1pn2/2pp4/1bPP4/P1NBPN2/1P3PPP/R1BQ1RK1 b - - 0 8',
				'c4d5' => 'r1bq1rk1/pp3ppp/2n1pn2/2pP4/1b1P4/2NBPN2/PP3PPP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2pp4/1bPP4/P1NBPN2/1P3PPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3'),
			moves => {
				'b4a5' => 'r1bq1rk1/pp3ppp/2n1pn2/b1pp4/2PP4/P1NBPN2/1P3PPP/R1BQ1RK1 w - - 1 9',
				'b4c3' => 'r1bq1rk1/pp3ppp/2n1pn2/2pp4/2PP4/P1bBPN2/1P3PPP/R1BQ1RK1 w - - 0 9',
				'c5d4' => 'r1bq1rk1/pp3ppp/2n1pn2/3p4/1bPp4/P1NBPN2/1P3PPP/R1BQ1RK1 w - - 0 9',
				'd5c4' => 'r1bq1rk1/pp3ppp/2n1pn2/2p5/1bpP4/P1NBPN2/1P3PPP/R1BQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2pp4/2PP4/P1PBPN2/5PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3 9. bxc3

			variation => N__('Nimzo-Indian: Main Line, 8...Bxc3'),
			moves => {
				'b7b6' => 'r1bq1rk1/p4ppp/1pn1pn2/2pp4/2PP4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 10',
				'd5c4' => 'r1bq1rk1/pp3ppp/2n1pn2/2p5/2pP4/P1PBPN2/5PPP/R1BQ1RK1 w - - 0 10',
				'd8c7' => 'r1b2rk1/ppq2ppp/2n1pn2/2pp4/2PP4/P1PBPN2/5PPP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/2pp4/2PP4/P1bBPN2/1P3PPP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E58',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Bxc3

			variation => N__('Nimzo-Indian: Main Line, 8...Bxc3'),
			moves => {
				'b2c3' => 'r1bq1rk1/pp3ppp/2n1pn2/2pp4/2PP4/P1PBPN2/5PPP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/3p4/1bPP4/P1NB1N2/1P3PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 cxd4 9. exd4

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 cxd4 9.exd4'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/3p4/1bPp4/P1NBPN2/1P3PPP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 cxd4

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 cxd4'),
			moves => {
				'e3d4' => 'r1bq1rk1/pp3ppp/2n1pn2/3p4/1bPP4/P1NB1N2/1P3PPP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/8/1bBP4/P1N2N2/1P3PPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4 10. exd4

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4 10.exd4'),
			moves => {
				'b4e7' => 'r1bq1rk1/pp2bppp/2n1pn2/8/2BP4/P1N2N2/1P3PPP/R1BQ1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/8/1bBp4/P1N1PN2/1P3PPP/R1BQ1RK1 w - - 0 10' => {
			eco => 'E57',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 cxd4

			variation => N__('Nimzo-Indian: Main Line, 8...dxc4 9.Bxc4 cxd4'),
			moves => {
				'e3d4' => 'r1bq1rk1/pp3ppp/2n1pn2/8/1bBP4/P1N2N2/1P3PPP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/b1p5/2BP4/P1N1PN2/1P3PPP/R1BQ1RK1 w - - 1 10' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 dxc4 9. Bxc4 Ba5

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 dxc4 9.Bxc4 Ba5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/b1pP4/3P4/P1NBPN2/1P3PPP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Ba5 9. cxd5

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 Ba5 9.cxd5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n1pn2/b1pp4/2PP4/P1NBPN2/1P3PPP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E56',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 c5 7. O-O Nc6 8. a3 Ba5

			variation => N__('Nimzo-Indian: Main Line, 7... Nc6 8. a3 Ba5'),
			moves => {
				'c4d5' => 'r1bq1rk1/pp3ppp/2n1pn2/b1pP4/3P4/P1NBPN2/1P3PPP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2n2B2/2b5/3p4/2N2NP1/PP2PPBP/R2Q1RK1 b - - 0 11' => {
			eco => 'D34l',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5 d4'),
			moves => {
				'd8f6' => 'r1b2rk1/pp3ppp/2n2q2/2b5/3p4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/2b3B1/3p4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 11' => {
			eco => 'D34l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5 Bxc5 10. Bg5 d4

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5 d4'),
			moves => {
				'g5f6' => 'r1bq1rk1/pp3ppp/2n2B2/2b5/3p4/2N2NP1/PP2PPBP/R2Q1RK1 b - - 0 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/2bp2B1/8/2N2NP1/PP2PPBP/R2Q1RK1 b - - 1 10' => {
			eco => 'D34k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5 Bxc5 10. Bg5

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5'),
			moves => {
				'c8e6' => 'r2q1rk1/pp3ppp/2n1bn2/2bp2B1/8/2N2NP1/PP2PPBP/R2Q1RK1 w - - 2 11',
				'd5d4' => 'r1bq1rk1/pp3ppp/2n2n2/2b3B1/3p4/2N2NP1/PP2PPBP/R2Q1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/2bp4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'D34i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5 Bxc5

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5'),
			moves => {
				'c1g5' => 'r1bq1rk1/pp3ppp/2n2n2/2bp2B1/8/2N2NP1/PP2PPBP/R2Q1RK1 b - - 1 10',
				'c3a4' => 'r1bq1rk1/pp3ppp/2n2n2/2bp4/N7/5NP1/PP2PPBP/R1BQ1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/2bp4/N7/5NP1/PP2PPBP/R1BQ1RK1 b - - 1 10' => {
			eco => 'D34j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O O-O 9. dxc5 Bxc5 10. Na4

			variation => N__('QGD Tarrasch: Reti Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/3p4/8/P1NB4/1PP2PPP/R1BQ1RK1 b - - 3 11' => {
			eco => 'B45u',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3, 9.ed5 ed5 10. Bd3 O-O'),
			moves => {
				'd5d4' => 'r1bq1rk1/pp3ppp/2n2n2/8/3p4/P1NB4/1PP2PPP/R1BQ1RK1 w - - 0 12',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/3p4/8/P1NB4/1PP2PPP/R1BQK2R w KQ - 2 11' => {
			eco => 'B45u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4 7. a3 Bxc3+ 8. Nxc3 d5 9. exd5 exd5 10. Bd3 O-O

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3, 9.ed5 ed5 10. Bd3 O-O'),
			moves => {
				'e1g1' => 'r1bq1rk1/pp3ppp/2n2n2/3p4/8/P1NB4/1PP2PPP/R1BQ1RK1 b - - 3 11',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/8/3p4/P1NB4/1PP2PPP/R1BQ1RK1 w - - 0 12' => {
			eco => 'B45v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4 7. a3 Bxc3+ 8. Nxc3 d5 9. exd5 exd5 10. Bd3 O-O 11. O-O d4

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3, 10. Bd3 O-O 11.O-O d4'),
			moves => {
				'c3e2' => 'r1bq1rk1/pp3ppp/2n2n2/8/3p4/P2B4/1PP1NPPP/R1BQ1RK1 b - - 1 12',
			},
		},
		'r1bq1rk1/pp3ppp/2n2n2/8/3p4/P2B4/1PP1NPPP/R1BQ1RK1 b - - 1 12' => {
			eco => 'B45w',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4 7. a3 Bxc3+ 8. Nxc3 d5 9. exd5 exd5 10. Bd3 O-O 11. O-O d4 12. Ne2

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3, 10. Bd3 O-O 11.O-O d4 12. Ne2'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n5/2bN4/3p4/5NP1/PP2PPBP/R2Q1RK1 w - - 2 13' => {
			eco => 'D34m',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5 d4 11.Bxf6 Qxf6 12. Nd5'),
			moves => {
				'f3d2' => 'r1bq1rk1/pp3ppp/2n5/2bN4/3p4/6P1/PP1NPPBP/R2Q1RK1 b - - 3 13',
			},
		},
		'r1bq1rk1/pp3ppp/2n5/2bN4/3p4/6P1/PP1NPPBP/R2Q1RK1 b - - 3 13' => {
			eco => 'D34m',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: 9.dxc5 Bxc5 10. Bg5 d4 11.Bxf6 Qxf6 12. Nd5'),
			moves => {
				'f8e8' => 'r1bqr1k1/pp3ppp/2n5/2bN4/3p4/6P1/PP1NPPBP/R2Q1RK1 w - - 4 14',
			},
		},
		'r1bq1rk1/pp3ppp/2n5/2bp2B1/8/5NP1/PP2PPBP/R2Q1RK1 b - - 1 11' => {
			eco => 'D41j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O 9. Nxd5 exd5 10. dxc5 Bxc5 11. Bg5

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9.Nxd5 exd5 10.dxc5 Bxc5 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2n5/2bp4/8/5NP1/PP2PPBP/R1BQ1RK1 w - - 0 11' => {
			eco => 'D41i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7 8. O-O O-O 9. Nxd5 exd5 10. dxc5 Bxc5

			variation => N__('QGD: Semi-Tarrasch, 6. g3, Main Line, 9.Nxd5 exd5 10.dxc5 Bxc5'),
			moves => {
				'c1g5' => 'r1bq1rk1/pp3ppp/2n5/2bp2B1/8/5NP1/PP2PPBP/R2Q1RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/pp3ppp/2nb1n2/3p4/3N4/1PN1P3/P3BPPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'D32x',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. e3 Nf6 7. Be2 cxd4 8. Nxd4 Bd6 9. O-O O-O 10. b3

			variation => N__('QGD Tarrasch: 5. Nf3 Nc6 6. e3 Nf6 7. Be2 cxd4 8.Nxd4 Bd6 9.O-O O-O 10. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2nb1n2/3p4/3N4/2N1P3/PP2BPPP/R1BQ1RK1 w - - 3 10' => {
			eco => 'D32w',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. e3 Nf6 7. Be2 cxd4 8. Nxd4 Bd6 9. O-O O-O

			variation => N__('QGD Tarrasch: 5. Nf3 Nc6 6. e3 Nf6 7. Be2 cxd4 8.Nxd4 Bd6 9.O-O O-O'),
			moves => {
				'b2b3' => 'r1bq1rk1/pp3ppp/2nb1n2/3p4/3N4/1PN1P3/P3BPPP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/pp3ppp/2nbpn2/2Pp4/8/2PBPN2/PP1N1PPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'D05m',
                        # TRANSLATORS:

			variation => N__('Colle: 5. c3 Nc6 6.Nbd2 Bd6 7.O-O O-O'),
			moves => {
				'd6c5' => 'r1bq1rk1/pp3ppp/2n1pn2/2bp4/8/2PBPN2/PP1N1PPP/R1BQ1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pp3ppp/2nbpn2/2pp4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8' => {
			eco => 'D40h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. e3 Nc6 6. Bd3 Bd6 7. O-O O-O

			variation => N__('QGD: Semi-Tarrasch, Symmetrical Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pp3ppp/2nbpn2/2pp4/3P4/2PBPN2/PP1N1PPP/R1BQ1RK1 w - - 5 8' => {
			eco => 'D05m',
                        # TRANSLATORS: 1. d4 d5 2. Nf3 Nf6 3. e3 e6 4. Bd3 c5 5. c3 Nc6 6. Nbd2 Bd6 7. O-O O-O

			variation => N__('Colle: 5. c3 Nc6 6.Nbd2 Bd6 7.O-O O-O'),
			moves => {
				'd4c5' => 'r1bq1rk1/pp3ppp/2nbpn2/2Pp4/8/2PBPN2/PP1N1PPP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pp3ppp/2nppn2/2p5/2PP4/P1PBP3/4NPPP/R1BQK2R w KQ - 0 9' => {
			eco => 'E29',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 d6

			variation => N__('Nimzo-Indian: Sämisch, 5...O-O 6. e3 c5 7. Bd3 Nc6 8. Ne2 d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp4bp/2nppnp1/2p5/2P1PPP1/2N5/PP2B2P/R1BQK1NR w KQ - 1 10' => {
			eco => 'E77',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Be2 c5 7. d5'),
			moves => {
				'h2h4' => 'r1bq1rk1/pp4bp/2nppnp1/2p5/2P1PPPP/2N5/PP2B3/R1BQK1NR b KQ h3 0 10',
			},
		},
		'r1bq1rk1/pp4bp/2nppnp1/2p5/2P1PPPP/2N5/PP2B3/R1BQK1NR b KQ h3 0 10' => {
			eco => 'E77',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Be2 c5 7. d5 e6 8. dxe6 fxe6 9. g4 Nc6 10. h4

			variation => N__('King\'s Indian: Four Pawns Attack, Six Pawns line'),
			moves => {
			},
		},
		'r1bq1rk1/pp4pp/2nbpn2/3p4/3P1B2/3B1N2/PP2NPPP/R2Q1RK1 b - - 5 12' => {
			eco => 'C06t',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 f6 9. exf6 Nxf6 10. O-O Bd6 11. Nf3 O-O 12. Bf4

			variation => N__('French: Tarrasch, Closed, 8... f6 9.exf6, 11...O-O 12. Bf4'),
			moves => {
			},
		},
		'r1bq1rk1/pp4pp/2nbpn2/3p4/3P4/3B1N2/PP2NPPP/R1BQ1RK1 w - - 4 12' => {
			eco => 'C06s',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ne2 cxd4 8. cxd4 f6 9. exf6 Nxf6 10. O-O Bd6 11. Nf3 O-O

			variation => N__('French: Tarrasch, Closed, 8... f6 9.exf6, 11...O-O'),
			moves => {
				'c1f4' => 'r1bq1rk1/pp4pp/2nbpn2/3p4/3P1B2/3B1N2/PP2NPPP/R2Q1RK1 b - - 5 12',
			},
		},
		'r1bq1rk1/ppn2pbp/3p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 w - - 8 11' => {
			eco => 'A73',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Na6 10. Nd2 Nc7

			variation => N__('Benoni: Classical, 9.O-O Na6 10. Nd2 Nc7'),
			moves => {
			},
		},
		'r1bq1rk1/ppn2pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w - - 7 11' => {
			eco => 'A62',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Na6 10. Nd2 Nc7

			variation => N__('Benoni: Fianchetto, 9.O-O Na6 10. Nd2 Nc7'),
			moves => {
			},
		},
		'r1bq1rk1/ppn2pbp/3ppnp1/2pP4/2P1P3/2NB1N1P/PP3PP1/R1BQ1RK1 w - - 4 10' => {
			eco => 'E90w',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 c5 7. d5 e6 8. Bd3 Na6 9. O-O Nc7

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 c5 7. d5 e6 8. Bd3 Na6 9.O-O Nc7'),
			moves => {
			},
		},
		'r1bq1rk1/ppnn1pbp/3p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQ1RK1 w - - 5 12' => {
			eco => 'A67s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O 10. Nf3 Na6 11. O-O Nc7

			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3 Na6 11.O-O Nc7'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1b1pp/2n1pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w - - 1 8' => {
			eco => 'A93',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 Nc6

			variation => N__('Dutch: Stonewall, Botvinnik, 7... Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1bppp/1nn5/4p3/8/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A29',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O Be7 8. d3 O-O

			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O Be7 8. d3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1bppp/2N1pn2/8/2pP4/6P1/PP2PPBP/RNBQ1RK1 b - - 0 8' => {
			eco => 'E05h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 dxc4 5. Nf3 Be7 6. O-O O-O 7. Ne5 Nc6 8. Nxc6

			variation => N__('Catalan: Open, Classical, 7. Ne5 Nc6 8.Nxc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1bppp/2n1pn2/4N3/2pP4/6P1/PP2PPBP/RNBQ1RK1 w - - 6 8' => {
			eco => 'E05g',
                        # TRANSLATORS:

			variation => N__('Catalan: Open, Classical, 7. Ne5'),
			moves => {
				'e5c6' => 'r1bq1rk1/ppp1bppp/2N1pn2/8/2pP4/6P1/PP2PPBP/RNBQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/ppp1bppp/2n1pn2/8/3PN3/P5N1/1PP1BPPP/R1BQK2R w KQ - 5 9' => {
			eco => 'C15u',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. Ne2 dxe4 5. a3 Be7 6. Nxe4 Nf6 7. N2g3 O-O 8. Be2 Nc6

			variation => N__('French: Winawer, Alekhine Gambit, Alatortsev Variation'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1bppp/2np1n2/4p3/4P3/1B1P1N2/PPP2PPP/RNBQ1RK1 w - - 0 7' => {
			eco => 'C55i',
                        # TRANSLATORS:

			variation => N__('Two Knights: 4. d3 Be7 5.O-O O-O 6. Bb3'),
			moves => {
				'c2c3' => 'r1bq1rk1/ppp1bppp/2np1n2/4p3/4P3/1BPP1N2/PP3PPP/RNBQ1RK1 b - - 0 7',
			},
		},
		'r1bq1rk1/ppp1bppp/2np1n2/4p3/4P3/1BPP1N2/PP3PPP/RNBQ1RK1 b - - 0 7' => {
			eco => 'C55j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d3 Be7 5. O-O O-O 6. Bb3 d6 7. c3

			variation => N__('Two Knights: 4. d3 Be7 5.O-O O-O 6. Bb3 d6 7. c3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1bppp/8/3p4/1nPPn3/5N2/PP2BPPP/RNBQ1RK1 w - - 3 10' => {
			eco => 'C42u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nf6 3. Nxe5 d6 4. Nf3 Nxe4 5. d4 d5 6. Bd3 Be7 7. O-O Nc6 8. c4 Nb4 9. Be2 O-O

			variation => N__('Russian Game: Classical, Jänisch, Main Line'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/2PPp3/4Pp2/2NN1P2/PP1BB1PP/R2Q1RK1 b - - 0 13' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Nd3 Nf6 12. Bd2 f4 13. c5

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Nd3 Nf6 12. Bd2 f4 13. c5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3Pp3/2P1Pp2/2NN1P2/PP1BB1PP/R2Q1RK1 w - - 0 13' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Nd3 Nf6 12. Bd2 f4

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Nd3 Nf6 12. Bd2 f4'),
			moves => {
				'c4c5' => 'r1bq1rk1/ppp1n1bp/3p1np1/2PPp3/4Pp2/2NN1P2/PP1BB1PP/R2Q1RK1 b - - 0 13',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P1P1/2N2P2/PP2B2P/R1BQNRK1 w - - 1 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. g4 Nf6

			variation => N__('King\'s Indian: Mar del Plata, Benko Attack, 11... Nf6'),
			moves => {
				'e1d3' => 'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P1P1/2NN1P2/PP2B2P/R1BQ1RK1 b - - 2 12',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P1P1/2NN1P2/PP2B2P/R1BQ1RK1 b - - 2 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. g4 Nf6 12. Nd3

			variation => N__('King\'s Indian: Mar del Plata, Benko Attack, 11... Nf6 12. Nd3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P3/2NN1P2/PP1BB1PP/R2Q1RK1 b - - 3 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Nd3 Nf6 12. Bd2

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Nd3 Nf6 12. Bd2'),
			moves => {
				'f5f4' => 'r1bq1rk1/ppp1n1bp/3p1np1/3Pp3/2P1Pp2/2NN1P2/PP1BB1PP/R2Q1RK1 w - - 0 13',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P3/2NN1P2/PP2B1PP/R1BQ1RK1 w - - 2 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Nd3 Nf6

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Nd3 Nf6'),
			moves => {
				'c1d2' => 'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P3/2NN1P2/PP1BB1PP/R2Q1RK1 b - - 3 12',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 w - - 2 12' => {
			eco => 'E98q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Nd3 f5 11. Bd2 Nf6

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Nd3 f5 11. Bd2 Nf6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3PppN1/1PP1P3/2N2B2/P4PPP/R1BQR1K1 b - - 3 12' => {
			eco => 'E97s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 f5 11. Ng5 Nf6 12. Bf3

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 f5 11. Ng5 Nf6 12. Bf3'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp2n1bp/2pp1np1/3PppN1/1PP1P3/2N2B2/P4PPP/R1BQR1K1 w - - 0 13',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p1np1/3PppN1/1PP1P3/2N5/P3BPPP/R1BQR1K1 w - - 2 12' => {
			eco => 'E97s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 f5 11. Ng5 Nf6

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 f5 11. Ng5 Nf6'),
			moves => {
				'e2f3' => 'r1bq1rk1/ppp1n1bp/3p1np1/3PppN1/1PP1P3/2N2B2/P4PPP/R1BQR1K1 b - - 3 12',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p2p1/3Ppp1n/1PP1P3/2N2N2/P3BPPP/R1BQR1K1 w - f6 0 11' => {
			eco => 'E97r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 f5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 f5'),
			moves => {
				'f3g5' => 'r1bq1rk1/ppp1n1bp/3p2p1/3PppNn/1PP1P3/2N5/P3BPPP/R1BQR1K1 b - - 1 11',
			},
		},
		'r1bq1rk1/ppp1n1bp/3p2p1/3PppNn/1PP1P3/2N5/P3BPPP/R1BQR1K1 b - - 1 11' => {
			eco => 'E97r',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 f5'),
			moves => {
				'h5f6' => 'r1bq1rk1/ppp1n1bp/3p1np1/3PppN1/1PP1P3/2N5/P3BPPP/R1BQR1K1 w - - 2 12',
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp1B1/2P1P3/2N2N2/PP2BPPP/R2Q1RK1 b - - 2 9' => {
			eco => 'E97d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Bg5

			variation => N__('King\'s Indian: Mar del Plata, Odessa Variation'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 b - b3 0 9' => {
			eco => 'E97f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack'),
			moves => {
				'a7a5' => 'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - a6 0 10',
				'f6e8' => 'r1bqnrk1/ppp1npbp/3p2p1/3Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - - 1 10',
				'f6h5' => 'r1bq1rk1/ppp1npbp/3p2p1/3Pp2n/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - - 1 10',
				'g8h8' => 'r1bq1r1k/ppp1npbp/3p1np1/3Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 2 9' => {
			eco => 'E85j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 Nc6 8. d5 Ne7 9. Qd2

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 Nc6 8. d5 Ne7 9. Qd2'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 1 9' => {
			eco => 'E85i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 Nc6 8. d5 Ne7

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 Nc6 8. d5 Ne7'),
			moves => {
				'd1d2' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N1BP2/PP1QN1PP/R3KB1R b KQ - 2 9',
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N2N2/PP1BBPPP/R2Q1RK1 b - - 2 9' => {
			eco => 'E97e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Bd2

			variation => N__('King\'s Indian: Mar del Plata, 9. Bd2'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E97d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7

			variation => N__('King\'s Indian: Mar del Plata, 8. d5 Ne7'),
			moves => {
				'b2b4' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 b - b3 0 9',
				'c1d2' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N2N2/PP1BBPPP/R2Q1RK1 b - - 2 9',
				'c1g5' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp1B1/2P1P3/2N2N2/PP2BPPP/R2Q1RK1 b - - 2 9',
				'f3d2' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 b - - 2 9',
				'f3e1' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 b - - 2 9',
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 9' => {
			eco => 'E62v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O e5 8. d5 Ne7 9. e4

			variation => N__('King\'s Indian: Fianchetto, Uhlmann/Szabo, 9. e4'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 b - - 2 9' => {
			eco => 'E97t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2'),
			moves => {
				'a7a5' => 'r1bq1rk1/1pp1npbp/3p1np1/p2Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - a6 0 10',
				'c7c5' => 'r1bq1rk1/pp2npbp/3p1np1/2pPp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - c6 0 10',
				'f6d7' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - - 3 10',
				'f6e8' => 'r1bqnrk1/ppp1npbp/3p2p1/3Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - - 3 10',
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 b - - 2 9' => {
			eco => 'E98a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp2npbp/3p1np1/2pPp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 w - c6 0 10',
				'f6d7' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 w - - 3 10',
				'f6e8' => 'r1bqnrk1/ppp1npbp/3p2p1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 w - - 3 10',
			},
		},
		'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E62u',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O e5 8. d5 Ne7

			variation => N__('King\'s Indian: Fianchetto, Uhlmann/Szabo Variation'),
			moves => {
				'e2e4' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 9',
			},
		},
		'r1bq1rk1/ppp1npbp/3p2p1/2PPp2n/1P2P3/2N2N2/P3BPPP/R1BQ1RK1 b - - 0 10' => {
			eco => 'E97m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. c5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... Nh5 10. c5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1npbp/3p2p1/3Pp2n/1PP1P3/2N2N2/P3BPPP/R1BQ1RK1 w - - 1 10' => {
			eco => 'E97l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... Nh5'),
			moves => {
				'c4c5' => 'r1bq1rk1/ppp1npbp/3p2p1/2PPp2n/1P2P3/2N2N2/P3BPPP/R1BQ1RK1 b - - 0 10',
				'f1e1' => 'r1bq1rk1/ppp1npbp/3p2p1/3Pp2n/1PP1P3/2N2N2/P3BPPP/R1BQR1K1 b - - 2 10',
				'g2g3' => 'r1bq1rk1/ppp1npbp/3p2p1/3Pp2n/1PP1P3/2N2NP1/P3BP1P/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/ppp1npbp/3p2p1/3Pp2n/1PP1P3/2N2N2/P3BPPP/R1BQR1K1 b - - 2 10' => {
			eco => 'E97o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... Nh5 10. Re1'),
			moves => {
				'a7a5' => 'r1bq1rk1/1pp1npbp/3p2p1/p2Pp2n/1PP1P3/2N2N2/P3BPPP/R1BQR1K1 w - a6 0 11',
				'f7f5' => 'r1bq1rk1/ppp1n1bp/3p2p1/3Ppp1n/1PP1P3/2N2N2/P3BPPP/R1BQR1K1 w - f6 0 11',
				'h5f4' => 'r1bq1rk1/ppp1npbp/3p2p1/3Pp3/1PP1Pn2/2N2N2/P3BPPP/R1BQR1K1 w - - 3 11',
			},
		},
		'r1bq1rk1/ppp1npbp/3p2p1/3Pp2n/1PP1P3/2N2NP1/P3BP1P/R1BQ1RK1 b - - 0 10' => {
			eco => 'E97n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. g3

			variation => N__('King\'s Indian: Mar del Plata, Bayonet Attack, 9... Nh5 10. g3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1npbp/3p2p1/3Pp3/1PP1Pn2/2N2N2/P3BPPP/R1BQR1K1 w - - 3 11' => {
			eco => 'E97q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. b4 Nh5 10. Re1 Nf4

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 Nf4'),
			moves => {
				'e2f1' => 'r1bq1rk1/ppp1npbp/3p2p1/3Pp3/1PP1Pn2/2N2N2/P4PPP/R1BQRBK1 b - - 4 11',
			},
		},
		'r1bq1rk1/ppp1npbp/3p2p1/3Pp3/1PP1Pn2/2N2N2/P4PPP/R1BQRBK1 b - - 4 11' => {
			eco => 'E97q',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, Bayonet, 9... Nh5 10. Re1 Nf4'),
			moves => {
				'a7a5' => 'r1bq1rk1/1pp1npbp/3p2p1/p2Pp3/1PP1Pn2/2N2N2/P4PPP/R1BQRBK1 w - a6 0 12',
			},
		},
		'r1bq1rk1/ppp1nppN/3p4/3P4/2B1R3/8/PP3PPP/R2Q2K1 b - - 0 14' => {
			eco => 'C54o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Nf6 5. d4 exd4 6. cxd4 Bb4+ 7. Nc3 Nxe4 8. O-O Bxc3 9. d5 Bf6 10. Re1 Ne7 11. Rxe4 d6 12. Bg5 Bxg5 13. Nxg5 O-O 14. Nxh7

			variation => N__('Giuoco Piano: Möller-Herzog Variation'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1nppp/3p1n2/1B2p1B1/1b2P2N/2NP4/PPP2PPP/R2Q1RK1 b - - 3 8' => {
			eco => 'C49k',
                        # TRANSLATORS:

			variation => N__('Four Knights: Symmetrical, Pillsbury Variation'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp2nppp/2pp1n2/1B2p1B1/1b2P2N/2NP4/PPP2PPP/R2Q1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/ppp1nppp/3p1n2/1B2p1B1/1b2P3/2NP1N2/PPP2PPP/R2Q1RK1 w - - 2 8' => {
			eco => 'C49k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Ne7

			variation => N__('Four Knights: Symmetrical, Pillsbury Variation'),
			moves => {
				'f3h4' => 'r1bq1rk1/ppp1nppp/3p1n2/1B2p1B1/1b2P2N/2NP4/PPP2PPP/R2Q1RK1 b - - 3 8',
			},
		},
		'r1bq1rk1/ppp1nppp/3p4/3P2N1/2B1R3/8/PP3PPP/R2Q2K1 w - - 1 14' => {
			eco => 'C54o',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: Möller (Therkatz) Attack'),
			moves => {
				'g5h7' => 'r1bq1rk1/ppp1nppN/3p4/3P4/2B1R3/8/PP3PPP/R2Q2K1 b - - 0 14',
			},
		},
		'r1bq1rk1/ppp1p1bp/2np1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5'),
			moves => {
				'c6a5' => 'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9',
				'c6e5' => 'r1bq1rk1/ppp1p1bp/3p1np1/3Pnp2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/ppp1p1bp/2np1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. b3

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. b3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/2np1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6'),
			moves => {
				'b2b3' => 'r1bq1rk1/ppp1p1bp/2np1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8',
				'd4d5' => 'r1bq1rk1/ppp1p1bp/2np1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/3PNp2/2P5/2N3P1/PP2PPBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5'),
			moves => {
				'd6e5' => 'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P5/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/3Pnp2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5'),
			moves => {
				'f3e5' => 'r1bq1rk1/ppp1p1bp/3p1np1/3PNp2/2P5/2N3P1/PP2PPBP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Na5

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Na5'),
			moves => {
				'd1d3' => 'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2NQ1NP1/PP2PPBP/R1B2RK1 b - - 2 9',
				'f3d2' => 'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 2 9',
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 b - - 2 9' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Na5 9. Nd2

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Na5 9. Nd2'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2NQ1NP1/PP2PPBP/R1B2RK1 b - - 2 9' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Na5 9. Qd3

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Na5 9. Qd3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P1P3/2N3P1/PP3PBP/R1BQ1RK1 b - e3 0 10' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5 dxe5 10. e4

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5 dxe5 10. e4'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P5/1QN3P1/PP2PPBP/R1B2RK1 b - - 1 10' => {
			eco => 'A89',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5 dxe5 10. Qb3

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5 dxe5 10. Qb3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P5/2N3P1/PP2PPBP/R1BQ1RK1 w - - 0 10' => {
			eco => 'A89',
                        # TRANSLATORS:

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5'),
			moves => {
				'd1b3' => 'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P5/1QN3P1/PP2PPBP/R1B2RK1 b - - 1 10',
				'e2e4' => 'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P1P3/2N3P1/PP3PBP/R1BQ1RK1 b - e3 0 10',
			},
		},
		'r1bq1rk1/ppp1p1bp/n2p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'A87',
                        # TRANSLATORS: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Na6

			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/1n4p1/n2P4/4P3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 10' => {
			eco => 'D76k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. d5 Na5 10. e4

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. d5 Na5 10. e4'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp2ppbp/1np3p1/n2P4/4P3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 11',
			},
		},
		'r1bq1rk1/ppp1ppbp/1n4p1/n2P4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 10' => {
			eco => 'D76j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. d5 Na5

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. d5 Na5'),
			moves => {
				'e2e4' => 'r1bq1rk1/ppp1ppbp/1n4p1/n2P4/4P3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 10',
			},
		},
		'r1bq1rk1/ppp1ppbp/1nn3p1/3P4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D76i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. d5

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. d5'),
			moves => {
				'c6a5' => 'r1bq1rk1/ppp1ppbp/1n4p1/n2P4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/ppp1ppbp/1nn3p1/8/3P4/2N1PNP1/PP3PBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'D76d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. e3

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. e3'),
			moves => {
				'e7e5' => 'r1bq1rk1/ppp2pbp/1nn3p1/4p3/3P4/2N1PNP1/PP3PBP/R1BQ1RK1 w - e6 0 10',
				'f8e8' => 'r1bqr1k1/ppp1ppbp/1nn3p1/8/3P4/2N1PNP1/PP3PBP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/ppp1ppbp/1nn3p1/8/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 4 9' => {
			eco => 'D76c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp1ppbp/1nn3p1/3P4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 9',
				'e2e3' => 'r1bq1rk1/ppp1ppbp/1nn3p1/8/3P4/2N1PNP1/PP3PBP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/ppp1ppbp/2n2np1/3p4/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w - - 6 7' => {
			eco => 'D77d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O Nc6

			variation => N__('Neo-Grünfeld, 6.O-O Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2n2np1/8/2BP4/1QN1PN2/PP3PPP/R1B1K2R w KQ - 1 8' => {
			eco => 'D95j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. e3 O-O 6. Qb3 dxc4 7. Bxc4 Nc6

			variation => N__('Grünfeld: 5. e3 O-O 6. Qb3 dxc4 7.Bxc4 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2n2np1/8/2NP4/5NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'D77j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O dxc4 7. Na3 Nc6 8. Nxc4

			variation => N__('Neo-Grünfeld, 6.O-O dxc4 7. Na3 Nc6 8.Nxc4'),
			moves => {
				'c8e6' => 'r2q1rk1/ppp1ppbp/2n1bnp1/8/2NP4/5NP1/PP2PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/ppp1ppbp/2n2np1/8/2QPP3/2N2N2/PP2BPPP/R1B1K2R b KQ - 2 8' => {
			eco => 'D97l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. Qb3 dxc4 6. Qxc4 O-O 7. e4 Nc6 8. Be2

			variation => N__('Grünfeld: Russian, Simagin, 8. Be2'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2n2np1/8/2QPP3/2N2N2/PP3PPP/R1B1KB1R w KQ - 1 8' => {
			eco => 'D97l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. Qb3 dxc4 6. Qxc4 O-O 7. e4 Nc6

			variation => N__('Grünfeld: Russian, Simagin Variation'),
			moves => {
				'f1e2' => 'r1bq1rk1/ppp1ppbp/2n2np1/8/2QPP3/2N2N2/PP2BPPP/R1B1K2R b KQ - 2 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/2n2np1/8/2pP4/N4NP1/PP2PPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'D77j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O dxc4 7. Na3 Nc6

			variation => N__('Neo-Grünfeld, 6.O-O dxc4 7. Na3 Nc6'),
			moves => {
				'a3c4' => 'r1bq1rk1/ppp1ppbp/2n2np1/8/2NP4/5NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/2n3p1/3n4/3P4/5NP1/PP2PPBP/RNBQ1RK1 w - - 2 8' => {
			eco => 'D74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nc6

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2n3p1/8/2BPP3/2P1B3/P3NPPP/R2QK2R b KQ - 6 9' => {
			eco => 'D86',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 Nc6 9. Be3

			variation => N__('Grünfeld: Classical Exchange, Simagin Improved, 9. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2n3p1/8/2BPP3/2P5/P3NPPP/R1BQ1RK1 b - - 6 9' => {
			eco => 'D86',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 Nc6 9. O-O

			variation => N__('Grünfeld: Classical Exchange, Simagin Improved, 9.O-O'),
			moves => {
				'b7b6' => 'r1bq1rk1/p1p1ppbp/1pn3p1/8/2BPP3/2P5/P3NPPP/R1BQ1RK1 w - - 0 10',
				'e7e5' => 'r1bq1rk1/ppp2pbp/2n3p1/4p3/2BPP3/2P5/P3NPPP/R1BQ1RK1 w - e6 0 10',
			},
		},
		'r1bq1rk1/ppp1ppbp/2n3p1/8/2BPP3/2P5/P3NPPP/R1BQK2R w KQ - 5 9' => {
			eco => 'D86',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 Nc6

			variation => N__('Grünfeld: Classical Exchange, Simagin Improved'),
			moves => {
				'c1e3' => 'r1bq1rk1/ppp1ppbp/2n3p1/8/2BPP3/2P1B3/P3NPPP/R2QK2R b KQ - 6 9',
				'e1g1' => 'r1bq1rk1/ppp1ppbp/2n3p1/8/2BPP3/2P5/P3NPPP/R1BQ1RK1 b - - 6 9',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/3P4/2P1P3/2N2N2/PP2BPPP/R1BQK2R b KQ - 0 7' => {
			eco => 'E91p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nc6 7. d5

			variation => N__('King\'s Indian: 6. Be2 Nc6 7. d5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/3P4/2P5/2N2NP1/PP2PPBP/R1BQK2R b KQ - 0 7' => {
			eco => 'E62m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. d5

			variation => N__('King\'s Indian: Fianchetto, 6... Nc6 7. d5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/4P3/3P1P2/2NB1N2/PPP3PP/R1BQK2R b KQ - 0 7' => {
			eco => 'B09k',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Nc6 7. e5

			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Nc6 7. e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/6B1/2PPP3/2N2P2/PP4PP/R2QKBNR w KQ - 3 7' => {
			eco => 'E81f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Bg5 Nc6

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Bg5 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 4 7' => {
			eco => 'E62n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O

			variation => N__('King\'s Indian: Fianchetto, 6... Nc6 7.O-O'),
			moves => {
				'a7a6' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
				'a8b8' => '1rbq1rk1/ppp1ppbp/2np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 8',
				'c8f5' => 'r2q1rk1/ppp1ppbp/2np1np1/5b2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 8',
				'c8g4' => 'r2q1rk1/ppp1ppbp/2np1np1/8/2PP2b1/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 8',
				'e7e5' => 'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - e6 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQ - 3 7' => {
			eco => 'E62l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6

			variation => N__('King\'s Indian: Fianchetto, 6... Nc6'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp1ppbp/2np1np1/3P4/2P5/2N2NP1/PP2PPBP/R1BQK2R b KQ - 0 7',
				'e1g1' => 'r1bq1rk1/ppp1ppbp/2np1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 4 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w - - 2 7' => {
			eco => 'E60n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nf3 Bg7 4. g3 O-O 5. Bg2 d6 6. O-O Nc6

			variation => N__('King\'s Indian: Fianchetto without Nc3, 6... Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP1Q2PP/R3KBNR b KQ - 4 7' => {
			eco => 'E83b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Qd2

			variation => N__('King\'s Indian: Sämisch, 6... Nc6 7. Qd2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP1Q2PP/R3KBNR w KQ - 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP2N1PP/R2QKB1R b KQ - 4 7' => {
			eco => 'E83d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6 7. Nge2

			variation => N__('King\'s Indian: Sämisch, 6... Nc6 7.Nge2'),
			moves => {
				'a7a6' => 'r1bq1rk1/1pp1ppbp/p1np1np1/8/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 0 8',
				'a8b8' => '1rbq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 5 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP4PP/R2QKBNR w KQ - 3 7' => {
			eco => 'E83a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nc6

			variation => N__('King\'s Indian: Sämisch, 6... Nc6'),
			moves => {
				'd1d2' => 'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP1Q2PP/R3KBNR b KQ - 4 7',
				'g1e2' => 'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N1BP2/PP2N1PP/R2QKB1R b KQ - 4 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQ - 4 7' => {
			eco => 'E91o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nc6

			variation => N__('King\'s Indian: 6. Be2 Nc6'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp1ppbp/2np1np1/3P4/2P1P3/2N2N2/PP2BPPP/R1BQK2R b KQ - 0 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2N3P1/PP3PBP/R1BQK1NR w KQ - 3 7' => {
			eco => 'E72',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. g3 O-O 6. Bg2 Nc6

			variation => N__('King\'s Indian: 4. e4 d6 5. g3 O-O 6. Bg2 Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2NB4/PP2NPPP/R1BQ1RK1 b - - 5 7' => {
			eco => 'E70n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Bd3 O-O 6. Nge2 Nc6 7. O-O

			variation => N__('King\'s Indian: 4. e4 d6 5. Bd3 O-O 6.Nge2 Nc6 7.O-O'),
			moves => {
				'e7e5' => 'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2NB4/PP2NPPP/R1BQ1RK1 w - e6 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2NB4/PP2NPPP/R1BQK2R w KQ - 4 7' => {
			eco => 'E70n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Bd3 O-O 6. Nge2 Nc6

			variation => N__('King\'s Indian: 4. e4 d6 5. Bd3 O-O 6.Nge2 Nc6'),
			moves => {
				'e1g1' => 'r1bq1rk1/ppp1ppbp/2np1np1/8/2PPP3/2NB4/PP2NPPP/R1BQ1RK1 b - - 5 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQ1RK1 w - - 6 7' => {
			eco => 'B08o',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Nc6

			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Nc6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQK2R w KQ - 5 7' => {
			eco => 'B09j',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Nc6

			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Nc6'),
			moves => {
				'e4e5' => 'r1bq1rk1/ppp1ppbp/2np1np1/4P3/3P1P2/2NB1N2/PPP3PP/R1BQK2R b KQ - 0 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/4P3/2PP1P2/2N2N2/PP4PP/R1BQKB1R b KQ - 0 7' => {
			eco => 'E76g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Nf3 Na6 7. e5

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Nf3 Na6 7. e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP2P/2N5/PP2BPP1/R2QK1NR b KQ h3 0 7' => {
			eco => 'E73q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6 7. h4

			variation => N__('King\'s Indian: Averbakh, 6... Na6 7. h4'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 7' => {
			eco => 'E90m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 Na6 7. Bg5

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 Na6 7. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N2N2/PP2BPPP/R2QK2R b KQ - 5 7' => {
			eco => 'E91l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Na6 7. Bg5

			variation => N__('King\'s Indian: Kazakh Variation, 7. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N4P/PP3PP1/R2QKBNR w KQ - 3 7' => {
			eco => 'E71k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. h3 O-O 6. Bg5 Na6

			variation => N__('King\'s Indian: Makagonov, 5...O-O 6. Bg5 Na6'),
			moves => {
				'f1d3' => 'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2NB3P/PP3PP1/R2QK1NR b KQ - 4 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N5/PP1QBPPP/R3K1NR b KQ - 5 7' => {
			eco => 'E73r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6 7. Qd2

			variation => N__('King\'s Indian: Averbakh, 6... Na6 7. Qd2'),
			moves => {
				'e7e5' => 'r1bq1rk1/ppp2pbp/n2p1np1/4p1B1/2PPP3/2N5/PP1QBPPP/R3K1NR w KQ e6 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N5/PP2BPPP/R2QK1NR w KQ - 4 7' => {
			eco => 'E73p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6

			variation => N__('King\'s Indian: Averbakh, 6... Na6'),
			moves => {
				'd1d2' => 'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N5/PP1QBPPP/R3K1NR b KQ - 5 7',
				'h2h4' => 'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP2P/2N5/PP2BPP1/R2QK1NR b KQ h3 0 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2NB3P/PP3PP1/R2QK1NR b KQ - 4 7' => {
			eco => 'E71l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. h3 O-O 6. Bg5 Na6 7. Bd3

			variation => N__('King\'s Indian: Makagonov, 5...O-O 6. Bg5 Na6 7. Bd3'),
			moves => {
				'e7e5' => 'r1bq1rk1/ppp2pbp/n2p1np1/4p1B1/2PPP3/2NB3P/PP3PP1/R2QK1NR w KQ e6 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPP3/2N2N1P/PP3PP1/R1BQKB1R w KQ - 1 7' => {
			eco => 'E90m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 Na6

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 Na6'),
			moves => {
				'c1g5' => 'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 5 7' => {
			eco => 'E91m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Na6 7. O-O

			variation => N__('King\'s Indian: Kazakh Variation, 7.O-O'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp2ppbp/n2p1np1/2p5/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - c6 0 8',
				'c7c6' => 'r1bq1rk1/pp2ppbp/n1pp1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQ - 4 7' => {
			eco => 'E91l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Na6

			variation => N__('King\'s Indian: Kazakh Variation'),
			moves => {
				'c1g5' => 'r1bq1rk1/ppp1ppbp/n2p1np1/6B1/2PPP3/2N2N2/PP2BPPP/R2QK2R b KQ - 5 7',
				'e1g1' => 'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 5 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPPP2/2N2N2/PP2B1PP/R1BQK2R b KQ - 4 7' => {
			eco => 'E76h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Nf3 Na6 7. Be2

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Nf3 Na6 7. Be2'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPPP2/2N2N2/PP4PP/R1BQKB1R w KQ - 3 7' => {
			eco => 'E76f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Nf3 Na6

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Nf3 Na6'),
			moves => {
				'e4e5' => 'r1bq1rk1/ppp1ppbp/n2p1np1/4P3/2PP1P2/2N2N2/PP4PP/R1BQKB1R b KQ - 0 7',
				'f1d3' => 'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPPP2/2NB1N2/PP4PP/R1BQK2R b KQ - 4 7',
				'f1e2' => 'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPPP2/2N2N2/PP2B1PP/R1BQK2R b KQ - 4 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/2PPPP2/2NB1N2/PP4PP/R1BQK2R b KQ - 4 7' => {
			eco => 'E76i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Nf3 Na6 7. Bd3

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Nf3 Na6 7. Bd3'),
			moves => {
				'e7e5' => 'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPPP2/2NB1N2/PP4PP/R1BQK2R w KQ e6 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQ1RK1 b - - 6 7' => {
			eco => 'B09l',
                        # TRANSLATORS:

			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp2ppbp/n2p1np1/2p5/3PPP2/2NB1N2/PPP3PP/R1BQ1RK1 w - c6 0 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQK2R w KQ - 5 7' => {
			eco => 'B09l',
                        # TRANSLATORS: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6

			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6'),
			moves => {
				'e1g1' => 'r1bq1rk1/ppp1ppbp/n2p1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQ1RK1 b - - 6 7',
			},
		},
		'r1bq1rk1/ppp1ppbp/n4np1/8/2QPP3/2N2N2/PP2BPPP/R1B1K2R b KQ - 2 8' => {
			eco => 'D97o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. Qb3 dxc4 6. Qxc4 O-O 7. e4 Na6 8. Be2

			variation => N__('Grünfeld: Russian, Prins, 8. Be2'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp2ppbp/n4np1/2p5/2QPP3/2N2N2/PP2BPPP/R1B1K2R w KQ c6 0 9',
			},
		},
		'r1bq1rk1/ppp1ppbp/n4np1/8/2QPP3/2N2N2/PP3PPP/R1B1KB1R w KQ - 1 8' => {
			eco => 'D97m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. Qb3 dxc4 6. Qxc4 O-O 7. e4 Na6

			variation => N__('Grünfeld: Russian, Prins Variation'),
			moves => {
				'c1f4' => 'r1bq1rk1/ppp1ppbp/n4np1/8/2QPPB2/2N2N2/PP3PPP/R3KB1R b KQ - 2 8',
				'f1e2' => 'r1bq1rk1/ppp1ppbp/n4np1/8/2QPP3/2N2N2/PP2BPPP/R1B1K2R b KQ - 2 8',
			},
		},
		'r1bq1rk1/ppp1ppbp/n4np1/8/2QPPB2/2N2N2/PP3PPP/R3KB1R b KQ - 2 8' => {
			eco => 'D97n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. Qb3 dxc4 6. Qxc4 O-O 7. e4 Na6 8. Bf4

			variation => N__('Grünfeld: Russian, Prins, 8. Bf4'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n4np1/8/2pP4/N4NP1/PP2PPBP/R1BQ1RK1 w - - 2 8' => {
			eco => 'D77g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. O-O dxc4 7. Na3 Na6

			variation => N__('Neo-Grünfeld, 6.O-O dxc4 7. Na3 Na6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n5p1/3n4/3P4/5NP1/PP2PPBP/RNBQ1RK1 w - - 2 8' => {
			eco => 'D74',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Na6

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Na6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/1n4p1/n2Pp3/8/2N1PNP1/PP3PBP/R1BQ1RK1 w - - 1 11' => {
			eco => 'D76f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. e3 e5 10. d5 Na5

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. e3 e5 10. d5 Na5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/1nn3p1/3Pp3/8/2N1PNP1/PP3PBP/R1BQ1RK1 b - - 0 10' => {
			eco => 'D76e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. e3 e5 10. d5

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. e3 e5 10. d5'),
			moves => {
				'c6a5' => 'r1bq1rk1/ppp2pbp/1n4p1/n2Pp3/8/2N1PNP1/PP3PBP/R1BQ1RK1 w - - 1 11',
			},
		},
		'r1bq1rk1/ppp2pbp/1nn3p1/4p3/3P4/2N1PNP1/PP3PBP/R1BQ1RK1 w - e6 0 10' => {
			eco => 'D76e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. g3 d5 4. Bg2 Bg7 5. Nf3 O-O 6. cxd5 Nxd5 7. O-O Nb6 8. Nc3 Nc6 9. e3 e5

			variation => N__('Neo-Grünfeld, 6.cxd5 Nxd5 7.O-O Nb6 8. Nc3 Nc6 9. e3 e5'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp2pbp/1nn3p1/3Pp3/8/2N1PNP1/PP3PBP/R1BQ1RK1 b - - 0 10',
			},
		},
		'r1bq1rk1/ppp2pbp/2n3p1/4p3/2BPP3/2P5/P3NPPP/R1BQ1RK1 w - e6 0 10' => {
			eco => 'D86',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Bc4 O-O 8. Ne2 Nc6 9. O-O e5

			variation => N__('Grünfeld: Classical Exchange, Simagin Improved, 9.O-O e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/3Pp3/2P1P3/2N1BP2/PP2N1PP/R2QKB1R b KQ - 0 8' => {
			eco => 'E85h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 Nc6 8. d5

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 Nc6 8. d5'),
			moves => {
				'c6e7' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 1 9',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E97d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5

			variation => N__('King\'s Indian: Mar del Plata, 8. d5'),
			moves => {
				'c6e7' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/3Pp3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E62t',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Fianchetto, Uhlmann/Szabo Variation'),
			moves => {
				'c6e7' => 'r1bq1rk1/ppp1npbp/3p1np1/3Pp3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4P3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E97b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. dxe5

			variation => N__('King\'s Indian: Mar del Plata, 8.dxe5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP1NP1/PP3PBP/R1BQ1RK1 b - - 4 8' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nf3 O-O 8. O-O

			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP1NP1/PP3PBP/R1BQK2R w KQ - 3 8' => {
			eco => 'A26',
                        # TRANSLATORS:

			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'e1g1' => 'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP1NP1/PP3PBP/R1BQ1RK1 b - - 4 8',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b - - 4 8' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nge2 O-O 8. O-O

			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQK2R w KQ - 3 8' => {
			eco => 'A26',
                        # TRANSLATORS:

			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'e1g1' => 'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b - - 4 8',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 b - - 5 8' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1

			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1'),
			moves => {
				'a7a5' => 'r1bq1rk1/1pp2pbp/2np1np1/p3p3/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 w - a6 0 9',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 4 8' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O

			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O'),
			moves => {
				'a1b1' => 'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 b - - 5 8',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - e6 0 8' => {
			eco => 'E62t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nc6 7. O-O e5

			variation => N__('King\'s Indian: Fianchetto, Uhlmann/Szabo Variation'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp2pbp/2np1np1/3Pp3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8' => {
			eco => 'E97c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. Be3

			variation => N__('King\'s Indian: Mar del Plata, 8. Be3'),
			moves => {
				'f6g4' => 'r1bq1rk1/ppp2pbp/2np2p1/4p3/2PPP1n1/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 2 8' => {
			eco => 'E85g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 Nc6

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 Nc6'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp2pbp/2np1np1/3Pp3/2P1P3/2N1BP2/PP2N1PP/R2QKB1R b KQ - 0 8',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E97a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6

			variation => N__('King\'s Indian: Mar del Plata'),
			moves => {
				'c1e3' => 'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8',
				'd4d5' => 'r1bq1rk1/ppp2pbp/2np1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 0 8',
				'd4e5' => 'r1bq1rk1/ppp2pbp/2np1np1/4P3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2N3P1/PP2NPBP/R1BQK2R w KQ - 2 8' => {
			eco => 'E72',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. g3 O-O 6. Bg2 e5 7. Nge2 Nc6

			variation => N__('King\'s Indian: Pomar System'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2PPP3/2NB4/PP2NPPP/R1BQ1RK1 w - e6 0 8' => {
			eco => 'E70n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Bd3 O-O 6. Nge2 Nc6 7. O-O e5

			variation => N__('King\'s Indian: 4. e4 d6 5. Bd3 O-O 6.Nge2 Nc6 7.O-O e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/2np2p1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 b - - 5 9' => {
			eco => 'E97c',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, 8. Be3 Ng4'),
			moves => {
				'f7f6' => 'r1bq1rk1/ppp3bp/2np1pp1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 w - - 0 10',
			},
		},
		'r1bq1rk1/ppp2pbp/2np2p1/4p3/2PPP1n1/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9' => {
			eco => 'E97c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. Be3 Ng4

			variation => N__('King\'s Indian: Mar del Plata, 8. Be3 Ng4'),
			moves => {
				'e3g5' => 'r1bq1rk1/ppp2pbp/2np2p1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 b - - 5 9',
			},
		},
		'r1bq1rk1/ppp2pbp/3p1np1/2n5/2PNP3/2N3P1/PP3PBP/R1BQ1RK1 w - - 1 10' => {
			eco => 'E68d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 exd4 9. Nxd4 Nc5

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 exd4 9.Nxd4 Nc5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/3p1np1/2n5/2PNP3/2N5/PP2BPPP/R1BQR1K1 w - - 1 10' => {
			eco => 'E95g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 exd4 9. Nxd4 Nc5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 exd4 9.Nxd4 Nc5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/3p1np1/2nPp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR w KQ - 1 9' => {
			eco => 'E73o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Nbd7 7. Qd2 e5 8. d5 Nc5

			variation => N__('King\'s Indian: Averbakh, 6...Nbd7 7. Qd2 e5 8. d5 Nc5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/3p1np1/2nPp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 1 9' => {
			eco => 'E94q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. d5 Nc5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. d5 Nc5'),
			moves => {
				'd1c2' => 'r1bq1rk1/ppp2pbp/3p1np1/2nPp3/2P1P3/2N2N2/PPQ1BPPP/R1B2RK1 b - - 2 9',
			},
		},
		'r1bq1rk1/ppp2pbp/3p1np1/2nPp3/2P1P3/2N2N2/PPQ1BPPP/R1B2RK1 b - - 2 9' => {
			eco => 'E94r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. d5 Nc5 9. Qc2

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. d5 Nc5 9. Qc2'),
			moves => {
				'a7a5' => 'r1bq1rk1/1pp2pbp/3p1np1/p1nPp3/2P1P3/2N2N2/PPQ1BPPP/R1B2RK1 w - a6 0 10',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/3Pp1B1/2P1P3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 8' => {
			eco => 'E90s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 e5 7. d5 Na6 8. Bg5

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 e5 7. d5 Na6 8. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR b KQ - 0 8' => {
			eco => 'E73s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6 7. Qd2 e5 8. d5

			variation => N__('King\'s Indian: Averbakh, 6... Na6 7. Qd2 e5 8. d5'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp3pbp/n1pp1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR w KQ - 0 9',
				'd8e8' => 'r1b1qrk1/ppp2pbp/n2p1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR w KQ - 1 9',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/3Pp3/2P1P3/2N1BN1P/PP3PP1/R2QKB1R b KQ - 2 8' => {
			eco => 'E90s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 e5 7. d5 Na6 8. Be3

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 e5 7. d5 Na6 8. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/3Pp3/2P1P3/2N2N1P/PP3PP1/R1BQKB1R w KQ - 1 8' => {
			eco => 'E90s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 e5 7. d5 Na6

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 e5 7. d5 Na6'),
			moves => {
				'c1e3' => 'r1bq1rk1/ppp2pbp/n2p1np1/3Pp3/2P1P3/2N1BN1P/PP3PP1/R2QKB1R b KQ - 2 8',
				'c1g5' => 'r1bq1rk1/ppp2pbp/n2p1np1/3Pp1B1/2P1P3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 8',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/4p1B1/2PPP3/2N5/PP1QBPPP/R3K1NR w KQ e6 0 8' => {
			eco => 'E73s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Na6 7. Qd2 e5

			variation => N__('King\'s Indian: Averbakh, 6... Na6 7. Qd2 e5'),
			moves => {
				'd4d5' => 'r1bq1rk1/ppp2pbp/n2p1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR b KQ - 0 8',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/4p1B1/2PPP3/2NB3P/PP3PP1/R2QK1NR w KQ e6 0 8' => {
			eco => 'E71l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. h3 O-O 6. Bg5 Na6 7. Bd3 e5

			variation => N__('King\'s Indian: Makagonov, 5...O-O 6. Bg5 Na6 7. Bd3 e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8' => {
			eco => 'E94l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Be3

			variation => N__('King\'s Indian: Glek, 8. Be3'),
			moves => {
				'f6g4' => 'r1bq1rk1/ppp2pbp/n2p2p1/4p3/2PPP1n1/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E94i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6

			variation => N__('King\'s Indian: Glek Variation'),
			moves => {
				'c1e3' => 'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8',
				'f1e1' => 'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b - - 3 8',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b - - 3 8' => {
			eco => 'E94j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Re1

			variation => N__('King\'s Indian: Glek, 8. Re1'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp3pbp/n1pp1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p1np1/4p3/2PPPP2/2NB1N2/PP4PP/R1BQK2R w KQ e6 0 8' => {
			eco => 'E76j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f4 O-O 6. Nf3 Na6 7. Bd3 e5

			variation => N__('King\'s Indian: Four Pawns Attack, 6. Nf3 Na6 7. Bd3 e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2pbp/n2p2p1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 b - - 5 9' => {
			eco => 'E94m',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Glek, 8. Be3 Ng4'),
			moves => {
				'd8e8' => 'r1b1qrk1/ppp2pbp/n2p2p1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 w - - 6 10',
			},
		},
		'r1bq1rk1/ppp2pbp/n2p2p1/4p3/2PPP1n1/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9' => {
			eco => 'E94m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Na6 8. Be3 Ng4

			variation => N__('King\'s Indian: Glek, 8. Be3 Ng4'),
			moves => {
				'e3g5' => 'r1bq1rk1/ppp2pbp/n2p2p1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 b - - 5 9',
			},
		},
		'r1bq1rk1/ppp2pp1/2np1n1p/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2Q1RK1 w - - 0 9' => {
			eco => 'C49m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3 8. bxc3 h6

			variation => N__('Four Knights: Symmetrical, 7. Bg5 Bxc3 8.bxc3 h6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2n1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 b - - 3 7' => {
			eco => 'E51',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6 7. O-O

			variation => N__('Nimzo-Indian: 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6 7.O-O'),
			moves => {
				'a7a6' => 'r1bq1rk1/1pp2ppp/p1n1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 0 8',
				'd5c4' => 'r1bq1rk1/ppp2ppp/2n1pn2/8/1bpP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/ppp2ppp/2n1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQK2R w KQ - 2 7' => {
			eco => 'E51',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6

			variation => N__('Nimzo-Indian: 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6'),
			moves => {
				'e1g1' => 'r1bq1rk1/ppp2ppp/2n1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 b - - 3 7',
			},
		},
		'r1bq1rk1/ppp2ppp/2n1pn2/8/1bpP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E51',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6 7. O-O dxc4

			variation => N__('Nimzo-Indian: 4. e3 O-O 5. Nf3 d5 6. Bd3 Nc6 7.O-O dxc4'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2n2n2/1B1pp3/4P3/2PP1N2/P1P2PPP/R1BQ1RK1 w - d6 0 8' => {
			eco => 'C49f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 Bxc3 7. bxc3 d5

			variation => N__('Four Knights: Svenonius Variation'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p1B1/1b2P3/2NP1N2/PPP2PPP/R2Q1RK1 b - - 1 7' => {
			eco => 'C49i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5

			variation => N__('Four Knights: Symmetrical, 7. Bg5'),
			moves => {
				'b4c3' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p1B1/4P3/2bP1N2/PPP2PPP/R2Q1RK1 w - - 0 8',
				'c6e7' => 'r1bq1rk1/ppp1nppp/3p1n2/1B2p1B1/1b2P3/2NP1N2/PPP2PPP/R2Q1RK1 w - - 2 8',
				'c8e6' => 'r2q1rk1/ppp2ppp/2npbn2/1B2p1B1/1b2P3/2NP1N2/PPP2PPP/R2Q1RK1 w - - 2 8',
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2Q1RK1 b - - 0 8' => {
			eco => 'C49l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3 8. bxc3

			variation => N__('Four Knights: Symmetrical, 7. Bg5 Bxc3'),
			moves => {
				'd8e7' => 'r1b2rk1/ppp1qppp/2np1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2Q1RK1 w - - 1 9',
				'h7h6' => 'r1bq1rk1/ppp2pp1/2np1n1p/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2Q1RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p1B1/4P3/2bP1N2/PPP2PPP/R2Q1RK1 w - - 0 8' => {
			eco => 'C49l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Bg5 Bxc3

			variation => N__('Four Knights: Symmetrical, 7. Bg5 Bxc3'),
			moves => {
				'b2c3' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p1B1/4P3/2PP1N2/P1P2PPP/R2Q1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/1b2P3/2NP1N2/PPP2PPP/R1BQ1RK1 w - - 0 7' => {
			eco => 'C49g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6

			variation => N__('Four Knights: Symmetrical Variation'),
			moves => {
				'c1g5' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p1B1/1b2P3/2NP1N2/PPP2PPP/R2Q1RK1 b - - 1 7',
				'c3e2' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/1b2P3/3P1N2/PPP1NPPP/R1BQ1RK1 b - - 1 7',
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/1b2P3/3P1N2/PPP1NPPP/R1BQ1RK1 b - - 1 7' => {
			eco => 'C49h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 d6 7. Ne2

			variation => N__('Four Knights: Symmetrical, Maroczy System'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/4P3/2PP1N2/P1P2PPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'C49f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Double Bishops, 6. d3 Bxc3'),
			moves => {
				'f1e1' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/4P3/2PP1N2/P1P2PPP/R1BQR1K1 b - - 1 8',
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/4P3/2PP1N2/P1P2PPP/R1BQR1K1 b - - 1 8' => {
			eco => 'C49f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 Bxc3 7. bxc3 d6 8. Re1

			variation => N__('Four Knights: Janowski Variation'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/2b1p3/2B1P3/2PP1N2/PP3PPP/RNBQ1RK1 w - - 2 7' => {
			eco => 'C54i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Nf6 5. d3 d6 6. O-O O-O

			variation => N__('Giuoco Pianissimo: 5. d3 d6 6.O-O O-O'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/2b1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w - - 2 7' => {
			eco => 'A29',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2 d6 6. O-O O-O

			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
				'd2d3' => 'r1bq1rk1/ppp2ppp/2np1n2/2b1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b - - 0 7',
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/2b1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b - - 0 7' => {
			eco => 'A29',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2 d6 6. O-O O-O 7. d3

			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp2ppp/2nppn2/8/1bPP4/2N2N2/PPQBPPPP/R3KB1R w KQ - 2 7' => {
			eco => 'E33',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 Nc6 5. Nf3 d6 6. Bd2 O-O

			variation => N__('Nimzo-Indian: Classical, Milner-Barry (Zurich), 6. Bd2 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/ppp3bp/2np1pp1/4p1B1/2PPP1n1/2N2N2/PP2BPPP/R2Q1RK1 w - - 0 10' => {
			eco => 'E97c',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, 8. Be3 Ng4'),
			moves => {
				'g5c1' => 'r1bq1rk1/ppp3bp/2np1pp1/4p3/2PPP1n1/2N2N2/PP2BPPP/R1BQ1RK1 b - - 1 10',
			},
		},
		'r1bq1rk1/ppp3bp/2np1pp1/4p3/2PPP1n1/2N2N2/PP2BPPP/R1BQ1RK1 b - - 1 10' => {
			eco => 'E97c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. Be3 Ng4 9. Bg5 f6 10. Bc1

			variation => N__('King\'s Indian: Mar del Plata, 8. Be3 Ng4'),
			moves => {
			},
		},
		'r1bq1rk1/ppp3bp/2np1pp1/4p3/2PPP1nB/2N2N2/PP2BPPP/R2QK2R w KQ - 2 10' => {
			eco => 'E92m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. Be3 Ng4 8. Bg5 f6 9. Bh4 Nc6

			variation => N__('King\'s Indian: Gligoric-Taimanov, Main Line'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pb1/3p1n1p/3Pp1p1/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ - 0 10' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5 h6 9. Bh4 g5

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6 9. Bh4 g5'),
			moves => {
				'h4g3' => 'r1bq1rk1/pppn1pb1/3p1n1p/3Pp1p1/2P1P3/2N2NB1/PP2BPPP/R2QK2R b KQ - 1 10',
			},
		},
		'r1bq1rk1/pppn1pb1/3p1n1p/3Pp1p1/2P1P3/2N2NB1/PP2BPPP/R2QK2R b KQ - 1 10' => {
			eco => 'E93',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6 9. Bh4 g5'),
			moves => {
				'f6h5' => 'r1bq1rk1/pppn1pb1/3p3p/3Pp1pn/2P1P3/2N2NB1/PP2BPPP/R2QK2R w KQ - 2 11',
			},
		},
		'r1bq1rk1/pppn1pb1/3p1npp/3Pp1B1/2P1P3/2N2N2/PP2BPPP/R2QK2R w KQ - 0 9' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5 h6

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pppn1pb1/3p1npp/3Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R b KQ - 1 9',
			},
		},
		'r1bq1rk1/pppn1pb1/3p1npp/3Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R b KQ - 1 9' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5 h6 9. Bh4

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6 9. Bh4'),
			moves => {
				'a7a5' => 'r1bq1rk1/1ppn1pb1/3p1npp/p2Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ a6 0 10',
				'a7a6' => 'r1bq1rk1/1ppn1pb1/p2p1npp/3Pp3/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ - 0 10',
				'g6g5' => 'r1bq1rk1/pppn1pb1/3p1n1p/3Pp1p1/2P1P2B/2N2N2/PP2BPPP/R2QK2R w KQ - 0 10',
			},
		},
		'r1bq1rk1/pppn1pb1/3p1npp/3Pp3/2P1P3/2N1B3/PP2BPPP/R2QK1NR w KQ - 1 9' => {
			eco => 'E73z',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 h6 7. Be3 e5 8. d5 Nbd7

			variation => N__('King\'s Indian: Averbakh, 6... h6 7. Be3 e5 8. d5 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pb1/3p1npp/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9' => {
			eco => 'E95c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 h6

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 h6'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pb1/3p3p/3Pp1pn/2P1P2P/2N2NB1/PP2BPP1/R2QK2R b KQ h3 0 11' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5 h6 9. Bh4 g5 10. Bg3 Nh5 11. h4

			variation => N__('King\'s Indian: Petrosian, Keres Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pb1/3p3p/3Pp1pn/2P1P3/2N2NB1/PP2BPPP/R2QK2R w KQ - 2 11' => {
			eco => 'E93',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5 h6 9. Bh4 g5'),
			moves => {
				'h2h4' => 'r1bq1rk1/pppn1pb1/3p3p/3Pp1pn/2P1P2P/2N2NB1/PP2BPP1/R2QK2R b KQ h3 0 11',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp1B1/2P1P3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 8' => {
			eco => 'E90r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 e5 7. d5 Nbd7 8. Bg5

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 e5 7. d5 Nbd7 8. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp1B1/2P1P3/2N2N2/PP2BPPP/R2QK2R b KQ - 2 8' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Bg5

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Bg5'),
			moves => {
				'h7h6' => 'r1bq1rk1/pppn1pb1/3p1npp/3Pp1B1/2P1P3/2N2N2/PP2BPPP/R2QK2R w KQ - 0 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR b KQ - 0 8' => {
			eco => 'E73o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Nbd7 7. Qd2 e5 8. d5

			variation => N__('King\'s Indian: Averbakh, 6...Nbd7 7. Qd2 e5 8. d5'),
			moves => {
				'd7c5' => 'r1bq1rk1/ppp2pbp/3p1np1/2nPp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR w KQ - 1 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N1BN1P/PP3PP1/R2QKB1R b KQ - 2 8' => {
			eco => 'E90r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 e5 7. d5 Nbd7 8. Be3

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 e5 7. d5 Nbd7 8. Be3'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N1BN2/PP2BPPP/R2QK2R b KQ - 2 8' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Be3

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Be3'),
			moves => {
				'f6g4' => 'r1bq1rk1/pppn1pbp/3p2p1/3Pp3/2P1P1n1/2N1BN2/PP2BPPP/R2QK2R w KQ - 3 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N1BP2/PP4PP/R2QKBNR w KQ - 1 8' => {
			eco => 'E87d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 Nbd7

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7. d5 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N2N1P/PP3PP1/R1BQKB1R w KQ - 1 8' => {
			eco => 'E90r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 e5 7. d5 Nbd7

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 e5 7. d5 Nbd7'),
			moves => {
				'c1e3' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N1BN1P/PP3PP1/R2QKB1R b KQ - 2 8',
				'c1g5' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp1B1/2P1P3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 8',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E94q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. d5

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. d5'),
			moves => {
				'd7c5' => 'r1bq1rk1/ppp2pbp/3p1np1/2nPp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQK2R w KQ - 1 8' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7

			variation => N__('King\'s Indian: Petrosian, Main Line'),
			moves => {
				'c1e3' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N1BN2/PP2BPPP/R2QK2R b KQ - 2 8',
				'c1g5' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp1B1/2P1P3/2N2N2/PP2BPPP/R2QK2R b KQ - 2 8',
				'd1c2' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N2N2/PPQ1BPPP/R1B1K2R b KQ - 2 8',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N2N2/PPQ1BPPP/R1B1K2R b KQ - 2 8' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Qc2

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Qc2'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E67e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. d5

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. d5'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4P3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E67f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. dxe5

			variation => N__('King\'s Indian: Fianchetto, Classical, 8.dxe5'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p1B1/2PPP3/2N5/PP1QBPPP/R3K1NR w KQ e6 0 8' => {
			eco => 'E73o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Nbd7 7. Qd2 e5

			variation => N__('King\'s Indian: Averbakh, 6...Nbd7 7. Qd2 e5'),
			moves => {
				'd4d5' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp1B1/2P1P3/2N5/PP1QBPPP/R3K1NR b KQ - 0 8',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8' => {
			eco => 'E67g',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. b3

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - e6 0 8' => {
			eco => 'E67d',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5

			variation => N__('King\'s Indian: Fianchetto, Classical Variation'),
			moves => {
				'b2b3' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b - - 0 8',
				'd1c2' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b - - 1 8',
				'd4d5' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
				'd4e5' => 'r1bq1rk1/pppn1pbp/3p1np1/4P3/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 0 8',
				'e2e4' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 8',
				'h2h3' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b - - 1 8' => {
			eco => 'E67h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. Qc2

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. Qc2'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 w - - 0 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 b - - 0 8' => {
			eco => 'E67j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. h3

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. h3'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/4p3/2PP4/2N2NPP/PP2PPB1/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - e6 0 8' => {
			eco => 'E60q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nf3 Bg7 4. g3 O-O 5. Bg2 d6 6. O-O Nbd7 7. Qc2 e5

			variation => N__('King\'s Indian: Fianchetto without Nc3, 6...Nbd7 7. Qc2 e5'),
			moves => {
				'f1d1' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/5NP1/PPQ1PPBP/RNBR2K1 b - - 1 8',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/5NP1/PPQ1PPBP/RNBR2K1 b - - 1 8' => {
			eco => 'E60q',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Fianchetto without Nc3, 6...Nbd7 7. Qc2 e5'),
			moves => {
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/4p3/2PP4/5NP1/PPQ1PPBP/RNBR2K1 w - - 2 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8' => {
			eco => 'E94u',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Be3

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Be3'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 0 9',
				'f6g4' => 'r1bq1rk1/pppn1pbp/3p2p1/4p3/2PPP1n1/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9',
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N1BP2/PP2N1PP/R2QKB1R w KQ - 2 8' => {
			eco => 'E85f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. Nge2 Nbd7

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7.Nge2 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 2 8' => {
			eco => 'E94p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7

			variation => N__('King\'s Indian: 7.O-O Nbd7'),
			moves => {
				'c1e3' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b - - 3 8',
				'd1c2' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 b - - 3 8',
				'd4d5' => 'r1bq1rk1/pppn1pbp/3p1np1/3Pp3/2P1P3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 0 8',
				'f1e1' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b - - 3 8',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b - - 3 8' => {
			eco => 'E95a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1'),
			moves => {
				'a7a5' => 'r1bq1rk1/1ppn1pbp/3p1np1/p3p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - a6 0 9',
				'c7c6' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9',
				'e5d4' => 'r1bq1rk1/pppn1pbp/3p1np1/8/2PpP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 4 9',
				'h7h6' => 'r1bq1rk1/pppn1pb1/3p1npp/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 b - - 3 8' => {
			eco => 'E94s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Qc2

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Qc2'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w - - 4 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 8' => {
			eco => 'E68a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4'),
			moves => {
				'a7a6' => 'r1bq1rk1/1ppn1pbp/p2p1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9',
				'c7c6' => 'r1bq1rk1/pp1n1pbp/2pp1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9',
				'e5d4' => 'r1bq1rk1/pppn1pbp/3p1np1/8/2PpP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9',
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/8/2PNP3/2N3P1/PP3PBP/R1BQ1RK1 b - - 0 9' => {
			eco => 'E68c',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 exd4'),
			moves => {
				'd7c5' => 'r1bq1rk1/ppp2pbp/3p1np1/2n5/2PNP3/2N3P1/PP3PBP/R1BQ1RK1 w - - 1 10',
				'f8e8' => 'r1bqr1k1/pppn1pbp/3p1np1/8/2PNP3/2N3P1/PP3PBP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/8/2PNP3/2N5/PP2BPPP/R1BQR1K1 b - - 0 9' => {
			eco => 'E95f',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 exd4'),
			moves => {
				'd7c5' => 'r1bq1rk1/ppp2pbp/3p1np1/2n5/2PNP3/2N5/PP2BPPP/R1BQR1K1 w - - 1 10',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/8/2PpP3/2N2N2/PP2BPPP/R1BQR1K1 w - - 0 9' => {
			eco => 'E95f',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Re1 exd4

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Re1 exd4'),
			moves => {
				'f3d4' => 'r1bq1rk1/pppn1pbp/3p1np1/8/2PNP3/2N5/PP2BPPP/R1BQR1K1 b - - 0 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p1np1/8/2PpP3/2N2NP1/PP3PBP/R1BQ1RK1 w - - 0 9' => {
			eco => 'E68c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O e5 8. e4 exd4

			variation => N__('King\'s Indian: Fianchetto, Classical, 8. e4 exd4'),
			moves => {
				'f3d4' => 'r1bq1rk1/pppn1pbp/3p1np1/8/2PNP3/2N3P1/PP3PBP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pppn1pbp/3p2p1/3Pp3/2P1P1n1/2N1BN2/PP2BPPP/R2QK2R w KQ - 3 9' => {
			eco => 'E93',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. d5 Nbd7 8. Be3 Ng4

			variation => N__('King\'s Indian: Petrosian, Main Line, 8. Be3 Ng4'),
			moves => {
			},
		},
		'r1bq1rk1/pppn1pbp/3p2p1/4p3/2PPP1n1/2N1BN2/PP2BPPP/R2Q1RK1 w - - 4 9' => {
			eco => 'E94v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nbd7 8. Be3 Ng4

			variation => N__('King\'s Indian: 7.O-O Nbd7 8. Be3 Ng4'),
			moves => {
			},
		},
		'r1bq1rk1/pppn2bp/3p2n1/3Pp1p1/P1P1Pp2/2N2P2/1P2BBPP/R2QNRK1 w - - 1 14' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Be3 f4 12. Bf2 g5 13. a4 Ng6

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Be3 f4 12. Bf2 g5 13. a4 Ng6'),
			moves => {
			},
		},
		'r1bq1rk1/pppn2bp/3p2p1/3Ppp1n/2P1P3/2N1BP2/PP1Q2PP/2KR1BNR w - - 2 10' => {
			eco => 'E87m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 Nh5 8. Qd2 f5 9. O-O-O Nd7

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7. d5 Nh5 8. Qd2 f5 9.O-O-O Nd7'),
			moves => {
				'f1d3' => 'r1bq1rk1/pppn2bp/3p2p1/3Ppp1n/2P1P3/2NBBP2/PP1Q2PP/2KR2NR b - - 3 10',
			},
		},
		'r1bq1rk1/pppn2bp/3p2p1/3Ppp1n/2P1P3/2NBBP2/PP1Q2PP/2KR2NR b - - 3 10' => {
			eco => 'E87n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 e5 7. d5 Nh5 8. Qd2 f5 9. O-O-O Nd7 10. Bd3

			variation => N__('King\'s Indian: Sämisch, Orthodox, 7. d5 Nh5 8. Qd2 f5 9.O-O-O Nd7 10. Bd3'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbBpp/3p1n2/4p1N1/3PP3/2N5/PPP2PPP/R1BQK2R b KQ - 0 7' => {
			eco => 'C41s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. Ng5 O-O 7. Bxf7+

			variation => N__('Philidor: Improved Hanham, Larobok Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbpp1/4pn1p/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 h6

			variation => N__('QGD: Orthodox, 7. Rc1 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pppnbpp1/4pn1p/3p4/2PP3B/2N1PN2/PP3PPP/2RQKB1R b K - 1 8',
			},
		},
		'r1bq1rk1/pppnbpp1/4pn1p/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 h6

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 h6'),
			moves => {
				'g5h4' => 'r1bq1rk1/pppnbpp1/4pn1p/3p4/2PP3B/2N1PN2/PPQ2PPP/R3KB1R b KQ - 1 8',
			},
		},
		'r1bq1rk1/pppnbpp1/4pn1p/3p4/2PP3B/2N1PN2/PP3PPP/2RQKB1R b K - 1 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 h6 8. Bh4

			variation => N__('QGD: Orthodox, 7. Rc1 h6 8. Bh4'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbpp1/4pn1p/3p4/2PP3B/2N1PN2/PPQ2PPP/R3KB1R b KQ - 1 8' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2 h6 8. Bh4

			variation => N__('QGD: Orthodox, Rubinstein, 7. Qc2 h6 8. Bh4'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp1nbpp1/4pn1p/2pp4/2PP3B/2N1PN2/PPQ2PPP/R3KB1R w KQ c6 0 9',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p1B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQ - 1 7' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 O-O

			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 O-O'),
			moves => {
				'd1c2' => 'r1bq1rk1/pppnbppp/3p1n2/4p1B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 2 7',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p1B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 2 7' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 O-O 7. Qc2

			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 O-O 7. Qc2'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p1B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p1N1/2BPP3/2N5/PPP2PPP/R1BQK2R w KQ - 7 7' => {
			eco => 'C41s',
                        # TRANSLATORS:

			variation => N__('Philidor: Improved Hanham, 6. Ng5'),
			moves => {
				'c4f7' => 'r1bq1rk1/pppnbBpp/3p1n2/4p1N1/3PP3/2N5/PPP2PPP/R1BQK2R b KQ - 0 7',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2BPP3/2N2N2/PPP1QPPP/R1B2RK1 b - - 8 7' => {
			eco => 'C41w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. Qe2

			variation => N__('Philidor: Improved Hanham, 7. Qe2'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2BPP3/2N2N2/PPP1QPPP/R1B2RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2BPP3/2N2N2/PPP2PPP/R1BQ1RK1 w - - 7 7' => {
			eco => 'C41u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O

			variation => N__('Philidor: Improved Hanham, Main Line'),
			moves => {
				'a2a4' => 'r1bq1rk1/pppnbppp/3p1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQ1RK1 b - a3 0 7',
				'd1e2' => 'r1bq1rk1/pppnbppp/3p1n2/4p3/2BPP3/2N2N2/PPP1QPPP/R1B2RK1 b - - 8 7',
				'f1e1' => 'r1bq1rk1/pppnbppp/3p1n2/4p3/2BPP3/2N2N2/PPP2PPP/R1BQR1K1 b - - 8 7',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2BPP3/2N2N2/PPP2PPP/R1BQR1K1 b - - 8 7' => {
			eco => 'C41x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. Re1

			variation => N__('Philidor: Improved Hanham, 7. Re1'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2BPP3/2N2N2/PPP2PPP/R1BQR1K1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 4 7' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O 7. O-O

			variation => N__('Old Indian: 5. g3 Be7'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQ - 3 7' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O

			variation => N__('Old Indian: 5. g3 Be7'),
			moves => {
				'e1g1' => 'r1bq1rk1/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 4 7',
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQ - 3 7' => {
			eco => 'A55i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 O-O

			variation => N__('Old Indian: 5. e4 Be7 6. Be2 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PPP3/2N2NP1/PP3P1P/R1BQKB1R w KQ - 1 7' => {
			eco => 'A55e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 O-O

			variation => N__('Old Indian: 5. e4 Be7 6. g3 O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQ1RK1 b - a3 0 7' => {
			eco => 'C41v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nf6 4. Nc3 Nbd7 5. Bc4 Be7 6. O-O O-O 7. a4

			variation => N__('Philidor: Improved Hanham, 7. a4'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/P1BPP3/2N2N2/1PP2PPP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4p3/3n2B1/3P4/2N1PN2/PP3PPP/R2QKB1R w KQ - 0 8' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. cxd5 Nxd5

			variation => N__('QGD: Orthodox, 7.cxd5 Nxd5'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3P2B1/3P4/2N1PN2/PP3PPP/R2QKB1R b KQ - 0 7' => {
			eco => 'D60',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox Defence'),
			moves => {
				'f6d5' => 'r1bq1rk1/pppnbppp/4p3/3n2B1/3P4/2N1PN2/PP3PPP/R2QKB1R w KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/1QN1PN2/PP3PPP/R3KB1R b KQ - 4 7' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qb3

			variation => N__('QGD: Orthodox, Rauzer Variation'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/1QN1PN2/PP3PPP/R3KB1R w KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R b K - 4 7' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1

			variation => N__('QGD: Orthodox, 7. Rc1'),
			moves => {
				'a7a6' => 'r1bq1rk1/1ppnbppp/p3pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8',
				'b7b6' => 'r1bq1rk1/p1pnbppp/1p2pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8',
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8',
				'd5c4' => 'r1bq1rk1/pppnbppp/4pn2/6B1/2pP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8',
				'f8e8' => 'r1bqr1k1/pppnbppp/4pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 5 8',
				'h7h6' => 'r1bq1rk1/pppnbpp1/4pn1p/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQ - 3 7' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7

			variation => N__('QGD: Orthodox Defence'),
			moves => {
				'a1c1' => 'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2N1PN2/PP3PPP/2RQKB1R b K - 4 7',
				'c4d5' => 'r1bq1rk1/pppnbppp/4pn2/3P2B1/3P4/2N1PN2/PP3PPP/R2QKB1R b KQ - 0 7',
				'd1b3' => 'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/1QN1PN2/PP3PPP/R3KB1R b KQ - 4 7',
				'd1c2' => 'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 4 7',
				'f1d3' => 'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R b KQ - 4 7',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R b KQ - 4 7' => {
			eco => 'D61',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Qc2

			variation => N__('QGD: Orthodox, Rubinstein Variation'),
			moves => {
				'a7a6' => 'r1bq1rk1/1ppnbppp/p3pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8',
				'c7c5' => 'r1bq1rk1/pp1nbppp/4pn2/2pp2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ c6 0 8',
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8',
				'h7h6' => 'r1bq1rk1/pppnbpp1/4pn1p/3p2B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2N2N2/PPQ1PPPP/R3KB1R w KQ - 8 7' => {
			eco => 'D53n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. Nf3 O-O 6. Qc2 Nbd7

			variation => N__('QGD: 4. Bg5 Be7 5. Nf3 O-O 6. Qc2 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R b KQ - 4 7' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3

			variation => N__('QGD: Orthodox, Botvinnik Variation'),
			moves => {
				'b7b6' => 'r1bq1rk1/p1pnbppp/1p2pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R w KQ - 0 8',
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R w KQ - 0 8',
				'd5c4' => 'r1bq1rk1/pppnbppp/4pn2/6B1/2pP4/2NBPN2/PP3PPP/R2QK2R w KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP1B2/2N1PN2/PP3PPP/R2QKB1R w KQ - 1 7' => {
			eco => 'D37j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 Be7 5. Bf4 O-O 6. e3 Nbd7

			variation => N__('QGD: Classical, 5...O-O 6. e3 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b - - 0 7' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. b3

			variation => N__('Catalan: Closed, 6...Nbd7 7. b3'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 7 7' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3

			variation => N__('Catalan: Closed, 6...Nbd7 7. Nc3'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
				'd5c4' => 'r1bq1rk1/pppnbppp/4pn2/8/2pP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/3Q1NP1/PP2PPBP/RNB2RK1 b - - 7 7' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qd3

			variation => N__('Catalan: Closed, 6...Nbd7 7. Qd3'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/5NP1/PP1NPPBP/R1BQ1RK1 b - - 7 7' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nbd2

			variation => N__('Catalan: Closed, 6...Nbd7 7.Nbd2'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w - - 6 7' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7

			variation => N__('Catalan: Closed, 6...Nbd7'),
			moves => {
				'b1c3' => 'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 7 7',
				'b1d2' => 'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/5NP1/PP1NPPBP/R1BQ1RK1 b - - 7 7',
				'b2b3' => 'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b - - 0 7',
				'd1c2' => 'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 b - - 7 7',
				'd1d3' => 'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/3Q1NP1/PP2PPBP/RNB2RK1 b - - 7 7',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 b - - 7 7' => {
			eco => 'E08a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Qc2

			variation => N__('Catalan: Closed, 7. Qc2'),
			moves => {
				'b7b6' => 'r1bq1rk1/p1pnbppp/1p2pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - - 0 8',
				'c7c5' => 'r1bq1rk1/pp1nbppp/4pn2/2pp4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - c6 0 8',
				'c7c6' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/6B1/2BP4/2N1PN2/PP3PPP/2RQK2R b K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS:

			variation => N__('QGD: Orthodox, 7. Rc1 dxc4'),
			moves => {
				'a7a6' => 'r1bq1rk1/1ppnbppp/p3pn2/6B1/2BP4/2N1PN2/PP3PPP/2RQK2R w K - 0 9',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/6B1/2BP4/2N1PN2/PP3PPP/R2QK2R b KQ - 0 8' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 dxc4 8. Bxc4

			variation => N__('QGD: Orthodox, Botvinnik, 7...dxc4 8.Bxc4'),
			moves => {
				'a7a6' => 'r1bq1rk1/1ppnbppp/p3pn2/6B1/2BP4/2N1PN2/PP3PPP/R2QK2R w KQ - 0 9',
				'c7c5' => 'r1bq1rk1/pp1nbppp/4pn2/2p3B1/2BP4/2N1PN2/PP3PPP/R2QK2R w KQ c6 0 9',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/6B1/2pP4/2N1PN2/PP3PPP/2RQKB1R w K - 0 8' => {
			eco => 'D63',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Rc1 dxc4

			variation => N__('QGD: Orthodox, 7. Rc1 dxc4'),
			moves => {
				'f1c4' => 'r1bq1rk1/pppnbppp/4pn2/6B1/2BP4/2N1PN2/PP3PPP/2RQK2R b K - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/6B1/2pP4/2NBPN2/PP3PPP/R2QK2R w KQ - 0 8' => {
			eco => 'D60',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 Be7 5. e3 O-O 6. Nf3 Nbd7 7. Bd3 dxc4

			variation => N__('QGD: Orthodox, Botvinnik, 7...dxc4'),
			moves => {
				'd3c4' => 'r1bq1rk1/pppnbppp/4pn2/6B1/2BP4/2N1PN2/PP3PPP/R2QK2R b KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/8/2pP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3 dxc4

			variation => N__('Catalan: Closed, 6...Nbd7 7. Nc3 dxc4'),
			moves => {
				'e2e4' => 'r1bq1rk1/pppnbppp/4pn2/8/2pPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 8',
			},
		},
		'r1bq1rk1/pppnbppp/4pn2/8/2pPP3/2N2NP1/PP3PBP/R1BQ1RK1 b - e3 0 8' => {
			eco => 'E07',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. g3 d5 4. Bg2 Be7 5. Nf3 O-O 6. O-O Nbd7 7. Nc3 dxc4 8. e4

			variation => N__('Catalan: Closed, 6...Nbd7 7. Nc3 dxc4'),
			moves => {
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Pp3/2P1Pp2/2N1BP2/PP2B1PP/R2QNRK1 w - - 0 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Be3 f4

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Be3 f4'),
			moves => {
				'e3f2' => 'r1bq1rk1/pppnn1bp/3p2p1/3Pp3/2P1Pp2/2N2P2/PP2BBPP/R2QNRK1 b - - 1 12',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Pp3/2P1Pp2/2N2P2/PP2BBPP/R2QNRK1 b - - 1 12' => {
			eco => 'E99',
                        # TRANSLATORS:

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Be3 f4'),
			moves => {
				'g6g5' => 'r1bq1rk1/pppnn1bp/3p4/3Pp1p1/2P1Pp2/2N2P2/PP2BBPP/R2QNRK1 w - - 0 13',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Pp3/2P1Pp2/2NN1P2/PP2B1PP/R1BQ1RK1 w - - 0 12' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Nd3 f4

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Nd3 f4'),
			moves => {
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P1P1/2N2P2/PP2B2P/R1BQNRK1 b - g3 0 11' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. g4

			variation => N__('King\'s Indian: Mar del Plata, Benko Attack'),
			moves => {
				'd7f6' => 'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P1P1/2N2P2/PP2B2P/R1BQNRK1 w - - 1 12',
				'g8h8' => 'r1bq1r1k/pppnn1bp/3p2p1/3Ppp2/2P1P1P1/2N2P2/PP2B2P/R1BQNRK1 w - - 1 12',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2N1B3/PP2BPPP/R2QNRK1 w - f6 0 11' => {
			eco => 'E98k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Be3 f5

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Be3 f5'),
			moves => {
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2N1BP2/PP2B1PP/R2QNRK1 b - - 1 11' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Be3

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Be3'),
			moves => {
				'f5f4' => 'r1bq1rk1/pppnn1bp/3p2p1/3Pp3/2P1Pp2/2N1BP2/PP2B1PP/R2QNRK1 w - - 0 12',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2N2P2/PP2B1PP/R1BQNRK1 w - f6 0 11' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5'),
			moves => {
				'c1e3' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2N1BP2/PP2B1PP/R2QNRK1 b - - 1 11',
				'e1d3' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN1P2/PP2B1PP/R1BQ1RK1 b - - 1 11',
				'g2g4' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P1P1/2N2P2/PP2B2P/R1BQNRK1 b - g3 0 11',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN1P2/PP2B1PP/R1BQ1RK1 b - - 1 11' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Nd3

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Nd3'),
			moves => {
				'd7f6' => 'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P3/2NN1P2/PP2B1PP/R1BQ1RK1 w - - 2 12',
				'f5f4' => 'r1bq1rk1/pppnn1bp/3p2p1/3Pp3/2P1Pp2/2NN1P2/PP2B1PP/R1BQ1RK1 w - - 0 12',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 b - - 1 11' => {
			eco => 'E98n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Nd3 f5 11. Bd2

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Nd3 f5 11. Bd2'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp1nn1bp/3p2p1/2pPpp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 w - c6 0 12',
				'd7f6' => 'r1bq1rk1/ppp1n1bp/3p1np1/3Ppp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 w - - 2 12',
				'g8h8' => 'r1bq1r1k/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 w - - 2 12',
			},
		},
		'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN4/PP2BPPP/R1BQ1RK1 w - f6 0 11' => {
			eco => 'E98m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Nd3 f5

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Nd3 f5'),
			moves => {
				'c1d2' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN4/PP1BBPPP/R2Q1RK1 b - - 1 11',
			},
		},
		'r1bq1rk1/pppnn1bp/3p4/3Pp1p1/2P1Pp2/2N2P2/PP2BBPP/R2QNRK1 w - - 0 13' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Be3 f4 12. Bf2 g5

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Be3 f4 12. Bf2 g5'),
			moves => {
				'a2a4' => 'r1bq1rk1/pppnn1bp/3p4/3Pp1p1/P1P1Pp2/2N2P2/1P2BBPP/R2QNRK1 b - a3 0 13',
			},
		},
		'r1bq1rk1/pppnn1bp/3p4/3Pp1p1/P1P1Pp2/2N2P2/1P2BBPP/R2QNRK1 b - a3 0 13' => {
			eco => 'E99',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3 f5 11. Be3 f4 12. Bf2 g5 13. a4

			variation => N__('King\'s Indian: Mar del Plata, 10. f3 f5 11. Be3 f4 12. Bf2 g5 13. a4'),
			moves => {
				'e7g6' => 'r1bq1rk1/pppn2bp/3p2n1/3Pp1p1/P1P1Pp2/2N2P2/1P2BBPP/R2QNRK1 w - - 1 14',
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/1PP1P3/2N5/P2NBPPP/R1BQ1RK1 b - b3 0 10' => {
			eco => 'E97v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 Nd7 10. b4

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 Nd7 10. b4'),
			moves => {
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N1B3/PP2BPPP/R2QNRK1 b - - 4 10' => {
			eco => 'E98j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Be3

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Be3'),
			moves => {
				'f7f5' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2N1B3/PP2BPPP/R2QNRK1 w - f6 0 11',
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N2P2/PP2B1PP/R1BQNRK1 b - - 0 10' => {
			eco => 'E98r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. f3

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. f3'),
			moves => {
				'f7f5' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2N2P2/PP2B1PP/R1BQNRK1 w - f6 0 11',
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N5/PP1NBPPP/R1BQ1RK1 w - - 3 10' => {
			eco => 'E97v',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Nd2 Nd7

			variation => N__('King\'s Indian: Mar del Plata, 9. Nd2 Nd7'),
			moves => {
				'b2b4' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/1PP1P3/2N5/P2NBPPP/R1BQ1RK1 b - b3 0 10',
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNR1K b - - 4 10' => {
			eco => 'E98i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Kh1

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Kh1'),
			moves => {
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNRK1 w - - 3 10' => {
			eco => 'E98h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7'),
			moves => {
				'c1e3' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N1B3/PP2BPPP/R2QNRK1 b - - 4 10',
				'e1d3' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2NN4/PP2BPPP/R1BQ1RK1 b - - 4 10',
				'f2f3' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N2P2/PP2B1PP/R1BQNRK1 b - - 0 10',
				'g1h1' => 'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2N5/PP2BPPP/R1BQNR1K b - - 4 10',
			},
		},
		'r1bq1rk1/pppnnpbp/3p2p1/3Pp3/2P1P3/2NN4/PP2BPPP/R1BQ1RK1 b - - 4 10' => {
			eco => 'E98l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 e5 7. O-O Nc6 8. d5 Ne7 9. Ne1 Nd7 10. Nd3

			variation => N__('King\'s Indian: Mar del Plata, 9. Ne1 Nd7 10. Nd3'),
			moves => {
				'f7f5' => 'r1bq1rk1/pppnn1bp/3p2p1/3Ppp2/2P1P3/2NN4/PP2BPPP/R1BQ1RK1 w - f6 0 11',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/4P3/2PP4/2N2N2/PP2BPPP/R1BQK2R b KQ - 0 7' => {
			eco => 'E91r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nbd7 7. e5

			variation => N__('King\'s Indian: 6. Be2 Nbd7 7. e5'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PP4/2N1PN2/PP2BPPP/R2QK2R b KQ - 2 7' => {
			eco => 'E61y',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 O-O 5. Bg5 d6 6. e3 Nbd7 7. Be2

			variation => N__('King\'s Indian: Smyslov System, 5...O-O 6. e3 Nbd7 7. Be2'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nppbp/2pp1np1/6B1/2PP4/2N1PN2/PP2BPPP/R2QK2R w KQ - 0 8',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQ - 1 7' => {
			eco => 'E61y',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 O-O 5. Bg5 d6 6. e3 Nbd7

			variation => N__('King\'s Indian: Smyslov System, 5...O-O 6. e3 Nbd7'),
			moves => {
				'f1e2' => 'r1bq1rk1/pppnppbp/3p1np1/6B1/2PP4/2N1PN2/PP2BPPP/R2QK2R b KQ - 2 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 7' => {
			eco => 'E90l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 Nbd7 7. Bg5

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 Nbd7 7. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N2N2/PP2BPPP/R2QK2R b KQ - 5 7' => {
			eco => 'E91q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nbd7 7. Bg5

			variation => N__('King\'s Indian: 6. Be2 Nbd7 7. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N4P/PP3PP1/R2QKBNR w KQ - 3 7' => {
			eco => 'E71j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. h3 O-O 6. Bg5 Nbd7

			variation => N__('King\'s Indian: Makagonov, 5...O-O 6. Bg5 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N5/PP1QBPPP/R3K1NR b KQ - 5 7' => {
			eco => 'E73n',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Nbd7 7. Qd2

			variation => N__('King\'s Indian: Averbakh, 6...Nbd7 7. Qd2'),
			moves => {
				'e7e5' => 'r1bq1rk1/pppn1pbp/3p1np1/4p1B1/2PPP3/2N5/PP1QBPPP/R3K1NR w KQ e6 0 8',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N5/PP2BPPP/R2QK1NR w KQ - 4 7' => {
			eco => 'E73m',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Be2 O-O 6. Bg5 Nbd7

			variation => N__('King\'s Indian: Averbakh, 6...Nbd7'),
			moves => {
				'd1d2' => 'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N5/PP1QBPPP/R3K1NR b KQ - 5 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 4 7' => {
			eco => 'E67b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7 7. O-O

			variation => N__('King\'s Indian: Fianchetto with 6... Nd7 7.O-O'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nppbp/2pp1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 0 8',
				'e7e5' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w - e6 0 8',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQ - 3 7' => {
			eco => 'E67a',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. Nf3 d6 5. g3 O-O 6. Bg2 Nbd7

			variation => N__('King\'s Indian: Fianchetto with 6... Nd7'),
			moves => {
				'e1g1' => 'r1bq1rk1/pppnppbp/3p1np1/8/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b - - 4 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w - - 2 7' => {
			eco => 'E60o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nf3 Bg7 4. g3 O-O 5. Bg2 d6 6. O-O Nbd7

			variation => N__('King\'s Indian: Fianchetto without Nc3, 6...Nbd7'),
			moves => {
				'd1c2' => 'r1bq1rk1/pppnppbp/3p1np1/8/2PP4/5NP1/PPQ1PPBP/RNB2RK1 b - - 3 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PP4/5NP1/PPQ1PPBP/RNB2RK1 b - - 3 7' => {
			eco => 'E60p',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nf3 Bg7 4. g3 O-O 5. Bg2 d6 6. O-O Nbd7 7. Qc2

			variation => N__('King\'s Indian: Fianchetto without Nc3, 6...Nbd7 7. Qc2'),
			moves => {
				'e7e5' => 'r1bq1rk1/pppn1pbp/3p1np1/4p3/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w - e6 0 8',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N1BP2/PP1Q2PP/R3KBNR b KQ - 4 7' => {
			eco => 'E81r',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nbd7 7. Qd2

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 Nbd7 7. Qd2'),
			moves => {
				'c7c5' => 'r1bq1rk1/pp1nppbp/3p1np1/2p5/2PPP3/2N1BP2/PP1Q2PP/R3KBNR w KQ c6 0 8',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N1BP2/PP4PP/R2QKBNR w KQ - 3 7' => {
			eco => 'E81q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. f3 O-O 6. Be3 Nbd7

			variation => N__('King\'s Indian: Sämisch, 5...O-O 6. Be3 Nbd7'),
			moves => {
				'd1d2' => 'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N1BP2/PP1Q2PP/R3KBNR b KQ - 4 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N2N1P/PP3PP1/R1BQKB1R w KQ - 1 7' => {
			eco => 'E90l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. h3 Nbd7

			variation => N__('King\'s Indian: 5. Nf3 O-O 6. h3 Nbd7'),
			moves => {
				'c1g5' => 'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N2N1P/PP3PP1/R2QKB1R b KQ - 2 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 5 7' => {
			eco => 'E91s',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nbd7 7. O-O

			variation => N__('King\'s Indian: 6. Be2 Nbd7 7.O-O'),
			moves => {
				'c7c6' => 'r1bq1rk1/pp1nppbp/2pp1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQ - 4 7' => {
			eco => 'E91q',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. Nf3 O-O 6. Be2 Nbd7

			variation => N__('King\'s Indian: 6. Be2 Nbd7'),
			moves => {
				'c1g5' => 'r1bq1rk1/pppnppbp/3p1np1/6B1/2PPP3/2N2N2/PP2BPPP/R2QK2R b KQ - 5 7',
				'e1g1' => 'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b - - 5 7',
				'e4e5' => 'r1bq1rk1/pppnppbp/3p1np1/4P3/2PP4/2N2N2/PP2BPPP/R1BQK2R b KQ - 0 7',
			},
		},
		'r1bq1rk1/pppnppbp/3p1np1/8/2PPP3/2N3P1/PP3PBP/R1BQK1NR w KQ - 3 7' => {
			eco => 'E72',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 Bg7 4. e4 d6 5. g3 O-O 6. Bg2 Nbd7

			variation => N__('King\'s Indian: 4. e4 d6 5. g3 O-O 6. Bg2 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/5np1/3p2B1/3P4/2P1PN2/PP1N1PPP/R2QKB1R w KQ - 1 7' => {
			eco => 'D03q',
                        # TRANSLATORS: 1. d4 d5 2. Nf3 Nf6 3. Bg5 g6 4. Nbd2 Bg7 5. e3 O-O 6. c3 Nbd7

			variation => N__('Torre Attack: 3... g6 4. e3 Bg7 5.Nbd2 O-O 6. c3 Nbd7'),
			moves => {
				'f1e2' => 'r1bq1rk1/pppnppbp/5np1/3p2B1/3P4/2P1PN2/PP1NBPPP/R2QK2R b KQ - 2 7',
			},
		},
		'r1bq1rk1/pppnppbp/5np1/3p2B1/3P4/2P1PN2/PP1NBPPP/R2QK2R b KQ - 2 7' => {
			eco => 'D03r',
                        # TRANSLATORS: 1. d4 d5 2. Nf3 Nf6 3. Bg5 g6 4. Nbd2 Bg7 5. e3 O-O 6. c3 Nbd7 7. Be2

			variation => N__('Torre Attack: 3... g6 4. e3 Bg7 5.Nbd2 O-O 6. c3 Nbd7 7. Be2'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/5np1/6N1/2BP4/1QN1P3/PP3PPP/R1B1K2R b KQ - 2 8' => {
			eco => 'D95i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. Nf3 Bg7 5. e3 O-O 6. Qb3 dxc4 7. Bxc4 Nbd7 8. Ng5

			variation => N__('Grünfeld: Pachman Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppnppbp/5np1/8/2BP4/1QN1PN2/PP3PPP/R1B1K2R w KQ - 1 8' => {
			eco => 'D95i',
                        # TRANSLATORS:

			variation => N__('Grünfeld: 5. e3 O-O 6. Qb3 dxc4 7.Bxc4'),
			moves => {
				'f3g5' => 'r1bq1rk1/pppnppbp/5np1/6N1/2BP4/1QN1P3/PP3PPP/R1B1K2R b KQ - 2 8',
			},
		},
		'r1bq1rk1/pppp1ppp/1bn2n2/1B2p1B1/3PP3/2P2N2/PP3PPP/RN1Q1RK1 b - - 2 7' => {
			eco => 'C65p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Bc5 5. c3 O-O 6. d4 Bb6 7. Bg5

			variation => N__('Spanish: Berlin, Benelux Variation, 7. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/1bn2n2/1B2p3/3PP3/2P2N2/PP3PPP/RNBQ1RK1 w - - 1 7' => {
			eco => 'C65o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Bc5 5. c3 O-O 6. d4 Bb6

			variation => N__('Spanish: Berlin, Benelux Variation'),
			moves => {
				'c1g5' => 'r1bq1rk1/pppp1ppp/1bn2n2/1B2p1B1/3PP3/2P2N2/PP3PPP/RN1Q1RK1 b - - 2 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2B2n2/4p3/1b2P3/2N2N2/PPPP1PPP/R1BQ1RK1 b - - 0 6' => {
			eco => 'C49d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. Bxc6

			variation => N__('Four Knights: Nimzowitsch (Paulsen) Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1B1Np3/1b2P3/5N2/PPPP1PPP/R1BQ1RK1 b - - 9 6' => {
			eco => 'C49c',
                        # TRANSLATORS:

			variation => N__('Four Knights: 4. Bb5 Bb4 5.O-O O-O'),
			moves => {
				'f6d5' => 'r1bq1rk1/pppp1ppp/2n5/1B1np3/1b2P3/5N2/PPPP1PPP/R1BQ1RK1 w - - 0 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1B2P3/1b1P1P2/2N5/PPP3PP/R1BQ1RK1 w - - 1 10' => {
			eco => 'C48f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bc5 5. O-O O-O 6. Nxe5 Nxe5 7. d4 Bd6 8. f4 Nc6 9. e5 Bb4

			variation => N__('Four Knights: Spanish, Classical, Bardeleben Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/1b2P3/2N2N2/PPPP1PPP/R1BQ1RK1 w - - 8 6' => {
			eco => 'C49c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O

			variation => N__('Four Knights: 4. Bb5 Bb4 5.O-O O-O'),
			moves => {
				'b5c6' => 'r1bq1rk1/pppp1ppp/2B2n2/4p3/1b2P3/2N2N2/PPPP1PPP/R1BQ1RK1 b - - 0 6',
				'c3d5' => 'r1bq1rk1/pppp1ppp/2n2n2/1B1Np3/1b2P3/5N2/PPPP1PPP/R1BQ1RK1 b - - 9 6',
				'd2d3' => 'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/1b2P3/2NP1N2/PPP2PPP/R1BQ1RK1 b - - 0 6',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/1b2P3/2NP1N2/PPP2PPP/R1BQ1RK1 b - - 0 6' => {
			eco => 'C49e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3

			variation => N__('Four Knights: Double Bishops, 6. d3'),
			moves => {
				'b4c3' => 'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/4P3/2bP1N2/PPP2PPP/R1BQ1RK1 w - - 0 7',
				'd7d6' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/1b2P3/2NP1N2/PPP2PPP/R1BQ1RK1 w - - 0 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/4P3/2PP1N2/P1P2PPP/R1BQ1RK1 b - - 0 7' => {
			eco => 'C49f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Double Bishops, 6. d3 Bxc3'),
			moves => {
				'd7d5' => 'r1bq1rk1/ppp2ppp/2n2n2/1B1pp3/4P3/2PP1N2/P1P2PPP/R1BQ1RK1 w - d6 0 8',
				'd7d6' => 'r1bq1rk1/ppp2ppp/2np1n2/1B2p3/4P3/2PP1N2/P1P2PPP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/4P3/2bP1N2/PPP2PPP/R1BQ1RK1 w - - 0 7' => {
			eco => 'C49f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. d3 Bxc3

			variation => N__('Four Knights: Double Bishops, 6. d3 Bxc3'),
			moves => {
				'b2c3' => 'r1bq1rk1/pppp1ppp/2n2n2/1B2p3/4P3/2PP1N2/P1P2PPP/R1BQ1RK1 b - - 0 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1Bb1N3/4P3/2N5/PPPP1PPP/R1BQ1RK1 b - - 0 6' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'c6d4' => 'r1bq1rk1/pppp1ppp/5n2/1Bb1N3/3nP3/2N5/PPPP1PPP/R1BQ1RK1 w - - 1 7',
				'c6e5' => 'r1bq1rk1/pppp1ppp/5n2/1Bb1n3/4P3/2N5/PPPP1PPP/R1BQ1RK1 w - - 0 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1Bb1p3/3PP3/2P2N2/PP3PPP/RNBQ1RK1 b - d3 0 6' => {
			eco => 'C65n',
                        # TRANSLATORS:

			variation => N__('Spanish: Berlin, Beverwijk, 5. c3'),
			moves => {
				'c5b6' => 'r1bq1rk1/pppp1ppp/1bn2n2/1B2p3/3PP3/2P2N2/PP3PPP/RNBQ1RK1 w - - 1 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1Bb1p3/4P3/2N2N2/PPPP1PPP/R1BQ1RK1 w - - 8 6' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'f3e5' => 'r1bq1rk1/pppp1ppp/2n2n2/1Bb1N3/4P3/2N5/PPPP1PPP/R1BQ1RK1 b - - 0 6',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 w - - 1 6' => {
			eco => 'C65n',
                        # TRANSLATORS:

			variation => N__('Spanish: Berlin, Beverwijk, 5. c3'),
			moves => {
				'd2d4' => 'r1bq1rk1/pppp1ppp/2n2n2/1Bb1p3/3PP3/2P2N2/PP3PPP/RNBQ1RK1 b - d3 0 6',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/3Np3/1bP5/4PN2/PPQP1PPP/R1B1KB1R b KQ - 4 6' => {
			eco => 'A28',
                        # TRANSLATORS:

			variation => N__('English: Four Knights, 4. e3 Bb4 5. Qc2'),
			moves => {
				'f8e8' => 'r1bqr1k1/pppp1ppp/2n2n2/3Np3/1bP5/4PN2/PPQP1PPP/R1B1KB1R w KQ - 5 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N1PN2/PPQP1PPP/R1B1KB1R w KQ - 3 6' => {
			eco => 'A28',
                        # TRANSLATORS:

			variation => N__('English: Four Knights, 4. e3 Bb4 5. Qc2'),
			moves => {
				'c3d5' => 'r1bq1rk1/pppp1ppp/2n2n2/3Np3/1bP5/4PN2/PPQP1PPP/R1B1KB1R b KQ - 4 6',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPPBP/R1BQ1RK1 b - - 4 6' => {
			eco => 'A29',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Bg2 O-O 6. O-O

			variation => N__('English: Four Knights, Main Line 6.O-O'),
			moves => {
				'e5e4' => 'r1bq1rk1/pppp1ppp/2n2n2/8/1bP1p3/2N2NP1/PP1PPPBP/R1BQ1RK1 w - - 0 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPPBP/R1BQK2R w KQ - 3 6' => {
			eco => 'A29',
                        # TRANSLATORS:

			variation => N__('English: Four Knights, 4. g3 Bb4 5. Bg2'),
			moves => {
				'e1g1' => 'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPPBP/R1BQ1RK1 b - - 4 6',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/8/1b2P3/2N1Q3/PPPB1PPP/2KR1BNR b - - 8 7' => {
			eco => 'C22',
                        # TRANSLATORS:

			variation => N__('Centre Game: Berger Variation'),
			moves => {
				'f8e8' => 'r1bqr1k1/pppp1ppp/2n2n2/8/1b2P3/2N1Q3/PPPB1PPP/2KR1BNR w - - 9 8',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/8/1b2P3/2N1Q3/PPPB1PPP/R3KBNR w KQ - 7 7' => {
			eco => 'C22',
                        # TRANSLATORS:

			variation => N__('Centre Game: Berger Variation'),
			moves => {
				'e1c1' => 'r1bq1rk1/pppp1ppp/2n2n2/8/1b2P3/2N1Q3/PPPB1PPP/2KR1BNR b - - 8 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/8/1bP1p3/2N2NP1/PP1PPPBP/R1BQ1RK1 w - - 0 7' => {
			eco => 'A29',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Bg2 O-O 6. O-O e4

			variation => N__('English: Four Knights, Main Line 6.O-O e4'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/b3N3/2BPP3/2P5/P4PPP/RNBQ1RK1 b - - 0 8' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. O-O Nf6 7. d4 O-O 8. Nxe5

			variation => N__('Evans Gambit: Richardson Attack'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/b3p3/2BPP3/2P2N2/P4PPP/RNBQ1RK1 w - - 1 8' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: 5... Ba5 6.O-O Nf6'),
			moves => {
				'f3e5' => 'r1bq1rk1/pppp1ppp/2n2n2/b3N3/2BPP3/2P5/P4PPP/RNBQ1RK1 b - - 0 8',
			},
		},
		'r1bq1rk1/pppp1ppp/2n5/1B1P4/1b2p3/5N2/PPPP1PPP/R1BQ1RK1 w - - 0 8' => {
			eco => 'C49c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bb4 5. O-O O-O 6. Nd5 Nxd5 7. exd5 e4

			variation => N__('Four Knights: Gunsberg Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/2n5/1B1Pp3/1b6/5N2/PPPP1PPP/R1BQ1RK1 b - - 0 7' => {
			eco => 'C49c',
                        # TRANSLATORS:

			variation => N__('Four Knights: 4. Bb5 Bb4 5.O-O O-O'),
			moves => {
				'e5e4' => 'r1bq1rk1/pppp1ppp/2n5/1B1P4/1b2p3/5N2/PPPP1PPP/R1BQ1RK1 w - - 0 8',
			},
		},
		'r1bq1rk1/pppp1ppp/2n5/1B1np3/1b2P3/5N2/PPPP1PPP/R1BQ1RK1 w - - 0 7' => {
			eco => 'C49c',
                        # TRANSLATORS:

			variation => N__('Four Knights: 4. Bb5 Bb4 5.O-O O-O'),
			moves => {
				'e4d5' => 'r1bq1rk1/pppp1ppp/2n5/1B1Pp3/1b6/5N2/PPPP1PPP/R1BQ1RK1 b - - 0 7',
			},
		},
		'r1bq1rk1/pppp1ppp/2nb1n2/1B2P3/3P1P2/2N5/PPP3PP/R1BQ1RK1 b - - 0 9' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'd6b4' => 'r1bq1rk1/pppp1ppp/2n2n2/1B2P3/1b1P1P2/2N5/PPP3PP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bq1rk1/pppp1ppp/2nb1n2/1B6/3PPP2/2N5/PPP3PP/R1BQ1RK1 w - - 1 9' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'e4e5' => 'r1bq1rk1/pppp1ppp/2nb1n2/1B2P3/3P1P2/2N5/PPP3PP/R1BQ1RK1 b - - 0 9',
			},
		},
		'r1bq1rk1/pppp1ppp/3b1n2/1B2n3/3PP3/2N5/PPP2PPP/R1BQ1RK1 w - - 1 8' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'f2f4' => 'r1bq1rk1/pppp1ppp/3b1n2/1B2n3/3PPP2/2N5/PPP3PP/R1BQ1RK1 b - f3 0 8',
			},
		},
		'r1bq1rk1/pppp1ppp/3b1n2/1B2n3/3PPP2/2N5/PPP3PP/R1BQ1RK1 b - f3 0 8' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'e5c6' => 'r1bq1rk1/pppp1ppp/2nb1n2/1B6/3PPP2/2N5/PPP3PP/R1BQ1RK1 w - - 1 9',
			},
		},
		'r1bq1rk1/pppp1ppp/5n2/1Bb1N3/3nP3/2N5/PPPP1PPP/R1BQ1RK1 w - - 1 7' => {
			eco => 'C48f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. Bb5 Bc5 5. O-O O-O 6. Nxe5 Nd4

			variation => N__('Four Knights: Spanish, Classical, Marshall Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/5n2/1Bb1n3/3PP3/2N5/PPP2PPP/R1BQ1RK1 b - d3 0 7' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'c5d6' => 'r1bq1rk1/pppp1ppp/3b1n2/1B2n3/3PP3/2N5/PPP2PPP/R1BQ1RK1 w - - 1 8',
			},
		},
		'r1bq1rk1/pppp1ppp/5n2/1Bb1n3/4P3/2N5/PPPP1PPP/R1BQ1RK1 w - - 0 7' => {
			eco => 'C48f',
                        # TRANSLATORS:

			variation => N__('Four Knights: Spanish, Classical Defence, 5.O-O'),
			moves => {
				'd2d4' => 'r1bq1rk1/pppp1ppp/5n2/1Bb1n3/3PP3/2N5/PPP2PPP/R1BQ1RK1 b - d3 0 7',
			},
		},
		'r1bq1rk1/ppppbppp/2n2n2/4p3/2B1P3/3P1N2/PPP2PPP/RNBQ1RK1 w - - 3 6' => {
			eco => 'C55h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d3 Be7 5. O-O O-O

			variation => N__('Two Knights: 4. d3 Be7 5.O-O O-O'),
			moves => {
				'c4b3' => 'r1bq1rk1/ppppbppp/2n2n2/4p3/4P3/1B1P1N2/PPP2PPP/RNBQ1RK1 b - - 4 6',
				'f1e1' => 'r1bq1rk1/ppppbppp/2n2n2/4p3/2B1P3/3P1N2/PPP2PPP/RNBQR1K1 b - - 4 6',
			},
		},
		'r1bq1rk1/ppppbppp/2n2n2/4p3/2B1P3/3P1N2/PPP2PPP/RNBQR1K1 b - - 4 6' => {
			eco => 'C55k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d3 Be7 5. O-O O-O 6. Re1

			variation => N__('Two Knights: 4. d3 Be7 5.O-O O-O 6. Re1'),
			moves => {
			},
		},
		'r1bq1rk1/ppppbppp/2n2n2/4p3/4P3/1B1P1N2/PPP2PPP/RNBQ1RK1 b - - 4 6' => {
			eco => 'C55i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d3 Be7 5. O-O O-O 6. Bb3

			variation => N__('Two Knights: 4. d3 Be7 5.O-O O-O 6. Bb3'),
			moves => {
				'd7d6' => 'r1bq1rk1/ppp1bppp/2np1n2/4p3/4P3/1B1P1N2/PPP2PPP/RNBQ1RK1 w - - 0 7',
			},
		},
		'r1bq1rk1/ppppbppp/2n2n2/4p3/4P3/1B1P1N2/PPP2PPP/RNBQK2R w KQ - 3 6' => {
			eco => 'C55e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. d3 Be7 5. Bb3 O-O

			variation => N__('Two Knights: 4. d3 Be7 5. Bb3 O-O'),
			moves => {
			},
		},
		'r1bq2r1/ppp2k1p/1b6/nP2B3/3PP3/8/P1P2PPP/RN1QK2R b KQ - 0 11' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Lange Variation'),
			moves => {
				'd8g5' => 'r1b3r1/ppp2k1p/1b6/nP2B1q1/3PP3/8/P1P2PPP/RN1QK2R w KQ - 1 12',
			},
		},
		'r1bq2r1/ppp2kBp/1b6/nP2p3/3PP3/8/P1P2PPP/RN1QK2R w KQ - 0 11' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Lange Variation'),
			moves => {
				'g7e5' => 'r1bq2r1/ppp2k1p/1b6/nP2B3/3PP3/8/P1P2PPP/RN1QK2R b KQ - 0 11',
			},
		},
		'r1bq3r/ppp2kpp/8/3p4/1n1Pn3/1Q3N2/PP3PPP/RN2K2R w KQ d6 0 11' => {
			eco => 'C54m',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 6.cxd4 Bb4+ 7. Bd2'),
			moves => {
				'f3e5' => 'r1bq3r/ppp2kpp/8/3pN3/1n1Pn3/1Q6/PP3PPP/RN2K2R b KQ - 1 11',
			},
		},
		'r1bq3r/ppp2kpp/8/3pN3/1n1Pn3/1Q6/PP3PPP/RN2K2R b KQ - 1 11' => {
			eco => 'C54m',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 6.cxd4 Bb4+ 7. Bd2'),
			moves => {
				'f7f6' => 'r1bq3r/ppp3pp/5k2/3pN3/1n1Pn3/1Q6/PP3PPP/RN2K2R w KQ - 2 12',
			},
		},
		'r1bq3r/ppp3pp/5k2/3pN3/1n1Pn3/1Q3P2/PP4PP/RN2K2R b KQ - 0 12' => {
			eco => 'C54m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Nf6 5. d4 exd4 6. cxd4 Bb4+ 7. Bd2 Nxe4 8. Bxb4 Nxb4 9. Bxf7+ Kxf7 10. Qb3+ d5 11. Ne5+ Kf6 12. f3

			variation => N__('Giuoco Piano: Krause Variation'),
			moves => {
			},
		},
		'r1bq3r/ppp3pp/5k2/3pN3/1n1Pn3/1Q6/PP3PPP/RN2K2R w KQ - 2 12' => {
			eco => 'C54m',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 6.cxd4 Bb4+ 7. Bd2'),
			moves => {
				'f2f3' => 'r1bq3r/ppp3pp/5k2/3pN3/1n1Pn3/1Q3P2/PP4PP/RN2K2R b KQ - 0 12',
			},
		},
		'r1bq3r/pppp1kpp/8/8/1n1Pn3/1Q3N2/PP3PPP/RN2K2R b KQ - 1 10' => {
			eco => 'C54m',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 6.cxd4 Bb4+ 7. Bd2'),
			moves => {
				'd7d5' => 'r1bq3r/ppp2kpp/8/3p4/1n1Pn3/1Q3N2/PP3PPP/RN2K2R w KQ d6 0 11',
			},
		},
		'r1bq3r/pppp1kpp/8/8/1n1Pn3/5N2/PP3PPP/RN1QK2R w KQ - 0 10' => {
			eco => 'C54m',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 6.cxd4 Bb4+ 7. Bd2'),
			moves => {
				'd1b3' => 'r1bq3r/pppp1kpp/8/8/1n1Pn3/1Q3N2/PP3PPP/RN2K2R b KQ - 1 10',
			},
		},
		'r1bq3r/ppppk1pp/2n2n2/2b1p1N1/4P3/1B6/PPPP1PPP/RNBQK2R b KQ - 2 6' => {
			eco => 'C57d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 Bc5 5. Bxf7+ Ke7 6. Bb3

			variation => N__('Two Knights: Traxler, 5.Bxf7+ Ke7 6. Bb3'),
			moves => {
			},
		},
		'r1bq3r/ppppk1pp/2n2n2/2bBp1N1/4P3/8/PPPP1PPP/RNBQK2R b KQ - 2 6' => {
			eco => 'C57d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 Bc5 5. Bxf7+ Ke7 6. Bd5

			variation => N__('Two Knights: Traxler, Chigorin/Pithart Variation'),
			moves => {
			},
		},
		'r1bq3r/ppppkBpp/2n2n2/2b1p1N1/4P3/8/PPPP1PPP/RNBQK2R w KQ - 1 6' => {
			eco => 'C57d',
                        # TRANSLATORS:

			variation => N__('Two Knights: Traxler, 5.Bxf7+'),
			moves => {
				'f7b3' => 'r1bq3r/ppppk1pp/2n2n2/2b1p1N1/4P3/1B6/PPPP1PPP/RNBQK2R b KQ - 2 6',
				'f7d5' => 'r1bq3r/ppppk1pp/2n2n2/2bBp1N1/4P3/8/PPPP1PPP/RNBQK2R b KQ - 2 6',
			},
		},
		'r1bqk1nr/1p1p1ppp/p1nQ4/4p3/4P3/8/PPP2PPP/RNB1KB1R b KQkq - 0 7' => {
			eco => 'B32k',
                        # TRANSLATORS:

			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'd8f6' => 'r1b1k1nr/1p1p1ppp/p1nQ1q2/4p3/4P3/8/PPP2PPP/RNB1KB1R w KQkq - 1 8',
			},
		},
		'r1bqk1nr/1p1p1ppp/p1nb4/4p3/4P3/8/PPP2PPP/RNBQKB1R w KQkq - 0 7' => {
			eco => 'B32k',
                        # TRANSLATORS:

			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'd1d6' => 'r1bqk1nr/1p1p1ppp/p1nQ4/4p3/4P3/8/PPP2PPP/RNB1KB1R b KQkq - 0 7',
			},
		},
		'r1bqk1nr/1p1pppbp/p1n3p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 0 6' => {
			eco => 'A37b',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 a6

			variation => N__('English: Symmetrical, 5. Nf3 a6'),
			moves => {
			},
		},
		'r1bqk1nr/1p2bppp/p1np4/1N2p3/2P1P3/2N5/PP3PPP/R1BQKB1R w KQkq - 0 8' => {
			eco => 'B32v',
                        # TRANSLATORS:

			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7.N1c3'),
			moves => {
				'b5a3' => 'r1bqk1nr/1p2bppp/p1np4/4p3/2P1P3/N1N5/PP3PPP/R1BQKB1R b KQkq - 1 8',
			},
		},
		'r1bqk1nr/1p2bppp/p1np4/4p3/2P1P3/N1N5/PP3PPP/R1BQKB1R b KQkq - 1 8' => {
			eco => 'B32v',
                        # TRANSLATORS:

			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7.N1c3'),
			moves => {
				'c8e6' => 'r2qk1nr/1p2bppp/p1npb3/4p3/2P1P3/N1N5/PP3PPP/R1BQKB1R w KQkq - 2 9',
			},
		},
		'r1bqk1nr/1pp2pbp/2np2p1/p3p3/2P5/2NP2P1/PP2PPBP/1RBQK1NR w Kkq a6 0 7' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Rb1 a5

			variation => N__('English: Closed, 5. d3 d6 6. Rb1 a5'),
			moves => {
			},
		},
		'r1bqk1nr/1pp2ppp/p1pb4/4p3/3PP3/5N2/PPP2PPP/RNBQ1RK1 b kq d3 0 6' => {
			eco => 'C68l',
                        # TRANSLATORS:

			variation => N__('Spanish: Exchange, 5.O-O Bd6'),
			moves => {
				'e5d4' => 'r1bqk1nr/1pp2ppp/p1pb4/8/3pP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 0 7',
			},
		},
		'r1bqk1nr/1pp2ppp/p1pb4/4p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 2 6' => {
			eco => 'C68l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O Bd6

			variation => N__('Spanish: Exchange, 5.O-O Bd6'),
			moves => {
				'd2d4' => 'r1bqk1nr/1pp2ppp/p1pb4/4p3/3PP3/5N2/PPP2PPP/RNBQ1RK1 b kq d3 0 6',
			},
		},
		'r1bqk1nr/1pp2ppp/p1pb4/8/3pP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 0 7' => {
			eco => 'C68m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Bxc6 dxc6 5. O-O Bd6 6. d4 exd4

			variation => N__('Spanish: Exchange, 5.O-O Bd6 6. d4 exd4'),
			moves => {
			},
		},
		'r1bqk1nr/1ppp1pbp/2n3p1/p3p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR w Kkq a6 0 6' => {
			eco => 'A25',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Rb1 a5

			variation => N__('English: Closed, 5. Rb1'),
			moves => {
			},
		},
		'r1bqk1nr/1ppp1ppp/p1n5/2b1p3/B3P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 2 5' => {
			eco => 'C70',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Bc5

			variation => N__('Spanish: Classical Deferred'),
			moves => {
			},
		},
		'r1bqk1nr/1ppp1ppp/p1n5/4p3/Bb2P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 2 5' => {
			eco => 'C70',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Bb4

			variation => N__('Spanish: Alapin Deferred'),
			moves => {
			},
		},
		'r1bqk1nr/1ppp1ppp/pbn5/4p3/PPB1P3/2N2N2/2PP1PPP/R1BQK2R b KQkq - 1 6' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. a4 a6 6. Nc3

			variation => N__('Evans Gambit Declined, Showalter Variation'),
			moves => {
			},
		},
		'r1bqk1nr/1ppp1ppp/pbn5/4p3/PPB1P3/5N2/2PP1PPP/RNBQK2R w KQkq - 0 6' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, 5. a4'),
			moves => {
				'b1c3' => 'r1bqk1nr/1ppp1ppp/pbn5/4p3/PPB1P3/2N2N2/2PP1PPP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/1ppp1ppp/pbn5/8/P3P3/1N6/1PP2PPP/RNBQKB1R w KQkq - 0 7' => {
			eco => 'C45q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6'),
			moves => {
				'b1c3' => 'r1bqk1nr/1ppp1ppp/pbn5/8/P3P3/1NN5/1PP2PPP/R1BQKB1R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/1ppp1ppp/pbn5/8/P3P3/1NN5/1PP2PPP/R1BQKB1R b KQkq - 1 7' => {
			eco => 'C45q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6 7. Nc3

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6 7. Nc3'),
			moves => {
				'd8f6' => 'r1b1k1nr/1ppp1ppp/pbn2q2/8/P3P3/1NN5/1PP2PPP/R1BQKB1R w KQkq - 2 8',
				'g8f6' => 'r1bqk2r/1ppp1ppp/pbn2n2/8/P3P3/1NN5/1PP2PPP/R1BQKB1R w KQkq - 2 8',
			},
		},
		'r1bqk1nr/1pppbppp/p1n5/4p3/B3P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 2 5' => {
			eco => 'C70',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Be7

			variation => N__('Spanish: 4. Ba4 Be7'),
			moves => {
			},
		},
		'r1bqk1nr/2pp1ppp/p1n5/1pb1p3/4P3/1B3N2/PPPP1PPP/RNBQK2R w KQkq - 2 6' => {
			eco => 'C70',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 b5 5. Bb3 Bc5

			variation => N__('Spanish: Graz Variation'),
			moves => {
			},
		},
		'r1bqk1nr/p1pp1ppp/2n5/bp2p3/2BPP3/2P2N2/P4PPP/RNBQK2R w KQkq b6 0 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 b5

			variation => N__('Evans Gambit: Leonhardt Variation'),
			moves => {
			},
		},
		'r1bqk1nr/p2pppbp/2p3p1/4P3/8/2N1B3/PPP2PPP/R2QKB1R w KQkq - 1 9' => {
			eco => 'B34s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6 bxc6 8. e5 Ng8

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6 bxc6 8. e5 Ng8'),
			moves => {
			},
		},
		'r1bqk1nr/pp1nbppp/2pp4/4P3/2B1P3/5N2/PPP2PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C41g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 d6 3. d4 Nd7 4. Bc4 c6 5. O-O Be7 6. dxe5

			variation => N__('Philidor: Hanham, Steiner Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1nbppp/2pp4/4p3/2BPP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 2 6' => {
			eco => 'C41g',
                        # TRANSLATORS:

			variation => N__('Philidor: Hanham, Krause Variation'),
			moves => {
				'd4e5' => 'r1bqk1nr/pp1nbppp/2pp4/4P3/2B1P3/5N2/PPP2PPP/RNBQ1RK1 b kq - 0 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p2P2/2B1P3/2N2N2/PPPP2PP/R1BQK2R b KQkq - 0 6' => {
			eco => 'B23v',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bc4 e6 6. f5

			variation => N__('Sicilian: Closed, Grand Prix, Schofman Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2B1PP2/2N2N2/PPPP2PP/R1BQK2R w KQkq - 0 6' => {
			eco => 'B23u',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bc4 e6

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bc4 e6'),
			moves => {
				'f4f5' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p2P2/2B1P3/2N2N2/PPPP2PP/R1BQK2R b KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A36p',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6

			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'g1e2' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq - 1 6' => {
			eco => 'A36p',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'g8e7' => 'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQkq - 2 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A36m',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e6

			variation => N__('English: Symmetrical, 5. e3 e6'),
			moves => {
				'g1e2' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq - 1 6' => {
			eco => 'A36m',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e6 6. Nge2

			variation => N__('English: Symmetrical, 5. e3 e6'),
			moves => {
				'g8e7' => 'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq - 2 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1PNP1/PP1P1PBP/R1BQK2R b KQkq - 0 6' => {
			eco => 'A37k',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. e3

			variation => N__('English: Symmetrical, 5. Nf3 e6 6. e3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 1 6' => {
			eco => 'A37l',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O

			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O'),
			moves => {
				'g8e7' => 'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 2 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 0 6' => {
			eco => 'A37i',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6

			variation => N__('English: Symmetrical, 5. Nf3 e6'),
			moves => {
				'd2d3' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 0 6',
				'e1g1' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 1 6',
				'e2e3' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1PNP1/PP1P1PBP/R1BQK2R b KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 0 6' => {
			eco => 'A37j',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. d3

			variation => N__('English: Symmetrical, 5. Nf3 e6 6. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A36h',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. a3 e6

			variation => N__('English: Symmetrical, 5. a3 e6'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6

			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6'),
			moves => {
				'c1e3' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR b KQkq - 1 6',
				'f2f4' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR b KQkq f3 0 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR b KQkq - 1 6' => {
			eco => 'B24',
                        # TRANSLATORS:

			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6'),
			moves => {
				'c6d4' => 'r1bqk1nr/pp1p1pbp/4p1p1/2p5/3nP3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq - 2 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq e3 0 6' => {
			eco => 'A04',
                        # TRANSLATORS: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 e6 6. e4

			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR b KQkq f3 0 6' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6 6. f4

			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6 6. f4'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 0 6' => {
			eco => 'A04',
                        # TRANSLATORS:

			variation => N__('Reti: 1... c5'),
			moves => {
				'e2e4' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq e3 0 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/1P2P3/5N2/P1PP1PPP/RNBQR1K1 b kq b3 0 6' => {
			eco => 'B31p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 e5 6. b4

			variation => N__('Sicilian: Rossolimo, Gurgenidze Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 w kq e6 0 6' => {
			eco => 'B31i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. c3 e5

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. c3 e5'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq - 0 6' => {
			eco => 'B31q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 e5 6. c3

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 e5'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq e6 0 6' => {
			eco => 'B31o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 e5

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 e5'),
			moves => {
				'b2b4' => 'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/1P2P3/5N2/P1PP1PPP/RNBQR1K1 b kq b3 0 6',
				'c2c3' => 'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq - 0 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq e6 0 6' => {
			eco => 'A36l',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e5

			variation => N__('English: Symmetrical, 5. e3 e5 (Botvinnik Reversed)'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 1 6' => {
			eco => 'A37u',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 0 7',
				'g8e7' => 'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 2 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq e6 0 6' => {
			eco => 'A37q',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5

			variation => N__('English: Symmetrical, 5. Nf3 e5'),
			moves => {
				'a2a3' => 'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/P1N2NP1/1P1PPPBP/R1BQK2R b KQkq - 0 6',
				'd2d3' => 'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 0 6',
				'e1g1' => 'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 1 6',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 0 6' => {
			eco => 'A37s',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. d3

			variation => N__('English: Symmetrical, 5. Nf3 e5 6. d3'),
			moves => {
				'g8e7' => 'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R w KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/P1N2NP1/1P1PPPBP/R1BQK2R b KQkq - 0 6' => {
			eco => 'A37r',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. a3

			variation => N__('English: Symmetrical, 5. Nf3 e5 6. a3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/4p1p1/2p5/3nP3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq - 2 7' => {
			eco => 'B24',
                        # TRANSLATORS:

			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6'),
			moves => {
				'c3e2' => 'r1bqk1nr/pp1p1pbp/4p1p1/2p5/3nP3/3PB1P1/PPP1NPBP/R2QK1NR b KQkq - 3 7',
			},
		},
		'r1bqk1nr/pp1p1pbp/4p1p1/2p5/3nP3/3PB1P1/PPP1NPBP/R2QK1NR b KQkq - 3 7' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6 6. Be3 Nd4 7. Nce2

			variation => N__('Sicilian: Closed, Smyslov Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1ppp/1bp5/1B2p3/3PP3/2P5/PP3PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C64t',
                        # TRANSLATORS:

			variation => N__('Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6. c3 Bb6 7. d4'),
			moves => {
				'b5a4' => 'r1bqk1nr/pp1p1ppp/1bp5/4p3/B2PP3/2P5/PP3PPP/RNBQ1RK1 b kq - 1 8',
			},
		},
		'r1bqk1nr/pp1p1ppp/1bp5/4p3/B2PP3/2P5/PP3PPP/RNBQ1RK1 b kq - 1 8' => {
			eco => 'C64u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4 Bxd4 6. c3 Bb6 7. d4 c6 8. Ba4

			variation => N__('Spanish: Classical, 6. c3 Bb6 7. d4 c6 8. Ba4'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp3ppp/1bpp4/4p3/B2PP3/2P5/PP3PPP/RNBQ1RK1 w kq - 0 9',
			},
		},
		'r1bqk1nr/pp1p1ppp/2n1p3/8/1b1NP3/2N5/PPP2PPP/R1BQKB1R w KQkq - 3 6' => {
			eco => 'B45b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Bb4

			variation => N__('Sicilian: Taimanov, 5. Nc3 Bb4'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1ppp/2p5/1Bb5/3pP3/3P4/PPP2PPP/RNBQ1RK1 w kq - 0 7' => {
			eco => 'C61k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nd4 4. Nxd4 exd4 5. O-O Bc5 6. d3 c6

			variation => N__('Spanish: Bird\'s, 5.O-O Bc5 6. d3 c6'),
			moves => {
				'b5c4' => 'r1bqk1nr/pp1p1ppp/2p5/2b5/2BpP3/3P4/PPP2PPP/RNBQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk1nr/pp1p1ppp/2p5/2b5/2BpP3/3P4/PPP2PPP/RNBQ1RK1 b kq - 1 7' => {
			eco => 'C61l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nd4 4. Nxd4 exd4 5. O-O Bc5 6. d3 c6 7. Bc4

			variation => N__('Spanish: Bird\'s, 5.O-O Bc5 6. d3 c6 7. Bc4'),
			moves => {
				'd7d5' => 'r1bqk1nr/pp3ppp/2p5/2bp4/2BpP3/3P4/PPP2PPP/RNBQ1RK1 w kq d6 0 8',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 5' => {
			eco => 'B31h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. c3

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. c3'),
			moves => {
				'e7e5' => 'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 w kq e6 0 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 2 5' => {
			eco => 'B31g',
                        # TRANSLATORS:

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O'),
			moves => {
				'c2c3' => 'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 5',
				'f1e1' => 'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 b kq - 3 5',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 b kq - 3 5' => {
			eco => 'B31j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1'),
			moves => {
				'e7e5' => 'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq e6 0 6',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq - 4 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4PP2/2N2N2/PPPP2PP/R1BQK2R b KQkq - 3 5' => {
			eco => 'B23w',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5'),
			moves => {
				'c6d4' => 'r1bqk1nr/pp1pppbp/6p1/1Bp5/3nPP2/2N2N2/PPPP2PP/R1BQK2R w KQkq - 4 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2B1PP2/2N2N2/PPPP2PP/R1BQK2R b KQkq - 3 5' => {
			eco => 'B23t',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bc4

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bc4'),
			moves => {
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2B1PP2/2N2N2/PPPP2PP/R1BQK2R w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq e3 0 5' => {
			eco => 'A36n',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4

			variation => N__('English: Symmetrical, Botvinnik System'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6',
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq - 1 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/1PN3P1/P2PPPBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'A36j',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. b3

			variation => N__('English: Symmetrical, 5. b3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'A36k',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3

			variation => N__('English: Symmetrical, 5. e3'),
			moves => {
				'e7e5' => 'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq e6 0 6',
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N1PN2/PP1P1PPP/R1BQKB1R w KQkq - 1 5' => {
			eco => 'A35f',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. Nf3 g6 4. e3 Bg7

			variation => N__('English: Symmetrical, 2. Nc3 Nc6 3. Nf3 g6 4. e3 Bg7'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq - 3 5' => {
			eco => 'A37a',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3

			variation => N__('English: Symmetrical, 5. Nf3'),
			moves => {
				'a7a6' => 'r1bqk1nr/1p1pppbp/p1n3p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 0 6',
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 0 6',
				'e7e5' => 'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq e6 0 6',
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 0 6',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 4 6',
				'g8h6' => 'r1bqk2r/pp1pppbp/2n3pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 4 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq - 2 5' => {
			eco => 'A36e',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7

			variation => N__('English: Symmetrical, 3. g3 g6'),
			moves => {
				'a2a3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR b KQkq - 0 5',
				'b2b3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/1PN3P1/P2PPPBP/R1BQK1NR b KQkq - 0 5',
				'd2d3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq - 0 5',
				'e2e3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq - 0 5',
				'e2e4' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq e3 0 5',
				'g1f3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq - 3 5',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'A36f',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3

			variation => N__('English: Symmetrical, 5. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'A36g',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. a3

			variation => N__('English: Symmetrical, 5. a3'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR w KQkq - 0 6',
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2N3P1/PPPP1PBP/R1BQK1NR w KQkq - 2 5' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7

			variation => N__('Sicilian: Closed, 3. g3 g6'),
			moves => {
				'd2d3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR b KQkq - 0 5',
				'f2f4' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/4PP2/2N3P1/PPPP2BP/R1BQK1NR b KQkq f3 0 5',
				'g1e2' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2N3P1/PPPPNPBP/R1BQK2R b KQkq - 3 5',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2N3P1/PPPPNPBP/R1BQK2R b KQkq - 3 5' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nge2

			variation => N__('Sicilian: Closed, 3. g3 g6, 5.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3

			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR w KQkq - 0 6',
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4PP2/2N2N2/PPPP2PP/R1BQKB1R w KQkq - 2 5' => {
			eco => 'B23s',
                        # TRANSLATORS:

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6'),
			moves => {
				'f1b5' => 'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4PP2/2N2N2/PPPP2PP/R1BQK2R b KQkq - 3 5',
				'f1c4' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/2B1PP2/2N2N2/PPPP2PP/R1BQK2R b KQkq - 3 5',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4PP2/2N3P1/PPPP2BP/R1BQK1NR b KQkq f3 0 5' => {
			eco => 'B24',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. f4

			variation => N__('Sicilian: Closed, 3. g3 g6, 5. f4'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq - 0 5' => {
			eco => 'A04',
                        # TRANSLATORS: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3

			variation => N__('Reti: 1... c5'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 0 6',
				'e7e6' => 'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 0 6',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 1 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq - 4 5' => {
			eco => 'A04',
                        # TRANSLATORS: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6

			variation => N__('Reti: 1... c5'),
			moves => {
				'd2d3' => 'r1bqk1nr/pp1pppbp/2n3p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq - 0 5',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/2P1P3/1N6/PP3PPP/RNBQKB1R b KQkq - 2 6' => {
			eco => 'B37b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nb3

			variation => N__('Sicilian: Maroczy Bind, 6. Nb3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/2P1P3/8/PPN2PPP/RNBQKB1R b KQkq - 2 6' => {
			eco => 'B37c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/8/2P1P3/8/PPN2PPP/RNBQKB1R w KQkq - 0 7',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/8/2P1P3/8/PPN2PPP/RNBQKB1R w KQkq - 3 7',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/2PNP3/4B3/PP3PPP/RN1QKB1R b KQkq - 2 6' => {
			eco => 'B38a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3

			variation => N__('Sicilian: Maroczy Bind, 6. Be3'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/8/2PNP3/4B3/PP3PPP/RN1QKB1R w KQkq - 0 7',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/8/2PNP3/4B3/PP3PPP/RN1QKB1R w KQkq - 3 7',
				'g8h6' => 'r1bqk2r/pp1pppbp/2n3pn/8/2PNP3/4B3/PP3PPP/RN1QKB1R w KQkq - 3 7',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/2PNP3/8/PP3PPP/RNBQKB1R w KQkq - 1 6' => {
			eco => 'B37a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7

			variation => N__('Sicilian: Maroczy Bind, 5... Bg7'),
			moves => {
				'c1e3' => 'r1bqk1nr/pp1pppbp/2n3p1/8/2PNP3/4B3/PP3PPP/RN1QKB1R b KQkq - 2 6',
				'd4b3' => 'r1bqk1nr/pp1pppbp/2n3p1/8/2P1P3/1N6/PP3PPP/RNBQKB1R b KQkq - 2 6',
				'd4c2' => 'r1bqk1nr/pp1pppbp/2n3p1/8/2P1P3/8/PPN2PPP/RNBQKB1R b KQkq - 2 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 3 6' => {
			eco => 'B34m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3'),
			moves => {
				'd7d6' => 'r1bqk1nr/pp2ppbp/2np2p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 0 7',
				'g8f6' => 'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 4 7',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq - 2 6' => {
			eco => 'B34j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 5... Bg7'),
			moves => {
				'c1e3' => 'r1bqk1nr/pp1pppbp/2n3p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 3 6',
				'd4b3' => 'r1bqk1nr/pp1pppbp/2n3p1/8/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq - 3 6',
				'd4e2' => 'r1bqk1nr/pp1pppbp/2n3p1/8/4P3/2N5/PPP1NPPP/R1BQKB1R b KQkq - 3 6',
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq - 3 6' => {
			eco => 'B34l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Nb3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Nb3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/4P3/2N5/PPP1NPPP/R1BQKB1R b KQkq - 3 6' => {
			eco => 'B34k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Nde2

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6.Nde2'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/6p1/1Bp5/3nPP2/2N2N2/PPPP2PP/R1BQ1RK1 b kq - 5 6' => {
			eco => 'B23y',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5 Nd4 6. O-O

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5 Nd4 6.O-O'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/6p1/1Bp5/3nPP2/2N2N2/PPPP2PP/R1BQK2R w KQkq - 4 6' => {
			eco => 'B23w',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5 Nd4

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5 Nd4'),
			moves => {
				'b5d3' => 'r1bqk1nr/pp1pppbp/6p1/2p5/3nPP2/2NB1N2/PPPP2PP/R1BQK2R b KQkq - 5 6',
				'e1g1' => 'r1bqk1nr/pp1pppbp/6p1/1Bp5/3nPP2/2N2N2/PPPP2PP/R1BQ1RK1 b kq - 5 6',
			},
		},
		'r1bqk1nr/pp1pppbp/6p1/2p5/3nPP2/2NB1N2/PPPP2PP/R1BQK2R b KQkq - 5 6' => {
			eco => 'B23x',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5 Nd4 6. Bd3

			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5 Nd4 6. Bd3'),
			moves => {
			},
		},
		'r1bqk1nr/pp2bppp/2n5/2pp2B1/3P4/2N2N2/PP2PPPP/R2QKB1R w KQkq - 4 7' => {
			eco => 'D32q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. Bg5 Be7

			variation => N__('QGD Tarrasch: 5. Nf3 Nc6 6. Bg5 Be7'),
			moves => {
			},
		},
		'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/2N5/PP3PPP/R1BQKB1R b KQkq - 2 7' => {
			eco => 'B32v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3

			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7.N1c3'),
			moves => {
				'a7a6' => 'r1bqk1nr/1p2bppp/p1np4/1N2p3/2P1P3/2N5/PP3PPP/R1BQKB1R w KQkq - 0 8',
			},
		},
		'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/8/PP2BPPP/RNBQK2R b KQkq - 2 7' => {
			eco => 'B32u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. Be2

			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7. Be2'),
			moves => {
			},
		},
		'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/8/PP3PPP/RNBQKB1R w KQkq - 1 7' => {
			eco => 'B32t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7

			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7'),
			moves => {
				'b1c3' => 'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/2N5/PP3PPP/R1BQKB1R b KQkq - 2 7',
				'f1e2' => 'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/8/PP2BPPP/RNBQK2R b KQkq - 2 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A36s',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6

			variation => N__('English: Symmetrical, Botvinnik, 5... d6'),
			moves => {
				'g1e2' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq - 1 6' => {
			eco => 'A36t',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2

			variation => N__('English: Symmetrical, Botvinnik, 5... d6 6.Nge2'),
			moves => {
				'g8f6' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQkq - 2 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 1 6' => {
			eco => 'A37g',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. O-O

			variation => N__('English: Symmetrical, 5. Nf3 d6 6.O-O'),
			moves => {
				'g8h6' => 'r1bqk2r/pp2ppbp/2np2pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 2 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 0 6' => {
			eco => 'A37e',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6

			variation => N__('English: Symmetrical, 5. Nf3 d6'),
			moves => {
				'd2d3' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 0 6',
				'e1g1' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 1 6',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 0 6' => {
			eco => 'A37f',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. d3

			variation => N__('English: Symmetrical, 5. Nf3 d6 6. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A36i',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. a3 d6

			variation => N__('English: Symmetrical, 5. a3 d6'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NP2P1/PPP1NPBP/R1BQK2R b KQkq - 1 6' => {
			eco => 'B25b',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nge2

			variation => N__('Sicilian: Closed, 3. g3, 5. d3 d6 6.Nge2'),
			moves => {
				'e7e5' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NP2P1/PPP1NPBP/R1BQK2R w KQkq e6 0 7',
				'e7e6' => 'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NP2P1/PPP1NPBP/R1BQK2R w KQkq - 0 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'B25a',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6

			variation => N__('Sicilian: Closed, 3. g3, 5. d3 d6'),
			moves => {
				'c1e3' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR b KQkq - 1 6',
				'f2f4' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR b KQkq f3 0 6',
				'g1e2' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NP2P1/PPP1NPBP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR b KQkq - 1 6' => {
			eco => 'B26a',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3

			variation => N__('Sicilian: Closed, 6. Be3'),
			moves => {
				'a8b8' => '1rbqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQk - 2 7',
				'e7e5' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq e6 0 7',
				'e7e6' => 'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq - 0 7',
				'g8f6' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq - 2 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq e3 0 6' => {
			eco => 'A04',
                        # TRANSLATORS: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 d6 6. e4

			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR b KQkq f3 0 6' => {
			eco => 'B25e',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4

			variation => N__('Sicilian: Closed, 6. f4'),
			moves => {
				'a8b8' => '1rbqk1nr/pp2ppbp/2np2p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQk - 1 7',
				'e7e5' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq e6 0 7',
				'e7e6' => 'r1bqk1nr/pp3pbp/2npp1p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq - 0 7',
				'g8f6' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 0 6' => {
			eco => 'A04',
                        # TRANSLATORS:

			variation => N__('Reti: 1... c5'),
			moves => {
				'e2e4' => 'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq e3 0 6',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/2P1P3/8/PPN1BPPP/RNBQK2R b KQkq - 1 7' => {
			eco => 'B37e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 d6 7. Be2

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 d6 7. Be2'),
			moves => {
				'g8h6' => 'r1bqk2r/pp2ppbp/2np2pn/8/2P1P3/8/PPN1BPPP/RNBQK2R w KQkq - 2 8',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/2P1P3/8/PPN2PPP/RNBQKB1R w KQkq - 0 7' => {
			eco => 'B37d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 d6

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 d6'),
			moves => {
				'f1e2' => 'r1bqk1nr/pp2ppbp/2np2p1/8/2P1P3/8/PPN1BPPP/RNBQK2R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R b KQkq - 1 7' => {
			eco => 'B38c',
                        # TRANSLATORS:

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 d6'),
			moves => {
				'g8h6' => 'r1bqk2r/pp2ppbp/2np2pn/8/2PNP3/2N1B3/PP3PPP/R2QKB1R w KQkq - 2 8',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/2PNP3/2N5/PP3PPP/R1BQKB1R w KQkq - 2 7' => {
			eco => 'B36b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 d6 6. Nc3 Bg7

			variation => N__('Sicilian: Maroczy Bind, 5... d6 6. Nc3 Bg7'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/2PNP3/4B3/PP3PPP/RN1QKB1R w KQkq - 0 7' => {
			eco => 'B38c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 d6

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 d6'),
			moves => {
				'b1c3' => 'r1bqk1nr/pp2ppbp/2np2p1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 0 7' => {
			eco => 'B34n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 d6

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 d6'),
			moves => {
				'd1d2' => 'r1bqk1nr/pp2ppbp/2np2p1/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R b KQkq - 1 7' => {
			eco => 'B34o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 d6 7. Qd2

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 d6 7. Qd2'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2p3p1/2p5/4P3/3P1N1P/PPP2PP1/RNBQK2R b KQkq - 0 6' => {
			eco => 'B31f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Bxc6 dxc6 5. d3 Bg7 6. h3

			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6 dxc6 5. d3 Bg7 6. h3'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2p3p1/2p5/4P3/3P1N2/PPP2PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'B31e',
                        # TRANSLATORS:

			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6 dxc6 5. d3'),
			moves => {
				'h2h3' => 'r1bqk1nr/pp2ppbp/2p3p1/2p5/4P3/3P1N1P/PPP2PP1/RNBQK2R b KQkq - 0 6',
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 0 7' => {
			eco => 'A37u',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6'),
			moves => {
				'd2d3' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq - 0 7',
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq - 0 7' => {
			eco => 'A37v',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6 7. d3

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6 7. d3'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2npbp/2np2p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w kq - 1 8',
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NP2P1/PPP1NPBP/R1BQK2R w KQkq e6 0 7' => {
			eco => 'B25c',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nge2 e5

			variation => N__('Sicilian: Closed, Botvinnik'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq e6 0 7' => {
			eco => 'B26d',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e5

			variation => N__('Sicilian: Closed, 6. Be3 e5'),
			moves => {
				'd1d2' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQkq - 1 7' => {
			eco => 'B26e',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e5 7. Qd2

			variation => N__('Sicilian: Closed, 6. Be3 e5 7. Qd2'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2npbp/2np2p1/2p1p3/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8',
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq - 1 7' => {
			eco => 'B25j',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5 7. Nf3

			variation => N__('Sicilian: Closed, 6. f4 e5 7. Nf3'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2npbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQK2R w KQkq - 2 8',
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq e6 0 7' => {
			eco => 'B25i',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5

			variation => N__('Sicilian: Closed, 6. f4 e5'),
			moves => {
				'g1f3' => 'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NP2P1/PPP1NPBP/R1BQK2R w KQkq - 0 7' => {
			eco => 'B25d',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nge2 e6

			variation => N__('Sicilian: Closed, 3. g3, 5. d3 d6 6.Nge2 e6'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq - 0 7' => {
			eco => 'B26k',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6

			variation => N__('Sicilian: Closed, 6. Be3 e6'),
			moves => {
				'd1d2' => 'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQkq - 1 7' => {
			eco => 'B26l',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2

			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2'),
			moves => {
				'a8b8' => '1rbqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQk - 2 8',
				'c6d4' => 'r1bqk1nr/pp3pbp/3pp1p1/2p5/3nP3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8',
				'd8a5' => 'r1b1k1nr/pp3pbp/2npp1p1/q1p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8',
				'g8e7' => 'r1bqk2r/pp2npbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8',
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq - 1 7' => {
			eco => 'B25m',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3

			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R w KQkq - 2 8',
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq - 0 7' => {
			eco => 'B25l',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6

			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'g1f3' => 'r1bqk1nr/pp3pbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/pp3pbp/3pp1p1/2p5/3nP3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8' => {
			eco => 'B26n',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Nd4

			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Nd4'),
			moves => {
			},
		},
		'r1bqk1nr/pp3ppp/1bpp4/4p3/B2PP3/2P5/PP3PPP/RNBQ1RK1 w kq - 0 9' => {
			eco => 'C64u',
                        # TRANSLATORS:

			variation => N__('Spanish: Classical, 6. c3 Bb6 7. d4 c6 8. Ba4'),
			moves => {
				'b1a3' => 'r1bqk1nr/pp3ppp/1bpp4/4p3/B2PP3/N1P5/PP3PPP/R1BQ1RK1 b kq - 1 9',
			},
		},
		'r1bqk1nr/pp3ppp/1bpp4/4p3/B2PP3/N1P5/PP3PPP/R1BQ1RK1 b kq - 1 9' => {
			eco => 'C64v',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4 Bxd4 6. c3 Bb6 7. d4 c6 8. Ba4 d6 9. Na3

			variation => N__('Spanish: Classical, 6. c3 Bb6 7. d4 c6 8. Ba4 d6 9. Na3'),
			moves => {
			},
		},
		'r1bqk1nr/pp3ppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQ1RK1 b kq - 1 8' => {
			eco => 'C09e',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 7.dxc5'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2nppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQ1RK1 w kq - 2 9',
			},
		},
		'r1bqk1nr/pp3ppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQK2R w KQkq - 0 8' => {
			eco => 'C09e',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 7.dxc5'),
			moves => {
				'e1g1' => 'r1bqk1nr/pp3ppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQ1RK1 b kq - 1 8',
			},
		},
		'r1bqk1nr/pp3ppp/2n5/3p4/1bpP4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq - 3 8' => {
			eco => 'D33e',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: Swedish, 7. Bg2'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2nppp/2n5/3p4/1bpP4/2N2NP1/PP2PPBP/R1BQ1RK1 w kq - 4 9',
			},
		},
		'r1bqk1nr/pp3ppp/2n5/3p4/1bpP4/2N2NP1/PP2PPBP/R1BQK2R w KQkq - 2 8' => {
			eco => 'D33e',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: Swedish, 7. Bg2'),
			moves => {
				'e1g1' => 'r1bqk1nr/pp3ppp/2n5/3p4/1bpP4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq - 3 8',
			},
		},
		'r1bqk1nr/pp3ppp/2nb4/1BPp4/8/5N2/PPPN1PPP/R1BQK2R b KQkq - 0 7' => {
			eco => 'C09e',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. dxc5

			variation => N__('French: Tarrasch, Open, 7.dxc5'),
			moves => {
				'd6c5' => 'r1bqk1nr/pp3ppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQK2R w KQkq - 0 8',
			},
		},
		'r1bqk1nr/pp3ppp/2nb4/1Bpp4/3P4/5N2/PPPN1PPP/R1BQ1RK1 b kq - 5 7' => {
			eco => 'C09d',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O

			variation => N__('French: Tarrasch, Open, 7.O-O'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2nppp/2nb4/1Bpp4/3P4/5N2/PPPN1PPP/R1BQ1RK1 w kq - 6 8',
			},
		},
		'r1bqk1nr/pp3ppp/2nb4/1Bpp4/3P4/5N2/PPPN1PPP/R1BQK2R w KQkq - 4 7' => {
			eco => 'C09d',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6

			variation => N__('French: Tarrasch, Open, 5.Ngf3 Nc6 6. Bb5 Bd6'),
			moves => {
				'd4c5' => 'r1bqk1nr/pp3ppp/2nb4/1BPp4/8/5N2/PPPN1PPP/R1BQK2R b KQkq - 0 7',
				'e1g1' => 'r1bqk1nr/pp3ppp/2nb4/1Bpp4/3P4/5N2/PPPN1PPP/R1BQ1RK1 b kq - 5 7',
			},
		},
		'r1bqk1nr/pp3ppp/2nbp3/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 1 6' => {
			eco => 'A08',
                        # TRANSLATORS:

			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'b1d2' => 'r1bqk1nr/pp3ppp/2nbp3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq - 2 6',
			},
		},
		'r1bqk1nr/pp3ppp/2nbp3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq - 2 6' => {
			eco => 'A08',
                        # TRANSLATORS:

			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'g8e7' => 'r1bqk2r/pp2nppp/2nbp3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq - 3 7',
			},
		},
		'r1bqk1nr/pp3ppp/2p5/2bp4/2BpP3/3P4/PPP2PPP/RNBQ1RK1 w kq d6 0 8' => {
			eco => 'C61m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nd4 4. Nxd4 exd4 5. O-O Bc5 6. d3 c6 7. Bc4 d5

			variation => N__('Spanish: Bird\'s, 5.O-O Bc5 6. d3 c6 7. Bc4 d5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1bppp/2np4/4p3/2BPP3/5N2/PPP2PPP/RNBQK2R w KQkq - 0 5' => {
			eco => 'C50g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Be7 4. d4 d6

			variation => N__('Italian: Hungarian Defence, 4. d4 d6'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1npbp/3p2p1/2PPp3/4P3/2N1B3/PP3PPP/R2QKBNR b KQkq - 0 7' => {
			eco => 'A42l',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5 Nce7 7. c5

			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5 Nce7 7. c5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1npbp/3p2p1/3Pp3/2P1P1P1/2N1B3/PP3P1P/R2QKBNR b KQkq g3 0 7' => {
			eco => 'A42m',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5 Nce7 7. g4

			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5 Nce7 7. g4'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1npbp/3p2p1/3Pp3/2P1P3/2N1B3/PP3PPP/R2QKBNR w KQkq - 1 7' => {
			eco => 'A42l',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5 Nce7

			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5 Nce7'),
			moves => {
				'c4c5' => 'r1bqk1nr/ppp1npbp/3p2p1/2PPp3/4P3/2N1B3/PP3PPP/R2QKBNR b KQkq - 0 7',
				'g2g4' => 'r1bqk1nr/ppp1npbp/3p2p1/3Pp3/2P1P1P1/2N1B3/PP3P1P/R2QKBNR b KQkq g3 0 7',
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/3P4/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq - 0 5' => {
			eco => 'A42h',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. d5

			variation => N__('Modern: Averbakh, Kotov, 5. d5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N1B3/PP3PPP/R2QKBNR b KQkq - 2 5' => {
			eco => 'A42i',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3

			variation => N__('Modern: Averbakh, Kotov, 5. Be3'),
			moves => {
				'e7e5' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2PPP3/2N1B3/PP3PPP/R2QKBNR w KQkq e6 0 6',
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq - 2 5' => {
			eco => 'A42g',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Nf3

			variation => N__('Modern: Averbakh, Kotov, 5. Nf3'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N5/PP2NPPP/R1BQKB1R b KQkq - 2 5' => {
			eco => 'A42g',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Nge2

			variation => N__('Modern: Averbakh, Kotov, 5.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq - 1 5' => {
			eco => 'A42g',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6

			variation => N__('Modern: Averbakh, Kotov Variation'),
			moves => {
				'c1e3' => 'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N1B3/PP3PPP/R2QKBNR b KQkq - 2 5',
				'd4d5' => 'r1bqk1nr/ppp1ppbp/2np2p1/3P4/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq - 0 5',
				'g1e2' => 'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N5/PP2NPPP/R1BQKB1R b KQkq - 2 5',
				'g1f3' => 'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq - 2 5',
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/3PPP2/2N5/PPP3PP/R1BQKBNR w KQkq - 1 5' => {
			eco => 'B06t',
                        # TRANSLATORS: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. f4 Nc6

			variation => N__('Modern: Pseudo-Austrian Attack'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/3Pp3/2P1P3/2N1B3/PP3PPP/R2QKBNR b KQkq - 0 6' => {
			eco => 'A42k',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5

			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5'),
			moves => {
				'c6e7' => 'r1bqk1nr/ppp1npbp/3p2p1/3Pp3/2P1P3/2N1B3/PP3PPP/R2QKBNR w KQkq - 1 7',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR b KQkq e3 0 6' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4

			variation => N__('English: Botvinnik System'),
			moves => {
				'g8e7' => 'r1bqk2r/ppp1npbp/2np2p1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR w KQkq - 1 7',
				'g8f6' => 'r1bqk2r/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR w KQkq - 1 7',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A25',
                        # TRANSLATORS:

			variation => N__('English: Closed, 5. e3'),
			moves => {
				'g1e2' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq - 1 6' => {
			eco => 'A25',
                        # TRANSLATORS:

			variation => N__('English: Closed, 5. e3'),
			moves => {
				'c8e6' => 'r2qk1nr/ppp2pbp/2npb1p1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq - 2 7',
				'g8h6' => 'r1bqk2r/ppp2pbp/2np2pn/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq - 2 7',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 1 6' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3

			variation => N__('English: Closed, 5. d3 d6 6. Nf3'),
			moves => {
				'g8f6' => 'r1bqk2r/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R w KQkq - 2 7',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP2P1/PP2PPBP/1RBQK1NR b Kkq - 1 6' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Rb1

			variation => N__('English: Closed, 5. d3 d6 6. Rb1'),
			moves => {
				'a7a5' => 'r1bqk1nr/1pp2pbp/2np2p1/p3p3/2P5/2NP2P1/PP2PPBP/1RBQK1NR w Kkq a6 0 7',
				'f7f5' => 'r1bqk1nr/ppp3bp/2np2p1/4pp2/2P5/2NP2P1/PP2PPBP/1RBQK1NR w Kkq f6 0 7',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR w KQkq - 0 6' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6

			variation => N__('English: Closed, 5. d3 d6'),
			moves => {
				'a1b1' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP2P1/PP2PPBP/1RBQK1NR b Kkq - 1 6',
				'e2e3' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NPP1P1/PP3PBP/R1BQK1NR b KQkq - 0 6',
				'e2e4' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR b KQkq e3 0 6',
				'g1f3' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NPP1P1/PP3PBP/R1BQK1NR b KQkq - 0 6' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e3

			variation => N__('English: Closed, 5. d3 d6 6. e3'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2PPP3/2N1B3/PP2NPPP/R2QKB1R b KQkq - 1 6' => {
			eco => 'A42j',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. Nge2

			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2PPP3/2N1B3/PP3PPP/R2QKBNR w KQkq e6 0 6' => {
			eco => 'A42j',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5

			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5'),
			moves => {
				'd4d5' => 'r1bqk1nr/ppp2pbp/2np2p1/3Pp3/2P1P3/2N1B3/PP3PPP/R2QKBNR b KQkq - 0 6',
				'g1e2' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2PPP3/2N1B3/PP2NPPP/R2QKB1R b KQkq - 1 6',
			},
		},
		'r1bqk1nr/ppp2ppp/1b1p4/n2P4/2B1P3/5N2/P4PPP/RNBQ1RK1 w kq - 1 10' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Anderssen Variation'),
			moves => {
				'c1b2' => 'r1bqk1nr/ppp2ppp/1b1p4/n2P4/2B1P3/5N2/PB3PPP/RN1Q1RK1 b kq - 2 10',
			},
		},
		'r1bqk1nr/ppp2ppp/1b1p4/n2P4/2B1P3/5N2/PB3PPP/RN1Q1RK1 b kq - 2 10' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bc5 6. d4 exd4 7. O-O d6 8. cxd4 Bb6 9. d5 Na5 10. Bb2

			variation => N__('Evans Gambit: Ulvestad Variation'),
			moves => {
				'g8e7' => 'r1bqk2r/ppp1nppp/1b1p4/n2P4/2B1P3/5N2/PB3PPP/RN1Q1RK1 w kq - 3 11',
			},
		},
		'r1bqk1nr/ppp2ppp/1b1p4/n5B1/2BPP3/2N2N2/P4PPP/R2Q1RK1 b kq - 4 10' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bc5 6. d4 exd4 7. O-O d6 8. cxd4 Bb6 9. Nc3 Na5 10. Bg5

			variation => N__('Evans Gambit: Göring Attack'),
			moves => {
				'f7f6' => 'r1bqk1nr/ppp3pp/1b1p1p2/n5B1/2BPP3/2N2N2/P4PPP/R2Q1RK1 w kq - 0 11',
			},
		},
		'r1bqk1nr/ppp2ppp/1b1p4/n7/2BPP3/2N2N2/P4PPP/R1BQ1RK1 w kq - 3 10' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Morphy Attack'),
			moves => {
				'c1g5' => 'r1bqk1nr/ppp2ppp/1b1p4/n5B1/2BPP3/2N2N2/P4PPP/R2Q1RK1 b kq - 4 10',
			},
		},
		'r1bqk1nr/ppp2ppp/1bnp4/3P4/2B1P3/5N2/P4PPP/RNBQ1RK1 b kq - 0 9' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bc5 6. d4 exd4 7. O-O d6 8. cxd4 Bb6 9. d5

			variation => N__('Evans Gambit: Anderssen Variation'),
			moves => {
				'c6a5' => 'r1bqk1nr/ppp2ppp/1b1p4/n2P4/2B1P3/5N2/P4PPP/RNBQ1RK1 w kq - 1 10',
			},
		},
		'r1bqk1nr/ppp2ppp/1bnp4/4p3/2BPP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 1 8' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. O-O d6 7. d4 Bb6

			variation => N__('Evans Gambit: Lasker\'s Defence'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/1bnp4/8/2BPP3/2N2N2/P4PPP/R1BQ1RK1 b kq - 2 9' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bc5 6. d4 exd4 7. O-O d6 8. cxd4 Bb6 9. Nc3

			variation => N__('Evans Gambit: Morphy Attack'),
			moves => {
				'c6a5' => 'r1bqk1nr/ppp2ppp/1b1p4/n7/2BPP3/2N2N2/P4PPP/R1BQ1RK1 w kq - 3 10',
				'c8g4' => 'r2qk1nr/ppp2ppp/1bnp4/8/2BPP1b1/2N2N2/P4PPP/R1BQ1RK1 w kq - 3 10',
			},
		},
		'r1bqk1nr/ppp2ppp/1bnp4/8/2BPP3/5N2/P4PPP/RNBQ1RK1 w kq - 1 9' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bc5 6. d4 exd4 7. O-O d6 8. cxd4 Bb6

			variation => N__('Evans Gambit: Normal Variation'),
			moves => {
				'b1c3' => 'r1bqk1nr/ppp2ppp/1bnp4/8/2BPP3/2N2N2/P4PPP/R1BQ1RK1 b kq - 2 9',
				'd4d5' => 'r1bqk1nr/ppp2ppp/1bnp4/3P4/2B1P3/5N2/P4PPP/RNBQ1RK1 b kq - 0 9',
			},
		},
		'r1bqk1nr/ppp2ppp/1bnp4/8/2BPP3/5N2/PP3PPP/RNBQK2R w KQkq - 1 7' => {
			eco => 'C53',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 d6 5. d4 exd4 6. cxd4 Bb6

			variation => N__('Giuoco Piano: LaBourdonnais Variation'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2n1p3/1B6/3Pp3/P1N5/1PP2PPP/R1BQK2R b KQkq - 2 7' => {
			eco => 'C15s',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. Ne2 dxe4 5. a3 Bxc3+ 6. Nxc3 Nc6 7. Bb5

			variation => N__('French: Winawer, Alekhine Gambit, Kan, 7. Bb5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2n1p3/3p4/1b1PP3/2N5/PPP1NPPP/R1BQKB1R w KQkq - 4 5' => {
			eco => 'C15p',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. Ne2 Nc6

			variation => N__('French: Winawer, Alekhine Gambit, 4... Nc6'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2n1p3/8/3Pp3/P1N5/1PP2PPP/R1BQKB1R w KQkq - 1 7' => {
			eco => 'C15r',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. Ne2 dxe4 5. a3 Bxc3+ 6. Nxc3 Nc6

			variation => N__('French: Winawer, Alekhine Gambit, Kan Variation'),
			moves => {
				'f1b5' => 'r1bqk1nr/ppp2ppp/2n1p3/1B6/3Pp3/P1N5/1PP2PPP/R1BQK2R b KQkq - 2 7',
			},
		},
		'r1bqk1nr/ppp2ppp/2n5/2bpp3/1PB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq d6 0 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 d5

			variation => N__('Evans Gambit: Hein Countergambit'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2n5/3p4/1b1P4/2NB4/PPP2PPP/R1BQK1NR w KQkq - 4 6' => {
			eco => 'C01w',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. exd5 exd5 4. Nc3 Bb4 5. Bd3 Nc6

			variation => N__('French: Exchange Winawer, 5. Bd3 Nc6'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 5' => {
			eco => 'C64o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O d6 5. c3

			variation => N__('Spanish: Classical, 4.O-O d6 5. c3'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 0 5' => {
			eco => 'C64n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O d6

			variation => N__('Spanish: Classical, 4.O-O d6'),
			moves => {
				'c2c3' => 'r1bqk1nr/ppp2ppp/2np4/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 5',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 0 5' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 4. c3'),
			moves => {
				'd2d4' => 'r1bqk1nr/ppp2ppp/2np4/2b1p3/2BPP3/2P2N2/PP3PPP/RNBQK2R b KQkq d3 0 5',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b1p3/2BPP3/2P2N2/PP3PPP/RNBQK2R b KQkq d3 0 5' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 4. c3'),
			moves => {
				'e5d4' => 'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/2P2N2/PP3PPP/RNBQK2R w KQkq - 0 6',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b5/2BPP3/5N2/P4PPP/RNBQ1RK1 b kq - 0 8' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'c5b6' => 'r1bqk1nr/ppp2ppp/1bnp4/8/2BPP3/5N2/P4PPP/RNBQ1RK1 w kq - 1 9',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b5/2BPP3/5N2/PP3PPP/RNBQK2R b KQkq - 0 6' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 4. c3'),
			moves => {
				'c5b6' => 'r1bqk1nr/ppp2ppp/1bnp4/8/2BPP3/5N2/PP3PPP/RNBQK2R w KQkq - 1 7',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'c3d4' => 'r1bqk1nr/ppp2ppp/2np4/2b5/2BPP3/5N2/P4PPP/RNBQ1RK1 b kq - 0 8',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/2P2N2/PP3PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C44y',
                        # TRANSLATORS:

			variation => N__('Scotch Gambit: 5.O-O'),
			moves => {
				'c8g4' => 'r2qk1nr/ppp2ppp/2np4/2b5/2BpP1b1/2P2N2/PP3PPP/RNBQ1RK1 w kq - 1 7',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/2P2N2/PP3PPP/RNBQK2R w KQkq - 0 6' => {
			eco => 'C53',
                        # TRANSLATORS:

			variation => N__('Giuoco Piano: 4. c3'),
			moves => {
				'c3d4' => 'r1bqk1nr/ppp2ppp/2np4/2b5/2BPP3/5N2/PP3PPP/RNBQK2R b KQkq - 0 6',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 0 6' => {
			eco => 'C44y',
                        # TRANSLATORS:

			variation => N__('Scotch Gambit: 5.O-O'),
			moves => {
				'c2c3' => 'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/2P2N2/PP3PPP/RNBQ1RK1 b kq - 0 6',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/8/1bB1P3/2N2N2/PP3PPP/R1BQ1RK1 b kq - 1 7' => {
			eco => 'C44w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. c3 dxc3 5. Nxc3 Bb4 6. Bc4 d6 7. O-O

			variation => N__('Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4 6. Bc4 d6 7.O-O'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/8/1bB1P3/2N2N2/PP3PPP/R1BQK2R w KQkq - 0 7' => {
			eco => 'C44w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. c3 dxc3 5. Nxc3 Bb4 6. Bc4 d6

			variation => N__('Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4 6. Bc4 d6'),
			moves => {
				'e1g1' => 'r1bqk1nr/ppp2ppp/2np4/8/1bB1P3/2N2N2/PP3PPP/R1BQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b3p1B1/2BPP3/2P2N2/P4PPP/RN1QK2R b KQkq - 1 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6 7. Bg5

			variation => N__('Evans Gambit: Sokolsky Variation'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQ1RK1 w kq - 0 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. O-O d6

			variation => N__('Evans Gambit: 5... Ba5 6.O-O d6'),
			moves => {
				'd2d4' => 'r1bqk1nr/ppp2ppp/2np4/b3p3/2BPP3/2P2N2/P4PPP/RNBQ1RK1 b kq d3 0 7',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b3p3/2BPP3/1QP2N2/P4PPP/RNB1K2R b KQkq - 1 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6 7. Qb3

			variation => N__('Evans Gambit: Tartakower Attack'),
			moves => {
				'd8d7' => 'r1b1k1nr/pppq1ppp/2np4/b3p3/2BPP3/1QP2N2/P4PPP/RNB1K2R w KQkq - 2 8',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b3p3/2BPP3/2P2N2/P4PPP/RNBQ1RK1 b kq d3 0 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. O-O d6 7. d4

			variation => N__('Evans Gambit: 5... Ba5 6.O-O d6 7. d4'),
			moves => {
				'a5b6' => 'r1bqk1nr/ppp2ppp/1bnp4/4p3/2BPP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 1 8',
				'c8d7' => 'r2qk1nr/pppb1ppp/2np4/b3p3/2BPP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 1 8',
				'c8g4' => 'r2qk1nr/ppp2ppp/2np4/b3p3/2BPP1b1/2P2N2/P4PPP/RNBQ1RK1 w kq - 1 8',
				'e5d4' => 'r1bqk1nr/ppp2ppp/2np4/b7/2BpP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b3p3/2BPP3/2P2N2/P4PPP/RNBQK2R w KQkq - 0 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 d6

			variation => N__('Evans Gambit: 5... Ba5 6. d4 d6'),
			moves => {
				'c1g5' => 'r1bqk1nr/ppp2ppp/2np4/b3p1B1/2BPP3/2P2N2/P4PPP/RN1QK2R b KQkq - 1 7',
				'd1b3' => 'r1bqk1nr/ppp2ppp/2np4/b3p3/2BPP3/1QP2N2/P4PPP/RNB1K2R b KQkq - 1 7',
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b7/2BpP3/1QP2N2/P4PPP/RNB2RK1 b kq - 1 8' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. O-O d6 7. d4 exd4 8. Qb3

			variation => N__('Evans Gambit: Waller Attack'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2ppp/2np4/b7/2BpP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: 5... Ba5 6.O-O d6 7. d4'),
			moves => {
				'd1b3' => 'r1bqk1nr/ppp2ppp/2np4/b7/2BpP3/1QP2N2/P4PPP/RNB2RK1 b kq - 1 8',
			},
		},
		'r1bqk1nr/ppp3bp/2np2p1/4pp2/2P5/2NP2P1/PP2PPBP/1RBQK1NR w Kkq f6 0 7' => {
			eco => 'A26',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Rb1 f5

			variation => N__('English: Closed, 5. d3 d6 6. Rb1 f5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp3pp/1b1p1p2/n5B1/2BPP3/2N2N2/P4PPP/R2Q1RK1 w kq - 0 11' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Göring Attack'),
			moves => {
				'g5e3' => 'r1bqk1nr/ppp3pp/1b1p1p2/n7/2BPP3/2N1BN2/P4PPP/R2Q1RK1 b kq - 1 11',
			},
		},
		'r1bqk1nr/ppp3pp/1b1p1p2/n7/2BPP3/2N1BN2/P4PPP/R2Q1RK1 b kq - 1 11' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bc5 6. d4 exd4 7. O-O d6 8. cxd4 Bb6 9. Nc3 Na5 10. Bg5 f6 11. Be3

			variation => N__('Evans Gambit: Steinitz Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppn1pbp/3p2p1/3Pp3/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq - 1 6' => {
			eco => 'A42u',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. d5 Nd7

			variation => N__('Modern: Averbakh, 4... e5 5. d5 Nd7'),
			moves => {
			},
		},
		'r1bqk1nr/pppn1pbp/3p2p1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R b KQkq - 3 6' => {
			eco => 'A42r',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nf3 Nd7 6. Be2

			variation => N__('Modern: Averbakh, 4... e5 5. Nf3 Nd7 6. Be2'),
			moves => {
				'g8e7' => 'r1bqk2r/pppnnpbp/3p2p1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQkq - 4 7',
			},
		},
		'r1bqk1nr/pppn1pbp/3p2p1/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R w KQkq - 2 6' => {
			eco => 'A42q',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nf3 Nd7

			variation => N__('Modern: Averbakh, 4... e5 5. Nf3 Nd7'),
			moves => {
				'f1e2' => 'r1bqk1nr/pppn1pbp/3p2p1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R b KQkq - 3 6',
			},
		},
		'r1bqk1nr/pppn1ppp/3p4/2b1p3/4PP2/2N2N2/PPPP2PP/R1BQKB1R w KQkq - 2 5' => {
			eco => 'C30l',
                        # TRANSLATORS: 1. e4 e5 2. f4 Bc5 3. Nf3 d6 4. Nc3 Nd7

			variation => N__('KGD: Classical, Hanham Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppnbppp/4p3/8/3PN3/5N2/PPP2PPP/R1BQKB1R w KQkq - 3 6' => {
			eco => 'C10i',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Be7

			variation => N__('French: Rubinstein, 5. Nf3 Be7'),
			moves => {
			},
		},
		'r1bqk1nr/pppnppbp/3p2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq - 2 5' => {
			eco => 'A42f',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nd7 5. Nf3

			variation => N__('Modern: Averbakh, 4... Nd7 5. Nf3'),
			moves => {
			},
		},
		'r1bqk1nr/pppnppbp/3p2p1/8/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq - 1 5' => {
			eco => 'A42f',
                        # TRANSLATORS: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nd7

			variation => N__('Modern: Averbakh, 4... Nd7'),
			moves => {
				'g1f3' => 'r1bqk1nr/pppnppbp/3p2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq - 2 5',
			},
		},
		'r1bqk1nr/pppp1Bpp/2n5/2b1p3/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq - 0 4' => {
			eco => 'C50h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. Bxf7+

			variation => N__('Giuoco Piano: Jerome Gambit'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq e3 0 5' => {
			eco => 'A25',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4

			variation => N__('English: Closed, 5. e4'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'A25',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3

			variation => N__('English: Closed, 5. e3'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR b Kkq - 3 5' => {
			eco => 'A25',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Rb1

			variation => N__('English: Closed, 5. Rb1'),
			moves => {
				'a7a5' => 'r1bqk1nr/1ppp1pbp/2n3p1/p3p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR w Kkq a6 0 6',
				'g8h6' => 'r1bqk2r/pppp1pbp/2n3pn/4p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR w Kkq - 4 6',
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq - 2 5' => {
			eco => 'A25',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7

			variation => N__('English: Closed'),
			moves => {
				'a1b1' => 'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR b Kkq - 3 5',
				'd2d3' => 'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq - 0 5',
				'e2e3' => 'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq - 0 5',
				'e2e4' => 'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq e3 0 5',
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq - 0 5' => {
			eco => 'A25',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3

			variation => N__('English: Closed, 5. d3'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR w KQkq - 0 6',
				'g8e7' => 'r1bqk2r/ppppnpbp/2n3p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR w KQkq - 1 6',
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 2 6' => {
			eco => 'C46l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 g6 4. d4 exd4 5. Nxd4 Bg7 6. Be3

			variation => N__('Three Knights: Steinitz, 4. d4 exd4 5.Nxd4 Bg7 6. Be3'),
			moves => {
				'g8f6' => 'r1bqk2r/pppp1pbp/2n2np1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 3 7',
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq - 1 6' => {
			eco => 'C46l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 g6 4. d4 exd4 5. Nxd4 Bg7

			variation => N__('Three Knights: Steinitz, 4. d4 exd4 5.Nxd4 Bg7'),
			moves => {
				'c1e3' => 'r1bqk1nr/pppp1pbp/2n3p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 2 6',
			},
		},
		'r1bqk1nr/pppp1ppp/1b6/1B2p3/3PP3/2P5/PP3PPP/RNBQ1RK1 b kq d3 0 7' => {
			eco => 'C64t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4 Bxd4 6. c3 Bb6 7. d4

			variation => N__('Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6. c3 Bb6 7. d4'),
			moves => {
				'c7c6' => 'r1bqk1nr/pp1p1ppp/1bp5/1B2p3/3PP3/2P5/PP3PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk1nr/pppp1ppp/1b6/1B2p3/4P3/2P5/PP1P1PPP/RNBQ1RK1 w kq - 1 7' => {
			eco => 'C64s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4 Bxd4 6. c3 Bb6

			variation => N__('Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6. c3 Bb6'),
			moves => {
				'd2d4' => 'r1bqk1nr/pppp1ppp/1b6/1B2p3/3PP3/2P5/PP3PPP/RNBQ1RK1 b kq d3 0 7',
			},
		},
		'r1bqk1nr/pppp1ppp/1b6/nP2N3/2B1P3/8/P1PP1PPP/RNBQK2R b KQkq - 0 6' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, 5. b5'),
			moves => {
				'd8g5' => 'r1b1k1nr/pppp1ppp/1b6/nP2N1q1/2B1P3/8/P1PP1PPP/RNBQK2R w KQkq - 1 7',
				'g8h6' => 'r1bqk2r/pppp1ppp/1b5n/nP2N3/2B1P3/8/P1PP1PPP/RNBQK2R w KQkq - 1 7',
			},
		},
		'r1bqk1nr/pppp1ppp/1b6/nP2p3/2B1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, 5. b5'),
			moves => {
				'f3e5' => 'r1bqk1nr/pppp1ppp/1b6/nP2N3/2B1P3/8/P1PP1PPP/RNBQK2R b KQkq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/1B2p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5' => {
			eco => 'C64c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. c3 Bb6

			variation => N__('Spanish: Classical, Charousek Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/1P2p3/2B1P3/5N2/P1PP1PPP/RNBQK2R b KQkq - 0 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. b5

			variation => N__('Evans Gambit Declined, 5. b5'),
			moves => {
				'c6a5' => 'r1bqk1nr/pppp1ppp/1b6/nP2p3/2B1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 1 6',
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/4p3/1PB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 1 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6

			variation => N__('Evans Gambit Declined'),
			moves => {
				'a2a4' => 'r1bqk1nr/pppp1ppp/1bn5/4p3/PPB1P3/5N2/2PP1PPP/RNBQK2R b KQkq a3 0 5',
				'b4b5' => 'r1bqk1nr/pppp1ppp/1bn5/1P2p3/2B1P3/5N2/P1PP1PPP/RNBQK2R b KQkq - 0 5',
				'c1b2' => 'r1bqk1nr/pppp1ppp/1bn5/4p3/1PB1P3/5N2/PBPP1PPP/RN1QK2R b KQkq - 2 5',
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/4p3/1PB1P3/5N2/PBPP1PPP/RN1QK2R b KQkq - 2 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. Bb2

			variation => N__('Evans Gambit Declined, Cordel Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/4p3/PPB1P3/5N2/2PP1PPP/RNBQK2R b KQkq a3 0 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bb6 5. a4

			variation => N__('Evans Gambit Declined, 5. a4'),
			moves => {
				'a7a6' => 'r1bqk1nr/1ppp1ppp/pbn5/4p3/PPB1P3/5N2/2PP1PPP/RNBQK2R w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/8/4P3/1N6/PPP2PPP/RNBQKB1R w KQkq - 3 6' => {
			eco => 'C45p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb6

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Nb3 Bb6'),
			moves => {
				'a2a4' => 'r1bqk1nr/pppp1ppp/1bn5/8/P3P3/1N6/1PP2PPP/RNBQKB1R b KQkq a3 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/1bn5/8/P3P3/1N6/1PP2PPP/RNBQKB1R b KQkq a3 0 6' => {
			eco => 'C45p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb6 6. a4

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Nb3 Bb6 6. a4'),
			moves => {
				'a7a6' => 'r1bqk1nr/1ppp1ppp/pbn5/8/P3P3/1N6/1PP2PPP/RNBQKB1R w KQkq - 0 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2B5/2b1p3/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq - 0 4' => {
			eco => 'C64a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. Bxc6

			variation => N__('Spanish: Classical, Exchange'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2N5/2b5/4P3/8/PPP2PPP/RNBQKB1R b KQkq - 0 5' => {
			eco => 'C45r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nxc6

			variation => N__('Scotch: 4.Nxd4 Bc5 5.Nxc6'),
			moves => {
				'd8f6' => 'r1b1k1nr/pppp1ppp/2N2q2/2b5/4P3/8/PPP2PPP/RNBQKB1R w KQkq - 1 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/1B2p3/1b2P3/2P2N2/PP1P1PPP/RNBQK2R b KQkq - 0 4' => {
			eco => 'C60i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bb4 4. c3

			variation => N__('Spanish: Alapin, 4. c3'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/1B2p3/1b2P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'C60h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bb4

			variation => N__('Spanish: Alapin Variation'),
			moves => {
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/1B2p3/1b2P3/2P2N2/PP1P1PPP/RNBQK2R b KQkq - 0 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/2N2N2/PPPP1PPP/R1BQK2R b KQkq - 5 4' => {
			eco => 'C46e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Bc5 4. Bb5

			variation => N__('Three Knights: 3... Bc5 4. Bb5'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R b KQkq - 0 4' => {
			eco => 'C64b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. c3

			variation => N__('Spanish: Classical, 4. c3'),
			moves => {
				'c5b6' => 'r1bqk1nr/pppp1ppp/1bn5/1B2p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
				'd8e7' => 'r1b1k1nr/ppppqppp/2n5/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
				'd8f6' => 'r1b1k1nr/pppp1ppp/2n2q2/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
				'f7f5' => 'r1bqk1nr/pppp2pp/2n5/1Bb1pp2/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq f6 0 5',
				'g8e7' => 'r1bqk2r/ppppnppp/2n5/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 5 4' => {
			eco => 'C64k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O

			variation => N__('Spanish: Classical, 4.O-O'),
			moves => {
				'c6d4' => 'r1bqk1nr/pppp1ppp/8/1Bb1p3/3nP3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 6 5',
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 0 5',
				'd8f6' => 'r1b1k1nr/pppp1ppp/2n2q2/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 6 5',
				'g8e7' => 'r1bqk2r/ppppnppp/2n5/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 6 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'C64a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5

			variation => N__('Spanish: Classical Defence'),
			moves => {
				'b5c6' => 'r1bqk1nr/pppp1ppp/2B5/2b1p3/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq - 0 4',
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/2P2N2/PP1P1PPP/RNBQK2R b KQkq - 0 4',
				'e1g1' => 'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 5 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1N3/4P3/2N5/PPPP1PPP/R1BQKB1R b KQkq - 0 4' => {
			eco => 'C46f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Bc5 4. Nxe5

			variation => N__('Three Knights: 3... Bc5 4.Nxe5'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1P3/4P3/8/PPP2PPP/RNBQKBNR w KQkq - 1 4' => {
			eco => 'B00u',
                        # TRANSLATORS: 1. e4 Nc6 2. d4 e5 3. dxe5 Bc5

			variation => N__('Nimzowitsch Defence: Bielefelder Gambit'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/1PB1P3/5N2/P1PP1PPP/RNBQK2R b KQkq b3 0 4' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4

			variation => N__('Evans Gambit'),
			moves => {
				'c5b4' => 'r1bqk1nr/pppp1ppp/2n5/4p3/1bB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 0 5',
				'c5b6' => 'r1bqk1nr/pppp1ppp/1bn5/4p3/1PB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 1 5',
				'c5e7' => 'r1bqk1nr/ppppbppp/2n5/4p3/1PB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 1 5',
				'd7d5' => 'r1bqk1nr/ppp2ppp/2n5/2bpp3/1PB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq d6 0 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P1Q1/2N5/PPPP1PPP/R1B1K1NR b KQkq - 5 4' => {
			eco => 'C25l',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. Bc4 Bc5 4. Qg4

			variation => N__('Vienna: 2... Nc6 3. Bc4 Bc5 4. Qg4'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2N2N2/PPPP1PPP/R1BQK2R b KQkq - 5 4' => {
			eco => 'C50i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. Nc3

			variation => N__('Giuoco Piano: Three Knights Variation'),
			moves => {
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2B1P3/2N2N2/PPPP1PPP/R1BQK2R w KQkq - 6 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2N5/PPPP1PPP/R1BQK1NR w KQkq - 4 4' => {
			eco => 'C25j',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. Bc4 Bc5

			variation => N__('Vienna: 2... Nc6 3. Bc4 Bc5'),
			moves => {
				'd1g4' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P1Q1/2N5/PPPP1PPP/R1B1K1NR b KQkq - 5 4',
				'd2d3' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2NP4/PPP2PPP/R1BQK1NR b KQkq - 0 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2NP4/PPP2PPP/R1BQK1NR b KQkq - 0 4' => {
			eco => 'C25k',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. Bc4 Bc5 4. d3

			variation => N__('Vienna: 2... Nc6 3. Bc4 Bc5 4. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'd2d4' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2BPP3/2P2N2/P4PPP/RNBQK2R b KQkq d3 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R b KQkq - 0 4' => {
			eco => 'C53',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3

			variation => N__('Giuoco Piano: 4. c3'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 0 5',
				'd8e7' => 'r1b1k1nr/ppppqppp/2n5/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq - 1 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2P5/PP1PQPPP/RNB1K1NR b KQkq - 0 4' => {
			eco => 'C23',
                        # TRANSLATORS:

			variation => N__('Bishop\'s Opening: 2... Bc5 3. Qe2'),
			moves => {
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2B1P3/2P5/PP1PQPPP/RNB1K1NR w KQkq - 1 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/3P1N2/PPP2PPP/RNBQK2R b KQkq - 0 4' => {
			eco => 'C50n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. d3

			variation => N__('Giuoco Pianissimo: 4. d3'),
			moves => {
				'f7f5' => 'r1bqk1nr/pppp2pp/2n5/2b1pp2/2B1P3/3P1N2/PPP2PPP/RNBQK2R w KQkq f6 0 5',
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2B1P3/3P1N2/PPP2PPP/RNBQK2R w KQkq - 1 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 5 4' => {
			eco => 'C50k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. O-O

			variation => N__('Giuoco Piano: 4.O-O'),
			moves => {
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2B1P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 6 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'C50h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5

			variation => N__('Giuoco Piano'),
			moves => {
				'b1c3' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2N2N2/PPPP1PPP/R1BQK2R b KQkq - 5 4',
				'b2b4' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/1PB1P3/5N2/P1PP1PPP/RNBQK2R b KQkq b3 0 4',
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2P2N2/PP1P1PPP/RNBQK2R b KQkq - 0 4',
				'c4f7' => 'r1bqk1nr/pppp1Bpp/2n5/2b1p3/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq - 0 4',
				'd2d3' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/3P1N2/PPP2PPP/RNBQK2R b KQkq - 0 4',
				'd2d4' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2BPP3/5N2/PPP2PPP/RNBQK2R b KQkq d3 0 4',
				'e1g1' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 5 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/8/PPPPQPPP/RNB1K1NR w KQkq - 4 4' => {
			eco => 'C23',
                        # TRANSLATORS:

			variation => N__('Bishop\'s Opening: 2... Bc5 3. Qe2'),
			moves => {
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2P5/PP1PQPPP/RNB1K1NR b KQkq - 0 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2BPP3/2P2N2/P4PPP/RNBQK2R b KQkq d3 0 6' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'e5d4' => 'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/2P2N2/P4PPP/RNBQK2R w KQkq - 0 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/2BPP3/5N2/PPP2PPP/RNBQK2R b KQkq d3 0 4' => {
			eco => 'C50h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. d4

			variation => N__('Giuoco Piano: Rosentreter Gambit'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/4P3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq - 4 4' => {
			eco => 'C46e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Bc5

			variation => N__('Three Knights: 3... Bc5'),
			moves => {
				'f1b5' => 'r1bqk1nr/pppp1ppp/2n5/1Bb1p3/4P3/2N2N2/PPPP1PPP/R1BQK2R b KQkq - 5 4',
				'f3e5' => 'r1bqk1nr/pppp1ppp/2n5/2b1N3/4P3/2N5/PPPP1PPP/R1BQKB1R b KQkq - 0 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1p3/4P3/2N3P1/PPPP1P1P/R1BQKBNR w KQkq - 1 4' => {
			eco => 'C25i',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nc6 3. g3 Bc5

			variation => N__('Vienna: Paulsen, 3... Bc5'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b3N1/2BpP3/8/PPP2PPP/RNBQK2R b KQkq - 3 5' => {
			eco => 'C44y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Bc4 Bc5 5. Ng5

			variation => N__('Scotch Gambit: 5. Ng5'),
			moves => {
				'g8h6' => 'r1bqk2r/pppp1ppp/2n4n/2b3N1/2BpP3/8/PPP2PPP/RNBQK2R w KQkq - 4 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/2B1P3/2Pp1N2/PP3PPP/RNBQK2R w KQkq - 0 6' => {
			eco => 'C44z',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Bc4 Bc5 5. c3 d3

			variation => N__('Scotch Gambit: 5. c3 d3'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/2P2N2/P4PPP/RNBQ1RK1 b kq - 1 7' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/2P2N2/P4PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/2P2N2/P4PPP/RNBQK2R w KQkq - 0 7' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'e1g1' => 'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/2P2N2/P4PPP/RNBQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/2P2N2/PP3PPP/RNBQK2R b KQkq - 0 5' => {
			eco => 'C44z',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Bc4 Bc5 5. c3

			variation => N__('Scotch Gambit: 5. c3'),
			moves => {
				'd4d3' => 'r1bqk1nr/pppp1ppp/2n5/2b5/2B1P3/2Pp1N2/PP3PPP/RNBQK2R w KQkq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/5N2/PPP2PPP/RNBQ1RK1 b kq - 3 5' => {
			eco => 'C44y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Bc4 Bc5 5. O-O

			variation => N__('Scotch Gambit: 5.O-O'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/2b5/2BpP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/5N2/PPP2PPP/RNBQK2R w KQkq - 2 5' => {
			eco => 'C44y',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Bc4 Bc5

			variation => N__('Scotch Gambit: 4... Bc5'),
			moves => {
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/2P2N2/PP3PPP/RNBQK2R b KQkq - 0 5',
				'e1g1' => 'r1bqk1nr/pppp1ppp/2n5/2b5/2BpP3/5N2/PPP2PPP/RNBQ1RK1 b kq - 3 5',
				'f3g5' => 'r1bqk1nr/pppp1ppp/2n5/2b3N1/2BpP3/8/PPP2PPP/RNBQK2R b KQkq - 3 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/3NP3/4B3/PPP2PPP/RN1QKB1R b KQkq - 2 5' => {
			eco => 'C45u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Be3

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Be3'),
			moves => {
				'd8f6' => 'r1b1k1nr/pppp1ppp/2n2q2/2b5/3NP3/4B3/PPP2PPP/RN1QKB1R w KQkq - 3 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/3NP3/8/PPP2PPP/RNBQKB1R w KQkq - 1 5' => {
			eco => 'C45n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5

			variation => N__('Scotch: 4.Nxd4 Bc5'),
			moves => {
				'c1e3' => 'r1bqk1nr/pppp1ppp/2n5/2b5/3NP3/4B3/PPP2PPP/RN1QKB1R b KQkq - 2 5',
				'd4b3' => 'r1bqk1nr/pppp1ppp/2n5/2b5/4P3/1N6/PPP2PPP/RNBQKB1R b KQkq - 2 5',
				'd4c6' => 'r1bqk1nr/pppp1ppp/2N5/2b5/4P3/8/PPP2PPP/RNBQKB1R b KQkq - 0 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b5/4P3/1N6/PPP2PPP/RNBQKB1R b KQkq - 2 5' => {
			eco => 'C45o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3

			variation => N__('Scotch: 4.Nxd4 Bc5 5. Nb3'),
			moves => {
				'c5b4' => 'r1bqk1nr/pppp1ppp/2n5/8/1b2P3/1N6/PPP2PPP/RNBQKB1R w KQkq - 3 6',
				'c5b6' => 'r1bqk1nr/pppp1ppp/1bn5/8/4P3/1N6/PPP2PPP/RNBQKB1R w KQkq - 3 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/3Np3/1b2P3/5N2/PPPP1PPP/R1BQKB1R b KQkq - 5 4' => {
			eco => 'C46h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Bb4 4. Nd5

			variation => N__('Three Knights: 3... Bb4 4. Nd5'),
			moves => {
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/3Np3/1b2P3/5N2/PPPP1PPP/R1BQKB1R w KQkq - 6 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/4p3/1b2P3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq - 4 4' => {
			eco => 'C46g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Bb4

			variation => N__('Three Knights: 3... Bb4'),
			moves => {
				'c3d5' => 'r1bqk1nr/pppp1ppp/2n5/3Np3/1b2P3/5N2/PPPP1PPP/R1BQKB1R b KQkq - 5 4',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/4p3/1bB1P3/2P2N2/P2P1PPP/RNBQK2R b KQkq - 0 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3

			variation => N__('Evans Gambit Accepted, 5. c3'),
			moves => {
				'b4a5' => 'r1bqk1nr/pppp1ppp/2n5/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6',
				'b4c5' => 'r1bqk1nr/pppp1ppp/2n5/2b1p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6',
				'b4d6' => 'r1bqk1nr/pppp1ppp/2nb4/4p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6',
				'b4e7' => 'r1bqk1nr/ppppbppp/2n5/4p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6',
				'b4f8' => 'r1bqkbnr/pppp1ppp/2n5/4p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/4p3/1bB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 0 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4

			variation => N__('Evans Gambit Accepted'),
			moves => {
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/4p3/1bB1P3/2P2N2/P2P1PPP/RNBQK2R b KQkq - 0 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/4p3/1bP5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq - 4 4' => {
			eco => 'A27',
                        # TRANSLATORS: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Bb4

			variation => N__('English: Three Knights, 3... Bb4'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1b1NP3/8/PPP2PPP/RNBQKB1R w KQkq - 1 5' => {
			eco => 'C45c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bb4+

			variation => N__('Scotch: 4.Nxd4 Bb4+'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1b2P3/1N6/PPP2PPP/RNBQKB1R w KQkq - 3 6' => {
			eco => 'C45o',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb4+

			variation => N__('Scotch: Romanishin Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1b2P3/2N2N2/PP3PPP/R1BQKB1R w KQkq - 1 6' => {
			eco => 'C44w',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. c3 dxc3 5. Nxc3 Bb4

			variation => N__('Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4'),
			moves => {
				'f1c4' => 'r1bqk1nr/pppp1ppp/2n5/8/1bB1P3/2N2N2/PP3PPP/R1BQK2R b KQkq - 2 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1b2P3/2P1Q3/PP3PPP/RNB1KBNR b KQkq - 0 5' => {
			eco => 'C22',
                        # TRANSLATORS:

			variation => N__('Centre Game: Paulsen Attack'),
			moves => {
				'b4e7' => 'r1bqk1nr/ppppbppp/2n5/8/4P3/2P1Q3/PP3PPP/RNB1KBNR w KQkq - 1 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1b2P3/4Q3/PPP2PPP/RNB1KBNR w KQkq - 3 5' => {
			eco => 'C22',
                        # TRANSLATORS:

			variation => N__('Centre Game: Paulsen Attack'),
			moves => {
				'c2c3' => 'r1bqk1nr/pppp1ppp/2n5/8/1b2P3/2P1Q3/PP3PPP/RNB1KBNR b KQkq - 0 5',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1bB1P3/2N2N2/PP3PPP/R1BQK2R b KQkq - 2 6' => {
			eco => 'C44w',
                        # TRANSLATORS:

			variation => N__('Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/8/1bB1P3/2N2N2/PP3PPP/R1BQK2R w KQkq - 0 7',
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/8/1bB1P3/2N2N2/PP3PPP/R1BQK2R w KQkq - 3 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/8/1bBpP3/5N2/PPP2PPP/RNBQK2R w KQkq - 2 5' => {
			eco => 'C44x',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Bc4 Bb4+

			variation => N__('Scotch Gambit: London Defence'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b3p3/2B1P3/1QP2N2/P2P1PPP/RNB1K2R b KQkq - 2 6' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. Qb3

			variation => N__('Evans Gambit: 5... Ba5 6. Qb3'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQ1RK1 b kq - 2 6' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. O-O

			variation => N__('Evans Gambit: 5... Ba5 6.O-O'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQ1RK1 w kq - 0 7',
				'g8f6' => 'r1bqk2r/pppp1ppp/2n2n2/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQ1RK1 w kq - 3 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5

			variation => N__('Evans Gambit: 5... Ba5'),
			moves => {
				'd1b3' => 'r1bqk1nr/pppp1ppp/2n5/b3p3/2B1P3/1QP2N2/P2P1PPP/RNB1K2R b KQkq - 2 6',
				'd2d4' => 'r1bqk1nr/pppp1ppp/2n5/b3p3/2BPP3/2P2N2/P4PPP/RNBQK2R b KQkq d3 0 6',
				'e1g1' => 'r1bqk1nr/pppp1ppp/2n5/b3p3/2B1P3/2P2N2/P2P1PPP/RNBQ1RK1 b kq - 2 6',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b3p3/2BPP3/2P2N2/P4PPP/RNBQK2R b KQkq d3 0 6' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4

			variation => N__('Evans Gambit: 5... Ba5 6. d4'),
			moves => {
				'b7b5' => 'r1bqk1nr/p1pp1ppp/2n5/bp2p3/2BPP3/2P2N2/P4PPP/RNBQK2R w KQkq b6 0 7',
				'd7d6' => 'r1bqk1nr/ppp2ppp/2np4/b3p3/2BPP3/2P2N2/P4PPP/RNBQK2R w KQkq - 0 7',
				'e5d4' => 'r1bqk1nr/pppp1ppp/2n5/b7/2BpP3/2P2N2/P4PPP/RNBQK2R w KQkq - 0 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b7/2B1P3/1Qp2N2/P4PPP/RNB2RK1 b kq - 1 8' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'd8f6' => 'r1b1k1nr/pppp1ppp/2n2q2/b7/2B1P3/1Qp2N2/P4PPP/RNB2RK1 w kq - 2 9',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b7/2B1P3/2p2N2/P4PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 exd4 7. O-O dxc3

			variation => N__('Evans Gambit: Compromised Defence'),
			moves => {
				'd1b3' => 'r1bqk1nr/pppp1ppp/2n5/b7/2B1P3/1Qp2N2/P4PPP/RNB2RK1 b kq - 1 8',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b7/2BpP3/2P2N2/P4PPP/RNBQ1RK1 b kq - 1 7' => {
			eco => 'C52',
                        # TRANSLATORS:

			variation => N__('Evans Gambit: 5... Ba5 6. d4 exd4'),
			moves => {
				'd4c3' => 'r1bqk1nr/pppp1ppp/2n5/b7/2B1P3/2p2N2/P4PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/b7/2BpP3/2P2N2/P4PPP/RNBQK2R w KQkq - 0 7' => {
			eco => 'C52',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Ba5 6. d4 exd4

			variation => N__('Evans Gambit: 5... Ba5 6. d4 exd4'),
			moves => {
				'e1g1' => 'r1bqk1nr/pppp1ppp/2n5/b7/2BpP3/2P2N2/P4PPP/RNBQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk1nr/pppp1ppp/2nb4/4p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Bd6

			variation => N__('Evans Gambit: Stone-Ware Defence'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/8/1B2p3/3bP3/2P5/PP1P1PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C64s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4 Bxd4 6. c3

			variation => N__('Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6. c3'),
			moves => {
				'd4b6' => 'r1bqk1nr/pppp1ppp/1b6/1B2p3/4P3/2P5/PP1P1PPP/RNBQ1RK1 w kq - 1 7',
			},
		},
		'r1bqk1nr/pppp1ppp/8/1B2p3/3bP3/8/PPPP1PPP/RNBQ1RK1 w kq - 0 6' => {
			eco => 'C64s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4 Bxd4

			variation => N__('Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4'),
			moves => {
				'c2c3' => 'r1bqk1nr/pppp1ppp/8/1B2p3/3bP3/2P5/PP1P1PPP/RNBQ1RK1 b kq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/8/1Bb1p3/1P1nP3/5N2/P1PP1PPP/RNBQ1RK1 b kq b3 0 5' => {
			eco => 'C64q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. b4

			variation => N__('Spanish: Classical, Zaitsev Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/8/1Bb1p3/3NP3/8/PPPP1PPP/RNBQ1RK1 b kq - 0 5' => {
			eco => 'C64s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Nxd4

			variation => N__('Spanish: Classical, 4.O-O Nd4 5.Nxd4'),
			moves => {
				'c5d4' => 'r1bqk1nr/pppp1ppp/8/1B2p3/3bP3/8/PPPP1PPP/RNBQ1RK1 w kq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/8/1Bb1p3/3nP3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 6 5' => {
			eco => 'C64p',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4

			variation => N__('Spanish: Classical, 4.O-O Nd4'),
			moves => {
				'b2b4' => 'r1bqk1nr/pppp1ppp/8/1Bb1p3/1P1nP3/5N2/P1PP1PPP/RNBQ1RK1 b kq b3 0 5',
				'b5c4' => 'r1bqk1nr/pppp1ppp/8/2b1p3/2BnP3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 7 5',
				'f3d4' => 'r1bqk1nr/pppp1ppp/8/1Bb1p3/3NP3/8/PPPP1PPP/RNBQ1RK1 b kq - 0 5',
			},
		},
		'r1bqk1nr/pppp1ppp/8/1Bb5/3pP3/3P4/PPP2PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C61j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nd4 4. Nxd4 exd4 5. O-O Bc5 6. d3

			variation => N__('Spanish: Bird\'s, 5.O-O Bc5 6. d3'),
			moves => {
				'c7c6' => 'r1bqk1nr/pp1p1ppp/2p5/1Bb5/3pP3/3P4/PPP2PPP/RNBQ1RK1 w kq - 0 7',
			},
		},
		'r1bqk1nr/pppp1ppp/8/1Bb5/3pP3/8/PPPP1PPP/RNBQ1RK1 w kq - 2 6' => {
			eco => 'C61i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nd4 4. Nxd4 exd4 5. O-O Bc5

			variation => N__('Spanish: Bird\'s, 5.O-O Bc5'),
			moves => {
				'd2d3' => 'r1bqk1nr/pppp1ppp/8/1Bb5/3pP3/3P4/PPP2PPP/RNBQ1RK1 b kq - 0 6',
			},
		},
		'r1bqk1nr/pppp1ppp/8/2b1p3/2BnP3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 7 5' => {
			eco => 'C64r',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. O-O Nd4 5. Bc4

			variation => N__('Spanish: Classical, 4.O-O Nd4 5. Bc4'),
			moves => {
			},
		},
		'r1bqk1nr/pppp2pp/2n5/1Bb1pp2/3PP3/2P2N2/PP3PPP/RNBQK2R b KQkq d3 0 5' => {
			eco => 'C64g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. c3 f5 5. d4

			variation => N__('Spanish: Classical, Cordel Gambit, 5. d4'),
			moves => {
			},
		},
		'r1bqk1nr/pppp2pp/2n5/1Bb1pp2/4P3/2P2N2/PP1P1PPP/RNBQK2R w KQkq f6 0 5' => {
			eco => 'C64f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Bc5 4. c3 f5

			variation => N__('Spanish: Classical, Cordel Gambit'),
			moves => {
				'd2d4' => 'r1bqk1nr/pppp2pp/2n5/1Bb1pp2/3PP3/2P2N2/PP3PPP/RNBQK2R b KQkq d3 0 5',
			},
		},
		'r1bqk1nr/pppp2pp/2n5/2b1p1N1/2B1Pp2/3P4/PPP2PPP/RNBQK2R w KQkq - 0 6' => {
			eco => 'C50n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. d3 f5 5. Ng5 f4

			variation => N__('Giuoco Pianissimo: Dubois Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp2pp/2n5/2b1pp2/2B1P3/3P1N2/PPP2PPP/RNBQK2R w KQkq f6 0 5' => {
			eco => 'C50n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. d3 f5

			variation => N__('Giuoco Pianissimo: Lucchini Gambit'),
			moves => {
				'f3g5' => 'r1bqk1nr/pppp2pp/2n5/2b1ppN1/2B1P3/3P4/PPP2PPP/RNBQK2R b KQkq - 1 5',
			},
		},
		'r1bqk1nr/pppp2pp/2n5/2b1ppN1/2B1P3/3P4/PPP2PPP/RNBQK2R b KQkq - 1 5' => {
			eco => 'C50n',
                        # TRANSLATORS:

			variation => N__('Giuoco Pianissimo: Lucchini Gambit'),
			moves => {
				'f5f4' => 'r1bqk1nr/pppp2pp/2n5/2b1p1N1/2B1Pp2/3P4/PPP2PPP/RNBQK2R w KQkq - 0 6',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/1B2p3/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'C60e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Be7

			variation => N__('Spanish: Lucena Defence'),
			moves => {
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/1PB1P3/5N2/P1PP1PPP/RNBQK2R w KQkq - 1 5' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Be7

			variation => N__('Evans Gambit Declined: 4... Be7'),
			moves => {
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/2B1P3/1QP2N2/P2P1PPP/RNB1K2R b KQkq - 2 6' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Be7 6. Qb3

			variation => N__('Evans Gambit: Anderssen, 6. Qb3'),
			moves => {
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/2B1P3/2P2N2/P2P1PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Be7

			variation => N__('Evans Gambit: Anderssen Variation'),
			moves => {
				'd1b3' => 'r1bqk1nr/ppppbppp/2n5/4p3/2B1P3/1QP2N2/P2P1PPP/RNB1K2R b KQkq - 2 6',
				'd2d4' => 'r1bqk1nr/ppppbppp/2n5/4p3/2BPP3/2P2N2/P4PPP/RNBQK2R b KQkq d3 0 6',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/2B1P3/5N2/PPPP1PPP/RNBQK2R w KQkq - 4 4' => {
			eco => 'C50e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Be7

			variation => N__('Italian: Hungarian Defence'),
			moves => {
				'd2d4' => 'r1bqk1nr/ppppbppp/2n5/4p3/2BPP3/5N2/PPP2PPP/RNBQK2R b KQkq d3 0 4',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/2BPP3/2P2N2/P4PPP/RNBQK2R b KQkq d3 0 6' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Be7 6. d4

			variation => N__('Evans Gambit: Anderssen, 6. d4'),
			moves => {
				'c6a5' => 'r1bqk1nr/ppppbppp/8/n3p3/2BPP3/2P2N2/P4PPP/RNBQK2R w KQkq - 1 7',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/2BPP3/5N2/PPP2PPP/RNBQK2R b KQkq d3 0 4' => {
			eco => 'C50e',
                        # TRANSLATORS:

			variation => N__('Italian: Hungarian Defence'),
			moves => {
				'd7d6' => 'r1bqk1nr/ppp1bppp/2np4/4p3/2BPP3/5N2/PPP2PPP/RNBQK2R w KQkq - 0 5',
				'e5d4' => 'r1bqk1nr/ppppbppp/2n5/8/2BpP3/5N2/PPP2PPP/RNBQK2R w KQkq - 0 5',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/4p3/4P3/2P2N2/PP1P1PPP/RNBQKB1R w KQkq - 1 4' => {
			eco => 'C44i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. c3 Be7

			variation => N__('Ponziani: Romanishin Variation'),
			moves => {
			},
		},
		'r1bqk1nr/ppppbppp/2n5/8/2BpP3/2P2N2/PP3PPP/RNBQK2R b KQkq - 0 5' => {
			eco => 'C50f',
                        # TRANSLATORS:

			variation => N__('Italian: Hungarian Defence, 4. d4 exd4'),
			moves => {
				'g8f6' => 'r1bqk2r/ppppbppp/2n2n2/8/2BpP3/2P2N2/PP3PPP/RNBQK2R w KQkq - 1 6',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/8/2BpP3/5N2/PPP2PPP/RNBQK2R w KQkq - 0 5' => {
			eco => 'C50f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Be7 4. d4 exd4

			variation => N__('Italian: Hungarian Defence, 4. d4 exd4'),
			moves => {
				'c2c3' => 'r1bqk1nr/ppppbppp/2n5/8/2BpP3/2P2N2/PP3PPP/RNBQK2R b KQkq - 0 5',
			},
		},
		'r1bqk1nr/ppppbppp/2n5/8/4P3/2P1Q3/PP3PPP/RNB1KBNR w KQkq - 1 6' => {
			eco => 'C22',
                        # TRANSLATORS: 1. e4 e5 2. d4 exd4 3. Qxd4 Nc6 4. Qe3 Bb4+ 5. c3 Be7

			variation => N__('Centre Game: Charousek Variation'),
			moves => {
			},
		},
		'r1bqk1nr/ppppbppp/8/n3p3/2BPP3/2P2N2/P4PPP/RNBQK2R w KQkq - 1 7' => {
			eco => 'C51',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. b4 Bxb4 5. c3 Be7 6. d4 Na5

			variation => N__('Evans Gambit: Cordel Variation'),
			moves => {
			},
		},
		'r1bqk1r1/ppp2BBp/1b6/nP2p3/3PP3/8/P1P2PPP/RN1QK2R b KQq - 0 10' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Lange Variation'),
			moves => {
				'e8f7' => 'r1bq2r1/ppp2kBp/1b6/nP2p3/3PP3/8/P1P2PPP/RN1QK2R w KQ - 0 11',
			},
		},
		'r1bqk1r1/ppp2pBp/1b6/nP2p3/2BPP3/8/P1P2PPP/RN1QK2R w KQq - 1 10' => {
			eco => 'C51',
                        # TRANSLATORS:

			variation => N__('Evans Gambit Declined, Lange Variation'),
			moves => {
				'c4f7' => 'r1bqk1r1/ppp2BBp/1b6/nP2p3/3PP3/8/P1P2PPP/RN1QK2R b KQq - 0 10',
			},
		},
		'r1bqk2r/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w kq - 0 8' => {
			eco => 'A55j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O a6

			variation => N__('Old Indian: 5. e4 Be7 6. Be2 c6 7.O-O a6'),
			moves => {
			},
		},
		'r1bqk2r/1p1nbppp/p2p1n2/4p3/4P3/2N3P1/PPP1NPBP/R1BQK2R w KQkq - 4 9' => {
			eco => 'B91m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. g3 e5 7. Nde2 Be7 8. Bg2 Nbd7

			variation => N__('Sicilian: Najdorf, 6. g3 e5 7.Nde2 Be7 8. Bg2 Nbd7'),
			moves => {
			},
		},
		'r1bqk2r/1p1nbppp/p2p1n2/4p3/P1B1PP2/2N2N2/1PP3PP/R1BQK2R b KQkq - 2 9' => {
			eco => 'B93r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bc4

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bc4'),
			moves => {
			},
		},
		'r1bqk2r/1p1nbppp/p2p1n2/4p3/P3PP2/2N2N2/1PP3PP/R1BQKB1R w KQkq - 1 9' => {
			eco => 'B93r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7'),
			moves => {
				'f1c4' => 'r1bqk2r/1p1nbppp/p2p1n2/4p3/P1B1PP2/2N2N2/1PP3PP/R1BQK2R b KQkq - 2 9',
				'f1d3' => 'r1bqk2r/1p1nbppp/p2p1n2/4p3/P3PP2/2NB1N2/1PP3PP/R1BQK2R b KQkq - 2 9',
			},
		},
		'r1bqk2r/1p1nbppp/p2p1n2/4p3/P3PP2/2NB1N2/1PP3PP/R1BQK2R b KQkq - 2 9' => {
			eco => 'B93s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3

			variation => N__('Sicilian: Najdorf, 6. f4 e5 7. Nf3 Nbd7 8. a4 Be7 9. Bd3'),
			moves => {
				'e8g8' => 'r1bq1rk1/1p1nbppp/p2p1n2/4p3/P3PP2/2NB1N2/1PP3PP/R1BQK2R w KQ - 3 10',
			},
		},
		'r1bqk2r/1p1nbppp/p2ppn2/6B1/3NP3/2N2Q2/PPP2PPP/2KR1B1R w kq - 4 9' => {
			eco => 'B95m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 a6 6. Bg5 e6 7. Qf3 Be7 8. O-O-O Nbd7

			variation => N__('Sicilian: Najdorf, 6... e6 7. Qf3 Be7 8.O-O-O Nbd7'),
			moves => {
			},
		},
		'r1bqk2r/1p1p1ppp/p1n1pn2/8/1bPNP3/2N5/PPB2PPP/R1BQK2R b KQkq - 6 8' => {
			eco => 'B41k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 a6 5. c4 Nf6 6. Nc3 Bb4 7. Bd3 Nc6 8. Bc2

			variation => N__('Sicilian: Kan, 5. c4 Nf6, Bronstein Variation'),
			moves => {
			},
		},
		'r1bqk2r/1p1p1ppp/p1n1pn2/8/1bPNP3/2NB4/PP3PPP/R1BQK2R w KQkq - 5 8' => {
			eco => 'B41k',
                        # TRANSLATORS:

			variation => N__('Sicilian: Kan, 5. c4 Nf6 6. Nc3 Bb4'),
			moves => {
				'd3c2' => 'r1bqk2r/1p1p1ppp/p1n1pn2/8/1bPNP3/2N5/PPB2PPP/R1BQK2R b KQkq - 6 8',
			},
		},
		'r1bqk2r/1p2bpp1/p1nppn1p/8/3NP3/2N1B3/PPPQ1PPP/2KR1B1R w kq - 2 10' => {
			eco => 'B66m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Be7

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Be7'),
			moves => {
				'f2f3' => 'r1bqk2r/1p2bpp1/p1nppn1p/8/3NP3/2N1BP2/PPPQ2PP/2KR1B1R b kq - 0 10',
				'f2f4' => 'r1bqk2r/1p2bpp1/p1nppn1p/8/3NPP2/2N1B3/PPPQ2PP/2KR1B1R b kq f3 0 10',
			},
		},
		'r1bqk2r/1p2bpp1/p1nppn1p/8/3NP3/2N1BP2/PPPQ2PP/2KR1B1R b kq - 0 10' => {
			eco => 'B66m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Be7 10. f3

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Be7 10. f3'),
			moves => {
				'c6d4' => 'r1bqk2r/1p2bpp1/p2ppn1p/8/3nP3/2N1BP2/PPPQ2PP/2KR1B1R w kq - 0 11',
			},
		},
		'r1bqk2r/1p2bpp1/p1nppn1p/8/3NPP2/2N1B3/PPPQ2PP/2KR1B1R b kq f3 0 10' => {
			eco => 'B66o',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Be7 10. f4

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Be7 10. f4'),
			moves => {
				'c6d4' => 'r1bqk2r/1p2bpp1/p2ppn1p/8/3nPP2/2N1B3/PPPQ2PP/2KR1B1R w kq - 0 11',
			},
		},
		'r1bqk2r/1p2bpp1/p2ppn1p/8/3BP3/2N2P2/PPPQ2PP/2KR1B1R b kq - 0 11' => {
			eco => 'B66n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Be7 10. f3 Nxd4 11. Bxd4

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Be7 10. f3 Nxd4 11.Bxd4'),
			moves => {
			},
		},
		'r1bqk2r/1p2bpp1/p2ppn1p/8/3nP3/2N1BP2/PPPQ2PP/2KR1B1R w kq - 0 11' => {
			eco => 'B66m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Be7 10. f3 Nxd4

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Be7 10. f3 Nxd4'),
			moves => {
				'e3d4' => 'r1bqk2r/1p2bpp1/p2ppn1p/8/3BP3/2N2P2/PPPQ2PP/2KR1B1R b kq - 0 11',
			},
		},
		'r1bqk2r/1p2bpp1/p2ppn1p/8/3nPP2/2N1B3/PPPQ2PP/2KR1B1R w kq - 0 11' => {
			eco => 'B66p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O h6 9. Be3 Be7 10. f4 Nxd4

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O h6 9. Be3 Be7 10. f4 Nxd4'),
			moves => {
			},
		},
		'r1bqk2r/1p2bppp/p1n1pn2/2p5/P1BP4/2N1PN2/1P2QPPP/R1B2RK1 b kq - 4 9' => {
			eco => 'D27s',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. O-O a6 7. a4 Nc6 8. Nc3 Be7 9. Qe2

			variation => N__('QGA: Classical, Rubinstein, 8. Nc3 Be7 9. Qe2'),
			moves => {
			},
		},
		'r1bqk2r/1p2bppp/p1n1pn2/2p5/P1BP4/2N1PN2/1P3PPP/R1BQ1RK1 w kq - 3 9' => {
			eco => 'D27r',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. O-O a6 7. a4 Nc6 8. Nc3 Be7

			variation => N__('QGA: Classical, Rubinstein, 8. Nc3 Be7'),
			moves => {
				'd1e2' => 'r1bqk2r/1p2bppp/p1n1pn2/2p5/P1BP4/2N1PN2/1P2QPPP/R1B2RK1 b kq - 4 9',
			},
		},
		'r1bqk2r/1p2bppp/p1n1pn2/8/P1BP4/5N2/1P2QPPP/RNBR2K1 b kq - 0 10' => {
			eco => 'D27v',
                        # TRANSLATORS:

			variation => N__('QGA: Classical, Rubinstein, 8. Qe2 cxd4'),
			moves => {
				'e8g8' => 'r1bq1rk1/1p2bppp/p1n1pn2/8/P1BP4/5N2/1P2QPPP/RNBR2K1 w - - 1 11',
			},
		},
		'r1bqk2r/1p2bppp/p1n1pn2/8/P1Bp4/4PN2/1P2QPPP/RNBR2K1 w kq - 2 10' => {
			eco => 'D27v',
                        # TRANSLATORS:

			variation => N__('QGA: Classical, Rubinstein, 8. Qe2 cxd4'),
			moves => {
				'e3d4' => 'r1bqk2r/1p2bppp/p1n1pn2/8/P1BP4/5N2/1P2QPPP/RNBR2K1 b kq - 0 10',
			},
		},
		'r1bqk2r/1p2bppp/p1nppn2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R w kq - 2 9' => {
			eco => 'B66d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O Be7

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O Be7'),
			moves => {
			},
		},
		'r1bqk2r/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP2BPPP/R1BQK2R b KQkq - 3 9' => {
			eco => 'B44u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7 9. Be2

			variation => N__('Sicilian, Taimanov, Szen, 7.N1c3 a6 8. Na3 Be7 9. Be2'),
			moves => {
				'b7b6' => 'r1bqk2r/4bppp/ppnppn2/8/2P1P3/N1N5/PP2BPPP/R1BQK2R w KQkq - 0 10',
				'e8g8' => 'r1bq1rk1/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP2BPPP/R1BQK2R w KQ - 4 10',
			},
		},
		'r1bqk2r/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP3PPP/R1BQKB1R w KQkq - 2 9' => {
			eco => 'B44t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7

			variation => N__('Sicilian, Taimanov, Szen, 7.N1c3 a6 8. Na3 Be7'),
			moves => {
				'f1e2' => 'r1bqk2r/1p2bppp/p1nppn2/8/2P1P3/N1N5/PP2BPPP/R1BQK2R b KQkq - 3 9',
			},
		},
		'r1bqk2r/1p2bppp/p1nppn2/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R w KQkq - 3 9' => {
			eco => 'B80i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be3 a6 7. f3 Nc6 8. Qd2 Be7

			variation => N__('Sicilian: Scheveningen, English, 7... Nc6 8. Qd2'),
			moves => {
			},
		},
		'r1bqk2r/1p2bppp/p2ppn2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R w kq - 1 10' => {
			eco => 'B66e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 a6 8. O-O-O Nxd4 9. Qxd4 Be7

			variation => N__('Sicilian: Richter-Rauzer, 7... a6 8.O-O-O Nxd4 9.Qxd4 Be7'),
			moves => {
			},
		},
		'r1bqk2r/1p3ppp/p1n1pn2/2b5/2B5/4PN2/PP2QPPP/RNB2RK1 w kq - 2 9' => {
			eco => 'D26n',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. Qe2 a6 7. dxc5 Bxc5 8. O-O Nc6

			variation => N__('QGA: Classical, Furman, 6... a6 7.dxc5 Bxc5 8.O-O Nc6'),
			moves => {
			},
		},
		'r1bqk2r/1p3ppp/p1n1pn2/2bp4/1PP5/P1N1PN2/5PPP/R1BQKB1R b KQkq b3 0 8' => {
			eco => 'D40q',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 5. e3 Nc6 6. a3 a6 7.dxc5 Bxc5'),
			moves => {
				'c5a7' => 'r1bqk2r/bp3ppp/p1n1pn2/3p4/1PP5/P1N1PN2/5PPP/R1BQKB1R w KQkq - 1 9',
			},
		},
		'r1bqk2r/1p3ppp/p1n1pn2/2bp4/2P5/P1N1PN2/1P3PPP/R1BQKB1R w KQkq - 0 8' => {
			eco => 'D40q',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. e3 Nc6 6. a3 a6 7. dxc5 Bxc5

			variation => N__('QGD: Semi-Tarrasch, 5. e3 Nc6 6. a3 a6 7.dxc5 Bxc5'),
			moves => {
				'b2b4' => 'r1bqk2r/1p3ppp/p1n1pn2/2bp4/1PP5/P1N1PN2/5PPP/R1BQKB1R b KQkq b3 0 8',
			},
		},
		'r1bqk2r/1pp1bppp/p1Bp1n2/4p3/4P3/5N2/PPPP1PPP/RNBQR1K1 b kq - 0 7' => {
			eco => 'C87c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. Bxc6+

			variation => N__('Spanish: Closed, Averbakh, 7.Bxc6+'),
			moves => {
				'b7c6' => 'r1bqk2r/2p1bppp/p1pp1n2/4p3/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq - 0 8',
			},
		},
		'r1bqk2r/1pp1bppp/p1np1n2/4p3/B3P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq - 0 7' => {
			eco => 'C87e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. c3

			variation => N__('Spanish: Closed, Averbakh, 7. c3'),
			moves => {
				'c8g4' => 'r2qk2r/1pp1bppp/p1np1n2/4p3/B3P1b1/2P2N2/PP1P1PPP/RNBQR1K1 w kq - 1 8',
				'e8g8' => 'r1bq1rk1/1pp1bppp/p1np1n2/4p3/B3P3/2P2N2/PP1P1PPP/RNBQR1K1 w - - 1 8',
			},
		},
		'r1bqk2r/1pp1bppp/p1np1n2/4p3/B3P3/5N2/PPPP1PPP/RNBQR1K1 w kq - 0 7' => {
			eco => 'C87b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6

			variation => N__('Spanish: Closed, Averbakh (Russian) Variation'),
			moves => {
				'a4c6' => 'r1bqk2r/1pp1bppp/p1Bp1n2/4p3/4P3/5N2/PPPP1PPP/RNBQR1K1 b kq - 0 7',
				'c2c3' => 'r1bqk2r/1pp1bppp/p1np1n2/4p3/B3P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq - 0 7',
			},
		},
		'r1bqk2r/1pp1bppp/p1np1n2/4p3/B3P3/5N2/PPPPQPPP/RNB2RK1 w kq - 0 7' => {
			eco => 'C86a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 d6

			variation => N__('Spanish: Worrall Attack, 6... d6'),
			moves => {
			},
		},
		'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/2N2N2/PPPP1PPP/R1BQ1RK1 b kq - 1 7' => {
			eco => 'C85d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. Nc3

			variation => N__('Spanish: Closed, Exchange, 7. Nc3'),
			moves => {
				'c8g4' => 'r2qk2r/1pp1bppp/p1p2n2/4p3/4P1b1/2N2N2/PPPP1PPP/R1BQ1RK1 w kq - 2 8',
			},
		},
		'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/3P1N2/PPP2PPP/RNBQ1RK1 b kq - 0 7' => {
			eco => 'C85j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3

			variation => N__('Spanish: Closed, Exchange, 7. d3'),
			moves => {
				'c8g4' => 'r2qk2r/1pp1bppp/p1p2n2/4p3/4P1b1/3P1N2/PPP2PPP/RNBQ1RK1 w kq - 1 8',
				'f6d7' => 'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/3P1N2/PPP2PPP/RNBQ1RK1 w kq - 1 8',
			},
		},
		'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/5N2/PPPP1PPP/RNB1QRK1 b kq - 1 7' => {
			eco => 'C85f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. Qe1

			variation => N__('Spanish: Closed, Exchange, 7. Qe1'),
			moves => {
				'c6c5' => 'r1bqk2r/1pp1bppp/p4n2/2p1p3/4P3/5N2/PPPP1PPP/RNB1QRK1 w kq - 0 8',
				'f6d7' => 'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/5N2/PPPP1PPP/RNB1QRK1 w kq - 2 8',
			},
		},
		'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 0 7' => {
			eco => 'C85b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6

			variation => N__('Spanish: Closed, Exchange'),
			moves => {
				'b1c3' => 'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/2N2N2/PPPP1PPP/R1BQ1RK1 b kq - 1 7',
				'd1e1' => 'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/5N2/PPPP1PPP/RNB1QRK1 b kq - 1 7',
				'd1e2' => 'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/5N2/PPPPQPPP/RNB2RK1 b kq - 1 7',
				'd2d3' => 'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/3P1N2/PPP2PPP/RNBQ1RK1 b kq - 0 7',
			},
		},
		'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/5N2/PPPPQPPP/RNB2RK1 b kq - 1 7' => {
			eco => 'C85c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. Qe2

			variation => N__('Spanish: Closed, Exchange, 7. Qe2'),
			moves => {
			},
		},
		'r1bqk2r/1pp1bppp/p4n2/2p1p3/4P3/5N2/PPPP1PPP/RNB1QRK1 w kq - 0 8' => {
			eco => 'C85g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. Qe1 c5

			variation => N__('Spanish: Closed, Exchange, 7. Qe1 c5'),
			moves => {
			},
		},
		'r1bqk2r/1pp2ppp/p1np1n2/2b1p3/2B1PP2/2NP1N2/PPP3PP/R1BQK2R w KQkq - 0 7' => {
			eco => 'C28l',
                        # TRANSLATORS: 1. e4 e5 2. Nc3 Nf6 3. Bc4 Nc6 4. d3 Bc5 5. f4 d6 6. Nf3 a6

			variation => N__('Vienna: 3. Bc4 Nc6 4. d3 Bc5 5. f4 d6 6. Nf3 a6'),
			moves => {
			},
		},
		'r1bqk2r/1pp2ppp/p1np1n2/2b1p3/4P3/2NP1NP1/PPP2PBP/R1BQK2R w KQkq - 0 7' => {
			eco => 'C47h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. g3 Bc5 5. Bg2 d6 6. d3 a6

			variation => N__('Four Knights: Glek, Main Line'),
			moves => {
			},
		},
		'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/1P3N2/P1PP1PPP/RNB1QRK1 b kq - 0 8' => {
			eco => 'C85i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. Qe1 Nd7 8. b3

			variation => N__('Spanish: Closed, Exchange, 7. Qe1 Nd7 8. b3'),
			moves => {
			},
		},
		'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/3P1N2/PPP2PPP/RNBQ1RK1 w kq - 1 8' => {
			eco => 'C85l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7'),
			moves => {
				'b1d2' => 'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/3P1N2/PPPN1PPP/R1BQ1RK1 b kq - 2 8',
			},
		},
		'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/3P1N2/PPPN1PPP/R1BQ1RK1 b kq - 2 8' => {
			eco => 'C85m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. d3 Nd7 8. Nbd2

			variation => N__('Spanish: Closed, Exchange, 7. d3 Nd7 8.Nbd2'),
			moves => {
				'e8g8' => 'r1bq1rk1/1ppnbppp/p1p5/4p3/4P3/3P1N2/PPPN1PPP/R1BQ1RK1 w - - 3 9',
			},
		},
		'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/5N2/PPPP1PPP/RNB1QRK1 w kq - 2 8' => {
			eco => 'C85h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6 dxc6 7. Qe1 Nd7

			variation => N__('Spanish: Closed, Exchange, 7. Qe1 Nd7'),
			moves => {
				'b2b3' => 'r1bqk2r/1ppnbppp/p1p5/4p3/4P3/1P3N2/P1PP1PPP/RNB1QRK1 b kq - 0 8',
			},
		},
		'r1bqk2r/1ppp1ppp/p1n2n2/2b1N3/B3P3/8/PPPP1PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C78c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Bc5 6. Nxe5

			variation => N__('Spanish: Möller Defence, 6.Nxe5'),
			moves => {
			},
		},
		'r1bqk2r/1ppp1ppp/p1n2n2/2b1p3/2B1P3/2PP1N2/PP3PPP/RNBQK2R w KQkq - 0 6' => {
			eco => 'C54e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Bc5 4. c3 Nf6 5. d3 a6

			variation => N__('Giuoco Pianissimo: 5. d3 a6'),
			moves => {
			},
		},
		'r1bqk2r/1ppp1ppp/p1n2n2/2b1p3/B3P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C78d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Bc5 6. c3

			variation => N__('Spanish: Möller Defence, 6. c3'),
			moves => {
			},
		},
		'r1bqk2r/1ppp1ppp/p1n2n2/2b1p3/B3P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 4 6' => {
			eco => 'C78b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Bc5

			variation => N__('Spanish: Möller Defence'),
			moves => {
				'c2c3' => 'r1bqk2r/1ppp1ppp/p1n2n2/2b1p3/B3P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 6',
				'f3e5' => 'r1bqk2r/1ppp1ppp/p1n2n2/2b1N3/B3P3/8/PPPP1PPP/RNBQ1RK1 b kq - 0 6',
			},
		},
		'r1bqk2r/1ppp1ppp/pbn2n2/8/P3P3/1NN5/1PP2PPP/R1BQKB1R w KQkq - 2 8' => {
			eco => 'C45q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. d4 exd4 4. Nxd4 Bc5 5. Nb3 Bb6 6. a4 a6 7. Nc3 Nf6

			variation => N__('Scotch: Geller Variation'),
			moves => {
			},
		},
		'r1bqk2r/1pppbppp/p1B2n2/4p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C85a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Bxc6

			variation => N__('Spanish: Closed, Exchange'),
			moves => {
				'd7c6' => 'r1bqk2r/1pp1bppp/p1p2n2/4p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 0 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4P3/B2p4/5N2/PPP2PPP/RNBQ1RK1 b kq - 0 7' => {
			eco => 'C84h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5

			variation => N__('Spanish: Closed, Centre Attack, 7. e5'),
			moves => {
				'f6e4' => 'r1bqk2r/1pppbppp/p1n5/4P3/B2pn3/5N2/PPP2PPP/RNBQ1RK1 w kq - 1 8',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B2PP3/5N2/PPP2PPP/RNBQ1RK1 b kq d3 0 6' => {
			eco => 'C84g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4

			variation => N__('Spanish: Closed, Centre Attack'),
			moves => {
				'e5d4' => 'r1bqk2r/1pppbppp/p1n2n2/8/B2pP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 0 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B2PP3/5N2/PPP2PPP/RNBQK2R w KQkq - 1 6' => {
			eco => 'C77n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. d4 Be7

			variation => N__('Spanish: Centre Attack, 5... Be7'),
			moves => {
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/2N2N2/PPPP1PPP/R1BQ1RK1 b kq - 5 6' => {
			eco => 'C84b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Nc3

			variation => N__('Spanish: Closed, 6. Nc3'),
			moves => {
				'b7b5' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/2N2N2/PPPP1PPP/R1BQ1RK1 w kq b6 0 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/3P1N2/PPP2PPP/RNBQ1RK1 b kq - 0 6' => {
			eco => 'C84e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d3

			variation => N__('Spanish: Closed, 6. d3'),
			moves => {
				'b7b5' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/3P1N2/PPP2PPP/RNBQ1RK1 w kq b6 0 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/5N2/PPPP1PPP/RNBQ1RK1 w kq - 4 6' => {
			eco => 'C84a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7

			variation => N__('Spanish: Closed System'),
			moves => {
				'a4c6' => 'r1bqk2r/1pppbppp/p1B2n2/4p3/4P3/5N2/PPPP1PPP/RNBQ1RK1 b kq - 0 6',
				'b1c3' => 'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/2N2N2/PPPP1PPP/R1BQ1RK1 b kq - 5 6',
				'd1e2' => 'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/5N2/PPPPQPPP/RNB2RK1 b kq - 5 6',
				'd2d3' => 'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/3P1N2/PPP2PPP/RNBQ1RK1 b kq - 0 6',
				'd2d4' => 'r1bqk2r/1pppbppp/p1n2n2/4p3/B2PP3/5N2/PPP2PPP/RNBQ1RK1 b kq d3 0 6',
				'f1e1' => 'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/5N2/PPPP1PPP/RNBQR1K1 b kq - 5 6',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/5N2/PPPP1PPP/RNBQR1K1 b kq - 5 6' => {
			eco => 'C87a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1

			variation => N__('Spanish: Closed, 6. Re1'),
			moves => {
				'b7b5' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/5N2/PPPP1PPP/RNBQR1K1 w kq b6 0 7',
				'd7d6' => 'r1bqk2r/1pp1bppp/p1np1n2/4p3/B3P3/5N2/PPPP1PPP/RNBQR1K1 w kq - 0 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/5N2/PPPPQPPP/RNB1K2R w KQkq - 4 6' => {
			eco => 'C77q',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. Qe2 Be7

			variation => N__('Spanish: Wormald Attack, 5... Be7'),
			moves => {
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/4p3/B3P3/5N2/PPPPQPPP/RNB2RK1 b kq - 5 6' => {
			eco => 'C86a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2

			variation => N__('Spanish: Worrall Attack'),
			moves => {
				'b7b5' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/5N2/PPPPQPPP/RNB2RK1 w kq b6 0 7',
				'd7d6' => 'r1bqk2r/1pp1bppp/p1np1n2/4p3/B3P3/5N2/PPPPQPPP/RNB2RK1 w kq - 0 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/8/B2pP3/5N2/PPP2PPP/RNBQ1RK1 w kq - 0 7' => {
			eco => 'C84g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4

			variation => N__('Spanish: Closed, Centre Attack'),
			moves => {
				'e4e5' => 'r1bqk2r/1pppbppp/p1n2n2/4P3/B2p4/5N2/PPP2PPP/RNBQ1RK1 b kq - 0 7',
				'f1e1' => 'r1bqk2r/1pppbppp/p1n2n2/8/B2pP3/5N2/PPP2PPP/RNBQR1K1 b kq - 1 7',
			},
		},
		'r1bqk2r/1pppbppp/p1n2n2/8/B2pP3/5N2/PPP2PPP/RNBQR1K1 b kq - 1 7' => {
			eco => 'C84l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. Re1

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1'),
			moves => {
				'b7b5' => 'r1bqk2r/2ppbppp/p1n2n2/1p6/B2pP3/5N2/PPP2PPP/RNBQR1K1 w kq b6 0 8',
				'e8g8' => 'r1bq1rk1/1pppbppp/p1n2n2/8/B2pP3/5N2/PPP2PPP/RNBQR1K1 w - - 2 8',
			},
		},
		'r1bqk2r/1pppbppp/p1n5/4P3/B2Nn3/8/PPP2PPP/RNBQ1RK1 b kq - 0 8' => {
			eco => 'C84i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4 8. Nxd4

			variation => N__('Spanish: Closed, Centre Attack, 7. e5 Ne4 8.Nxd4'),
			moves => {
				'c6d4' => 'r1bqk2r/1pppbppp/p7/4P3/B2nn3/8/PPP2PPP/RNBQ1RK1 w kq - 0 9',
				'e8g8' => 'r1bq1rk1/1pppbppp/p1n5/4P3/B2Nn3/8/PPP2PPP/RNBQ1RK1 w - - 1 9',
			},
		},
		'r1bqk2r/1pppbppp/p1n5/4P3/B2pn3/2P2N2/PP3PPP/RNBQ1RK1 b kq - 0 8' => {
			eco => 'C84h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4 8. c3

			variation => N__('Spanish: Closed, Centre Attack, Basque Gambit'),
			moves => {
				'd4c3' => 'r1bqk2r/1pppbppp/p1n5/4P3/B3n3/2p2N2/PP3PPP/RNBQ1RK1 w kq - 0 9',
			},
		},
		'r1bqk2r/1pppbppp/p1n5/4P3/B2pn3/5N2/PPP2PPP/RNBQ1RK1 w kq - 1 8' => {
			eco => 'C84h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4

			variation => N__('Spanish: Closed, Centre Attack, 7. e5 Ne4'),
			moves => {
				'b2b4' => 'r1bqk2r/1pppbppp/p1n5/4P3/BP1pn3/5N2/P1P2PPP/RNBQ1RK1 b kq b3 0 8',
				'c2c3' => 'r1bqk2r/1pppbppp/p1n5/4P3/B2pn3/2P2N2/PP3PPP/RNBQ1RK1 b kq - 0 8',
				'f3d4' => 'r1bqk2r/1pppbppp/p1n5/4P3/B2Nn3/8/PPP2PPP/RNBQ1RK1 b kq - 0 8',
			},
		},
		'r1bqk2r/1pppbppp/p1n5/4P3/B3n3/2p2N2/PP3PPP/RNBQ1RK1 w kq - 0 9' => {
			eco => 'C84h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4 8. c3 dxc3

			variation => N__('Spanish: Closed, Centre Attack, Basque Gambit Accepted'),
			moves => {
			},
		},
		'r1bqk2r/1pppbppp/p1n5/4P3/BP1pn3/5N2/P1P2PPP/RNBQ1RK1 b kq b3 0 8' => {
			eco => 'C84h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4 8. b4

			variation => N__('Spanish: Closed, Centre Attack, 7. e5 Ne4 8. b4'),
			moves => {
			},
		},
		'r1bqk2r/1pppbppp/p1n5/4p3/B2Pn3/5N2/PPP2PPP/RNBQ1RK1 w kq - 1 7' => {
			eco => 'C80h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Nxe4 6. d4 Be7

			variation => N__('Spanish: Open, 6. d4 Be7'),
			moves => {
			},
		},
		'r1bqk2r/1pppbppp/p7/4P3/B2nn3/8/PPP2PPP/RNBQ1RK1 w kq - 0 9' => {
			eco => 'C84j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. e5 Ne4 8. Nxd4 Nxd4

			variation => N__('Spanish: Closed, Centre Attack, 7. e5 Ne4 8.Nxd4 Nxd4'),
			moves => {
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p1Np3/4P3/1B3N2/PPPP1PPP/R1BQ1RK1 b kq - 1 8' => {
			eco => 'C84d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Nc3 b5 7. Bb3 d6 8. Nd5

			variation => N__('Spanish: Closed, 6. Nc3 b5 7. Bb3 d6 8. Nd5'),
			moves => {
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/3PP3/1B3N2/PPP1QPPP/RNB1K2R w KQkq - 0 8' => {
			eco => 'C77u',
                        # TRANSLATORS:

			variation => N__('Spanish: Wormald Attack, 5... b5 6. Bb3 Be7'),
			moves => {
				'c2c3' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/3PP3/1BP2N2/PP2QPPP/RNB1K2R b KQkq - 0 8',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/3PP3/1B3N2/PPP2PPP/RNBQR1K1 b kq d3 0 8' => {
			eco => 'C88c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 d6 8. d4

			variation => N__('Spanish: Closed, 7... d6 8. d4'),
			moves => {
				'c6d4' => 'r1bqk2r/2p1bppp/p2p1n2/1p2p3/3nP3/1B3N2/PPP2PPP/RNBQR1K1 w kq - 0 9',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/3PP3/1BP2N2/PP2QPPP/RNB1K2R b KQkq - 0 8' => {
			eco => 'C77u',
                        # TRANSLATORS:

			variation => N__('Spanish: Wormald Attack, 5... b5 6. Bb3 Be7'),
			moves => {
				'c8g4' => 'r2qk2r/2p1bppp/p1np1n2/1p2p3/3PP1b1/1BP2N2/PP2QPPP/RNB1K2R w KQkq - 1 9',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 w kq - 0 8' => {
			eco => 'C88c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 d6

			variation => N__('Spanish: Closed 7... d6'),
			moves => {
				'c2c3' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 b kq - 0 8',
				'd2d4' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/3PP3/1B3N2/PPP2PPP/RNBQR1K1 b kq d3 0 8',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB2RK1 w kq - 0 8' => {
			eco => 'C86c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 d6

			variation => N__('Spanish: Worrall Attack, 7... d6'),
			moves => {
				'c2c3' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 b kq - 0 8',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1BN2N2/PPPP1PPP/R1BQ1RK1 w kq - 0 8' => {
			eco => 'C84c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Nc3 b5 7. Bb3 d6

			variation => N__('Spanish: Closed, 6. Nc3 b5 7. Bb3 d6'),
			moves => {
				'c3d5' => 'r1bqk2r/2p1bppp/p1np1n2/1p1Np3/4P3/1B3N2/PPPP1PPP/R1BQ1RK1 b kq - 1 8',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 b kq - 0 8' => {
			eco => 'C88d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 d6 8. c3

			variation => N__('Spanish: Closed 7... d6 8. c3'),
			moves => {
				'c6a5' => 'r1bqk2r/2p1bppp/p2p1n2/np2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 w kq - 1 9',
				'c8g4' => 'r2qk2r/2p1bppp/p1np1n2/1p2p3/4P1b1/1BP2N2/PP1P1PPP/RNBQR1K1 w kq - 1 9',
			},
		},
		'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1BP2N2/PP1PQPPP/RNB2RK1 b kq - 0 8' => {
			eco => 'C86d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3 d6 8. c3

			variation => N__('Spanish: Worrall Attack, 7... d6 8. c3'),
			moves => {
			},
		},
		'r1bqk2r/2p1bppp/p1pp1n2/4p3/3PP3/5N2/PPP2PPP/RNBQR1K1 b kq d3 0 8' => {
			eco => 'C87c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. Bxc6+ bxc6 8. d4

			variation => N__('Spanish: Closed, Averbakh, 7.Bxc6+ bxc6 8. d4'),
			moves => {
				'e5d4' => 'r1bqk2r/2p1bppp/p1pp1n2/8/3pP3/5N2/PPP2PPP/RNBQR1K1 w kq - 0 9',
			},
		},
		'r1bqk2r/2p1bppp/p1pp1n2/4p3/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq - 0 8' => {
			eco => 'C87c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, Averbakh, 7.Bxc6+'),
			moves => {
				'd2d4' => 'r1bqk2r/2p1bppp/p1pp1n2/4p3/3PP3/5N2/PPP2PPP/RNBQR1K1 b kq d3 0 8',
			},
		},
		'r1bqk2r/2p1bppp/p1pp1n2/8/3NP3/8/PPP2PPP/RNBQR1K1 b kq - 0 9' => {
			eco => 'C87d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. Bxc6+ bxc6 8. d4 exd4 9. Nxd4

			variation => N__('Spanish: Closed, Averbakh, 7.Bxc6+ bxc6 8. d4 exd4'),
			moves => {
			},
		},
		'r1bqk2r/2p1bppp/p1pp1n2/8/3pP3/5N2/PPP2PPP/RNBQR1K1 w kq - 0 9' => {
			eco => 'C87d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 d6 7. Bxc6+ bxc6 8. d4 exd4

			variation => N__('Spanish: Closed, Averbakh, 7.Bxc6+ bxc6 8. d4 exd4'),
			moves => {
				'f3d4' => 'r1bqk2r/2p1bppp/p1pp1n2/8/3NP3/8/PPP2PPP/RNBQR1K1 b kq - 0 9',
			},
		},
		'r1bqk2r/2p1bppp/p2p1n2/1p2p3/3NP3/1B6/PPP2PPP/RNBQR1K1 b kq - 0 9' => {
			eco => 'C88c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, 7... d6 8. d4'),
			moves => {
				'e5d4' => 'r1bqk2r/2p1bppp/p2p1n2/1p6/3pP3/1B6/PPP2PPP/RNBQR1K1 w kq - 0 10',
			},
		},
		'r1bqk2r/2p1bppp/p2p1n2/1p2p3/3nP3/1B3N2/PPP2PPP/RNBQR1K1 w kq - 0 9' => {
			eco => 'C88c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, 7... d6 8. d4'),
			moves => {
				'f3d4' => 'r1bqk2r/2p1bppp/p2p1n2/1p2p3/3NP3/1B6/PPP2PPP/RNBQR1K1 b kq - 0 9',
			},
		},
		'r1bqk2r/2p1bppp/p2p1n2/1p6/3QP3/1B6/PPP2PPP/RNB1R1K1 b kq - 0 10' => {
			eco => 'C88c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, 7... d6 8. d4'),
			moves => {
				'c7c5' => 'r1bqk2r/4bppp/p2p1n2/1pp5/3QP3/1B6/PPP2PPP/RNB1R1K1 w kq c6 0 11',
			},
		},
		'r1bqk2r/2p1bppp/p2p1n2/1p6/3pP3/1B6/PPP2PPP/RNBQR1K1 w kq - 0 10' => {
			eco => 'C88c',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed, 7... d6 8. d4'),
			moves => {
				'd1d4' => 'r1bqk2r/2p1bppp/p2p1n2/1p6/3QP3/1B6/PPP2PPP/RNB1R1K1 b kq - 0 10',
			},
		},
		'r1bqk2r/2p1bppp/p2p1n2/np2p3/4P3/1BP2N2/PP1P1PPP/RNBQR1K1 w kq - 1 9' => {
			eco => 'C88e',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 d6 8. c3 Na5

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'b3c2' => 'r1bqk2r/2p1bppp/p2p1n2/np2p3/4P3/2P2N2/PPBP1PPP/RNBQR1K1 b kq - 2 9',
			},
		},
		'r1bqk2r/2p1bppp/p2p1n2/np2p3/4P3/2P2N2/PPBP1PPP/RNBQR1K1 b kq - 2 9' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'c7c5' => 'r1bqk2r/4bppp/p2p1n2/npp1p3/4P3/2P2N2/PPBP1PPP/RNBQR1K1 w kq c6 0 10',
			},
		},
		'r1bqk2r/2p2ppp/p1np1n2/1pb1p3/3PP3/1BP2N2/PP3PPP/RNBQ1RK1 b kq d3 0 8' => {
			eco => 'C78k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. c3 d6 8. d4

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7. c3 d6 8. d4'),
			moves => {
			},
		},
		'r1bqk2r/2p2ppp/p1np1n2/1pb1p3/4P3/1BP2N2/PP1P1PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C78j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. c3 d6

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7. c3 d6'),
			moves => {
				'd2d4' => 'r1bqk2r/2p2ppp/p1np1n2/1pb1p3/3PP3/1BP2N2/PP3PPP/RNBQ1RK1 b kq d3 0 8',
			},
		},
		'r1bqk2r/2p2ppp/p4n2/1pbPR1N1/3n4/1B6/PPPP1PPP/RNBQ2K1 b kq - 0 10' => {
			eco => 'C78f',
                        # TRANSLATORS:

			variation => N__('Spanish: 5.O-O b5 6. Bb3 d6'),
			moves => {
				'e8f8' => 'r1bq1k1r/2p2ppp/p4n2/1pbPR1N1/3n4/1B6/PPPP1PPP/RNBQ2K1 w - - 1 11',
			},
		},
		'r1bqk2r/2p2ppp/p4n2/1pbPp1N1/3n4/1B6/PPPP1PPP/RNBQR1K1 w kq - 3 10' => {
			eco => 'C78f',
                        # TRANSLATORS:

			variation => N__('Spanish: 5.O-O b5 6. Bb3 d6'),
			moves => {
				'e1e5' => 'r1bqk2r/2p2ppp/p4n2/1pbPR1N1/3n4/1B6/PPPP1PPP/RNBQ2K1 b kq - 0 10',
			},
		},
		'r1bqk2r/2pp1ppp/p1n2n2/1pb1N3/4P3/1B6/PPPP1PPP/RNBQ1RK1 b kq - 0 7' => {
			eco => 'C78i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. Nxe5

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7.Nxe5'),
			moves => {
				'c6e5' => 'r1bqk2r/2pp1ppp/p4n2/1pb1n3/4P3/1B6/PPPP1PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/4P3/1B3N2/PPPP1PPP/RNBQ1RK1 w kq - 2 7' => {
			eco => 'C78h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5'),
			moves => {
				'a2a4' => 'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/P3P3/1B3N2/1PPP1PPP/RNBQ1RK1 b kq a3 0 7',
				'c2c3' => 'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/4P3/1BP2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 7',
				'f3e5' => 'r1bqk2r/2pp1ppp/p1n2n2/1pb1N3/4P3/1B6/PPPP1PPP/RNBQ1RK1 b kq - 0 7',
			},
		},
		'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/4P3/1B3N2/PPPPQPPP/RNB1K2R w KQkq - 2 7' => {
			eco => 'C77s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. Qe2 b5 6. Bb3 Bc5

			variation => N__('Spanish: Wormald Attack, 5... b5 6. Bb3 Bc5'),
			moves => {
				'c2c3' => 'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/4P3/1BP2N2/PP1PQPPP/RNB1K2R b KQkq - 0 7',
			},
		},
		'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/4P3/1BP2N2/PP1P1PPP/RNBQ1RK1 b kq - 0 7' => {
			eco => 'C78j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. c3

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7. c3'),
			moves => {
				'd7d6' => 'r1bqk2r/2p2ppp/p1np1n2/1pb1p3/4P3/1BP2N2/PP1P1PPP/RNBQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/4P3/1BP2N2/PP1PQPPP/RNB1K2R b KQkq - 0 7' => {
			eco => 'C77t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. Qe2 b5 6. Bb3 Bc5 7. c3

			variation => N__('Spanish: Wormald Attack, 5... b5 6. Bb3 Bc5 7. c3'),
			moves => {
			},
		},
		'r1bqk2r/2pp1ppp/p1n2n2/1pb1p3/P3P3/1B3N2/1PPP1PPP/RNBQ1RK1 b kq a3 0 7' => {
			eco => 'C78l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. a4

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7. a4'),
			moves => {
				'a8b8' => '1rbqk2r/2pp1ppp/p1n2n2/1pb1p3/P3P3/1B3N2/1PPP1PPP/RNBQ1RK1 w k - 1 8',
			},
		},
		'r1bqk2r/2pp1ppp/p4n2/1pb1n3/3PP3/1B6/PPP2PPP/RNBQ1RK1 b kq d3 0 8' => {
			eco => 'C78i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Bc5 7. Nxe5 Nxe5 8. d4

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7.Nxe5'),
			moves => {
			},
		},
		'r1bqk2r/2pp1ppp/p4n2/1pb1n3/4P3/1B6/PPPP1PPP/RNBQ1RK1 w kq - 0 8' => {
			eco => 'C78i',
                        # TRANSLATORS:

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Bc5 7.Nxe5'),
			moves => {
				'd2d4' => 'r1bqk2r/2pp1ppp/p4n2/1pb1n3/3PP3/1B6/PPP2PPP/RNBQ1RK1 b kq d3 0 8',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2P3/B2p4/5N2/PPP2PPP/RNBQR1K1 b kq - 0 8' => {
			eco => 'C84n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. Re1 b5 8. e5

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1 b5 8. e5'),
			moves => {
				'c6e5' => 'r1bqk2r/2ppbppp/p4n2/1p2n3/B2p4/5N2/PPP2PPP/RNBQR1K1 w kq - 0 9',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/3PP3/1B3N2/PPP1QPPP/RNB1K2R b KQkq d3 0 7' => {
			eco => 'C77u',
                        # TRANSLATORS:

			variation => N__('Spanish: Wormald Attack, 5... b5 6. Bb3 Be7'),
			moves => {
				'd7d6' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/3PP3/1B3N2/PPP1QPPP/RNB1K2R w KQkq - 0 8',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/3PP3/1B3N2/PPP2PPP/RNBQ1RK1 b kq d3 0 7' => {
			eco => 'C78g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Be7 7. d4

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Be7'),
			moves => {
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B1P1N2/PPP2PPP/RNBQ1RK1 b kq - 1 7' => {
			eco => 'C84f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d3 b5 7. Bb3

			variation => N__('Spanish: Closed, 6. d3 b5'),
			moves => {
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQ1RK1 w kq - 2 7' => {
			eco => 'C78g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O b5 6. Bb3 Be7

			variation => N__('Spanish: 5.O-O b5 6. Bb3 Be7'),
			moves => {
				'd2d4' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/3PP3/1B3N2/PPP2PPP/RNBQ1RK1 b kq d3 0 7',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 b kq - 1 7' => {
			eco => 'C88a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3

			variation => N__('Spanish: Closed, 6... b5 7. Bb3'),
			moves => {
				'c8b7' => 'r2qk2r/1bppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 w kq - 2 8',
				'd7d6' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 w kq - 0 8',
				'e8g8' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 w - - 2 8',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB1K2R w KQkq - 2 7' => {
			eco => 'C77u',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. Qe2 b5 6. Bb3 Be7

			variation => N__('Spanish: Wormald Attack, 5... b5 6. Bb3 Be7'),
			moves => {
				'd2d4' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/3PP3/1B3N2/PPP1QPPP/RNB1K2R b KQkq d3 0 7',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB2RK1 b kq - 1 7' => {
			eco => 'C86b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5 7. Bb3

			variation => N__('Spanish: Worrall Attack, 6... b5 7. Bb3'),
			moves => {
				'd7d6' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB2RK1 w kq - 0 8',
				'e8g8' => 'r1bq1rk1/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB2RK1 w - - 2 8',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1BN2N2/PPPP1PPP/R1BQ1RK1 b kq - 1 7' => {
			eco => 'C84c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Nc3 b5 7. Bb3

			variation => N__('Spanish: Closed, 6. Nc3 b5 7. Bb3'),
			moves => {
				'd7d6' => 'r1bqk2r/2p1bppp/p1np1n2/1p2p3/4P3/1BN2N2/PPPP1PPP/R1BQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/2N2N2/PPPP1PPP/R1BQ1RK1 w kq b6 0 7' => {
			eco => 'C84c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Nc3 b5

			variation => N__('Spanish: Closed, 6. Nc3 b5'),
			moves => {
				'a4b3' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1BN2N2/PPPP1PPP/R1BQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/3P1N2/PPP2PPP/RNBQ1RK1 w kq b6 0 7' => {
			eco => 'C84f',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d3 b5

			variation => N__('Spanish: Closed, 6. d3 b5'),
			moves => {
				'a4b3' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B1P1N2/PPP2PPP/RNBQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/5N2/PPPP1PPP/RNBQR1K1 w kq b6 0 7' => {
			eco => 'C88a',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5

			variation => N__('Spanish: Closed, 6... b5'),
			moves => {
				'a4b3' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPP1PPP/RNBQR1K1 b kq - 1 7',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p2p3/B3P3/5N2/PPPPQPPP/RNB2RK1 w kq b6 0 7' => {
			eco => 'C86b',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Qe2 b5

			variation => N__('Spanish: Worrall Attack, 6... b5'),
			moves => {
				'a4b3' => 'r1bqk2r/2ppbppp/p1n2n2/1p2p3/4P3/1B3N2/PPPPQPPP/RNB2RK1 b kq - 1 7',
			},
		},
		'r1bqk2r/2ppbppp/p1n2n2/1p6/B2pP3/5N2/PPP2PPP/RNBQR1K1 w kq b6 0 8' => {
			eco => 'C84m',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. Re1 b5

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1 b5'),
			moves => {
				'e4e5' => 'r1bqk2r/2ppbppp/p1n2n2/1p2P3/B2p4/5N2/PPP2PPP/RNBQR1K1 b kq - 0 8',
			},
		},
		'r1bqk2r/2ppbppp/p4n2/1p2R3/B2p4/5N2/PPP2PPP/RNBQ2K1 b kq - 0 9' => {
			eco => 'C84n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. Re1 b5 8. e5 Nxe5 9. Rxe5

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1 b5 8. e5 Nxe5 9.Rxe5'),
			moves => {
			},
		},
		'r1bqk2r/2ppbppp/p4n2/1p2n3/B2p4/5N2/PPP2PPP/RNBQR1K1 w kq - 0 9' => {
			eco => 'C84n',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. d4 exd4 7. Re1 b5 8. e5 Nxe5

			variation => N__('Spanish: Closed, Centre Attack, 7. Re1 b5 8. e5 Nxe5'),
			moves => {
				'e1e5' => 'r1bqk2r/2ppbppp/p4n2/1p2R3/B2p4/5N2/PPP2PPP/RNBQ2K1 b kq - 0 9',
			},
		},
		'r1bqk2r/4bppp/p1n1pn2/1pp5/3P4/1BN1PN2/PP2QPPP/R1B2RK1 w kq - 4 10' => {
			eco => 'D28n',
                        # TRANSLATORS: 1. d4 d5 2. c4 dxc4 3. Nf3 Nf6 4. e3 e6 5. Bxc4 c5 6. O-O a6 7. Qe2 b5 8. Bb3 Nc6 9. Nc3 Be7

			variation => N__('QGA: Classical, 7. Qe2 b5 8. Bb3 Nc6 9. Nc3 Be7'),
			moves => {
			},
		},
		'r1bqk2r/4bppp/p1np1B2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R b KQkq - 0 10' => {
			eco => 'B33m',
                        # TRANSLATORS:

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7'),
			moves => {
				'e7f6' => 'r1bqk2r/5ppp/p1np1b2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R w KQkq - 0 11',
			},
		},
		'r1bqk2r/4bppp/p1np1n2/1p1Np1B1/4P3/N7/PPP2PPP/R2QKB1R w KQkq - 2 10' => {
			eco => 'B33m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7'),
			moves => {
				'g5f6' => 'r1bqk2r/4bppp/p1np1B2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R b KQkq - 0 10',
			},
		},
		'r1bqk2r/4bppp/p2p1n2/1pp5/3QP3/1B6/PPP2PPP/RNB1R1K1 w kq c6 0 11' => {
			eco => 'C88c',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 a6 4. Ba4 Nf6 5. O-O Be7 6. Re1 b5 7. Bb3 d6 8. d4 Nxd4 9. Nxd4 exd4 10. Qxd4 c5

			variation => N__('Spanish: Closed, Noah\'s Ark Trap'),
			moves => {
			},
		},
		'r1bqk2r/4bppp/p2p1n2/npp1p3/3PP3/2P2N2/PPB2PPP/RNBQR1K1 b kq d3 0 10' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'd8c7' => 'r1b1k2r/2q1bppp/p2p1n2/npp1p3/3PP3/2P2N2/PPB2PPP/RNBQR1K1 w kq - 1 11',
			},
		},
		'r1bqk2r/4bppp/p2p1n2/npp1p3/4P3/2P2N2/PPBP1PPP/RNBQR1K1 w kq c6 0 10' => {
			eco => 'C88e',
                        # TRANSLATORS:

			variation => N__('Spanish: Closed 7... d6 8. c3 Na5'),
			moves => {
				'd2d4' => 'r1bqk2r/4bppp/p2p1n2/npp1p3/3PP3/2P2N2/PPB2PPP/RNBQR1K1 b kq d3 0 10',
			},
		},
		'r1bqk2r/4bppp/ppnppn2/8/2P1P3/N1N5/PP2BPPP/R1BQK2R w KQkq - 0 10' => {
			eco => 'B44u',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nb5 d6 6. c4 Nf6 7. N1c3 a6 8. Na3 Be7 9. Be2 b6

			variation => N__('Sicilian, Taimanov, Szen, 7.N1c3 a6 8. Na3 Be7 9. Be2 b6'),
			moves => {
			},
		},
		'r1bqk2r/5pbp/p1np1p2/1p1Np3/4P3/N2B4/PPP2PPP/R2QK2R b KQkq - 3 11' => {
			eco => 'B33t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 Bg7 11. Bd3

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10. Nd5 Bg7 11. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/5pbp/p1np1p2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R w KQkq - 2 11' => {
			eco => 'B33s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 Bg7

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10. Nd5 Bg7'),
			moves => {
				'f1d3' => 'r1bqk2r/5pbp/p1np1p2/1p1Np3/4P3/N2B4/PPP2PPP/R2QK2R b KQkq - 3 11',
			},
		},
		'r1bqk2r/5pbp/p1np4/1p1NpP2/8/N1P5/PP3PPP/R2QKB1R b KQkq - 0 12' => {
			eco => 'B33v',
                        # TRANSLATORS:

			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7'),
			moves => {
				'c8f5' => 'r2qk2r/5pbp/p1np4/1p1Npb2/8/N1P5/PP3PPP/R2QKB1R w KQkq - 0 13',
			},
		},
		'r1bqk2r/5pbp/p1np4/1p1Npp2/4P3/N1P5/PP3PPP/R2QKB1R w KQkq - 1 12' => {
			eco => 'B33v',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3 Bg7

			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7'),
			moves => {
				'e4f5' => 'r1bqk2r/5pbp/p1np4/1p1NpP2/8/N1P5/PP3PPP/R2QKB1R b KQkq - 0 12',
			},
		},
		'r1bqk2r/5ppp/4pn2/1p2N3/1b1p4/3B1Q2/PP3PPP/R1B1K2R w KQkq - 2 14' => {
			eco => 'D49g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 a6 9. e4 c5 10. e5 cxd4 11. Nxb5 Nxe5 12. Nxe5 axb5 13. Qf3 Bb4+

			variation => N__('Semi-Slav: Meran, Sozin, Stahlberg Attack, 13... Bb4'),
			moves => {
			},
		},
		'r1bqk2r/5ppp/p1np1b2/1p1Np3/4P3/N1P5/PP3PPP/R2QKB1R b KQkq - 0 11' => {
			eco => 'B33n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3'),
			moves => {
				'e8g8' => 'r1bq1rk1/5ppp/p1np1b2/1p1Np3/4P3/N1P5/PP3PPP/R2QKB1R w KQ - 1 12',
				'f6g5' => 'r1bqk2r/5ppp/p1np4/1p1Np1b1/4P3/N1P5/PP3PPP/R2QKB1R w KQkq - 1 12',
			},
		},
		'r1bqk2r/5ppp/p1np1b2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R w KQkq - 0 11' => {
			eco => 'B33m',
                        # TRANSLATORS:

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7'),
			moves => {
				'c2c3' => 'r1bqk2r/5ppp/p1np1b2/1p1Np3/4P3/N1P5/PP3PPP/R2QKB1R b KQkq - 0 11',
			},
		},
		'r1bqk2r/5ppp/p1np4/1p1Np1b1/4P3/N1P5/PP3PPP/R2QKB1R w KQkq - 1 12' => {
			eco => 'B33p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 Bg5

			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 Bg5'),
			moves => {
			},
		},
		'r1bqk2r/bp3ppp/p1n1pn2/3p4/1PP5/P1N1PN2/5PPP/R1BQKB1R w KQkq - 1 9' => {
			eco => 'D40r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. e3 Nc6 6. a3 a6 7. dxc5 Bxc5 8. b4 Ba7

			variation => N__('QGD: Semi-Tarrasch, 5. e3 Nc6 6. a3 a6 7.dxc5 Bxc5 8. b4 Ba7'),
			moves => {
			},
		},
		'r1bqk2r/p1p2ppp/2p2n2/3P4/1b6/2NB4/PPP2PPP/R1BQK2R b KQkq - 0 8' => {
			eco => 'C47t',
                        # TRANSLATORS:

			variation => N__('Four Knights: Scotch, Main Line, 7... d5'),
			moves => {
				'c6d5' => 'r1bqk2r/p1p2ppp/5n2/3p4/1b6/2NB4/PPP2PPP/R1BQK2R w KQkq - 0 9',
			},
		},
		'r1bqk2r/p1p2ppp/2p2n2/3p4/1b2P3/2NB4/PPP2PPP/R1BQK2R w KQkq d6 0 8' => {
			eco => 'C47t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5

			variation => N__('Four Knights: Scotch, Main Line, 7... d5'),
			moves => {
				'e4d5' => 'r1bqk2r/p1p2ppp/2p2n2/3P4/1b6/2NB4/PPP2PPP/R1BQK2R b KQkq - 0 8',
			},
		},
		'r1bqk2r/p1p2ppp/5n2/3p4/1b6/2NB4/PPP2PPP/R1BQ1RK1 b kq - 1 9' => {
			eco => 'C47t',
                        # TRANSLATORS:

			variation => N__('Four Knights: Scotch, Main Line, 7... d5 8.exd5 cxd5'),
			moves => {
				'e8g8' => 'r1bq1rk1/p1p2ppp/5n2/3p4/1b6/2NB4/PPP2PPP/R1BQ1RK1 w - - 2 10',
			},
		},
		'r1bqk2r/p1p2ppp/5n2/3p4/1b6/2NB4/PPP2PPP/R1BQK2R w KQkq - 0 9' => {
			eco => 'C47t',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3 d5 8. exd5 cxd5

			variation => N__('Four Knights: Scotch, Main Line, 7... d5 8.exd5 cxd5'),
			moves => {
				'e1g1' => 'r1bqk2r/p1p2ppp/5n2/3p4/1b6/2NB4/PPP2PPP/R1BQ1RK1 b kq - 1 9',
			},
		},
		'r1bqk2r/p1pp1ppp/2p2n2/8/1b2P3/2N5/PPP2PPP/R1BQKB1R w KQkq - 0 7' => {
			eco => 'C47s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6

			variation => N__('Four Knights: Scotch, 4...exd4 5.Nxd4 Bb4 6.Nxc6 bxc6'),
			moves => {
				'f1d3' => 'r1bqk2r/p1pp1ppp/2p2n2/8/1b2P3/2NB4/PPP2PPP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1bqk2r/p1pp1ppp/2p2n2/8/1b2P3/2NB4/PPP2PPP/R1BQK2R b KQkq - 1 7' => {
			eco => 'C47s',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Nc3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Nxc6 bxc6 7. Bd3

			variation => N__('Four Knights: Scotch, Main Line'),
			moves => {
				'd7d5' => 'r1bqk2r/p1p2ppp/2p2n2/3p4/1b2P3/2NB4/PPP2PPP/R1BQK2R w KQkq d6 0 8',
			},
		},
		'r1bqk2r/p1ppbppp/2p5/4Pn2/8/5N2/PPP1QPPP/RNB2RK1 w kq - 1 9' => {
			eco => 'C67i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Be7 6. Qe2 Nd6 7. Bxc6 bxc6 8. dxe5 Nf5

			variation => N__('Spanish: Open Berlin, Cordel Variation'),
			moves => {
			},
		},
		'r1bqk2r/p1ppbppp/2pn4/4P3/8/5N2/PPP1QPPP/RNB2RK1 b kq - 0 8' => {
			eco => 'C67i',
                        # TRANSLATORS:

			variation => N__('Spanish: Open Berlin, 5... Be7 6. Qd2 Nd6'),
			moves => {
				'd6b7' => 'r1bqk2r/pnppbppp/2p5/4P3/8/5N2/PPP1QPPP/RNB2RK1 w kq - 1 9',
				'd6f5' => 'r1bqk2r/p1ppbppp/2p5/4Pn2/8/5N2/PPP1QPPP/RNB2RK1 w kq - 1 9',
			},
		},
		'r1bqk2r/p1ppbppp/2pn4/4p3/3P4/5N2/PPP1QPPP/RNB2RK1 w kq - 0 8' => {
			eco => 'C67i',
                        # TRANSLATORS:

			variation => N__('Spanish: Open Berlin, 5... Be7 6. Qd2 Nd6'),
			moves => {
				'd4e5' => 'r1bqk2r/p1ppbppp/2pn4/4P3/8/5N2/PPP1QPPP/RNB2RK1 b kq - 0 8',
			},
		},
		'r1bqk2r/p2n1ppp/2pbpn2/1p6/3P4/2NBPN2/PP3PPP/R1BQK2R w KQkq - 2 9' => {
			eco => 'D47h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 Bd6

			variation => N__('Semi-Slav: Meran, 8. Bd3 Bd6'),
			moves => {
			},
		},
		'r1bqk2r/p2nbppp/2p1pn2/8/1p1PN3/3BPN2/PP3PPP/R1BQ1RK1 b kq - 3 10' => {
			eco => 'D47l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 b4 9. Ne4 Be7 10. O-O

			variation => N__('Semi-Slav: Meran, Lundin, 9. Ne4 Be7 10.O-O'),
			moves => {
			},
		},
		'r1bqk2r/p2nbppp/2p1pn2/8/1p1PN3/3BPN2/PP3PPP/R1BQK2R w KQkq - 2 10' => {
			eco => 'D47l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 dxc4 7. Bxc4 b5 8. Bd3 b4 9. Ne4 Be7

			variation => N__('Semi-Slav: Meran, Lundin, 9. Ne4 Be7'),
			moves => {
				'e1g1' => 'r1bqk2r/p2nbppp/2p1pn2/8/1p1PN3/3BPN2/PP3PPP/R1BQ1RK1 b kq - 3 10',
			},
		},
		'r1bqk2r/p2p1ppp/2p1pn2/8/1bP1P3/2N5/PP3PPP/R1BQKB1R w KQkq - 0 8' => {
			eco => 'B44f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. c4 Nf6 6. Nc3 Bb4 7. Nxc6 bxc6

			variation => N__('Sicilian: Taimanov, 5. c4 Nf6 6. Nc3 Bb4 7.Nxc6 bxc6'),
			moves => {
			},
		},
		'r1bqk2r/p2pppbp/2p2np1/4P3/8/2N1B3/PPP2PPP/R2QKB1R b KQkq - 0 8' => {
			eco => 'B34r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6 bxc6 8. e5

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6 bxc6 8. e5'),
			moves => {
				'f6g8' => 'r1bqk1nr/p2pppbp/2p3p1/4P3/8/2N1B3/PPP2PPP/R2QKB1R w KQkq - 1 9',
			},
		},
		'r1bqk2r/p2pppbp/2p2np1/8/4P3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 0 8' => {
			eco => 'B34r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6 bxc6

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6 bxc6'),
			moves => {
				'e4e5' => 'r1bqk2r/p2pppbp/2p2np1/4P3/8/2N1B3/PPP2PPP/R2QKB1R b KQkq - 0 8',
			},
		},
		'r1bqk2r/p4pp1/2p2n1p/n1b1N3/4p3/8/PPPPBPPP/RNBQK2R w KQkq - 2 11' => {
			eco => 'C59d',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bc5

			variation => N__('Two Knights: Morphy, Steiner Variation'),
			moves => {
			},
		},
		'r1bqk2r/p4pp1/2pb1n1p/n3N3/3Pp3/8/PPP1BPPP/RNBQK2R b KQkq d3 0 11' => {
			eco => 'C59j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. d4

			variation => N__('Two Knights: Main Line, 11. d4'),
			moves => {
				'd8c7' => 'r1b1k2r/p1q2pp1/2pb1n1p/n3N3/3Pp3/8/PPP1BPPP/RNBQK2R w KQkq - 1 12',
				'e4d3' => 'r1bqk2r/p4pp1/2pb1n1p/n3N3/8/3p4/PPP1BPPP/RNBQK2R w KQkq - 0 12',
			},
		},
		'r1bqk2r/p4pp1/2pb1n1p/n3N3/4p3/8/PPPPBPPP/RNBQK2R w KQkq - 2 11' => {
			eco => 'C59g',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6

			variation => N__('Two Knights: Main Line 10... Bd6'),
			moves => {
				'd2d4' => 'r1bqk2r/p4pp1/2pb1n1p/n3N3/3Pp3/8/PPP1BPPP/RNBQK2R b KQkq d3 0 11',
				'f2f4' => 'r1bqk2r/p4pp1/2pb1n1p/n3N3/4pP2/8/PPPPB1PP/RNBQK2R b KQkq f3 0 11',
			},
		},
		'r1bqk2r/p4pp1/2pb1n1p/n3N3/4pP2/8/PPPPB1PP/RNBQK2R b KQkq f3 0 11' => {
			eco => 'C59h',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. f4

			variation => N__('Two Knights: Main Line, 11. f4'),
			moves => {
				'e4f3' => 'r1bqk2r/p4pp1/2pb1n1p/n3N3/8/5p2/PPPPB1PP/RNBQK2R w KQkq - 0 12',
			},
		},
		'r1bqk2r/p4pp1/2pb1n1p/n3N3/8/3p4/PPP1BPPP/RNBQK2R w KQkq - 0 12' => {
			eco => 'C59k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. d4 exd3

			variation => N__('Two Knights: Main Line, 11. d4 exd3'),
			moves => {
				'e5d3' => 'r1bqk2r/p4pp1/2pb1n1p/n7/8/3N4/PPP1BPPP/RNBQK2R b KQkq - 0 12',
			},
		},
		'r1bqk2r/p4pp1/2pb1n1p/n3N3/8/5p2/PPPPB1PP/RNBQK2R w KQkq - 0 12' => {
			eco => 'C59i',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bc4 Nf6 4. Ng5 d5 5. exd5 Na5 6. Bb5+ c6 7. dxc6 bxc6 8. Be2 h6 9. Nf3 e4 10. Ne5 Bd6 11. f4 exf3

			variation => N__('Two Knights: Main Line, 11. f4 exf3'),
			moves => {
			},
		},
		'r1bqk2r/p4pp1/2pb1n1p/n7/8/3N4/PPP1BPPP/RNBQK2R b KQkq - 0 12' => {
			eco => 'C59k',
                        # TRANSLATORS:

			variation => N__('Two Knights: Main Line, 11. d4 exd3'),
			moves => {
				'd8c7' => 'r1b1k2r/p1q2pp1/2pb1n1p/n7/8/3N4/PPP1BPPP/RNBQK2R w KQkq - 1 13',
			},
		},
		'r1bqk2r/pnppbppp/2p5/4P3/2P5/5N2/PP2QPPP/RNB2RK1 b kq c3 0 9' => {
			eco => 'C67j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Be7 6. Qe2 Nd6 7. Bxc6 bxc6 8. dxe5 Nb7 9. c4

			variation => N__('Spanish: Open Berlin, Zukertort Variation'),
			moves => {
			},
		},
		'r1bqk2r/pnppbppp/2p5/4P3/3N4/8/PPP1QPPP/RNB2RK1 b kq - 2 9' => {
			eco => 'C67k',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Be7 6. Qe2 Nd6 7. Bxc6 bxc6 8. dxe5 Nb7 9. Nd4

			variation => N__('Spanish: Open Berlin, Winawer Attack'),
			moves => {
			},
		},
		'r1bqk2r/pnppbppp/2p5/4P3/8/1P3N2/P1P1QPPP/RNB2RK1 b kq - 0 9' => {
			eco => 'C67j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Be7 6. Qe2 Nd6 7. Bxc6 bxc6 8. dxe5 Nb7 9. b3

			variation => N__('Spanish: Open Berlin, Pillsbury Variation'),
			moves => {
			},
		},
		'r1bqk2r/pnppbppp/2p5/4P3/8/2N2N2/PPP1QPPP/R1B2RK1 b kq - 2 9' => {
			eco => 'C67l',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Be7 6. Qe2 Nd6 7. Bxc6 bxc6 8. dxe5 Nb7 9. Nc3

			variation => N__('Spanish: Open Berlin, Main Line 9. Nc3 (Rio de Janerio)'),
			moves => {
			},
		},
		'r1bqk2r/pnppbppp/2p5/4P3/8/5N2/PPP1QPPP/RNB2RK1 w kq - 1 9' => {
			eco => 'C67j',
                        # TRANSLATORS: 1. e4 e5 2. Nf3 Nc6 3. Bb5 Nf6 4. O-O Nxe4 5. d4 Be7 6. Qe2 Nd6 7. Bxc6 bxc6 8. dxe5 Nb7

			variation => N__('Spanish: Open Berlin, 5... Be7 6. Qd2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7'),
			moves => {
				'b1c3' => 'r1bqk2r/pnppbppp/2p5/4P3/8/2N2N2/PPP1QPPP/R1B2RK1 b kq - 2 9',
				'b2b3' => 'r1bqk2r/pnppbppp/2p5/4P3/8/1P3N2/P1P1QPPP/RNB2RK1 b kq - 0 9',
				'c2c4' => 'r1bqk2r/pnppbppp/2p5/4P3/2P5/5N2/PP2QPPP/RNB2RK1 b kq c3 0 9',
				'f3d4' => 'r1bqk2r/pnppbppp/2p5/4P3/3N4/8/PPP1QPPP/RNB2RK1 b kq - 2 9',
			},
		},
		'r1bqk2r/pp1n1pbp/3p1np1/1BpP4/4PP2/2N5/PP4PP/R1BQK1NR w KQkq - 3 9' => {
			eco => 'A67b',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7

			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'e4e5' => 'r1bqk2r/pp1n1pbp/3p1np1/1BpPP3/5P2/2N5/PP4PP/R1BQK1NR b KQkq - 0 9',
			},
		},
		'r1bqk2r/pp1n1pbp/3p1np1/1BpPP3/5P2/2N5/PP4PP/R1BQK1NR b KQkq - 0 9' => {
			eco => 'A67b',
                        # TRANSLATORS:

			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'd6e5' => 'r1bqk2r/pp1n1pbp/5np1/1BpPp3/5P2/2N5/PP4PP/R1BQK1NR w KQkq - 0 10',
			},
		},
		'r1bqk2r/pp1n1pbp/4P1p1/1BpP3n/8/2N5/PP4PP/R1BQK1NR b KQkq - 0 11' => {
			eco => 'A67c',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7 9. e5 dxe5 10. fxe5 Nh5 11. e6

			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7 9. e5 dxe5 10.fxe5 Nh5 11. e6'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1pbp/5np1/1BpPP3/8/2N5/PP4PP/R1BQK1NR b KQkq - 0 10' => {
			eco => 'A67b',
                        # TRANSLATORS:

			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'f6h5' => 'r1bqk2r/pp1n1pbp/6p1/1BpPP2n/8/2N5/PP4PP/R1BQK1NR w KQkq - 1 11',
			},
		},
		'r1bqk2r/pp1n1pbp/5np1/1BpPp3/5P2/2N5/PP4PP/R1BQK1NR w KQkq - 0 10' => {
			eco => 'A67b',
                        # TRANSLATORS:

			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'f4e5' => 'r1bqk2r/pp1n1pbp/5np1/1BpPP3/8/2N5/PP4PP/R1BQK1NR b KQkq - 0 10',
			},
		},
		'r1bqk2r/pp1n1pbp/6p1/1BpPP2n/8/2N5/PP4PP/R1BQK1NR w KQkq - 1 11' => {
			eco => 'A67b',
                        # TRANSLATORS:

			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'e5e6' => 'r1bqk2r/pp1n1pbp/4P1p1/1BpP3n/8/2N5/PP4PP/R1BQK1NR b KQkq - 0 11',
			},
		},
		'r1bqk2r/pp1n1pp1/2pbp2p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R b KQkq - 0 10' => {
			eco => 'B17m',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4 10. Qxe4

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3, Main Line 10.Qxe4'),
			moves => {
				'd7f6' => 'r1bqk2r/pp3pp1/2pbpn1p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R w KQkq - 1 11',
				'd8c7' => 'r1b1k2r/ppqn1pp1/2pbp2p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R w KQkq - 1 11',
			},
		},
		'r1bqk2r/pp1n1pp1/2pbp2p/8/3Pn3/3B1N2/PPP1QPPP/R1B1K2R w KQkq - 0 10' => {
			eco => 'B17l',
                        # TRANSLATORS:

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'e2e4' => 'r1bqk2r/pp1n1pp1/2pbp2p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R b KQkq - 0 10',
			},
		},
		'r1bqk2r/pp1n1pp1/2pbpn1p/3p4/2PP2P1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 0 8' => {
			eco => 'D45l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. g4 h6

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. g4 h6'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1pp1/2pbpn1p/6N1/3P4/3B1N2/PPP1QPPP/R1B1K2R w KQkq - 0 9' => {
			eco => 'B17l',
                        # TRANSLATORS:

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'g5e4' => 'r1bqk2r/pp1n1pp1/2pbpn1p/8/3PN3/3B1N2/PPP1QPPP/R1B1K2R b KQkq - 1 9',
			},
		},
		'r1bqk2r/pp1n1pp1/2pbpn1p/8/3PN3/3B1N2/PPP1QPPP/R1B1K2R b KQkq - 1 9' => {
			eco => 'B17l',
                        # TRANSLATORS:

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'f6e4' => 'r1bqk2r/pp1n1pp1/2pbp2p/8/3Pn3/3B1N2/PPP1QPPP/R1B1K2R w KQkq - 0 10',
			},
		},
		'r1bqk2r/pp1n1pp1/3p1n1p/2pPp3/2P4B/2P1PP2/P5PP/R2QKBNR w KQkq - 1 10' => {
			eco => 'E31o',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Bg5 h6 5. Bh4 c5 6. d5 d6 7. e3 Bxc3+ 8. bxc3 e5 9. f3 Nbd7

			variation => N__('Nimzo-Indian: Leningrad, Main Line, 8... e5 9. f3 Nbd7'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2n1p3/2bpP3/3N1P2/2N1B3/PPP3PP/R2QKB1R w KQkq - 1 9' => {
			eco => 'C11n',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. e5 Nfd7 5. f4 c5 6. Nf3 Nc6 7. Be3 cxd4 8. Nxd4 Bc5

			variation => N__('French: Steinitz, Boleslavsky, 7...cxd4 8.Nxd4 Bc5'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2n1p3/2bpP3/5P2/P1N5/1PP3PP/R1BQKBNR w KQkq - 0 8' => {
			eco => 'C11g',
                        # TRANSLATORS:

			variation => N__('French: Steinitz, 6.dxc5 Nc6'),
			moves => {
				'd1g4' => 'r1bqk2r/pp1n1ppp/2n1p3/2bpP3/5PQ1/P1N5/1PP3PP/R1B1KBNR b KQkq - 1 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2n1p3/2bpP3/5PQ1/P1N5/1PP3PP/R1B1KBNR b KQkq - 1 8' => {
			eco => 'C11g',
                        # TRANSLATORS:

			variation => N__('French: Steinitz, 6.dxc5 Nc6'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1n1ppp/2n1p3/2bpP3/5PQ1/P1N5/1PP3PP/R1B1KBNR w KQ - 2 9',
			},
		},
		'r1bqk2r/pp1n1ppp/2p1pn2/3p4/1bPP2P1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 1 8' => {
			eco => 'D45n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. g4 Bb4

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. g4 Bb4'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2p1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 7' => {
			eco => 'D46b',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bb4 7. O-O

			variation => N__('Semi-Slav: Romih, 7.O-O'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1n1ppp/2p1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2p1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQK2R w KQkq - 3 7' => {
			eco => 'D46b',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bb4

			variation => N__('Semi-Slav: Romih Variation'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1n1ppp/2p1pn2/3p4/1bPP4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 7',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbp3/3p4/2PP2n1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 0 8' => {
			eco => 'D45o',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. g4 Nxg4

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. g4 Nxg4'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP2P1/2N1PN2/PPQ2P1P/R1B1KB1R b KQkq g3 0 7' => {
			eco => 'D45l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. g4

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. g4'),
			moves => {
				'd5c4' => 'r1bqk2r/pp1n1ppp/2pbpn2/8/2pP2P1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 0 8',
				'd6b4' => 'r1bqk2r/pp1n1ppp/2p1pn2/3p4/1bPP2P1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 1 8',
				'f6g4' => 'r1bqk2r/pp1n1ppp/2pbp3/3p4/2PP2n1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 0 8',
				'h7h6' => 'r1bqk2r/pp1n1pp1/2pbpn1p/3p4/2PP2P1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q2PPP/R1B1KB1R b KQkq - 0 7' => {
			eco => 'D45p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. b3

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. b3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q2PPP/R1B1KB1R w KQ - 1 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B1K2R b KQkq - 4 7' => {
			eco => 'D45v',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. Be2

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. Be2'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B1K2R w KQ - 5 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ2PPP/R1B1KB1R w KQkq - 3 7' => {
			eco => 'D45k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6

			variation => N__('Semi-Slav: 6. Qc2 Bd6'),
			moves => {
				'b2b3' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/1PN1PN2/P1Q2PPP/R1B1KB1R b KQkq - 0 7',
				'c1d2' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQB1PPP/R3KB1R b KQkq - 4 7',
				'e3e4' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PPP3/2N2N2/PPQ2PPP/R1B1KB1R b KQkq - 0 7',
				'f1e2' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQ1BPPP/R1B1K2R b KQkq - 4 7',
				'g2g4' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP2P1/2N1PN2/PPQ2P1P/R1B1KB1R b KQkq g3 0 7',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2N1PN2/PPQB1PPP/R3KB1R b KQkq - 4 7' => {
			eco => 'D45u',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. Bd2

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. Bd2'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 7' => {
			eco => 'D46n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. O-O

			variation => N__('Semi-Slav: Chigorin, 7.O-O'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQK2R w KQkq - 3 7' => {
			eco => 'D46e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6

			variation => N__('Semi-Slav: Chigorin Defence'),
			moves => {
				'd1c2' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PPQ2PPP/R1B1K2R b KQkq - 4 7',
				'e1g1' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 7',
				'e3e4' => 'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PPP3/2NB1N2/PP3PPP/R1BQK2R b KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PPQ2PPP/R1B1K2R b KQkq - 4 7' => {
			eco => 'D46f',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2

			variation => N__('Semi-Slav: Chigorin, 7. Qc2'),
			moves => {
				'd5c4' => 'r1bqk2r/pp1n1ppp/2pbpn2/8/2pP4/2NBPN2/PPQ2PPP/R1B1K2R w KQkq - 0 8',
				'e8g8' => 'r1bq1rk1/pp1n1ppp/2pbpn2/3p4/2PP4/2NBPN2/PPQ2PPP/R1B1K2R w KQ - 5 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PPP3/2N2N2/PPQ2PPP/R1B1KB1R b KQkq - 0 7' => {
			eco => 'D45k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. e4

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. e4'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/3p4/2PPP3/2NB1N2/PP3PPP/R1BQK2R b KQkq - 0 7' => {
			eco => 'D46e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. e4

			variation => N__('Semi-Slav: Chigorin, 7. e4'),
			moves => {
				'd5e4' => 'r1bqk2r/pp1n1ppp/2pbpn2/8/2PPp3/2NB1N2/PP3PPP/R1BQK2R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/6N1/3P4/3B1N2/PPP1QPPP/R1B1K2R b KQkq - 3 8' => {
			eco => 'B17l',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'h7h6' => 'r1bqk2r/pp1n1pp1/2pbpn1p/6N1/3P4/3B1N2/PPP1QPPP/R1B1K2R w KQkq - 0 9',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/6N1/3P4/3B1N2/PPP2PPP/R1BQK2R w KQkq - 2 8' => {
			eco => 'B17k',
                        # TRANSLATORS:

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6'),
			moves => {
				'd1e2' => 'r1bqk2r/pp1n1ppp/2pbpn2/6N1/3P4/3B1N2/PPP1QPPP/R1B1K2R b KQkq - 3 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B1K2R b KQkq - 0 8' => {
			eco => 'D46g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 dxc4 8. Bxc4

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 dxc4'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/8/2PPN3/3B1N2/PP3PPP/R1BQK2R b KQkq - 0 8' => {
			eco => 'D46e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. e4 dxe4 8. Nxe4

			variation => N__('Semi-Slav: Chigorin, 7. e4 dxe4 8.Nxe4'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/8/2PPp3/2NB1N2/PP3PPP/R1BQK2R w KQkq - 0 8' => {
			eco => 'D46e',
                        # TRANSLATORS:

			variation => N__('Semi-Slav: Chigorin, 7. e4'),
			moves => {
				'c3e4' => 'r1bqk2r/pp1n1ppp/2pbpn2/8/2PPN3/3B1N2/PP3PPP/R1BQK2R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/8/2pP2P1/2N1PN2/PPQ2P1P/R1B1KB1R w KQkq - 0 8' => {
			eco => 'D45m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Bd6 7. g4 dxc4

			variation => N__('Semi-Slav: 6. Qc2 Bd6 7. g4 dxc4'),
			moves => {
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/8/2pP4/2NBPN2/PPQ2PPP/R1B1K2R w KQkq - 0 8' => {
			eco => 'D46g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Bd6 7. Qc2 dxc4

			variation => N__('Semi-Slav: Chigorin, 7. Qc2 dxc4'),
			moves => {
				'd3c4' => 'r1bqk2r/pp1n1ppp/2pbpn2/8/2BP4/2N1PN2/PPQ2PPP/R1B1K2R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1nbp1p/2n1p3/2ppP1p1/3P4/2PB1N2/PP1N1PPP/R1BQ1RK1 w kq g6 0 9' => {
			eco => 'C05t',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ngf3 Be7 8. O-O g5

			variation => N__('French: Tarrasch, Closed, 7.Ngf3 Be7 8.O-O g5'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2n1p3/2ppP3/3P4/2PB1N2/PP1N1PPP/R1BQ1RK1 b kq - 4 8' => {
			eco => 'C05s',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Closed, 7.Ngf3 Be7'),
			moves => {
				'g7g5' => 'r1bqk2r/pp1nbp1p/2n1p3/2ppP1p1/3P4/2PB1N2/PP1N1PPP/R1BQ1RK1 w kq g6 0 9',
			},
		},
		'r1bqk2r/pp1nbppp/2n1p3/2ppP3/3P4/2PB1N2/PP1N1PPP/R1BQK2R w KQkq - 3 8' => {
			eco => 'C05s',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 Nf6 4. e5 Nfd7 5. Bd3 c5 6. c3 Nc6 7. Ngf3 Be7

			variation => N__('French: Tarrasch, Closed, 7.Ngf3 Be7'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1nbppp/2n1p3/2ppP3/3P4/2PB1N2/PP1N1PPP/R1BQ1RK1 b kq - 4 8',
			},
		},
		'r1bqk2r/pp1nbppp/2npp3/6P1/3NP3/2N1B3/PPP2P1P/R2QKB1R w KQkq - 3 9' => {
			eco => 'B81t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. g4 Nc6 7. g5 Nd7 8. Be3 Be7

			variation => N__('Sicilian: Scheveningen, Keres, 6... Nc6 7. g5 Nd7 8. Be3 Be7'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p1pn2/3p2B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 1 7' => {
			eco => 'D43i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 Be7 6. e3 Nbd7

			variation => N__('Semi-Slav: Botvinnik, 5... Be7 6. e3 Nbd7'),
			moves => {
				'f1d3' => 'r1bqk2r/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R b KQkq - 2 7',
			},
		},
		'r1bqk2r/pp1nbppp/2p1pn2/3p2B1/2PP4/2NBPN2/PP3PPP/R2QK2R b KQkq - 2 7' => {
			eco => 'D43j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. Bg5 Be7 6. e3 Nbd7 7. Bd3

			variation => N__('Semi-Slav: Botvinnik, 5... Be7 6. e3 Nbd7 7. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p1pn2/3p4/2PP4/2N1PN2/PPQ2PPP/R1B1KB1R w KQkq - 3 7' => {
			eco => 'D45j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Qc2 Be7

			variation => N__('Semi-Slav: 6. Qc2 Be7'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p1pn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 7' => {
			eco => 'D46c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Be7 7. O-O

			variation => N__('Semi-Slav: Bogoljubow, 7.O-O'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1nbppp/2p1pn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 8',
			},
		},
		'r1bqk2r/pp1nbppp/2p1pn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQK2R w KQkq - 3 7' => {
			eco => 'D46c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c6 5. e3 Nbd7 6. Bd3 Be7

			variation => N__('Semi-Slav: Bogoljubow Variation'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1nbppp/2p1pn2/3p4/2PP4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 7',
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2N1P3/PPQ2PPP/R3KBNR w KQkq - 1 8' => {
			eco => 'D36h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7

			variation => N__('QGD: Exchange, 6. Qc2 Be7 7. e3 Nbd7'),
			moves => {
				'f1d3' => 'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ2PPP/R3K1NR b KQkq - 2 8',
				'g1f3' => 'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R b KQkq - 2 8',
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2N1PN2/PPQ2PPP/R3KB1R b KQkq - 2 8' => {
			eco => 'D36h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Nf3

			variation => N__('QGD: Exchange, 6. Qc2 Be7 7. e3 Nbd7 8. Nf3'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2N2N2/PPQ1PPPP/R3KB1R w KQkq - 4 8' => {
			eco => 'D36c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. Nf3 Nbd7

			variation => N__('QGD: Exchange, 6. Qc2 Be7 7. Nf3 Nbd7'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PP2NPPP/R2QK2R b KQkq - 4 8' => {
			eco => 'D35t',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. e3 Be7 7. Bd3 Nbd7 8. Nge2

			variation => N__('QGD: Exchange, 5. Bg5 c6 6. e3 Be7 7. Bd3 Nbd7 8.Nge2'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PP3PPP/R2QK1NR w KQkq - 3 8' => {
			eco => 'D35t',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. e3 Be7 7. Bd3 Nbd7

			variation => N__('QGD: Exchange, 5. Bg5 c6 6. e3 Be7 7. Bd3 Nbd7'),
			moves => {
				'g1e2' => 'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PP2NPPP/R2QK2R b KQkq - 4 8',
				'g1f3' => 'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBPN2/PP3PPP/R2QK2R b KQkq - 4 8',
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ2PPP/R3K1NR b KQkq - 2 8' => {
			eco => 'D36i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. Qc2 Be7 7. e3 Nbd7 8. Bd3

			variation => N__('QGD: Exchange, 6. Qc2 Be7 7. e3 Nbd7 8. Bd3'),
			moves => {
				'd7f8' => 'r1bqkn1r/pp2bppp/2p2n2/3p2B1/3P4/2NBP3/PPQ2PPP/R3K1NR w KQkq - 3 9',
				'e8g8' => 'r1bq1rk1/pp1nbppp/2p2n2/3p2B1/3P4/2NBP3/PPQ2PPP/R3K1NR w KQ - 3 9',
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p2B1/3P4/2NBPN2/PP3PPP/R2QK2R b KQkq - 4 8' => {
			eco => 'D35u',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Bg5 c6 6. e3 Be7 7. Bd3 Nbd7 8. Nf3

			variation => N__('QGD: Exchange, 5. Bg5 c6 6. e3 Be7 7. Bd3 Nbd7 8. Nf3'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2p2n2/3p4/3P1B2/2N2N2/PPQ1PPPP/R3KB1R w KQkq - 2 8' => {
			eco => 'D35h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. cxd5 exd5 5. Nf3 Be7 6. Bf4 c6 7. Qc2 Nbd7

			variation => N__('QGD: Exchange, 5. Nf3 Be7 6. Bf4 c6 7. Qc2 Nbd7'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p1B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQkq - 0 7' => {
			eco => 'A54',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 c6

			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 c6'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b kq - 1 7' => {
			eco => 'A55j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O

			variation => N__('Old Indian: 5. e4 Be7 6. Be2 c6 7.O-O'),
			moves => {
				'a7a6' => 'r1bqk2r/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w kq - 0 8',
				'e8g8' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w - - 2 8',
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQkq - 0 7' => {
			eco => 'A55j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6

			variation => N__('Old Indian: 5. e4 Be7 6. Be2 c6'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b kq - 1 7',
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3P1P/R1BQKB1R w KQkq - 0 7' => {
			eco => 'A55e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6'),
			moves => {
				'f1g2' => 'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQK2R b KQkq - 1 7',
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQK2R b KQkq - 1 7' => {
			eco => 'A55e',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2

			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQK2R w KQ - 2 8',
			},
		},
		'r1bqk2r/pp1nbppp/2pp3n/4p1N1/2BPPP2/8/PPP3PP/RNBQ1RK1 b kq - 2 7' => {
			eco => 'C41g',
                        # TRANSLATORS:

			variation => N__('Philidor: Hanham, Kmoch Variation'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1nbppp/2pp3n/4p1N1/2BPPP2/8/PPP3PP/RNBQ1RK1 w - - 3 8',
			},
		},
		'r1bqk2r/pp1nbppp/2pp3n/4p1N1/2BPPP2/8/PPP3PP/RNBQK2R w KQkq - 1 7' => {
			eco => 'C41g',
                        # TRANSLATORS:

			variation => N__('Philidor: Hanham, Kmoch Variation'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1nbppp/2pp3n/4p1N1/2BPPP2/8/PPP3PP/RNBQ1RK1 b kq - 2 7',
			},
		},
		'r1bqk2r/pp1nbppp/4pn2/2pp2B1/3P4/2P1PN2/PP1N1PPP/R2QKB1R w KQkq - 1 7' => {
			eco => 'D03j',
                        # TRANSLATORS: 1. d4 d5 2. Nf3 Nf6 3. Bg5 e6 4. e3 c5 5. Nbd2 Be7 6. c3 Nbd7

			variation => N__('Torre Attack: 3... e6 4. e3 c5'),
			moves => {
			},
		},
		'r1bqk2r/pp1nppbp/2np2p1/8/2P1P3/2N5/PPN1BPPP/R1BQK2R w KQkq - 2 9' => {
			eco => 'B37j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2 Nd7

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2 Nd7'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1pbp/6p1/1N2p3/2PnP3/4B3/PP3PPP/R2QKB1R b KQkq - 1 10' => {
			eco => 'B39f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 e5 10. Nb5

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 e5 10. Nb5'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1pbp/6p1/4p3/2PnP3/2N1B3/PP3PPP/R2QKB1R w KQkq e6 0 10' => {
			eco => 'B39e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 e5

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 e5'),
			moves => {
				'c3b5' => 'r1bqk2r/pp1p1pbp/6p1/1N2p3/2PnP3/4B3/PP3PPP/R2QKB1R b KQkq - 1 10',
				'f1d3' => 'r1bqk2r/pp1p1pbp/6p1/4p3/2PnP3/2NBB3/PP3PPP/R2QK2R b KQkq - 1 10',
			},
		},
		'r1bqk2r/pp1p1pbp/6p1/4p3/2PnP3/2NBB3/PP3PPP/R2QK2R b KQkq - 1 10' => {
			eco => 'B39g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 e5 10. Bd3

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 e5 10. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2N1pn2/8/1bP1P3/2N5/PP3PPP/R1BQKB1R b KQkq - 0 7' => {
			eco => 'B44e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. c4 Nf6 6. Nc3 Bb4 7. Nxc6

			variation => N__('Sicilian: Taimanov, 5. c4 Nf6 6. Nc3 Bb4 7.Nxc6'),
			moves => {
				'b7c6' => 'r1bqk2r/p2p1ppp/2p1pn2/8/1bP1P3/2N5/PP3PPP/R1BQKB1R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1p3/2P5/1bP1n3/1QN2N2/PP1BPPPP/R3KB1R b KQkq - 4 7' => {
			eco => 'E23',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Spielmann, 4... c5 5.dxc5 Nc6 6. Nf3 Ne4'),
			moves => {
				'e4c5' => 'r1bqk2r/pp1p1ppp/2n1p3/2n5/1bP5/1QN2N2/PP1BPPPP/R3KB1R w KQkq - 0 8',
				'e4d2' => 'r1bqk2r/pp1p1ppp/2n1p3/2P5/1bP5/1QN2N2/PP1nPPPP/R3KB1R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1p3/2P5/1bP1n3/1QN2N2/PP2PPPP/R1B1KB1R w KQkq - 3 7' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6 6. Nf3 Ne4

			variation => N__('Nimzo-Indian: Spielmann, 4... c5 5.dxc5 Nc6 6. Nf3 Ne4'),
			moves => {
				'c1d2' => 'r1bqk2r/pp1p1ppp/2n1p3/2P5/1bP1n3/1QN2N2/PP1BPPPP/R3KB1R b KQkq - 4 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1p3/2P5/1bP5/1QN2N2/PP1nPPPP/R3KB1R w KQkq - 0 8' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6 6. Nf3 Ne4 7. Bd2 Nxd2

			variation => N__('Nimzo-Indian: Spielmann, Carlsbad Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1p3/2n5/1bP5/1QN2N2/PP1BPPPP/R3KB1R w KQkq - 0 8' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6 6. Nf3 Ne4 7. Bd2 Nxc5

			variation => N__('Nimzo-Indian: Spielmann, San Remo Variation'),
			moves => {
				'b3c2' => 'r1bqk2r/pp1p1ppp/2n1p3/2n5/1bP5/2N2N2/PPQBPPPP/R3KB1R b KQkq - 1 8',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1p3/2n5/1bP5/2N2N2/PPQBPPPP/R3KB1R b KQkq - 1 8' => {
			eco => 'E23',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Spielmann, San Remo Variation'),
			moves => {
				'f7f5' => 'r1bqk2r/pp1p2pp/2n1p3/2n2p2/1bP5/2N2N2/PPQBPPPP/R3KB1R w KQkq f6 0 9',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/1N6/1b2P3/2N5/PPP2PPP/R1BQKB1R w KQkq - 5 7' => {
			eco => 'B45p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4'),
			moves => {
				'a2a3' => 'r1bqk2r/pp1p1ppp/2n1pn2/1N6/1b2P3/P1N5/1PP2PPP/R1BQKB1R b KQkq - 0 7',
				'b5d6' => 'r1bqk2r/pp1p1ppp/2nNpn2/8/1b2P3/2N5/PPP2PPP/R1BQKB1R b KQkq - 6 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/1N6/1b2P3/P1N5/1PP2PPP/R1BQKB1R b KQkq - 0 7' => {
			eco => 'B45r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4 7. a3

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3'),
			moves => {
				'b4c3' => 'r1bqk2r/pp1p1ppp/2n1pn2/1N6/4P3/P1b5/1PP2PPP/R1BQKB1R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/1N6/1bP5/2N5/PP2PPPP/R1BQKB1R w KQkq - 4 7' => {
			eco => 'A33h',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 Bb4

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 Bb4'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/1N6/4P3/P1b5/1PP2PPP/R1BQKB1R w KQkq - 0 8' => {
			eco => 'B45r',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3'),
			moves => {
				'b5c3' => 'r1bqk2r/pp1p1ppp/2n1pn2/8/4P3/P1N5/1PP2PPP/R1BQKB1R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2P3B1/1bP5/1QN5/PP2PPPP/R3KBNR b KQkq - 2 6' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6 6. Bg5

			variation => N__('Nimzo-Indian: Spielmann, 4... c5 5.dxc5 Nc6 6. Bg5'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2P5/1bP5/1QN2N2/PP2PPPP/R1B1KB1R b KQkq - 2 6' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6 6. Nf3

			variation => N__('Nimzo-Indian: Spielmann, 4... c5 5.dxc5 Nc6 6. Nf3'),
			moves => {
				'f6e4' => 'r1bqk2r/pp1p1ppp/2n1p3/2P5/1bP1n3/1QN2N2/PP2PPPP/R1B1KB1R w KQkq - 3 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2P5/1bP5/1QN5/PP2PPPP/R1B1KBNR w KQkq - 1 6' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6

			variation => N__('Nimzo-Indian: Spielmann, 4... c5 5.dxc5 Nc6'),
			moves => {
				'c1g5' => 'r1bqk2r/pp1p1ppp/2n1pn2/2P3B1/1bP5/1QN5/PP2PPPP/R3KBNR b KQkq - 2 6',
				'g1f3' => 'r1bqk2r/pp1p1ppp/2n1pn2/2P5/1bP5/1QN2N2/PP2PPPP/R1B1KB1R b KQkq - 2 6',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2P5/1bP5/2N5/PPQ1PPPP/R1B1KBNR w KQkq - 1 6' => {
			eco => 'E38h',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 Nc6

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2b5/2PN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq - 1 7' => {
			eco => 'A33n',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Bc5

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. g3 Bc5'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2b5/2PN4/P1N5/1P2PPPP/R1BQKB1R w KQkq - 1 7' => {
			eco => 'A33f',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. a3 Bc5

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. a3 Bc5'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/1bPP4/2N2NP1/PP2PP1P/R1BQKB1R w KQkq - 2 6' => {
			eco => 'E20',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. g3 c5 5. Nf3 Nc6

			variation => N__('Nimzo-Indian: Romanishin, 4... c5 5. Nf3 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBP3/PP2NPPP/R1BQK2R b KQkq - 3 6' => {
			eco => 'E41',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 c5 5. Bd3 Nc6 6. Ne2

			variation => N__('Nimzo-Indian: 4. e3 c5 5. Bd3 Nc6 6. Ne2'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBP3/PP3PPP/R1BQK1NR w KQkq - 2 6' => {
			eco => 'E41',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 c5 5. Bd3 Nc6

			variation => N__('Nimzo-Indian: 4. e3 c5 5. Bd3 Nc6'),
			moves => {
				'g1e2' => 'r1bqk2r/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBP3/PP2NPPP/R1BQK2R b KQkq - 3 6',
				'g1f3' => 'r1bqk2r/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBPN2/PP3PPP/R1BQK2R b KQkq - 3 6',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/1bPP4/2NBPN2/PP3PPP/R1BQK2R b KQkq - 3 6' => {
			eco => 'E41',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 c5 5. Bd3 Nc6 6. Nf3

			variation => N__('Nimzo-Indian: 4. e3 c5 5. Bd3 Nc6 6. Nf3'),
			moves => {
				'b4c3' => 'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/2bBPN2/PP3PPP/R1BQK2R w KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/2PBPN2/P4PPP/R1BQK2R b KQkq - 0 7' => {
			eco => 'E41',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: 4. e3 c5 5. Bd3 Nc6 6. Nf3'),
			moves => {
				'd7d6' => 'r1bqk2r/pp3ppp/2nppn2/2p5/2PP4/2PBPN2/P4PPP/R1BQK2R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/2bBPN2/PP3PPP/R1BQK2R w KQkq - 0 7' => {
			eco => 'E41',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: 4. e3 c5 5. Bd3 Nc6 6. Nf3'),
			moves => {
				'b2c3' => 'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/2PBPN2/P4PPP/R1BQK2R b KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/P1P1P3/5PPP/R1BQKBNR w KQkq - 1 7' => {
			eco => 'E26',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 c5 6. e3 Nc6

			variation => N__('Nimzo-Indian: Sämisch, 5... c5 6. e3 Nc6'),
			moves => {
				'f1d3' => 'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBP3/5PPP/R1BQK1NR b KQkq - 2 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2p5/2PP4/P1PBP3/5PPP/R1BQK1NR b KQkq - 2 7' => {
			eco => 'E26',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. a3 Bxc3+ 5. bxc3 c5 6. e3 Nc6 7. Bd3

			variation => N__('Nimzo-Indian: Sämisch, 5... c5 6. e3 Nc6 7. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1b1NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 5 7' => {
			eco => 'B45i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Be3 Bb4

			variation => N__('Sicilian: Taimanov, Four Knights, 6. Be3 Bb4'),
			moves => {
				'f1d3' => 'r1bqk2r/pp1p1ppp/2n1pn2/8/1b1NP3/2NBB3/PPP2PPP/R2QK2R b KQkq - 6 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1b1NP3/2NBB3/PPP2PPP/R2QK2R b KQkq - 6 7' => {
			eco => 'B45j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Be3 Bb4 7. Bd3

			variation => N__('Sicilian: Taimanov, Four Knights, 6. Be3 Bb4 7. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1bPN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq - 1 7' => {
			eco => 'A33m',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Bb4

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. g3 Bb4'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1bPN4/2N5/PP1BPPPP/R2QKB1R w KQkq - 4 7' => {
			eco => 'A32',
                        # TRANSLATORS: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Bb4 6. Bd2 Nc6

			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Bb4 6. Bd2 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1bPNP3/2N5/PP3PPP/R1BQKB1R w KQkq - 3 7' => {
			eco => 'B44e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. c4 Nf6 6. Nc3 Bb4

			variation => N__('Sicilian: Taimanov, 5. c4 Nf6 6. Nc3 Bb4'),
			moves => {
				'd4c6' => 'r1bqk2r/pp1p1ppp/2N1pn2/8/1bP1P3/2N5/PP3PPP/R1BQKB1R b KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/4P3/P1N5/1PP2PPP/R1BQKB1R b KQkq - 0 8' => {
			eco => 'B45r',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3'),
			moves => {
				'd7d5' => 'r1bqk2r/pp3ppp/2n1pn2/3p4/4P3/P1N5/1PP2PPP/R1BQKB1R w KQkq d6 0 9',
			},
		},
		'r1bqk2r/pp1p1ppp/2nNpn2/8/1b2P3/2N5/PPP2PPP/R1BQKB1R b KQkq - 6 7' => {
			eco => 'B45q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4 7. Nd6+

			variation => N__('Sicilian: Taimanov, Four Knights, American Attack'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/4pn2/2n5/1PP5/P1Q5/4PPPP/R1B1KBNR b KQkq b3 0 8' => {
			eco => 'E38k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 Na6 6. a3 Bxc3+ 7. Qxc3 Nxc5 8. b4

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6: 8. b4'),
			moves => {
				'c5e4' => 'r1bqk2r/pp1p1ppp/4pn2/8/1PP1n3/P1Q5/4PPPP/R1B1KBNR w KQkq - 1 9',
			},
		},
		'r1bqk2r/pp1p1ppp/4pn2/2n5/2P5/P1Q5/1P2PPPP/R1B1KBNR w KQkq - 0 8' => {
			eco => 'E38j',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6 6. a3 Bxc3+ 7.Qxc3'),
			moves => {
				'b2b4' => 'r1bqk2r/pp1p1ppp/4pn2/2n5/1PP5/P1Q5/4PPPP/R1B1KBNR b KQkq b3 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/4pn2/8/1PP1n3/P1Q5/4PPPP/R1B1KBNR w KQkq - 1 9' => {
			eco => 'E38k',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 Na6 6. a3 Bxc3+ 7. Qxc3 Nxc5 8. b4 Nce4

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6: 8. b4 Nce4'),
			moves => {
				'c3d4' => 'r1bqk2r/pp1p1ppp/4pn2/8/1PPQn3/P7/4PPPP/R1B1KBNR b KQkq - 2 9',
			},
		},
		'r1bqk2r/pp1p1ppp/4pn2/8/1PPQn3/P7/4PPPP/R1B1KBNR b KQkq - 2 9' => {
			eco => 'E38l',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 Na6 6. a3 Bxc3+ 7. Qxc3 Nxc5 8. b4 Nce4 9. Qd4

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6: 8. b4 Nce4 9. Qd4'),
			moves => {
				'd7d5' => 'r1bqk2r/pp3ppp/4pn2/3p4/1PPQn3/P7/4PPPP/R1B1KBNR w KQkq d6 0 10',
			},
		},
		'r1bqk2r/pp1p1ppp/n3pn2/2P5/1bP5/1QN5/PP2PPPP/R1B1KBNR w KQkq - 1 6' => {
			eco => 'E22',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Na6

			variation => N__('Nimzo-Indian: Spielmann, 4... c5 5.dxc5 Na6'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/n3pn2/2P5/1bP5/2N5/PPQ1PPPP/R1B1KBNR w KQkq - 1 6' => {
			eco => 'E38i',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 Na6

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6'),
			moves => {
				'a2a3' => 'r1bqk2r/pp1p1ppp/n3pn2/2P5/1bP5/P1N5/1PQ1PPPP/R1B1KBNR b KQkq - 0 6',
			},
		},
		'r1bqk2r/pp1p1ppp/n3pn2/2P5/1bP5/P1N5/1PQ1PPPP/R1B1KBNR b KQkq - 0 6' => {
			eco => 'E38i',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6'),
			moves => {
				'b4c3' => 'r1bqk2r/pp1p1ppp/n3pn2/2P5/2P5/P1b5/1PQ1PPPP/R1B1KBNR w KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1p1ppp/n3pn2/2P5/2P5/P1Q5/1P2PPPP/R1B1KBNR b KQkq - 0 7' => {
			eco => 'E38j',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 c5 5. dxc5 Na6 6. a3 Bxc3+ 7. Qxc3

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6 6. a3 Bxc3+ 7.Qxc3'),
			moves => {
				'a6c5' => 'r1bqk2r/pp1p1ppp/4pn2/2n5/2P5/P1Q5/1P2PPPP/R1B1KBNR w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1p1ppp/n3pn2/2P5/2P5/P1b5/1PQ1PPPP/R1B1KBNR w KQkq - 0 7' => {
			eco => 'E38i',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Classical, 4... c5 5.dxc5 Na6'),
			moves => {
				'c2c3' => 'r1bqk2r/pp1p1ppp/n3pn2/2P5/2P5/P1Q5/1P2PPPP/R1B1KBNR b KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1p2pp/2n1p3/2n2p2/1bP5/2N2N2/PPQBPPPP/R3KB1R w KQkq f6 0 9' => {
			eco => 'E23',
                        # TRANSLATORS:

			variation => N__('Nimzo-Indian: Spielmann, San Remo Variation'),
			moves => {
				'g2g3' => 'r1bqk2r/pp1p2pp/2n1p3/2n2p2/1bP5/2N2NP1/PPQBPP1P/R3KB1R b KQkq - 0 9',
			},
		},
		'r1bqk2r/pp1p2pp/2n1p3/2n2p2/1bP5/2N2NP1/PPQBPP1P/R3KB1R b KQkq - 0 9' => {
			eco => 'E23',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qb3 c5 5. dxc5 Nc6 6. Nf3 Ne4 7. Bd2 Nxc5 8. Qc2 f5 9. g3

			variation => N__('Nimzo-Indian: Spielmann, Stahlberg Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp1pbppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 5 6' => {
			eco => 'A36c',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'd7d5' => 'r1bqk2r/pp2bppp/2n1pn2/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq d6 0 7',
			},
		},
		'r1bqk2r/pp1pbppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 4 6' => {
			eco => 'A36c',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1pbppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 5 6',
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 b kq - 3 7' => {
			eco => 'A36p',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 w - - 4 8',
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQkq - 2 7' => {
			eco => 'A36p',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7

			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 b kq - 3 7',
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq - 2 7' => {
			eco => 'A36m',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e6 6. Nge2 Nge7

			variation => N__('English: Symmetrical, 5. e3 e6'),
			moves => {
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N1PNP1/PP1P1PBP/R1BQ1RK1 b kq - 0 7' => {
			eco => 'A37m',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. e3

			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. e3'),
			moves => {
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 2 7' => {
			eco => 'A37l',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7

			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7'),
			moves => {
				'd2d3' => 'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq - 0 7',
				'e2e3' => 'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N1PNP1/PP1P1PBP/R1BQ1RK1 b kq - 0 7',
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq - 0 7' => {
			eco => 'A37n',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. d3

			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. d3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 1 8',
			},
		},
		'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 2 7' => {
			eco => 'A37x',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7'),
			moves => {
				'd2d3' => 'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq - 0 7',
			},
		},
		'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq - 0 7' => {
			eco => 'A37y',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7 7. d3

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7 7. d3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w - - 1 8',
			},
		},
		'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R w KQkq - 1 7' => {
			eco => 'A37t',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. d3 Nge7

			variation => N__('English: Symmetrical, 5. Nf3 e5 6. d3 Nge7'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2N2np1/8/4P3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 0 7' => {
			eco => 'B34r',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6'),
			moves => {
				'b7c6' => 'r1bqk2r/p2pppbp/2p2np1/8/4P3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq - 0 6' => {
			eco => 'B31l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQR1K1 w - - 1 7',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq - 4 6' => {
			eco => 'B31k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6

			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6'),
			moves => {
				'c2c3' => 'r1bqk2r/pp1pppbp/2n2np1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq - 0 6',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq - 1 6' => {
			eco => 'A36o',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 Nf6

			variation => N__('English: Symmetrical, Botvinnik, 5... Nf6'),
			moves => {
				'g1e2' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq - 2 6',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq - 2 6' => {
			eco => 'A36o',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 Nf6 6. Nge2

			variation => N__('English: Symmetrical, Botvinnik, 5... Nf6 6.Nge2'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 5 6' => {
			eco => 'A38c',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O

			variation => N__('English: Symmetrical, Main Line, 6.O-O'),
			moves => {
				'd7d5' => 'r1bqk2r/pp2ppbp/2n2np1/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq d6 0 7',
				'd7d6' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 0 7',
				'e8g8' => 'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w - - 6 7',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 4 6' => {
			eco => 'A38a',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6

			variation => N__('English: Symmetrical, Main Line'),
			moves => {
				'd2d4' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq d3 0 6',
				'e1g1' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 5 6',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq d3 0 6' => {
			eco => 'A38b',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. d4

			variation => N__('English: Symmetrical, Main Line, 6. d4'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq e3 0 6' => {
			eco => 'A04',
                        # TRANSLATORS: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 Nf6 6. e4

			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq - 1 6' => {
			eco => 'A04',
                        # TRANSLATORS: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 Nf6

			variation => N__('Reti: 1... c5'),
			moves => {
				'e2e4' => 'r1bqk2r/pp1pppbp/2n2np1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq e3 0 6',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R b KQkq - 5 7' => {
			eco => 'B35a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4'),
			moves => {
				'd8a5' => 'r1b1k2r/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQkq - 6 8',
				'e8g8' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQ - 6 8',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/2P1P3/2N5/PPN2PPP/R1BQKB1R b KQkq - 4 7' => {
			eco => 'B37g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3'),
			moves => {
				'd7d6' => 'r1bqk2r/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN2PPP/R1BQKB1R w KQkq - 0 8',
				'e8g8' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2P1P3/2N5/PPN2PPP/R1BQKB1R w KQ - 5 8',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/2P1P3/8/PPN2PPP/RNBQKB1R w KQkq - 3 7' => {
			eco => 'B37g',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6'),
			moves => {
				'b1c3' => 'r1bqk2r/pp1pppbp/2n2np1/8/2P1P3/2N5/PPN2PPP/R1BQKB1R b KQkq - 4 7',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R b KQkq - 4 7' => {
			eco => 'B38d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 Nf6 7. Nc3'),
			moves => {
				'd7d6' => 'r1bqk2r/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R w KQkq - 0 8',
				'e8g8' => 'r1bq1rk1/pp1pppbp/2n2np1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R w KQ - 5 8',
				'f6g4' => 'r1bqk2r/pp1pppbp/2n3p1/8/2PNP1n1/2N1B3/PP3PPP/R2QKB1R w KQkq - 5 8',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/2PNP3/4B3/PP3PPP/RN1QKB1R w KQkq - 3 7' => {
			eco => 'B38d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 Nf6'),
			moves => {
				'b1c3' => 'r1bqk2r/pp1pppbp/2n2np1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R b KQkq - 4 7',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R b KQkq - 5 7' => {
			eco => 'B34t',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQ - 6 8',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq - 4 7' => {
			eco => 'B34p',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6'),
			moves => {
				'd4c6' => 'r1bqk2r/pp1pppbp/2N2np1/8/4P3/2N1B3/PPP2PPP/R2QKB1R b KQkq - 0 7',
				'f1c4' => 'r1bqk2r/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R b KQkq - 5 7',
				'f1e2' => 'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R b KQkq - 5 7',
				'f2f3' => 'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1BP2/PPP3PP/R2QKB1R b KQkq - 0 7',
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1BP2/PPP3PP/R2QKB1R b KQkq - 0 7' => {
			eco => 'B34q',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. f3

			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. f3'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n3p1/8/2PNP1Q1/2N1B3/PP3PPP/R3KB1R b KQkq - 0 8' => {
			eco => 'B39b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4'),
			moves => {
				'c6d4' => 'r1bqk2r/pp1pppbp/6p1/8/2PnP1Q1/2N1B3/PP3PPP/R3KB1R w KQkq - 0 9',
			},
		},
		'r1bqk2r/pp1pppbp/2n3p1/8/2PNP1n1/2N1B3/PP3PPP/R2QKB1R w KQkq - 5 8' => {
			eco => 'B39a',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4

			variation => N__('Sicilian: Maroczy Bind, Breyer Variation'),
			moves => {
				'd1g4' => 'r1bqk2r/pp1pppbp/2n3p1/8/2PNP1Q1/2N1B3/PP3PPP/R3KB1R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp1pppbp/2n3pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 5 6' => {
			eco => 'A37d',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nh6 6. O-O

			variation => N__('English: Symmetrical, 5. Nf3 Nh6 6.O-O'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n3pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq - 4 6' => {
			eco => 'A37c',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nh6

			variation => N__('English: Symmetrical, 5. Nf3 Nh6'),
			moves => {
				'e1g1' => 'r1bqk2r/pp1pppbp/2n3pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 5 6',
			},
		},
		'r1bqk2r/pp1pppbp/2n3pn/8/2PNP3/4B3/PP3PPP/RN1QKB1R w KQkq - 3 7' => {
			eco => 'B38b',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nh6

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 Nh6'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/4n1p1/8/2P1P3/2N1B3/PP1Q1PPP/R3KB1R b KQkq - 3 10' => {
			eco => 'B39i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 Ne6 10. Qd2

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 Ne6 10. Qd2'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/4n1p1/8/2P1P3/2N1B3/PP3PPP/2RQKB1R b Kkq - 3 10' => {
			eco => 'B39j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 Ne6 10. Rc1

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 Ne6 10. Rc1'),
			moves => {
				'd8a5' => 'r1b1k2r/pp1pppbp/4n1p1/q7/2P1P3/2N1B3/PP3PPP/2RQKB1R w Kkq - 4 11',
			},
		},
		'r1bqk2r/pp1pppbp/4n1p1/8/2P1P3/2N1B3/PP3PPP/R2QKB1R w KQkq - 2 10' => {
			eco => 'B39h',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1 Ne6

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1 Ne6'),
			moves => {
				'a1c1' => 'r1bqk2r/pp1pppbp/4n1p1/8/2P1P3/2N1B3/PP3PPP/2RQKB1R b Kkq - 3 10',
				'd1d2' => 'r1bqk2r/pp1pppbp/4n1p1/8/2P1P3/2N1B3/PP1Q1PPP/R3KB1R b KQkq - 3 10',
			},
		},
		'r1bqk2r/pp1pppbp/6p1/8/2PnP1Q1/2N1B3/PP3PPP/R3KB1R w KQkq - 0 9' => {
			eco => 'B39c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4'),
			moves => {
				'g4d1' => 'r1bqk2r/pp1pppbp/6p1/8/2PnP3/2N1B3/PP3PPP/R2QKB1R b KQkq - 1 9',
			},
		},
		'r1bqk2r/pp1pppbp/6p1/8/2PnP3/2N1B3/PP3PPP/R2QKB1R b KQkq - 1 9' => {
			eco => 'B39d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 Ng4 8. Qxg4 Nxd4 9. Qd1

			variation => N__('Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9. Qd1'),
			moves => {
				'd4e6' => 'r1bqk2r/pp1pppbp/4n1p1/8/2P1P3/2N1B3/PP3PPP/R2QKB1R w KQkq - 2 10',
				'e7e5' => 'r1bqk2r/pp1p1pbp/6p1/4p3/2PnP3/2N1B3/PP3PPP/R2QKB1R w KQkq e6 0 10',
			},
		},
		'r1bqk2r/pp2bpp1/2np1n1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQ1RK1 b kq - 0 9' => {
			eco => 'B58l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nf3 h6 8. O-O Be7 9. h3

			variation => N__('Sicilian: Boleslavsky, 7. Nf3 h6 8.O-O Be7 9. h3'),
			moves => {
			},
		},
		'r1bqk2r/pp2bpp1/2np1n1p/4p3/4P3/2N2N2/PPP1BPPP/R1BQ1RK1 w kq - 2 9' => {
			eco => 'B58k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nf3 h6 8. O-O Be7

			variation => N__('Sicilian: Boleslavsky, 7. Nf3 h6 8.O-O Be7'),
			moves => {
				'f1e1' => 'r1bqk2r/pp2bpp1/2np1n1p/4p3/4P3/2N2N2/PPP1BPPP/R1BQR1K1 b kq - 3 9',
				'h2h3' => 'r1bqk2r/pp2bpp1/2np1n1p/4p3/4P3/2N2N1P/PPP1BPP1/R1BQ1RK1 b kq - 0 9',
			},
		},
		'r1bqk2r/pp2bpp1/2np1n1p/4p3/4P3/2N2N2/PPP1BPPP/R1BQR1K1 b kq - 3 9' => {
			eco => 'B58m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nf3 h6 8. O-O Be7 9. Re1

			variation => N__('Sicilian: Boleslavsky, 7. Nf3 h6 8.O-O Be7 9. Re1'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bpp1/2np1n1p/4p3/4P3/2N2N2/PPP1BPPP/R1BQR1K1 w - - 4 10',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/2pn4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq - 4 8' => {
			eco => 'D41c',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. g3 Nc6 7. Bg2 Be7'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n1p3/2pn4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 9',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/2pn4/3P4/2N2NP1/PP2PPBP/R1BQK2R w KQkq - 3 8' => {
			eco => 'D41c',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. g3 Nc6 7. Bg2 Be7

			variation => N__('QGD: Semi-Tarrasch, 6. g3 Nc6 7. Bg2 Be7'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2bppp/2n1p3/2pn4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq - 4 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/2pn4/3P4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 8' => {
			eco => 'D42b',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bd3 Be7'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n1p3/2pn4/3P4/2NBPN2/PP3PPP/R1BQ1RK1 w - - 5 9',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/2pn4/3P4/2NBPN2/PP3PPP/R1BQK2R w KQkq - 3 8' => {
			eco => 'D42b',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 Be7

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bd3 Be7'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2bppp/2n1p3/2pn4/3P4/2NBPN2/PP3PPP/R1BQ1RK1 b kq - 4 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/2pn4/8/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 0 8' => {
			eco => 'A36d',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2 Be7 6. O-O d5 7. cxd5 Nxd5

			variation => N__('English: Symmetrical, Keres-Parma, Main Line Exchange'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQ1RK1 b kq - 2 9' => {
			eco => 'D41r',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bc4'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQ1RK1 w - - 3 10',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQK2R w KQkq - 1 9' => {
			eco => 'D41r',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bc4 cxd4 8. exd4 Be7

			variation => N__('QGD: Semi-Tarrasch, 6. e3 Nc6 7. Bc4'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2bppp/2n1p3/3n4/2BP4/2N2N2/PP3PPP/R1BQ1RK1 b kq - 2 9',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQ1RK1 b kq - 2 9' => {
			eco => 'D42e',
                        # TRANSLATORS:

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQ1RK1 w - - 3 10',
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQK2R w KQkq - 1 9' => {
			eco => 'D42e',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. cxd5 Nxd5 6. e3 Nc6 7. Bd3 cxd4 8. exd4 Be7

			variation => N__('QGD: Semi-Tarrasch, 7. Bd3 Main Line'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2bppp/2n1p3/3n4/3P4/2NB1N2/PP3PPP/R1BQ1RK1 b kq - 2 9',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2Pp2B1/3P4/2N2N2/PP3PPP/R2QKB1R b KQkq - 0 8' => {
			eco => 'B13z',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7 8. c5

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variaiton, 8. c5'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n1pn2/2Pp2B1/3P4/2N2N2/PP3PPP/R2QKB1R w KQ - 1 9',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2Pp4/3P4/P1N2N2/1P3PPP/R1BQKB1R b KQkq - 0 8' => {
			eco => 'D40m',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. e3 Nc6 6. a3 cxd4 7. exd4 Be7 8. c5

			variation => N__('QGD: Semi-Tarrasch, 5. e3 Nc6 6. a3 cxd4 7.exd4 Be7 8. c5'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pP4/8/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 0 7' => {
			eco => 'A36c',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'f6d5' => 'r1bqk2r/pp2bppp/2n1p3/2pn4/8/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 0 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq d6 0 7' => {
			eco => 'A36c',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'c4d5' => 'r1bqk2r/pp2bppp/2n1pn2/2pP4/8/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq - 0 7',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq e3 0 7' => {
			eco => 'A08',
                        # TRANSLATORS: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 Be7 7. e4

			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w - - 1 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq - 3 7' => {
			eco => 'A08',
                        # TRANSLATORS:

			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 'r1bqk2r/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq e3 0 7',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3P2B1/3QP3/2N5/PP3PPP/R3KBNR w KQkq - 1 8' => {
			eco => 'D50',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 c5 5. cxd5 cxd4 6. Qxd4 Be7 7. e4 Nc6

			variation => N__('QGD: Dutch-Peruvian, Prins, 6.Qxd4 Be7 7. e4 Nc6'),
			moves => {
				'd4d2' => 'r1bqk2r/pp2bppp/2n1pn2/3P2B1/4P3/2N5/PP1Q1PPP/R3KBNR b KQkq - 2 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3P2B1/4P3/2N5/PP1Q1PPP/R3KBNR b KQkq - 2 8' => {
			eco => 'D50',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Bg5 c5 5. cxd5 cxd4 6. Qxd4 Be7 7. e4 Nc6 8. Qd2

			variation => N__('QGD: Dutch-Peruvian, Prins, 6.Qxd4 Be7 7. e4 Nc6 8. Qd2'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3p2B1/2PP4/2N2N2/PP3PPP/R2QKB1R w KQkq - 2 8' => {
			eco => 'B13y',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation'),
			moves => {
				'c4c5' => 'r1bqk2r/pp2bppp/2n1pn2/2Pp2B1/3P4/2N2N2/PP3PPP/R2QKB1R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3p4/2PP4/P1N2N2/1P3PPP/R1BQKB1R w KQkq - 1 8' => {
			eco => 'D40l',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. e3 Nc6 6. a3 cxd4 7. exd4 Be7

			variation => N__('QGD: Semi-Tarrasch, 5. e3 Nc6 6. a3 cxd4 7.exd4 Be7'),
			moves => {
				'c4c5' => 'r1bqk2r/pp2bppp/2n1pn2/2Pp4/3P4/P1N2N2/1P3PPP/R1BQKB1R b KQkq - 0 8',
				'f1d3' => 'r1bqk2r/pp2bppp/2n1pn2/3p4/2PP4/P1NB1N2/1P3PPP/R1BQK2R b KQkq - 2 8',
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3p4/2PP4/P1NB1N2/1P3PPP/R1BQK2R b KQkq - 2 8' => {
			eco => 'D40n',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 Nf6 4. Nf3 c5 5. e3 Nc6 6. a3 cxd4 7. exd4 Be7 8. Bd3

			variation => N__('QGD: Semi-Tarrasch, 5. e3 Nc6 6. a3 cxd4 7.exd4 Be7 8. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3p4/3P1B2/2N1PN2/PP3PPP/R2QKB1R w KQkq - 1 8' => {
			eco => 'D13r',
                        # TRANSLATORS: 1. d4 d5 2. c4 c6 3. Nf3 Nf6 4. cxd5 cxd5 5. Nc3 Nc6 6. Bf4 e6 7. e3 Be7

			variation => N__('Slav: Exchange, 6. Bf4 e6 7. e3 Be7'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/2Pp4/8/2N1PN2/PP2BPPP/R1BQK2R b KQkq - 0 8' => {
			eco => 'D32u',
                        # TRANSLATORS:

			variation => N__('QGD Tarrasch: 5. Nf3 Nc6 6. e3 Nf6 7. Be2 Be7'),
			moves => {
				'e7c5' => 'r1bqk2r/pp3ppp/2n2n2/2bp4/8/2N1PN2/PP2BPPP/R1BQK2R w KQkq - 0 9',
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/2pp4/3P4/2N1PN2/PP2BPPP/R1BQK2R w KQkq - 3 8' => {
			eco => 'D32u',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. e3 Nf6 7. Be2 Be7

			variation => N__('QGD Tarrasch: 5. Nf3 Nc6 6. e3 Nf6 7. Be2 Be7'),
			moves => {
				'd4c5' => 'r1bqk2r/pp2bppp/2n2n2/2Pp4/8/2N1PN2/PP2BPPP/R1BQK2R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/2pp4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq - 4 8' => {
			eco => 'D34a',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7 8. O-O

			variation => N__('QGD Tarrasch: 7. Bg2 Be7 8.O-O'),
			moves => {
				'c8e6' => 'r2qk2r/pp2bppp/2n1bn2/2pp4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w kq - 5 9',
				'e8g8' => 'r1bq1rk1/pp2bppp/2n2n2/2pp4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 w - - 5 9',
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/2pp4/3P4/2N2NP1/PP2PPBP/R1BQK2R w KQkq - 3 8' => {
			eco => 'D34a',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 Be7

			variation => N__('QGD Tarrasch: 7. Bg2 Be7'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2bppp/2n2n2/2pp4/3P4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq - 4 8',
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/2pp4/3P4/5NP1/PP2PPBP/RNBQ1RK1 w kq - 5 8' => {
			eco => 'D30i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nf3 c5 4. cxd5 exd5 5. g3 Nc6 6. Bg2 Nf6 7. O-O Be7

			variation => N__('QGD: Tarrasch without Nc3: Main Line'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/3p4/3N4/2N3P1/PP2PPBP/R1BQ1RK1 b kq - 2 9' => {
			eco => 'D33p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 cxd4 8. Nxd4 Be7 9. O-O

			variation => N__('QGD Tarrasch: 7. Bg2 cxd4 8.Nxd4 Be7'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2n2n2/3p4/3N4/2N3P1/PP2PPBP/R1BQ1RK1 w - - 3 10',
			},
		},
		'r1bqk2r/pp2bppp/2n2n2/3p4/3N4/2N3P1/PP2PPBP/R1BQK2R w KQkq - 1 9' => {
			eco => 'D33p',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 Nf6 7. Bg2 cxd4 8. Nxd4 Be7

			variation => N__('QGD Tarrasch: 7. Bg2 cxd4 8.Nxd4 Be7'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2bppp/2n2n2/3p4/3N4/2N3P1/PP2PPBP/R1BQ1RK1 b kq - 2 9',
			},
		},
		'r1bqk2r/pp2bppp/2np1n2/4p1B1/4P3/1NN5/PPP1BPPP/R2QK2R b KQkq - 3 8' => {
			eco => 'B59d',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. Bg5

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8. Bg5'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2np1n2/4p3/3NPP2/2N1BQ2/PPP3PP/R3KB1R w KQkq - 0 9' => {
			eco => 'B82',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. f4 Nc6 7. Be3 Be7 8. Qf3 e5

			variation => N__('Sicilian: Scheveningen, Tal, 8... e5'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2np1n2/4p3/4P3/1NN1B3/PPP1BPPP/R2QK2R b KQkq - 3 8' => {
			eco => 'B59e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. Be3

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8. Be3'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2np1n2/4p3/4P3/1NN2P2/PPP3PP/R1BQKB1R w KQkq - 2 8' => {
			eco => 'B56n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. f3 e5 7. Nb3 Be7

			variation => N__('Sicilian: Open, 2... d6, 5. Nc3 Nc6 6. f3 e5 7. Nb3 Be7'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1RK1 b kq - 3 8' => {
			eco => 'B59f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7 8. O-O

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7 8.O-O'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1RK1 w - - 4 9',
			},
		},
		'r1bqk2r/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQK2R w KQkq - 2 8' => {
			eco => 'B59c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Be2 e5 7. Nb3 Be7

			variation => N__('Sicilian: Boleslavsky, 7. Nb3 Be7'),
			moves => {
				'c1e3' => 'r1bqk2r/pp2bppp/2np1n2/4p3/4P3/1NN1B3/PPP1BPPP/R2QK2R b KQkq - 3 8',
				'c1g5' => 'r1bqk2r/pp2bppp/2np1n2/4p1B1/4P3/1NN5/PPP1BPPP/R2QK2R b KQkq - 3 8',
				'e1g1' => 'r1bqk2r/pp2bppp/2np1n2/4p3/4P3/1NN5/PPP1BPPP/R1BQ1RK1 b kq - 3 8',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/6B1/3NP3/2N5/PPP1BPPP/R2QK2R w KQkq - 2 8' => {
			eco => 'B62c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Be2 Be7

			variation => N__('Sicilian: Richter-Rauzer, 6... e6 7. Be2 Be7'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R b kq - 3 8' => {
			eco => 'B63l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7 8.O-O-O'),
			moves => {
				'c6d4' => 'r1bqk2r/pp2bppp/3ppn2/6B1/3nP3/2N5/PPPQ1PPP/2KR1B1R w kq - 0 9',
				'e8g8' => 'r1bq1rk1/pp2bppp/2nppn2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R w - - 4 9',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/6B1/3NP3/2N5/PPPQ1PPP/R3KB1R w KQkq - 2 8' => {
			eco => 'B63k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7'),
			moves => {
				'e1c1' => 'r1bqk2r/pp2bppp/2nppn2/6B1/3NP3/2N5/PPPQ1PPP/2KR1B1R b kq - 3 8',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/2B1P3/2N2N2/PP2QPPP/R1B2RK1 w kq - 4 9' => {
			eco => 'B21z',
                        # TRANSLATORS: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O Nf6 8. Qe2 Be7

			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O Nf6 8. Qe2 Be7'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/R3K2R b KQkq - 5 8' => {
			eco => 'B89',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Be3 Be7 8. Qe2

			variation => N__('Sicilian: Velimirovic'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/R3K2R w KQ - 6 9',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQkq - 4 8' => {
			eco => 'B89',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Be3 Be7

			variation => N__('Sicilian: Sozin, 7. Be3 Be7'),
			moves => {
				'd1e2' => 'r1bqk2r/pp2bppp/2nppn2/8/2BNP3/2N1B3/PPP1QPPP/R3K2R b KQkq - 5 8',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/3NP3/1BN1B3/PPP2PPP/R2QK2R b KQkq - 5 8' => {
			eco => 'B88',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Bc4 Nc6 7. Bb3 Be7 8. Be3

			variation => N__('Sicilian: Sozin-Scheveningen, 7. Bb3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2nppn2/8/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ - 6 9',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/3NP3/1BN5/PPP2PPP/R1BQK2R w KQkq - 4 8' => {
			eco => 'B88',
                        # TRANSLATORS:

			variation => N__('Sicilian: Sozin-Scheveningen, 7. Bb3'),
			moves => {
				'c1e3' => 'r1bqk2r/pp2bppp/2nppn2/8/3NP3/1BN1B3/PPP2PPP/R2QK2R b KQkq - 5 8',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 5 8' => {
			eco => 'B83',
                        # TRANSLATORS:

			variation => N__('Sicilian: Scheveningen, Modern'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/2nppn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w - - 6 9',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/3NP3/2N5/PPP1BPPP/R1BQ1RK1 w kq - 4 8' => {
			eco => 'B83',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. Be2 Nc6 7. O-O Be7

			variation => N__('Sicilian: Scheveningen, Modern'),
			moves => {
				'c1e3' => 'r1bqk2r/pp2bppp/2nppn2/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 5 8',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/3NPP2/2N1B3/PPP3PP/R2QKB1R w KQkq - 3 8' => {
			eco => 'B82',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. f4 Nc6 7. Be3 Be7

			variation => N__('Sicilian: Scheveningen, 6. f4 Nc6 7. Be3 Be7'),
			moves => {
				'd1f3' => 'r1bqk2r/pp2bppp/2nppn2/8/3NPP2/2N1BQ2/PPP3PP/R3KB1R b KQkq - 4 8',
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/3NPP2/2N1BQ2/PPP3PP/R3KB1R b KQkq - 4 8' => {
			eco => 'B82',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e6 6. f4 Nc6 7. Be3 Be7 8. Qf3

			variation => N__('Sicilian: Scheveningen, Tal Variation'),
			moves => {
				'd8c7' => 'r1b1k2r/ppq1bppp/2nppn2/8/3NPP2/2N1BQ2/PPP3PP/R3KB1R w KQkq - 5 9',
				'e6e5' => 'r1bqk2r/pp2bppp/2np1n2/4p3/3NPP2/2N1BQ2/PPP3PP/R3KB1R w KQkq - 0 9',
			},
		},
		'r1bqk2r/pp2bppp/3ppn2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R b kq - 0 9' => {
			eco => 'B63m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O Nxd4 9. Qxd4

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7 8.O-O-O Nxd4'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2bppp/3ppn2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R w - - 1 10',
			},
		},
		'r1bqk2r/pp2bppp/3ppn2/6B1/3nP3/2N5/PPPQ1PPP/2KR1B1R w kq - 0 9' => {
			eco => 'B63m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Qd2 Be7 8. O-O-O Nxd4

			variation => N__('Sicilian: Richter-Rauzer, 7. Qd2 Be7 8.O-O-O Nxd4'),
			moves => {
				'd2d4' => 'r1bqk2r/pp2bppp/3ppn2/6B1/3QP3/2N5/PPP2PPP/2KR1B1R b kq - 0 9',
			},
		},
		'r1bqk2r/pp2npbp/2n1p1p1/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w kq - 1 8' => {
			eco => 'A08',
                        # TRANSLATORS: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2 g6 7. e4 Bg7

			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w kq - 1 8' => {
			eco => 'A37v',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6 7. d3 Nge7

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6 7. d3 Nge7'),
			moves => {
				'a2a3' => 'r1bqk2r/pp2npbp/2np2p1/2p1p3/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b kq - 0 8',
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b kq - 0 8' => {
			eco => 'A37w',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6 7. d3 Nge7 8. a3

			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6 7. d3 Nge7 8. a3'),
			moves => {
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8' => {
			eco => 'B26f',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e5 7. Qd2 Nge7

			variation => N__('Sicilian: Closed, 6. Be3 e5 7. Qd2 Nge7'),
			moves => {
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 b kq - 3 8' => {
			eco => 'B25k',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5 7. Nf3 Nge7 8. O-O

			variation => N__('Sicilian: Closed, 6. f4 e5 7. Nf3 Nge7 8.O-O'),
			moves => {
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQK2R w KQkq - 2 8' => {
			eco => 'B25j',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5 7. Nf3 Nge7

			variation => N__('Sicilian: Closed, 6. f4 e5 7. Nf3 Nge7'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2npbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 b kq - 3 8',
			},
		},
		'r1bqk2r/pp2npbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq - 2 8' => {
			eco => 'B26o',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Nge7

			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Nge7'),
			moves => {
			},
		},
		'r1bqk2r/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 b kq - 3 8' => {
			eco => 'B25o',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O

			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 w - - 4 9',
			},
		},
		'r1bqk2r/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R w KQkq - 2 8' => {
			eco => 'B25n',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7

			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 b kq - 3 8',
			},
		},
		'r1bqk2r/pp2nppp/1bn5/1B1p4/8/1N3N2/PPP2PPP/R1BQ1RK1 w kq - 2 10' => {
			eco => 'C09k',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3 Bb6

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bb6'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n1p3/2ppP3/3P2Q1/P1P5/2P2PPP/R1B1KBNR w KQkq - 3 8' => {
			eco => 'C18k',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Qg4 Nbc6

			variation => N__('French: Winawer, 6... Ne7 7. Qg4 Nbc6'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n1p3/2ppP3/3P3P/P1P2N2/2P2PP1/R1BQKB1R b KQkq - 2 8' => {
			eco => 'C18i',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. h4 Nbc6 8. Nf3

			variation => N__('French: Winawer, 6... Ne7 7. h4 Nbc6'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n1p3/2ppP3/3P3P/P1P5/2P2PP1/R1BQKBNR w KQkq - 1 8' => {
			eco => 'C18i',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. h4 Nbc6

			variation => N__('French: Winawer, 6... Ne7 7. h4 Nbc6'),
			moves => {
				'g1f3' => 'r1bqk2r/pp2nppp/2n1p3/2ppP3/3P3P/P1P2N2/2P2PP1/R1BQKB1R b KQkq - 2 8',
			},
		},
		'r1bqk2r/pp2nppp/2n1p3/2ppP3/3P4/P1P2N2/2P2PPP/R1BQKB1R w KQkq - 3 8' => {
			eco => 'C19l',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Nf3 Nbc6

			variation => N__('French: Winawer, 6... Ne7 7. Nf3 Nbc6'),
			moves => {
				'a3a4' => 'r1bqk2r/pp2nppp/2n1p3/2ppP3/P2P4/2P2N2/2P2PPP/R1BQKB1R b KQkq - 0 8',
			},
		},
		'r1bqk2r/pp2nppp/2n1p3/2ppP3/P2P4/2P2N2/2P2PPP/R1BQKB1R b KQkq - 0 8' => {
			eco => 'C19m',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Bb4 4. e5 c5 5. a3 Bxc3+ 6. bxc3 Ne7 7. Nf3 Nbc6 8. a4

			variation => N__('French: Winawer, 6... Ne7 7. Nf3 Nbc6 8. a4'),
			moves => {
				'c8d7' => 'r2qk2r/pp1bnppp/2n1p3/2ppP3/P2P4/2P2N2/2P2PPP/R1BQKB1R w KQkq - 1 9',
				'd8a5' => 'r1b1k2r/pp2nppp/2n1p3/q1ppP3/P2P4/2P2N2/2P2PPP/R1BQKB1R w KQkq - 1 9',
			},
		},
		'r1bqk2r/pp2nppp/2n5/1Bbp4/8/1N3N2/PPP2PPP/R1BQ1RK1 b kq - 1 9' => {
			eco => 'C09k',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3'),
			moves => {
				'c5b6' => 'r1bqk2r/pp2nppp/1bn5/1B1p4/8/1N3N2/PPP2PPP/R1BQ1RK1 w kq - 2 10',
				'c5d6' => 'r1bqk2r/pp2nppp/2nb4/1B1p4/8/1N3N2/PPP2PPP/R1BQ1RK1 w kq - 2 10',
			},
		},
		'r1bqk2r/pp2nppp/2n5/1Bbp4/8/2P2N2/PP1N1PPP/R1BQ1RK1 b kq - 0 9' => {
			eco => 'C09g',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. c3

			variation => N__('French: Tarrasch, Open, Main Line, 9. c3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2nppp/2n5/1Bbp4/8/2P2N2/PP1N1PPP/R1BQ1RK1 w - - 1 10',
			},
		},
		'r1bqk2r/pp2nppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQ1RK1 w kq - 0 9' => {
			eco => 'C09d',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 7.O-O'),
			moves => {
				'c2c3' => 'r1bqk2r/pp2nppp/2n5/1Bbp4/8/2P2N2/PP1N1PPP/R1BQ1RK1 b kq - 0 9',
				'd2b3' => 'r1bqk2r/pp2nppp/2n5/1Bbp4/8/1N3N2/PPP2PPP/R1BQ1RK1 b kq - 1 9',
			},
		},
		'r1bqk2r/pp2nppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQ1RK1 w kq - 2 9' => {
			eco => 'C09f',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. dxc5 Bxc5 8. O-O Ne7

			variation => N__('French: Tarrasch, Open, Main Line'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n5/3p4/1bpP4/2N2NP1/PP1BPPBP/R2Q1RK1 b kq - 5 9' => {
			eco => 'D33h',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 c4 7. Bg2 Bb4 8. O-O Nge7 9. Bd2

			variation => N__('QGD Tarrasch: Swedish, 9. Bd2'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n5/3p4/1bpP4/2N2NP1/PP2PPBP/R1BQ1RK1 w kq - 4 9' => {
			eco => 'D33f',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 c4 7. Bg2 Bb4 8. O-O Nge7

			variation => N__('QGD Tarrasch: Swedish, 7. Bg2 Bb4 8.O-O Nge7'),
			moves => {
				'a2a3' => 'r1bqk2r/pp2nppp/2n5/3p4/1bpP4/P1N2NP1/1P2PPBP/R1BQ1RK1 b kq - 0 9',
				'c1d2' => 'r1bqk2r/pp2nppp/2n5/3p4/1bpP4/2N2NP1/PP1BPPBP/R2Q1RK1 b kq - 5 9',
				'e2e4' => 'r1bqk2r/pp2nppp/2n5/3p4/1bpPP3/2N2NP1/PP3PBP/R1BQ1RK1 b kq e3 0 9',
				'f3e5' => 'r1bqk2r/pp2nppp/2n5/3pN3/1bpP4/2N3P1/PP2PPBP/R1BQ1RK1 b kq - 5 9',
			},
		},
		'r1bqk2r/pp2nppp/2n5/3p4/1bpP4/P1N2NP1/1P2PPBP/R1BQ1RK1 b kq - 0 9' => {
			eco => 'D33g',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 c4 7. Bg2 Bb4 8. O-O Nge7 9. a3

			variation => N__('QGD Tarrasch: Swedish, 9. a3'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n5/3p4/1bpPP3/2N2NP1/PP3PBP/R1BQ1RK1 b kq e3 0 9' => {
			eco => 'D33j',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 c4 7. Bg2 Bb4 8. O-O Nge7 9. e4

			variation => N__('QGD Tarrasch: Swedish, 9. e4'),
			moves => {
				'd5e4' => 'r1bqk2r/pp2nppp/2n5/8/1bpPp3/2N2NP1/PP3PBP/R1BQ1RK1 w kq - 0 10',
			},
		},
		'r1bqk2r/pp2nppp/2n5/3pN3/1bpP4/2N3P1/PP2PPBP/R1BQ1RK1 b kq - 5 9' => {
			eco => 'D33i',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 c4 7. Bg2 Bb4 8. O-O Nge7 9. Ne5

			variation => N__('QGD Tarrasch: Swedish, 9. Ne5'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2n5/8/1bpPp3/2N2NP1/PP3PBP/R1BQ1RK1 w kq - 0 10' => {
			eco => 'D33k',
                        # TRANSLATORS: 1. d4 d5 2. c4 e6 3. Nc3 c5 4. cxd5 exd5 5. Nf3 Nc6 6. g3 c4 7. Bg2 Bb4 8. O-O Nge7 9. e4 dxe4

			variation => N__('QGD Tarrasch: Swedish, 9. e4 dxe4'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2nb4/1B1p2B1/8/1N3N2/PPP2PPP/R2Q1RK1 b kq - 3 10' => {
			eco => 'C09n',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3 Bd6 10. Bg5

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bd6 10. Bg5'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2nb4/1B1p4/3N4/5N2/PPP2PPP/R1BQ1RK1 b kq - 3 10' => {
			eco => 'C09m',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3 Bd6 10. Nbd4

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bd6 10.Nbd4'),
			moves => {
			},
		},
		'r1bqk2r/pp2nppp/2nb4/1B1p4/8/1N3N2/PPP2PPP/R1BQ1RK1 w kq - 2 10' => {
			eco => 'C09l',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3 Bd6

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bd6'),
			moves => {
				'b3d4' => 'r1bqk2r/pp2nppp/2nb4/1B1p4/3N4/5N2/PPP2PPP/R1BQ1RK1 b kq - 3 10',
				'c1g5' => 'r1bqk2r/pp2nppp/2nb4/1B1p2B1/8/1N3N2/PPP2PPP/R2Q1RK1 b kq - 3 10',
				'f1e1' => 'r1bqk2r/pp2nppp/2nb4/1B1p4/8/1N3N2/PPP2PPP/R1BQR1K1 b kq - 3 10',
			},
		},
		'r1bqk2r/pp2nppp/2nb4/1B1p4/8/1N3N2/PPP2PPP/R1BQR1K1 b kq - 3 10' => {
			eco => 'C09o',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nd2 c5 4. exd5 exd5 5. Ngf3 Nc6 6. Bb5 Bd6 7. O-O Ne7 8. dxc5 Bxc5 9. Nb3 Bd6 10. Re1

			variation => N__('French: Tarrasch, Open, Main Line, 9. Nb3 Bd6 10. Re1'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2nppp/2nb4/1B1p4/8/1N3N2/PPP2PPP/R1BQR1K1 w - - 4 11',
			},
		},
		'r1bqk2r/pp2nppp/2nb4/1BPp4/8/5N2/PPPN1PPP/R1BQ1RK1 b kq - 0 8' => {
			eco => 'C09d',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 7.O-O'),
			moves => {
				'd6c5' => 'r1bqk2r/pp2nppp/2n5/1Bbp4/8/5N2/PPPN1PPP/R1BQ1RK1 w kq - 0 9',
			},
		},
		'r1bqk2r/pp2nppp/2nb4/1Bpp4/3P4/5N2/PPPN1PPP/R1BQ1RK1 w kq - 6 8' => {
			eco => 'C09d',
                        # TRANSLATORS:

			variation => N__('French: Tarrasch, Open, 7.O-O'),
			moves => {
				'd4c5' => 'r1bqk2r/pp2nppp/2nb4/1BPp4/8/5N2/PPPN1PPP/R1BQ1RK1 b kq - 0 8',
			},
		},
		'r1bqk2r/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq e3 0 7' => {
			eco => 'A08',
                        # TRANSLATORS: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2 Nge7 7. e4

			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w - - 1 8',
			},
		},
		'r1bqk2r/pp2nppp/2nbp3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq - 3 7' => {
			eco => 'A08',
                        # TRANSLATORS:

			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 'r1bqk2r/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq e3 0 7',
			},
		},
		'r1bqk2r/pp2nppp/3p1n2/2pPp3/2P1P3/2PB1N2/P4PPP/R1BQK2R w KQkq - 1 10' => {
			eco => 'E41',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. e3 c5 5. Bd3 Nc6 6. Nf3 Bxc3+ 7. bxc3 d6 8. e4 e5 9. d5 Ne7

			variation => N__('Nimzo-Indian: Hübner, 8. e4 e5 9. d5 Ne7'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2n2np1/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq d6 0 7' => {
			eco => 'A38d',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O d5

			variation => N__('English: Symmetrical, Main Line, 6.O-O d5'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2n3p1/2p5/3PP3/2P1BN2/P4PPP/R2QKB1R w KQkq - 2 9' => {
			eco => 'D85t',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 g6 3. Nc3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. bxc3 Bg7 7. Nf3 c5 8. Be3 Nc6

			variation => N__('Grünfeld: Modern Exchange, 8. Be3 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 b kq - 3 7' => {
			eco => 'A36u',
                        # TRANSLATORS:

			variation => N__('English: Symmetrical, Botvinnik, 5... d6 6.Nge2 Nf6'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 w - - 4 8',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQkq - 2 7' => {
			eco => 'A36u',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6

			variation => N__('English: Symmetrical, Botvinnik, 5... d6 6.Nge2 Nf6'),
			moves => {
				'e1g1' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 b kq - 3 7',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 0 7' => {
			eco => 'A38e',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O d6

			variation => N__('English: Symmetrical, Main Line, 6.O-O d6'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq - 2 7' => {
			eco => 'B26b',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Nf6

			variation => N__('Sicilian: Closed, 6. Be3 Nf6'),
			moves => {
				'h2h3' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/4P3/2NPB1PP/PPP2PB1/R2QK1NR b KQkq - 0 7',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4P3/2NPB1PP/PPP2PB1/R2QK1NR b KQkq - 0 7' => {
			eco => 'B26c',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Nf6 7. h3

			variation => N__('Sicilian: Closed, 6. Be3 Nf6 7. h3'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq - 2 7' => {
			eco => 'B25h',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 Nf6 7. Nf3

			variation => N__('Sicilian: Closed, 6. f4 Nf6 7. Nf3'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq - 1 7' => {
			eco => 'B25g',
                        # TRANSLATORS: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 Nf6

			variation => N__('Sicilian: Closed, 6. f4 Nf6'),
			moves => {
				'g1f3' => 'r1bqk2r/pp2ppbp/2np1np1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq - 2 7',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQkq - 4 8' => {
			eco => 'B72',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Bc4 Nc6

			variation => N__('Sicilian: Dragon, 6. Be3 Bg7 7. Bc4 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/2BNP3/2N1BP2/PPP3PP/R2QK2R b KQkq - 2 8' => {
			eco => 'B75',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 Nc6 8. Bc4

			variation => N__('Sicilian: Dragon, Yugoslav, 7... Nc6 8. Bc4'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN1BPPP/R1BQK2R b KQkq - 1 8' => {
			eco => 'B37i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6 8. Be2

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6 8. Be2'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN1BPPP/R1BQK2R w KQ - 2 9',
				'f6d7' => 'r1bqk2r/pp1nppbp/2np2p1/8/2P1P3/2N5/PPN1BPPP/R1BQK2R w KQkq - 2 9',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN2PPP/R1BQKB1R w KQkq - 0 8' => {
			eco => 'B37i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 Nf6 7. Nc3 d6

			variation => N__('Sicilian: Maroczy Bind, 6. Nc2 Nf6 7. Nc3 d6'),
			moves => {
				'f1e2' => 'r1bqk2r/pp2ppbp/2np1np1/8/2P1P3/2N5/PPN1BPPP/R1BQK2R b KQkq - 1 8',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R b KQkq - 1 8' => {
			eco => 'B38f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 d6 8. Be2

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 Nf6 7. Nc3 d6 8. Be2'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP3PPP/R2QKB1R w KQkq - 0 8' => {
			eco => 'B38e',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 Nf6 7. Nc3 d6

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 Nf6 7. Nc3 d6'),
			moves => {
				'f1e2' => 'r1bqk2r/pp2ppbp/2np1np1/8/2PNP3/2N1B3/PP2BPPP/R2QK2R b KQkq - 1 8',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 5 8' => {
			eco => 'B73',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. O-O

			variation => N__('Sicilian: Dragon, Classical, 8.O-O'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w - - 6 9',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQkq - 4 8' => {
			eco => 'B72',
                        # TRANSLATORS:

			variation => N__('Sicilian: Dragon, Classical Attack'),
			moves => {
				'd1d2' => 'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/R3K2R b KQkq - 5 8',
				'd4b3' => 'r1bqk2r/pp2ppbp/2np1np1/8/4P3/1NN1B3/PPP1BPPP/R2QK2R b KQkq - 5 8',
				'e1g1' => 'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b kq - 5 8',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/R3K2R b KQkq - 5 8' => {
			eco => 'B72',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. Qd2

			variation => N__('Sicilian: Dragon, Classical, Amsterdam Variation'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/2N1B3/PPPQBPPP/R3K2R w KQ - 6 9',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1BP2/PPP3PP/R2QKB1R w KQkq - 1 8' => {
			eco => 'B75',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 Nc6

			variation => N__('Sicilian: Dragon, Yugoslav, 7... Nc6'),
			moves => {
				'd1d2' => 'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R b KQkq - 2 8',
				'f1c4' => 'r1bqk2r/pp2ppbp/2np1np1/8/2BNP3/2N1BP2/PPP3PP/R2QK2R b KQkq - 2 8',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R b KQkq - 2 8' => {
			eco => 'B75',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. f3 Nc6 8. Qd2

			variation => N__('Sicilian: Dragon, Yugoslav, 7... Nc6 8. Qd2'),
			moves => {
				'c8d7' => 'r2qk2r/pp1bppbp/2np1np1/8/3NP3/2N1BP2/PPPQ2PP/R3KB1R w KQkq - 3 9',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/3NP3/2N5/PPP1BPPP/R1BQ1RK1 w kq - 4 8' => {
			eco => 'B70l',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O Nc6

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O Nc6'),
			moves => {
				'd4b3' => 'r1bqk2r/pp2ppbp/2np1np1/8/4P3/1NN5/PPP1BPPP/R1BQ1RK1 b kq - 5 8',
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/4P3/1NN1B3/PPP1BPPP/R2QK2R b KQkq - 5 8' => {
			eco => 'B72',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be3 Bg7 7. Be2 Nc6 8. Nb3

			variation => N__('Sicilian: Dragon, Classical, Nottingham Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/8/4P3/1NN5/PPP1BPPP/R1BQ1RK1 b kq - 5 8' => {
			eco => 'B70m',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 d6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 g6 6. Be2 Bg7 7. O-O Nc6 8. Nb3

			variation => N__('Sicilian: Dragon, 6. Be2 Bg7 7.O-O Nc6 8. Nb3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/2np1np1/8/4P3/1NN5/PPP1BPPP/R1BQ1RK1 w - - 6 9',
			},
		},
		'r1bqk2r/pp2ppbp/2np2pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq - 2 7' => {
			eco => 'A37h',
                        # TRANSLATORS: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. O-O Nh6

			variation => N__('English: Symmetrical, 5. Nf3 d6 6.O-O Nh6'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np2pn/8/2P1P3/8/PPN1BPPP/RNBQK2R w KQkq - 2 8' => {
			eco => 'B37f',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Nc2 d6 7. Be2 Nh6

			variation => N__('Sicilian: Maroczy Bind, Simagin Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np2pn/8/2PNP3/2N1B3/PP3PPP/R2QKB1R w KQkq - 2 8' => {
			eco => 'B38c',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Bg7 6. Be3 d6 7. Nc3 Nh6

			variation => N__('Sicilian: Maroczy Bind, 6. Be3 d6 7. Nc3 Nh6'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/3p1np1/6B1/2PQP3/2N5/PP2BPPP/R3K2R b KQkq - 2 9' => {
			eco => 'B36k',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. Bg5

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9. Bg5'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/3p1np1/6B1/2PQP3/2N5/PP2BPPP/R3K2R w KQ - 3 10',
			},
		},
		'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N1B3/PP2BPPP/R3K2R b KQkq - 2 9' => {
			eco => 'B36n',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. Be3

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9. Be3'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp2ppbp/3p1np1/8/2PQP3/2N1B3/PP2BPPP/R3K2R w KQ - 3 10',
			},
		},
		'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N5/PP2BPPP/R1B1K2R w KQkq - 1 9' => {
			eco => 'B36i',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7'),
			moves => {
				'c1e3' => 'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N1B3/PP2BPPP/R3K2R b KQkq - 2 9',
				'c1g5' => 'r1bqk2r/pp2ppbp/3p1np1/6B1/2PQP3/2N5/PP2BPPP/R3K2R b KQkq - 2 9',
				'e1g1' => 'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N5/PP2BPPP/R1B2RK1 b kq - 2 9',
			},
		},
		'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N5/PP2BPPP/R1B2RK1 b kq - 2 9' => {
			eco => 'B36j',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7 9. O-O

			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7 9.O-O'),
			moves => {
			},
		},
		'r1bqk2r/pp3p2/2n1p1pp/2ppP3/3P2Q1/2PB1N2/P1PK1PPP/R6R w kq - 2 12' => {
			eco => 'C12r',
                        # TRANSLATORS: 1. e4 e6 2. d4 d5 3. Nc3 Nf6 4. Bg5 Bb4 5. e5 h6 6. Bd2 Bxc3 7. bxc3 Ne4 8. Qg4 g6 9. Bd3 Nxd2 10. Kxd2 c5 11. Nf3 Nc6

			variation => N__('French: MacCutcheon, Main Line, 11. Nf3 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp3pbp/2n1pnp1/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w kq - 1 8' => {
			eco => 'A08',
                        # TRANSLATORS: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 g6 7. e4 Bg7

			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp3pp1/2pbpn1p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R w KQkq - 1 11' => {
			eco => 'B17o',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4 10. Qxe4 Nf6

			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3, Main Line 10.Qxe4 Nf6'),
			moves => {
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/2Pp4/2P1n3/P7/1PQ1PPPP/R1B1KBNR w KQkq - 1 9' => {
			eco => 'E37',
                        # TRANSLATORS: 1. d4 Nf6 2. c4 e6 3. Nc3 Bb4 4. Qc2 d5 5. a3 Bxc3+ 6. Qxc3 Ne4 7. Qc2 c5 8. dxc5 Nc6

			variation => N__('Nimzo-Indian: Classical, Noa, Main Line, 7. Qc2 c5 8.dxc5 Nc6'),
			moves => {
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2N2N2/PP1B1PPP/R2QKB1R w KQkq - 2 9' => {
			eco => 'B14w',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Bd2 Nc6

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2 Nc6'),
			moves => {
				'f1d3' => 'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PP1B1PPP/R2QK2R b KQkq - 3 9',
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2N2N2/PPQ2PPP/R1B1KB1R w KQkq - 2 9' => {
			eco => 'B14t',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Qc2 Nc6

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Qc2 Nc6'),
			moves => {
				'f1d3' => 'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PPQ2PPP/R1B1K2R b KQkq - 3 9',
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PP1B1PPP/R2QK2R b KQkq - 3 9' => {
			eco => 'B14w',
                        # TRANSLATORS:

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2 Nc6'),
			moves => {
				'e8g8' => 'r1bq1rk1/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PP1B1PPP/R2QK2R w KQ - 4 10',
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PPQ2PPP/R1B1K2R b KQkq - 3 9' => {
			eco => 'B14u',
                        # TRANSLATORS: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Qc2 Nc6 9. Bd3

			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Qc2 Nc6 9. Bd3'),
			moves => {
			},
		},
		'r1bqk2r/pp3ppp/2n1pn2/3P4/8/P1N5/1PP2PPP/R1BQKB1R b KQkq - 0 9' => {
			eco => 'B45s',
                        # TRANSLATORS:

			variation => N__('Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7. a3 Bxc3+ 8.Nxc3 d5'),
			moves => {
				'e6d5' => 'r1bqk2r/pp3ppp/2n2n2/3p4/8/P1N5/1PP2PPP/R1BQKB1R w KQkq - 0 10',
			},
		},
		'r1bqk2r/pp3ppp/2n1pn2/3p4/4P3/P1N5/1PP2PPP/R1BQKB1R w KQkq d6 0 9' => {
			eco => 'B45s',
                        # TRANSLATORS: 1. e4 c5 2. Nf3 e6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 Nf6 6. Ndb5 Bb4 7. a3 Bxc3+ 8. Nxc3 d5

			variation => N_