package Import::Into::As;
use strict;
use warnings;

our $VERSION = '0.000001';

use Import::Into;
use Carp qw/croak/;

sub import::into::as {
    my $class = shift;
    my ($target, $rename, @args) = @_;

    # need to bump level by 2, 1 for the call to _get_imported_syms,
    # another for the call to import::into
    my $bump_level = 2;

    my ($dest, $level);

    if (ref $target) {
        $dest = $target->{package} if exists $target->{package};
        if (exists $target->{level}) {
            $level = $target->{level};
            $dest  ||= caller($level);
            $target->{level} += $bump_level;
        }
    }
    elsif ($target =~ m/^\d+$/) {
        $level = $target;
        $dest  = caller($level);
        $target += $bump_level;
    }
    else {
        $dest = $target;
    }

    croak "unable to find destination package!"
        unless $dest;

    # Get all the symbols that should be imported. This is a hashref of symbol
    # names where value is an arrayref of glob entries.
    my %syms = _get_imported_syms($class, $target, \@args, $dest);

    # Load each import symbol into the destination namespace
    for my $sym (keys %syms) {
        next if $sym eq '__ANON__';

        # Coderefs can be renamed.
        my $subname = $rename->{$sym} || $sym;

        no strict 'refs';

        # Only support the big 4 types in the glob.
        *{"$dest\::$subname"} = $syms{$sym}{CODE}   if $syms{$sym}{CODE};
        *{"$dest\::$sym"}     = $syms{$sym}{SCALAR} if $syms{$sym}{SCALAR};
        *{"$dest\::$sym"}     = $syms{$sym}{HASH}   if $syms{$sym}{HASH};
        *{"$dest\::$sym"}     = $syms{$sym}{ARRAY}  if $syms{$sym}{ARRAY};
    }
}

# This will import the symbols into the desired namespace, but will localize
# the namespace to ensure no changes are actually made. It will then return all
# the symbols that were imported into the namespace and restore the
# namespace to its previous state.
sub _get_imported_syms {
    my ($module, $target, $args, $dest) = @_;

    # This block does some crazy things, we need to turn off strict refs
    no strict 'refs';

    # Localize the entire destination namespace so that we don't actually do
    # anything to it. This also means we do not need to turn off redefine
    # warnings.
    local *{"$dest\::"};

    # Import into the namespace we just protected with local.
    $module->import::into($target, @$args);

    # Generate a (sym => { CODE => ..., SCALAR => ..., ... }, ...) hash.
    return map {
        my $code   = *{"$dest\::$_"}{CODE};
        my $array  = *{"$dest\::$_"}{ARRAY};
        my $hash   = *{"$dest\::$_"}{HASH};
        my $scalar = *{"$dest\::$_"}{SCALAR};

        # Scalars in globs are crazy... this is "good enough", though it will
        # fail for anonymous scalar exports that are set to undef.
        $scalar = undef unless defined($$scalar) || $scalar == \${"$module\::$_"};

        $_ => {
            $code   ? (CODE   => $code)   : (),
            $array  ? (ARRAY  => $array)  : (),
            $hash   ? (HASH   => $hash)   : (),
            $scalar ? (SCALAR => $scalar) : (),
        }
    } keys %{"$dest\::"};
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Import::Into::As - Wrapper around Import::Into that lets you rename subs.

=head1 DESCRIPTION

This wrapper provides a new function C<import::into::as> which works just like
C<import::into()>, except that you can rename the subs being exported around.

This is a wrapper around L<Import::Into>, which is an awesome tool that lets
you import any module into any other module. The problem is that not all
modules you are importing from have the ability to rename exports. This is not
a problem if you are exporting from modules that use L<Sub::Exporter> or
L<Exporter::Declare>, but it is a problem if you want to rename a sub exported
from L<Exporter>.

B<Note:> If you are exporting from modules that allow you to rename the exports
in their own syntax then you should use that. This module will be slower than
using the exporters rename syntax when available. This module also has some
limitations, see the L</LIMITATIONS> section.

=head1 SYNOPSYS

    use Import::Into::As;

    # Export 'foo' and 'baz' from Some::Exporter into Destination::Package.
    # Rename 'foo' to 'bar' in the process.
    Some::Exporter->import_into_as('Destination::Package', {foo => bar}, qw/foo baz/);

=head1 METHODS

=head2 $package->import::into::as($target, \%renames, @arguments);

=over 4

=item $package

Package to export from.

=item $target

This can be a target package, a call level, or a hashref, see
L<Import::Into/METHODS> for an expanded explanation of C<$target>.

=item \%renames

This must be a hashref where the keys are the names of a subs exported by the
package, and the values are the new names you want them to have.

    { 'export_name' => 'new_name', ... }

=item \@arguments

Import arguments for the package.

=back

=head1 HOW IT WORKS

This package will localize the taget packages namespace, temporarily making it
empty. This will then use L<Import::Into> to import into the now empty
namespace. Once the import is complete the new symbol table is copied. Once the
original symbol table is restored the new imports will be placed into the
actual namespace, but under the new names.

=head2 WHY IS IT SO COMPLICATED!?

There are some problems that have to be worked around:

=over 4

=item Not all export tools support export renaming

This is not something we can directly do anything about, so we have to do our
own rename post-import.

=item It can be hard to determine if an exporter does support renaming

This makes it hard to defer to the exporter when it is possible.

=item There may already be a sub defined in the package with the export name

This is actually pretty easy to solve. We can look for an existing sub for any
key in the rename hash, store it, and restore it after the import. None of this
localizing the package would be necessary if this was the last problem.

=item There may NOT already be a sub defined in the package with the export name

This is suprisingly a MUCH harder problem to solve. The import process will put
a sub into the package namespace under the original name. We can then copy the
sub to the new name easily enough. The hard part is removing the sub from the
old name, it is surprisingly difficult!

=back

=head1 LIMITATIONS

=over 4

=item Only subs can be renamed (this may change in the future)

It would not be difficult to add renaming support for other types, but there
has not been any need yet.

=item Only CODE, SCALAR, HASH, and ARRAY exports are allowed

Typeglobs have other keys such as C<FORMATTER>, but the big 4 listed here
shoudl be sufficient for nearly all use cases.

=item Exporting endefined scalars that do not exist in the exporting namespace fail

This is a pretty rare edge case, but it can happen.

If the exporter does this, then the scalar export will not work:

    sub import {
        my $caller = caller;

        my $FOO = undef;

        *{"$caller\::FOO"} = \$FOO;
    }

This reasonf or this is that a typeglob always has a refernece to undef in the
SCALAR slot. We cannot easily check if a scalar was imported into a specific
name whent he scalar is undef. As a backup we check if the reference is the
same reference as the one in the exporting package, but in cases like the one
above that is not helpful.

=back

=head1 SOURCE

The source code repository for Import-Into-As can be found at
F<http://github.com/exodist/Import-Into-As>.

=head1 MAINTAINERS

=over 4

=item Chad Granum E<lt>exodist@cpan.orgE<gt>

=back

=head1 AUTHORS

=over 4

=item Chad Granum E<lt>exodist@cpan.orgE<gt>

=back

=head1 COPYRIGHT

Copyright 2015 Chad Granum E<lt>exodist7@gmail.comE<gt>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See F<http://dev.perl.org/licenses/>

=cut
