# NAME

Import::Into::As - Wrapper around Import::Into that lets you rename subs.

# DESCRIPTION

This wrapper provides a new function `import::into::as` which works just like
`import::into()`, except that you can rename the subs being exported around.

This is a wrapper around [Import::Into](https://metacpan.org/pod/Import::Into), which is an awesome tool that lets
you import any module into any other module. The problem is that not all
modules you are importing from have the ability to rename exports. This is not
a problem if you are exporting from modules that use [Sub::Exporter](https://metacpan.org/pod/Sub::Exporter) or
[Exporter::Declare](https://metacpan.org/pod/Exporter::Declare), but it is a problem if you want to rename a sub exported
from [Exporter](https://metacpan.org/pod/Exporter).

**Note:** If you are exporting from modules that allow you to rename the exports
in their own syntax then you should use that. This module will be slower than
using the exporters rename syntax when available. This module also has some
limitations, see the ["LIMITATIONS"](#limitations) section.

# SYNOPSYS

    use Import::Into::As;

    # Export 'foo' and 'baz' from Some::Exporter into Destination::Package.
    # Rename 'foo' to 'bar' in the process.
    Some::Exporter->import_into_as('Destination::Package', {foo => bar}, qw/foo baz/);

# METHODS

## $package->import::into::as($target, \\%renames, @arguments);

- $package

    Package to export from.

- $target

    This can be a target package, a call level, or a hashref, see
    ["METHODS" in Import::Into](https://metacpan.org/pod/Import::Into#METHODS) for an expanded explanation of `$target`.

- \\%renames

    This must be a hashref where the keys are the names of a subs exported by the
    package, and the values are the new names you want them to have.

        { 'export_name' => 'new_name', ... }

- \\@arguments

    Import arguments for the package.

# HOW IT WORKS

This package will localize the taget packages namespace, temporarily making it
empty. This will then use [Import::Into](https://metacpan.org/pod/Import::Into) to import into the now empty
namespace. Once the import is complete the new symbol table is copied. Once the
original symbol table is restored the new imports will be placed into the
actual namespace, but under the new names.

## WHY IS IT SO COMPLICATED!?

There are some problems that have to be worked around:

- Not all export tools support export renaming

    This is not something we can directly do anything about, so we have to do our
    own rename post-import.

- It can be hard to determine if an exporter does support renaming

    This makes it hard to defer to the exporter when it is possible.

- There may already be a sub defined in the package with the export name

    This is actually pretty easy to solve. We can look for an existing sub for any
    key in the rename hash, store it, and restore it after the import. None of this
    localizing the package would be necessary if this was the last problem.

- There may NOT already be a sub defined in the package with the export name

    This is suprisingly a MUCH harder problem to solve. The import process will put
    a sub into the package namespace under the original name. We can then copy the
    sub to the new name easily enough. The hard part is removing the sub from the
    old name, it is surprisingly difficult!

# LIMITATIONS

- Only subs can be renamed (this may change in the future)

    It would not be difficult to add renaming support for other types, but there
    has not been any need yet.

- Only CODE, SCALAR, HASH, and ARRAY exports are allowed

    Typeglobs have other keys such as `FORMATTER`, but the big 4 listed here
    shoudl be sufficient for nearly all use cases.

- Exporting endefined scalars that do not exist in the exporting namespace fail

    This is a pretty rare edge case, but it can happen.

    If the exporter does this, then the scalar export will not work:

        sub import {
            my $caller = caller;

            my $FOO = undef;

            *{"$caller\::FOO"} = \$FOO;
        }

    This reasonf or this is that a typeglob always has a refernece to undef in the
    SCALAR slot. We cannot easily check if a scalar was imported into a specific
    name whent he scalar is undef. As a backup we check if the reference is the
    same reference as the one in the exporting package, but in cases like the one
    above that is not helpful.

# SOURCE

The source code repository for Import-Into-As can be found at
`http://github.com/exodist/Import-Into-As`.

# MAINTAINERS

- Chad Granum &lt;exodist@cpan.org>

# AUTHORS

- Chad Granum &lt;exodist@cpan.org>

# COPYRIGHT

Copyright 2015 Chad Granum &lt;exodist7@gmail.com>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See `http://dev.perl.org/licenses/`
