package TM::IP::Surface::Controller::Root;

use strict;
use warnings;
use Switch;

use Data::Dumper;
use HTTP::Status qw(:constants);
use Image::Magick;

use base 'Catalyst::Controller::REST';

=pod

=head1 NAME

TM::IP::Surface::Controller::Root - Root Controller for TM::IP::Surface

=head1 DESCRIPTION

TMIP is a suite of RESTful protocols to communicate with Topic Maps (TM) based servers. This
controller deals with the I<surface subspace>, i.e. an aspect of a topic map which visualizes its 3D
structure.

=head2 Map Addressing

Like for all TMIP protocols, the topic map is addressed first using a URL structure like

  http://my.server/internet/web/

where C</> is the root map, C<internet> one submap and C<web> a submap under that. Underneath that
mount point the surface maps reside, by selected with C<.surface/>:

  http://my.server/internet/web/.surface/x1.png
  http://my.server/internet/web/.surface/x2.png
  ...
  http://my.server/internet/web/.surface/x9.png

The factors C<x1>, C<x2>, ..., C<x9> represents different zoom levels of the map. Each
zoom factor has it own set of tiles, such as

  http://my.server/internet/web/.surface/x3/60x40/20x3.png

The directory name I<60x40> defines the number of tiles, the file name I<20x3>inside the directory
addresses one particular tile. The tiles are autogenerated whenever they are first requested.

=cut

__PACKAGE__->config->{namespace} = '';
__PACKAGE__->config->{default} = 'text/x-yaml';

=pod

=head2 HTTP Methods

The operations offered by this subspace are (the C<__> are only to separate relevant pieces, they do
not appear in URLs):

=over

=item B<surface> GET

I<GET  ...../.surface/>__I<x>__n__.png

This method retrieves the named PNG surface file. The resolution factor C<n> can be from C<1> to
C<9>. Only PNG is supported.

=cut

sub surface  : Regex('(.+)\.surface/x([1-9]).png$')  : ActionClass('REST') { }

sub surface_GET {
    my ($self, $c) = @_;

    my ($mappath, $zoom) = @{ $c->req->snippets };
    my $file = $c->config->{mapbase} . $mappath . '.surface/x' . $zoom . '.png';         # produce the real path
    $c->log->debug ( "file: $file, $zoom" );
    
    if (! -f $file) {
	$self->status_not_found ($c, message => "Cannot find object '$file'" );
    } elsif (! -r $file) {
	$self->status_bad_request($c, message => "Cannot do what you have asked!");
    } else {
	_send_file ($c, $file);
    }
}

=pod

=item B<surface> PUT

I<PUT  ...../.surface/>__I<x>__n__.png

This method takes the message body, tests whether this is a PNG file (currently looking only at the
C<Content-Type> header) and tries to store that as the surface with the given zoom factor (that
between C<1> and C<9>). If the file already exists, it will be overwritten.

=cut

sub surface_PUT {
    my ($self, $c) = @_;
    my ($mappath, $zoom) = @{ $c->req->snippets };
    my $file = $c->config->{mapbase} . $mappath . '.surface/x' . $zoom . '.png';         # produce the real path
    $c->log->debug ( "PUT file: $file, $zoom" );
    $c->log->debug ( $c->request->content_type );

    if ($c->request->content_type ne 'image/png') {
	$self->status_bad_request($c, message => "Cannot store non-PNG file '$file'" );
    } elsif (rename $c->request->body, $file) {
	$self->status_created ($c, 
			       location => $c->req->uri->as_string,
			       entity   => undef);
    } else {
	$self->status_bad_request($c, message => "Cannot store file '$file'" );
    }
}

=pod

=item B<tile> GET

I<GET  ...../.surface/>__I<x>__n__I</>__nxm__I</>__nxm__.png

This method retrieves the addressed tile from the surface. Each zoom factor C<n> has its own
directory. In that is one directory for one set of tiles, depending on the size of the tiles. Each
tile inside one such set is addressed by its coordinate, (0, 0) being the the left-top tile.

=cut

sub tile : Regex('(.+)\.surface/x([1-9])/(\d+x\d+)/(\d+x\d+).png$')  : ActionClass('REST') { }

sub tile_GET {
    my ($self, $c) = @_;
    my ($mappath, $zoom, $resolution, $position) = @{ $c->req->snippets };

    my $dir   = $c->config->{mapbase} . "$mappath.surface/x$zoom";
    my $tfile = "$dir/$resolution/$position.png";
    my $mfile = "$dir.png";

    $c->log->debug ( "file: $mfile, $tfile" );

    if (! -f $mfile) {                                                              # the master file does not exist
 	$self->status_not_found ($c, message => "Cannot find object '$mfile'" );
    } elsif (! -f $tfile                                                            # tile does not exist
             ||
             (stat $tfile)[9] < (stat $mfile)[9]                                    # or it exists and is older than the map
	     ) {                                                  
 	_mk_tiles  ($c, $dir, $resolution);                                         # create the tiles
	if (! -f $tfile) {                                                          # still not here?
	    $self->status_not_found ($c, message => "Cannot find/generate object '$tfile'" );
	} else {
	    _send_file ($c, $tfile);
	}
    } else {                                                                        # tile file obviously already exists and is up to date
 	_send_file ($c, $tfile);
    }
}

sub _send_file {
    my $c = shift;
    my $file = shift;

    use File::Slurp;
    my $content = read_file ( $file );                                                # try to find tile, should be there
    $c->response->body           ($content);
    $c->response->content_length (length ($content) );
    $c->response->content_type   ('image/png');
}


sub _mk_tiles {
    my $c          = shift;                                                           # only needed for log
    my $dir        = shift;
    my $resolution = shift;
    my ($m, $n) = ($resolution =~ /(\d+)x(\d+)/);

    $c->log->debug ( "tiles $resolution $m $n" );

    mkdir "$dir"             unless -d "$dir";
    mkdir "$dir/$resolution" unless -d "$dir/$resolution";

    my $image = new Image::Magick;
    $image->Read("$dir.png");
    my ($X, $Y) = $image->Get('width', 'height');
    my ($tile_x, $tile_y) = (int($X/$m), int($Y/$n));

    $c->log->debug ( "tiles width $X $Y, tilex $tile_x, $tile_y" );

    for (    my $m = 0; (my $x = $m * $tile_x) < $X; $m++) {
	for (my $n = 0; (my $y = $n * $tile_y) < $Y; $n++) {
            my $tile = $image->Clone;
            $tile->Crop  (geometry => "${tile_x}x${tile_y}+$x+$y");
            $tile->Write ("$dir/$resolution/${m}x${n}.png");
	}
    }

}

sub end   :                              ActionClass('Serialize') {}

=pod

=back

=head1 AUTHOR

Robert Barta, C<< <rho at devc.at> >>

=head1 SEE ALSO

L<TM::IP>

L<http://kill.devc.at/internet/semantic-web/topic-maps/tmip>

=head1 COPYRIGHT & LICENSE

Copyright 200[9] Robert Barta, all rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl
itself.

=cut

our $VERSION = '0.02';

1;

__END__

