package Dist::Zilla::Plugin::Test::TidyAll;

use strict;
use warnings;
use namespace::autoclean;

our $VERSION = '0.01';

use B;
use Dist::Zilla::File::InMemory;

use Moose;

has conf_file => (
    is        => 'ro',
    isa       => 'Str',
    predicate => '_has_conf_file',
);

with qw(
    Dist::Zilla::Role::FileGatherer
    Dist::Zilla::Role::PrereqSource
);

sub register_prereqs {
    my ($self) = @_;

    $self->zilla->register_prereqs(
        {
            type  => 'requires',
            phase => 'develop',
        },

        # XXX - remove this once JSWARTZ gives me perms on Test::Code::TidyAll
        'Code::TidyAll'       => '0.24',
        'Test::Code::TidyAll' => '0.24',
        'Test::More'          => '0.88',
    );

    return;
}

sub gather_files {
    my ($self) = @_;

    $self->add_file(
        Dist::Zilla::File::InMemory->new(
            {
                name    => 'xt/release/tidyall.t',
                content => $self->_file_content(),
            }
        ),
    );

    return;
}

my $head = <<'EOF';
#!perl
# This file was automatically generated by Dist::Zilla::Plugin::Test::TidyAll

use Test::Code::TidyAll 0.24;
use Test::More 0.88;

EOF

sub _file_content {
    my $self = shift;

    my $content = $head;
    my $args    = q{};
    if ( $self->_has_conf_file() ) {
        ## no critic (Subroutines::ProhibitCallsToUnexportedSubs)
        $args = ' conf_file => ' . B::perlstring( $self->conf_file() ) . q{ };
        ## use critic
    }

    $content .= <<"EOF";
tidyall_ok($args);

done_testing();
EOF

    return $content;
}

__PACKAGE__->meta->make_immutable;

1;

# ABSTRACT: Adds a tidyall test to your distro

__END__

=pod

=head1 NAME

Dist::Zilla::Plugin::Test::TidyAll - Adds a tidyall test to your distro

=head1 VERSION

version 0.01

=head1 SYNOPSIS

  [Test::TidyAll]

=head1 DESCRIPTION

This is a L<Dist::Zilla> plugin that create a tidyall test in your distro
using L<Test::Code::TidyAll>'s C<tidyall_ok()> sub.

L<Code::TidyAll> C<0.24> and L<Test::More> C<0.88> will be added as C<develop
requires> dependencies.

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Test::TidyAll

=head1 CONFIGURATION

This plugin accepts the following configuration options:

=head2 conf_file

If this is provided, it will be passed to the C<tidyall_ok()> sub.

Note that you must provide a configuration file, either by using one of the
default files that L<Test::Code::TidyAll> looks for, or by providing another
file via this option.

=head1 WHAT TO IGNORE IN YOUR TIDYALL CONFIG

Many other plugins also add files to the final distro, and these may not pass
your tidyall checks. You will need to ignore these files files in your tidyall
config.

Because of the way tidyall works, you'll also want to ignore the F<blib>
directory. Here is a suggested set of C<ignore> directives for a dzil-based
distro.

    ignore = t/00-*
    ignore = t/author-*
    ignore = t/release-*
    ignore = blib/**/*
    ignore = .build/**/*
    ignore = {{Your-Plugin-Name}}*/**/*

This presumes that you will not create any tests of your own that start with
"00-".

=head1 AUTHOR

Dave Rolsky <autarch@urth.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by Dave Rolsky.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut
