#!perl -T
use 5.006;
use strict;
use warnings FATAL => 'all';
use Test::More;
use Time::Zone::Olson();
use POSIX();

$ENV{PATH} = '/bin:/usr/bin';
delete @ENV{'IFS', 'CDPATH', 'ENV', 'BASH_ENV'};

my $perl_date = 0;
my $bsd_date = 0;
my $test_gnu_date = `TZ="Australia/Melbourne" date -d "2015/02/28 11:00:00" +"%Y/%m/%d %H:%M:%S" 2>&1`;
chomp $test_gnu_date;
diag "Output of gnu date command:$test_gnu_date";
if (($test_gnu_date) && ($test_gnu_date eq '2015/02/28 11:00:00')) {
} else {
	my $test_bsd_date = `TZ="Australia/Melbourne" date -r 1425081600 +"%Y/%m/%d %H:%M:%S" 2>&1`;
	chomp $test_bsd_date;
	diag "Output of bsd date command:$test_bsd_date";
	if (($test_bsd_date) && ($test_bsd_date eq '2015/02/28 11:00:00')) {
		$bsd_date = 1;
	} else {
		$perl_date = 1;
	}
}

my $pack_q_ok = 0;
eval { my $q = pack 'q>', 2 ** 33; my $p = unpack 'q>', $q; $pack_q_ok = 1; };
diag"Results of unpack:$@";

$ENV{TZ} ||= 'Australia/Melbourne';
diag("TZ environment variable is $ENV{TZ}");
my $zoneinfo = Time::Zone::Olson->new();
ok($zoneinfo, "Time::Zone::Olson->new() generates an object");
ok($zoneinfo->directory() eq '/usr/share/zoneinfo', "\$zoneinfo->directory() returns the correct directory");
ok($zoneinfo->timezone() =~ /^\w+(\/[\w\-\/]+)?$/, "\$zoneinfo->timezone() parses correctly");
ok($zoneinfo->area() . '/' . $zoneinfo->location() eq $ENV{TZ}, "\$zoneinfo->area() and \$zoneinfo->location() contain the area and location of the current timezone");
ok((grep /^Australia$/, $zoneinfo->areas()), "Found 'Australia' in \$zoneinfo->areas()");
ok((grep /^Melbourne$/, $zoneinfo->locations('Australia')), "Found 'Melbourne' in \$zoneinfo->areas('Australia')");
ok($zoneinfo->comment('Australia/Melbourne') eq 'Victoria', "\$zoneinfo->comment('Australia/Melbourne') returns 'Victoria'");
my $now = time;
my @correct_localtime = localtime $now;
my @test_localtime = $zoneinfo->local_time($now);
my $matched = 1;
foreach my $index (0 .. (( scalar @correct_localtime )- 1)) {
	if ($correct_localtime[$index] ne $test_localtime[$index]) {
		$matched = 0;
	}
}
foreach my $index (0 .. (( scalar @test_localtime )- 1)) {
	if ($correct_localtime[$index] ne $test_localtime[$index]) {
		$matched = 0;
	}
}
ok($matched, "Matched wantarray localtime");
foreach my $area ($zoneinfo->areas()) {
	foreach my $location ($zoneinfo->locations($area)) {
		my $correct_date = get_external_date($area, $location, $now);
		$zoneinfo->timezone("$area/$location");
		my $test_date = POSIX::strftime('%Y/%m/%d %H:%M:%S', $zoneinfo->local_time($now));
		ok($test_date eq $correct_date, "Matched $test_date to $correct_date for $area/$location");
		my @local_time = $zoneinfo->local_time($now);
		my $revert_time = $zoneinfo->time_local(@local_time);
		ok($revert_time == $now, "\$zoneinfo->time_local(\$zoneinfo->local_time(\$now)) == \$now where $revert_time = $now with a difference of " . ($revert_time - $now)); 
		my @leap_seconds = $zoneinfo->leap_seconds();
		die "Leap seconds found in $area/$location" if (scalar @leap_seconds);
	}
}

Test::More::done_testing();

sub get_external_date {
	my ($area, $location, $unix_time) = @_;
	my $untainted_unix_time;
	if ($unix_time =~ /^(\-?\d+)$/) {
		($untainted_unix_time) = ($1);
	} else {
		die "Failed to parse transition time $unix_time";
	}
	my $formatted_date;
	if ($perl_date) {
		$formatted_date = `TZ="$area/$location" perl -MPOSIX -e 'print POSIX::strftime("%Y/%m/%d %H:%M:%S", localtime($untainted_unix_time))'`;
	} elsif ($bsd_date) {
		$formatted_date = `TZ="$area/$location" date -r $untainted_unix_time +"%Y/%m/%d %H:%M:%S"`;
	} else {
		my $gm_strftime = POSIX::strftime("%Y/%m/%d %H:%M:%S GMT", gmtime $untainted_unix_time);
		$formatted_date = `TZ="$area/$location" date -d "$gm_strftime" +"%Y/%m/%d %H:%M:%S"`;
	}
	chomp $formatted_date;
	return $formatted_date;
}
