package Archive::Har::Page::PageTimings;

use warnings;
use strict;
use Carp();

our $VERSION = 0.08;

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	if (defined $params) {
		$self->on_content_load($params->{onContentLoad});
		$self->on_load($params->{onLoad});
		if (defined $params->{comment}) {
			$self->comment($params->{comment});
		}
		foreach my $key (sort { $a cmp $b } keys %{$params}) {
			if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
				$self->$key($params->{$key});
			}
		}
	}
	return $self;
}

sub on_content_load {
	my ($self, $new) = @_;
	my $old = $self->{onContentLoad};
	if (@_ > 1) {
		if (defined $new) {
			if ($new =~ /^(\d+|-1)$/smx) { 
				$self->{onContentLoad} = $1 + 0;
			} else {
				Carp::croak("onContentLoad must be a positive number or -1");
			}
		} else {
			$self->{onContentLoad} = -1;
		}
	}
	if ((defined $old) && ($old == -1)) {
		return;
	} else {
		return $old;
	}
}

sub on_load {
	my ($self, $new) = @_;
	my $old = $self->{onLoad};
	if (@_ > 1) {
		if (defined $new) {
			if ($new =~ /^(\d+|\-1)$/smx) { 
				$self->{onLoad} = $1 + 0;
			} else {
				Carp::croak("onLoad must be a positive number or -1");
			}
		} else {
			$self->{onLoad} = -1;
		}
	}
	if ((defined $old) && ($old == -1)) {
		return;
	} else {
		return $old;
	}
	return $old;
}

sub comment {
	my ($self, $new) = @_;
	my $old = $self->{comment};
	if (@_ > 1) {
		$self->{comment} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my ($self, $new) = @_;

	my $name = $Archive::Har::Page::PageTimings::AUTOLOAD;
	$name =~ s/.*://smx;   # strip fully-qualified portion

	my $old;
	if ($name =~ /^_[[:alnum:]]+$/smx) { # private fields
		$old = $self->{$name};
		if (@_ > 1) {
			$self->{$name} = $new;
		}
	} else {
		Carp::croak("$name is not specified in the HAR 1.2 spec and does not start with an underscore");
	}
	return $old;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = { };
	if (defined $self->on_content_load()) {
		$json->{onContentLoad} = $self->on_content_load();
	} else {
		$json->{onContentLoad} = -1;
	}
	if (defined $self->on_load()) {
		$json->{onLoad} = $self->on_load();
	} else {
		$json->{onLoad} = -1;
	}
	if (defined $self->comment()) {
		$json->{comment} = $self->comment();
	}
	foreach my $key (sort { $a cmp $b } keys %{$self}) {
		next if (!defined $self->{$key});
		if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
			$json->{$key} = $self->{$key};
		}
	}
	return $json;
}

1;
__END__
=head1 NAME

Archive::Har::Page::PageTimings - Represents detailed timing of page within the HTTP Archive

=head1 VERSION

Version 0.08

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $page ($har->pages()) {
        my $timings = $page->pageTimings();
	print "onContentLoad for " . $page->title() . ": " . $timings->on_content_load() . "\n";
	print "onLoad for " . $page->title() . ": " . $timings->on_load() . "\n";
        print "Comment for " . $page->title() . ": " . $timings->comment() . "\n";
    }

=head1 SUBROUTINES/METHODS

=head2 on_content_load

returns the number of milliseconds since $har->page()->startedDateTime() that the content of the page loaded or undef if the timing does not apply

=head2 on_load

returns the number of milliseconds since $har->page()->startedDateTime() that the page loaded or undef if the timing does not apply

=head2 comment

returns the comment about the page timing

