package Bio::MUST::Core::SeqId;
# ABSTRACT: Modern and legacy MUST-compliant sequence id
# CONTRIBUTOR: Mick VAN VLIERBERGHE <mvanvlierberghe@doct.uliege.be>
$Bio::MUST::Core::SeqId::VERSION = '0.203490';
use Moose;
use namespace::autoclean;

# AUTOGENERATED CODE! DO NOT MODIFY THIS FILE!

use autodie;
use feature qw(say);

use Smart::Comments;

use Carp;
use Const::Fast;

use Bio::MUST::Core::Types;
use Bio::MUST::Core::Constants qw(:ncbi :seqids);


has 'full_id' => (
    is       => 'ro',
    isa      => 'Str',
    required => 1,
    writer   => '_set_full_id',
);


has 'is_' . $_ => (
    traits   => ['Bool'],
    is       => 'ro',
    isa      => 'Bool',
    init_arg => undef,
    default  => 0,
    handles  => {
        '_set_' . $_ => 'set',
    },
) for qw(foreign new);


has $_ => (
    is       => 'ro',
    isa      => 'Maybe[Str]',
    init_arg => undef,
    default  => undef,
    writer   => '_set_' . $_,
) for qw(family tag genus species strain accession tail
         taxon_id gca gca_novers gca_vers gca_prefix gca_number
         gi database identifier);


# array of parts from NCBI FASTA-style GNL ids
# Note: all_parts accessor returns an empty list if undef
has 'parts' => (
    traits   => ['Array'],
    is       => 'ro',
    isa      => 'ArrayRef[Str]',
    init_arg => undef,
    default  => sub { [] },
    writer   => '_set_parts',
    handles  => {
        count_parts => 'count',
          all_parts => 'elements',
          get_part  => 'get',
    },
);

# file-scope constants

# regexes for full_id (legacy and new) components
const my $FAMILY     => qr{ # underscore allowed
    [^\-\s@\#]+?            # single word without dash, space, '@', '#'
}xms;

const my $TAG        => qr{
    [a-z]{1,4}              # 1-4 letter(s) (lowercase)
}xms;

# list of standard tags:
#     c: contaminant
#     d: divergent
#     u: unknown taxonomy
#     cp: encoded in plastid genome
#     mt: encoded in mitochondrial genome
#     nm: encoded in nucleomorph genome
#     nucp: nuclear copy of plastid gene
#     numt: nuclear copy of mitochondrial gene

const my $GENUS      => qr{ # underscore and dash allowed
    [^\s@]+?                # single word without space, '@'
}xms;

const my $SPECIES    => qr{ # dash allowed
    [^_\s@]+?               # single word without underscore, space, '@'
}xms;

const my $STRAIN     => qr{ # as genus
    $GENUS
}xms;

const my $ACCESSION  => qr{
    $GENUS                  # as genus
}xms;

const my $TAIL       => qr{
    $GENUS                  # as genus
}xms;

# regex for parsing (valid) full_id
const my $FULL_ID    => qr{
    \A                      # begins with...
    (?: ($FAMILY)           # optional family
    - )?                    #   followed by a literal dash (-)
    (?: ($TAG)              # optional tag
    \# )?                   #   followed by a literal number sign (#)
    ($GENUS)                # genus
    \s+                     # whitespace
    ($SPECIES)              # species
    (?: _                   # optional literal underscore (_)
    ($STRAIN) )?            #   followed by strain
    @                       # literal at sign (@)
    ($ACCESSION)            # accession
    (?: \.{3}               # optional literal ellipsis (...)
    ($TAIL) )?              #   followed by a single-word tail
    ($NEW_TAG)?             # optional #NEW# tag
    \z                      # ... at the end
}xms;

# species regex for genus-only ids
const my $GENUSONLY  => qr{
    \A                      # nothing before!
    (?:
    sspp | ssp | spp | sp   # either species (sp), species pluralis (spp),
    )                       # subspecies (ssp) or subspecies pluralis (sspp)pa
    \.?                     # optionally followed by a dot
    \z                      # and nothing after!
}xms;

# tag regex for doubtful organisms (from a taxonomic POV)
const my $DOUBTFUL   => qr{
    \A
    [cdu]
    \z
}xms;

# regex for parsing taxonomy-enabled abbreviated ids
const my $TAXABBR_ID => qr{
    \A                      # begins with...
    ($NCBIPKEY|$NCBIGCA)    # NCBI primary key (= taxid) or NCBI GCA/GCF
    \|                      # literal pipe sign (|)
    ($ACCESSION)            # accession
    \z                      # ... at the end
}xms;

# regex for parsing NCBI FASTA-style GI ids
# TODO: merge with $GI_ID in Constants?
# TODO: improve handling of partial ids (e.g., gi|404160475| would fail)
const my $NCBIGI_ID => qr{
    \A                      # begins with...
    gi                      # literal 'gi'
    \|                      # literal pipe sign (|)
    ($NCBIPKEY)             # NCBI primary key (= GI number)
    (?: \|                  # optional literal pipe sign (|)
     $NCBIDBABBR            #   followed by NCBI database abbr (not captured)
    \|                      #   then literal pipe sign (|)
    ($NCBIACC) ) ?          #   then NCBI accession
                            # no end anchor to allow for additional stuff
}xms;

# regex for parsing NCBI FASTA-style GNL ids
# TODO: merge with $GNL_ID in Constants?
# Note: this regex actually returns the full_id for subsequent splitting
const my $NCBIGNL_ID => qr{
    \A                      # begins with...
    (gnl                    # literal 'gnl'
    (?: \|                  #   literal pipe sign (|)
    $NCBIPART)              #   NCBI part (very accommodating)
    {2,} )                  # at least twice but allowing for more
}xms;

BEGIN{

no warnings "qw";

# static lists of hyphenated genera and underscored species
# Note: in BEGIN block because the hash must be built before any call to BUILD

const my @GENERA  => qw(
    1,4-dioxane-degrading
    1-butene
    1-hexene
    16SrXXII-A
    16SrXXII-B
    2,3-D
    2,4-D
    2,4-D-degrading
    229E-related
    4-nitrophenol
    A-2
    acetamiprid-degrading
    Activation-tagging
    Adeno-Associated
    Adeno-associated
    Aids-associated
    alk-system
    Altai-like
    Amalga-like
    ammonia-oxidizing
    aniline-degrading
    ANME-1
    ANME-2
    ANMV-1
    Anopheline-associated
    antibiotic-resistant
    antimony-reducing
    Apple-associated
    Araraquara-like
    Argemone-Cactus
    Arrabida-like
    arsenate-reducing
    arsenic-metabolizing
    arsenic-oxidizing
    arsenite-oxidising
    arsenite-oxidizing
    Arthrobacter-like
    ARV-138
    ARV-176
    ASFV-like
    Avian-like
    Avon-Heathcote
    Aydin-like
    B-lymphotropic
    Bacteroides-like
    Banna-like
    Barkedji-like
    Bas-Congo
    Bastrovirus-like_virus/VietNam/Bat/17819_21
    BD1-11
    BD1-7
    BD2-11
    Bean-associated
    Bean-pod
    Bear-associated
    Begomovirus-associated
    Bemisia-associated
    benzene-decomposing
    benzene-degrading
    benzoapyrene-degrading
    Benzoyl-CoA
    beta-symbiont
    Bi-allelically
    Bi-directional
    Bi-molecular
    Bidnavirus-like
    biocide-degrading
    biphenthrin-degrading
    biphenyl-degrading
    Birds-foot
    Black-and-white
    Black-eyed
    Black-headed
    Blackcurrant-associated
    Bo-Circo-like
    Bombus-associated
    Bos-associated
    Bovine-like
    Broad-leafed
    bromate-reducing
    bromoform-metabolizing
    Bromus-associated
    BTEX-degrading
    BtMd-BetaCoV/HuB2013
    BtMf-AlphaCoV/AH2011
    BtMf-AlphaCoV/FJ2012
    BtMf-AlphaCoV/GD2012
    BtMf-AlphaCoV/GD2012-a
    BtMf-AlphaCoV/GD2012-b
    BtMf-AlphaCoV/HeN2013
    BtMf-AlphaCoV/HeN2013-a
    BtMf-AlphaCoV/HeN2013-b
    BtMf-AlphaCoV/HuB2013
    BtMf-AlphaCoV/HuB2013-a
    BtMf-AlphaCoV/JX2012
    BtMr-AlphaCoV/SAX2011
    BtNv-AlphaCoV/SC2013
    BtPa-BetaCoV/GD2013
    BtRf-AlphaCoV/HuB2013
    BtRf-AlphaCoV/YN2012
    BtRf-BetaCoV/HeB2013
    BtRf-BetaCoV/HeN2013
    BtRf-BetaCoV/HuB2013
    BtRf-BetaCoV/JL2012
    BtRf-BetaCoV/SX2013
    BtRs-BetaCoV/GX2013
    BtRs-BetaCoV/HuB2013
    BtRs-BetaCoV/YN2013
    BtTp-BetaCoV/GX2012
    BtVs-BetaCoV/SC2013
    bubaline-associated
    Bufavirus-1
    Bufavirus-2
    Bufavirus-3
    Bunya-like
    Burke-Gilman
    butyrate-producing
    Buxtonella-like
    C-terminal
    C.glutamicum-E.coli
    Cachavirus-1A
    Cachavirus-1B
    Cafeteriavirus-dependent
    Calf-giraffe
    Calhevirus-2a
    Calhevirus-2b
    Campylobacter-like
    carbazole-degrading
    Carpediemonas-like
    carrageenase-producing
    Cas-Br-E
    Cas-NS-1
    CASV-2
    CAT-levansucrase
    Catch-me-cave
    CBA120-like
    Cell-free
    Ceratium-like
    CFB-group
    chain-forming
    Chaq-like
    Chiba-040502
    chlamydia-associated
    Chlorante-Aster
    Chlorella-like
    chloromethane-metabolizing
    chlortetracycline-resistant
    Choricystis-like
    Chryso-hypnum
    Chrysovirus-like
    cilia-associated
    Circo-like
    Circovirus-like
    Citrus-associated
    Clavibacter-like
    Clostridium-like
    Co-expression
    Co-reporter
    Colwellia-like
    Corn-reddening
    Corynebacterium-like
    COS-degrading
    Cotton-top
    Counter-selectable
    Cowpox-Vaccinia
    Coxiella-like
    Cr(VI)-reducing
    Crane-associated
    crAss-like
    Cre-expression
    Cre-lox
    Crimean-Congo
    CRISPR-cas9
    CRISPR-mediated
    Crow-dipper
    Crucivirus-102
    Crucivirus-111
    Crucivirus-112
    Crucivirus-113
    Crucivirus-119
    Crucivirus-122
    Crucivirus-124
    Crucivirus-133
    Crucivirus-141
    Crucivirus-146
    Crucivirus-147
    Crucivirus-148
    Crucivirus-152
    Crucivirus-158
    Crucivirus-163
    Crucivirus-168
    Crucivirus-170
    Crucivirus-174
    Crucivirus-175
    Crucivirus-182
    Crucivirus-183
    Crucivirus-184
    Crucivirus-190
    Crucivirus-195
    Crucivirus-204
    Crucivirus-205
    Crucivirus-207
    Crucivirus-211
    Crucivirus-212
    Crucivirus-224
    Crucivirus-234
    Crucivirus-241
    Crucivirus-243
    Crucivirus-245
    Crucivirus-250
    Crucivirus-251
    Crucivirus-255
    Crucivirus-264
    Crucivirus-265
    Crucivirus-266
    Crucivirus-272
    Crucivirus-275
    Crucivirus-276
    Crucivirus-277
    Crucivirus-278
    Crucivirus-281
    Crucivirus-282
    Crucivirus-291
    Crucivirus-293
    Crucivirus-295
    Crucivirus-298
    Crucivirus-304
    Crucivirus-305
    Crucivirus-306
    Crucivirus-309
    Crucivirus-310
    Crucivirus-311
    Crucivirus-312
    Crucivirus-322
    Crucivirus-323
    Crucivirus-324
    Crucivirus-326
    Crucivirus-339
    Crucivirus-340
    Crucivirus-345
    Crucivirus-349
    Crucivirus-350
    Crucivirus-353
    Crucivirus-359
    Crucivirus-362
    Crucivirus-364
    Crucivirus-371
    Crucivirus-372
    Crucivirus-373
    Crucivirus-380
    Crucivirus-384
    Crucivirus-387
    Crucivirus-389
    Crucivirus-392
    Crucivirus-402
    Crucivirus-406
    Crucivirus-407
    Crucivirus-409
    Crucivirus-410
    Crucivirus-411
    Crucivirus-416
    Crucivirus-419
    Crucivirus-427
    Crucivirus-429
    Crucivirus-432
    Crucivirus-438
    Crucivirus-453
    Crucivirus-454
    Crucivirus-455
    Crucivirus-456
    Crucivirus-458
    Crucivirus-468
    Crucivirus-472
    Crucivirus-474
    Crucivirus-475
    Crucivirus-479
    Crucivirus-480
    Crucivirus-482
    Crucivirus-483
    Crucivirus-495
    Crucivirus-497
    Crucivirus-499
    Crucivirus-501
    Crucivirus-505
    Crucivirus-506
    Crucivirus-509
    Crucivirus-513
    Crucivirus-514
    Crucivirus-520
    Crucivirus-523
    Crucivirus-527
    Crucivirus-528
    Crucivirus-530
    Crucivirus-534
    Crucivirus-536
    Crucivirus-538
    Crucivirus-539
    Crucivirus-540
    Crucivirus-81
    Crucivirus-88
    Crucivirus-93
    Crucivirus-94
    Crucivirus-97
    Crucivirus-99
    Crucivirus-like
    Cryphonectria-Endothia
    Cryptaulaxoides-like
    Ctenophore-associated
    Culex-associated
    Culicine-associated
    cyanobacteria-associated
    Cyrto-hypnum
    Cysteine-free
    Cytophaga-like
    DCM-degrading
    deep-sea
    Deep-sea
    Deinococcus-like
    Deinococcus-Thermus
    Dendro-hypnum
    Dermatophilus-like
    DG-75
    dibenzofuran-degrading
    dichloromethane-degrading
    Dietzia-Escherichia
    Dimorpha-like
    Diporeia-associated
    DMSP-degrading
    DNA-binding
    Dobrava-Belgrade
    Dragonfly-associated
    Drosophila-associated
    DSMP-degrading
    Dual-tagging
    Duck-dominant
    dye-degrading
    EDTA-degrading
    Ehrlichia-like
    EIAV-based
    Endornavirus-like
    Entamoeba-associated
    Epomophorini-Myonycterini
    Epstein-barr
    Erythrobacter-like
    Escherichia-Pseudomonas
    estrogen-degrading
    ethanol-producing
    EX-H1
    F-11
    Faecal-associated
    FCV-like
    Fe(III)-reducing
    Fe-oxidizing
    fenpropathrin-degrading
    fenthion-degrading
    Fermo-like
    Fesa-like
    Fibropapilloma-associated
    Finkel-Biskis-Jinkins
    Fisa-like
    FLAG-tagging
    Flavi-like
    Flavobacterium-like
    Flexibacter-like
    Flexistipes-like
    Flying-fox
    Fonticula-like
    Food-grade
    Foot-and-mouth
    Francisella-like
    FRT-Tp-FRT
    furfural-degrading
    gamma-symbiont
    Ganwon-do
    Gardner-Arnstein
    GBV-A-like
    Gemmata-like
    Gene-editing
    Gene-trap
    Gene-trapping
    Giardia-associated
    Gill-associated
    Ginkgoaceae-associated
    Glaucous-winged
    Glaucus-gull
    Gluconacetobacter-like
    glycerol-degrading
    GO-respiring
    Golden-headed
    Gonad-specific
    Goose-dominant
    Gordonia-like
    Gram-negative
    Gram-positive
    Grapevine-associated
    Grass-Cactus
    Gray-bellied
    Gyrovirus-like
    H-1
    H-Pelican
    H-Stinger
    H2/CO2-metabolizing
    Hantaan-like
    Hardy-Zuckerman
    Haruka-associated
    HCV/GBV-B
    Hedyotis-Oldenlandia
    Helper-independent
    Hemibarbus-Squalidus
    HERV-H/env59
    HERV-H/env60
    HERV-H/env62
    Hibiscus-infecting
    Himar1-delivery
    his-3
    HIV-1
    HIV-2
    HIV-2.D205
    HIV-like
    HMWdDNA-degrading
    Hom-1
    HPIV-1
    HSV-tk-containing
    HTLV-1
    HTLV-3
    Humaita-Tubiacanga
    Hump-back
    Husa-like
    HVAC-associated
    Hyphomicrobium-like
    Ichthyobodo-related
    IncP-1
    IncP-1beta
    IncQ-like
    Insect-associated
    iodate-reducing
    iodide-accumulating
    iodide-oxidizing
    IRE/CTVM19-associated
    iron-reducing
    Isosphaera-like
    Issyk-Kul
    J-virus
    Janibacter-like
    jellyfish-degrading
    Jiangxi12-B14
    Jodiemicrovirus-1
    Johnson-sea-linkia
    Juquitiba-like
    K139-like
    Kineococcus-like
    Knock-in
    Korle-bu
    LANV-2
    Large-insert
    Leafhopper-borne
    Legionella-like
    Leguminosae-associated
    Leptotrichia-like
    lignocellulose-degrading
    LMWdDNA-degrading
    Long-fingered
    long-tailed
    LPP-group
    Luna-1
    Luna-2
    lung-eye-trachea
    Lupinus-Cactus
    Maedi-Visna-related
    magnetite-containing
    magneto-ovoid
    Magpie-robin
    Maize-associated
    male-killer
    male-killing
    manganese-oxidation
    manganese-oxidizing
    Marinobacter-like
    Mason-Pfizer
    MAST-4
    Maus-Elberfeld
    Maverick-related
    MCG-1
    MCG-15
    MCG-6
    Meaban-like
    Melon-headed
    mercury-resistant
    Merida-like
    metal-contaminated
    methane-oxidizing
    methane-seep
    methylotroph-related
    Metopion-like
    Mimivirus-dependent
    mini-chromosome
    mini-Tn5*7-lux
    Mini-Tn7
    Mint-like
    Mirim-like
    Mito-like
    MLV-like
    MLV-related
    MMTV-luciferase
    monochloroacetic-acid-degrading
    Mononegavirus-like
    mosquito-borne
    mouse-rat
    Mu-like
    Multi-host
    Murine-associated
    Mx-Lox
    Mycobacteria-E.coli
    N-peptide-mCherry
    N-terminal
    N-tropic
    Nanovirus-like
    naphthalene-utilizing
    narna-like
    Nebraska-like
    Negelike-virus
    Negev-like
    Neo-uvaria
    NFAT-DsRed
    nitrite-oxidizing
    nitrogen-fixing
    Nivun-Haamir
    NL63-related
    Non-A,
    non-culturable
    Non-geniculate
    Non-human
    non-mammal
    non-primate
    Non-primate
    Norovirus/GII.4/1993-6/UK
    Norovirus/Hu/GII.2/V1/09/18-Jan-2009/Slovenia
    Norovirus/Hu/GII.4/1732/07/07-Jun-2007/Slovenia
    Norovirus/Hu/GII.4/1991/07/26-Jun-2007/Slovenia
    Norovirus/Hu/GII.4/220/10/28-Jan-2010/Slovenia
    Norovirus/Hu/GII.4/2791/09/09-Oct-2009/Slovenia
    Norovirus/Hu/GII.4/2830/09/15-Oct-2009/Slovenia
    Norovirus/Hu/GII.4/288/05/04-Feb-2005/Slovenia
    Norovirus/Hu/GII.4/2889/09/23-Oct-2009/Slovenia
    Norovirus/Hu/GII.4/290/05/04-Feb-2005/Slovenia
    Norovirus/Hu/GII.4/580/07/26-Feb-2007/Slovenia
    Norovirus/Hu/GII.4/755/10/17-Mar-2010/Slovenia
    Norovirus/Hu/GII.4/756/10/17-Mar-2010/Slovenia
    Norovirus/Hu/GII.4/757/10/17-Mar-2010/Slovenia
    Norovirus/Hu/GII.4/V17/08/05-Feb-2009/Slovenia
    Norovirus/Hu/GII.4/V2/09/18-Jan-2009/Slovenia
    Norovirus/Hu/GII.4/V3/09/18-Jan-2009/Slovenia
    Norovirus/Hu/GII.4/V47/09/25-Feb-2009/Slovenia
    Norovirus/Hu/GII.4/V7/09/11-Jan-2009/Slovenia
    Norwalk-like
    Nuclearia-like
    Oak-Vale
    Oceanospirillum-like
    Ochlerotatus-associated
    Odonata-associated
    oil-degrading
    oil-susceptible
    Onyong-nyong
    oomycete-like
    oral-nasopharyngeal
    Orange-breasted
    Orange-spotted
    Ourmia-like
    oxamyl-degrading
    P-decomposing
    P-element
    P1-specific
    PAH-contaminated
    PAH-degrading
    Paracoccus-like
    Parecho-like
    Paromomycin-resistant
    Partiti-like
    Partitivirus-like
    Parvo-like
    PCB-dechlorinating
    Pechuel-loeschea
    Peptococcus-like
    perchlorate-reducing
    Perchlorate-reducing
    Perkinsela-like
    Perkinsiella-like
    pesticide-degrading
    Pfiesteria-like
    pharmaceutical-degrading
    phenanthrene-degrading
    phenanthrene-metabolizing
    phenol-degrading
    Phlebotomus-associated
    Phoma-like
    phosphorous-removal
    Picorna-like
    Pig-tailed
    Pigeon-dominant
    piggyBac-based
    piggyBac-helper
    Pike-perch
    Pink-eared
    Piscirickettsia-like
    Pistacia-associated
    pkBIG-alpha
    pLEV-TeLS
    pnpB-pnpA
    Po-Circo-like
    Poaceae-associated
    Point-Douro
    Polycystine-like
    porcine-murine
    Porphyromonas-like
    Port-miou
    Posa-like
    PreXMRV-1
    PreXMRV-1/PreXMRV-2
    PreXMRV-2
    Promoter-probe
    Promoter-trap
    Pseudo-nitzschia
    Pseudoalteromonas-like
    PTE-S5
    Puumala-like
    PVB_OTU_4-like
    Qinvirus-like
    R6K-based
    Raccoon-associated
    RDHV-like
    Red-bellied
    Red-capped
    Red-crowned
    Red-eared
    Red-faced
    Red-fronted
    Red-handed
    Red-necked
    Red-recombineering
    Regulator-reporter
    Reptile-associated
    Retro-megavirus
    Rigidomastix-like
    RIOMV-3
    RIOMV-4
    Rm-Bat-CoV
    RML-105355
    rod-shaped
    root-associated
    Rosaceae-associated
    Roseobacter-like
    Rous-associated
    rpL23-fusion
    Rubrivivax-like
    Rudolf-kamelinia
    RV-African
    RV-Bower
    RV-Brook
    RV-Brown
    RV-Common
    RV-Edible
    RV-European
    RV-Freshwater
    RV-Gharial
    RV-Horse
    RV-Iberian
    RV-Leopard
    RV-Painted
    RV-Palmate
    RV-Pit
    RV-Pufferfish
    RV-Rhinatremid
    RV-Rocket
    RV-Slider
    RV-Stickleback
    RV-Stripe
    RV-Tiger
    RV-Tinamou
    RV-Tuatara
    RVA/Human-wt/MWI/1473/2001/G8P4
    RVA/Human-wt/ZAF/3133WC/2009/G12P4
    RVA/Human-wt/ZAF/3176WC/2009/G12P6
    RVA/Human-wt/ZAF/3203WC/2009/G2P4
    S10-like
    S2-like
    S5-like
    S7-like
    Santee-Cooper
    Sapelovirus-like
    Sapporo-HK299
    SARS-like
    SARS-related
    Saxo-fridericia
    Sciuro-hypnum
    selenate-reducing
    Selenomonas-like
    Self-inactivating
    Sewage-associated
    Sharp-tailed
    Short-finned
    short-tailed
    Shuttle-expression
    Shuttle/allelic-replacement
    Sikhote-Alin
    simazine-degrading
    Simian-Human
    Sindbis-like
    Single-copy
    Sint-Jan
    Site-specific
    SIV-derived
    SIV-wrc
    Snail-Cactus
    SNAP-XDocII
    Snyder-Theilen
    Sodalis-like
    Soft-shell
    Soft-shelled
    Soil-borne
    Solms-laubachia
    Soybean-associated
    Spawner-isolated
    Sphingobacterium-like
    Sphingomonas-like
    Sphinx1.76-related
    Sphinx2.36-related
    Spiroplasma-symbiont
    Spirosoma-like
    Spumella-like
    Spur-thighed
    St-Valerien
    star-like
    Stenotrophomonas-like
    Sticholonche-like
    Stolbur-It
    Stolbur-Rus
    Stripa-derived
    Stx1-converting
    Stx2-converting
    sulfate-reducing
    Sulfate-reducing
    sulfide-oxidizing
    sulfite-reducing
    sulfo-oxidizing
    sulfur-oxidizing
    Sunn-hemp
    Synechococcus-like
    Synechocystis-like
    T-DNA
    T-vector
    T4-like
    T7-like
    Taro-associated
    Taxeobacter-like
    Temperature-sensitive
    Tent-making
    Tetracycline-inducible
    Theileria-related
    Theilers-like
    thiocyanate-degrading
    thiosulfate-disproportionating
    thiosulfate-reducing
    Thrips-associated
    Ti-curing
    Tick-associated
    Tick-borne
    tick-borne
    toluene-degrading
    Tombus-like
    Totivirus-like
    Tri-shuttle
    trichloroacetic-acid-degrading
    triclosan-degrading
    Trifolium-associated
    TSE-associated
    TTV-like
    Twisted-stalk
    TYLCAxV-Sic1-IT:Sic2/2:04
    TYLCAxV-Sic2-IT:Sic2/5:04
    TYLCCNV-Y322
    tylosin-resistant
    Type-B
    Type-C
    Type-D
    Type-E
    Type-F
    U1Ap-mCherry
    UAG-readthrough
    UAS-less
    UAS-Red
    UAS-Stinger
    Ui-te-Rangiora
    Van-royena
    VDV-1/DWV
    VESV-like
    Violaceae-associated
    Virga-like
    Viroid-like
    Virus-associated
    Virus-induced
    Virus-like
    Visna-maedi
    Vittaforma-like
    WA-like
    wall-less
    Weivirus-like
    White-eye
    White-headed
    white-rot
    White-sloanea
    White-tailed
    white-throated
    White-tufted-ear
    Whitefly-associated
    Whitefly-transmitted
    WPS-2
    X-cell
    X-symbiont
    Xanthomonas-like
    XMRV-like
    XMRV-related
    Y-symbiont
    Yaba-7
    Yaba-like
    Yado-kari
    Yado-nushi
    Yaounde-like
    Yellow-bellied
    Yellow-breasted
    Yellow-wax
    );

const my @SPECIES => qw(
    04_1_1
    070605-23_N11_16
    100_oclvp78
    101_oclvp44
    102_oclvp93
    103_oclvp94
    104_oclvp99
    105_oclvp561
    106_oclvp92
    107_oclvp18
    108_oclvp22
    109_oclvp514
    10_xplvp428
    110_oclvp17
    111_oclvp16
    112_oclvp26
    113_oclvp23
    114_oclvp28
    115_oclvp27
    1168_2009
    116_oclvp532
    1178_2009
    117_oclvp558
    118_39
    118_oclvp546
    119_MR-2014
    119_xplvp419
    11_xplvp442
    11GM_10_3
    120_oclvp102
    121_oclvp513
    1228_2009
    122_oclvp550
    123_oclvp549
    124_oclvp198
    125_oclvp12
    126_oclvp498
    127_xplvp336
    128_xplvp197
    129_MR-2014
    129_xplvp391
    12_oclvp105
    130_MR-2014
    130_xplvp347
    131_xplvp388
    132_xplvp394
    133_xplvp560
    134_xplvp393
    135_xplvp202
    136_MR-2014
    136_xplvp446
    137_MR-2014
    137_xplvp455
    138_xplvp451
    139_xplvp295
    13_1_20CM_2_51_10
    13_1_20CM_2_51_12
    13_1_20CM_2_54_9
    13_1_20CM_2_63_93
    13_1_20CM_3_53_6
    13_1_20CM_4_52_8
    13_1_20CM_52_20
    13_1_40CM_2_52_13
    13_1_40CM_2_52_4
    13_1_40CM_2_55_8
    13_1_40CM_3_53_8
    13_1_40CM_4_53_4
    13_1_40CM_4_53_8
    13_2_20CM_2_52_21
    13_2_20CM_2_53_6
    13_xplvp186
    140_xplvp374
    141_xplvp294
    142_xplvp291
    143_xplvp334
    144_xplvp290
    145_xplvp335
    146_xplvp378
    147_xplvp204
    1489_2009
    148_xplvp1
    1490_2009
    149_xplvp363
    14_xplvp362
    150907-44_B22_ML8_968
    150907-44_D20_ML1_713
    150907-44_F20_ML2_781
    150907-44_H20_ML3_785
    150907-44_L20_ML5_758
    150907-44_N20_ML6_768
    150907-44_P20_ML7_756
    150_xplvp379
    151_xplvp405
    152_xplvp404
    153_xplvp376
    154_xplvp377
    155_xplvp369
    156_xplvp375
    157_xplvp365
    158_xplvp352
    159_xplvp212
    15_xplvp210
    160_xplvp214
    161_xplvp465
    162_xplvp226
    163_xplvp569
    164_xplvp565
    165_MR-2014
    165_xplvp467
    166_oclvp485b
    167_xplvp346
    168_xplvp345
    169_xplvp344
    16_xplvp576
    16F2019_GII.p16_Guangdong_CHN_2016
    16F2025_GII.p16_Guangdong_CHN_2016
    16F2026_GII.p16_Guangdong_CHN_2016
    16F2029_GII.p16_Guangdong_CHN_2016
    16F21132_GII.p16_Guangdong_CHN_2016
    16F21134_GII.p16_Guangdong_CHN_2016
    16F2149_GII.2_Guangdong_CHN_2016
    16F2152_GII.2_Guangdong_CHN_2016
    16F2161_GII.2_Guangdong_CHN_2016
    16F2284_GII.p16_Guangdong_CHN_2016
    16F2285_GII.p16_Guangdong_CHN_2016
    16SF2025_GII.2_Guangdong_CHN_2016
    16SF2026_GII.2_Guangdong_CHN_2016
    16SF2029_GII.2_Guangdong_CHN_2016
    16SF2030_GII.2_Guangdong_CHN_2016
    16SF21132_GII.2_Guangdong_CHN_2016
    16SF21134_GII.2_Guangdong_CHN_2016
    16SF2151_GII.2_Guangdong_CHN_2016
    16SF2153_GII.2_Guangdong_CHN_2016
    16SF2285_GII.2_Guangdong_CHN_2016
    170_xplvp340
    171_xplvp255
    172_xplvp245
    173_xplvp338
    174_xplvp339
    175_xplvp267
    176_xplvp289
    177_xplvp288
    178_oclvp200
    179_oclvp274
    17_xplvp121
    180_oclvp537
    181_MR-2014
    181_oclvp548
    182_oclvp528
    183_oclvp263
    184_MR-2014
    184_oclvp523
    185_oclvp234
    186_oclvp236
    187_oclvp241
    188_oclvp265
    189_oclvp280
    18_xplvp100
    190_oclvp281
    191_oclvp493
    192_oclvp270
    193_oclvp262
    194_oclvp520
    195_oclvp221
    196_oclvp220
    197_oclvp508
    198_oclvp491
    199_oclvp489
    19_xplvp109
    1_xplvp296
    1AZ_DGR
    200_oclvp497
    2010_Msc_17636_VA3/HMO
    2010_Msc_18001_MLB
    201_oclvp283
    202_oclvp284
    203_oclvp551
    204_oclvp480
    205_oclvp481
    206_oclvp485
    207_oclvp487
    20_xplvp148
    21_oclvp54
    21a7_large
    21a7_orange
    21a7_pink
    21a7_small
    22_MR-2014
    22_xplvp150
    22B9_orange
    22C12_green
    22c12_orange
    22E1_Frozen
    23_oclvp517
    24_xplvp459
    25_xplvp140
    26_xplvp434
    27_xplvp395
    28_xplvp13
    29_xplvp147
    2_xplvp435
    2A3B_DGR
    2ANA_DGR
    3-5_YC-ZSS-LKJ-3024
    3-7_YC-ZSS-LKJ-1723
    30_xplvp384
    31_xplvp325
    32_xplvp209
    33_17
    33_xplvp322
    34_xplvp425
    35_xplvp171
    36_xplvp225
    37_xplvp454
    380_MR-2014
    38_xplvp95
    397_MR-2014
    39_xplvp47
    3_xplvp302
    4-1_YC-ZSS-LKJ-010-0902
    40_xplvp46
    41_xplvp469
    42_11
    42_xplvp429
    43_xplvp216
    44_xplvp252
    453a_as
    453b_as
    45_oclvp525
    46_xplvp399
    47_oclvp120
    480_MR-2014
    48_xplvp213
    495_MR-2014
    49_oclvp161
    4_57p
    4_xplvp418
    507_MR-2014
    50_oclvp203
    51_oclvp114
    52_oclvp282
    533_MR-2014
    537_MR-2014
    53_oclvp165
    54_oclvp90
    55_oclvp119
    56_oclvp163
    57_xplvp332
    58_oclvp104
    59_13
    59_oclvp166
    5_25as
    5_xplvp440
    5P_1
    5P_2
    5P_3
    603_MR-2014
    60_oclvp187
    61_oclvp108
    62_oclvp107
    63_oclvp173
    64_12
    64_oclvp174
    65_10
    65_oclvp553
    66_1
    66_12
    66_oclvp134
    67_12
    67_oclvp138
    68_12
    68_oclvp142
    69_oclvp189
    6_xplvp63
    70_oclvp488
    71_18
    71_oclvp188
    72_oclvp145
    73_oclvp578
    73_VP7_H6
    73_VP8_A3
    73_VP8_A4
    73_VP8_B3
    73_VP8_B4
    73_VP8_C3
    73_VP8_C4
    73_VP8_D3
    73_VP8_D4
    73_VP8_E3
    73_VP8_E4
    73_VP8_F3
    73_VP8_G3
    73_VP8_H3
    73_VP9_A1
    74_oclvp132
    755_MR-2014
    75_oclvp484
    763_MR-2014
    76_oclvp68
    774_MR-2014
    77_oclvp66
    78_oclvp237
    79_oclvp69
    7_oclvp249
    80_oclvp279
    81_oclvp45
    82_oclvp510
    82_VP7_A8
    82_VP7_B8
    82_VP7_D8
    82_VP7_E7
    82_VP7_E8
    82_VP7_F8
    82_VP7_G7
    82_VP7_H7
    82_VP9_B1
    82_VP9_D1
    82_VP9_E1
    83_oclvp219
    84_MR-2014
    84_oclvp575
    85_oclvp50
    86_oclvp5
    87_oclvp554
    89_oclvp526
    8_xplvp75
    90_oclvp79
    91_11
    91_27
    91_oclvp479
    92_oclvp33
    93_oclvp84
    94_oclvp83
    95_oclvp82
    96_oclvp81
    97_xplvp333
    98_oclvp43
    99_oclvp542
    9_xplvp439
    A18_CLW
    A19_CLW
    A1_395
    A22_CLW
    A3M_VP7_A10
    A3M_VP7_A12
    A3M_VP7_A9
    A3M_VP7_B10
    A3M_VP7_B11
    A3M_VP7_B9
    A3M_VP7_C10
    A3M_VP7_C12
    A3M_VP7_C9
    A3M_VP7_D10
    A3M_VP7_D11
    A3M_VP7_D12
    A3M_VP7_D9
    A3M_VP7_E10
    A3M_VP7_E9
    A3M_VP7_F10
    A3M_VP7_F11
    A3M_VP7_F9
    A3M_VP7_G10
    A3M_VP7_G9
    A3M_VP7_H10
    A3M_VP7_H8
    A3S_VP5_A6
    A3S_VP5_A8
    A3S_VP5_B7
    A3S_VP5_B8
    A3S_VP5_C7
    A3S_VP5_C8
    A3S_VP5_D7
    A3S_VP5_D8
    A3S_VP5_E7
    A3S_VP5_E8
    A3S_VP5_F7
    A3S_VP5_G6
    A3S_VP5_G8
    A3S_VP8_A5
    A3S_VP8_B5
    A3S_VP8_C5
    A3S_VP8_G4
    A3T_VP5_A5
    A3T_VP5_B4
    A3T_VP5_B5
    A3T_VP5_C4
    A3T_VP5_D4
    A3T_VP5_D5
    A3T_VP5_E4
    A3T_VP5_E5
    A3T_VP5_F4
    A3T_VP5_F5
    A3T_VP5_G4
    A3T_VP5_G5
    A3T_VP7_C6
    A3T_VP7_G12
    A3T_VP7_H12
    A3T_VP8_A1
    A3T_VP8_B1
    A3T_VP8_C1
    A3T_VP8_D1
    A3T_VP8_E1
    A3T_VP8_F1
    A3T_VP8_G1
    A4M_VP5_A10
    A4M_VP5_A11
    A4M_VP5_A9
    A4M_VP5_B10
    A4M_VP5_B9
    A4M_VP5_C10
    A4M_VP5_C9
    A4M_VP5_D10
    A4M_VP5_D9
    A4M_VP5_E10
    A4M_VP5_E9
    A4M_VP5_F10
    A4M_VP5_F9
    A4M_VP5_G10
    A4M_VP5_G9
    A4M_VP5_H10
    A4M_VP8_A6
    A4M_VP8_C6
    A4M_VP8_D5
    A4M_VP8_D6
    A4M_VP8_F5
    A4M_VP8_F6
    A4M_VP8_G5
    A4M_VP8_G6
    A4M_VP8_H5
    A4S_VP6_A4
    A4S_VP6_B3
    A4S_VP6_B6
    A4S_VP6_C3
    A4S_VP6_C4
    A4S_VP6_C5
    A4S_VP6_D3
    A4S_VP6_F3
    A4S_VP6_G2
    A4S_VP6_G4
    A4S_VP6_G5
    A4S_VP6_H2
    A4S_VP6_H3
    A4S_VP8_A11
    A4S_VP8_H10
    A4T_VP5_A12
    A4T_VP5_B12
    A4T_VP5_C12
    A4T_VP5_D11
    A4T_VP5_D12
    A4T_VP5_E11
    A4T_VP5_E12
    A4T_VP5_F11
    A4T_VP5_F12
    A4T_VP5_G11
    A4T_VP5_G12
    A4T_VP5_H11
    A4T_VP5_H12
    A4T_VP6_A1
    A4T_VP6_A2
    A4T_VP6_B1
    A4T_VP6_B2
    A4T_VP6_C1
    A4T_VP6_C2
    A4T_VP6_D1
    A4T_VP6_D2
    A4T_VP6_E1
    A4T_VP6_E2
    A4T_VP6_F2
    A5M_1
    A5M_2
    A5M_VP8_B2
    A5M_VP8_C2
    A5M_VP8_D2
    A5M_VP8_E2
    A5M_VP8_F2
    A5M_VP8_G2
    A5M_VP8_H2
    A5S_VP6_A10
    A5S_VP6_B10
    A5S_VP6_B11
    A5S_VP6_D11
    A5S_VP6_F10
    A5S_VP6_F9
    A5S_VP6_G11
    A5S_VP6_G9
    A5S_VP6_H11
    A5S_VP6_H9
    A5S_VP9_B2
    A5T_VP6_A7
    A5T_VP6_A8
    A5T_VP6_A9
    A5T_VP6_B7
    A5T_VP6_B9
    A5T_VP6_C6
    A5T_VP6_C7
    A5T_VP6_C9
    A5T_VP6_E6
    A5T_VP6_E7
    A5T_VP6_E8
    A5T_VP6_F6
    A5T_VP6_F8
    A5T_VP6_G6
    A5T_VP6_G7
    A5T_VP6_G8
    A5T_VP6_H7
    A5T_VP8_A12
    A5T_VP8_B12
    A5T_VP8_E12
    A5T_VP8_H11
    A6M_VP6_B12
    A6M_VP6_C12
    A6M_VP6_D12
    A6M_VP6_H12
    A6M_VP7_A2
    A6M_VP7_A3
    A6M_VP7_B2
    A6M_VP7_C1
    A6M_VP7_D1
    A6M_VP7_D2
    A6M_VP7_E1
    A6M_VP7_F1
    A6M_VP7_G1
    A6M_VP7_H1
    A6M_VP8_B11
    A6M_VP8_C11
    A6M_VP8_D11
    A6M_VP8_E11
    A6M_VP8_F11
    A6S_VP7_A6
    A6S_VP7_B5
    A6S_VP7_B6
    A6S_VP7_C5
    A6S_VP7_D5
    A6S_VP7_D6
    A6S_VP7_E5
    A6S_VP7_E6
    A6S_VP7_F4
    A6S_VP7_F5
    A6S_VP7_F6
    A6S_VP7_G4
    A6S_VP7_G5
    A6S_VP7_H4
    A6S_VP7_H5
    A6S_VP8_A10
    A6S_VP8_A9
    A6S_VP8_B10
    A6S_VP8_F9
    A6S_VP8_H8
    A6S_VP8_H9
    A6T_VP7_A4
    A6T_VP7_A5
    A6T_VP7_B3
    A6T_VP7_B4
    A6T_VP7_C3
    A6T_VP7_C4
    A6T_VP7_D3
    A6T_VP7_D4
    A6T_VP7_E2
    A6T_VP7_E3
    A6T_VP7_E4
    A6T_VP7_F2
    A6T_VP7_F3
    A6T_VP7_G2
    A6T_VP7_G3
    A6T_VP7_H3
    A6T_VP8_A8
    A6T_VP8_B7
    A6T_VP8_D7
    A6T_VP8_F7
    A6T_VP8_G7
    A6T_VP8_H6
    A_20
    ADB_BD003
    ADB_BD015
    ADB_BD042
    ADB_BD111
    AF_NAK1-3
    Al31_1
    AO2_32
    apr34_000142F
    apr34_1784
    apr34_1789
    apr34_1792
    ARS04068_00002
    ARS04158_00005
    ARS04414_00003
    ARS04571_00004
    ARS06627_00002
    ARS06759_00001
    ARS06980_00002
    ARS07370_00002
    astrovirus_12
    AstV_Rat1
    AstV_Rat4
    ATA3_5Q_KO9_cl1
    ATA3_5Q_KO9_cl2
    AZD10a_EZLN1
    AZD10a_EZLN2
    AZD11_3
    AZD5_G10
    AZD8_5
    AZD9_B1
    AZD9_B2
    AZD9_W1
    AZD9_W2
    B01_8wb_b10_2005-12-15
    B15_CLW
    B18_CLW
    B24_CLW
    B29_CLW
    B3_2_cr
    B3_3_cr
    B7_CLW
    B_36
    B_A17_127
    B_A17_147
    B_C14_122
    B_C14_139
    B_C14_162
    B_C14_165
    B_C49_12
    B_C49_13
    B_C49_14
    B_C49_51
    B_C49_9
    B_C49_91
    B_D75_40
    B_D75_46
    B_D75_48
    B_D75_5
    B_D75_6
    B_D75_66
    B_D75_7
    B_D75_82
    B_EHO
    B_UC1
    Bact_1
    Bact_10
    Bact_11
    Bact_12
    Bact_13
    Bact_14
    Bact_15
    Bact_16
    Bact_17
    Bact_18
    Bact_19
    Bact_2
    Bact_20
    Bact_21
    Bact_22
    Bact_23
    Bact_24
    Bact_25
    Bact_26
    Bact_27
    Bact_28
    Bact_29
    Bact_3
    Bact_30
    Bact_31
    Bact_32
    Bact_33
    Bact_34
    Bact_35
    Bact_36
    Bact_37
    Bact_38
    Bact_39
    Bact_4
    Bact_40
    Bact_41
    Bact_42
    Bact_5
    Bact_6
    Bact_7
    Bact_8
    Bact_9
    bacteriumRM12EL_02D
    bacteriumRM12EL_05A
    BCoV/PB675/Art_lit/PAN/2011
    Beta_07_inf_18
    BgL12_3S_Black
    BgL12_3S_Purple
    BgL12_3S_Red
    BJCX4_extB9.1
    BM0354_1
    BM0354_2
    Bog1183_53
    Bog1249_12
    Bog5275_51
    Bog5712_52
    Bog8989_22
    Bog9017_22
    bovine/GIII.2/216_0114/2006/NOR
    bovine/GIII.2/240_0243/2005/NOR
    bovine/GIII.2/300_0250/2006/NOR
    bovine/GIII.2/312_0529/2006/NOR
    bovine/GIII.2/340_1235/2006/NOR
    bovine/GIII.2/471_0790/2005/NOR
    bovine/GIII.2/541_0448/2005/NOR
    bovine/GIII.2/584_3248/2005/NOR
    bovine/GIII.2/670_0799/2006/NOR
    bovine/GIII.2/718_0785/2006/NOR
    bovine/GIII/chimeric/107_0485/2005/NOR
    bovine/GIII/chimeric/661_1570/2006/NOR
    bovine/GIII/chimeric/752_3024/2006/NOR
    bovine/GIII/chimeric/785_0449/2006/NOR
    BRH_c32
    BtCoV/BRA100/Car_per/BRA/2009
    BtCoV/BRA114/Car_bre/BRA/2009
    BtCoV/BRA118/Car_per/BRA/2009
    BtCoV/BRA119/Car_per/BRA/2009
    BtCoV/BRA182/Mol_ruf/BRA/2009
    BtCoV/BRA344/Car_bre/BRA/2009
    BtCoV/BRAP103/Mol_cur/BRA/2009
    BtCoV/GrNC1/Neo_cap/RSA/2012
    BtCoV/GrNC2/Neo_cap/RSA/2012
    BtCoV/GrNC3/Neo_cap/RSA/2012
    BtCoV/GrNC4/Neo_cap/RSA/2012
    BtCoV/GrNC5/Neo_cap/RSA/2012
    BtCoV/GrNC6/Neo_cap/RSA/2012
    BtCoV/GrNC7/Neo_cap/RSA/2012
    BtCoV/GrNC8/Neo_cap/RSA/2012
    BtCoV/KCR15/Pte_par/CRC/2012
    BtCoV/KCR155/Pte_par/CRC/2012
    BtCoV/KCR180/Pte_par/CRC/2012
    BtCoV/KCR216/Car_per/CRC/2010
    BtCoV/KCR22/Pte_par/CRC/2012
    BtCoV/KCR230/Pte_par/CRC/2010
    BtCoV/KCR24/Ano_geo/CRC/2010
    BtCoV/KCR252/Car_per/CRC/2010
    BtCoV/KCR253/Car_per/CRC/2010
    BtCoV/KCR260/Car_per/CRC/2012
    BtCoV/KCR289/Ano_geo/CRC/2010
    BtCoV/KCR291/Ano_geo/CRC/2010
    BtCoV/KCR293/Ano_geo/CRC/2010
    BtCoV/KCR370/Pte_par/CRC/2011
    BtCoV/KCR372/Car_per/CRC/2010
    BtCoV/KCR90/Car_per/CRC/2010
    BtCoV/KCR91/Car_per/CRC/2010
    BtCoV/KP256/Art_jam/PAN/2010
    BtCoV/KP524/Art_jam/PAN/2010
    BtCoV/KP534/Art_lit/PAN/2010
    BtCoV/KP565/Art_jam/PAN/2010
    BtCoV/KP816/Phy_dis/PAN/2011
    BtCoV/KP817/Phy_dis/PAN/2011
    BtCoV/KW2E-F53/Nyc_spec/GHA/2011
    BtCoV/KW2E-F82/Nyc_spec/GHA/2011
    BtCoV/KW2E-F93/Nyc_spec/GHA/2010
    BtCoV/MSTM2/Min_nat/RSA/2010
    BtCoV/MSTM6/Min_nat/RSA/2010
    BtCoV/NCL_MCO1/Mop_con/RSA/2012
    BtCoV/OCR11/Pte_par/CRC/2011
    BtCoV/Rhi_bla/BB98-22/BGR/2008
    BtCoV/Rhi_eur/BB98-92/BGR/2008
    BtCoV/Rhi_eur/BB98-98/BGR/2008
    BtCoV/Rhi_eur/BB99-04/BGR/2009
    BtCoV/Rhi_fer/FR0711-B11/FRA/2011
    BtCoV/Rhi_fer/FR0711-B3/FRA/2011
    BtCoV/Rhi_fer/It1/ITA/2009
    BtCoV/Rhi_fer/It13/ITA/2009
    BtCoV/Rhi_fer/It15/ITA/2009
    BtCoV/Rhi_fer/It17/ITA/2009
    BtCoV/Rhi_fer/It2/ITA/2009
    BtCoV/Rhi_hip/R07-09/SPA/2010
    BtCoV/Rhi_hip/R13-08/SPA/2010
    BtCoV/Rhi_hip/R46-03/SPA/2010
    BtCoV/Rhi_hip/R7-08/SPA/2010
    BtCoV/Rhi_hip/R77-02/SPA/2010
    BtCoV/Rhi_hip/R8-09/SPA/2010
    BtCoV/Rhi_hip/Slo48/SLO/2009
    BtCoV/Rhi_hip/Slo52/SLO/2009
    BtCoV/Rhi_hip/Slo53/SLO/2009
    BtCoV/Rhi_hip/Slo54/SLO/2009
    BtCoV/Rhi_hip/Slo57/SLO/2009
    BtCoV/Rhi_hip/Slo69/SLO/2009
    BtCoV/VH_NC2/Neo_cap/RSA/2012
    C01_2wb_b10_2005-12-15
    C02_1ub_b10_2005-12-15
    C100_CLW
    C151_oct11_cons
    C18_CLW
    C39_CLW
    C47_CLW
    C49_CLW
    C60I_1
    C60I_10
    C60I_11B
    C60I_12
    C60I_14
    C60I_15
    C60I_16
    C60I_2
    C60I_4
    C60I_8
    C60I_9
    C67_CLW
    C72_CLW
    C79_CLW
    C84_CLW
    C92_CLW
    cat_SB2894
    CCSD_DF2030_Ag37_isolate_1
    CCSD_DF2030_Ag37_isolate_2
    CCSD_DF2030_Ag68_isolate_1
    CCSD_DF2030_Ag68_isolate_2
    CCSD_DF2030_Ag68_isolate_3
    CCSD_DF2030_CL68_isolate_2
    CCSD_DF2030_TE37_isolate_1
    CCSD_DF2030_TE37_isolate_2
    Ced_B08
    CG06_land_8_20_14_3_00_33_50
    CG06_land_8_20_14_3_00_37_11
    CG07_land_8_20_14_0_80_38_8
    CG09_39_24
    CG10_37_50
    CG10_46_32
    CG10_49_38
    CG10_big_fil_rev_8_21_14_0_10_33_18
    CG10_big_fil_rev_8_21_14_0_10_43_11
    CG17_big_fil_post_rev_8_21_14_2_50_64_8
    CG1_02_42_9
    CG2_30_31_98
    CG2_30_33_46
    CG2_30_37_16
    CG2_30_40_12
    CG2_30_54_10
    CG_4_10_14_0_2_um_filter_33_32
    CG_4_10_14_0_2_um_filter_48_144
    CG_4_10_14_0_2_um_filter_Archaea_38_6
    CG_4_10_14_0_8_um_filter_33_57
    CG_4_8_14_3_um_filter_33_28
    CG_4_8_14_3_um_filter_38_5
    CG_4_9_14_3_um_filter_33_26
    CG_4_9_14_3_um_filter_65_15
    CH30_LM136
    CH311c_11T
    CH336b_11D
    CH35_1T
    CH64b_2B
    CN30_LM100
    Col_A
    cr106_1
    cr107_1
    cr108_1
    cr109_1
    cr10_1
    cr110_1
    cr111_1
    cr112_1
    cr113_1
    cr114_1
    cr115_1
    cr116_1
    cr118_1
    cr11_1
    cr124_1
    cr125_1
    cr126_1
    cr127_1
    cr128_1
    cr130_1
    cr131_1
    cr1_1
    CR1_10_MR-20148
    CR1_11_MR-2014
    cr271_1
    cr272_1
    cr273_1
    cr3_1
    cr4_1
    cr50_1
    cr52_1
    cr53_1
    cr55_1
    cr56_1
    CR5_1_MR-2014
    CR5_2_MR-2014
    cr60_1
    cr6_1
    cr7_1
    CR7_6_MR-2014
    cr85_1
    cr8_1
    CRF01_AE
    CRF02_AG:08GQ032
    CRF02_AG:08GQ057
    CRF02_AG:08GQ058
    CRF02_AG:08GQ060
    CRF02_AG:08GQ061
    CRF02_AG:08GQ092
    CRF02_AG:08GQ093
    CRF02_AG:08GQ166
    CRF02_AG:08GQ200
    CRF02_AG:08GQ203
    CRF02_AG:08GQ239
    CRF02_AG:08GQ258
    CRF02_AG:08GQ269
    CRF02_AG:08GQ342
    CRF02_AG:08GQ370
    CRF02_AG:08GQ378
    CRF02_AG:08GQ385
    CRF02_AG:08GQ386
    CRF02_AG:08GQ392
    CRF02_AG:08GQ415
    CRF02_AG:08GQ433
    CRF02_AG:08GQ463
    CRF03_AB
    CRF04_cpx
    CRF06_cpx:08GQ191
    CRF08_BC
    CRF09_cpx:08GQ399
    CRF11_cpx:08GQ380
    CRF22_01A1:08GQ346
    CRF26_A5U:08GQ019
    CRF26_A5U:08GQ323
    CRO1006_1
    CRO1006_2
    CRO1091_1
    CRO1091_2
    CRO23_2
    CRO23_3
    CRO372_2
    CRO462_3
    CRO477_3
    CRO479_2
    CRO479_3
    CRO480_1
    CRO480_2
    CRO480_3
    CRO588_1
    CRO588_2
    CRO5_2
    CRO65_3
    CRO707_2
    CRO729_1
    CRO729_2
    CRO777_1
    CRO777_2
    CRO832_1
    CRO832_2
    CRO834_3
    Cry7_14
    Cry7_9.1
    Cry7_extC18.1
    CSeA30_26
    CSeA30_42
    CSeA30_59
    CSeA30_60
    CSu4_9
    D01_9wb_b10_2005-12-15
    D02_1pb_b10_2005-12-15
    D2_1_1F-16sF
    D2_1_3F-16sF
    D3_1_1F-16sF
    D3_1_3_2F-16sF
    DAB_16s_R32(2)
    DB_16s_F5
    DB_16s_F77
    DB_16s_F8
    DB_16s_R8
    Djaks_4
    Djaks_6
    DmelCS_103
    dog/GVI.1/HKU_Ca026F/2007/HKG
    dog/GVI.1/HKU_Ca035F/2007/HKG
    DOLJORAL78_63_78
    DOLJORAL78_65_58
    DOLZORAL124_38_8
    DOLZORAL124_64_63
    DSFBPG_3R2A
    DSFBPG_3R2B
    DSFBPG_4R3A
    DSFBPG_4R3B
    DSFBPG_4R3C
    DSFBPG_5R1B
    DSFBPG_5R3A
    DSFBPG_UR1A
    DSFBPG_UR1B
    DSFBPG_UR1C
    DSFBPG_UR1D
    DSFBPG_UR1E
    DSFBPG_UR2A
    DSFBPG_UR2B
    DSFBPG_UR2D
    DSFBPG_UR2E
    DSFBPG_UR2F
    DSFBPG_UR2G
    DSFBPG_UR3A
    DSFBPG_UR3B
    DSFBPG_UR3C
    DSFBPG_UR3D
    DSFBPG_UR3E
    DSFBPG_UR3F
    DSFBPG_URC2C
    DSFBPS_4R3A
    DSFBPS_UR1A
    DSFBPS_UR1B
    DSFBPS_UR1C
    DSFBPS_UR1D
    DSFBPS_UR1E
    DSFBPS_UR1F
    DSFBPS_UR1G
    DSFBPS_UR1H
    DSFBPS_UR2B
    DSFBPS_UR2C
    DSFBPS_UR2D
    DSFBPS_UR2E
    DSFBPS_UR2F
    DSFBPS_UR3A
    DSFBPS_UR3B
    DSFBPS_UR3C
    DSS3_MAL1
    DSS3_PM1
    DSS3_VP1
    E01_10wb_b10_2005-12-15
    E56_UTI
    E60_UTI
    E81_UTI
    Ep06i_A10B
    Ep06i_A1B
    Ep06i_A4C
    Eptesicus/13RS384_26/Italy/2012
    EqCoPV_11
    EqCoPV_8
    EqCoPV_9
    ES5_1357
    ES6_1418
    ES6_478
    ES_ALL_000190F
    ex4484_74
    Ex_DG74
    F01_14wb_b10_2005-12-15
    F3_an4
    F3_an5
    F3H1_a10
    FA1-2_000172F
    FA_121
    FA_130
    FA_132
    FA_139
    FA_149
    FA_151
    FA_152
    FA_183B
    FA_2CB*
    FA_86
    FAKO05_000032F
    FAKO27_000238F
    FAKO27_000271F
    Fen2266_11
    Fen418_41
    Fen4707_41
    Fen51_42
    Fen672_31
    Fen685_11
    Fen7786_21
    Fen7875_21
    Fen7895_21
    Fen7918_21
    Fen7940_21
    FGPV_KD09.ZAF
    flour_#16
    FMAS_AP8
    FMAS_AP9
    FMAS_PD2
    FMAS_PN2
    FMAS_PN3
    FMAS_PN4
    FMAS_PN5
    FR1_11-MR-2014
    FR1_16_MR-2014
    FR3_2_MR-2014
    G01_1mb_b10_2005-12-15
    G3_GreenLake
    G5_GreenLake
    G7_8_4BO2
    G_01
    G_02
    G_05
    G_10
    G_14
    G_20
    G_22
    G_23
    geoffroys_cat/URU-01/2009
    GF1-2_000079F
    GI/Hu/BG/2011/GI.P3_GI.3/RV1253
    GI/Hu/BG/2011/GI.Pb_GI.6/NV2(341)
    GI/Hu/JP/2007/GI.P3_GI.3/Shimizu/KK2866
    GI/Hu/JP/2007/GI.P8_GI.8/Nagoya/KY531
    GI/Hu/JP/FukuokaP/2015/GI.3/FE49_2014
    GI/Hu/KH/2004-2006/GI.P8_GI.8/NP-451_1
    GI/Hu/KH/2004-2006/GI.Pc_GI.5/NP-093_2
    GI/PBV/bovine/BRA/PBV18_PR/2014
    GII.4/Cordoba/Sds_August09/2009/ARG
    GII.4/Cordoba/Sds_July09-2/2009/ARG
    GII.4/Cordoba/Sds_July09-3/2009/ARG
    GII.4/Cordoba/Sds_July09-4/2009/ARG
    GII.4/Cordoba/Sds_July09-5/2009/ARG
    GII.4/Cordoba/Sds_July09-6/2009/ARG
    GII.4/Cordoba/Sds_July09/2009/ARG
    GII.4/Cordoba/Sds_June09/2009/ARG
    GII.4/Cordoba/Sus_August09/2009/ARG
    GII.4/Cordoba/Sus_July09-4/2009/ARG
    GII.4/Cordoba/Sus_July09-5/2009/ARG
    GII.4/Cordoba/Sus_July09/2009/ARG
    GII.4/Cordoba/WTPi_August09/2009/ARG
    GII.4/Cordoba/WTPi_July09-1/2009/ARG
    GII.4/Cordoba/WTPi_July09-2/2009/ARG
    GII.4/Cordoba/WTPi_July09-3/2009/ARG
    GII.4/Cordoba/WTPi_July09-4/2009/ARG
    GII.4/Cordoba/WTPi_July09/2009/ARG
    GII.4/Cordoba/WTPo_August09/2009/ARG
    GII.4/Cordoba/WTPo_June09/2009/ARG
    GII.4_2006b_Minerva
    GII.4_2010_New
    GII.g/Cordoba/Sds_July09-1/2009/ARG
    GII.g/Cordoba/Sus_July09-1/2009/ARG
    GII.g/Cordoba/Sus_July09-2/2009/ARG
    GII.g/Cordoba/Sus_July09-3/2009/ARG
    GII.g/Cordoba/WTPi_June09/2009/ARG
    GII.g/Cordoba/WTPo_July09-1/2009/ARG
    GII.g/Cordoba/WTPo_July09-2/2009/ARG
    GII.g/Cordoba/WTPo_July09-3/2009/ARG
    GII.g/Cordoba/WTPo_July09/2009/ARG
    GII.P12_GII.3
    GII.Pe_GII.4
    GII/Hu/BG/2009/GII.P21_GII.3/RV638
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV672
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV703
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV737
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV746
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV752
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV107
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV154
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV26
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV28
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV100
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV127
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV21
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV27
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV272
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV6
    GII/Hu/BG/2010/GII.P4_Den_Haag_2006b/RV107
    GII/Hu/BG/2010/GII.P4_Den_Haag_2006b/RV154
    GII/Hu/BG/2010/GII.P4_Den_Haag_2006b/RV26
    GII/Hu/BG/2010/GII.P4_New_Orleans_2009/RV21
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1389
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1521
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1540
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1543
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1368
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1389
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1521
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1528
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1540
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1543
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1223
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1251
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1256
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1278
    GII/Hu/BG/2012/GII.P7_GII.6/RV84
    GII/Hu/BG/2012/GII.P7_GII.6/RV85
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Aracaju19862
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Aracaju20078
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Aracaju20341
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Itabaiana20321
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Linhares20116
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Linhares20236
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Linhares20400
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Machado20361
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Nepomuceno20170
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Recife20276
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Ribamar20310
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Salvador20137
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Salvador20148
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis19516
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20190
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20196
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20292
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20302
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Tubarao19873
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Zortea20377
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Angra_dos_Reis21429
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Blumenau21624
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Fraiburgo21677
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Mantenopolos21548
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Paco_do_Lumiar21659
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Rio_de_Janeiro21389
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Sao_Jeronimo21393
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Sao_Luis21652
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Sao_Luis21666
    GII/Hu/ESP/2010/GII.P16_GII.13/SanSebastian356037
    GII/Hu/ESP/2012/GII.P7_GII.6/SanSebastian75500004
    GII/Hu/HK/2016/GII.P16_GII.2/CUHK-NS-1082
    GII/Hu/HKG/2015/GII.Pe_GII.17/CUHK-NS-682
    GII/Hu/IN/2005/GII.P4_GII.4_Hunter2004/Pune-PC09
    GII/Hu/IN/2006/GII.P4_GII.4_Hunter2004/Pune-PC12
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC11
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC13
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC16
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC18
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC19
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC20
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC21
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC23
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-0715450
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-0716584
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC37
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC45
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC46
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC47
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC49
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC56
    GII/Hu/IN/2007/GII.P4_GII.4_Hunter2004/Pune-PC31
    GII/Hu/IN/2007/GII.Pe_GII.4_Osaka2007/Pune-0713603
    GII/Hu/IN/2007/GII.Pe_GII.4_Osaka2007/Pune-0716588
    GII/Hu/IN/2008/GII.P4_GII.4_Apeldoorn2007/Pune-083366
    GII/Hu/IN/2008/GII.P4_GII.4_Apeldoorn2007/Pune-086619
    GII/Hu/IN/2008/GII.P4_GII.4_NewOrleans2009/Pune-0816948
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-0816943
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-0816955
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-0816962
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084093
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084097
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084105
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084695
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-08496
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-091056
    GII/Hu/IN/2009/GII.P4_GII.4_NewOrleans2009/Pune-093745
    GII/Hu/IN/2009/GII.Pe_GII.4_Osaka2007/Pune-091101
    GII/Hu/IN/2009/GII.Pe_GII.4_Osaka2007/Pune-092434
    GII/Hu/IN/2010/GII.P4_GII.4_NewOrleans2009/Pune-1022675
    GII/Hu/IN/2010/GII.P4_GII.4_NewOrleans2009/Pune-1038540
    GII/Hu/IN/2010/GII.P4_GII.4_NewOrleans2009/Pune-1038546
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-112370
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-113700
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-116686
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-116704
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-116710
    GII/Hu/IN/2012/GII.P4_GII.4_NewOrleans2009/Pune-124748
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-1310786
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-131742
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-1324318
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-136935
    GII/Hu/JP/2000/GII.P12_GII.12/Saitama/KU16
    GII/Hu/JP/2001/GII.P12_GII.12/Saitama/T15
    GII/Hu/JP/2001/GII.P12_GII.12/Saitama/T18
    GII/Hu/JP/2001/GII.P16_GII.17/OC01031-1
    GII/Hu/JP/2001/GII.P22_GII.5/Saitama/T49
    GII/Hu/JP/2002/GII.P12_GII.13/Saitama/T80
    GII/Hu/JP/2002/GII.P16_GII.17/Saitama/T87
    GII/Hu/JP/2002/GII.P5_GII.5/Saitama/T52
    GII/Hu/JP/2007/GII.P15_GII.15/Sapporo/HK299
    GII/Hu/JP/2007/GII.P21_GII.21/Kawasaki/YO284
    GII/Hu/JP/2007/GII.P7_GII.14/Fukuoka/KK282
    GII/Hu/JP/2007/GII.P7_GII.14/Sendai/YG99
    GII/Hu/JP/2010/GII.P4_GII.4/Shimada/ASC96
    GII/Hu/JP/2010/GII.P7_GII.7/Musashimurayama/TAKAsanKimchi
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/1
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/2
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/3
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/5
    GII/Hu/JP/2016/GII.P16_GII.4_Sydney2012/OC16023
    GII/Hu/JP/2016/GII.P16_GII.4_Sydney2012/OH16002
    GII/Hu/KH/2004-2006/GII.P12_GII.12/NP-162_1
    GII/Hu/KH/2004-2006/GII.P12_GII.12/NP-328_1
    GII/Hu/KH/2004-2006/GII.P12_GII.13/NP-259_1
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NP-011_2
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NP-014_2
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NP-312_1
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NPC-350_1
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NP-234_3
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NPC-239_1
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NPC-268_1
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NPC-269_1
    GII/Hu/KH/2004-2006/GII.P16_GII.17/NP-209_1
    GII/Hu/KH/2004-2006/GII.P21_GII.3/NP-355_1
    GII/Hu/KH/2004-2006/GII.P2_GII.2/NP-455_1
    GII/Hu/KH/2004-2006/GII.P3_GII.3/NP-122_1
    GII/Hu/KH/2004-2006/GII.P3_GII.3/NP-406_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Asia_2003/NP-593_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Asia_2003/NPC-405_2
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-328_3
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-330_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-353_2
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-356_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-422_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NPC-355_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NPC-405_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Yerseke_2006a/NP-544_3
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Yerseke_2006a/NP-561_3
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Yerseke_2006a/NP-598_3
    GII/Hu/KH/2004-2006/GII.P6_GII.6/NP-544_1
    GII/Hu/KH/2004-2006/GII.P7_GII.14/NP-255_1
    GII/Hu/KH/2004-2006/GII.P7_GII.14/NP-318_1
    GII/Hu/KH/2004-2006/GII.P7_GII.14/NP-593_3
    GII/Hu/KH/2004-2006/GII.P7_GII.20/NP-428_1
    GII/Hu/KH/2004-2006/GII.P7_GII.20/NP-432_1
    GII/Hu/KH/2004-2006/GII.P7_GII.20/NP-492_2
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-435_5
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-483_1
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-525_3
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-526_2
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-565_2
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NPC-031_1
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1006
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana108
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1087
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana124
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1331
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1363
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1367
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1383
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1560
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana181
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana192
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana212
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2172
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2192
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2395
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2398
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2414
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2443
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2481
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2487
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2515
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2536
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2612
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2619
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2629
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2698
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2721
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2763
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2768
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2820
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana34
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana36
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana362
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana456
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana464
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana75
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana795
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1025
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1026
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1096
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1314
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1399
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1447
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1457
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1537
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1643
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana181
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana251
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana504
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana539
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana566
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana58
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana655
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana76
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana774
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana826
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/105
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/106
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/108
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/109
    GII/Hu/ZA/2012/GII.Pe_GII.4
    GII/Hu/ZA/2012/GII.Pg_GII.12/Bushbuckridge9306
    GII/Hu/ZA/2013/GII.P7_GII.6/Johannesburg130930
    GII/Hu/ZAF/2009/GII.P4_GII.4/Johannesburg/4175
    GII/Hu/ZAF/2011/GII.P4_GII.4/Empangeni/8501
    GII/Hu/ZAF/2011/GII.P4_GII.4/Johannesburg/7028
    GII/Hu/ZAF/2012/GII.P4_GII.4/CapeTown/10105
    GII/Hu/ZAF/2012/GII.P4_GII.4/CapeTown/9772
    GII/Hu/ZAF/2012/GII.P4_GII.4/Empangeni/8604
    GII/Hu/ZAF/2012/GII.P4_GII.4/Empangeni/9693
    GII/Hu/ZAF/2012/GII.P4_GII.4/Johannesburg/BW
    GS2_2_25
    GS2_2_29
    GS3_1_3
    GSM_I_36B_cl_10_2
    GSM_I_36B_cl_11_3
    GW2011_AR1
    GW2011_AR10
    GW2011_AR11
    GW2011_AR13
    GW2011_AR15
    GW2011_AR16
    GW2011_AR17
    GW2011_AR18
    GW2011_AR19
    GW2011_AR20
    GW2011_AR21
    GW2011_AR3
    GW2011_AR4
    GW2011_AR5
    GW2011_AR6
    GW2011_AR9
    H01_1mb1_b10_2005-12-15
    H9_DGR
    H_A17_151
    H_C14_114
    H_C14_141
    H_C14_142
    H_C14_166
    H_C49_23
    H_C49_53
    H_C49_54
    H_C49_57
    H_C49_69
    H_C49_72
    H_C49_73
    Hare_CoV/La06_377/Lep_gra/ESP/2006
    Hare_CoV/La06_378/Lep_gra/ESP/2006
    Hare_CoV/La06_379/Lep_gra/ESP/2006
    Hare_CoV/La06_380/Lep_gra/ESP/2006
    Hare_CoV/La06_382/Lep_gra/ESP/2006
    Hare_CoV/La06_383/Lep_gra/ESP/2006
    Hare_CoV/La08_60/Lep_gra/ESP/2008
    Hare_CoV/La08_96/Lep_gra/ESP/2008
    Hare_CoV/La08_99/Lep_gra/ESP/2008
    HMP_JCVI_SC0102
    HMP_JCVI_SC0110
    HMP_JCVI_SC0119
    HMP_JCVI_SC0121
    HMP_JCVI_SC0124
    HMP_JCVI_SC0125
    HMP_JCVI_SC0126
    HMP_JCVI_SC0127
    HMP_JCVI_SC0141
    HMP_JCVI_SC0143
    HMP_JCVI_SC0175
    HMP_JCVI_SC0203
    HMP_JCVI_SC0224
    HMP_JCVI_SC0234
    HMP_JCVI_SC0278
    HMP_JCVI_SC0281
    hmp_mda_pilot_jcvi_0005
    Horo_12
    Horo_15
    Hu/Caterer_Company_1_pat_1/10-04-2003/NL
    Hu/Caterer_Company_1_pat_2/10-04-2003/NL
    Hu/Caterer_Company_2_pat_1/12-04-2003/NL
    Hu/Caterer_Company_2_pat_2/12-04-2003/NL
    Hu/Caterer_personel_1/16-04-2003/NL
    Hu/Caterer_personel_2/16-04-2003/NL
    Hu/Caterer_personel_3/16-04-2003/NL
    Hu/Caterer_personel_4/16-04-2003/NL
    Hu/Caterer_personel_5/16-04-2003/NL
    Hu/Daycare-center_A/21-02-2002/NL
    Hu/Daycare-center_B/12-12-2002/NL
    Hu/Diner_Amsterdam/08-07-2002/NL
    Hu/G1/FH-0931_Kr/2010
    Hu/G1/FH-0933_Kr/2010
    Hu/G2/FH-0901_Kr/2009
    Hu/G2/FH-0902_Kr/2009
    Hu/G2/FH-0903_Kr/2009
    Hu/G2/FH-0904_Kr/2009
    Hu/G2/FH-0905_Kr/2009
    Hu/G2/FH-0906_Kr/2009
    Hu/G2/FH-0907_Kr/2009
    Hu/G2/FH-0908_Kr/2009
    Hu/G2/FH-0909_Kr/2009
    Hu/G2/FH-0910_Kr/2009
    Hu/G2/FH-0911_Kr/2009
    Hu/G2/FH-0912_Kr/2009
    Hu/G2/FH-0913_Kr/2009
    Hu/G2/FH-0914_Kr/2009
    Hu/G2/FH-0915_Kr/2009
    Hu/G2/FH-0916_Kr/2009
    Hu/G2/FH-0917_Kr/2009
    Hu/G2/FH-0918_Kr/2009
    Hu/G2/FH-0919_Kr/2009
    Hu/G2/FH-0920_Kr/2009
    Hu/G2/FH-0921_Kr/2009
    Hu/G2/FH-0922_Kr/2009
    Hu/G2/FH-0923_Kr/2009
    Hu/G2/FH-0924_Kr/2009
    Hu/G2/FH-0925_Kr/2009
    Hu/G2/FH-0926_Kr/2009
    Hu/G2/FH-0927_Kr/2009
    Hu/G2/FH-0928_Kr/2010
    Hu/G2/FH-0929_Kr/2010
    Hu/G2/FH-0930_Kr/2010
    Hu/G2/FH-0932_Kr/2010
    Hu/G2/FH-0934_Kr/2009
    Hu/GGI.2/Treated_Wastewater/Llobregat_S6a/Jan2008/SP
    Hu/GGI.2/Wastewater/Llobregat_S5/Jan2008/SP
    Hu/GGI.3/Wastewater/Llobregat_S5/Apr2008/SP
    Hu/GGI.4/River_Water/Llobregat_S2/Dec2008/SP
    Hu/GGI.4/River_Water/Llobregat_S2/Jan2008/SP
    Hu/GGI.4/River_Water/Llobregat_S3/Dec2008/SP
    Hu/GGI.4/River_Water/Llobregat_S3/Feb2009/SP
    Hu/GGI.4/River_Water/Llobregat_S3/Nov2008/SP
    Hu/GGI.4/River_Water/Llobregat_S4/Feb2009/SP
    Hu/GGI.4/River_Water/Llobregat_S7/Apr2008/SP
    Hu/GGI.4/River_Water/Llobregat_S9/Apr2008/SP
    Hu/GGI.4/River_Water/Llobregat_S9/Feb2009/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Apr2008/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Feb2009/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Jan2009/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Jul2008/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Jun2008/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6b/Jan2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Apr2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Dec2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Feb2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Jan2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Jul2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Oct2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Sep2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Feb2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Feb2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Jan2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Jun2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Mar2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Sep2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Jan2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Nov2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Oct2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Sep2008/SP
    Hu/GGI.5/Wastewater/Llobregat_S8/Feb2008/SP
    Hu/GGI.5/Wastewater/Llobregat_S8b/May2008/SP
    Hu/GGI.5/Wastewater/Llobregat_S8c/May2008/SP
    Hu/GGI.7/ASPB06/2010/10_11180/Jan2010/SP
    Hu/GGI.7/ASPB08/2010/10_01949/Jan2010/SP
    Hu/GGI.7/ASPB80/2010/10_31750/Dec2010/SP
    Hu/GGI.7/Wastewater/Llobregat_S8/Jun2008/SP
    Hu/GGI.8/Wastewater/Llobregat_S8a/May2008/SP
    Hu/GGI.NA2/River_Water/Llobregat_S7/Jun2008/SP
    Hu/GGI.NA2/Treated_Wastewater/Llobregat_S6/Dec2008/SP
    Hu/GGI.NA3/River_Water/Llobregat_S4/Apr2008/SP
    Hu/GGII-4/Badalona/1586/BNM_05/06/Sp
    Hu/GGII-4/Banyoles/1474/Gi_10/06/Sp
    Hu/GGII-4/Barcelona/1258/BNM_08/05
    Hu/GGII-4/Blanes/1483/Gi_22/06/Sp
    Hu/GGII-4/Cabrera/1590/RCSP_01/Sp
    Hu/GGII-4/Calonge/Gi_03/1216/05
    Hu/GGII-4/Cambrils/1105/TA_10/05
    Hu/GGII-4/Corbera/RCSP_68/1134/04
    Hu/GGII-4/Els_Pallers/1617/TA_02/06/Sp
    Hu/GGII-4/Roses/1491/Gi_23/06/Sp
    Hu/GGII-4/Sabadell/1196/RC94_04/04
    Hu/GGII-4/Sabadell/1236/RC_08/05
    Hu/GGII-4/Sabadell/1510/RC_02/06/Sp
    Hu/GGII-4/Sant_Cugat/1527/RC_06/06/Sp
    Hu/GGII-4/Sant_Quirze/1092/RC_80/04
    Hu/GGII-4/Sant_Sadurni/1087/RC_84/04
    Hu/GGII-4/Tarragona/1613/TA_01/06/Sp
    Hu/GGII-4/Tarragona/1624/TA_03/06/Sp
    Hu/GGII-4/Tarragona/1636/TA_05/06/Sp
    Hu/GGII-4/Tremp/1567/Lleida_03/06/Sp
    Hu/GGII-4/Valencia_HCUV/593/02
    Hu/GGII-4/Vall_de_Bianya/1477/Gi_19/06/Sp
    Hu/GGII.12/ASPB80/2010/10_31978/Dec2010/SP
    Hu/GGII.13/Wastewater/Llobregat_S12/Apr2008/SP
    Hu/GGII.14/River_Water/Llobregat_S3/Feb2009/SP
    Hu/GGII.21/ASPB11/2010/10_01694/Feb2010/SP
    Hu/GGII.21/River_Water/Llobregat_S4/Jan2008/SP
    Hu/GGII.21/River_Water/Llobregat_S9b/Apr2008/SP
    Hu/GGII.21/Treated_Wastewater/Llobregat_S6/Feb2008/SP
    Hu/GGII.21/Treated_Wastewater/Llobregat_S6/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S12/Feb2009/SP
    Hu/GGII.21/Wastewater/Llobregat_S12/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S12/Oct2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S5/Feb2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S5/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S5/Jun2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S8/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S8/Jan2009/SP
    Hu/GGII.3/River_Water/Llobregat_S2/Dec2008/SP
    Hu/GGII.3/River_Water/Llobregat_S3/Dec2008/SP
    Hu/GGII.3/Wastewater/Llobregat_S5b/Apr2008/SP
    Hu/GGII.4/ASPB01/2011/11_00122/Jan2011/SP
    Hu/GGII.4/ASPB02/2010/10_00563/Jan2010/SP
    Hu/GGII.4/ASPB03/2011/11_02329/Jan2011/SP
    Hu/GGII.4/ASPB05/2010/10_01034/Jan2010/SP
    Hu/GGII.4/ASPB05/2010/10_01737/Jan2010/SP
    Hu/GGII.4/ASPB05/2010/10_01739/Jan2010/SP
    Hu/GGII.4/ASPB109/2012/12_22463/Oct2012/SP
    Hu/GGII.4/ASPB11/2010/10_01644/Feb2010/SP
    Hu/GGII.4/ASPB111/2011/11_21457/Oct2011/SP
    Hu/GGII.4/ASPB111/2012/12_22947/SP
    Hu/GGII.4/ASPB14/2011/11_05091/March2011/SP
    Hu/GGII.4/ASPB15/2011/11_05033/March2011/SP
    Hu/GGII.4/ASPB15/2011/11_05034/March2011/SP
    Hu/GGII.4/ASPB25/2010/10_05477/March2010/SP
    Hu/GGII.4/ASPB25/2010/10_05657/March2010/SP
    Hu/GGII.4/ASPB74/2010/10_30425/Nov2010/SP
    Hu/GGII.4/River_Water/Llobregat_S2/Jan2008/SP
    Hu/GGII.4/River_Water/Llobregat_S3/Feb2008/SP
    Hu/GGII.4/River_Water/Llobregat_S3/Nov2008/SP
    Hu/GGII.4/River_Water/Llobregat_S4b/Jan2008/SP
    Hu/GGII.4/River_Water/Llobregat_S7/Apr2008/SP
    Hu/GGII.4/River_Water/Llobregat_S9/Feb2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S12/Mar2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S5/Dec2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S5/Oct2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S8/Feb2008/SP
    Hu/GGII.6/ASPB105/2012/12_21910/Oct2012/SP
    Hu/GGII.6/ASPB108/2012/12_22457/Oct2012/SP
    Hu/GGII.6/ASPB38/2012/12_09395/May2012/SP
    Hu/GGII.6/ASPB79/2012/12_18747/Sep2012/SP
    Hu/GGII.6/River_Water/Llobregat_S3/Apr2008/SP
    Hu/GGII.6/River_Water/Llobregat_S9/Apr2008/SP
    Hu/GGII.6/Wastewater/Llobregat_S5/Apr2008/SP
    Hu/GGII.6/Wastewater/Llobregat_S5/Jan2009/SP
    Hu/GGII.6/Wastewater/Llobregat_S8/Oct2008/SP
    Hu/GGII.7/ASPB01/2011/11_00124/Jan2011/SP
    Hu/GGII.7/ASPB105/2012/12_22088/Oct2012/SP
    Hu/GGII.7/River_Water/Llobregat_S7/May2008/SP
    Hu/GI.1/CHA6A003_20091028/2009/USA
    Hu/GI.1/CHA6A003_20091031/2009/USA
    Hu/GI.1/CHA6A003_20091104/2009/USA
    Hu/GI.1/CHA9A004_20110419/2011/USA
    Hu/GI.1/CHA9A004_20110426/2011/USA
    Hu/GI.3/JKPG_881/SWE/2007
    Hu/GI.3/JKPG_882/SWE/2007
    Hu/GI.3/JKPG_883/SWE/2007
    Hu/GI.4/Muonio/11V1867_1/2011/FIN
    Hu/GI.4/Muonio/11V1869_1/2011/FIN
    Hu/GI.4/Muonio/11V1869_2/2011/FIN
    Hu/GI.6/M16_003/2016/JP
    Hu/GI.6/wastewater/Llobregat_S5/Oct2008/SP
    Hu/GI/GP1_Aug09_4102/ZAF
    Hu/GI/GP1_Feb10_4826/ZAF
    Hu/GI/GP1_Feb10_4832/ZAF
    Hu/GI/GP1_Jun09_3613/ZAF
    Hu/GI/GP1_Mar10_5068/ZAF
    Hu/GI/GP1_Mar11_7137/ZAF
    Hu/GI/GP1_May09_3474/ZAF
    Hu/GI/GP1_Oct09_4398/ZAF
    Hu/GI/GP2_Aug10_2676/ZAF
    Hu/GI/GP2_Feb08_1335/ZAF
    Hu/GI/GP2_Feb11_2905/ZAF
    Hu/GI/GP2_Jul08_1520/ZAF
    Hu/GI/GP2_Jun08_1496/ZAF
    Hu/GI/GP2_May08_1454/ZAF
    Hu/GI/InDRE2883_Tamps/2006/MEX
    Hu/GI/KZN_Dec10_6692/ZAF
    Hu/GI/MP_Aug10_6150/ZAF
    Hu/GI/MP_Dec11_8408/ZAF
    Hu/GI/MP_Jul11_3115/ZAF
    Hu/GI/MP_May10_5433/ZAF
    Hu/GI/MP_May10_5522/ZAF
    Hu/GI/MP_May11_7519/ZAF
    Hu/GI/WC_Feb11_6926/ZAF
    Hu/GI/WC_Jan11_6837/ZAF
    Hu/GI/WC_Jan11_6844/ZAF
    Hu/GII-4/2007a_ORF2-3/JP
    Hu/GII-4/2007a_ORF2/JP
    Hu/GII-4/2007b_ORF1/JP
    Hu/GII-4/2007b_ORF2-3/JP
    Hu/GII-4/2007b_ORF2/JP
    Hu/GII-4/2008a_ORF1/JP
    Hu/GII-4/2008a_ORF2-3/JP
    Hu/GII-4/2008a_ORF2/JP
    Hu/GII-4/2008b_ORF1/JP
    Hu/GII-4/2008b_ORF2-3/JP
    Hu/GII-4/2008b_ORF2/JP
    Hu/GII-4/Berga/RC69_02/926/03/Sp
    Hu/GII-4/Can_Sans/787/02/Sp
    Hu/GII-4/Manresa/81_02/914/02/Sp
    Hu/GII-4/Pineda_de_Mar/792/02/Sp
    Hu/GII-4/Res._S.Patricio/757/02/Sp
    Hu/GII-4/Res_Enbellpuig/793/02/Sp
    Hu/GII-4/Sallent/RC_73_02/921/02/Sp
    Hu/GII.13/ASPB03/2010/10_04332/March2010/SP
    Hu/GII.13/ASPB37/2010/10_15477/June2010/SP
    Hu/GII.2/Akita7_2014_JP
    Hu/GII.2/Akita8_2012_JP
    Hu/GII.2/Akita8_2014_JP
    Hu/GII.2/Aomori7_2014_JP
    Hu/GII.2/Ehime1_2012_JP
    Hu/GII.2/Ehime2_2012_JP
    Hu/GII.2/Ehime4_2014_JP
    Hu/GII.2/Ehime5_2014_JP
    Hu/GII.2/Fukui1_2012_JP
    Hu/GII.2/Fukui2_2012_JP
    Hu/GII.2/Hiroshima6_2014_JP
    Hu/GII.2/Hiroshimacity1_2012_JP
    Hu/GII.2/Hiroshimacity2_2012_JP
    Hu/GII.2/Hiroshimacity5_2014_JP
    Hu/GII.2/Hiroshimacity6_2014_JP
    Hu/GII.2/Miyagi1_2012_JP
    Hu/GII.2/Miyagi2_2014_JP
    Hu/GII.2/Niigata5_2012_JP
    Hu/GII.2/Osaka4_2014_JP
    Hu/GII.2/Osaka5_2014_JP
    Hu/GII.2/Osakacity5_2012_JP
    Hu/GII.2/Saitama4_2012_JP
    Hu/GII.2/Saitama5_2012_JP
    Hu/GII.4/Airv_B1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/Airv_C1_R10/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R10/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R11/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R12/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R13/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R14/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R17/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R18/2012/Gothenburg/Sweden
    Hu/GII.4/InDRE0644_Chis/MEX/2006
    Hu/GII.4/InDRE1307_Camp/MEX/2006
    Hu/GII.4/InDRE1401_Chis/MEX/2006
    Hu/GII.4/InDRE1697_Chis/MEX/2006
    Hu/GII.4/InDRE1839_Camp/MEX/2006
    Hu/GII.4/InDRE2115_Hgo/MEX/2006
    Hu/GII.4/InDRE2154_QRoo/MEX/2006
    Hu/GII.4/InDRE2256_QRoo/MEX/2006
    Hu/GII.4/InDRE2266_Mich/MEX/2006
    Hu/GII.4/InDRE2307_Camp/MEX/2006
    Hu/GII.4/InDRE2378_QRoo/MEX/2006
    Hu/GII.4/InDRE2384_Tamps/MEX/2006
    Hu/GII.4/InDRE2389_Tamps/MEX/2006
    Hu/GII.4/InDRE2395_Tamps/MEX/2006
    Hu/GII.4/InDRE2600_Camp/MEX/2006
    Hu/GII.4/InDRE2627_QRoo/MEX/2006
    Hu/GII.4/InDRE2628_QRoo/MEX/2006
    Hu/GII.4/InDRE2659_Camp/MEX/2006
    Hu/GII.4/InDRE2661_Camp/MEX/2006
    Hu/GII.4/InDRE2691_Chis/MEX/2006
    Hu/GII.4/InDRE2788_Tamps/MEX/2006
    Hu/GII.4/InDRE2812_Tamps/MEX/2006
    Hu/GII.4/InDRE2822_Tamps/MEX/2006
    Hu/GII.4/InDRE2835_QRoo/MEX/2006
    Hu/GII.4/InDRE2872_Chis/MEX/2006
    Hu/GII.4/InDRE2900_QRoo/MEX/2006
    Hu/GII.4/InDRE2913_QRoo/MEX/2006
    Hu/GII.4/InDRE2915_QRoo/MEX/2006
    Hu/GII.4/InDRE2917_QRoo/MEX/2006
    Hu/GII.4/InDRE2983_Tamps/MEX/2006
    Hu/GII.4/InDRE2987_Tamps/MEX/2006
    Hu/GII.4/InDRE2991_Camp/MEX/2006
    Hu/GII.4/InDRE3013_Ver/MEX/2006
    Hu/GII.4/InDRE3019_Ver/MEX/2006
    Hu/GII.4/InDRE3098_Yuc/MEX/2006
    Hu/GII.4/InDRE3101_Yuc/MEX/2006
    Hu/GII.4/InDRE3103_Yuc/MEX/2006
    Hu/GII.4/InDRE3114_Yuc/MEX/2006
    Hu/GII.4/InDRE3116_Yuc/MEX/2006
    Hu/GII.4/InDRE3129_Yuc/MEX/2006
    Hu/GII.4/InDRE3144_Tamps/MEX/2006
    Hu/GII.4/InDRE3156_Tamps/MEX/2006
    Hu/GII.4/InDRE3159_Tamps/MEX/2006
    Hu/GII.4/InDRE3186_Ver/MEX/2006
    Hu/GII.4/InDRE3203_QRoo/MEX/2006
    Hu/GII.4/InDRE3228_QRoo/MEX/2006
    Hu/GII.4/InDRE3238_Tamps/MEX/2006
    Hu/GII.4/InDRE3242_Tamps/MEX/2006
    Hu/GII.4/InDRE3246_Tamps/MEX/2006
    Hu/GII.4/InDRE3251_Tamps/MEX/2006
    Hu/GII.4/InDRE3268_Tamps/MEX/2006
    Hu/GII.4/InDRE3293_Tamps/MEX/2006
    Hu/GII.4/InDRE3296_Tamps/MEX/2006
    Hu/GII.4/InDRE3300_Tamps/MEX/2006
    Hu/GII.4/InDRE3344_Tamps/MEX/2006
    Hu/GII.4/InDRE3387_Tamps/MEX/2006
    Hu/GII.4/InDRE3400_Ver/MEX/2006
    Hu/GII.4/InDRE3402_Ver/MEX/2006
    Hu/GII.4/InDRE3437_Tamps/MEX/2006
    Hu/GII.4/InDRE3927_Mich/MEX/2006
    Hu/GII.4/InDRE3928_Mich/MEX/2006
    Hu/GII.4/InDRE3933_Mich/MEX/2006
    Hu/GII.4/InDRE3934_Mich/MEX/2006
    Hu/GII.4/P_02-2010_1/GER/2010
    Hu/GII.4/P_02-2010_10/GER/2010
    Hu/GII.4/P_02-2010_11/GER/2010
    Hu/GII.4/P_02-2010_4/GER/2010
    Hu/GII.4/P_02-2010_5/GER/2010
    Hu/GII.4/P_02-2010_6/GER/2010
    Hu/GII.4/P_02-2010_7/GER/2010
    Hu/GII.4/P_02-2010_8/GER/2010
    Hu/GII.4/P_02-2010_9/GER/2010
    Hu/GII.4/P_02-2020_3/GER/2010
    Hu/GII.4/P_04-2009_10/GER/2009
    Hu/GII.4/P_04-2009_11/GER/2009
    Hu/GII.4/P_04-2009_13/GER/2009
    Hu/GII.4/P_04-2009_3/GER/2009
    Hu/GII.4/P_04-2009_5/GER/2009
    Hu/GII.4/P_04-2009_6/GER/2009
    Hu/GII.4/P_04-2009_8/GER/2009
    Hu/GII.4/P_04-2009_9/GER/2009
    Hu/GII.4/P_09-2009_1/GER/2009
    Hu/GII.4/P_09-2009_10/GER/2009
    Hu/GII.4/P_09-2009_2/GER/2009
    Hu/GII.4/P_09-2009_3/GER/2009
    Hu/GII.4/P_09-2009_4/GER/2009
    Hu/GII.4/P_09-2009_5/GER/2009
    Hu/GII.4/P_09-2009_6/GER/2009
    Hu/GII.4/P_09-2009_7/GER/2009
    Hu/GII.4/P_09-2009_8/GER/2009
    Hu/GII.4/P_09-2009_9/GER/2009
    Hu/GII.4/patient_A+5mo/2010/USA
    Hu/GII.4/patient_A/2010/USA
    Hu/GII.4/patient_B/2010/USA
    Hu/GII.4/patient_C/2010/USA
    Hu/GII.4/PR_BRA_1008_2011
    Hu/GII.4/PR_BRA_1382_2001
    Hu/GII.4/PR_BRA_2082_2011
    Hu/GII.4/PR_BRA_2901_2011
    Hu/GII.4/PR_BRA_3074_2011
    Hu/GII.4/PR_BRA_3157_2011
    Hu/GII.4/PR_BRA_3172_2011
    Hu/GII.4/PR_BRA_3228_2011
    Hu/GII.4/PR_BRA_3270_2011
    Hu/GII.4/PR_BRA_3329_2011
    Hu/GII.4/PR_BRA_3353_2011
    Hu/GII.4/PR_BRA_3484_2011
    Hu/GII.4/PR_BRA_4287_2010
    Hu/GII.4/PR_BRA_762_2011
    Hu/GII.4/Ptab_B1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/Ptab_C1_R7/2012/Gothenburg/Sweden
    Hu/GII.4/R_02-2010_1/GER/2010
    Hu/GII.4/R_02-2010_11/GER/2010
    Hu/GII.4/R_02-2010_2/GER/2010
    Hu/GII.4/R_02-2010_3/GER/2010
    Hu/GII.4/R_02-2010_4/GER/2010
    Hu/GII.4/R_02-2010_5/GER/2010
    Hu/GII.4/R_02-2010_6/GER/2010
    Hu/GII.4/R_02-2010_7/GER/2010
    Hu/GII.4/R_02-2010_8/GER/2010
    Hu/GII.4/R_09-2009_1/GER/2009
    Hu/GII.4/R_09-2009_10/GER/2009
    Hu/GII.4/R_09-2009_11/GER/2009
    Hu/GII.4/R_09-2009_3/GER/2009
    Hu/GII.4/R_09-2009_5/GER/2009
    Hu/GII.4/R_09-2009_6/GER/2009
    Hu/GII.4/R_09-2009_8/GER/2009
    Hu/GII.4/R_09-2009_9/GER/2009
    Hu/GII.4/Salo/11V2359_1/2011/FIN
    Hu/GII.4/Vt_A1_R10/2012/Gothenburg/Sweden
    Hu/GII.4/Vt_A1_R11/2012/Gothenburg/Sweden
    Hu/GII.4/Vt_B2_R15/2012/Gothenburg/Sweden
    Hu/GII.4/whb_B1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/whb_C1_R7/2012/Gothenburg/Sweden
    Hu/GII.4_variant
    Hu/GII.7/A_03-2006_11/GER/2006
    Hu/GII.7/A_03-2006_12/GER/2006
    Hu/GII.7/A_03-2006_2/GER/2006
    Hu/GII.7/A_03-2006_3/GER/2006
    Hu/GII.7/A_03-2006_4/GER/2006
    Hu/GII.7/A_03-2006_5/GER/2006
    Hu/GII.7/A_03-2006_6/GER/2006
    Hu/GII.7/A_03-2006_7/GER/2006
    Hu/GII.7/A_03-2006_9/GER/2006
    Hu/GII.7/A_06-2006_1/GER/2006
    Hu/GII.7/A_06-2006_2/GER/2006
    Hu/GII.7/A_06-2006_3/GER/2006
    Hu/GII.7/A_06-2006_4/GER/2006
    Hu/GII.7/A_06-2006_5/GER/2006
    Hu/GII.7/A_06-2006_7/GER/2006
    Hu/GII.7/A_06-2006_8/GER/2006
    Hu/GII.7/A_06-2006_9/GER/2006
    Hu/GII.7/GII.6/LNOV_PA16014/BRAZIL/2016
    Hu/GII.7/Muonio/11V1867_2/2011/FIN
    Hu/GII.8/GII.8/LNOV_PA16001/BRAZIL/2016
    Hu/GII.g/Salo/11V2362_1/2011/FIN
    Hu/GII.P.17/GII.17/LNOV_PA16004/BRAZIL/2016
    Hu/GII.P.17/GII.17/LNOV_PA16005/BRAZIL/2016
    Hu/GII.P.17/GII.17/LNOV_PA16007/BRAZIL/2016
    Hu/GII.P.17/GII.17/LNOV_PA16009/BRAZIL/2016
    Hu/GII.P16_GII.13/PR1395/2012/ITA
    Hu/GII.P16_GII.13/PR6717/2010/ITA
    Hu/GII.P16_GII.3/PR11068/2011/ITA
    Hu/GII.P16_GII.3/PR4526/2012/ITA
    Hu/GII.P17_GII.17/
    Hu/GII.P17_GII.17/Jinshan15/Shanghai/2014/CHN
    Hu/GII.P17_GII.17/NSW6016/2014/AU
    Hu/GII.P17_GII.17/PA31/2016/ITA
    Hu/GII.P17_GII.17/PA39/2016/ITA
    Hu/GII.P17_GII.17/PR668/2015/ITA
    Hu/GII.P4_GII.4/Minerva/2006/USA
    Hu/GII.P6_GII.6/1401Y107/CHN/2014
    Hu/GII.P7_GII.6/1401Y075/CHN/2014
    Hu/GII.P7_GII.6/1401Y127/CHN/2014
    Hu/GII.P7_GII.6/1403Y001/CHN/2014
    Hu/GII.P7_GII.6/1405Y017/CHN/2014
    Hu/GII.P7_GII.6/1405Y024/CHN/2014
    Hu/GII.P7_GII.6/1405Y035/CHN/2014
    Hu/GII.P7_GII.6/1406Y114/CHN/2014
    Hu/GII.P7_GII.6/1415Y002/CHN/2014
    Hu/GII.P7_GII.6/Maryland/2014/USA
    Hu/GII.P7_GII.7/1401Y082/CHN/2014
    Hu/GII.P7_GII.7/1406Y012/CHN/2014
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16002/BRAZIL/2016
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16003/BRAZIL/2016
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16010/BRAZIL/2016
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16016/BRAZIL/2016
    Hu/GII.Pg/GII.1/LNOV_PA16008/BRAZIL/2016
    Hu/GII.Pg/GII.1/LNOV_PA16011/BRAZIL/2016
    Hu/GII/AUS/2015/GII.P17_GII.17/Brisbane462
    Hu/GII/CN/2013/GII.P17_GII.17/Nanjing010141
    Hu/GII/GP1_Apr11_7390/ZAF
    Hu/GII/GP1_Feb11_6939/ZAF
    Hu/GII/GP1_Mar12_8762/ZAF
    Hu/GII/GP1_Nov10_6514/ZAF
    Hu/GII/GP1_Oct10_6381/ZAF
    Hu/GII/GP1_Sep11_8120/ZAF
    Hu/GII/GP2_Feb08_1351/ZAF
    Hu/GII/GP2_Mar11_2963/ZAF
    Hu/GII/GP2_Nov08_1706/ZAF
    Hu/GII/JP/2004/GII.P12_GII.2/Tochigi-92
    Hu/GII/JP/2004/GII.P2_GII.2/Hokkaido-13
    Hu/GII/JP/2004/GII.P2_GII.2/Tochigi-85
    Hu/GII/JP/2004/GII.P2_GII.2/Tochigi-86
    Hu/GII/JP/2004/GII.P2_GII.2/Tochigi-87
    Hu/GII/JP/2006/GII.P2_GII.2/Hokkaido-14
    Hu/GII/JP/2008/GII.P2_GII.2/Hokkaido-15
    Hu/GII/JP/2009/GII.P16_GII.2/Kanagawa-49
    Hu/GII/JP/2010/GII.P16_GII.2/Ehime-43
    Hu/GII/JP/2010/GII.P16_GII.2/Ehime-44
    Hu/GII/JP/2010/GII.P16_GII.2/Kanagawa-50
    Hu/GII/JP/2010/GII.P16_GII.2/Kanagawa-51
    Hu/GII/JP/2010/GII.P16_GII.2/Osaka-019
    Hu/GII/JP/2010/GII.P2_GII.2/Hiroshima-17
    Hu/GII/JP/2010/GII.P2_GII.2/Hiroshima-18
    Hu/GII/JP/2010/GII.P2_GII.2/Hiroshima-19
    Hu/GII/JP/2010/GII.P2_GII.2/Hokkaido-16
    Hu/GII/JP/2011/GII.P16_GII.2/Ehime-45
    Hu/GII/JP/2011/GII.P16_GII.2/Hiroshima-26
    Hu/GII/JP/2011/GII.P16_GII.2/Hokkaido-17
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-023
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-038
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-18
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-26
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-9
    Hu/GII/JP/2011/GII.P16_GII.2/Saitama-51
    Hu/GII/JP/2011/GII.P16_GII.2/Yamaguchi-4
    Hu/GII/JP/2012/GII.P16_GII.2/Ehime-46
    Hu/GII/JP/2012/GII.P16_GII.2/Hokkaido-18
    Hu/GII/JP/2012/GII.P16_GII.2/Saitama-121
    Hu/GII/JP/2012/GII.P16_GII.2/Saitama-122
    Hu/GII/JP/2012/GII.P16_GII.2/Tochigi-26
    Hu/GII/JP/2012/GII.P16_GII.2/Tochigi-30
    Hu/GII/JP/2013/GII.P16_GII.2/Ehime-6
    Hu/GII/JP/2013/GII.P16_GII.2/Ehime-8
    Hu/GII/JP/2013/GII.P16_GII.2/Ehime-9
    Hu/GII/JP/2013/GII.P16_GII.2/Miyagi-7
    Hu/GII/JP/2013/GII.P16_GII.2/Miyagi-8
    Hu/GII/JP/2013/GII.P16_GII.2/Saitama-123
    Hu/GII/JP/2013/GII.P16_GII.2/Saitama-124
    Hu/GII/JP/2013/GII.P16_GII.2/Saitama-125
    Hu/GII/JP/2013/GII.P16_GII.2/Tochigi-46
    Hu/GII/JP/2013/GII.P17_GII.17/FE21_2013
    Hu/GII/JP/2013/GII.P17_GII.17/Saitama5203
    Hu/GII/JP/2013/GII.P17_GII.17/Saitama5309
    Hu/GII/JP/2014/GII.P16_GII.2/Hiroshima-30
    Hu/GII/JP/2014/GII.P16_GII.2/Kanagawa-52
    Hu/GII/JP/2014/GII.P16_GII.2/Osaka-225
    Hu/GII/JP/2014/GII.P16_GII.2/Osaka246
    Hu/GII/JP/2014/GII.P16_GII.2/Saitama-126
    Hu/GII/JP/2014/GII.P16_GII.2/Tochigi-17
    Hu/GII/JP/2014/GII.P17_GII.17/Kawasaki323
    Hu/GII/JP/2014/GII.P17_GII.17/Nagano7-1
    Hu/GII/JP/2014/GII.P17_GII.17/Nagano8-1
    Hu/GII/JP/2014/GII.P2_GII.2/Yamaguchi-014
    Hu/GII/JP/2014/GII.Pe_GII.2/Saitama-127
    Hu/GII/JP/2015/GII.P17_GII.17/FE20-2015
    Hu/GII/JP/2015/GII.P17_GII.17/FE44-2014
    Hu/GII/JP/2015/GII.P17_GII.17/FE47-2014
    Hu/GII/JP/2015/GII.P17_GII.17/FE58-2014
    Hu/GII/JP/2015/GII.P17_GII.17/Kawasaki308
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster1-2015
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster2-2015
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster3-2015
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster4-2015
    Hu/GII/JP/2015/GII.P2_GII.2/Miyagi-63
    Hu/GII/JP/2015/GII.P2_GII.2/Saitama-169
    Hu/GII/JP/2015/GII.Pe_GII.4/Osaka/OSF78
    Hu/GII/JP/2016/GII.P16_GII.4_Sydney2012/Kawasaki194
    Hu/GII/JP/2016/P16_GII.2/Aichi430
    Hu/GII/JP/2016/P16_GII.2/AichiF219
    Hu/GII/JP/2016/P16_GII.2/AichiF243
    Hu/GII/JP/2017/GII.P17_GII.17/Tokyo330021
    Hu/GII/JP/2017/P16_GII.2/Aichi156
    Hu/GII/MP_Jul09_3880/ZAF
    Hu/GII/MP_Mar11_2949/ZAF
    Hu/GII/OuagadougouBF102/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF128/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF136/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF146/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF152/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF25/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF27/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF37/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF38/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouTB04/GII.4_Sydney2012/2011/BFA
    Hu/GII/PA/VIR_16F/BR/04-2012
    Hu/GII/PA/VIR_17F/BR/04-2012
    Hu/GII/PA/VIR_18F/BR/04-2012
    Hu/GII/PA/VIR_19F/BR/04-2012
    Hu/GII/PA/VIR_1F/BR/03-2012
    Hu/GII/PA/VIR_20F/BR/04-2012
    Hu/GII/PA/VIR_22F/BR/05-2012
    Hu/GII/PA/VIR_24F/BR/05-2012
    Hu/GII/PA/VIR_25F/BR/05-2012
    Hu/GII/PA/VIR_28F/BR/05-2012
    Hu/GII/PA/VIR_2F/BR/03-2012
    Hu/GII/PA/VIR_31F/BR/06-2012
    Hu/GII/PA/VIR_32F/BR/06-2012
    Hu/GII/PA/VIR_35F/BR/06-2012
    Hu/GII/PA/VIR_3F/BR/03-2012
    Hu/GII/PA/VIR_45F/BR/07-2012
    Hu/GII/PA/VIR_45S/BR/07-2012
    Hu/GII/PA/VIR_63F/BR/08-2012
    Hu/GII/PA/VIR_73F/BR/08-2012
    Hu/GII/PA/VIR_76F/BR/08-2012
    Hu/GII/PA/VIR_82F/BR/08-2012
    Hu/GII/PA/VIR_82S/BR/08-2012
    Hu/GII/RO/2015/GII.P17_GII.17/Dezna187865
    Hu/GII/RO/2015/GII.P17_GII.17/Sebis187601
    Hu/GII/RO/2015/GII.P17_GII.17/Sebis187602
    Hu/GII/WC_Jun10_5755/ZAF
    Hu/GIV/GP1_Jun10_5795/ZAF
    Hu/GIV/GP1_Mar10_5040/ZAF
    Hu/GIV/GP1_Mar10_5065/ZAF
    Hu/GIV/GP1_Mar11_7139/ZAF
    Hu/GIV/GP1_Sep09_4231/ZAF
    Hu/GIV/GP2_Dec08_1766/ZAF
    Hu/GIV/GP2_Jul10_2631/ZAF
    Hu/GIV/GP3_Mar08_1383/ZAF
    Hu/GIV/MP_Mar10_5036/ZAF
    Hu/GIV/MP_Mar11_8760/ZAF
    Hu/GIV/MP_May10_5510/ZAF
    Hu/GP1_Aug09_4102/ZAF
    Hu/GP1_Feb10_4826/ZAF
    Hu/GP1_Feb10_4832/ZAF
    Hu/GP1_Jun09_3613/ZAF
    Hu/GP1_Jun10_5795/ZAF
    Hu/GP1_Nov10_6514/ZAF
    Hu/GP2_Apr08_1416/ZAF
    Hu/GP2_Aug10_2676/ZAF
    Hu/GP2_Feb08_1351/ZAF
    Hu/GP2_Jun08_1496/ZAF
    Hu/GP2_Jun08_1520/ZAF
    Hu/GP2_Mar11_2949/ZAF
    Hu/Hospital_AA_PAAZ_pat_1/15-07-2002/NL
    Hu/Hospital_AA_PAAZ_personel_1/18-07-2002/NL
    Hu/Hospital_AB_A1_pat_104-02-2003/NL
    Hu/Hospital_AB_A1_pat_205-03-2003/NL
    Hu/Hospital_AB_A3_pat_1/10-02-2003/NL
    Hu/Hospital_AB_A3_personel_1/10-02-2003/NL
    Hu/Hospital_AB_A7_pat_1/06-01-2003/NL
    Hu/Hospital_AB_A8_pat_1/21-01-2003/NL
    Hu/Hospital_AB_A9_pat_1/18-12-2002/NL
    Hu/Hospital_AB_B1_pat_1/22-01-2003/NL
    Hu/Hospital_AB_B1_pat_2/28-02-2003/NL
    Hu/Hospital_AB_B1_personel_1/27-01-2003/NL
    Hu/Hospital_AB_PAAZ_pat07-02-2003/NL2.
    Hu/Hospital_AB_PAAZ_pat_3/11-02-2003/NL
    Hu/InDRE0644_Chis/MEX/2006
    Hu/InDRE1307_Camp/MEX/2006
    Hu/InDRE1401_Chis/MEX/2006
    Hu/InDRE1406_DF/MEX/2006
    Hu/InDRE1697_Chis/MEX/2006
    Hu/InDRE1835_NL/MEX/2006
    Hu/InDRE1843_Camp/MEX/2006
    Hu/InDRE2010_DF/MEX/2006
    Hu/InDRE2115_Hgo/MEX/2006
    Hu/InDRE2154_QRoo/MEX/2006
    Hu/InDRE2256_QRoo/MEX/2006
    Hu/InDRE2257_QRoo/MEX/2006
    Hu/InDRE2264_Mich/MEX/2006
    Hu/InDRE2265_Mich/MEX/2006
    Hu/InDRE2266_Mich/MEX/2006
    Hu/InDRE2283_QRoo/MEX/2006
    Hu/InDRE2289_QRoo/MEX/2006
    Hu/InDRE2295_QRoo/MEX/2006
    Hu/InDRE2389_Tamps/MEX/2006
    Hu/InDRE2395_Tamps/MEX/2006
    Hu/InDRE2627_QRoo/MEX/2006
    Hu/InDRE2628_QRoo/MEX/2006
    Hu/InDRE2641_QRoo/MEX/2006
    Hu/InDRE2659_Camp/MEX/2006
    Hu/InDRE2679_Chis/MEX/2006
    Hu/InDRE2812_Tamps/MEX/2006
    Hu/InDRE2913_QRoo/MEX/2006
    Hu/InDRE2915_QRoo/MEX/2006
    Hu/InDRE2917_QRoo/MEX/2006
    Hu/InDRE2943_QRoo/MEX/2006
    Hu/InDRE2987_Tamps/MEX/2006
    Hu/InDRE2991_Camp/MEX/2006
    Hu/InDRE3144_Tamps/MEX/2006
    Hu/InDRE3156_Tamps/MEX/2006
    Hu/InDRE3186_Ver/MEX/2006
    Hu/InDRE3228_QRoo/MEX/2006
    Hu/InDRE3238_Tamps/MEX/2006
    Hu/InDRE3242_Tamps/MEX/2006
    Hu/InDRE3244_Tamps/MEX/2006
    Hu/InDRE3246_Tamps/MEX/2006
    Hu/InDRE3344_Tamps/MEX/2006
    Hu/InDRE3358_Tamps/MEX/2006
    Hu/InDRE3919_Mich/MEX/2006
    Hu/InDRE3925_Mich/MEX/2006
    Hu/InDRE3926_Mich/MEX/2006
    Hu/InDRE3927_Mich/MEX/2006
    Hu/InDRE3928_Mich/MEX/2006
    Hu/InDRE3933_Mich/MEX/2006
    Hu/InDRE3934_Mich/MEX/2006
    Hu/KZN_Dec10_6692/ZAF
    Hu/Lelystad_Hospital/16-01-2003/NL
    Hu/Lelystad_Nursing-home/21-07-2002/NL
    Hu/Lelystad_Nursing-home/22-07-2002/NL
    Hu/MP_Dec11_8408/ZAF
    Hu/MP_Jul09_3880/ZAF
    Hu/MP_Jul11_3115/ZAF
    Hu/MP_May10_5433/ZAF
    Hu/MP_May11_7519/ZAF
    Hu/NoV/05_1/2002/SE
    Hu/NoV/05_2/2002/SE
    Hu/NoV/07_1/2005/SE
    Hu/NoV/07_2/2005/SE
    Hu/NoV/10_1/2006/SE
    Hu/NoV/10_2/2006/SE
    Hu/NoV/13_1/2006/SE
    Hu/NoV/13_2/2006/SE
    Hu/NoV/22_1/2006/SE
    Hu/NoV/22_2/2006/SE
    Hu/NoV/23_1/2006/SE
    Hu/NoV/23_2/2006/SE
    Hu/Nursing-home_A/28-12-2001/NL
    Hu/Nursing-home_B/05-05-2002/NL
    Hu/Nursing-home_C/06-06-2002/NL
    Hu/Nursing-home_D/19-11-2002/NL
    Hu/Nursing-home_E_pat_1/08-01-2003/NL/
    Hu/Nursing-home_E_pat_2/10-01-2003/NL
    Hu/Nursing-home_F/22-01-2003/NL
    Hu/Nursing-home_G/27-01-2003/NL
    Hu/Nursing-home_H/03-02-2003/NL
    Hu/PV_Voelk0017/2013/DEU
    Hu/PV_Voelk0018/2013/DEU
    Hu/River_Cruising_ship_1/16-05-2001/NL
    Hu/River_Cruising_Ship_2/08-04-2003/NL
    Hu/RU/2013/GII.4_Sydney2012/Nizhny_Novgorod4874
    Hu/RU/2014/GII.6/Nizhny_Novgorod1404
    Hu/RU/2014/GII.6/Nizhny_Novgorod1470
    Hu/RU/2014/GII.6/Nizhny_Novgorod1760
    Hu/RU/2014/GII.6/Nizhny_Novgorod18
    Hu/RU/2014/GII.6/Nizhny_Novgorod1908
    Hu/RU/2014/GII.6/Nizhny_Novgorod3125
    Hu/USA/2011/GI.P7_GI.7/CS5567
    Hu/USA/2014/GI.P7_GI.7/GA5043
    Hu/USA/2015/GII.P16_GII.4_Sydney/CA3477
    Hu/USA/2016/GI.P9_GI.9/SC6350
    Hu/WC_Feb11_6926/ZAF
    Hu/WC_Jun10_5755/ZAF
    Human/GII/InDRE1839_Camp/2006/Mexico
    Human/GII/InDRE2307_Camp/2006/Mexico
    Human/GII/InDRE2378_QRoo/2006/Mexico
    Human/GII/InDRE2384_Tamps/2006/Mexico
    Human/GII/InDRE2600_Camp/2006/Mexico
    Human/GII/InDRE2661_Camp/2006/Mexico
    Human/GII/InDRE2691_Chis/2006/Mexico
    Human/GII/InDRE2788_Tamps/2006/Mexico
    Human/GII/InDRE2822_Tamps/2006/Mexico
    Human/GII/InDRE2835_QRoo/2006/Mexico
    Human/GII/InDRE2872_Chis/2006/Mexico
    Human/GII/InDRE2900_QRoo/2006/Mexico
    Human/GII/InDRE2919_QRoo/2006/Mexico
    Human/GII/InDRE2983_Tamps/2006/Mexico
    Human/GII/InDRE3013_Ver/2006/Mexico
    Human/GII/InDRE3019_Ver/2006/Mexico
    Human/GII/InDRE3098_Yuc/2006/Mexico
    Human/GII/InDRE3101_Yuc/2006/Mexico
    Human/GII/InDRE3103_Yuc/2006/Mexico
    Human/GII/InDRE3107_Yuc/2006/Mexico
    Human/GII/InDRE3114_Yuc/2006/Mexico
    Human/GII/InDRE3115_Yuc/2006/Mexico
    Human/GII/InDRE3124_Yuc/2006/Mexico
    Human/GII/InDRE3129_Yuc/2006/Mexico
    Human/GII/InDRE3159_Tamps/2006/Mexico
    Human/GII/InDRE3203_QRoo/2006/Mexico
    Human/GII/InDRE3251_Tamps/2006/Mexico
    Human/GII/InDRE3260_Tamps/2006/Mexico
    Human/GII/InDRE3268_Tamps/2006/Mexico
    Human/GII/InDRE3293_Tamps/2006/Mexico
    Human/GII/InDRE3296_Tamps/2006/Mexico
    Human/GII/InDRE3297_Tamps/2006/Mexico
    Human/GII/InDRE3300_Tamps/2006/Mexico
    Human/GII/InDRE3372_Tamps/2006/Mexico
    Human/GII/InDRE3387_Tamps/2006/Mexico
    Human/GII/InDRE3400_Ver/2006/Mexico
    Human/GII/InDRE3402_Ver/2006/Mexico
    Human/GII/InDRE3432_Tamps/2006/Mexico
    Human/GII/InDRE3437_Tamps/2006/Mexico
    I_35_MF_cl1
    I_35_MF_cl2
    idli_24
    idli_26
    idli_29
    idli_44
    idli_45
    IFC_YC12
    IIIL_D
    IK_MB1-800R
    IK_MB10-800R
    IK_MB11-800R
    IK_MB12-800R
    IK_MB13-800R
    IK_MB14-800R
    IK_MB15-800R
    IK_MB16-800R
    IK_MB2-800R
    IK_MB3-800R
    IK_MB4-800R
    IK_MB5-800R
    IK_MB6-800R
    IK_MB7-800R
    IK_MB8-800R
    IK_MB9-800R
    IM_14
    IM_15
    IM_4
    IM_4B
    IM_5
    IM_8A
    IM_9A
    IM_9B
    INBio_4054J
    INBio_4099AF
    INBio_4100K
    INBio_4507A
    INBio_4510E
    Ips_1
    Ips_2
    Ips_3
    Ips_4
    Ips_5
    Ips_6
    Ips_7
    IRQ_01
    IRQ_02
    IRQ_03
    IRQ_04
    IRQ_05
    IRQ_06
    IRQ_07
    IRQ_09
    IRQ_11
    IRQ_13
    IRQ_14
    IRQ_15
    IRQ_16
    IRQ_18
    IRQ_19
    IRQ_20
    IRQ_21
    IRQ_22
    IRQ_24
    IRQ_25
    IRQ_26
    IRQ_28
    IRQ_29
    IRQ_31
    IRQ_33
    IRQ_34
    IRQ_35
    IRQ_36
    IRQ_37
    IRQ_38
    IRQ_40
    IRQ_41
    IRQ_43
    IRQ_47
    IRQ_48
    IRQ_54
    IRQ_56
    IRQ_57
    IRQ_58
    IRQ_59
    IRQ_60
    IRQ_61
    IRQ_62
    Iso11_2
    ISW_15
    ISW_2
    JCVI_control_MDA_Ecoli_02
    JCVI_control_MDA_Saureus_10
    JL4_S
    JN18_A7_F*
    Jomon_CT89
    JPL_7
    K12_27
    K12_28b
    kmd_008
    kmd_043
    kmd_065
    kmd_070
    kmd_071
    kmd_072
    kmd_076
    kmd_078
    kmd_116
    kmd_123
    kmd_151
    kmd_152
    kmd_160
    kmd_217
    kmd_222
    kmd_278
    kmd_307
    kmd_323
    kmd_327
    kmd_332
    kmd_340
    kmd_360
    L3_Loihi
    LaBonte_alg10
    Lad_11K
    Lad_23K
    Liv16S_LC_L1
    LUC13011_P14
    M12R1_907R_F06
    M:02_AG
    M:A1_BA106
    M:A1_BA110
    M:A1_BA117
    M:A1_BA147
    M:A1_BA153
    M:A1_BA18
    M:A1_BA200
    M:A1_BA23
    M:A1_BA232
    M:A1_BA3
    M:A1_BA31
    M:A1_BA32
    M:A1_BA41
    M:A1_BA44
    M:A1_BA49
    M:A1_BA52
    M:A1_BA56
    M:A1_BA88
    M:A1_LBS024
    M:A3/U_MAU1876
    M:A3/U_MAU2044
    M:A3/U_MAU432
    M:A3_1223SC
    M:A3_125HPD
    M:A3_4769RTHALD
    M:A3_492HALD
    M:A3_K1245
    M:A3_K1339
    M:A3_L044
    M:A3_MAU159
    M:A3_MAU1838
    M:A3_MAU759
    M:A_06TG.HT032
    M:A_06TG.HT045
    M:A_06TG.HT055
    M:A_06TG.HT077
    M:A_06TG.HT085
    M:A_06TG.HT126
    M:A_06TG.HT152
    M:A_06TG.HT247
    M:A_15HALD
    M:A_171HPD
    M:A_2178/2012/KWT
    M:A_227HALD
    M:A_5HALD
    M:A_69HPD
    M:A_CHU12
    M:A_K1111
    M:A_K1113
    M:A_K1143
    M:A_K1200
    M:A_K1210
    M:A_K1212
    M:A_K1229
    M:A_K1245
    M:A_K1256
    M:A_K1259
    M:A_K1263
    M:A_K1290
    M:A_K1304
    M:A_K1305
    M:A_K1345
    M:A_K1355
    M:A_K1360
    M:A_K1363
    M:A_K1369
    M:A_K1394
    M:A_K2025
    M:A_K2032
    M:A_K2033
    M:A_KS049
    M:A_L019
    M:A_L044
    M:A_LBS024
    M:A_MP1294
    M:A_MP1392
    M:A_MP582
    M:A_MP640
    M:A_MP804
    M:A_MP812
    M:A_ORAV459
    M:A_U0434
    M:A_U1524
    M:A_U1803
    M:A_U1808
    M:A_U2226
    M:A_U3121
    M:B/A/CR14_BG_PT107
    M:B/CRF14_BG_PT176
    M:B/D_PT508
    M:B/URF_PT38
    M:B_00NOVAX03
    M:B_00NOVAX06
    M:B_00NOVAX10
    M:B_00NOVAX15
    M:B_00NOVAX20
    M:B_00NOVAX38
    M:B_00NOVAX40
    M:B_01NOVAX15
    M:B_02NOVAX30
    M:B_03NOVAX01
    M:B_03NOVAX02
    M:B_03NOVAX04
    M:B_03NOVAX06
    M:B_03NOVAX12
    M:B_03NOVAX13
    M:B_03NOVAX14
    M:B_03NOVAX15
    M:B_03NOVAX16
    M:B_03NOVAX19
    M:B_03NOVAX29
    M:B_03NOVAX40
    M:B_03NOVAX41
    M:B_04NOVAX01
    M:B_04NOVAX02
    M:B_04NOVAX12
    M:B_04NOVAX14
    M:B_04NOVAX15
    M:B_04NOVAX19
    M:B_04NOVAX20
    M:B_04NOVAX22
    M:B_04NOVAX26
    M:B_04NOVAX29
    M:B_04NOVAX36
    M:B_04NOVAX41
    M:B_04SN.MS019
    M:B_04SN.MS103
    M:B_04SN.MS482
    M:B_04SN.MS483
    M:B_04SN.MS605
    M:B_04SN.MS614
    M:B_04SN.MS619
    M:B_04SN.MS651
    M:B_04SN.MS802
    M:B_04SN.MS812
    M:B_04SN.MS830
    M:B_04SN.MS838
    M:B_04SN.MS868
    M:B_05NOVAX02
    M:B_05NOVAX23
    M:B_05NOVAX38
    M:B_06NOVAX03
    M:B_06NOVAX07
    M:B_06NOVAX10
    M:B_06NOVAX11
    M:B_06NOVAX12
    M:B_06NOVAX14
    M:B_06NOVAX19
    M:B_06NOVAX21
    M:B_06NOVAX28
    M:B_06NOVAX30
    M:B_06NOVAX32
    M:B_06NOVAX34
    M:B_06NOVAX37
    M:B_07CN.HN007
    M:B_07CN.HN009
    M:B_07CN.HN022
    M:B_07CN.HN023
    M:B_07CN.HN029
    M:B_07CN.HN030
    M:B_07CN.HN033
    M:B_07CN.HN063
    M:B_07CN.HN064
    M:B_07CN.HN079
    M:B_101Hald
    M:B_17HALD
    M:B_215HPD
    M:B_21831/2012/KWT
    M:B_21HALD
    M:B_486HALD
    M:B_513/2012/KW
    M:B_5827/2012/KWT
    M:B_7750/2012/KWT
    M:B_8830/2012/KWT
    M:B_89.6
    M:B_95NOVAX41
    M:B_96NOVAX01
    M:B_96NOVAX14
    M:B_96NOVAX23
    M:B_96NOVAX41
    M:B_9773/2012/KWT
    M:B_97NOVAX01
    M:B_97NOVAX02
    M:B_97NOVAX07
    M:B_97NOVAX13
    M:B_97NOVAX14
    M:B_97NOVAX26
    M:B_98NOVAX16
    M:B_98NOVAX19
    M:B_98NOVAX21
    M:B_98NOVAX28
    M:B_98NOVAX29
    M:B_98NOVAX32
    M:B_99NOVAX04
    M:B_99NOVAX19
    M:B_99NOVAX22
    M:B_99NOVAX37
    M:B_ARV2/SF2
    M:B_BA111
    M:B_BA205
    M:B_BM2010-1
    M:B_BM2010-2
    M:B_HXB2R
    M:B_K1043
    M:B_Lai
    M:B_MAU153
    M:B_MN
    M:B_MP051
    M:B_MP077
    M:B_PT073
    M:B_PT107
    M:B_PT114
    M:B_PT1147
    M:B_PT16
    M:B_PT176
    M:B_PT2
    M:B_PT24
    M:B_PT3013
    M:B_PT317
    M:B_PT38
    M:B_PT423
    M:B_PT471
    M:B_PT508
    M:B_PT6060
    M:B_PT695
    M:B_PT709
    M:B_PT85
    M:B_PT8731
    M:B_PT893
    M:B_PT919
    M:B_PT955
    M:B_PT983
    M:BD_472HALD
    M:BD_K1165
    M:C/U/C_MAU420-PROT
    M:C_03NOVAX25
    M:C_04NOVAX25
    M:C_04SN.MS002
    M:C_04SN.MS007
    M:C_04SN.MS010
    M:C_04SN.MS011
    M:C_04SN.MS015
    M:C_04SN.MS029
    M:C_04SN.MS034
    M:C_04SN.MS036
    M:C_04SN.MS245
    M:C_04SN.MS262
    M:C_04SN.MS281
    M:C_04SN.MS422
    M:C_04SN.MS440
    M:C_04SN.MS448
    M:C_04SN.MS456
    M:C_04SN.MS475
    M:C_04SN.MS477
    M:C_04SN.MS481
    M:C_04SN.MS487
    M:C_04SN.MS492
    M:C_04SN.MS522
    M:C_04SN.MS540
    M:C_04SN.MS700
    M:C_04SN.MS779
    M:C_04SN.MS816
    M:C_04SN.MS821
    M:C_04SN.MS835
    M:C_04SN.MS855
    M:C_04SN.MS883
    M:C_05NOVAX25
    M:C_07CN.HN003
    M:C_07CN.HN004
    M:C_10/93_KC005
    M:C_100/93_KC005
    M:C_101/93_KC005
    M:C_102HALD
    M:C_1032Hald-RT
    M:C_1083
    M:C_109/93_KC005
    M:C_11/93_KC005
    M:C_1119
    M:C_1186
    M:C_1189
    M:C_130/93_KC005
    M:C_131/93_KC005
    M:C_132/93_KC005
    M:C_138/93_KC005
    M:C_14/93_KC005
    M:C_141/93_KC005
    M:C_142/93_KC005
    M:C_143/93_KC005
    M:C_144/93_KC005
    M:C_145/93_KC005
    M:C_148/1997_KC005
    M:C_148/93_KC005
    M:C_14Fann
    M:C_15/93_KC005
    M:C_154/93_KC005
    M:C_154HALD
    M:C_16/93_KC005
    M:C_162/93_KC005
    M:C_164/93_KC005
    M:C_169/93_KC005
    M:C_176/93_KC005
    M:C_179/93_KC005
    M:C_181/93_KC005
    M:C_183/93_KC005
    M:C_19/93_KC005
    M:C_198/93_KC005
    M:C_224/1997_KC005
    M:C_23/93_KC005
    M:C_24/93_KC005
    M:C_25Fann
    M:C_260HALD
    M:C_2658HALD
    M:C_2909HALD
    M:C_2911HALD
    M:C_2931Hald
    M:C_2936HALD
    M:C_3/93_KC005
    M:C_3076HALD
    M:C_30801/2012/KWT
    M:C_32/93_KC005
    M:C_321HALD
    M:C_33/93_KC005
    M:C_34/93_KC005
    M:C_35/93_KC005
    M:C_36/93_KC005
    M:C_38/93_KC005
    M:C_39HALD
    M:C_40/93_KC005
    M:C_41/93_KC005
    M:C_4276/2011/KWT
    M:C_43/93_KC005
    M:C_44/93_KC005
    M:C_463HALD
    M:C_478HALD
    M:C_49/93_KC005
    M:C_51/93_KC005
    M:C_510HALD
    M:C_571/2012/KWT
    M:C_58/93_KC005
    M:C_59/93_KC005
    M:C_62/93_KC005
    M:C_64/93_KC005
    M:C_68/93_KC005
    M:C_6920/2011/KWT
    M:C_72/93_KC005
    M:C_77/1997_KC005
    M:C_79/93_KC005
    M:C_86/93_KC005
    M:C_8618/2012/KWT
    M:C_88/93_KC005
    M:C_9/93_KC005
    M:C_92/93_KC005
    M:C_92BR025
    M:C_94/93_KC005
    M:C_96/93_KC005
    M:C_96NOVAX25
    M:C_97/93_KC005
    M:C_99/93_KC005
    M:C_B1054
    M:C_B1085
    M:C_B1100
    M:C_BA12
    M:C_BA28
    M:C_ETH2220
    M:C_K1165
    M:C_K1335
    M:C_L065
    M:C_L160
    M:C_MAU331
    M:C_MAU420
    M:C_MP102
    M:C_MP116
    M:C_SF130
    M:C_SF431
    M:C_SF621
    M:C_U0126
    M:C_U0211
    M:C_U0221
    M:C_U0333
    M:C_U0427
    M:C_U0434
    M:C_U0435
    M:C_U0505
    M:C_U0636
    M:C_U0702
    M:C_U0705
    M:C_U0727
    M:C_U0735
    M:C_U0808
    M:C_U0812
    M:C_U0820
    M:C_U1622
    M:C_U1803
    M:C_U2132
    M:CFR02/01_04SN.MS291
    M:CRF01_03NOVAX31
    M:CRF01_04NOVAX31
    M:CRF01_97NOVAX31
    M:CRF01_AE
    M:CRF01_AE_10988/2012/KWT
    M:CRF01_AE_5717/2011/KWT
    M:CRF01_BA1
    M:CRF01_BA11
    M:CRF01_BA118
    M:CRF01_BA121
    M:CRF01_BA128
    M:CRF01_BA13
    M:CRF01_BA131
    M:CRF01_BA132
    M:CRF01_BA133
    M:CRF01_BA146
    M:CRF01_BA148
    M:CRF01_BA151
    M:CRF01_BA156
    M:CRF01_BA165
    M:CRF01_BA166
    M:CRF01_BA179
    M:CRF01_BA219
    M:CRF01_BA231
    M:CRF01_BA30
    M:CRF01_BA33
    M:CRF01_BA67
    M:CRF01_BA76
    M:CRF01_BA77
    M:CRF01_BA8
    M:CRF01_BA94
    M:CRF01_K1104
    M:CRF01_MP365
    M:CRF01_MP38
    M:CRF01_MP540
    M:CRF01_R_ADN_ge_R1
    M:CRF01_R_ADN_ge_R10
    M:CRF01_R_ADN_ge_R11
    M:CRF01_R_ADN_ge_R12
    M:CRF01_R_ADN_ge_R13
    M:CRF01_R_ADN_ge_R14
    M:CRF01_R_ADN_ge_R15
    M:CRF01_R_ADN_ge_R16
    M:CRF01_R_ADN_ge_R17
    M:CRF01_R_ADN_ge_R18
    M:CRF01_R_ADN_ge_R2
    M:CRF01_R_ADN_ge_R3
    M:CRF01_R_ADN_ge_R5
    M:CRF01_R_ADN_ge_R6
    M:CRF01_R_ADN_ge_R7
    M:CRF01_R_ADN_ge_R9
    M:CRF01_R_ARN_pl_R11
    M:CRF01_R_ARN_pl_R19
    M:CRF01_R_ARN_pl_R2
    M:CRF01_R_ARN_pl_R21
    M:CRF01_R_ARN_pl_R3
    M:CRF01_R_ARN_pl_R4
    M:CRF01_R_ARN_pl_R5
    M:CRF01_R_ARN_pl_R8
    M:CRF01_R_ARN_pl_R9
    M:CRF02-AG_MAU46
    M:CRF02/06/09_04SN.MS235
    M:CRF02/06_04SN.MS258
    M:CRF02_04SN.MS013
    M:CRF02_04SN.MS178
    M:CRF02_04SN.MS206
    M:CRF02_04SN.MS253
    M:CRF02_04SN.MS268
    M:CRF02_04SN.MS474
    M:CRF02_04SN.MS516
    M:CRF02_04SN.MS618
    M:CRF02_04SN.MS636
    M:CRF02_04SN.MS641
    M:CRF02_04SN.MS668
    M:CRF02_04SN.MS831
    M:CRF02_04SN.MS840
    M:CRF02_04SN.MS842
    M:CRF02_04SN.MS869
    M:CRF02_125HPD
    M:CRF02_163HALD
    M:CRF02_167HPD
    M:CRF02_16HALD
    M:CRF02_171HPD
    M:CRF02_18HALD
    M:CRF02_191HPD
    M:CRF02_214HPD
    M:CRF02_217HALD
    M:CRF02_227HALD
    M:CRF02_232HALD
    M:CRF02_233HALD
    M:CRF02_234HALD
    M:CRF02_265HPD
    M:CRF02_283HPD
    M:CRF02_29HALD
    M:CRF02_303HALD
    M:CRF02_304HALD
    M:CRF02_304HPD
    M:CRF02_307HPD
    M:CRF02_309HALD
    M:CRF02_311HALD
    M:CRF02_323HALD
    M:CRF02_324HALD
    M:CRF02_34HALD
    M:CRF02_383HALD
    M:CRF02_416HALD
    M:CRF02_418HALD
    M:CRF02_420HALD
    M:CRF02_422HALD
    M:CRF02_424HALD
    M:CRF02_426HALD
    M:CRF02_42HPD
    M:CRF02_455HALD
    M:CRF02_456HALD
    M:CRF02_459HALD
    M:CRF02_466HALD
    M:CRF02_469HALD
    M:CRF02_46HALD
    M:CRF02_471HALD
    M:CRF02_474HALD
    M:CRF02_477HALD
    M:CRF02_479HALD
    M:CRF02_481HALD
    M:CRF02_484HALD
    M:CRF02_492HALD
    M:CRF02_496HALD
    M:CRF02_49HALD
    M:CRF02_97HALD
    M:CRF02_A_ADN_ge_A11
    M:CRF02_A_ADN_ge_A12
    M:CRF02_A_ADN_ge_A13
    M:CRF02_A_ADN_ge_A14
    M:CRF02_A_ADN_ge_A15
    M:CRF02_A_ADN_ge_A16
    M:CRF02_A_ADN_ge_A17
    M:CRF02_A_ADN_ge_A18
    M:CRF02_A_ADN_ge_A19
    M:CRF02_A_ADN_ge_A1_2
    M:CRF02_A_ADN_ge_A20
    M:CRF02_A_ADN_ge_A2_2
    M:CRF02_A_ADN_ge_A3_1
    M:CRF02_A_ADN_ge_A3_2
    M:CRF02_A_ADN_ge_A4_1
    M:CRF02_A_ADN_ge_A4_2
    M:CRF02_A_ADN_ge_A5_1
    M:CRF02_A_ADN_ge_A8_1
    M:CRF02_A_ADN_ge_A8_2
    M:CRF02_A_ADN_ge_A9_1
    M:CRF02_A_ADN_ge_A9_2
    M:CRF02_A_ARN_ge_A1
    M:CRF02_A_ARN_ge_A10
    M:CRF02_A_ARN_ge_A11
    M:CRF02_A_ARN_ge_A12
    M:CRF02_A_ARN_ge_A13
    M:CRF02_A_ARN_ge_A16
    M:CRF02_A_ARN_ge_A17
    M:CRF02_A_ARN_ge_A18
    M:CRF02_A_ARN_ge_A19
    M:CRF02_A_ARN_ge_A3
    M:CRF02_A_ARN_ge_A6
    M:CRF02_A_ARN_ge_A7
    M:CRF02_A_ARN_pl_A10_1
    M:CRF02_A_ARN_pl_A10_2
    M:CRF02_A_ARN_pl_A12
    M:CRF02_A_ARN_pl_A16
    M:CRF02_A_ARN_pl_A18_1
    M:CRF02_A_ARN_pl_A18_2
    M:CRF02_A_ARN_pl_A2_1
    M:CRF02_A_ARN_pl_A2_2
    M:CRF02_A_ARN_pl_A3_1
    M:CRF02_A_ARN_pl_A3_2
    M:CRF02_A_ARN_pl_A4_1
    M:CRF02_A_ARN_pl_A6_1
    M:CRF02_A_ARN_pl_A6_2
    M:CRF02_A_ARN_pl_A8
    M:CRF02_AG
    M:CRF02_AG/A3/CRF02_AG_MAU216
    M:CRF02_AG/A3_8253Hald
    M:CRF02_AG/A3_MAU2140
    M:CRF02_AG/A3_MAU251
    M:CRF02_AG/URF_PT836
    M:CRF02_AG_00082500314
    M:CRF02_AG_00129346294
    M:CRF02_AG_00136644558
    M:CRF02_AG_00384364598
    M:CRF02_AG_00384760994
    M:CRF02_AG_00401150434
    M:CRF02_AG_00446856847
    M:CRF02_AG_00486811743
    M:CRF02_AG_01161893839
    M:CRF02_AG_01835871153
    M:CRF02_AG_03099751427
    M:CRF02_AG_03328130847
    M:CRF02_AG_04319750683
    M:CRF02_AG_04566394139
    M:CRF02_AG_04732291679
    M:CRF02_AG_05655055625
    M:CRF02_AG_07821754717
    M:CRF02_AG_08351618287
    M:CRF02_AG_08633910237
    M:CRF02_AG_08855259237
    M:CRF02_AG_08TG.C7003
    M:CRF02_AG_08TG.C7017
    M:CRF02_AG_08TG.C7023
    M:CRF02_AG_08TG.C7024
    M:CRF02_AG_08TG.C7028
    M:CRF02_AG_08TG.C7030
    M:CRF02_AG_08TG.E5013
    M:CRF02_AG_08TG.E5017
    M:CRF02_AG_08TG.E5025
    M:CRF02_AG_08TG.E5027
    M:CRF02_AG_08TG.E5032
    M:CRF02_AG_08TG.E5043
    M:CRF02_AG_08TG.E5044
    M:CRF02_AG_08TG.E5053
    M:CRF02_AG_08TG.E5068
    M:CRF02_AG_08TG.T3003
    M:CRF02_AG_08TG.T3008
    M:CRF02_AG_08TG.T3017
    M:CRF02_AG_08TG.T3029
    M:CRF02_AG_08TG.T3031
    M:CRF02_AG_08TG.T3043
    M:CRF02_AG_08TG.T3046
    M:CRF02_AG_08TG.T3049
    M:CRF02_AG_08TG.T3061
    M:CRF02_AG_08TG.T3069
    M:CRF02_AG_08TG.T3073
    M:CRF02_AG_09360633833
    M:CRF02_AG_09826618155
    M:CRF02_AG_1393HALD
    M:CRF02_AG_1837Hald
    M:CRF02_AG_20029892406
    M:CRF02_AG_20069834746
    M:CRF02_AG_20154094266
    M:CRF02_AG_20728228596
    M:CRF02_AG_2107Hald
    M:CRF02_AG_2340Hald
    M:CRF02_AG_2413Hald-PROT
    M:CRF02_AG_2413Hald-RT
    M:CRF02_AG_2613Hald
    M:CRF02_AG_2698Hald
    M:CRF02_AG_30111376596
    M:CRF02_AG_30220614916
    M:CRF02_AG_4039Hald
    M:CRF02_AG_4921HALD
    M:CRF02_AG_49HALD
    M:CRF02_AG_50067984846
    M:CRF02_AG_50314106694
    M:CRF02_AG_50936804498
    M:CRF02_AG_60159306954
    M:CRF02_AG_80394360234
    M:CRF02_AG_80880292104
    M:CRF02_AG_8253Hald
    M:CRF02_AG_90684486142
    M:CRF02_AG_929Hald
    M:CRF02_AG_MAU1086
    M:CRF02_AG_MAU1120
    M:CRF02_AG_MAU1138
    M:CRF02_AG_MAU1151
    M:CRF02_AG_MAU1247
    M:CRF02_AG_MAU1298
    M:CRF02_AG_MAU1339
    M:CRF02_AG_MAU1399
    M:CRF02_AG_MAU145
    M:CRF02_AG_MAU1567
    M:CRF02_AG_MAU161
    M:CRF02_AG_MAU1674
    M:CRF02_AG_MAU1743
    M:CRF02_AG_MAU1860
    M:CRF02_AG_MAU1911
    M:CRF02_AG_MAU2048
    M:CRF02_AG_MAU2059
    M:CRF02_AG_MAU2073
    M:CRF02_AG_MAU2115
    M:CRF02_AG_MAU2117
    M:CRF02_AG_MAU222
    M:CRF02_AG_MAU339
    M:CRF02_AG_MAU406
    M:CRF02_AG_MAU471
    M:CRF02_AG_MAU513
    M:CRF02_AG_MAU515
    M:CRF02_AG_MAU519
    M:CRF02_AG_MAU521
    M:CRF02_AG_MAU523
    M:CRF02_AG_MAU53
    M:CRF02_AG_MAU577
    M:CRF02_AG_MAU624
    M:CRF02_AG_MAU631
    M:CRF02_AG_MAU635
    M:CRF02_AG_MAU657
    M:CRF02_AG_MAU664
    M:CRF02_AG_MAU670
    M:CRF02_AG_MAU683
    M:CRF02_AG_MAU722
    M:CRF02_AG_MAU765
    M:CRF02_AG_MAU938
    M:CRF02_AG_MAU938-PROT
    M:CRF02_AG_PT4132
    M:CRF02_AG_PT4773
    M:CRF02_AG_PT727
    M:CRF02_AG_PT836
    M:CRF02_AG_PT9000
    M:CRF02_AG_PT988
    M:CRF02_B_ADN_ge_B1
    M:CRF02_B_ADN_ge_B10
    M:CRF02_B_ADN_ge_B11
    M:CRF02_B_ADN_ge_B12
    M:CRF02_B_ADN_ge_B13
    M:CRF02_B_ADN_ge_B15
    M:CRF02_B_ADN_ge_B16
    M:CRF02_B_ADN_ge_B17
    M:CRF02_B_ADN_ge_B18
    M:CRF02_B_ADN_ge_B19
    M:CRF02_B_ADN_ge_B20
    M:CRF02_B_ADN_ge_B21
    M:CRF02_B_ADN_ge_B24
    M:CRF02_B_ADN_ge_B25
    M:CRF02_B_ADN_ge_B3
    M:CRF02_B_ADN_ge_B4
    M:CRF02_B_ADN_ge_B5
    M:CRF02_B_ADN_ge_B6
    M:CRF02_B_ADN_ge_B8
    M:CRF02_B_ADN_ge_B9
    M:CRF02_B_ARN_ge_B1
    M:CRF02_B_ARN_ge_B10
    M:CRF02_B_ARN_ge_B11
    M:CRF02_B_ARN_ge_B12
    M:CRF02_B_ARN_ge_B13
    M:CRF02_B_ARN_ge_B15
    M:CRF02_B_ARN_ge_B16
    M:CRF02_B_ARN_ge_B17
    M:CRF02_B_ARN_ge_B18
    M:CRF02_B_ARN_ge_B19
    M:CRF02_B_ARN_ge_B2
    M:CRF02_B_ARN_ge_B3
    M:CRF02_B_ARN_ge_B4
    M:CRF02_B_ARN_ge_B5
    M:CRF02_B_ARN_ge_B6
    M:CRF02_B_ARN_ge_B7
    M:CRF02_B_ARN_ge_B8
    M:CRF02_B_ARN_ge_B9
    M:CRF02_B_ARN_pl_B10_1
    M:CRF02_B_ARN_pl_B10_2
    M:CRF02_B_ARN_pl_B11
    M:CRF02_B_ARN_pl_B14
    M:CRF02_B_ARN_pl_B15
    M:CRF02_B_ARN_pl_B16
    M:CRF02_B_ARN_pl_B1_2
    M:CRF02_B_ARN_pl_B2_1
    M:CRF02_B_ARN_pl_B2_2
    M:CRF02_B_ARN_pl_B3_1
    M:CRF02_B_ARN_pl_B3_2
    M:CRF02_B_ARN_pl_B4_1
    M:CRF02_B_ARN_pl_B4_2
    M:CRF02_B_ARN_pl_B5_1
    M:CRF02_B_ARN_pl_B6_1
    M:CRF02_B_ARN_pl_B6_2
    M:CRF02_B_ARN_pl_B7_2
    M:CRF02_B_ARN_pl_B9_1
    M:CRF02_B_ARN_pl_B9_2
    M:CRF02_BA190
    M:CRF02_BA213
    M:CRF02_BA215
    M:CRF02_BA244
    M:CRF02_BA26
    M:CRF02_BA35
    M:CRF02_BA53
    M:CRF02_BA58
    M:CRF02_BA59
    M:CRF02_BA60
    M:CRF02_H_ARN_ge_H10
    M:CRF02_H_ARN_ge_H11
    M:CRF02_K1086
    M:CRF02_K1103
    M:CRF02_K1110
    M:CRF02_K1114
    M:CRF02_K1118
    M:CRF02_K1129
    M:CRF02_K1131
    M:CRF02_K1163
    M:CRF02_K1166
    M:CRF02_K1170
    M:CRF02_K1171
    M:CRF02_K1175
    M:CRF02_K1178
    M:CRF02_K1194
    M:CRF02_K1227
    M:CRF02_K1230
    M:CRF02_K1243
    M:CRF02_K1263
    M:CRF02_K1268
    M:CRF02_K1281
    M:CRF02_K1282
    M:CRF02_K1283
    M:CRF02_K1292
    M:CRF02_K1300
    M:CRF02_K1303
    M:CRF02_K1306
    M:CRF02_K1311
    M:CRF02_K1316
    M:CRF02_K1318
    M:CRF02_K1326
    M:CRF02_K1331
    M:CRF02_K1333
    M:CRF02_K1343
    M:CRF02_K1344
    M:CRF02_K1370
    M:CRF02_K1371
    M:CRF02_K1380
    M:CRF02_K1425
    M:CRF02_K2002
    M:CRF02_K2005
    M:CRF02_K2007
    M:CRF02_K2023
    M:CRF02_K2033
    M:CRF02_K2035
    M:CRF02_K2053
    M:CRF02_K2055
    M:CRF02_K_ADN_ge_K10_1
    M:CRF02_K_ADN_ge_K10_2
    M:CRF02_K_ADN_ge_K11
    M:CRF02_K_ADN_ge_K12
    M:CRF02_K_ADN_ge_K13
    M:CRF02_K_ADN_ge_K16
    M:CRF02_K_ADN_ge_K18
    M:CRF02_K_ADN_ge_K19
    M:CRF02_K_ADN_ge_K1_1
    M:CRF02_K_ADN_ge_K1_2
    M:CRF02_K_ADN_ge_K20
    M:CRF02_K_ADN_ge_K2_1
    M:CRF02_K_ADN_ge_K2_2
    M:CRF02_K_ADN_ge_K3_2
    M:CRF02_K_ADN_ge_K4_1
    M:CRF02_K_ADN_ge_K4_2
    M:CRF02_K_ADN_ge_K5_1
    M:CRF02_K_ADN_ge_K5_2
    M:CRF02_K_ADN_ge_K6_1
    M:CRF02_K_ADN_ge_K6_2
    M:CRF02_K_ADN_ge_K7_2
    M:CRF02_K_ADN_ge_K9_1
    M:CRF02_K_ADN_ge_K9_2
    M:CRF02_K_ARN_ge_K1
    M:CRF02_K_ARN_ge_K10
    M:CRF02_K_ARN_ge_K12
    M:CRF02_K_ARN_ge_K13
    M:CRF02_K_ARN_ge_K14
    M:CRF02_K_ARN_ge_K15
    M:CRF02_K_ARN_ge_K16
    M:CRF02_K_ARN_ge_K17
    M:CRF02_K_ARN_ge_K19
    M:CRF02_K_ARN_ge_K2
    M:CRF02_K_ARN_ge_K3
    M:CRF02_K_ARN_ge_K4
    M:CRF02_K_ARN_ge_K7
    M:CRF02_K_ARN_ge_K9
    M:CRF02_K_ARN_pl_K11_1
    M:CRF02_K_ARN_pl_K12
    M:CRF02_K_ARN_pl_K13
    M:CRF02_K_ARN_pl_K17
    M:CRF02_K_ARN_pl_K18
    M:CRF02_K_ARN_pl_K19
    M:CRF02_K_ARN_pl_K1_1
    M:CRF02_K_ARN_pl_K1_2
    M:CRF02_K_ARN_pl_K2
    M:CRF02_K_ARN_pl_K20
    M:CRF02_K_ARN_pl_K5_1
    M:CRF02_K_ARN_pl_K5_2
    M:CRF02_K_ARN_pl_K6
    M:CRF02_K_ARN_pl_K7_1
    M:CRF02_K_ARN_pl_K7_2
    M:CRF02_K_ARN_pl_K8_1
    M:CRF02_K_ARN_pl_K8_2
    M:CRF02_K_ARN_pl_K9
    M:CRF02_L001
    M:CRF02_L003
    M:CRF02_L006
    M:CRF02_L009
    M:CRF02_L013
    M:CRF02_L014
    M:CRF02_L019
    M:CRF02_L035
    M:CRF02_L040
    M:CRF02_L056
    M:CRF02_L057
    M:CRF02_L074
    M:CRF02_L080
    M:CRF02_L084
    M:CRF02_L086
    M:CRF02_L091
    M:CRF02_L117
    M:CRF02_MP1071
    M:CRF02_MP1075
    M:CRF02_MP1277
    M:CRF02_MP1378
    M:CRF02_MP1539
    M:CRF02_MP1540
    M:CRF02_MP540
    M:CRF02_MP578
    M:CRF02_MP642
    M:CRF02_MP802
    M:CRF02_MP803
    M:CRF02_MP804
    M:CRF02_MP812
    M:CRF02_MP815
    M:CRF02_ORAV227
    M:CRF02_ORAV455
    M:CRF02_ORAV461
    M:CRF02_ORAV465
    M:CRF03_H_ARN_ge_H2
    M:CRF04_H_ARN_ge_H5
    M:CRF04_K2049
    M:CRF05_H_ARN_ge_H3
    M:CRF05_LBTB018
    M:CRF06_1223SC
    M:CRF06_127HPD
    M:CRF06_cpx
    M:CRF06_cpx/CRF02_AG_2401Hald
    M:CRF06_cpx_08TG.E5016
    M:CRF06_cpx_08TG.E5020
    M:CRF06_cpx_08TG.E5059
    M:CRF06_cpx_08TG.E5061
    M:CRF06_cpx_08TG.T3034
    M:CRF06_cpx_08TG.T3042
    M:CRF06_cpx_2401Hald
    M:CRF06_cpx_MAU1213
    M:CRF06_cpx_MAU1554
    M:CRF06_cpx_MAU25
    M:CRF06_cpx_MAU284
    M:CRF06_cpx_MAU60
    M:CRF06_cpx_MAU60-PROT
    M:CRF06_cpx_MAU922
    M:CRF06_H_ARN_ge_H16
    M:CRF06_K1211
    M:CRF06_L021
    M:CRF06_L058
    M:CRF06_L136
    M:CRF06_MP1075
    M:CRF06_ORAV207
    M:CRF06_ORAV208
    M:CRF06_ORAV215
    M:CRF06_ORAV216
    M:CRF06_ORAV217
    M:CRF06_ORAV224
    M:CRF06_ORAV227
    M:CRF06_ORAV405
    M:CRF06_ORAV436
    M:CRF06_ORAV452
    M:CRF06_ORAV453
    M:CRF06_ORAV455
    M:CRF06_ORAV456
    M:CRF06_ORAV459
    M:CRF06_ORAV460
    M:CRF06_ORAV462
    M:CRF06_U2929
    M:CRF07_BC
    M:CRF07_H_ARN_ge_H8
    M:CRF08_BC
    M:CRF08_H_ARN_ge_H9
    M:CRF09_04SN.MS445
    M:CRF09_04SN.MS463
    M:CRF09_04SN.MS823
    M:CRF09_16HALD
    M:CRF09_cpx
    M:CRF09_H_ARN_ge_H17
    M:CRF09_ORAV224
    M:CRF10_H_ARN_ge_H15
    M:CRF10_U1916
    M:CRF10_U1929
    M:CRF11_413HALD
    M:CRF11_BA109
    M:CRF11_BA116
    M:CRF11_BA123
    M:CRF11_BA124
    M:CRF11_BA126
    M:CRF11_BA137
    M:CRF11_BA14
    M:CRF11_BA149
    M:CRF11_BA150
    M:CRF11_BA159
    M:CRF11_BA173
    M:CRF11_BA175
    M:CRF11_BA177
    M:CRF11_BA184
    M:CRF11_BA185
    M:CRF11_BA189
    M:CRF11_BA192
    M:CRF11_BA195
    M:CRF11_BA197
    M:CRF11_BA199
    M:CRF11_BA20
    M:CRF11_BA220
    M:CRF11_BA224
    M:CRF11_BA225
    M:CRF11_BA226
    M:CRF11_BA227
    M:CRF11_BA230
    M:CRF11_BA237
    M:CRF11_BA238
    M:CRF11_BA242
    M:CRF11_BA25
    M:CRF11_BA29
    M:CRF11_BA34
    M:CRF11_BA36
    M:CRF11_BA37
    M:CRF11_BA48
    M:CRF11_BA50
    M:CRF11_BA51
    M:CRF11_BA68
    M:CRF11_BA7
    M:CRF11_BA72
    M:CRF11_BA78
    M:CRF11_BA80
    M:CRF11_BA87
    M:CRF11_BA9
    M:CRF11_C_ARN_ge_C1
    M:CRF11_C_ARN_ge_C11
    M:CRF11_C_ARN_ge_C12
    M:CRF11_C_ARN_ge_C13
    M:CRF11_C_ARN_ge_C17
    M:CRF11_C_ARN_ge_C2
    M:CRF11_C_ARN_ge_C4
    M:CRF11_C_ARN_ge_C5
    M:CRF11_C_ARN_ge_C6
    M:CRF11_C_ARN_ge_C7
    M:CRF11_C_ARN_ge_C8
    M:CRF11_C_ARN_ge_C9
    M:CRF11_C_ARN_pl_C1
    M:CRF11_C_ARN_pl_C11
    M:CRF11_C_ARN_pl_C12
    M:CRF11_C_ARN_pl_C13
    M:CRF11_C_ARN_pl_C14
    M:CRF11_C_ARN_pl_C16
    M:CRF11_C_ARN_pl_C17
    M:CRF11_C_ARN_pl_C5
    M:CRF11_C_ARN_pl_C6
    M:CRF11_C_ARN_pl_C7
    M:CRF11_C_ARN_pl_C8
    M:CRF11_cpx_MAU963
    M:CRF11_E_ADN_ge_E10
    M:CRF11_E_ADN_ge_E12
    M:CRF11_E_ADN_ge_E14
    M:CRF11_E_ADN_ge_E18
    M:CRF11_E_ADN_ge_E1_1
    M:CRF11_E_ADN_ge_E1_2
    M:CRF11_E_ADN_ge_E1_3
    M:CRF11_E_ADN_ge_E2_1
    M:CRF11_E_ADN_ge_E2_2
    M:CRF11_E_ADN_ge_E2_3
    M:CRF11_E_ADN_ge_E3_1
    M:CRF11_E_ADN_ge_E3_2
    M:CRF11_E_ADN_ge_E4_1
    M:CRF11_E_ADN_ge_E4_3
    M:CRF11_E_ADN_ge_E5_1
    M:CRF11_E_ADN_ge_E5_2
    M:CRF11_E_ADN_ge_E6_1
    M:CRF11_E_ADN_ge_E6_2
    M:CRF11_E_ADN_ge_E6_3
    M:CRF11_E_ADN_ge_E7_1
    M:CRF11_E_ADN_ge_E7_2
    M:CRF11_E_ADN_ge_E8_1
    M:CRF11_E_ADN_ge_E8_2
    M:CRF11_E_ADN_ge_E9_1
    M:CRF11_E_ADN_ge_E9_2
    M:CRF11_E_ARN_ge_E1
    M:CRF11_E_ARN_ge_E10
    M:CRF11_E_ARN_ge_E13
    M:CRF11_E_ARN_ge_E15
    M:CRF11_E_ARN_ge_E16
    M:CRF11_E_ARN_ge_E2
    M:CRF11_E_ARN_ge_E20
    M:CRF11_E_ARN_ge_E3
    M:CRF11_E_ARN_ge_E7
    M:CRF11_E_ARN_ge_E8
    M:CRF11_E_ARN_pl_E1
    M:CRF11_E_ARN_pl_E10_1
    M:CRF11_E_ARN_pl_E10_2
    M:CRF11_E_ARN_pl_E11
    M:CRF11_E_ARN_pl_E12
    M:CRF11_E_ARN_pl_E14
    M:CRF11_E_ARN_pl_E15
    M:CRF11_E_ARN_pl_E16
    M:CRF11_E_ARN_pl_E17
    M:CRF11_E_ARN_pl_E19
    M:CRF11_E_ARN_pl_E2_2
    M:CRF11_E_ARN_pl_E3_1
    M:CRF11_E_ARN_pl_E3_2
    M:CRF11_E_ARN_pl_E4_1
    M:CRF11_E_ARN_pl_E4_2
    M:CRF11_E_ARN_pl_E5_1
    M:CRF11_E_ARN_pl_E6_1
    M:CRF11_E_ARN_pl_E6_2
    M:CRF11_E_ARN_pl_E7_1
    M:CRF11_E_ARN_pl_E7_2
    M:CRF11_E_ARN_pl_E8_1
    M:CRF11_E_ARN_pl_E8_2
    M:CRF11_E_ARN_pl_E8_3
    M:CRF11_E_ARN_pl_E9_1
    M:CRF11_E_ARN_pl_E9_2
    M:CRF11_F_ARN_ge_F1
    M:CRF11_F_ARN_ge_F13
    M:CRF11_F_ARN_ge_F14
    M:CRF11_F_ARN_ge_F16
    M:CRF11_F_ARN_ge_F19
    M:CRF11_F_ARN_ge_F2
    M:CRF11_F_ARN_ge_F4
    M:CRF11_F_ARN_pl_F1
    M:CRF11_F_ARN_pl_F12
    M:CRF11_F_ARN_pl_F13
    M:CRF11_F_ARN_pl_F14
    M:CRF11_F_ARN_pl_F15
    M:CRF11_F_ARN_pl_F16
    M:CRF11_F_ARN_pl_F3_1
    M:CRF11_F_ARN_pl_F4_1
    M:CRF11_F_ARN_pl_F4_2
    M:CRF11_F_ARN_pl_F5
    M:CRF11_F_ARN_pl_F6
    M:CRF11_F_ARN_pl_F7_1
    M:CRF11_F_ARN_pl_F9
    M:CRF11_G_ADN_ge_G1
    M:CRF11_G_ADN_ge_G10
    M:CRF11_G_ADN_ge_G11
    M:CRF11_G_ADN_ge_G12
    M:CRF11_G_ADN_ge_G13
    M:CRF11_G_ADN_ge_G14
    M:CRF11_G_ADN_ge_G15
    M:CRF11_G_ADN_ge_G16
    M:CRF11_G_ADN_ge_G17
    M:CRF11_G_ADN_ge_G18
    M:CRF11_G_ADN_ge_G19
    M:CRF11_G_ADN_ge_G2
    M:CRF11_G_ADN_ge_G20
    M:CRF11_G_ADN_ge_G3
    M:CRF11_G_ADN_ge_G4
    M:CRF11_G_ADN_ge_G5
    M:CRF11_G_ADN_ge_G7
    M:CRF11_G_ADN_ge_G8
    M:CRF11_G_ADN_ge_G9
    M:CRF11_G_ARN_ge_G10
    M:CRF11_G_ARN_ge_G11
    M:CRF11_G_ARN_ge_G14
    M:CRF11_G_ARN_ge_G15
    M:CRF11_G_ARN_ge_G16
    M:CRF11_G_ARN_ge_G17
    M:CRF11_G_ARN_ge_G18
    M:CRF11_G_ARN_ge_G3
    M:CRF11_G_ARN_ge_G4
    M:CRF11_G_ARN_ge_G5
    M:CRF11_G_ARN_ge_G6
    M:CRF11_G_ARN_ge_G7
    M:CRF11_G_ARN_ge_G9
    M:CRF11_G_ARN_pl_G1
    M:CRF11_G_ARN_pl_G10
    M:CRF11_G_ARN_pl_G11
    M:CRF11_G_ARN_pl_G12
    M:CRF11_G_ARN_pl_G13
    M:CRF11_G_ARN_pl_G14
    M:CRF11_G_ARN_pl_G15
    M:CRF11_G_ARN_pl_G16
    M:CRF11_G_ARN_pl_G18
    M:CRF11_G_ARN_pl_G2
    M:CRF11_G_ARN_pl_G3
    M:CRF11_G_ARN_pl_G4
    M:CRF11_G_ARN_pl_G7
    M:CRF11_G_ARN_pl_G8
    M:CRF11_G_ARN_pl_G9
    M:CRF11_H_ARN_ge_H4
    M:CRF11_K1140
    M:CRF11_K1186
    M:CRF11_K1213
    M:CRF11_K1244
    M:CRF11_K1288
    M:CRF11_K1356
    M:CRF11_KS049
    M:CRF11_L018
    M:CRF11_L_ADN_ge_L10_1
    M:CRF11_L_ADN_ge_L10_2
    M:CRF11_L_ADN_ge_L1_1
    M:CRF11_L_ADN_ge_L1_3
    M:CRF11_L_ADN_ge_L2_2
    M:CRF11_L_ADN_ge_L3_1
    M:CRF11_L_ADN_ge_L3_2
    M:CRF11_L_ADN_ge_L3_3
    M:CRF11_L_ADN_ge_L4_1
    M:CRF11_L_ADN_ge_L4_2
    M:CRF11_L_ADN_ge_L5_2
    M:CRF11_L_ADN_ge_L5_3
    M:CRF11_L_ADN_ge_L6_1
    M:CRF11_L_ADN_ge_L6_2
    M:CRF11_L_ADN_ge_L6_3
    M:CRF11_L_ADN_ge_L7_1
    M:CRF11_L_ADN_ge_L7_2
    M:CRF11_L_ADN_ge_L7_3
    M:CRF11_L_ADN_ge_L8_2
    M:CRF11_L_ADN_ge_L8_3
    M:CRF11_L_ADN_ge_L9_1
    M:CRF11_L_ADN_ge_L9_2
    M:CRF11_L_ADN_ge_L9_3
    M:CRF11_L_ARN_ge_L1
    M:CRF11_L_ARN_ge_L11
    M:CRF11_L_ARN_ge_L12
    M:CRF11_L_ARN_ge_L13
    M:CRF11_L_ARN_ge_L14
    M:CRF11_L_ARN_ge_L15
    M:CRF11_L_ARN_ge_L16
    M:CRF11_L_ARN_ge_L17
    M:CRF11_L_ARN_ge_L18
    M:CRF11_L_ARN_ge_L19
    M:CRF11_L_ARN_ge_L2
    M:CRF11_L_ARN_ge_L3
    M:CRF11_L_ARN_ge_L4
    M:CRF11_L_ARN_ge_L5
    M:CRF11_L_ARN_ge_L6
    M:CRF11_L_ARN_ge_L7
    M:CRF11_L_ARN_ge_L8
    M:CRF11_L_ARN_pl_L10_2
    M:CRF11_L_ARN_pl_L11
    M:CRF11_L_ARN_pl_L17
    M:CRF11_L_ARN_pl_L19
    M:CRF11_L_ARN_pl_L1_1
    M:CRF11_L_ARN_pl_L1_2
    M:CRF11_L_ARN_pl_L20
    M:CRF11_L_ARN_pl_L2_2
    M:CRF11_L_ARN_pl_L3_1
    M:CRF11_L_ARN_pl_L3_2
    M:CRF11_L_ARN_pl_L4
    M:CRF11_L_ARN_pl_L6
    M:CRF11_L_ARN_pl_L7_1
    M:CRF11_L_ARN_pl_L7_2
    M:CRF11_MP703
    M:CRF11_MP809
    M:CRF11_MP818
    M:CRF12_H_ARN_ge_H14
    M:CRF13_BA182
    M:CRF13_BA40
    M:CRF13_BA64
    M:CRF13_cpx_2829Hald
    M:CRF13_H_ARN_ge_H1
    M:CRF13_K1209
    M:CRF13_K1212
    M:CRF13_MP1034
    M:CRF13_MP816
    M:CRF13_N_ADN_ge_N10_1
    M:CRF13_N_ADN_ge_N10_2
    M:CRF13_N_ADN_ge_N11
    M:CRF13_N_ADN_ge_N12
    M:CRF13_N_ADN_ge_N13_1
    M:CRF13_N_ADN_ge_N13_2
    M:CRF13_N_ADN_ge_N14
    M:CRF13_N_ADN_ge_N15
    M:CRF13_N_ADN_ge_N16
    M:CRF13_N_ADN_ge_N17
    M:CRF13_N_ADN_ge_N18
    M:CRF13_N_ADN_ge_N19
    M:CRF13_N_ADN_ge_N1_2
    M:CRF13_N_ADN_ge_N20
    M:CRF13_N_ADN_ge_N2_1
    M:CRF13_N_ADN_ge_N3_2
    M:CRF13_N_ADN_ge_N4_1
    M:CRF13_N_ADN_ge_N4_2
    M:CRF13_N_ADN_ge_N5_1
    M:CRF13_N_ADN_ge_N5_2
    M:CRF13_N_ADN_ge_N6_1
    M:CRF13_N_ADN_ge_N6_2
    M:CRF13_N_ADN_ge_N7_1
    M:CRF13_N_ADN_ge_N7_2
    M:CRF13_N_ADN_ge_N8_1
    M:CRF13_N_ADN_ge_N8_2
    M:CRF13_N_ADN_ge_N9_1
    M:CRF13_N_ADN_ge_N9_2
    M:CRF13_N_ARN_ge_N1
    M:CRF13_N_ARN_ge_N10
    M:CRF13_N_ARN_ge_N11
    M:CRF13_N_ARN_ge_N12
    M:CRF13_N_ARN_ge_N13
    M:CRF13_N_ARN_ge_N14
    M:CRF13_N_ARN_ge_N15
    M:CRF13_N_ARN_ge_N16
    M:CRF13_N_ARN_ge_N17
    M:CRF13_N_ARN_ge_N18
    M:CRF13_N_ARN_ge_N19
    M:CRF13_N_ARN_ge_N1_2
    M:CRF13_N_ARN_ge_N2
    M:CRF13_N_ARN_ge_N2_2
    M:CRF13_N_ARN_ge_N3
    M:CRF13_N_ARN_ge_N3_1
    M:CRF13_N_ARN_ge_N4
    M:CRF13_N_ARN_ge_N5
    M:CRF13_N_ARN_ge_N6
    M:CRF13_N_ARN_ge_N7
    M:CRF13_N_ARN_ge_N8
    M:CRF13_N_ARN_ge_N9
    M:CRF13_N_ARN_pl_N11
    M:CRF13_N_ARN_pl_N13
    M:CRF13_N_ARN_pl_N14
    M:CRF13_N_ARN_pl_N15
    M:CRF13_N_ARN_pl_N19
    M:CRF13_N_ARN_pl_N1_2
    M:CRF13_N_ARN_pl_N20
    M:CRF13_N_ARN_pl_N2_2
    M:CRF13_N_ARN_pl_N3_2
    M:CRF13_N_ARN_pl_N4_2
    M:CRF13_N_ARN_pl_N6_1
    M:CRF13_N_ARN_pl_N7
    M:CRF13_N_ARN_pl_N8_1
    M:CRF13_N_ARN_pl_N8_2
    M:CRF14_H_ARN_pl_H2
    M:CRF15_BA162
    M:CRF15_BA219
    M:CRF15_BA49
    M:CRF15_H_ARN_pl_H9
    M:CRF16_H_ARN_pl_H4_2
    M:CRF17_H_ARN_pl_H5_1
    M:CRF18_H_ARN_pl_H4_1
    M:CRF18_K1111
    M:CRF18_MP1294
    M:CRF18_MP1335
    M:CRF19_H_ARN_pl_H6_2
    M:CRF20_H_ARN_pl_H8
    M:CRF21_H_ARN_pl_H13
    M:CRF22_01A1
    M:CRF22_316HALD
    M:CRF22_413HALD
    M:CRF22_H_ARN_pl_H6_1
    M:CRF22_K1116
    M:CRF22_K1135
    M:CRF22_K1161
    M:CRF22_K1166
    M:CRF22_K1179
    M:CRF22_K1182
    M:CRF22_K1183
    M:CRF22_K1214
    M:CRF22_K1223
    M:CRF22_K1274
    M:CRF22_K1301
    M:CRF22_K1314
    M:CRF22_K1369
    M:CRF22_K2006
    M:CRF22_K2030
    M:CRF22_K2031
    M:CRF22_K2049
    M:CRF22_K2051
    M:CRF22_MP1036
    M:CRF22_MP1073
    M:CRF22_MP1237
    M:CRF22_MP512
    M:CRF22_MP811
    M:CRF22_U1617
    M:CRF22_U2929
    M:CRF23_H_ARN_pl_H7_2
    M:CRF24_H_ARN_pl_H7_1
    M:CRF24_K1254
    M:CRF24_MP1416
    M:CRF25_10HALD
    M:CRF25_H_ARN_pl_H15
    M:CRF25_K1123
    M:CRF26_H_ARN_pl_H14
    M:CRF37_cpx/U_MAU969
    M:CRF37_K1163
    M:CRF43_K1348
    M:CRF43_K1349
    M:CRF45_316HALD
    M:CRF45_413HALD
    M:CRF45_cpx/U_1523Hald
    M:CRF45_K2023
    M:CRF49_cpx
    M:CRF_43
    M:D/A1_MAU2002
    M:D/B_PT317
    M:D_03NOVAX33
    M:D_305HALD
    M:D_3259Hald
    M:D_472HALD
    M:D_97NOVAX33
    M:D_AN010
    M:D_AN019
    M:D_AN034
    M:D_AN113
    M:D_AN174
    M:D_AN175
    M:D_AN176
    M:D_AN177
    M:D_ARV32
    M:D_BA160
    M:D_BA66
    M:D_BA69
    M:D_BA85
    M:D_BR014
    M:D_BR015
    M:D_BR019
    M:D_CA035
    M:D_CA048
    M:D_CA067
    M:D_CA080
    M:D_CHU07
    M:D_CHU19
    M:D_CHU27
    M:D_CHU39
    M:D_CHU53
    M:D_CP03
    M:D_K1176
    M:D_K1185
    M:D_K1186
    M:D_K1192
    M:D_K1291
    M:D_K1299
    M:D_K1300
    M:D_K1371
    M:D_K1417
    M:D_K2055
    M:D_L016
    M:D_MBFE252
    M:D_MP1017
    M:D_NA113
    M:D_NA125
    M:D_NA126
    M:D_NA127
    M:D_NA128
    M:D_NA129
    M:D_NA130
    M:D_NA131
    M:D_NA132
    M:D_NA133
    M:D_NA134
    M:D_NA135
    M:D_PO011
    M:D_PO012
    M:D_PO022
    M:D_PT317
    M:D_RE056
    M:D_RE083
    M:D_RE142
    M:D_RE207
    M:D_RE236
    M:D_RE237
    M:D_RE238
    M:D_RO001
    M:D_RO002
    M:D_RO003
    M:D_RO004
    M:D_RO005
    M:D_RO006
    M:D_U0106
    M:D_U1916
    M:F1_93BR020
    M:F1_BA216
    M:F1_BA93
    M:F1_BA97
    M:F1_KBS154
    M:F1_MP1035
    M:F1_MP1436
    M:F1_MP411
    M:F1_MP577
    M:F1_PT658
    M:F1_VI850
    M:F2_K1125
    M:F2_K1130
    M:F2_K1135
    M:F2_K1191
    M:F2_K1240
    M:F2_K1308
    M:F2_K1346
    M:F2_K1361
    M:F2_K2001
    M:F2_K2006
    M:F2_K2009
    M:F2_MP1017
    M:F2_MP1031
    M:F2_MP1035
    M:F2_MP1236
    M:F2_MP1367
    M:F2_MP255C
    M:F2_MP257C
    M:F2_MP576
    M:F2_MP577
    M:F_CHU51
    M:G/CRF02_04SN.MS642
    M:G_04SN.MS209
    M:G_04SN.MS210
    M:G_04SN.MS279
    M:G_04SN.MS297
    M:G_04SN.MS486
    M:G_06TG.HT023
    M:G_06TG.HT049
    M:G_06TG.HT056
    M:G_06TG.HT065
    M:G_06TG.HT068
    M:G_06TG.HT072
    M:G_06TG.HT081
    M:G_06TG.HT082
    M:G_06TG.HT086
    M:G_06TG.HT087
    M:G_06TG.HT116
    M:G_06TG.HT139
    M:G_06TG.HT149
    M:G_08TG.E5011
    M:G_92NG083
    M:G_ARV93
    M:G_BA125
    M:G_BA134
    M:G_BA152
    M:G_BA157
    M:G_BA172
    M:G_BA202
    M:G_BA233
    M:G_BA234
    M:G_BA240
    M:G_CHU06
    M:G_CHU11
    M:G_CHU16
    M:G_CHU38
    M:G_CHU46
    M:G_CHU56
    M:G_CP06
    M:G_GB03
    M:G_K1167
    M:G_K1184
    M:G_K1243
    M:G_K1247
    M:G_K1254
    M:G_K1256
    M:G_K1291
    M:G_K1293
    M:G_K1295
    M:G_K1316
    M:G_K2005
    M:G_L136
    M:G_MP1033
    M:G_MP1367
    M:G_MP1369
    M:G_MP1372
    M:G_MP1397
    M:G_MP1416
    M:G_MP1417
    M:G_MP694
    M:G_MP801
    M:G_MP962
    M:G_PT114
    M:G_PT13
    M:G_PT1544
    M:G_PT1631
    M:G_PT176
    M:G_PT227
    M:G_PT251
    M:G_PT271
    M:G_PT3
    M:G_PT4132
    M:G_PT471
    M:G_PT4773
    M:G_PT482
    M:G_PT5236
    M:G_PT6060
    M:G_PT658
    M:G_PT709
    M:G_PT7139
    M:G_PT72
    M:G_PT727
    M:G_PT806
    M:G_PT836
    M:G_PT85
    M:G_PT9000
    M:G_PT91
    M:G_PT9108
    M:G_PT919
    M:G_PT9456
    M:G_PT955
    M:G_PT98
    M:G_PT988
    M:G_PTR251
    M:G_SE6165
    M:G_U1720
    M:H_316HALD
    M:H_90CF056
    M:H_BA16
    M:H_BA201
    M:H_BA208
    M:H_CP07
    M:H_KTB238
    M:H_MAU757
    M:H_MP964
    M:H_U3121
    M:H_VI991
    M:J_CHU29
    M:J_CP14
    M:J_KTB238
    M:J_SE9173
    M:J_SE9280
    M:K_96CM-MP535
    M:K_97ZR-EQTB11
    M:K_U2226
    M:RecCpx_D_ADN_ge_D1
    M:RecCpx_D_ADN_ge_D10
    M:RecCpx_D_ADN_ge_D11
    M:RecCpx_D_ADN_ge_D12
    M:RecCpx_D_ADN_ge_D13
    M:RecCpx_D_ADN_ge_D14
    M:RecCpx_D_ADN_ge_D15
    M:RecCpx_D_ADN_ge_D16
    M:RecCpx_D_ADN_ge_D18
    M:RecCpx_D_ADN_ge_D19
    M:RecCpx_D_ADN_ge_D2
    M:RecCpx_D_ADN_ge_D3
    M:RecCpx_D_ADN_ge_D4
    M:RecCpx_D_ADN_ge_D5
    M:RecCpx_D_ADN_ge_D6
    M:RecCpx_D_ADN_ge_D7
    M:RecCpx_D_ADN_ge_D8
    M:RecCpx_D_ADN_ge_D9
    M:RecCpx_D_ARN_ge_D1
    M:RecCpx_D_ARN_ge_D10
    M:RecCpx_D_ARN_ge_D11
    M:RecCpx_D_ARN_ge_D12
    M:RecCpx_D_ARN_ge_D13
    M:RecCpx_D_ARN_ge_D14
    M:RecCpx_D_ARN_ge_D15
    M:RecCpx_D_ARN_ge_D16
    M:RecCpx_D_ARN_ge_D17
    M:RecCpx_D_ARN_ge_D18
    M:RecCpx_D_ARN_ge_D19
    M:RecCpx_D_ARN_ge_D2
    M:RecCpx_D_ARN_ge_D3
    M:RecCpx_D_ARN_ge_D4
    M:RecCpx_D_ARN_ge_D5
    M:RecCpx_D_ARN_ge_D6
    M:RecCpx_D_ARN_ge_D7
    M:RecCpx_D_ARN_ge_D8
    M:RecCpx_D_ARN_ge_D9
    M:RecCpx_D_ARN_pl_D1
    M:RecCpx_D_ARN_pl_D13
    M:RecCpx_D_ARN_pl_D14
    M:RecCpx_D_ARN_pl_D15
    M:RecCpx_D_ARN_pl_D16
    M:RecCpx_D_ARN_pl_D17
    M:RecCpx_D_ARN_pl_D18
    M:RecCpx_D_ARN_pl_D19
    M:RecCpx_D_ARN_pl_D2
    M:RecCpx_D_ARN_pl_D4
    M:RecCpx_D_ARN_pl_D5
    M:RecCpx_D_ARN_pl_D6
    M:RecCpx_D_ARN_pl_D7
    M:RecCpx_D_ARN_pl_D9
    M:RecCpx_M_ADN_ge_M10
    M:RecCpx_M_ADN_ge_M12
    M:RecCpx_M_ADN_ge_M13
    M:RecCpx_M_ADN_ge_M14
    M:RecCpx_M_ADN_ge_M15
    M:RecCpx_M_ADN_ge_M17
    M:RecCpx_M_ADN_ge_M1_1
    M:RecCpx_M_ADN_ge_M1_2
    M:RecCpx_M_ADN_ge_M2_1
    M:RecCpx_M_ADN_ge_M2_2
    M:RecCpx_M_ADN_ge_M3_1
    M:RecCpx_M_ADN_ge_M3_2
    M:RecCpx_M_ADN_ge_M4_1
    M:RecCpx_M_ADN_ge_M4_2
    M:RecCpx_M_ADN_ge_M5_1
    M:RecCpx_M_ADN_ge_M6_1
    M:RecCpx_M_ADN_ge_M6_2
    M:RecCpx_M_ADN_ge_M8
    M:RecCpx_M_ADN_ge_M9_1
    M:RecCpx_M_ADN_ge_M9_2
    M:RecCpx_M_ARN_ge_M1
    M:RecCpx_M_ARN_ge_M10
    M:RecCpx_M_ARN_ge_M11
    M:RecCpx_M_ARN_ge_M12
    M:RecCpx_M_ARN_ge_M13
    M:RecCpx_M_ARN_ge_M14
    M:RecCpx_M_ARN_ge_M15
    M:RecCpx_M_ARN_ge_M16
    M:RecCpx_M_ARN_ge_M17
    M:RecCpx_M_ARN_ge_M18
    M:RecCpx_M_ARN_ge_M19
    M:RecCpx_M_ARN_ge_M1_2
    M:RecCpx_M_ARN_ge_M2
    M:RecCpx_M_ARN_ge_M2(2)
    M:RecCpx_M_ARN_ge_M3
    M:RecCpx_M_ARN_ge_M3_2
    M:RecCpx_M_ARN_ge_M4
    M:RecCpx_M_ARN_ge_M5
    M:RecCpx_M_ARN_ge_M6
    M:RecCpx_M_ARN_ge_M7
    M:RecCpx_M_ARN_ge_M8
    M:RecCpx_M_ARN_ge_M9
    M:RecCpx_M_ARN_pl_M10_1
    M:RecCpx_M_ARN_pl_M10_2
    M:RecCpx_M_ARN_pl_M12
    M:RecCpx_M_ARN_pl_M13
    M:RecCpx_M_ARN_pl_M14
    M:RecCpx_M_ARN_pl_M16
    M:RecCpx_M_ARN_pl_M17
    M:RecCpx_M_ARN_pl_M18
    M:RecCpx_M_ARN_pl_M19
    M:RecCpx_M_ARN_pl_M1_1
    M:RecCpx_M_ARN_pl_M2
    M:RecCpx_M_ARN_pl_M20
    M:RecCpx_M_ARN_pl_M3_2
    M:RecCpx_M_ARN_pl_M4
    M:RecCpx_M_ARN_pl_M7_1
    M:RecCpx_M_ARN_pl_M7_2
    M:RecCpx_M_ARN_pl_M8_1
    M:RecCpx_M_ARN_pl_M8_2
    M:RecCpx_O_ARN_ge_O1
    M:RecCpx_O_ARN_ge_O10
    M:RecCpx_O_ARN_ge_O11
    M:RecCpx_O_ARN_ge_O12
    M:RecCpx_O_ARN_ge_O13
    M:RecCpx_O_ARN_ge_O14
    M:RecCpx_O_ARN_ge_O16
    M:RecCpx_O_ARN_ge_O18
    M:RecCpx_O_ARN_ge_O1_1
    M:RecCpx_O_ARN_ge_O2
    M:RecCpx_O_ARN_ge_O20
    M:RecCpx_O_ARN_ge_O2_2
    M:RecCpx_O_ARN_ge_O3
    M:RecCpx_O_ARN_ge_O3_2
    M:RecCpx_O_ARN_ge_O4
    M:RecCpx_O_ARN_ge_O5
    M:RecCpx_O_ARN_ge_O6
    M:RecCpx_O_ARN_ge_O7
    M:RecCpx_O_ARN_ge_O8
    M:RecCpx_O_ARN_ge_O9
    M:RecCpx_O_ARN_pl_O10_1
    M:RecCpx_O_ARN_pl_O12
    M:RecCpx_O_ARN_pl_O13
    M:RecCpx_O_ARN_pl_O14
    M:RecCpx_O_ARN_pl_O15
    M:RecCpx_O_ARN_pl_O1_1
    M:RecCpx_O_ARN_pl_O1_2
    M:RecCpx_O_ARN_pl_O2
    M:RecCpx_O_ARN_pl_O3_1
    M:RecCpx_O_ARN_pl_O3_2
    M:RecCpx_O_ARN_pl_O4_1
    M:RecCpx_O_ARN_pl_O4_2
    M:RecCpx_O_ARN_pl_O5_1
    M:RecCpx_O_ARN_pl_O5_2
    M:RecCpx_O_ARN_pl_O6_1
    M:RecCpx_O_ARN_pl_O6_2
    M:RecCpx_O_ARN_pl_O7_1
    M:RecCpx_O_ARN_pl_O8_1
    M:RecCpx_O_ARN_pl_O8_2
    M:RecCpx_O_ARN_pl_O9_1
    M:RecCpx_O_ARN_pl_O9_2
    M:U/A3_4769PROTHALD
    M:U/CRF01_AE_MAU509
    M:U/CRF45_cpx_1523Hald
    M:U/G_04SN.MS285
    M:U_225HALD
    M:U_29HALD
    M:U_305HALD
    M:U_483HALD
    M:U_K1200
    M:U_K1273
    M:U_K1381
    M:U_MP1069
    M:U_MP816
    M:U_ORAV208
    M:U_U1720
    M:URF(A/CRF02_AG)
    M:URF(A3/CRF02_AG)
    M:URF(C/CRF02_AG)
    M:URF(CRF02/A3)_08TG.T3039
    M:URF(CRF02/A3)_08TG.T3041
    M:URF(CRF02/CRF06/CRF02)_08TG.E5009
    M:URF(CRF02/U/CRF02)_08TG.T3047
    M:URF(CRF02_AG/A3)
    M:URF(CRF02_AG/A3/CRF02_AG)
    M:URF(CRF02_AG/A3/CRF02_AG/A3)
    M:URF(CRF02_AG/C)
    M:URF(CRF02_AG/CRF06_cpx)
    M:URF(CRF02_AG/CRF06_cpx/CRF02_AG)
    M:URF(CRF02_AG/CRF06_cpx/CRF02_AG/CRF06_cpx)
    M:URF(CRF02_AG/CRF06_cpx/CRF02_AG/CRF06_cpx/U)
    M:URF(CRF02_AG/CRF09_cpx/CRF02_AG)
    M:URF(CRF02_AG/G)
    M:URF(CRF02_AG/U)
    M:URF(CRF02_AG/U/A3)
    M:URF(CRF06/G)_08TG.E5047
    M:URF(CRF06_cpx/CRF02_AG/A)
    M:URF(CRF06_cpx/CRF02_AG/CRF06_cpx)
    M:URF(CRF37_cpx/A/CRF37_cpx)
    M:URF(CRF45_cpx/C)
    M:URF(D/CRF02_AG/D)
    M:URF(G/A)_08TG.T3067
    M:URF(G/A/CRF02)_08TG.E5024
    M:URF(G/A3)_08TG.T3071
    M:URF(G/CRF02/G/CRF02)_08TG.C7044
    M:URF(G/CRF02/G/CRF02/G)_08TG.C7022
    M:URF(G/CRF06_cpx)
    M:URF(J/U/K/J)_08TG.C7012
    M:URF(U/CRF02_AG)_08TG.T3013
    M:URF(U/CRF06/CRF02/CRF06/U)_08TG.C7004
    M:URF/D_PT317
    M:URF_03092610735
    M:URF_06526755339
    M:URF_07415052079
    M:URF_09504133825
    M:URF_30631572644
    M_02CD.KS069
    M_02CD.LBTB032
    M_02CD.LBTB084
    M_02CD.MBTB047
    M_97CD.KFE267
    M_97CD.KMST91
    M_97CD.KTB119
    M_97CD.MBFE250
    M_Sw_oHS_04/11_10_3(2)
    M_Sw_oHS_08/11_6_2(1)
    MA_11
    MA_26
    MAR_274_D06_2010-01-20
    MC1_11
    MC1_16
    MC1_2
    MC1_26
    MC2_24_MR-2014
    MC3_11
    MEP_002
    MEP_003
    MEP_007
    MEP_009
    MEP_010
    MEP_014
    Mex_CoV-1
    Mex_CoV-10
    Mex_CoV-11a
    Mex_CoV-11b
    Mex_CoV-2
    Mex_CoV-3
    Mex_CoV-4
    Mex_CoV-5a
    Mex_CoV-5b
    Mex_CoV-6
    Mex_CoV-7
    Mex_CoV-8
    Mex_CoV-9
    MF1_21
    MF1_22_MR-2014
    MF_isolate_1
    MF_isolate_10
    MF_isolate_11
    MF_isolate_13
    MF_isolate_14
    MF_isolate_15
    MF_isolate_16
    MF_isolate_17
    MF_isolate_18
    MF_isolate_22
    MF_isolate_26
    MF_isolate_28
    MF_isolate_29
    MF_isolate_31
    MF_isolate_33
    MF_isolate_35
    MF_isolate_4
    MF_isolate_41
    MF_isolate_42
    MF_isolate_43
    MF_isolate_8
    MPB_DPBA28
    MPB_DPFF10B
    MPB_DPFF1a
    MPB_DPFF2
    MPB_DPFF3a
    MPB_DPFF9B
    MPB_DPMF2
    MPB_JP_a
    MPB_JP_b
    MPB_LP_a
    MPB_LP_b
    MPB_LP_c
    MPB_LP_d
    MPB_LP_e
    MPB_LPxJP_a
    MPB_LPxJP_b
    MPB_LPxJP_c
    MPB_LPxJP_d
    MPB_LPxJP_e
    MPB_LPxJP_f
    MPB_LPxJP_g
    MPB_LPxJP_h
    MPB_OSA52A
    MPB_OSA52B
    MSX_63619
    Myotis/13rs384_31/Italy/2012
    N11_011-8
    N_OV_001
    N_OV_003
    N_OV_008
    N_OV_011
    N_OV_013
    N_OV_039
    N_OV_064
    N_OV_080
    N_OV_137
    N_YBF106
    N_YBF30
    NG_camel27
    NG_camel37
    NG_camel40
    NG_chicken
    NG_chicken11
    NG_chicken12
    NG_chicken13
    NG_chicken14
    NG_chicken15
    NG_chicken16
    NG_chicken17
    NG_chicken18
    NG_chicken19
    NG_chicken2
    NG_chicken20
    NG_chicken21
    NG_chicken22
    NG_chicken23
    NG_chicken24
    NG_chicken25
    NG_chicken26
    NG_chicken3
    NG_chicken31
    NG_chicken32
    NG_chicken33
    NG_chicken36
    NG_chicken37
    NG_chicken38
    NG_chicken39
    NG_chicken5
    NG_chicken6
    NG_chicken7
    NG_chicken8
    NG_chicken9
    NG_cow12
    NG_cow2
    NG_cow23
    NG_sheep50
    NIO_00081
    NQ1K_cl_15_1d
    NQ1K_cl_15_2b
    NQ1K_cl_15_3b
    NQ1K_cl_15_5d
    NQ2B_9
    NSA_12
    NSA_23
    O_ANT70
    O_MVP5180
    OSW_10
    OU_20
    OX_B
    OX_D
    OX_LEAF1
    OX_LEAF2
    OX_LEAF3
    OX_LEAF4
    OX_LONAR
    OX_M
    OX_R
    OX_T1
    OX_T2
    OX_T3
    OX_V1
    OX_V2
    OX_Y
    p5_H12
    p62_G06
    p8_G05
    PE_BacIPD31B
    PE_IP16
    PE_IP5a
    PE_IPC15A
    PE_IPFFA5
    PE_IPFFA6
    PE_IPFFB6
    PE_IPFFB7
    PE_IPFFB9
    PE_IPFFC12
    PE_IPFFC13
    PE_IPFFC14
    PE_IPFFC15-B
    PE_IPFFC18
    PE_IPFFC19
    PE_IPFFD20
    PE_IPFFD21
    PE_IPMFB27
    PE_IPMFB28
    pEBO_BDBV-GP
    PEPV_San92
    PGPV10K_LD1
    PGPV10K_RP1
    PGPV10K_SP1
    PGPV11K_FP1
    PGPV11K_FP2
    PGPV11K_LD2
    PGPV11K_Pi5
    PGPV93K_RP2
    PGPVO_Pi1
    PGPVO_Pi2
    PGPVO_Pi4
    Ph1_Loihi
    Ph2_Loihi
    phage_Deep-GF0-KM16-C193
    phage_Deep1-GF2-KM23-C739
    phage_MedDCM-OCT-S28-C10
    phage_MedDCM-OCT-S28-C3
    phage_MedDCM-OCT-S30-C28
    phage_MedDCM-OCT-S31-C1
    phage_MedDCM-OCT-S35-C6
    phage_MedDCM-OCT-S37-C6
    phage_MedDCM-OCT-S38-C3
    phage_MedDCM-OCT-S39-C11
    phage_MedDCM-OCT-S42-C7
    phage_MedDCM-OCT-S45-C18
    phage_MedDCM-OCT-S45-C4
    phage_MedDCM-OCT-S46-C10
    pHC_pUC_ori
    PI_GH1.1.B5
    PI_GH2.1.C2
    pig/P15_2/2004/SVN
    pig/sw-90_1/SWE/2010
    pig/sw-92_2/SWE/2010
    pig/Swebo_1/SWE
    PII_GH2.2.C11
    Pip1_Cr_FR_2014
    Pip2_Cr_FR_2014
    Pip3_M_FR_2014
    pLC_pBBR1_ori
    PM58B_RA
    PREDICT_AdV-13
    PREDICT_AdV-14
    PREDICT_AdV-15
    PREDICT_AdV-16
    PREDICT_AdV-17
    PREDICT_AdV-18
    PREDICT_AdV-19
    PREDICT_AdV-2
    PREDICT_AdV-20
    PREDICT_AdV-21
    PREDICT_AdV-22
    PREDICT_AdV-23
    PREDICT_AdV-24
    PREDICT_AdV-25
    PREDICT_AdV-26
    PREDICT_AdV-27
    PREDICT_AdV-28
    PREDICT_AdV-29
    PREDICT_AdV-30
    PREDICT_AdV-31
    PREDICT_AdV-32
    PREDICT_AdV-33
    PREDICT_AdV-34
    PREDICT_AdV-35
    PREDICT_AdV-36
    PREDICT_AdV-37
    PREDICT_AdV-38
    PREDICT_AdV-39
    PREDICT_AdV-40
    PREDICT_AdV-41
    PREDICT_AdV-42
    PREDICT_AdV-43
    PREDICT_AdV-44
    PREDICT_AdV-45
    PREDICT_AdV-46
    PREDICT_AdV-47
    PREDICT_AdV-48
    PREDICT_AdV-49
    PREDICT_AdV-50
    PREDICT_AdV-51
    PREDICT_AdV-52
    PREDICT_AdV-53
    PREDICT_AdV-54
    PREDICT_AdV-55
    PREDICT_AdV-56
    PREDICT_AdV-57
    PREDICT_AdV-58
    PREDICT_AdV-59
    PREDICT_AdV-60
    PREDICT_AdV-61
    PREDICT_AdV-62
    PREDICT_AdV-63
    PREDICT_AdV-64
    PREDICT_AdV-65
    PREDICT_AdV-66
    PREDICT_AdV-67
    PREDICT_AdV-68
    PREDICT_AdV-69
    PREDICT_AdV-70
    PREDICT_AdV-71
    PREDICT_AdV-72
    PREDICT_AdV-73
    PREDICT_AdV-74
    PREDICT_AdV-75
    PREDICT_AdV-77
    PREDICT_AdV-78
    PREDICT_AdV-79
    PREDICT_AdV-80
    PREDICT_AdV-81
    PREDICT_AdV-87
    PREDICT_AdV-90
    PREDICT_AdV-91
    PREDICT_CoV-49
    PREDICT_CoV-50
    PREDICT_Orbi-6
    PREDICT_PapV-1
    PREDICT_PbV-1
    PREDICT_PbV-10
    PREDICT_PbV-100
    PREDICT_PbV-101
    PREDICT_PbV-102
    PREDICT_PbV-103
    PREDICT_PbV-104
    PREDICT_PbV-105
    PREDICT_PbV-106
    PREDICT_PbV-107
    PREDICT_PbV-108
    PREDICT_PbV-109
    PREDICT_PbV-11
    PREDICT_PbV-110
    PREDICT_PbV-111
    PREDICT_PbV-112
    PREDICT_PbV-113
    PREDICT_PbV-114
    PREDICT_PbV-115
    PREDICT_PbV-116
    PREDICT_PbV-117
    PREDICT_PbV-118
    PREDICT_PbV-119
    PREDICT_PbV-12
    PREDICT_PbV-120
    PREDICT_PbV-121
    PREDICT_PbV-13
    PREDICT_PbV-14
    PREDICT_PbV-15
    PREDICT_PbV-16
    PREDICT_PbV-17
    PREDICT_PbV-18
    PREDICT_PbV-19
    PREDICT_PbV-2
    PREDICT_PbV-20
    PREDICT_PbV-21
    PREDICT_PbV-22
    PREDICT_PbV-23
    PREDICT_PbV-24
    PREDICT_PbV-25
    PREDICT_PbV-26
    PREDICT_PbV-27
    PREDICT_PbV-28
    PREDICT_PbV-29
    PREDICT_PbV-3
    PREDICT_PbV-30
    PREDICT_PbV-31
    PREDICT_PbV-32
    PREDICT_PbV-33
    PREDICT_PbV-34
    PREDICT_PbV-35
    PREDICT_PbV-36
    PREDICT_PbV-37
    PREDICT_PbV-38
    PREDICT_PbV-39
    PREDICT_PbV-4
    PREDICT_PbV-40
    PREDICT_PbV-41
    PREDICT_PbV-42
    PREDICT_PbV-43
    PREDICT_PbV-44
    PREDICT_PbV-45
    PREDICT_PbV-46
    PREDICT_PbV-47
    PREDICT_PbV-48
    PREDICT_PbV-49
    PREDICT_PbV-5
    PREDICT_PbV-50
    PREDICT_PbV-51
    PREDICT_PbV-52
    PREDICT_PbV-53
    PREDICT_PbV-54
    PREDICT_PbV-55
    PREDICT_PbV-57
    PREDICT_PbV-58
    PREDICT_PbV-59
    PREDICT_PbV-6
    PREDICT_PbV-60
    PREDICT_PbV-61
    PREDICT_PbV-62
    PREDICT_PbV-63
    PREDICT_PbV-64
    PREDICT_PbV-65
    PREDICT_PbV-66
    PREDICT_PbV-67
    PREDICT_PbV-68
    PREDICT_PbV-69
    PREDICT_PbV-7
    PREDICT_PbV-70
    PREDICT_PbV-71
    PREDICT_PbV-72
    PREDICT_PbV-73
    PREDICT_PbV-74
    PREDICT_PbV-75
    PREDICT_PbV-76
    PREDICT_PbV-77
    PREDICT_PbV-78
    PREDICT_PbV-79
    PREDICT_PbV-8
    PREDICT_PbV-80
    PREDICT_PbV-81
    PREDICT_PbV-82
    PREDICT_PbV-83
    PREDICT_PbV-84
    PREDICT_PbV-85
    PREDICT_PbV-86
    PREDICT_PbV-87
    PREDICT_PbV-88
    PREDICT_PbV-89
    PREDICT_PbV-9
    PREDICT_PbV-90
    PREDICT_PbV-91
    PREDICT_PbV-92
    PREDICT_PbV-93
    PREDICT_PbV-94
    PREDICT_PbV-95
    PREDICT_PbV-96
    PREDICT_PbV-97
    PREDICT_PbV-98
    PREDICT_PbV-99
    PREDICT_PicornaV-1
    PREDICT_PicornaV-2
    PREDICT_PicornaV-3
    PREDICT_PicornaV-4
    PREDICT_PMV-10
    PREDICT_PMV-100
    PREDICT_PMV-101
    PREDICT_PMV-102
    PREDICT_PMV-103
    PREDICT_PMV-104
    PREDICT_PMV-105
    PREDICT_PMV-106
    PREDICT_PMV-107
    PREDICT_PMV-108
    PREDICT_PMV-109
    PREDICT_PMV-11
    PREDICT_PMV-110
    PREDICT_PMV-113
    PREDICT_PMV-114
    PREDICT_PMV-115
    PREDICT_PMV-116
    PREDICT_PMV-117
    PREDICT_PMV-118
    PREDICT_PMV-119
    PREDICT_PMV-12
    PREDICT_PMV-120
    PREDICT_PMV-121
    PREDICT_PMV-122
    PREDICT_PMV-124
    PREDICT_PMV-125
    PREDICT_PMV-126
    PREDICT_PMV-127
    PREDICT_PMV-128
    PREDICT_PMV-13
    PREDICT_PMV-131
    PREDICT_PMV-132
    PREDICT_PMV-133
    PREDICT_PMV-137
    PREDICT_PMV-138
    PREDICT_PMV-139
    PREDICT_PMV-14
    PREDICT_PMV-140
    PREDICT_PMV-141
    PREDICT_PMV-142
    PREDICT_PMV-143
    PREDICT_PMV-144
    PREDICT_PMV-145
    PREDICT_PMV-146
    PREDICT_PMV-147
    PREDICT_PMV-148
    PREDICT_PMV-149
    PREDICT_PMV-15
    PREDICT_PMV-150
    PREDICT_PMV-151
    PREDICT_PMV-152
    PREDICT_PMV-153
    PREDICT_PMV-155
    PREDICT_PMV-156
    PREDICT_PMV-159
    PREDICT_PMV-16
    PREDICT_PMV-160
    PREDICT_PMV-161
    PREDICT_PMV-163
    PREDICT_PMV-167
    PREDICT_PMV-168
    PREDICT_PMV-169
    PREDICT_PMV-17
    PREDICT_PMV-170
    PREDICT_PMV-171
    PREDICT_PMV-172
    PREDICT_PMV-173
    PREDICT_PMV-174
    PREDICT_PMV-175
    PREDICT_PMV-176
    PREDICT_PMV-18
    PREDICT_PMV-19
    PREDICT_PMV-2
    PREDICT_PMV-20
    PREDICT_PMV-21
    PREDICT_PMV-22
    PREDICT_PMV-23
    PREDICT_PMV-24
    PREDICT_PMV-25
    PREDICT_PMV-26
    PREDICT_PMV-27
    PREDICT_PMV-28
    PREDICT_PMV-29
    PREDICT_PMV-3
    PREDICT_PMV-30
    PREDICT_PMV-31
    PREDICT_PMV-32
    PREDICT_PMV-33
    PREDICT_PMV-34
    PREDICT_PMV-35
    PREDICT_PMV-36
    PREDICT_PMV-37
    PREDICT_PMV-38
    PREDICT_PMV-39
    PREDICT_PMV-4
    PREDICT_PMV-40
    PREDICT_PMV-41
    PREDICT_PMV-42
    PREDICT_PMV-43
    PREDICT_PMV-44
    PREDICT_PMV-45
    PREDICT_PMV-46
    PREDICT_PMV-47
    PREDICT_PMV-48
    PREDICT_PMV-49
    PREDICT_PMV-5
    PREDICT_PMV-50
    PREDICT_PMV-51
    PREDICT_PMV-52
    PREDICT_PMV-53
    PREDICT_PMV-54
    PREDICT_PMV-55
    PREDICT_PMV-56
    PREDICT_PMV-57
    PREDICT_PMV-58
    PREDICT_PMV-59
    PREDICT_PMV-6
    PREDICT_PMV-60
    PREDICT_PMV-62
    PREDICT_PMV-63
    PREDICT_PMV-64
    PREDICT_PMV-65
    PREDICT_PMV-66
    PREDICT_PMV-67
    PREDICT_PMV-68
    PREDICT_PMV-69
    PREDICT_PMV-70
    PREDICT_PMV-71
    PREDICT_PMV-72
    PREDICT_PMV-73
    PREDICT_PMV-74
    PREDICT_PMV-75
    PREDICT_PMV-76
    PREDICT_PMV-77
    PREDICT_PMV-78
    PREDICT_PMV-79
    PREDICT_PMV-80
    PREDICT_PMV-81
    PREDICT_PMV-82
    PREDICT_PMV-83
    PREDICT_PMV-85
    PREDICT_PMV-86
    PREDICT_PMV-91
    PREDICT_PMV-92
    PREDICT_PMV-93
    PREDICT_PMV-94
    PREDICT_PMV-95
    PREDICT_PMV-97
    PREDICT_PMV-98
    PREDICT_PMV-99
    PREDICT_PyV-14
    PREDICT_PyV-15
    PREDICT_PyV-22
    PREDICT_PyV-23
    PREDICT_PyV-24
    PREDICT_PyV-25
    PREDICT_PyV-28
    PREDICT_PyV-30
    PREDICT_PyV-31
    PREDICT_PyV-32
    PREDICT_PyV-34
    PREDICT_PyV-5
    PREDICT_PyV-6
    PREDICT_PyV-7
    PREDICT_PyV-9
    PREDICT_RbdV-14
    PREDICT_RbdV-15
    pRS315_AnSuc1:pep1
    pRS315_Ppdc_xyr1V821F:ace1
    pRS315_Pxyn1_ReCel3a:slp1
    pRS315_res1
    pStA0.BBa_J23100
    pStA0.BBa_J23102
    pStA0.BBa_J23107
    pStA0.BBa_J23113
    pStA0.BBa_J23116
    pStA0.BBa_J23118
    pT7SA11NSP1_3xFL_UnaG
    Q_sR_OV_011
    Q_sR_OV_019
    Q_sR_OV_020
    Q_sR_OV_022
    Q_sR_OV_023
    Q_sR_OV_025
    Q_sR_OV_036
    Q_sR_OV_041
    Q_sR_OV_042
    R2_11
    R2_8
    Rabbit_CoV/La06_256/Ory_cun/ESP/2006
    Rabbit_CoV/La09_24/Ory_cun/ESP/2009
    Rabbit_CoV/La09_31/Ory_cun/ESP/2009
    Rabbit_CoV/La09_32/Ory_cun/ESP/2009
    Rabbit_CoV/La09_48/Ory_cun/ESP/2009
    RBC_A_MPN10_2
    RBC_B_mpn_10_2_GM
    RBG_16_50_20
    RCoV/Bor473/Rat_rat/MYS/2012
    RCoV/D_RMU10_1055/Myo_gla/GER/2010
    RCoV/D_RMU10_1087/Apo_fla/GER/2010
    RCoV/D_RMU10_1182/Mic_agr/GER/2010
    RCoV/D_RMU10_1222/Mus_dom/GER/2010
    RCoV/D_RMU10_1226/Myo_gla/GER/2010
    RCoV/D_RMU10_1228/Mus_dom/GER/2010
    RCoV/D_RMU10_1919/Myo_gla/GER/2010
    RCoV/D_RMU10_1940/Myo_gla/GER/2010
    RCoV/D_RMU10_1972/Myo_gla/GER/2010
    RCoV/D_RMU10_1974/Myo_gla/GER/2010
    RCoV/D_RMU10_1985/Myo_gla/GER/2010
    RCoV/D_RMU10_1989/Myo_gla/GER/2010
    RCoV/D_RMU10_1993/Myo_gla/GER/2010
    RCoV/D_RMU10_2065/Myo_gla/GER/2010
    RCoV/D_RMU10_2066/Myo_gla/GER/2010
    RCoV/D_RMU10_2070/Myo_gla/GER/2010
    RCoV/D_RMU10_2089/Myo_gla/GER/2010
    RCoV/D_RMU10_590/Mic_arv/GER/2010
    RCoV/D_RMU10_592/Mic_arv/GER/2010
    RCoV/D_RMU10_974/Myo_gla/GER/2010
    RCoV/D_RMU10_986/Myo_gla/GER/2010
    RCoV/KS10_2636/Mic_arv/GER/2010
    RCoV/KS10_2743/Mic_arv/GER/2010
    RCoV/KS10_2810/Mic_arv/GER/2010
    RCoV/KS10_2864/Myo_glar/GER/2010
    RCoV/KS11_0514/Mic_arv/GER/2011
    RCoV/KS11_0997/Apo_agr/GER/2011
    RCoV/LEO/Rat_sik/CHN/2007
    RCoV/Lmg115/Sig_spec/MEX/2011
    RCoV/NL_69/Mic_arv/NED/2007
    RCoV/NSP32/Rha_pum/RSA/2008
    RCoV/NSP63/Rha_pum/RSA/2008
    RCoV/NSP65/Rha_pum/RSA/2008
    RCoV/RMU10_2010/Myo_gla/GER/2010
    RCoV/RMU10_3052/Apo_fla/GER/2010
    RCoV/RMU10_3147/Apo_fla/GER/2010
    RCoV/RMU10_3212/Myo_gla/GER/2010
    RCoV/RMU10_620/Apo_agr/GER/2010
    RCoV/SPB40/Liomys_spec/MEX/2011
    RCoV/TP3_22/Ban_ind/THA/2005
    RCoV/TP3_26/Rat_rat/THA/2005
    Ri1Ps_4703
    Ri1Ps_4749
    Ri1Pw_5477
    Ri1Pw_5610
    Ri1Pw_5615
    Ri2Ps_4855
    Ri2Ps_4998
    Ri2Ps_5096
    Ri2Ps_5130
    Ri2Pt_4001
    Ri2Pt_4081
    Ri2Pw_6101
    Ri3Ps_5313
    Ri3Pw_6170
    Ri3Pw_6328
    RK10_tank
    RK10a_tank
    RK11_tank
    RK13_tank
    RK14_tank
    RK16_1_tank
    RK17_tank
    RK18_tank
    RK1_tank
    RK20_tank
    RK23_tank
    RK24_tank
    RK25_tank
    RK27_tank
    RK28_tank
    RK2a_tank
    RK31_tank
    RK31a_tank
    RK39_tank
    RK4_tank
    RK5_tank
    RK5a2_tank
    RK5a_tank
    RK65_tank
    RK66_tank
    RK67_tank
    RK69_tank
    RK70_tank
    RK73_tank
    RK77_tank
    RK78_tank
    RK79_tank
    RK7_tank
    RK80_tank
    RK81_tank
    RK84_tank
    RK85_tank
    RK88_tank
    RK89_tank
    RK8_tank
    RK91_tank
    RK92_tank
    RK97_tank
    RK9_tank
    RM12EL_21F
    Rn/GV/HKU_CT2/HKG/2011
    Rn/GV/HKU_KT/HKG/2012
    RO_4
    RO_isolate_1
    RO_isolate_11
    RO_isolate_12
    RO_isolate_14
    RO_isolate_17
    RO_isolate_18
    RO_isolate_19
    RO_isolate_2
    RO_isolate_23
    RO_isolate_26
    RO_isolate_29
    RO_isolate_3
    RO_isolate_32
    RO_isolate_6
    RO_isolate_7
    S1_MeH
    S2_MeH
    S8A-217_16S
    S_A17_125
    S_A17_144
    S_A17_145
    S_C14_118
    S_C14_131
    S_C14_133
    S_C14_134
    S_C14_135
    S_C14_136
    S_C49_155
    S_C49_24
    S_C49_26
    S_C49_70
    S_C49_74
    S_D75_1
    S_D75_109
    S_D75_157
    S_D75_164
    S_D75_168
    S_D75_169
    S_D75_33
    S_D75_81
    S_D75_83
    S_D75_84
    S_D75_94
    SA08_00821
    SA_07_02434
    SA_0800821
    SAL_1
    SAL_18
    SAL_7
    salicis_cluster2
    salicis_cluster3
    SB_Bac1
    SB_Bac10
    SB_Bac11
    SB_Bac2
    SB_Bac4
    SB_Bac5
    SB_Bac7
    SB_Bac9
    SB_Bact3
    SBN1_DGR
    SBOS_U
    SC_4_H6H8_2017
    Schreyahn_AOB_Sali_Kultur_19
    Schreyahn_AOB_Sali_Kultur_8
    Schreyahn_AOB_SSU_Aster_6
    Schreyahn_AOB_SSU_Aster_8
    Schreyahn_K10.Sali
    Schreyahn_K9.Sali
    SCS_04Z1_1
    SCS_04Z1_26_2
    SCS_04Z1_29
    SCS_04Z1_3
    SCS_04Z1_41_2
    SCS_04Z1_41_3
    SCS_04Z1_42_2
    SCS_04Z1_47_3
    SCS_04Z1_60_1
    SCS_04Z1_60_2
    SCS_04Z1_62
    SCS_04Z1_68_2
    SCS_30Z0_1
    SCS_30Z0_10_3
    SCS_30Z0_12
    SCS_30Z0_20_2
    SCS_30Z0_23
    SCS_30Z0_37_2
    SCS_30Z0_4
    SCS_30Z0_45_2
    SCS_30Z0_49_3
    SCS_30Z0_6_2
    SCS_30Z1_11
    SCS_30Z1_12_2
    SCS_30Z1_14_2
    SCS_30Z1_20_3
    SCS_30Z1_24
    SCS_30Z1_28
    SCS_30Z1_37_2
    SCS_30Z1_39_2
    SCS_30Z1_4_2
    SCS_30Z1_8
    Se1Ps_1969
    Se1Pw_3014
    Se1Pw_3206
    Se2Ps_2218
    Se2Ps_2269
    Se2Ps_2350
    Se2Ps_2423
    Se2Ps_2430
    Se2Ps_2446
    Se2Ps_2490
    Se2Ps_640
    Se3Ps_2605
    Se3Ps_2679
    Se3Ps_2758
    Se3Pw_3619
    SM23_31
    SM23_57
    SM_12
    SM_3
    SNISO_A2
    SNISO_B12
    SNISO_C1
    SNISO_C6
    SNISO_D1
    SNISO_E1
    SNISO_E12
    SNISO_E3
    SNISO_F2
    SNISO_F3
    SNISO_F4
    SNISO_I12
    SNISO_J2
    SNISO_J4
    So1Ps_682
    So1Pw_1092
    So1Pw_968
    So1Pw_972
    So1Pw_974
    So3Ps_1722
    So3Ps_1723
    So3Ps_1725
    So3Pw_1027
    So3Pw_2867
    SOB_395
    sp.48_UnivF-1183
    sp.R0101_021
    sp_1_GP
    sp_2_GP
    sp_AL
    Spd2B_4
    Sq1_host
    SRC_DS3
    SRC_DS3a
    SRC_DSE1
    SRC_DSE11
    SRC_DSE3
    SRC_DSE5
    SRC_DSE6
    SRC_DSE9
    SRC_DSI1
    SRC_DSI2
    SRU_1
    SRU_17
    SRU_35
    SUND_HBM1
    SV72_D
    TC1_29
    TC1_4
    TC2_12
    TC2_13
    TC3_16_MR-2014
    TC3_23_MR-2014
    TC3_29
    TC3_7_MR-2014
    terrain_en_construction
    TF4_16
    TF4_17
    TF4_17_MR-2014
    TF5_22_MR-2014
    TPMV_P-EGFP
    TPMV_P-EGFP_H-His6
    TPMV_P-EGFP_HaCD20
    TPMV_P-EGFP_HaEGFR
    TsCyV-1_JP-NUBS-2014
    UAB_Phi87
    UBP10_UBA1149
    UBP10_UBA1160
    UBP11_UBA4055
    UBP12_UBA5184
    UBP13_UBA5359
    UBP14_UBA6098
    UBP15_UBA6099
    UBP16_UBA6123
    UBP17_UBA6191
    UBP18_UBA7526
    UBP1_UBA2172
    UBP1_UBA4771
    UBP1_UBA4783
    UBP2_UBA2226
    UBP2_UBA2255
    UBP2_UBA2291
    UBP2_UBA4780
    UBP2_UBA5202
    UBP2_UBA5599
    UBP3_UBA1247
    UBP3_UBA1439
    UBP3_UBA3054
    UBP3_UBA6266
    UBP4_UBA4092
    UBP4_UBA6127
    UBP5_UBA1551
    UBP5_UBA1559
    UBP6_UBA1177
    UBP6_UBA1196
    UBP6_UBA1209
    UBP6_UBA3056
    UBP7_UBA6624
    UBP7_UBA7824
    UBP8_UBA6595
    UBP8_UBA7324
    UBP8_UBA7816
    UBP8_UBA817
    UBP8_UBA827
    UBP9_UBA1085
    UBP9_UBA11836
    UBP9_UBA1432
    UBP9_UBA4627
    UBP9_UBA4705
    UBP9_UBA5552
    UBP9_UBA6111
    UBP9_UBA6348
    UBP9_UBA6438
    Udayana_MDPVA
    Udayana_MDPVB
    vB_NpeS-2AV2
    VPXV_CA85
    W13C10_B
    W13C11_B
    W13C12_B
    W13C14_B
    W13C15_B
    W13C16_B
    W13C17_B
    W13C18_B
    W13C21_B
    W13C22_B
    W13C23_B
    W13C24_B
    W13C25_B
    W13C27_B
    W13C28_B
    W13C2_B
    W13C30_B
    W13C32_B
    W13C33_B
    W13C34_B
    W13C5_B
    W13C7_B
    W13M2_F
    W13M4_B
    W13M60_B
    W13M61a_B
    W13M62b_B
    W13M7_B
    W13S12_B
    W13S13_B
    W13S16_B
    W13S18_B
    W13S21_B
    W13S22_B
    W13S23_B
    W13S26_B
    W13S28_F
    W13S29_B
    W13S2_B
    W13S30_B
    W13S31_B
    W13S32_B
    W13S34_B
    W13S46_B
    W13S49_B
    W13S4_B
    W13S51_B
    W13S54_B
    W13S8_B
    W15C15_F
    W15C16_F
    W15C18a_F
    W15C28_F
    W15C2_F
    W15M16_F
    W15M1a_F
    W15M34_F
    W15S10_F
    W15S11_F
    W15S14_F
    W15S18_F
    W15S24b_F
    W15S32_F
    W15S58a_F
    W15S67_F
    W15S83_F
    W15S84a_F
    W15S84b_F
    W15S87a_F
    wastewater/GI.1/GP_Dec11_1395_1/2011/ZAF
    wastewater/GI.1/GP_Oct11_1145_3/2011/ZAF
    wastewater/GI.1/MP_Aug10_992/2010/ZAF
    wastewater/GI.1/MP_Nov10_1391/2010/ZAF
    wastewater/GI.1/MP_Oct11_1168/2011/ZAF
    wastewater/GI.1/MP_Sep10_1108/2010/ZAF
    wastewater/GI.3/GP_Oct11_1145_1/2011/ZAF
    wastewater/GI.3/GP_Oct11_1145_5/2011/ZAF
    wastewater/GI.3/NW_Nov11_1266/2011/ZAF
    wastewater/GI.4/GP_Sep11_1051/2011/ZAF
    wastewater/GI.8/NW_Aug11_957/2011/ZAF
    wastewater/GI/GP_Dec11_1395_2/2011/ZAF
    wastewater/GI/GP_Oct11_1085/2011/ZAF
    wastewater/GI/GP_Oct11_1145_2/2011/ZAF
    wastewater/GI/NW_Dec11_1394/2011/ZAF
    wastewater/GI/NW_Jul11_759/2011/ZAF
    wastewater/GI/NW_Nov11_1268/2011/ZAF
    wastewater/GII.12/NW_Aug11_957_1/2011/ZAF
    wastewater/GII.13/MP_Aug10_1012_3/2010/ZAF
    wastewater/GII.13/MP_Oct11_1086_5/2011/ZAF
    wastewater/GII.13/MP_Oct11_1168_2/2011/ZAF
    wastewater/GII.17/GP_Dec11_1395_1/2011/ZAF
    wastewater/GII.17/MP_Aug10_1012_5/2010/ZAF
    wastewater/GII.17/MP_Jul11_862_1/2011/ZAF
    wastewater/GII.17/MP_Sep10_1108/2010/ZAF
    wastewater/GII.17/NW_Aug11_1035_1/2011/ZAF
    wastewater/GII.17/NW_Aug11_908/2011/ZAF
    wastewater/GII.17/NW_Jul11_759/2011/ZAF
    wastewater/GII.17/NW_Jul11_874_1/2011/ZAF
    wastewater/GII.17/NW_Nov11_1268/2011/ZAF
    wastewater/GII.2/GP_Oct11_1083/2011/ZAF
    wastewater/GII.2/GP_Oct11_1145/2011/ZAF
    wastewater/GII.2/GP_Oct11_1147/2011/ZAF
    wastewater/GII.2/GP_Sep11_1051/2011/ZAF
    wastewater/GII.2/MP_Aug10_985/2010/ZAF
    wastewater/GII.2/MP_Oct11_1168_3/2011/ZAF
    wastewater/GII.2/MP_Sep10_1124/2010/ZAF
    wastewater/GII.2/NW_Aug11_909_3/2011/ZAF
    wastewater/GII.2/NW_Dec11_1394/2011/ZAF
    wastewater/GII.2/NW_Nov11_1266_1/2011/ZAF
    wastewater/GII.2/NW_Oct11_1077_1/2011/ZAF
    wastewater/GII.2/NW_Oct11_1077_2/2011/ZAF
    wastewater/GII.3/MP_Aug10_1010_4/2010/ZAF
    wastewater/GII.3/MP_Aug10_1012_4/2010/ZAF
    wastewater/GII.3/MP_Dec10_1493/2010/ZAF
    wastewater/GII.3/MP_Dec10_1521_6/2010/ZAF
    wastewater/GII.3/MP_Oct10_1229/2010/ZAF
    wastewater/GII.4/GP_Dec11_1395_2/2011/ZAF
    wastewater/GII.4/GP_Dec11_1395_3/2011/ZAF
    wastewater/GII.4/KZN_Nov11_1306/2011/ZAF
    wastewater/GII.4/KZN_Sep10_1052/2010/ZAF
    wastewater/GII.4/MP_Aug10_1010_2/2010/ZAF
    wastewater/GII.4/MP_Aug10_1010_3/2010/ZAF
    wastewater/GII.4/MP_Aug10_1011_1/2010/ZAF
    wastewater/GII.4/MP_Aug10_1011_2/2010/ZAF
    wastewater/GII.4/MP_Oct11_1086_1/2011/ZAF
    wastewater/GII.4/MP_Oct11_1168_1/2011/ZAF
    wastewater/GII.4/NW_Aug11_957_4/2011/ZAF
    wastewater/GII.4/NW_Jul11_757_1/2011/ZAF
    wastewater/GII.4/NW_Nov11_1266_3/2011/ZAF
    wastewater/GII.6/MP_Aug10_1011_3/2010/ZAF
    wastewater/GII.6/MP_Aug10_995/2010/ZAF
    wastewater/GII.6/MP_Dec10_1521_3/2010/ZAF
    wastewater/GII.6/NW_Aug11_909_1/2011/ZAF
    wastewater/GII.6/NW_Aug11_909_2/2011/ZAF
    wastewater/GII.6/NW_Jul11_757_4/2011/ZAF
    wastewater/GII.6/NW_Nov11_1266_2/2011/ZAF
    wastewater/GII.7/MP_Aug10_1012_1/2010/ZAF
    wastewater/GII.7/MP_Dec10_1521_4/2010/ZAF
    wastewater/GP_Dec11_1395/ZAF
    wastewater/GP_Oct11_1145/ZAF
    wastewater/MP_Aug10_1010/ZAF
    wastewater/MP_Aug10_1011/ZAF
    wastewater/MP_Aug10_985/ZAF
    wastewater/MP_Aug10_992/ZAF
    wastewater/MP_Aug10_995/ZAF
    wastewater/MP_Jul11_862/ZAF
    wastewater/MP_Oct10_1229/ZAF
    wastewater/MP_Oct11_1086/ZAF
    wastewater/MP_Sep10_1102/ZAF
    wastewater/MP_Sep10_1124/ZAF
    wastewater/NW_Aug11_909/ZAF
    wastewater/NW_Dec11_1396/ZAF
    wastewater/NW_Jul11_759/ZAF
    wastewater/NW_Nov11_1268/ZAF
    water/LP_Jan12_1/ZAF
    water/LP_Jan12_23/ZAF
    water/LP_Jan12_5/ZAF
    water/LP_Jan12_7/ZAF
    water/LP_Mar12_7/ZAF
    water/LP_Mar12_PVT/ZAF
    WBUT_C01
    WM06i_A10G
    WM06i_A11B
    WM06i_A1D
    WM06i_A5D
    WM06i_B11A
    WM06i_B11H
    WM06i_B1G
    WM06i_B6A
    WM06i_B8F
    WM06i_B9F
    WOR_2
    WS01_1416
    WS01_480
    WS06_481
    WS08_522
    WS10_380
    WS12_472
    WS13_484
    WS18_483
    WS19_479
    XAN_1
    XAN_14
    XAN_8
    XZ110_XZ_CHN_2017
    XZ111_XZ_CHN_2017
    XZ112_XZ_CHN_2017
    XZ114_XZ_CHN_2017
    XZ115_XZ_CHN_2017
    XZ93_XZ_CHN_2017
    XZ97_XZ_CHN_2017
    XZ98_XZ_CHN_2017
    XZ99_XZ_CHN_2017
    Y4_579_1
    YS1-2_2434
    YS1-2_2437
    );

my %is_hyphenated  = map { $_ => 1 } @GENERA;
my %is_underscored = map { $_ => 1 } @SPECIES;

sub _set_gca_and_or_taxon_id {
    my $self   = shift;
    my $strain = shift;

    # Note: we use named captures here (?<capture name> ... )

    ## no critic (ProhibitUnusedCapture)

    # handle GCA/GCF acting as taxon_ids
    if ($strain =~ m/\A (:?(?<strain>.*)_)? (?<gca>$NCBIGCA) \z/xms) {

        # truncate strain to optional non-GCA/GCF part (excl. underscore)...
        # ... and set taxon_id to full GCA/GCF
        $self->_set_strain($+{strain});
        my $gca = $+{gca};
        $self->_set_taxon_id($gca);
        $self->_set_gca(     $gca);     # alias

        # further store GCA/GCF components for easier comparison
        my ($gca_novers, $gca_vers) = split /\./xms, $gca;
        $self->_set_gca_novers($gca_novers);
        $self->_set_gca_vers(  $gca_vers  );
        my ($gca_prefix, $gca_number) = split /_/xms, $gca_novers;
        $self->_set_gca_prefix($gca_prefix);
        $self->_set_gca_number($gca_number);
    }

    # handle regular taxon_ids
    elsif ($strain =~ m/\A (:?(?<strain>.*)_)? (?<taxon_id>$NCBIPKEY) \z/xms) {
        # truncate strain to optional non-taxon_id part (excl. underscore)...
        # ... and set taxon_id
        $self->_set_strain(  $+{strain  });
        $self->_set_taxon_id($+{taxon_id});
    }

    ## use critic

    # ignore legacy strains
    return;
}

sub BUILD {
    my $self = shift;

    # parse potential taxonomy-aware abbreviated id
    # Note: such ids are still considered as foreign_ids
    {
        my ($strain, $acc) = $self->full_id =~ $TAXABBR_ID;
        if (defined $strain) {
            $self->_set_gca_and_or_taxon_id($strain);
            $self->_set_accession($acc);
            $self->_set_foreign;
            return;
        }
    }

    # parse potential NCBI FASTA-style GI id
    # Note: such ids are still considered as foreign_ids
    {
        my ($gi, $acc) = $self->full_id =~ $NCBIGI_ID;
        if (defined $gi) {
            $self->_set_gi($gi);
            $self->_set_accession($acc);
            $self->_set_foreign;
            return;
        }
    }

    # parse potential NCBI FASTA-style GNL id
    # Note: such ids are still considered as foreign_ids
    {
        my ($match) = $self->full_id =~ $NCBIGNL_ID;
        if ($match) {
            $self->_set_parts( [ split /\|/xms, $match ] );
            $self->_set_database(   $self->get_part(1) );   # official
            $self->_set_identifier( $self->get_part(2) );   # aliases
            $self->_set_foreign;
            return;
        }
    }

    # skip RiboDB ids that look like weird full_ids
    if ($self->full_id =~ m/~$NCBIGCA/xms) {
        $self->_set_foreign;
        return;
    }

    # check full_id validity
    my  ($family, $tag, $genus, $species, $strain, $acc, $tail, $new)
        = $self->full_id =~ $FULL_ID;
    unless (defined $genus) {

        # First try to coerce foreign full_id by replacing 1st '_' by ' '. If
        # this does not work, keep the original full_id and flag it as foreign.
        # This approach allows the transparent conversion of valid full_ids
        # from foreign software able to handle unlimited gap-free ids.
        # Note: This will fails if the optional family part contains an '_'.

        my $cand_id = $self->full_id =~ s{_}{ }xmsr;
        ($family, $tag, $genus, $species, $strain, $acc, $tail, $new)
            = $cand_id =~ $FULL_ID;
        unless (defined $genus) {
            $self->_set_foreign;
            return;
        }
        $self->_set_full_id($cand_id);
    }

    # handle hyphenated genera that could interfere with family definition
    if (defined $family) {
        my $hyphenated = $family . '-' . $genus;
        if (defined $is_hyphenated{$hyphenated}) {
            $family = undef;
            $genus = $hyphenated;
        }
    }

    # handle underscored species that could interfere with species definition
    if (defined $strain) {
        my $underscored = $species . '_' . $strain;
        if (defined $is_underscored{$underscored}) {
            $strain = undef;
            $species = $underscored;
        }
    }

    # populate legacy components from full_id (and tail)
    $self->_set_family($family);
    $self->_set_tag($tag);
    $self->_set_genus($genus);
    $self->_set_species($species);
    $self->_set_strain($strain);
    $self->_set_accession($acc);
    $self->_set_tail($tail);

    # populate modern components if available
    $self->_set_gca_and_or_taxon_id($strain) if defined $strain;
    $self->_set_gi($acc)                     if $acc =~ $PKEYONLY;

    # set new flag if needed
    $self->_set_new if $new;

    return;
}

}   # end of BEGIN block


around qr{ is_new | is_genus_only | is_doubtful | org $  }xms => sub {
    my $method = shift;
    my $self   = shift;

    # Note: we return an explicit undef to emulate other accessor behavior
    return undef if $self->is_foreign;      ## no critic (ProhibitExplicitReturnUndef)
    return $self->$method(@_);
};



sub is_genus_only {
    my $self = shift;
    return 1 if $self->species =~ $GENUSONLY;
    return 0;
}


sub is_doubtful {
    my $self = shift;
    return 1 if defined $self->tag && $self->tag =~ $DOUBTFUL;
    return 0;
}


sub org {
    my $self = shift;
    return $self->genus . q{ } . $self->species;
}


sub abbr_org {
    my $self = shift;
    return substr($self->genus, 0, 1) . q{. } . $self->species;
}


sub full_org {
    my $self = shift;
    my $sep  = shift // '_';        # default to underscore

    my $org = $self->org;
       $org .= $sep . $self->strain   if $self->strain;
       $org .= $sep . $self->taxon_id if $self->taxon_id;
                                    # also works with GCA/GCF!
    return $org;
}


sub family_then_full_org {                  ## no critic (RequireArgUnpacking)
    my $self = shift;

    my $family = $self->family // q{};
    $family .= '-' if $family;

    return $family . $self->full_org(@_);   # note the currying
}


sub contam_org {
    my $self = shift;

    return undef unless $self->tail;        ## no critic (ProhibitExplicitReturnUndef)
    # Note: we return an explicit undef to emulate other accessor behavior

    # only replace the first underscore by a space (for consistency)
    return $self->tail =~ s{_}{ }xmsr;
}


sub foreign_id {
    my $self = shift;

    # do not alter full_id if already foreign!
    return $self->full_id if $self->is_foreign;

    # otherwise, only replace the first space by an underscore
    return $self->full_id =~ s{\ }{_}xmsr;
}


sub nexus_id {
    my $self = shift;

    # quoted named are not escaped by Bio::Phylo...
    # ... but cannot contain single quotes

    my $full_id = $self->full_id =~ tr/'//dr;       # remove quotes...
	return qq{'$full_id'};                          # ... and requote
}

# from Bio::Phylo::PhyloRole

# =item get_nexus_name()
#
# Gets invocant's name, modified to be safely used in nexus files. This means that:
#
# =item names with spaces in them that aren't 'single quoted' have their spaces replaced
# with underscores
#
# =item names with any of the following characters in them are single quoted:
#  -^*(){}[]+=;:"\<>/,
#
# =item names with single quotes inside them (i.e. not around them) are "double quoted"
#
#  Type    : Accessor
#  Title   : get_nexus_name
#  Usage   : my $name = $obj->get_nexus_name;
#  Function: Returns the object's name.
#  Returns : A string
#  Args    : None
#
# =cut
#
#     sub get_nexus_name {
#         my $self = shift;
#         my $name = $self->get_internal_name;
#         if ( $name =~ /\s/ && $name !~ /^'.+'$/ ) {
#             $name =~ s/\s/_/g;
#         }
#         if (   $name =~ /(?:\-|\^|\*|\(|\)|{|}|\[|\]|\+|=|;|:|"|\\|<|>|\/|,)/
#             && $name !~ /^'.+'$/ )
#         {
#             $name = "'${name}'";
#         }
#         if ( $name =~ /'/ && $name !~ /^".+"$/ && $name !~ /^'.+'$/ ) {
#             $name = "\"${name}\"";
#         }
#         return $name;
#     }


# class methods to build modern MUST-compliant id from NCBI components


sub new_with {                              ## no critic (RequireArgUnpacking)
    my $class = shift;
    my %args  = @_;                         # TODO: handle HashRef?

    my ($org, $taxon_id, $accession, $keep_strain)
        = @args{ qw(org taxon_id accession keep_strain) };
    $accession //= $args{gi};       # fall back to legacy argument name

    # extract components from organism name
    my ($genus, $species, $strain) = $class->parse_ncbi_name($org);
    # Note: genus, species and strain will have been cleaned-up at this stage

    # truncate name to Genus species (or sp. if none)
    # append strain (if asked to do so)
    # append NCBI taxon_id or GCA/GCF as pseudo-strain
    # append accession number (if provided)
    my $full_id
        = $genus . ' '
        . ($species   ? ( $species                               ) : 'sp.')
        . ($strain    ? ( $keep_strain ? ('_' . $strain  ) : q{} ) :  q{} )
        . ($taxon_id  ? ('_' . $taxon_id                         ) :  q{} )
        . ($accession ? ('@' . $accession                        ) :  q{} )
    ;

    return $class->new( full_id => $full_id );
}



sub parse_ncbi_name {
    my $class = shift;
    my $org   = shift;

    # clean org name
    $org = $class->clean_ncbi_name($org);

    # split org name into max 3 components: genus, species and strain
    # strain is a greedy component for trailing information
    my ($genus, $species, $strain) = split /\s+/xms, $org, 3;

    # clean strain of unwanted prefices and characters (if any)
    $strain = $class->clean_strain($strain) if $strain;

    return ($genus, $species, $strain);
}


sub clean_ncbi_name {
    my $class = shift;
    my $org   = shift;

    # remove unwanted prefices
    $org =~ s{uncultured \s+ candidatus \s+}{}xmsgi;
    $org =~ s{candidatus \s+}{}xmsgi;

    # remove cf.
    $org =~ s{\b cf \.? \s+}{}xmsgi
        if $org =~ m/\b cf \.? \s+ \D+ \s+/xmsgi;
    # Note: delete only if followed by a word without digits (not a strain)
    # hence when 'cf' does not stand for a species itself

    # remove unwanted characters
    $org =~ s{[\[\]\']}{}xmsgi;

    return $org;
}


sub clean_strain {
    my $class  = shift;
    my $strain = shift;

    # remove unwanted prefices and characters (if any)
    $strain =~ s{\b substr \b}{}xmsgi;
    $strain =~ s{\b strain \b}{}xmsgi;
    $strain =~ s{\b subsp  \b}{}xmsgi;
    $strain =~ s{\b ssp\.    }{}xmsgi;
    $strain =~ s{\b str    \b}{}xmsgi;
    $strain =~ tr/A-Za-z0-9//cd;        # delete non-alphanumeric chars

    return $strain;
}

__PACKAGE__->meta->make_immutable;
1;

__END__

=pod

=head1 NAME

Bio::MUST::Core::SeqId - Modern and legacy MUST-compliant sequence id

=head1 VERSION

version 0.203490

=head1 SYNOPSIS

    # TODO

=head1 DESCRIPTION

    # TODO

=head1 METHODS

=head2 is_genus_only

=head2 is_doubtful

=head2 org

=head2 abbr_org

=head2 full_org

=head2 family_then_full_org

=head2 contam_org

=head2 foreign_id

=head2 nexus_id

=head2 new_with

=head2 parse_ncbi_name

=head2 clean_ncbi_name

=head2 clean_strain

=head1 AUTHOR

Denis BAURAIN <denis.baurain@uliege.be>

=head1 CONTRIBUTOR

=for stopwords Mick VAN VLIERBERGHE

Mick VAN VLIERBERGHE <mvanvlierberghe@doct.uliege.be>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2013 by University of Liege / Unit of Eukaryotic Phylogenomics / Denis BAURAIN.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
