
package TipJar::MTA::dateheader;

sub TIESCALAR{
	my $x;
	bless \$x;
};
@days=qw/Sun Mon Tue Wed Thu Fri Sat/;
@months=qw/Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec/;

sub FETCH{
   my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday)
   =  gmtime(time);
   #adjust date for printability:
   $year += 1900;
   # zero-pad time-of-day components
   $hour = substr("0$hour", -2);
   $min = substr("0$min", -2);
   $sec = substr("0$sec", -2);

   return
   "Date: $days[$wday], $mday $months[$mon] $year $hour:$min:$sec +0000";
};


package TipJar::MTA;




use strict;
use warnings;
use Carp;

tie my $dateheader, 'TipJar::MTA::dateheader';

use vars qw/
	$VERSION $MyDomain $interval $basedir
	$ReturnAddress $Recipient 
	$AgeBeforeDeferralReport
	$LogToStdout
	$OnlyOnce
	$LastChild
	$TimeStampFrequency
	$timeout
/;

$TimeStampFrequency = 200; # just under an hour at 17 seconds each

sub CRLF(){
	"\015\012"
};

use Fcntl ':flock'; # import LOCK_* constants
$interval = 17;
$AgeBeforeDeferralReport = 4 * 3600; # four hours

$VERSION = '0.10';

$SIG{CHLD} = 'IGNORE';

use Sys::Hostname;

$MyDomain = ( hostname() || 'sys.hostname.returned.false' );

my $time;
sub newmessage($);

sub OneWeek(){ 7 * 24 * 3600; };
sub SixHours(){ 6 * 3600; };

sub import{
	shift;	#package name
	$basedir = shift;
	$basedir ||= './MTAdir';
};

$LogToStdout = 0;

{
my $LogTime = 0;



sub mylog(@);
sub mylog(@){

	if (time - $LogTime > 30){
		$LogTime = time;
		mylog scalar localtime;

	};

	open LOG, ">>$basedir/log/current" or print(@_,"\n") and return;
	flock LOG, LOCK_EX or die "flock: $!";
	if($LogToStdout){
		seek STDOUT,2,0;
		print "$$ $Recipient ",@_;
		print "\n";
	}else{
		seek LOG,2,0;
		print LOG "$$ $Recipient ",@_;
		print LOG "\n";
	};
	flock LOG, LOCK_UN;	# flushes before unlocking
};

};

# END {
# 	mylog "exiting";
# };

sub run(){

	$Recipient = '';

	$SIG{ALRM} = sub { mylog 'TIMEOUT -- caught alarm signal'; 
			$timeout = 1;
			};

	-d $basedir
		or mkdir $basedir,0770
		or die "could not mkdir $basedir: $!" ;

	-w $basedir or croak "base dir <$basedir> must be writable!";

	# log dir contains logs (duh)
	-d "$basedir/log"
		or mkdir "$basedir/log",0770
		or die "could not mkdir $basedir/log: $!" ;

	# queue dir contains deferred messageobjects
	-d "$basedir/queue"
		or mkdir "$basedir/queue",0770
		or die "could not mkdir $basedir/queue: $!" ;

	# temp dir contains message objects under construction
	-d "$basedir/temp" or mkdir "$basedir/temp",0770
		or die "could not mkdir $basedir/temp: $!" ;


	{	# only one MTA at a time, so we can run this 
		# from cron
	open PID, ">>$basedir/temp/MTApid"; # "touch" sort of
	open PID, "+<$basedir/temp/MTApid"
		or die "could not open pid file '$basedir/temp/MTApid'";
	flock PID, LOCK_EX;
	chomp ( my $oldpid = <PID>);

	if ($oldpid and kill 0, $oldpid){
		print "$$ MTA process number $oldpid is still running\n";
		mylog "MTA process number $oldpid is still running";
		exit;
	};

	seek PID,0,0;
	print PID "$$\n";
	flock PID, LOCK_UN;
	close PID;
	}

	# immediate dir contains reprioritized deferred objects
	-d "$basedir/immediate" or mkdir "$basedir/immediate",0770
		or die "could not mkdir $basedir/immediate: $!" ;

	# endless top level loop
	mylog "starting fork-and-wait loop: will launch every $interval seconds.";
	my $count;
	for(;;){
		++$count % $TimeStampFrequency or
			mylog(time,": ",scalar( localtime)," ",$count);

		# new child drops out of the waiting loop
		$LastChild = fork or last;	
		if($OnlyOnce){
			mylog "OnlyOnce flag set to [$OnlyOnce]";
			return $OnlyOnce;
		};
		sleep $interval;
	};


	$time=time;

	# process new files if any
	opendir BASEDIR, $basedir;
	my @entries = readdir BASEDIR;
	for my $file (@entries){
		-f "$basedir/$file" or next;
		-s "$basedir/$file" or next;
		mylog "processing new message file $file";
		# expand and write into temp, then try to
		# deliver each file as it is expanded
		unless(open MESSAGE0, "<$basedir/$file"){
			mylog "Could not open $basedir/$file for reading";
			unless(unlink "$basedir/$file"){
				mylog "Could not unlink $basedir/$file";
			};
			next;
		};

		flock MESSAGE0, LOCK_EX|LOCK_NB or next;

		unless(unlink "$basedir/$file"){
			mylog "Could not unlink $basedir/$file";
			next;
		};

		my @MessData = (<MESSAGE0>);
		mylog @MessData;

		chomp(my $FirstLine = shift @MessData);
		mylog "from [[$FirstLine]]";
		# never mind $FirstLine =~ s/\s*<*([^<>\s]*).*$/$1/s;

		my @RecipList;
		my $Recip;

		for(;;){
			chomp(my $Recip = shift @MessData);
			unless (@MessData){
				die "no body in message";
			};
			# never mind $Recip =~ s/\s*<*([^<>\s]+\@[\w\-\.]+).*$/$1/s or last;
			mylog "for $Recip";
			$Recip =~ /\@/ or last;
			push @RecipList, $Recip;
			mylog "Recipients: @RecipList";
		};


		my $string = 'a';
		foreach $Recip (@RecipList){
			$string++;
			open TEMP, ">$basedir/temp/$time.$$.$string";
			print TEMP "$FirstLine\n$Recip\n",@MessData,"\n";
			close TEMP;
			rename 
			"$basedir/temp/$time.$$.$string",
			"$basedir/immediate/$time.$$.$string";
		};

	};

	# process all messages in immediate directory
	opendir BASEDIR, "$basedir/immediate"
		or die "could not open immediate dir: $!";
	@entries = readdir BASEDIR;
	for my $file (@entries){
		my $M = newmessage "$basedir/immediate/$file" or next;
		$M->attempt();	# will skip or requeue or delete
	};

	# reprioritize deferred messages
	my($minieon,$microeon) = $time =~ /^(\d+)(\d\d)\d\d$/;
	opendir QDIR, "$basedir/queue";
	my @directories =
	  grep { /^\d+$/ and $_ <= $minieon and -d "$basedir/queue/$_" }
	    readdir QDIR;

	for my $dir (@directories){	
		opendir QDIR, "$basedir/queue/$dir";
		my @directories2 =
		    grep { /\w/ } (readdir QDIR);

		unless (@directories2){
			mylog "removing directory queue/$dir";
			rmdir "$basedir/queue/$dir";
			next;
		};

		@directories2 = grep {
	 /\d/ and ($dir * 100 + $_) < ($minieon*100 + $microeon)
			} @directories2;

		#move files in these directories into the immediate directory
		for my $dir2 (@directories2){
			opendir QDIR, "$basedir/queue/$dir/$dir2";
			for (   readdir QDIR ){
				-f "$basedir/queue/$dir/$dir2/$_" or next;
				mylog "reprioritizing queue/$dir/$dir2/$_";
				rename "$basedir/queue/$dir/$dir2/$_", "$basedir/immediate/$_";
			};
			mylog "removing inner directory queue/$dir/$dir2";
			rmdir "$basedir/queue/$dir/$dir2";
		};
	};	
	exit;
};


# only one active message per process.
# (MESSAGE, $ReturnAddress, $Recipient) are all global.


sub newmessage($){
	#my $pack = shift;
	my $messageID = shift;
	-f $messageID or return undef;
	open MESSAGE, "<$messageID" or return undef;
	flock MESSAGE, LOCK_EX|LOCK_NB or return undef;
	chomp ($ReturnAddress = <MESSAGE>);
	chomp ($Recipient = <MESSAGE>);
	bless \$messageID;
};

use Socket;

{ no warnings; sub dnsmx($){
	# look up MXes for domain
	my @mxresults = sort {$a <=> $b} `dnsmx $_[0]`;
	# djbdns program dnsmx provides lines of form /\d+ $domain\n
	return map {/\d+ (\S+)/; $1} @mxresults;
};};

# my $calls;
# sub SOCKready(){
#         my $rin='';	
#         vec($rin,fileno('SOCK'),1) = 1;	
# 	my ($n, $tl) = select(my $r=$rin,undef,undef,0.25);
# 	print "$calls\n";
# 	$calls++ > 200 and exit;
# 	return $n;
# };

my $CRLF = CRLF;

sub eofSOCK(){

	seek SOCK, 0, 1;
	my $syserr = $!;
	$syserr =~ /^Illegal seek/ and return 0;

	# a seek on a closed socket gives a different error.

	mylog $syserr;
	return 1;
};

sub getresponse($){

	# mylog "sending: [$_[0]]";

	if(eofSOCK){
		mylog "problem with SOCK";
		return undef;
	};

	$timeout = 0;
	alarm 130;
	unless(print SOCK  "$_[0]$CRLF"){
		mylog "print SOCK: $!";
		return undef;
	};
	# mylog "sent $_[0]";

	my ($dash,$response) = ('-','');
	while($dash eq '-'){
		my $letter;
		my @letters;
		my $i=0;
		my $more = 1;
		my $BOL = 1;	# "beginning of line"
		do {
			if($timeout){
				mylog "timeout in getresponse";
				return undef;
			};
			if(eofSOCK){
				mylog "eofSOCK";
				return undef;
			};
			sysread(SOCK,$letter,1);
			if ($letter eq "\r" or $letter eq "\n"){
				$more = $BOL;
			}else{
				$BOL = 0;
				if(length($letter)){
					$letters[$i++] = $letter;
					# mylog @letters;
				}else{
					sleep 1;
				};
			};
		} while( $more );

		my $line = join('',@letters);

	#	mylog "received: [$line]";
		$response .= $line;
		($dash) = $line =~ /^\d+([\-\ ])/;
	};
	$response;
};


sub attempt{
	# deliver and delete, or requeue; also send bounces if appropriate
	my $message = shift;
	mylog "Attempting [$ReturnAddress] -> [$Recipient]";
	# Message Data is supposed to start on third line

	my ($Domain) = $Recipient =~ /\@([^\s>]+)/ or goto GoodDelivery;

	my @dnsmxes;
	@dnsmxes = dnsmx($Domain);
	my $dnsmx_count = @dnsmxes;
	mylog "[[$Domain]] MX handled by @dnsmxes";
	unless ( @dnsmxes ){
		mylog "requeueing due to empty dnsmx result";
		goto ReQueue_unconnected;
	};
	my $Peerout;

	my $line;

	TryAgain:

	while($Peerout = shift @dnsmxes){
		mylog "attempting $Peerout";

		# connect to $Peerout, smtp
		my @GHBNres;
		unless ( @GHBNres = gethostbyname($Peerout)){
			if ($dnsmx_count == 1 and
			    $Peerout eq $Domain){
				mylog $line="Apparently there is no valid MX for $Domain";
				goto Bounce;
			};
			next;
		};
		my $iaddr = $GHBNres[4]	or next;
        	my $paddr   = sockaddr_in(25, $iaddr);
        	socket(SOCK,
			PF_INET,
			SOCK_STREAM,
			getprotobyname('tcp'))
			or die "$$ socket: $!";

		connect(SOCK, $paddr)  || next ;
		mylog "connected to $Peerout";
         	my $oldfh = select(SOCK); $| = 1; select($oldfh);
		goto SMTPsession;

	};

	mylog "Unable to establish SMTP connection to $Domain MX";
	goto ReQueue_unconnected;


	# talk SMTP
	SMTPsession:	

        # expect 220
        alarm 60;
        eval { $line = <SOCK>; };
	if($@){
		mylog $@;
		close SOCK;
		goto TryAgain;
	};
        mylog "$line";
	unless($line =~ /^2/){
		mylog "Weird greeting: [$line]";
		close SOCK;
		goto TryAgain;
	};

        # print SOCK "HELO $MyDomain",CRLF;
        # expect 250
        $line = getresponse "HELO $MyDomain" or goto TryAgain;
	mylog $line;
        unless($line =~ /^250 /){
		mylog "peer not happy with HELO: [$line]";
		close SOCK;
		goto TryAgain;
	};
        # print SOCK "RSET",CRLF;
        $line = getresponse "RSET" or goto TryAgain;
        # expect 250
        # $line = getresponse;
	# mylog "RSET and got [$line]";
        unless($line =~ /^250 /){
		mylog "peer not happy with RSET: [$line]\n";
		close SOCK;
		goto TryAgain;
	};


	# remove angle brackets if any
	$ReturnAddress =~ s/^.*<//;
	$ReturnAddress =~ s/>.*$//;

        $line = getresponse "MAIL FROM: <$ReturnAddress>"  or goto TryAgain;
        mylog "$line";
        unless($line =~ /^[2]/){
		mylog "peer not happy with return address: [$line]";
		if ($line =~ /^[4]/){
			mylog "requeueing";
			goto ReQueue;
		};
		if ($line =~ /^[5]/){
			mylog "bouncing";
			goto Bounce;
		};
		mylog "and response was neither 2,4 or 5 coded.";
		goto TryAgain;
	};

        # print SOCK "RCPT TO: <$Recipient>\r\n";
        # expect 250
	
	# remove angle brackets if any
	$Recipient =~ s/^.*<//;
	$Recipient =~ s/>.*$//;

        $line = getresponse "RCPT TO: <$Recipient>" or goto TryAgain;
        unless($line =~ /^2/){
		mylog "peer not happy with recipient: [$line]";
		if ($line =~ /^4/){
			mylog "requeueing";
			goto ReQueue;
		};
		if ($line =~ /^5/){
			mylog "bouncing";
			goto Bounce;
		};
		mylog "reporting noncompliant SMTP peer [$Peerout]";
		goto TryAgain;
	};


        # print SOCK "DATA\r\n";
        # expect 354
        $line = getresponse 'DATA' or goto TryAgain;
        unless($line =~ /^354 /){
		mylog "peer not happy with DATA: [$line]";
		if ($line =~ /^4/){
			mylog "requeueing";
			goto ReQueue;
		};
		if ($line =~ /^5/){
			mylog "bouncing";
			goto Bounce;
		};
		mylog "reporting noncompliant SMTP peer [$Peerout]";
		goto TryAgain;
	};
	my $linecount;
	my $bytecount;
	while (<MESSAGE>){
		$linecount++;
		$bytecount += length;
		chomp;
		eval{
        	alarm 60;
		if ($_ eq '.'){
			print SOCK "..\r\n" or die $!;
		}else{
			print SOCK $_,"\r\n" or die $!;
		};
		};
		if ($@){
			mylog $@;
			goto TryAgain;
		};
	};
	# print SOCK ".\r\n";
        # expect 250
	mylog "$linecount lines ($bytecount chars) of message data";
        $line = getresponse '.' or goto TryAgain;
        unless($line =~ /^2/){
		mylog "peer not happy with message body: [$line]";
		if ($line =~ /^4/){
			mylog "requeueing";
			goto ReQueue;
		};
		if ($line =~ /^5/){
			mylog "bouncing";
			goto Bounce;
		};
		mylog "reporting noncompliant SMTP peer [$Peerout]";
		goto TryAgain;
	};

	goto GoodDelivery;

	ReQueue:
	mylog getresponse 'QUIT';
	close SOCK;
	ReQueue_unconnected:
	$message->requeue($line);
	return undef;

	Bounce:

	$ReturnAddress =~ /\@/ or goto GoodDelivery; #suppress doublebounces
	my $filename = join '.',time,$$,'bounce',rand(10000000);
	open BOUNCE, ">$basedir/temp/$filename";
	print BOUNCE <<EOF;
<>
$ReturnAddress
$dateheader
From: MAILER-DAEMON
To: $ReturnAddress
Subject: delivery failure to <$Recipient>
Content-type: text/plain

While connected to SMTP peer $Peerout,
the $MyDomain e-mail system received the error message

$line

which indicates a permanent error.
The first hundred and fifty lines of the message follow below:
-------------------------------------------------------------
EOF

	seek(MESSAGE,0,0);
	for(1..150){
		defined(my $lin = <MESSAGE>) or last;
		print BOUNCE $lin;
	};
	close BOUNCE;
	rename "$basedir/temp/$filename","$basedir/immediate/$filename";

	GoodDelivery:
	mylog getresponse 'QUIT';
	close SOCK;
	return unlink $$message;	# "true"

};

sub requeue{
	my $message = shift;
	my $reason = shift;
	my ($fdir,$fname) = $$message =~ m#^(.+)/([^/]+)$#;
	my @stat = stat($$message);
	my $age = time - $stat[9];

	if ($age > OneWeek){

		$ReturnAddress =~ /\@/ or goto unlinkme; #suppress doublebounces
		my $filename = join '.',time,$$,'bounce',rand(10000000);
		open BOUNCE, ">$basedir/temp/$filename";
		print BOUNCE <<EOF;
<>
$ReturnAddress
$dateheader
From: MAILER-DAEMON
To: $ReturnAddress
Subject: delivery failure to <$Recipient>
Content-type: text/plain

A message has been enqueued for delivery for over a week,
the $MyDomain e-mail system is deleting it.

Final temporary deferral reason:
$reason

The first hundred and fifty lines of the message follow below:
-------------------------------------------------------------
EOF

		seek(MESSAGE,0,0);
		for(1..150){
			defined(my $lin = <MESSAGE>) or last;
			print BOUNCE $lin;
		};
		close BOUNCE;
		rename "$basedir/temp/$filename","$basedir/immediate/$filename";

		unlinkme:
		unlink $$message;
	};

	if (
		$age > $AgeBeforeDeferralReport and
		$reason and
		$ReturnAddress =~ /\@/ # suppress doublebounces
	){
	my $filename = join '.',time,$$,'bounce',rand(10000000);
	open BOUNCE, ">$basedir/temp/$filename";
	print BOUNCE <<EOF;
<>
$ReturnAddress
$dateheader
From: MAILER-DAEMON
To: $ReturnAddress
Subject: delivery deferral to <$Recipient>
Content-type: text/plain

The $MyDomain e-mail system is not able to deliver
a message to $Recipient right now.
Attempts will continue until the message is over a week old.

Temporary deferral reason:
$reason

The first hundred and fifty lines of the message follow below:
-------------------------------------------------------------
EOF

	seek(MESSAGE,0,0);
	for(1..150){
		defined(my $lin = <MESSAGE>) or last;
		print BOUNCE $lin;
	};
	close BOUNCE;
	rename "$basedir/temp/$filename","$basedir/immediate/$filename";

		$message->deferralmessage("Will keep attempting until message is over a week old");
	};

	my $futuretime = int(time + 100 + ( $age * ( 3 + rand(2)) / 4));
	# print "futuretime will be $futuretime\n";
	my ($dir,$subdir) = ($futuretime =~ m/^(\d+)(\d\d)\d\d$/);
	# print "dir,subdir is $dir,$subdir\n";
	
	-d "$basedir/queue/$dir"
	or mkdir "$basedir/queue/$dir", 0777
	or croak "$$ Permissions problems: $basedir/queue/$dir [$!]\n";

	-d "$basedir/queue/$dir/$subdir"
	or mkdir "$basedir/queue/$dir/$subdir", 0777
	or croak "$$ Permissions problems: $basedir/queue/$dir/$subdir [$!]\n";

	# $fname = FIXME -- something to do with the domain?

	rename $$message, "$basedir/queue/$dir/$subdir/$fname";
	mylog "message queued to $basedir/queue/$dir/$subdir/$fname";


};


1;
__END__

=head1 NAME

TipJar::MTA - outgoing SMTP with exponential random backoff.

=head1 SYNOPSIS

  use TipJar::MTA '/var/spool/MTA';	# must be a writable -d
					# defaults to ./MTAdir
  $TipJar::MTA::interval='100';		# the default is 17
  $TipJar::MTA::TimeStampFrequency='35';	# the default is 200
  $TipJar::MTA::AgeBeforeDeferralReport=7000;	# default is 4 hours
  $TipJar::MTA::MyDomain='peanut.af.mil';	# defaults to `hostname`
					# And away we go,
  TipJar::MTA::run();			# logging to /var/spool/MTA/log/current
  

=head1 DESCRIPTION

On startup, we identify the base directory and make sure we
can write to it, check for and create a few subdirectories,
check if there is an MTA already running and stop if there is,
so that TipJar::MTA can be restarted from cron.

We are not concerned with either listening on port 25 or with
local delivery.  This module implements outgoing SMTP with
exponentially deferred random backoffs on temporary failure.
Future delivery scheduling is determined by what directory
a message appears in.  File age, according to C<stat()>, is
used to determine repeated deferral.

Every C<$interval> seconds,  we fork a child process.

A new child process first goes through all new outbound messages
and expands them into individual messages 
and tries to send them.  New messages are to be formatted with the return
address on the first line, then recipient addresses on subsequent lines,
then a blank line (rather, a line with no @ sign), then the body of the message.
The L<TipJar::MTA::queue>
module will help compose such files if needed.

Messages are rewritten into multiple messages when they are for
multiple recipients, and then attempted in the order that the
recipients appeared in the file.

After attempting new messages, a child process attempts all messages
in the "immediate" directory.

After attempting all messages in the immediate directory, a child
process moves deferred messages whose times have arrived into the
immediate directory for processing by later children.

Deferred messages are stored in directories named according
to when a message is to be reattempted. Reattempt times are
assigned at requeueing time to be now plus between three and five
quarters of the message age. Messages more than a week old are
not reattempted.  An undeliverable message that got the maximum
deferrment after getting attempted just shy of the one-week deadline
could conceivably be attempted for the final time fifteen and three
quarters days after it was originally enqueued.  Then it would be
deleted.

The format for new messages is as follows:

=over 4

=item return address

The first line of the message contains the return address.  It
can be bare or contained in angle-brackets.  If there are
angle brackets, the part of the line not in them is discarded.

=item recipient list

All recipients are listed each on their own line.  Recipients
must have at-signs in them.

=item blank line

The first line (after the first line) that does not contain a
@ symbol marks the end of the recipients.  We are not concerned
with local delivery.

=item data

Follow the routing information with the data, starting with
header lines.

=back

=head2 EXPORT

None.


=head1 DEPENDENCIES

the C<dnsmx()> function uses the dnsmx program from the djbdns tool package:
it is abstracted into a function for easy replacement with your preferred
MX lookup tool.

The file system holding the queue must support reading from a file handle
after the file has been unlinked from its directory.  If your computer
can't do this, see the spot in the code near the phrase "UNLINK ISSUE"
and follow the instructions.

For that matter, we also generate some long file names with lots
of dots in them, which could conceivably not be portable.

=head1 HISTORY

=over 8

=item 0.03 17 April 2003

threw away some inefficient archtecture ideas, such as
per-domain queues for connection reuse, in order to have
a working system ASAP.  Testing kill-zero functionality in
test script.

=item 0.04 20 April 2003

logging to $basedir/log/current instead of stdout, unless
$LogToStdout is true.
$AgeBeforeDeferralReport
variable to suppress deferral
bounces when a message has been queued for less than an
interval.


=item 0.05 22 April 2003

slight code and documentation cleanup

=item 0.06 6 May 2003

Testing, testing, testing!  make test on TipJar::MTA::queue before
making test on this module, and you will send me two e-mails.  Now
using Sys::Hostname instead of `hostname` and gracefully handling
absolutely any combination of carriage-returns and line-feeds
as valid line termination.

=item 0.07 1 June 2003

Wrapped all reads and writes to the SMTP socket in C<eval> blocks,
and installed a ALRM signal handler, for better handling of time-out 
conditions.  Also added a $TipJar::MTA::TimeStampFrequency variable
which is how many iterations of the main fork-and-send loop to make
before logging a timestamp.  The default is 200.

=item 0.08 10 June 2003

minor cleanup.

=item 0.09 12 June 2003

AOL and Yahoo.com and who knows how many other sticklers require
angle brackets around return addresses and recipients.  Improved
handling of MXes that we cannot connect to, by defining a
C<ReQueue_unconnected>
entry point in addition to the C<ReQueue> one that we had already..

=item 0.09 19 June 2003

We now bounce mail to domains that ( have no MX records OR there
is only one MX record and it is the same as the domain name ) AND
we could not resolve the one name. Previously it had been given
the full benefit of the doubt.


=item 0.10 24 June 2003

Better handling of slow peers.  C<sysread> does not block and C<eof> cannot
be used on sockets, did you know that?  We check for socket openness by
seeking and looking at the text of the resulting error message.  Instead
of using an OO interface.  Also timeouts are now handled with a global
variable instead of C<die> because C<die>ing from a signal handler does
not get appear to get caught by an eval enclosing the point of execution
at the time of the signal.  Is this a bug?

Anyway TipJar::MTA.pm has been handling outgoing production e-mail for
a couple weeks now.

=back

=head1 To-do list and Known Bugs

Patches are welcome.

=over 4

=item log rolling

there is no rotation of the log in the C<mylog()> function.
C<mylog> does
repoen the file by name on every logging event, though.  Rewriting mylog to
use L<Unix::Syslog> or L<Sys::Syslog> would be cool, but would add dependencies.
Mailing the log to the postmaster every once in a while is easy enough
to do from L<cron>.

=item connection reuse and per-domain queues

have deferred messages organized by peer, when the
deferral is because of connection problems, possibly by grouping the
"immediate" messages by domain so we can reuse a connection
instead of trying to make a new connection

=item ESMTP

take advantage of post-RFC-821 features

=item QMTP

use QMTP when available.

=item local deliveries

add MBOX and MailDir deliveries and some kind of
configuration interface

=back

=head1 AUTHOR

David Nicol, E<lt>davidnico@cpan.orgE<gt>

=head1 SEE ALSO

L<TipJar::MTA::queue>.

=cut
