#!/usr/bin/perl

# $Id: create_object_security.pl,v 1.2 2003/06/10 15:49:13 lachoy Exp $

# create_object_security.pl
#
#   Take a class name and scope information and create security
#   settings for all objects in the class. (See pod)

use strict;
use Getopt::Long             qw( GetOptions );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX DEBUG LOG );
use OpenInteract2::Setup;
use SPOPS::Secure            qw( :level :scope );

my %VALID_SCOPE = (
   world => SEC_SCOPE_WORLD,
   user  => SEC_SCOPE_USER,
   group => SEC_SCOPE_GROUP );

my %VALID_LEVEL = (
    none  => SEC_LEVEL_NONE,
    read  => SEC_LEVEL_READ,
    write => SEC_LEVEL_WRITE );

{
    $| = 1;
    my ( $OPT_scope, $OPT_scope_id, $OPT_level, $OPT_class, $OPT_iterator,
         $OPT_website_dir, $OPT_where, $OPT_verbose, $OPT_help );
    GetOptions( 'scope=s'       => \$OPT_scope,
                'scope_id=s'    => \$OPT_scope_id,
                'level=s'       => \$OPT_level,
                'class=s'       => \$OPT_class,
                'where=s'       => \$OPT_where,
                'iterator=s'    => \$OPT_iterator,
                'verbose'       => \$OPT_verbose,
                'help'          => \$OPT_help,
                'website_dir=s' => \$OPT_website_dir );
    if ( $OPT_help ) {
        print usage();
        exit(0);
    }
    unless ( $OPT_website_dir ) {
        $OPT_website_dir = $ENV{OPENINTERACT2};
    }

    # Be sure all the necessary parameters have been issued before we
    # create the context

    my @errors = ();
    unless ( -d $OPT_website_dir ) {
        push @errors, "* --website_dir or 'OPENINTERACT2' env must be set to website dir\n";
    }
    unless ( $OPT_scope =~ /^(world|group|user)$/ ) {
        push @errors, "* --scope must be set to 'world', 'group', or 'user'\n";
    }
    if ( $OPT_scope ne 'world' and ! $OPT_scope_id ) {
        push @errors, "* --scope_id must be set if the --scope is set to 'group' or 'user'\n";
    }
    unless ( $OPT_level =~ /^(none|read|write)$/ ) {
        push @errors, "* --level must be set to 'none', 'read' or 'write'\n";
    }
    unless ( $OPT_class ) {
        push @errors, "* --class must be set to the class of the object you want to\n",
                      "set security for (e.g., 'OpenInteract2::News')\n";
    }

    my $scope = $VALID_SCOPE{ lc $OPT_scope };
    unless ( $scope ) {
        push @errors, "Invalid scope given! (Scope: $OPT_scope)\n";
    }

    my $level = $VALID_LEVEL{ lc $OPT_level };
    unless ( $level ) {
        push @errors, "Invalid level given! (Level: $OPT_level)\n";
    }

    if ( scalar @errors ) {
        die "Incomplete parameters.\n\n", join( "\n", @errors ), "\n";
    }

    @errors = ();

    # Turn off debugging or we'll blind people with messages (before
    # AND after creation)

    CTX->_SET_DEBUG(0) unless ( $OPT_verbose );
    OpenInteract2::Context->create({ website_dir => $OPT_website_dir });
    CTX->_SET_DEBUG(0) unless ( $OPT_verbose );

    # Now that the context is created, be sure the capabilities of the
    # class match what is asked of it.

    unless ( $OPT_class->can( 'fetch' ) ) {
        push @errors, "* It doesn't appear that ($OPT_class) is a valid class for an\n",
                      "SPOPS class defined in your OpenInteract setup.\n";
    }

    unless ( $OPT_class->isa( 'SPOPS::Secure' ) ) {
        warn "It doesn't appear that ($OPT_class) is using security. I can create\n",
             "security settings for it but they won't get used until the class\n",
             "uses security. To do so, ensure that 'SPOPS::Secure' or a child of\n",
             "it is in the 'isa' of the class.\n";
    }
    if ( $OPT_iterator ne 'no' and ! $OPT_class->can( 'fetch_iterator' ) ) {
        warn  "It doesn't appear that ($OPT_class) has the ability to\n",
              "retrieve an iterator. I'll modify the call to return a list\n",
              "of objects. The action should still work, it just might take\n",
              "little more time and memory. In the future, you might want to\n",
              "set '--iterator=no'.\n";
        $OPT_iterator = 'no';
    }

    if ( scalar @errors ) {
        die "Incomplete parameters.\n\n", join( "\n", @errors ), "\n";
    }

    $OPT_iterator ||= 'yes';

    my $default_objects = CTX->server_config->{default_objects};

    if ( $OPT_scope_id and $default_objects->{ $OPT_scope_id } ) {
        $OPT_scope_id = $default_objects->{ $OPT_scope_id };
    }

    my $begin_time = time;
    $OPT_verbose && print "Results for each object:\n";

    my ( $object_store );
    if ( $OPT_iterator eq 'yes' ) {
        $object_store = $OPT_class->fetch_iterator(
                              { skip_security => 1,
                                column_group  => '_id_field',
                                where         => $OPT_where });
    }
    else {
        $object_store = $OPT_class->fetch_group(
                              { skip_security => 1,
                                column_group  => '_id_field',
                                where         => $OPT_where });
    }

    my ( $count, $failure ) = ( 0, 0 );

    my $sec_class = CTX->security_object;

    while ( my $object = get_from_store( $object_store ) ) {
        $sec_class->new({ class          => $OPT_class,
                          object_id      => $object->id,
                          scope          => $scope,
                          scope_id       => $OPT_scope_id,
                          security_level => $level } )->save();
        if ( $@ ) {
            $OPT_verbose && print "  FAIL: ", $object->id, " ($@)\n";
            $failure++;
        }
        else {
            $OPT_verbose && print "  OK: ", $object->id, "\n";
        }
        $count++;
        unless ( $OPT_verbose ) {
            printf( '%8d', $count ) if ( $count % 250 == 0 );
            print "\n"              if ( $count % 2000 == 0 );
        }
    }
    $OPT_verbose && print "\nComplete!\n",
                          "Records:  $count\n",
                          "Success:  ", $count - $failure, "\n",
                          "Failed:   $failure\n",
                          "Started:  ", scalar localtime( $begin_time ), "\n",
                          "Finished: ", scalar localtime, "\n";
}


# Use either an iterator or a list

sub get_from_store {
    my ( $store ) = @_;
    return undef unless ( $store );
    if ( ref $store eq 'ARRAY' ) {
        return undef unless ( scalar @{ $store } );
        return shift @{ $store };
    }
    return $store->get_next;
}


sub usage {
    return <<'USAGE';
create_object_security.pl - Create security settings for all objects
of a particular class

Usage:

 create_object_security.pl [options]

Options:

 --class        class of object (e.g., 'OpenInteract2::News')

 --level        Security level to set (none|read|write)

 --scope        Scope to set security for (world|group|user)

 --scope_id     ID of user/group -- ignored for 'world'; you can use
                any key found in the 'default_objects' hash of your
                server configuration (e.g., 'site_admite_group')

 --website_dir  Path to OI website (or set \$ENV{OIWEBSITE2})

 --where        SQL statement to restrict results (optional)

 --iterator     Whether to use an Iterator to return results 
                (optional; default 'yes')

 --verbose      Display verbose messages. We normally turn off
                debugging since we can potentially deal with so many
                objects at once, but if you specify this your
                debugging level will not be touched.

 --help         Display this message

Example:

Create security setting so that all members of group with ID 3 get
write permission for all 'OpenInteract2::News' objects.

 $ perl create_object_security.pl --class=OpenInteract2::News \
                                  --level=write
                                  --scope=group \
                                  --scope_id=3 \
                                  --website_dir=/home/httpd/mysite.com
USAGE
}

__END__

=head1 NAME

create_object_security.pl - Batch-create security settings for objects

=head1 SYNOPSIS

 # Create security setting so that all members of group with ID 3 get
 # write permission for all 'OpenInteract2::News' objects.
 
 perl create_object_security.pl --class=OpenInteract2::News \
                                --level=write
                                --scope=group \
                                --scope_id=3 \
                                --website_dir=/home/httpd/mysite.com

=head1 DESCRIPTION

Creates security settings for all objects in a class. Run with
'--help' to get syntax and more info.

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>

