package OpenInteract::Handler::Security;

# $Id: Security.pm,v 1.4 2001/07/13 15:03:27 lachoy Exp $

use strict;
use Data::Dumper             qw( Dumper );
use SPOPS::Secure            qw( :level :scope );
use SPOPS::Secure::Hierarchy qw( $ROOT_OBJECT_NAME );

@OpenInteract::Handler::Security::ISA     = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::Security::VERSION = sprintf("%d.%02d", q$Revision: 1.4 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Security::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Security::default_method    = 'simple_show';
@OpenInteract::Handler::Security::forbidden_methods = ();
%OpenInteract::Handler::Security::security          = ( 
  simple_show    => SEC_LEVEL_WRITE,
  simple_edit    => SEC_LEVEL_WRITE,
  hierarchy_specify => SEC_LEVEL_WRITE,
  hierarchy_show => SEC_LEVEL_WRITE,
  show           => SEC_LEVEL_WRITE,
  edit           => SEC_LEVEL_WRITE,
);

use constant MAIN_SCRIPT => '/Security';

# Deprecated methods

sub simple_show { return show( @_ ) }
sub simple_edit { return edit( @_ ) }


# Blah -- need to modify the UI (or something -- lookup_module?) to
# call a template directly

sub hierarchy_specify {
    my ( $class, $p ) = @_;
    return OpenInteract::Request->instance->template->handler( 
                                      {}, {}, { db      => 'hierarchy_specify',
                                                package => 'base_security' } );
}


sub hierarchy_show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $object = $p->{object};
    my ( $object_class, $oid );
    if ( ref $object ) {
        $object_class = ref $object;
        $oid          = $object->id;
    }
    else {
        $object_class = $R->apache->param( 'object_class' );
        $oid          = $R->apache->param( 'oid' ) || $R->apache->param( 'object_id' );
    }

    # Retrieve the security levels so we can display them
 
    my ( $track, $first, $check_list ) = 
            SPOPS::Secure::Hierarchy->get_hierarchy_levels({ 
                class                 => $object_class, 
                object_id             => $oid,
                security_object_class => $R->security_object,
		        user                  => $R->{auth}->{user},
                group                 => $R->{auth}->{group} });

    my @check_list_items = map { { oid => $_, security_defined => $track->{ $_ } } } 
                           @{ $check_list };

    my $params = { main_script  => MAIN_SCRIPT,
                   object       => $object,
                   object_class => $object_class,
                   oid          => $oid,
                   check_list   => \@check_list_items,
                   ROOT_OBJECT_NAME => $ROOT_OBJECT_NAME };
    $R->{page}->{title} = 'Object Hierarchy Security';
    return $R->template->handler( {}, $params, 
                                { db      => 'hierarchy_security', 
                                  package => 'base_security' } ); 
}


sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $object = $p->{object};
    my $instantiate = $R->apache->param( 'instantiate' );
    my ( $object_class, $oid );

    # Ensure we have a class and oid since this isn't meant to be
    # used with class-level (e.g., modular) security
    #
    # Note that if we are told not to instantiate an object, we don't. We
    # just use the object class as a categorical identifier and oid as a
    # specific one.

    unless ( $object ) {
        $object_class = $R->apache->param( 'object_class' );
        $oid          = $R->apache->param( 'oid' ) || $R->apache->param( 'object_id' );
        if ( $instantiate ne 'no' ) {
            $object = eval { $object_class->fetch( $oid ) };
            if ( ! $object or $@ ) {
                return '<h2>Cannot Create Object</h2>' .
                       '<p>Cannot create object using information given.';
            }
        }
    }

    if ( $object ) {
        unless ( $object->isa( 'SPOPS::Secure' )  ) {
            return <<CANNOTSECURE;
<h2>Object Not Securable</h2>
<p>This object (class $object_class) is not a subclass of the Security
implementation, meaning it is not under security protection.  Please
check its <tt>spops.perl</tt> configuration and ensure that
<tt>SPOPS::Secure</tt> is in the 'isa' tag.
CANNOTSECURE
        }
        $R->DEBUG && $R->scrib( 1, "Using object ", ref $object, " (", $object->id, ") for security box editor." ); 
    }
    else {
        $R->DEBUG && $R->scrib( 1, "Using class $object_class ($oid) for security box editor." ); 
    }

    my $params = { main_script  => MAIN_SCRIPT, 
                   object       => $object,
                   object_class => $object_class,
                   oid          => $oid,
                   instantiate  => $instantiate };

    # Now fetch the security info -- we want to see who already has
    # security set so we can display that information next to the name
    # of the group/user or world in the listing

    my $security = eval { $R->security_object->fetch_by_object( 
                                 $object, 
							     { class     => $object_class, 
                                   object_id => $oid,
							       group     => 'all' } ) };
    $R->DEBUG && $R->scrib( 1, "Security fetched: ", Dumper( $security ) );;
    if ( $@ ) {
        my $ei = OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 308 } );
        return "<h2>Error!</h2><p>Error retrieving security: $ei->{system_msg}";
    }

    # First item in the scope is the WORLD setting   

    my $world_level = $security->{ SEC_SCOPE_WORLD() };
    my @scopes = ({ scope => SEC_SCOPE_WORLD, 
                    name  => 'World',
                    level => $world_level });

    # Retrieve groups and match with security level

    my $group_list = eval { $R->group->fetch_group( { order => 'name' } ) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 403 } );
        $group_list = [];
    }

    foreach my $group ( @{ $group_list } ) {
        my $gid = $group->{group_id};
        my $level = $security->{ SEC_SCOPE_GROUP() }->{ $gid };
        push @scopes, { scope => SEC_SCOPE_GROUP, 
                        scope_id => $gid,
                        name  => $group->{name}, 
                        level => $level };
    }

    # NOTE: See discussion and code at bottom of file regarding
    # user-level security (if you're interested)

    $params->{scope_list} = \@scopes;
    $R->{page}->{title} = 'Edit Object Security';
    return $R->template->handler( {}, $params, 
                                  { db      => 'assign_object_security',
                                    package => 'base_security' } ); 
}

# Edit security for a particular object -- note that the widget
# currently only supports setting one level for many scopes rather
# than many levels for many scopes.

sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance; 

    my $level       = $R->apache->param( 'level' );
    my $obj_class   = $R->apache->param( 'object_class' );
    my $oid         = $R->apache->param( 'oid' ) || $R->apache->param( 'object_id' );
    my $instantiate = $R->apache->param( 'instantiate' );
    my @scope       = map { [ split /;/, $_ ] } $R->apache->param( 'scope' );

    my ( $total, $success );
    my $secure_class = 'SPOPS::Secure';
    foreach my $info ( @scope ) {
        $total++;
        $R->DEBUG && $R->scrib( 1, "Trying to set security ($level) for $obj_class ($oid) in scope $info->[0] ($info->[1])" );
        if ( $level eq 'clear' ) {
            $success += eval { $secure_class->remove_item_security({ 
                                        security_object_class => $R->security_object,
                                        class     => $obj_class, 
                                        object_id => $oid,
                                        scope     => $info->[0], 
                                        scope_id  => $info->[1] }) };
        }
        else {
            $success += eval { $secure_class->set_item_security({ 
                                        security_object_class => $R->security_object,
                                        class     => $obj_class, 
                                        object_id => $oid,
                                        scope     => $info->[0], 
                                        scope_id  => $info->[1],
                                        level     => $level }) };
        }
        if ( $@ ) {
            my $ei = OpenInteract::Error->set( SPOPS::Error->get );
            $R->scrib( 0, "Error trying to set security!\n$@\n", Dumper( $ei ) ); 
            $R->throw({ code => 406 });
        }
    }
    my $return_url = $R->apache->param( 'return_url' ) || 
                     MAIN_SCRIPT . "/show/?object_class=$obj_class;oid=$oid;instantiate=$instantiate";
    my ( $action_class, $action_method ) = $R->lookup_action( 'redirect' );
    return $action_class->$action_method( { url => $return_url } );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Security - Process changes to security made by users

=head1 SYNOPSIS

=head1 DESCRIPTION

Handler to display and process the results of object-level security setting.

=head1 METHODS

B<hierarchy_show>

Feeds the widget that displays the parents of a particular object and
whether each one has security currently defined or not.

B<show>

Feeds the widget that allows users to edit security on a single object
or item.

B<edit>

Processes the results of the 'show' page.

=head1 NOTES

B<Deprecated: simple_*>

The C<simple_show> and C<simple_edit> methods are now deprecated. Use
C<show> and C<edit> instead.

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut

# From simple_show():
#
# NOTE - We've commented this out while considering different options
# for assigning security to users. For a site with even more than a
# handful of users, plowing through a list of them is going to be a
# pain not only for the user but also for the browser (!) Until we
# have a good reason to do so, this tool will only assign security for
# WORLD and the different groups created in the system. In any case,
# it's better to do per-group security anyway :)
#
# Perhaps we can extend the tool to allow a user to do a username
# lookup (login_name/last_name) and add them to the list. This would
# put the burden on the administrator to know who he/she is assigning
# rights to beforehand.

# # Now fetch the users and match each up with a security level
# my $user_list  = eval { $R->user->fetch_group( { order => 'login_name' } ) };
# if ( $@ ) {
#   OpenInteract::Error->set( SPOPS::Error->get );
#   $R->throw( { code => 403 } );
#   $user_list = [];
# }

# foreach my $user ( @{ $user_list } ) {
#   my $uid = $user->{user_id};
#   my $level = $security->{ SEC_SCOPE_USER() }->{ $uid };
#   push @scopes, { scope => SEC_SCOPE_USER, scope_id => $uid,
#                   name => "U: $user->{login_name} ($LEVEL{ $level })" };
# }

