package OpenInteract::SiteTemplate;

# $Id: SiteTemplate.pm,v 1.2 2001/05/30 03:13:03 lachoy Exp $

use strict;
use Data::Dumper qw( Dumper );

$OpenInteract::SiteTemplate::VERSION = sprintf("%d.%02d", q$Revision: 1.2 $ =~ /(\d+)\.(\d+)/);

sub fetch_by_name {
  my ( $class, $name, $package ) = @_;
  $name = lc $name;
  my $R = OpenInteract::Request->instance;
  $R->DEBUG && $R->scrib( 1, "Trying to retrieve template for name ($name) and ",
                "package ($package)" );

  my $where = 'name = ?';
  my @value = ( $name );
  if ( $package ) {
    $where   .= ' AND package = ?';
    push @value, $package;
  }
  my $obj = eval { $class->fetch_group({ 
                       where => $where, value => \@value 
                   }) };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $OpenInteract::Error::user_msg = "Cannot retrieve site template for name " .
                                     "<<$name>> and package <<$package>>";
    die $OpenInteract::Error::user_msg;
  }
  if ( scalar @{ $obj } > 1 ) {
    $R->scrib( 0, "Too many items found in response to name <<$name>> ",
                  "and package <<$package>>! Returning ",
                  "$obj->[0]->{name} - $obj->[0]->{package}" );
  }
  $R->DEBUG && $R->scrib( 2, "Template object(s) found:\n", Dumper( $obj ) );
  return $obj->[0];
}

1;

__END__

=pod

=head1 NAME

OpenInteract::SiteTemplate - Object to represent templates and accompanying info

=head1 SYNOPSIS

 my $R = OpenInteract::Request->instance;

 # Retreive a single template based on name

 my $tmpl = eval { $R->site_template->fetch_by_name( 
                            'user_info_box', 
                            'base_box' 
                   ) };
 die "Cannot retrieve box: $@" if ( $@ );
 print "Template contents: $tmpl->{template}\n";

 # Retrieve multiple templates from a package

 my $tmpl_list = eval { $R->site_template->fetch_group({
                            package => 'base_box',
                        }) };
 die "Cannot retrieve templates from base_box: $@" if ( $@ );
 foreach my $tmpl ( @{ $tmpl_list } ) {
    print "Template contents: $tmpl->{template}\n";
 }

=head1 DESCRIPTION

SiteTemplate objects are used throughout OpenInteract -- in fact, on
every single request multiple objects will be retrieved from the
database.

Each object represents a template which can be interpreted by the
template processing engine (normally L<Template Toolkit>) and replaced
with information from the OpenInteract environment along with data
that you decide to display on a page.

However, most of the time you will not deal directly with template
objects. The core OpenInteract modules L<OpenInteract::Template> and
L<OpenInteract::Template::Toolkit> will retrieve templates for you
based on the name and package specified.

=head1 METHODS

B<fetch_by_name( template_name, [ package_name ])>

Retrieves template(s) by name and/or package. (Just from the database
for now, not from the filesystem.)

Parameters:

 template_name ($)

   Name of the template to retrieve. Currently this name must be
   unique among all templates.

 package_name ($) (optional)

   Name of the package to use for retrieving the template. If you have
   this information you should always pass it in -- future versions of
   this module might require it to ensure a unique template namespace
   for each package.

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
